/*!
    \file    gd32e25x_fmc.h
    \brief   definitions for the FMC

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32E25X_FMC_H
#define GD32E25X_FMC_H

#include "gd32e25x.h"

/* FMC and option byte definition */
#define FMC                     FMC_BASE                    /*!< FMC register base address */
#define OB                      OB_BASE                     /*!< option byte base address */

/* registers definitions */
#define FMC_WS                  REG32((FMC) + 0x00U)        /*!< FMC wait state register */
#define FMC_KEY                 REG32((FMC) + 0x04U)        /*!< FMC unlock key register */
#define FMC_OBKEY               REG32((FMC) + 0x08U)        /*!< FMC option bytes unlock key register */
#define FMC_STAT                REG32((FMC) + 0x0CU)        /*!< FMC status register */
#define FMC_CTL                 REG32((FMC) + 0x10U)        /*!< FMC control register */
#define FMC_ADDR                REG32((FMC) + 0x14U)        /*!< FMC address register */
#define FMC_OBSTAT              REG32((FMC) + 0x1CU)        /*!< FMC option bytes status register */
#define FMC_WP                  REG32((FMC) + 0x20U)        /*!< FMC write protection register */
#define FMC_ECCCS               REG32((FMC) + 0x30U)        /*!< FMC ECC control and status register  */
#define FMC_ECCADDR             REG32((FMC) + 0x34U)        /*!< FMC ECC address register  */
#define FMC_PID                 REG32((FMC) + 0x100U)       /*!< FMC product ID register */

#define OB_SPC                  REG16((OB) + 0x00U)         /*!< option byte security protection value, user value and data*/
#define OB_USER                 REG16((OB) + 0x02U)         /*!< option byte user value*/
#define OB_DATA                 REG32((OB) + 0x04U)         /*!< option byte data value*/
#define OB_WP                   REG32((OB) + 0x08U)         /*!< option byte write protection */

/* bits definitions */
/* FMC_WS */
#define FMC_WS_WSCNT            BITS(0,2)                   /*!< wait state counter */
#define FMC_WS_PFEN             BIT(4)                      /*!< pre-fetch enable */

/* FMC_KEY */
#define FMC_KEY_KEY             BITS(0,31)                  /*!< FMC main flash unlock key bits */

/* FMC_OBKEY */
#define FMC_OBKEY_OBKEY         BITS(0,31)                  /*!< option bytes unlock key bits */

/* FMC_STAT */
#define FMC_STAT_BUSY           BIT(0)                      /*!< flash busy flag bit */
#define FMC_STAT_PGERR          BIT(2)                      /*!< flash program error flag bit */
#define FMC_STAT_PGAERR         BIT(3)                      /*!< program alignment error flag bit */
#define FMC_STAT_WPERR          BIT(4)                      /*!< flash write protection error flag bit */
#define FMC_STAT_ENDF           BIT(5)                      /*!< end of operation flag bit */
#define FMC_STAT_SRAM_PARITYENF BIT(6)                      /*!< SRAM parity check enable flag bit */
#define FMC_STAT_SRAM_ECCENF    BIT(7)                      /*!< SRAM ECC check enable flag bit */

/* FMC_CTL */
#define FMC_CTL_PG              BIT(0)                      /*!< main flash program command bit */
#define FMC_CTL_PER             BIT(1)                      /*!< main flash page erase bit */
#define FMC_CTL_MER             BIT(2)                      /*!< main flash mass erase bit */
#define FMC_CTL_OBPG            BIT(4)                      /*!< option bytes program command bit */
#define FMC_CTL_OBER            BIT(5)                      /*!< option bytes erase command bit */
#define FMC_CTL_START           BIT(6)                      /*!< send erase command to FMC bit */
#define FMC_CTL_LK              BIT(7)                      /*!< flash lock bit */
#define FMC_CTL_OBWEN           BIT(9)                      /*!< option bytes erase/program enable bit */
#define FMC_CTL_ERRIE           BIT(10)                     /*!< error interrupt enable bit */
#define FMC_CTL_ENDIE           BIT(12)                     /*!< end of operation interrupt enable bit */
#define FMC_CTL_OBRLD           BIT(13)                     /*!< option bytes reload bit */
#define FMC_CTL_RPCS            BIT(14)                     /*!< ram parity check control selection */
#define FMC_CTL_RPCD            BIT(15)                     /*!< ram parity check disable */
#define FMC_CTL_NRSTMD_VALUE    BIT(16)                     /*!< PB11 pin as NRST or GPIO */
#define FMC_CTL_NRSTMD_REG      BIT(17)                     /*!< PB11 pin is controlled by register */

/* FMC_ADDR */
#define FMC_ADDR_ADDR           BITS(0,31)                  /*!< flash command address bits */

/* FMC_OBSTAT */
#define FMC_OBSTAT_OBERR        BIT(0)                      /*!< option bytes read error bit */
#define FMC_OBSTAT_PLEVEL       BITS(1,2)                   /*!< protection level bit*/
#define FMC_OBSTAT_USER         BITS(8,15)                  /*!< option bytes user bits */
#define FMC_OBSTAT_DATA         BITS(16,31)                 /*!< option byte data bits */

/* FMC_WP */
#define FMC_WP_WP               BITS(0,15)                  /*!< store WP[15:0] of option byte block after system reset */

/* FMC_ECCCS */
#define FMC_ECCCS_ECCCOR        BIT(0)                      /*!< one bit error detected and correct flag */
#define FMC_ECCCS_ECCDET        BIT(1)                      /*!< two bit errors detect flag */
#define FMC_ECCCS_ECCDETIE      BIT(14)                     /*!< two bit errors detect interrupt enable */
#define FMC_ECCCS_ECCCORIE      BIT(15)                     /*!< one bit error detected and correct interrupt enable */
#define FMC_ECCCS_ECCINJDATA    BITS(24,25)                 /*!< flash ECC fault injection data */
#define FMC_ECCCS_ECCINJEN      BIT(31)                     /*!< flash ECC fault injection enable */

/* FMC_ECCADDR */
#define FMC_ECCADDR_ECCADDR     BITS(0,31)                  /*!< ECC error logic address */

/* FMC_PID */
#define FMC_PID_PID             BITS(0,31)                  /*!< product ID bits */

/* option bytes write protection */
#define OB_WP_0                        BIT(0)                                        /*!< erase/program protection of page 0 ~ 3   */
#define OB_WP_1                        BIT(1)                                        /*!< erase/program protection of page 4 ~ 7   */
#define OB_WP_2                        BIT(2)                                        /*!< erase/program protection of page 8 ~ 11  */
#define OB_WP_3                        BIT(3)                                        /*!< erase/program protection of page 12 ~ 15 */
#define OB_WP_4                        BIT(4)                                        /*!< erase/program protection of page 16 ~ 19 */
#define OB_WP_5                        BIT(5)                                        /*!< erase/program protection of page 20 ~ 23 */
#define OB_WP_6                        BIT(6)                                        /*!< erase/program protection of page 24 ~ 27 */
#define OB_WP_7                        BIT(7)                                        /*!< erase/program protection of page 28 ~ 31 */
#define OB_WP_8                        BIT(8)                                        /*!< erase/program protection of page 32 ~ 35 */
#define OB_WP_9                        BIT(9)                                        /*!< erase/program protection of page 36 ~ 39 */
#define OB_WP_10                       BIT(10)                                       /*!< erase/program protection of page 40 ~ 43 */
#define OB_WP_11                       BIT(11)                                       /*!< erase/program protection of page 44 ~ 47 */
#define OB_WP_12                       BIT(12)                                       /*!< erase/program protection of page 48 ~ 51 */
#define OB_WP_13                       BIT(13)                                       /*!< erase/program protection of page 52 ~ 55 */
#define OB_WP_14                       BIT(14)                                       /*!< erase/program protection of page 56 ~ 59 */
#define OB_WP_15                       BIT(15)                                       /*!< erase/program protection of page 60 ~ 63 */
#define OB_WP_ALL                      BITS(0,15)                                    /*!< erase/program protection of all pages    */

/* constants definitions */
/* fmc state */
typedef enum
{
    FMC_READY,                                              /*!< the operation has been completed */
    FMC_BUSY,                                               /*!< the operation is in progress */
    FMC_PGERR,                                              /*!< program error */
    FMC_PGAERR,                                             /*!< program alignment error */
    FMC_WPERR,                                              /*!< erase/program protection error */
    FMC_TOERR,                                              /*!< timeout error */
    FMC_OB_HSPC,                                            /*!< option byte security protection code high */
    FMC_UNDEFINEDERR                                        /*!< undefined error for function input parameter checking */
}fmc_state_enum;

/* option byte */
typedef struct
{
    uint16_t ob_spc;                                        /*!< SPC of option byte */
    uint16_t ob_user;                                       /*!< USER of option byte */
    uint32_t ob_data;                                       /*!< DATA of option byte */
    uint32_t ob_wp;                                         /*!< WP of option byte */
}option_byte_struct;

/* two 64 bits of option byte */
typedef enum
{
    DOUBLEWORD_SPC_USER_DATA = 0,                           /*!< SPC, USER, DATA of option byte */
    DOUBLEWORD_WP = 1                                       /*!< WP of option byte */
}option_byte_enum;

/* unlock key */
#define UNLOCK_KEY0                ((uint32_t)0x45670123U)  /*!< unlock key 0 */
#define UNLOCK_KEY1                ((uint32_t)0xCDEF89ABU)  /*!< unlock key 1 */

/* wait state counter value */
#define WS_WSCNT_0                 ((uint8_t)0x00U)         /*!< 0 wait state added */
#define WS_WSCNT_1                 ((uint8_t)0x01U)         /*!< 1 wait state added */
#define WS_WSCNT_2                 ((uint8_t)0x02U)         /*!< 2 wait state added */

/* read protect configure */
#define FMC_NSPC                   ((uint16_t)0x5AA5U)      /*!< no security protection */
#define FMC_LSPC                   ((uint16_t)0x44BBU)      /*!< low security protection, any value except 0xA5 or 0xCC */
#define FMC_HSPC                   ((uint16_t)0x33CCU)      /*!< high security protection */

/* option byte software/hardware free watchdog timer */
#define OB_FWDGT_HW                ((uint8_t)(~BIT(0)))     /*!< hardware free watchdog timer */
#define OB_FWDGT_SW                ((uint8_t)BIT(0))        /*!< software free watchdog timer */

/* option byte reset or not entering deep sleep mode */
#define OB_DEEPSLEEP_RST           ((uint8_t)(~BIT(1)))     /*!< generate a reset instead of entering deepsleep mode */
#define OB_DEEPSLEEP_NRST          ((uint8_t)BIT(1))        /*!< no reset when entering deepsleep mode */

/* option byte reset or not entering standby mode */
#define OB_STDBY_RST               ((uint8_t)(~BIT(2)))     /*!< generate a reset instead of entering standby mode */
#define OB_STDBY_NRST              ((uint8_t)BIT(2))        /*!< no reset when entering deepsleep mode */

/* option byte OB_BOOT0_n set */
#define OB_BOOT0_SET_1             ((uint8_t)(~BIT(3)))     /*!< BOOT0 bit is 1 */
#define OB_BOOT0_SET_0             ((uint8_t)BIT(3))        /*!< BOOT0 bit is 0 */

/* option byte OB_BOOT1_n set */
#define OB_BOOT1_SET_1             ((uint8_t)(~BIT(4)))     /*!< BOOT1 bit is 1 */
#define OB_BOOT1_SET_0             ((uint8_t)BIT(4))        /*!< BOOT1 bit is 0 */

/* option byte VDDA monitor enable/disable */
#define OB_VDDA_DISABLE            ((uint8_t)(~BIT(5)))     /*!< disable VDDA monitor */
#define OB_VDDA_ENABLE             ((uint8_t)BIT(5))        /*!< enable VDDA monitor */

/* option byte SRAM parity enable/disable */
#define OB_SRAM_PARITY_ENABLE      ((uint8_t)(~BIT(6)))     /*!< enable SRAM parity check */
#define OB_SRAM_PARITY_DISABLE     ((uint8_t)BIT(6))        /*!< disable SRAM parity check */

/* option byte NRSTMD set */
#define OB_NRSTMD_PB11             ((uint8_t)(~BIT(7)))     /*!< PB11 is GPIO*/
#define OB_NRSTMD_NRST             ((uint8_t)BIT(7))        /*!< PB11 is NRST */

/* option byte security protection level in FMC_OBSTAT register */
#define OB_OBSTAT_PLEVEL_NO        ((uint8_t)0x00U)         /*!< no security protection */
#define OB_OBSTAT_PLEVEL_LOW       ((uint8_t)0x01U)         /*!< low security protection */
#define OB_OBSTAT_PLEVEL_HIGH      ((uint8_t)0x03U)         /*!< high security protection */

/* option byte data address */
#define OB_DATA_ADDR0              ((uint32_t)0x1FFFF804U)  /*!< option byte data address 0 */
#define OB_DATA_ADDR1              ((uint32_t)0x1FFFF806U)  /*!< option byte data address 1 */

/* flash ECC error injection type */
#define FMC_NO_ECC_ERROR           ((uint8_t)0x00U)         /*!< no flash ECC error fault injection*/
#define FMC_ONE_BIT_ECC_ERROR_1    ((uint8_t)0x01U)         /*!< flash one bit ECC error fault injection */
#define FMC_ONE_BIT_ECC_ERROR_2    ((uint8_t)0x02U)         /*!< flash one bit ECC error fault injection */
#define FMC_TWO_BITS_ECC_ERROR     ((uint8_t)0x03U)         /*!< flash two bits ECC error fault injection */

/* define the FMC bit position and its register index offset */
#define FMC_REGIDX_BIT(regidx, bitpos)  (((uint32_t)(regidx) << 6) | (uint32_t)(bitpos))
#define FMC_REG_VAL(offset)             (REG32(FMC + (((uint32_t)(offset) & 0x0000FFFFU) >> 6)))
#define FMC_BIT_POS(val)                ((uint32_t)(val) & 0x0000001FU)
#define FMC_REGIDX_BIT2(regidx, bitpos, regidx2, bitpos2)   (((uint32_t)(regidx2) << 22) | (uint32_t)((bitpos2) << 16)\
                                                             | (((uint32_t)(regidx) << 6) | (uint32_t)(bitpos)))
#define FMC_REG_VAL2(offset)            (REG32(FMC + ((uint32_t)(offset) >> 22)))
#define FMC_BIT_POS2(val)               (((uint32_t)(val) & 0x001F0000U) >> 16)

/* register offset */
#define FMC_STAT_REG_OFFSET            ((uint32_t)0x0000000CU)                  /*!< STAT0 register offset */
#define FMC_CTL_REG_OFFSET             ((uint32_t)0x00000010U)                  /*!< CTL0 register offset */
#define FMC_ECCCS_REG_OFFSET           ((uint32_t)0x00000030U)                  /*!< ECCCS register offset */

/* FMC flags */
typedef enum {
    /* flags in STAT register */
    FMC_FLAG_BUSY = FMC_REGIDX_BIT(FMC_STAT_REG_OFFSET, 0U),                    /*!< flash busy flag */
    FMC_FLAG_PGERR = FMC_REGIDX_BIT(FMC_STAT_REG_OFFSET, 2U),                   /*!< flash program error flag */
    FMC_FLAG_PGAERR = FMC_REGIDX_BIT(FMC_STAT_REG_OFFSET, 3U),                  /*!< flash program alignment error flag */
    FMC_FLAG_WPERR = FMC_REGIDX_BIT(FMC_STAT_REG_OFFSET, 4U),                   /*!< flash erase/program protection error flag */
    FMC_FLAG_ENDF = FMC_REGIDX_BIT(FMC_STAT_REG_OFFSET, 5U),                    /*!< flash end of operation flag */
    FMC_FLAG_PARITYEN = FMC_REGIDX_BIT(FMC_STAT_REG_OFFSET, 6U),                /*!< SRAM parity check enable flag */
    FMC_FLAG_ECCEN = FMC_REGIDX_BIT(FMC_STAT_REG_OFFSET, 7U),                   /*!< SRAM ECC check enable flag */
    /* flags in ECCCS register */
    FMC_FLAG_ECCCOR = FMC_REGIDX_BIT(FMC_ECCCS_REG_OFFSET, 0U),                 /*!< flash single bit ECC error flag */
    FMC_FLAG_ECCDET = FMC_REGIDX_BIT(FMC_ECCCS_REG_OFFSET, 1U)                  /*!< flash double bit ECC error flag */
} fmc_flag_enum;

/* FMC interrupt flags */
typedef enum {
    /* interrupt flags in STAT0 register */
    FMC_INT_FLAG_PGERR = FMC_REGIDX_BIT2(FMC_CTL_REG_OFFSET, 10U, FMC_STAT_REG_OFFSET, 2U),         /*!< flash program error interrupt flag */
    FMC_INT_FLAG_PGAERR = FMC_REGIDX_BIT2(FMC_CTL_REG_OFFSET, 10U, FMC_STAT_REG_OFFSET, 3U),        /*!< flash program alignment error interrupt flag */
    FMC_INT_FLAG_WPERR = FMC_REGIDX_BIT2(FMC_CTL_REG_OFFSET, 10U, FMC_STAT_REG_OFFSET, 4U),         /*!< flash erase/program protection error interrupt flag */
    FMC_INT_FLAG_END = FMC_REGIDX_BIT2(FMC_CTL_REG_OFFSET, 12U, FMC_STAT_REG_OFFSET, 5U),           /*!< flash end of operation interrupt flag */
    /* interrupt flags in ECCCS register */
    FMC_INT_FLAG_ECCCOR = FMC_REGIDX_BIT2(FMC_ECCCS_REG_OFFSET, 15U, FMC_ECCCS_REG_OFFSET, 0U),     /*!< flash single bit ECC error interrupt flag */
    FMC_INT_FLAG_ECCDET = FMC_REGIDX_BIT2(FMC_ECCCS_REG_OFFSET, 14U, FMC_ECCCS_REG_OFFSET, 1U)      /*!< flash double bits ECC error interrupt flag */
} fmc_interrupt_flag_enum;

/* FMC interrupt */
typedef enum {
    /* interrupt in CTL0 register */
    FMC_INT_ERR = FMC_REGIDX_BIT(FMC_CTL_REG_OFFSET, 10U),                       /*!< FMC flash error interrupt */
    FMC_INT_END = FMC_REGIDX_BIT(FMC_CTL_REG_OFFSET, 12U),                       /*!< FMC flash end of operation interrupt */
    /* interrupt in ECCCS register */
    FMC_INT_ECCDET = FMC_REGIDX_BIT(FMC_ECCCS_REG_OFFSET, 14U),                  /*!< FMC flash double bits ECC error interrupt */
    FMC_INT_ECCCOR = FMC_REGIDX_BIT(FMC_ECCCS_REG_OFFSET, 15U)                   /*!< FMC flash single bit ECC error interrupt */
} fmc_interrupt_enum;

/* FMC time out */
#define FMC_TIMEOUT_COUNT          ((uint32_t)0x000F0000U)  /*!< count to judge of FMC timeout 0x000F0000U */

#define MAIN_FLASH_BASE_ADDRESS    ((uint32_t)0x08000000U)  /*!< main flash base address */
#define MAIN_FLASH_SIZE            ((uint32_t)0x00010000U)  /*!< main flash size */
#define MAIN_FLASH_PAGE_SIZE       ((uint32_t)0x00000400U)  /*!< main flash sub page size */
#define MAIN_FLASH_PAGE_TOTAL_NUM  ((uint32_t)0x00000040U)  /*!< main flash page total number */


/* parameter check definitions */
#ifdef FW_DEBUG_ERR_REPORT
/* check address */
#define NOT_FMC_VALID_ADDRESS(address)                            (((address) < MAIN_FLASH_BASE_ADDRESS) || ((address) > (MAIN_FLASH_BASE_ADDRESS + MAIN_FLASH_SIZE)))
#endif /* FW_DEBUG_ERR_REPORT */


/* function declarations */
/* FMC main memory programming functions */
/* unlock the main FMC operation */
void fmc_unlock(void);
/* lock the main FMC operation */
void fmc_lock(void);
/* set the wait state counter value */
void fmc_wscnt_set(uint8_t wscnt);

/* pre-fetch enable */
void fmc_prefetch_enable(void);
/* pre-fetch disable */
void fmc_prefetch_disable(void);
/* FMC erase page */
fmc_state_enum fmc_page_erase(uint32_t page_number);
/* FMC erase whole chip */
fmc_state_enum fmc_mass_erase(void);
/* FMC program a double word at the corresponding address */
fmc_state_enum fmc_doubleword_program(uint32_t address, uint64_t data);
/* SRAM parity check enable, SRAM ECC disable */
void fmc_sramparity_enable(void);
/* SRAM parity check disable, SRAM ECC enable */
void fmc_sramparity_disable(void);
/* PB11 pin as NRST */
void fmc_nrstmdvalue_enable(void);
/* PB11 pin as GPIO */
void fmc_nrstmdvalue_disable(void);	
/* FMC option bytes programming functions */
/* unlock the option byte operation */
void ob_unlock(void);
/* lock the option byte operation */
void ob_lock(void);
/* reload the option byte and generate a system reset */
void ob_reset(void);
/* erase option byte */
fmc_state_enum ob_erase(void);
/* enable option byte write protection (OB_WP) */
fmc_state_enum ob_write_protection_enable(uint16_t ob_wp);
/* configure read out protect */
fmc_state_enum ob_security_protection_config(uint16_t ob_spc);
/* write the FMC option byte user */
fmc_state_enum ob_user_write(uint8_t ob_user);
/* write the FMC option byte data */
fmc_state_enum ob_data_program(uint16_t data);
/* get the FMC option byte OB_USER */
uint8_t ob_user_get(void);
/* get the FMC option byte OB_DATA */
uint16_t ob_data_get(void);
/* get the FMC option byte write protection */
uint16_t ob_write_protection_get(void);
/* get the value of FMC option byte security protection level (PLEVEL) in FMC_OBSTAT register */
uint32_t ob_obstat_plevel_get(void);

/* FMC ECC related functions */
/* get the FMC ECC error address in register FMC_ECCADDR */
uint32_t fmc_ecc_error_address_get(void);
/* FMC ECC error injection enable */
void fmc_ecc_error_injection_enable(void);
/* FMC ECC error injection disable */
void fmc_ecc_error_injection_disable(void);
/* FMC ECC error injection */
void fmc_ecc_error_injection(uint8_t error_typ);
    
/* FMC interrupts and flags management functions */
/* enable FMC interrupt */
void fmc_interrupt_enable(fmc_interrupt_enum interrupt);
/* disable FMC interrupt */
void fmc_interrupt_disable(fmc_interrupt_enum interrupt);
/* get flag set or reset */
FlagStatus fmc_flag_get(fmc_flag_enum flag);
/* clear the FMC pending flag */
void fmc_flag_clear(fmc_flag_enum flag);
/* get interrupt flag set or reset */
FlagStatus fmc_interrupt_flag_get(fmc_interrupt_flag_enum int_flag);
/* clear the FMC interrupt pending flag by writing 1 */
void fmc_interrupt_flag_clear(fmc_interrupt_flag_enum int_flag);
/* return the FMC state */
fmc_state_enum fmc_state_get(void);
/* check FMC ready or not */
fmc_state_enum fmc_ready_wait(uint32_t timeout);

#endif /* GD32E25X_FMC_H */
