/*!
    \file    gd32e25x_cla.h
    \brief   definitions for the CLA

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32E25X_CLA_H
#define GD32E25X_CLA_H

#include "gd32e25x.h"

/*CLA definitions */
#define CLA                                CLA_BASE

/* registers definitions */
#define CLA_GCTL                           REG32((CLA) + 0x00000000U)                  /*!< CLA global control register */
#define CLA_INTE                           REG32((CLA) + 0x00000004U)                  /*!< CLA interrupt flag enable register */
#define CLA_INTF                           REG32((CLA) + 0x00000008U)                  /*!< CLA interrupt flag register */
#define CLA_STAT                           REG32((CLA) + 0x0000000CU)                  /*!< CLA status register */
#define CLA_SIGS(CLAx)                     REG32((CLA) + 0x10U + ((CLAx) * 0x0CU))     /*!< CLA signal selection register */
#define CLA_LCUCTL(CLAx)                   REG32((CLA) + 0x14U + ((CLAx) * 0x0CU))     /*!< CLA LCU control register */
#define CLA_CTL(CLAx)                      REG32((CLA) + 0x18U + ((CLAx) * 0x0CU))     /*!< CLA control register */

/* bits definitions */
/* CLA_GCTL */
#define CLA_GCTL_CLA0EN                    BIT(0)                                      /*!< CLA0 unit enable */
#define CLA_GCTL_CLA1EN                    BIT(1)                                      /*!< CLA1 unit enable */
#define CLA_GCTL_CLA2EN                    BIT(2)                                      /*!< CLA2 unit enable */
#define CLA_GCTL_CLA3EN                    BIT(3)                                      /*!< CLA3 unit enable */
                                                                                       
/* CLA_INTE */                                                                         
#define CLA_INTE_CLA0NIE                   BIT(0)                                      /*!< CLA0 uint negedge interrupt enable */
#define CLA_INTE_CLA0PIE                   BIT(1)                                      /*!< CLA0 uint posedge interrupt enable */
#define CLA_INTE_CLA1NIE                   BIT(2)                                      /*!< CLA1 uint negedge interrupt enable */
#define CLA_INTE_CLA1PIE                   BIT(3)                                      /*!< CLA1 uint posedge interrupt enable */
#define CLA_INTE_CLA2NIE                   BIT(4)                                      /*!< CLA2 uint negedge interrupt enable */
#define CLA_INTE_CLA2PIE                   BIT(5)                                      /*!< CLA2 uint posedge interrupt enable */
#define CLA_INTE_CLA3NIE                   BIT(6)                                      /*!< CLA3 uint negedge interrupt enable */
#define CLA_INTE_CLA3PIE                   BIT(7)                                      /*!< CLA3 uint posedge interrupt enable */
                                                                                       
/* CLA_INTF */                                                                         
#define CLA_INTF_CLA0NF                    BIT(0)                                      /*!< CLA0 uint negedge flag */
#define CLA_INTF_CLA0PF                    BIT(1)                                      /*!< CLA0 uint posedge flag */
#define CLA_INTF_CLA1NF                    BIT(2)                                      /*!< CLA1 uint negedge flag */
#define CLA_INTF_CLA1PF                    BIT(3)                                      /*!< CLA1 uint posedge flag */
#define CLA_INTF_CLA2NF                    BIT(4)                                      /*!< CLA2 uint negedge flag */
#define CLA_INTF_CLA2PF                    BIT(5)                                      /*!< CLA2 uint posedge flag */
#define CLA_INTF_CLA3NF                    BIT(6)                                      /*!< CLA3 uint negedge flag */
#define CLA_INTF_CLA3PF                    BIT(7)                                      /*!< CLA3 uint posedge flag */
                                                                                       
/* CLA_STAT */                                                                         
#define CLA_STAT_CLA0OUT                   BIT(0)                                      /*!< CLA0 unit output state */
#define CLA_STAT_CLA1OUT                   BIT(1)                                      /*!< CLA1 unit output state */
#define CLA_STAT_CLA2OUT                   BIT(2)                                      /*!< CLA2 unit output state */
#define CLA_STAT_CLA3OUT                   BIT(3)                                      /*!< CLA3 unit output state */
                                                                                       
/* CLAx_SIGSS */                                                                       
#define CLA_SIGS_SIGS1                     BITS(0,3)                                   /*!< signal selector 1 input selection */
#define CLA_SIGS_SIGS0                     BITS(4,7)                                   /*!< signal selector 0 input selection */
                                                                                       
/* CLAx_LCUCTL */                                                                      
#define CLA_LCU                            BITS(0,7)                                   /*!< LCU control */
                                                                                       
/* CLAx_CTL */                                                                         
#define CLA_CTL_CSEL                       BITS(0,1)                                   /*!< clock selection */
#define CLA_CTL_CPOL                       BIT(2)                                      /*!< clock polarity*/
#define CLA_CTL_FFRST                      BIT(3)                                      /*!< Flip- flop output reset */
#define CLA_CTL_OEN                        BIT(6)                                      /*!< output enable */
#define CLA_CTL_OSEL                       BIT(7)                                      /*!< output selection */
#define CLA_CTL_SWB1                       BIT(8)                                      /*!< SIG1 Software input value */
#define CLA_CTL_SWB0                       BIT(9)                                      /*!< SIG0 Software input value */
#define CLA_CTL_SWBEN1                     BIT(10)                                     /*!< SIG1 Software input enable */
#define CLA_CTL_SWBEN0                     BIT(11)                                     /*!< SIG0 Software input enable */
                                                                                       
/* constants definitions */                                                            
/* output of SIGS  */
typedef enum
{
    SIGS_OUTPUT_SWB,
    SIGS_OUTPUT_SELECT_INPUT
}cla_sigs_output_enum;

/* CLA flags */                                                                        
typedef enum                                                                           
{                                                                                      
    /* flags in INTF register */                                                       
    CLA_FLAG_CLA0NF,                                                                   /*!< CLA0 unit negedge flag */
    CLA_FLAG_CLA0PF,                                                                   /*!< CLA0 unit posedge flag */
    CLA_FLAG_CLA1NF,                                                                   /*!< CLA1 unit negedge flag */
    CLA_FLAG_CLA1PF,                                                                   /*!< CLA1 unit posedge flag */
    CLA_FLAG_CLA2NF,                                                                   /*!< CLA2 unit negedge flag */
    CLA_FLAG_CLA2PF,                                                                   /*!< CLA2 unit posedge flag */
    CLA_FLAG_CLA3NF,                                                                   /*!< CLA3 unit negedge flag */
    CLA_FLAG_CLA3PF                                                                    /*!< CLA3 unit posedge flag */
}cla_flag_enum;                                                                        
                                                                                       
/* CLA interrupt flags */                                                              
typedef enum                                                                           
{                                                                                      
    /* interrupt flags in INTF register */                                             
    CLA_INT_FLAG_CLA0NF,                                                               /*!< CLA0 unit negedge interrupt flag */
    CLA_INT_FLAG_CLA0PF,                                                               /*!< CLA0 unit posedge interrupt flag */
    CLA_INT_FLAG_CLA1NF,                                                               /*!< CLA1 unit negedge interrupt flag */
    CLA_INT_FLAG_CLA1PF,                                                               /*!< CLA1 unit posedge interrupt flag */
    CLA_INT_FLAG_CLA2NF,                                                               /*!< CLA2 unit negedge interrupt flag */
    CLA_INT_FLAG_CLA2PF,                                                               /*!< CLA2 unit posedge interrupt flag */
    CLA_INT_FLAG_CLA3NF,                                                               /*!< CLA3 unit negedge interrupt flag */
    CLA_INT_FLAG_CLA3PF                                                                /*!< CLA3 unit posedge interrupt flag */
}cla_interrupt_flag_enum;                                                              

/* CLA units */
typedef enum
{
    CLA0,
    CLA1,
    CLA2,
    CLA3
}cla_enum;

/* CLA signal selector */
typedef enum
{
    SIGS0,
    SIGS1
}cla_sigs_enum;

/* output state of CLAx unit */
typedef enum
{
    CLA_OUTPUT_LOW,
    CLA_OUTPUT_HIGH
}cla_outputstatus_enum;

typedef FlagStatus out_value;

/* negedge interrupt */
#define CLA0_INT_NEGEDGE                          CLA_INTE_CLA0NIE                   /*!< CLA0 unit negedge interrupt */
#define CLA1_INT_NEGEDGE                          CLA_INTE_CLA1NIE                   /*!< CLA1 unit negedge interrupt */
#define CLA2_INT_NEGEDGE                          CLA_INTE_CLA2NIE                   /*!< CLA2 unit negedge interrupt */
#define CLA3_INT_NEGEDGE                          CLA_INTE_CLA3NIE                   /*!< CLA3 unit negedge interrupt */

/* posedge interrupt enable */
#define CLA0_INT_POSEDGE                          CLA_INTE_CLA0PIE                   /*!< CLA0 unit  posedge interrupt */
#define CLA1_INT_POSEDGE                          CLA_INTE_CLA1PIE                   /*!< CLA1 unit  posedge interrupt */
#define CLA2_INT_POSEDGE                          CLA_INTE_CLA2PIE                   /*!< CLA2 unit  posedge interrupt */
#define CLA3_INT_POSEDGE                          CLA_INTE_CLA3PIE                   /*!< CLA3 unit  posedge interrupt */

/* signal selector 0 input selection */
#define CLA_SIGS0(regval)                 (BITS(4,7) & ((uint32_t)(regval) << 4))
/* signal selector 1 input selection */
#define CLA_SIGS1(regval)                 (BITS(0,3) & ((uint32_t)(regval) << 0))
typedef enum
{
    /* SIGS0 input signal */
    CLA0SIGS0_CLA0_ASYNC_OUT = CLA_SIGS0(0),                        /*!< the input of CLA0SIGS0 is CLA0_ASYNC_OUT */
    CLA1SIGS0_CLA0_ASYNC_OUT = CLA_SIGS0(0),                        /*!< the input of CLA1SIGS0 is CLA0_ASYNC_OUT */
    CLA2SIGS0_CLA0_ASYNC_OUT = CLA_SIGS0(0),                        /*!< the input of CLA2SIGS0 is CLA0_ASYNC_OUT */
    CLA3SIGS0_CLA0_ASYNC_OUT = CLA_SIGS0(0),                        /*!< the input of CLA3SIGS0 is CLA0_ASYNC_OUT */
    CLA0SIGS0_CLA1_ASYNC_OUT = CLA_SIGS0(1),                        /*!< the input of CLA0SIGS0 is CLA1_ASYNC_OUT */
    CLA1SIGS0_CLA1_ASYNC_OUT = CLA_SIGS0(1),                        /*!< the input of CLA1SIGS0 is CLA1_ASYNC_OUT */
    CLA2SIGS0_CLA1_ASYNC_OUT = CLA_SIGS0(1),                        /*!< the input of CLA2SIGS0 is CLA1_ASYNC_OUT */
    CLA3SIGS0_CLA1_ASYNC_OUT = CLA_SIGS0(1),                        /*!< the input of CLA3SIGS0 is CLA1_ASYNC_OUT */
    CLA0SIGS0_CLA2_ASYNC_OUT = CLA_SIGS0(2),                        /*!< the input of CLA0SIGS0 is CLA2_ASYNC_OUT */
    CLA1SIGS0_CLA2_ASYNC_OUT = CLA_SIGS0(2),                        /*!< the input of CLA1SIGS0 is CLA2_ASYNC_OUT */
    CLA2SIGS0_CLA2_ASYNC_OUT = CLA_SIGS0(2),                        /*!< the input of CLA2SIGS0 is CLA2_ASYNC_OUT */
    CLA3SIGS0_CLA2_ASYNC_OUT = CLA_SIGS0(2),                        /*!< the input of CLA3SIGS0 is CLA2_ASYNC_OUT */
    CLA0SIGS0_CLA3_ASYNC_OUT = CLA_SIGS0(3),                        /*!< the input of CLA0SIGS0 is CLA3_ASYNC_OUT */
    CLA1SIGS0_CLA3_ASYNC_OUT = CLA_SIGS0(3),                        /*!< the input of CLA1SIGS0 is CLA3_ASYNC_OUT */
    CLA2SIGS0_CLA3_ASYNC_OUT = CLA_SIGS0(3),                        /*!< the input of CLA2SIGS0 is CLA3_ASYNC_OUT */
    CLA3SIGS0_CLA3_ASYNC_OUT = CLA_SIGS0(3),                        /*!< the input of CLA3SIGS0 is CLA3_ASYNC_OUT */
    CLA0SIGS0_TIMER1_TRGO = CLA_SIGS0(4),                           /*!< the input of CLA0SIGS0 is TIMER1_TRGO */
    CLA1SIGS0_TIMER2_TRGO = CLA_SIGS0(4),                           /*!< the input of CLA1SIGS0 is TIMER2_TRGO */
    CLA2SIGS0_TIMER5_TRGO = CLA_SIGS0(4),                           /*!< the input of CLA2SIGS0 is TIMER5_TRGO */
    CLA3SIGS0_TIMER6_TRGO = CLA_SIGS0(4),                           /*!< the input of CLA3SIGS0 is TIMER6_TRGO */
    CLA0SIGS0_TIMER0_CH0 = CLA_SIGS0(5),                            /*!< the input of CLA0SIGS0 is TIMER0_CH0 */
    CLA1SIGS0_TIMER0_CH0 = CLA_SIGS0(5),                            /*!< the input of CLA1SIGS0 is TIMER0_CH0 */
    CLA2SIGS0_TIMER0_CH1 = CLA_SIGS0(5),                            /*!< the input of CLA2SIGS0 is TIMER0_CH1 */
    CLA3SIGS0_TIMER0_CH2 = CLA_SIGS0(5),                            /*!< the input of CLA3SIGS0 is TIMER0_CH2 */
    CLA0SIGS0_TIMER0_CH1 = CLA_SIGS0(6),                            /*!< the input of CLA0SIGS0 is TIMER0_CH1 */
    CLA1SIGS0_TIMER0_CH3 = CLA_SIGS0(6),                            /*!< the input of CLA0SIGS0 is TIMER0_CH3 */
    CLA2SIGS0_TIMER0_CH3 = CLA_SIGS0(6),                            /*!< the input of CLA0SIGS0 is TIMER0_CH3 */
    CLA3SIGS0_TIMER1_CH0 = CLA_SIGS0(6),                            /*!< the input of CLA0SIGS0 is TIMER1_CH0 */
    CLA0SIGS0_TIMER0_CH2 = CLA_SIGS0(7),                            /*!< the input of CLA0SIGS0 is TIMER0_CH2 */
    CLA1SIGS0_TIMER1_CH0 = CLA_SIGS0(7),                            /*!< the input of CLA1SIGS0 is TIMER1_CH0 */
    CLA2SIGS0_TIMER1_CH1 = CLA_SIGS0(7),                            /*!< the input of CLA2SIGS0 is TIMER1_CH1 */
    CLA3SIGS0_TIMER1_CH1 = CLA_SIGS0(7),                            /*!< the input of CLA3SIGS0 is TIMER1_CH1 */
    CLA0SIGS0_CLAIN0 = CLA_SIGS0(8),                                /*!< the input of CLA0SIGS0 is CLAIN0(PA15), */
    CLA1SIGS0_CLAIN4 = CLA_SIGS0(8),                                /*!< the input of CLA1SIGS0 is CLAIN4(PB6), */
    CLA2SIGS0_CLAIN0 = CLA_SIGS0(8),                                /*!< the input of CLA2SIGS0 is CLAIN0(PA15), */
    CLA3SIGS0_CLAIN2 = CLA_SIGS0(8),                                /*!< the input of CLA3SIGS0 is CLAIN2(PB4), */
    CLA0SIGS0_CLAIN2 = CLA_SIGS0(9),                                /*!< the input of CLA0SIGS0 is CLAIN2(PB4), */
    CLA1SIGS0_CLAIN5 = CLA_SIGS0(9),                                /*!< the input of CLA1SIGS0 is CLAIN5(PB7), */
    CLA2SIGS0_CLAIN1 = CLA_SIGS0(9),                                /*!< the input of CLA2SIGS0 is CLAIN1(PB3), */
    CLA3SIGS0_CLAIN3 = CLA_SIGS0(9),                                /*!< the input of CLA3SIGS0 is CLAIN3(PB5), */
    CLA0SIGS0_CLAIN4 = CLA_SIGS0(10),                               /*!< the input of CLA0SIGS0 is CLAIN4(PB6), */
    CLA1SIGS0_CLAIN8 = CLA_SIGS0(10),                               /*!< the input of CLA1SIGS0 is CLAIN8(PB0), */
    CLA2SIGS0_CLAIN8 = CLA_SIGS0(10),                               /*!< the input of CLA2SIGS0 is CLAIN8(PB0), */
    CLA3SIGS0_CLAIN6 = CLA_SIGS0(10),                               /*!< the input of CLA3SIGS0 is CLAIN6(PB8), */
    CLA0SIGS0_CLAIN6 = CLA_SIGS0(11),                               /*!< the input of CLA0SIGS0 is CLAIN6(PB8), */
    CLA1SIGS0_CLAIN10 = CLA_SIGS0(11),                              /*!< the input of CLA1SIGS0 is CLAIN10(PB2), */
    CLA2SIGS0_CLAIN9 = CLA_SIGS0(11),                               /*!< the input of CLA2SIGS0 is CLAIN9(PB1), */
    CLA3SIGS0_CLAIN7 = CLA_SIGS0(11),                               /*!< the input of CLA3SIGS0 is CLAIN7(PB9), */
    CLA0SIGS0_CLAIN8 = CLA_SIGS0(12),                               /*!< the input of CLA0SIGS0 is CLAIN8(PB0), */
    CLA1SIGS0_CLAIN12 = CLA_SIGS0(12),                              /*!< the input of CLA1SIGS0 is CLAIN12(PA9), */
    CLA2SIGS0_CLAIN14 = CLA_SIGS0(12),                              /*!< the input of CLA2SIGS0 is CLAIN14(PA11), */
    CLA3SIGS0_CLAIN10 = CLA_SIGS0(12),                              /*!< the input of CLA3SIGS0 is CLAIN10(PB2), */
    CLA0SIGS0_CLAIN10 = CLA_SIGS0(13),                              /*!< the input of CLA0SIGS0 is CLAIN10(PB2), */
    CLA1SIGS0_CLAIN13 = CLA_SIGS0(13),                              /*!< the input of CLA1SIGS0 is CLAIN13(PA10), */
    CLA2SIGS0_CLAIN15 = CLA_SIGS0(13),                              /*!< the input of CLA2SIGS0 is CLAIN15(PA12), */
    CLA3SIGS0_CLAIN11 = CLA_SIGS0(13),                              /*!< the input of CLA3SIGS0 is CLAIN11(PA8), */
    CLA0SIGS0_CLAIN12 = CLA_SIGS0(14),                              /*!< the input of CLA0SIGS0 is CLAIN12(PA9), */
    CLA1SIGS0_CLAIN16 = CLA_SIGS0(14),                              /*!< the input of CLA1SIGS0 is CLAIN16(PF0), */
    CLA2SIGS0_CLAIN16 = CLA_SIGS0(14),                              /*!< the input of CLA2SIGS0 is CLAIN16(PF0), */
    CLA3SIGS0_CLAIN18 = CLA_SIGS0(14),                              /*!< the input of CLA3SIGS0 is CLAIN18(PA0), */
    CLA0SIGS0_CLAIN14 = CLA_SIGS0(15),                              /*!< the input of CLA0SIGS0 is CLAIN14(PA11), */
    CLA1SIGS0_CLAIN18 = CLA_SIGS0(15),                              /*!< the input of CLA1SIGS0 is CLAIN18(PA0), */
    CLA2SIGS0_CLAIN17 = CLA_SIGS0(15),                              /*!< the input of CLA2SIGS0 is CLAIN17(PF1), */
    CLA3SIGS0_CLAIN19 = CLA_SIGS0(15),                              /*!< the input of CLA3SIGS0 is CLAIN19(PA1) */
    CLAxSIGS0_SOFTWARE = CLA_CTL_SWBEN0,                            /*!< the input of CLA0SIGS0 is software */
    /* SIGS1 input signal */
    CLA0SIGS1_CLA0_ASYNC_OUT = CLA_SIGS1(0),                        /*!< the input of CLA0SIGS1 is CLA0_ASYNC_OUT */
    CLA1SIGS1_CLA0_ASYNC_OUT = CLA_SIGS1(0),                        /*!< the input of CLA1SIGS1 is CLA0_ASYNC_OUT */
    CLA2SIGS1_CLA0_ASYNC_OUT = CLA_SIGS1(0),                        /*!< the input of CLA2SIGS1 is CLA0_ASYNC_OUT */
    CLA3SIGS1_CLA0_ASYNC_OUT = CLA_SIGS1(0),                        /*!< the input of CLA3SIGS1 is CLA0_ASYNC_OUT */
    CLA0SIGS1_CLA1_ASYNC_OUT = CLA_SIGS1(1),                        /*!< the input of CLA0SIGS1 is CLA1_ASYNC_OUT */
    CLA1SIGS1_CLA1_ASYNC_OUT = CLA_SIGS1(1),                        /*!< the input of CLA1SIGS1 is CLA1_ASYNC_OUT */
    CLA2SIGS1_CLA1_ASYNC_OUT = CLA_SIGS1(1),                        /*!< the input of CLA2SIGS1 is CLA1_ASYNC_OUT */
    CLA3SIGS1_CLA1_ASYNC_OUT = CLA_SIGS1(1),                        /*!< the input of CLA3SIGS1 is CLA1_ASYNC_OUT */
    CLA0SIGS1_CLA2_ASYNC_OUT = CLA_SIGS1(2),                        /*!< the input of CLA0SIGS1 is CLA2_ASYNC_OUT */
    CLA1SIGS1_CLA2_ASYNC_OUT = CLA_SIGS1(2),                        /*!< the input of CLA1SIGS1 is CLA2_ASYNC_OUT */
    CLA2SIGS1_CLA2_ASYNC_OUT = CLA_SIGS1(2),                        /*!< the input of CLA2SIGS1 is CLA2_ASYNC_OUT */
    CLA3SIGS1_CLA2_ASYNC_OUT = CLA_SIGS1(2),                        /*!< the input of CLA3SIGS1 is CLA2_ASYNC_OUT */
    CLA0SIGS1_CLA3_ASYNC_OUT = CLA_SIGS1(3),                        /*!< the input of CLA0SIGS1 is CLA3_ASYNC_OUT */
    CLA1SIGS1_CLA3_ASYNC_OUT = CLA_SIGS1(3),                        /*!< the input of CLA1SIGS1 is CLA3_ASYNC_OUT */
    CLA2SIGS1_CLA3_ASYNC_OUT = CLA_SIGS1(3),                        /*!< the input of CLA2SIGS1 is CLA3_ASYNC_OUT */
    CLA3SIGS1_CLA3_ASYNC_OUT = CLA_SIGS1(3),                        /*!< the input of CLA3SIGS1 is CLA3_ASYNC_OUT */
    CLA0SIGS1_ADC_CONV = CLA_SIGS1(4),                              /*!< the input of CLA0SIGS1 is ADC_CONV */
    CLA1SIGS1_ADC_CONV = CLA_SIGS1(4),                              /*!< the input of CLA1SIGS1 is ADC_CONV */
    CLA2SIGS1_ADC_CONV = CLA_SIGS1(4),                              /*!< the input of CLA2SIGS1 is ADC_CONV */
    CLA3SIGS1_ADC_CONV = CLA_SIGS1(4),                              /*!< the input of CLA3SIGS1 is ADC_CONV */
    CLA0SIGS1_TIMER0_CH3 = CLA_SIGS1(5),                            /*!< the input of CLA0SIGS1 is TIMER0_CH3 */
    CLA1SIGS1_TIMER0_CH1 = CLA_SIGS1(5),                            /*!< the input of CLA1SIGS1 is TIMER0_CH1 */
    CLA2SIGS1_TIMER0_CH0 = CLA_SIGS1(5),                            /*!< the input of CLA2SIGS1 is TIMER0_CH0 */
    CLA3SIGS1_TIMER0_CH0 = CLA_SIGS1(5),                            /*!< the input of CLA3SIGS1 is TIMER0_CH0 */
    CLA0SIGS1_TIMER1_CH0 = CLA_SIGS1(6),                            /*!< the input of CLA0SIGS1 is TIMER1_CH0 */
    CLA1SIGS1_TIMER0_CH2 = CLA_SIGS1(6),                            /*!< the input of CLA1SIGS1 is TIMER0_CH2 */
    CLA2SIGS1_TIMER0_CH2 = CLA_SIGS1(6),                            /*!< the input of CLA2SIGS1 is TIMER0_CH2 */
    CLA3SIGS1_TIMER0_CH1 = CLA_SIGS1(6),                            /*!< the input of CLA3SIGS1 is TIMER0_CH1 */
    CLA0SIGS1_TIMER1_CH1 = CLA_SIGS1(7),                            /*!< the input of CLA0SIGS1 is TIMER2_CH1 */
    CLA1SIGS1_TIMER1_CH1 = CLA_SIGS1(7),                            /*!< the input of CLA1SIGS1 is TIMER2_CH1 */
    CLA2SIGS1_TIMER1_CH0 = CLA_SIGS1(7),                            /*!< the input of CLA2SIGS1 is TIMER1_CH0 */
    CLA3SIGS1_TIMER0_CH3 = CLA_SIGS1(7),                            /*!< the input of CLA3SIGS1 is TIMER0_CH3 */
    CLA0SIGS1_CLAIN1 = CLA_SIGS1(8),                                /*!< the input of CLA0SIGS1 is CLAIN1(PB3), */
    CLA1SIGS1_CLAIN6 = CLA_SIGS1(8),                                /*!< the input of CLA1SIGS1 is CLAIN6(PB8), */
    CLA2SIGS1_CLAIN2 = CLA_SIGS1(8),                                /*!< the input of CLA2SIGS1 is CLAIN2(PB4), */
    CLA3SIGS1_CLAIN0 = CLA_SIGS1(8),                                /*!< the input of CLA3SIGS1 is CLAIN0(PA15), */
    CLA0SIGS1_CLAIN3 = CLA_SIGS1(9),                                /*!< the input of CLA0SIGS1 is CLAIN3(PB5), */
    CLA1SIGS1_CLAIN7 = CLA_SIGS1(9),                                /*!< the input of CLA1SIGS1 is CLAIN7(PB9), */
    CLA2SIGS1_CLAIN3 = CLA_SIGS1(9),                                /*!< the input of CLA2SIGS1 is CLAIN3(PB5), */
    CLA3SIGS1_CLAIN1 = CLA_SIGS1(9),                                /*!< the input of CLA3SIGS1 is CLAIN1(PB3), */
    CLA0SIGS1_CLAIN5 = CLA_SIGS1(10),                               /*!< the input of CLA0SIGS1 is CLAIN5(PB7), */
    CLA1SIGS1_CLAIN9 = CLA_SIGS1(10),                               /*!< the input of CLA1SIGS1 is CLAIN9(PB1), */
    CLA2SIGS1_CLAIN10 = CLA_SIGS1(10),                              /*!< the input of CLA2SIGS1 is CLAIN10(PB2), */
    CLA3SIGS1_CLAIN4 = CLA_SIGS1(10),                               /*!< the input of CLA3SIGS1 is CLAIN4(PB6), */
    CLA0SIGS1_CLAIN7 = CLA_SIGS1(11),                               /*!< the input of CLA0SIGS1 is CLAIN6(PB8), */
    CLA1SIGS1_CLAIN11 = CLA_SIGS1(11),                              /*!< the input of CLA1SIGS1 is CLAIN11(PA8), */
    CLA2SIGS1_CLAIN11 = CLA_SIGS1(11),                              /*!< the input of CLA2SIGS1 is CLAIN11(PA8), */
    CLA3SIGS1_CLAIN5 = CLA_SIGS1(11),                               /*!< the input of CLA3SIGS1 is CLAIN5(PB7), */
    CLA0SIGS1_CLAIN9 = CLA_SIGS1(12),                               /*!< the input of CLA0SIGS1 is CLAIN9(PB1), */
    CLA1SIGS1_CLAIN14 = CLA_SIGS1(12),                              /*!< the input of CLA1SIGS1 is CLAIN14(PA11), */
    CLA2SIGS1_CLAIN12 = CLA_SIGS1(12),                              /*!< the input of CLA2SIGS1 is CLAIN12(PA9), */
    CLA3SIGS1_CLAIN8 = CLA_SIGS1(12),                               /*!< the input of CLA3SIGS1 is CLAIN8(PB0), */
    CLA0SIGS1_CLAIN11 = CLA_SIGS1(13),                              /*!< the input of CLA0SIGS1 is CLAIN11(PA8), */
    CLA1SIGS1_CLAIN15 = CLA_SIGS1(13),                              /*!< the input of CLA1SIGS1 is CLAIN15(PA12), */
    CLA2SIGS1_CLAIN13 = CLA_SIGS1(13),                              /*!< the input of CLA2SIGS1 is CLAIN13(PA10), */
    CLA3SIGS1_CLAIN9 = CLA_SIGS1(13),                               /*!< the input of CLA3SIGS1 is CLAIN9(PB1), */
    CLA0SIGS1_CLAIN13 = CLA_SIGS1(14),                              /*!< the input of CLA0SIGS1 is CLAIN13(PA10), */
    CLA1SIGS1_CLAIN17 = CLA_SIGS1(14),                              /*!< the input of CLA1SIGS1 is CLAIN17(PF1), */
    CLA2SIGS1_CLAIN18 = CLA_SIGS1(14),                              /*!< the input of CLA2SIGS1 is CLAIN18(PA0), */
    CLA3SIGS1_CLAIN16 = CLA_SIGS1(14),                              /*!< the input of CLA3SIGS1 is CLAIN16(PF0), */
    CLA0SIGS1_CLAIN15 = CLA_SIGS1(15),                              /*!< the input of CLA0SIGS1 is CLAIN15(PA12), */
    CLA1SIGS1_CLAIN19 = CLA_SIGS1(15),                              /*!< the input of CLA1SIGS1 is CLAIN19(PA1), */
    CLA2SIGS1_CLAIN19 = CLA_SIGS1(15),                              /*!< the input of CLA2SIGS1 is CLAIN19(PA1), */
    CLA3SIGS1_CLAIN17 = CLA_SIGS1(15),                              /*!< the input of CLA3SIGS1 is CLAIN17(PF1) */
    CLAxSIGS1_SOFTWARE = CLA_CTL_SWBEN1                             /*!< the input of CLA0SIGS1 is software */
}sigs_input_enum;

/* LCU control */
#define LCUCTL(regval)                   (BITS(0,7) & ((uint32_t)(regval) << 0U))

/* CLA output selection */
#define FLIP_FLOP_OUTPUT                 ((uint32_t )0x00000000U)            /*!< flip-flop output is selected as CLAx output */
#define LCU_RESULT                       CLA_CTL_OSEL                       /*!< LCU result is selected as CLAx output */

/* flip-flop clock polarity selection */
#define CLA_CLOCKPOLARITY_POSEDGE        ((uint32_t)0x00000000U)            /*!< the clock polarity of flip-flop is posedge */
#define CLA_CLOCKPOLARITY_NEGEDGE        CLA_CTL_CPOL                       /*!< the clock polarity of flip-flop is negedge */

/* flip-flop clock source selection */
#define PRE_CLA_LCU_RESULT               ((uint32_t)0x00000000U)            /*!< the LCU result of the previous CLA units */
#define SIGS0_OUTPUT                     ((uint32_t)0x00000001U)            /*!< the signal selector output of SIGS0 */
#define HCLK                             ((uint32_t)0x00000002U)            /*!< HCLK */
#define TIMER_TRGO                       ((uint32_t)0x00000003U)            /*!< TIMER_TRGO */

/* function declarations */
/* CLA initialization and configuration functions */
/* reset CLA */
void cla_deinit(void);
/* enable CLA */
void cla_enable(cla_enum cla_periph);
/* disable CLA */
void cla_disable(cla_enum cla_periph);
/* get CLA output state */
cla_outputstatus_enum cla_output_state_get(cla_enum cla_periph);
/* configure signal selector input */
void cla_sigs_input_config(cla_enum cla_periph, cla_sigs_enum sigs, sigs_input_enum input);
/* configure CLA LCU control register value */
void cla_lcu_control_config(cla_enum cla_periph, uint8_t lcuctl_value);
/* configure CLA output */
void cla_output_config(cla_enum cla_periph, uint32_t output);
/* enable CLA output */
void cla_output_enable(cla_enum cla_periph);
/* disable CLA output */
void cla_output_disable(cla_enum cla_periph);
/* configure multiplexer software output */
void cla_sigs_software_output_set(cla_enum cla_periph, cla_sigs_enum sigs, cla_sigs_output_enum sigs_output);

/* flip-flop configuration */
/* reset the flip-flop output asynchronously */
void cla_flip_flop_output_reset(cla_enum cla_periph);
/* configure clock polarity of flip-flop */
void cla_flip_flop_clockpolarity_config(cla_enum cla_periph, uint32_t polarity);
/* configure clock source of flip-flop */
void cla_flip_flop_clocksource_config(cla_enum cla_periph, uint32_t clock_source);

/* flag and interrupt functions */
/* check CLA flag is set or not */
FlagStatus cla_flag_get(cla_flag_enum flag);
/*clear CLA flag */
void cla_flag_clear(cla_flag_enum flag);
/* enable CLA negedge interrupt */
void cla_negedge_interrupt_enable(uint32_t interrupt);
/* disable CLA negedge interrupt */
void cla_negedge_interrupt_disable(uint32_t interrupt);
/* enable CLA posedge interrupt */
void cla_posedge_interrupt_enable(uint32_t interrupt);
/* disable CLA posedge interrupt */
void cla_posedge_interrupt_disable(uint32_t interrupt);
/* check CLA interrupt flag is set or not */
FlagStatus cla_interrupt_flag_get(cla_interrupt_flag_enum int_flag);
/* clear CLA interrupt flag */
void cla_interrupt_flag_clear(cla_interrupt_flag_enum int_flag);

#endif /* GD32E25X_CLA_H */
