/*!
    \file    gd32e25x_bkp.h
    \brief   definitions for the bkp

    \version 2025-07-25, V0.2.0, firmware for GD32E25x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32E25X_BKP_H
#define GD32E25X_BKP_H

#include "gd32e25x.h"

/* BKP definitions */
#define BKP                                 BKP_BASE                                    /*!< BKP base address */

/* registers definitions */
#define BKP_STAT                            REG32((BKP) + 0x0000000CU)                  /*!< BKP status register */
#define BKP_TAMP                            REG32((BKP) + 0x00000040U)                  /*!< BKP tamper register */
#define BKP_DATA0                           REG32((BKP) + 0x00000050U)                  /*!< BKP backup 0 register */
#define BKP_DATA1                           REG32((BKP) + 0x00000054U)                  /*!< BKP backup 1 register */
#define BKP_DATA2                           REG32((BKP) + 0x00000058U)                  /*!< BKP backup 2 register */
#define BKP_DATA3                           REG32((BKP) + 0x0000005CU)                  /*!< BKP backup 3 register */
#define BKP_DATA4                           REG32((BKP) + 0x00000060U)                  /*!< BKP backup 4 register */

/* bits definitions */
/* BKP_STAT */
#define BKP_STAT_TP1F                       BIT(14)                                     /*!< BKP tamp 1 detected flag */

/* BKP_TAMP */
#define BKP_TAMP_TPIE                       BIT(2)                                      /*!< tamper detection interrupt enable */
#define BKP_TAMP_TP1EN                      BIT(3)                                      /*!< tamper 1 detection enable */
#define BKP_TAMP_TP1EG                      BIT(4)                                      /*!< tamper 1 event trigger edge for BKP tamp 1 input */
#define BKP_TAMP_FREQ                       BITS(8,10)                                  /*!< sample frequency of tamper event detection */
#define BKP_TAMP_FLT                        BITS(11,12)                                 /*!< BKP tamp x filter count setting */
#define BKP_TAMP_PRCH                       BITS(13,14)                                 /*!< precharge duration time of BKP tamp x */
#define BKP_TAMP_DISPU                      BIT(15)                                     /*!< BKP tamp x pull up disable bit */

/* BKP_DATA0 */
#define BKP_DATA0_DATA                      BITS(0,31)                                  /*!< backup domain register 0 */

/* BKP_DATA1 */
#define BKP_DATA1_DATA                      BITS(0,31)                                  /*!< backup domain register 1 */

/* BKP_DATA2 */
#define BKP_DATA2_DATA                      BITS(0,31)                                  /*!< backup domain register 2 */

/* BKP_DATA3 */
#define BKP_DATA3_DATA                      BITS(0,31)                                  /*!< backup domain register 3 */

/* BKP_DATA4 */
#define BKP_DATA4_DATA                      BITS(0,31)                                  /*!< backup domain register 4 */

/* constants definitions */
/* BKP register */
#define BKP_DATA0_4(number)                 REG32((BKP) + 0x50U + (number) * 0x04U)

/* get data of BKP data register */
#define BKP_DATA_GET(regval)                GET_BITS((uint32_t)(regval), 0, 31)

/* sample frequency */
#define TAMP_FREQ(regval)                   (BITS(8,10) & ((uint32_t)(regval) << 8U))   /*!< write value to BKP_TAMP_FREQ bit field */
#define BKP_FREQ_DIV32768                   TAMP_FREQ(0)                                /*!< sample once every 32768 BKPCLK(1Hz if BKPCLK=32.768KHz) */
#define BKP_FREQ_DIV16384                   TAMP_FREQ(1)                                /*!< sample once every 16384 BKPCLK(2Hz if BKPCLK=32.768KHz) */
#define BKP_FREQ_DIV8192                    TAMP_FREQ(2)                                /*!< sample once every 8192 BKPCLK(4Hz if BKPCLK=32.768KHz) */
#define BKP_FREQ_DIV4096                    TAMP_FREQ(3)                                /*!< sample once every 4096 BKPCLK(8Hz if BKPCLK=32.768KHz) */
#define BKP_FREQ_DIV2048                    TAMP_FREQ(4)                                /*!< sample once every 2048 BKPCLK(16Hz if BKPCLK=32.768KHz) */
#define BKP_FREQ_DIV1024                    TAMP_FREQ(5)                                /*!< sample once every 1024 BKPCLK(32Hz if BKPCLK=32.768KHz) */
#define BKP_FREQ_DIV512                     TAMP_FREQ(6)                                /*!< sample once every 512 BKPCLK(64Hz if BKPCLK=32.768KHz) */
#define BKP_FREQ_DIV256                     TAMP_FREQ(7)                                /*!< sample once every 256 BKPCLK(128Hz if BKPCLK=32.768KHz) */

/* filter count */
#define TAMP_FLT(regval)                    (BITS(11,12) & ((uint32_t)(regval) << 11U)) /*!< write value to BKP_TAMP_FLT bit field */
#define BKP_FLT_EDGE                        TAMP_FLT(0)                                 /*!< detecting tamper event using edge mode. precharge duration is disabled automatically */
#define BKP_FLT_2S                          TAMP_FLT(1)                                 /*!< detecting tamper event using level mode.2 consecutive valid level samples will make an effective tamper event */
#define BKP_FLT_4S                          TAMP_FLT(2)                                 /*!< detecting tamper event using level mode.4 consecutive valid level samples will make an effective tamper event */
#define BKP_FLT_8S                          TAMP_FLT(3)                                 /*!< detecting tamper event using level mode.8 consecutive valid level samples will make an effective tamper event */

/* precharge time before each sampling */
#define TAMP_PRCH(regval)                   (BITS(13,14) & ((uint32_t)(regval) << 13U)) /*!< write value to BKP_TAMP_PRCH bit field */
#define BKP_PRCH_1C                         TAMP_PRCH(0)                                /*!< 1 BKP clock precharge time before each sampling */
#define BKP_PRCH_2C                         TAMP_PRCH(1)                                /*!< 2 BKP clock precharge time before each sampling  */
#define BKP_PRCH_4C                         TAMP_PRCH(2)                                /*!< 4 BKP clock precharge time before each sampling */
#define BKP_PRCH_8C                         TAMP_PRCH(3)                                /*!< 8 BKP clock precharge time before each sampling */

/* BKP tamp event trigger edge */
#define BKP_TRIG_RISING_LOW                 ((uint32_t)0x00000000U)                     /*!< tamper 1 event trigger edge low level */
#define BKP_TRIG_FALLING_HIGH               BKP_TAMP_TP1EG                              /*!< tamper 1 event trigger edge high level */

/* BKP tamp detect */
#define BKP_TAMPER1                         BKP_TAMP_TP1EN                              /*!< tamper 1 detection enable */

/* BKP interrupt source */
#define BKP_INT_TAMP                        BKP_TAMP_TPIE                               /*!< tamper detection interrupt enable */

/* BKP flag */
#define BKP_FLAG_TAMP1                      BKP_STAT_TP1F                               /*!< tamper 1 event flag */

/* BKP data register number */
typedef enum {
    BKP_DATA_0 = 0,                                                                     /*!< BKP data register 0 */
    BKP_DATA_1,                                                                         /*!< BKP data register 1 */
    BKP_DATA_2,                                                                         /*!< BKP data register 2 */
    BKP_DATA_3,                                                                         /*!< BKP data register 3 */
    BKP_DATA_4                                                                          /*!< BKP data register 4 */
} bkp_data_register_enum;

/* mask definitions for internal use */
#define SAMP_FREQ_MASK                  ((uint32_t)0x00000700U)                         /*!< the sampling frequency mask */
#define DURATION_TIME_MASK              ((uint32_t)0x00006000U)                         /*!< pre-charge duration time mask */
#define FLT_COUNT_MASK                  ((uint32_t)0x00001800U)                         /*!< number of consecutive valid level mask */
#define TRIG_EDGE_MASK                  ((uint32_t)0x00000010U)                         /*!< trigger edge mask */

/* parameter check section */
#ifdef FW_DEBUG_ERR_REPORT
#define NOT_BKP_REGISTER(x)        ((x) > BKP_DATA_4)
#define NOT_BKP_FLAG(x)            ((x) != BKP_FLAG_TAMP1)
#define NOT_BKP_SAMPLE_FREQ(x)     ((x) & (~SAMP_FREQ_MASK))
#define NOT_BKP_DURATION_TIME(x)   ((x) & (~DURATION_TIME_MASK))
#define NOT_BKP_FILTER_COUNT(x)    ((x) & (~FLT_COUNT_MASK))
#define NOT_BKP_TRIG_EDGE(x)       ((x) & (~TRIG_EDGE_MASK))
#endif

/* function declarations */
/* reset BKP registers */
void bkp_deinit(void);

/* tamper functions */
/* enable tamper detection */
void bkp_tamper_detection_enable(void);
/* disable tamper detection */
void bkp_tamper_detection_disable(void);
/* configure tamper detection sampling frequency */
void bkp_tamper_sampling_frequency_config(uint32_t frequency);
/* configure pre-charge duration time */
void bkp_sampling_duration_time_config(uint32_t time);
/* enable pull up pre-charge duration */
void bkp_pull_up_enable(void);
/* disable pull up pre-charge duration */
void bkp_pull_up_disable(void);
/* configure filter count */
void bkp_filter_count_config(uint32_t count);
/* configure event trigger edge */
void bkp_tamper_trigger_edge_config(uint32_t edge);

/* BKP data functions */
/* write BKP data register */
void bkp_write_data(bkp_data_register_enum register_number, uint32_t data);
/* read BKP data register */
uint32_t bkp_read_data(bkp_data_register_enum register_number);

/* flag and interrupt functions */
/* enable tamper interrupt */
void bkp_tamper_interrupt_enable(void);
/* disable tamper interrupt */
void bkp_tamper_interrupt_disable(void);
/* get BKP flag status */
FlagStatus bkp_flag_get(uint32_t flag);
/* clear BKP flag */
void bkp_flag_clear(uint32_t flag);
#endif /* GD32E25X_BKP_H */
