/*!
    \file    hs0_usbh_standard_hid.c
    \brief   USB host HID keyboard and mouse driver

    \version 2024-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "usbh_standard_hid.h"
#include <stdbool.h>

hid_keybd_info hs0_keybd_info;
hid_mouse_info hs0_mouse_info;

uint32_t hs0_mouse_report_data[2];
uint32_t hs0_keybd_report_data[2];

/* local constants */
static const uint8_t hs0_hid_keybrd_codes[] =
{
    0,      0,      0,      0,      31,     50,     48,     33,
    19,     34,     35,     36,     24,     37,     38,     39,         /* 0x00 - 0x0F */
    52,     51,     25,     26,     17,     20,     32,     21,
    23,     49,     18,     47,     22,     46,     2,      3,          /* 0x10 - 0x1F */
    4,      5,      6,      7,      8,      9,      10,     11,
    43,     110,    15,     16,     61,     12,     13,     27,         /* 0x20 - 0x2F */
    28,     29,     42,     40,     41,     1,      53,     54,
    55,     30,     112,    113,    114,    115,    116,    117,        /* 0x30 - 0x3F */
    118,    119,    120,    121,    122,    123,    124,    125,
    126,    75,     80,     85,     76,     81,     86,     89,         /* 0x40 - 0x4F */
    79,     84,     83,     90,     95,     100,    105,    106,
    108,    93,     98,     103,    92,     97,     102,    91,         /* 0x50 - 0x5F */
    96,     101,    99,     104,    45,     129,    0,      0,
    0,      0,      0,      0,      0,      0,      0,      0,          /* 0x60 - 0x6F */
    0,      0,      0,      0,      0,      0,      0,      0,
    0,      0,      0,      0,      0,      0,      0,      0,          /* 0x70 - 0x7F */
    0,      0,      0,      0,      0,      107,    0,      56,
    0,      0,      0,      0,      0,      0,      0,      0,          /* 0x80 - 0x8F */
    0,      0,      0,      0,      0,      0,      0,      0,
    0,      0,      0,      0,      0,      0,      0,      0,          /* 0x90 - 0x9F */
    0,      0,      0,      0,      0,      0,      0,      0,
    0,      0,      0,      0,      0,      0,      0,      0,          /* 0xA0 - 0xAF */
    0,      0,      0,      0,      0,      0,      0,      0,
    0,      0,      0,      0,      0,      0,      0,      0,          /* 0xB0 - 0xBF */
    0,      0,      0,      0,      0,      0,      0,      0,
    0,      0,      0,      0,      0,      0,      0,      0,          /* 0xC0 - 0xCF */
    0,      0,      0,      0,      0,      0,      0,      0,
    0,      0,      0,      0,      0,      0,      0,      0,          /* 0xD0 - 0xDF */
    58,     44,     60,     127,    64,     57,     62,     128         /* 0xE0 - 0xE7 */
};

#ifdef QWERTY_KEYBOARD

static const int8_t hs0_hid_keybrd_key[] =
{
    '\0',   '`',    '1',    '2',    '3',    '4',    '5',    '6',
    '7',    '8',    '9',    '0',    '-',    '=',    '\0',   '\r',
    '\t',   'q',    'w',    'e',    'r',    't',    'y',    'u',
    'i',    'o',    'p',    '[',    ']',    '\\',
    '\0',   'a',    's',    'd',    'f',    'g',    'h',    'j',
    'k',    'l',    ';',    '\'',   '\0',   '\n',
    '\0',   '\0',   'z',    'x',    'c',    'v',    'b',    'n',
    'm',    ',',    '.',    '/',    '\0',   '\0',
    '\0',   '\0',   '\0',   ' ',    '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\r',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '7',    '4',    '1',
    '\0',   '/',    '8',    '5',    '2',
    '0',    '*',    '9',    '6',    '3',
    '.',    '-',    '+',    '\0',   '\n',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0'
};

static const int8_t hs0_hid_keybrd_shiftkey[] = {
    '\0',   '~',    '!',    '@',    '#',    '$',    '%',    '^',    '&',    '*',    '(',    ')',
    '_',    '+',    '\0',   '\0',   '\0',   'Q',    'W',    'E',    'R',    'T',    'Y',    'U',
    'I',    'O',    'P',    '{',    '}',    '|',    '\0',   'A',    'S',    'D',    'F',    'G',
    'H',    'J',    'K',    'L',    ':',    '"',    '\0',   '\n',   '\0',   '\0',   'Z',    'X',
    'C',    'V',    'B',    'N',    'M',    '<',    '>',    '?',    '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0'
};

#else

static const int8_t hs0_hid_keybrd_key[] = {
    '\0',   '`',    '1',    '2',    '3',    '4',    '5',    '6',    '7',    '8',    '9',    '0',
    '-',    '=',    '\0',   '\r',   '\t',   'a',    'z',    'e',    'r',    't',    'y',    'u',
    'i',    'o',    'p',    '[',    ']',    '\\',   '\0',   'q',    's',    'd',    'f',    'g',
    'h',    'j',    'k',    'l',    'm',    '\0',   '\0',   '\n',   '\0',   '\0',   'w',    'x',
    'c',    'v',    'b',    'n',    ',',    ';',    ':',    '!',    '\0',   '\0',   '\0',   '\0',
    '\0',   ' ',    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\r',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '7',    '4',    '1',    '\0',   '/',
    '8',    '5',    '2',    '0',    '*',    '9',    '6',    '3',    '.',    '-',    '+',    '\0',
    '\n',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0'
};

static const int8_t hs0_hid_keybrd_shiftkey[] = {
    '\0',   '~',    '!',    '@',    '#',    '$',    '%',    '^',    '&',    '*',    '(',    ')',    '_',
    '+',    '\0',   '\0',   '\0',   'A',    'Z',    'E',    'R',    'T',    'Y',    'U',    'I',    'O',
    'P',    '{',    '}',    '*',    '\0',   'Q',    'S',    'D',    'F',    'G',    'H',    'J',    'K',
    'L',    'M',    '%',    '\0',   '\n',   '\0',   '\0',   'W',    'X',    'C',    'V',    'B',    'N',
    '?',    '.',    '/',    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',
    '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0',   '\0'
};

#endif

/*!
    \brief      initialize the mouse function
    \param[in]  udev: pointer to USB core instance
    \param[in]  uhost: pointer to USB host
    \param[out] none
    \retval     none
*/
usbh_status hs0_usbh_hid_mouse_init (usb_core_driver *udev, hs0_usbh_host *uhost)
{
    hs0_usbh_hid_handler *hid = (hs0_usbh_hid_handler *)uhost->active_class->class_data;

    hs0_mouse_info.x = 0U;
    hs0_mouse_info.y = 0U;
    hs0_mouse_info.buttons[0] = 0U;
    hs0_mouse_info.buttons[1] = 0U;
    hs0_mouse_info.buttons[2] = 0U;

    hs0_mouse_report_data[0] = 0U;

    if(hid->len > sizeof(hs0_mouse_report_data)) {
        hid->len = sizeof(hs0_mouse_report_data);
    }

    hid->pdata = (uint8_t *)(void *)hs0_mouse_report_data;

    hs0_usbh_hid_fifo_init(&hid->fifo, uhost->dev_prop.data, HID_QUEUE_SIZE * sizeof(hs0_mouse_report_data));

    hs0_usr_mouse_init();

    return USBH_OK;
}

/*!
    \brief      initialize the keyboard function
    \param[in]  udev: pointer to USB core instance
    \param[in]  uhost: pointer to USB host
    \param[out] none
    \retval     operation status
*/
usbh_status hs0_usbh_hid_keybd_init (usb_core_driver *udev, hs0_usbh_host *uhost)
{
    hs0_usbh_hid_handler *hid = (hs0_usbh_hid_handler *)uhost->active_class->class_data;

    hs0_keybd_info.lctrl = hs0_keybd_info.lshift = 0U;
    hs0_keybd_info.lalt  = hs0_keybd_info.lgui   = 0U;
    hs0_keybd_info.rctrl = hs0_keybd_info.rshift = 0U;
    hs0_keybd_info.ralt  = hs0_keybd_info.rgui   = 0U;

    for (uint32_t x = 0U; x < (sizeof(hs0_keybd_report_data) / sizeof(uint32_t)); x++) {
        hs0_keybd_report_data[x] = 0U;
    }

    if (hid->len > (sizeof(hs0_keybd_report_data) / sizeof(uint32_t))) {
        hid->len = (sizeof(hs0_keybd_report_data) / sizeof(uint32_t));
    }

    hid->pdata = (uint8_t*)(void *)hs0_keybd_report_data;

    hs0_usbh_hid_fifo_init (&hid->fifo, uhost->dev_prop.data, HID_QUEUE_SIZE * sizeof(hs0_keybd_report_data));

    /* call user initialization*/
    hs0_usr_keybrd_init();

    return USBH_OK;
}

/*!
    \brief      decode mouse information
    \param[in]  data: pointer to inut data
    \param[out] none
    \retval     operation status
*/
usbh_status hs0_usbh_hid_mouse_decode(uint8_t *data)
{
    hs0_mouse_info.buttons[0] = data[0] & MOUSE_BUTTON_1;
    hs0_mouse_info.buttons[1] = data[0] & MOUSE_BUTTON_2;
    hs0_mouse_info.buttons[2] = data[0] & MOUSE_BUTTON_3;

    hs0_mouse_info.x = data[1];
    hs0_mouse_info.y = data[2];

    /* handle mouse data position */
    hs0_usr_mouse_process_data(&hs0_mouse_info);

    return USBH_FAIL;
}

/*!
    \brief      decode keyboard information
    \param[in]  udev: pointer to USB core instance
    \param[in]  uhost: pointer to USB host
    \param[out] none
    \retval     operation status
*/
usbh_status hs0_usbh_hid_keybrd_decode (uint8_t *data)
{
    uint8_t output;

    hs0_keybd_info.lshift = data[0] & KBD_LEFT_SHIFT;
    hs0_keybd_info.rshift = data[0] & KBD_RIGHT_SHIFT;

    hs0_keybd_info.keys[0] = data[2];

    if (hs0_keybd_info.lshift || hs0_keybd_info.rshift) {
        output = hs0_hid_keybrd_shiftkey[hs0_hid_keybrd_codes[hs0_keybd_info.keys[0]]];
    } else {
        output = hs0_hid_keybrd_key[hs0_hid_keybrd_codes[hs0_keybd_info.keys[0]]];
    }

    if (0U != output) {
        hs0_usr_keybrd_process_data(output);
    }

    return USBH_OK;
}

/*!
    \brief      read a hid report item
    \param[in]  ri: pointer to report item
    \param[in]  ndx: report index
    \param[out] none
    \retval     operation status (0: fail otherwise: item value)
*/
uint32_t hid_item_read (hid_report_item *ri, uint8_t ndx)
{
    uint32_t val = 0U;
    uint32_t bofs = 0U;
    uint8_t *data = ri->data;
    uint8_t shift = ri->shift;

    /* get the logical value of the item */

    /* if this is an array, wee may need to offset ri->data.*/
    if (ri->count > 0U) {
        /* if APP tries to read outside of the array. */
        if (ri->count <= ndx) {
            return(0U);
        }

        /* calculate bit offset */
        bofs = ndx * ri->size;
        bofs += shift;

        /* calculate byte offset + shift pair from bit offset. */
        data += bofs / 8U;
        shift = (uint8_t)(bofs % 8U);
    }

    /* read data bytes in little endian order */
    for (uint32_t x = 0U; x < ((ri->size & 0x7U) ? (ri->size / 8U) + 1U : (ri->size / 8U)); x++) {
        val=(uint32_t)((uint32_t)(*data) << (x * 8U));
    }

    val=(val >> shift) & ((1U << ri->size) - 1U);

    if ((val < ri->logical_min) || (val > ri->logical_max)) {
        return(0U);
    }

    /* convert logical value to physical value */
    /* see if the number is negative or not. */
    if ((ri->sign) && (val & (1U << (ri->size - 1U)))) {
        /* yes, so sign extend value to 32 bits. */
        uint32_t vs = (uint32_t)((0xffffffffU & ~((1U << (ri->size)) - 1U)) | val);

        if (1U == ri->resolution) {
            return((uint32_t)vs);
        }
        return((uint32_t)(vs * ri->resolution));
    } else {
        if (1U == ri->resolution) {
            return(val);
        }

        return (val * ri->resolution);
    }
}

/*!
    \brief      write a hid report item
    \param[in]  ri: pointer to report item
    \param[in]  value: the value to be write
    \param[in]  ndx: report index
    \param[out] none
    \retval     operation status (1: fail 0: OK)
*/
uint32_t hid_item_write(hid_report_item *ri, uint32_t value, uint8_t ndx)
{
    uint32_t mask;
    uint32_t bofs;
    uint8_t *data = ri->data;
    uint8_t shift = ri->shift;

    if ((value < ri->physical_min) || (value > ri->physical_max)) {
        return(1U);
    }

    /* if this is an array, wee may need to offset ri->data.*/
    if (ri->count > 0U) {
        /* if APP tries to read outside of the array. */
        if (ri->count >= ndx) {
            return(0U);
        }

        /* calculate bit offset */
        bofs = ndx * ri->size;
        bofs += shift;

        /* calculate byte offset + shift pair from bit offset. */
        data += bofs / 8U;
        shift = (uint8_t)(bofs % 8U);
    }

    /* convert physical value to logical value. */
    if (1U != ri->resolution) {
        value = value / ri->resolution;
    }

    /* write logical value to report in little endian order. */
    mask = (1U << ri->size) - 1U;
    value = (value & mask) << shift;

    for (uint32_t x = 0U; x < ((ri->size & 0x7U) ? (ri->size / 8U) + 1U : (ri->size / 8U)); x++) {
        *(ri->data + x) = (uint8_t)((*(ri->data+x) & ~(mask>>(x* 8U))) | ((value >> (x * 8U)) & (mask >> (x * 8U))));
    }

    return 0U;
}
