/*!
    \file    usbd_storage_msd.c
    \brief   this file provides the disk operations functions

    \version 2024-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "hs0_usbd_msc_mem.h"

/* pages 0 and 1 base and end addresses */
#define HS0_NAND_FLASH_BASE_ADDRESS     0x8010000U
#define HS0_PAGE_SIZE                   0x1000U

#define HS0_ISFLASH_BLOCK_SIZE         4096U
#define HS0_ISFLASH_BLOCK_NUM          64U

/* function declarations */
/* initialize the nand flash */
uint32_t  hs0_flash_init (void);
/* read data from multiple blocks of nand flash */
uint32_t  hs0_flash_multi_blocks_read (uint8_t* pBuf, uint32_t read_addr, uint16_t block_size, uint32_t block_num);
/* write data to multiple blocks of flash */
uint32_t  hs0_flash_multi_blocks_write (uint8_t* pBuf, uint32_t write_addr, uint16_t block_size, uint32_t block_num);

/* USB mass storage standard inquiry data */

const int8_t HS0_STORAGE_InquiryData[] =
{
    /* LUN 0 */
    0x00,
    0x80,
    0x00,
    0x01,
    (HS0_USBD_STD_INQUIRY_LENGTH - 5U),
    0x00,
    0x00,
    0x00,
    'G', 'D', '3', '2', ' ', ' ', ' ', ' ', /* Manufacturer : 8 bytes */
    'I', 'n', 't', 'e', 'r', 'n', 'a', 'l', /* Product      : 16 Bytes */
    ' ', 'f', 'l', 'a', 's', 'h', ' ', ' ',
    '1', '.', '0' ,'0',                     /* Version      : 4 Bytes */
};

static int8_t  HS0_STORAGE_Init             (uint8_t Lun);
static int8_t  HS0_STORAGE_IsReady          (uint8_t Lun);
static int8_t  HS0_STORAGE_IsWriteProtected (uint8_t Lun);
static int8_t  HS0_STORAGE_GetMaxLun        (void);

static int8_t  HS0_STORAGE_Read             (uint8_t Lun,
                                             uint8_t *buf,
                                             uint32_t BlkAddr,
                                             uint16_t BlkLen);

static int8_t  HS0_STORAGE_Write            (uint8_t Lun,
                                             uint8_t *buf,
                                             uint32_t BlkAddr,
                                             uint16_t BlkLen);


hs0_usbd_mem_cb HS0_USBD_Internal_Storage_fops =
{
    .mem_init      = HS0_STORAGE_Init,
    .mem_ready     = HS0_STORAGE_IsReady,
    .mem_protected = HS0_STORAGE_IsWriteProtected,
    .mem_read      = HS0_STORAGE_Read,
    .mem_write     = HS0_STORAGE_Write,
    .mem_maxlun    = HS0_STORAGE_GetMaxLun,

    .mem_inquiry_data = {(uint8_t *)HS0_STORAGE_InquiryData},
    .mem_block_size   = {HS0_ISFLASH_BLOCK_SIZE},
    .mem_block_len    = {HS0_ISFLASH_BLOCK_NUM}
};

hs0_usbd_mem_cb *hs0_usbd_mem_fops = &HS0_USBD_Internal_Storage_fops;

/*!
    \brief      initialize the storage medium
    \param[in]  Lun: logical unit number
    \param[out] none
    \retval     status
*/
static int8_t  HS0_STORAGE_Init (uint8_t Lun)
{
    return 0;
}

/*!
    \brief      check whether the medium is ready
    \param[in]  Lun: logical unit number
    \param[out] none
    \retval     status
*/
static int8_t  HS0_STORAGE_IsReady (uint8_t Lun)
{
    hs0_flash_init ();

    return 0;
}

/*!
    \brief      check whether the medium is write-protected
    \param[in]  Lun: logical unit number
    \param[out] none
    \retval     status
*/
static int8_t  HS0_STORAGE_IsWriteProtected (uint8_t Lun)
{
    return 0;
}

/*!
    \brief      read data from the medium
    \param[in]  Lun: logical unit number
    \param[in]  buf: pointer to the buffer to save data
    \param[in]  BlkAddr: address of 1st block to be read
    \param[in]  BlkLen: number of blocks to be read
    \param[out] none
    \retval     status
*/
static int8_t  HS0_STORAGE_Read (uint8_t Lun,
                            uint8_t *buf,
                            uint32_t BlkAddr,
                            uint16_t BlkLen)
{
    if (hs0_flash_multi_blocks_read (buf, BlkAddr, HS0_ISFLASH_BLOCK_SIZE, BlkLen) != 0U) {
        return 5;
    }

    return 0;
}

/*!
    \brief      write data to the medium
    \param[in]  Lun: logical unit number
    \param[in]  buf: pointer to the buffer to write
    \param[in]  BlkAddr: address of 1st block to be written
    \param[in]  BlkLen: number of blocks to be write
    \param[out] none
    \retval     status
*/
static int8_t  HS0_STORAGE_Write (uint8_t Lun,
                             uint8_t *buf,
                             uint32_t BlkAddr,
                             uint16_t BlkLen)
{
    if (hs0_flash_multi_blocks_write (buf, BlkAddr, HS0_ISFLASH_BLOCK_SIZE, BlkLen) != 0U) {
        return 5;
    }

    return (0);
}

/*!
    \brief      get number of supported logical unit
    \param[in]  none
    \param[out] none
    \retval     number of logical unit
*/
static int8_t HS0_STORAGE_GetMaxLun (void)
{
    return (HS0_MEM_LUN_NUM - 1);
}

/*!
    \brief      initialize the internal flash
    \param[in]  none
    \param[out] none
    \retval     status
  */
uint32_t  hs0_flash_init ()
{
    hal_fmc_unlock();

    return 0U;
}

/*!
    \brief      read data from multiple blocks of internal flash
    \param[in]  pBuf: pointer to user buffer
    \param[in]  read_addr: address to be read
    \param[in]  block_size: size of block
    \param[in]  block_num: number of block
    \param[out] none
    \retval     status
*/
uint32_t  hs0_flash_multi_blocks_read (uint8_t *pBuf, uint32_t read_addr, uint16_t block_size, uint32_t block_num)
{
    uint32_t i;
    uint8_t *pSource = (uint8_t *)(read_addr + HS0_NAND_FLASH_BASE_ADDRESS);

    /* Data transfer */
    while (block_num--) {
        for (i = 0U; i < block_size; i++) {
            *pBuf++ = *pSource++;
        }
    }

    return 0U;
}

/*!
    \brief      write data to multiple blocks of flash
    \param[in]  pBuf: pointer to user buffer
    \param[in]  write_addr: address to be write
    \param[in]  block_size: block size
    \param[in]  block_num: number of block
    \param[out] none
    \retval     status
*/
uint32_t hs0_flash_multi_blocks_write (uint8_t *pBuf, uint32_t write_addr, uint16_t block_size, uint32_t block_num)
{
    uint32_t i, page;
    uint32_t start_page = (write_addr / HS0_PAGE_SIZE) * HS0_PAGE_SIZE + HS0_NAND_FLASH_BASE_ADDRESS;
    uint32_t *ptrs = (uint32_t *)pBuf;

    page = block_num;

    for(; page > 0U; page--){
        hals_fmc_sector_erase(start_page);

        i = 0U;

        do{
            hals_fmc_word_program(start_page, *ptrs++);
            start_page += 4U;
        }while(++i < 1024U);
    }

    return 0U;
}
