/*!
    \file    dfu_core.c
    \brief   USB DFU device class core functions

    \version 2024-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "drv_usb_hw.h"
#include "dfu_core.h"
#include "dfu_mal.h"
#include "hs1_inter_flash_if.h"
#include <string.h>
#include "usbd_core.h"

#define USBD_VID                     0x28E9U
#define USBD_PID                     0x0189U

/* local function prototypes ('static') */
static uint8_t hs1_dfu_init(usb_dev *udev, uint8_t config_index);
static uint8_t hs1_dfu_deinit(usb_dev *udev, uint8_t config_index);
static uint8_t hs1_dfu_req_handler(usb_dev *udev, usb_req *req);
static uint8_t hs1_dfu_ctlx_in(usb_dev *udev);
static void hs1_dfu_detach(usb_dev *udev, usb_req *req);
static void hs1_dfu_dnload(usb_dev *udev, usb_req *req);
static void hs1_dfu_upload(usb_dev *udev, usb_req *req);
static void hs1_dfu_getstatus(usb_dev *udev, usb_req *req);
static void hs1_dfu_clrstatus(usb_dev *udev, usb_req *req);
static void hs1_dfu_getstate(usb_dev *udev, usb_req *req);
static void hs1_dfu_abort(usb_dev *udev, usb_req *req);
static void hs1_dfu_mode_leave(usb_dev *udev);
static uint8_t hs1_dfu_getstatus_complete(usb_dev *udev);

extern void *hs1_dfu_strings_desc[];

static void (*dfu_request_process[])(usb_dev *udev, usb_req *req) = {
    [DFU_DETACH]    = hs1_dfu_detach,
    [DFU_DNLOAD]    = hs1_dfu_dnload,
    [DFU_UPLOAD]    = hs1_dfu_upload,
    [DFU_GETSTATUS] = hs1_dfu_getstatus,
    [DFU_CLRSTATUS] = hs1_dfu_clrstatus,
    [DFU_GETSTATE]  = hs1_dfu_getstate,
    [DFU_ABORT]     = hs1_dfu_abort
};

typedef struct
{
    uint8_t bStatus;
    uint8_t bwPollTimeout0;
    uint8_t bwPollTimeout1;
    uint8_t bwPollTimeout2;
    uint8_t bState;
    uint8_t iString;

    uint8_t manifest_state;
    uint32_t data_len;
    uint16_t block_num;
    uint32_t base_addr;

    uint8_t buf[HS1_TRANSFER_SIZE];
} hs1_usbd_dfu_handler;

/* note:it should use the c99 standard when compiling the below codes */
/* USB standard device descriptor */
const __ALIGN_BEGIN usb_desc_dev hs1_dfu_dev_desc __ALIGN_END = {
    .header =
    {
        .bLength          = USB_DEV_DESC_LEN,
        .bDescriptorType  = USB_DESCTYPE_DEV
    },
    .bcdUSB                = 0x0200U,
    .bDeviceClass          = 0x00U,
    .bDeviceSubClass       = 0x00U,
    .bDeviceProtocol       = 0x00U,
    .bMaxPacketSize0       = USB_FS_EP0_MAX_LEN,
    .idVendor              = USBD_VID,
    .idProduct             = USBD_PID,
    .bcdDevice             = 0x0100U,
    .iManufacturer         = STR_IDX_MFC,
    .iProduct              = STR_IDX_PRODUCT,
    .iSerialNumber         = STR_IDX_SERIAL,
    .bNumberConfigurations = HS1_DEVICE_MAX_CONF_NUM
};

/* USB device configuration descriptor */
const __ALIGN_BEGIN usb_dfu_desc_config_set hs1_dfu_config_desc __ALIGN_END = {
    .config =
    {
        .header =
        {
            .bLength         = sizeof(usb_desc_config),
            .bDescriptorType = USB_DESCTYPE_CONFIG
        },
        .wTotalLength         = USB_DFU_CONFIG_DESC_SIZE,
        .bNumInterfaces       = 0x01U,
        .bConfigurationValue  = 0x01U,
        .iConfiguration       = 0x00U,
        .bmAttributes         = (HS1_DEVICE_SELF_POWERED << 6U),
        .bMaxPower            = 0x32U
    },

    .dfu_itf =
    {
        .header =
        {
            .bLength         = sizeof(usb_desc_itf),
            .bDescriptorType = USB_DESCTYPE_ITF
        },
        .bInterfaceNumber     = 0x00U,
        .bAlternateSetting    = 0x00U,
        .bNumEndpoints        = 0x00U,
        .bInterfaceClass      = USB_DFU_CLASS,
        .bInterfaceSubClass   = USB_DFU_SUBCLASS_UPGRADE,
        .bInterfaceProtocol   = USB_DFU_PROTOCL_DFU,
        .iInterface           = 0x05U
    },

    .dfu_func =
    {
        .header =
        {
            .bLength          = sizeof(usb_desc_dfu_func),
            .bDescriptorType  = DFU_DESC_TYPE
        },
        .bmAttributes         = USB_DFU_CAN_DOWNLOAD | USB_DFU_CAN_UPLOAD | USB_DFU_WILL_DETACH,
        .wDetachTimeOut       = 0x00FFU,
        .wTransferSize        = HS1_TRANSFER_SIZE,
        .bcdDFUVersion        = 0x011AU,
    },
};

usb_desc hs1_dfu_desc = {
    .dev_desc    = (uint8_t *) &hs1_dfu_dev_desc,
    .config_desc = (uint8_t *) &hs1_dfu_config_desc,
    .strings     = hs1_dfu_strings_desc
};

usb_class_core hs1_dfu_class = {
    .init            = hs1_dfu_init,
    .deinit          = hs1_dfu_deinit,
    .req_proc        = hs1_dfu_req_handler,
    .ctlx_in         = hs1_dfu_ctlx_in
};

/*!
    \brief      initialize the DFU device
    \param[in]  udev: pointer to USB device instance
    \param[in]  config_index: configuration index
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t hs1_dfu_init(usb_dev *udev, uint8_t config_index)
{
    static hs1_usbd_dfu_handler dfu_handler;

    /* unlock the internal flash */
    hs1_dfu_mal_init();

    memset((void *)&dfu_handler, 0, sizeof(hs1_usbd_dfu_handler));

    dfu_handler.base_addr = HS1_APP_LOADED_ADDR;
    dfu_handler.manifest_state = MANIFEST_COMPLETE;
    dfu_handler.bState = STATE_DFU_IDLE;
    dfu_handler.bStatus = STATUS_OK;

    udev->dev.class_data[HS1_USBD_DFU_INTERFACE] = (void *)&dfu_handler;

    return USBD_OK;
}

/*!
    \brief      de-initialize the DFU device
    \param[in]  udev: pointer to USB device instance
    \param[in]  config_index: configuration index
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t hs1_dfu_deinit(usb_dev *udev, uint8_t config_index)
{
    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    /* restore device default state */
    memset(udev->dev.class_data[HS1_USBD_DFU_INTERFACE], 0, sizeof(hs1_usbd_dfu_handler));

    dfu->bState = STATE_DFU_IDLE;
    dfu->bStatus = STATUS_OK;

    /* lock the internal flash */
    hs1_dfu_mal_deinit();

    return USBD_OK;
}

/*!
    \brief      handle the DFU class-specific requests
    \param[in]  udev: pointer to USB device instance
    \param[in]  req: device class-specific request
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t hs1_dfu_req_handler(usb_dev *udev, usb_req *req)
{
    if(req->bRequest < DFU_REQ_MAX) {
        dfu_request_process[req->bRequest](udev, req);
    } else {
        return USBD_FAIL;
    }

    return USBD_OK;
}

/*!
    \brief      handle data Stage
    \param[in]  udev: pointer to USB device instance
    \param[in]  ep_num: the endpoint number
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t hs1_dfu_ctlx_in(usb_dev *udev)
{
    hs1_dfu_getstatus_complete(udev);

    return USBD_OK;
}

/*!
    \brief      leave DFU mode and reset device to jump to user loaded code
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     none
*/
static void hs1_dfu_mode_leave(usb_dev *udev)
{
    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    dfu->manifest_state = MANIFEST_COMPLETE;

    if(hs1_dfu_config_desc.dfu_func.bmAttributes & 0x04U) {
        dfu->bState = STATE_DFU_MANIFEST_SYNC;
    } else {
        dfu->bState = STATE_DFU_MANIFEST_WAIT_RESET;

        /* lock the internal flash */
        hs1_dfu_mal_deinit();

        /* generate system reset to allow jumping to the user code */
        NVIC_SystemReset();
    }
}

/*!
    \brief      handle data IN stage in control endpoint 0
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
  */
static uint8_t hs1_dfu_getstatus_complete(usb_dev *udev)
{
    uint32_t addr;

    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    if(STATE_DFU_DNBUSY == dfu->bState) {
        /* decode the special command */
        if(0U == dfu->block_num) {
            if(1U == dfu->data_len) {
                if(GET_COMMANDS == dfu->buf[0]) {
                    /* no operation */
                }
            } else if(5U == dfu->data_len) {
                if(SET_ADDRESS_POINTER == dfu->buf[0]) {
                    /* set flash operation address */
                    dfu->base_addr = *(uint32_t *)(dfu->buf + 1U);
                } else if(ERASE == dfu->buf[0]) {
                    dfu->base_addr = *(uint32_t *)(dfu->buf + 1U);

                    hs1_dfu_mal_erase(dfu->base_addr);
                } else {
                    /* no operation */
                }
            } else {
                /* no operation */
            }
        } else if(dfu->block_num > 1U) {   /* regular download command */
            /* decode the required address */
            addr = (dfu->block_num - 2U) * HS1_TRANSFER_SIZE + dfu->base_addr;

            hs1_dfu_mal_write(dfu->buf, addr, dfu->data_len);

            dfu->block_num = 0U;
        } else {
            /* no operation */
        }

        dfu->data_len = 0U;

        /* update the device state and poll timeout */
        dfu->bState = STATE_DFU_DNLOAD_SYNC;

        return USBD_OK;
    } else if(STATE_DFU_MANIFEST == dfu->bState) {  /* manifestation in progress */
        /* start leaving DFU mode */
        hs1_dfu_mode_leave(udev);
    } else {
        /* no operation */
    }

    return USBD_OK;
}

/*!
    \brief      handle the DFU_DETACH request
    \param[in]  udev: pointer to USB device instance
    \param[in]  req: DFU class request
    \param[out] none
    \retval     none.
*/
static void hs1_dfu_detach(usb_dev *udev, usb_req *req)
{
    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    switch(dfu->bState) {
    case STATE_DFU_IDLE:
    case STATE_DFU_DNLOAD_SYNC:
    case STATE_DFU_DNLOAD_IDLE:
    case STATE_DFU_MANIFEST_SYNC:
    case STATE_DFU_UPLOAD_IDLE:
        dfu->bStatus = STATUS_OK;
        dfu->bState = STATE_DFU_IDLE;
        dfu->iString = 0U; /* iString */

        dfu->block_num = 0U;
        dfu->data_len = 0U;
        break;

    default:
        break;
    }

    /* check the detach capability in the DFU functional descriptor */
    if(hs1_dfu_config_desc.dfu_func.wDetachTimeOut & DFU_DETACH_MASK) {
        hs1_usbd_disconnect(udev);

        hs1_usbd_connect(udev);
    } else {
        /* wait for the period of time specified in detach request */
        usb_mdelay(4U);
    }
}

/*!
    \brief      handle the DFU_DNLOAD request
    \param[in]  udev: pointer to USB device instance
    \param[in]  req: DFU class request
    \param[out] none
    \retval     none
*/
static void hs1_dfu_dnload(usb_dev *udev, usb_req *req)
{
    usb_transc *transc = &udev->dev.transc_out[0];
    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    switch(dfu->bState) {
    case STATE_DFU_IDLE:
    case STATE_DFU_DNLOAD_IDLE:
        if(req->wLength > 0U) {
            /* update the global length and block number */
            dfu->block_num = req->wValue;
            dfu->data_len = req->wLength;

            dfu->bState = STATE_DFU_DNLOAD_SYNC;

            transc->remain_len = dfu->data_len;
            transc->xfer_buf = dfu->buf;
        } else {
            dfu->manifest_state = MANIFEST_IN_PROGRESS;
            dfu->bState = STATE_DFU_MANIFEST_SYNC;
        }
        break;

    default:
        break;
    }
}

/*!
    \brief      handles the DFU_UPLOAD request.
    \param[in]  udev: pointer to USB device instance
    \param[in]  req: DFU class request
    \param[out] none
    \retval     none
*/
static void hs1_dfu_upload(usb_dev *udev, usb_req *req)
{
    uint8_t *phy_addr = NULL;
    uint32_t addr = 0U;
    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    usb_transc *transc = &udev->dev.transc_in[0];

    if(req->wLength <= 0U) {
        dfu->bState = STATE_DFU_IDLE;
        return;
    }

    switch(dfu->bState) {
    case STATE_DFU_IDLE:
    case STATE_DFU_UPLOAD_IDLE:
        /* update the global length and block number */
        dfu->block_num = req->wValue;
        dfu->data_len = req->wLength;

        /* DFU get command */
        if(0U == dfu->block_num) {
            /* update the state machine */
            dfu->bState = (dfu->data_len > 3U) ? STATE_DFU_IDLE : STATE_DFU_UPLOAD_IDLE;

            /* store the values of all supported commands */
            dfu->buf[0] = GET_COMMANDS;
            dfu->buf[1] = SET_ADDRESS_POINTER;
            dfu->buf[2] = ERASE;

            /* send the status data over EP0 */
            transc->xfer_buf = &(dfu->buf[0]);
            transc->remain_len = 3U;
        } else if(dfu->block_num > 1U) {
            dfu->bState = STATE_DFU_UPLOAD_IDLE;

            /* change is accelerated */
            addr = (dfu->block_num - 2U) * HS1_TRANSFER_SIZE + dfu->base_addr;

            /* return the physical address where data are stored */
            phy_addr = hs1_dfu_mal_read(dfu->buf, addr, dfu->data_len);

            /* send the status data over EP0 */
            transc->xfer_buf = phy_addr;
            transc->remain_len = dfu->data_len;
        } else {
            dfu->bState = STATUS_ERR_STALLEDPKT;
        }
        break;

    default:
        dfu->data_len = 0U;
        dfu->block_num = 0U;
        break;
    }
}

/*!
    \brief      handle the DFU_GETSTATUS request
    \param[in]  udev: pointer to USB device instance
    \param[in]  req: DFU class request
    \param[out] none
    \retval     none
*/
static void hs1_dfu_getstatus(usb_dev *udev, usb_req *req)
{
    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    usb_transc *transc = &udev->dev.transc_in[0];

    switch(dfu->bState) {
    case STATE_DFU_DNLOAD_SYNC:
        if(0U != dfu->data_len) {
            dfu->bState = STATE_DFU_DNBUSY;

            if(0U == dfu->block_num) {
                if(ERASE == dfu->buf[0]) {
                    hs1_dfu_mal_getstatus(dfu->base_addr, CMD_ERASE, (uint8_t *)&dfu->bwPollTimeout0);
                } else {
                    hs1_dfu_mal_getstatus(dfu->base_addr, CMD_WRITE, (uint8_t *)&dfu->bwPollTimeout0);
                }
            }
        } else {
            dfu->bState = STATE_DFU_DNLOAD_IDLE;
        }
        break;

    case STATE_DFU_MANIFEST_SYNC:
        if(MANIFEST_IN_PROGRESS == dfu->manifest_state) {
            dfu->bState = STATE_DFU_MANIFEST;
            dfu->bwPollTimeout0 = 1U;
        } else if((MANIFEST_COMPLETE == dfu->manifest_state) && \
                  (hs1_dfu_config_desc.dfu_func.bmAttributes & 0x04U)) {
            dfu->bState = STATE_DFU_IDLE;
            dfu->bwPollTimeout0 = 0U;
        } else {
            /* no operation */
        }
        break;

    default:
        break;
    }

    /* send the status data of DFU interface to host over EP0 */
    transc->xfer_buf = (uint8_t *) & (dfu->bStatus);
    transc->remain_len = 6U;
}

/*!
    \brief      handle the DFU_CLRSTATUS request
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     none
*/
static void hs1_dfu_clrstatus(usb_dev *udev, usb_req *req)
{
    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    if(STATE_DFU_ERROR == dfu->bState) {
        dfu->bStatus = STATUS_OK;
        dfu->bState = STATE_DFU_IDLE;
    } else {
        /* state error */
        dfu->bStatus = STATUS_ERR_UNKNOWN;
        dfu->bState = STATE_DFU_ERROR;
    }

    dfu->iString = 0U; /* iString: index = 0 */
}

/*!
    \brief      handle the DFU_GETSTATE request
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     none
*/
static void hs1_dfu_getstate(usb_dev *udev, usb_req *req)
{
    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    usb_transc *transc = &udev->dev.transc_in[0];

    /* send the current state of the DFU interface to host */
    transc->xfer_buf = &(dfu->bState);
    transc->remain_len = 1U;
}

/*!
    \brief      handle the DFU_ABORT request
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     none
*/
static void hs1_dfu_abort(usb_dev *udev, usb_req *req)
{
    hs1_usbd_dfu_handler *dfu = (hs1_usbd_dfu_handler *)udev->dev.class_data[HS1_USBD_DFU_INTERFACE];

    switch(dfu->bState) {
    case STATE_DFU_IDLE:
    case STATE_DFU_DNLOAD_SYNC:
    case STATE_DFU_DNLOAD_IDLE:
    case STATE_DFU_MANIFEST_SYNC:
    case STATE_DFU_UPLOAD_IDLE:
        dfu->bStatus = STATUS_OK;
        dfu->bState = STATE_DFU_IDLE;
        dfu->iString = 0U; /* iString: index = 0 */

        dfu->block_num = 0U;
        dfu->data_len = 0U;
        break;

    default:
        break;
    }
}
