[#ftl]
[#-- V1.2.1 --]
/**
 * @file
 * Ethernet Interface for standalone applications (without RTOS)
 *
 */

/*
 * Copyright (c) 2001-2004 Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 *
 * Author: Adam Dunkels <adam@sics.se>
 *
 */

/* External Includes*/
#include "lwip/mem.h"
#include "netif/etharp.h"
#include "ethernetif.h"
#include "gd32h7xx_hal_enet.h"
#include "gd32h7xx_hal_init.h"
#include "main.h"
#include <string.h>
/* user code [External Includes] begin */

/* user code [External Includes] end */

/* Private Type Definitions */
/* user code [Private Type Definitions] begin */

/* user code [Private Type Definitions] end */

/* Private Macros */
/* network interface name */
#define IFNAME0 'G'
#define IFNAME1 '0'
#define IFNAME3 '1'
#define IFNAME4 'D'
/* user code [Private Macros] begin */

/* user code [Private Macros] end */

/* Private Constants */
#define ENET_DMA_TRANSMIT_TIMEOUT    20
#define ENET_TRAMSIT_BUFF_SIZE       4
/* user code [Private Constants] begin */

/* user code [Private Constants] end */

/* Private Variables */
/* ENET RxDMA/TxDMA descriptor */
[#if Peripherals?contains("ENET0") == true]
#if defined   (__CC_ARM) /*!< ARM compiler */
__attribute__((section(".ARM.__at_0x30000000")))   hal_enet_descriptors_struct  rxdesc_tab0[ENET_RXDESC_NUM];        /*!< ENET RxDMA descriptor */
__attribute__((section(".ARM.__at_0x30000160")))   hal_enet_descriptors_struct  txdesc_tab0[ENET_TXDESC_NUM];        /*!< ENET TxDMA descriptor */

#elif defined ( __ICCARM__ ) /*!< IAR compiler */
#pragma location=0x30000000
hal_enet_descriptors_struct  rxdesc_tab0[ENET_RXDESC_NUM];        /*!< ENET RxDMA descriptor */
#pragma location=0x30000160
hal_enet_descriptors_struct  txdesc_tab0[ENET_TXDESC_NUM];        /*!< ENET TxDMA descriptor */

#elif defined (__GNUC__)        /* GNU Compiler */
hal_enet_descriptors_struct  rxdesc_tab0[ENET_RXDESC_NUM] __attribute__((section(".ARM.__at_0x30000000")));          /*!< ENET RxDMA descriptor */
hal_enet_descriptors_struct  txdesc_tab0[ENET_TXDESC_NUM] __attribute__((section(".ARM.__at_0x30000160")));          /*!< ENET TxDMA descriptor */
#endif
[/#if]
[#if Peripherals?contains("ENET1") == true]
#if defined (__CC_ARM)  /*!< ARM compiler */
__attribute__((section(".ARM.__at_0x30004000")))   hal_enet_descriptors_struct  rxdesc_tab1[ENET_RXDESC_NUM];        /*!< ENET RxDMA descriptor */
__attribute__((section(".ARM.__at_0x30004160")))   hal_enet_descriptors_struct  txdesc_tab1[ENET_TXDESC_NUM];        /*!< ENET TxDMA descriptor */

#elif defined ( __ICCARM__ ) /*!< IAR compiler */
#pragma location=0x30004000
hal_enet_descriptors_struct  rxdesc_tab1[ENET_RXDESC_NUM];        /*!< ENET RxDMA descriptor */
#pragma location=0x30004160
hal_enet_descriptors_struct  txdesc_tab1[ENET_TXDESC_NUM];        /*!< ENET TxDMA descriptor */

#elif defined (__GNUC__) /* GNU Compiler */
hal_enet_descriptors_struct  rxdesc_tab1[ENET_RXDESC_NUM] __attribute__((section(".ARM.__at_0x30004000")));          /*!< ENET RxDMA descriptor */
hal_enet_descriptors_struct  txdesc_tab1[ENET_TXDESC_NUM] __attribute__((section(".ARM.__at_0x30004160")));          /*!< ENET TxDMA descriptor */
#endif
[/#if]

/* preserve another ENET RxDMA/TxDMA ptp descriptor for normal mode */
[#if Peripherals?contains("ENET0") == true]
hal_enet_transmit_packet_config_struct tx0_packet_config;
[#if ENET0_PTPSynchronization = "TRUE"]
hal_enet_descriptors_struct  ptp0_txstructure[ENET_TXDESC_NUM];
hal_enet_descriptors_struct  ptp0_rxstructure[ENET_RXDESC_NUM];
[/#if]
[/#if]
[#if Peripherals?contains("ENET1") == true]
hal_enet_transmit_packet_config_struct tx1_packet_config;
[#if ENET1_PTPSynchronization = "TRUE"]
hal_enet_descriptors_struct  ptp1_txstructure[ENET_TXDESC_NUM];
hal_enet_descriptors_struct  ptp1_rxstructure[ENET_RXDESC_NUM];
[/#if]
[/#if]

[#if Peripherals?contains("ENET0") == true]
uint8_t rx0_buff[ENET_RXDESC_NUM][ENET_MAX_FRAME_SIZE] __attribute__((section(".ARM.__at_0x30000300")));
[/#if]
[#if Peripherals?contains("ENET1") == true]
uint8_t rx1_buff[ENET_RXDESC_NUM][ENET_MAX_FRAME_SIZE] __attribute__((section(".ARM.__at_0x30004300")));
[/#if]
/* user code [Private Variables] begin */

/* user code [Private Variables] end */

/* Private Function Declaration */
/* user code [Private Function Declaration] begin */

/* user code [Private Function Declaration] end */

/* Extern Variables */
/* user code [Extern Variables] begin */

/* user code [Extern Variables] end */

[#if Peripherals?contains("ENET0") == true]
/**
 * In this function, the hardware should be initialized.
 * Called from ethernetif_init().
 *
 * @param netif the already initialized lwip network interface structure
 *        for this ethernetif
 */
static void low_level_init(struct netif *netif)
{
    uint32_t i = 0U;
    uint8_t tx0_buff[ENET_TXDESC_NUM][ENET_TRAMSIT_BUFF_SIZE];

    /* set tx desc address */
    for(i = 0; i < ENET_TXDESC_NUM; i++) {
        enet0_info.tx_desc_list.tx_desc[i] = (uint32_t)&txdesc_tab0[i];
    }

    /* set rx desc address */
    for(i = 0; i < ENET_RXDESC_NUM; i++) {
        enet0_info.rx_desc_list.rx_desc[i] = (uint32_t)&rxdesc_tab0[i];
    }

    /* struct init tx packet config */
    hal_enet_struct_init(HAL_ENET_TRANSMIT_PACKET_CONFIG_STRUCT, &tx0_packet_config);

    /* set MAC hardware address length */
    netif->hwaddr_len = ETHARP_HWADDR_LEN;

    /* set MAC hardware address */
    netif->hwaddr[0] =  BOARD0_MAC_ADDR0;
    netif->hwaddr[1] =  BOARD0_MAC_ADDR1;
    netif->hwaddr[2] =  BOARD0_MAC_ADDR2;
    netif->hwaddr[3] =  BOARD0_MAC_ADDR3;
    netif->hwaddr[4] =  BOARD0_MAC_ADDR4;
    netif->hwaddr[5] =  BOARD0_MAC_ADDR5;

    /* initialize MAC address in ethernet MAC */
    hal_enet_mac_address_set(&enet0_info, ENET_MAC_ADDRESS0, netif->hwaddr);

    /* maximum transfer unit */
    netif->mtu = 1500;

    /* device capabilities */
    /* don't set NETIF_FLAG_ETHARP if this device is not an ethernet one */
    netif->flags = NETIF_FLAG_BROADCAST | NETIF_FLAG_ETHARP | NETIF_FLAG_LINK_UP;

[#if ENET0_GeneralConfiguration_DescriptorPattern == "Enhance Descriptor Pattern"]
    /* initialize descriptors list: chain/ring mode */
    hal_enet_descriptors_chain_init(&enet0_info, ENET_DMA_TX, (uint8_t *)tx0_buff, SET);
    hal_enet_descriptors_chain_init(&enet0_info, ENET_DMA_RX, (uint8_t *)rx0_buff, SET);
[#else]
    /* initialize descriptors list: chain/ring mode */
    hal_enet_descriptors_chain_init(&enet0_info, ENET_DMA_TX, (uint8_t *)tx0_buff, RESET);
    hal_enet_descriptors_chain_init(&enet0_info, ENET_DMA_RX, (uint8_t *)rx0_buff, RESET);
[/#if]

    /* enable ethernet Rx interrrupt */
    {
        for(i = 0; i < ENET_RXDESC_NUM; i++) {
            hal_enet_rx_desc_immediate_receive_complete_interrupt(&enet0_info, \
                                                                  (hal_enet_descriptors_struct *)enet0_info.rx_desc_list.rx_desc[i], \
                                                                  NULL);
        }
    }

[#if LWIP_LWIPEnable_TRUELWIPChecksumConfiguration_ChecksumByHardware == "Enable"]
    /* enable the TCP, UDP and ICMP checksum insertion for the Tx frames */
    for(i = 0; i < ENET_TXDESC_NUM; i++) {
        hal_enet_transmit_checksum_config((hal_enet_descriptors_struct *)enet0_info.tx_desc_list.tx_desc[i], \
                                           ENET_CHECKSUM_TCPUDPICMP_FULL);
    }
[/#if]

    /* note: TCP, UDP, ICMP checksum checking for received frame are enabled in DMA config */
    /* enable MAC and DMA transmission and reception */
    hal_enet_start(&enet0_info);
}

/**
 * This function should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @param p the MAC packet to send (e.g. IP packet including MAC addresses and type)
 * @return ERR_OK if the packet could be sent
 *         an err_t value if the packet couldn't be sent
 *
 * @note Returning ERR_MEM here if a DMA queue of your MAC is full can lead to
 *       strange results. You might consider waiting for space in the DMA queue
 *       to become availale since the stack doesn't retry to send a packet
 *       dropped because of memory failure (except for the TCP timers).
 */

static err_t low_level_output(struct netif *netif, struct pbuf *p)
{
    struct pbuf *q;
    hal_enet_descriptors_struct *tx_desc;

    uint32_t i = 0U;
    uint32_t current_index = 0U;

    hal_enet_transfer_buffer_struct tx_buffer[ENET_TXDESC_NUM];
    memset(tx_buffer, 0U, ENET_TXDESC_NUM * sizeof(hal_enet_transfer_buffer_struct));

    current_index = enet0_info.tx_desc_list.tx_current_desc_index;
    tx_desc = (hal_enet_descriptors_struct *)enet0_info.tx_desc_list.tx_desc[current_index];

    while((uint32_t)RESET != ((tx_desc->status & ENET_TDES0_DAV) >> 31U));

    /* copy frame from pbufs to driver buffers */
    for(q = p; q != NULL; q = q->next) {
        if(i >= ENET_TXDESC_NUM) {
          return ERR_IF;
        } else {
          tx_buffer[i].buffer      = q->payload;
          tx_buffer[i].buffer_size = q->len;

          if(i > 0U) {
            tx_buffer[i-1].next_buff = &tx_buffer[i];
          } else {
              /* do nothing */
          }

          if(q->next == NULL) {
            tx_buffer[i].next_buff = NULL;
          } else {
              /* do nothing */
          }

          i++;
        }

[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
[#else]
        SCB_CleanDCache_by_Addr((uint32_t*)q->payload, q->len);
[/#if]
    }
[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
[#else]
    /* note: padding and CRC for transmitted frame are automatically inserted by DMA */
    SCB_CleanDCache_by_Addr((uint32_t*)tx_buffer, sizeof(tx_buffer));
    SCB_CleanDCache_by_Addr((uint32_t*)tx_desc, sizeof(hal_enet_descriptors_struct));
[/#if]

    /* transmit descriptors to give to DMA */
    tx0_packet_config.tx_packet_feature                = HAL_ENET_TX_FEATURE_CM;
    tx0_packet_config.checksum_mode                    = ENET_CHECKSUM_TCPUDPICMP_FULL;

    tx0_packet_config.tx_buffer                        = tx_buffer;
    tx0_packet_config.tx_buffer->next_buff             = tx_buffer->next_buff;
    tx0_packet_config.tx_buffer->secondary_buff        = NULL;
    tx0_packet_config.tx_buffer->buffer_size           = p->tot_len;
    tx0_packet_config.tx_buffer->secondary_buffer_size = 0U;
[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
[#else]
    SCB_CleanDCache_by_Addr((uint32_t*)&tx0_packet_config, sizeof(tx0_packet_config));
[/#if]

    hal_enet_frame_transmit(&enet0_info, &tx0_packet_config, ENET_DMA_TRANSMIT_TIMEOUT);

    return ERR_OK;
}

/**
 * Should allocate a pbuf and transfer the bytes of the incoming
 * packet from the interface into the pbuf.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return a pbuf filled with the received packet (including MAC header)
 *         NULL on memory error
 */
static struct pbuf *low_level_input(struct netif *netif)
{
    struct pbuf *p, *q;
    uint8_t *buffer;
    hal_enet_descriptors_struct *rx_desc;

    u16_t len;
    int l = 0;
    uint32_t current_index = 0U;

    p = NULL;

    /* obtain the size of the packet and put it into the "len" variable. */
    current_index = enet0_info.rx_desc_list.rx_current_desc_index;
    rx_desc = (hal_enet_descriptors_struct *)enet0_info.rx_desc_list.rx_desc[current_index];

    SCB_InvalidateDCache_by_Addr((uint32_t*)rx_desc, sizeof(hal_enet_descriptors_struct));

    len = hal_enet_desc_information_get(&enet0_info, rx_desc, RXDESC_FRAME_LENGTH);
    buffer = (uint8_t *)(hal_enet_desc_information_get(&enet0_info, rx_desc, RXDESC_BUFFER_1_ADDR));

    SCB_InvalidateDCache_by_Addr((uint32_t *)buffer, (len + 31) & ~31);

    /* we allocate a pbuf chain of pbufs from the Lwip buffer pool */
[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
    p = pbuf_alloc(PBUF_RAW, len, PBUF_POOL);
[#else]
    p = pbuf_alloc(PBUF_RAW, len, PBUF_RAM);
[/#if]

    if(p != NULL) {
        for(q = p; q != NULL; q = q->next) {
            SCB_InvalidateDCache_by_Addr((uint32_t *)q->payload, (q->len + 31) & ~31);
            memcpy((uint8_t *)q->payload, (u8_t *)&buffer[l], q->len);
[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
[#else]
            SCB_CleanDCache_by_Addr((uint32_t *)q->payload, (q->len + 31) & ~31);
[/#if]
            l = l + q->len;
        }
    }

    hal_enet_frame_receive(&enet0_info);

    return p;
}

/**
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface. Then the type of the received packet is determined and
 * the appropriate input function is called.
 *
 * @param netif the lwip network interface structure for this ethernetif
 */
err_t ethernetif_input(struct netif *netif)
{
    err_t err;
    struct pbuf *p;

    /* move received packet into a new pbuf */
    p = low_level_input(netif);

    /* no packet could be read, silently ignore this */
    if(p == NULL) {
        return ERR_MEM;
    }

    /* entry point to the LwIP stack */
    err = netif->input(p, netif);

    if(err != ERR_OK) {
        LWIP_DEBUGF(NETIF_DEBUG, ("ethernetif_input: IP input error\n"));
        pbuf_free(p);
        p = NULL;
    }

    return err;
}

/**
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 * This function should be passed as a parameter to netif_add().
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return ERR_OK if the loopif is initialized
 *         ERR_MEM if private data couldn't be allocated
 *         any other err_t on error
 */
err_t ethernetif_init(struct netif *netif)
{
    LWIP_ASSERT("netif != NULL", (netif != NULL));

#if LWIP_NETIF_HOSTNAME
    /* Initialize interface hostname */
    netif->hostname = "Gigadevice.COM_lwip";
#endif /* LWIP_NETIF_HOSTNAME */

    netif->name[0] = IFNAME0;
    netif->name[1] = IFNAME1;

    /* We directly use etharp_output() here to save a function call.
     * You can instead declare your own function an call etharp_output()
     * from it if you have to do some checks before sending (e.g. if link
     * is available...) */
    netif->output = etharp_output;
    netif->linkoutput = low_level_output;

    /* initialize the hardware */
    low_level_init(netif);

    return ERR_OK;
}
[/#if]

[#if Peripherals?contains("ENET1") == true]
/**
 * In this function, the hardware should be initialized.
 * Called from ethernetif_init().
 *
 * @param netif the already initialized lwip network interface structure
 *        for this ethernetif
 */
static void low_level_init1(struct netif *netif)
{
    uint32_t i = 0U;
    uint8_t tx1_buff[ENET_TXDESC_NUM][ENET_TRAMSIT_BUFF_SIZE];

    for(i = 0; i < ENET_TXDESC_NUM; i++) {
        enet1_info.tx_desc_list.tx_desc[i] = (uint32_t)&txdesc_tab1[i];
    }

    for(i = 0; i < ENET_RXDESC_NUM; i++) {
        enet1_info.rx_desc_list.rx_desc[i] = (uint32_t)&rxdesc_tab1[i];
    }

    /* struct init tx packet config */
    hal_enet_struct_init(HAL_ENET_TRANSMIT_PACKET_CONFIG_STRUCT, &tx1_packet_config);

    /* set MAC hardware address length */
    netif->hwaddr_len = ETHARP_HWADDR_LEN;

    /* set MAC hardware address */
    netif->hwaddr[0] =  BOARD1_MAC_ADDR0;
    netif->hwaddr[1] =  BOARD1_MAC_ADDR1;
    netif->hwaddr[2] =  BOARD1_MAC_ADDR2;
    netif->hwaddr[3] =  BOARD1_MAC_ADDR3;
    netif->hwaddr[4] =  BOARD1_MAC_ADDR4;
    netif->hwaddr[5] =  BOARD1_MAC_ADDR5;

    /* initialize MAC address in ethernet MAC */
    hal_enet_mac_address_set(&enet1_info, ENET_MAC_ADDRESS1, netif->hwaddr);

    /* maximum transfer unit */
    netif->mtu = 1500;

    /* device capabilities */
    /* don't set NETIF_FLAG_ETHARP if this device is not an ethernet one */
    netif->flags = NETIF_FLAG_BROADCAST | NETIF_FLAG_ETHARP | NETIF_FLAG_LINK_UP;

    /* initialize descriptors list: chain/ring mode */
[#if ENET1_GeneralConfiguration_DescriptorPattern == "Enhance Descriptor Pattern"]
    /* initialize descriptors list: chain/ring mode */
    hal_enet_descriptors_chain_init(&enet1_info, ENET_DMA_TX, (uint8_t *)tx1_buff, SET);
    hal_enet_descriptors_chain_init(&enet1_info, ENET_DMA_RX, (uint8_t *)rx1_buff, SET);
[#else]
    /* initialize descriptors list: chain/ring mode */
    hal_enet_descriptors_chain_init(&enet1_info, ENET_DMA_TX, (uint8_t *)tx1_buff, RESET);
    hal_enet_descriptors_chain_init(&enet1_info, ENET_DMA_RX, (uint8_t *)rx1_buff, RESET);
[/#if]

    /* enable ethernet Rx interrrupt */
    {
        for(i = 0; i < ENET_RXDESC_NUM; i++) {
            hal_enet_rx_desc_immediate_receive_complete_interrupt(&enet1_info, \
                                                                  (hal_enet_descriptors_struct *)enet1_info.rx_desc_list.rx_desc[i], \
                                                                  NULL);
        }
    }

[#if LWIP_LWIPEnable_TRUELWIPChecksumConfiguration_ChecksumByHardware == "Enable"]
    /* enable the TCP, UDP and ICMP checksum insertion for the Tx frames */
    for(i = 0; i < ENET_TXDESC_NUM; i++) {
        hal_enet_transmit_checksum_config((hal_enet_descriptors_struct *)enet1_info.tx_desc_list.tx_desc[i], \
                                           ENET_CHECKSUM_TCPUDPICMP_FULL);
    }
[/#if]

    /* note: TCP, UDP, ICMP checksum checking for received frame are enabled in DMA config */
    /* enable MAC and DMA transmission and reception */
    hal_enet_start(&enet1_info);
}

/**
 * This function should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @param p the MAC packet to send (e.g. IP packet including MAC addresses and type)
 * @return ERR_OK if the packet could be sent
 *         an err_t value if the packet couldn't be sent
 *
 * @note Returning ERR_MEM here if a DMA queue of your MAC is full can lead to
 *       strange results. You might consider waiting for space in the DMA queue
 *       to become availale since the stack doesn't retry to send a packet
 *       dropped because of memory failure (except for the TCP timers).
 */

static err_t low_level_output1(struct netif *netif, struct pbuf *p)
{
    struct pbuf *q;
    hal_enet_descriptors_struct *tx_desc;

    uint32_t i = 0U;
    uint32_t current_index = 0U;

    hal_enet_transfer_buffer_struct tx_buffer[ENET_TXDESC_NUM];
    memset(tx_buffer, 0U, ENET_TXDESC_NUM * sizeof(hal_enet_transfer_buffer_struct));

    current_index = enet1_info.tx_desc_list.tx_current_desc_index;
    tx_desc = (hal_enet_descriptors_struct *)enet1_info.tx_desc_list.tx_desc[current_index];

    while((uint32_t)RESET != ((tx_desc->status & ENET_TDES0_DAV) >> 31U));

    /* copy frame from pbufs to driver buffers */
    for(q = p; q != NULL; q = q->next) {
        if(i >= ENET_TXDESC_NUM) {
          return ERR_IF;
        } else {
          tx_buffer[i].buffer      = q->payload;
          tx_buffer[i].buffer_size = q->len;

          if(i > 0U) {
            tx_buffer[i-1].next_buff = &tx_buffer[i];
          } else {
              /* do nothing */
          }

          if(q->next == NULL) {
            tx_buffer[i].next_buff = NULL;
          } else {
              /* do nothing */
          }

          i++;
        }
[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
[#else]
        SCB_CleanDCache_by_Addr((uint32_t*)q->payload, q->len);
[/#if]
    }
[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
[#else]

    /* note: padding and CRC for transmitted frame are automatically inserted by DMA */
    SCB_CleanDCache_by_Addr((uint32_t*)tx_buffer, sizeof(tx_buffer));
    SCB_CleanDCache_by_Addr((uint32_t*)tx_desc, sizeof(hal_enet_descriptors_struct));
[/#if]

    /* transmit descriptors to give to DMA */
    tx1_packet_config.tx_packet_feature                = HAL_ENET_TX_FEATURE_CM;
    tx1_packet_config.checksum_mode                    = ENET_CHECKSUM_TCPUDPICMP_FULL;

    tx1_packet_config.tx_buffer                        = tx_buffer;
    tx1_packet_config.tx_buffer->next_buff             = tx_buffer->next_buff;
    tx1_packet_config.tx_buffer->secondary_buff        = NULL;
    tx1_packet_config.tx_buffer->buffer_size           = p->tot_len;
    tx1_packet_config.tx_buffer->secondary_buffer_size = 0U;
[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
[#else]

    SCB_CleanDCache_by_Addr((uint32_t*)&tx1_packet_config, sizeof(tx1_packet_config));
[/#if]
    hal_enet_frame_transmit(&enet1_info, &tx1_packet_config, ENET_DMA_TRANSMIT_TIMEOUT);

    return ERR_OK;
}

/**
 * Should allocate a pbuf and transfer the bytes of the incoming
 * packet from the interface into the pbuf.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return a pbuf filled with the received packet (including MAC header)
 *         NULL on memory error
 */
static struct pbuf *low_level_input1(struct netif *netif)
{
    struct pbuf *p, *q;
    uint8_t *buffer;
    hal_enet_descriptors_struct *rx_desc;

    uint16_t len = 0U;
    uint32_t l = 0;
    uint32_t current_index = 0U;

    p = NULL;

    /* obtain the size of the packet and put it into the "len" variable. */
    current_index = enet1_info.rx_desc_list.rx_current_desc_index;
    rx_desc = (hal_enet_descriptors_struct *)enet1_info.rx_desc_list.rx_desc[current_index];

    SCB_InvalidateDCache_by_Addr((uint32_t*)rx_desc, sizeof(hal_enet_descriptors_struct));

    len = hal_enet_desc_information_get(&enet1_info, rx_desc, RXDESC_FRAME_LENGTH);
    buffer = (uint8_t *)(hal_enet_desc_information_get(&enet1_info, rx_desc, RXDESC_BUFFER_1_ADDR));

    SCB_InvalidateDCache_by_Addr((uint32_t *)buffer, (len + 31) & ~31);

    /* we allocate a pbuf chain of pbufs from the Lwip buffer pool */
[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
    p = pbuf_alloc(PBUF_RAW, len, PBUF_POOL);
[#else]
    p = pbuf_alloc(PBUF_RAW, len, PBUF_RAM);
[/#if]

    if(p != NULL) {
        for(q = p; q != NULL; q = q->next) {
            SCB_InvalidateDCache_by_Addr((uint32_t *)q->payload, (q->len + 31) & ~31);
            memcpy((uint8_t *)q->payload, (u8_t *)&buffer[l], q->len);
[#if SYS_CPUDCacheDisable ??== true && SYS_CPUDCacheDisable == "TRUE"]
[#else]
            SCB_CleanDCache_by_Addr((uint32_t *)q->payload, (q->len + 31) & ~31);
[/#if]
            l = l + q->len;
        }
    }

    hal_enet_frame_receive(&enet1_info);

    return p;
}

/**
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface. Then the type of the received packet is determined and
 * the appropriate input function is called.
 *
 * @param netif the lwip network interface structure for this ethernetif
 */
err_t ethernetif_input1(struct netif *netif)
{
    err_t err;
    struct pbuf *p;

    /* move received packet into a new pbuf */
    p = low_level_input1(netif);

    /* no packet could be read, silently ignore this */
    if(p == NULL) {
        return ERR_MEM;
    }

    /* entry point to the LwIP stack */
    err = netif->input(p, netif);

    if(err != ERR_OK) {
        LWIP_DEBUGF(NETIF_DEBUG, ("ethernetif_input1: IP input error\n"));
        pbuf_free(p);
        p = NULL;
    }

    return err;
}

/**
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 * This function should be passed as a parameter to netif_add().
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return ERR_OK if the loopif is initialized
 *         ERR_MEM if private data couldn't be allocated
 *         any other err_t on error
 */
err_t ethernetif_init1(struct netif *netif)
{
    LWIP_ASSERT("netif != NULL", (netif != NULL));

#if LWIP_NETIF_HOSTNAME
    /* Initialize interface hostname */
    netif->hostname = "Gigadevice.COM_lwip";
#endif /* LWIP_NETIF_HOSTNAME */

    netif->name[0] = IFNAME3;
    netif->name[1] = IFNAME4;

    /* We directly use etharp_output() here to save a function call.
     * You can instead declare your own function an call etharp_output()
     * from it if you have to do some checks before sending (e.g. if link
     * is available...) */
    netif->output = etharp_output;
    netif->linkoutput = low_level_output1;

    /* initialize the hardware */
    low_level_init1(netif);

    return ERR_OK;
}
[/#if]
