/*!
    \file    hs0_usbd_core.c
    \brief   USB device mode core functions

    \version 2025-10-21, V1.0.0,  firmware for GD32H75E
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "usbd_core.h"
#include "usbd_enum.h"
#include "drv_usb_hw.h"

/*!
    \brief      flush the endpoint FIFOs
    \param[in]  udev: pointer to USB core instance
    \param[in]  ep_addr: endpoint address
                  in this parameter:
                    bit0..bit6: endpoint number (0..7)
                    bit7: endpoint direction which can be IN(1) or OUT(0)
    \param[out] none
    \retval     none
*/
uint32_t hs0_usbd_fifo_flush(usb_core_driver *udev, uint8_t ep_addr)
{
    if(EP_DIR(ep_addr)) {
        (void)usb_txfifo_flush(&udev->regs, EP_ID(ep_addr));
    } else {
        (void)usb_rxfifo_flush(&udev->regs);
    }

    return (0U);
}

/*!
    \brief      device connect
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     none
*/
void hs0_usbd_connect(usb_core_driver *udev)
{
#ifndef HS0_USE_OTG_MODE
    /* connect device */
    usb_dev_connect(udev);

    usb_mdelay(3U);
#endif /* HS0_USE_OTG_MODE */
}

/*!
    \brief      device disconnect
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     none
*/
void hs0_usbd_disconnect(usb_core_driver *udev)
{
#ifndef HS0_USE_OTG_MODE
    /* disconnect device for 3ms */
    usb_dev_disconnect(udev);

    usb_mdelay(3U);
#endif /* HS0_USE_OTG_MODE */
}

/*!
    \brief      initializes the USB device-mode stack and load the class driver
    \param[in]  udev: pointer to USB core instance
    \param[in]  desc: pointer to USB descriptor
    \param[in]  class_core: class driver
    \param[out] none
    \retval     none
*/
void hs0_usbd_init(usb_core_driver *udev, usb_desc *desc, usb_class_core *class_core)
{
    udev->dev.desc = desc;

    /* class callbacks */
    udev->dev.class_core = class_core;

    hs0_string_get();

    /* create serial string */
    hs0_serial_string_get(udev->dev.desc->strings[STR_IDX_SERIAL]);

    /* configure USB capabilities */
    (void)usb_basic_init(&udev->bp, &udev->regs);

    /* initializes the USB core*/
    (void)hs0_usb_core_init(udev->bp, &udev->regs);

    /* set device disconnect */
    hs0_usbd_disconnect(udev);

    /* initializes device mode */
    (void)hs0_usb_devcore_init(udev);

    /* set device connect */
    hs0_usbd_connect(udev);

    udev->dev.cur_status = (uint8_t)USBD_DEFAULT;
}
