[#ftl]
[#-- V1.2.0 --]
/*!
    \file    hs0_usbd_conf.c
    \brief   the source file of USB device configuration

    \version 2025-10-21, V1.0.0,  firmware for GD32H75E
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "usbd_core.h"
#include "usbd_enum.h"
#include "dfu_mem.h"
#include "dfu_core.h"

#define TIM_MSEC_DELAY                          0x01U
#define TIM_USEC_DELAY                          0x02U

__IO uint32_t hs0_delay_time = 0U;
__IO uint16_t timer_prescaler = 5U;

extern usb_core_driver hs0_usbd_core;

static void hw_time_set(uint8_t unit);
static void hw_delay(uint32_t ntime, uint8_t unit);

void hs0_usbd_basic_parameter_init(usb_core_basic *usb_basic);

/*!
    \brief      configure USBFS device initialization
    \param[in]  none
    \param[out] none
    \retval     none
*/
void msd_usbhs0_device_init(void)
{
${mwGetCode}
[#if GeneralConfiguration_Speed == "Full Speed"]
    hs0_usb_para_init(&hs0_usbd_core, USBHS0, USB_SPEED_FULL);
[#elseif GeneralConfiguration_Speed == "High Speed"]
    hs0_usb_para_init(&hs0_usbd_core, USBHS0, USB_SPEED_HIGH);
[#elseif GeneralConfiguration_Speed == "Low Speed"]
    hs0_usb_para_init(&hs0_usbd_core, USBHS0, USB_SPEED_LOW);
[/#if]
    hs0_usbd_basic_parameter_init(&hs0_usbd_core.bp);
    hs0_usbd_init (&hs0_usbd_core, &hs0_dfu_desc, &hs0_dfu_class);
}

/*!
    \brief      configure USB basic parameter
    \param[in]  usb_basic: pointer to USB capabilities
    \param[out] none
    \retval     none
*/
void hs0_usbd_basic_parameter_init(usb_core_basic *usb_basic)
{
[#if GeneralConfiguration_Speed == "Full Speed"]
    usb_basic->core_speed = (uint8_t)USB_SPEED_FULL;
[#elseif GeneralConfiguration_Speed == "High Speed"]
    usb_basic->core_speed = (uint8_t)USB_SPEED_HIGH;
[#elseif GeneralConfiguration_Speed == "High Speed"]
    usb_basic->core_speed = (uint8_t)USB_SPEED_LOW;
[/#if]
    usb_basic->sof_enable = HS0_USB_SOF_OUTPUT;
    usb_basic->low_power = HS0_USB_LOW_POWER;
    usb_basic->lpm_enable = HS0_LPM_ENABLE;
    usb_basic->vbus_sensing_enable = HS0_VBUS_SENSING;
}

/*!
    \brief      configure  endpoint FIFO size
    \param[in]  usb_regs: USB core registers
    \param[out] none
    \retval     none
*/
void hs0_usbd_fifo_size_init(usb_core_regs *usb_regs)
{
    usb_set_rxfifo(usb_regs, HS0_RX_FIFO_SIZE);
    usb_set_txfifo(usb_regs, 0, HS0_TX0_FIFO_SIZE);
    usb_set_txfifo(usb_regs, 1, HS0_TX1_FIFO_SIZE);
    usb_set_txfifo(usb_regs, 2, HS0_TX2_FIFO_SIZE);
    usb_set_txfifo(usb_regs, 3, HS0_TX3_FIFO_SIZE);
    usb_set_txfifo(usb_regs, 4, HS0_TX4_FIFO_SIZE);
    usb_set_txfifo(usb_regs, 5, HS0_TX5_FIFO_SIZE);
    usb_set_txfifo(usb_regs, 6, HS0_TX6_FIFO_SIZE);
    usb_set_txfifo(usb_regs, 7, HS0_TX7_FIFO_SIZE);
}

/*!
    \brief      delay in microseconds
    \param[in]  usec: value of delay required in microseconds
    \param[out] none
    \retval     none
*/
void usb_udelay(const uint32_t usec)
{
    hw_delay(usec, TIM_USEC_DELAY);
}

/*!
    \brief      delay in milli seconds
    \param[in]  msec: value of delay required in milli seconds
    \param[out] none
    \retval     none
*/
void usb_mdelay(const uint32_t msec)
{
    hw_delay(msec, TIM_MSEC_DELAY);
}

/*!
    \brief      delay routine based on TIMER2
    \param[in]  nTime: delay Time
    \param[in]  unit: delay Time unit = milliseconds / microseconds
    \param[out] none
    \retval     none
*/
static void hw_delay(uint32_t ntime, uint8_t unit)
{
    hs0_delay_time = ntime;

    hw_time_set(unit);

    while(0U != hs0_delay_time) {
    }

    timer_disable(TIMER2);
}

/*!
    \brief      configures TIMER for delay routine based on Timer2
    \param[in]  unit: msec /usec
    \param[out] none
    \retval     none
*/
static void hw_time_set(uint8_t unit)
{
    timer_parameter_struct  timer_basestructure;

    timer_prescaler = ((rcu_clock_freq_get(CK_APB1) / 1000000U * 2U) / 12U) - 1U;

    timer_disable(TIMER2);
    timer_interrupt_disable(TIMER2, TIMER_INT_UP);

    if(TIM_USEC_DELAY == unit) {
        timer_basestructure.period = 11U;
    } else if(TIM_MSEC_DELAY == unit) {
        timer_basestructure.period = 11999U;
    } else {
        /* no operation */
    }

    timer_basestructure.prescaler         = timer_prescaler;
    timer_basestructure.alignedmode       = TIMER_COUNTER_EDGE;
    timer_basestructure.counterdirection  = TIMER_COUNTER_UP;
    timer_basestructure.clockdivision     = TIMER_CKDIV_DIV1;
    timer_basestructure.repetitioncounter = 0U;

    timer_init(TIMER2, &timer_basestructure);

    timer_interrupt_flag_clear(TIMER2, TIMER_INT_FLAG_UP);

    timer_auto_reload_shadow_enable(TIMER2);

    /* TIMER2 interrupt enable */
    timer_interrupt_enable(TIMER2, TIMER_INT_UP);

    /* TIMER2 enable counter */
    timer_enable(TIMER2);
}

