[#ftl]
/*!
    \file    hs_usbh_usr.c
    \brief   some user routines

    \version 2025-07-30, V1.0.0, firmware for GD32F5xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include <string.h>
#include "hs_usbh_usr.h"
#include "drv_usb_hw.h"
#include "usbh_msc_core.h"
#include "usbh_msc_scsi.h"
#include "usbh_msc_bbb.h"
#include "ff.h"

/* External Includes*/
/* user code [External Includes] begin */

/* user code [External Includes] end */

/* Private Type Definitions */
/* user code [Private Type Definitions] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
#define User_Debug_Log(...) do { \
                                 \
                                 printf(__VA_ARGS__);\
                               } while (0)
#else
#define User_Debug_Log(__type__,...) \
    do { \
        User_Debug_LCD_Log(__type__, __VA_ARGS__); \
    } while (0)
#endif
__ALIGN_BEGIN char ReadTextBuff[100] __ALIGN_END;
__ALIGN_BEGIN char WriteTextBuff1[] __ALIGN_END = "GD32 USB Host Demo application using FAT_FS   ";
uint16_t bytesWritten, bytesToWrite, bytesRead;
uint8_t line_idx;
uint8_t hs_usbh_usr_application_state = HS_USBH_USR_FS_INIT;
FIL file;

#ifdef USE_USBFS_HOST_MSC
extern FATFS fatfs[2];
#else
FATFS fatfs[2];
PARTITION VolToPart[] = {0x00};
#endif
/* user code [Private Type Definitions] end */

/* user code [Private Macros] begin */

/* user code [Private Macros] end */

/* Private Constants */
/* user code [Private Constants] begin */

/* user code [Private Constants] end */

/* Extern Variables */
/* user code [Extern Variables] begin */
extern usb_core_driver hs_usbh_core;
extern hs_usbh_host hs_usb_host;
/* user code [Extern Variables] end */

/* user code [Private Function Declaration] begin */
/* local function prototypes ('static') */
static uint8_t explore_disk(char *path, uint8_t recu_level);
/* user code [Private Function Declaration] end */

/* Points to the DEVICE_PROP structure of current device */
usbh_user_cb hs_usr_cb =
{
    hs_usbh_user_init,
    hs_usbh_user_deinit,
    hs_usbh_user_device_connected,
    hs_usbh_user_device_reset,
    hs_usbh_user_device_disconnected,
    hs_usbh_user_over_current_detected,
    hs_usbh_user_device_speed_detected,
    hs_usbh_user_device_desc_available,
    hs_usbh_user_device_address_assigned,
    hs_usbh_user_configuration_descavailable,
    hs_usbh_user_manufacturer_string,
    hs_usbh_user_product_string,
    hs_usbh_user_serialnum_string,
    hs_usbh_user_enumeration_finish,
    hs_usbh_user_userinput,
    hs_usbh_user_msc_application,
    hs_usbh_user_device_not_supported,
    hs_usbh_user_unrecovered_error
};

const uint8_t HS_MSG_HOST_HEADER[] = "USBHS MSC Host";
const uint8_t HS_MSG_HOST_FOOTER[] = "USB Host Library v3.0.0";

/*!
    \brief      user operation for host-mode initialization
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_init(void)
{
    static uint8_t startup = 0U;
    /* user code [hs_usbh_user_init local 0] begin */

    /* user code [hs_usbh_user_init local 0] end */
    if(0U == startup) {
        startup = 1U;
        /* user code [hs_usbh_user_init local 1] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> USB host library started \r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "> USB host library started \r\n");
#endif
        /* user code [hs_usbh_user_init local 1] end */
    }
    /* user code [hs_usbh_user_init local 2] begin */

    /* user code [hs_usbh_user_init local 2] end */
}

/*!
    \brief      deinit user state and associated variables
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_deinit(void)
{
    /* user code [hs_usbh_user_deinit local 0] begin */
    hs_usbh_usr_application_state = HS_USBH_USR_FS_INIT;
    /* user code [hs_usbh_user_deinit local 0] end */
}

/*!
    \brief      user operation for device attached
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_device_connected(void)
{
    /* user code [hs_usbh_user_device_connected local 0] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Device Attached.\r\n");
#else
    User_Debug_Log(LCD_LOG_TYPE_USER, "> Device Attached.\r\n");
#endif
    /* user code [hs_usbh_user_device_connected local 0] end */
}

/*!
    \brief      user operation for unrecovered error happens
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_unrecovered_error(void)
{
    /* user code [hs_usbh_user_unrecovered_error local 0] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Unrecovered Error State.\r\n");
#else
    User_Debug_Log(LCD_LOG_TYPE_ERROR, "> Unrecovered Error State.\r\n");
#endif
    /* user code [hs_usbh_user_unrecovered_error local 0] end */
}

/*!
    \brief      user operation for device disconnect event
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_device_disconnected(void)
{
    /* user code [hs_usbh_user_device_disconnected local 0] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Device Disconnected.\r\n");
#else
    User_Debug_Log(LCD_LOG_TYPE_USER, "> Device Disconnected.\r\n");
#endif
    /* user code [hs_usbh_user_device_disconnected local 0] end */
}

/*!
    \brief      user operation for reset USB Device
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_device_reset(void)
{
    /* user code [hs_usbh_user_device_reset local 0] begin */
    /* users can do their application actions here for the USB-Reset */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Reset the USB device.\r\n");
#else
    User_Debug_Log(LCD_LOG_TYPE_USER, "> Reset the USB device.\r\n");
#endif
    /* user code [hs_usbh_user_device_reset local 0] end */
}

/*!
    \brief      user operation for detecting device speed
    \param[in]  DeviceSpeed: device speed
    \param[out] none
    \retval     none
*/
void hs_usbh_user_device_speed_detected(uint32_t device_speed)
{
    /* user code [hs_usbh_user_device_speed_detected local 0] begin */
    if (PORT_SPEED_HIGH == device_speed) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> High speed device detected.\r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "> High speed device detected.\r\n");
#endif
    } else if(PORT_SPEED_FULL == device_speed) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> Full speed device detected.\r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "> Full speed device detected.\r\n");
#endif
    } else if(PORT_SPEED_LOW == device_speed) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> Low speed device detected.\r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "> Low speed device detected.\r\n");
#endif
    } else {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> Device Fault.\r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_ERROR, "> Device Fault.\r\n");
#endif
    }
    /* user code [hs_usbh_user_device_speed_detected local 0] end */
}

/*!
    \brief      user operation when device descriptor is available
    \param[in]  device_desc: device descriptor
    \param[out] none
    \retval     none
*/
void hs_usbh_user_device_desc_available(void *device_desc)
{
    /* user code [hs_usbh_user_device_desc_available local 0] begin */
    usb_desc_dev *pDevStr = device_desc;
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> VID: %04Xh\r\n", (uint32_t)pDevStr->idVendor);
    User_Debug_Log("> PID: %04Xh\r\n", (uint32_t)pDevStr->idProduct);
#else
    User_Debug_Log(LCD_LOG_TYPE_INFO, "> VID: %04Xh\r\n", (uint32_t)pDevStr->idVendor);
    User_Debug_Log(LCD_LOG_TYPE_INFO, "> PID: %04Xh\r\n", (uint32_t)pDevStr->idProduct);
#endif
    /* user code [hs_usbh_user_device_desc_available local 0] end */
}

/*!
    \brief      USB device is successfully assigned the address
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_device_address_assigned(void)
{
    /* user code [hs_usbh_user_device_address_assigned local 0] begin */

    /* user code [hs_usbh_user_device_address_assigned local 0] end */
}

/*!
    \brief      user operation when configuration descriptor is available
    \param[in]  cfg_desc: pointer to configuration descriptor
    \param[in]  itf_desc: pointer to interface descriptor
    \param[in]  ep_desc: pointer to endpoint descriptor
    \param[out] none
    \retval     none
*/
void hs_usbh_user_configuration_descavailable(usb_desc_config *cfg_desc,
                                               usb_desc_itf *itf_desc,
                                               usb_desc_ep *ep_desc)
{
    usb_desc_itf *id = itf_desc;
    /* user code [hs_usbh_user_configuration_descavailable local 0] begin */
    if (0x08U == (*id).bInterfaceClass) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> Mass storage device connected.\r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "> Mass storage device connected.\r\n");
#endif
    } else if (0x03U == (*id).bInterfaceClass) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> HID device connected.\r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "> HID device connected.\r\n");
#endif
    }
    /* user code [hs_usbh_user_configuration_descavailable local 0] end */
}

/*!
    \brief      user operation when manufacturer string exists
    \param[in]  manufacturer_string: manufacturer string of usb device
    \param[out] none
    \retval     none
*/
void hs_usbh_user_manufacturer_string(void *manufacturer_string)
{
    /* user code [hs_usbh_user_manufacturer_string local 0] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Manufacturer: %s\r\n", (char *)manufacturer_string);
#else
    User_Debug_Log(LCD_LOG_TYPE_INFO, "> Manufacturer: %s\r\n", (char *)manufacturer_string);
#endif
    /* user code [hs_usbh_user_manufacturer_string local 0] end */
}

/*!
    \brief      user operation when manufacturer string exists
    \param[in]  product_string: product string of usb device
    \param[out] none
    \retval     none
*/
void hs_usbh_user_product_string(void *product_string)
{
    /* user code [hs_usbh_user_product_string local 0] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Product: %s\r\n", (char *)product_string);
#else
    User_Debug_Log(LCD_LOG_TYPE_INFO, "> Product: %s\r\n", (char *)product_string);
#endif
    /* user code [hs_usbh_user_product_string local 0] end */
}

/*!
    \brief      user operation when serialnum string exists
    \param[in]  serial_num_string: serialNum string of usb device
    \param[out] none
    \retval     none
*/
void hs_usbh_user_serialnum_string(void *serial_num_string)
{
    /* user code [hs_usbh_user_serialnum_string local 0] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Serial Number: %s\r\n", (char *)serial_num_string);
#else
    User_Debug_Log(LCD_LOG_TYPE_INFO, "> Serial Number: %s\r\n", (char *)serial_num_string);
#endif
    /* user code [hs_usbh_user_serialnum_string local 0] end */
}

/*!
    \brief      user response request is displayed to ask for application jump to class
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_enumeration_finish(void)
{
    /* user code [hs_usbh_user_enumeration_finish local 0] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Enumeration completed.\r\n");
    User_Debug_Log("-------------------------------\r\n");
    User_Debug_Log("> To see the disk information:\r\n");
#else
    User_Debug_Log(LCD_LOG_TYPE_USER, "> Enumeration completed.\r\n");
    User_Debug_Log(LCD_LOG_TYPE_USER, "-------------------------------\r\n");
    User_Debug_Log(LCD_LOG_TYPE_USER, "> To see the disk information:\r\n");
#endif
    /* user code [hs_usbh_user_enumeration_finish local 0] end */
}

/*!
    \brief      user operation when device is not supported
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_device_not_supported(void)
{
    /* user code [hs_usbh_user_device_not_supported local 0] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Device not supported.\r\n");
#else
    User_Debug_Log(LCD_LOG_TYPE_ERROR, "> Device not supported.\r\n");
#endif
    /* user code [hs_usbh_user_device_not_supported local 0] end */
}

/*!
    \brief      user action for application state entry
    \param[in]  none
    \param[out] none
    \retval     user response for user key
*/
usbh_user_status hs_usbh_user_userinput(void)
{
    usbh_user_status usbh_usr_status = USR_IN_NO_RESP;
    /* user code [hs_usbh_user_userinput local 0] begin */
    usbh_usr_status = USR_IN_RESP_OK;
    /* user code [hs_usbh_user_userinput local 0] end */

    return usbh_usr_status;
}

/*!
    \brief      user action for device overcurrent detection event
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs_usbh_user_over_current_detected(void)
{
    /* user code [hs_usbh_user_over_current_detected local 0] begin */
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
    User_Debug_Log("> Overcurrent detected.\r\n");
#else
    User_Debug_Log(LCD_LOG_TYPE_ERROR, "> Overcurrent detected.\r\n");
#endif
    /* user code [hs_usbh_user_over_current_detected local 0] end */
}

/*!
    \brief      init mouse window
    \param[in]  none
    \param[out] none
    \retval     none
*/
int hs_usbh_user_msc_application(void)
{
    /* user code [hs_usbh_user_msc_application local 0] begin */
    FRESULT res;
    msc_lun info;
    char hs_disk_number[4];
    snprintf(hs_disk_number, sizeof(hs_disk_number), "%d:/", HS_USBH_USR_MSC_DISK);
    uint8_t WriteTextBuff[] = "GD32 Connectivity line Host Demo application using FAT_FS   ";
    uint16_t bytesWritten, bytesToWrite;

    switch(hs_usbh_usr_application_state) {
    case HS_USBH_USR_FS_INIT:
        /* initializes the file system*/
        if(FR_OK != f_mount(&fatfs[1], hs_disk_number, 0)) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
            User_Debug_Log("> Cannot initialize File System.\r\n");
#else
            User_Debug_Log(LCD_LOG_TYPE_ERROR, "> Cannot initialize File System.\r\n");
#endif
            return(-1);
        }
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> File System initialized.\r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "> File System initialized.\r\n");
#endif
        if(USBH_OK == hs_usbh_msc_lun_info_get(&hs_usb_host, 0, &info)) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
            User_Debug_Log("> Disk capacity: %ud Bytes.\r\n", info.capacity.block_nbr * info.capacity.block_size);
#else
            User_Debug_Log(LCD_LOG_TYPE_USER, "> Disk capacity: %ud Bytes.\r\n", info.capacity.block_nbr * info.capacity.block_size);
#endif
        } else {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
            User_Debug_Log("> Cannot get disk information.\r\n");
#else
            User_Debug_Log(LCD_LOG_TYPE_ERROR, "> Cannot get disk information.\r\n");
#endif
        }

        hs_usbh_usr_application_state = HS_USBH_USR_FS_READLIST;
        break;

    case HS_USBH_USR_FS_READLIST:
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> Exploring disk flash ...\r\n");
        User_Debug_Log("> To see the root content of disk \r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "> Exploring disk flash ...\r\n");
        User_Debug_Log(LCD_LOG_TYPE_USER, "> To see the root content of disk \r\n");
#endif
        usb_mdelay(100);

        explore_disk(hs_disk_number, 1);
        line_idx = 0;
        hs_usbh_usr_application_state = HS_USBH_USR_FS_WRITEFILE;
        break;

    case HS_USBH_USR_FS_WRITEFILE:
        usb_mdelay(100);
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("-------------------------------\r\n");
        User_Debug_Log("> Writing File to disk flash...\r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "-------------------------------\r\n");
        User_Debug_Log(LCD_LOG_TYPE_USER, "> Writing File to disk flash...\r\n");
#endif
        /* register work area for logical drives */
        f_mount(&fatfs[1], hs_disk_number, 1);
        char file_path[11];
        snprintf(file_path, sizeof(file_path), "%d:GD32.TXT", HS_USBH_USR_MSC_DISK);
        if(FR_OK == f_open(&file, file_path, FA_CREATE_ALWAYS | FA_WRITE)) {
            /* write buffer to file */
            bytesToWrite = sizeof(WriteTextBuff);
            res = f_write(&file, WriteTextBuff, bytesToWrite, (void *)&bytesWritten);
            /* EOF or error */
            if((0U == bytesWritten) || (FR_OK != res)) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
                User_Debug_Log("> GD32.TXT cannot be written.\r\n");
#else
                User_Debug_Log(LCD_LOG_TYPE_ERROR, "> GD32.TXT cannot be written.\r\n");
#endif
            } else {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
                User_Debug_Log("> GD32.TXT created in the disk.\r\n");
#else
                User_Debug_Log(LCD_LOG_TYPE_USER, "> GD32.TXT created in the disk.\r\n");
#endif
            }

            /* close file and filesystem */
            f_close(&file);
            f_mount(NULL, hs_disk_number, 1);
        } else {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
                User_Debug_Log("> GD32.TXT cannot be created.\r\n");
#else
                User_Debug_Log(LCD_LOG_TYPE_ERROR, "> GD32.TXT cannot be created.\r\n");
#endif
        }

        hs_usbh_usr_application_state = HS_USBH_USR_FS_DEMOEND;
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
        User_Debug_Log("> The MSC host demo is end.\r\n");
#else
        User_Debug_Log(LCD_LOG_TYPE_USER, "> The MSC host demo is end.\r\n");
#endif
        break;

    case HS_USBH_USR_FS_DEMOEND:
        break;

    default:
        break;
    }
    /* user code [hs_usbh_user_msc_application local 0] end */

    return(0);
}


/* user code [Private Function Implementations] begin */
/*!
    \brief      displays disk content
    \param[in]  path: pointer to root path
    \param[in]  recu_level: recursive level
    \param[out] none
    \retval     status
*/
static uint8_t explore_disk(char *path, uint8_t recu_level)
{
    FRESULT res;
    FILINFO fno;
    DIR dir;
    char *fn;

    res = f_opendir(&dir, path);

    if(FR_OK == res) {
        while((hs_usbh_core.host.connect_status)) {
            res = f_readdir(&dir, &fno);
            if(FR_OK != res || 0U == fno.fname[0]) {
                break;
            }

            if('.' == fno.fname[0]) {
                continue;
            }

            fn = fno.fname;

            line_idx++;

            if(line_idx > 4U) {
                line_idx = 0U;
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
                User_Debug_Log("                             \r\n");
#else
                User_Debug_Log(LCD_LOG_TYPE_USER, "                             \r\n");
#endif
            }

            if(1U == recu_level) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
                User_Debug_Log(">    |__ \r\n");
#else
                User_Debug_Log(LCD_LOG_TYPE_USER, ">    |__ \r\n");
#endif
            } else if(2U == recu_level) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
                User_Debug_Log(">    |   |__ \r\n");
#else
                User_Debug_Log(LCD_LOG_TYPE_USER, ">    |   |__ \r\n");
#endif
            }

            if(AM_DIR == fno.fattrib) {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
                User_Debug_Log("%s\r\n", fno.fname);
#else
                User_Debug_Log(LCD_LOG_TYPE_USER, "%s\r\n", fno.fname);
#endif
            } else {
#if (1U ==  HS_HOST_USER_DEBUG_LOG)
                User_Debug_Log("%s\r\n", fno.fname);
#else
                User_Debug_Log(LCD_LOG_TYPE_USER, "%s\r\n", fno.fname);
#endif
            }

            if((AM_DIR == fno.fattrib) && (1U == recu_level)) {
                explore_disk(fn, 2);
            }
        }
    }

    return res;
}

/* user code [Private Function Implementations] end */

/* user code [Public Functions Implementations] begin */

/* user code [Public Functions Implementations] end */