/*!
    \file    fs_audio_core.c
    \brief   USB audio device class core functions

    \version 2025-02-23, V1.0.0, firmware for GD32F5xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "audio_out_itf.h"
#include "fs_audio_core.h"
#include <string.h>
#include <math.h>
#include "usbd_endpoint.h"
#include "usbd_core.h"

#define VOL_MIN                      0U    /* volume Minimum Value */
#define VOL_MAX                      100U  /* volume Maximum Value */
#define VOL_RES                      1U    /* volume Resolution */
#define VOL_0dB                      70U   /* 0dB is in the middle of VOL_MIN and VOL_MAX */

#ifdef USE_USB_AD_MICPHONE
extern volatile uint32_t count_data;
extern const char wavetestdata[];
#define LENGTH_DATA    (1747 * 32)
#endif /* USE_USB_AD_MICPHONE */

__ALIGN_BEGIN fs_usbd_audio_handler fs_audio_handler __ALIGN_END;

extern void *fs_uac_strings_desc[];

/* local function prototypes ('static') */
static uint8_t fs_audio_init (usb_dev *udev, uint8_t config_index);
static uint8_t fs_audio_deinit (usb_dev *udev, uint8_t config_index);
static uint8_t fs_audio_req_handler (usb_dev *udev, usb_req *req);
static uint8_t fs_audio_set_intf (usb_dev *udev, usb_req *req);
static uint8_t fs_audio_ctlx_out (usb_dev *udev);
static uint8_t fs_audio_data_in (usb_dev *udev, uint8_t ep_num);
static uint8_t fs_audio_data_out (usb_dev *udev, uint8_t ep_num);
static uint8_t fs_audio_sof (usb_dev *udev);
static uint8_t fs_audio_iso_in_incomplete (usb_dev *udev);
static uint8_t fs_audio_iso_out_incomplete (usb_dev *udev);
static uint32_t fs_usbd_audio_spk_get_feedback(usb_dev *udev);
static void fs_usbd_get_feedback_fs_rate(uint32_t rate, uint8_t *buf);
static uint32_t FS_I2S_ACTUAL_SAM_FREQ(uint32_t set_freq);

usb_class_core fs_usbd_audio_cb = {
    .init      = fs_audio_init,
    .deinit    = fs_audio_deinit,
    .req_proc  = fs_audio_req_handler,
    .set_intf  = fs_audio_set_intf,
    .ctlx_out  = fs_audio_ctlx_out,
    .data_in   = fs_audio_data_in,
    .data_out  = fs_audio_data_out,
    .SOF       = fs_audio_sof,
    .incomplete_isoc_in = fs_audio_iso_in_incomplete,
    .incomplete_isoc_out = fs_audio_iso_out_incomplete
};

/* note:it should use the c99 standard when compiling the below codes */
/* USB standard device descriptor */
__ALIGN_BEGIN const usb_desc_dev fs_audio_dev_desc __ALIGN_END =
{
    .header =
     {
         .bLength          = USB_DEV_DESC_LEN,
         .bDescriptorType  = USB_DESCTYPE_DEV
     },
    .bcdUSB                = 0x0200U,
    .bDeviceClass          = 0x00U,
    .bDeviceSubClass       = 0x00U,
    .bDeviceProtocol       = 0x00U,
    .bMaxPacketSize0       = USB_EP0_MAX_LEN,
    .idVendor              = FS_VID_STR,
    .idProduct             = FS_PID_STR,
    .bcdDevice             = 0x0100U,
    .iManufacturer         = STR_IDX_MFC,
    .iProduct              = STR_IDX_PRODUCT,
    .iSerialNumber         = STR_IDX_SERIAL,
    .bNumberConfigurations = FS_DEVICE_MAX_CONF_NUM
};

/* USB device configuration descriptor */
__ALIGN_BEGIN const fs_usb_desc_config_set fs_audio_config_set __ALIGN_END =
{
    .config =
    {
        .header =
         {
             .bLength         = sizeof(usb_desc_config),
             .bDescriptorType = USB_DESCTYPE_CONFIG
         },
        .wTotalLength         = FS_AD_CONFIG_DESC_SET_LEN,
        .bNumInterfaces       = 0x01U + FS_CONFIG_DESC_AS_ITF_COUNT,
        .bConfigurationValue  = 0x01U,
        .iConfiguration       = 0x00U,
        .bmAttributes         = 0x80U | (FS_DEVICE_SELF_POWERED << 6U),
        .bMaxPower            = 0x32U
    },

    .std_itf =
    {
        .header =
         {
             .bLength         = sizeof(usb_desc_itf),
             .bDescriptorType = USB_DESCTYPE_ITF
         },
         .bInterfaceNumber    = 0x00U,
         .bAlternateSetting   = 0x00U,
         .bNumEndpoints       = 0x00U,
         .bInterfaceClass     = FS_USB_CLASS_AUDIO,
         .bInterfaceSubClass  = FS_AD_SUBCLASS_CONTROL,
         .bInterfaceProtocol  = FS_AD_PROTOCOL_UNDEFINED,
         .iInterface          = 0x00U
    },

    .ac_itf =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_AC_itf),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = 0x01U,
         .bcdADC              = 0x0100U,
         .wTotalLength        = FS_AC_ITF_TOTAL_LEN,
         .bInCollection       = FS_CONFIG_DESC_AS_ITF_COUNT,
#ifdef USE_USB_AD_MICPHONE
         .baInterfaceNr0       = 0x01U,
#endif /* USE_USB_AD_MICPHONE */

#ifdef USE_USB_AD_SPEAKER
         .baInterfaceNr1       = 0x02U
#endif /* USE_USB_AD_SPEAKER */
    },

#ifdef USE_USB_AD_MICPHONE
    .mic_in_terminal =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_input_terminal),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = 0x02U,
         .bTerminalID         = 0x01U,
         .wTerminalType       = 0x0201U,
         .bAssocTerminal      = 0x00U,
         .bNrChannels         = 0x02U,
         .wChannelConfig      = 0x0003U,
         .iChannelNames       = 0x00U,
         .iTerminal           = 0x00U
    },

    .mic_feature_unit =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_mono_feature_unit),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = FS_AD_CONTROL_FEATURE_UNIT,
         .bUnitID             = FS_AD_IN_STREAMING_CTRL,
         .bSourceID           = 0x01U,
         .bControlSize        = 0x01U,
         .bmaControls0        = FS_AD_CONTROL_MUTE | FS_AD_CONTROL_VOLUME,
         .bmaControls1        = 0x00U,
         .iFeature            = 0x00U
    },

    .mic_out_terminal =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_output_terminal),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = FS_AD_CONTROL_OUTPUT_TERMINAL,
         .bTerminalID         = 0x03U,
         .wTerminalType       = 0x0101U,
         .bAssocTerminal      = 0x00U,
         .bSourceID           = 0x02U,
         .iTerminal           = 0x00U
    },
#endif /* USE_USB_AD_MICPHONE */

#ifdef USE_USB_AD_SPEAKER
    .speak_in_terminal =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_input_terminal),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = FS_AD_CONTROL_INPUT_TERMINAL,
         .bTerminalID         = 0x04U,
         .wTerminalType       = 0x0101U,
         .bAssocTerminal      = 0x00U,
         .bNrChannels         = 0x02U,
         .wChannelConfig      = 0x0003U,
         .iChannelNames       = 0x00U,
         .iTerminal           = 0x00U
    },

    .speak_feature_unit =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_mono_feature_unit),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = FS_AD_CONTROL_FEATURE_UNIT,
         .bUnitID             = FS_AD_OUT_STREAMING_CTRL,
         .bSourceID           = 0x04U,
         .bControlSize        = 0x01U,
         .bmaControls0        = FS_AD_CONTROL_MUTE | FS_AD_CONTROL_VOLUME,
         .bmaControls1        = 0x00U,
         .iFeature            = 0x00U
    },

    .speak_out_terminal =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_output_terminal),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = FS_AD_CONTROL_OUTPUT_TERMINAL,
         .bTerminalID         = 0x06U,
         .wTerminalType       = 0x0301U,
         .bAssocTerminal      = 0x00U,
         .bSourceID           = 0x05U,
         .iTerminal           = 0x00U
    },
#endif /* USE_USB_AD_SPEAKER */

#ifdef USE_USB_AD_MICPHONE
    .mic_std_as_itf_zeroband =
    {
        .header =
         {
             .bLength         = sizeof(usb_desc_itf),
             .bDescriptorType = USB_DESCTYPE_ITF
         },
         .bInterfaceNumber    = 0x01U,
         .bAlternateSetting   = 0x00U,
         .bNumEndpoints       = 0x00U,
         .bInterfaceClass     = FS_USB_CLASS_AUDIO,
         .bInterfaceSubClass  = FS_AD_SUBCLASS_AUDIOSTREAMING,
         .bInterfaceProtocol  = FS_AD_PROTOCOL_UNDEFINED,
         .iInterface          = 0x00U
    },

    .mic_std_as_itf_opera =
    {
        .header =
         {
             .bLength         = sizeof(usb_desc_itf),
             .bDescriptorType = USB_DESCTYPE_ITF
         },
         .bInterfaceNumber    = 0x01U,
         .bAlternateSetting   = 0x01U,
         .bNumEndpoints       = 0x01U,
         .bInterfaceClass     = FS_USB_CLASS_AUDIO,
         .bInterfaceSubClass  = FS_AD_SUBCLASS_AUDIOSTREAMING,
         .bInterfaceProtocol  = FS_AD_PROTOCOL_UNDEFINED,
         .iInterface          = 0x00U
    },

    .mic_as_itf =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_AS_itf),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = FS_AD_STREAMING_GENERAL,
         .bTerminalLink       = 0x03U,
         .bDelay              = 0x01U,
         .wFormatTag          = 0x0001U,
    },

    .mic_format_typeI =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_format_type),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = FS_AD_STREAMING_FORMAT_TYPE,
         .bFormatType         = FS_AD_FORMAT_TYPE_I,
         .bNrChannels         = FS_MIC_IN_CHANNEL_NBR,
         .bSubFrameSize       = 0x02U,
         .bBitResolution      = FS_MIC_IN_BIT_RESOLUTION,
         .bSamFreqType        = 0x01U,
         .bSamFreq[0]         = (uint8_t)FS_USBD_MIC_FREQ,
         .bSamFreq[1]         = FS_USBD_MIC_FREQ >> 8U,
         .bSamFreq[2]         = FS_USBD_MIC_FREQ >> 16U
    },

    .mic_std_endpoint =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_std_ep),
             .bDescriptorType = USB_DESCTYPE_EP
         },
         .bEndpointAddress    = FS_AD_IN_EP,
         .bmAttributes        = FS_USB_ENDPOINT_TYPE_ISOCHRONOUS,
         .wMaxPacketSize      = FS_MIC_IN_PACKET,
         .bInterval           = 0x01U,
         .bRefresh            = 0x00U,
         .bSynchAddress       = 0x00U
    },

    .mic_as_endpoint =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_AS_ep),
             .bDescriptorType = FS_AD_DESCTYPE_ENDPOINT
         },
         .bDescriptorSubtype  = FS_AD_ENDPOINT_GENERAL,
         .bmAttributes        = 0x00U,
         .bLockDelayUnits     = 0x00U,
         .wLockDelay          = 0x0000U,
    },
#endif /* USE_USB_AD_MICPHONE */

#ifdef USE_USB_AD_SPEAKER
    .speak_std_as_itf_zeroband =
    {
        .header =
         {
             .bLength         = sizeof(usb_desc_itf),
             .bDescriptorType = USB_DESCTYPE_ITF
         },
         .bInterfaceNumber    = 0x02U,
         .bAlternateSetting   = 0x00U,
         .bNumEndpoints       = 0x00U,
         .bInterfaceClass     = FS_USB_CLASS_AUDIO,
         .bInterfaceSubClass  = FS_AD_SUBCLASS_AUDIOSTREAMING,
         .bInterfaceProtocol  = FS_AD_PROTOCOL_UNDEFINED,
         .iInterface          = 0x00U
    },

    .speak_std_as_itf_opera =
    {
        .header =
         {
             .bLength         = sizeof(usb_desc_itf),
             .bDescriptorType = USB_DESCTYPE_ITF
         },
         .bInterfaceNumber    = 0x02U,
         .bAlternateSetting   = 0x01U,
         .bNumEndpoints       = 0x02U,
         .bInterfaceClass     = FS_USB_CLASS_AUDIO,
         .bInterfaceSubClass  = FS_AD_SUBCLASS_AUDIOSTREAMING,
         .bInterfaceProtocol  = FS_AD_PROTOCOL_UNDEFINED,
         .iInterface          = 0x00U
    },

    .speak_as_itf =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_AS_itf),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = FS_AD_STREAMING_GENERAL,
         .bTerminalLink       = 0x04U,
         .bDelay              = 0x01U,
         .wFormatTag          = 0x0001U,
    },

    .speak_format_typeI =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_format_type),
             .bDescriptorType = FS_AD_DESCTYPE_INTERFACE
         },
         .bDescriptorSubtype  = FS_AD_STREAMING_FORMAT_TYPE,
         .bFormatType         = FS_AD_FORMAT_TYPE_I,
         .bNrChannels         = FS_SPEAKER_OUT_CHANNEL_NBR,
         .bSubFrameSize       = 0x02U,
         .bBitResolution      = FS_SPEAKER_OUT_BIT_RESOLUTION,
         .bSamFreqType        = 0x01U,
         .bSamFreq[0]         = (uint8_t)FS_USBD_SPEAKER_FREQ,
         .bSamFreq[1]         = FS_USBD_SPEAKER_FREQ >> 8U,
         .bSamFreq[2]         = FS_USBD_SPEAKER_FREQ >> 16U
    },

    .speak_std_endpoint =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_std_ep),
             .bDescriptorType = USB_DESCTYPE_EP
         },
         .bEndpointAddress    = FS_AD_OUT_EP,
         .bmAttributes        = USB_EP_ATTR_ISO | USB_EP_ATTR_ASYNC,
         .wMaxPacketSize      = FS_SPEAKER_OUT_PACKET,
         .bInterval           = 0x01U,
         .bRefresh            = 0x00U,
         .bSynchAddress       = FS_AD_FEEDBACK_IN_EP,
    },

    .speak_as_endpoint =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_AS_ep),
             .bDescriptorType = FS_AD_DESCTYPE_ENDPOINT
         },
         .bDescriptorSubtype  = FS_AD_ENDPOINT_GENERAL,
         .bmAttributes        = 0x00U,
         .bLockDelayUnits     = 0x00U,
         .wLockDelay          = 0x0000U,
    },

    .speak_feedback_endpoint =
    {
        .header =
         {
             .bLength         = sizeof(fs_usb_desc_FeedBack_ep),
             .bDescriptorType = USB_DESCTYPE_EP
         },
         .bEndpointAddress    = FS_AD_FEEDBACK_IN_EP,
         .bmAttributes        = USB_EP_ATTR_ISO | USB_EP_ATTR_ASYNC | USB_EP_ATTR_FEEDBACK,
         .wMaxPacketSize      = FS_FEEDBACK_IN_PACKET,
         .bInterval           = 0x01U,
         .Refresh             = FS_FEEDBACK_IN_INTERVAL, /* refresh every 32(2^5) ms */
         .bSynchAddress       = 0x00U,
    },
#endif /* USE_USB_AD_SPEAKER */
};

/* USB descriptor configure */
usb_desc fs_audio_desc = {
    .dev_desc    = (uint8_t *)&fs_audio_dev_desc,
    .config_desc = (uint8_t *)&fs_audio_config_set,
    .strings     = fs_uac_strings_desc
};

/*!
    \brief      initialize the AUDIO device
    \param[in]  udev: pointer to USB device instance
    \param[in]  config_index: configuration index
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_init (usb_dev *udev, uint8_t config_index)
{
    memset((void *)&fs_audio_handler, 0, sizeof(fs_usbd_audio_handler));

#ifdef USE_USB_AD_MICPHONE
{
    fs_usb_desc_std_ep std_ep = fs_audio_config_set.mic_std_endpoint;

    usb_desc_ep ep = {
        .header           = std_ep.header,
        .bEndpointAddress = std_ep.bEndpointAddress,
        .bmAttributes     = std_ep.bmAttributes,
        .wMaxPacketSize   = std_ep.wMaxPacketSize,
        .bInterval        = std_ep.bInterval
    };

    /* initialize TX endpoint */
    usbd_ep_setup (udev, &ep);
}
#endif /* USE_USB_AD_MICPHONE */

#ifdef USE_USB_AD_SPEAKER
{
    fs_audio_handler.isoc_out_rdptr = fs_audio_handler.isoc_out_buff;
    fs_audio_handler.isoc_out_wrptr = fs_audio_handler.isoc_out_buff;

    fs_usb_desc_std_ep std_ep = fs_audio_config_set.speak_std_endpoint;

    usb_desc_ep ep1 = {
        .header           = std_ep.header,
        .bEndpointAddress = std_ep.bEndpointAddress,
        .bmAttributes     = std_ep.bmAttributes,
        .wMaxPacketSize   = FS_SPEAKER_OUT_MAX_PACKET,
        .bInterval        = std_ep.bInterval
    };

    /* initialize RX endpoint */
    usbd_ep_setup (udev, &ep1);

    /* prepare out endpoint to receive next audio packet */
    usbd_ep_recev (udev, FS_AD_OUT_EP, fs_audio_handler.usb_rx_buffer, FS_SPEAKER_OUT_MAX_PACKET);

    /* initialize the audio output hardware layer */
    if (USBD_OK != audio_out_fops.audio_init(FS_USBD_SPEAKER_FREQ, FS_DEFAULT_VOLUME)) {
        return USBD_FAIL;
    }

    fs_usb_desc_FeedBack_ep feedback_ep = fs_audio_config_set.speak_feedback_endpoint;

    usb_desc_ep ep2 = {
        .header           = feedback_ep.header,
        .bEndpointAddress = feedback_ep.bEndpointAddress,
        .bmAttributes     = feedback_ep.bmAttributes,
        .wMaxPacketSize   = feedback_ep.wMaxPacketSize,
        .bInterval        = feedback_ep.bInterval
    };

    /* initialize Tx endpoint */
    usbd_ep_setup (udev, &ep2);
}
#endif /* USE_USB_AD_SPEAKER */

    return USBD_OK;
}

/*!
    \brief      de-initialize the AUDIO device
    \param[in]  udev: pointer to USB device instance
    \param[in]  config_index: configuration index
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_deinit (usb_dev *udev, uint8_t config_index)
{
#ifdef USE_USB_AD_MICPHONE
    /* deinitialize AUDIO endpoints */
    usbd_ep_clear(udev, FS_AD_IN_EP);
#endif /* USE_USB_AD_MICPHONE */

#ifdef USE_USB_AD_SPEAKER
    /* deinitialize AUDIO endpoints */
    usbd_ep_clear(udev, FS_AD_OUT_EP);

    /* deinitialize the audio output hardware layer */
    if (USBD_OK != audio_out_fops.audio_deinit()) {
        return USBD_FAIL;
    }

    /* deinitialize AUDIO endpoints */
    usbd_ep_clear(udev, FS_AD_FEEDBACK_IN_EP);
#endif /* USE_USB_AD_SPEAKER */

    return USBD_OK;
}

/*!
    \brief      handle the AUDIO class-specific requests
    \param[in]  udev: pointer to USB device instance
    \param[in]  req: device class-specific request
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_req_handler (usb_dev *udev, usb_req *req)
{
    uint8_t status = REQ_NOTSUPP;

    usb_transc *transc_in = &udev->dev.transc_in[0];
    usb_transc *transc_out = &udev->dev.transc_out[0];

    switch (req->bRequest) {
    case FS_AD_REQ_GET_CUR:
        transc_in->xfer_buf = fs_audio_handler.audioctl;
        transc_in->remain_len = req->wLength;

        status = REQ_SUPP;
        break;

    case FS_AD_REQ_SET_CUR:
        if (req->wLength) {
            transc_out->xfer_buf = fs_audio_handler.audioctl;
            transc_out->remain_len = req->wLength;

            udev->dev.class_core->command = FS_AD_REQ_SET_CUR;

            fs_audio_handler.audioctl_len = req->wLength;
            fs_audio_handler.audioctl_unit = BYTE_HIGH(req->wIndex);

            status = REQ_SUPP;
        }
        break;

    case FS_AD_REQ_GET_MIN:
        *((uint16_t *)fs_audio_handler.audioctl) = VOL_MIN;
        transc_in->xfer_buf = fs_audio_handler.audioctl;
        transc_in->remain_len = req->wLength;
        status = REQ_SUPP;
        break;

    case FS_AD_REQ_GET_MAX:
        *((uint16_t *)fs_audio_handler.audioctl) = VOL_MAX;
        transc_in->xfer_buf = fs_audio_handler.audioctl;
        transc_in->remain_len = req->wLength;
        status = REQ_SUPP;
        break;

    case FS_AD_REQ_GET_RES:
        *((uint16_t *)fs_audio_handler.audioctl) = VOL_RES;
        transc_in->xfer_buf = fs_audio_handler.audioctl;
        transc_in->remain_len = req->wLength;
        status = REQ_SUPP;
        break;

    default:
        break;
    }

    return status;
}

/*!
    \brief      handle the AUDIO set interface requests
    \param[in]  udev: pointer to USB device instance
    \param[in]  req: device class-specific request
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_set_intf(usb_dev *udev, usb_req *req)
{
    udev->dev.class_core->alter_set = req->wValue;

    if(0xFF != req->wValue){
        if (req->wValue != 0){
            /* deinit audio handler */
            memset((void *)&fs_audio_handler, 0, sizeof(fs_usbd_audio_handler));

            fs_audio_handler.play_flag = 0;
            fs_audio_handler.isoc_out_rdptr = fs_audio_handler.isoc_out_buff;
            fs_audio_handler.isoc_out_wrptr = fs_audio_handler.isoc_out_buff;

            /* feedback calculate sample freq */
            fs_audio_handler.actual_freq = FS_I2S_ACTUAL_SAM_FREQ(FS_USBD_SPEAKER_FREQ);
            fs_usbd_get_feedback_fs_rate(fs_audio_handler.actual_freq, fs_audio_handler.feedback_freq);

            /* send feedback data of estimated frequence*/
            usbd_ep_send(udev, FS_AD_FEEDBACK_IN_EP, fs_audio_handler.feedback_freq, FS_FEEDBACK_IN_PACKET);
        } else {
            /* stop audio output */
            audio_out_fops.audio_cmd(fs_audio_handler.isoc_out_rdptr, FS_SPEAKER_OUT_PACKET/2, AD_CMD_STOP);

            fs_audio_handler.play_flag = 0;
            fs_audio_handler.isoc_out_rdptr = fs_audio_handler.isoc_out_buff;
            fs_audio_handler.isoc_out_wrptr = fs_audio_handler.isoc_out_buff;

            fs_usbd_fifo_flush (udev, FS_AD_IN_EP);
            fs_usbd_fifo_flush (udev, FS_AD_FEEDBACK_IN_EP);
            fs_usbd_fifo_flush (udev, FS_AD_OUT_EP);
        }
    }

    return 0;
}

/*!
    \brief      handles the control transfer OUT callback
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_ctlx_out (usb_dev *udev)
{
#ifdef USE_USB_AD_SPEAKER
    /* handles audio control requests data */
    /* check if an audio_control request has been issued */
    if (FS_AD_REQ_SET_CUR == udev->dev.class_core->command) {
        /* in this driver, to simplify code, only SET_CUR request is managed */

        /* check for which addressed unit the audio_control request has been issued */
        if (FS_AD_OUT_STREAMING_CTRL == fs_audio_handler.audioctl_unit) {
            /* in this driver, to simplify code, only one unit is manage */

            /* reset the audioctl_cmd variable to prevent re-entering this function */
            udev->dev.class_core->command = 0U;

            fs_audio_handler.audioctl_len = 0U;
        }
    }
#endif /* USE_USB_AD_SPEAKER */

    return USBD_OK;
}

/*!
    \brief      handles the audio IN data stage
    \param[in]  udev: pointer to USB device instance
    \param[in]  ep_num: endpoint number
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_data_in (usb_dev *udev, uint8_t ep_num)
{
#ifdef USE_USB_AD_MICPHONE
    if(ep_num == EP_ID(FS_AD_IN_EP)){
        if(count_data < LENGTH_DATA){
            /* Prepare next buffer to be sent: dummy data */
            usbd_ep_send(udev, FS_AD_IN_EP,(uint8_t*)&wavetestdata[count_data],FS_MIC_IN_PACKET);
            count_data += FS_MIC_IN_PACKET;
        } else {
            usbd_ep_send(udev, FS_AD_IN_EP,(uint8_t*)wavetestdata,FS_MIC_IN_PACKET);
            count_data = FS_MIC_IN_PACKET;
        }
    }
#endif /* USE_USB_AD_MICPHONE */

#ifdef USE_USB_AD_SPEAKER
    if(ep_num == EP_ID(FS_AD_FEEDBACK_IN_EP)){
        /* calculate feedback actual freq */
        fs_audio_handler.actual_freq = fs_usbd_audio_spk_get_feedback(udev);
        fs_usbd_get_feedback_fs_rate(fs_audio_handler.actual_freq, fs_audio_handler.feedback_freq);

        usbd_ep_send(udev, FS_AD_FEEDBACK_IN_EP, fs_audio_handler.feedback_freq, FS_FEEDBACK_IN_PACKET);
    }
#endif /* USE_USB_AD_SPEAKER */

    return USBD_OK;
}

/*!
    \brief      handles the audio OUT data stage
    \param[in]  udev: pointer to USB device instance
    \param[in]  ep_num: endpoint number
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_data_out (usb_dev *udev, uint8_t ep_num)
{
    uint16_t usb_rx_length, tail_len;

    /* get receive length */
    usb_rx_length = ((usb_core_driver *)udev)->dev.transc_out[ep_num].xfer_count;

    if(fs_audio_handler.isoc_out_wrptr >= fs_audio_handler.isoc_out_rdptr){
        fs_audio_handler.buf_free_size = FS_TOTAL_OUT_BUF_SIZE + fs_audio_handler.isoc_out_rdptr - fs_audio_handler.isoc_out_wrptr;
    }else{
        fs_audio_handler.buf_free_size = fs_audio_handler.isoc_out_rdptr - fs_audio_handler.isoc_out_wrptr;
    }

    /* free buffer enough to save rx data */
    if(fs_audio_handler.buf_free_size > usb_rx_length){
        if(fs_audio_handler.isoc_out_wrptr >= fs_audio_handler.isoc_out_rdptr){
            tail_len = fs_audio_handler.isoc_out_buff + FS_TOTAL_OUT_BUF_SIZE - fs_audio_handler.isoc_out_wrptr;

            if(tail_len >= usb_rx_length){
                memcpy(fs_audio_handler.isoc_out_wrptr, fs_audio_handler.usb_rx_buffer, usb_rx_length);

                /* increment the buffer pointer */
                fs_audio_handler.isoc_out_wrptr += usb_rx_length;

                /* increment the Buffer pointer or roll it back when all buffers are full */
                if(fs_audio_handler.isoc_out_wrptr >= (fs_audio_handler.isoc_out_buff + FS_TOTAL_OUT_BUF_SIZE)){
                    /* all buffers are full: roll back */
                    fs_audio_handler.isoc_out_wrptr = fs_audio_handler.isoc_out_buff;
                }
            }else{
                memcpy(fs_audio_handler.isoc_out_wrptr, fs_audio_handler.usb_rx_buffer, tail_len);
                /* adjust write pointer */
                fs_audio_handler.isoc_out_wrptr = fs_audio_handler.isoc_out_buff;

                memcpy(fs_audio_handler.isoc_out_wrptr, &fs_audio_handler.usb_rx_buffer[tail_len], usb_rx_length - tail_len);
                /* adjust write pointer */
                fs_audio_handler.isoc_out_wrptr += usb_rx_length - tail_len;
            }
        }else{
            memcpy(fs_audio_handler.isoc_out_wrptr, fs_audio_handler.usb_rx_buffer, usb_rx_length);

            /* increment the buffer pointer */
            fs_audio_handler.isoc_out_wrptr += usb_rx_length;
        }
    }

    /* Toggle the frame index */
    udev->dev.transc_out[ep_num].frame_num = (udev->dev.transc_out[ep_num].frame_num)? 0U:1U;

    /* prepare out endpoint to receive next audio packet */
    usbd_ep_recev (udev, FS_AD_OUT_EP, fs_audio_handler.usb_rx_buffer, FS_SPEAKER_OUT_MAX_PACKET);

    if(fs_audio_handler.isoc_out_wrptr >= fs_audio_handler.isoc_out_rdptr){
        fs_audio_handler.buf_free_size = FS_TOTAL_OUT_BUF_SIZE + fs_audio_handler.isoc_out_rdptr - fs_audio_handler.isoc_out_wrptr;
    }else{
        fs_audio_handler.buf_free_size = fs_audio_handler.isoc_out_rdptr - fs_audio_handler.isoc_out_wrptr;
    }

    if ((0U == fs_audio_handler.play_flag) && (fs_audio_handler.buf_free_size < FS_TOTAL_OUT_BUF_SIZE/2)) {
        /* enable start of streaming */
        fs_audio_handler.play_flag = 1U;

        /* initialize the audio output hardware layer */
        if (USBD_OK != audio_out_fops.audio_cmd(fs_audio_handler.isoc_out_rdptr, FS_SPEAKER_OUT_MAX_PACKET/2, AD_CMD_PLAY)) {
            return USBD_FAIL;
        }

        fs_audio_handler.dam_tx_len = FS_SPEAKER_OUT_MAX_PACKET;
    }

    return USBD_OK;
}

/*!
    \brief      handles the SOF event (data buffer update and synchronization)
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_sof (usb_dev *udev)
{
    return USBD_OK;
}

/*!
    \brief      handles the audio ISO IN Incomplete event
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_iso_in_incomplete (usb_dev *udev)
{
    (void)usb_txfifo_flush (&udev->regs, EP_ID(FS_AD_FEEDBACK_IN_EP));

    fs_audio_handler.actual_freq = fs_usbd_audio_spk_get_feedback(udev);
    fs_usbd_get_feedback_fs_rate(fs_audio_handler.actual_freq, fs_audio_handler.feedback_freq);

    /* send feedback data of estimated frequence*/
    usbd_ep_send(udev, FS_AD_FEEDBACK_IN_EP, fs_audio_handler.feedback_freq, FS_FEEDBACK_IN_PACKET);

    return USBD_OK;
}

/*!
    \brief      handles the audio ISO OUT Incomplete event
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     USB device operation status
*/
static uint8_t fs_audio_iso_out_incomplete (usb_dev *udev)
{
    return USBD_OK;
}

/*!
    \brief      calculate feedback sample frequency
    \param[in]  udev: pointer to USB device instance
    \param[out] none
    \retval     feedback frequency value
*/
static uint32_t fs_usbd_audio_spk_get_feedback(usb_dev *udev)
{
    static uint32_t fb_freq;

    /* calculate buffer free size */
    if(fs_audio_handler.isoc_out_wrptr >= fs_audio_handler.isoc_out_rdptr){
        fs_audio_handler.buf_free_size = FS_TOTAL_OUT_BUF_SIZE + fs_audio_handler.isoc_out_rdptr - fs_audio_handler.isoc_out_wrptr;
    }else{
        fs_audio_handler.buf_free_size = fs_audio_handler.isoc_out_rdptr - fs_audio_handler.isoc_out_wrptr;
    }

    /* calculate feedback frequency */
    if(fs_audio_handler.buf_free_size <= (FS_TOTAL_OUT_BUF_SIZE/4)){
        fb_freq = FS_I2S_ACTUAL_SAM_FREQ(FS_USBD_SPEAKER_FREQ) - FS_FEEDBACK_FREQ_OFFSET;
    }else if(fs_audio_handler.buf_free_size >= (FS_TOTAL_OUT_BUF_SIZE*3/4)){
        fb_freq = FS_I2S_ACTUAL_SAM_FREQ(FS_USBD_SPEAKER_FREQ) + FS_FEEDBACK_FREQ_OFFSET;
    }else{
        fb_freq = FS_I2S_ACTUAL_SAM_FREQ(FS_USBD_SPEAKER_FREQ);
    }

    return fb_freq;
}

/*!
    \brief      get feedback value from rate in USB full speed
    \param[in]  rate: sample frequence
    \param[in]  buf: pointer to result buffer
    \param[out] none
    \retval     USB device operation status
*/
static void fs_usbd_get_feedback_fs_rate(uint32_t rate, uint8_t *buf)
{
    rate = ((rate / 1000) << 14) | ((rate % 1000) << 4);

    buf[0] = rate;
    buf[1] = rate >> 8;
    buf[2] = rate >> 16;
}
/*!
    \brief      get the calculate value of i2s sample frequency
    \param[in]  set_freq: setting sample frequence
    \param[out] none
    \retval     i2s sample frequency
*/
static uint32_t FS_I2S_ACTUAL_SAM_FREQ(uint32_t set_freq)
{
    uint32_t actual_freq;

    switch(set_freq){

    case 48000:
        actual_freq = 46875;
        break;

    case 44100:
        actual_freq = 41666;
        break;

    case 16000:
        actual_freq = 16304;
        break;

    case 8000:
        actual_freq = 7979;
        break;

    default:
        actual_freq = set_freq;
        break;
    }

    return actual_freq;
}
