/*!
    \file    gd32h7xx_hal_vref.c
    \brief   VREF driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define VREF_TIMEOUT_VALUE               (600U)                                      /*!< VREF initialize timeout */

/*!
    \brief      deinitialize VREF
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_vref_deinit(void)
{
    hal_rcu_periph_reset_enable(RCU_VREFRST);
    hal_rcu_periph_reset_disable(RCU_VREFRST);
}

/*!
    \brief      initialize VREF
    \param[in]  p_vref_init: hal_vref_init_struct
                vref_mode: VREF mode select,
                only one parameter can be selected which is shown as below:
      \arg        HAL_VREF_MODE_DISABLED: VREF disabled, VREFP pin pulled-down to VSSA
      \arg        HAL_VREF_MODE_EXTERNAL: external voltage reference mode
      \arg        HAL_VREF_MODE_INTERNAL: internal voltage reference mode
      \arg        HAL_VREF_MODE_HOLD: hold mode
                vref_calibration_mode: VREF calibration mode select,
                only one parameter can be selected which is shown as below:
      \arg        HAL_VREF_CALIB_MODE_FACTORY: factory calibration
      \arg        HAL_VREF_CALIB_MODE_USER: user calibration
                vref_user_calibration: calibration value (0x00 - 0x3F)
                vref_ref_voltage: voltage reference select,
                only one parameter can be selected which is shown as below:
      \arg        VREF_CS_2V5: VREF voltage reference select 2.5 V
      \arg        VREF_CS_2V048: VREF voltage reference select 2.048 V
      \arg        VREF_CS_1V8: VREF voltage reference select 1.8 V
      \arg        VREF_CS_1V5: VREF voltage reference select 1.5 V
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_vref_init(hal_vref_init_struct *p_vref_init)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;
    __IO uint32_t tick_start;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_vref_init) {
        HAL_DEBUGE("pointer [p_vref_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* Check that vref_calibration_mode is a valid value */
    if((p_vref_init->vref_calibration_mode != HAL_VREF_CALIB_MODE_FACTORY) && \
       (p_vref_init->vref_calibration_mode != HAL_VREF_CALIB_MODE_USER)) {
        HAL_DEBUGE("parameter [vref_calibration_mode] value is invalid");
        return HAL_ERR_VAL;
    }

    /* Check that vref_mode is a valid value */
    if((p_vref_init->vref_mode != HAL_VREF_MODE_DISABLED) && (p_vref_init->vref_mode != HAL_VREF_MODE_EXTERNAL) && \
       (p_vref_init->vref_mode != HAL_VREF_MODE_INTERNAL) && (p_vref_init->vref_mode != HAL_VREF_MODE_HOLD)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    /* Check that vref_ref_voltage is a valid value */
    if((VREF_CS_2V5 != p_vref_init->vref_ref_voltage) && (VREF_CS_2V048 != p_vref_init->vref_ref_voltage) && \
       (VREF_CS_1V8 != p_vref_init->vref_ref_voltage) && (VREF_CS_1V5   != p_vref_init->vref_ref_voltage)) {
        HAL_DEBUGE("parameter [vref_ref_voltage] value is invalid");
        return HAL_ERR_VAL;
    }

#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Configuration the calibration and mode */
    VREF_CS &= ~(VREF_EN | VREF_HIGH_IMPEDANCE_MODE);

    if(HAL_VREF_CALIB_MODE_USER == p_vref_init->vref_calibration_mode) {
        /* Apply user-defined calibration value */
        VREF_CALIB = (VREF_CALIB_VREFCAL & p_vref_init->vref_user_calibration);
    } else {
        /* Apply calibration value stored in Flash, no configuration required */
    }

    /* Config the VREF voltage reference */
    VREF_CS &= ~VREF_CS_VREFS;
    VREF_CS |= (VREF_CS_VREFS & (uint32_t)p_vref_init->vref_ref_voltage);

    /* Config VREF MODE */
    if(HAL_VREF_MODE_DISABLED == p_vref_init->vref_mode) {
        /* Configure the VREFP pin pulled-down to VSSA */
        VREF_CS &= ~(VREF_CS_HIPM | VREF_EN);
    } else if (HAL_VREF_MODE_EXTERNAL == p_vref_init->vref_mode) {
        /* Configure the external voltage reference */
        VREF_CS |= VREF_CS_HIPM;
        VREF_CS &= ~VREF_EN;
    } else if (HAL_VREF_MODE_INTERNAL == p_vref_init->vref_mode) {
        /* Configure the internal voltage reference */
        VREF_CS &= ~VREF_CS_HIPM;
        VREF_CS |= VREF_EN;
    } else {
        /* Configure the hold mode */
        VREF_CS |= (VREF_CS_HIPM | VREF_EN);
    }

    /* Get the tick count */
    tick_start = hal_sys_basetick_count_get();

    /* Wait for VREF ready */
    while(RESET == (VREF_CS & VREF_CS_VREFRDY)) {
        /* Check for the timeout */
        if(SET == hal_sys_basetick_timeout_check(tick_start, VREF_TIMEOUT_VALUE)) {
            HAL_DEBUGE("vref init timeout");
            retval = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* Do nothing */
        }
    }

    return retval;
}

/*!
    \brief      initialize the VREF structure
    \param[in]  hal_struct_type: refer to hal_vref_struct_type_enum
                  only one parameter can be selected which is shown as below:
      \arg        HAL_VREF_INIT_STRUCT: VREF initialization structure
    \param[out] p_struct: point to VREF structure that contains the configuration information
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_vref_struct_init(hal_vref_struct_type_enum hal_struct_type, void *p_struct)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* Check the parametUers */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_VREF_INIT_STRUCT:
        /* Initialize VREF initialization structure with the init values */
        ((hal_vref_init_struct *)p_struct)->vref_ref_voltage      = VREF_CS_2V5;
        ((hal_vref_init_struct *)p_struct)->vref_user_calibration = 0U;
        ((hal_vref_init_struct *)p_struct)->vref_mode             = HAL_VREF_MODE_EXTERNAL;
        ((hal_vref_init_struct *)p_struct)->vref_calibration_mode = HAL_VREF_CALIB_MODE_FACTORY;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefined");
        retval = HAL_ERR_VAL;
        break;
    }

    return retval;
}

/*!
    \brief      pin high impedance mode disable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_vref_hipm_disable(void)
{
    VREF_CS &= ~VREF_HIGH_IMPEDANCE_MODE;
}

/*!
    \brief      pin high impedance mode enable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_vref_hipm_enable(void)
{
    VREF_CS |= VREF_HIGH_IMPEDANCE_MODE;
}

/*!
    \brief      get the calibration value of VREF
    \param[in]  none
    \param[out] none
    \retval     uint8_t: calibration value (0x00 - 0x3F)
*/
uint8_t hal_vref_cali_value_get(void)
{
    return (uint8_t)(VREF_CALIB_VREFCAL & VREF_CALIB);
}

/*!
    \brief      voltage reference disable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_vref_disable(void)
{
    VREF_CS &= ~VREF_EN;
}

/*!
    \brief      voltage reference enable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_vref_enable(void)
{
    VREF_CS |= VREF_EN;
}

/*!
    \brief      set the calibration value of VREF
    \param[in]  vref_user_calibration: calibration value (0x00 - 0x3F)
    \param[out] none
    \retval     none
*/
void hal_vref_cali_value_set(uint8_t vref_user_calibration)
{
    VREF_CALIB = (VREF_CALIB_VREFCAL & (uint32_t)vref_user_calibration);
}

/*!
    \brief      configure the VREF voltage reference
    \param[in]  vref_ref_voltage: voltage reference select,
                only one parameter can be selected which is shown as below:
      \arg        VREF_CS_2V5: VREF voltage reference select 2.5 V
      \arg        VREF_CS_2V048: VREF voltage reference select 2.048 V
      \arg        VREF_CS_1V8: VREF voltage reference select 1.8 V
      \arg        VREF_CS_1V5: VREF voltage reference select 1.5 V
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_vref_config(uint8_t vref_ref_voltage)
{
/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((VREF_CS_2V5 != vref_ref_voltage) && (VREF_CS_2V048 != vref_ref_voltage) && \
       (VREF_CS_1V8 != vref_ref_voltage) && (VREF_CS_1V5   != vref_ref_voltage)) {
        HAL_DEBUGE("parameter [vref_voltage] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* config the VREF voltage reference */
    VREF_CS &= ~VREF_CS_VREFS;
    VREF_CS |= (VREF_CS_VREFS & (uint32_t)vref_ref_voltage);

    return HAL_ERR_NONE;
}
