/*!
    \file    gd32h7xx_hal_usart_com.c
    \brief   USART driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* USART register bit offset */
#define CTL1_ADDR0_OFFSET    ((uint32_t)24U) /* bit offset of ADDR0 in USART_CTL1 */
#define CTL2_ADDR1_OFFSET    ((uint32_t)24U) /* bit offset of ADDR1 in USART_CTL2 */
#define FCS_RFCNT_3_4_OFFSET ((uint32_t)3U)  /* bit offset of RFCNT[4:3] in USART_FCS */

/*!
    \brief      configure USART baud rate value
    \param[in]  usart_periph: USARTx(x=0,1,2,5)
    \param[in]  baudval: baud rate value
    \param[out] none
    \retval     none
*/
void hals_usart_baudrate_set(uint32_t usart_periph, uint32_t baudval)
{
    uint32_t uclk = 0U, intdiv = 0U, fradiv = 0U, udiv = 0U;
    switch(usart_periph) {
    /* get clock frequency */
    case USART0:
        /* get USART0 clock */
        uclk = hal_rcu_clock_freq_get(CK_USART0);
        break;
    case USART1:
        /* get USART1 clock */
        uclk = hal_rcu_clock_freq_get(CK_USART1);
        break;
    case USART2:
        /* get USART2 clock */
        uclk = hal_rcu_clock_freq_get(CK_USART2);
        break;
    case UART3:
        /* get UART3 clock */
        uclk = hal_rcu_clock_freq_get(CK_APB1);
        break;
    case UART4:
        /* get UART4 clock */
        uclk = hal_rcu_clock_freq_get(CK_APB1);
        break;
    case USART5:
        /* get USART5 clock */
        uclk = hal_rcu_clock_freq_get(CK_USART5);
        break;
    case UART6:
        /* get UART6 clock */
        uclk = hal_rcu_clock_freq_get(CK_APB1);
        break;
    case UART7:
        /* get UART7 clock */
        uclk = hal_rcu_clock_freq_get(CK_APB1);
        break;
    default:
        break;
    }

    if(USART_CTL0(usart_periph) & USART_CTL0_OVSMOD) {
        /* oversampling by 8, configure the value of USART_BAUD */
        udiv   = ((2U * uclk) + baudval / 2U) / baudval;
        intdiv = udiv & 0x0000fff0U;
        fradiv = (udiv >> 1U) & 0x00000007U;
        USART_BAUD(usart_periph) = ((USART_BAUD_FRADIV | USART_BAUD_INTDIV) & (intdiv | fradiv));
    } else {
        /* oversampling by 16, configure the value of USART_BAUD */
        udiv   = (uclk + baudval / 2U) / baudval;
        intdiv = udiv & 0x0000fff0U;
        fradiv = udiv & 0x0000000fU;
        USART_BAUD(usart_periph) = ((USART_BAUD_FRADIV | USART_BAUD_INTDIV) & (intdiv | fradiv));
    }
}

/*!
    \brief      enable USART
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[out] none
    \retval     none
*/
void hals_usart_enable(uint32_t usart_periph)
{
    USART_CTL0(usart_periph) |= USART_CTL0_UEN;
}

/*!
    \brief      disable USART
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[out] none
    \retval     none
*/
void hals_usart_disable(uint32_t usart_periph)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);
}

/*!
    \brief      configure USART transmitter
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  txconfig: enable or disable USART transmitter
                only one parameter can be selected which is shown as below:
      \arg        USART_TRANSMIT_ENABLE: enable USART transmission
      \arg        USART_TRANSMIT_DISABLE: enable USART transmission
    \param[out] none
    \retval     none
*/
void hals_usart_transmit_config(uint32_t usart_periph, uint32_t txconfig)
{
    USART_CTL0(usart_periph) &= ~USART_CTL0_TEN;
    /* configure transfer mode */
    USART_CTL0(usart_periph) |= txconfig;
}

/*!
    \brief      configure USART receiver
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  rxconfig: enable or disable USART receiver
                only one parameter can be selected which is shown as below:
      \arg        USART_RECEIVE_ENABLE: enable USART reception
      \arg        USART_RECEIVE_DISABLE: disable USART reception
    \param[out] none
    \retval     none
*/
void hals_usart_receive_config(uint32_t usart_periph, uint32_t rxconfig)
{
    USART_CTL0(usart_periph) &= ~USART_CTL0_REN;
    /* configure receiver mode */
    USART_CTL0(usart_periph) |= rxconfig;
}

/*!
    \brief      USART inverted configure
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  invertpara: refer to usart_invert_enum
                only one parameter can be selected which is shown as below:
      \arg        USART_DINV_ENABLE: data bit level inversion
      \arg        USART_DINV_DISABLE: data bit level not inversion
      \arg        USART_TXPIN_ENABLE: TX pin level inversion
      \arg        USART_TXPIN_DISABLE: TX pin level not inversion
      \arg        USART_RXPIN_ENABLE: RX pin level inversion
      \arg        USART_RXPIN_DISABLE: RX pin level not inversion
      \arg        USART_SWAP_ENABLE: swap TX/RX pins
      \arg        USART_SWAP_DISABLE: not swap TX/RX pins
    \param[out] none
    \retval     none
*/
void hals_usart_invert_config(uint32_t usart_periph, hal_usart_invert_enum invertpara)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);

    /* inverted or not the specified signal */
    switch(invertpara) {
    case USART_DINV_ENABLE:
        USART_CTL1(usart_periph) |= USART_CTL1_DINV;
        break;
    case USART_DINV_DISABLE:
        USART_CTL1(usart_periph) &= ~(USART_CTL1_DINV);
        break;
    case USART_TXPIN_ENABLE:
        USART_CTL1(usart_periph) |= USART_CTL1_TINV;
        break;
    case USART_TXPIN_DISABLE:
        USART_CTL1(usart_periph) &= ~(USART_CTL1_TINV);
        break;
    case USART_RXPIN_ENABLE:
        USART_CTL1(usart_periph) |= USART_CTL1_RINV;
        break;
    case USART_RXPIN_DISABLE:
        USART_CTL1(usart_periph) &= ~(USART_CTL1_RINV);
        break;
    case USART_SWAP_ENABLE:
        USART_CTL1(usart_periph) |= USART_CTL1_STRP;
        break;
    case USART_SWAP_DISABLE:
        USART_CTL1(usart_periph) &= ~(USART_CTL1_STRP);
        break;
    default:
        break;
    }
}

/*!
    \brief      USART transmit data function
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  data: data of transmission
    \param[out] none
    \retval     none
*/
void hals_usart_data_transmit(uint32_t usart_periph, uint16_t data)
{
    USART_TDATA(usart_periph) = USART_TDATA_TDATA & (uint32_t)data;
}

/*!
    \brief      USART receive data function
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[out] none
    \retval     data of received
*/
uint16_t hals_usart_data_receive(uint32_t usart_periph)
{
    return (uint16_t)(GET_BITS(USART_RDATA(usart_periph), 0U, 8U));
}

/*!
    \brief      enable USART command
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  cmdtype: command type
                only one parameter can be selected which is shown as below:
      \arg        USART_CMD_SBKCMD: send break command
      \arg        USART_CMD_MMCMD: mute mode command
      \arg        USART_CMD_RXFCMD: receive data flush command
      \arg        USART_CMD_TXFCMD: transmit data flush request
    \param[out] none
    \retval     none
*/
void hals_usart_command_enable(uint32_t usart_periph, uint32_t cmdtype)
{
    USART_CMD(usart_periph) &= ~(USART_CMD_SBKCMD | USART_CMD_MMCMD | USART_CMD_RXFCMD | USART_CMD_TXFCMD);
    USART_CMD(usart_periph) |= (cmdtype);
}

/*!
    \brief      configure USART DMA reception
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  dmacmd: enable or disable DMA for reception
                only one parameter can be selected which is shown as below:
      \arg        USART_RECEIVE_DMA_ENABLE: USART enable DMA for reception
      \arg        USART_RECEIVE_DMA_DISABLE: USART disable DMA for reception
    \param[out] none
    \retval     none
*/
void hals_usart_dma_receive_config(uint32_t usart_periph, uint32_t dmacmd)
{
    uint32_t ctl = 0U;

    ctl = USART_CTL2(usart_periph);
    ctl &= ~USART_CTL2_DENR;
    ctl |= dmacmd;

    /* configure DMA reception */
    USART_CTL2(usart_periph) = ctl;
}

/*!
    \brief      configure USART DMA transmission
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  dmacmd: enable or disable DMA for transmission
                only one parameter can be selected which is shown as below:
      \arg        USART_TRANSMIT_DMA_ENABLE: USART enable DMA for transmission
      \arg        USART_TRANSMIT_DMA_DISABLE: USART disable DMA for transmission
    \param[out] none
    \retval     none
*/
void hals_usart_dma_transmit_config(uint32_t usart_periph, uint32_t dmacmd)
{
    uint32_t ctl = 0U;

    ctl = USART_CTL2(usart_periph);
    ctl &= ~USART_CTL2_DENT;
    ctl |= dmacmd;

    /* configure DMA transmission */
    USART_CTL2(usart_periph) = ctl;
}

/*!
    \brief      enable address 0 match mode
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[out] none
    \retval     none
*/
void hals_usart_address_0_match_mode_enable(uint32_t usart_periph)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);
    USART_CTL1(usart_periph) |= USART_CTL1_AMEN0;
}

/*!
    \brief      disable address 0 match mode
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[out] none
    \retval     none
*/
void hals_usart_address_0_match_mode_disable(uint32_t usart_periph)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);
    USART_CTL1(usart_periph) &= ~(USART_CTL1_AMEN0);
}

/*!
    \brief      enable address 1 match mode
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[out] none
    \retval     none
*/
void hals_usart_address_1_match_mode_enable(uint32_t usart_periph)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);
    USART_CTL2(usart_periph) |= USART_CTL2_AMEN1;
}

/*!
    \brief      disable address 1 match mode
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[out] none
    \retval     none
*/
void hals_usart_address_1_match_mode_disable(uint32_t usart_periph)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);
    USART_CTL2(usart_periph) &= ~(USART_CTL2_AMEN1);
}

/*!
    \brief      configure address 0 of the USART
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  addr: 0x00-0xFF, address of USART terminal
    \param[out] none
    \retval     none
*/
void hals_usart_address_0_config(uint32_t usart_periph, uint8_t addr)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);
    USART_CTL1(usart_periph) &= ~(USART_CTL1_ADDR0);
    USART_CTL1(usart_periph) |= (USART_CTL1_ADDR0 & (((uint32_t)addr) << CTL1_ADDR0_OFFSET));
}

/*!
    \brief      configure address 1 of the USART
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  addr: 0x00-0xFF, address of USART terminal
    \param[out] none
    \retval     none
*/
void hals_usart_address_1_config(uint32_t usart_periph, uint8_t addr)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);
    USART_CTL2(usart_periph) &= ~(USART_CTL2_ADDR1);
    USART_CTL2(usart_periph) |= (USART_CTL2_ADDR1 & (((uint32_t)addr) << CTL2_ADDR1_OFFSET));
}

/*!
    \brief      configure address 0 detection mode
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  addmod: address detection mode
                only one parameter can be selected which is shown as below:
      \arg        USART_ADDM0_4BIT: 4 bits
      \arg        USART_ADDM0_FULLBIT: full bits
    \param[out] none
    \retval     none
*/
void hals_usart_address_0_detection_mode_config(uint32_t usart_periph, uint32_t addmod)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);
    USART_CTL1(usart_periph) &= ~(USART_CTL1_ADDM0);
    USART_CTL1(usart_periph) |= USART_CTL1_ADDM0 & (addmod);
}

/*!
    \brief      configure address 1 detection mode
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  addmod: address detection mode
                only one parameter can be selected which is shown as below:
      \arg        USART_ADDM1_4BIT: 4 bits
      \arg        USART_ADDM1_FULLBIT: full bits
    \param[out] none
    \retval     none
*/
void hals_usart_address_1_detection_mode_config(uint32_t usart_periph, uint32_t addmod)
{
    USART_CTL0(usart_periph) &= ~(USART_CTL0_UEN);
    USART_CTL2(usart_periph) &= ~(USART_CTL2_ADDM1);
    USART_CTL2(usart_periph) |= USART_CTL2_ADDM1 & (addmod);
}

/*!
   \brief      configure hardware flow control coherence mode
   \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
   \param[in]  hcm: hardware flow control coherence mode
               only one parameter can be selected which is shown as below:
     \arg        USART_HCM_NONE: nRTS signal equals to the rxne status register
     \arg        USART_HCM_EN:   nRTS signal is set when the last data bit has been sampled
   \param[out] none
   \retval     none
*/
void hals_usart_hardware_flow_coherence_config(uint32_t usart_periph, uint32_t hcm)
{
    USART_CHC(usart_periph) &= ~(USART_CHC_HCM);
    USART_CHC(usart_periph) |= (USART_CHC_HCM & hcm);
}

/*!
    \brief      read receive FIFO counter number
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[out] none
    \retval     uint8_t: receive FIFO counter number, 0 - 0xFF
*/
uint8_t hals_usart_receive_fifo_counter_number(uint32_t usart_periph)
{
    uint32_t cnt = 0U;

    cnt |= GET_BITS(USART_FCS(usart_periph), 12U, 14U);
    cnt |= (GET_BITS(USART_FCS(usart_periph), 1U, 2U) << FCS_RFCNT_3_4_OFFSET);

    return (uint8_t)(cnt);
}

/*!
    \brief      get flag in STAT/CHC/FCS register
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  flag: flag type
                only one parameter can be selected which is shown as below:
      \arg        USART_FLAG_PERR: parity error flag
      \arg        USART_FLAG_FERR: frame error flag
      \arg        USART_FLAG_NERR: noise error flag
      \arg        USART_FLAG_ORERR: overrun error
      \arg        USART_FLAG_IDLE: idle line detected flag
      \arg        USART_FLAG_RFN: receive FIFO not empty(when FIFO is enabled)
      \arg        USART_FLAG_RBNE: read data buffer not empty(when FIFO is disabled)
      \arg        USART_FLAG_TC: transmission completed
      \arg        USART_FLAG_TFN: transmit FIFO not full(when FIFO is enabled)
      \arg        USART_FLAG_TBE: transmit data register empty(when FIFO is disabled)
      \arg        USART_FLAG_LBD: LIN break detected flag
      \arg        USART_FLAG_CTSF: CTS change flag
      \arg        USART_FLAG_CTS: CTS level
      \arg        USART_FLAG_RT: receiver timeout flag
      \arg        USART_FLAG_EB: end of block flag
      \arg        USART_FLAG_BSY: busy flag
      \arg        USART_FLAG_AM0: address 0 match flag
      \arg        USART_FLAG_AM1: address 1 match flag
      \arg        USART_FLAG_SB: send break flag
      \arg        USART_FLAG_RWU: receiver wakeup from mute mode.
      \arg        USART_FLAG_WU: wakeup from deep-sleep mode flag
      \arg        USART_FLAG_TEA: transmit enable acknowledge flag
      \arg        USART_FLAG_REA: receive enable acknowledge flag
      \arg        USART_FLAG_EPERR: early parity error flag
      \arg        USART_FLAG_RFE: receive FIFO empty flag
      \arg        USART_FLAG_RFF: receive FIFO full flag
      \arg        USART_FLAG_TFTIF: transmit FIFO threshold interrupt flag
      \arg        USART_FLAG_TFEIF: transmit FIFO empty interrupt flag
      \arg        USART_FLAG_RFTIF: receive FIFO threshold interrupt flag
      \arg        USART_FLAG_RFFIF: receive FIFO full interrupt flag
      \arg        USART_FLAG_TFT: transmit FIFO threshold flag
      \arg        USART_FLAG_TFE: transmit FIFO empty flag
      \arg        USART_FLAG_TFF: transmit FIFO full flag
      \arg        USART_FLAG_RFT: receive FIFO threshold flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_usart_flag_get(uint32_t usart_periph, hal_usart_flag_enum flag)
{
    FlagStatus ret = RESET;

    if(RESET != (USART_REG_VAL(usart_periph, flag) & BIT(USART_BIT_POS(flag)))) {
        return SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      clear USART status
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  flag: flag type
                only one parameter can be selected which is shown as below:
      \arg        USART_FLAG_PERR: parity error flag
      \arg        USART_FLAG_FERR: frame error flag
      \arg        USART_FLAG_NERR: noise detected flag
      \arg        USART_FLAG_ORERR: overrun error flag
      \arg        USART_FLAG_IDLE: idle line detected flag
      \arg        USART_FLAG_TC: transmission complete flag
      \arg        USART_FLAG_LBD: LIN break detected flag
      \arg        USART_FLAG_CTSF: CTS change flag
      \arg        USART_FLAG_RT: receiver timeout flag
      \arg        USART_FLAG_EB: end of block flag
      \arg        USART_FLAG_AM0: address 0 match flag
      \arg        USART_FLAG_AM1: address 1 match flag
      \arg        USART_FLAG_WU: wakeup from deep-sleep mode flag
      \arg        USART_FLAG_EPERR: early parity error flag
      \arg        USART_FLAG_TFE: transmit FIFO empty flag
    \param[out] none
    \retval     none
*/
void hals_usart_flag_clear(uint32_t usart_periph, hal_usart_flag_enum flag)
{
    if(USART_FLAG_AM1 == flag) {
        USART_INTC(usart_periph) |= USART_INTC_AMC1;
    } else if(USART_FLAG_EPERR == flag) {
        USART_CHC(usart_periph) &= (uint32_t)(~USART_CHC_EPERR);
    } else if(USART_FLAG_TFE == flag) {
        USART_FCS(usart_periph) |= USART_FCS_TFEC;
    } else {
        USART_INTC(usart_periph) |= BIT(USART_BIT_POS(flag));
    }
}

/*!
    \brief      enable USART interrupt
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  interrupt: interrupt type
                only one parameter can be selected which is shown as below:
      \arg        USART_INT_IDLE: idle interrupt
      \arg        USART_INT_RFNE: receive FIFO not empty interrupt and overrun error interrupt(when FIFO is enabled)
      \arg        USART_INT_RBNE: read data buffer not empty interrupt and
                                  overrun error interrupt enable interrupt(when FIFO is disabled)
      \arg        USART_INT_TC: transmission complete interrupt
      \arg        USART_INT_TFNF: transmit FIFO not full interrupt(when FIFO is enabled)
      \arg        USART_INT_TBE: transmit data register empty interrupt(when FIFO is disabled)
      \arg        USART_INT_PERR: parity error interrupt
      \arg        USART_INT_AM0: address 0 match interrupt
      \arg        USART_INT_AM1: address 1 match interrupt
      \arg        USART_INT_RT: receiver timeout interrupt
      \arg        USART_INT_EB: end of block interrupt
      \arg        USART_INT_LBD: LIN break detection interrupt
      \arg        USART_INT_ERR: error interrupt enable in multibuffer communication
      \arg        USART_INT_CTS: CTS interrupt
      \arg        USART_INT_WU: wakeup from deep-sleep mode interrupt
      \arg        USART_INT_TFE: transmit FIFO empty interrupt
      \arg        USART_INT_TFT: transmit FIFO threshold interrupt
      \arg        USART_INT_RFT: receive FIFO threshold interrupt
      \arg        USART_INT_RFF: receive FIFO full interrupt
    \param[out] none
    \retval     none
*/
void hals_usart_interrupt_enable(uint32_t usart_periph, hal_usart_interrupt_enum interrupt)
{
    USART_REG_VAL(usart_periph, interrupt) |= BIT(USART_BIT_POS(interrupt));
}

/*!
    \brief      disable USART interrupt
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  interrupt: interrupt type
                only one parameter can be selected which is shown as below:
      \arg        USART_INT_IDLE: idle interrupt
      \arg        USART_INT_RFNE: receive FIFO not empty interrupt and overrun error interrupt(when FIFO is enabled)
      \arg        USART_INT_RBNE: read data buffer not empty interrupt and
                                  overrun error interrupt enable interrupt(when FIFO is disabled)
      \arg        USART_INT_TC: transmission complete interrupt
      \arg        USART_INT_TFNF: transmit FIFO not full interrupt(when FIFO is enabled)
      \arg        USART_INT_TBE: transmit data register empty interrupt(when FIFO is disabled)
      \arg        USART_INT_PERR: parity error interrupt
      \arg        USART_INT_AM0: address 0 match interrupt
      \arg        USART_INT_AM1: address 1 match interrupt
      \arg        USART_INT_RT: receiver timeout interrupt
      \arg        USART_INT_EB: end of block interrupt
      \arg        USART_INT_LBD: LIN break detection interrupt
      \arg        USART_INT_ERR: error interrupt enable in multibuffer communication
      \arg        USART_INT_CTS: CTS interrupt
      \arg        USART_INT_WU: wakeup from deep-sleep mode interrupt
      \arg        USART_INT_TFE: transmit FIFO empty interrupt
      \arg        USART_INT_TFT: transmit FIFO threshold interrupt
      \arg        USART_INT_RFT: receive FIFO threshold interrupt
      \arg        USART_INT_RFF: receive FIFO full interrupt
    \param[out] none
    \retval     none
*/
void hals_usart_interrupt_disable(uint32_t usart_periph, hal_usart_interrupt_enum interrupt)
{
    USART_REG_VAL(usart_periph, interrupt) &= ~BIT(USART_BIT_POS(interrupt));
}

/*!
    \brief      get USART interrupt and flag status
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  int_flag: interrupt and flag type, refer to usart_interrupt_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        USART_INT_FLAG_EB: end of block interrupt and flag
      \arg        USART_INT_FLAG_RT: receiver timeout interrupt and flag
      \arg        USART_INT_FLAG_AM0: address 0 match interrupt and flag
      \arg        USART_INT_FLAG_AM1: address 1 match interrupt and flag
      \arg        USART_INT_FLAG_PERR: parity error interrupt and flag
      \arg        USART_INT_FLAG_TBE: transmitter buffer empty interrupt and flag(when FIFO is disabled)
      \arg        USART_INT_FLAG_TFNF: transmit FIFO not full interrupt and flag(when FIFO is enabled)
      \arg        USART_INT_FLAG_TC: transmission complete interrupt and flag
      \arg        USART_INT_FLAG_RBNE: read data buffer not empty interrupt and flag(when FIFO is disabled)
      \arg        USART_INT_FLAG_RFNE: receive FIFO not empty interrupt and flag(when FIFO is enabled)
      \arg        USART_INT_FLAG_RBNE_ORERR: read data buffer not empty interrupt and
                                             overrun error flag(when FIFO is disabled)
      \arg        USART_INT_FLAG_RFNE_ORERR: receive FIFO not empty interrupt and
                                             overrun error flag(when FIFO is enabled)
      \arg        USART_INT_FLAG_IDLE: IDLE line detected interrupt and flag
      \arg        USART_INT_FLAG_LBD: LIN break detected interrupt and flag
      \arg        USART_INT_FLAG_WU: wakeup from deep-sleep mode interrupt and flag
      \arg        USART_INT_FLAG_CTS: CTS interrupt and flag
      \arg        USART_INT_FLAG_ERR_NERR: error interrupt and noise error flag
      \arg        USART_INT_FLAG_ERR_ORERR: error interrupt and overrun error
      \arg        USART_INT_FLAG_ERR_FERR: error interrupt and frame error flag
      \arg        USART_INT_FLAG_TFT: transmit FIFO threshold interrupt and flag
      \arg        USART_INT_FLAG_TFE: transmit FIFO empty interrupt and flag
      \arg        USART_INT_FLAG_RFT: receive FIFO threshold interrupt and flag
      \arg        USART_INT_FLAG_RFF: receive FIFO full interrupt and flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_usart_interrupt_flag_get(uint32_t usart_periph, hal_usart_interrupt_flag_enum int_flag)
{
    FlagStatus ret = RESET;
    uint32_t intenable = 0U, flagstatus = 0U;

    /* get the interrupt enable bit status */
    intenable = (USART_REG_VAL(usart_periph, int_flag) & BIT(USART_BIT_POS(int_flag)));
    /* get the corresponding flag bit status */
    flagstatus = (USART_REG_VAL2(usart_periph, int_flag) & BIT(USART_BIT_POS2(int_flag)));

    if(flagstatus && intenable) {
        return SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      clear USART interrupt flag
    \param[in]  usart_periph: USARTx(x=0,1,2,5), UARTx(x=3,4,6,7)
    \param[in]  int_flag: USART interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        USART_INT_FLAG_PERR: parity error flag
      \arg        USART_INT_FLAG_ERR_FERR: frame error flag
      \arg        USART_INT_FLAG_ERR_NERR: noise detected flag
      \arg        USART_INT_FLAG_RBNE_ORERR: read data buffer not empty interrupt and overrun error flag
      \arg        USART_INT_FLAG_ERR_ORERR: error interrupt and overrun error
      \arg        USART_INT_FLAG_IDLE: idle line detected flag
      \arg        USART_INT_FLAG_TC: transmission complete flag
      \arg        USART_INT_FLAG_LBD: LIN break detected flag
      \arg        USART_INT_FLAG_CTS: CTS change flag
      \arg        USART_INT_FLAG_RT: receiver timeout flag
      \arg        USART_INT_FLAG_EB: end of block flag
      \arg        USART_INT_FLAG_AM0: address 0 match flag
      \arg        USART_INT_FLAG_AM1: address 1 match flag
      \arg        USART_INT_FLAG_WU: wakeup from deep-sleep mode flag
      \arg        USART_INT_FLAG_RFF: receive FIFO full interrupt and flag
      \arg        USART_INT_FLAG_TFE: transmit FIFO empty interrupt and flag
    \param[out] none
    \retval     none
*/
void hals_usart_interrupt_flag_clear(uint32_t usart_periph, hal_usart_interrupt_flag_enum int_flag)
{
    if(USART_INT_FLAG_TFE == int_flag) {
        USART_FCS(usart_periph) |= USART_FCS_TFEC;
    } else if(USART_INT_FLAG_RFF == int_flag) {
        USART_FCS(usart_periph) &= (uint32_t)(~USART_FCS_RFFIF);
    } else {
        USART_INTC(usart_periph) |= BIT(USART_BIT_POS2(int_flag));
    }
}
