/*!
    \file    gd32h7xx_hal_syscfg.c
    \brief   SYSCFG driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/*!
    \brief      deinitialize SYSCFG
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_syscfg_deinit(void)
{
    hal_rcu_periph_reset_enable(RCU_SYSCFGRST);
    hal_rcu_periph_reset_disable(RCU_SYSCFGRST);
}

/*!
    \brief      enable I2Cx(x=0,1,2,3) fast mode plus or I2C fast mode plus PBx(x=6,7,8,9)
    \param[in]  i2c_fmp: I2C fast mode plus or I2C fast mode plus PBx(x=6,7,8,9)
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_I2C0_FMP: I2C0 fast mode plus
      \arg        SYSCFG_I2C1_FMP: I2C1 fast mode plus
      \arg        SYSCFG_I2C2_FMP: I2C2 fast mode plus
      \arg        SYSCFG_I2C3_FMP: I2C3 fast mode plus
      \arg        SYSCFG_I2C_FMP_PB6: I2C fast mode plus on PB6 pin
      \arg        SYSCFG_I2C_FMP_PB7: I2C fast mode plus on PB7 pin
      \arg        SYSCFG_I2C_FMP_PB8: I2C fast mode plus on PB8 pin
      \arg        SYSCFG_I2C_FMP_PB9: I2C fast mode plus on PB9 pin
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_i2c_fast_mode_plus_enable(hal_syscfg_i2cx_fast_mode_plus_enum i2c_fmp)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYSCFG_I2C0_FMP    != i2c_fmp) && (SYSCFG_I2C1_FMP != i2c_fmp)    && \
       (SYSCFG_I2C2_FMP    != i2c_fmp) && (SYSCFG_I2C3_FMP != i2c_fmp)    && \
       (SYSCFG_I2C_FMP_PB6 != i2c_fmp) && (SYSCFG_I2C_FMP_PB7 != i2c_fmp) && \
       (SYSCFG_I2C_FMP_PB8 != i2c_fmp) && (SYSCFG_I2C_FMP_PB9 != i2c_fmp)) {
        HAL_DEBUGE("parameter [i2c_fmp] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SYSCFG_PMCFG |= i2c_fmp;

    return HAL_ERR_NONE;
}

/*!
    \brief      disable I2Cx(x=0,1,2,3) fast mode plus or I2C fast mode plus PBx(x=6,7,8,9)
    \param[in]  i2c_fmp: I2C fast mode plus or I2C fast mode plus PBx(x=6,7,8,9)
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_I2C0_FMP: I2C0 fast mode plus
      \arg        SYSCFG_I2C1_FMP: I2C1 fast mode plus
      \arg        SYSCFG_I2C2_FMP: I2C2 fast mode plus
      \arg        SYSCFG_I2C3_FMP: I2C3 fast mode plus
      \arg        SYSCFG_I2C_FMP_PB6: I2C fast mode plus on PB6 pin
      \arg        SYSCFG_I2C_FMP_PB7: I2C fast mode plus on PB7 pin
      \arg        SYSCFG_I2C_FMP_PB8: I2C fast mode plus on PB8 pin
      \arg        SYSCFG_I2C_FMP_PB9: I2C fast mode plus on PB9 pin
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_i2c_fast_mode_plus_disable(hal_syscfg_i2cx_fast_mode_plus_enum i2c_fmp)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYSCFG_I2C0_FMP    != i2c_fmp) && (SYSCFG_I2C1_FMP != i2c_fmp)    && \
       (SYSCFG_I2C2_FMP    != i2c_fmp) && (SYSCFG_I2C3_FMP != i2c_fmp)    && \
       (SYSCFG_I2C_FMP_PB6 != i2c_fmp) && (SYSCFG_I2C_FMP_PB7 != i2c_fmp) && \
       (SYSCFG_I2C_FMP_PB8 != i2c_fmp) && (SYSCFG_I2C_FMP_PB9 != i2c_fmp)) {
        HAL_DEBUGE("parameter [i2c_fmp] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SYSCFG_PMCFG &= ~(uint32_t)(i2c_fmp);

    return HAL_ERR_NONE;
}

/*!
    \brief      open analog switch
    \param[in]  gpio_answ: GPIO analog switch
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_PA0_ANALOG_SWITCH: PA0 analog switch
      \arg        SYSCFG_PA1_ANALOG_SWITCH: PA1 analog switch
      \arg        SYSCFG_PC2_ANALOG_SWITCH: PC2 analog switch
      \arg        SYSCFG_PC3_ANALOG_SWITCH: PC3 analog switch
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_analog_switch_enable(hal_syscfg_pin_analog_enum gpio_answ)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYSCFG_PA0_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PA1_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PC2_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PC3_ANALOG_SWITCH != gpio_answ)) {
        HAL_DEBUGE("parameter [gpio_answ] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SYSCFG_PMCFG |= (uint32_t)(gpio_answ);

    return HAL_ERR_NONE;
}

/*!
    \brief      close analog switch
    \param[in]  gpio_answ: GPIO analog switch
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_PA0_ANALOG_SWITCH: PA0 analog switch
      \arg        SYSCFG_PA1_ANALOG_SWITCH: PA1 analog switch
      \arg        SYSCFG_PC2_ANALOG_SWITCH: PC2 analog switch
      \arg        SYSCFG_PC3_ANALOG_SWITCH: PC3 analog switch
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_analog_switch_disable(hal_syscfg_pin_analog_enum gpio_answ)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYSCFG_PA0_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PA1_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PC2_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PC3_ANALOG_SWITCH != gpio_answ)) {
        HAL_DEBUGE("parameter [gpio_answ] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SYSCFG_PMCFG &= ~(uint32_t)(gpio_answ);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the PHY interface for the ethernet MAC
    \param[in]  ethernet: ENETx(x=0,1)
    \param[in]  phy_interface: specifies the media interface mode
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_ENET_PHY_MII: MII mode is selected
      \arg        SYSCFG_ENET_PHY_RMII: RMII mode is selected
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_enet_phy_interface_config(uint32_t ethernet, uint32_t phy_interface)
{
    uint32_t reg = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((ENET0 != ethernet) && (ENET1 != ethernet)) {
        HAL_DEBUGE("parameter [ethernet] value is undefine");
        return HAL_ERR_VAL;
    }

    if((SYSCFG_ENET_PHY_MII != phy_interface) && (SYSCFG_ENET_PHY_RMII != phy_interface)) {
        HAL_DEBUGE("parameter [phy_interface] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    reg = SYSCFG_PMCFG;

    /* configure the ENET media interface */
    if(ENET0 == ethernet) {
        reg &= ~SYSCFG_PMCFG_ENET0_PHY_SEL;
        reg |= ENET0_MEDIA_INTERFACE(phy_interface);
    } else {
        reg &= ~SYSCFG_PMCFG_ENET1_PHY_SEL;
        reg |= ENET1_MEDIA_INTERFACE(phy_interface);
    }

    SYSCFG_PMCFG = reg;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the GPIO pin as EXTI Line
    \param[in]  exti_port: specify the GPIO port used in EXTI
                only one parameter can be selected which is shown as below:
      \arg        EXTI_SOURCE_GPIOx(x = A,B,C,D,E,F,G,H,J,K): EXTI GPIO port
    \param[in]  exti_pin: specify the EXTI line
                only one parameter can be selected which is shown as below:
      \arg        EXTI_SOURCE_PINx(GPIOA x = 0..15,GPIOB x = 0..15,GPIOC x = 0..15,GPIOD x = 0..15,
                  GPIOE x = 0..15, GPIOF x = 0..15,GPIOG x = 0..15,GPIOH x = 0..15,GPIOI x = 0..15,
                  GPIOJ x = 8,9,10,11, GPIOK x = 0,1,2,4,5,6): EXTI GPIO pin
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_exti_config(uint32_t exti_port, uint32_t exti_pin)
{
    uint32_t clear_exti_mask  = ~((uint32_t)EXTI_SS_MASK << (EXTI_SS_MSTEP(exti_pin)));
    uint32_t config_exti_mask = ((uint32_t)exti_port) << (EXTI_SS_MSTEP(exti_pin));
    int32_t  ret              = HAL_ERR_NONE;

    switch(exti_pin / EXTI_SS_JSTEP) {
    case EXTISS0:
        /* clear EXTI source line(0..3) */
        SYSCFG_EXTISS0 &= clear_exti_mask;
        /* configure EXTI source line(0..3) */
        SYSCFG_EXTISS0 |= config_exti_mask;
        break;
    case EXTISS1:
        /* clear EXTI source line(4..7) */
        SYSCFG_EXTISS1 &= clear_exti_mask;
        /* configure EXTI source line(4..7) */
        SYSCFG_EXTISS1 |= config_exti_mask;
        break;
    case EXTISS2:
        /* clear EXTI source line(8..11) */
        SYSCFG_EXTISS2 &= clear_exti_mask;
        /* configure EXTI source line(8..11) */
        SYSCFG_EXTISS2 |= config_exti_mask;
        break;
    case EXTISS3:
        /* clear EXTI source line(12..15) */
        SYSCFG_EXTISS3 &= clear_exti_mask;
        /* configure EXTI source line(12..15) */
        SYSCFG_EXTISS3 |= config_exti_mask;
        break;
    default:
        HAL_DEBUGE("parameter [exti_pin] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      select timer channel input source
    \param[in]  timer_input: system timer input source,
                the argument could be selected from enumeration <hal_syscfg_timer_channel_input_enum>
    \param[out] none
    \retval     none
*/
void hal_syscfg_timer_input_source_select(hal_syscfg_timer_channel_input_enum timer_input)
{
    uint32_t clear_timer_mask  = ~((uint32_t)TIMER_IS_MASK << (TIMER_BIT_POS(timer_input)));
    uint32_t config_timer_mask = (TIMER_SEL_VAL(timer_input) << TIMER_BIT_POS(timer_input));

    switch(TIMER_REG_INDEX(timer_input)) {
    case TIMERCISEL0:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL0 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL0 |= config_timer_mask;
        break;
    case TIMERCISEL1:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL1 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL1 |= config_timer_mask;
        break;
    case TIMERCISEL2:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL2 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL2 |= config_timer_mask;
        break;
    case TIMERCISEL3:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL3 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL3 |= config_timer_mask;
        break;
    case TIMERCISEL4:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL4 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL4 |= config_timer_mask;
        break;
    case TIMERCISEL5:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL5 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL5 |= config_timer_mask;
        break;
    case TIMERCISEL6:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL6 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL6 |= config_timer_mask;
        break;
    default:
        break;
    }
}

/*!
    \brief      enable TCM wait state
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_syscfg_tcm_wait_state_enable(void)
{
    SYSCFG_SRAMCFG1 |= SYSCFG_SRAMCFG1_TCM_WAITSTATE;
}

/*!
    \brief      disable TCM wait state
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_syscfg_tcm_wait_state_disable(void)
{
    SYSCFG_SRAMCFG1 &= ~SYSCFG_SRAMCFG1_TCM_WAITSTATE;
}

/*!
    \brief      enable module lockup function (function can be disabled by system reset)
    \param[in]  lockup: module lockup function
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_LVD_LOCKUP: LVD signal
      \arg        SYSCFG_CPU_LOCKUP: CPU lockup signal
      \arg        SYSCFG_BKPRAM_LOCKUP: Region 2 backup SRAM ECC double error signal
      \arg        SYSCFG_SRAM1_LOCKUP: Region 1 SRAM1 ECC double error signal
      \arg        SYSCFG_SRAM0_LOCKUP: Region 1 SRAM0 ECC double error signal
      \arg        SYSCFG_DTCM_LOCKUP: Region 0 DTCM ECC double error signal
      \arg        SYSCFG_ITCM_LOCKUP: Region 0 ITCM-RAM ECC double error signal
      \arg        SYSCFG_AXIRAM_LOCKUP: Region 0 AXI-SRAM ECC double error signal
    \param[out] none
    \retval     none
*/
void hals_syscfg_lockup_enable(hal_syscfg_module_lockup_enum lockup)
{
    SYSCFG_LKCTL |= lockup;
}

/*!
    \brief    configure the I/O compensation cell
    \param[in]  syscfg_cps: specifies the I/O compensation cell mode
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_IO_COMPENSATION_ENABLE: I/O compensation cell is enabled
      \arg        SYSCFG_IO_COMPENSATION_DISABLE: I/O compensation cell is disabled
    \param[out] none
    \retval     none
*/
void hals_syscfg_io_compensation_config(uint32_t syscfg_cps)
{
    uint32_t reg = 0U;

    /* reset the SYSCFG_CPSCTL_CPS_EN bit and set according to syscfg_compensation */
    reg = SYSCFG_CPSCTL;
    reg &= ~SYSCFG_CPSCTL_CPS_EN;
    SYSCFG_CPSCTL = (reg | syscfg_cps);
}

/*!
    \brief      set P/N MOS compensation value
    \param[in]  mos: specifies the P/N MOS type
                only one parameter can be selected which is shown as below:
      \arg        NMOS_COMPENSATION: NMOS
      \arg        PMOS_COMPENSATION: PMOS
    \param[in]  code: P/N MOS compensation value
    \param[out] none
    \retval     none
*/
void hals_syscfg_pnmos_compensation_code_set(uint32_t mos, uint32_t code)
{
    uint32_t value = 0U;

    value = SYSCFG_CPSCCCFG;

    if(NMOS_COMPENSATION == mos) {
        value &= ~SYSCFG_CPSCCCFG_NCPSCC;
        value |= (code & 0x0FU);
    } else {
        value &= ~SYSCFG_CPSCCCFG_PCPSCC;
        value |= ((code & 0x0FU) << 4U);
    }

    SYSCFG_CPSCCCFG = value;
}

/*!
    \brief      enable I/O speed optimization, high-speed at low-voltage
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_syscfg_io_low_voltage_speed_optimization_enable(void)
{
    SYSCFG_CPSCTL |= SYSCFG_CPSCTL_IOSPDOP;
}

/*!
    \brief      disable I/O speed optimization, high-speed at low-voltage
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_syscfg_io_low_voltage_speed_optimization_disable(void)
{
    SYSCFG_CPSCTL &= ~SYSCFG_CPSCTL_IOSPDOP;
}

/*!
    \brief      set secure SRAM size
    \param[in]  size: secure SRAM size
                one or more parameters can be selected which are shown as below:
      \arg        SECURE_SRAM_SIZE_0KB: secure SRAM size is 0KB
      \arg        SECURE_SRAM_SIZE_32KB: secure SRAM size is 32KB
      \arg        SECURE_SRAM_SIZE_64KB: secure SRAM size is 64KB
      \arg        SECURE_SRAM_SIZE_128KB: secure SRAM size is 128KB
    \param[out] none
    \retval     none
*/
void hals_syscfg_secure_sram_size_set(uint32_t size)
{
    SYSCFG_SRAMCFG0 &= (uint32_t)(~SYSCFG_SRAMCFG0_SECURE_SRAM_SIZE);
    SYSCFG_SRAMCFG0 |= size;
}

/*!
    \brief      get secure SRAM size
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0x00000000 - 0xFFFFFFFF
*/
uint32_t hals_syscfg_secure_sram_size_get(void)
{
    return (SYSCFG_SRAMCFG0 & SYSCFG_SRAMCFG0_SECURE_SRAM_SIZE);
}

/*!
    \brief      get BOOT mode
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0x00000000 - 0xFFFFFFFF
*/
uint32_t hals_syscfg_bootmode_get(void)
{
    return ((SYSCFG_USERCFG & SYSCFG_USERCFG_BOOT_MODE) >> 4U);
}

/*!
    \brief      enable FPU interrupt
    \param[in]  interrupt: FPU interrupt
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_FPUINT_INEXACT: inexact interrupt
      \arg        SYSCFG_FPUINT_INPUT_ABNORMAL: input abnormal interrupt
      \arg        SYSCFG_FPUINT_OVERFLOW: overflow interrupt
      \arg        SYSCFG_FPUINT_UNDERFLOW: underflow interrupt
      \arg        SYSCFG_FPUINT_DIV0: divide-by-zero interrupt
      \arg        SYSCFG_FPUINT_INVALID_OPERATION: invalid operation interrupt
    \param[out] none
    \retval     none
*/
void hals_syscfg_fpu_interrupt_enable(hal_syscfg_fpuint_enum interrupt)
{
    SYSCFG_FPUINTEN |= (uint32_t)interrupt;
}

/*!
    \brief      disable FPU interrupt
    \param[in]  interrupt: FPU interrupt
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_FPUINT_INEXACT: inexact interrupt
      \arg        SYSCFG_FPUINT_INPUT_ABNORMAL: input abnormal interrupt
      \arg        SYSCFG_FPUINT_OVERFLOW: overflow interrupt
      \arg        SYSCFG_FPUINT_UNDERFLOW: underflow interrupt
      \arg        SYSCFG_FPUINT_DIV0: divide-by-zero interrupt
      \arg        SYSCFG_FPUINT_INVALID_OPERATION: invalid operation interrupt
    \param[out] none
    \retval     none
*/
void hals_syscfg_fpu_interrupt_disable(hal_syscfg_fpuint_enum interrupt)
{
    SYSCFG_FPUINTEN &= ~(uint32_t)interrupt;
}

/*!
    \brief      get compensation cell flags
    \param[in]  flag: compensation flag
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_FLAG_IO_LOW_VOLTAGE: I/O in low voltage state flag,
                                              product supply voltage is working below 2.5V
      \arg        SYSCFG_FLAG_COMPENSATION_READY: I/O compensation cell ready flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_syscfg_compensation_flag_get(uint32_t flag)
{
    FlagStatus ret = RESET;

    if(SYSCFG_CPSCTL & flag) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      get the ICACHE or DCACHE detection and error information
    \param[in]  cache: select cache
                only one parameter can be selected which is shown as below:
      \arg        ICACHE_STATUS: select ICACHE
      \arg        DCACHE_STATUS: select DCACHE
    \param[in]  status: select detection or error information
                only one parameter can be selected which is shown as below:
      \arg        CPU_CACHE_ERROR_DETECTION: select detection information
      \arg        CPU_CACHE_ERROR_BANK: select error information
    \param[out] none
    \retval     uint32_t: 0x00000000 - 0xFFFFFFFF
*/
uint32_t hals_syscfg_cpu_cache_status_get(uint32_t cache, uint32_t status)
{
    uint32_t value = 0U;

    switch(cache) {
    /* get ICACHE information */
    case ICACHE_STATUS:
        if(CPU_CACHE_ERROR_DETECTION == status) {
            /* return detection information */
            value = (uint32_t)((SYSCFG_CPUICAC & SYSCFG_CPUICAC_CPU_ICDET) >> 28U);
        } else {
            /* return error bank information */
            value = (uint32_t)((SYSCFG_CPUICAC & SYSCFG_CPUICAC_CPU_ICERR) >> 6U);
        }
        break;
    /* get DCACHE information */
    case DCACHE_STATUS:
        if(CPU_CACHE_ERROR_DETECTION == status) {
            /* return detection information */
            value = (uint32_t)((SYSCFG_CPUDCAC & SYSCFG_CPUICAC_CPU_ICDET) >> 28U);
        } else {
            /* return error bank information */
            value = (uint32_t)((SYSCFG_CPUDCAC & SYSCFG_CPUICAC_CPU_ICERR) >> 6U);
        }
        break;
    default:
        break;
    }

    return value;
}

/*!
    \brief      get brownout reset threshold level
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0x00000000 - 0xFFFFFFFF
*/
uint32_t hals_syscfg_brownout_reset_threshold_level_get(void)
{
    return (SYSCFG_USERCFG & SYSCFG_USERCFG_BOR_TH);
}
