/*!
    \file    gdh7xx_hal_smbus.c
    \brief   SMBUS driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define SMBUS_MAX_BYTE_SIZE                   (255U)
#define SMBUS_DATA_SEND_FINISH_LENGTH         (0U)
#define SMBUS_DATA_RECEIVED_FINISH_LENGTH     (0U)
#define I2C_CTL1_UPDATA_MASK                  ((uint32_t) ~(uint32_t)(I2C_CTL1_SADDRESS | I2C_CTL1_BYTENUM | \
                                                                      I2C_CTL1_RELOAD | I2C_CTL1_AUTOEND | \
                                                                      I2C_CTL1_TRDIR | I2C_CTL1_START | I2C_CTL1_STOP | \
                                                                      I2C_CTL1_PECTRANS))

/* wait the flag status until timeout */
static int32_t _smbus_wait_flag_timeout(uint32_t smbus_periph, hal_i2c_flag_enum flag, \
                                        FlagStatus status, uint32_t timeout_ms);
static void _smbus_master_transmit_config(uint32_t smbus_periph, uint32_t length, uint32_t mode, uint32_t request);

static void _smbus_flush_tdata_register(uint32_t smbus_periph);

static void _smbus_master_transfer_interrupt(void *smbus_dev);
static void _smbus_master_received_interrupt(void *smbus_dev);
static void _smbus_slave_transmit_interrupt(void *smbus_dev);
static void _smbus_slave_received_interrupt(void *smbus_dev);
static void _smbus_address_listen_interrupt(void *smbus_dev);

/*!
    \brief      initialize the SMBUS structure with the default values
    \param[in]  struct_type: refer to hal_smbus_struct_type_enum
                  only one parameter can be selected which is shown as below:
      \arg        HAL_SMBUS_INIT_STRUCT: SMBUS initialization structure
      \arg        HAL_SMBUS_DEV_STRUCT: SMBUS device information structure
      \arg        HAL_SMBUS_IRQ_STRUCT: SMBUS interrupt callback function structure
      \arg        HAL_SMBUS_BUFFER_STRUCT: SMBUS buffer structure
      \arg        HAL_SMBUS_IRQ_USER_CALLBACK_STRUCT: SMBUS interrupt callback function structure
    \param[out] p_struct: point to SMBUS structure that contains the configuration information
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_struct_init(hal_smbus_struct_type_enum struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(struct_type) {
    case HAL_SMBUS_INIT_STRUCT:
        /* initialize SMBUS initialization structure with the default values */
        ((hal_smbus_init_struct *)p_struct)->scl_stretch    = SMBUS_SCLSTRETCH_DISABLE;
        ((hal_smbus_init_struct *)p_struct)->address_format = SMBUS_ADDFORMAT_7BITS;
        ((hal_smbus_init_struct *)p_struct)->own_address1   = 0U;
        ((hal_smbus_init_struct *)p_struct)->dual_address   = SMBUS_DUADEN_DISABLE;
        ((hal_smbus_init_struct *)p_struct)->address2_mask  = ADDRESS2_NO_MASK;
        ((hal_smbus_init_struct *)p_struct)->own_address2   = 0U;
        ((hal_smbus_init_struct *)p_struct)->general_call   = SMBUS_GCEN_DISABLE;
        ((hal_smbus_init_struct *)p_struct)->time           = 0U;
        ((hal_smbus_init_struct *)p_struct)->digital_filter = DIGITAL_FILTER_DISABLE;
        ((hal_smbus_init_struct *)p_struct)->analog_filter  = SMBUS_DUADEN_DISABLE;
        ((hal_smbus_init_struct *)p_struct)->smbus_pec      = SMBUS_PEC_DISABLE;
        ((hal_smbus_init_struct *)p_struct)->smbus_type     = SMBUS_SLAVE;
        ((hal_smbus_init_struct *)p_struct)->smbus_timeout  = 0U;
        break;
    case HAL_SMBUS_DEV_STRUCT:
        /* initialize SMBUS device information structure with the default values */
        ((hal_smbus_dev_struct *)p_struct)->periph                 = 0U;
        ((hal_smbus_dev_struct *)p_struct)->smbus_irq.event_handle = NULL;
        ((hal_smbus_dev_struct *)p_struct)->smbus_irq.error_handle = NULL;
        ((hal_smbus_dev_struct *)p_struct)->txbuffer.buffer        = NULL;
        ((hal_smbus_dev_struct *)p_struct)->txbuffer.length        = 0U;
        ((hal_smbus_dev_struct *)p_struct)->txbuffer.pos           = 0U;
        ((hal_smbus_dev_struct *)p_struct)->rxbuffer.buffer        = NULL;
        ((hal_smbus_dev_struct *)p_struct)->rxbuffer.length        = 0U;
        ((hal_smbus_dev_struct *)p_struct)->rxbuffer.pos           = 0U;
        ((hal_smbus_dev_struct *)p_struct)->smbus_pec_transfer     = SMBUS_PECTRANS_DISABLE;
        ((hal_smbus_dev_struct *)p_struct)->error_state            = HAL_SMBUS_ERROR_NONE;
        ((hal_smbus_dev_struct *)p_struct)->tx_state               = HAL_SMBUS_STATE_FREE;
        ((hal_smbus_dev_struct *)p_struct)->rx_state               = HAL_SMBUS_STATE_FREE;
        ((hal_smbus_dev_struct *)p_struct)->last_error             = (uint16_t)HAL_SMBUS_ERROR_NONE;
        ((hal_smbus_dev_struct *)p_struct)->rx_callback            = NULL;
        ((hal_smbus_dev_struct *)p_struct)->tx_callback            = NULL;
        ((hal_smbus_dev_struct *)p_struct)->addr_callback          = NULL;
        ((hal_smbus_dev_struct *)p_struct)->priv                   = NULL;
        ((hal_smbus_dev_struct *)p_struct)->mutex                  = HAL_MUTEX_UNLOCKED;
        break;
    case HAL_SMBUS_IRQ_STRUCT:
        /* initialize SMBS irq structure with the default values */
        ((hal_smbus_irq_struct *)p_struct)->error_handle = NULL;
        ((hal_smbus_irq_struct *)p_struct)->event_handle = NULL;
        break;
    case HAL_SMBUS_BUFFER_STRUCT:
        ((hal_smbus_buffer_struct *)p_struct)->buffer = NULL;
        ((hal_smbus_buffer_struct *)p_struct)->length = 0U;
        ((hal_smbus_buffer_struct *)p_struct)->pos    = 0U;
        break;
    case HAL_SMBUS_IRQ_USER_CALLBACK_STRUCT:
        ((hal_smbus_irq_user_callback_struct *)p_struct)->rx_callback    = NULL;
        ((hal_smbus_irq_user_callback_struct *)p_struct)->tx_callback    = NULL;
        ((hal_smbus_irq_user_callback_struct *)p_struct)->addr_callback  = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      initialize smbus
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which SMBUS is initialized
    \param[in]  smbus_init: the initialization data needed to initialize SMBUS
                  scl_stretch:
                    only one parameter can be selected which is shown as below:
        \arg        SMBUS_SCLSTRETCH_ENABLE: SMBUS SCL stretch is enabled
        \arg        SMBUS_SCLSTRETCH_DISABLE: SMBUS SCL stretch is disabled
                  address_format:
                    only one parameter can be selected which is shown as below:
       \arg         SMBUS_ADDFORMAT_7BITS: SMBUS address format is 7-bit
       \arg         SMBUS_ADDFORMAT_10BITS: SMBUS address format is 10-bit
                  own_address1: SMBUS address
                  dual_address:
                    only one parameter can be selected which is shown as below:
       \arg         SMBUS_DUADEN_ENABLE: SMBUS dual-address mode is enabled
       \arg         SMBUS_DUADEN_DISABLE: SMBUS dual-address mode is disabled
                  own_address2: the second address in dual-address mode
                  general_call:
                    only one parameter can be selected which is shown as below:
        \arg        SMBUS_GCEN_ENABLE: SMBUS general call is enabled
        \arg        SMBUS_GCEN_DISABLE: SMBUS general call is disabled
                  time: the I2C clock timing register value
                  digital_filter: the digital filter value, refer to hal_i2c_digital_filter_enum
                    only one parameter can be selected which is shown as below:
        \arg        DIGITAL_FILTER_DISABLE: digital filter is disabled
        \arg        DIGITAL_FILTER_LESSTHAN_1CLK: filter out spikes with pulse width less than 1 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_2CLK: filter out spikes with pulse width less than 2 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_3CLK: filter out spikes with pulse width less than 3 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_4CLK: filter out spikes with pulse width less than 4 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_5CLK: filter out spikes with pulse width less than 5 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_6CLK: filter out spikes with pulse width less than 6 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_7CLK: filter out spikes with pulse width less than 7 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_8CLK: filter out spikes with pulse width less than 8 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_9CLK: filter out spikes with pulse width less than 9 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_10CLK: filter out spikes with pulse width less than 10 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_11CLK: filter out spikes with pulse width less than 11 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_12CLK: filter out spikes with pulse width less than 12 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_13CLK: filter out spikes with pulse width less than 13 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_14CLK: filter out spikes with pulse width less than 14 tI2CCLK
        \arg        DIGITAL_FILTER_LESSTHAN_15CLK: filter out spikes with pulse width less than 15 tI2CCLK
                  analog_filter:
                    only one parameter can be selected which is shown as below:
        \arg        ANALOG_FILTER_ENABLE: analog noise filter is enabled
        \arg        ANALOG_FILTER_DISABLE: analog noise filter is disabled
                  smbus_pec:
                    only one parameter can be selected which is shown as below:
        \arg        SMBUS_PEC_ENABLE: SMBUS PEC is enabled
        \arg        SMBUS_PEC_DISABLE: SMBUS PEC is disabled
                  smbus_type:
                    only one parameter can be selected which is shown as below:
       \arg         SMBUS_DEVICE: SMBUS device mode
       \arg         SMBUS_HOST: SMBUS host mode
                  smbus_timeout: the I2C timeout register value
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_init(hal_smbus_dev_struct *smbus_dev, uint32_t periph, hal_smbus_init_struct *smbus_init)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == smbus_dev) && (NULL == smbus_init)) {
        HAL_DEBUGE("pointer [smbus_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    smbus_dev->periph = periph;

    /* disable peripheral */
    HAL_I2C_DISABLE(smbus_dev->periph);

    /* configure I2C clock */
    I2C_TIMING(smbus_dev->periph) = smbus_init->time;

    /* configure I2C Timeout */
    I2C_TIMEOUT(smbus_dev->periph) = smbus_init->smbus_timeout;

    /* configure analog noise filter */
    if(ANALOG_FILTER_ENABLE == smbus_init->analog_filter) {
        I2C_CTL0(smbus_dev->periph) &= ~I2C_CTL0_ANOFF;
    } else {
        I2C_CTL0(smbus_dev->periph) |= I2C_CTL0_ANOFF;
    }

    /* configure digital noise filter */
    if(DIGITAL_FILTER_DISABLE != smbus_init->digital_filter) {
        I2C_CTL0(smbus_dev->periph) |= (uint32_t)((uint32_t)smbus_init->digital_filter << CTL0_DNF_OFFSET);
    } else {
        I2C_CTL0(smbus_dev->periph) |= (uint32_t)((uint32_t)DIGITAL_FILTER_DISABLE << CTL0_DNF_OFFSET);
    }

    /* configure I2C address */
    if(HAL_ERR_NONE != hals_i2c_device_address_config(smbus_dev->periph, smbus_init->own_address1, \
                                                      smbus_init->address_format)) {
        HAL_DEBUGE("parameter [smbus_init->own_address1] value is invalid");
        smbus_dev->error_state = HAL_SMBUS_ERROR_ADDRESS;
        ret = HAL_ERR_VAL;
    } else {
        /* configure dual-address mode */
        if(I2C_ADDFORMAT_10BITS == smbus_init->address_format) {
            hals_i2c_address10_enable(smbus_dev->periph);
        } else {
            /* do nothing */
        }

        /* configure dual-address mode */
        if(I2C_DUADEN_ENABLE == smbus_init->dual_address) {
            smbus_init->own_address2 = smbus_init->own_address2;
            if(HAL_ERR_NONE != hals_i2c_device_second_address_enable(smbus_dev->periph, smbus_init->own_address2, \
                                                                     smbus_init->address2_mask)) {
                HAL_DEBUGE("parameter [smbus_init->own_address2] value is invalid");
                smbus_dev->error_state = HAL_SMBUS_ERROR_ADDRESS;
                ret = HAL_ERR_VAL;
            } else {
                /* do nothing */
            }
        } else {
            hals_i2c_device_second_address_disable(smbus_dev->periph);
        }

        /* configure whether to stretch SCL low when data is not ready in slave mode */
        hals_i2c_stretch_scl_low_config(smbus_dev->periph, smbus_init->scl_stretch);

        /* whether or not to response to a general call */
        hals_i2c_slave_response_to_gcall_config(smbus_dev->periph, smbus_init->general_call);

        hals_smbus_type_config(smbus_dev->periph, smbus_init->smbus_type);

        hals_i2c_smbus_pec_config(smbus_dev->periph, smbus_init->smbus_pec);

        /* enable peripheral */
        HAL_I2C_ENABLE(smbus_dev->periph);

        /* enable acknowledge */
        if(RESET == ((I2C_SADDR0(periph) & I2C_SADDR0_ADDFORMAT) >> 10)) {
            hals_i2c_ack_config(smbus_dev->periph, SMBUS_ACK_ENABLE);
        } else {
            /* do nothing */
        }

        smbus_dev->tx_state = HAL_SMBUS_STATE_READY;
        smbus_dev->rx_state = HAL_SMBUS_STATE_READY;
    }

    return ret;
}

/*!
    \brief      deinitialize smbus
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE ,HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_deinit(hal_smbus_dev_struct *smbus_dev)
{
    int32_t  ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smbus_dev) {
        HAL_DEBUGE("pointer [smbus_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(smbus_dev->periph) {
    case I2C0:
        hal_rcu_periph_reset_enable(RCU_I2C1RST);
        hal_rcu_periph_reset_disable(RCU_I2C1RST);
        hal_smbus_struct_init(HAL_SMBUS_DEV_STRUCT, smbus_dev);
        smbus_dev->tx_state = HAL_SMBUS_STATE_READY;
        smbus_dev->rx_state = HAL_SMBUS_STATE_READY;
        smbus_dev->error_state = HAL_SMBUS_ERROR_NONE;
    break;
    case I2C1:
        hal_rcu_periph_reset_enable(RCU_I2C2RST);
        hal_rcu_periph_reset_disable(RCU_I2C2RST);
        hal_smbus_struct_init(HAL_SMBUS_DEV_STRUCT, smbus_dev);
        smbus_dev->tx_state = HAL_SMBUS_STATE_READY;
        smbus_dev->rx_state = HAL_SMBUS_STATE_READY;
        smbus_dev->error_state = HAL_SMBUS_ERROR_NONE;
    break;
    case I2C2:
        hal_rcu_periph_reset_enable(RCU_I2C2RST);
        hal_rcu_periph_reset_disable(RCU_I2C2RST);
        hal_smbus_struct_init(HAL_SMBUS_DEV_STRUCT, smbus_dev);
        smbus_dev->tx_state = HAL_SMBUS_STATE_READY;
        smbus_dev->rx_state = HAL_SMBUS_STATE_READY;
        smbus_dev->error_state = HAL_SMBUS_ERROR_NONE;
    break;
    case I2C3:
        hal_rcu_periph_reset_enable(RCU_I2C3RST);
        hal_rcu_periph_reset_disable(RCU_I2C3RST);
        hal_smbus_struct_init(HAL_SMBUS_DEV_STRUCT, smbus_dev);
        smbus_dev->tx_state = HAL_SMBUS_STATE_READY;
        smbus_dev->rx_state = HAL_SMBUS_STATE_READY;
        smbus_dev->error_state = HAL_SMBUS_ERROR_NONE;
    break;
    default:
        HAL_DEBUGE("parameter [smbus_dev->periph] value is invalid");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      SMBUS interrupt handler content function,which is merely used in i2c_event_handler
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_event_irq(hal_smbus_dev_struct *smbus_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smbus_dev) {
        HAL_DEBUGE("pointer [smbus_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(NULL != smbus_dev->smbus_irq.event_handle) {
        smbus_dev->smbus_irq.event_handle(smbus_dev);
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      SMBUS interrupt handler content function,which is merely used in i2c_error_handler
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_error_irq(hal_smbus_dev_struct *smbus_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smbus_dev) {
        HAL_DEBUGE("pointer [smbus_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* PEC error */
    if(hals_i2c_interrupt_flag_get(smbus_dev->periph, I2C_INT_FLAG_PECERR)) {
        hals_i2c_interrupt_flag_clear(smbus_dev->periph, I2C_INT_FLAG_PECERR);
        smbus_dev->error_state |= (uint32_t)HAL_SMBUS_ERROR_PECERR;
        smbus_dev->last_error = (uint16_t)HAL_SMBUS_ERROR_PECERR;
    } else {
        /* do nothing */
    }

    /* over-run or under-run when SCL stretch is disabled */
    if(hals_i2c_interrupt_flag_get(smbus_dev->periph, I2C_INT_FLAG_OUERR)) {
        hals_i2c_interrupt_flag_clear(smbus_dev->periph, I2C_INT_FLAG_OUERR);
        smbus_dev->error_state |= (uint32_t)HAL_I2C_ERROR_OUERR;
        smbus_dev->last_error = (uint16_t)HAL_I2C_ERROR_OUERR;
    } else {
        /* do nothing */
    }

    /* arbitration lost */
    if(hals_i2c_interrupt_flag_get(smbus_dev->periph, I2C_INT_FLAG_LOSTARB)) {
        hals_i2c_interrupt_flag_clear(smbus_dev->periph, I2C_INT_FLAG_LOSTARB);
        smbus_dev->error_state |= (uint32_t)HAL_I2C_ERROR_LOSTARB;
        smbus_dev->last_error = (uint16_t)HAL_I2C_ERROR_LOSTARB;
    } else {
        /* do nothing */
    }

    /* bus error */
    if(hals_i2c_interrupt_flag_get(smbus_dev->periph, I2C_INT_FLAG_BERR)) {
        hals_i2c_interrupt_flag_clear(smbus_dev->periph, I2C_INT_FLAG_BERR);
        smbus_dev->error_state |= (uint32_t)HAL_I2C_ERROR_BERR;
        smbus_dev->last_error = (uint16_t)HAL_I2C_ERROR_BERR;
    } else {
        /* do nothing */
    }

    /* CRC value doesn't match */
    if(hals_i2c_interrupt_flag_get(smbus_dev->periph, I2C_INT_FLAG_PECERR)) {
        hals_i2c_interrupt_flag_clear(smbus_dev->periph, I2C_INT_FLAG_PECERR);
        smbus_dev->error_state |= (uint32_t)HAL_I2C_ERROR_PECERR;
        smbus_dev->last_error = (uint16_t)HAL_I2C_ERROR_PECERR;
    } else {
        /* do nothing */
    }

    /* BERR error */
    if(hals_i2c_interrupt_flag_get(smbus_dev->periph, I2C_INT_FLAG_BERR)) {
        hals_i2c_interrupt_flag_clear(smbus_dev->periph, I2C_INT_FLAG_BERR);
        smbus_dev->error_state |= (uint32_t)HAL_SMBUS_ERROR_BERR;
        smbus_dev->last_error = (uint16_t)HAL_SMBUS_ERROR_BERR;
    } else {
        /* do nothing */
    }

    /* LOSTARB error */
    if(hals_i2c_interrupt_flag_get(smbus_dev->periph, I2C_INT_FLAG_LOSTARB)) {
        hals_i2c_interrupt_flag_clear(smbus_dev->periph, I2C_INT_FLAG_LOSTARB);
        smbus_dev->error_state |= (uint32_t)HAL_SMBUS_ERROR_LOSTARB;
        smbus_dev->last_error = (uint16_t)HAL_SMBUS_ERROR_LOSTARB;
    } else {
        /* do nothing */
    }

    /* OUERR error */
    if(hals_i2c_interrupt_flag_get(smbus_dev->periph, I2C_INT_FLAG_OUERR)) {
        hals_i2c_interrupt_flag_clear(smbus_dev->periph, I2C_INT_FLAG_OUERR);
        smbus_dev->error_state |= (uint32_t)HAL_SMBUS_ERROR_OUERR;
        smbus_dev->last_error = (uint16_t)HAL_SMBUS_ERROR_OUERR;
    } else {
        /* do nothing */
    }

    /* SMBUS alert */
    if(hals_i2c_interrupt_flag_get(smbus_dev->periph, I2C_INT_FLAG_SMBALT)) {
        hals_i2c_interrupt_flag_clear(smbus_dev->periph, I2C_INT_FLAG_SMBALT);
        smbus_dev->error_state |= (uint32_t)HAL_SMBUS_ERROR_SMBALT;
        smbus_dev->last_error = (uint16_t)HAL_SMBUS_ERROR_SMBALT;
    } else {
        /* do nothing */
    }

    /* SMBUS no error */
    if(smbus_dev->error_state != HAL_SMBUS_ERROR_NONE) {
        if(NULL != smbus_dev->smbus_irq.error_handle) {
            smbus_dev->smbus_irq.error_handle(smbus_dev);
            smbus_dev->error_state = HAL_SMBUS_ERROR_NONE;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to SMBUS interrupt callback functions structure
                  The structure member can be assigned as following
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                  the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and
                  disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_irq_handle_set(hal_smbus_dev_struct *smbus_dev, hal_smbus_irq_struct *p_irq)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == smbus_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [smbus_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* event interrupt handler set */
    if(NULL != p_irq->event_handle) {
        smbus_dev->smbus_irq.event_handle = p_irq->event_handle;
        hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_TX);
        hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_LISTEN);
    } else {
        smbus_dev->smbus_irq.event_handle = NULL;
        hals_i2c_interrupt_disable(smbus_dev->periph, I2C_INT_TX);
        hals_i2c_interrupt_disable(smbus_dev->periph, I2C_INT_LISTEN);
    }

    /* error interrupt handler set */
    if(NULL != p_irq->error_handle) {
        smbus_dev->smbus_irq.error_handle = p_irq->error_handle;
        hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_ERR);
    } else {
        smbus_dev->smbus_irq.error_handle = NULL;
        hals_i2c_interrupt_disable(smbus_dev->periph, I2C_INT_ERR);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_irq_handle_all_reset(hal_smbus_dev_struct *smbus_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smbus_dev) {
        HAL_DEBUGE("pointer [smbus_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    smbus_dev->smbus_irq.event_handle = NULL;
    smbus_dev->smbus_irq.error_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      config the SMBUS alert mode in master mode
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_smbus_alert_master_config(uint32_t smbus_periph)
{
    hals_smbus_alert_master_device_config(smbus_periph);

    /* enable SMBus alert */
    hals_smbus_alert_enable(smbus_periph);

    /* clear ALERT flag */
    hals_i2c_flag_clear(smbus_periph, I2C_FLAG_SMBALT);

    /* enable Alert Interrupt */
    hals_i2c_interrupt_enable(smbus_periph, I2C_INT_ERR);
}

/*!
    \brief      config the SMBUS alert mode in slave mode
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_smbus_alert_slave_config(uint32_t smbus_periph)
{
    hals_smbus_alert_slave_device_config(smbus_periph);

    /* enable SMBus alert */
    hals_smbus_alert_disable(smbus_periph);

    /* disable Alert Interrupt */
    hals_i2c_interrupt_disable(smbus_periph, I2C_INT_ERR);
}

/*!
    \brief      SMBUS master transmit in interrupt mode
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  slave_address: 0-0x3FF except reserved address, SMBUS slave address to be sent
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_master_transmit_interrupt(hal_smbus_dev_struct *smbus_dev, uint16_t slave_address, \
                                            uint8_t *p_buffer, uint32_t length, \
                                            hal_smbus_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == smbus_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [smbus_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smbus_dev);

    /* check the tx_state whether it is busy or not */
    if(HAL_SMBUS_STATE_BUSY == smbus_dev->tx_state) {
        HAL_DEBUGE("i2c tx has already been used, please wait until run_state change to free");
        smbus_dev->error_state = HAL_SMBUS_ERROR_BERR;
        ret = HAL_ERR_BUSY;
    } else {
        /* wait until I2C bus is idle */
        if(HAL_ERR_NONE != _smbus_wait_flag_timeout(smbus_dev->periph, I2C_FLAG_I2CBUSY, RESET, SMBUS_BUSY_TIMEOUT)) {
            HAL_DEBUGE("i2c busy timeout");
            smbus_dev->error_state = HAL_SMBUS_ERROR_SMBTO;
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* initialize transmit parameters */
            smbus_dev->tx_state   = HAL_SMBUS_STATE_BUSY;
            smbus_dev->last_error = (uint16_t)HAL_I2C_ERROR_NONE;

            smbus_dev->txbuffer.buffer = (uint8_t *)p_buffer;
            smbus_dev->txbuffer.length = length;
            smbus_dev->txbuffer.pos    = 0U;

            smbus_dev->slave_address.device_address = slave_address;
            smbus_dev->smbus_irq.event_handle       = _smbus_master_transfer_interrupt;
            smbus_dev->rx_callback                  = NULL;


            if(NULL != p_user_func) {
              if(NULL != p_user_func->tx_callback) {
                  smbus_dev->rx_callback = (void *)p_user_func->tx_callback;
              } else {
                   /* do nothing */
              }
            } else {
                /* do nothing */
            }

            /* configuration slave address */
            if(HAL_ERR_NONE != hals_i2c_master_addressing(smbus_dev->periph, (uint32_t)smbus_dev->slave_address.device_address)) {
                HAL_DEBUGE("SMBUS slave address is invalid value");
                ret = HAL_ERR_VAL;
            } else {
                /* configuration nbyte and data length */
                if(SMBUS_MAX_BYTE_SIZE <= smbus_dev->txbuffer.length) {
                    smbus_dev->txbuffer.pos = SMBUS_MAX_BYTE_SIZE;
                    _smbus_master_transmit_config(smbus_dev->periph, smbus_dev->txbuffer.pos, \
                                                  I2C_RELOAD_MODE, I2C_GENERATE_START_WRITE);
                } else {
                    smbus_dev->txbuffer.pos = smbus_dev->txbuffer.length;
                    if(SET == ((I2C_CTL1(smbus_dev->periph) & I2C_CTL1_PECTRANS) >> 26)) {
                        _smbus_master_transmit_config(smbus_dev->periph, smbus_dev->txbuffer.pos + 1U, \
                                                      I2C_AUTOEND_MODE, I2C_GENERATE_START_WRITE);
                    } else {
                        _smbus_master_transmit_config(smbus_dev->periph, smbus_dev->txbuffer.pos, \
                                                      I2C_AUTOEND_MODE, I2C_GENERATE_START_WRITE);
                    }
                }

                /* enable the I2C interrupt */
                hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_TX);
            }
        }
    }

    HAL_UNLOCK(smbus_dev);
    return ret;
}

/*!
    \brief      SMBUS master receive with interrupt
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be read
    \param[in]  slave_address: 0-0x3FF except reserved address, SMBUS slave address to be sent
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_master_receive_interrupt(hal_smbus_dev_struct *smbus_dev, uint16_t slave_address, \
                                           uint8_t *p_buffer, uint32_t length, \
                                           hal_smbus_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == smbus_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [smbus_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smbus_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_SMBUS_STATE_BUSY == smbus_dev->tx_state) {
        HAL_DEBUGE("i2c tx has already been used, please wait until run_state change to free ");
        smbus_dev->error_state = HAL_SMBUS_ERROR_BERR;
        ret = HAL_ERR_BUSY;
    } else {
        /* wait until I2C bus is idle */
        if(HAL_ERR_NONE != _smbus_wait_flag_timeout(smbus_dev->periph, I2C_FLAG_I2CBUSY, RESET, SMBUS_BUSY_TIMEOUT)) {
            HAL_DEBUGE("i2c busy timeout");
            smbus_dev->error_state = HAL_SMBUS_ERROR_SMBTO;
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* update the values of SMBUS struct */
            smbus_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
            smbus_dev->rxbuffer.length = length;
            smbus_dev->rxbuffer.pos    = 0U;

            smbus_dev->rx_callback = NULL;
            smbus_dev->slave_address.device_address = slave_address;

            if(NULL != p_user_func) {
              if(NULL != p_user_func->rx_callback) {
                  smbus_dev->rx_callback = (void *)p_user_func->rx_callback;
              } else {
                   /* do nothing */
              }
            } else {
                /* do nothing */
            }

            /* update the values of SMBUS struct */
            smbus_dev->rx_state               = HAL_SMBUS_STATE_BUSY;
            smbus_dev->smbus_irq.event_handle = _smbus_master_received_interrupt;

            /* configuration slave address */
            if(HAL_ERR_NONE != hals_i2c_master_addressing(smbus_dev->periph, (uint32_t)smbus_dev->slave_address.device_address)) {
                HAL_DEBUGE("i2c slave address is invalid value");
                smbus_dev->error_state = HAL_SMBUS_ERROR_ADDRESS;
                ret = HAL_ERR_VAL;
            } else {
                /* configuration nbyte and data length */
                if(SMBUS_MAX_BYTE_SIZE <= smbus_dev->rxbuffer.length) {
                    smbus_dev->rxbuffer.pos = SMBUS_MAX_BYTE_SIZE;
                    _smbus_master_transmit_config(smbus_dev->periph, smbus_dev->rxbuffer.pos, \
                                                  I2C_RELOAD_MODE, I2C_GENERATE_START_READ);
                } else {
                    smbus_dev->rxbuffer.pos = smbus_dev->rxbuffer.length;
                    _smbus_master_transmit_config(smbus_dev->periph, smbus_dev->rxbuffer.pos, \
                                                  I2C_AUTOEND_MODE, I2C_GENERATE_START_READ);
                }

                /* enable the I2C interrupt */
                hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_RX);
            }
        }
    }

    HAL_UNLOCK(smbus_dev);

    return ret;
}

/*!
    \brief      SMBUS slave transmit in interrupt mode
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_slave_transmit_interrupt(hal_smbus_dev_struct *smbus_dev, uint8_t *p_buffer, \
                                           uint32_t length, hal_smbus_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == smbus_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [smbus_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smbus_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_SMBUS_STATE_BUSY == smbus_dev->tx_state) {
        HAL_DEBUGE("i2c tx has already been used, please wait until run_state change to free ");
        smbus_dev->error_state = HAL_SMBUS_ERROR_BERR;
        ret = HAL_ERR_BUSY;
    } else {
        /* wait until I2C bus is idle */
        if(HAL_ERR_NONE != _smbus_wait_flag_timeout(smbus_dev->periph, I2C_FLAG_I2CBUSY, RESET, SMBUS_BUSY_TIMEOUT)) {
            HAL_DEBUGE("i2c busy timeout");
            smbus_dev->error_state = HAL_SMBUS_ERROR_SMBTO;
            ret = HAL_ERR_TIMEOUT;
        } else {
            if(SET == ((I2C_CTL1(smbus_dev->periph) & I2C_CTL1_PECTRANS) >> 26)) {
                I2C_CTL0(smbus_dev->periph) |= I2C_CTL0_SBCTL;
            } else {
                /* do nothing */
            }

            /* initialize transmit parameters */
            smbus_dev->tx_state   = HAL_SMBUS_STATE_BUSY;
            smbus_dev->last_error = (uint16_t)HAL_I2C_ERROR_NONE;

            smbus_dev->txbuffer.buffer = (uint8_t *)p_buffer;
            smbus_dev->txbuffer.length = length;
            smbus_dev->txbuffer.pos    = 0U;

            smbus_dev->smbus_irq.event_handle = _smbus_slave_transmit_interrupt;
            smbus_dev->tx_callback            = NULL;
            smbus_dev->addr_callback          = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->tx_callback) {
                    smbus_dev->tx_callback = (void *)p_user_func->tx_callback;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->addr_callback) {
                    smbus_dev->addr_callback = (void *)p_user_func->addr_callback;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            if(I2C_CTL0(smbus_dev->periph) & I2C_CTL0_SS) {
                /* transmit data form TDATA register */
                hals_i2c_data_transmit(smbus_dev->periph, (uint32_t)*smbus_dev->txbuffer.buffer);
                smbus_dev->txbuffer.buffer++;
                smbus_dev->txbuffer.length--;
                smbus_dev->txbuffer.pos--;
            } else {
                /* do nothing */
            }

            /* enable the I2C interrupt */
            hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_TX);
            hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_LISTEN);
        }
    }

    HAL_UNLOCK(smbus_dev);

    return ret;
}

/*!
    \brief      SMBUS slave receive with interrupt mode
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be read
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_slave_receive_interrupt(hal_smbus_dev_struct *smbus_dev, \
                                          uint8_t *p_buffer, uint32_t length, \
                                          hal_smbus_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == smbus_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [smbus_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smbus_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_SMBUS_STATE_BUSY == smbus_dev->rx_state) {
        HAL_DEBUGE("i2c tx has already been used, please wait until run_state change to free ");
        smbus_dev->error_state = HAL_SMBUS_ERROR_BERR;
        ret = HAL_ERR_BUSY;
    } else {
        /* wait until I2C bus is idle */
        if(HAL_ERR_NONE != _smbus_wait_flag_timeout(smbus_dev->periph, I2C_FLAG_I2CBUSY, RESET, SMBUS_BUSY_TIMEOUT)) {
            HAL_DEBUGE("i2c busy timeout");
            smbus_dev->error_state = HAL_SMBUS_ERROR_SMBTO;
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }

        if(SET == ((I2C_CTL1(smbus_dev->periph) & I2C_CTL1_PECTRANS) >> 26)) {
            I2C_CTL0(smbus_dev->periph) |= I2C_CTL0_SBCTL;
            I2C_CTL0(smbus_dev->periph) &= ~I2C_CTL1_RELOAD;
        } else {
            /* do nothing */
        }

        /* update the values of SMBUS struct */
        smbus_dev->rx_state = HAL_SMBUS_STATE_BUSY;
        smbus_dev->last_error = (uint16_t)HAL_I2C_ERROR_NONE;

        smbus_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
        smbus_dev->rxbuffer.length = length;
        smbus_dev->rxbuffer.pos = length;

        smbus_dev->smbus_irq.event_handle = _smbus_slave_received_interrupt;

        smbus_dev->rx_callback = NULL;
        smbus_dev->addr_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->rx_callback) {
                smbus_dev->rx_callback = (void *)p_user_func->rx_callback;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->addr_callback) {
                smbus_dev->addr_callback = (void *)p_user_func->addr_callback;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable the I2C interrupt */
        hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_RX);
        hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_LISTEN);
    }

    HAL_UNLOCK(smbus_dev);

    return ret;
}

/*!
    \brief      abort a master/host SMBUS process communication in interrupt mode.
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, SMBUS slave address to be sent
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_master_abort_interrupt(hal_smbus_dev_struct *smbus_dev, uint16_t slave_address)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == smbus_dev)) {
        HAL_DEBUGE("pointer [smbus_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smbus_dev);

    if((I2C_CTL1(smbus_dev->periph) & I2C_CTL1_TRDIR)) {
        /* disable interrupt and store previous state */
        if((HAL_SMBUS_STATE_BUSY == smbus_dev->tx_state)) {
            hals_i2c_interrupt_disable(smbus_dev->periph, I2C_INT_TX);
        } else if((HAL_SMBUS_STATE_BUSY == smbus_dev->rx_state)) {
            hals_i2c_interrupt_disable(smbus_dev->periph, I2C_INT_RX);
            hals_i2c_interrupt_disable(smbus_dev->periph, I2C_INT_LISTEN);
        } else {
            /* do nothing */
        }

        smbus_dev->tx_state = HAL_SMBUS_STATE_READY;

        /* configuration slave address */
        hals_i2c_master_addressing(smbus_dev->periph, (uint32_t)slave_address);
        /* set nbyte to 1 to generate a dummy read on I2C peripheral
           set AUTOEND mode, this will generate a NACK then STOP condition to abort the current transfer */
        _smbus_master_transmit_config(smbus_dev->periph, 1U, I2C_AUTOEND_MODE, I2C_GENERATE_STOP);

        /* i2c interrupts must be enabled to avoid the risk of i2c interrupt handle execution */
        hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_STPDETIE);
        hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_TCIE);
    } else {
        HAL_DEBUGE("i2c is not in master mode");
        smbus_dev->error_state = HAL_SMBUS_ERROR_LOSTARB;
        ret = HAL_ERR_ABORT;
    }

    HAL_UNLOCK(smbus_dev);

    return ret;
}

/*!
    \brief      check whether the device is ready for access
    \param[in]  smbus_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_device_ready_check(hal_smbus_dev_struct *smbus_dev)
{
    int32_t ret = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == smbus_dev)) {
        HAL_DEBUGE("pointer [SMBUS] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smbus_dev);

    while(1) {
        if(RESET == (I2C_CTL1(smbus_dev->periph) & I2C_CTL1_RELOAD)) {
            /* wait until I2C bus is idle */
            while(hals_i2c_flag_get(smbus_dev->periph, I2C_FLAG_I2CBUSY));
            if(HAL_ERR_NONE != _smbus_wait_flag_timeout(smbus_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
                HAL_DEBUGE("i2c busy timeout");
                smbus_dev->error_state = (uint16_t)HAL_I2C_ERROR_BERR;
                ret = HAL_ERR_TIMEOUT;
            } else {
                /* check if I2C is already enabled */
                if(I2C_CTL0_I2CEN != (I2C_CTL0(smbus_dev->periph) & I2C_CTL0_I2CEN)) {
                    HAL_I2C_ENABLE(smbus_dev->periph);
                } else {
                    /* do nothing */
                }

                if((HAL_SMBUS_STATE_READY != smbus_dev->tx_state) || (HAL_SMBUS_STATE_READY != smbus_dev->rx_state)) {
                    smbus_dev->error_state = (uint16_t)HAL_I2C_ERROR_BERR;
                    ret = HAL_ERR_BUSY;
                } else {
                    /* do nothing */
                }
            }
        } else {
            if(HAL_ERR_NONE != _smbus_wait_flag_timeout(smbus_dev->periph, I2C_FLAG_TCR, SET, I2C_BUSY_TIMEOUT)) {
                HAL_DEBUGE("i2c busy timeout");
                smbus_dev->error_state = (uint16_t)HAL_I2C_ERROR_BERR;
                ret = HAL_ERR_TIMEOUT;
            } else {
                /* do nothing */
            }
        }

        ret = HAL_ERR_NONE;
        break;
    }

    HAL_UNLOCK(smbus_dev);

    return ret;
}

/*!
    \brief      enable the address listen mode with Interrupt
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_address_listen_interrupt_enable(hal_smbus_dev_struct *smbus_dev, \
                                                  hal_smbus_irq_user_callback_struct *p_user_func)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smbus_dev) {
        HAL_DEBUGE("pointer [smbus_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(0U != smbus_dev->txbuffer.length) {
        smbus_dev->tx_state = HAL_SMBUS_STATE_LISTEN;
    } else if(0U != smbus_dev->rxbuffer.length) {
        smbus_dev->rx_state = HAL_SMBUS_STATE_LISTEN;
    } else {
        /* do nothing */
    }

    smbus_dev->rx_callback = NULL;

    if(NULL != p_user_func) {
        if(NULL != p_user_func->rx_callback) {
            smbus_dev->rx_callback = (void *)p_user_func->rx_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    smbus_dev->smbus_irq.event_handle = _smbus_address_listen_interrupt;

    /* enable the address match interrupt */
    hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_ADDMIE);
    hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_NACKIE);
    hals_i2c_interrupt_enable(smbus_dev->periph, I2C_INT_STPDETIE);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable the address listen mode with Interrupt
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_smbus_address_listen_interrupt_disable(hal_smbus_dev_struct *smbus_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smbus_dev) {
        HAL_DEBUGE("pointer [smbus_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(0U != smbus_dev->txbuffer.length) {
        smbus_dev->tx_state = HAL_SMBUS_STATE_READY;
    } else if(0U != smbus_dev->rxbuffer.length) {
        smbus_dev->rx_state = HAL_SMBUS_STATE_READY;
    } else {
        /* do nothing */
    }

    /* disable the address match interrupt */
    hals_i2c_interrupt_disable(smbus_dev->periph, I2C_INT_ADDMIE);
    hals_i2c_interrupt_disable(smbus_dev->periph, I2C_INT_NACKIE);
    hals_i2c_interrupt_disable(smbus_dev->periph, I2C_INT_STPDETIE);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure digital noise filter
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[in]  filter_length: the length of filter spikes
                only one parameter can be selected which is shown as below:
      \arg        DIGITAL_FILTER_DISABLE: digital filter is disabled
      \arg        DIGITAL_FILTER_LESSTHAN_1CLK: filter out spikes with pulse width less than 1 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_2CLK: filter out spikes with pulse width less than 2 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_3CLK: filter out spikes with pulse width less than 3 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_4CLK: filter out spikes with pulse width less than 4 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_5CLK: filter out spikes with pulse width less than 5 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_6CLK: filter out spikes with pulse width less than 6 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_7CLK: filter out spikes with pulse width less than 7 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_8CLK: filter out spikes with pulse width less than 8 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_9CLK: filter out spikes with pulse width less than 9 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_10CLK: filter out spikes with pulse width less than 10 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_11CLK: filter out spikes with pulse width less than 11 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_12CLK: filter out spikes with pulse width less than 12 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_13CLK: filter out spikes with pulse width less than 13 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_14CLK: filter out spikes with pulse width less than 14 tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_15CLK: filter out spikes with pulse width less than 15 tI2CCLK
    \param[out] none
    \retval     none
*/
void hal_smbus_digital_noise_filter_config(uint32_t smbus_periph, hal_i2c_digital_filter_enum filter_length)
{
    I2C_CTL0(smbus_periph) &= (uint32_t)(~I2C_CTL0_DNF);
    I2C_CTL0(smbus_periph) |= (uint32_t)((uint32_t)filter_length << CTL0_DNF_OFFSET);
}

/*!
    \brief      configure analog noise filter
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[in]  state:  enable or disable the analog noise filter.
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the analog noise filter
      \arg        DISABLE: disable the analog noise filter
    \param[out] none
    \retval     none
*/
void hal_smbus_analog_noise_filter_config(uint32_t smbus_periph, ControlStatus state)
{
    if(ENABLE == state) {
        I2C_CTL0(smbus_periph) &= ~I2C_CTL0_ANOFF;
    } else {
        I2C_CTL0(smbus_periph) |= I2C_CTL0_ANOFF;
    }
}

/*!
    \brief      return the smbus tx state
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_smbus_run_state_enum
*/
hal_smbus_run_state_enum hal_smbus_tx_state_get(hal_smbus_dev_struct *smbus_dev)
{
    return (hal_smbus_run_state_enum)smbus_dev->tx_state;
}

/*!
    \brief      return the smbus rx state
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_smbus_run_state_enum
*/
hal_smbus_run_state_enum hal_smbus_rx_state_get(hal_smbus_dev_struct *smbus_dev)
{
    return (hal_smbus_run_state_enum)smbus_dev->rx_state;
}

/*!
    \brief      return the smbus error code
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: 0 - 0xFFFFFFFFU
*/
uint32_t hal_smbus_error_code_get(hal_smbus_dev_struct *smbus_dev)
{
    return smbus_dev->error_state;
}

/*!
    \brief      select SMBus type
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[in]  type: select SMBus type.
                only one parameter can be selected which is shown as below:
      \arg        SMBUS_HOST: SMBus mode device type
      \arg        SMBUS_SLAVE: SMBus mode host type
      \arg        SMBUS_SLAVE_ARP:SMBus mode slave arp type
    \param[out] none
    \retval     none
*/
void hals_smbus_type_config(uint32_t smbus_periph, uint32_t type)
{
    if(SMBUS_HOST == type) {
        hals_smbus_alert_master_device_config(smbus_periph);
    } else if(SMBUS_SLAVE == type) {
        I2C_CTL0(smbus_periph) |= (I2C_CTL0_I2CEN);
    } else {
        hals_smbus_default_addr_enable(smbus_periph);
    }
}

/*!
    \brief      enable SMBus device default address
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_smbus_default_addr_enable(uint32_t smbus_periph)
{
    I2C_CTL0(smbus_periph) |= I2C_CTL0_SMBDAEN;
}

/*!
    \brief      disable SMBus device default address
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_smbus_default_addr_disable(uint32_t smbus_periph)
{
    I2C_CTL0(smbus_periph) &= ~I2C_CTL0_SMBDAEN;
}

/*!
    \brief      enable SMBus host address
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_smbus_alert_master_device_config(uint32_t smbus_periph)
{
    I2C_CTL0(smbus_periph) |= I2C_CTL0_SMBHAEN;
}

/*!
    \brief      disable SMBus host address
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_smbus_alert_slave_device_config(uint32_t smbus_periph)
{
    I2C_CTL0(smbus_periph) &= ~I2C_CTL0_SMBHAEN;
}

/*!
    \brief      enable SMBus alert
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_smbus_alert_enable(uint32_t smbus_periph)
{
    I2C_CTL0(smbus_periph) |= I2C_CTL0_SMBALTEN;
}

/*!
    \brief      disable SMBus alert
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_smbus_alert_disable(uint32_t smbus_periph)
{
    I2C_CTL0(smbus_periph) &= ~I2C_CTL0_SMBALTEN;
}

/*!
    \brief      configure I2C PEC calculation
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[in]  pec_state: only one parameter can be selected which is shown as below:
      \arg        I2C_PEC_ENABLE: PEC calculation on
      \arg        I2C_PEC_DISABLE: PEC calculation off
    \param[out] none
    \retval     none
*/
void hals_i2c_smbus_pec_config(uint32_t smbus_periph, uint32_t pec_state)
{
    uint32_t ctl = 0U;

    ctl = I2C_CTL0(smbus_periph);
    ctl &= ~(I2C_CTL0_PECEN);
    ctl |= pec_state;

    I2C_CTL0(smbus_periph) = ctl;
}

/*!
    \brief enable the SMBUS fast mode plus driving capability
    \param[in] config_fastmode: Selects the pin.
               only one parameter can be selected which is shown as below:
      \arg       SMBUS_FMP_NOT_SUPPORTED: Fast Mode Plus not supported
      \arg       SMBUS_FASTMODEPLUS_PB6 : enable Fast Mode Plus on PB6
      \arg       SMBUS_FASTMODEPLUS_PB7 : enable Fast Mode Plus on PB7
      \arg       SMBUS_FASTMODEPLUS_PB8 : enable Fast Mode Plus on PB8
      \arg       SMBUS_FASTMODEPLUS_PB9 : enable Fast Mode Plus on PB9
      \arg       SMBUS_FASTMODEPLUS_I2C1: enable Fast Mode Plus on I2C1 pins
      \arg       SMBUS_FASTMODEPLUS_I2C2: enable Fast Mode Plus on I2C2 pins
      \arg       SMBUS_FASTMODEPLUS_I2C3: enable Fast Mode Plus on I2C3 pins
      \arg       SMBUS_FASTMODEPLUS_I2C4: enable Fast Mode Plus on I2C4 pins
    \param[out] none
    \retval     none
*/
void hals_smbus_enable_fastplus_mode(uint32_t config_fastmode)
{
    uint32_t reg0 = 0U;

    /* enable SYSCFG clock */
    if(RESET == (RCU_APB4EN & RCU_APB4EN_SYSCFGEN)) {
        hal_rcu_periph_clock_enable(RCU_SYSCFG);
    } else {
        /* do nothing */
    }

    /* enable fast mode plus driving capability for selected pin */
    reg0 = SYSCFG_PMCFG;
    reg0 |= (uint32_t)config_fastmode;

    SYSCFG_PMCFG = reg0;
}

/*!
    \brief disable the SMBUS fast mode plus driving capability
    \param[in] config_fastmode: Selects the pin.
               only one parameter can be selected which is shown as below:
      \arg       SMBUS_FMP_NOT_SUPPORTED: Fast Mode Plus not supported
      \arg       SMBUS_FASTMODEPLUS_PB6 : enable Fast Mode Plus on PB6
      \arg       SMBUS_FASTMODEPLUS_PB7 : enable Fast Mode Plus on PB7
      \arg       SMBUS_FASTMODEPLUS_PB8 : enable Fast Mode Plus on PB8
      \arg       SMBUS_FASTMODEPLUS_PB9 : enable Fast Mode Plus on PB9
      \arg       SMBUS_FASTMODEPLUS_I2C1: enable Fast Mode Plus on I2C1 pins
      \arg       SMBUS_FASTMODEPLUS_I2C2: enable Fast Mode Plus on I2C2 pins
      \arg       SMBUS_FASTMODEPLUS_I2C3: enable Fast Mode Plus on I2C3 pins
      \arg       SMBUS_FASTMODEPLUS_I2C4: enable Fast Mode Plus on I2C4 pins
    \param[out] none
    \retval     none
*/
void hals_smbus_disable_fastplus_mode(uint32_t config_fastmode)
{
    uint32_t reg0 = 0U;

    /* enable SYSCFG clock */
    if(RESET == (RCU_APB4EN & RCU_APB4EN_SYSCFGEN)) {
        hal_rcu_periph_clock_enable(RCU_SYSCFG);
    } else {
        /* do nothing */
    }

    /* enable fast mode plus driving capability for selected pin */
    reg0 = SYSCFG_PMCFG;
    reg0 &= ~(uint32_t)config_fastmode;

    SYSCFG_PMCFG = reg0;
}

/*!
    \brief      enable extended clock timeout detection
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_extented_clock_timeout_enable(uint32_t smbus_periph)
{
    I2C_TIMEOUT(smbus_periph) |= I2C_TIMEOUT_EXTOEN;
}

/*!
    \brief      disable extended clock timeout detection
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_extented_clock_timeout_disable(uint32_t smbus_periph)
{
    I2C_TIMEOUT(smbus_periph) &= ~I2C_TIMEOUT_EXTOEN;
}

/*!
    \brief      enable clock timeout detection
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_clock_timeout_enable(uint32_t smbus_periph)
{
    I2C_TIMEOUT(smbus_periph) |= I2C_TIMEOUT_TOEN;
}

/*!
    \brief      disable clock timeout detection
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_clock_timeout_disable(uint32_t smbus_periph)
{
    I2C_TIMEOUT(smbus_periph) &= ~I2C_TIMEOUT_TOEN;
}

/*!
    \brief      configure bus timeout B
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[in]  timeout: bus timeout B
    \param[out] none
    \retval     none
*/
void hals_i2c_bus_timeout_b_config(uint32_t smbus_periph, uint32_t timeout)
{
    I2C_TIMEOUT(smbus_periph) &= ~I2C_TIMEOUT_BUSTOB;
    I2C_TIMEOUT(smbus_periph) |= (uint32_t)(timeout << TIMEOUT_BUSTOB_OFFSET);
}

/*!
    \brief      configure bus timeout A
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[in]  timeout: bus timeout A
    \param[out] none
    \retval     none
*/
void hals_i2c_bus_timeout_a_config(uint32_t smbus_periph, uint32_t timeout)
{
    I2C_TIMEOUT(smbus_periph) &= ~I2C_TIMEOUT_BUSTOA;
    I2C_TIMEOUT(smbus_periph) |= timeout;
}

/*!
    \brief      configure idle clock timeout detection
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[in]  timeout_detection: enable or disable the analog noise filter.
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the analog noise filter
      \arg        DISABLE: disable the analog noise filter
    \param[out] none
    \retval     none
*/
void hals_i2c_idle_clock_timeout_config(uint32_t smbus_periph, ControlStatus timeout_detection)
{
    I2C_TIMEOUT(smbus_periph) &= ~I2C_TIMEOUT_TOIDLE;
    I2C_TIMEOUT(smbus_periph) |= timeout_detection;
}

/*!
    \brief      wait the flag status until timeout
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[in]  flag: I2C flags, refer to hal_i2c_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        I2C_FLAG_TBE: I2C_TDATA is empty during transmitting
      \arg        I2C_FLAG_TI: I2C_TDATA is empty during transmitting and ready to transfer data
      \arg        I2C_FLAG_RBNE: I2C_RDATA is not empty during receiving
      \arg        I2C_FLAG_ADDSEND: address is sent in master mode or received and matches in slave mode
      \arg        I2C_FLAG_NACK: received nack request
      \arg        I2C_FLAG_STPDET: stop condition detected in slave mode
      \arg        I2C_FLAG_TC: transfer end in master mode
      \arg        I2C_FLAG_TCR: transfer reload complete
      \arg        I2C_FLAG_BERR: a bus error occurs indication a unexpected start or stop condition on I2C bus
      \arg        I2C_FLAG_LOSTARB: arbitration lost in master mode
      \arg        I2C_FLAG_OUERR: over-run or under-run situation occurs in slave mode
      \arg        I2C_FLAG_PECERR: PEC error when receiving data
      \arg        I2C_FLAG_TIMEOUT: timeout signal
      \arg        I2C_FLAG_SMBALT: SMBus alert status
      \arg        I2C_FLAG_I2CBUSY: busy flag
      \arg        I2C_FLAG_TR: whether the I2C is a transmitter or a receiver
    \param[in]  status: the status of I2C flag to wait
    \param[in]  timeout_ms: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _smbus_wait_flag_timeout(uint32_t smbus_periph, hal_i2c_flag_enum flag, \
                                        FlagStatus status, uint32_t timeout_ms)
{
    __IO uint32_t tick_start = 0U;
    int32_t  ret = HAL_ERR_NONE;

    tick_start = hal_sys_basetick_count_get();
    /* wait flag status RESET */
    if(RESET == status) {
        while(SET == hals_i2c_flag_get(smbus_periph, flag)) {
            if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                    HAL_DEBUGE("i2c get flag timeout");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    } else {
        /* wait flag status SET */
        while(RESET == hals_i2c_flag_get(smbus_periph, flag)) {
            if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                    HAL_DEBUGE("i2c get flag timeout");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    }

    return ret;
}

/*!
    \brief      master sends device address for write request
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[in]  length: the length of send data
    \param[in]  mode: smbus mode select,
                only one parameter can be selected which is shown as below:
      \arg        SMBUS_RELOAD_MODE: SMBUS reload mode
      \arg        SMBUS_AUTOEND_MODE: SMBUS automatic end mode
      \arg        SMBUS_SOFTEND_MODE: SMBUS software end mode
    \param[in]  request: SMBUS request select,
                only one parameter can be selected which is shown as below:
      \arg        SMBUS_NO_STARTSTOP: no start or stop generation
      \arg        SMBUS_GENERATE_STOP: generate stop condition after transfer
      \arg        SMBUS_GENERATE_START_READ: generate start and restart condition for read request
      \arg        SMBUS_GENERATE_START_WRITE: generate start and restart condition for write request
    \param[out] none
    \retval     none
*/
static void _smbus_master_transmit_config(uint32_t smbus_periph, uint32_t length, uint32_t mode, uint32_t request)
{
    hals_i2c_transfer_byte_number_config(smbus_periph, length);

    if(I2C_RELOAD_MODE == mode) {
        hals_i2c_reload_enable(smbus_periph);
    } else if(I2C_AUTOEND_MODE == mode) {
        hals_i2c_automatic_end_enable(smbus_periph);
    } else if(I2C_SOFTEND_MODE == mode) {
        hals_i2c_reload_disable(smbus_periph);
        hals_i2c_automatic_end_disable(smbus_periph);
    } else {
        /* do nothing */
    }

    switch(request) {
    case I2C_NO_STARTSTOP:
        I2C_CTL1(smbus_periph) &= ~(I2C_CTL1_START | I2C_CTL1_STOP);
        break;
    case I2C_GENERATE_STOP:
            hal_i2c_stop_on_bus(smbus_periph);
    break;
        case I2C_GENERATE_START_READ:
            hals_i2c_master_status(smbus_periph, I2C_MASTER_RECEIVE);
            hal_i2c_start_on_bus(smbus_periph);
        break;
    case I2C_GENERATE_START_WRITE:
            hals_i2c_master_status(I2C1, I2C_MASTER_TRANSMIT);
            hal_i2c_start_on_bus(smbus_periph);
        break;
    default:
        break;
    }
}

/*!
    \brief      I2C Tx data register flush process.
    \param[in]  smbus_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
static void _smbus_flush_tdata_register(uint32_t smbus_periph)
{
    /* Write a dummy data in TDATA to clear it */
    if(SET == hals_i2c_flag_get(smbus_periph, I2C_FLAG_TI)) {
        I2C_STAT(smbus_periph) |= I2C_STAT_TBE;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      event handler in SMBUS master transmit mode
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _smbus_master_transfer_interrupt(void *smbus_dev)
{
    hal_smbus_dev_struct *p_smbus = smbus_dev;
    hal_smbus_user_cb p_func = (hal_smbus_user_cb)p_smbus->tx_callback;

    if((SET ==hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_STPDET))) {
        p_smbus->tx_state = HAL_SMBUS_STATE_READY;

        /* clear STPDET Flag */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_STPDET);

        /* disable interrupt */
        hals_i2c_interrupt_disable(p_smbus->periph, I2C_INT_ALL);

        /* user configuration tx_callback*/
        if(NULL != p_func) {
            p_func(p_smbus);
        } else {
            /* do nothing */
        }
    } else if((SET ==hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_TI))) {
        /* transmit data from tdata */
        I2C_TDATA(p_smbus->periph) = *p_smbus->txbuffer.buffer;
        p_smbus->txbuffer.buffer++;
        p_smbus->txbuffer.pos--;
        p_smbus->txbuffer.length--;
    } else if((SET ==hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_TC))) {
        p_smbus->tx_state = HAL_SMBUS_STATE_READY;

        /* generator stop signal */
        hal_i2c_stop_on_bus(p_smbus->periph);

        /* user configuration tx_callback*/
        if(NULL != p_func) {
            p_func(p_smbus);
        } else {
            /* do nothing */
        }
    } else if((SET ==hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_TCR))) {
        /* transmit reload complete interrupt */
        if((SMBUS_DATA_SEND_FINISH_LENGTH != p_smbus->txbuffer.length) && \
           (SMBUS_DATA_SEND_FINISH_LENGTH == p_smbus->txbuffer.pos)) {
            if(SMBUS_MAX_BYTE_SIZE <= p_smbus->txbuffer.length) {
                 p_smbus->txbuffer.pos = SMBUS_MAX_BYTE_SIZE;
                _smbus_master_transmit_config(p_smbus->periph, p_smbus->txbuffer.pos, \
                                              I2C_RELOAD_MODE, I2C_NO_STARTSTOP);
            } else {
                p_smbus->txbuffer.pos = p_smbus->txbuffer.length;
                _smbus_master_transmit_config(p_smbus->periph, p_smbus->txbuffer.pos, \
                                              I2C_AUTOEND_MODE, I2C_NO_STARTSTOP);
            }
        } else {
            /* do nothing */
        }
    } else if((SET ==hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_NACK))) {
        if((SMBUS_DATA_SEND_FINISH_LENGTH != p_smbus->txbuffer.length)) {
            p_smbus->error_state = (uint16_t)HAL_I2C_ERROR_MASTER_TX;
        } else {
            /* do nothing */
        }

        /* clear nack Flag */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_NACK);

        /* clear tdata register */
        _smbus_flush_tdata_register(p_smbus->periph);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      event handler in SMBUS master received mode
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _smbus_master_received_interrupt(void *smbus_dev)
{
    hal_smbus_dev_struct *p_smbus = smbus_dev;
    hal_smbus_user_cb p_func = (hal_smbus_user_cb)p_smbus->rx_callback;

    if((SET ==hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_STPDET))) {
        p_smbus->rx_state = HAL_SMBUS_STATE_READY;

        /* clear STPDET Flag */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_STPDET);

        /* disable interrupt */
        hals_i2c_interrupt_disable(p_smbus->periph, I2C_INT_ALL);

        /* user configuration rx_callback*/
        if(NULL != p_func) {
            p_func(p_smbus);
        } else {
            /* do nothing */
        }
    } else if((SET ==hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_RBNE))) {
        /* received data */
        *p_smbus->rxbuffer.buffer = hals_i2c_data_receive(p_smbus->periph);
         p_smbus->rxbuffer.buffer++;
         p_smbus->rxbuffer.pos--;
         p_smbus->rxbuffer.length--;
    } else if((SET ==hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_TC))) {
        p_smbus->rx_state = HAL_SMBUS_STATE_READY;

        /* generator stop signal */
        hal_i2c_stop_on_bus(p_smbus->periph);

        /* user configuration rx_callback*/
        if(NULL != p_func) {
            p_func(p_smbus);
        } else {
            /* do nothing */
        }
    } else if((SET ==hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_TCR))) {
        /* received reload complete interrupt */
        if((SMBUS_DATA_RECEIVED_FINISH_LENGTH != p_smbus->rxbuffer.length) && \
           (SMBUS_DATA_RECEIVED_FINISH_LENGTH == p_smbus->rxbuffer.pos)) {
            if(SMBUS_MAX_BYTE_SIZE <= p_smbus->rxbuffer.length) {
                p_smbus->rxbuffer.pos = SMBUS_MAX_BYTE_SIZE;
                _smbus_master_transmit_config(p_smbus->periph, p_smbus->rxbuffer.pos, \
                                              I2C_RELOAD_MODE, I2C_NO_STARTSTOP);
            } else {
                p_smbus->rxbuffer.pos = p_smbus->rxbuffer.length;
                _smbus_master_transmit_config(p_smbus->periph, p_smbus->rxbuffer.pos, \
                                              I2C_AUTOEND_MODE, I2C_NO_STARTSTOP);
            }
        } else {
            /* do nothing */
        }
    } else if((SET == hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_NACK))) {
        if((SMBUS_DATA_RECEIVED_FINISH_LENGTH != p_smbus->rxbuffer.length)) {
            p_smbus->error_state = (uint16_t)HAL_I2C_ERROR_MASTER_RX;
        } else {
            /* do nothing */
        }

        /* clear nack Flag */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_NACK);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      event handler in SMBUS slave transmit mode
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _smbus_slave_transmit_interrupt(void *smbus_dev)
{
    hal_smbus_dev_struct *p_smbus = smbus_dev;
    hal_smbus_user_cb p_func = (hal_smbus_user_cb)p_smbus->tx_callback;
    hal_smbus_user_cb p_func_addr = (hal_smbus_user_cb)p_smbus->addr_callback;

    if((SET == hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_STPDET))) {
        p_smbus->tx_state = HAL_SMBUS_STATE_READY;

        /* clear the STOP bit */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_STPDET);
        /* disable interrupt */
        hals_i2c_interrupt_disable(p_smbus->periph, I2C_INT_ALL);

        /* user configuration tx_callback*/
        if(NULL != p_func) {
            p_func(p_smbus);
        } else {
            /* do nothing */
        }
    } else if((SET == hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_ADDSEND))) {
        /* clear the ADDSEND bit */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_ADDSEND);

        /* clear I2C_TDATA register */
        _smbus_flush_tdata_register(p_smbus->periph);

        /* user configuration addr_callback */
        if(NULL != p_func_addr) {
            p_func_addr(p_smbus);
        } else {
            /* do nothing */
        }
    } else if(SET == hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_TI)) {
        /* transmit data from TDATA */
        hals_i2c_data_transmit(p_smbus->periph, (uint32_t)*p_smbus->txbuffer.buffer);
        p_smbus->txbuffer.buffer++;
        p_smbus->txbuffer.length--;
        p_smbus->txbuffer.pos--;
    } else if(SET == hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_NACK)) {
        if((0U != p_smbus->txbuffer.length)) {
            p_smbus->error_state = (uint16_t)HAL_I2C_ERROR_SLAVE_TX;
        } else {
            /* do nothing */
        }

        /* clear the ADDSEND bit */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_NACK);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      event handler in SMBUS slave received mode
    \param[in]  smbus_dev: SMBUS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _smbus_slave_received_interrupt(void *smbus_dev)
{
    hal_smbus_dev_struct *p_smbus = smbus_dev;
    hal_smbus_user_cb p_func = (hal_smbus_user_cb)p_smbus->rx_callback;
    hal_smbus_user_cb p_func_addr = (hal_smbus_user_cb)p_smbus->addr_callback;

    if((SET == hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_STPDET))) {
        p_smbus->rx_state = HAL_SMBUS_STATE_READY;

        /* clear the STOP bit */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_STPDET);

        /* user configuration rx_callback*/
        if(NULL != p_func) {
            p_func(p_smbus);
        } else {
            /* do nothing */
        }
    } else if((SET == hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_ADDSEND))) {
        /* clear the ADDSEND bit */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_ADDSEND);

        /* clear I2C_TDATA register */
        _smbus_flush_tdata_register(p_smbus->periph);

        /* user configuration addr_callback */
        if(NULL != p_func_addr) {
            p_func_addr(p_smbus);
        } else {
            /* do nothing */
        }
    } else if((SET == hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_RBNE))) {
        /* Read data from RXDR */
        *p_smbus->rxbuffer.buffer = hals_i2c_data_receive(p_smbus->periph);
         p_smbus->rxbuffer.buffer++;
         p_smbus->rxbuffer.length--;
         p_smbus->rxbuffer.pos--;
    } else if(SET == hals_i2c_interrupt_flag_get(p_smbus->periph, I2C_INT_FLAG_NACK)) {
        if((SMBUS_DATA_RECEIVED_FINISH_LENGTH != p_smbus->rxbuffer.length)) {
            p_smbus->error_state = (uint16_t)HAL_I2C_ERROR_SLAVE_RX;
        } else {
            /* do nothing */
        }

        /* clear the NACK bit */
        hals_i2c_interrupt_flag_clear(p_smbus->periph, I2C_INT_FLAG_NACK);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      event handler for address listen in slave mode
    \param[in]  smbus_dev: SMBus device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _smbus_address_listen_interrupt(void *smbus_dev)
{
    hal_smbus_dev_struct *p_smbus = smbus_dev;
    hal_smbus_user_cb p_func    = (hal_smbus_user_cb)p_smbus->rx_callback;

    hals_i2c_interrupt_disable(p_smbus->periph, I2C_INT_ADDMIE);
    hals_i2c_interrupt_disable(p_smbus->periph, I2C_INT_NACKIE);
    hals_i2c_interrupt_disable(p_smbus->periph, I2C_INT_STPDETIE);

    /* check transfer direction in p_func, so that the user knows
    which function to call next */
    if(NULL != p_func) {
        p_func(p_smbus);
    } else {
        /* do nothing */
    }
}
