/*!
    \file    gd32h7xx_hal_sdio_sdcard.c
    \brief   SDIO SDCARD driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

uint32_t g_buf_tuning[64]; /* store the data read from the card */

/* store the data read from the card */
uint32_t g_buf_tuning_standard[16] = {0x00FF0FFFU, 0xCCC3CCFFU, 0xFFCC3CC3U, 0xEFFEFFFEU, 0xDDFFDFFFU, 0xFBFFFBFFU, \
                                    0xFF7FFFBFU, 0xEFBDF777U, 0xF0FFF0FFU, 0x3CCCFC0FU, 0xCFCC33CCU, 0xEEFFEFFFU, \
                                    0xFDFFFDFFU, 0xFFBFFFDFU, 0xFFF7FFBBU, 0xDE7B7FF7U};

/* check if the command sent error occurs */
static hal_sdio_sdcard_error_enum _cmdsent_error_check(hal_sdio_sdcard_dev_struct *sd_dev);
/* check if error occurs for R1 response */
static hal_sdio_sdcard_error_enum _r1_error_check(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t cmdindex);
/* check if error type for R1 response */
static hal_sdio_sdcard_error_enum _r1_error_type_check(uint32_t resp);
/* check if error occurs for R2 response */
static hal_sdio_sdcard_error_enum _r2_error_check(hal_sdio_sdcard_dev_struct *sd_dev);
/* check if error occurs for R3 response */
static hal_sdio_sdcard_error_enum _r3_error_check(hal_sdio_sdcard_dev_struct *sd_dev);
/* check if error occurs for R6 response */
static hal_sdio_sdcard_error_enum _r6_error_check(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t cmdindex, uint16_t *prca);
/* check if error occurs for R7 response */
static hal_sdio_sdcard_error_enum _r7_error_check(hal_sdio_sdcard_dev_struct *sd_dev);
/* configure the sd card init */
static hal_sdio_sdcard_error_enum _sd_card_init(hal_sdio_sdcard_dev_struct *sd_dev);
/* configure the sd power on */
static hal_sdio_sdcard_error_enum _sd_power_on(hal_sdio_sdcard_dev_struct *sd_dev);
/* configure the sd card select deselect */
static hal_sdio_sdcard_error_enum _sd_card_select_deselect(hal_sdio_sdcard_dev_struct *sd_dev);

/* configure the bus width mode */
static hal_sdio_sdcard_error_enum _sd_bus_width_config(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t buswidth);
/* get the SCR of corresponding card */
static hal_sdio_sdcard_error_enum _sd_scr_get(hal_sdio_sdcard_dev_struct *sd_dev);
/* send transfer stop cmd */
static hal_sdio_sdcard_error_enum _sd_transfer_stop(hal_sdio_sdcard_dev_struct *sd_dev);
/* switch 1.8V power level of SD card */
static hal_sdio_sdcard_error_enum _sd_card_voltage_switch(hal_sdio_sdcard_dev_struct *sd_dev);
/* get the data block size */
static uint32_t _sd_datablocksize_get(uint16_t bytesnumber);
/* read block interrupt service */
static void _sd_read_interrupt(void *sd_dev);
/* write block interrupt service */
static void _sd_write_interrupt(void *sd_dev);
/* enable SD Transceiver 1.8V mode */
static void _sd_transceiver_enable(FlagStatus flag);
/* perform sampling point tuning using tuning command */
static hal_sdio_sdcard_error_enum _sd_tuning(hal_sdio_sdcard_dev_struct *sd_dev);
/* get the card state */
static hal_sdio_sdcard_error_enum _sd_card_state_get(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t *pcardstate);
/* sdcard deinit */
static void _sdcard_deinit(uint32_t sdio_periph);
/* configuration cpdm time sync */
static void _cpdm_config(void);
/* get the card status whose response format R1 contains a 32-bit field */
static hal_sdio_sdcard_error_enum _sdcard_cardstatus_get(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pcardstatus);
static hal_sdio_sdcard_error_enum _sdcard_cardstate_get(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t *pcardstate);
/* lock or unlock a card */
static hal_sdio_sdcard_error_enum _sdcard_lock_unlock(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t lockstate);
/* configure the bus mode */
static hal_sdio_sdcard_error_enum _sdcard_bus_mode_config(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t busmode, uint32_t speed);

/* configure error handle */
static void _sd_transmit_received_handle(void *sd_dev);
/* configure idma error handle */
static void _sd_idma_complete_handle(void *sd_dev);
/* configure error handle */
static void _sd_error_handle(void *sd_dev);

/*!
    \brief      initialize sdcard
    \param[in]  sd_dev: SD card device information structure
    \param[in]  periph: SDIO peripheral address
    \param[in]  p_sd_init: the initialization data needed to initialize SDIO
                clock_division: clock division factor
                clock_edge: clock edge type
                only one parameter can be selected which is shown as below:
      \arg        SDIO_SDIOCLKEDGE_RISING: rising edge of SDIOCLK
      \arg        SDIO_SDIOCLKEDGE_FALLING: falling edge of SDIOCLK
                clock_powersave: clock power save mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_CLOCKPWRSAVE_DISABLE: clock power save disable
      \arg        SDIO_CLOCKPWRSAVE_ENABLE: clock power save enable
                bus_speed_mode: bus speed mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_BUSSPEED_LOW: low speed mode
      \arg        SDIO_BUSSPEED_HIGHT: high speed mode
                data_rate: data rate mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_DATA_RATE_SDR: single data rate
      \arg        SDIO_DATA_RATE_DDR: double data rate
                hardware_control: hardware flow control
                only one parameter can be selected which is shown as below:
      \arg        SDIO_HARDWARE_DISABLE: hardware flow control disable
      \arg        SDIO_HARDWARE_ENABLE: hardware flow control enable
                transceiver: transceiver enable
                only one parameter can be selected which is shown as below:
      \arg        DISABLE: transceiver disable
      \arg        ENABLE: transceiver enable
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
hal_sdio_sdcard_error_enum hal_sdio_sdcard_init(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t periph, hal_sdio_sdcard_init_struct *p_sd_init)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    hal_sdio_sdcard_card_csd_struct csd;
    uint32_t cardstate = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == sd_dev) || (NULL == p_sd_init)) {
        HAL_DEBUGE("pointer [sd_dev] or [p_sd_init] address is invalid");
        return HAL_SD_ERROR;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* deinitialize the SDIO */
    _sdcard_deinit(periph);

    sd_dev->periph = periph;

    if((SDIO_SPEED_DEFAULT != p_sd_init->bus_speed_mode) && (SDIO_SPEED_HIGH != p_sd_init->bus_speed_mode)) {
        sd_dev->card_info.card_speed = SDCAED_VOLTAGE_18V;
    } else {
        sd_dev->card_info.card_speed = SDCAED_VOLTAGE_33V;
    }

    /* if transceiver present */
    if(NULL == sd_dev->transceiver_1_8v_callback) {
        sd_dev->transceiver_1_8v_callback = &_sd_transceiver_enable;
    } else {
        /* do nothing */
    }

    /* configure the clock and work voltage */
    status = _sd_power_on(sd_dev);
    if(HAL_SD_OK == status) {

        /* initialize the card and get CID and CSD of the card */
        status = _sd_card_init(sd_dev);
        if(HAL_SD_OK == status) {

            /* configure the SDIO peripheral */
            hals_sdio_clock_config(periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, SD_CLK_DIV_INIT);
            hals_sdio_bus_mode_set(periph, SDIO_BUSMODE_1BIT);
            hals_sdio_hardware_clock_disable(periph);

            hal_sdio_sdcard_csd_get(sd_dev, &csd);
            _sd_card_select_deselect(sd_dev);
            status = _sdcard_cardstatus_get(sd_dev, &cardstate);
            if(cardstate & 0x02000000U) {
                status = _sdcard_lock_unlock(sd_dev, SD_UNLOCK);
                if(HAL_SD_OK != status) {

                    status = HAL_SD_LOCK_UNLOCK_FAILED;
                } else {
                    /* do nothing */
                }
            }

            if((HAL_SD_OK == status) && (!(cardstate & 0x02000000U))) {
                /* set bus mode */
                status = _sdcard_bus_mode_config(sd_dev, p_sd_init->bus_width, p_sd_init->bus_speed_mode);
            } else {
                /* do nothing */
            }

            if((HAL_SD_OK == status) && (SDCAED_VOLTAGE_18V == sd_dev->card_info.card_speed)) {
                /* UHS-I Hosts can perform sampling point tuning using tuning command  */
                status = _sd_tuning(sd_dev);
            } else {
                /* do nothing */
            }

            /* set the SD state */
            sd_dev->state = HAL_SD_STATE_READY;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }


    return status;
}

/*!
    \brief      initialize the sdcard structure with the default values
    \param[in]  hal_struct_type: type of sdcard structure for initialization
                only one parameters can be selected which are shown as below:
      \arg        HAL_SD_INIT_STRUCT: initialization structure
      \arg        HAL_SD_DEV_STRUCT: device information structure
      \arg        HAL_SD_IRQ_INIT_STRUCT: interrupt callback initialization structure
    \param[out] p_struct: pointer to sdcard structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_struct_init(hal_sdio_sdcard_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t reval = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_SD_INIT_STRUCT:
        /* initialize SDIO initialization structure with the default values */
        ((hal_sdio_sdcard_init_struct *)p_struct)->clock_division   = 0U;
        ((hal_sdio_sdcard_init_struct *)p_struct)->clock_edge       = SDIO_SDIOCLKEDGE_RISING;
        ((hal_sdio_sdcard_init_struct *)p_struct)->clock_powersave  = SDIO_CLOCKPWRSAVE_DISABLE;
        ((hal_sdio_sdcard_init_struct *)p_struct)->bus_speed_mode   = SDIO_BUSSPEED_LOW;
        ((hal_sdio_sdcard_init_struct *)p_struct)->hardware_control = SDIO_HARDWARE_DISABLE;
        ((hal_sdio_sdcard_init_struct *)p_struct)->transceiver      = DISABLE;
        break;
    case HAL_SD_DEV_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_sdcard_dev_struct *)p_struct)->periph                              = 0U;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->p_txbuffer                          = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->tx_length                           = 0U;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->p_rxbuffer                          = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->rx_length                           = 0U;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->transfer                            = SD_TRANSFER_NONE;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->state                               = HAL_SD_STATE_RESET;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->error_state                         = HAL_SD_OK;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->mutex                               = HAL_MUTEX_UNLOCKED;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->transmit_complete_callback          = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->receive_complete_callback           = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->transfer_error_callback             = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->write_dma_buffer0_complete_callback = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->write_dma_buffer1_complete_callback = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->read_dma_buffer0_complete_callback  = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->read_dma_buffer1_complete_callback  = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->abort_complete_callback             = NULL;
        ((hal_sdio_sdcard_dev_struct *)p_struct)->transceiver_1_8v_callback           = NULL;
        break;
    case HAL_SD_IRQ_INIT_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_sdcard_irq_struct *)p_struct)->transmit_fifo_half_empty_handle  = NULL;
        ((hal_sdio_sdcard_irq_struct *)p_struct)->receive_fifo_half_empty_handle   = NULL;
        ((hal_sdio_sdcard_irq_struct *)p_struct)->transfer_complete_handle         = NULL;
        ((hal_sdio_sdcard_irq_struct *)p_struct)->idma_complete_handle             = NULL;
        ((hal_sdio_sdcard_irq_struct *)p_struct)->transfer_error_handle            = NULL;
        break;
    case HAL_SD_CARD_CID_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_sdcard_card_cid_struct *)p_struct)->mid     = 0U;
        ((hal_sdio_sdcard_card_cid_struct *)p_struct)->oid     = 0U;
        ((hal_sdio_sdcard_card_cid_struct *)p_struct)->pnm0    = 0U;
        ((hal_sdio_sdcard_card_cid_struct *)p_struct)->pnm1    = 0U;
        ((hal_sdio_sdcard_card_cid_struct *)p_struct)->prv     = 0U;
        ((hal_sdio_sdcard_card_cid_struct *)p_struct)->psn     = 0U;
        ((hal_sdio_sdcard_card_cid_struct *)p_struct)->mdt     = 0U;
        ((hal_sdio_sdcard_card_cid_struct *)p_struct)->cid_crc = 0U;
        break;
    case HAL_SD_CARD_CSD_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->csd_struct         = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->taac               = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->nsac               = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->tran_speed         = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->ccc                = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->read_bl_len        = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->read_bl_partial    = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->write_blk_misalign = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->read_blk_misalign  = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->dsp_imp            = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->c_size             = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->vdd_r_curr_min     = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->vdd_r_curr_max     = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->vdd_w_curr_min     = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->vdd_w_curr_max     = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->c_size_mult        = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->erase_blk_en       = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->sector_size        = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->wp_grp_size        = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->wp_grp_enable      = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->r2w_factor         = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->write_bl_len       = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->write_bl_partial   = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->file_format_grp    = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->copy_flag          = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->perm_write_protect = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->tmp_write_protect  = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->file_format        = 0U;
        ((hal_sdio_sdcard_card_csd_struct *)p_struct)->csd_crc            = 0U;
        break;
    case HAL_SD_CARDINFO_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_sdcard_cardinfo_struct *)p_struct)->card_type       = 0U;
        ((hal_sdio_sdcard_cardinfo_struct *)p_struct)->card_speed      = 0U;
        ((hal_sdio_sdcard_cardinfo_struct *)p_struct)->card_rca        = 0U;
        ((hal_sdio_sdcard_cardinfo_struct *)p_struct)->card_blocknum   = 0U;
        ((hal_sdio_sdcard_cardinfo_struct *)p_struct)->card_blocksize  = 0U;
        ((hal_sdio_sdcard_cardinfo_struct *)p_struct)->logic_blocknum  = 0U;
        ((hal_sdio_sdcard_cardinfo_struct *)p_struct)->logic_blocksize = 0U;
        break;
    case HAL_SD_IRQ_USER_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_sdcard_irq_user_callback_struct *)p_struct)->transmit_complete_func          = NULL;
        ((hal_sdio_sdcard_irq_user_callback_struct *)p_struct)->receive_complete_func           = NULL;
        ((hal_sdio_sdcard_irq_user_callback_struct *)p_struct)->read_dma_buffer0_complete_func  = NULL;
        ((hal_sdio_sdcard_irq_user_callback_struct *)p_struct)->read_dma_buffer1_complete_func  = NULL;
        ((hal_sdio_sdcard_irq_user_callback_struct *)p_struct)->write_dma_buffer0_complete_func = NULL;
        ((hal_sdio_sdcard_irq_user_callback_struct *)p_struct)->write_dma_buffer1_complete_func = NULL;
        ((hal_sdio_sdcard_irq_user_callback_struct *)p_struct)->transfer_error_func             = NULL;
        ((hal_sdio_sdcard_irq_user_callback_struct *)p_struct)->abort_complete_func             = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        reval =  HAL_ERR_VAL;
        break;
    }

    return reval;
}

/*!
    \brief      deinitialize the sdcard
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_deinit(hal_sdio_sdcard_dev_struct *sd_dev)
{
    int32_t reval = HAL_ERR_NONE;
    uint32_t periph;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sd_dev) {
        HAL_DEBUGE("pointer [sd_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    periph = sd_dev->periph;
    if((SDIO0 == periph) || (SDIO1 == periph)) {
        /* reset SD Transceiver 1.8V mode via callback */
        if(NULL != sd_dev->transceiver_1_8v_callback) {
            sd_dev->transceiver_1_8v_callback(RESET);
        } else {
            /* do nothing */
        }
        /* deinitialize the periph and the device information structure */
        _sdcard_deinit(periph);
        sd_dev->state = HAL_SD_STATE_RESET;
    } else {
        HAL_DEBUGE("parameter [sd_dev->periph] value is invalid");
        reval = HAL_ERR_VAL;
    }

    return reval;
}

/*!
    \brief      set user-defined transceiver callback function
    \param[in]  sd_dev: sdcard device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_callback: point to sdcard transceiver callback functions
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_transceiver_callback_register(hal_sdio_sdcard_dev_struct *sd_dev, transceiver_callback p_user_callback)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and p_user_callback address */
    if((NULL == sd_dev) || (NULL == p_user_callback)) {
        HAL_DEBUGE("pointer [sd_dev] or [p_user_callback] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    sd_dev->transceiver_1_8v_callback = p_user_callback;

    return HAL_ERR_NONE;
}

/*!
    \brief      reset user-defined transceiver callback function
    \param[in]  sd_dev: sdcard device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_transceiver_callback_unregister(hal_sdio_sdcard_dev_struct *sd_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer address */
    if((NULL == sd_dev)) {
        HAL_DEBUGE("pointer [sd_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    sd_dev->transceiver_1_8v_callback = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  sd_dev: sdcard device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to sdcard interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_irq_handle_set(hal_sdio_sdcard_dev_struct *sd_dev, hal_sdio_sdcard_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and p_irq address */
    if((NULL == sd_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [sd_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set user-defined transmit complete interrupt callback */
    if(NULL != p_irq->transmit_fifo_half_empty_handle) {
        sd_dev->sd_irq.transmit_fifo_half_empty_handle = p_irq->transmit_fifo_half_empty_handle;
        hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_TFH | SDIO_INT_DTEND);
    } else {
        sd_dev->sd_irq.transmit_fifo_half_empty_handle = NULL;
        hals_sdio_interrupt_disable(sd_dev->periph, SDIO_INT_TFH | SDIO_INT_DTEND);
    }

    /* set user-defined receive complete interrupt callback */
    if(NULL != p_irq->receive_fifo_half_empty_handle) {
        sd_dev->sd_irq.receive_fifo_half_empty_handle = p_irq->receive_fifo_half_empty_handle;
        hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_RFH | SDIO_INT_DTEND);
    } else {
        sd_dev->sd_irq.receive_fifo_half_empty_handle = NULL;
        hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_RFH | SDIO_INT_DTEND);
    }

    /* set user-defined transfer error interrupt callback */
    if(NULL != p_irq->transfer_error_handle) {
        sd_dev->sd_irq.transfer_error_handle = p_irq->transfer_error_handle;
        hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                                   SDIO_INT_TXURE | SDIO_INT_RXORE | SDIO_INT_IDMAERR);
    } else {
        sd_dev->sd_irq.transfer_error_handle = NULL;
        hals_sdio_interrupt_disable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                                    SDIO_INT_TXURE | SDIO_INT_RXORE | SDIO_INT_IDMAERR);
    }

    /* set user-defined internal DMA transfer complete interrupt callback */
    if(NULL != p_irq->idma_complete_handle) {
        sd_dev->sd_irq.idma_complete_handle = p_irq->idma_complete_handle;
        hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_IDMAEND | SDIO_INT_DTEND);
    } else {
        sd_dev->sd_irq.idma_complete_handle = NULL;
        hals_sdio_interrupt_disable(sd_dev->periph,  SDIO_INT_IDMAEND | SDIO_INT_DTEND);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_irq_handle_all_reset(hal_sdio_sdcard_dev_struct *sd_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer address */
    if(NULL == sd_dev) {
        HAL_DEBUGE("pointer [sd_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    sd_dev->sd_irq.transmit_fifo_half_empty_handle      = NULL;
    sd_dev->sd_irq.receive_fifo_half_empty_handle       = NULL;
    sd_dev->sd_irq.transfer_complete_handle             = NULL;
    sd_dev->sd_irq.idma_complete_handle                 = NULL;
    sd_dev->sd_irq.transfer_error_handle                = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      SD card interrupt handler content function,which is merely used in SDIO_IRQHandler
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_sdio_sdcard_irq(hal_sdio_sdcard_dev_struct *sd_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sd_dev) {
        HAL_DEBUGE("pointer [sd_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check for SDMMC interrupt flags */
    if(RESET != hals_sdio_interrupt_flag_get(sd_dev->periph, SDIO_INT_FLAG_RFH)) {
        hals_sdio_interrupt_flag_clear(sd_dev->periph, SDIO_INT_FLAG_RFH);
        if(NULL != sd_dev->sd_irq.receive_fifo_half_empty_handle) {
            sd_dev->sd_irq.receive_fifo_half_empty_handle(sd_dev);
        } else {
            /* do nothing */
        }
    } else if(RESET != hals_sdio_interrupt_flag_get(sd_dev->periph, SDIO_INT_FLAG_DTEND)) {
        /* clear DTEND flag */
        hals_sdio_interrupt_flag_clear(sd_dev->periph, SDIO_INT_FLAG_DTEND);
        if(NULL != sd_dev->sd_irq.transfer_complete_handle) {
            sd_dev->sd_irq.transfer_complete_handle(sd_dev);
        } else {
            /* do nothing */
        }
    } else if(RESET != hals_sdio_interrupt_flag_get(sd_dev->periph, SDIO_INT_FLAG_TFH)) {
        hals_sdio_interrupt_flag_clear(sd_dev->periph, SDIO_INT_FLAG_TFH);
        if(NULL != sd_dev->sd_irq.transmit_fifo_half_empty_handle) {
            sd_dev->sd_irq.transmit_fifo_half_empty_handle(sd_dev);
        } else {
            /* do nothing */
        }
    } else if(RESET != hals_sdio_interrupt_flag_get(sd_dev->periph, SDIO_INT_FLAG_DTCRCERR | SDIO_INT_FLAG_DTTMOUT | \
                                                             SDIO_INT_FLAG_TXURE | SDIO_INT_FLAG_RXORE)) {
        if(NULL != sd_dev->sd_irq.transfer_error_handle) {
            sd_dev->sd_irq.transfer_error_handle(sd_dev);
        } else {
            /* do nothing */
        }
    } else if(RESET != hals_sdio_interrupt_flag_get(sd_dev->periph, SDIO_INT_FLAG_IDMAEND)) {
        hals_sdio_interrupt_flag_clear(sd_dev->periph, SDIO_INT_FLAG_IDMAEND);
        if(NULL != sd_dev->sd_irq.idma_complete_handle) {
            sd_dev->sd_irq.idma_complete_handle(sd_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      configure the bus mode
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  busmode: the bus mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_BUSMODE_1BIT: 1-bit SDIO card bus mode
      \arg        SDIO_BUSMODE_4BIT: 4-bit SDIO card bus mode
      \arg        SDIO_BUSMODE_8BIT: 8-bit SDIO card bus mode (MMC only)
    \param[in]  speed: the bus speed mode
                only one parameter can be selected which is shown as below:
      \arg        SD_SPEED_DEFAULT: default bus speed
      \arg        SD_SPEED_HIGH: high bus speed
      \arg        SD_SPEED_SDR25: SDR25 bus speed
      \arg        SD_SPEED_SDR50: SDR50 bus speed
      \arg        SD_SPEED_SDR104: SDR104 bus speed
      \arg        SD_SPEED_DDR50: DDR50 bus speed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_bus_mode_config(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t busmode, uint32_t speed)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer address */
    if(NULL == sd_dev) {
        HAL_DEBUGE("pointer [sd_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    status = _sdcard_bus_mode_config(sd_dev, busmode, speed);
    if(HAL_SD_OK != status) {
        reval = HAL_ERR_VAL;
    } else {
        reval = HAL_ERR_NONE;
    }
    return reval;
}

/*!
    \brief      configure dma dual buffer mode. the data transfer is managed by an internal dma
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  data_buffer0: pointer to the buffer0 that will contain/receive the transferred data
    \param[in]  data_buffer1: pointer to the buffer1 that will contain/receive the transferred data
    \param[in]  buffer_size: size of buffer
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_dma_buffer_config(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *data_buffer0, uint32_t *data_buffer1, \
                                     uint32_t buffer_size)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer address */
    if((NULL == sd_dev) || (NULL == data_buffer0) || (NULL == data_buffer1)) {
        HAL_DEBUGE("pointer [sd_dev] or [data_buffer] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    hals_sdio_idma_set(sd_dev->periph, SDIO_IDMA_DOUBLE_BUFFER, buffer_size);
    hals_sdio_idma_buffer0_address_set(sd_dev->periph, (uint32_t)data_buffer0);
    hals_sdio_idma_buffer1_address_set(sd_dev->periph, (uint32_t)data_buffer1);

    return HAL_ERR_NONE;
}

/*!
    \brief      change the dma buffer0 or buffer1 address on the fly
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  buffer_select: the buffer to be used
                only one parameter can be selected which is shown as below:
      \arg        SDIO_IDMA_BUFFER0: select buffer0
      \arg        SDIO_IDMA_BUFFER1: select buffer1
    \param[in]  data_buffer: The new address
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_change_dma_buffer(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t buffer_select, \
                                        uint32_t *data_buffer)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer address */
    if((NULL == sd_dev) || (NULL == data_buffer)) {
        HAL_DEBUGE("pointer [sd_dev] or [data_buffer] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(SDIO_IDMA_BUFFER0 == buffer_select) {
        hals_sdio_idma_buffer0_address_set(sd_dev->periph, (uint32_t)data_buffer);
    } else {
        hals_sdio_idma_buffer1_address_set(sd_dev->periph, (uint32_t)data_buffer);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      read data into a buffer from the specified address of a card
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  blockaddr: the read data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be read
    \param[out] pdata: a pointer that store multiple blocks read data
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_readblocks_poll(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                   uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t count = 0U, align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE, *ptempbuff = pdata;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pdata address */
    if((NULL == sd_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [sd_dev] or [pdata] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_READY != sd_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > SD_MAX_DATA_LENGTH) {
        /* exceeds the maximum length */
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(sd_dev);

        sd_dev->state = HAL_SD_STATE_BUSY;

        /* clear all DSM configuration */
        hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(sd_dev->periph);
        hals_sdio_idma_disable(sd_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
            sd_dev->error_state = HAL_SD_LOCK_UNLOCK_FAILED;
            sd_dev->state = HAL_SD_STATE_READY;
            HAL_UNLOCK(sd_dev);
            reval = HAL_ERR_ABORT;
        } else {

            /* blocksize is fixed in 512B for SDHC card */
            if(SDIO_HIGH_CAPACITY_SD_CARD != sd_dev->card_info.card_type) {
                blockaddr *= 512U;
            } else {
                blocksize = 512U;
            }

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _sd_datablocksize_get(blocksize);
                /* send CMD16(SET_BLOCKLEN) to set the block length */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, (uint32_t)blocksize, \
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    reval = HAL_ERR_ABORT;
                    HAL_UNLOCK(sd_dev);
                } else {
                    /* do nothing */
                }
            } else {
                sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
                sd_dev->state = HAL_SD_STATE_READY;
                HAL_UNLOCK(sd_dev);
                reval = HAL_ERR_ABORT;
            }

            if(HAL_ERR_NONE == reval) {

                /* configure SDIO data transmission */
                hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, ((uint32_t)blocknumber * blocksize), datablksize);
                hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
                hals_sdio_trans_start_enable(sd_dev->periph);

                if(blocknumber > 1U) {
                    sd_dev->transfer = SD_TRANSFER_READ_MULTIPLE_BLOCK;
                    /* send CMD18(READ_MULTIPLE_BLOCK) to read multiple blocks */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_READ_MULTIPLE_BLOCK, blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);

                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_READ_MULTIPLE_BLOCK);
                    if(HAL_SD_OK != status) {
                        sd_dev->state    = HAL_SD_STATE_READY;
                        sd_dev->transfer = SD_TRANSFER_NONE;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_ABORT;
                    } else {
                        /* do nothing */
                    }
                } else {
                    sd_dev->transfer = SD_TRANSFER_READ_SINGLE_BLOCK;
                    /* send CMD17(READ_SINGLE_BLOCK) to read a block */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_READ_SINGLE_BLOCK, (uint32_t)blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);
                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_READ_SINGLE_BLOCK);
                    if(HAL_SD_OK != status) {
                        sd_dev->state    = HAL_SD_STATE_READY;
                        sd_dev->transfer = SD_TRANSFER_NONE;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_ABORT;
                    } else {
                        /* do nothing */
                    }
                }

                if(HAL_ERR_NONE == reval) {
                    /* polling mode */
                    while(!hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR | SDIO_FLAG_DTTMOUT | SDIO_FLAG_RXORE | \
                                                            SDIO_FLAG_DTBLKEND | SDIO_FLAG_DTEND)) {
                        if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_RFH)) {
                            /* at least 8 words can be read in the FIFO */
                            for(count = 0U; count < SD_FIFOHALF_WORDS; count++) {
                                *(ptempbuff + count) = hals_sdio_data_read(sd_dev->periph);
                            }
                            ptempbuff += SD_FIFOHALF_WORDS;
                        } else {
                            /* do nothing */
                        }
                    }

                    /* whether some error occurs and return it */
                    if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR)) {
                        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTCRCERR);
                        sd_dev->error_state = HAL_SD_DATA_CRC_ERROR;
                        sd_dev->state    = HAL_SD_STATE_READY;
                        sd_dev->transfer = SD_TRANSFER_NONE;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_ABORT;
                    } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTTMOUT)) {
                        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTTMOUT);
                        sd_dev->error_state = HAL_SD_DATA_TIMEOUT;
                        sd_dev->state    = HAL_SD_STATE_READY;
                        sd_dev->transfer = SD_TRANSFER_NONE;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_ABORT;
                    } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_RXORE)) {
                        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_RXORE);
                        sd_dev->error_state = HAL_SD_RX_OVERRUN_ERROR;
                        sd_dev->state    = HAL_SD_STATE_READY;
                        sd_dev->transfer = SD_TRANSFER_NONE;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_ABORT;
                    } else {
                        /* do nothing */
                    }

                    if(HAL_ERR_NONE == reval) {
                        while((SET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_RFE)) && \
                            (SET == hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DATSTA))) {
                            *ptempbuff = hals_sdio_data_read(sd_dev->periph);
                            ++ptempbuff;
                        }

                        hals_sdio_trans_start_disable(sd_dev->periph);

                        if((RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTEND)) && (blocknumber > 1U)) {
                            if((SDIO_STD_CAPACITY_SD_CARD_V1_1 == sd_dev->card_info.card_type) || \
                            (SDIO_STD_CAPACITY_SD_CARD_V2_0 == sd_dev->card_info.card_type) || \
                            (SDIO_HIGH_CAPACITY_SD_CARD == sd_dev->card_info.card_type)) {
                                /* send CMD12(STOP_TRANSMISSION) to stop transmission */
                                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_STOP_TRANSMISSION, 0U, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(sd_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(sd_dev, SD_CMD_STOP_TRANSMISSION);
                                if(HAL_SD_OK != status) {
                                    sd_dev->state    = HAL_SD_STATE_READY;
                                    sd_dev->transfer = SD_TRANSFER_NONE;
                                    HAL_UNLOCK(sd_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                /* do nothing */
                            }
                        } else {
                            /* do nothing */
                        }

                        /* clear the DATA_FLAGS flags */
                        hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_DATA_FLAGS);

                        sd_dev->state    = HAL_SD_STATE_READY;
                        sd_dev->transfer = SD_TRANSFER_NONE;

                        /* process unlocked */
                        HAL_UNLOCK(sd_dev);
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      read data into a buffer from the specified address of a card by interrupt method
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  blockaddr: the read data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be read
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] pdata: a pointer that store multiple blocks read data
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY,HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_readblocks_interrupt(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                             uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                             hal_sdio_sdcard_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pdata address */
    if((NULL == sd_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [sd_dev] or [pdata] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_READY != sd_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    }else if((uint32_t)blocknumber * blocksize > SD_MAX_DATA_LENGTH) {
        /* exceeds the maximum length */
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {
        /* process locked */
        HAL_LOCK(sd_dev);

        sd_dev->state = HAL_SD_STATE_BUSY;

        /* blocksize is fixed in 512B for SDHC card */
        if(SDIO_HIGH_CAPACITY_SD_CARD != sd_dev->card_info.card_type) {
            blockaddr *= 512U;
        } else {
            blocksize = 512U;
        }

        sd_dev->p_rxbuffer = pdata;
        sd_dev->rx_length  = (uint32_t)blocksize * blocknumber;

        sd_dev->sd_irq.receive_fifo_half_empty_handle = &_sd_read_interrupt;
        sd_dev->sd_irq.transfer_complete_handle       = &_sd_transmit_received_handle;
        sd_dev->sd_irq.transfer_error_handle          = &_sd_error_handle;

        sd_dev->receive_complete_callback        = NULL;
        sd_dev->transfer_error_callback          = NULL;
        sd_dev->abort_complete_callback          = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->receive_complete_func) {
                sd_dev->receive_complete_callback = (void *)p_user_func->receive_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                sd_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                sd_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(sd_dev->periph);
        hals_sdio_idma_disable(sd_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
            sd_dev->error_state = HAL_SD_LOCK_UNLOCK_FAILED;
            sd_dev->state = HAL_SD_STATE_READY;
            HAL_UNLOCK(sd_dev);
            reval = HAL_ERR_ABORT;
        } else {

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _sd_datablocksize_get(blocksize);
                /* send CMD16(SET_BLOCKLEN) to set the block length */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, (uint32_t)blocksize, \
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    /* do nothing */
                }
            } else {
                sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
                sd_dev->state = HAL_SD_STATE_READY;
                HAL_UNLOCK(sd_dev);
                reval = HAL_ERR_ABORT;
            }

            if(HAL_ERR_NONE == reval) {

                /* configure SDIO data transmission */
                hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, ((uint32_t)blocknumber * blocksize), datablksize);
                hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
                hals_sdio_trans_start_enable(sd_dev->periph);

                if(blocknumber > 1U) {
                    sd_dev->transfer = (SD_TRANSFER_READ_MULTIPLE_BLOCK | SD_TRANSFER_IT);
                    /* send CMD18(READ_MULTIPLE_BLOCK) to read multiple blocks */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_READ_MULTIPLE_BLOCK, blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);

                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_READ_MULTIPLE_BLOCK);
                    if(HAL_SD_OK != status) {
                        sd_dev->state    = HAL_SD_STATE_READY;
                        sd_dev->transfer = SD_TRANSFER_NONE;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_ABORT;
                    } else {
                        /* do nothing */
                    }
                } else {
                    sd_dev->transfer = (SD_TRANSFER_READ_SINGLE_BLOCK | SD_TRANSFER_IT);
                    /* send CMD17(READ_SINGLE_BLOCK) to read a block */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_READ_SINGLE_BLOCK, (uint32_t)blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);
                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_READ_SINGLE_BLOCK);
                    if(HAL_SD_OK != status) {
                        sd_dev->state    = HAL_SD_STATE_READY;
                        sd_dev->transfer = SD_TRANSFER_NONE;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_ABORT;
                    } else {
                        /* do nothing */
                    }
                }
                if (HAL_ERR_NONE == reval) {
                    /* enable the SDIO corresponding interrupts */
                    hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                                            SDIO_INT_RFH | SDIO_INT_RXORE | SDIO_INT_DTEND);

                    /* process unlocked */
                    HAL_UNLOCK(sd_dev);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      read data into a buffer from the specified address of a card by dma method
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  blockaddr: the read data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be read
    \param[in]  p_user_func: pointer to the sdcard irq user callback struct
    \param[out] pdata: a pointer that store multiple blocks read data
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_readblocks_dma(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, uint32_t blockaddr, \
                                       uint16_t blocksize, uint16_t blocknumber, \
                                       hal_sdio_sdcard_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pdata address */
    if((NULL == sd_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [sd_dev] or [pdata] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_READY != sd_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > SD_MAX_DATA_LENGTH) {
        /* exceeds the maximum length */
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {
        /* process locked */
        HAL_LOCK(sd_dev);

        sd_dev->state = HAL_SD_STATE_BUSY;

        /* blocksize is fixed in 512B for SDHC card */
        if(SDIO_HIGH_CAPACITY_SD_CARD != sd_dev->card_info.card_type) {
            blockaddr *= 512U;
        } else {
            blocksize = 512U;
        }

        sd_dev->p_rxbuffer = pdata;
        sd_dev->rx_length  = (uint32_t)blocksize * blocknumber;

        sd_dev->sd_irq.receive_fifo_half_empty_handle = &_sd_read_interrupt;
        sd_dev->sd_irq.transfer_complete_handle = &_sd_transmit_received_handle;
        sd_dev->sd_irq.transfer_error_handle = &_sd_error_handle;

        sd_dev->receive_complete_callback = NULL;
        sd_dev->transfer_error_callback = NULL;
        sd_dev->abort_complete_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->receive_complete_func) {
                sd_dev->receive_complete_callback = (void *)p_user_func->receive_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                sd_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                sd_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(sd_dev->periph);
        hals_sdio_idma_disable(sd_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
            sd_dev->error_state = HAL_SD_LOCK_UNLOCK_FAILED;
            sd_dev->state = HAL_SD_STATE_READY;
            HAL_UNLOCK(sd_dev);
            reval = HAL_ERR_ABORT;
        } else {

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _sd_datablocksize_get(blocksize);
                /* send CMD16(SET_BLOCKLEN) to set the block length */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, (uint32_t)blocksize, \
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    /* do nothing */
                }
            } else {
                sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
                sd_dev->state = HAL_SD_STATE_READY;
                HAL_UNLOCK(sd_dev);
                reval = HAL_ERR_ABORT;
            }

            if(HAL_ERR_NONE == reval) {

                /* DMA mode */
                hals_sdio_idma_set(sd_dev->periph, SDIO_IDMA_SINGLE_BUFFER, (uint32_t)((uint32_t)blocksize >> 5U));
                hals_sdio_idma_buffer0_address_set(sd_dev->periph, (uint32_t)pdata);
                hals_sdio_idma_enable(sd_dev->periph);

                /* configure SDIO data transmission */
                hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, (uint32_t)blocknumber * blocksize, datablksize);
                hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
                hals_sdio_trans_start_enable(sd_dev->periph);

                if(blocknumber > 1U) {
                    sd_dev->transfer = (SD_TRANSFER_READ_MULTIPLE_BLOCK | SD_TRANSFER_DMA);
                    /* send CMD18(READ_MULTIPLE_BLOCK) to read multiple blocks */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_READ_MULTIPLE_BLOCK, blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);
                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_READ_MULTIPLE_BLOCK);
                } else {
                    sd_dev->transfer = (SD_TRANSFER_READ_SINGLE_BLOCK | SD_TRANSFER_DMA);
                    /* send CMD17(READ_SINGLE_BLOCK) to read a block */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_READ_SINGLE_BLOCK, (uint32_t)blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);
                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_READ_SINGLE_BLOCK);
                }

                if(HAL_SD_OK != status) {
                    sd_dev->state    = HAL_SD_STATE_READY;
                    sd_dev->transfer = SD_TRANSFER_NONE;
                } else {
                    /* do nothing */
                }

                /* enable the SDIO corresponding interrupts and DMA function */
                hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_RXORE | SDIO_INT_DTEND);

                /* process unlocked */
                HAL_UNLOCK(sd_dev);
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      read data into a buffer from the specified address of a card by dma method
    \param[in]  sd_dev: sdcard device information structure
                 the structure is not necessary to be reconfigured after structure initialization,
                 the structure parameters altering is automatically configured by core
    \param[in]  pbuf0: the idma buffer 0
    \param[in]  pbuf1: the idma buffer 1
    \param[in]  blockaddr: the read data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be read
    \param[in]  p_user_func: pointer to the sdcard irq user callback struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_readblocks_multibuffer_dma(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pbuf0, uint32_t *pbuf1, \
                                                   uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                                   hal_sdio_sdcard_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pbuf address */
    if((NULL == sd_dev) || (NULL == pbuf0) || (NULL == pbuf1)) {
        HAL_DEBUGE("pointer [sd_dev] or [pbuf0] or [pbuf1] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_READY != sd_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pbuf0) || (NULL == pbuf1) || (0U == blocknumber)) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else  if((uint32_t)blocknumber * blocksize > SD_MAX_DATA_LENGTH) {
        /* exceeds the maximum length */
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(sd_dev);

        sd_dev->state = HAL_SD_STATE_BUSY;

        /* blocksize is fixed in 512B for SDHC card */
        if(SDIO_HIGH_CAPACITY_SD_CARD != sd_dev->card_info.card_type) {
            blockaddr *= 512U;
        } else {
            blocksize = 512U;
        }

        sd_dev->sd_irq.receive_fifo_half_empty_handle = &_sd_read_interrupt;
        sd_dev->sd_irq.transfer_complete_handle = &_sd_transmit_received_handle;
        sd_dev->sd_irq.transfer_error_handle = &_sd_error_handle;

        sd_dev->read_dma_buffer0_complete_callback = NULL;
        sd_dev->read_dma_buffer1_complete_callback = NULL;
        sd_dev->abort_complete_callback = NULL;
        sd_dev->transfer_error_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->read_dma_buffer0_complete_func) {
                sd_dev->read_dma_buffer1_complete_callback = (void *)p_user_func->read_dma_buffer0_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->read_dma_buffer1_complete_func) {
                sd_dev->read_dma_buffer1_complete_callback = (void *)p_user_func->read_dma_buffer1_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                sd_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                sd_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(sd_dev->periph);
        hals_sdio_idma_disable(sd_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
            sd_dev->error_state = HAL_SD_LOCK_UNLOCK_FAILED;
            sd_dev->state = HAL_SD_STATE_READY;
            HAL_UNLOCK(sd_dev);
            reval = HAL_ERR_ABORT;
        } else {

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _sd_datablocksize_get(blocksize);
                /* send CMD16(SET_BLOCKLEN) to set the block length */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, (uint32_t)blocksize, \
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    /* do nothing */
                }
            } else {
                sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
                sd_dev->state = HAL_SD_STATE_READY;
                HAL_UNLOCK(sd_dev);
                reval = HAL_ERR_ABORT;
            }

            if(HAL_ERR_NONE == reval) {
                /* DMA mode */
                hals_sdio_idma_set(sd_dev->periph, SDIO_IDMA_DOUBLE_BUFFER, (uint32_t)((uint32_t)blocksize >> 5U));
                hals_sdio_idma_buffer0_address_set(sd_dev->periph, (uint32_t)pbuf0);
                hals_sdio_idma_buffer1_address_set(sd_dev->periph, (uint32_t)pbuf1);
                hals_sdio_idma_enable(sd_dev->periph);

                /* configure SDIO data transmission */
                hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, (uint32_t)blocknumber * blocksize, datablksize);
                hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
                hals_sdio_trans_start_enable(sd_dev->periph);

                sd_dev->transfer = (SD_TRANSFER_READ_MULTIPLE_BLOCK | SD_TRANSFER_DMA);
                /* send CMD18(READ_MULTIPLE_BLOCK) to read multiple blocks */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_READ_MULTIPLE_BLOCK, blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_READ_MULTIPLE_BLOCK);
                if(HAL_SD_OK != status) {
                    sd_dev->state    = HAL_SD_STATE_READY;
                    sd_dev->transfer = SD_TRANSFER_NONE;
                } else {
                    /* do nothing */
                }

                /* enable the SDIO corresponding interrupts and DMA function */
                hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_RXORE | \
                                                        SDIO_INT_DTEND | SDIO_INT_IDMAERR | SDIO_INT_IDMAEND);

                /* process unlocked */
                HAL_UNLOCK(sd_dev);
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      write data to the specified address of a card
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  pdata: a pointer that store multiple blocks data to be transferred
    \param[in]  blockaddr: the write data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be written
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY,HAL_ERR_ABORT,HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_writeblocks_poll(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                         uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint8_t cardstate        = 0U;
    uint32_t count = 0U, align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE, *ptempbuff = pdata;
    uint32_t transbytes = 0U, restwords = 0U, response = 0U;
    __IO uint32_t timeout = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pdata address */
    if((NULL == sd_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [sd_dev] or [pdata] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_READY != sd_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > SD_MAX_DATA_LENGTH) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(sd_dev);

        sd_dev->state = HAL_SD_STATE_BUSY;

        /* clear all DSM configuration */
        hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(sd_dev->periph);
        hals_sdio_idma_disable(sd_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
            sd_dev->error_state = HAL_SD_LOCK_UNLOCK_FAILED;
            sd_dev->state = HAL_SD_STATE_READY;
            HAL_UNLOCK(sd_dev);
            reval = HAL_ERR_ABORT;
        } else {

            /* blocksize is fixed in 512B for SDHC card */
            if(SDIO_HIGH_CAPACITY_SD_CARD != sd_dev->card_info.card_type) {
                blockaddr *= 512U;
            } else {
                blocksize = 512U;
            }

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _sd_datablocksize_get(blocksize);
                /* send CMD16(SET_BLOCKLEN) to set the block length */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, (uint32_t)blocksize, \
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    /* do nothing */
                }
            } else {
                sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
                sd_dev->state = HAL_SD_STATE_READY;
                HAL_UNLOCK(sd_dev);
                reval = HAL_ERR_ABORT;
            }

            if(HAL_ERR_NONE == reval) {

                /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS,
                                                (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SEND_STATUS);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {

                    response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                    timeout  = 100000U;

                    while((0U == (response & SD_R1_READY_FOR_DATA)) && (timeout > 0U)) {
                        /* continue to send CMD13 to polling the state of card until buffer empty or timeout */
                        --timeout;
                        /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                        hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS, \
                                                        (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, \
                                                        SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(sd_dev->periph);
                        /* check if some error occurs */
                        status = _r1_error_check(sd_dev, SD_CMD_SEND_STATUS);
                        if(HAL_SD_OK != status) {
                            sd_dev->state = HAL_SD_STATE_READY;
                            HAL_UNLOCK(sd_dev);
                            reval = HAL_ERR_ABORT;
                            break;
                        } else{
                            response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                        }
                    }
                    if (HAL_ERR_NONE == reval){
                        if(0U == timeout) {
                            sd_dev->state = HAL_SD_STATE_READY;
                            HAL_UNLOCK(sd_dev);
                            reval = HAL_ERR_TIMEOUT;
                        } else {
                            /* do nothing */
                        }
                        if (HAL_ERR_NONE == reval) {
                            if(blocknumber > 1U) {
                                if((SDIO_STD_CAPACITY_SD_CARD_V1_1 == sd_dev->card_info.card_type) || \
                                (SDIO_STD_CAPACITY_SD_CARD_V2_0 == sd_dev->card_info.card_type) || \
                                (SDIO_HIGH_CAPACITY_SD_CARD == sd_dev->card_info.card_type)) {
                                    /* send CMD55(APP_CMD) to indicate next command is application specific command */
                                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_APP_CMD, \
                                                                    (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, \
                                                                    SDIO_RESPONSETYPE_SHORT);
                                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                    hals_sdio_csm_enable(sd_dev->periph);
                                    /* check if some error occurs */
                                    status = _r1_error_check(sd_dev, SD_CMD_APP_CMD);
                                    if(HAL_SD_OK != status) {
                                        sd_dev->state = HAL_SD_STATE_READY;
                                        HAL_UNLOCK(sd_dev);
                                        reval = HAL_ERR_ABORT;
                                    } else {

                                        /* send ACMD23(SET_WR_BLK_ERASE_COUNT) to set the number of write blocks to be preerased before writing */
                                        hals_sdio_command_response_config(sd_dev->periph, SD_APPCMD_SET_WR_BLK_ERASE_COUNT, (uint32_t)blocknumber, \
                                                                        SDIO_RESPONSETYPE_SHORT);
                                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                        hals_sdio_csm_enable(sd_dev->periph);
                                        /* check if some error occurs */
                                        status = _r1_error_check(sd_dev, SD_APPCMD_SET_WR_BLK_ERASE_COUNT);
                                        if(HAL_SD_OK != status) {
                                            sd_dev->state = HAL_SD_STATE_READY;
                                            HAL_UNLOCK(sd_dev);
                                            reval = HAL_ERR_ABORT;
                                        } else {
                                            /* do nothing */
                                        }
                                    }
                                }
                            } else {
                                /* do nothing */
                            }

                            /* configure the SDIO data transmission */
                            hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, (uint32_t)blocknumber * blocksize, datablksize);
                            hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
                            hals_sdio_trans_start_enable(sd_dev->periph);

                            if(blocknumber > 1U) {
                                sd_dev->transfer = SD_TRANSFER_WRITE_MULTIPLE_BLOCK;
                                /* send CMD25(WRITE_MULTIPLE_BLOCK) to continuously write blocks of data */
                                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_WRITE_MULTIPLE_BLOCK, blockaddr, \
                                                                SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(sd_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(sd_dev, SD_CMD_WRITE_MULTIPLE_BLOCK);
                                if(HAL_SD_OK != status) {
                                    sd_dev->state    = HAL_SD_STATE_READY;
                                    sd_dev->transfer = SD_TRANSFER_NONE;
                                    HAL_UNLOCK(sd_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                sd_dev->transfer = SD_TRANSFER_WRITE_SINGLE_BLOCK;
                                /* send CMD24(WRITE_BLOCK) to write a block */
                                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_WRITE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(sd_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(sd_dev, SD_CMD_WRITE_BLOCK);
                                if(HAL_SD_OK != status) {
                                    sd_dev->state    = HAL_SD_STATE_READY;
                                    sd_dev->transfer = SD_TRANSFER_NONE;
                                    HAL_UNLOCK(sd_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* do nothing */
                                }
                            }
                            if (HAL_ERR_NONE == reval){
                                /* polling mode */
                                while(!hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR | SDIO_FLAG_DTTMOUT | SDIO_FLAG_TXURE | \
                                                                            SDIO_FLAG_DTBLKEND | SDIO_FLAG_DTEND)) {
                                    if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_TFH)) {
                                        /* at least 8 words can be written into the FIFO */
                                        if((((uint32_t)blocknumber * blocksize) - transbytes) < SD_FIFOHALF_BYTES) {
                                            restwords = (((uint32_t)blocknumber * blocksize) - transbytes + 3U) / 4U;
                                            for(count = 0U; count < restwords; count++) {
                                                hals_sdio_data_write(sd_dev->periph, *ptempbuff);
                                                ++ptempbuff;
                                            }
                                            ptempbuff -= restwords;
                                        } else {
                                            for(count = 0U; count < SD_FIFOHALF_WORDS; count++) {
                                                hals_sdio_data_write(sd_dev->periph, *(ptempbuff + count));
                                            }
                                            /* 8 words(32 bytes) has been transferred */
                                            ptempbuff += SD_FIFOHALF_WORDS;
                                            transbytes += SD_FIFOHALF_BYTES;
                                        }
                                    } else {
                                        /* do nothing */
                                    }
                                }

                                hals_sdio_trans_start_disable(sd_dev->periph);

                                /* whether some error occurs and return it */
                                if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR)) {
                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTCRCERR);
                                    sd_dev->error_state = HAL_SD_DATA_CRC_ERROR;
                                    sd_dev->state    = HAL_SD_STATE_READY;
                                    sd_dev->transfer = SD_TRANSFER_NONE;
                                    HAL_UNLOCK(sd_dev);
                                    reval = HAL_ERR_ABORT;
                                } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTTMOUT)) {
                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTTMOUT);
                                    sd_dev->error_state = HAL_SD_DATA_TIMEOUT;
                                    sd_dev->state    = HAL_SD_STATE_READY;
                                    sd_dev->transfer = SD_TRANSFER_NONE;
                                    HAL_UNLOCK(sd_dev);
                                    reval = HAL_ERR_ABORT;
                                } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_TXURE)) {
                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_TXURE);
                                    sd_dev->error_state = HAL_SD_TX_UNDERRUN_ERROR;
                                    sd_dev->state    = HAL_SD_STATE_READY;
                                    sd_dev->transfer = SD_TRANSFER_NONE;
                                    HAL_UNLOCK(sd_dev);
                                    reval = HAL_ERR_ABORT;
                                } else if((RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTEND)) && (blocknumber > 1U)) {
                                    if((SDIO_STD_CAPACITY_SD_CARD_V1_1 == sd_dev->card_info.card_type) || \
                                    (SDIO_STD_CAPACITY_SD_CARD_V2_0 == sd_dev->card_info.card_type) || \
                                    (SDIO_HIGH_CAPACITY_SD_CARD == sd_dev->card_info.card_type)) {
                                        /* send CMD12(STOP_TRANSMISSION) to stop transmission */
                                        hals_sdio_command_response_config(sd_dev->periph, SD_CMD_STOP_TRANSMISSION, 0U, SDIO_RESPONSETYPE_SHORT);
                                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                        hals_sdio_csm_enable(sd_dev->periph);
                                        /* check if some error occurs */
                                        status = _r1_error_check(sd_dev, SD_CMD_STOP_TRANSMISSION);
                                        if(HAL_SD_OK != status) {
                                            sd_dev->state    = HAL_SD_STATE_READY;
                                            sd_dev->transfer = SD_TRANSFER_NONE;
                                            HAL_UNLOCK(sd_dev);
                                            reval = HAL_ERR_ABORT;
                                        } else {
                                            /* do nothing */
                                        }
                                    } else {
                                        /* do nothing */
                                    }
                                } else {
                                    /* do nothing */
                                }
                                if (HAL_ERR_NONE == reval){
                                    /* clear the DATA_FLAGS flags */
                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_DATA_FLAGS);
                                    /* get the card state and wait the card is out of programming and receiving state */
                                    status = _sdcard_cardstate_get(sd_dev, &cardstate);
                                    while((HAL_SD_OK == status) && ((SD_CARDSTATE_PROGRAMMING == cardstate) || (SD_CARDSTATE_RECEIVING == cardstate))) {
                                        status = _sdcard_cardstate_get(sd_dev, &cardstate);
                                    }

                                    sd_dev->state    = HAL_SD_STATE_READY;
                                    sd_dev->transfer = SD_TRANSFER_NONE;

                                    /* process unlocked */
                                    HAL_UNLOCK(sd_dev);
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                /* do nothing */
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      write data to the specified address of a card by interrupt method
    \param[in]  sd_dev: sdcard device information structure
                 the structure is not necessary to be reconfigured after structure initialization,
                 the structure parameters altering is automatically configured by core
    \param[in]  pdata: a pointer that store multiple blocks data to be transferred
    \param[in]  blockaddr: the write data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be written
    \param[in]  p_user_func: pointer to the sdcard irq user callback struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ABORT, HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_writeblocks_interrupt(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, \
                                              uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                              hal_sdio_sdcard_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
    uint32_t response     = 0U;
    __IO uint32_t timeout = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pdata address */
    if((NULL == sd_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [sd_dev] or [pdata] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_READY != sd_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > SD_MAX_DATA_LENGTH) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(sd_dev);

        sd_dev->state = HAL_SD_STATE_BUSY;
        /* blocksize is fixed in 512B for SDHC card */
        if(SDIO_HIGH_CAPACITY_SD_CARD != sd_dev->card_info.card_type) {
            blockaddr *= 512U;
        } else {
            blocksize = 512U;
        }

        sd_dev->p_txbuffer = pdata;
        sd_dev->tx_length  = (uint32_t)blocksize * blocknumber;

        sd_dev->sd_irq.transmit_fifo_half_empty_handle = &_sd_write_interrupt;
        sd_dev->sd_irq.transfer_complete_handle        = &_sd_transmit_received_handle;
        sd_dev->sd_irq.transfer_error_handle           = &_sd_error_handle;

        sd_dev->transmit_complete_callback        = NULL;
        sd_dev->transfer_error_callback           = NULL;
        sd_dev->abort_complete_callback           = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->transmit_complete_func) {
                sd_dev->transmit_complete_callback = (void *)p_user_func->transmit_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                sd_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                sd_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(sd_dev->periph);
        hals_sdio_idma_disable(sd_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
            sd_dev->error_state = HAL_SD_LOCK_UNLOCK_FAILED;
            sd_dev->state = HAL_SD_STATE_READY;
            HAL_UNLOCK(sd_dev);
            reval = HAL_ERR_ABORT;
        } else {

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _sd_datablocksize_get(blocksize);
                /* send CMD16(SET_BLOCKLEN) to set the block length */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, (uint32_t)blocksize, \
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    /* do nothing */
                }
            } else {
                sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
                sd_dev->state = HAL_SD_STATE_READY;
                HAL_UNLOCK(sd_dev);
                reval = HAL_ERR_ABORT;
            }
            if(HAL_ERR_NONE == reval) {
                /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS,
                                                (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SEND_STATUS);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {

                    response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                    timeout  = 100000U;

                    while((0U == (response & SD_R1_READY_FOR_DATA)) && (timeout > 0U)) {
                        /* continue to send CMD13 to polling the state of card until buffer empty or timeout */
                        --timeout;
                        /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                        hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS, \
                                                        (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, \
                                                        SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(sd_dev->periph);
                        /* check if some error occurs */
                        status = _r1_error_check(sd_dev, SD_CMD_SEND_STATUS);
                        if(HAL_SD_OK != status) {
                            sd_dev->state = HAL_SD_STATE_READY;
                            reval = HAL_ERR_ABORT;
                            break;
                        } else {
                            response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                        }
                    }

                    if(0U == timeout) {
                        sd_dev->state = HAL_SD_STATE_READY;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_TIMEOUT;
                    } else {
                        /* do nothing */
                    }
                    if(HAL_ERR_NONE == reval) {
                        if(blocknumber > 1U) {
                            if((SDIO_STD_CAPACITY_SD_CARD_V1_1 == sd_dev->card_info.card_type) || \
                            (SDIO_STD_CAPACITY_SD_CARD_V2_0 == sd_dev->card_info.card_type) || \
                            (SDIO_HIGH_CAPACITY_SD_CARD == sd_dev->card_info.card_type)) {
                                /* send CMD55(APP_CMD) to indicate next command is application specific command */
                                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_APP_CMD, \
                                                                (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, \
                                                                SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(sd_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(sd_dev, SD_CMD_APP_CMD);
                                if(HAL_SD_OK != status) {
                                    sd_dev->state = HAL_SD_STATE_READY;
                                    /* process unlocked */
                                    HAL_UNLOCK(sd_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {

                                    /* send ACMD23(SET_WR_BLK_ERASE_COUNT) to set the number of write blocks to be preerased before writing */
                                    hals_sdio_command_response_config(sd_dev->periph, SD_APPCMD_SET_WR_BLK_ERASE_COUNT, (uint32_t)blocknumber, \
                                                                    SDIO_RESPONSETYPE_SHORT);
                                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                    hals_sdio_csm_enable(sd_dev->periph);
                                    /* check if some error occurs */
                                    status = _r1_error_check(sd_dev, SD_APPCMD_SET_WR_BLK_ERASE_COUNT);
                                    if(HAL_SD_OK != status) {
                                        sd_dev->state = HAL_SD_STATE_READY;
                                        HAL_UNLOCK(sd_dev);
                                        reval = HAL_ERR_ABORT;
                                    } else {
                                        /* do nothing */
                                    }
                                }
                            } else {
                                /* do nothing */
                            }
                        } else {
                            /* do nothing */
                        }

                        if(HAL_ERR_NONE == reval) {
                            /* configure the SDIO data transmission */
                            hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, (uint32_t)blocknumber * blocksize, datablksize);
                            hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
                            hals_sdio_trans_start_enable(sd_dev->periph);

                            if(blocknumber > 1U) {
                                sd_dev->transfer = (SD_TRANSFER_WRITE_MULTIPLE_BLOCK | SD_TRANSFER_IT);
                                /* send CMD25(WRITE_MULTIPLE_BLOCK) to continuously write blocks of data */
                                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_WRITE_MULTIPLE_BLOCK, blockaddr, \
                                                                SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(sd_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(sd_dev, SD_CMD_WRITE_MULTIPLE_BLOCK);
                            } else {
                                sd_dev->transfer = (SD_TRANSFER_WRITE_SINGLE_BLOCK | SD_TRANSFER_IT);
                                /* send CMD24(WRITE_BLOCK) to write a block */
                                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_WRITE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(sd_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(sd_dev, SD_CMD_WRITE_BLOCK);
                            }

                            if(HAL_SD_OK != status) {
                                /* clear the DATA_FLAGS flags */
                                hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);
                                sd_dev->state    = HAL_SD_STATE_READY;
                                sd_dev->transfer = SD_TRANSFER_NONE;
                                HAL_UNLOCK(sd_dev);
                                reval = HAL_ERR_ABORT;
                            } else {
                                /* enable the SDIO corresponding interrupts and DMA */
                                hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                                                        SDIO_INT_TFH | SDIO_INT_TXURE | SDIO_INT_DTEND);

                                /* process unlocked */
                                HAL_UNLOCK(sd_dev);
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      read data into a buffer from the specified address of a card by dma method
    \param[in]  sd_dev: sdcard device information structure
                the structure is not necessary to be reconfigured after structure initialization,
                the structure parameters altering is automatically configured by core
    \param[in]  blockaddr: the read data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be read
    \param[in]  p_user_func: pointer to the sdcard irq user callback struct
    \param[out] pdata: a pointer that store multiple blocks read data
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_writeblocks_dma(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pdata, uint32_t blockaddr, \
                                        uint16_t blocksize, uint16_t blocknumber, \
                                        hal_sdio_sdcard_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
    uint32_t response     = 0U;
    __IO uint32_t timeout = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pdata address */
    if((NULL == sd_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [sd_dev] or [pdata] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_READY != sd_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > SD_MAX_DATA_LENGTH) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(sd_dev);

        sd_dev->state = HAL_SD_STATE_BUSY;

        blocksize = 512U;
        sd_dev->p_txbuffer = pdata;
        sd_dev->tx_length = (uint32_t)blocksize * blocknumber;

        sd_dev->sd_irq.transmit_fifo_half_empty_handle = &_sd_write_interrupt;
        sd_dev->sd_irq.transfer_complete_handle = &_sd_transmit_received_handle;
        sd_dev->sd_irq.transfer_error_handle = &_sd_error_handle;

        sd_dev->transmit_complete_callback = NULL;
        sd_dev->transfer_error_callback = NULL;
        sd_dev->abort_complete_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->transmit_complete_func) {
                sd_dev->transmit_complete_callback = (void *)p_user_func->transmit_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                sd_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                sd_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(sd_dev->periph);
        hals_sdio_idma_disable(sd_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
            sd_dev->error_state = HAL_SD_LOCK_UNLOCK_FAILED;
            sd_dev->state = HAL_SD_STATE_READY;
            HAL_UNLOCK(sd_dev);
            reval = HAL_ERR_ABORT;
        } else {

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _sd_datablocksize_get(blocksize);
                /* send CMD16(SET_BLOCKLEN) to set the block length */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, (uint32_t)blocksize, \
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    /* do nothing */
                }
            } else {
                sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
                sd_dev->state = HAL_SD_STATE_READY;
                HAL_UNLOCK(sd_dev);
                reval = HAL_ERR_ABORT;
            }
            if(HAL_ERR_NONE == reval) {

                /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS, \
                                                (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SEND_STATUS);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                    timeout  = 100000U;

                    while((0U == (response & SD_R1_READY_FOR_DATA)) && (timeout > 0U)) {
                        /* continue to send CMD13 to polling the state of card until buffer empty or timeout */
                        --timeout;
                        /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                        hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS, \
                                                        (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, \
                                                        SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(sd_dev->periph);
                        /* check if some error occurs */
                        status = _r1_error_check(sd_dev, SD_CMD_SEND_STATUS);
                        if(HAL_SD_OK != status) {
                            sd_dev->state = HAL_SD_STATE_READY;
                            HAL_UNLOCK(sd_dev);
                            reval = HAL_ERR_ABORT;
                            break;
                        } else {
                            response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                        }
                    }
                    if(HAL_ERR_NONE == reval) {
                        if(0U == timeout) {
                            sd_dev->state = HAL_SD_STATE_READY;
                            HAL_UNLOCK(sd_dev);
                            reval = (int32_t)HAL_SD_ERROR;
                        } else {
                            /* do nothing */
                        }
                        if(HAL_ERR_NONE == reval) {
                            if(blocknumber > 1U) {
                                if((SDIO_STD_CAPACITY_SD_CARD_V1_1 == sd_dev->card_info.card_type) || \
                                (SDIO_STD_CAPACITY_SD_CARD_V2_0 == sd_dev->card_info.card_type) || \
                                (SDIO_HIGH_CAPACITY_SD_CARD == sd_dev->card_info.card_type)) {
                                    /* send CMD55(APP_CMD) to indicate next command is application specific command */
                                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_APP_CMD, \
                                                                    (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT,
                                                                    SDIO_RESPONSETYPE_SHORT);
                                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                    hals_sdio_csm_enable(sd_dev->periph);
                                    /* check if some error occurs */
                                    status = _r1_error_check(sd_dev, SD_CMD_APP_CMD);
                                    if(HAL_SD_OK != status) {
                                        sd_dev->state = HAL_SD_STATE_READY;
                                        HAL_UNLOCK(sd_dev);
                                        reval = HAL_ERR_ABORT;
                                    } else {
                                        /* send ACMD23(SET_WR_BLK_ERASE_COUNT) to set the number of write blocks to be preerased before writing */
                                        hals_sdio_command_response_config(sd_dev->periph, SD_APPCMD_SET_WR_BLK_ERASE_COUNT, (uint32_t)blocknumber, \
                                                                        SDIO_RESPONSETYPE_SHORT);
                                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                        hals_sdio_csm_enable(sd_dev->periph);
                                        /* check if some error occurs */
                                        status = _r1_error_check(sd_dev, SD_APPCMD_SET_WR_BLK_ERASE_COUNT);
                                        if(HAL_SD_OK != status) {
                                            sd_dev->state = HAL_SD_STATE_READY;
                                            HAL_UNLOCK(sd_dev);
                                            reval = HAL_ERR_ABORT;
                                        } else {
                                            /* do nothing */
                                        }
                                    }
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                /* do nothing */
                            }
                            if(HAL_ERR_NONE == reval) {
                                /* DMA mode */
                                hals_sdio_idma_set(sd_dev->periph, SDIO_IDMA_SINGLE_BUFFER, (uint32_t)((uint32_t)blocksize >> 5U));
                                hals_sdio_idma_buffer0_address_set(sd_dev->periph, (uint32_t)pdata);
                                hals_sdio_idma_enable(sd_dev->periph);

                                /* configure the SDIO data transmission */
                                hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, (uint32_t)blocknumber * blocksize, datablksize);
                                hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
                                hals_sdio_trans_start_enable(sd_dev->periph);

                                if(blocknumber > 1U) {
                                    sd_dev->transfer = (SD_TRANSFER_WRITE_MULTIPLE_BLOCK | SD_TRANSFER_DMA);
                                    /* send CMD25(WRITE_MULTIPLE_BLOCK) to continuously write blocks of data */
                                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_WRITE_MULTIPLE_BLOCK, blockaddr, \
                                                                    SDIO_RESPONSETYPE_SHORT);
                                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                    hals_sdio_csm_enable(sd_dev->periph);
                                    /* check if some error occurs */
                                    status = _r1_error_check(sd_dev, SD_CMD_WRITE_MULTIPLE_BLOCK);
                                } else {
                                    sd_dev->transfer = (SD_TRANSFER_WRITE_SINGLE_BLOCK | SD_TRANSFER_DMA);
                                    /* send CMD24(WRITE_BLOCK) to write a block */
                                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_WRITE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                                    hals_sdio_csm_enable(sd_dev->periph);
                                    /* check if some error occurs */
                                    status = _r1_error_check(sd_dev, SD_CMD_WRITE_BLOCK);
                                }

                                if(HAL_SD_OK != status) {
                                    /* clear the DATA_FLAGS flags */
                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);
                                    sd_dev->state    = HAL_SD_STATE_READY;
                                    sd_dev->transfer = SD_TRANSFER_NONE;
                                    HAL_UNLOCK(sd_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* enable the SDIO corresponding interrupts and DMA */
                                    hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_TXURE | SDIO_INT_DTEND);
                                    /* process unlocked */
                                    HAL_UNLOCK(sd_dev);
                                }
                            } else {
                            /* do nothing */
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      write data to the specified address of a card by dma method
    \param[in]  sd_dev: sdcard device information structure
                 the structure is not necessary to be reconfigured after structure initialization,
                 the structure parameters altering is automatically configured by core
    \param[in]  pbuf0: the idma buffer 0
    \param[in]  pbuf1: the idma buffer 1
    \param[in]  blockaddr: the write data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be written
    \param[in]  p_user_func: pointer to the sdcard irq user callback struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_writeblocks_multibuffer_dma(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pbuf0, uint32_t *pbuf1, \
                                                    uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                                    hal_sdio_sdcard_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pbuf address */
    if((NULL == sd_dev) || (NULL == pbuf0) || (NULL == pbuf1)) {
        HAL_DEBUGE("pointer [sd_dev] or [pbuf0] or [pbuf1] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_READY != sd_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pbuf0) || (NULL == pbuf1) || (0U == blocknumber)) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > SD_MAX_DATA_LENGTH) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(sd_dev);

        sd_dev->state = HAL_SD_STATE_BUSY;

        sd_dev->sd_irq.transmit_fifo_half_empty_handle = &_sd_write_interrupt;
        sd_dev->sd_irq.transfer_complete_handle = &_sd_transmit_received_handle;
        sd_dev->sd_irq.idma_complete_handle = &_sd_idma_complete_handle;
        sd_dev->sd_irq.transfer_error_handle = &_sd_error_handle;

        sd_dev->write_dma_buffer0_complete_callback = NULL;
        sd_dev->write_dma_buffer1_complete_callback = NULL;
        sd_dev->abort_complete_callback = NULL;
        sd_dev->transfer_error_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->write_dma_buffer0_complete_func) {
                sd_dev->write_dma_buffer0_complete_callback = (void *)p_user_func->write_dma_buffer0_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->write_dma_buffer1_complete_func) {
                sd_dev->write_dma_buffer1_complete_callback = (void *)p_user_func->write_dma_buffer1_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                sd_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                sd_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(sd_dev->periph);
        hals_sdio_idma_disable(sd_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
            status        = HAL_SD_LOCK_UNLOCK_FAILED;
            sd_dev->state = HAL_SD_STATE_READY;
            HAL_UNLOCK(sd_dev);
            reval = HAL_ERR_ABORT;
        } else {

            blocksize = 512U;
            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _sd_datablocksize_get(blocksize);
                /* send CMD16(SET_BLOCKLEN) to set the block length */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, (uint32_t)blocksize, \
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                /* do nothing */
            }
            } else {
                sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
                sd_dev->state = HAL_SD_STATE_READY;
                HAL_UNLOCK(sd_dev);
                reval = HAL_ERR_ABORT;
            }
            if (HAL_ERR_NONE == reval){
                /* DMA mode */
                hals_sdio_idma_set(sd_dev->periph, SDIO_IDMA_DOUBLE_BUFFER, (uint32_t)((uint32_t)blocksize >> 5U));
                hals_sdio_idma_buffer0_address_set(sd_dev->periph, (uint32_t)pbuf0);
                hals_sdio_idma_buffer1_address_set(sd_dev->periph, (uint32_t)pbuf1);
                hals_sdio_idma_enable(sd_dev->periph);

                /* configure the SDIO data transmission */
                hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, (uint32_t)blocknumber * blocksize, datablksize);
                hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
                hals_sdio_trans_start_enable(sd_dev->periph);

                sd_dev->transfer = (SD_TRANSFER_WRITE_MULTIPLE_BLOCK | SD_TRANSFER_DMA);
                /* send CMD25(WRITE_MULTIPLE_BLOCK) to continuously write blocks of data */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_WRITE_MULTIPLE_BLOCK, blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_WRITE_MULTIPLE_BLOCK);
                if(HAL_SD_OK != status) {
                    /* clear the DATA_FLAGS flags */
                    hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);
                    sd_dev->state = HAL_SD_STATE_READY;
                    sd_dev->transfer = SD_TRANSFER_NONE;
                    HAL_UNLOCK(sd_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    /* enable the SDIO corresponding interrupts and DMA */
                    hals_sdio_interrupt_enable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_TXURE | \
                                                            SDIO_INT_DTEND | SDIO_INT_IDMAERR | SDIO_INT_IDMAEND);

                    /* process unlocked */
                    HAL_UNLOCK(sd_dev);
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      erase a continuous area of a card
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  startaddr: the start address
    \param[in]  endaddr: the end address
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_erase(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t startaddr, uint32_t endaddr)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t count = 0U, clkdiv = 0U;
    __IO uint32_t delay = 0U;
    uint8_t cardstate = 0U, tempbyte = 0U;
    uint16_t tempccc = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer address */
    if((NULL == sd_dev)) {
        HAL_DEBUGE("pointer [sd_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_READY != sd_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if(endaddr < startaddr) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(sd_dev);

        sd_dev->state = HAL_SD_STATE_BUSY;

        /* get the card command classes from CSD */
        tempbyte = (uint8_t)((sd_dev->csd_data[1] & SD_MASK_24_31BITS) >> 24U);
        tempccc  = (uint16_t)((uint16_t)tempbyte << 4U);
        tempbyte = (uint8_t)((sd_dev->csd_data[1] & SD_MASK_16_23BITS) >> 16U);
        tempccc |= ((uint16_t)tempbyte & 0xF0U) >> 4U;
        if(0U == (tempccc & SD_CCC_ERASE)) {
            /* don't support the erase command */
            sd_dev->error_state = HAL_SD_FUNCTION_UNSUPPORTED;
            sd_dev->state = HAL_SD_STATE_READY;
            HAL_UNLOCK(sd_dev);
            reval = HAL_ERR_ABORT;
        } else {

            clkdiv = (SDIO_CLKCTL(sd_dev->periph) & SDIO_CLKCTL_DIV);
            if(0U == clkdiv) {
               clkdiv = 1U;
            } else {
               clkdiv *= 2U;
            }
            delay = 168000U / clkdiv;

            /* check whether the card is locked */
            if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
                sd_dev->error_state = HAL_SD_LOCK_UNLOCK_FAILED;
                sd_dev->state = HAL_SD_STATE_READY;
                HAL_UNLOCK(sd_dev);
                reval = HAL_ERR_ABORT;
            } else {

                /* blocksize is fixed in 512B for SDHC card */
                if(SDIO_HIGH_CAPACITY_SD_CARD != sd_dev->card_info.card_type) {
                    startaddr *= 512U;
                    endaddr *= 512U;
                } else {
                    /* do nothing */
                }

                if((SDIO_STD_CAPACITY_SD_CARD_V1_1 == sd_dev->card_info.card_type) || \
                (SDIO_STD_CAPACITY_SD_CARD_V2_0 == sd_dev->card_info.card_type) || \
                (SDIO_HIGH_CAPACITY_SD_CARD == sd_dev->card_info.card_type)) {
                    /* send CMD32(ERASE_WR_BLK_START) to set the address of the first write block to be erased */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_ERASE_WR_BLK_START, startaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);
                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_ERASE_WR_BLK_START);
                    if(HAL_SD_OK != status) {
                        sd_dev->state = HAL_SD_STATE_READY;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_ABORT;
                    } else {

                        /* send CMD33(ERASE_WR_BLK_END) to set the address of the last write block of the continuous range to be erased */
                        hals_sdio_command_response_config(sd_dev->periph, SD_CMD_ERASE_WR_BLK_END, endaddr, SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(sd_dev->periph);
                        /* check if some error occurs */
                        status = _r1_error_check(sd_dev, SD_CMD_ERASE_WR_BLK_END);
                        if(HAL_SD_OK != status) {
                            sd_dev->state = HAL_SD_STATE_READY;
                            HAL_UNLOCK(sd_dev);
                            reval = HAL_ERR_ABORT;
                        } else {
                            /* do nothing */
                        }
                    }
                } else {
                    /* do nothing */
                }
                if (HAL_ERR_NONE == reval){

                    /* send CMD38(ERASE) to set the address of the first write block to be erased */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_ERASE, 0U, SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);

                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_ERASE);
                    if(HAL_SD_OK != status) {
                        sd_dev->state = HAL_SD_STATE_READY;
                        HAL_UNLOCK(sd_dev);
                        reval = HAL_ERR_ABORT;
                    } else {

                        /* loop until the counter is reach to the calculated time */
                        for(count = 0U; count < delay; count++) {
                            /* empty loop */
                        }

                        /* get the card state and wait the card is out of programming and receiving state */
                        status = _sdcard_cardstate_get(sd_dev, &cardstate);
                        while((HAL_SD_OK == status) && ((SD_CARDSTATE_PROGRAMMING == cardstate) || (SD_CARDSTATE_RECEIVING == cardstate))) {
                            status = _sdcard_cardstate_get(sd_dev, &cardstate);
                        }

                        sd_dev->state = HAL_SD_STATE_READY;

                        /* process unlocked */
                        HAL_UNLOCK(sd_dev);
                    }
                } else {
                    /* do nothing */
                }
            }
        }
    }

    return reval;
}

/*!
    \brief      abort an ongoing data transfer
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_abort(hal_sdio_sdcard_dev_struct *sd_dev)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer address */
    if((NULL == sd_dev)) {
        HAL_DEBUGE("pointer [sd_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SD_STATE_BUSY == sd_dev->state) {
        /* disable the SDIO corresponding interrupts */
        hals_sdio_interrupt_disable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_RXORE | \
                                                    SDIO_INT_TXURE | SDIO_INT_DTEND);
        hals_sdio_trans_start_disable(sd_dev->periph);
        status = _sd_transfer_stop(sd_dev);
        if(HAL_SD_OK != status) {
            reval = HAL_ERR_VAL;
        } else {
            hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DAT0BSYEND);
            hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_DATA_FLAGS);
            hals_sdio_idma_disable(sd_dev->periph);

            sd_dev->state = HAL_SD_STATE_READY;
            sd_dev->transfer = SD_TRANSFER_NONE;
        }
    } else {
        /* do nothing */
    }

    return reval;
}

/*!
    \brief      abort an ongoing data transfer
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_abort_interrupt(hal_sdio_sdcard_dev_struct *sd_dev)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    hal_sdio_sdcard_user_cb p_func = NULL;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer address */
    if((NULL == sd_dev)) {
        HAL_DEBUGE("pointer [sd_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable the SDIO corresponding interrupts */
    hals_sdio_interrupt_disable(sd_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_RXORE | \
                                                SDIO_INT_TXURE | SDIO_INT_DTEND);
    hals_sdio_idma_disable(sd_dev->periph);

    /* clear all the SDIO_INTC flags */
    hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_DATA_FLAGS);

    status = _sd_transfer_stop(sd_dev);
    if(HAL_SD_OK != status) {
        reval = HAL_ERR_VAL;
    } else {
        p_func = (hal_sdio_sdcard_user_cb)sd_dev->abort_complete_callback;
        if(NULL != p_func) {
            p_func(sd_dev);
        } else {
            /* do nothing */
        }
    }

    sd_dev->state = HAL_SD_STATE_READY;
    sd_dev->transfer = SD_TRANSFER_NONE;

    return reval;
}

/*!
    \brief      lock and unlock sdcard
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  lockstate: the lock state
                only one parameter can be selected which is shown as below:
      \arg        SD_LOCK: lock the SD card
      \arg        SD_UNLOCK: unlock the SD card
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_lock_unlock(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t lockstate)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;

#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer address */
    if((NULL == sd_dev)) {
        HAL_DEBUGE("pointer [sd_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    status = _sdcard_lock_unlock(sd_dev, lockstate);
    if(HAL_SD_OK != status) {
        reval = HAL_ERR_ABORT;
    } else {
        reval = HAL_ERR_NONE;
    }
    return reval;
}

/*!
    \brief      get the state which the card is in
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] pcardstate: a pointer that store the card state
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_cardstate_get(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t *pcardstate)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;

#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pcardstate address */
    if((NULL == sd_dev) || (NULL == pcardstate)) {
        HAL_DEBUGE("pointer [sd_dev] or [pcardstate] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    status = _sdcard_cardstate_get(sd_dev, pcardstate);
    if(HAL_SD_OK == status) {
        reval = HAL_ERR_NONE;
    } else {
        reval = HAL_ERR_ABORT;
    }
    return reval;
}

/*!
    \brief      get the information of the card which are stored on
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_cid: sdcard cid data information structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_cid_get(hal_sdio_sdcard_dev_struct *sd_dev, hal_sdio_sdcard_card_cid_struct *p_cid)
{
    uint8_t tempbyte = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and p_cid address */
    if((NULL == sd_dev) || (NULL == p_cid)) {
        HAL_DEBUGE("pointer [sd_dev] or [p_cid] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* CID byte 0 */
    tempbyte = (uint8_t)((sd_dev->cid_data[0] & SD_MASK_24_31BITS) >> 24U);
    p_cid->mid = tempbyte;

    /* CID byte 1 */
    tempbyte = (uint8_t)((sd_dev->cid_data[0] & SD_MASK_16_23BITS) >> 16U);
    p_cid->oid = (uint16_t)((uint16_t)tempbyte << 8U);

    /* CID byte 2 */
    tempbyte = (uint8_t)((sd_dev->cid_data[0] & SD_MASK_8_15BITS) >> 8U);
    p_cid->oid |= (uint16_t)tempbyte;

    /* CID byte 3 */
    tempbyte = (uint8_t)(sd_dev->cid_data[0] & SD_MASK_0_7BITS);
    p_cid->pnm0 = (uint32_t)((uint32_t)tempbyte << 24U);

    /* CID byte 4 */
    tempbyte = (uint8_t)((sd_dev->cid_data[1] & SD_MASK_24_31BITS) >> 24U);
    p_cid->pnm0 |= (uint32_t)((uint32_t)tempbyte << 16U);

    /* CID byte 5 */
    tempbyte = (uint8_t)((sd_dev->cid_data[1] & SD_MASK_16_23BITS) >> 16U);
    p_cid->pnm0 |= (uint32_t)((uint32_t)tempbyte << 8U);

    /* CID byte 6 */
    tempbyte = (uint8_t)((sd_dev->cid_data[1] & SD_MASK_8_15BITS) >> 8U);
    p_cid->pnm0 |= (uint32_t)(tempbyte);

    /* CID byte 7 */
    tempbyte = (uint8_t)(sd_dev->cid_data[1] & SD_MASK_0_7BITS);
    p_cid->pnm1 = tempbyte;

    /* CID byte 8 */
    tempbyte = (uint8_t)((sd_dev->cid_data[2] & SD_MASK_24_31BITS) >> 24U);
    p_cid->prv = tempbyte;

    /* CID byte 9 */
    tempbyte = (uint8_t)((sd_dev->cid_data[2] & SD_MASK_16_23BITS) >> 16U);
    p_cid->psn = (uint32_t)((uint32_t)tempbyte << 24U);

    /* CID byte 10 */
    tempbyte = (uint8_t)((sd_dev->cid_data[2] & SD_MASK_8_15BITS) >> 8U);
    p_cid->psn |= (uint32_t)((uint32_t)tempbyte << 16U);

    /* CID byte 11 */
    tempbyte = (uint8_t)(sd_dev->cid_data[2] & SD_MASK_0_7BITS);
    p_cid->psn |= (uint32_t)tempbyte;

    /* CID byte 12 */
    tempbyte = (uint8_t)((sd_dev->cid_data[3] & SD_MASK_24_31BITS) >> 24U);
    p_cid->psn |= (uint32_t)tempbyte;

    /* CID byte 13 */
    tempbyte = (uint8_t)((sd_dev->cid_data[3] & SD_MASK_16_23BITS) >> 16U);
    p_cid->mdt = ((uint16_t)tempbyte & 0x0FU) << 8U;

    /* CID byte 14 */
    tempbyte = (uint8_t)((sd_dev->cid_data[3] & SD_MASK_8_15BITS) >> 8U);
    p_cid->mdt |= (uint16_t)tempbyte;

    /* CID byte 15 */
    tempbyte = (uint8_t)(sd_dev->cid_data[3] & SD_MASK_0_7BITS);
    p_cid->cid_crc = (tempbyte & 0xFEU) >> 1U;

    return HAL_ERR_NONE;
}

/*!
    \brief      get the information of the card which are stored on
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_csd: sdcard csd data information structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_csd_get(hal_sdio_sdcard_dev_struct *sd_dev, hal_sdio_sdcard_card_csd_struct *p_csd)
{
    uint8_t tempbyte = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and p_csd address */
    if((NULL == sd_dev) || (NULL == p_csd)) {
        HAL_DEBUGE("pointer [sd_dev] or [p_cid] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* CSD byte 0 */
    tempbyte = (uint8_t)((sd_dev->csd_data[0] & SD_MASK_24_31BITS) >> 24U);
    p_csd->csd_struct = (tempbyte & 0xC0U) >> 6U;

    /* CSD byte 1 */
    tempbyte = (uint8_t)((sd_dev->csd_data[0] & SD_MASK_16_23BITS) >> 16U);
    p_csd->taac = tempbyte;

    /* CSD byte 2 */
    tempbyte = (uint8_t)((sd_dev->csd_data[0] & SD_MASK_8_15BITS) >> 8U);
    p_csd->nsac = tempbyte;

    /* CSD byte 3 */
    tempbyte = (uint8_t)(sd_dev->csd_data[0] & SD_MASK_0_7BITS);
    p_csd->tran_speed = tempbyte;

    /* CSD byte 4 */
    tempbyte = (uint8_t)((sd_dev->csd_data[1] & SD_MASK_24_31BITS) >> 24U);
    p_csd->ccc = (uint16_t)((uint16_t)tempbyte << 4U);

    /* CSD byte 5 */
    tempbyte = (uint8_t)((sd_dev->csd_data[1] & SD_MASK_16_23BITS) >> 16U);
    p_csd->ccc |= ((uint16_t)tempbyte & 0xF0U) >> 4U;
    p_csd->read_bl_len = tempbyte & 0x0FU;

    /* CSD byte 6 */
    tempbyte = (uint8_t)((sd_dev->csd_data[1] & SD_MASK_8_15BITS) >> 8U);
    p_csd->read_bl_partial    = (tempbyte & 0x80U) >> 7U;
    p_csd->write_blk_misalign = (tempbyte & 0x40U) >> 6U;
    p_csd->read_blk_misalign  = (tempbyte & 0x20U) >> 5U;
    p_csd->dsp_imp = (tempbyte & 0x10U) >> 4U;

    if((SDIO_STD_CAPACITY_SD_CARD_V1_1 == sd_dev->card_info.card_type) || \
       (SDIO_STD_CAPACITY_SD_CARD_V2_0 == sd_dev->card_info.card_type)) {
        /* card is SDSC card, CSD version 1.0 */
        p_csd->c_size = ((uint32_t)tempbyte & 0x03U) << 10U;

        /* CSD byte 7 */
        tempbyte = (uint8_t)(sd_dev->csd_data[1] & SD_MASK_0_7BITS);
        p_csd->c_size |= (uint32_t)((uint32_t)tempbyte << 2U);

        /* CSD byte 8 */
        tempbyte = (uint8_t)((sd_dev->csd_data[2] & SD_MASK_24_31BITS) >> 24U);
        p_csd->c_size |= ((uint32_t)tempbyte & 0xC0U) >> 6U;
        p_csd->vdd_r_curr_min = (tempbyte & 0x38U) >> 3U;
        p_csd->vdd_r_curr_max = tempbyte & 0x07U;

        /* CSD byte 9 */
        tempbyte = (uint8_t)((sd_dev->csd_data[2] & SD_MASK_16_23BITS) >> 16U);
        p_csd->vdd_w_curr_min = (tempbyte & 0xE0U) >> 5U;
        p_csd->vdd_w_curr_max = (tempbyte & 0x1CU) >> 2U;
        p_csd->c_size_mult    = (tempbyte & 0x03U) << 1U;

        /* CSD byte 10 */
        tempbyte = (uint8_t)((sd_dev->csd_data[2] & SD_MASK_8_15BITS) >> 8U);
        p_csd->c_size_mult |= (tempbyte & 0x80U) >> 7U;

        /* calculate the card block size and capacity */
        sd_dev->card_info.card_blocksize = ((uint32_t)1U << (p_csd->read_bl_len));
        sd_dev->card_info.card_blocknum  = p_csd->c_size + 1U;
        sd_dev->card_info.card_blocknum *= ((uint32_t)1U << (p_csd->c_size_mult + 2U));
        sd_dev->card_info.card_blocknum *= sd_dev->card_info.card_blocksize;

        sd_dev->card_info.logic_blocknum = (sd_dev->card_info.card_blocknum) * ((sd_dev->card_info.card_blocksize) / 512U);
        sd_dev->card_info.logic_blocksize = 512U;
    } else if(SDIO_HIGH_CAPACITY_SD_CARD == sd_dev->card_info.card_type) {
        /* card is SDHC card, CSD version 2.0 */
        /* CSD byte 7 */
        tempbyte = (uint8_t)(sd_dev->csd_data[1] & SD_MASK_0_7BITS);
        p_csd->c_size = ((uint32_t)tempbyte & 0x3FU) << 16U;

        /* CSD byte 8 */
        tempbyte = (uint8_t)((sd_dev->csd_data[2] & SD_MASK_24_31BITS) >> 24U);
        p_csd->c_size |= ((uint32_t)tempbyte << 8U);

        /* CSD byte 9 */
        tempbyte = (uint8_t)((sd_dev->csd_data[2] & SD_MASK_16_23BITS) >> 16U);
        p_csd->c_size |= (uint32_t)tempbyte;

        /* calculate the card block size and capacity */
        sd_dev->card_info.card_blocksize = 512U;
        sd_dev->card_info.card_blocknum  = (p_csd->c_size + 1U) * 512U;
        sd_dev->card_info.logic_blocknum = sd_dev->card_info.card_blocknum;
        sd_dev->card_info.logic_blocksize = sd_dev->card_info.card_blocksize;
    } else {
        /* do nothing */
    }

    p_csd->erase_blk_en = (tempbyte & 0x40U) >> 6U;
    p_csd->sector_size = (tempbyte & 0x3FU) << 1U;

    /* CSD byte 11 */
    tempbyte = (uint8_t)(sd_dev->csd_data[2] & SD_MASK_0_7BITS);
    p_csd->sector_size |= (tempbyte & 0x80U) >> 7U;
    p_csd->wp_grp_size = (tempbyte & 0x7FU);

    /* CSD byte 12 */
    tempbyte = (uint8_t)((sd_dev->csd_data[3] & SD_MASK_24_31BITS) >> 24U);
    p_csd->wp_grp_enable = (tempbyte & 0x80U) >> 7U;
    p_csd->r2w_factor = (tempbyte & 0x1CU) >> 2U;
    p_csd->write_bl_len = (tempbyte & 0x03U) << 2U;

    /* CSD byte 13 */
    tempbyte = (uint8_t)((sd_dev->csd_data[3] & SD_MASK_16_23BITS) >> 16U);
    p_csd->write_bl_len |= (tempbyte & 0xC0U) >> 6U;
    p_csd->write_bl_partial = (tempbyte & 0x20U) >> 5U;

    /* CSD byte 14 */
    tempbyte = (uint8_t)((sd_dev->csd_data[3] & SD_MASK_8_15BITS) >> 8U);
    p_csd->file_format_grp = (tempbyte & 0x80U) >> 7U;
    p_csd->copy_flag = (tempbyte & 0x40U) >> 6U;
    p_csd->perm_write_protect = (tempbyte & 0x20U) >> 5U;
    p_csd->tmp_write_protect = (tempbyte & 0x10U) >> 4U;
    p_csd->file_format = (tempbyte & 0x0CU) >> 2U;

    /* CSD byte 15 */
    tempbyte = (uint8_t)(sd_dev->csd_data[3] & SD_MASK_0_7BITS);
    p_csd->csd_crc = (tempbyte & 0xFEU) >> 1U;

    return HAL_ERR_NONE;
}

/*!
    \brief      get the card status whose response format R1 contains a 32-bit field
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] pcardstatus: a pointer that store card status
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_cardstatus_get(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pcardstatus)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and pcardstatus address */
    if((NULL == sd_dev) || (NULL == pcardstatus)) {
        HAL_DEBUGE("pointer [sd_dev] or [pcardstatus] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(NULL == pcardstatus) {
        sd_dev->error_state = HAL_SD_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        status = _sdcard_cardstatus_get(sd_dev, pcardstatus);
        if(HAL_SD_OK != status) {
            reval = HAL_ERR_ABORT;
        } else {
            reval = HAL_ERR_NONE;
        }
    }
    return reval;
}

/*!
    \brief      get the detailed information of the SD card based on received CID and CSD
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_cardinfo: sdcard information structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_sdcard_information_get(hal_sdio_sdcard_dev_struct *sd_dev, hal_sdio_sdcard_cardinfo_struct *p_cardinfo)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check sdcard pointer and p_cardinfo address */
    if((NULL == sd_dev) || (NULL == p_cardinfo)) {
        HAL_DEBUGE("pointer [sd_dev] or [p_cardinfo] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    p_cardinfo->card_type       = sd_dev->card_info.card_type;
    p_cardinfo->card_speed      = sd_dev->card_info.card_speed;
    p_cardinfo->card_rca        = sd_dev->card_info.card_rca;
    p_cardinfo->card_blocknum   = sd_dev->card_info.card_blocknum;
    p_cardinfo->card_blocksize  = sd_dev->card_info.card_blocksize;
    p_cardinfo->logic_blocknum  = sd_dev->card_info.logic_blocknum;
    p_cardinfo->logic_blocksize = sd_dev->card_info.logic_blocksize;

    return HAL_ERR_NONE;
}

/*!
    \brief      return the sdcard state
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_state_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
hal_sdio_sdcard_state_enum hal_sdio_sdcard_state_get(hal_sdio_sdcard_dev_struct *sd_dev)
{
    return sd_dev->state;
}

/*!
    \brief      return the sdcard error code
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
hal_sdio_sdcard_error_enum hal_sdio_sdcard_error_get(hal_sdio_sdcard_dev_struct *sd_dev)
{
    return sd_dev->error_state;
}

/*!
    \brief      initialize the card and get CID and CSD of the card
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sd_card_init(hal_sdio_sdcard_dev_struct *sd_dev)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint16_t temp_rca        = 0x01U;

    if(SDIO_POWER_OFF == hals_sdio_power_state_get(sd_dev->periph)) {
        status = HAL_SD_OPERATION_IMPROPER;
    } else {

        /* the card is not I/O only card */
        if(SDIO_SECURE_DIGITAL_IO_CARD != sd_dev->card_info.card_type) {
            /* send CMD2(SD_CMD_ALL_SEND_CID) to get the CID numbers */
            hals_sdio_command_response_config(sd_dev->periph, SD_CMD_ALL_SEND_CID, 0U, SDIO_RESPONSETYPE_LONG);
            hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
            hals_sdio_csm_enable(sd_dev->periph);
            /* check if some error occurs */
            status = _r2_error_check(sd_dev);
            if(HAL_SD_OK == status) {

                /* store the CID numbers */
                sd_dev->cid_data[0] = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                sd_dev->cid_data[1] = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE1);
                sd_dev->cid_data[2] = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE2);
                sd_dev->cid_data[3] = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE3);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        if (HAL_SD_OK == status){
            /* the card is SD memory card or the I/O card has the memory portion */
            if((SDIO_STD_CAPACITY_SD_CARD_V1_1 == sd_dev->card_info.card_type) || \
            (SDIO_STD_CAPACITY_SD_CARD_V2_0 == sd_dev->card_info.card_type) || \
            (SDIO_STD_CAPACITY_SD_CARD_V3_0 == sd_dev->card_info.card_type) || \
            (SDIO_HIGH_CAPACITY_SD_CARD == sd_dev->card_info.card_type)     || \
            (SDIO_SECURE_DIGITAL_IO_COMBO_CARD == sd_dev->card_info.card_type)) {
                /* send CMD3(SEND_RELATIVE_ADDR) to ask the card to publish a new relative address (RCA) */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_RELATIVE_ADDR, 0U, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);
                /* check if some error occurs */
                status = _r6_error_check(sd_dev, SD_CMD_SEND_RELATIVE_ADDR, &temp_rca);
            } else {
                /* do nothing */
            }
            if (HAL_SD_OK == status){

                if(SDIO_SECURE_DIGITAL_IO_CARD != sd_dev->card_info.card_type) {
                    /* the card is not I/O only card */
                    sd_dev->card_info.card_rca = temp_rca;

                    /* send CMD9(SEND_CSD) to get the addressed card's card-specific data (CSD) */
                    hals_sdio_command_response_config(sd_dev->periph, (uint32_t)SD_CMD_SEND_CSD, \
                                                    (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_LONG);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);
                    /* check if some error occurs */
                    status = _r2_error_check(sd_dev);
                    if(HAL_SD_OK == status) {

                        /* store the card-specific data (CSD) */
                        sd_dev->csd_data[0] = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                        sd_dev->csd_data[1] = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE1);
                        sd_dev->csd_data[2] = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE2);
                        sd_dev->csd_data[3] = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE3);
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }
        }
    }

    return status;
}

/*!
    \brief      configure the clock and the work voltage, and get the card type
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sd_power_on(hal_sdio_sdcard_dev_struct *sd_dev)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t response = 0U, count = 0U;
    uint8_t busyflag    = 0U;
    uint32_t timedelay  = 0U;
    uint32_t sdcardtype = SD_STD_CAPACITY;

    /* configure the SDIO peripheral */
    hals_sdio_clock_config(sd_dev->periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, SD_CLK_DIV_INIT);
    hals_sdio_bus_mode_set(sd_dev->periph, SDIO_BUSMODE_1BIT);
    hals_sdio_hardware_clock_disable(sd_dev->periph);

    if(SDCAED_VOLTAGE_18V == sd_dev->card_info.card_speed) {
        hals_sdio_hardware_clock_enable(sd_dev->periph);
        hals_sdio_direction_polarity_set(sd_dev->periph, SDIO_DIRECTION_SIGNAL_HIGH);
    } else {
        /* do nothing */
    }

    hals_sdio_power_state_set(sd_dev->periph, SDIO_POWER_ON);

    /* time delay for power up */
    timedelay = 500U;
    while(timedelay > 0U) {
        timedelay--;
    }

    /* send CMD0(GO_IDLE_STATE) to reset the card */
    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_GO_IDLE_STATE, 0U, SDIO_RESPONSETYPE_NO);
    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
    /* enable the CSM */
    hals_sdio_csm_enable(sd_dev->periph);

    /* check if command sent error occurs */
    status = _cmdsent_error_check(sd_dev);
    if(HAL_SD_OK == status) {

        /* send CMD8(SEND_IF_COND) to get SD memory card interface condition */
        hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_IF_COND, SD_CHECK_PATTERN, SDIO_RESPONSETYPE_SHORT);
        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
        hals_sdio_csm_enable(sd_dev->periph);

        if(HAL_SD_OK == _r7_error_check(sd_dev)) {
            /* SD Card 2.0 */
            sd_dev->card_info.card_type = SDIO_STD_CAPACITY_SD_CARD_V2_0;
            sdcardtype = SD_HIGH_CAPACITY;
        } else {
            /* do nothing */
        }

        /* send CMD55(APP_CMD) to indicate next command is application specific command */
        hals_sdio_command_response_config(sd_dev->periph, SD_CMD_APP_CMD, 0U, SDIO_RESPONSETYPE_SHORT);
        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
        hals_sdio_csm_enable(sd_dev->periph);

        if(HAL_SD_OK == _r1_error_check(sd_dev, SD_CMD_APP_CMD)) {
            /* SD memory card */
            while((!busyflag) && (count < SD_MAX_VOLT_VALIDATION)) {
                /* send CMD55(APP_CMD) to indicate next command is application specific command */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_APP_CMD, 0U, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_APP_CMD);
                if(HAL_SD_OK == status) {
                    /* send ACMD41(SD_SEND_OP_COND) to get host capacity support information (HCS) and OCR content */
                    hals_sdio_command_response_config(sd_dev->periph, SD_APPCMD_SD_SEND_OP_COND, \
                                                    (SD_VOLTAGE_WINDOW | SD_VOLTAGE_18V | sdcardtype), \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);
                    /* check if some error occurs */
                    status = _r3_error_check(sd_dev);
                    if(HAL_SD_OK == status) {
                        /* get the response and check card power up status bit(busy) */
                        response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                        busyflag = (uint8_t)((response >> 31U) & 0x01U);
                        ++count;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
                if(HAL_SD_OK != status) {
                    break;
                } else {
                    /* do nothing */
                }
            }
            if (HAL_SD_OK == status){
                if(count >= SD_MAX_VOLT_VALIDATION) {
                    status = HAL_SD_VOLTRANGE_INVALID;

                } else {

                    if(response & SD_HIGH_CAPACITY) {
                        sd_dev->card_info.card_blocknum = SD_SDHC_SDXC;
                        sd_dev->card_info.card_type     = SDIO_HIGH_CAPACITY_SD_CARD;

                        /* SDHC card */
                        if(response & SD_VOLTAGE_18V) {
                            /* need 1.8V power switch*/
                            if(SDCAED_VOLTAGE_18V == sd_dev->card_info.card_speed) {
                                /*step1: set VSEN for starting the Voltage switch */
                                hals_sdio_voltage_switch_enable(sd_dev->periph);

                                /*step2: send the CMD11 and wait clock stopping */
                                status = _sd_card_voltage_switch(sd_dev);
                                /* check if some error occurs */
                                if(HAL_SD_OK == status) {

                                    /* wait CLKSTOP */
                                    timedelay = 0U;
                                    while(SET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_CLKSTOP)) {
                                        /* time delay for SDIO_CLK stopped and restarted in voltage switch procedure*/
                                        timedelay++;
                                        if(timedelay > SD_DATATIMEOUT) {
                                            status = HAL_SD_ERROR;

                                            break;
                                        } else {
                                            /* do nothing */
                                        }
                                    }
                                    if (HAL_SD_OK == status){
                                        /*step3: clear the CLKSTOP and enable the transceiver */
                                        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CLKSTOP);
                                        /* whether data0 is busy */
                                        if(SET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DAT0BSY)) {
                                            status = HAL_SD_FUNCTION_UNSUPPORTED;

                                        } else {
                                            /* enable SD Transceiver 1.8V mode */
                                            if(NULL != sd_dev->transceiver_1_8v_callback) {
                                                sd_dev->transceiver_1_8v_callback(SET);
                                            } else {
                                                /* do nothing */
                                            }
                                            /* enable the  transceiver */
                                            hals_sdio_voltage_switch_sequence_enable(sd_dev->periph);
                                            timedelay = 0U;
                                            while(SET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_VOLSWEND)) {
                                                /* time delay for wait 1.8V power switch sequence is completed*/
                                                timedelay++;
                                                if(timedelay > SD_DATATIMEOUT) {
                                                    status = HAL_SD_ERROR;

                                                    break;
                                                } else {
                                                    /* do nothing */
                                                }
                                            }
                                            if (HAL_SD_OK == status){
                                                /* clear the SDIO_FLAG_VOLSWEND flags */
                                                hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_VOLSWEND);
                                                if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DAT0BSY)) {
                                                    status = HAL_SD_VOLTRANGE_INVALID;

                                                } else {
                                                    /* switch to 1.8V power OK */

                                                    /*step4: disable  VSSTART and VSEN */
                                                    hals_sdio_voltage_switch_sequence_disable(sd_dev->periph);
                                                    hals_sdio_voltage_switch_disable(sd_dev->periph);
                                                    /* clear the SDIO_INTC flags */
                                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);
                                                }
                                            } else {
                                                /* do nothing */
                                            }
                                        }
                                    } else {
                                        /* do nothing */
                                    }
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                /* do nothing */
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      select or deselect a card
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sd_card_select_deselect(hal_sdio_sdcard_dev_struct *sd_dev)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;

    /* send CMD7(SELECT/DESELECT_CARD) to select or deselect the card */
    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SELECT_DESELECT_CARD, \
                                      (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(sd_dev->periph);

    status = _r1_error_check(sd_dev, SD_CMD_SELECT_DESELECT_CARD);

    return status;
}

/*!
    \brief      switch 1.8V power level of SD card
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sd_card_voltage_switch(hal_sdio_sdcard_dev_struct *sd_dev)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;

    /* send CMD11(SD_CMD_VOLATAGE_SWITCH) switch to 1.8V bus signaling level */
    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_VOLATAGE_SWITCH, 0U, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(sd_dev->periph);

    status = _r1_error_check(sd_dev, SD_CMD_VOLATAGE_SWITCH);

    return status;
}

/*!
    \brief      handle the receive interrupt
    \param[in]  sd_dev: pointer to a sd device information structure
    \param[out] none
    \retval     none
*/
static void _sd_read_interrupt(void *sd_dev)
{
    uint32_t count;
    uint32_t restwords;
    uint32_t *ptempbuff;

    hal_sdio_sdcard_dev_struct *p_sd = sd_dev;
    ptempbuff = p_sd->p_rxbuffer;

    if(p_sd->rx_length >= SD_FIFOHALF_BYTES) {
        /* at least 8 words can be read in the FIFO */
        for(count = 0U; count < SD_FIFOHALF_WORDS; count++) {
            *(ptempbuff + count) = hals_sdio_data_read(p_sd->periph);
        }
        ptempbuff += SD_FIFOHALF_WORDS;
        p_sd->rx_length -= SD_FIFOHALF_BYTES;
    } else {
        restwords = (p_sd->rx_length + 3U) / 4U;
        for(count = 0U; count < restwords; count++) {
            *ptempbuff = hals_sdio_data_read(p_sd->periph);
            ++ptempbuff;
        }
        ptempbuff -= restwords;
    }

    p_sd->p_rxbuffer = ptempbuff;
}

/*!
    \brief      handle the transmit interrupt
    \param[in]  sd_dev: pointer to a sd device information structure
    \param[out] none
    \retval     none
*/
static void _sd_write_interrupt(void *sd_dev)
{
    uint32_t count;
    uint32_t restwords;
    uint32_t *ptempbuff;

    hal_sdio_sdcard_dev_struct *p_sd = sd_dev;
    ptempbuff = p_sd->p_txbuffer;

    /* at least 8 words can be written into the FIFO */
    if(p_sd->tx_length >= SD_FIFOHALF_BYTES) {
        for(count = 0U; count < SD_FIFOHALF_WORDS; count++) {
            hals_sdio_data_write(p_sd->periph, *(ptempbuff + count));
        }
        /* 8 words(32 bytes) has been transferred */
        ptempbuff += SD_FIFOHALF_WORDS;
        p_sd->tx_length -= SD_FIFOHALF_BYTES;
    } else {
        restwords = (p_sd->tx_length + 3U) / 4U;
        for(count = 0U; count < restwords; count++) {
            hals_sdio_data_write(p_sd->periph, *ptempbuff);
            ++ptempbuff;
        }
        ptempbuff -= restwords;
    }

    p_sd->p_txbuffer = ptempbuff;
}

/*!
    \brief      stop an ongoing data transfer
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sd_transfer_stop(hal_sdio_sdcard_dev_struct *sd_dev)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;

    /* send CMD12(STOP_TRANSMISSION) to stop transmission */
    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_STOP_TRANSMISSION, 0U, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_trans_stop_enable(sd_dev->periph);
    hals_sdio_trans_start_disable(sd_dev->periph);
    hals_sdio_csm_enable(sd_dev->periph);

    /* check if some error occurs */
    status = _r1_error_check(sd_dev, SD_CMD_STOP_TRANSMISSION);
    hals_sdio_trans_stop_disable(sd_dev->periph);

    return status;
}

/*!
    \brief      check if the command sent error occurs
     \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _cmdsent_error_check(hal_sdio_sdcard_dev_struct *sd_dev)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t timeout         = 100000U;
    /* check command sent flag */
    while((RESET == hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_CMDSEND)) && (timeout > 0U)) {
        --timeout;
    }

    /* command response is timeout */
    if(0U == timeout) {
        status = HAL_SD_CMD_RESP_TIMEOUT;

    } else {

        /* if the command is sent, clear the CMD_FLAGS flags */
        hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_CMD_FLAGS);
    }

    return status;
}

/*!
    \brief      check if error type for R1 response
    \param[in]  resp: content of response
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _r1_error_type_check(uint32_t resp)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_ERROR;
    /* check which error occurs */
    if(resp & SD_R1_OUT_OF_RANGE) {
        status = HAL_SD_OUT_OF_RANGE;
    } else if(resp & SD_R1_ADDRESS_ERROR) {
        status = HAL_SD_ADDRESS_ERROR;
    } else if(resp & SD_R1_BLOCK_LEN_ERROR) {
        status = HAL_SD_BLOCK_LEN_ERROR;
    } else if(resp & SD_R1_ERASE_SEQ_ERROR) {
        status = HAL_SD_ERASE_SEQ_ERROR;
    } else if(resp & SD_R1_ERASE_PARAM) {
        status = HAL_SD_ERASE_PARAM;
    } else if(resp & SD_R1_WP_VIOLATION) {
        status = HAL_SD_WP_VIOLATION;
    } else if(resp & SD_R1_LOCK_UNLOCK_FAILED) {
        status = HAL_SD_LOCK_UNLOCK_FAILED;
    } else if(resp & SD_R1_COM_CRC_ERROR) {
        status = HAL_SD_COM_CRC_ERROR;
    } else if(resp & SD_R1_ILLEGAL_COMMAND) {
        status = HAL_SD_ILLEGAL_COMMAND;
    } else if(resp & SD_R1_CARD_ECC_FAILED) {
        status = HAL_SD_CARD_ECC_FAILED;
    } else if(resp & SD_R1_CC_ERROR) {
        status = HAL_SD_CC_ERROR;
    } else if(resp & SD_R1_GENERAL_UNKNOWN_ERROR) {
        status = HAL_SD_GENERAL_UNKNOWN_ERROR;
    } else if(resp & SD_R1_CSD_OVERWRITE) {
        status = HAL_SD_CSD_OVERWRITE;
    } else if(resp & SD_R1_WP_ERASE_SKIP) {
        status = HAL_SD_WP_ERASE_SKIP;
    } else if(resp & SD_R1_CARD_ECC_DISABLED) {
        status = HAL_SD_CARD_ECC_DISABLED;
    } else if(resp & SD_R1_ERASE_RESET) {
        status = HAL_SD_ERASE_RESET;
    } else if(resp & SD_R1_AKE_SEQ_ERROR) {
        status = HAL_SD_AKE_SEQ_ERROR;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      check if error occurs for R1 response
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  cmdindex: the index of command
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _r1_error_check(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t cmdindex)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t reg_status = 0U, resp_r1 = 0U;

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(sd_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(sd_dev->periph);
    }

    /* check whether an error or timeout occurs or command response received */
    if(reg_status & SDIO_FLAG_CCRCERR) {
        status = HAL_SD_CMD_CRC_ERROR;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CCRCERR);

    } else if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_SD_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CMDTMOUT);

    } else {

        /* check whether the last response command index is the desired one */
        if(hals_sdio_command_index_get(sd_dev->periph) != cmdindex) {
            status = HAL_SD_ILLEGAL_COMMAND;

        } else {

            /* clear all the CMD_FLAGS flags */
            hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_CMD_FLAGS);

            /* get the SDIO response register 0 for checking */
            resp_r1 = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
            if(SD_ALLZERO == (resp_r1 & SD_R1_ERROR_BITS)) {
                /* no error occurs, return HAL_SD_OK */
                status = HAL_SD_OK;

            } else {

                /* if some error occurs, return the error type */
                status = _r1_error_type_check(resp_r1);
            }
        }
    }

    return status;
}

/*!
    \brief      check if error occurs for R2 response
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _r2_error_check(hal_sdio_sdcard_dev_struct *sd_dev)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t reg_status      = 0U;

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(sd_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(sd_dev->periph);
    }

    /* check whether an error or timeout occurs or command response received */
    if(reg_status & SDIO_FLAG_CCRCERR) {
        status = HAL_SD_CMD_CRC_ERROR;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CCRCERR);

    } else if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_SD_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CMDTMOUT);

    } else {
        /* clear all the CMD_FLAGS flags */
        hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_CMD_FLAGS);
    }

    return status;
}

/*!
    \brief      check if error occurs for R3 response
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _r3_error_check(hal_sdio_sdcard_dev_struct *sd_dev)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t reg_status      = 0U;

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(sd_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(sd_dev->periph);
    }

    if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_SD_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CMDTMOUT);

    } else {

        /* clear all the CMD_FLAGS flags */
        hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_CMD_FLAGS);
    }

    return status;
}

/*!
    \brief      check if error occurs for R6 response
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  cmdindex: the index of command
    \param[out] prca: a pointer that store the RCA of card
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _r6_error_check(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t cmdindex, uint16_t *prca)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t reg_status = 0U, response = 0U;

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(sd_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(sd_dev->periph);
    }

    /* check whether an error or timeout occurs or command response received */
    if(reg_status & SDIO_FLAG_CCRCERR) {
        status = HAL_SD_CMD_CRC_ERROR;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CCRCERR);

    } else if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_SD_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CMDTMOUT);

    } else {

        /* check whether the last response command index is the desired one */
        if(hals_sdio_command_index_get(sd_dev->periph) != cmdindex) {
            status = HAL_SD_ILLEGAL_COMMAND;

        } else {

            /* clear all the CMD_FLAGS flags */
            hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_CMD_FLAGS);
            /* get the SDIO response register 0 for checking */
            response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);

            if(SD_ALLZERO == (response & (SD_R6_COM_CRC_ERROR | SD_R6_ILLEGAL_COMMAND | SD_R6_GENERAL_UNKNOWN_ERROR))) {
                *prca = (uint16_t)(response >> 16U);

            } else {

                /* if some error occurs, return the error type */
                if(response & SD_R6_COM_CRC_ERROR) {
                    status = HAL_SD_COM_CRC_ERROR;
                } else if(response & SD_R6_ILLEGAL_COMMAND) {
                    status = HAL_SD_ILLEGAL_COMMAND;
                } else if(response & SD_R6_GENERAL_UNKNOWN_ERROR) {
                    status = HAL_SD_GENERAL_UNKNOWN_ERROR;
                } else {
                    /* do nothing */
                }
            }
        }
    }

    return status;
}

/*!
    \brief      check if error occurs for R7 response
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _r7_error_check(hal_sdio_sdcard_dev_struct *sd_dev)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_ERROR;
    uint32_t reg_status = 0U, timeout = 100000U;

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(sd_dev->periph);
    while((0U == (reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) && (timeout > 0U)) {
        reg_status = SDIO_STAT(sd_dev->periph);
        --timeout;
    }

    /* check the flags */
    if((reg_status & SDIO_FLAG_CMDTMOUT) || (0U == timeout)) {
        status = HAL_SD_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CMDTMOUT);

    } else {
        if(reg_status & SDIO_FLAG_CMDRECV) {
            status = HAL_SD_OK;
            hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CMDRECV);

        } else {
            /* do nothing */
        }
    }

    return status;
}

/*!
    \brief      configure the bus width mode
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  buswidth: the bus width
                only one parameter can be selected which is shown as below:
      \arg        SD_BUS_WIDTH_1BIT: 1-bit bus width
      \arg        SD_BUS_WIDTH_4BIT: 4-bit bus width
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sd_bus_width_config(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t buswidth)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    /* check whether the card is locked */
    if(hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0) & SD_CARDSTATE_LOCKED) {
        status = HAL_SD_LOCK_UNLOCK_FAILED;

    } else {

        /* get the SCR register */
        status = _sd_scr_get(sd_dev);
        if(HAL_SD_OK == status) {

            if(SD_BUS_WIDTH_1BIT == buswidth) {
                if(SD_ALLZERO != (sd_dev->scr_data[1] & buswidth)) {
                    /* send CMD55(APP_CMD) to indicate next command is application specific command */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_APP_CMD, \
                                                    (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);

                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_APP_CMD);
                    if(HAL_SD_OK == status) {

                        /* send ACMD6(SET_BUS_WIDTH) to define the data bus width */
                        hals_sdio_command_response_config(sd_dev->periph, SD_APPCMD_SET_BUS_WIDTH, 0U, SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(sd_dev->periph);

                        /* check if some error occurs */
                        status = _r1_error_check(sd_dev, SD_APPCMD_SET_BUS_WIDTH);

                    } else {
                        /* do nothing */
                    }
                } else {
                    status = HAL_SD_OPERATION_IMPROPER;
                }

            } else if(SD_BUS_WIDTH_4BIT == buswidth) {
                if(SD_ALLZERO != (sd_dev->scr_data[1] & buswidth)) {
                    /* send CMD55(APP_CMD) to indicate next command is application specific command */
                    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_APP_CMD, \
                                                    (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(sd_dev->periph);

                    /* check if some error occurs */
                    status = _r1_error_check(sd_dev, SD_CMD_APP_CMD);
                    if(HAL_SD_OK == status) {

                        /* send ACMD6(SET_BUS_WIDTH) to define the data bus width */
                        hals_sdio_command_response_config(sd_dev->periph, SD_APPCMD_SET_BUS_WIDTH, 2U, SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(sd_dev->periph);

                        /* check if some error occurs */
                        status = _r1_error_check(sd_dev, SD_APPCMD_SET_BUS_WIDTH);

                    } else {
                        /* do nothing */
                    }
                } else {
                    status = HAL_SD_OPERATION_IMPROPER;
                }

            } else {
                status = HAL_SD_PARAMETER_INVALID;

            }
        } else {
            /* do nothing */
        }
    }
    return status;
}

/*!
    \brief      get the SCR of corresponding card
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sd_scr_get(hal_sdio_sdcard_dev_struct *sd_dev)
{
    uint32_t temp_scr[2] = {0, 0}, idx_scr = 0U;
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;

    /* send CMD16(SET_BLOCKLEN) to set block length */
    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, 8U, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(sd_dev->periph);

    /* check if some error occurs */
    status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
    if(HAL_SD_OK == status) {

        /* send CMD55(APP_CMD) to indicate next command is application specific command */
        hals_sdio_command_response_config(sd_dev->periph, SD_CMD_APP_CMD, \
                                        (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
        hals_sdio_csm_enable(sd_dev->periph);

        /* check if some error occurs */
        status = _r1_error_check(sd_dev, SD_CMD_APP_CMD);
        if(HAL_SD_OK == status) {

            /* configure SDIO data */
            hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, 8U, SDIO_DATABLOCKSIZE_8BYTES);
            hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
            hals_sdio_dsm_enable(sd_dev->periph);

            /* send ACMD51(SEND_SCR) to read the SD configuration register */
            hals_sdio_command_response_config(sd_dev->periph, SD_APPCMD_SEND_SCR, 0U, SDIO_RESPONSETYPE_SHORT);
            hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
            hals_sdio_csm_enable(sd_dev->periph);

            /* check if some error occurs */
            status = _r1_error_check(sd_dev, SD_APPCMD_SEND_SCR);
            if(HAL_SD_OK == status) {

                /* store the received SCR */
                while(!hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR | SDIO_FLAG_DTTMOUT | SDIO_FLAG_RXORE | \
                                                        SDIO_FLAG_DTBLKEND | SDIO_FLAG_DTEND)) {
                    if((SET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_RFE)) && \
                    (SET == hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DATSTA))) {
                        *(temp_scr + idx_scr) = hals_sdio_data_read(sd_dev->periph);
                        ++idx_scr;
                    } else {
                        /* do nothing */
                    }
                }

                /* check whether some error occurs */
                if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR)) {
                    status = HAL_SD_DATA_CRC_ERROR;
                    hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTCRCERR);

                } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTTMOUT)) {
                    status = HAL_SD_DATA_TIMEOUT;
                    hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTTMOUT);

                } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_RXORE)) {
                    status = HAL_SD_RX_OVERRUN_ERROR;
                    hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_RXORE);

                } else {
                    /* do nothing */
                }

                /* clear all the SDIO_INTC flags */
                hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);
                /* readjust the temp SCR value */
                sd_dev->scr_data[0] = ((temp_scr[1] & SD_MASK_0_7BITS) << 24U) | ((temp_scr[1] & SD_MASK_8_15BITS) << 8U) | \
                                    ((temp_scr[1] & SD_MASK_16_23BITS) >> 8U) | ((temp_scr[1] & SD_MASK_24_31BITS) >> 24U);
                sd_dev->scr_data[1] = ((temp_scr[0] & SD_MASK_0_7BITS) << 24U) | ((temp_scr[0] & SD_MASK_8_15BITS) << 8U) | \
                                    ((temp_scr[0] & SD_MASK_16_23BITS) >> 8U) | ((temp_scr[0] & SD_MASK_24_31BITS) >> 24U);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
    return status;
}

/*!
    \brief     perform sampling point tuning using tuning command
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sd_tuning(hal_sdio_sdcard_dev_struct *sd_dev)
{
    /* initialize the variables */
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t count = 0U, temp = 0U, tuning_count = 0U, totalnumber_bytes = 0U, *ptempbuff = g_buf_tuning;
    __IO uint32_t i = 0U;
    uint8_t cardstate = 0U;

    /* clear all DSM configuration */
    hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
    hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSDIRECTION_TOCARD, SDIO_TRANSMODE_BLOCKCOUNT);
    hals_sdio_dsm_disable(sd_dev->periph);

    /* cpdm config */
    _cpdm_config();

    for(i = 0xBU; i > 0U; i--) {
        CPDM_CTL(CPDM_SDIO0) = CPDM_CTL_DLSEN;
        temp = CPDM_CFG(CPDM_SDIO0);
        temp &= 0xFFFFFFF0U;
        CPDM_CFG(CPDM_SDIO0) = temp | i;
        CPDM_CTL(CPDM_SDIO0) = CPDM_CTL_CPDMEN;

        for(tuning_count = 0U; tuning_count < 16U; tuning_count++) {
            g_buf_tuning[tuning_count] = 0U;
        }
        tuning_count      = 0U;
        totalnumber_bytes = 64U;

        if(i == 8U) {
            __NOP();
        } else {
            /* do nothing */
        }

        /* configure SDIO data transmission */
        hals_sdio_data_config(sd_dev->periph, 0xFFFFFFFU, totalnumber_bytes, SDIO_DATABLOCKSIZE_64BYTES);
        hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSDIRECTION_TOSDIO, SDIO_TRANSMODE_BLOCKCOUNT);
        hals_sdio_dsm_enable(sd_dev->periph);

        /* send CMD19(SEND_TUNING_PATTERN) to read a block */
        hals_sdio_command_response_config(sd_dev->periph, SD_SEND_TUNING_PATTERN, (uint32_t)0, SDIO_RESPONSETYPE_SHORT);
        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
        hals_sdio_csm_enable(sd_dev->periph);

        /* check if some error occurs */
        status = _r1_error_check(sd_dev, 19U);
        if(HAL_SD_OK == status) {
            ptempbuff = g_buf_tuning;
            /* polling mode */
            while(!hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR | SDIO_FLAG_DTTMOUT | SDIO_FLAG_RXORE | \
                                                    SDIO_FLAG_DTBLKEND | SDIO_FLAG_DTEND)) {
                if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_RFH)) {
                    /* at least 8 words can be read in the FIFO */
                    for(count = 0U; count < SD_FIFOHALF_WORDS; count++) {
                        *(ptempbuff + count) = hals_sdio_data_read(sd_dev->periph);
                    }
                    ptempbuff += SD_FIFOHALF_WORDS;
                } else {
                    /* do nothing */
                }
            }

            hals_sdio_trans_start_disable(sd_dev->periph);

            /* whether some error occurs and return it */
            if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR)) {
                status = HAL_SD_DATA_CRC_ERROR;
                hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTCRCERR);
                hals_sdio_fifo_reset_enable(sd_dev->periph);
                hals_sdio_fifo_reset_disable(sd_dev->periph);
                continue;
            } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTTMOUT)) {
                status = HAL_SD_DATA_TIMEOUT;
                hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTTMOUT);
                hals_sdio_fifo_reset_enable(sd_dev->periph);
                hals_sdio_fifo_reset_disable(sd_dev->periph);
                continue;
            } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_RXORE)) {
                status = HAL_SD_RX_OVERRUN_ERROR;
                hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_RXORE);
                hals_sdio_fifo_reset_enable(sd_dev->periph);
                hals_sdio_fifo_reset_disable(sd_dev->periph);
                continue;
            } else {
                /* do nothing */
            }


            /* clear the SDIO_INTC flags */
            hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);

            /* get the card state and wait the card is out of programming and receiving state */
            status = _sd_card_state_get(sd_dev, &cardstate);
            while((HAL_SD_OK == status) && \
                ((SD_CARDSTATE_PROGRAMMING == cardstate) || (SD_CARDSTATE_RECEIVING == cardstate))) {
                status = _sd_card_state_get(sd_dev, &cardstate);
            }
            status = HAL_SD_OK;

            for(tuning_count = 0U; tuning_count < 16U; tuning_count++) {
                if(g_buf_tuning[tuning_count] != g_buf_tuning_standard[tuning_count]) {
                    break;
                } else {
                    /* do nothing */
                }
            }

            if(16U == tuning_count) {
                break;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        hals_sdio_hardware_clock_disable(sd_dev->periph);
    }

    if((0U == i) & (16U != tuning_count)) {
        status = HAL_SD_ERROR;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      get the data block size
    \param[in]  bytesnumber: the number of bytes
    \param[out] none
    \retval     data block size
                only one parameter can be selected which is shown as below:
      \arg        SDIO_DATABLOCKSIZE_1BYTE: block size = 1 byte
      \arg        SDIO_DATABLOCKSIZE_2BYTES: block size = 2 bytes
      \arg        SDIO_DATABLOCKSIZE_4BYTES: block size = 4 bytes
      \arg        SDIO_DATABLOCKSIZE_8BYTES: block size = 8 bytes
      \arg        SDIO_DATABLOCKSIZE_16BYTES: block size = 16 bytes
      \arg        SDIO_DATABLOCKSIZE_32BYTES: block size = 32 bytes
      \arg        SDIO_DATABLOCKSIZE_64BYTES: block size = 64 bytes
      \arg        SDIO_DATABLOCKSIZE_128BYTES: block size = 128 bytes
      \arg        SDIO_DATABLOCKSIZE_256BYTES: block size = 256 bytes
      \arg        SDIO_DATABLOCKSIZE_512BYTES: block size = 512 bytes
      \arg        SDIO_DATABLOCKSIZE_1024BYTES: block size = 1024 bytes
      \arg        SDIO_DATABLOCKSIZE_2048BYTES: block size = 2048 bytes
      \arg        SDIO_DATABLOCKSIZE_4096BYTES: block size = 4096 bytes
      \arg        SDIO_DATABLOCKSIZE_8192BYTES: block size = 8192 bytes
      \arg        SDIO_DATABLOCKSIZE_16384BYTES: block size = 16384 bytes
*/
static uint32_t _sd_datablocksize_get(uint16_t bytesnumber)
{
    uint8_t exp_val = 0U;

    /* calculate the exponent of 2 */
    while(1U != bytesnumber) {
        bytesnumber >>= 1U;
        ++exp_val;
    }

    return DATACTL_BLKSZ(exp_val);
}

/*!
    \brief      get the card state
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] pcardstate: a pointer that store the card state
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sd_card_state_get(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t *pcardstate)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    __IO uint32_t reg_status = 0U, response = 0U;

    /* send CMD13(SEND_STATUS), addressed card sends its status register */
    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS, \
                                      (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(sd_dev->periph);

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(sd_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(sd_dev->periph);
    }

    /* check whether an error or timeout occurs or command response received */
    if(reg_status & SDIO_FLAG_CCRCERR) {
        status = HAL_SD_CMD_CRC_ERROR;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CCRCERR);

    } else if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_SD_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CMDTMOUT);

    } else {

        /* command response received, store the response command index */
        reg_status = (uint32_t)hals_sdio_command_index_get(sd_dev->periph);
        if((uint32_t)SD_CMD_SEND_STATUS != reg_status) {
            status = HAL_SD_ILLEGAL_COMMAND;

        } else {

            /* clear all the SDIO_INTC flags */
            hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);
            /* get the SDIO response register 0 for checking */
            response    = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
            *pcardstate = (uint8_t)((response >> 9U) & 0x0000000FU);

            if(SD_ALLZERO == (response & SD_R1_ERROR_BITS)) {
                /* no error occurs, return SD_OK */
                status = HAL_SD_OK;

            } else {
                /* do nothing */
            }

            /* if some error occurs, return the error type */
            status = _r1_error_type_check(response);
        }
    }

    return status;
}

/*!
    \brief      enable SD Transceiver 1.8V mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
static void _sd_transceiver_enable(FlagStatus flag)
{
    /* NOTE : add code for enable SD Transceiver 1.8V mode */
    if(SET == flag) {
        hal_gpio_bit_set(GPIOC, GPIO_PIN_0);
    } else {
        hal_gpio_bit_reset(GPIOC, GPIO_PIN_0);
    }

}

/*!
    \brief      deinitialize the SDIO
    \param[in]  sdio_periph: SDIOx(x=0,1)
    \param[out] none
    \retval     none
*/
static void _sdcard_deinit(uint32_t sdio_periph)
{
    switch(sdio_periph) {
    /* deinitialize SDIO0 */
    case SDIO0:
        hal_rcu_periph_reset_enable(RCU_SDIO0RST);
        hal_rcu_periph_reset_disable(RCU_SDIO0RST);
        break;
    /* deinitialize SDIO1 */
    case SDIO1:
        hal_rcu_periph_reset_enable(RCU_SDIO1RST);
        hal_rcu_periph_reset_disable(RCU_SDIO1RST);
        break;
    default:
        break;
    }
}

/*!
    \brief      configure CPDM
    \param[in]  none
    \param[out] none
    \retval     none
*/
static void _cpdm_config(void)
{
    uint32_t temp, i;

    /* apply the delay settings */
    CPDM_CTL(CPDM_SDIO0) = CPDM_CTL_CPDMEN | CPDM_CTL_DLSEN;
    temp = CPDM_MAX_PHASE ;

    for(i = 0U; i < ((uint32_t)0x00000080U); i++) {
        CPDM_CFG(CPDM_SDIO0) = temp | (i << 8);

        while(CPDM_CFG_DLLENF == (CPDM_CFG(CPDM_SDIO0) & CPDM_CFG_DLLENF)) {
           /* empty loop */
        }

        while(RESET == (CPDM_CFG(CPDM_SDIO0) & CPDM_CFG_DLLENF)) {
           /* empty loop */
        }

        if(((((uint32_t)CPDM_CFG(CPDM_SDIO0) >> 16U) & 0x7FFU) > 0U) && \
           ((RESET == (CPDM_CFG(CPDM_SDIO0) & 0x04000000U)) || (RESET ==(CPDM_CFG(CPDM_SDIO0) & 0x08000000U)))) {
            temp = CPDM_CFG(CPDM_SDIO0);
            temp &= 0xFFFFFFF0U;

            CPDM_CFG(CPDM_SDIO0) = temp | 0xAU;
            CPDM_CTL(CPDM_SDIO0) = CPDM_CTL_CPDMEN;
            break;
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      get the card status whose response format R1 contains a 32-bit field
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] pcardstatus: a pointer that store card status
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sdcard_cardstatus_get(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t *pcardstatus)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;

    /* send CMD13(SEND_STATUS), addressed card sends its status register */
    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS, \
                                      (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(sd_dev->periph);

    /* check if some error occurs */
    status = _r1_error_check(sd_dev, SD_CMD_SEND_STATUS);
    if(HAL_SD_OK == status) {

        *pcardstatus = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
    } else {
        /* do nothing */
    }
    return status;
}

/*!
    \brief      lock or unlock a card
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  lockstate: the lock state
                only one parameter can be selected which is shown as below:
      \arg        SD_LOCK: lock the SD card
      \arg        SD_UNLOCK: unlock the SD card
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sdcard_lock_unlock(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t lockstate)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint8_t cardstate = 0U, tempbyte = 0U;
    uint32_t pwd1 = 0U, pwd2 = 0U, response = 0U, timeout = 0U;
    uint16_t tempccc = 0U;
    if(HAL_SD_STATE_READY != sd_dev->state) {

        status = HAL_SD_ERROR;
    } else {

        /* get the card command classes from CSD */
        tempbyte = (uint8_t)((sd_dev->csd_data[1] & SD_MASK_24_31BITS) >> 24U);
        tempccc  = ((uint16_t)tempbyte << 4U);
        tempbyte = (uint8_t)((sd_dev->csd_data[1] & SD_MASK_16_23BITS) >> 16U);
        tempccc |= (((uint16_t)tempbyte & 0xF0U) >> 4U);

        if(0U == (tempccc & SD_CCC_LOCK_CARD)) {
            /* don't support the lock command */
            status        = HAL_SD_FUNCTION_UNSUPPORTED;
            sd_dev->state = HAL_SD_STATE_READY;

        } else {
            /* password pattern */
            pwd1 = (0x01020600U | lockstate);
            pwd2 = 0x03040506U;

            /* clear all DSM configuration */
            hals_sdio_data_config(sd_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
            hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
            hals_sdio_dsm_disable(sd_dev->periph);
            hals_sdio_idma_disable(sd_dev->periph);

            /* send CMD16(SET_BLOCKLEN) to set the block length */
            hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, 8U, SDIO_RESPONSETYPE_SHORT);
            hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
            hals_sdio_csm_enable(sd_dev->periph);

            /* check if some error occurs */
            status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
            if(HAL_SD_OK != status) {
                sd_dev->state = HAL_SD_STATE_READY;

            } else {

                /* send CMD13(SEND_STATUS), addressed card sends its status register */
                hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS, \
                                                (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_CMD_SEND_STATUS);
                if(HAL_SD_OK != status) {
                    sd_dev->state = HAL_SD_STATE_READY;

                } else {

                    response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                    timeout  = 100000U;
                    while((0U == (response & SD_R1_READY_FOR_DATA)) && (timeout > 0U)) {
                        /* continue to send CMD13 to polling the state of card until buffer empty or timeout */
                        --timeout;
                        /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                        hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS, \
                                                        (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT,
                                                        SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(sd_dev->periph);
                        /* check if some error occurs */
                        status = _r1_error_check(sd_dev, SD_CMD_SEND_STATUS);
                        if(HAL_SD_OK != status) {
                            sd_dev->state = HAL_SD_STATE_READY;

                            break;
                        } else {
                        response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
                        }
                    }
                    if (HAL_SD_OK == status){
                        if(0U == timeout) {
                            status        = HAL_SD_ERROR;
                            sd_dev->state = HAL_SD_STATE_READY;

                        } else {

                            /* send CMD42(LOCK_UNLOCK) to set/reset the password or lock/unlock the card */
                            hals_sdio_command_response_config(sd_dev->periph, SD_CMD_LOCK_UNLOCK, 0U, SDIO_RESPONSETYPE_SHORT);
                            hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                            hals_sdio_csm_enable(sd_dev->periph);
                            /* check if some error occurs */
                            status = _r1_error_check(sd_dev, SD_CMD_LOCK_UNLOCK);
                            if(HAL_SD_OK != status) {
                                sd_dev->state = HAL_SD_STATE_READY;

                            } else {

                                response = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);

                                /* configure the SDIO data transmission */
                                hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, 8U, SDIO_DATABLOCKSIZE_8BYTES);
                                hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
                                hals_sdio_dsm_enable(sd_dev->periph);

                                /* write password pattern */
                                hals_sdio_data_write(sd_dev->periph, pwd1);
                                hals_sdio_data_write(sd_dev->periph, pwd2);

                                /* whether some error occurs and return it */
                                if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR)) {
                                    status = HAL_SD_DATA_CRC_ERROR;
                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTCRCERR);
                                    sd_dev->state = HAL_SD_STATE_READY;

                                } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTTMOUT)) {
                                    status = HAL_SD_DATA_TIMEOUT;
                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_DTTMOUT);
                                    sd_dev->state = HAL_SD_STATE_READY;

                                } else if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_TXURE)) {
                                    status = HAL_SD_TX_UNDERRUN_ERROR;
                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_TXURE);
                                    sd_dev->state = HAL_SD_STATE_READY;

                                } else {
                                    /* clear the SDIO_INTC flags */
                                    hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);
                                    /* get the card state and wait the card is out of programming and receiving state */
                                    status = _sdcard_cardstate_get(sd_dev, &cardstate);
                                    while((HAL_SD_OK == status) && ((SD_CARDSTATE_PROGRAMMING == cardstate) || (SD_CARDSTATE_RECEIVING == cardstate))) {
                                        status = _sdcard_cardstate_get(sd_dev, &cardstate);
                                    }

                                    sd_dev->state = HAL_SD_STATE_READY;
                                }
                            }
                        }
                    } else {
                        /* do nothing */
                    }
                }
            }
        }
    }

    return status;
}

/*!
    \brief      configure the bus mode
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  busmode: the bus mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_BUSMODE_1BIT: 1-bit SDIO card bus mode
      \arg        SDIO_BUSMODE_4BIT: 4-bit SDIO card bus mode
      \arg        SDIO_BUSMODE_8BIT: 8-bit SDIO card bus mode (MMC only)
    \param[in]  speed: the bus speed mode
                only one parameter can be selected which is shown as below:
      \arg        SD_SPEED_DEFAULT: default bus speed
      \arg        SD_SPEED_HIGH: high bus speed
      \arg        SD_SPEED_SDR25: SDR25 bus speed
      \arg        SD_SPEED_SDR50: SDR50 bus speed
      \arg        SD_SPEED_SDR104: SDR104 bus speed
      \arg        SD_SPEED_DDR50: DDR50 bus speed
    \param[out] none
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sdcard_bus_mode_config(hal_sdio_sdcard_dev_struct *sd_dev, uint32_t busmode, uint32_t speed)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    uint32_t count, clk_div;
    if(SDIO_MULTIMEDIA_CARD == sd_dev->card_info.card_type) {
        /* MMC card doesn't support this function */
        status = HAL_SD_FUNCTION_UNSUPPORTED;

    } else if((SDIO_STD_CAPACITY_SD_CARD_V1_1 == sd_dev->card_info.card_type) || \
              (SDIO_STD_CAPACITY_SD_CARD_V2_0 == sd_dev->card_info.card_type) || \
              (SDIO_HIGH_CAPACITY_SD_CARD == sd_dev->card_info.card_type)) {
        if(SDIO_BUSMODE_8BIT == busmode) {
            /* 8 bit bus mode doesn't support */
            status = HAL_SD_FUNCTION_UNSUPPORTED;

        } else if(SDIO_BUSMODE_4BIT == busmode) {
            /* configure SD bus width and the SDIO */
            status = _sd_bus_width_config(sd_dev, SD_BUS_WIDTH_4BIT);
            if(HAL_SD_OK == status) {
                hals_sdio_bus_mode_set(sd_dev->periph, busmode);
            } else {
                /* do nothing */
            }
        } else if(SDIO_BUSMODE_1BIT == busmode) {
            /* configure SD bus width and the SDIO */
            status = _sd_bus_width_config(sd_dev, SD_BUS_WIDTH_1BIT);
            if(HAL_SD_OK == status) {
                hals_sdio_bus_mode_set(sd_dev->periph, busmode);
            } else {
                /* do nothing */
            }
        } else {
            status = HAL_SD_PARAMETER_INVALID;
        }
    } else {
        /* do nothing */
    }

    if(HAL_SD_OK == status) {
        if((speed != SDIO_SPEED_DEFAULT) && (speed != SDIO_SPEED_HIGH)) {
            /* switch UHS-I speed mode */
            switch(speed) {
            case SDIO_SPEED_SDR25:
                clk_div = SD_CLK_DIV_TRANS_SDR25SPEED;
                break;
            case SDIO_SPEED_SDR50:
                clk_div = SD_CLK_DIV_TRANS_SDR50SPEED;
                break;
            case SDIO_SPEED_SDR104:
                clk_div = SD_CLK_DIV_TRANS_SDR104SPEED;
                break;
            case SDIO_SPEED_DDR50:
                clk_div = SD_CLK_DIV_TRANS_DDR50SPEED;
                break;
            default:
                clk_div = SD_CLK_DIV_TRANS_DSPEED;
                break;
            }

            /* send CMD16(SET_BLOCKLEN) to set the block length */
            hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SET_BLOCKLEN, 64U, SDIO_RESPONSETYPE_SHORT);
            hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
            hals_sdio_csm_enable(sd_dev->periph);

            /* check if some error occurs */
            status = _r1_error_check(sd_dev, SD_CMD_SET_BLOCKLEN);
            if(HAL_SD_OK == status) {

                hals_sdio_data_config(sd_dev->periph, SD_DATATIMEOUT, 64U, SDIO_DATABLOCKSIZE_64BYTES);
                hals_sdio_data_transfer_config(sd_dev->periph, SDIO_TRANSDIRECTION_TOSDIO, SDIO_TRANSMODE_BLOCKCOUNT);
                hals_sdio_dsm_enable(sd_dev->periph);

                /* SDR25 0x80FFFF01U  SDR50 0x80FF1F02U  SDR104 0x80FF1F03U  DDR50 0x80FF1F04U */
                hals_sdio_command_response_config(sd_dev->periph, SD_APPCMD_SET_BUS_WIDTH, speed, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(sd_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(sd_dev, SD_APPCMD_SET_BUS_WIDTH);
                if(HAL_SD_OK == status) {

                    while(!hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_DTCRCERR | SDIO_FLAG_DTTMOUT | SDIO_FLAG_RXORE | \
                                                            SDIO_FLAG_DTBLKEND | SDIO_FLAG_DTEND)) {
                        if(RESET != hals_sdio_flag_get(sd_dev->periph, SDIO_FLAG_RFH)) {
                            /* at least 8 words can be read in the FIFO */
                            for(count = 0U; count < SD_FIFOHALF_WORDS; count++) {
                                hals_sdio_data_read(sd_dev->periph);
                            }
                        } else {
                            /* do nothing */
                        }
                    }

                    /* clear the SDIO_INTC flags */
                    hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);
                    /* change the clock to UHS-I speed , user according to the speed configuration */
                    hals_sdio_clock_config(sd_dev->periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, clk_div);
                    if(speed == SDIO_SPEED_DDR50) {
                        /* set SDIO data rate */
                        hals_sdio_data_rate_set(sd_dev->periph, SDIO_DATA_RATE_DDR);
                    } else {
                        /* do nothing */
                    }
                    hals_sdio_hardware_clock_enable(sd_dev->periph);

                    if(SDCAED_VOLTAGE_18V == sd_dev->card_info.card_speed) {
                        hals_sdio_clock_receive_set(sd_dev->periph, SDIO_RECEIVECLOCK_FBCLK);
                        hals_sdio_hardware_clock_disable(sd_dev->periph);
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else if(speed == SDIO_SPEED_HIGH) {
            /* change the clock to high speed , user according to the speed configuration */
            hals_sdio_clock_config(sd_dev->periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, \
                                SD_CLK_DIV_TRANS_HSPEED);
            hals_sdio_hardware_clock_enable(sd_dev->periph);
        } else if(speed == SDIO_SPEED_DEFAULT) {
            /* change the clock to default speed , user according to the speed configuration */
            hals_sdio_clock_config(sd_dev->periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, \
                                SD_CLK_DIV_TRANS_DSPEED);
            hals_sdio_hardware_clock_enable(sd_dev->periph);
        } else {
            /* do nothing */
        }
    }

    return status;
}

/*!
    \brief      get the card state
    \param[in]  sd_dev: sdcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] pcardstate: a pointer that store the card state
    \retval     hal_sdio_sdcard_error_enum details refer to gd32h7xx_hal_sdio_sdcard.h
*/
static hal_sdio_sdcard_error_enum _sdcard_cardstate_get(hal_sdio_sdcard_dev_struct *sd_dev, uint8_t *pcardstate)
{
    hal_sdio_sdcard_error_enum status = HAL_SD_OK;
    __IO uint32_t reg_status = 0U, response = 0U;

    /* send CMD13(SEND_STATUS), addressed card sends its status register */
    hals_sdio_command_response_config(sd_dev->periph, SD_CMD_SEND_STATUS, \
                                      (uint32_t)sd_dev->card_info.card_rca << SD_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(sd_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(sd_dev->periph);

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(sd_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(sd_dev->periph);
    }

    /* check whether an error or timeout occurs or command response received */
    if(reg_status & SDIO_FLAG_CCRCERR) {
        status = HAL_SD_CMD_CRC_ERROR;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CCRCERR);
    } else if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_SD_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(sd_dev->periph, SDIO_FLAG_CMDTMOUT);
    } else {

        /* command response received, store the response command index */
        reg_status = (uint32_t)hals_sdio_command_index_get(sd_dev->periph);
        if(reg_status != (uint32_t)SD_CMD_SEND_STATUS) {
            status = HAL_SD_ILLEGAL_COMMAND;
        } else {

            /* clear all the SDIO_INTC flags */
            hals_sdio_flag_clear(sd_dev->periph, SDIO_MASK_INTC_FLAGS);
            /* get the SDIO response register 0 for checking */
            response    = hals_sdio_response_get(sd_dev->periph, SDIO_RESPONSE0);
            *pcardstate = (uint8_t)((response >> 9U) & 0x0000000FU);

            if(SD_ALLZERO == (response & SD_R1_ERROR_BITS)) {
                /* no error occurs, return SD_OK */
                status = HAL_SD_OK;
            } else {

                /* if some error occurs, return the error type */
                status = _r1_error_type_check(response);
            }
        }
    }

    return status;
}

/*!
    \brief      sd transmit/received handle
    \param[in]  sd_dev: pointer to a sd device information structure
    \param[out] none
    \retval     none
*/
static void _sd_transmit_received_handle(void *sd_dev)
{
    hal_sdio_sdcard_dev_struct *p_sd = sd_dev;
    hal_sdio_sdcard_user_cb p_func;

    __IO uint32_t errorstate;

    /* disable idma for idma transfer */
    hals_sdio_idma_disable(p_sd->periph);

    /* disable all the interrupts */
    hals_sdio_interrupt_disable(p_sd->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_DTEND | SDIO_INT_TFH | \
                                              SDIO_INT_RFH | SDIO_INT_TXURE | SDIO_INT_RXORE);
    hals_sdio_trans_start_disable(p_sd->periph);

    if((0U != (p_sd->transfer & SD_TRANSFER_READ_MULTIPLE_BLOCK)) || \
       (0U != (p_sd->transfer & SD_TRANSFER_WRITE_MULTIPLE_BLOCK))) {
        errorstate = _sd_transfer_stop(p_sd);
        if(HAL_SD_OK != errorstate) {
            if(NULL != p_sd->sd_irq.transfer_error_handle) {
                p_sd->sd_irq.transfer_error_handle(p_sd);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* clear all the static flags */
    hals_sdio_flag_clear(p_sd->periph, SDIO_MASK_DATA_FLAGS);

    if(0U != (p_sd->transfer & SD_TRANSFER_IT)) {
        if((0U != (p_sd->transfer & SD_TRANSFER_READ_SINGLE_BLOCK)) || \
           (0U != (p_sd->transfer & SD_TRANSFER_READ_MULTIPLE_BLOCK))) {
            p_func = (hal_sdio_sdcard_user_cb)p_sd->receive_complete_callback;
            if(NULL != p_func) {
                p_func(p_sd);
            } else {
                /* do nothing */
            }
        } else {
            p_func = (hal_sdio_sdcard_user_cb)p_sd->transmit_complete_callback;
            if(NULL != p_func) {
                p_func(p_sd);
            } else {
                /* do nothing */
            }
        }
    } else if(0U != (p_sd->transfer & SD_TRANSFER_DMA)) {
        if((0U != (p_sd->transfer & SD_TRANSFER_WRITE_SINGLE_BLOCK)) || \
           (0U != (p_sd->transfer & SD_TRANSFER_WRITE_MULTIPLE_BLOCK))) {
            p_func = (hal_sdio_sdcard_user_cb)p_sd->transmit_complete_callback;
            if(NULL != p_func) {
                p_func(p_sd);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        if((0U != (p_sd->transfer & SD_TRANSFER_READ_SINGLE_BLOCK)) || \
           (0U != (p_sd->transfer & SD_TRANSFER_READ_MULTIPLE_BLOCK))) {
            p_func = (hal_sdio_sdcard_user_cb)p_sd->receive_complete_callback;
            if(NULL != p_func) {
                p_func(p_sd);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    p_sd->state    = HAL_SD_STATE_READY;
    p_sd->transfer = SD_TRANSFER_NONE;
}

/*!
    \brief      sd idma error handle
    \param[in]  sd_dev: pointer to a sd device information structure
    \param[out] none
    \retval     none
*/
static void _sd_idma_complete_handle(void *sd_dev)
{
    hal_sdio_sdcard_dev_struct *p_sd = (hal_sdio_sdcard_dev_struct *)sd_dev;
    hal_sdio_sdcard_user_cb p_func = NULL;

    hals_sdio_interrupt_flag_clear(p_sd->periph, SDIO_INT_FLAG_IDMAEND);

    if(hals_sdio_buffer_selection_get(p_sd->periph) == SDIO_IDMA_BUFFER0)
    {
      /* Current buffer is buffer0, Transfer complete for buffer1 */
      if(0U != (p_sd->transfer & SD_TRANSFER_WRITE_MULTIPLE_BLOCK)) {
        p_func = (hal_sdio_sdcard_user_cb)p_sd->write_dma_buffer1_complete_callback;
        if(NULL != p_func) {
            p_func(p_sd);
        } else {
            /* do nothing */
        }
      } else {
        p_func = (hal_sdio_sdcard_user_cb)p_sd->read_dma_buffer1_complete_callback;
        if(NULL != p_func) {
            p_func(p_sd);
        } else {
            /* do nothing */
        }
      }
    } else {
      /* Current buffer is buffer1, Transfer complete for buffer0 */
      if(0U != (p_sd->transfer & SD_TRANSFER_WRITE_MULTIPLE_BLOCK)) {
        p_func = (hal_sdio_sdcard_user_cb)p_sd->write_dma_buffer0_complete_callback;
        if(NULL != p_func) {
            p_func(p_sd);
        } else {
            /* do nothing */
        }
      } else {
        p_func = (hal_sdio_sdcard_user_cb)p_sd->read_dma_buffer0_complete_callback;
        if(NULL != p_func) {
            p_func(p_sd);
        } else {
            /* do nothing */
        }
      }
    }
}

/*!
    \brief      sd error handle
    \param[in]  sd_dev: pointer to a sd device information structure
    \param[out] none
    \retval     none
*/
static void _sd_error_handle(void *sd_dev)
{
    hal_sdio_sdcard_dev_struct *p_sd = sd_dev;
    hal_sdio_sdcard_user_cb p_func   = (hal_sdio_sdcard_user_cb)p_sd->transfer_error_callback;

    __IO uint32_t errorstate;

    /* set different errors */
    if(RESET != hals_sdio_interrupt_flag_get(p_sd->periph, SDIO_INT_FLAG_DTCRCERR)) {
        errorstate = HAL_SD_DATA_CRC_ERROR;
    } else {
        /* do nothing */
    }

    if(RESET != hals_sdio_interrupt_flag_get(p_sd->periph, SDIO_INT_FLAG_DTTMOUT)) {
        errorstate = HAL_SD_DATA_TIMEOUT;
    } else {
        /* do nothing */
    }

    if(RESET != hals_sdio_interrupt_flag_get(p_sd->periph, SDIO_INT_FLAG_TXURE)) {
        errorstate = HAL_SD_TX_UNDERRUN_ERROR;
    } else {
        /* do nothing */
    }

    if(RESET != hals_sdio_interrupt_flag_get(p_sd->periph, SDIO_INT_FLAG_RXORE)) {
        errorstate = HAL_SD_RX_OVERRUN_ERROR;
    } else {
        /* do nothing */
    }

    /* clear data flags */
    hals_sdio_flag_clear(p_sd->periph, SDIO_MASK_DATA_FLAGS);
    /* disable all the interrupts */
    hals_sdio_interrupt_disable(p_sd->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                              SDIO_INT_DTEND | SDIO_INT_TXURE | SDIO_INT_RXORE);

    hals_sdio_trans_start_disable(p_sd->periph);
    hals_sdio_fifo_reset_enable(p_sd->periph);
    hals_sdio_fifo_reset_disable(p_sd->periph);

    /* send CMD12 to stop data transfer in multiple blocks operation */
    errorstate = _sd_transfer_stop(p_sd);
    hals_sdio_flag_clear(p_sd->periph, SDIO_FLAG_DTABORT);

    if(0U != (p_sd->transfer & SD_TRANSFER_IT)) {
        /* set the SD state to ready to be able to start again the process */
        p_sd->state    = HAL_SD_STATE_READY;
        p_sd->transfer = SD_TRANSFER_NONE;

        if(NULL != p_func) {
            p_func(p_sd);
        } else {
            /* do nothing */
        }
    } else if(0U != (p_sd->transfer & SD_TRANSFER_DMA)) {
        if(HAL_SD_OK != errorstate) {
            /* disable internal dma */
            hals_sdio_idma_disable(p_sd->periph);
            if(NULL != p_func) {
                p_func(p_sd);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}
