/*!
    \file    gd32h7xx_hal_sdio_emmc.c
    \brief   eMMC driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* check if the command sent error occurs */
static hal_sdio_emmc_error_enum _cmdsent_error_check(hal_sdio_emmc_dev_struct *emmc_dev);
/* check if error occurs for R1 response */
static hal_sdio_emmc_error_enum _r1_error_check(hal_sdio_emmc_dev_struct *emmc_dev, uint8_t cmdindex);
/* check if error type for R1 response */
static hal_sdio_emmc_error_enum _r1_error_type_check(uint32_t resp);
/* check if error occurs for R2 response */
static hal_sdio_emmc_error_enum _r2_error_check(hal_sdio_emmc_dev_struct *emmc_dev);
/* check if error occurs for R3 response */
static hal_sdio_emmc_error_enum _r3_error_check(hal_sdio_emmc_dev_struct *emmc_dev);
/* check if error occurs for R6 response */
static hal_sdio_emmc_error_enum _r6_error_check(hal_sdio_emmc_dev_struct *emmc_dev, uint8_t cmdindex, uint16_t *prca);
/* initialize the card and get CID and CSD of the card */
static hal_sdio_emmc_error_enum _emmc_card_init(hal_sdio_emmc_dev_struct *emmc_dev);
/* configure the clock and the work voltage, and get the card type */
static hal_sdio_emmc_error_enum _emmc_power_on(hal_sdio_emmc_dev_struct *emmc_dev);
/* select or deselect a card */
static hal_sdio_emmc_error_enum _emmc_card_select_deselect(hal_sdio_emmc_dev_struct *emmc_dev);
/* get the EXT_CSD register detailed information of the eMMC */
static hal_sdio_emmc_error_enum _emmc_card_extcsd_get(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *ext_csd);
/* get the EXT_CSD register detailed information of the eMMC */
static hal_sdio_emmc_error_enum _emmc_card_csd_get(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_cardcsd_struct *p_csd);
/* read the EXT_CSD register detailed information of the eMMC */
static hal_sdio_emmc_error_enum _emmc_card_extcsd_read(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *filed_data, \
                                                  uint16_t filed_index);
/* configure the bus width mode */
static hal_sdio_emmc_error_enum _emmc_bus_width_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t buswidth);
/* send transfer stop cmd */
static hal_sdio_emmc_error_enum _emmc_transfer_stop(hal_sdio_emmc_dev_struct *emmc_dev);
/* get the state which the card is in */
static hal_sdio_emmc_error_enum _emmc_cardstate_get(hal_sdio_emmc_dev_struct *emmc_dev, uint8_t *pcardstate);
/* configure the bus mode */
static hal_sdio_emmc_error_enum _emmc_bus_mode_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t busmode, uint32_t speed);
/* get the data block size */
static uint32_t _emmc_datablocksize_get(uint16_t bytesnumber);
/* eMMC deinit */
static void _emmc_deinit(uint32_t sdio_periph);
/* read block interrupt service */
static void _emmc_read_interrupt(void *emmc_dev);
/* write block interrupt service */
static void _emmc_write_interrupt(void *emmc_dev);
/* configure transmit received handle */
static void _emmc_transmit_received_handle(void *emmc_dev);
/* configure idma error handle */
static void _emmc_idma_complete_handle(void *emmc_dev);
/* configure error handle */
static void _emmc_error_handle(void *emmc_dev);
/* get value position */
static uint8_t _position_val_get(uint32_t data);

/*!
    \brief      initialize eMMC
    \param[in]  emmc_dev: eMMC device information structure
    \param[in]  periph: SDIO peripheral address
    \param[in]  p_emmc_init: the initialization data needed to initialize SDIO
                clock_division: Manual input is required and range is 0-3FF
                clock_edge: clock edge type
                only one parameter can be selected which is shown as below:
      \arg        SDIO_SDIOCLKEDGE_RISING: rising edge of SDIOCLK
      \arg        SDIO_SDIOCLKEDGE_FALLING: falling edge of SDIOCLK
                clock_powersave: clock power save mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_CLOCKPWRSAVE_DISABLE: clock power save disable
      \arg        SDIO_CLOCKPWRSAVE_ENABLE: clock power save enable
                bus_speed_mode: bus speed mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_BUSSPEED_LOW: low speed mode
      \arg        SDIO_BUSSPEED_HIGHT: high speed mode
                data_rate: data rate mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_DATA_RATE_SDR: single data rate
      \arg        SDIO_DATA_RATE_DDR: double data rate
                hardware_control: hardware flow control
                only one parameter can be selected which is shown as below:
      \arg        SDIO_HARDWARE_DISABLE: hardware flow control disable
      \arg        SDIO_HARDWARE_ENABLE: hardware flow control enable
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
hal_sdio_emmc_error_enum hal_sdio_emmc_init(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t periph, hal_sdio_emmc_init_struct *p_emmc_init)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    hal_sdio_emmc_cardcsd_struct card_csd;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == emmc_dev) || (NULL == p_emmc_init)) {
        HAL_DEBUGE("pointer [emmc_dev] or [p_emmc_init] address is invalid");
        return HAL_EMMC_ERROR;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    emmc_dev->periph = periph;

    /* configure the clock and work voltage */
    status = _emmc_power_on(emmc_dev);
    if(HAL_EMMC_OK == status) {

        /* initialize the card and get CID and CSD of the card */
        status = _emmc_card_init(emmc_dev);
        if(HAL_EMMC_OK == status) {

            /* configure the SDIO peripheral */
            hals_sdio_clock_config(emmc_dev->periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, EMMC_CLK_DIV_INIT);
            hals_sdio_bus_mode_set(emmc_dev->periph, SDIO_BUSMODE_1BIT);
            hals_sdio_hardware_clock_disable(emmc_dev->periph);

            /* configuration emmc card */
            status = _emmc_card_select_deselect(emmc_dev);
            if(HAL_EMMC_OK == status) {
                status = _emmc_card_csd_get(emmc_dev, &card_csd);
            } else {
                /* do nothing */
            }

            if(HAL_EMMC_OK == status) {
                status = _emmc_card_extcsd_get(emmc_dev, emmc_dev->ext_csd);
            } else {
                /* do nothing */
            }

            /* configuration emmc bus mode */
            if(HAL_EMMC_OK == status) {
                status = _emmc_bus_mode_config(emmc_dev, p_emmc_init->bus_width, p_emmc_init->data_rate);
            } else {
                /* do nothing */
            }

            /* set the SD state */
            emmc_dev->state = HAL_EMMC_STATE_READY;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
    return status;
}

/*!
    \brief      initialize the eMMC structure with the default values
    \param[in]  hal_struct_type: type of sdcard structure for initialization
                only one parameters can be selected which are shown as below:
      \arg        HAL_EMMC_INIT_STRUCT: initialization structure
      \arg        HAL_EMMC_DEV_STRUCT: device information structure
      \arg        HAL_EMMC_IRQ_INIT_STRUCT: interrupt callback initialization structure
    \param[out] p_struct: pointer to eMMC structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_struct_init(hal_sdio_emmc_struct_type_enum hal_struct_type, void *p_struct)
{
    int reval = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_EMMC_INIT_STRUCT:
        /* initialize SDIO initialization structure with the default values */
        ((hal_sdio_emmc_init_struct *)p_struct)->clock_division   = 0U;
        ((hal_sdio_emmc_init_struct *)p_struct)->clock_edge       = SDIO_SDIOCLKEDGE_RISING;
        ((hal_sdio_emmc_init_struct *)p_struct)->clock_powersave  = SDIO_CLOCKPWRSAVE_DISABLE;
        ((hal_sdio_emmc_init_struct *)p_struct)->bus_speed_mode   = SDIO_BUSSPEED_LOW;
        ((hal_sdio_emmc_init_struct *)p_struct)->hardware_control = SDIO_HARDWARE_DISABLE;
        ((hal_sdio_emmc_init_struct *)p_struct)->bus_width        = SDIO_BUSMODE_1BIT;
        ((hal_sdio_emmc_init_struct *)p_struct)->data_rate        = SDIO_DATA_RATE_DDR;
        break;
    case HAL_EMMC_DEV_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_emmc_dev_struct *)p_struct)->periph                              = 0U;
        ((hal_sdio_emmc_dev_struct *)p_struct)->p_txbuffer                          = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->tx_length                           = 0U;
        ((hal_sdio_emmc_dev_struct *)p_struct)->p_rxbuffer                          = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->rx_length                           = 0U;
        ((hal_sdio_emmc_dev_struct *)p_struct)->transfer                            = EMMC_TRANSFER_NONE;
        ((hal_sdio_emmc_dev_struct *)p_struct)->state                               = HAL_EMMC_STATE_RESET;
        ((hal_sdio_emmc_dev_struct *)p_struct)->error_state                         = HAL_EMMC_OK;
        ((hal_sdio_emmc_dev_struct *)p_struct)->transmit_complete_callback          = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->receive_complete_callback           = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->transfer_error_callback             = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->write_dma_buffer0_complete_callback = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->write_dma_buffer1_complete_callback = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->read_dma_buffer0_complete_callback  = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->read_dma_buffer1_complete_callback  = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->abort_complete_callback             = NULL;
        ((hal_sdio_emmc_dev_struct *)p_struct)->mutex                               = HAL_MUTEX_UNLOCKED;
        break;
    case HAL_EMMC_IRQ_INIT_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_emmc_irq_struct *)p_struct)->transmit_fifo_half_empty_handle = NULL;
        ((hal_sdio_emmc_irq_struct *)p_struct)->receive_fifo_half_empty_handle  = NULL;
        ((hal_sdio_emmc_irq_struct *)p_struct)->transfer_complete_handle        = NULL;
        ((hal_sdio_emmc_irq_struct *)p_struct)->idma_complete_handle            = NULL;
        ((hal_sdio_emmc_irq_struct *)p_struct)->transfer_error_handle           = NULL;
        break;
    case HAL_EMMC_CARDID_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_emmc_cardcid_struct *)p_struct)->manufacturerid = 0U;
        ((hal_sdio_emmc_cardcid_struct *)p_struct)->oem_appliid    = 0U;
        ((hal_sdio_emmc_cardcid_struct *)p_struct)->prodname1      = 0U;
        ((hal_sdio_emmc_cardcid_struct *)p_struct)->prodname2      = 0U;
        ((hal_sdio_emmc_cardcid_struct *)p_struct)->prodrev        = 0U;
        ((hal_sdio_emmc_cardcid_struct *)p_struct)->reserved1      = 0U;
        ((hal_sdio_emmc_cardcid_struct *)p_struct)->manufactdate   = 0U;
        ((hal_sdio_emmc_cardcid_struct *)p_struct)->cid_crc        = 0U;
        ((hal_sdio_emmc_cardcid_struct *)p_struct)->reserved2      = 1U;
        break;
    case HAL_EMMC_CARDSD_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->csdstruct           = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->sysspecversion      = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->reserved1           = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->taac                = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->nsac                = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->maxbusclkfrec       = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->cardcomdclasses     = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->rdblocklen          = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->partblockread       = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->wrblockmisalign     = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->rdblockmisalign     = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->dsrimpl             = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->reserved2           = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->devicesize          = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->maxrdcurrentvddmin  = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->maxrdcurrentvddmax  = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->maxwrcurrentvddmin  = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->maxwrcurrentvddmax  = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->devicesizemul       = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->erasegrsize         = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->erasegrmul          = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->wrprotectgrsize     = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->wrprotectgrenable   = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->mandeflecc          = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->wrspeedfact         = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->maxwrblocklen       = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->writeblockpapartial = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->reserved3           = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->contentprotectappli = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->fileformatgroup     = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->copyflag            = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->permwrprotect       = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->tempwrprotect       = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->fileformat          = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->ecc                 = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->csd_crc             = 0U;
        ((hal_sdio_emmc_cardcsd_struct *)p_struct)->reserved4           = 0U;
        break;
    case HAL_EMMC_CARD_INFO_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_emmc_cardinfo_struct *)p_struct)->card_type       = 0U;
        ((hal_sdio_emmc_cardinfo_struct *)p_struct)->card_speed      = 0U;
        ((hal_sdio_emmc_cardinfo_struct *)p_struct)->card_rca        = 0U;
        ((hal_sdio_emmc_cardinfo_struct *)p_struct)->card_blocknum   = 0U;
        ((hal_sdio_emmc_cardinfo_struct *)p_struct)->card_blocksize  = 0U;
        ((hal_sdio_emmc_cardinfo_struct *)p_struct)->logic_blocknum  = 0U;
        ((hal_sdio_emmc_cardinfo_struct *)p_struct)->logic_blocksize = 0U;
        break;
    case HAL_EMMC_IRQ_USER_CALLBACK_STRUCT:
        /* initialize SDIO device information structure with the default values */
        ((hal_sdio_emmc_irq_user_callback_struct *)p_struct)->transmit_complete_func          = NULL;
        ((hal_sdio_emmc_irq_user_callback_struct *)p_struct)->receive_complete_func           = NULL;
        ((hal_sdio_emmc_irq_user_callback_struct *)p_struct)->read_dma_buffer0_complete_func  = NULL;
        ((hal_sdio_emmc_irq_user_callback_struct *)p_struct)->read_dma_buffer1_complete_func  = NULL;
        ((hal_sdio_emmc_irq_user_callback_struct *)p_struct)->write_dma_buffer0_complete_func = NULL;
        ((hal_sdio_emmc_irq_user_callback_struct *)p_struct)->write_dma_buffer1_complete_func = NULL;
        ((hal_sdio_emmc_irq_user_callback_struct *)p_struct)->abort_complete_func             = NULL;
        ((hal_sdio_emmc_irq_user_callback_struct *)p_struct)->transfer_error_func             = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        reval = HAL_ERR_VAL;
        break;
    }

    return reval;
}

/*!
    \brief      deinitialize the eMMC
    \param[in]  emmc_dev: emmc device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_deinit(hal_sdio_emmc_dev_struct *emmc_dev)
{
    int32_t reval = HAL_ERR_NONE;
    uint32_t periph;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    periph = emmc_dev->periph;
    if((SDIO0 == periph) || (SDIO1 == periph)) {
        /* deinitialize the periph and the device information structure */
        _emmc_deinit(periph);
        emmc_dev->state = HAL_EMMC_STATE_RESET;
    } else {
        HAL_DEBUGE("parameter [emmc_dev->periph] value is invalid");
        reval = HAL_ERR_VAL;
    }

    return reval;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to emmc interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_irq_handle_set(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer and p_irq address */
    if((NULL == emmc_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [emmc_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set user-defined transmit fifo half complete interrupt callback */
    if(NULL != p_irq->transmit_fifo_half_empty_handle) {
        emmc_dev->emmc_irq.transmit_fifo_half_empty_handle = p_irq->transmit_fifo_half_empty_handle;
        hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_TFH | SDIO_INT_DTEND);
    } else {
        emmc_dev->emmc_irq.transmit_fifo_half_empty_handle = NULL;
        hals_sdio_interrupt_disable(emmc_dev->periph, SDIO_INT_TFH | SDIO_INT_DTEND);
    }

    /* set user-defined receive fifo half complete interrupt callback */
    if(NULL != p_irq->receive_fifo_half_empty_handle) {
        emmc_dev->emmc_irq.receive_fifo_half_empty_handle = p_irq->receive_fifo_half_empty_handle;
        hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_RFH | SDIO_INT_DTEND);
    } else {
        emmc_dev->emmc_irq.receive_fifo_half_empty_handle = NULL;
        hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_RFH | SDIO_INT_DTEND);
    }

    /* set user-defined transfer error interrupt callback */
    if(NULL != p_irq->transfer_error_handle) {
        emmc_dev->emmc_irq.transfer_error_handle = p_irq->transfer_error_handle;
        hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                                     SDIO_INT_TXURE | SDIO_INT_RXORE | SDIO_INT_IDMAERR);
    } else {
        emmc_dev->emmc_irq.transfer_error_handle = p_irq->transfer_error_handle;
        hals_sdio_interrupt_disable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                                      SDIO_INT_TXURE | SDIO_INT_RXORE | SDIO_INT_IDMAERR);
    }

    /* set user-defined internal DMA transfer complete interrupt callback */
    if(NULL != p_irq->idma_complete_handle) {
        emmc_dev->emmc_irq.idma_complete_handle = p_irq->idma_complete_handle;
        hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_IDMAEND | SDIO_INT_DTEND);
    } else {
        emmc_dev->emmc_irq.idma_complete_handle = NULL;
        hals_sdio_interrupt_disable(emmc_dev->periph,  SDIO_INT_IDMAEND | SDIO_INT_DTEND);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_irq_handle_all_reset(hal_sdio_emmc_dev_struct *emmc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    emmc_dev->emmc_irq.transmit_fifo_half_empty_handle    = NULL;
    emmc_dev->emmc_irq.receive_fifo_half_empty_handle     = NULL;
    emmc_dev->emmc_irq.transfer_complete_handle           = NULL;
    emmc_dev->emmc_irq.idma_complete_handle               = NULL;
    emmc_dev->emmc_irq.transfer_error_handle              = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      eMMC interrupt handler content function,which is merely used in SDIO_IRQHandler
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_sdio_emmc_irq(hal_sdio_emmc_dev_struct *emmc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* receive fifo half empty handle */
    if(RESET != hals_sdio_interrupt_flag_get(emmc_dev->periph, SDIO_INT_FLAG_RFH)) {
        hals_sdio_interrupt_flag_clear(emmc_dev->periph, SDIO_INT_FLAG_RFH);
        if(NULL != emmc_dev->emmc_irq.receive_fifo_half_empty_handle) {
            emmc_dev->emmc_irq.receive_fifo_half_empty_handle(emmc_dev);
        } else {
            /* do nothing */
        }
    } else if(RESET != hals_sdio_interrupt_flag_get(emmc_dev->periph, SDIO_INT_FLAG_DTEND)) {
        /* transmit/received complete handle */
        hals_sdio_interrupt_flag_clear(emmc_dev->periph, SDIO_INT_FLAG_DTEND);
        if(NULL != emmc_dev->emmc_irq.transfer_complete_handle) {
            emmc_dev->emmc_irq.transfer_complete_handle(emmc_dev);
        } else {
            /* do nothing */
        }
    } else if(RESET != hals_sdio_interrupt_flag_get(emmc_dev->periph, SDIO_INT_FLAG_TFH)) {
        hals_sdio_interrupt_flag_clear(emmc_dev->periph, SDIO_INT_FLAG_TFH);
        if(NULL != emmc_dev->emmc_irq.transmit_fifo_half_empty_handle) {
            emmc_dev->emmc_irq.transmit_fifo_half_empty_handle(emmc_dev);
        } else {
            /* do nothing */
        }
    } else if(RESET != hals_sdio_interrupt_flag_get(emmc_dev->periph, SDIO_INT_FLAG_DTCRCERR | SDIO_INT_FLAG_DTTMOUT | \
                                                                      SDIO_INT_FLAG_TXURE | SDIO_INT_FLAG_RXORE)) {
        /* transmit error handle */
        if(NULL != emmc_dev->emmc_irq.transfer_error_handle) {
            emmc_dev->emmc_irq.transfer_error_handle(emmc_dev);
        } else {
            /* do nothing */
        }
    } else if(RESET != hals_sdio_interrupt_flag_get(emmc_dev->periph, SDIO_INT_FLAG_IDMAEND)) {
        hals_sdio_interrupt_flag_clear(emmc_dev->periph, SDIO_INT_FLAG_IDMAEND);
        if(NULL != emmc_dev->emmc_irq.idma_complete_handle) {
            emmc_dev->emmc_irq.idma_complete_handle(emmc_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      configure the bus mode
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  busmode: the bus mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_BUSMODE_1BIT: 1-bit SDIO card bus mode
      \arg        SDIO_BUSMODE_4BIT: 4-bit SDIO card bus mode
      \arg        SDIO_BUSMODE_8BIT: 8-bit SDIO card bus mode
    \param[in]  speed: the bus speed mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_DATA_RATE_SDR: SDR selected
      \arg        SDIO_DATA_RATE_DDR: DDR selected

    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_bus_mode_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t busmode, uint32_t speed)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    status = _emmc_bus_mode_config(emmc_dev, busmode, speed);
    if(HAL_EMMC_OK != status) {
        reval = HAL_ERR_VAL;
    } else {
        reval = HAL_ERR_NONE;
    }
    return reval;
}

/*!
    \brief      configure dma dual buffer mode. the data transfer is managed by an internal dma
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  data_buffer0: pointer to the buffer0 that will contain/receive the transferred data
    \param[in]  data_buffer1: pointer to the buffer1 that will contain/receive the transferred data
    \param[in]  buffer_size: size of buffer
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_dma_buffer_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *data_buffer0, uint32_t *data_buffer1, \
                                        uint32_t buffer_size)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    hals_sdio_idma_set(emmc_dev->periph, SDIO_IDMA_DOUBLE_BUFFER, buffer_size);
    hals_sdio_idma_buffer0_address_set(emmc_dev->periph, (uint32_t)data_buffer0);
    hals_sdio_idma_buffer1_address_set(emmc_dev->periph, (uint32_t)data_buffer1);

    return HAL_ERR_NONE;
}

/*!
    \brief      change the dma buffer0 or buffer1 address on the fly
    \param[in]  emmc_dev: emmc device information structure
                 the structure is not necessary to be reconfigured after structure initialization,
                 the structure parameters altering is automatically configured by core
    \param[in]  buffer_select: the buffer to be used
                only one parameter can be selected which is shown as below:
      \arg        SDIO_IDMA_BUFFER0: select buffer0
      \arg        SDIO_IDMA_BUFFER1: select buffer1
    \param[in]  data_buffer: The new buffer address
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_change_dma_buffer(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t buffer_select, uint32_t *data_buffer)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(SDIO_IDMA_BUFFER0 == buffer_select) {
        hals_sdio_idma_buffer0_address_set(emmc_dev->periph, (uint32_t)data_buffer);
    } else {
        hals_sdio_idma_buffer1_address_set(emmc_dev->periph, (uint32_t)data_buffer);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      read data into a buffer from the specified address of a card
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  blockaddr: the read data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be read
    \param[out] pdata: a pointer that store multiple blocks read data
    \retval     error code: HAL_ERR_ABORT, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_readblocks_poll(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, uint32_t blockaddr, \
                                      uint16_t blocksize, uint16_t blocknumber)
{
    int32_t reval = HAL_ERR_NONE;
    /* initialize the variables */
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t count = 0U, align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE, *ptempbuff = pdata;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_READY != emmc_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        emmc_dev->error_state  = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > EMMC_MAX_DATA_LENGTH) {
        /* exceeds the maximum length */
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(emmc_dev);
        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        /* clear all DSM configuration */
        hals_sdio_data_config(emmc_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(emmc_dev->periph);
        hals_sdio_idma_disable(emmc_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) {
            emmc_dev->error_state = HAL_EMMC_LOCK_UNLOCK_FAILED;
            emmc_dev->state = HAL_EMMC_STATE_READY;
            HAL_UNLOCK(emmc_dev);
            reval = HAL_ERR_ABORT;
        } else {

            blocksize = 512U;

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _emmc_datablocksize_get(blocksize);
            } else {
                emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_VAL;
            }
            if(HAL_ERR_NONE == reval) {
                /* configure SDIO data transmission */
                hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, ((uint32_t)blocksize * blocknumber), datablksize);
                hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
                hals_sdio_trans_start_enable(emmc_dev->periph);

                if(blocknumber > 1U) {
                    emmc_dev->transfer = EMMC_TRANSFER_READ_MULTIPLE_BLOCK;
                    /* send CMD18(READ_MULTIPLE_BLOCK) to read multiple blocks */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_READ_MULTIPLE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);

                    /* check if some error occurs */
                    status = _r1_error_check(emmc_dev, EMMC_CMD_READ_MULTIPLE_BLOCK);
                    if(HAL_EMMC_OK != status) {
                        emmc_dev->state = HAL_EMMC_STATE_READY;
                        emmc_dev->transfer = EMMC_TRANSFER_NONE;
                        HAL_UNLOCK(emmc_dev);
                        reval = HAL_ERR_ABORT;
                    } else {
                        /* do nothing */
                    }
                } else {
                    emmc_dev->transfer = EMMC_TRANSFER_READ_SINGLE_BLOCK;
                    /* send CMD17(READ_SINGLE_BLOCK) to read a block */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_READ_SINGLE_BLOCK, (uint32_t)blockaddr,
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);
                    /* check if some error occurs */
                    status = _r1_error_check(emmc_dev, EMMC_CMD_READ_SINGLE_BLOCK);
                    if(HAL_EMMC_OK != status) {
                        emmc_dev->state = HAL_EMMC_STATE_READY;
                        emmc_dev->transfer = EMMC_TRANSFER_NONE;
                        HAL_UNLOCK(emmc_dev);
                        reval = HAL_ERR_ABORT;
                    } else {
                        /* do nothing */
                    }
                }
                if(HAL_ERR_NONE == reval) {
                    /* polling mode */
                    while(!hals_sdio_flag_get(emmc_dev->periph,
                                            SDIO_FLAG_DTCRCERR | SDIO_FLAG_DTTMOUT | SDIO_FLAG_RXORE | SDIO_FLAG_DTBLKEND | SDIO_FLAG_DTEND)) {
                        if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_RFH)) {
                            /* at least 8 words can be read in the FIFO */
                            for(count = 0U; count < EMMC_FIFOHALF_WORDS; count++) {
                                *(ptempbuff + count) = hals_sdio_data_read(emmc_dev->periph);
                            }
                            ptempbuff += EMMC_FIFOHALF_WORDS;
                        } else {
                            /* do nothing */
                        }
                    }

                    /* whether some error occurs and return it */
                    if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTCRCERR)) {
                        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DTCRCERR);
                        emmc_dev->error_state = HAL_EMMC_DATA_CRC_ERROR;
                        emmc_dev->state = HAL_EMMC_STATE_READY;
                        emmc_dev->transfer = EMMC_TRANSFER_NONE;
                        HAL_UNLOCK(emmc_dev);
                        reval = HAL_ERR_ABORT;
                    } else if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTTMOUT)) {
                        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DTTMOUT);
                        emmc_dev->error_state = HAL_EMMC_DATA_TIMEOUT;
                        emmc_dev->state = HAL_EMMC_STATE_READY;
                        emmc_dev->transfer = EMMC_TRANSFER_NONE;
                        HAL_UNLOCK(emmc_dev);
                        reval = HAL_ERR_ABORT;
                    } else if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_RXORE)) {
                        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_RXORE);
                        emmc_dev->error_state = HAL_EMMC_RX_OVERRUN_ERROR;
                        emmc_dev->state = HAL_EMMC_STATE_READY;
                        emmc_dev->transfer = EMMC_TRANSFER_NONE;
                        HAL_UNLOCK(emmc_dev);
                        reval = HAL_ERR_ABORT;
                    } else {

                        while((SET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_RFE))
                            && (SET == hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DATSTA))) {
                            *ptempbuff = hals_sdio_data_read(emmc_dev->periph);
                            ++ptempbuff;
                        }

                        hals_sdio_trans_start_disable(emmc_dev->periph);

                        if((RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTEND)) && (blocknumber > 1U)) {
                            /* send CMD12(STOP_TRANSMISSION) to stop transmission */
                            hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_STOP_TRANSMISSION, 0U, SDIO_RESPONSETYPE_SHORT);
                            hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                            hals_sdio_csm_enable(emmc_dev->periph);
                            /* check if some error occurs */
                            status = _r1_error_check(emmc_dev, EMMC_CMD_STOP_TRANSMISSION);
                            if(HAL_EMMC_OK != status) {
                                emmc_dev->state = HAL_EMMC_STATE_READY;
                                emmc_dev->transfer = EMMC_TRANSFER_NONE;
                                HAL_UNLOCK(emmc_dev);
                                reval = HAL_ERR_ABORT;
                            } else {
                                /* do nothing */
                            }
                        } else {

                            /* clear the DATA_FLAGS flags */
                            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_DATA_FLAGS);

                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            emmc_dev->transfer = EMMC_TRANSFER_NONE;

                            /* process unlocked */
                            HAL_UNLOCK(emmc_dev);
                        }
                    }
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      read data into a buffer from the specified address of a card by interrupt method
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  blockaddr: the read data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be read
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] pdata: a pointer that store multiple blocks read data
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ABORT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_readblocks_interrupt(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, uint32_t blockaddr, \
                                           uint16_t blocksize, uint16_t blocknumber, \
                                           hal_sdio_emmc_irq_user_callback_struct *p_user_func)
{
    int32_t reval = HAL_ERR_NONE;
    /* initialize the variables */
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_READY != emmc_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > EMMC_MAX_DATA_LENGTH) {
        /* exceeds the maximum length */
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(emmc_dev);
        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        blocksize = 512U;
        emmc_dev->p_rxbuffer = pdata;
        emmc_dev->rx_length = (uint32_t)blocksize * blocknumber;

        emmc_dev->emmc_irq.receive_fifo_half_empty_handle = &_emmc_read_interrupt;
        emmc_dev->emmc_irq.transfer_complete_handle = &_emmc_transmit_received_handle;
        emmc_dev->emmc_irq.transfer_error_handle = &_emmc_error_handle;

        emmc_dev->receive_complete_callback = NULL;
        emmc_dev->transfer_error_callback = NULL;
        emmc_dev->abort_complete_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->receive_complete_func) {
                emmc_dev->receive_complete_callback = (void *)p_user_func->receive_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                emmc_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                emmc_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(emmc_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(emmc_dev->periph);
        hals_sdio_idma_disable(emmc_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) {
            emmc_dev->error_state = HAL_EMMC_LOCK_UNLOCK_FAILED;
            emmc_dev->state = HAL_EMMC_STATE_READY;
            HAL_UNLOCK(emmc_dev);
            reval = HAL_ERR_ABORT;
        } else {

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _emmc_datablocksize_get(blocksize);
            } else {
                emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_ABORT;
            }
            if(HAL_ERR_NONE == reval) {
                /* configure SDIO data transmission */
                hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, ((uint32_t)blocksize * blocknumber), datablksize);
                hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
                hals_sdio_trans_start_enable(emmc_dev->periph);

                if(blocknumber > 1U) {
                    emmc_dev->transfer = (EMMC_TRANSFER_READ_MULTIPLE_BLOCK | EMMC_TRANSFER_IT);
                    /* send CMD18(READ_MULTIPLE_BLOCK) to read multiple blocks */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_READ_MULTIPLE_BLOCK, blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);

                    /* check if some error occurs */
                    status = _r1_error_check(emmc_dev, EMMC_CMD_READ_MULTIPLE_BLOCK);
                    if(HAL_EMMC_OK != status) {
                        emmc_dev->state = HAL_EMMC_STATE_READY;
                        emmc_dev->transfer = EMMC_TRANSFER_NONE;
                        HAL_UNLOCK(emmc_dev);
                        reval = HAL_ERR_ABORT;
                    } else {
                        /* do nothing */
                    }
                } else {
                    emmc_dev->transfer = (EMMC_TRANSFER_READ_SINGLE_BLOCK | EMMC_TRANSFER_IT);
                    /* send CMD17(READ_SINGLE_BLOCK) to read a block */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_READ_SINGLE_BLOCK, (uint32_t)blockaddr, \
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);
                    /* check if some error occurs */
                    status = _r1_error_check(emmc_dev, EMMC_CMD_READ_SINGLE_BLOCK);
                    if(HAL_EMMC_OK != status) {
                        emmc_dev->state = HAL_EMMC_STATE_READY;
                        emmc_dev->transfer = EMMC_TRANSFER_NONE;
                        HAL_UNLOCK(emmc_dev);
                        reval = HAL_ERR_ABORT;
                    } else {
                        /* do nothing */
                    }
                }
                if(HAL_ERR_NONE == reval) {
                    /* enable the SDIO corresponding interrupts */
                    hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                                                SDIO_INT_RFH | SDIO_INT_RXORE | SDIO_INT_DTEND);

                    /* process unlocked */
                    HAL_UNLOCK(emmc_dev);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      read data into a buffer from the specified address of a card by dma method
    \param[in]  emmc_dev: eMMC device information structure
                   the structure is not necessary to be reconfigured after structure initialization,
                   the structure parameters altering is automatically configured by core
    \param[in]  blockaddr: the read data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be read
    \param[in]  p_user_func: pointer to the emmc irq user callback struct
    \param[out] pdata: a pointer that store multiple blocks read data
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ABORT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_readblocks_dma(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, uint32_t blockaddr, \
                                     uint16_t blocksize, uint16_t blocknumber, \
                                     hal_sdio_emmc_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_READY != emmc_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > EMMC_MAX_DATA_LENGTH) {
        /* exceeds the maximum length */
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        blocksize = 512U;
        emmc_dev->p_rxbuffer = pdata;
        emmc_dev->rx_length = (uint32_t)blocksize * blocknumber;

        emmc_dev->emmc_irq.receive_fifo_half_empty_handle = &_emmc_read_interrupt;
        emmc_dev->emmc_irq.transfer_complete_handle = &_emmc_transmit_received_handle;
        emmc_dev->emmc_irq.transfer_error_handle = &_emmc_error_handle;

        emmc_dev->receive_complete_callback = NULL;
        emmc_dev->transfer_error_callback = NULL;
        emmc_dev->abort_complete_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->receive_complete_func) {
                emmc_dev->receive_complete_callback = (void *)p_user_func->receive_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                emmc_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                emmc_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(emmc_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(emmc_dev->periph);
        hals_sdio_idma_disable(emmc_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) {
            emmc_dev->error_state = HAL_EMMC_LOCK_UNLOCK_FAILED;
            emmc_dev->state = HAL_EMMC_STATE_READY;
            reval = HAL_ERR_ABORT;
        } else {

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _emmc_datablocksize_get(blocksize);
            } else {
                emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_ABORT;
            }
            if(HAL_ERR_NONE == reval) {
                /* DMA mode */
                hals_sdio_idma_set(emmc_dev->periph, SDIO_IDMA_SINGLE_BUFFER, (uint32_t)((uint32_t)blocksize >> 5U));
                hals_sdio_idma_buffer0_address_set(emmc_dev->periph, (uint32_t)pdata);
                hals_sdio_idma_enable(emmc_dev->periph);

                /* configure SDIO data transmission */
                hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, ((uint32_t)blocknumber * blocksize), datablksize);
                hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
                hals_sdio_trans_start_enable(emmc_dev->periph);

                if(blocknumber > 1U) {
                    emmc_dev->transfer = (EMMC_TRANSFER_READ_MULTIPLE_BLOCK | EMMC_TRANSFER_DMA);
                    /* send CMD18(READ_MULTIPLE_BLOCK) to read multiple blocks */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_READ_MULTIPLE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);
                    /* check if some error occurs */
                    status = _r1_error_check(emmc_dev, EMMC_CMD_READ_MULTIPLE_BLOCK);

                } else {
                    emmc_dev->transfer = (EMMC_TRANSFER_READ_SINGLE_BLOCK | EMMC_TRANSFER_DMA);
                    /* send CMD17(READ_SINGLE_BLOCK) to read a block */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_READ_SINGLE_BLOCK, (uint32_t)blockaddr,
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);
                    /* check if some error occurs */
                    status = _r1_error_check(emmc_dev, EMMC_CMD_READ_SINGLE_BLOCK);
                }
                if(HAL_EMMC_OK != status) {
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                    reval = HAL_ERR_ABORT;
                } else {

                    /* enable the SDIO corresponding interrupts and DMA function */
                    hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_RXORE | SDIO_INT_DTEND);
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      read data into a buffer from the specified address of a card by dma method
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  pbuf0: pointer to the idma buffer0
    \param[in]  pbuf1: pointer to the idma buffer1
    \param[in]  blockaddr: the read data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be read
    \param[in]  p_user_func: pointer to the emmc irq user callback struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ABORT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_readblocks_multibuffer_dma(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pbuf0, uint32_t *pbuf1, \
                                                uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                                hal_sdio_emmc_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_READY != emmc_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pbuf0) || (NULL == pbuf1) || (0U == blocknumber)) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > EMMC_MAX_DATA_LENGTH) {
        /* exceeds the maximum length */
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        emmc_dev->emmc_irq.receive_fifo_half_empty_handle = &_emmc_read_interrupt;
        emmc_dev->emmc_irq.idma_complete_handle = &_emmc_idma_complete_handle;
        emmc_dev->emmc_irq.transfer_error_handle = &_emmc_error_handle;

        emmc_dev->read_dma_buffer0_complete_callback = NULL;
        emmc_dev->read_dma_buffer1_complete_callback = NULL;
        emmc_dev->abort_complete_callback = NULL;
        emmc_dev->transfer_error_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->read_dma_buffer0_complete_func) {
                emmc_dev->read_dma_buffer1_complete_callback = (void *)p_user_func->read_dma_buffer0_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->read_dma_buffer1_complete_func) {
                emmc_dev->read_dma_buffer1_complete_callback = (void *)p_user_func->read_dma_buffer1_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                emmc_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                emmc_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(emmc_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(emmc_dev->periph);
        hals_sdio_idma_disable(emmc_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) {
            emmc_dev->error_state = HAL_EMMC_LOCK_UNLOCK_FAILED;
            emmc_dev->state = HAL_EMMC_STATE_READY;
            reval = HAL_ERR_ABORT;
        } else {

            blocksize = 512U;
            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _emmc_datablocksize_get(blocksize);
            } else {
                emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_ABORT;
            }
            if(HAL_ERR_NONE == reval) {
                /* DMA mode */
                hals_sdio_idma_set(emmc_dev->periph, SDIO_IDMA_DOUBLE_BUFFER, (uint32_t)((uint32_t)blocksize >> 5U));
                hals_sdio_idma_buffer0_address_set(emmc_dev->periph, (uint32_t)pbuf0);
                hals_sdio_idma_buffer1_address_set(emmc_dev->periph, (uint32_t)pbuf1);
                hals_sdio_idma_enable(emmc_dev->periph);

                /* configure SDIO data transmission */
                hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, ((uint32_t)blocknumber * blocksize), datablksize);
                hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
                hals_sdio_trans_start_enable(emmc_dev->periph);

                emmc_dev->transfer = (EMMC_TRANSFER_READ_MULTIPLE_BLOCK | EMMC_TRANSFER_DMA);
                /* send CMD18(READ_MULTIPLE_BLOCK) to read multiple blocks */
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_READ_MULTIPLE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(emmc_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(emmc_dev, EMMC_CMD_READ_MULTIPLE_BLOCK);
                if(HAL_EMMC_OK != status) {
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                    reval = HAL_ERR_ABORT;
                } else {

                    /* enable the SDIO corresponding interrupts and DMA function */
                    hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_RXORE | \
                                                            SDIO_INT_DTEND | SDIO_INT_IDMAERR | SDIO_INT_IDMAEND);
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      write data to the specified address of a card
    \param[in]  emmc_dev: SD eMMC device information structure
    \param[in]  pdata: a pointer that store multiple blocks data to be transferred
    \param[in]  blockaddr: the write data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be written
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ABORT, HAL_ERR_VAL, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_writeblocks_poll(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, uint32_t blockaddr, \
                                       uint16_t blocksize, uint16_t blocknumber)
{
    int32_t reval = HAL_ERR_NONE;
    /* initialize the variables */
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint8_t cardstate = 0U;
    uint32_t count = 0U, align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE, *ptempbuff = pdata;
    uint32_t transbytes = 0U, restwords = 0U, response = 0U;
    __IO uint32_t timeout = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_READY != emmc_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > EMMC_MAX_DATA_LENGTH) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(emmc_dev);
        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        /* clear all DSM configuration */
        hals_sdio_data_config(emmc_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(emmc_dev->periph);
        hals_sdio_idma_disable(emmc_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) {
            emmc_dev->error_state = HAL_EMMC_LOCK_UNLOCK_FAILED;
            emmc_dev->state = HAL_EMMC_STATE_READY;
            HAL_UNLOCK(emmc_dev);
            reval = HAL_ERR_ABORT;
        } else {

            blocksize = 512U;

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _emmc_datablocksize_get(blocksize);
            } else {
                emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_ABORT;
            }
            if(HAL_ERR_NONE == reval) {
                /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(emmc_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                if(HAL_EMMC_OK != status) {
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    reval = HAL_ERR_ABORT;
                } else {

                    response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                    timeout = 100000U;

                    while((0U == (response & EMMC_R1_READY_FOR_DATA)) && (timeout > 0U)) {
                        /* continue to send CMD13 to polling the state of card until buffer empty or timeout */
                        --timeout;
                        /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                        (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(emmc_dev->periph);
                        /* check if some error occurs */
                        status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                        if(HAL_EMMC_OK != status) {
                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            HAL_UNLOCK(emmc_dev);
                            reval = HAL_ERR_ABORT;
                            break;
                        } else {
                            response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                        }
                    }
                    if(HAL_ERR_NONE == reval) {
                        if(0U == timeout) {
                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            HAL_UNLOCK(emmc_dev);
                            reval = HAL_ERR_TIMEOUT;
                        } else {

                        }
                        if(HAL_ERR_NONE == reval) {
                            if(blocknumber > 1U) {
                                /* send CMD23(EMMC_SET_BLOCK_COUNT) to set the number of write blocks to be preerased before writing */
                                hals_sdio_command_response_config(emmc_dev->periph, EMMC_SET_BLOCK_COUNT, (uint32_t)blocknumber, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(emmc_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(emmc_dev, EMMC_SET_BLOCK_COUNT);
                                if(HAL_EMMC_OK != status) {
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    HAL_UNLOCK(emmc_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                /* do nothing */
                            }

                            /* configure the SDIO data transmission */
                            hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, ((uint32_t)blocknumber * blocksize), datablksize);
                            hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
                            hals_sdio_trans_start_enable(emmc_dev->periph);

                            if(blocknumber > 1U) {
                                emmc_dev->transfer = EMMC_TRANSFER_WRITE_MULTIPLE_BLOCK;
                                /* send CMD25(WRITE_MULTIPLE_BLOCK) to continuously write blocks of data */
                                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_WRITE_MULTIPLE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(emmc_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(emmc_dev, EMMC_CMD_WRITE_MULTIPLE_BLOCK);
                                if(HAL_EMMC_OK != status) {
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                                    HAL_UNLOCK(emmc_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                emmc_dev->transfer = EMMC_TRANSFER_WRITE_SINGLE_BLOCK;
                                /* send CMD24(WRITE_BLOCK) to write a block */
                                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_WRITE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(emmc_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(emmc_dev, EMMC_CMD_WRITE_BLOCK);
                                if(HAL_EMMC_OK != status) {
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                                    HAL_UNLOCK(emmc_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* do nothing */
                                }
                            }
                            if(HAL_ERR_NONE == reval) {
                                /* polling mode */
                                while(!hals_sdio_flag_get(emmc_dev->periph,
                                                        SDIO_FLAG_DTCRCERR | SDIO_FLAG_DTTMOUT | SDIO_FLAG_TXURE | SDIO_FLAG_DTBLKEND | SDIO_FLAG_DTEND)) {
                                    if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_TFH)) {
                                        /* at least 8 words can be written into the FIFO */
                                        if((((uint32_t)blocknumber * blocksize) - transbytes) < EMMC_FIFOHALF_BYTES) {
                                            restwords = (((uint32_t)blocknumber * blocksize) - transbytes + 3U) / 4U;
                                            for(count = 0U; count < restwords; count++) {
                                                hals_sdio_data_write(emmc_dev->periph, *ptempbuff);
                                                ++ptempbuff;
                                            }
                                            ptempbuff -= restwords;
                                        } else {
                                            for(count = 0U; count < EMMC_FIFOHALF_WORDS; count++) {
                                                hals_sdio_data_write(emmc_dev->periph, *(ptempbuff + count));
                                            }
                                            /* 8 words(32 bytes) has been transferred */
                                            ptempbuff += EMMC_FIFOHALF_WORDS;
                                            transbytes += EMMC_FIFOHALF_BYTES;
                                        }
                                    } else {
                                        /* do nothing */
                                    }
                                }
                                hals_sdio_trans_start_disable(emmc_dev->periph);

                                /* whether some error occurs and return it */
                                if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTCRCERR)) {
                                    hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DTCRCERR);
                                    emmc_dev->error_state = HAL_EMMC_DATA_CRC_ERROR;
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                                    HAL_UNLOCK(emmc_dev);
                                    reval = HAL_ERR_ABORT;
                                } else if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTTMOUT)) {
                                    hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DTTMOUT);
                                    emmc_dev->error_state = HAL_EMMC_DATA_TIMEOUT;
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                                    HAL_UNLOCK(emmc_dev);
                                    reval = HAL_ERR_ABORT;
                                } else if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_TXURE)) {
                                    hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_TXURE);
                                    emmc_dev->error_state = HAL_EMMC_TX_UNDERRUN_ERROR;
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                                    HAL_UNLOCK(emmc_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {

                                    /* clear the DATA_FLAGS flags */
                                    hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_DATA_FLAGS);
                                    /* get the card state and wait the card is out of programming and receiving state */
                                    status = _emmc_cardstate_get(emmc_dev, &cardstate);
                                    while((HAL_EMMC_OK == status) && ((EMMC_CARDSTATE_PROGRAMMING == cardstate)
                                                                    || (EMMC_CARDSTATE_RECEIVING == cardstate))) {
                                        status = _emmc_cardstate_get(emmc_dev, &cardstate);
                                    }

                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    emmc_dev->transfer = EMMC_TRANSFER_NONE;

                                    /* process unlocked */
                                    HAL_UNLOCK(emmc_dev);
                                }
                            } else {
                                /* do nothing */
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      write data to the specified address of a card by interrupt method
    \param[in]  emmc_dev: SD eMMC device information structure
    \param[in]  pdata: a pointer that store multiple blocks data to be transferred
    \param[in]  blockaddr: the write data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be written
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ABORT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_writeblocks_interrupt(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, uint32_t blockaddr, \
                                            uint16_t blocksize, uint16_t blocknumber, \
                                            hal_sdio_emmc_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
    uint32_t response = 0U;
    __IO uint32_t timeout = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_READY != emmc_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > EMMC_MAX_DATA_LENGTH) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(emmc_dev);

        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        blocksize = 512U;
        emmc_dev->p_txbuffer = pdata;
        emmc_dev->tx_length = (uint32_t)blocksize * blocknumber;

        emmc_dev->emmc_irq.transmit_fifo_half_empty_handle = &_emmc_write_interrupt;
        emmc_dev->emmc_irq.transfer_complete_handle = &_emmc_transmit_received_handle;
        emmc_dev->emmc_irq.transfer_error_handle = &_emmc_error_handle;

        emmc_dev->transmit_complete_callback = NULL;
        emmc_dev->transfer_error_callback = NULL;
        emmc_dev->abort_complete_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->transmit_complete_func) {
                emmc_dev->transmit_complete_callback = (void *)p_user_func->transmit_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                emmc_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                emmc_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(emmc_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(emmc_dev->periph);
        hals_sdio_idma_disable(emmc_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) {
            emmc_dev->error_state = HAL_EMMC_LOCK_UNLOCK_FAILED;
            emmc_dev->state = HAL_EMMC_STATE_READY;
            HAL_UNLOCK(emmc_dev);
            reval = HAL_ERR_ABORT;
        } else {

            blocksize = 512U;
            emmc_dev->p_txbuffer = pdata;
            emmc_dev->tx_length = (uint32_t)blocksize * blocknumber;

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _emmc_datablocksize_get(blocksize);
            } else {
                emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_ABORT;
            }
            if(HAL_ERR_NONE == reval) {
                /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(emmc_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                if(HAL_EMMC_OK != status) {
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    HAL_UNLOCK(emmc_dev);
                    reval = HAL_ERR_ABORT;
                } else {

                    response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                    timeout = 100000U;

                    while((0U == (response & EMMC_R1_READY_FOR_DATA)) && (timeout > 0U)) {
                        /* continue to send CMD13 to polling the state of card until buffer empty or timeout */
                        --timeout;
                        /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                        (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(emmc_dev->periph);
                        /* check if some error occurs */
                        status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                        if(HAL_EMMC_OK != status) {
                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            HAL_UNLOCK(emmc_dev);
                            reval = HAL_ERR_ABORT;
                            break;
                        } else {
                            response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                        }
                    }
                    if(HAL_ERR_NONE == reval) {
                        if(0U == timeout) {
                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            HAL_UNLOCK(emmc_dev);
                            reval = HAL_ERR_ABORT;
                        } else {
                            /* do nothing */
                        }

                        if(HAL_ERR_NONE == reval) {
                            if(blocknumber > 1U) {
                                /* send CMD23(EMMC_SET_BLOCK_COUNT) to set the number of write blocks to be preerased before writing */
                                hals_sdio_command_response_config(emmc_dev->periph, EMMC_SET_BLOCK_COUNT, (uint32_t)blocknumber, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(emmc_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(emmc_dev, EMMC_SET_BLOCK_COUNT);
                                if(HAL_EMMC_OK != status) {
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    HAL_UNLOCK(emmc_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {
                                        /* do nothing */
                                }
                            } else {
                                /* do nothing */
                            }

                            /* configure the SDIO data transmission */
                            hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, ((uint32_t)blocknumber * blocksize), datablksize);
                            hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
                            hals_sdio_trans_start_enable(emmc_dev->periph);

                            if(blocknumber > 1U) {
                                emmc_dev->transfer = (EMMC_TRANSFER_WRITE_MULTIPLE_BLOCK | EMMC_TRANSFER_IT);
                                /* send CMD25(WRITE_MULTIPLE_BLOCK) to continuously write blocks of data */
                                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_WRITE_MULTIPLE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(emmc_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(emmc_dev, EMMC_CMD_WRITE_MULTIPLE_BLOCK);
                                if(HAL_EMMC_OK != status) {
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                emmc_dev->transfer = (EMMC_TRANSFER_WRITE_SINGLE_BLOCK | EMMC_TRANSFER_IT);
                                /* send CMD24(WRITE_BLOCK) to write a block */
                                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_WRITE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(emmc_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(emmc_dev, EMMC_CMD_WRITE_BLOCK);
                                if(HAL_EMMC_OK != status) {
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* do nothing */
                                }
                            }
                            if(HAL_ERR_NONE == reval) {
                                /* enable the SDIO corresponding interrupts and DMA */
                                hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                                        SDIO_INT_TFH | SDIO_INT_TXURE | SDIO_INT_DTEND);

                                /* process unlocked */
                                HAL_UNLOCK(emmc_dev);
                            } else {
                                /* do nothing */
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval ;
}

/*!
    \brief      write data to the specified address of a card by dma method
    \param[in]  emmc_dev: SD eMMC device information structure
    \param[in]  pdata: a pointer that store multiple blocks data to be transferred
    \param[in]  blockaddr: the write data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be written
    \param[in]  p_user_func: pointer to the emmc irq user callback struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_writeblocks_dma(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, uint32_t blockaddr, \
                                      uint16_t blocksize, uint16_t blocknumber, \
                                      hal_sdio_emmc_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
    uint32_t response = 0U;
    __IO uint32_t timeout = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_READY != emmc_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pdata) || (0U == blocknumber)) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > EMMC_MAX_DATA_LENGTH) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(emmc_dev);

        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        blocksize = 512U;
        emmc_dev->p_txbuffer = pdata;
        emmc_dev->tx_length = (uint32_t)blocksize * blocknumber;

        emmc_dev->emmc_irq.transmit_fifo_half_empty_handle = &_emmc_write_interrupt;
        emmc_dev->emmc_irq.transfer_complete_handle = &_emmc_transmit_received_handle;
        emmc_dev->emmc_irq.transfer_error_handle = &_emmc_error_handle;

        emmc_dev->transmit_complete_callback = NULL;
        emmc_dev->abort_complete_callback = NULL;
        emmc_dev->transfer_error_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->transmit_complete_func) {
                emmc_dev->transmit_complete_callback = (void *)p_user_func->transmit_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                emmc_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                emmc_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(emmc_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(emmc_dev->periph);
        hals_sdio_idma_disable(emmc_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) {
            emmc_dev->error_state = HAL_EMMC_LOCK_UNLOCK_FAILED;
            emmc_dev->state = HAL_EMMC_STATE_READY;
            HAL_UNLOCK(emmc_dev);
            reval = HAL_ERR_ABORT;
        } else {

            blocksize = 512U;
            emmc_dev->p_txbuffer = pdata;
            emmc_dev->tx_length = (uint32_t)blocksize * blocknumber;

            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _emmc_datablocksize_get(blocksize);
            } else {
                emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_ABORT;
            }
            if(HAL_ERR_NONE == reval) {
                /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(emmc_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                if(HAL_EMMC_OK != status) {
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    HAL_UNLOCK(emmc_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                    timeout = 100000U;

                    while((0U == (response & EMMC_R1_READY_FOR_DATA)) && (timeout > 0U)) {
                        /* continue to send CMD13 to polling the state of card until buffer empty or timeout */
                        --timeout;
                        /* send CMD13(SEND_STATUS), addressed card sends its status registers */
                        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                        (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(emmc_dev->periph);
                        /* check if some error occurs */
                        status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                        if(HAL_EMMC_OK != status) {
                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            HAL_UNLOCK(emmc_dev);
                            reval = HAL_ERR_ABORT;
                            break;
                        } else {
                            response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                        }
                    }
                    if(HAL_ERR_NONE == reval) {
                        if(0U == timeout) {
                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            HAL_UNLOCK(emmc_dev);
                            reval = HAL_ERR_ABORT;
                        }else {
                            /* do nothing */
                        }
                        if(HAL_ERR_NONE == reval) {
                            if(blocknumber > 1U) {
                                /* send CMD23(EMMC_SET_BLOCK_COUNT) to set the number of write blocks to be preerased before writing */
                                hals_sdio_command_response_config(emmc_dev->periph, EMMC_SET_BLOCK_COUNT, (uint32_t)blocknumber, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(emmc_dev->periph);
                                /* check if some error occurs */
                                status = _r1_error_check(emmc_dev, EMMC_SET_BLOCK_COUNT);
                                if(HAL_EMMC_OK != status) {
                                    emmc_dev->state = HAL_EMMC_STATE_READY;
                                    HAL_UNLOCK(emmc_dev);
                                    reval = HAL_ERR_ABORT;
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                /* do nothing */
                            }

                            /* DMA mode */
                            hals_sdio_idma_set(emmc_dev->periph, SDIO_IDMA_SINGLE_BUFFER, (uint32_t)((uint32_t)blocksize >> 5U));
                            hals_sdio_idma_buffer0_address_set(emmc_dev->periph, (uint32_t)pdata);
                            hals_sdio_idma_enable(emmc_dev->periph);

                            /* configure the SDIO data transmission */
                            hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, ((uint32_t)blocksize * blocknumber), datablksize);
                            hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
                            hals_sdio_trans_start_enable(emmc_dev->periph);

                            emmc_dev->transfer = (EMMC_TRANSFER_WRITE_MULTIPLE_BLOCK | EMMC_TRANSFER_DMA);
                            /* send CMD25(WRITE_MULTIPLE_BLOCK) to continuously write blocks of data */
                            hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_WRITE_MULTIPLE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                            hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                            hals_sdio_csm_enable(emmc_dev->periph);
                            /* check if some error occurs */
                            status = _r1_error_check(emmc_dev, EMMC_CMD_WRITE_MULTIPLE_BLOCK);

                            if(HAL_EMMC_OK != status) {
                                /* clear the DATA_FLAGS flags */
                                hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
                                emmc_dev->state = HAL_EMMC_STATE_READY;
                                emmc_dev->transfer = EMMC_TRANSFER_NONE;
                                HAL_UNLOCK(emmc_dev);
                                reval = HAL_ERR_ABORT;
                            } else {
                                /* enable the SDIO corresponding interrupts and DMA */
                                hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_TXURE | SDIO_INT_DTEND);

                                /* process unlocked */
                                HAL_UNLOCK(emmc_dev);
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      write data to the specified address of a card by dma method
    \param[in]  emmc_dev: SD eMMC device information structure
    \param[in]  pbuf0: pointer to the idma buffer0
    \param[in]  pbuf1: pointer to the idma buffer1
    \param[in]  blockaddr: the write data address
    \param[in]  blocksize: the data block size
    \param[in]  blocknumber: number of blocks that will be written
    \param[in]  p_user_func: pointer to the emmc irq user callback struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_writeblocks_multibuffer_dma(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pbuf0, uint32_t *pbuf1, \
                                                  uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                                  hal_sdio_emmc_irq_user_callback_struct *p_user_func)
{
    /* initialize the variables */
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t align = 0U, datablksize = SDIO_DATABLOCKSIZE_1BYTE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_READY != emmc_dev->state) {
        reval = HAL_ERR_BUSY;
    } else if((NULL == pbuf0) || (NULL == pbuf1) || (0U == blocknumber)) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else if((uint32_t)blocknumber * blocksize > EMMC_MAX_DATA_LENGTH) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* process locked */
        HAL_LOCK(emmc_dev);

        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        emmc_dev->emmc_irq.transmit_fifo_half_empty_handle = &_emmc_write_interrupt;
        emmc_dev->emmc_irq.transfer_complete_handle = &_emmc_transmit_received_handle;
        emmc_dev->emmc_irq.idma_complete_handle = &_emmc_idma_complete_handle;
        emmc_dev->emmc_irq.transfer_error_handle = &_emmc_error_handle;

        emmc_dev->write_dma_buffer0_complete_callback = NULL;
        emmc_dev->write_dma_buffer1_complete_callback = NULL;
        emmc_dev->abort_complete_callback = NULL;
        emmc_dev->transfer_error_callback = NULL;

        if(NULL != p_user_func) {
            if(NULL != p_user_func->write_dma_buffer0_complete_func) {
                emmc_dev->write_dma_buffer0_complete_callback = (void *)p_user_func->write_dma_buffer0_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->write_dma_buffer1_complete_func) {
                emmc_dev->write_dma_buffer1_complete_callback = (void *)p_user_func->write_dma_buffer1_complete_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->transfer_error_func) {
                emmc_dev->transfer_error_callback = (void *)p_user_func->transfer_error_func;
            } else {
                /* do nothing */
            }
            if(NULL != p_user_func->abort_complete_func) {
                emmc_dev->abort_complete_callback = (void *)p_user_func->abort_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all DSM configuration */
        hals_sdio_data_config(emmc_dev->periph, 0U, 0U, SDIO_DATABLOCKSIZE_1BYTE);
        hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
        hals_sdio_dsm_disable(emmc_dev->periph);
        hals_sdio_idma_disable(emmc_dev->periph);

        /* check whether the card is locked */
        if(hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) {
            emmc_dev->error_state = HAL_EMMC_LOCK_UNLOCK_FAILED;
            emmc_dev->state = HAL_EMMC_STATE_READY;
            HAL_UNLOCK(emmc_dev);
            reval = HAL_ERR_ABORT;
        } else {

            blocksize = 512U;
            align = (uint32_t)blocksize & ((uint32_t)blocksize - 1U);
            if((blocksize > 0U) && (blocksize <= 2048U) && (0U == align)) {
                datablksize = _emmc_datablocksize_get(blocksize);
            } else {
                emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_ABORT;
            }
            if(HAL_ERR_NONE == reval) {
                /* DMA mode */
                hals_sdio_idma_set(emmc_dev->periph, SDIO_IDMA_DOUBLE_BUFFER, (uint32_t)((uint32_t)blocksize >> 5U));
                hals_sdio_idma_buffer0_address_set(emmc_dev->periph, (uint32_t)pbuf0);
                hals_sdio_idma_buffer1_address_set(emmc_dev->periph, (uint32_t)pbuf1);
                hals_sdio_idma_enable(emmc_dev->periph);

                /* configure the SDIO data transmission */
                hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, ((uint32_t)blocksize * blocknumber), datablksize);
                hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOCARD);
                hals_sdio_trans_start_enable(emmc_dev->periph);

                emmc_dev->transfer = (EMMC_TRANSFER_WRITE_MULTIPLE_BLOCK | EMMC_TRANSFER_DMA);
                /* send CMD25(WRITE_MULTIPLE_BLOCK) to continuously write blocks of data */
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_WRITE_MULTIPLE_BLOCK, blockaddr, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(emmc_dev->periph);
                /* check if some error occurs */
                status = _r1_error_check(emmc_dev, EMMC_CMD_WRITE_MULTIPLE_BLOCK);

                if(HAL_EMMC_OK != status) {
                    /* clear the DATA_FLAGS flags */
                    hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    emmc_dev->transfer = EMMC_TRANSFER_NONE;
                    HAL_UNLOCK(emmc_dev);
                    reval = HAL_ERR_ABORT;
                } else {
                    /* enable the SDIO corresponding interrupts and DMA */
                    hals_sdio_interrupt_enable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_TXURE | \
                                                                SDIO_INT_DTEND | SDIO_INT_IDMAERR | SDIO_INT_IDMAEND);

                    /* process unlocked */
                    HAL_UNLOCK(emmc_dev);
                }
            } else {
                /* do nothing */
            }
        }
    }

    return reval;
}

/*!
    \brief      erase eMMC device
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  startaddr: erase start address
    \param[in]  endaddr: erase end address
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_erase(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t startaddr, uint32_t endaddr)
{
    int32_t reval = HAL_ERR_NONE;
    uint32_t errorstate;
    uint32_t start_addr = startaddr;
    uint32_t end_addr = endaddr;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_READY == emmc_dev->state) {
        /* process locked */
        HAL_LOCK(emmc_dev);
        emmc_dev->error_state = HAL_EMMC_OK;

        if(end_addr < start_addr) {
            emmc_dev->error_state = HAL_EMMC_ERASE_PARAM;
            HAL_UNLOCK(emmc_dev);
            reval = HAL_ERR_VAL;
        } else if(end_addr > (emmc_dev->card_info.logic_blocknum)) {
            emmc_dev->error_state = HAL_EMMC_OUT_OF_RANGE;
            HAL_UNLOCK(emmc_dev);
            reval = HAL_ERR_VAL;
        } else if(((emmc_dev->ext_csd[(61u/4u)] >> 8u) & 0x000000FFu) != 0x0u) {
            if(((start_addr % 8u) != 0u) || ((end_addr % 8u) != 0u)) {
                emmc_dev->error_state |= HAL_EMMC_ADDRESS_ERROR;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_VAL;
            } else {
                /* do nothing */
            }
        } else {

            emmc_dev->state = HAL_EMMC_STATE_BUSY;
            if(((emmc_dev->card_info.card_speed) & 0x00000020u) == 0u) {
                hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
                emmc_dev->error_state |= HAL_EMMC_FUNCTION_UNSUPPORTED;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_ABORT;
            } else if((hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) == \
            EMMC_CARDSTATE_LOCKED) {
                 /* check whether the card is locked */
                emmc_dev->error_state |= HAL_EMMC_LOCK_UNLOCK_FAILED;
                emmc_dev->state = HAL_EMMC_STATE_READY;
                HAL_UNLOCK(emmc_dev);
                reval = HAL_ERR_ABORT;
            } else {

                if((emmc_dev->card_info.card_type) != EMMC_HIGH_CAPACITY_CARD) {
                    start_addr *= 512U;
                    end_addr   *= 512U;
                } else {
                    /* do nothing */
                }

                /* Send CMD35 MMC_ERASE_GRP_START with argument as addr  */
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_ERASE_GROUP_START,
                                                (uint32_t)start_addr, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(emmc_dev->periph);
                errorstate = _r1_error_check(emmc_dev, EMMC_CMD_ERASE_GROUP_START);
                if(HAL_EMMC_OK != errorstate)
                {

                    /* Clear All flags */
                    hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
                    emmc_dev->error_state |= (hal_sdio_emmc_error_enum)errorstate;
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    HAL_UNLOCK(emmc_dev);
                    reval = HAL_ERR_ABORT;
                } else {

                    /* Send CMD36 MMC_ERASE_GRP_END with argument as addr  */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_ERASE_GROUP_END,
                                                    (uint32_t)end_addr, SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);
                    errorstate = _r1_error_check(emmc_dev, EMMC_CMD_ERASE_GROUP_END);
                    if(HAL_EMMC_OK != errorstate)
                    {
                        /* Clear All flags */
                        hals_sdio_flag_clear(emmc_dev->periph, (uint32_t)SDIO_MASK_INTC_FLAGS);
                        emmc_dev->error_state |= (hal_sdio_emmc_error_enum)errorstate;
                        emmc_dev->state = HAL_EMMC_STATE_READY;
                        HAL_UNLOCK(emmc_dev);
                        reval = HAL_ERR_ABORT;
                    } else {

                        /* Send CMD38 ERASE */
                        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_ERASE,
                                                        (uint32_t)0, SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(emmc_dev->periph);
                        errorstate = _r1_error_check(emmc_dev, EMMC_CMD_ERASE);
                        if(HAL_EMMC_OK != errorstate) {
                            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
                            emmc_dev->error_state |= (hal_sdio_emmc_error_enum)errorstate;
                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            HAL_UNLOCK(emmc_dev);
                            reval = HAL_ERR_ABORT;
                        } else {

                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            /* process unlock */
                            HAL_UNLOCK(emmc_dev);

                            reval = HAL_ERR_NONE;
                        }
                    }
                }
            }
        }
    } else {
        reval = HAL_ERR_BUSY;
    }
    return reval;
}

/*!
    \brief      abort an ongoing data transfer
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_TIMEOUT, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_abort(hal_sdio_emmc_dev_struct *emmc_dev)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status;
    uint32_t tickstart;

#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if(NULL == emmc_dev) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(HAL_EMMC_STATE_BUSY == emmc_dev->state) {
        /* disable the SDIO corresponding interrupts */
        hals_sdio_interrupt_disable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                    SDIO_INT_RXORE | SDIO_INT_TXURE | SDIO_INT_DTEND);
        hals_sdio_trans_start_disable(emmc_dev->periph);

        status = _emmc_transfer_stop(emmc_dev);
        emmc_dev->error_state = status;
        if((HAL_EMMC_OK != status) && (HAL_EMMC_CMD_RESP_TIMEOUT != status)) {
            reval = HAL_ERR_ABORT;
        } else{

            /* configure timeout */
            tickstart = hal_sys_basetick_count_get();
            if((SDIO_DATACTL(emmc_dev->periph) & SDIO_DATACTL_DATADIR) == SDIO_TRANSDIRECTION_TOCARD) {
                if(HAL_EMMC_OK == emmc_dev->error_state) {
                    while(RESET == hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTABORT | \
                                                                    SDIO_FLAG_DAT0BSYEND)) {
                        if(SET == hal_sys_basetick_timeout_check(tickstart, HAL_TIMEOUT_FOREVER)) {
                            emmc_dev->error_state = HAL_EMMC_DATA_TIMEOUT;
                            emmc_dev->state = HAL_EMMC_STATE_READY;
                            reval = HAL_ERR_TIMEOUT;
                            break;
                        } else {
                            /* do nothing */
                        }
                    }
                } else {
                    /* do nothing */
                }
                if(HAL_ERR_NONE == reval) {
                    if(HAL_EMMC_CMD_RESP_TIMEOUT == emmc_dev->error_state)
                    {
                        while(RESET == hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTEND)) {
                            if(SET == hal_sys_basetick_timeout_check(tickstart, HAL_TIMEOUT_FOREVER)) {
                                emmc_dev->error_state = HAL_EMMC_DATA_TIMEOUT;
                                emmc_dev->state = HAL_EMMC_STATE_READY;
                                reval = HAL_ERR_TIMEOUT;
                                break;
                            } else {
                                /* do nothing */
                            }
                        }
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            } else if((SDIO_DATACTL(emmc_dev->periph) & SDIO_DATACTL_DATADIR) == SDIO_TRANSDIRECTION_TOSDIO) {
                while(RESET == hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTABORT | \
                                                                SDIO_FLAG_DTEND)) {
                    if(SET == hal_sys_basetick_timeout_check(tickstart, HAL_TIMEOUT_FOREVER)) {
                        emmc_dev->error_state = HAL_EMMC_DATA_TIMEOUT;
                        emmc_dev->state = HAL_EMMC_STATE_READY;
                        reval = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            } else {
                /* do nothing */
            }
            if(HAL_ERR_NONE == reval) {
                hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DAT0BSYEND);
                hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_DATA_FLAGS);
                hals_sdio_idma_disable(emmc_dev->periph);

                emmc_dev->state = HAL_EMMC_STATE_READY;
                emmc_dev->transfer = EMMC_TRANSFER_NONE;
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    return reval;
}

/*!
    \brief      abort an ongoing data transfer in interrupt
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_abort_interrupt(hal_sdio_emmc_dev_struct *emmc_dev)
{
    int32_t reval = HAL_ERR_NONE;
    uint8_t cardstate;
    hal_emmc_user_cb p_func = NULL;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer address */
    if((NULL == emmc_dev)) {
        HAL_DEBUGE("pointer [emmc_dev] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable all the interrupts */
    hals_sdio_interrupt_disable(emmc_dev->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | \
                                SDIO_INT_RXORE | SDIO_INT_TXURE | SDIO_INT_DTEND);
    hals_sdio_idma_disable(emmc_dev->periph);

    hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);

    _emmc_cardstate_get(emmc_dev, &cardstate);

    emmc_dev->state = HAL_EMMC_STATE_READY;

    if((0x00000005U == cardstate) || (0x00000006U == cardstate)) {
        emmc_dev->error_state = _emmc_transfer_stop(emmc_dev);
    } else {
        /* do nothing */
    }

    if(HAL_EMMC_OK != emmc_dev->error_state) {
        reval = HAL_ERR_ABORT;
    } else {
        p_func = (hal_emmc_user_cb)emmc_dev->abort_complete_callback;
        if(NULL != p_func) {
            p_func(emmc_dev);
        }else {
            /* do nothing */
        }
    }

    return reval;

}

/*!
    \brief      perform specific commands sequence for the different type of erase
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  erase_type: specifies the type of erase to be performed
    \param[in]  startaddr: block start address
    \param[in]  endaddr: block end address
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_sequence_erase(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t erase_type,
                                     uint32_t startaddr, uint32_t endaddr)
{
    int32_t reval = HAL_ERR_NONE;
    uint32_t errorstate;
    uint32_t start_addr = startaddr;
    uint32_t end_addr = endaddr;
    uint32_t tickstart;

    /* Check the coherence between start and end address */
    if(end_addr < start_addr) {
        emmc_dev->error_state = HAL_EMMC_ERASE_PARAM;
        reval = HAL_ERR_ABORT;
    } else if(end_addr > (emmc_dev->card_info.logic_blocknum)) {
        emmc_dev->error_state = HAL_EMMC_OUT_OF_RANGE;
        reval = HAL_ERR_ABORT;
    } else if(((emmc_dev->ext_csd[(61u/4u)] >> 8u) & 0x000000FFu) != 0x0u) {
        if(((start_addr % 8u) != 0u) || ((end_addr % 8u) != 0u)) {
            emmc_dev->error_state |= HAL_EMMC_ADDRESS_ERROR;
            reval = HAL_ERR_ABORT;
        }else {
            /* do nothing */
        }
    }else {
        /* do nothing */
    }
    if(HAL_ERR_NONE == reval) {
        emmc_dev->state = HAL_EMMC_STATE_BUSY;
        if(((emmc_dev->card_info.card_speed) & 0x00000020u) == 0u) {
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
            emmc_dev->error_state |= HAL_EMMC_FUNCTION_UNSUPPORTED;
            emmc_dev->state = HAL_EMMC_STATE_READY;
            reval = HAL_ERR_ABORT;
        } else {

            /* check the state of the driver */
            if(HAL_EMMC_STATE_READY == emmc_dev->state) {
                /* Change State */
                emmc_dev->state = HAL_EMMC_STATE_BUSY;

                /* check whether the card is locked */
                if((hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) == \
                    EMMC_CARDSTATE_LOCKED) {
                    emmc_dev->error_state |= HAL_EMMC_LOCK_UNLOCK_FAILED;
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    reval = HAL_ERR_ABORT;
                } else {

                    if((emmc_dev->card_info.card_type) != EMMC_HIGH_CAPACITY_CARD) {
                        start_addr *= 512U;
                        end_addr   *= 512U;
                    } else {
                        /* do nothing */
                    }

                    tickstart = hal_sys_basetick_count_get();

                    /* Send CMD35 MMC_ERASE_GRP_START with argument as addr  */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_ERASE_GROUP_START,
                                                    (uint32_t)start_addr, SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);
                    errorstate = _r1_error_check(emmc_dev, EMMC_CMD_ERASE_GROUP_START);
                    if(HAL_EMMC_OK == errorstate) {
                        /* Send CMD36 MMC_ERASE_GRP_END with argument as addr  */
                        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_ERASE_GROUP_END,
                                                        (uint32_t)end_addr, SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(emmc_dev->periph);
                        errorstate = _r1_error_check(emmc_dev, EMMC_CMD_ERASE_GROUP_END);
                        if(HAL_EMMC_OK == errorstate) {
                            /* Send CMD38 ERASE */
                            hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_ERASE,
                                                            (uint32_t)erase_type, SDIO_RESPONSETYPE_SHORT);
                            hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                            hals_sdio_csm_enable(emmc_dev->periph);
                            errorstate = _r1_error_check(emmc_dev, EMMC_CMD_ERASE);
                            if(HAL_EMMC_OK == errorstate) {
                                if((erase_type == HAL_EMMC_SECURE_ERASE) || (erase_type == HAL_EMMC_SECURE_TRIM_STEP2)) {
                                    while(RESET == hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DAT0BSYEND)) {
                                        if(SET == hal_sys_basetick_timeout_check(tickstart, HAL_TIMEOUT_FOREVER)) {
                                            errorstate = HAL_EMMC_DATA_TIMEOUT;
                                        } else {
                                            /* do nothing */
                                        }
                                    }
                                } else {
                                    /* do nothing */
                                }

                                hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DAT0BSYEND);
                            } else {
                                /* do nothing */
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }

                    /* Change State */
                    emmc_dev->state = HAL_EMMC_STATE_READY;

                    /* Manage errors */
                    if(HAL_EMMC_OK != errorstate) {
                        /* Clear all the static flags */
                        hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
                        emmc_dev->error_state |= (hal_sdio_emmc_error_enum)errorstate;

                        if(HAL_EMMC_DATA_TIMEOUT != errorstate) {
                            reval = HAL_ERR_ABORT;
                        } else {
                            reval = HAL_ERR_TIMEOUT;
                        }
                    } else {
                        reval = HAL_ERR_NONE;
                    }
                }
            } else {
                reval = HAL_ERR_BUSY;
            }
        }
    }else {
        /* do nothing */
    }
    return reval;
}

/*!
    \brief      perform sanitize operation on the device
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_sanitize(hal_sdio_emmc_dev_struct *emmc_dev)
{
    int32_t reval = HAL_ERR_NONE;
    uint32_t errorstate;
    uint32_t response = 0U;
    uint32_t count;
    uint32_t tickstart = hal_sys_basetick_count_get();

    /* Check the state of the driver */
    if(HAL_EMMC_STATE_READY == emmc_dev->state) {
        /* Change State */
        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        /* Index : 165 - Value : 0x01 */
        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SWITCH_FUNC,
                                        (uint32_t)0x03A50100U, SDIO_RESPONSETYPE_SHORT);
        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
        hals_sdio_csm_enable(emmc_dev->periph);
        errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SWITCH_FUNC);
        if(HAL_EMMC_OK == errorstate) {
            /* Wait that the device is ready by checking the D0 line */
            while((RESET == hals_sdio_flag_get(emmc_dev->periph,SDIO_FLAG_DAT0BSYEND)) && \
                                                (HAL_EMMC_OK == errorstate)) {
                if(SET == hal_sys_basetick_timeout_check(tickstart, HAL_TIMEOUT_FOREVER)) {
                    emmc_dev->error_state = HAL_EMMC_DATA_TIMEOUT;
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    reval = HAL_ERR_TIMEOUT;
                    break;
                }else {
                    /* do nothing */
                }
            }
            if(HAL_ERR_NONE == reval) {
                /* Clear the flag corresponding to end D0 bus line */
                hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DAT0BSYEND);

                if(HAL_EMMC_OK == errorstate) {
                    /* While card is not ready for data and trial number for sending CMD13 is not exceeded */
                    count = 0xFFFFu;
                    do {
                        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                        (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(emmc_dev->periph);
                        errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                        if(HAL_EMMC_OK != errorstate) {
                            break;
                        } else {
                            /* do nothing */
                        }
                        /* Get command response */
                        response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                        count--;
                    } while ((0U == (response & 0x100U)) && (0U != count));

                    /* Check the status after the switch command execution */
                    if((0U != count) && (HAL_EMMC_OK == errorstate)) {
                        /* Check the bit SWITCH_ERROR of the device status */
                        if(0U != (response & 0x80U)) {
                            errorstate = HAL_EMMC_GENERAL_UNKNOWN_ERROR;
                        } else {
                            /* do nothing */
                        }
                    } else if(0U == count) {
                        errorstate = HAL_EMMC_DATA_TIMEOUT;
                    } else {
                    /* do nothing */
                    }
                }else {
                    /* do nothing */
                }
            }else {
                /* do nothing */
            }
        }else {
            /* do nothing */
        }
        if(HAL_ERR_NONE == reval) {
            /* Change State */
            emmc_dev->state = HAL_EMMC_STATE_READY;

            /* Manage errors */
            if(HAL_EMMC_OK != errorstate) {
                /* Clear all the static flags */
                hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
                emmc_dev->error_state |= (hal_sdio_emmc_error_enum)errorstate;

                if(HAL_EMMC_DATA_TIMEOUT != errorstate) {
                    reval = HAL_ERR_ABORT;
                } else {
                    reval = HAL_ERR_TIMEOUT;
                }
            } else {
                reval = HAL_ERR_NONE;
            }
        }else {
            /* do nothing */
        }
    } else {
        reval = HAL_ERR_BUSY;
    }
    return reval;
}

/*!
    \brief      configure the SRT in the Extended CSD register
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  srt_mode: specifies the type of erase to be performed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_BUSY, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_srt_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t srt_mode)
{
    int32_t reval = HAL_ERR_NONE;
    uint32_t srt;
    uint32_t errorstate;
    uint32_t response = 0U;
    uint32_t count;

    /* Check the state of the driver */
    if(HAL_EMMC_STATE_READY == emmc_dev->state) {
        /* get the supported values by the device */
        srt = (emmc_dev->ext_csd[4] & 0x0000000FU);
        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        /* Check the value passed as parameter is supported by the device */
        if(0U != (srt_mode & srt)) {
            /* index : 16 - value : srt mode */
            srt |= ((uint32_t)(_position_val_get(srt_mode)) << 4U);
            hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                            (uint32_t)0x03100000U | (srt << 8U), SDIO_RESPONSETYPE_SHORT);
            hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
            hals_sdio_csm_enable(emmc_dev->periph);
            errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
            if(HAL_EMMC_OK == errorstate) {
                /* While card is not ready for data and trial number for sending CMD13 is not exceeded */
                count = 0xFFFFU;
                do {
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                    (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);
                    errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                    if(HAL_EMMC_OK != errorstate) {
                        break;
                    } else {
                        /* do nothing */
                    }

                    /* Get command response */
                    response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                    count--;
                } while ((0U == (response & 0x100U)) && (0U != count));

                /* Check the status after the switch command execution */
                if((0U != count) && (HAL_EMMC_OK == errorstate)) {
                    /* Check the bit SWITCH_ERROR of the device status */
                    if(0U != (response & 0x80U)) {
                        errorstate = HAL_EMMC_GENERAL_UNKNOWN_ERROR;
                    } else {
                        /* do nothing */
                    }
                } else if(0U == count) {
                    errorstate = HAL_EMMC_DATA_TIMEOUT;
                } else {
                    /* do nothing */
                }
            }else {
                /* do nothing */
            }
        } else {
            errorstate = HAL_EMMC_FUNCTION_UNSUPPORTED;
        }

        /* Change State */
        emmc_dev->state = HAL_EMMC_STATE_READY;

        /* Manage errors */
        if(HAL_EMMC_OK != errorstate) {
            /* Clear all the static flags */
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
            emmc_dev->error_state |= (hal_sdio_emmc_error_enum)errorstate;
            reval = HAL_ERR_ABORT;
        } else {
            reval = HAL_ERR_NONE;
        }
    } else {
        reval = HAL_ERR_BUSY;
    }
    return reval;
}

/*!
    \brief      gets the supported values of the the SRT
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] srt: pointer for supported srt value
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_supported_srt_get(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *srt)
{
    int32_t reval = HAL_ERR_NONE;
    /* Check the state of the driver */
    if(HAL_EMMC_STATE_READY == emmc_dev->state) {
        /* Change State */
        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        /* Read field SECURE_REMOVAL_TYPE [16 = 4*4] of the Extended CSD register */
        *srt = (emmc_dev->ext_csd[4] & 0x0000000FU); /* Bits [3:0] of field 16 */

        /* Change State */
        emmc_dev->state = HAL_EMMC_STATE_READY;

        reval = HAL_ERR_NONE;
    } else {
        reval = HAL_ERR_BUSY;
    }
    return reval;
}

/*!
    \brief      switch the device from standby state to sleep state.
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_device_sleep(hal_sdio_emmc_dev_struct *emmc_dev)
{
    int32_t reval = HAL_ERR_NONE;
    uint32_t errorstate,
             sleep_timeout,
             timeout,
             count,
             response = 0U;
    uint8_t cardstate;
    uint32_t tickstart = hal_sys_basetick_count_get();

    /* Check the state of the driver */
    if(HAL_EMMC_STATE_READY == emmc_dev->state) {
        /* change State */
        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        /* Set the power-off notification to powered-on : Ext_CSD[34] = 1 */
        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                        (uint32_t)0x03220100U, SDIO_RESPONSETYPE_SHORT);
        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
        hals_sdio_csm_enable(emmc_dev->periph);
        errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
        if(HAL_EMMC_OK == errorstate) {
            /* While card is not ready for data and trial number for sending CMD13 is not exceeded */
            count = 0xFFFFU;
            do {
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(emmc_dev->periph);
                errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                if(HAL_EMMC_OK != errorstate) {
                    break;
                }else {
                    /* do nothing */
                }

                /* Get command response */
                response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                count--;
            } while ((0U == (response & 0x100U)) && (0U != count));

            /* Check the status after the switch command execution */
            if(0U == count) {
                errorstate = HAL_EMMC_DATA_TIMEOUT;
            }
            else if(HAL_EMMC_OK == errorstate) {
                /* Check the bit SWITCH_ERROR of the device status */
                if(0U != (response & 0x80U)) {
                    errorstate = HAL_EMMC_FUNCTION_UNSUPPORTED;
                } else {
                    /* Set the power-off notification to sleep notification : Ext_CSD[34] = 4 */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                    (uint32_t)0x03220400U, SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    hals_sdio_csm_enable(emmc_dev->periph);
                    errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                    if(HAL_EMMC_OK == errorstate) {
                        /* Field SLEEP_NOTIFICATION_TIME [216] */
                        sleep_timeout = ((emmc_dev->ext_csd[(216 / 4)]) & 0x000000FFU);

                        /* Sleep/Awake Timeout = 10us * 2^SLEEP_NOTIFICATION_TIME */
                        /* In HAL, the tick interrupt occurs each ms */
                        if((0U == sleep_timeout) || (sleep_timeout > 0x17U)) {
                            sleep_timeout = 0x17U; /* Max register value defined is 0x17 */
                        }else {
                            /* do nothing */
                        }
                        timeout = (((1UL << sleep_timeout) / 100U) + 1U);

                        /* Wait that the device is ready by checking the D0 line */
                        while((RESET == hals_sdio_flag_get(emmc_dev->periph,SDIO_FLAG_DAT0BSYEND)) && \
                                                            (HAL_EMMC_OK == errorstate)) {
                            if(SET == hal_sys_basetick_timeout_check(tickstart, timeout)) {
                                emmc_dev->error_state = HAL_EMMC_DATA_TIMEOUT;
                                emmc_dev->state = HAL_EMMC_STATE_READY;
                                reval = HAL_ERR_TIMEOUT;
                                break;
                            }else {
                                /* do nothing */
                            }
                        }
                        if(HAL_ERR_NONE == reval) {
                            /* Clear the flag corresponding to end D0 bus line */
                            hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DAT0BSYEND);
                            if(HAL_EMMC_OK == errorstate) {
                                /* While card is not ready for data and trial number for sending CMD13 is not exceeded */
                                count = 0xFFFFU;
                                do {
                                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                                    (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                    hals_sdio_csm_enable(emmc_dev->periph);
                                    errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                                    if(HAL_EMMC_OK != errorstate) {
                                        break;
                                    }else {
                                        /* do nothing */
                                    }

                                    /* Get command response */
                                    response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                                    count--;
                                } while ((0U == (response & 0x100U)) && (0U != count));

                                /* Check the status after the switch command execution */
                                if(0U == count) {
                                    errorstate = HAL_EMMC_DATA_TIMEOUT;
                                } else if(HAL_EMMC_OK == errorstate) {
                                    /* Check the bit SWITCH_ERROR of the device status */
                                    if(0U != (response & 0x80U)) {
                                        errorstate = HAL_EMMC_FUNCTION_UNSUPPORTED;
                                    } else {
                                        /* Switch the device in stand-by mode */
                                        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SELECT_DESELECT_CARD,
                                                                        (uint32_t)0u, SDIO_RESPONSETYPE_SHORT);
                                        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                        hals_sdio_csm_enable(emmc_dev->periph);
                                        _r1_error_check(emmc_dev, EMMC_CMD_SELECT_DESELECT_CARD);

                                        /* Field S_A_TIEMOUT [217] */
                                        sleep_timeout = ((emmc_dev->ext_csd[(217u / 4u)] >> 8u) & 0x000000FFU);

                                        /* Sleep/Awake Timeout = 100ns * 2^S_A_TIMEOUT */
                                        /* In HAL, the tick interrupt occurs each ms */
                                        if((0U == sleep_timeout) || (sleep_timeout > 0x17U)) {
                                            sleep_timeout = 0x17U; /* Max register value defined is 0x17 */
                                        }else {
                                            /* do nothing */
                                        }
                                        timeout = (((1UL << sleep_timeout) / 10000U) + 1U);
                                        _emmc_cardstate_get(emmc_dev,&cardstate);
                                        if(HAL_EMMC_CARD_STANDBY == cardstate) {
                                            /* Send CMD5 CMD_MMC_SLEEP_AWAKE with RCA and SLEEP as argument */
                                            hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SLEEP_AWAKE,
                                                                            (uint32_t)((uint32_t)(emmc_dev->card_info.card_rca) << 16u) | ((uint32_t)0x01u << 15u), \
                                                                                SDIO_RESPONSETYPE_SHORT);
                                            hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                            hals_sdio_csm_enable(emmc_dev->periph);
                                            errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SLEEP_AWAKE);
                                            if(HAL_EMMC_OK == errorstate) {
                                                /* Wait that the device is ready by checking the D0 line */
                                                while((RESET == hals_sdio_flag_get(emmc_dev->periph,SDIO_FLAG_DAT0BSYEND)) && \
                                                                                    (HAL_EMMC_OK == errorstate)) {
                                                    if(SET == hal_sys_basetick_timeout_check(tickstart, timeout)) {
                                                        emmc_dev->error_state = HAL_EMMC_DATA_TIMEOUT;
                                                        emmc_dev->state = HAL_EMMC_STATE_READY;
                                                        reval = HAL_ERR_TIMEOUT;
                                                        break;
                                                    }else {
                                                        /* do nothing */
                                                    }
                                                }

                                                /* Clear the flag corresponding to end D0 bus line */
                                                hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DAT0BSYEND);
                                            }else {
                                                /* do nothing */
                                            }
                                        } else {
                                            errorstate = HAL_EMMC_FUNCTION_UNSUPPORTED;
                                        }
                                    }
                                } else {
                                /* do nothing */
                                }
                            }else {
                                /* do nothing */
                            }
                        }else {
                            /* do nothing */
                        }
                    }else {
                        /* do nothing */
                    }
                }
            } else {
            /* do nothing */
            }
        }else {
            /* do nothing */
        }

        /* Change State */
        emmc_dev->state = HAL_EMMC_STATE_READY;

        /* Manage errors */
        if(HAL_EMMC_OK != errorstate) {
            /* Clear all the static flags */
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
            emmc_dev->error_state |= (hal_sdio_emmc_error_enum)errorstate;

            if(HAL_EMMC_DATA_TIMEOUT != errorstate) {
                reval = HAL_ERR_ABORT;
            } else {
                reval = HAL_ERR_TIMEOUT;
            }
        } else {
            reval = HAL_ERR_NONE;
        }
    } else {
        reval = HAL_ERR_BUSY;
    }
    return reval;
}

/*!
    \brief      switch the device from sleep State to standby state
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_device_awake(hal_sdio_emmc_dev_struct *emmc_dev)
{
    int32_t reval = HAL_ERR_NONE;
    uint32_t errorstate;
    uint32_t sleep_timeout;
    uint32_t timeout;
    uint32_t count;
    uint32_t response = 0U;
    uint8_t cardstate;
    uint32_t tickstart = hal_sys_basetick_count_get();

    /* Check the state of the driver */
    if(HAL_EMMC_STATE_READY == emmc_dev->state) {
        /* Change State */
        emmc_dev->state = HAL_EMMC_STATE_BUSY;

        /* Field S_A_TIEMOUT [217] */
        sleep_timeout = ((emmc_dev->ext_csd[(217 / 4)] >> 8) & 0x000000FFU);

        /* Sleep/Awake Timeout = 100ns * 2^S_A_TIMEOUT */
        /* In HAL, the tick interrupt occurs each ms */
        if((0U == sleep_timeout) || (sleep_timeout > 0x17U)) {
            sleep_timeout = 0x17U; /* Max register value defined is 0x17 */
        }else {
            /* do nothing */
        }
        timeout = (((1UL << sleep_timeout) / 10000U) + 1U);

        /* Send CMD5 CMD_MMC_SLEEP_AWAKE with RCA and SLEEP as argument */
        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SLEEP_AWAKE,
                                        (uint32_t)((uint32_t)(emmc_dev->card_info.card_rca) << 16u), \
                                        SDIO_RESPONSETYPE_SHORT);
        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
        hals_sdio_csm_enable(emmc_dev->periph);
        errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SLEEP_AWAKE);
        if(HAL_EMMC_OK == errorstate) {
            /* Wait that the device is ready by checking the D0 line */
            while((RESET == hals_sdio_flag_get(emmc_dev->periph,SDIO_FLAG_DAT0BSYEND)) && \
                                                (HAL_EMMC_OK == errorstate)) {
                if(SET == hal_sys_basetick_timeout_check(tickstart, timeout)) {
                    emmc_dev->error_state = HAL_EMMC_DATA_TIMEOUT;
                    emmc_dev->state = HAL_EMMC_STATE_READY;
                    reval = HAL_ERR_TIMEOUT;
                    break;
                }else {
                    /* do nothing */
                }
            }
            if(HAL_ERR_NONE == reval) {
                /* Clear the flag corresponding to end D0 bus line */
                hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DAT0BSYEND);
                if(HAL_EMMC_OK == errorstate) {
                    _emmc_cardstate_get(emmc_dev,&cardstate);
                    if(HAL_EMMC_CARD_STANDBY == cardstate) {
                        /* Switch the device in transfer mode */
                        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SELECT_DESELECT_CARD,
                                                        (uint32_t)((uint32_t)(emmc_dev->card_info.card_rca) << 16U), SDIO_RESPONSETYPE_SHORT);
                        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                        hals_sdio_csm_enable(emmc_dev->periph);
                        errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SELECT_DESELECT_CARD);
                        if(HAL_EMMC_OK == errorstate) {
                            _emmc_cardstate_get(emmc_dev,&cardstate);
                            if(HAL_EMMC_CARD_TRANSFER == cardstate) {
                                /* Set the power-off notification to powered-on : Ext_CSD[34] = 1 */
                                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SWITCH_FUNC,
                                                                (uint32_t)0x03220100U, SDIO_RESPONSETYPE_SHORT);
                                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                hals_sdio_csm_enable(emmc_dev->periph);
                                errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SWITCH_FUNC);
                                if(HAL_EMMC_OK == errorstate) {
                                    /* While card is not ready for data and trial number for sending CMD13 is not exceeded */
                                    count = 0xFFFFU;
                                    do {
                                        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                                                        (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
                                        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                                        hals_sdio_csm_enable(emmc_dev->periph);
                                        errorstate = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
                                        if(HAL_EMMC_OK != errorstate) {
                                            break;
                                        } else {
                                            /* do nothing */
                                        }

                                        /* Get command response */
                                        response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                                        count--;
                                    } while ((0U == (response & 0x100U)) && (0U != count));

                                    /* Check the status after the switch command execution */
                                    if(0U == count) {
                                        errorstate = HAL_EMMC_DATA_TIMEOUT;
                                    } else if(HAL_EMMC_OK == errorstate) {
                                        /* Check the bit SWITCH_ERROR of the device status */
                                        if(0U != (response & 0x80U)) {
                                            errorstate = HAL_EMMC_FUNCTION_UNSUPPORTED;
                                        } else {
                                            /* do nothing */
                                        }
                                    } else {
                                        /* do nothing */
                                    }
                                } else {
                                    /* do nothing */
                                }
                            } else {
                                errorstate = HAL_EMMC_ILLEGAL_COMMAND;
                            }
                        } else {
                            /* do nothing */
                        }
                    } else {
                        errorstate = HAL_EMMC_ILLEGAL_COMMAND;
                    }
                }else {
                    /* do nothing */
                }
            }else {
                /* do nothing */
            }
        }else {
            /* do nothing */
        }
        if(HAL_ERR_NONE == reval) {
            /* Change State */
            emmc_dev->state = HAL_EMMC_STATE_READY;

            /* Manage errors */
            if(HAL_EMMC_OK != errorstate) {
                /* Clear all the static flags */
                hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
                emmc_dev->error_state |= (hal_sdio_emmc_error_enum)errorstate;

                if(HAL_EMMC_DATA_TIMEOUT != errorstate) {
                    reval = HAL_ERR_ABORT;
                } else {
                    reval = HAL_ERR_TIMEOUT;
                }
            } else {
                reval = HAL_ERR_NONE;
            }
        }else {
            /* do nothing */
        }
    } else {
        reval = HAL_ERR_BUSY;
    }
    return reval;
}

/*!
    \brief      get the information of the card which are stored on
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_cid: eMMC device cid data information structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_card_cid_get(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_cardcid_struct *p_cid)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer and p_cid address */
    if((NULL == emmc_dev) || (NULL == p_cid)) {
        HAL_DEBUGE("pointer [emmc_dev] or [p_cid] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    p_cid->manufacturerid = (uint8_t)((emmc_dev->cid_data[0] & 0xff000000u) >> 24u);
    p_cid->oem_appliid = (uint16_t)((emmc_dev->cid_data[0] & 0x00ffff00u) >> 8u);
    p_cid->prodname1 = (((emmc_dev->cid_data[0] & 0x000000ffu) << 24u) | ((emmc_dev->cid_data[1] & 0xffffff00u) >> 8u));
    p_cid->prodname2 = (uint8_t)(emmc_dev->cid_data[1] & 0x000000ffu);
    p_cid->prodrev = (uint8_t)((emmc_dev->cid_data[2] & 0xff000000u) >> 24u);
    p_cid->prodsn = (((emmc_dev->cid_data[2] & 0x00ffffffu) << 8u) | ((emmc_dev->cid_data[3] & 0xff000000u) >> 24u));
    p_cid->reserved1 = (uint8_t)((emmc_dev->cid_data[3] & 0x00f00000u) >> 20u);
    p_cid->manufactdate = (uint16_t)((emmc_dev->cid_data[3] & 0x000fff00u) >> 8u);
    p_cid->cid_crc = (uint8_t)((emmc_dev->cid_data[3] & 0x000000feu) >> 1u);
    p_cid->reserved2 = 1u;

    return HAL_ERR_NONE;
}

/*!
    \brief      get the information of the card which are stored on
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_csd: eMMC device csd data information structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_card_csd_get(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_cardcsd_struct *p_csd)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer and p_csd address */
    if((NULL == emmc_dev) || (NULL == p_csd)) {
        HAL_DEBUGE("pointer [emmc_dev] or [p_csd] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    status = _emmc_card_csd_get(emmc_dev, p_csd);
    if(HAL_EMMC_OK != status) {
        reval = HAL_ERR_VAL;
    } else {
        reval = HAL_ERR_NONE;
    }
    return reval;
}

/*!
    \brief      get the EXT_CSD register detailed information of the eMMC
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] ext_csd: pointer to the buffer that stores the EXT_CSD data
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_card_extcsd_get(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *ext_csd)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer and ext_csd address */
    if((NULL == emmc_dev) || (NULL == ext_csd)) {
        HAL_DEBUGE("pointer [emmc_dev] or [ext_csd] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    status = _emmc_card_extcsd_get(emmc_dev, ext_csd);
    if(HAL_EMMC_OK != status) {
        reval = HAL_ERR_VAL;
    } else {
        reval = HAL_ERR_NONE;
    }
  return reval;
}

/*!
    \brief      get the detailed information of the SD card based on received CID and CSD
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_cardinfo: sdcard information structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_card_information_get(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_cardinfo_struct *p_cardinfo)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer and p_cardinfo address */
    if((NULL == emmc_dev) || (NULL != p_cardinfo)) {
        HAL_DEBUGE("pointer [sd_dev] or [p_cardinfo] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    p_cardinfo->card_type = emmc_dev->card_info.card_type;
    p_cardinfo->card_speed = emmc_dev->card_info.card_speed;
    p_cardinfo->card_rca = emmc_dev->card_info.card_rca;
    p_cardinfo->card_blocknum = emmc_dev->card_info.card_blocknum;
    p_cardinfo->card_blocksize = emmc_dev->card_info.card_blocksize;
    p_cardinfo->logic_blocknum = emmc_dev->card_info.logic_blocknum;
    p_cardinfo->logic_blocksize = emmc_dev->card_info.logic_blocksize;

    return HAL_ERR_NONE;
}

/*!
    \brief      get the state which the card is in
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] pcardstate: a pointer that store the card state
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ABORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_cardstate_get(hal_sdio_emmc_dev_struct *emmc_dev, uint8_t *pcardstate)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer and pcardstate address */
    if((NULL == emmc_dev) || (NULL == pcardstate)) {
        HAL_DEBUGE("pointer [emmc_dev] or [pcardstate] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    status = _emmc_cardstate_get(emmc_dev, pcardstate);
    if(HAL_EMMC_OK != status) {
        reval = HAL_ERR_ABORT;
    } else {
        reval = HAL_ERR_NONE;
    }
    return reval;
}

/*!
    \brief      get the card status whose response format R1 contains a 32-bit field
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] pcardstatus: a pointer that store card status
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ABORT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sdio_emmc_cardstatus_get(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pcardstatus)
{
    int32_t reval = HAL_ERR_NONE;
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
#if (1U == HAL_PARAMETER_CHECK)
    /* check emmc pointer and pcardstatus address */
    if((NULL == emmc_dev) || (NULL == pcardstatus)) {
        HAL_DEBUGE("pointer [emmc_dev] or [pcardstatus] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    if(NULL == pcardstatus) {
        emmc_dev->error_state = HAL_EMMC_PARAMETER_INVALID;
        reval = HAL_ERR_VAL;
    } else {

        /* send CMD13(SEND_STATUS), addressed card sends its status register */
        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                        (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
        hals_sdio_csm_enable(emmc_dev->periph);
        /* check if some error occurs */
        status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
        if(HAL_EMMC_OK != status) {
            emmc_dev->error_state = status;
            reval = HAL_ERR_ABORT;
        } else{
            *pcardstatus = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
        }
    }
    return reval;
}

/*!
    \brief      return the emmc state
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_state_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
hal_sdio_emmc_state_enum hal_sdio_emmc_state_get(hal_sdio_emmc_dev_struct *emmc_dev)
{
    return emmc_dev->state;
}

/*!
    \brief      return the emmc error code
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
hal_sdio_emmc_error_enum hal_sdio_emmc_error_get(hal_sdio_emmc_dev_struct *emmc_dev)
{
    return emmc_dev->error_state;
}

/*!
    \brief      initialize the card and get CID and CSD of the card
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_card_init(hal_sdio_emmc_dev_struct *emmc_dev)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint16_t temp_rca = 0x01U;

    if(SDIO_POWER_OFF == hals_sdio_power_state_get(emmc_dev->periph)) {
        status = HAL_EMMC_OPERATION_IMPROPER;
    } else {

        /* send CMD2(EMMC_CMD_ALL_SEND_CID) to get the CID numbers */
        hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_ALL_SEND_CID, 0U, SDIO_RESPONSETYPE_LONG);
        hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
        hals_sdio_csm_enable(emmc_dev->periph);
        /* check if some error occurs */
        status = _r2_error_check(emmc_dev);
        if(HAL_EMMC_OK == status) {

            /* store the CID numbers */
            emmc_dev->cid_data[0] = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
            emmc_dev->cid_data[1] = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE1);
            emmc_dev->cid_data[2] = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE2);
            emmc_dev->cid_data[3] = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE3);

            /* send CMD3(SEND_RELATIVE_ADDR) to ask the eMMC to publish a new relative address (RCA) */
            hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_RELATIVE_ADDR, 0U,
                                            SDIO_RESPONSETYPE_SHORT);
            hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
            hals_sdio_csm_enable(emmc_dev->periph);
            /* check if some error occurs */
            status = _r6_error_check(emmc_dev, EMMC_CMD_SEND_RELATIVE_ADDR, &temp_rca);
            if(HAL_EMMC_OK == status) {

                /* the card is not I/O only card */
                emmc_dev->card_info.card_rca = temp_rca;

                /* send CMD9(SEND_CSD) to get the addressed card's card-specific data (CSD) */
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_CSD,
                                                (uint32_t)((uint32_t)(emmc_dev->card_info.card_rca) << EMMC_RCA_SHIFT), SDIO_RESPONSETYPE_LONG);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                hals_sdio_csm_enable(emmc_dev->periph);
                /* check if some error occurs */
                status = _r2_error_check(emmc_dev);
                if(HAL_EMMC_OK == status) {

                    /* store the card-specific data (CSD) */
                    emmc_dev->csd_data[0] = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                    emmc_dev->csd_data[1] = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE1);
                    emmc_dev->csd_data[2] = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE2);
                    emmc_dev->csd_data[3] = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE3);
                }else {
                    /* do nothing */
                }
            }else {
                /* do nothing */
            }
        }else {
            /* do nothing */
        }
    }

    return status;
}

/*!
    \brief      configure the clock and the work voltage, and get the card type
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_power_on(hal_sdio_emmc_dev_struct *emmc_dev)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t response = 0U;
    uint32_t timedelay = 0U;

    /* configure the SDIO peripheral */
    hals_sdio_clock_config(emmc_dev->periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, EMMC_CLK_DIV_INIT);
    hals_sdio_bus_mode_set(emmc_dev->periph, SDIO_BUSMODE_1BIT);
    hals_sdio_hardware_clock_disable(emmc_dev->periph);
    hals_sdio_power_state_set(emmc_dev->periph, SDIO_POWER_ON);

    /* time delay for power up*/
    timedelay = 500U;
    while(timedelay > 0U) {
        timedelay--;
    }

    /* send CMD0(GO_IDLE_STATE) to reset the card */
    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_GO_IDLE_STATE, 0U, SDIO_RESPONSETYPE_NO);
    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
    /* enable the CSM */
    hals_sdio_csm_enable(emmc_dev->periph);

    /* check if command sent error occurs */
    status = _cmdsent_error_check(emmc_dev);
    if(HAL_EMMC_OK == status) {

        timedelay = 0x4000U;
        while(timedelay--) {
            hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_OP_COND, (uint32_t)0x40ff8000U,
                                            SDIO_RESPONSETYPE_SHORT);
            hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
            /* enable the CSM */
            hals_sdio_csm_enable(emmc_dev->periph);

            /* check if some error occurs */
            status = _r3_error_check(emmc_dev);
            if(HAL_EMMC_OK == status) {

                if(timedelay) {
                    /* get the response and check card power up status bit(busy) */
                    response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
                    if(0x80000000U == (response & 0x80000000U)) {
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }else {
                /* do nothing */
            }
        }
        if(HAL_EMMC_OK == status) {
            if(((response & 0xFF000000U) >> 24U) == 0xC0U) {
                emmc_dev->card_info.card_type = EMMC_HIGH_CAPACITY_CARD;
            } else {
                emmc_dev->card_info.card_type = EMMC_HIGH_CAPACITY_CARD;
            }
        }else {
            /* do nothing */
        }
    }else {
        /* do nothing */
    }
    return status;
}

/*!
    \brief      select or deselect a card
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_card_select_deselect(hal_sdio_emmc_dev_struct *emmc_dev)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    /* send CMD7(SELECT/DESELECT_CARD) to select or deselect the card */
    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SELECT_DESELECT_CARD,
                                      (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(emmc_dev->periph);

    status = _r1_error_check(emmc_dev, EMMC_CMD_SELECT_DESELECT_CARD);
    return status;
}

/*!
    \brief    configure the bus width mode
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  buswidth: SD eMMC device information structure
                only one parameter can be selected which is shown as below:
      \arg        EMMC_BUS_WIDTH_1BIT: 1-bit bus width
      \arg        EMMC_BUS_WIDTH_4BIT: 4-bit bus width
      \arg        EMMC_BUS_WIDTH_8BIT: 8-bit bus width
    \param[out] none
    \retval     hal_sdio_emmc_state_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_bus_width_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t buswidth)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;

    (void)buswidth;
    /* check whether the card is locked */
    if(hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0) & EMMC_CARDSTATE_LOCKED) {
        status = HAL_EMMC_LOCK_UNLOCK_FAILED;
    } else {
        /* do nothing */
    }
    return status;
}

/*!
    \brief      get the information of the card which are stored on
    \param[in]  emmc_dev: eMMC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_card_csd_get(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_cardcsd_struct *p_csd)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t block_nbr = 0u;

    p_csd->csdstruct = (uint8_t)((emmc_dev->csd_data[0] & 0xc0000000u) >> 30u);
    p_csd->sysspecversion = (uint8_t)((emmc_dev->csd_data[0] & 0x3c000000u) >> 26u);
    p_csd->reserved1 = (uint8_t)((emmc_dev->csd_data[0] & 0x03000000u) >> 24u);
    p_csd->taac = (uint8_t)((emmc_dev->csd_data[0] & 0x00ff0000u) >> 16u);
    p_csd->nsac = (uint8_t)((emmc_dev->csd_data[0] & 0x0000ff00u) >> 8u);
    p_csd->maxbusclkfrec = (uint8_t)(emmc_dev->csd_data[0] & 0x000000ffu);
    p_csd->cardcomdclasses = (uint16_t)((emmc_dev->csd_data[1] & 0xfff00000u) >> 20u);
    p_csd->rdblocklen = (uint8_t)((emmc_dev->csd_data[1] & 0x000f0000u) >> 16u);
    p_csd->partblockread   = (uint8_t)((emmc_dev->csd_data[1] & 0x00008000u) >> 15u);
    p_csd->wrblockmisalign = (uint8_t)((emmc_dev->csd_data[1] & 0x00004000u) >> 14u);
    p_csd->rdblockmisalign = (uint8_t)((emmc_dev->csd_data[1] & 0x00002000u) >> 13u);
    p_csd->dsrimpl = (uint8_t)((emmc_dev->csd_data[1] & 0x00001000u) >> 12u);
    p_csd->reserved2 = 0u; /*!< reserved */

    status = _emmc_card_extcsd_read(emmc_dev, &block_nbr, 212u);
    if(HAL_EMMC_OK == status) {  /* Field SEC_COUNT [215:212] */

        if(emmc_dev->card_info.card_type == EMMC_LOW_CAPACITY_CARD) {
            p_csd->devicesize = (((emmc_dev->csd_data[1] & 0x000003ffu) << 2u) | ((emmc_dev->csd_data[2] & 0xc0000000u) >> 30u));
            p_csd->maxrdcurrentvddmin = (uint8_t)((emmc_dev->csd_data[2] & 0x38000000u) >> 27u);
            p_csd->maxrdcurrentvddmax = (uint8_t)((emmc_dev->csd_data[2] & 0x07000000u) >> 24u);
            p_csd->maxwrcurrentvddmin = (uint8_t)((emmc_dev->csd_data[2] & 0x00e00000u) >> 21u);
            p_csd->maxwrcurrentvddmax = (uint8_t)((emmc_dev->csd_data[2] & 0x001c0000u) >> 18u);
            p_csd->devicesizemul = (uint8_t)((emmc_dev->csd_data[2] & 0x00038000u) >> 15u);
            emmc_dev->card_info.card_blocknum  = (p_csd->devicesize + 1U) ;
            emmc_dev->card_info.card_blocknum *= (1UL << ((p_csd->devicesizemul & 0x07U) + 2U));
            emmc_dev->card_info.card_blocksize = (1UL << (p_csd->rdblocklen & 0x0FU));
            emmc_dev->card_info.logic_blocknum = (emmc_dev->card_info.card_blocknum) * ((emmc_dev->card_info.card_blocksize) / 512U);
            emmc_dev->card_info.logic_blocksize = 512U;
        } else if(emmc_dev->card_info.card_type == EMMC_HIGH_CAPACITY_CARD) {
            emmc_dev->card_info.card_blocknum = block_nbr;
            emmc_dev->card_info.logic_blocknum = emmc_dev->card_info.card_blocknum;
            emmc_dev->card_info.card_blocksize = 512u;
            emmc_dev->card_info.logic_blocksize = emmc_dev->card_info.card_blocksize;
        } else {
            /* clear the SDIO_INTC flags */
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
            status = HAL_EMMC_FUNCTION_UNSUPPORTED;
        }
        if(HAL_EMMC_OK == status) {
            p_csd->erasegrsize = (uint8_t)((emmc_dev->csd_data[2] & 0x00004000u) >> 14u);
            p_csd->erasegrmul = (uint8_t)((emmc_dev->csd_data[2] & 0x00003f80u) >> 7u);
            p_csd->wrprotectgrsize = (uint8_t)(emmc_dev->csd_data[2] & 0x0000007fu);
            p_csd->wrprotectgrenable = (uint8_t)((emmc_dev->csd_data[3] & 0x80000000u) >> 31u);
            p_csd->mandeflecc = (uint8_t)((emmc_dev->csd_data[3] & 0x60000000u) >> 29u);
            p_csd->wrspeedfact = (uint8_t)((emmc_dev->csd_data[3] & 0x1c000000u) >> 26u);
            p_csd->maxwrblocklen = (uint8_t)((emmc_dev->csd_data[3] & 0x03c00000u) >> 22u);
            p_csd->writeblockpapartial = (uint8_t)((emmc_dev->csd_data[3] & 0x00200000u) >> 21u);
            p_csd->reserved3 = 0u;
            p_csd->contentprotectappli = (uint8_t)((emmc_dev->csd_data[3] & 0x00010000u) >> 16u);
            p_csd->fileformatgroup = (uint8_t)((emmc_dev->csd_data[3] & 0x00008000u) >> 15u);
            p_csd->copyflag = (uint8_t)((emmc_dev->csd_data[3] & 0x00004000u) >> 14u);
            p_csd->permwrprotect = (uint8_t)((emmc_dev->csd_data[3] & 0x00002000u) >> 13u);
            p_csd->tempwrprotect = (uint8_t)((emmc_dev->csd_data[3] & 0x00001000u) >> 12u);
            p_csd->fileformat = (uint8_t)((emmc_dev->csd_data[3] & 0x00000c00u) >> 10u);
            p_csd->ecc = (uint8_t)((emmc_dev->csd_data[3] & 0x00000300u) >> 8u);
            p_csd->csd_crc = (uint8_t)((emmc_dev->csd_data[3] & 0x000000feu) >> 1u);
            p_csd->reserved4 = 1u;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      get the EXT_CSD register detailed information of the eMMC
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_card_extcsd_get(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *ext_csd)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t count = 0U, *ptempbuff = ext_csd;

    /* configure SDIO data transmission */
    hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, 512U, SDIO_DATABLOCKSIZE_512BYTES);
    hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);
    hals_sdio_trans_start_enable(emmc_dev->periph);

    /* send CMD8(READ_SINGLE_BLOCK) to read a block */
    hals_sdio_command_response_config(emmc_dev->periph, (uint32_t)EMMC_CMD_SEND_EXT_CSD, 0U,
                                      SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(emmc_dev->periph);
    /* check if some error occurs */
    status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_EXT_CSD);
    if(HAL_EMMC_OK == status) {
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);

        /* polling mode */
        while(!hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTCRCERR | SDIO_FLAG_DTTMOUT | SDIO_FLAG_RXORE | \
                                                    SDIO_FLAG_DTBLKEND | SDIO_FLAG_DTEND)) {
            if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_RFH)) {
                /* at least 8 words can be read in the FIFO */
                for(count = 0U; count < EMMC_FIFOHALF_WORDS; count++) {
                    *(ptempbuff + count) = hals_sdio_data_read(emmc_dev->periph);
                }
                ptempbuff += EMMC_FIFOHALF_WORDS;
            } else {
                /* do nothing */
            }
        }

        hals_sdio_trans_start_disable(emmc_dev->periph);
        /* whether some error occurs and return it */
        if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTCRCERR)) {
            status = HAL_EMMC_DATA_CRC_ERROR;
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DTCRCERR);

        } else if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTTMOUT)) {
            status = HAL_EMMC_DATA_TIMEOUT;
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DTTMOUT);

        } else if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_RXORE)) {
            status = HAL_EMMC_RX_OVERRUN_ERROR;
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_RXORE);
        } else {
            /* clear the SDIO_INTC flags */
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
        }
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      read the EXT_CSD register detailed information of the eMMC
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] filed_data: pointer to the read buffer
    \param[out] filed_index: index of the field to be read
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_card_extcsd_read(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *filed_data, \
                                                  uint16_t filed_index)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t count = 0U;
    uint32_t i = 0U;
    uint32_t tmp_data;

    /* configure SDIO data transmission */
    hals_sdio_data_config(emmc_dev->periph, EMMC_DATATIMEOUT, 512U, SDIO_DATABLOCKSIZE_512BYTES);
    hals_sdio_data_transfer_config(emmc_dev->periph, SDIO_TRANSMODE_BLOCKCOUNT, SDIO_TRANSDIRECTION_TOSDIO);

    /* send CMD8(READ_SINGLE_BLOCK) to read a block */
    hals_sdio_command_response_config(emmc_dev->periph, (uint32_t)EMMC_CMD_SEND_EXT_CSD, 0U,
                                      SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(emmc_dev->periph);
    /* check if some error occurs */
    status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_EXT_CSD);
    if(HAL_EMMC_OK == status) {
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);

        /* polling mode */
        while(!hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTCRCERR | SDIO_FLAG_DTTMOUT | SDIO_FLAG_RXORE | \
                                                    SDIO_FLAG_DTBLKEND | SDIO_FLAG_DTEND)) {
            if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_RFH)) {
                /* at least 8 words can be read in the FIFO */
                for(count = 0U; count < EMMC_FIFOHALF_WORDS; count++) {
                    tmp_data = hals_sdio_data_read(emmc_dev->periph);
                    if((i + count) == ((uint32_t)filed_index / 4U))
                    {
                        *filed_data = tmp_data;
                    } else {
                        /* do nothing */
                    }
                }
                i += EMMC_FIFOHALF_WORDS;
            } else {
                /* do nothing */
            }
        }

        /* whether some error occurs and return it */
        if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTCRCERR)) {
            status = HAL_EMMC_DATA_CRC_ERROR;
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DTCRCERR);

        } else if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_DTTMOUT)) {
            status = HAL_EMMC_DATA_TIMEOUT;
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_DTTMOUT);

        } else if(RESET != hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_RXORE)) {
            status = HAL_EMMC_RX_OVERRUN_ERROR;
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_RXORE);
        } else {

            /* send CMD13(SEND_STATUS), addressed card sends its status registers */
            hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                            (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
            hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
            hals_sdio_csm_enable(emmc_dev->periph);
            /* check if some error occurs */
            status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);

            /* clear the SDIO_INTC flags */
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
        }
    } else {
        /* do nothing */
    }

    return status;
}
/*!
    \brief      check if the command sent error occurs
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _cmdsent_error_check(hal_sdio_emmc_dev_struct *emmc_dev)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t timeout = 100000U;
    /* check command sent flag */
    while((RESET == hals_sdio_flag_get(emmc_dev->periph, SDIO_FLAG_CMDSEND)) && (timeout > 0U)) {
        --timeout;
    }
    /* command response is timeout */
    if(0U == timeout) {
        status = HAL_EMMC_CMD_RESP_TIMEOUT;
    } else {
        /* if the command is sent, clear the CMD_FLAGS flags */
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_CMD_FLAGS);
    }
    return status;
}

/*!
    \brief      check if error type for R1 response
    \param[in]  resp: content of response
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _r1_error_type_check(uint32_t resp)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_ERROR;

    /* check which error occurs */
    if(resp & EMMC_R1_OUT_OF_RANGE) {
        status = HAL_EMMC_OUT_OF_RANGE;
    } else if(resp & EMMC_R1_ADDRESS_ERROR) {
        status = HAL_EMMC_ADDRESS_ERROR;
    } else if(resp & EMMC_R1_BLOCK_LEN_ERROR) {
        status = HAL_EMMC_BLOCK_LEN_ERROR;
    } else if(resp & EMMC_R1_ERASE_SEQ_ERROR) {
        status = HAL_EMMC_ERASE_SEQ_ERROR;
    } else if(resp & EMMC_R1_ERASE_PARAM) {
        status = HAL_EMMC_ERASE_PARAM;
    } else if(resp & EMMC_R1_WP_VIOLATION) {
        status = HAL_EMMC_WP_VIOLATION;
    } else if(resp & EMMC_R1_LOCK_UNLOCK_FAILED) {
        status = HAL_EMMC_LOCK_UNLOCK_FAILED;
    } else if(resp & EMMC_R1_COM_CRC_ERROR) {
        status = HAL_EMMC_COM_CRC_ERROR;
    } else if(resp & EMMC_R1_ILLEGAL_COMMAND) {
        status = HAL_EMMC_ILLEGAL_COMMAND;
    } else if(resp & EMMC_R1_CARD_ECC_FAILED) {
        status = HAL_EMMC_CARD_ECC_FAILED;
    } else if(resp & EMMC_R1_CC_ERROR) {
        status = HAL_EMMC_CC_ERROR;
    } else if(resp & EMMC_R1_GENERAL_UNKNOWN_ERROR) {
        status = HAL_EMMC_GENERAL_UNKNOWN_ERROR;
    } else if(resp & EMMC_R1_CSD_OVERWRITE) {
        status = HAL_EMMC_CSD_OVERWRITE;
    } else if(resp & EMMC_R1_WP_ERASE_SKIP) {
        status = HAL_EMMC_WP_ERASE_SKIP;
    } else if(resp & EMMC_R1_CARD_ECC_DISABLED) {
        status = HAL_EMMC_CARD_ECC_DISABLED;
    } else if(resp & EMMC_R1_ERASE_RESET) {
        status = HAL_EMMC_ERASE_RESET;
    } else if(resp & EMMC_R1_AKE_SEQ_ERROR) {
        status = HAL_EMMC_AKE_SEQ_ERROR;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      check if error type for R1 response
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _r1_error_check(hal_sdio_emmc_dev_struct *emmc_dev, uint8_t cmdindex)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t reg_status = 0U, resp_r1 = 0U;

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(emmc_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(emmc_dev->periph);
    }

    /* check whether an error or timeout occurs or command response received */
    if(reg_status & SDIO_FLAG_CCRCERR) {
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_CCRCERR);
        status = HAL_EMMC_CMD_CRC_ERROR;
    } else if(reg_status & SDIO_FLAG_CMDTMOUT) {
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_CMDTMOUT);
        status = HAL_EMMC_CMD_RESP_TIMEOUT;
    } else {

        /* check whether the last response command index is the desired one */
        if(hals_sdio_command_index_get(emmc_dev->periph) != cmdindex) {
            status = HAL_EMMC_ILLEGAL_COMMAND;
        } else {
            /* clear all the CMD_FLAGS flags */
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_CMD_FLAGS);
            /* get the SDIO response register 0 for checking */
            resp_r1 = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
            if(EMMC_ALLZERO == (resp_r1 & EMMC_R1_ERROR_BITS)) {
                /* no error occurs, return HAL_EMMC_OK */
                status = HAL_EMMC_OK;
            } else {

                /* if some error occurs, return the error type */
                status = _r1_error_type_check(resp_r1);
            }
        }
    }

    return status;
}

/*!
    \brief      check if error occurs for R2 response
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _r2_error_check(hal_sdio_emmc_dev_struct *emmc_dev)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t reg_status = 0U;

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(emmc_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(emmc_dev->periph);
    }
    /* check whether an error or timeout occurs or command response received */
    if(reg_status & SDIO_FLAG_CCRCERR) {
        status = HAL_EMMC_CMD_CRC_ERROR;
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_CCRCERR);
    } else if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_EMMC_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_CMDTMOUT);
    } else {
        /* clear all the CMD_FLAGS flags */
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_CMD_FLAGS);
    }

    return status;
}

/*!
    \brief      check if error occurs for R3 response
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _r3_error_check(hal_sdio_emmc_dev_struct *emmc_dev)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t reg_status = 0U;

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(emmc_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(emmc_dev->periph);
    }
    if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_EMMC_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_CMDTMOUT);
    } else {
        /* clear all the CMD_FLAGS flags */
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_CMD_FLAGS);
    }
    return status;
}

/*!
    \brief      check if error occurs for R6 response
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cmdindex: the index of command
    \param[out] prca: a pointer that store the RCA of card
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _r6_error_check(hal_sdio_emmc_dev_struct *emmc_dev, uint8_t cmdindex, uint16_t *prca)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    uint32_t reg_status = 0U, response = 0U;

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(emmc_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(emmc_dev->periph);
    }
    /* check whether an error or timeout occurs or command response received */
    if(reg_status & SDIO_FLAG_CCRCERR) {
        status = HAL_EMMC_CMD_CRC_ERROR;
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_CCRCERR);
    } else if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_EMMC_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_CMDTMOUT);
    } else {

        /* check whether the last response command index is the desired one */
        if(hals_sdio_command_index_get(emmc_dev->periph) != cmdindex) {
            status = HAL_EMMC_ILLEGAL_COMMAND;
        } else {
            /* clear all the CMD_FLAGS flags */
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_CMD_FLAGS);
            /* get the SDIO response register 0 for checking */
            response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);

            if(EMMC_ALLZERO == (response & (EMMC_R6_COM_CRC_ERROR | EMMC_R6_ILLEGAL_COMMAND | EMMC_R6_GENERAL_UNKNOWN_ERROR))) {
                *prca = (uint16_t)(response >> 16U);
            } else {
                /* if some error occurs, return the error type */
                if(response & EMMC_R6_COM_CRC_ERROR) {
                    status = HAL_EMMC_COM_CRC_ERROR;
                } else if(response & EMMC_R6_ILLEGAL_COMMAND) {
                    status = HAL_EMMC_ILLEGAL_COMMAND;
                } else if(response & EMMC_R6_GENERAL_UNKNOWN_ERROR) {
                    status = HAL_EMMC_GENERAL_UNKNOWN_ERROR;
                } else {
                    /* do nothing */
                }
            }
        }
    }
    return status;
}

/*!
    \brief      get the data block size
    \param[in]  bytesnumber: the number of bytes
    \param[out] none
    \retval     data block size
                only one parameter can be selected which is shown as below:
      \arg        SDIO_DATABLOCKSIZE_1BYTE: block size = 1 byte
      \arg        SDIO_DATABLOCKSIZE_2BYTES: block size = 2 bytes
      \arg        SDIO_DATABLOCKSIZE_4BYTES: block size = 4 bytes
      \arg        SDIO_DATABLOCKSIZE_8BYTES: block size = 8 bytes
      \arg        SDIO_DATABLOCKSIZE_16BYTES: block size = 16 bytes
      \arg        SDIO_DATABLOCKSIZE_32BYTES: block size = 32 bytes
      \arg        SDIO_DATABLOCKSIZE_64BYTES: block size = 64 bytes
      \arg        SDIO_DATABLOCKSIZE_128BYTES: block size = 128 bytes
      \arg        SDIO_DATABLOCKSIZE_256BYTES: block size = 256 bytes
      \arg        SDIO_DATABLOCKSIZE_512BYTES: block size = 512 bytes
      \arg        SDIO_DATABLOCKSIZE_1024BYTES: block size = 1024 bytes
      \arg        SDIO_DATABLOCKSIZE_2048BYTES: block size = 2048 bytes
      \arg        SDIO_DATABLOCKSIZE_4096BYTES: block size = 4096 bytes
      \arg        SDIO_DATABLOCKSIZE_8192BYTES: block size = 8192 bytes
      \arg        SDIO_DATABLOCKSIZE_16384BYTES: block size = 16384 bytes
*/
static uint32_t _emmc_datablocksize_get(uint16_t bytesnumber)
{
    uint8_t exp_val = 0U;
    /* calculate the exponent of 2 */
    while(1U != bytesnumber) {
        bytesnumber >>= 1U;
        ++exp_val;
    }
    return DATACTL_BLKSZ(exp_val);
}

/*!
    \brief      deinitialize the SDIO
    \param[in]  sdio_periph: SDIOx(x=0,1)
    \param[out] none
    \retval     none
*/
static void _emmc_deinit(uint32_t sdio_periph)
{
    switch(sdio_periph) {
    /* deinitialize SDIO0 */
    case SDIO0:
        hal_rcu_periph_reset_enable(RCU_SDIO0RST);
        hal_rcu_periph_reset_disable(RCU_SDIO0RST);
        break;
    /* deinitialize SDIO1 */
    case SDIO1:
        hal_rcu_periph_reset_enable(RCU_SDIO1RST);
        hal_rcu_periph_reset_disable(RCU_SDIO1RST);
        break;
    default:
        break;
    }
}

/*!
    \brief      configure the bus mode
    \param[in]  emmc_dev: SD eMMC device information structure
    \param[in]  busmode: the bus mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_BUSMODE_1BIT: 1-bit SDIO card bus mode
      \arg        SDIO_BUSMODE_4BIT: 4-bit SDIO card bus mode
      \arg        SDIO_BUSMODE_8BIT: 8-bit SDIO card bus mode
    \param[in]  speed: the bus speed mode
                only one parameter can be selected which is shown as below:
      \arg        SDIO_DATA_RATE_SDR: SDR selected
      \arg        SDIO_DATA_RATE_DDR: DDR selected

    \param[out] none
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_bus_mode_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t busmode, uint32_t speed)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;

    /* send CMD13(SEND_STATUS), addressed card sends its status registers */
    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                      (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(emmc_dev->periph);
    /* check if some error occurs */
    status = _r1_error_check(emmc_dev, EMMC_CMD_SEND_STATUS);
    if(HAL_EMMC_OK == status) {

        if(SDIO_DATA_RATE_DDR == speed) {
            /* send CMD6(EMMC_CMD_SWITCH) to set the width */
            hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SWITCH_FUNC, (uint32_t)0x03B90100,
                                            SDIO_RESPONSETYPE_SHORT);
            hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
            /* enable the CSM */
            hals_sdio_csm_enable(emmc_dev->periph);

            /* check if some error occurs */
            status = _r1_error_check(emmc_dev, EMMC_CMD_SWITCH_FUNC);
            if(HAL_EMMC_OK == status) {

                if(SDIO_BUSMODE_8BIT == busmode) {
                    /* send CMD6(EMMC_CMD_SWITCH) to set the width */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SWITCH_FUNC, (uint32_t)0x03B70600,
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    /* enable the CSM */
                    hals_sdio_csm_enable(emmc_dev->periph);

                    /* check if some error occurs */
                    status = _r1_error_check(emmc_dev, EMMC_CMD_SWITCH_FUNC);
                    if(HAL_EMMC_OK == status) {
                        /* configure eMMC bus width and the SDIO */
                        status = _emmc_bus_width_config(emmc_dev, EMMC_BUS_WIDTH_8BIT);
                        if(HAL_EMMC_OK == status) {
                            hals_sdio_clock_config(emmc_dev->periph, SDIO_SDIOCLKEDGE_FALLING, SDIO_CLOCKPWRSAVE_DISABLE,
                                                EMMC_CLK_DIV_TRANS_DSPEED);
                            hals_sdio_bus_mode_set(emmc_dev->periph, busmode);
                            hals_sdio_data_rate_set(emmc_dev->periph, SDIO_DATA_RATE_DDR);
                            hals_sdio_bus_speed_set(emmc_dev->periph, SDIO_BUSSPEED_HIGH);
                            hals_sdio_hardware_clock_enable(emmc_dev->periph);
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                } else if(SDIO_BUSMODE_4BIT == busmode) {
                    /* send CMD6(EMMC_CMD_SWITCH) to set the width */
                    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SWITCH_FUNC, (uint32_t)0x03B70500,
                                                    SDIO_RESPONSETYPE_SHORT);
                    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                    /* enable the CSM */
                    hals_sdio_csm_enable(emmc_dev->periph);

                    /* check if some error occurs */
                    status = _r1_error_check(emmc_dev, EMMC_CMD_SWITCH_FUNC);
                    if(HAL_EMMC_OK == status) {
                        /* configure eMMC bus width and the SDIO */
                        status = _emmc_bus_width_config(emmc_dev, EMMC_BUS_WIDTH_4BIT);
                        if(HAL_EMMC_OK == status) {
                            hals_sdio_clock_config(emmc_dev->periph, SDIO_SDIOCLKEDGE_FALLING, SDIO_CLOCKPWRSAVE_DISABLE,
                                                EMMC_CLK_DIV_TRANS_DSPEED);
                            hals_sdio_bus_mode_set(emmc_dev->periph, busmode);
                            hals_sdio_data_rate_set(emmc_dev->periph, SDIO_DATA_RATE_DDR);
                            hals_sdio_bus_speed_set(emmc_dev->periph, SDIO_BUSSPEED_HIGH);
                            hals_sdio_hardware_clock_enable(emmc_dev->periph);
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                } else {
                    status = HAL_EMMC_PARAMETER_INVALID;
                }
            } else {
                /* do nothing */
            }
        } else {
            if(SDIO_BUSMODE_8BIT == busmode) {
                /* send CMD6(EMMC_CMD_SWITCH) to set the width */
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SWITCH_FUNC, (uint32_t)0x03B70200,
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                /* enable the CSM */
                hals_sdio_csm_enable(emmc_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(emmc_dev, EMMC_CMD_SWITCH_FUNC);
                if(HAL_EMMC_OK == status) {

                    /* configure eMMC bus width and the SDIO */
                    status = _emmc_bus_width_config(emmc_dev, EMMC_BUS_WIDTH_8BIT);
                    if(HAL_EMMC_OK == status) {
                        hals_sdio_clock_config(emmc_dev->periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, EMMC_CLK_DIV_TRANS_DSPEED);
                        hals_sdio_bus_mode_set(emmc_dev->periph, busmode);
                        hals_sdio_hardware_clock_enable(emmc_dev->periph);
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            } else if(SDIO_BUSMODE_4BIT == busmode) {
                /* send CMD6(EMMC_CMD_SWITCH) to set the width */
                hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SWITCH_FUNC, (uint32_t)0x03B70100,
                                                SDIO_RESPONSETYPE_SHORT);
                hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
                /* enable the CSM */
                hals_sdio_csm_enable(emmc_dev->periph);

                /* check if some error occurs */
                status = _r1_error_check(emmc_dev, EMMC_CMD_SWITCH_FUNC);
                if(HAL_EMMC_OK == status) {
                    /* configure eMMC bus width and the SDIO */
                    status = _emmc_bus_width_config(emmc_dev, EMMC_BUS_WIDTH_4BIT);
                    if(HAL_EMMC_OK == status) {
                        hals_sdio_clock_config(emmc_dev->periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, EMMC_CLK_DIV_TRANS_DSPEED);
                        hals_sdio_bus_mode_set(emmc_dev->periph, busmode);
                        hals_sdio_hardware_clock_enable(emmc_dev->periph);
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            } else if(SDIO_BUSMODE_1BIT == busmode) {
                /* configure eMMC bus width and the SDIO */
                status = _emmc_bus_width_config(emmc_dev, EMMC_BUS_WIDTH_1BIT);
                if(HAL_EMMC_OK == status) {
                    hals_sdio_clock_config(emmc_dev->periph, SDIO_SDIOCLKEDGE_RISING, SDIO_CLOCKPWRSAVE_DISABLE, EMMC_CLK_DIV_TRANS_DSPEED);
                    hals_sdio_bus_mode_set(emmc_dev->periph, busmode);
                    hals_sdio_hardware_clock_enable(emmc_dev->periph);
                } else {
                    /* do nothing */
                }
            } else {
                status = HAL_EMMC_PARAMETER_INVALID;
            }
        }
        if(HAL_EMMC_OK == status) {
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      get the state which the card is in
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] pcardstate: a pointer that store the card state
    \retval     hal_sdio_emmc_error_enum details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_cardstate_get(hal_sdio_emmc_dev_struct *emmc_dev, uint8_t *pcardstate)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;
    __IO uint32_t reg_status = 0U, response = 0U;

    /* send CMD13(SEND_STATUS), addressed card sends its status register */
    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_SEND_STATUS,
                                      (uint32_t)emmc_dev->card_info.card_rca << EMMC_RCA_SHIFT, SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_csm_enable(emmc_dev->periph);

    /* store the content of SDIO_STAT */
    reg_status = SDIO_STAT(emmc_dev->periph);
    while(!(reg_status & (SDIO_FLAG_CCRCERR | SDIO_FLAG_CMDTMOUT | SDIO_FLAG_CMDRECV))) {
        reg_status = SDIO_STAT(emmc_dev->periph);
    }
    /* check whether an error or timeout occurs or command response received */
    if(reg_status & SDIO_FLAG_CCRCERR) {
        status = HAL_EMMC_CMD_CRC_ERROR;
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_CCRCERR);
    } else if(reg_status & SDIO_FLAG_CMDTMOUT) {
        status = HAL_EMMC_CMD_RESP_TIMEOUT;
        hals_sdio_flag_clear(emmc_dev->periph, SDIO_FLAG_CMDTMOUT);
    } else {

        /* command response received, store the response command index */
        reg_status = (uint32_t)hals_sdio_command_index_get(emmc_dev->periph);
        if((uint32_t)EMMC_CMD_SEND_STATUS != reg_status) {
            status = HAL_EMMC_ILLEGAL_COMMAND;
        } else {
            /* clear all the SDIO_INTC flags */
            hals_sdio_flag_clear(emmc_dev->periph, SDIO_MASK_INTC_FLAGS);
            /* get the SDIO response register 0 for checking */
            response = hals_sdio_response_get(emmc_dev->periph, SDIO_RESPONSE0);
            *pcardstate = (uint8_t)((response >> 9U) & 0x0000000FU);

            if(EMMC_ALLZERO == (response & EMMC_R1_ERROR_BITS)) {
                /* no error occurs, return HAL_EMMC_OK */
                status = HAL_EMMC_OK;
            } else {
                /* if some error occurs, return the error type */
                status = _r1_error_type_check(response);
            }
        }
    }

    return status;
}

/*!
    \brief      read block interrupt service
    \param[in]  emmc_dev: pointer to a emmc device information structure
    \param[out] none
    \retval     none
*/
static void _emmc_read_interrupt(void *emmc_dev)
{
    uint32_t count;
    uint32_t restwords;
    uint32_t *ptempbuff;
    hal_sdio_emmc_dev_struct *p_emmc = (hal_sdio_emmc_dev_struct *)emmc_dev;

    ptempbuff = p_emmc->p_rxbuffer;

    if(p_emmc->rx_length >= EMMC_FIFOHALF_BYTES) {
        /* at least 8 words can be read in the FIFO */
        for(count = 0U; count < EMMC_FIFOHALF_WORDS; count++) {
            *(ptempbuff + count) = hals_sdio_data_read(p_emmc->periph);
        }
        ptempbuff += EMMC_FIFOHALF_WORDS;
        p_emmc->rx_length -= EMMC_FIFOHALF_BYTES;
    } else {
        restwords = (p_emmc->rx_length + 3U) / 4U;
        for(count = 0U; count < restwords; count++) {
            *ptempbuff = hals_sdio_data_read(p_emmc->periph);
            ++ptempbuff;
        }
        ptempbuff -= restwords;
    }

    p_emmc->p_rxbuffer = ptempbuff;
}

/*!
    \brief      write block interrupt service.
    \param[in]  emmc_dev: pointer to a emmc device information structure
    \param[out] none
    \retval     none
*/
static void _emmc_write_interrupt(void *emmc_dev)
{
    uint32_t count;
    uint32_t restwords;
    uint32_t *ptempbuff;
    hal_sdio_emmc_dev_struct *p_emmc = (hal_sdio_emmc_dev_struct *)emmc_dev;

    ptempbuff = p_emmc->p_txbuffer;

    /* at least 8 words can be written into the FIFO */
    if(p_emmc->tx_length >= EMMC_FIFOHALF_BYTES) {
        for(count = 0U; count < EMMC_FIFOHALF_WORDS; count++) {
            hals_sdio_data_write(p_emmc->periph, *(ptempbuff + count));
        }
        /* 8 words(32 bytes) has been transferred */
        ptempbuff += EMMC_FIFOHALF_WORDS;
        p_emmc->tx_length -= EMMC_FIFOHALF_BYTES;
    } else {
        restwords = (p_emmc->tx_length + 3U) / 4U;
        for(count = 0U; count < restwords; count++) {
            hals_sdio_data_write(p_emmc->periph, *ptempbuff);
            ++ptempbuff;
        }
        ptempbuff -= restwords;
    }

    p_emmc->p_txbuffer = ptempbuff;
}

/*!
    \brief      emmc transmit/received handle
    \param[in]  emmc_dev: pointer to a emmc device information structure
    \param[out] none
    \retval     none
*/
static void _emmc_transmit_received_handle(void *emmc_dev)
{
    hal_sdio_emmc_dev_struct *p_emmc = (hal_sdio_emmc_dev_struct *)emmc_dev;
    hal_emmc_user_cb p_func = NULL;

    __IO uint32_t errorstate;

    /* disable idma for idma transfer */
    hals_sdio_idma_disable(p_emmc->periph);

    /* disable all the interrupts */
    hals_sdio_interrupt_disable(p_emmc->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_DTEND | \
                                                SDIO_INT_TFH | SDIO_INT_RFH | SDIO_INT_TXURE | SDIO_INT_RXORE);
    hals_sdio_trans_start_disable(p_emmc->periph);

    if(0U != (p_emmc->transfer & EMMC_TRANSFER_READ_MULTIPLE_BLOCK)) {
        errorstate = _emmc_transfer_stop(p_emmc);
        if(HAL_EMMC_OK != errorstate) {
            if(NULL != p_emmc->emmc_irq.transfer_error_handle) {
                p_emmc->emmc_irq.transfer_error_handle(p_emmc);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* clear all the static flags */
    hals_sdio_flag_clear(p_emmc->periph, SDIO_MASK_DATA_FLAGS);

    p_emmc->state = HAL_EMMC_STATE_READY;
    if(0U != (p_emmc->transfer & EMMC_TRANSFER_IT)) {
        if((0U != (p_emmc->transfer & EMMC_TRANSFER_READ_SINGLE_BLOCK)) || \
           (0U != (p_emmc->transfer & EMMC_TRANSFER_READ_MULTIPLE_BLOCK))) {
            p_func = (hal_emmc_user_cb)p_emmc->receive_complete_callback;
            if(NULL != p_func) {
                p_func(p_emmc);
            } else {
                /* do nothing */
            }
        } else {
            p_func = (hal_emmc_user_cb)p_emmc->transmit_complete_callback;
            if(NULL != p_func) {
                p_func(p_emmc);
            } else {
                /* do nothing */
            }
        }
    } else if(0U != (p_emmc->transfer & EMMC_TRANSFER_DMA)) {
        /* stop Transfer for Write Multi blocks or Read Multi blocks */
        if(0U != (p_emmc->transfer & EMMC_TRANSFER_READ_MULTIPLE_BLOCK)) {
            errorstate = _emmc_transfer_stop(emmc_dev);
            if(HAL_EMMC_OK != errorstate) {
                p_func = (hal_emmc_user_cb)p_emmc->transfer_error_callback;
                if(NULL != p_func) {
                    p_func(p_emmc);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear all the static flags */
        hals_sdio_flag_clear(p_emmc->periph, SDIO_MASK_DATA_FLAGS);

        /* stop transfer for write multi blocks or read multi blocks */
        if((0U != (p_emmc->transfer & EMMC_TRANSFER_WRITE_SINGLE_BLOCK)) || \
           (0U != (p_emmc->transfer & EMMC_TRANSFER_WRITE_MULTIPLE_BLOCK))) {
            p_func = (hal_emmc_user_cb)p_emmc->transmit_complete_callback;
            if(NULL != p_func) {
                p_func(p_emmc);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        if((0U != (p_emmc->transfer & EMMC_TRANSFER_READ_SINGLE_BLOCK)) || \
           (0U != (p_emmc->transfer & EMMC_TRANSFER_READ_MULTIPLE_BLOCK))) {
            p_func = (hal_emmc_user_cb)p_emmc->receive_complete_callback;
            if(NULL != p_func) {
                p_func(p_emmc);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      emmc idma complete handle
    \param[in]  emmc_dev: pointer to a emmc device information structure
    \param[out] none
    \retval     none
*/
static void _emmc_idma_complete_handle(void *emmc_dev)
{
    hal_sdio_emmc_dev_struct *p_emmc = (hal_sdio_emmc_dev_struct *)emmc_dev;
    hal_emmc_user_cb p_func = NULL;

    hals_sdio_interrupt_flag_clear(p_emmc->periph, SDIO_INT_FLAG_IDMAEND);

    if(hals_sdio_buffer_selection_get(p_emmc->periph) == SDIO_IDMA_BUFFER0)
    {
        /* Current buffer is buffer0, Transfer complete for buffer1 */
        if(0U != (p_emmc->transfer & EMMC_TRANSFER_WRITE_MULTIPLE_BLOCK)) {
            p_func = (hal_emmc_user_cb)p_emmc->write_dma_buffer1_complete_callback;
            if(NULL != p_func) {
                p_func(p_emmc);
            } else {
                /* do nothing */
            }
        } else {
            p_func = (hal_emmc_user_cb)p_emmc->read_dma_buffer1_complete_callback;
            if(NULL != p_func) {
                p_func(p_emmc);
            } else {
                /* do nothing */
            }
        }
    } else {
        /* Current buffer is buffer1, Transfer complete for buffer0 */
        if(0U != (p_emmc->transfer & EMMC_TRANSFER_WRITE_MULTIPLE_BLOCK)) {
            p_func = (hal_emmc_user_cb)p_emmc->write_dma_buffer0_complete_callback;
            if(NULL != p_func) {
                p_func(p_emmc);
            } else {
                /* do nothing */
            }
        } else {
            p_func = (hal_emmc_user_cb)p_emmc->read_dma_buffer0_complete_callback;
            if(NULL != p_func) {
                p_func(p_emmc);
            } else {
                /* do nothing */
            }
        }
    }
}

/*!
    \brief      emmc error handle
    \param[in]  emmc_dev: pointer to a emmc device information structure
    \param[out] none
    \retval     none
*/
static void _emmc_error_handle(void *emmc_dev)
{
    hal_sdio_emmc_dev_struct *p_emmc = (hal_sdio_emmc_dev_struct *)emmc_dev;
    hal_emmc_user_cb p_func = NULL;

    __IO uint32_t errorstate;

    /* set different errors */
    if(RESET != hals_sdio_interrupt_flag_get(p_emmc->periph, SDIO_INT_FLAG_DTCRCERR)) {
        errorstate = HAL_EMMC_DATA_CRC_ERROR;
    } else {
        /* do nothing */
    }

    if(RESET != hals_sdio_interrupt_flag_get(p_emmc->periph, SDIO_INT_FLAG_DTTMOUT)) {
        errorstate = HAL_EMMC_DATA_TIMEOUT;
    } else {
        /* do nothing */
    }

    if(RESET != hals_sdio_interrupt_flag_get(p_emmc->periph, SDIO_INT_FLAG_TXURE)) {
        errorstate = HAL_EMMC_TX_UNDERRUN_ERROR;
    } else {
        /* do nothing */
    }

    if(RESET != hals_sdio_interrupt_flag_get(p_emmc->periph, SDIO_INT_FLAG_RXORE)) {
        errorstate = HAL_EMMC_RX_OVERRUN_ERROR;
    } else {
        /* do nothing */
    }

    hals_sdio_interrupt_flag_clear(p_emmc->periph, SDIO_MASK_DATA_FLAGS);

    /* disable all the interrupts */
    hals_sdio_interrupt_disable(p_emmc->periph, SDIO_INT_DTCRCERR | SDIO_INT_DTTMOUT | SDIO_INT_DTEND | \
                                                SDIO_INT_TXURE | SDIO_INT_RXORE);

    hals_sdio_trans_start_disable(p_emmc->periph);
    hals_sdio_fifo_reset_enable(p_emmc->periph);
    hals_sdio_fifo_reset_disable(p_emmc->periph);

    /* send CMD12 to stop data transfer in multiple blocks operation */
    errorstate = _emmc_transfer_stop(p_emmc);
    hals_sdio_flag_clear(p_emmc->periph, SDIO_FLAG_DTABORT);

    if(0U != (p_emmc->transfer & EMMC_TRANSFER_IT)) {
        /* set the state to ready to be able to start again the process */
        p_emmc->state = HAL_EMMC_STATE_READY;
        p_func = (hal_emmc_user_cb)p_emmc->transfer_error_callback;
        if(NULL != p_func) {
            p_func(p_emmc);
        } else {
            /* do nothing */
        }
    } else if(0U != (p_emmc->transfer & EMMC_TRANSFER_DMA)) {
        if(HAL_EMMC_OK != errorstate) {
            /* disable internal dma */
            hals_sdio_idma_disable(p_emmc->periph);
            p_func = (hal_emmc_user_cb)p_emmc->transfer_error_callback;
            if(NULL != p_func) {
                p_func(p_emmc);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      stop an ongoing data transfer
    \param[in]  emmc_dev: emmc device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sdio_emmc_error_enum  details refer to gd32h7xx_hal_sdio_emmc.h
*/
static hal_sdio_emmc_error_enum _emmc_transfer_stop(hal_sdio_emmc_dev_struct *emmc_dev)
{
    hal_sdio_emmc_error_enum status = HAL_EMMC_OK;

    /* send CMD12(STOP_TRANSMISSION) to stop transmission */
    hals_sdio_command_response_config(emmc_dev->periph, EMMC_CMD_STOP_TRANSMISSION, 0U,
                                      SDIO_RESPONSETYPE_SHORT);
    hals_sdio_wait_type_set(emmc_dev->periph, SDIO_WAITTYPE_NO);
    hals_sdio_trans_stop_enable(emmc_dev->periph);
    hals_sdio_trans_start_disable(emmc_dev->periph);
    hals_sdio_csm_enable(emmc_dev->periph);
    /* check if some error occurs */
    status = _r1_error_check(emmc_dev, EMMC_CMD_STOP_TRANSMISSION);
    hals_sdio_trans_stop_disable(emmc_dev->periph);

    if(HAL_EMMC_OUT_OF_RANGE == status)
    {
        status = HAL_EMMC_OK;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      position value get
    \param[in]  data : the input data
    \param[out] none
    \retval     none
*/
static uint8_t _position_val_get(uint32_t data)
{
    uint8_t count = 0U;
    uint32_t mask = 0x80000000U;

    if(0U == data) {
        count = 32U;
    } else {
        while (0U == (data & mask)) {
            count += 1U;
            mask = mask >> 1U;
        }
    }

    return count;
}
