/*!
    \file    gd32h7xx_hal_rtc.c
    \brief   RTC driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* RTC timeout value */
#define RTC_WTWF_TIMEOUT     ((uint32_t)0x00004000U) /*!< wakeup timer can be written flag timeout */
#define RTC_INITM_TIMEOUT    ((uint32_t)0x00004000U) /*!< initialization state flag timeout */
#define RTC_RSYNF_TIMEOUT    ((uint32_t)0x00008000U) /*!< register synchronization flag timeout */
#define RTC_HRFC_TIMEOUT     ((uint32_t)0x20000000U) /*!< recalibration pending flag timeout */
#define RTC_SHIFTCTL_TIMEOUT ((uint32_t)0x00001000U) /*!< shift function operation pending flag timeout */
#define RTC_ALRMXWF_TIMEOUT  ((uint32_t)0x00008000U) /*!< alarm configuration can be written flag timeout */

/* alarm0 interrupt handler function */
static void _rtc_alarm0_handler(void *rtc);
/* alarm1 interrupt handler function */
static void _rtc_alarm1_handler(void *rtc);
/* tamper0 interrupt handler function */
static void _rtc_tamper0_handler(void *rtc);
/* tamper1 interrupt handler function */
static void _rtc_tamper1_handler(void *rtc);
/* timestamp interrupt handler function */
static void _rtc_timestamp_handler(void *rtc);
/* wakeup interrupt handler function */
static void _rtc_wakeup_handler(void *rtc);

/*!
    \brief      reset most of the RTC registers
    \param[in]  rtc_dev: RTCS device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hal_rtc_deinit(hal_rtc_dev_struct *rtc_dev)
{

    ErrStatus error_status       = ERROR;
    volatile uint32_t time_index = RTC_WTWF_TIMEOUT;
    uint32_t flag_status         = RESET;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] address is invalid");
        /* return function state */
        return ERROR;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /* After Backup domain reset, some of the RTC registers are write-protected: RTC_TIME, RTC_DATE, RTC_PSC,
       RTC_HRFC, RTC_SHIFTCTL, the bit INITM in RTC_STAT and the bits CS, S1H, A1H, REFEN in RTC_CTL. */

    /* RTC_TAMP register is not under write protection */
    rtc_dev->state = HAL_RTC_STATE_BUSY;

    RTC_TAMP = RTC_REGISTER_RESET;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* RTC alarmx related registers are not under the protection of RTC_WPK, different from former GD32MCU */
    RTC_CTL &= ((uint32_t)~(RTC_CTL_ALARM0EN | RTC_CTL_ALARM1EN));
    /* to write RTC_ALRMxTD and RTC_ALRMxSS register, 1  ALRMxEN bit in RTC_CTL register should be reset as the
      condition 2 or in INIT mode */
    RTC_ALRM0TD = RTC_REGISTER_RESET;
    RTC_ALRM1TD = RTC_REGISTER_RESET;
    RTC_ALRM0SS = RTC_REGISTER_RESET;
    RTC_ALRM1SS = RTC_REGISTER_RESET;

    /* 1 only when RTC_CTL_WTEN=0 and RTC_STAT_WTWF=1 can write RTC_CTL[2: 0]
       2 or configure the wakeup timer in INIT mode*/
    RTC_CTL &= ((uint32_t)~RTC_CTL_WTEN);
    /* wait until the WTWF flag to be set */
    do {
        flag_status = RTC_STAT & RTC_STAT_WTWF;
    } while(((--time_index) > 0U) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_status = ERROR;
    } else {
        RTC_CTL &= ((uint32_t)~RTC_CTL_WTCS);
        RTC_WUT = RTC_WUT_RESET;

        /* reset RTC_CTL register, this can be done without the init mode */
        RTC_CTL &= RTC_REGISTER_RESET;

        /* enter init mode */
        error_status = hals_rtc_init_mode_enter();

        if(ERROR != error_status) {
            /* before reset RTC_TIME and RTC_DATE, BPSHAD bit in RTC_CTL should be reset as the condition.
               in order to read calendar from shadow register, not the real registers being reset */
            RTC_TIME = RTC_REGISTER_RESET;
            RTC_DATE = RTC_DATE_RESET;

            RTC_PSC = RTC_PSC_RESET;
            RTC_CTL = RTC_REGISTER_RESET;

            /* reset RTC_STAT register, also exit init mode.
               at the same time, RTC_STAT_SOPF bit is reset, as the condition to reset RTC_SHIFTCTL register later */
            RTC_STAT = RTC_STAT_RESET;

            /* to write RTC_ALRM0SS register, ALRM0EN bit in RTC_CTL register should be reset as the condition */
            RTC_ALRM0TD = RTC_REGISTER_RESET;
            RTC_ALRM0SS = RTC_REGISTER_RESET;

            RTC_ALRM1TD = RTC_REGISTER_RESET;
            RTC_ALRM1SS = RTC_REGISTER_RESET;

            /* reset RTC_SHIFTCTL and RTC_HRFC register, and the bits S1H, A1H, REFEN in RTC_CTL, these can be done
             * without the init mode */
            RTC_SHIFTCTL = RTC_REGISTER_RESET;
            RTC_HRFC     = RTC_REGISTER_RESET;

            error_status = hals_rtc_register_sync_wait();
        } else {
            /* do nothing */
        }

        /* enable the write protection */
        RTC_WPK = RTC_LOCK_KEY;
    }

    /* change RTC error state */
    rtc_dev->error_state = HAL_RTC_ERROR_NONE;

    /* change state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* return error status */
    return error_status;
}

/*!
    \brief      initialize RTC structure
    \param[in]  hal_struct_type:refer to hal_rtc_struct_type_enum
    \param[in]  p_struct:pointer to RTC structure that contains the configuration information
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_struct_init(hal_rtc_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    /* initialize RTC basic init configuration structure with the default values */
    case HAL_RTC_INIT_STRUCT:
        ((hal_rtc_init_struct *)p_struct)->year            = 0U;
        ((hal_rtc_init_struct *)p_struct)->month           = RTC_JAN;
        ((hal_rtc_init_struct *)p_struct)->date            = 1U;
        ((hal_rtc_init_struct *)p_struct)->day_of_week     = RTC_MONDAY;
        ((hal_rtc_init_struct *)p_struct)->hour            = 0U;
        ((hal_rtc_init_struct *)p_struct)->minute          = 0U;
        ((hal_rtc_init_struct *)p_struct)->second          = 0U;
        ((hal_rtc_init_struct *)p_struct)->factor_asyn     = 127U;
        ((hal_rtc_init_struct *)p_struct)->factor_syn      = 255U;
        ((hal_rtc_init_struct *)p_struct)->am_pm           = RTC_AM;
        ((hal_rtc_init_struct *)p_struct)->display_format  = RTC_24HOUR;
        ((hal_rtc_init_struct *)p_struct)->daylight_saving = RTC_DAYLIGHT_SAVING_NONE;
        break;
    /* initialize RTC alarm output configuration structure with the default values */
    case HAL_RTC_ALARM_OUTPUT_CONFIG_STRUCT:
        ((hal_rtc_alarm_output_config_struct *)p_struct)->rtc_alarm_output_polarity = RTC_ALARM0_HIGH;
        ((hal_rtc_alarm_output_config_struct *)p_struct)->rtc_alarm_output_type     = RTC_ALARM_OUTPUT_OD;
        break;
    /* initialize RTC alarm configuration structure with the default values */
    case HAL_RTC_ALARM_CONFIG_STRUCT:
        ((hal_rtc_alarm_config_struct *)p_struct)->alarmx                = RTC_ALARM0;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_weekday_or_date = RTC_ALARM_DATE_SELECTED;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_day             = RTC_ALARM_MONDAY;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_hour            = 0U;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_minute          = 0U;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_second          = 0U;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_am_pm           = RTC_ALARM_AM;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_subsecond       = 0U;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_weekday_mask    = RTC_ALARM_WEEKDAY_MASK_DISABLE;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_hour_mask       = RTC_ALARM_HOUR_MASK_DISABLE;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_minute_mask     = RTC_ALARM_MINUTE_MASK_DISABLE;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_second_mask     = RTC_ALARM_SECOND_MASK_DISABLE;
        ((hal_rtc_alarm_config_struct *)p_struct)->alarm_subsecond_mask  = RTC_MSKSSC_0_14;
        break;
    /* initialize tamper configuration structure with the default values */
    case HAL_RTC_TAMPER_CONFIG_STRUCT:
        ((hal_rtc_tamper_config_struct *)p_struct)->tamper_source           = RTC_TAMPER0;
        ((hal_rtc_tamper_config_struct *)p_struct)->tamper_tamper0_trigger  = RTC_TAMPER0_TRIGGER_EDGE_RISING_LOW;
        ((hal_rtc_tamper_config_struct *)p_struct)->tamper_tamper1_trigger  = RTC_TAMPER1_TRIGGER_EDGE_RISING_LOW;
        ((hal_rtc_tamper_config_struct *)p_struct)->tamper_filter           = RTC_FLT_EDGE;
        ((hal_rtc_tamper_config_struct *)p_struct)->tamper_sample_frequency = RTC_FREQ_DIV32768;
        ((hal_rtc_tamper_config_struct *)p_struct)->tamper_precharge_enable = RTC_TAMPER_PRECHARGE_ENABLE;
        ((hal_rtc_tamper_config_struct *)p_struct)->tamper_precharge_time   = RTC_PRCH_1C;
        ((hal_rtc_tamper_config_struct *)p_struct)->tamper_with_timestamp   = RTC_TAMPER_WITH_TIMESTAMP_DISABLE;
        break;
    case HAL_RTC_TIMESTAMP_CONFIG_STRUCT:
        /* initialize calibration configuration structure with the default values */
        ((hal_rtc_timestamp_config_struct *)p_struct)->timestamp_month     = RTC_JAN;
        ((hal_rtc_timestamp_config_struct *)p_struct)->timestamp_date      = 1U;
        ((hal_rtc_timestamp_config_struct *)p_struct)->timestamp_weekday   = 0U;
        ((hal_rtc_timestamp_config_struct *)p_struct)->timestamp_hour      = 0U;
        ((hal_rtc_timestamp_config_struct *)p_struct)->timestamp_minute    = 0U;
        ((hal_rtc_timestamp_config_struct *)p_struct)->timestamp_second    = 0U;
        ((hal_rtc_timestamp_config_struct *)p_struct)->timestamp_subsecond = 0U;
        ((hal_rtc_timestamp_config_struct *)p_struct)->timestamp_am_pm     = RTC_ALARM_AM;
        break;
    /* initialize RTC user interrupt callback structure with the default values */
    case HAL_RTC_USER_CALLBACK_STRUCT:
        ((hal_rtc_user_callback_struct *)p_struct)->alarm0_func  = NULL;
        ((hal_rtc_user_callback_struct *)p_struct)->alarm1_func  = NULL;
        ((hal_rtc_user_callback_struct *)p_struct)->tamper0_func = NULL;
        ((hal_rtc_user_callback_struct *)p_struct)->tamper1_func = NULL;
        ((hal_rtc_user_callback_struct *)p_struct)->wakeup_func  = NULL;
        break;
    /* initialize calibration configuration structure with the default values */
    case HAL_RTC_CALIB_CONFIG_STRUCT:
        ((hal_rtc_calib_config_struct *)p_struct)->calib_shielding_number = 0U;
        ((hal_rtc_calib_config_struct *)p_struct)->calib_cycle            = RTC_CALIBRATION_WINDOW_32S;
        ((hal_rtc_calib_config_struct *)p_struct)->calib_frequency        = RTC_CALIBRATION_PLUS_RESET;
        ((hal_rtc_calib_config_struct *)p_struct)->calib_output           = 0U;
        break;
    /* initialize RTC irq handle configuration structure with the default values */
    case HAL_RTC_IRQ_STRUCT:
        ((hal_rtc_irq_struct *)p_struct)->rtc_timestamp_handle = NULL;
        ((hal_rtc_irq_struct *)p_struct)->rtc_alarm0_handle    = NULL;
        ((hal_rtc_irq_struct *)p_struct)->rtc_alarm1_handle    = NULL;
        ((hal_rtc_irq_struct *)p_struct)->rtc_tamper0_handle   = NULL;
        ((hal_rtc_irq_struct *)p_struct)->rtc_tamper1_handle   = NULL;
        ((hal_rtc_irq_struct *)p_struct)->rtc_wakeup_handle    = NULL;
        break;
    /* initialize RTC dev info configuration structure with the default values */
    case HAL_RTC_DEV_STRUCT:
        ((hal_rtc_dev_struct *)p_struct)->rtc_irq.rtc_timestamp_handle = NULL;
        ((hal_rtc_dev_struct *)p_struct)->rtc_irq.rtc_alarm0_handle    = NULL;
        ((hal_rtc_dev_struct *)p_struct)->rtc_irq.rtc_alarm1_handle    = NULL;
        ((hal_rtc_dev_struct *)p_struct)->rtc_irq.rtc_tamper0_handle   = NULL;
        ((hal_rtc_dev_struct *)p_struct)->rtc_irq.rtc_tamper1_handle   = NULL;
        ((hal_rtc_dev_struct *)p_struct)->rtc_irq.rtc_wakeup_handle    = NULL;
        ((hal_rtc_dev_struct *)p_struct)->error_state                  = HAL_RTC_ERROR_NONE;
        ((hal_rtc_dev_struct *)p_struct)->state                        = HAL_RTC_STATE_NONE;
        ((hal_rtc_dev_struct *)p_struct)->mutex                        = HAL_MUTEX_UNLOCKED;
        ((hal_rtc_dev_struct *)p_struct)->priv                         = NULL;
        break;
    case HAL_RTC_WAKEUP_CLK_STRUCT:
        ((hal_rtc_wakeup_clk_struct *)p_struct)->wakeup_clock = WAKEUP_RTCCK_DIV16;
        ((hal_rtc_wakeup_clk_struct *)p_struct)->wakeup_timer = 0U;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        error_code = HAL_ERR_VAL;
        break;
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      initialize RTC registers
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtc_init: pointer to a rtc_parameter_struct structure which contains
                parameters for initialization of the RTC peripheral
                members of the structure and the member values are shown as below:
                  year: 0x0 - 0x99(BCD format)
                  month: RTC_JAN, RTC_FEB, RTC_MAR, RTC_APR, RTC_MAY, RTC_JUN,
                             RTC_JUL, RTC_AUG, RTC_SEP, RTC_OCT, RTC_NOV, RTC_DEC
                  date: 0x1 - 0x31(BCD format)
                  day_of_week: RTC_MONDAY, RTC_TUESDAY, RTC_WEDNESDAY, RTC_THURSDAY
                                   RTC_FRIDAY, RTC_SATURDAY, RTC_SUNDAY
                  hour: 0x0 - 0x12(BCD format) or 0x0 - 0x23(BCD format) depending on the RTC display_format chose
                  minute: 0x0 - 0x59(BCD format)
                  second: 0x0 - 0x59(BCD format)
                  factor_asyn: 0x0 - 0x7F
                  factor_syn: 0x0 - 0x7FFF
                  am_pm: RTC_AM, RTC_PM
                  display_format: RTC_24HOUR, RTC_12HOUR
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_init(hal_rtc_dev_struct *rtc_dev, hal_rtc_init_struct *rtc_init)
{
    uint32_t reg_time      = 0U;
    uint32_t reg_date      = 0U;
    int32_t  error_code    = HAL_ERR_NONE;
    ErrStatus error_status = ERROR;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == rtc_dev) || (NULL == rtc_init)) {
        HAL_DEBUGE("pointer [rtc_dev] or pointer [rtc_init] address is invalid");
        /* return function state */
        return HAL_ERR_ADDRESS;
    }
    /* check parameter rtc_init->year */
    if(rtc_init->year > 99U) {
        HAL_DEBUGE("parameter [rtc_init->rtc_year] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_init->date */
    if((rtc_init->date < 1U) || (rtc_init->date > 31U)) {
        HAL_DEBUGE("parameter [rtc_init->rtc_date] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_init->display_format */
    if((RTC_24HOUR != rtc_init->display_format) && (RTC_12HOUR != rtc_init->display_format)) {
        HAL_DEBUGE("parameter [rtc_init->display_format] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_init->hour, and rtc_init->am_pm */
    if(RTC_12HOUR == rtc_init->display_format) {
        if((rtc_init->hour < 1U) || (rtc_init->hour > 12U)) {
            HAL_DEBUGE("parameter [rtc_init->hour] value is invalid");
            return HAL_ERR_VAL;
        }
        if((RTC_AM != rtc_init->am_pm) && (RTC_PM != rtc_init->am_pm)) {
            HAL_DEBUGE("parameter [rtc_init->rtc_am_pm] value is invalid");
            return HAL_ERR_VAL;
        }
    } else {
        if(rtc_init->hour > 23U) {
            HAL_DEBUGE("parameter [rtc_init->rtc_hour] value is invalid");
            return HAL_ERR_VAL;
        }
        rtc_init->am_pm = RTC_AM;
    }
    /* check parameter rtc_init->minute */
    if(rtc_init->minute > 59U) {
        HAL_DEBUGE("parameter [rtc_init->minute] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_init->second */
    if(rtc_init->second > 59U) {
        HAL_DEBUGE("parameter [rtc_init->second] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_init->factor_syn */
    if(rtc_init->factor_syn > 0x7FFFU) {
        HAL_DEBUGE("parameter [rtc_init->factor_syn] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_init->rtc_factor_asyn */
    if(rtc_init->factor_asyn > 0x7FU) {
        HAL_DEBUGE("parameter [rtc_init->factor_asyn] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* change state */
    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* 1st: disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* 2nd: enter init mode */
    error_status = hals_rtc_init_mode_enter();

    if(ERROR != error_status) {
        RTC_PSC = (uint32_t)(PSC_FACTOR_A(rtc_init->factor_asyn) | PSC_FACTOR_S(rtc_init->factor_syn));

        reg_date = (uint32_t)(DATE_YR(hals_rtc_normal_2_bcd(rtc_init->year))
                              | DATE_DOW(hals_rtc_normal_2_bcd(rtc_init->day_of_week))
                              | DATE_MON(hals_rtc_normal_2_bcd(rtc_init->month))
                              | DATE_DAY(hals_rtc_normal_2_bcd(rtc_init->date)));

        reg_time = (uint32_t)(TIME_HR(hals_rtc_normal_2_bcd(rtc_init->hour)) | TIME_MN(hals_rtc_normal_2_bcd(rtc_init->minute))
                       | TIME_SC(hals_rtc_normal_2_bcd(rtc_init->second)));

        RTC_TIME = (uint32_t)reg_time;
        RTC_DATE = (uint32_t)reg_date;

        /* reset RTC time format */
        RTC_TIME &= (uint32_t)(~RTC_PM);
        /* set RTC time format */
        RTC_TIME |= (uint32_t)rtc_init->am_pm;

        RTC_CTL &= (uint32_t)(~RTC_CTL_CS);
        RTC_CTL |= (uint32_t)rtc_init->display_format;

        /* reset daylight saving bits */
        RTC_CTL &= (uint32_t)(~(RTC_DAYLIGHT_SAVING_ADD_1H | RTC_DAYLIGHT_SAVING_SUB_1H | RTC_CTL_DSM));
        /* set daylight saving bits */
        RTC_CTL |= (uint32_t)rtc_init->daylight_saving;

        /* 3rd: exit init mode */
        hals_rtc_init_mode_exit();

        /* 4th: wait the RSYNF flag to set */
        error_status = hals_rtc_register_sync_wait();
    } else {
        /* do nothing */
    }

    /* 5th: enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
    if(ERROR == error_status) {
        error_code = HAL_ERR_BUSY;
    } else {
        /* change RTC error state */
        rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    }

    /* change state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC alarm output source
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtc_alarm_output_config: specify signal to output
    \arg        rtc_alarm_output_polarity:
                only one parameter can be selected which is shown as below:
                      RTC_ALARM0_HIGH: when the  alarm0 flag is set, the output pin is high
                      RTC_ALARM0_LOW: when the  alarm0 flag is set, the output pin is low
                      RTC_ALARM1_HIGH: when the  alarm1 flag is set, the output pin is high
                      RTC_ALARM1_LOW: when the  alarm1 flag is set, the output pin is low
                      RTC_WAKEUP_HIGH: when the  wakeup flag is set, the output pin is high
                      RTC_WAKEUP_LOW: when the  wakeup flag is set, the output pin is low
      \arg      rtc_alarm_output_type:
                  mode: specify the output pin mode when output alarm signal or auto wakeup signal
                      RTC_ALARM_OUTPUT_OD: open drain mode
                      RTC_ALARM_OUTPUT_PP: push pull mode
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_alarm_output_config(hal_rtc_dev_struct *rtc_dev, hal_rtc_alarm_output_config_struct *rtc_alarm_output_config)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == rtc_dev) || (NULL == rtc_alarm_output_config)) {
        HAL_DEBUGE("pointer [rtc_dev] or [rtc_alarm_output_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);

    rtc_dev->state = HAL_RTC_STATE_BUSY;
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* reset output polarity */
    RTC_CTL &= ~(RTC_CTL_OS | RTC_CTL_OPOL);
    /* reset alarm output type */
    RTC_CFG &= ~(uint32_t)RTC_CFG_ALRMOUTTYPE;

    RTC_CTL |= (uint32_t)(rtc_alarm_output_config->rtc_alarm_output_polarity);
    /* alarm output */
    RTC_CFG |= (uint32_t)(rtc_alarm_output_config->rtc_alarm_output_type);
    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* change RTC error state */
    rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    /* change state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC alarm
    \param[in]  rtc_dev: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtc_alarm_config: the pointer of RTC alarm configure structure
                  rtc_alarm_am_pm:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_AM: AM format
      \arg          RTC_ALARM_PM: PM format
                  rtc_alarm_weekday_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_WEEKDAY_MASK_DISABLE: RTC alarm weekday no mask
      \arg          RTC_ALARM_WEEKDAY_MASK_ENABLE: RTC alarm weekday mask
                  rtc_alarm_hour_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_HOUR_MASK_DISABLE: RTC alarm hour no mask
      \arg          RTC_ALARM_HOUR_MASK_ENABLE: RTC alarm hour mask
                  rtc_alarm_minute_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_MINUTE_MASK_DISABLE: RTC alarm minute no mask
      \arg          RTC_ALARM_MINUTE_MASK_ENABLE: RTC alarm minute mask
                  rtc_alarm_second_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_SECOND_MASK_DISABLE: RTC alarm second no mask
      \arg          RTC_ALARM_SECOND_MASK_ENABLE: RTC alarm second mask
                  rtc_alarm_subsecond_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_MASKSSC_0_14: mask alarm subsecond configuration
      \arg          RTC_MASKSSC_1_14: mask RTC_ALRMXSS_SSC[14:1], and RTC_ALRMXSS_SSC[0] is to be compared
      \arg          RTC_MASKSSC_2_14: mask RTC_ALRMXSS_SSC[14:2], and RTC_ALRMXSS_SSC[1:0] is to be compared
      \arg          RTC_MASKSSC_3_14: mask RTC_ALRMXSS_SSC[14:3], and RTC_ALRMXSS_SSC[2:0] is to be compared
      \arg          RTC_MASKSSC_4_14: mask RTC_ALRMXSS_SSC[14:4], and RTC_ALRMXSS_SSC[3:0] is to be compared
      \arg          RTC_MASKSSC_5_14: mask RTC_ALRMXSS_SSC[14:5], and RTC_ALRMXSS_SSC[4:0] is to be compared
      \arg          RTC_MASKSSC_6_14: mask RTC_ALRMXSS_SSC[14:6], and RTC_ALRMXSS_SSC[5:0] is to be compared
      \arg          RTC_MASKSSC_7_14: mask RTC_ALRMXSS_SSC[14:7], and RTC_ALRMXSS_SSC[6:0] is to be compared
      \arg          RTC_MASKSSC_8_14: mask RTC_ALRMXSS_SSC[14:8], and RTC_ALRMXSS_SSC[7:0] is to be compared
      \arg          RTC_MASKSSC_9_14: mask RTC_ALRMXSS_SSC[14:9], and RTC_ALRMXSS_SSC[8:0] is to be compared
      \arg          RTC_MASKSSC_10_14: mask RTC_ALRMXSS_SSC[14:10], and RTC_ALRMXSS_SSC[9:0] is to be compared
      \arg          RTC_MASKSSC_11_14: mask RTC_ALRMXSS_SSC[14:11], and RTC_ALRMXSS_SSC[10:0] is to be compared
      \arg          RTC_MASKSSC_12_14: mask RTC_ALRMXSS_SSC[14:12], and RTC_ALRMXSS_SSC[11:0] is to be compared
      \arg          RTC_MASKSSC_13_14: mask RTC_ALRMXSS_SSC[14:13], and RTC_ALRMXSS_SSC[12:0] is to be compared
      \arg          RTC_MASKSSC_14: mask RTC_ALRMXSS_SSC[14], and RTC_ALRMXSS_SSC[13:0] is to be compared
      \arg          RTC_MASKSSC_NONE: mask none, and RTC_ALRMXSS_SSC[14:0] is to be compared
                  rtc_weekday_or_date:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_SELECT_WEEKDAY: RTC alarm select weekday
      \arg          RTC_ALARM_SELECT_DATE: RTC alarm select date
                  rtc_alarm_day: 1 - 31 or below
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_MONDAY: RTC alarm Monday
      \arg          RTC_ALARM_TUESDAY: RTC alarm Tuesday
      \arg          RTC_ALARM_WEDSDAY: RTC alarm Wednesday
      \arg          RTC_ALARM_THURSDAY: RTC alarm Thursday
      \arg          RTC_ALARM_FRIDAY: RTC alarm Friday
      \arg          RTC_ALARM_SATURDAY: RTC alarm Saturday
      \arg          RTC_ALARM_SUNDAY: RTC alarm Sunday
                  rtc_hour: 0 - 24 or 1 - 12
                  rtc_minute: 0 - 59
                  rtc_second: 0 - 59
                  rtc_alarm_subsecond: 0x00~0x7FFF
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_alarm_config(hal_rtc_dev_struct *rtc_dev, hal_rtc_alarm_config_struct *rtc_alarm_config)
{
    int32_t error_code  = HAL_ERR_NONE;
    uint32_t reg_alrmtd = 0x00000000U;
    uint32_t reg_alrmss = 0x00000000U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    uint32_t rtc_display_format = 0U;
    if((NULL == rtc_dev) || (NULL == rtc_alarm_config)) {
        HAL_DEBUGE("pointer [rtc_dev] or pointer [rtc_alarm_config] address is invalid");
        /* return function state */
        return HAL_ERR_ADDRESS;
    }
    /* check parameter rtc_alarm_time->rtc_alarm_mask */
    if(0x80808080U != ((rtc_alarm_config->alarm_hour_mask | rtc_alarm_config->alarm_minute_mask
            | rtc_alarm_config->alarm_second_mask) | 0x80808080U)) {
        HAL_DEBUGE("parameter [rtc_alarm_time->rtc_alarm_mask] value is invalid");
        return HAL_ERR_VAL;
    }
    rtc_display_format = RTC_CTL & RTC_CTL_CS;
    /* check parameter rtc_alarm_config->alarm_hour, and rtc_alarm_config->alarm_am_pm */
    if(0U != rtc_display_format) {
        if((rtc_alarm_config->alarm_hour < 1U) || (rtc_alarm_config->alarm_hour > 12U)) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_hour] value is invalid");
            return HAL_ERR_VAL;
        }
        if((RTC_ALARM_AM != rtc_alarm_config->alarm_am_pm) && (RTC_ALARM_PM != rtc_alarm_config->alarm_am_pm)) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_am_pm] value is invalid");
            return HAL_ERR_VAL;
        }
    } else {
        if(rtc_alarm_config->alarm_hour > 23U) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_hour] value is invalid");
            return HAL_ERR_VAL;
        }
        rtc_alarm_config->alarm_am_pm = RTC_ALARM_AM;
    }
    /* check parameter rtc_alarm_time->alarm_minute */
    if(rtc_alarm_config->alarm_minute > 59U) {
        HAL_DEBUGE("parameter [rtc_alarm_config->alarm_minute] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_alarm_time->alarm_second */
    if(rtc_alarm_config->alarm_second > 59U) {
        HAL_DEBUGE("parameter [rtc_alarm_config->alarm_second] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_alarm_time->weekday_or_date, and rtc_alarm_time->alarm_day */
    if(RTC_ALARM_DATE_SELECTED == rtc_alarm_config->alarm_weekday_or_date) {
        if((rtc_alarm_config->alarm_day < 1U) || (rtc_alarm_config->alarm_day > 31U)) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_weekday_or_date] value is invalid");
            return HAL_ERR_VAL;
        }
    } else if(RTC_ALARM_WEEKDAY_SELECTED == rtc_alarm_config->alarm_weekday_or_date) {
        if((rtc_alarm_config->alarm_day < 1U) || (rtc_alarm_config->alarm_day > 7U)) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_weekday_or_date] value is invalid");
            return HAL_ERR_VAL;
        }
    } else {
        HAL_DEBUGE("parameter [rtc_alarm_config->rtc_weekday_or_date] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check parameter rtc_alarm_time->alarm_subsecond_mask */
    if(RTC_ALRM0SS_MASKSSC != (rtc_alarm_config->alarm_subsecond_mask | RTC_ALRM0SS_MASKSSC)) {
        HAL_DEBUGE("parameter [rtc_alarm_config->alarm_subsecond_mask] value is invalid");
        return HAL_ERR_VAL;
    }

    if((RTC_ALARM0 != rtc_alarm_config->alarmx) && (RTC_ALARM1 != rtc_alarm_config->alarmx)) {
        HAL_DEBUGE("parameter [rtc_alarm_config->alarmx] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);

    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* alarm config */
    reg_alrmtd = (uint32_t)(rtc_alarm_config->alarm_am_pm | rtc_alarm_config->alarm_weekday_mask
                            | rtc_alarm_config->alarm_hour_mask | rtc_alarm_config->alarm_minute_mask
                            | rtc_alarm_config->alarm_second_mask | rtc_alarm_config->alarm_weekday_or_date
                            | ALRMTD_DAY(hals_rtc_normal_2_bcd(rtc_alarm_config->alarm_day))
                            | ALRMTD_HR(hals_rtc_normal_2_bcd(rtc_alarm_config->alarm_hour))
                            | ALRMTD_MN(hals_rtc_normal_2_bcd(rtc_alarm_config->alarm_minute))
                            | ALRMTD_SC(hals_rtc_normal_2_bcd(rtc_alarm_config->alarm_second)));
    reg_alrmss = rtc_alarm_config->alarm_subsecond_mask | SS_SSC(rtc_alarm_config->alarm_subsecond);

    /* disable RTC alarm */
    if((RTC_ALARM0 == rtc_alarm_config->alarmx) && (RESET != (RTC_CTL & RTC_CTL_ALARM0EN))) {
        if(ERROR == hal_rtc_alarm_disable(RTC_ALARM0)) {
            HAL_DEBUGE("disable RTC alarm 0 error");
            error_code = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if((RTC_ALARM1 == rtc_alarm_config->alarmx) && (RESET != (RTC_CTL & RTC_CTL_ALARM1EN))) {
        if(ERROR == hal_rtc_alarm_disable(RTC_ALARM1)) {
            HAL_DEBUGE("disable RTC alarm 1 error");
            error_code = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == error_code) {
        /* disable the write protection */
        RTC_WPK = RTC_UNLOCK_KEY1;
        RTC_WPK = RTC_UNLOCK_KEY2;

        /* check parameter ALARM0 or ALARM1 */
        if(RTC_ALARM0 == rtc_alarm_config->alarmx) {
            RTC_ALRM0TD = (uint32_t)reg_alrmtd;
            RTC_ALRM0SS = reg_alrmss;
        } else {
            RTC_ALRM1TD = (uint32_t)reg_alrmtd;
            RTC_ALRM1SS = reg_alrmss;
        }

        /* enable the write protection */
        RTC_WPK = RTC_LOCK_KEY;
        /* change RTC error state */
        rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    } else {
        /* do nothing */
    }

    /* change RTC state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* lock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC alarm
    \param[in]  rtc_dev: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtc_alarm_config: the pointer of RTC alarm configure structure
                  rtc_alarm_am_pm:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_AM: AM format
      \arg          RTC_ALARM_PM: PM format
                  rtc_alarm_weekday_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_WEEKDAY_MASK_DISABLE: RTC alarm weekday no mask
      \arg          RTC_ALARM_WEEKDAY_MASK_ENABLE: RTC alarm weekday mask
                  rtc_alarm_hour_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_HOUR_MASK_DISABLE: RTC alarm hour no mask
      \arg          RTC_ALARM_HOUR_MASK_ENABLE: RTC alarm hour mask
                  rtc_alarm_minute_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_MINUTE_MASK_DISABLE: RTC alarm minute no mask
      \arg          RTC_ALARM_MINUTE_MASK_ENABLE: RTC alarm minute mask
                  rtc_alarm_second_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_SECOND_MASK_DISABLE: RTC alarm second no mask
      \arg          RTC_ALARM_SECOND_MASK_ENABLE: RTC alarm second mask
                  rtc_alarm_subsecond_mask:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_MASKSSC_0_14: RTC alarm second no mask
      \arg          RTC_MASKSSC_1_14: RTC alarm second mask
      \arg          RTC_MASKSSC_2_14: RTC alarm second no mask
      \arg          RTC_MASKSSC_3_14: RTC alarm second mask
      \arg          RTC_MASKSSC_4_14: RTC alarm second no mask
      \arg          RTC_MASKSSC_5_14: RTC alarm second mask
      \arg          RTC_MASKSSC_6_14: RTC alarm second no mask
      \arg          RTC_MASKSSC_7_14: RTC alarm second mask
      \arg          RTC_MASKSSC_8_14: RTC alarm second no mask
      \arg          RTC_MASKSSC_9_14: RTC alarm second mask
      \arg          RTC_MASKSSC_10_14: RTC alarm second no mask
      \arg          RTC_MASKSSC_11_14: RTC alarm second mask
      \arg          RTC_MASKSSC_12_14: RTC alarm second no mask
      \arg          RTC_MASKSSC_13_14: RTC alarm second no mask
      \arg          RTC_MASKSSC_14: RTC alarm second mask
      \arg          RTC_MASKSSC_NONE: RTC alarm second mask
                  rtc_weekday_or_date:
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_SELECT_WEEKDAY: RTC alarm select weekday
      \arg          RTC_ALARM_SELECT_DATE: RTC alarm select date
                  rtc_alarm_day: 1 - 31 or below
                  only one parameter can be selected which is shown as below:
      \arg          RTC_ALARM_MONDAY: RTC alarm Monday
      \arg          RTC_ALARM_TUESDAY: RTC alarm Tuesday
      \arg          RTC_ALARM_WEDSDAY: RTC alarm Wednesday
      \arg          RTC_ALARM_THURSDAY: RTC alarm Thursday
      \arg          RTC_ALARM_FRIDAY: RTC alarm Friday
      \arg          RTC_ALARM_SATURDAY: RTC alarm Saturday
      \arg          RTC_ALARM_SUNDAY: RTC alarm Sunday
                  rtc_hour: 0 - 24 or 1 - 12
                  rtc_minute: 0 - 59
                  rtc_second: 0 - 59
                  rtc_alarm_subsecond: 0x00~0x7FFF
    \param[in]  p_user_func: pointer to a hal_rtc_user_callback_struct structure
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_alarm_config_interrupt(hal_rtc_dev_struct *rtc_dev, hal_rtc_alarm_config_struct *rtc_alarm_config,
                                       hal_rtc_user_callback_struct *p_user_func)
{
    int32_t error_code  = HAL_ERR_NONE;
    uint32_t reg_alrmtd = 0x00000000U;
    uint32_t reg_alrmss = 0x00000000U;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    uint32_t rtc_display_format = 0U;
    if((NULL == rtc_dev) || (NULL == rtc_alarm_config)) {
        HAL_DEBUGE("pointer [rtc_dev] or pointer [rtc_alarm_config] address is invalid");
        /* return function state */
        return HAL_ERR_ADDRESS;
    }
    /* check parameter rtc_alarm_time->rtc_alarm_mask */
    if(0x80808080U
       != ((rtc_alarm_config->alarm_hour_mask | rtc_alarm_config->alarm_minute_mask
            | rtc_alarm_config->alarm_second_mask)
           | 0x80808080U)) {
        HAL_DEBUGE("parameter [rtc_alarm_time->rtc_alarm_mask] value is invalid");
        return HAL_ERR_VAL;
    }
    rtc_display_format = RTC_CTL & RTC_CTL_CS;
    /* check parameter rtc_alarm_config->alarm_hour, and rtc_alarm_config->alarm_am_pm */
    if(0U != rtc_display_format) {
        if((rtc_alarm_config->alarm_hour < 1U) || (rtc_alarm_config->alarm_hour > 12U)) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_hour] value is invalid");
            return HAL_ERR_VAL;
        }
        if((RTC_ALARM_AM != rtc_alarm_config->alarm_am_pm) && (RTC_ALARM_PM != rtc_alarm_config->alarm_am_pm)) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_am_pm] value is invalid");
            return HAL_ERR_VAL;
        }
    } else {
        if(rtc_alarm_config->alarm_hour > 23U) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_hour] value is invalid");
            return HAL_ERR_VAL;
        }
        rtc_alarm_config->alarm_am_pm = RTC_ALARM_AM;
    }
    /* check parameter rtc_alarm_time->alarm_minute */
    if(rtc_alarm_config->alarm_minute > 59U) {
        HAL_DEBUGE("parameter [rtc_alarm_config->alarm_minute] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_alarm_time->alarm_second */
    if(rtc_alarm_config->alarm_second > 59U) {
        HAL_DEBUGE("parameter [rtc_alarm_config->alarm_second] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_alarm_time->weekday_or_date, and rtc_alarm_time->alarm_day */
    if(RTC_ALARM_DATE_SELECTED == rtc_alarm_config->alarm_weekday_or_date) {
        if((rtc_alarm_config->alarm_day < 1U) || (rtc_alarm_config->alarm_day > 31U)) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_weekday_or_date] value is invalid");
            return HAL_ERR_VAL;
        }
    } else if(RTC_ALARM_WEEKDAY_SELECTED == rtc_alarm_config->alarm_weekday_or_date) {
        if((rtc_alarm_config->alarm_day < 1U) || (rtc_alarm_config->alarm_day > 7U)) {
            HAL_DEBUGE("parameter [rtc_alarm_config->alarm_weekday_or_date] value is invalid");
            return HAL_ERR_VAL;
        }
    } else {
        HAL_DEBUGE("parameter [rtc_alarm_config->rtc_weekday_or_date] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter rtc_alarm_time->alarm_subsecond_mask */
    if(RTC_ALRM0SS_MASKSSC != (rtc_alarm_config->alarm_subsecond_mask | RTC_ALRM0SS_MASKSSC)) {
        HAL_DEBUGE("parameter [rtc_alarm_config->alarm_subsecond_mask] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check parameter p_user_func */
    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);

    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* clear alram0 and alarm1 callback */
    rtc_dev->alarm0_callback = NULL;
    rtc_dev->alarm1_callback = NULL;

    /* alarm config */
    reg_alrmtd = (uint32_t)(rtc_alarm_config->alarm_am_pm | rtc_alarm_config->alarm_weekday_mask
                            | rtc_alarm_config->alarm_hour_mask | rtc_alarm_config->alarm_minute_mask
                            | rtc_alarm_config->alarm_second_mask | rtc_alarm_config->alarm_weekday_or_date
                            | ALRMTD_DAY(hals_rtc_normal_2_bcd(rtc_alarm_config->alarm_day))
                            | ALRMTD_HR(hals_rtc_normal_2_bcd(rtc_alarm_config->alarm_hour))
                            | ALRMTD_MN(hals_rtc_normal_2_bcd(rtc_alarm_config->alarm_minute))
                            | ALRMTD_SC(hals_rtc_normal_2_bcd(rtc_alarm_config->alarm_second)));

    reg_alrmss = rtc_alarm_config->alarm_subsecond_mask | SS_SSC(rtc_alarm_config->alarm_subsecond);

    /* disable RTC alarm */
    if((RTC_ALARM0 == rtc_alarm_config->alarmx) && (RESET != (RTC_CTL & RTC_CTL_ALARM0EN))) {
        if(ERROR == hal_rtc_alarm_disable(RTC_ALARM0)) {
            HAL_DEBUGE("disable RTC alarm 0 error");
            error_code = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if((RTC_ALARM1 == rtc_alarm_config->alarmx) && (RESET != (RTC_CTL & RTC_CTL_ALARM1EN))) {
        if(ERROR == hal_rtc_alarm_disable(RTC_ALARM1)) {
            HAL_DEBUGE("disable RTC alarm 1 error");
            error_code = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if (HAL_ERR_NONE == error_code)
    {
        /* disable the write protection */
        RTC_WPK = RTC_UNLOCK_KEY1;
        RTC_WPK = RTC_UNLOCK_KEY2;

        /* check parameter ALARM0 or ALARM1 */
        if(RTC_ALARM0 == rtc_alarm_config->alarmx) {
            RTC_ALRM0TD = (uint32_t)reg_alrmtd;
            RTC_ALRM0SS = reg_alrmss;
        } else {
            RTC_ALRM1TD = (uint32_t)reg_alrmtd;
            RTC_ALRM1SS = reg_alrmss;
        }

        /* configure ALARM0 or ALARM1 interrupt*/
        if(RTC_ALARM0 == rtc_alarm_config->alarmx) {
            /* enable alarm0 interrupt */
            hals_rtc_flag_clear(RTC_FLAG_ALARM0);
            hals_rtc_interrupt_enable(RTC_INT_ALARM0);
            rtc_dev->alarm0_callback           = (void *)p_user_func->alarm0_func;
            rtc_dev->rtc_irq.rtc_alarm0_handle = _rtc_alarm0_handler;
        } else {
            /* enable alarm1 interrupt */
            hals_rtc_flag_clear(RTC_FLAG_ALARM1);
            hals_rtc_interrupt_enable(RTC_INT_ALARM1);
            rtc_dev->alarm1_callback           = (void *)p_user_func->alarm1_func;
            rtc_dev->rtc_irq.rtc_alarm1_handle = _rtc_alarm1_handler;
        }

        /* enable RTC interrupt line */
        hal_exti_internal_init(EXTI_LINE_17_RTC_ALARM, EXTI_INTERRUPT_TRIG_RISING);
        hals_rtc_alarm_enable(rtc_alarm_config->alarmx);

        /* enable the write protection */
        RTC_WPK = RTC_LOCK_KEY;
        /* change RTC error state */
        rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    }

    /* change RTC state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC tamper
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtc_tamper_config: pointer to a rtc_tamper_struct structure which contains
                parameters for RTC tamper configuration
                members of the structure and the member values are shown as below:
                  detecting tamper event can using edge mode or level mode
                  (1) using edge mode configuration:
                  tamper_source: RTC_TAMPER0, RTC_TAMPER1
                  tamper_tamper0_trigger: RTC_TAMPER0_TRIGGER_EDGE_RISING_LOW, RTC_TAMPER0_TRIGGER_EDGE_FALLING_HIGH
                  tamper_tamper1_trigger: RTC_TAMPER1_TRIGGER_EDGE_RISING_LOW, RTC_TAMPER1_TRIGGER_EDGE_FALLING_HIGH
                  tamper_filter: RTC_FLT_EDGE
                  tamper_with_timestamp: RTC_TAMPER_WITH_TIMESTAMP_DISABLE, RTC_TAMPER_WITH_TIMESTAMP_ENABLE
                  (2) using level mode configuration:
                  tamper_source: RTC_TAMPER0, RTC_TAMPER1
                  tamper_trigger:RTC_TAMPER_TRIGGER_LEVEL_LOW, RTC_TAMPER_TRIGGER_LEVEL_HIGH
                  tamper_filter: RTC_FLT_2S, RTC_FLT_4S, RTC_FLT_8S
                  tamper_sample_frequency: RTC_FREQ_DIV32768, RTC_FREQ_DIV16384, RTC_FREQ_DIV8192,
                                               RTC_FREQ_DIV4096, RTC_FREQ_DIV2048, RTC_FREQ_DIV1024,
                                               RTC_FREQ_DIV512, RTC_FREQ_DIV256
                  tamper_precharge_enable: DISABLE, ENABLE
                  tamper_precharge_time: RTC_PRCH_1C, RTC_PRCH_2C, RTC_PRCH_4C, RTC_PRCH_8C
                  tamper_with_timestamp: RTC_TAMPER_WITH_TIMESTAMP_DISABLE, RTC_TAMPER_WITH_TIMESTAMP_ENABLE
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_tamper_config(hal_rtc_dev_struct *rtc_dev, hal_rtc_tamper_config_struct *rtc_tamper_config)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == rtc_dev) || (NULL == rtc_tamper_config)) {
        HAL_DEBUGE("pointer [rtc_dev] or [rtc_tamper_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);

    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* reset tamper source*/
    /* tamper filter must be used when the tamper source is voltage level detection */
    RTC_TAMP &= ~(uint32_t)(RTC_TAMP_DISPU | RTC_TAMP_PRCH | RTC_TAMP_FREQ | RTC_TAMP_FLT | RTC_TAMP_TPTS
                            | RTC_TAMP_TP0EG | RTC_TAMP_TP1EG | RTC_TAMPER0 | RTC_TAMPER1);
    RTC_TAMP |= (uint32_t)(rtc_tamper_config->tamper_filter | rtc_tamper_config->tamper_sample_frequency
                           | rtc_tamper_config->tamper_precharge_enable | rtc_tamper_config->tamper_precharge_time
                           | rtc_tamper_config->tamper_with_timestamp | rtc_tamper_config->tamper_tamper1_trigger
                           | rtc_tamper_config->tamper_tamper0_trigger);

    /* enable tamper*/
    RTC_TAMP |= (uint32_t)(rtc_tamper_config->tamper_source);
    /* change RTC error state */
    rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    /* change RTC state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC tamper interrupt mode
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtc_tamper_config: pointer to a rtc_tamper_struct structure which contains
                parameters for RTC tamper configuration
                members of the structure and the member values are shown as below:
                  detecting tamper event can using edge mode or level mode
                  (1) using edge mode configuration:
                  tamper_source: RTC_TAMPER0, RTC_TAMPER1
                  tamper_tamper0_trigger: RTC_TAMPER0_TRIGGER_EDGE_RISING_LOW, RTC_TAMPER0_TRIGGER_EDGE_FALLING_HIGH
                  tamper_tamper1_trigger: RTC_TAMPER1_TRIGGER_EDGE_RISING_LOW, RTC_TAMPER1_TRIGGER_EDGE_FALLING_HIGH
                  tamper_filter: RTC_FLT_EDGE
                  tamper_with_timestamp: RTC_TAMPER_WITH_TIMESTAMP_DISABLE, RTC_TAMPER_WITH_TIMESTAMP_ENABLE
                  (2) using level mode configuration:
                  tamper_source: RTC_TAMPER0, RTC_TAMPER1
                  tamper_trigger:RTC_TAMPER_TRIGGER_LEVEL_LOW, RTC_TAMPER_TRIGGER_LEVEL_HIGH
                  tamper_filter: RTC_FLT_2S, RTC_FLT_4S, RTC_FLT_8S
                  tamper_sample_frequency: RTC_FREQ_DIV32768, RTC_FREQ_DIV16384, RTC_FREQ_DIV8192,
                                               RTC_FREQ_DIV4096, RTC_FREQ_DIV2048, RTC_FREQ_DIV1024,
                                               RTC_FREQ_DIV512, RTC_FREQ_DIV256
                  tamper_precharge_enable: DISABLE, ENABLE
                  tamper_precharge_time: RTC_PRCH_1C, RTC_PRCH_2C, RTC_PRCH_4C, RTC_PRCH_8C
                  tamper_with_timestamp: RTC_TAMPER_WITH_TIMESTAMP_DISABLE, RTC_TAMPER_WITH_TIMESTAMP_ENABLE
    \param[in]  p_user_func: RTC irq user callback function pointer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_tamper_config_interrupt(hal_rtc_dev_struct *rtc_dev, hal_rtc_tamper_config_struct *rtc_tamper_config,
                                        hal_rtc_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == rtc_dev) || (NULL == rtc_tamper_config)) {
        HAL_DEBUGE("pointer [rtc_dev] or [rtc_tamper_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);

    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* reset tamper source*/
    /* tamper filter must be used when the tamper source is voltage level detection */
    RTC_TAMP &= ~(uint32_t)(RTC_TAMP_DISPU | RTC_TAMP_PRCH | RTC_TAMP_FREQ | RTC_TAMP_FLT | RTC_TAMP_TPTS
                            | RTC_TAMP_TP0EG | RTC_TAMP_TP1EG | RTC_TAMPER0 | RTC_TAMPER1);

    RTC_TAMP |= (uint32_t)(rtc_tamper_config->tamper_filter | rtc_tamper_config->tamper_sample_frequency
                           | rtc_tamper_config->tamper_precharge_enable | rtc_tamper_config->tamper_precharge_time
                           | rtc_tamper_config->tamper_with_timestamp | rtc_tamper_config->tamper_tamper1_trigger
                           | rtc_tamper_config->tamper_tamper0_trigger);

    if(RTC_TAMPER0 == (rtc_tamper_config->tamper_source & RTC_TAMPER0)) {
        hals_rtc_flag_clear(RTC_FLAG_TP0);
        hals_rtc_interrupt_disable(RTC_INT_TAMP_ALL);
        rtc_dev->tamper0_callback           = (void *)p_user_func->tamper0_func;
        rtc_dev->rtc_irq.rtc_tamper0_handle = _rtc_tamper0_handler;
    } else {
        /* do nothing */
    }

    if(RTC_TAMPER1 == (rtc_tamper_config->tamper_source & RTC_TAMPER1)) {
        hals_rtc_flag_clear(RTC_FLAG_TP1);
        hals_rtc_interrupt_disable(RTC_INT_TAMP_ALL);
        rtc_dev->tamper1_callback           = (void *)p_user_func->tamper1_func;
        rtc_dev->rtc_irq.rtc_tamper1_handle = _rtc_tamper1_handler;
    } else {
        /* do nothing */
    }

    /* enable tamper detection interrupt */
    hals_rtc_interrupt_enable(RTC_INT_TAMP_ALL);
    hal_exti_internal_init(EXTI_LINE_18_RTC_TAMPER_TIMESTAMP, EXTI_INTERRUPT_TRIG_RISING);

    /* enable tamper*/
    RTC_TAMP |= (uint32_t)(rtc_tamper_config->tamper_source);
    /* change RTC error state */
    rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    /* change RTC state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC smooth calibration
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtc_calib_config:
                calib_cycle: select calibration window
      \arg        RTC_CALIBRATION_WINDOW_32S: 2exp20 RTCCLK cycles, 32s if RTCCLK = 32768 Hz
      \arg        RTC_CALIBRATION_WINDOW_16S: 2exp19 RTCCLK cycles, 16s if RTCCLK = 32768 Hz
      \arg        RTC_CALIBRATION_WINDOW_8S: 2exp18 RTCCLK cycles, 8s if RTCCLK = 32768 Hz
                calib_frequency: add RTC clock or not
      \arg        RTC_CALIBRATION_PLUS_SET: add one RTC clock every 2048 RTC clock
      \arg        RTC_CALIBRATION_PLUS_RESET: no effect
                calib_shielding_number: the RTC clock to minus during the calibration window(0x0 - 0x1FF)
                calib_output: RTC output selection
                  RTC_CALIBRATION_512HZ: calibration output of 512Hz is enable
                  RTC_CALIBRATION_1HZ: calibration output of 1Hz is enable
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE,HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_calib_config(hal_rtc_dev_struct *rtc_dev, hal_rtc_calib_config_struct *rtc_calib_config)
{
    int32_t error_code           = HAL_ERR_NONE;
    volatile uint32_t time_index = RTC_HRFC_TIMEOUT;
    ErrStatus error_status       = ERROR;
    uint32_t flag_status         = RESET;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == rtc_dev) || (NULL == rtc_calib_config)) {
        HAL_DEBUGE("pointer [rtc_dev] or [rtc_calib_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */


    /* lock rtc */
    HAL_LOCK(rtc_dev);

    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* check if a smooth calibration operation is ongoing */
    do {
        flag_status = RTC_STAT & RTC_STAT_SCPF;
    } while(((--time_index) > 0U) && ((uint32_t)RESET != flag_status));

    if((uint32_t)RESET == flag_status) {
        RTC_HRFC     = (uint32_t)(rtc_calib_config->calib_cycle | rtc_calib_config->calib_frequency
                              | HRFC_CMSK(rtc_calib_config->calib_shielding_number));
        error_status = SUCCESS;
    } else {
        /*do nothing*/
    }

    RTC_CTL &= (uint32_t)~(RTC_CTL_COEN | RTC_CTL_COS);
    RTC_CTL |= (uint32_t)(rtc_calib_config->calib_output);

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    if(ERROR == error_status) {
        HAL_DEBUGE(" wait SCPF reset timeout");
        error_code = HAL_ERR_BUSY;
    } else {
        /* change RTC error state */
        rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    }

    /* change state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC reference clock detection function
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  refclock_detection:Disable or Enable reference clock detection
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE,HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_refclock_detection_config(hal_rtc_dev_struct *rtc_dev, uint32_t refclock_detection)
{
    int32_t error_code     = HAL_ERR_NONE;
    ErrStatus error_status = ERROR;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);
    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* enter init mode */
    error_status = hals_rtc_init_mode_enter();

    if(ERROR != error_status) {
        /* reset reference clock detection */
        RTC_CTL &= ~(uint32_t)RTC_CTL_REFEN;
        /* set reference clock detection */
        RTC_CTL |= (uint32_t)refclock_detection;
        /* exit init mode */
        hals_rtc_init_mode_exit();
    } else {
        /* do nothing */
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    if(ERROR != error_status) {
        /* change RTC error state */
        rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    } else {
        /* change RTC error state */
        rtc_dev->error_state = HAL_RTC_ERROR_SYSTEM;
        error_code = HAL_ERR_HARDWARE;
    }

    /* change RTC state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC bypass shadow registers function
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  shadow_disable:Disable or Enable bypass shadow registers
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_shadow_config(hal_rtc_dev_struct *rtc_dev, uint32_t shadow_disable)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);

    rtc_dev->state     = HAL_RTC_STATE_BUSY;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL &= ~(uint32_t)RTC_CTL_BPSHAD;
    RTC_CTL |= (uint32_t)shadow_disable;

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* change state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC time-stamp internal event
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtc_timestamp_internal: specify which internal or external event to be detected
                only one parameter can be selected which is shown as below:
      \arg        RTC_ITSEN_DISABLE: disable RTC time-stamp internal event
      \arg        RTC_ITSEN_ENABLE: enable RTC time-stamp internal event
    \param[in]  rtc_timestamp_edge: specify which edge to detect of time-stamp
                only one parameter can be selected which is shown as below:
      \arg        RTC_TIMESTAMP_RISING_EDGE: rising edge is valid event edge for timestamp event
      \arg        RTC_TIMESTAMP_FALLING_EDGE: falling edge is valid event edge for timestamp event
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_timestamp_config(hal_rtc_dev_struct *rtc_dev, uint32_t rtc_timestamp_internal, uint32_t rtc_timestamp_edge)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t reg_ctl   = 0x00000000U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);
    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* set RTC time-stamp internal event */
    if(RTC_ITSEN_ENABLE == rtc_timestamp_internal) {
        /* set RTC_CTL ITSEN */
        RTC_CTL |= RTC_CTL_ITSEN;
        /* Enable bypass shadow registers */
        reg_ctl = (uint32_t)(RTC_CTL & (uint32_t)(~(RTC_CTL_TSEG | RTC_CTL_TSEN)));

        /* new configuration */
        reg_ctl |= (uint32_t)(rtc_timestamp_edge | RTC_CTL_TSEN);

        RTC_CTL = (uint32_t)reg_ctl;
    } else {
        /* reset RTC_CTL ITSEN */
        RTC_CTL &= ~(uint32_t)RTC_CTL_ITSEN;
        /* Disable bypass shadow registers */
        RTC_CTL &= ~(uint32_t)(RTC_CTL_TSEG | RTC_CTL_TSEN);
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* change RTC error state */
    rtc_dev->error_state = HAL_RTC_ERROR_NONE;

    /* change state */
    rtc_dev->state = HAL_RTC_STATE_READY;
    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC time-stamp internal event interrupt mode
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtc_timestamp_internal: specify which internal or external event to be detected
                only one parameter can be selected which is shown as below:
      \arg        RTC_ITSEN_DISABLE: disable RTC time-stamp internal event
      \arg        RTC_ITSEN_ENABLE: enable RTC time-stamp internal event
    \param[in]  rtc_timestamp_edge: specify which edge to detect of time-stamp
                only one parameter can be selected which is shown as below:
      \arg        RTC_TIMESTAMP_RISING_EDGE: rising edge is valid event edge for timestamp event
      \arg        RTC_TIMESTAMP_FALLING_EDGE: falling edge is valid event edge for timestamp event
    \param[in]  p_user_func: pointer to a hal_rtc_user_callback_struct structure that contains
                  the callback function pointer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_timestamp_config_interrupt(hal_rtc_dev_struct *rtc_dev, uint32_t rtc_timestamp_internal,
                                           uint32_t rtc_timestamp_edge, hal_rtc_user_callback_struct *p_user_func)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t reg_ctl   = 0x00000000U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] or [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);
    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* set RTC time-stamp internal event */
    if(RTC_ITSEN_ENABLE == rtc_timestamp_internal) {
        /* set RTC_CTL ITSEN */
        RTC_CTL |= RTC_CTL_ITSEN;
        /* Enable bypass shadow registers */
        reg_ctl = (uint32_t)(RTC_CTL & (uint32_t)(~(RTC_CTL_TSEG | RTC_CTL_TSEN)));

        /* new configuration */
        reg_ctl |= (uint32_t)(rtc_timestamp_edge | RTC_CTL_TSEN);

        RTC_CTL = (uint32_t)reg_ctl;
    } else {
        /* reset RTC_CTL ITSEN */
        RTC_CTL &= ~(uint32_t)RTC_CTL_ITSEN;
        /* Disable bypass shadow registers */
        RTC_CTL &= ~(uint32_t)(RTC_CTL_TSEG | RTC_CTL_TSEN);
    }

    /* set timestamp handler */
    rtc_dev->rtc_irq.rtc_timestamp_handle = _rtc_timestamp_handler;

    /* set timestamp callback function */
    if(NULL != p_user_func) {
        rtc_dev->timestamp_callback = (void *)p_user_func->timestamp_func;
    } else {
        /* do nothing */
    }

    /* enable time-stamp interrupt */
    hals_rtc_flag_clear(RTC_FLAG_TS);
    hals_rtc_flag_clear(RTC_FLAG_TSOVR);
    hals_rtc_interrupt_enable(RTC_INT_TIMESTAMP);
    hal_exti_internal_init(EXTI_LINE_18_RTC_TAMPER_TIMESTAMP, EXTI_INTERRUPT_TRIG_RISING);

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* change RTC error state */
    rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    /* change state */
    rtc_dev->state = HAL_RTC_STATE_READY;

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      set RTC auto wakeup timer clock and wakeup timer value
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  wakeup_clk:
                wakeup_clock:
      \arg        WAKEUP_RTCCK_DIV16: RTC auto wakeup timer clock is RTC clock divided by 16
      \arg        WAKEUP_RTCCK_DIV8: RTC auto wakeup timer clock is RTC clock divided by 8
      \arg        WAKEUP_RTCCK_DIV4: RTC auto wakeup timer clock is RTC clock divided by 4
      \arg        WAKEUP_RTCCK_DIV2: RTC auto wakeup timer clock is RTC clock divided by 2
      \arg        WAKEUP_CKSPRE: RTC auto wakeup timer clock is ckspre
      \arg        WAKEUP_CKSPRE_2EXP16: RTC auto wakeup timer clock is ckspre and wakeup timer add 2exp16
                wakeup_timer: 0x0000-0xffff
    \param[out] none
    \retval     error_code: HAL_ERR_ADDRESS,HAL_ERR_TIMEOUT,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_wake_up_config(hal_rtc_dev_struct *rtc_dev, hal_rtc_wakeup_clk_struct *wakeup_clk)
{
    int32_t error_code           = HAL_ERR_NONE;
    volatile uint32_t time_index = RTC_WTWF_TIMEOUT;
    uint32_t flag_status         = RESET;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);

    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* set auto wake up timer clock */
    /* only when RTC_CTL_WTEN=0 and RTC_STAT_WTWF=1 can write RTC_CTL[2:0] */
    /* wait until the WTWF flag to be set */
    do {
        flag_status = RTC_STAT & RTC_STAT_WTWF;
    } while((((--time_index)) > 0U) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_code = HAL_ERR_TIMEOUT;
    } else {
        RTC_CTL &= (uint32_t)~RTC_CTL_WTCS;
        RTC_CTL |= (uint32_t)wakeup_clk->wakeup_clock;
        /* wait until the WTWF flag to be set */
        RTC_WUT = (uint32_t)wakeup_clk->wakeup_timer - 1U;
    }

    if(HAL_ERR_NONE == error_code) {
        /* enable the write protection */
        RTC_WPK = RTC_LOCK_KEY;
        /* change RTC error state */
        rtc_dev->error_state = HAL_RTC_ERROR_NONE;
    } else {
        /* do nothing */
    }

    /* change state */
    rtc_dev->state = HAL_RTC_STATE_READY;
    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      set RTC auto wakeup timer clock and wakeup timer value and enable wakeup interrupt
    \param[in]  rtc_dev: RTCS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  wakeup_clk:
                wakeup_clock:
      \arg        WAKEUP_RTCCK_DIV16: RTC auto wakeup timer clock is RTC clock divided by 16
      \arg        WAKEUP_RTCCK_DIV8: RTC auto wakeup timer clock is RTC clock divided by 8
      \arg        WAKEUP_RTCCK_DIV4: RTC auto wakeup timer clock is RTC clock divided by 4
      \arg        WAKEUP_RTCCK_DIV2: RTC auto wakeup timer clock is RTC clock divided by 2
      \arg        WAKEUP_CKSPRE: RTC auto wakeup timer clock is ckspre
      \arg        WAKEUP_CKSPRE_2EXP16: RTC auto wakeup timer clock is ckspre and wakeup timer add 2exp16
                wakeup_timer: 0x0000-0xffff
    \param[in]  p_user_func: user callback function
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_TIMEOUT,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_wake_up_config_interrupt(hal_rtc_dev_struct *rtc_dev, hal_rtc_wakeup_clk_struct *wakeup_clk,
                                         hal_rtc_user_callback_struct *p_user_func)
{
    int32_t error_code           = HAL_ERR_NONE;
    volatile uint32_t time_index = RTC_WTWF_TIMEOUT;
    uint32_t flag_status         = RESET;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);
    rtc_dev->state = HAL_RTC_STATE_BUSY;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* set auto wake up timer clock */
    /* only when RTC_CTL_WTEN=0 and RTC_STAT_WTWF=1 can write RTC_CTL[2:0] */
    /* wait until the WTWF flag to be set */
    do {
        flag_status = RTC_STAT & RTC_STAT_WTWF;
    } while((((--time_index)) > 0U) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_code = HAL_ERR_TIMEOUT;
    } else {
        RTC_CTL &= (uint32_t)~RTC_CTL_WTCS;
        RTC_CTL |= (uint32_t)wakeup_clk->wakeup_clock;
        /* wait until the WTWF flag to be set */
        RTC_WUT = (uint32_t)wakeup_clk->wakeup_timer - 1U;
    }

    rtc_dev->rtc_irq.rtc_wakeup_handle = _rtc_wakeup_handler;

    /* set callback function */
    if(NULL != p_user_func) {
        rtc_dev->wakeup_callback = (void *)p_user_func->wakeup_func;
    } else {
        /* do nothing */
    }

    hals_rtc_flag_clear(RTC_FLAG_WT);
    /* enable wake up interrupt */
    hals_rtc_interrupt_enable(RTC_INT_WAKEUP);
    hal_exti_internal_init(EXTI_LINE_19_RTC_WAKEUP, EXTI_INTERRUPT_TRIG_RISING);

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* change RTC error state */
    rtc_dev->error_state = HAL_RTC_ERROR_NONE;

    /* change state */
    rtc_dev->state = HAL_RTC_STATE_READY;
    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      select the RTC output pin
    \param[in]  outputpin: specify the RTC output pin is PC13 or PB2
      \arg        RTC_OUT_PC13: the RTC output pin is PC13
      \arg        RTC_OUT_PB2: the RTC output pin is PB2
    \param[out] none
    \retval     none
*/
void hal_rtc_output_pin_select(uint32_t outputpin)
{
    ErrStatus error_status = ERROR;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* enter init mode */
    error_status = hals_rtc_init_mode_enter();

    if(ERROR != error_status) {
        RTC_CFG &= (uint32_t)(~RTC_CFG_OUT2EN);
        RTC_CFG |= (uint32_t)(outputpin);
        /* exit init mode */
        hals_rtc_init_mode_exit();
    } else {
        /* do nothing */
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      RTC interrupt handler content function,which is merely used in rtc_handler
    \param[in]  rtc_dev: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE
*/
int32_t hal_rtc_irq(hal_rtc_dev_struct *rtc_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check whether the alarm0 event flag set or not set */
    if(SET == (hals_rtc_flag_get(RTC_FLAG_ALARM0))) {
        /* clear the alarm0 event flag */
        hals_rtc_flag_clear(RTC_FLAG_ALARM0);
        /* alarm0 event handle */
        if(NULL != rtc_dev->rtc_irq.rtc_alarm0_handle) {
            rtc_dev->rtc_irq.rtc_alarm0_handle(rtc_dev);
        } else {
            /* do nothing */
        }
        hals_exti_flag_clear(EXTI_17);
    } else {
        /* do nothing */
    }

    /* check whether the alarm1 event flag set or not set */
    if(SET == (hals_rtc_flag_get(RTC_FLAG_ALARM1))) {
        /* clear the alarm1 event flag */
        hals_rtc_flag_clear(RTC_FLAG_ALARM1);
        /* alarm1 event handle */
        if(NULL != rtc_dev->rtc_irq.rtc_alarm1_handle) {
            rtc_dev->rtc_irq.rtc_alarm1_handle(rtc_dev);
        } else {
            /* do nothing */
        }
        hals_exti_flag_clear(EXTI_17);
    } else {
        /* do nothing */
    }

    /* check whether the wake up flag set or not set */
    if(SET == (hals_rtc_flag_get(RTC_FLAG_WT))) {
        /* clear the wake up flag */
        hals_rtc_flag_clear(RTC_FLAG_WT);
        /* wake up handle */
        if(NULL != rtc_dev->rtc_irq.rtc_wakeup_handle) {
            rtc_dev->rtc_irq.rtc_wakeup_handle(rtc_dev);
        } else {
            /* do nothing */
        }
        /* clear exti wekeup irq */
        hals_exti_flag_clear(EXTI_19);
    } else {
        /* do nothing */
    }

    /* check whether the tamper 0 detected flag set or not set */
    if(SET == (hals_rtc_flag_get(RTC_FLAG_TP0))) {
        /* clear the tamper 0 detected flag */
        hals_rtc_flag_clear(RTC_FLAG_TP0);
        /* tamper 0 detected handle */
        if(NULL != rtc_dev->rtc_irq.rtc_tamper0_handle) {
            rtc_dev->rtc_irq.rtc_tamper0_handle(rtc_dev);
        } else {
            /* do nothing */
        }
        hals_exti_flag_clear(EXTI_18);
    } else {
        /* do nothing */
    }

    /* check whether the tamper 1 detected flag set or not set */
    if(SET == (hals_rtc_flag_get(RTC_FLAG_TP1))) {
        /* clear the tamper 1 detected flag */
        hals_rtc_flag_clear(RTC_FLAG_TP1);
        /* tamper 1 detected handle */
        if(NULL != rtc_dev->rtc_irq.rtc_tamper1_handle) {
            rtc_dev->rtc_irq.rtc_tamper1_handle(rtc_dev);
        } else {
            /* do nothing */
        }
        hals_exti_flag_clear(EXTI_18);
    } else {
        /* do nothing */
    }

    /* check whether the time-stamp flag set or not set */
    if(SET == (hals_rtc_flag_get(RTC_FLAG_TS))) {
        /* time-stamp handle */
        if(NULL != rtc_dev->rtc_irq.rtc_timestamp_handle) {
            rtc_dev->rtc_irq.rtc_timestamp_handle(rtc_dev);
        } else {
            /* do nothing */
        }
        /* clear the time-stamp flag */
        hals_rtc_flag_clear(RTC_FLAG_TS);
        hals_exti_flag_clear(EXTI_18);
    } else {
        /* do nothing */
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      set user-defined interrupt callback function which will be registered and called when corresponding
   interrupt be triggered
    \param[in]  rtc_dev: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to RTC interrupt callback functions structure
                  please refer to hal_rtc_irq_struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_irq_handle_set(hal_rtc_dev_struct *rtc_dev, hal_rtc_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(NULL == p_irq) {
        HAL_DEBUGE("pointer [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* RTC alarm0 event handle set */
    if(NULL != p_irq->rtc_alarm0_handle) {
        rtc_dev->rtc_irq.rtc_alarm0_handle = p_irq->rtc_alarm0_handle;
        /* enable alarm0 interrupt */
        hals_rtc_flag_clear(RTC_FLAG_ALARM0);
        hals_rtc_interrupt_enable(RTC_INT_ALARM0);
        /* enable RTC alarm */
        hal_exti_internal_init(EXTI_LINE_17_RTC_ALARM, EXTI_INTERRUPT_TRIG_RISING);
        hals_rtc_alarm_enable(RTC_ALARM0);
    } else {
        rtc_dev->rtc_irq.rtc_alarm0_handle = NULL;
        /* disable alarm0 interrupt */
        hals_rtc_interrupt_disable(RTC_INT_ALARM0);
        /* disable RTC alarm */
        hal_rtc_alarm_disable(RTC_ALARM0);
        EXTI_PD0 = (uint32_t)EXTI_LINE_17_RTC_ALARM;
    }

    /* RTC alarm1 handle set */
    if(NULL != p_irq->rtc_alarm1_handle) {
        rtc_dev->rtc_irq.rtc_alarm1_handle = p_irq->rtc_alarm1_handle;
        hals_rtc_flag_clear(RTC_FLAG_ALARM1);
        hals_rtc_interrupt_enable(RTC_FLAG_ALARM1);
        /* enable RTC alarm */
        hal_exti_internal_init(EXTI_LINE_17_RTC_ALARM, EXTI_INTERRUPT_TRIG_RISING);
        hals_rtc_alarm_enable(RTC_ALARM1);
    } else {
        rtc_dev->rtc_irq.rtc_alarm1_handle = NULL;
        /* disable alarm1 interrupt */
        hals_rtc_interrupt_disable(RTC_INT_ALARM1);
        /* disable RTC alarm */
        hal_rtc_alarm_disable(RTC_ALARM1);
        EXTI_PD0 = (uint32_t)EXTI_LINE_17_RTC_ALARM;
    }

    /* RTC wake up handle set */
    if(NULL != p_irq->rtc_wakeup_handle) {
        rtc_dev->rtc_irq.rtc_wakeup_handle = p_irq->rtc_wakeup_handle;
        hals_rtc_flag_clear(RTC_FLAG_WT);
        /* enable wake up interrupt */
        hals_rtc_interrupt_enable(RTC_INT_WAKEUP);
        hal_exti_internal_init(EXTI_LINE_19_RTC_WAKEUP, EXTI_INTERRUPT_TRIG_RISING);
    } else {
        rtc_dev->rtc_irq.rtc_wakeup_handle = NULL;
        /* disable wake up interrupt */
        hals_rtc_interrupt_disable(RTC_INT_WAKEUP);
        EXTI_PD0 = (uint32_t)EXTI_LINE_19_RTC_WAKEUP;
    }

    /* RTC time-stamp handle set */
    if(NULL != p_irq->rtc_timestamp_handle) {
        rtc_dev->rtc_irq.rtc_timestamp_handle = p_irq->rtc_timestamp_handle;
        /* enable time-stamp interrupt */
        hals_rtc_flag_clear(RTC_FLAG_TS);
        hals_rtc_flag_clear(RTC_FLAG_TSOVR);
        hals_rtc_interrupt_enable(RTC_INT_TIMESTAMP);
        hal_exti_internal_init(EXTI_LINE_18_RTC_TAMPER_TIMESTAMP, EXTI_INTERRUPT_TRIG_RISING);
    } else {
        rtc_dev->rtc_irq.rtc_timestamp_handle = NULL;
        /* disable time-stamp interrupt */
        hals_rtc_interrupt_disable(RTC_INT_TIMESTAMP);
        EXTI_PD0 = (uint32_t)EXTI_LINE_18_RTC_TAMPER_TIMESTAMP;
    }

    /* RTC tamper detection handle set */
    if((NULL != p_irq->rtc_tamper0_handle) || (NULL != p_irq->rtc_tamper1_handle)) {
        if(NULL == p_irq->rtc_tamper0_handle) {
            rtc_dev->rtc_irq.rtc_tamper0_handle = NULL;
            rtc_dev->rtc_irq.rtc_tamper1_handle = p_irq->rtc_tamper1_handle;
            hal_rtc_tamper_disable(RTC_TAMPER0);
            hal_rtc_tamper_enable(RTC_TAMPER1);
        } else if(NULL == p_irq->rtc_tamper1_handle) {
            rtc_dev->rtc_irq.rtc_tamper1_handle = NULL;
            rtc_dev->rtc_irq.rtc_tamper0_handle = p_irq->rtc_tamper0_handle;
            hal_rtc_tamper_disable(RTC_TAMPER1);
            hal_rtc_tamper_enable(RTC_TAMPER0);
        } else {
            rtc_dev->rtc_irq.rtc_tamper0_handle = p_irq->rtc_tamper0_handle;
            rtc_dev->rtc_irq.rtc_tamper1_handle = p_irq->rtc_tamper1_handle;
            hal_rtc_tamper_enable(RTC_TAMPER0);
            hal_rtc_tamper_enable(RTC_TAMPER1);
        }
        hals_rtc_flag_clear(RTC_FLAG_TP0);
        hals_rtc_flag_clear(RTC_FLAG_TP1);
        /* enable tamper detection interrupt */
        hals_rtc_interrupt_enable(RTC_INT_TAMP_ALL);
        hal_exti_internal_init(EXTI_LINE_18_RTC_TAMPER_TIMESTAMP, EXTI_INTERRUPT_TRIG_RISING);
    } else {
        rtc_dev->rtc_irq.rtc_tamper0_handle = NULL;
        rtc_dev->rtc_irq.rtc_tamper1_handle = NULL;
        /* disable tamper detection interrupt */
        hals_rtc_interrupt_disable(RTC_INT_TAMP_ALL);
        hal_rtc_tamper_disable(RTC_TAMPER1);
        hal_rtc_tamper_enable(RTC_TAMPER0);
        EXTI_PD0 = (uint32_t)EXTI_LINE_18_RTC_TAMPER_TIMESTAMP;
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      reset all user-defined interrupt callback function, which will be registered and called when
                corresponding interrupt be triggered which will be registered and called when corresponding interrupt be triggered
    \param[in]  rtc_dev: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_irq_handle_all_reset(hal_rtc_dev_struct *rtc_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_dev) {
        HAL_DEBUGE("pointer [rtc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* RTC interrupt handle reset */
    rtc_dev->rtc_irq.rtc_timestamp_handle = NULL;
    rtc_dev->rtc_irq.rtc_alarm0_handle    = NULL;
    rtc_dev->rtc_irq.rtc_alarm1_handle    = NULL;
    rtc_dev->rtc_irq.rtc_tamper0_handle   = NULL;
    rtc_dev->rtc_irq.rtc_tamper1_handle   = NULL;
    rtc_dev->rtc_irq.rtc_wakeup_handle    = NULL;

    /* return error code */
    return error_code;
}

/*!
    \brief      reset rtc alternate output source
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rtc_reset_output(void)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL &= (uint32_t)~(RTC_CTL_COEN | RTC_CTL_OS | RTC_CTL_OPOL | RTC_CTL_COS);

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      get current time and date
    \param[in]  none
    \param[out] rtc_calendar: pointer to a hal_rtc_init_struct structure which contains
                parameters for initialization of the RTC peripheral
                members of the structure and the member values are shown as below:
                  year: 0x0 - 0x99(BCD format)
                  month: RTC_JAN, RTC_FEB, RTC_MAR, RTC_APR, RTC_MAY, RTC_JUN,
                         RTC_JUL, RTC_AUG, RTC_SEP, RTC_OCT, RTC_NOV, RTC_DEC
                  date: 0x1 - 0x31(BCD format)
                  day_of_week: RTC_MONDAY, RTC_TUESDAY, RTC_WEDNESDAY, RTC_THURSDAY
                               RTC_FRIDAY, RTC_SATURDAY, RTC_SUNDAY
                  hour: 0x0 - 0x12(BCD format) or 0x0 - 0x23(BCD format) depending on the RTC display_format chose
                  minute: 0x0 - 0x59(BCD format)
                  second: 0x0 - 0x59(BCD format)
                  factor_asyn: 0x0 - 0x7F
                  factor_syn: 0x0 - 0x7FFF
                  am_pm: RTC_AM, RTC_PM
                  display_format: RTC_24HOUR, RTC_12HOUR
    \param[out] rtc_subsecond: rtc subsecond value
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_current_time_get(hal_rtc_init_struct *rtc_calendar, uint32_t *rtc_subsecond)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t temp_tr   = 0U;
    uint32_t temp_dr   = 0U;
    uint32_t temp_pscr = 0U;
    uint32_t temp_ctlr = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_calendar) {
        HAL_DEBUGE("pointer [rtc_calendar] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* if BPSHAD bit is reset, reading RTC_SS will lock RTC_TIME and RTC_DATE automatically */
    temp_tr   = (uint32_t)RTC_TIME;
    temp_dr   = (uint32_t)RTC_DATE;
    temp_pscr = (uint32_t)RTC_PSC;
    temp_ctlr = (uint32_t)RTC_CTL;
    /* read RTC_DATE to unlock the 3 shadow registers */
    (void)(RTC_DATE);

    /* get current time and construct hal_rtc_init_struct structure */
    rtc_calendar->year            = hals_rtc_bcd_2_normal((uint8_t)GET_DATE_YR(temp_dr));
    rtc_calendar->month           = hals_rtc_bcd_2_normal((uint8_t)GET_DATE_MON(temp_dr));
    rtc_calendar->date            = hals_rtc_bcd_2_normal((uint8_t)GET_DATE_DAY(temp_dr));
    rtc_calendar->day_of_week     = hals_rtc_bcd_2_normal((uint8_t)GET_DATE_DOW(temp_dr));
    rtc_calendar->hour            = hals_rtc_bcd_2_normal((uint8_t)GET_TIME_HR(temp_tr));
    rtc_calendar->minute          = hals_rtc_bcd_2_normal((uint8_t)GET_TIME_MN(temp_tr));
    rtc_calendar->second          = hals_rtc_bcd_2_normal((uint8_t)GET_TIME_SC(temp_tr));
    rtc_calendar->daylight_saving = (uint32_t)(temp_ctlr & (RTC_CTL_A1H | RTC_CTL_S1H));
    rtc_calendar->factor_asyn     = (uint16_t)GET_PSC_FACTOR_A(temp_pscr);
    rtc_calendar->factor_syn      = (uint16_t)GET_PSC_FACTOR_S(temp_pscr);
    rtc_calendar->am_pm           = (uint32_t)(temp_tr & RTC_TIME_PM);
    rtc_calendar->display_format  = (uint32_t)(temp_ctlr & RTC_CTL_CS);

    *rtc_subsecond = hals_rtc_subsecond_get();

    /* return error code */
    return error_code;
}

/*!
    \brief      get RTC alarm
    \param[in]  rtc_alarm: RTC_ALARM0 or RTC_ALARM1
    \param[out] rtc_alarm_time: pointer to a hal_rtc_alarm_config_struct structure which contains
                parameters for RTC alarm configuration
                members of the structure and the member values are shown as below:
                  alarm_mask: RTC_ALARM_NONE_MASK, RTC_ALARM_DATE_MASK, RTC_ALARM_HOUR_MASK
                              RTC_ALARM_MINUTE_MASK, RTC_ALARM_SECOND_MASK, RTC_ALARM_ALL_MASK
                  weekday_or_date: RTC_ALARM_DATE_SELECTED, RTC_ALARM_WEEKDAY_SELECTED
                  alarm_day: 1) 0x1 - 0x31(BCD format) if RTC_ALARM_DATE_SELECTED is set
                                 2) RTC_MONDAY, RTC_TUESDAY, RTC_WEDNESDAY, RTC_THURSDAY, RTC_FRIDAY,
                                    RTC_SATURDAY, RTC_SUNDAY if RTC_ALARM_WEEKDAY_SELECTED is set
                  alarm_hour: 0x0 - 0x12(BCD format) or 0x0 - 0x23(BCD format) depending on the RTC display_format
                  alarm_minute: 0x0 - 0x59(BCD format)
                  alarm_second: 0x0 - 0x59(BCD format)
                  am_pm: RTC_AM, RTC_PM
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_alarm_get(uint8_t rtc_alarm, hal_rtc_alarm_config_struct *rtc_alarm_time)
{
    int32_t error_code  = HAL_ERR_NONE;
    uint32_t reg_alrmtd = 0U;
    uint32_t reg_alrmss = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == rtc_alarm_time) {
        HAL_DEBUGE("pointer [rtc_alarm_time] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get the value of RTC_ALRM0TD register */
    if(RTC_ALARM0 == rtc_alarm) {
        reg_alrmtd = RTC_ALRM0TD;
    } else if(RTC_ALARM1 == rtc_alarm) {
        reg_alrmtd = RTC_ALRM1TD;
    } else {
        HAL_DEBUGE("parameter [rtc_alarm] value is invalid");
        error_code = HAL_ERR_VAL;
    }

    if(HAL_ERR_NONE == error_code) {
        /* get the value of RTC_ALRM0TD and RTC_ALRM0SS register */
        if(RTC_ALARM0 == rtc_alarm_time->alarmx) {
            rtc_alarm_time->alarm_subsecond      = (uint32_t)(reg_alrmss & RTC_ALRM0SS_SSC);
            rtc_alarm_time->alarm_subsecond_mask = reg_alrmss & RTC_ALRM0SS_MASKSSC;
        } else {
            rtc_alarm_time->alarm_subsecond      = (uint32_t)(reg_alrmss & RTC_ALRM1SS_SSC);
            rtc_alarm_time->alarm_subsecond_mask = reg_alrmss & RTC_ALRM1SS_MASKSSC;
        }

        /* get alarm parameters and construct the rtc_alarm_struct structure */
        rtc_alarm_time->alarm_am_pm           = (uint32_t)(reg_alrmtd & RTC_ALRMXTD_PM);
        rtc_alarm_time->alarm_weekday_mask    = (uint32_t)(reg_alrmtd & RTC_ALARM_WEEKDAY_MASK_ENABLE);
        rtc_alarm_time->alarm_hour_mask       = reg_alrmtd & RTC_ALARM_HOUR_MASK_ENABLE;
        rtc_alarm_time->alarm_minute_mask     = reg_alrmtd & RTC_ALARM_MINUTE_MASK_ENABLE;
        rtc_alarm_time->alarm_second_mask     = reg_alrmtd & RTC_ALARM_SECOND_MASK_ENABLE;
        rtc_alarm_time->alarm_weekday_or_date = (uint32_t)(reg_alrmtd & RTC_ALRMXTD_DOWS);
        rtc_alarm_time->alarm_day             = hals_rtc_bcd_2_normal((uint8_t)GET_ALRMTD_DAY(reg_alrmtd));
        rtc_alarm_time->alarm_hour            = hals_rtc_bcd_2_normal((uint8_t)GET_ALRMTD_HR(reg_alrmtd));
        rtc_alarm_time->alarm_minute          = hals_rtc_bcd_2_normal((uint8_t)GET_ALRMTD_MN(reg_alrmtd));
        rtc_alarm_time->alarm_second          = hals_rtc_bcd_2_normal((uint8_t)GET_ALRMTD_SC(reg_alrmtd));
    } else {
        /* do nothing */
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      poll for RTC alarm0 event
    \param[in]  timeout_ms: the time cost for event polling
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_alarm0_event_poll(uint32_t timeout_ms)
{
    int32_t error_code   = HAL_ERR_NONE;
    uint32_t flag_status = RESET;
    uint32_t tick_start  = 0U;

    tick_start = hal_sys_basetick_count_get();
    /* wait until ALRM0F flag to be set after the alarm is enabled */
    do {
        flag_status = RTC_STAT & RTC_STAT_ALRM0F;
    } while((SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_code = HAL_ERR_TIMEOUT;
    } else {
        hals_rtc_flag_clear(RTC_FLAG_ALARM0);
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      poll for RTC alarm1 event
    \param[in]  timeout_ms: the time cost for event polling
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_alarm1_event_poll(uint32_t timeout_ms)
{
    int32_t error_code   = HAL_ERR_NONE;
    uint32_t flag_status = RESET;
    uint32_t tick_start  = 0U;

    tick_start = hal_sys_basetick_count_get();
    /* wait until ALRM0F flag to be set after the alarm is enabled */
    do {
        flag_status = RTC_STAT & RTC_STAT_ALRM1F;
    } while((SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_code = HAL_ERR_TIMEOUT;
    } else {
        hals_rtc_flag_clear(RTC_FLAG_ALARM1);
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      poll for RTC timestamp event
    \param[in]  timeout_ms: the time cost for event polling
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_HARDWARE, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_timestamp_event_poll(uint32_t timeout_ms)
{
    int32_t error_code   = HAL_ERR_NONE;
    uint32_t flag_status = RESET;
    uint32_t tick_start  = 0U;

    /* wait until TSF flag to be set */
    flag_status = RTC_STAT & RTC_STAT_TSF;
    tick_start  = hal_sys_basetick_count_get();
    while((SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) && ((uint32_t)RESET == flag_status)) {
        flag_status = RTC_STAT & RTC_STAT_TSF;
        if((uint32_t)RESET != (RTC_STAT & RTC_STAT_TSOVRF)) {
            error_code = HAL_ERR_HARDWARE;
            break;
        } else {
            /* do nothing */
        }
    }

    if(HAL_ERR_NONE == error_code) {
        if((uint32_t)RESET == flag_status) {
            error_code = HAL_ERR_TIMEOUT;
        } else {
            hals_rtc_flag_clear(RTC_FLAG_TS);
            hals_rtc_flag_clear(RTC_FLAG_TSOVR);
        }
    } else {
        /* do nothing */
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      poll for RTC tamper0 event
    \param[in]  timeout_ms: the time cost for event polling
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_tamper0_event_poll(uint32_t timeout_ms)
{
    uint32_t flag_status = RESET;
    uint32_t tick_start  = 0U;
    int32_t  error_code  = HAL_ERR_NONE;

    tick_start = hal_sys_basetick_count_get();
    /* wait until TP0F flag to be set */
    do {
        flag_status = RTC_STAT & RTC_STAT_TP0F;
    } while((SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_code = HAL_ERR_TIMEOUT;
    } else {
        hals_rtc_flag_clear(RTC_FLAG_TP0);
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      poll for RTC tamper1 event
    \param[in]  timeout_ms: the time cost for event polling
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_tamper1_event_poll(uint32_t timeout_ms)
{
    int32_t error_code   = HAL_ERR_NONE;
    uint32_t flag_status = RESET;
    uint32_t tick_start  = 0U;

    tick_start = hal_sys_basetick_count_get();
    /* wait until TP1F flag to be set */
    do {
        flag_status = RTC_STAT & RTC_STAT_TP1F;
    } while((SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_code = HAL_ERR_TIMEOUT;
    } else {
        hals_rtc_flag_clear(RTC_FLAG_TP1);
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      poll for RTC wake up event
    \param[in]  timeout_ms: the time cost for event polling
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_wakeup_event_poll(uint32_t timeout_ms)
{
    int32_t error_code   = HAL_ERR_NONE;
    uint32_t flag_status = RESET;
    uint32_t tick_start  = 0U;

    tick_start = hal_sys_basetick_count_get();
    /* wait until WTF flag to be set */
    do {
        flag_status = RTC_STAT & RTC_STAT_WTF;
    } while((SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_code = HAL_ERR_TIMEOUT;
    } else {
        hals_rtc_flag_clear(RTC_FLAG_WT);
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      enable RTC time-stamp
    \param[in]  edge: specify which edge to detect of time-stamp
                only one parameter can be selected which is shown as below:
      \arg        RTC_TIMESTAMP_RISING_EDGE: rising edge is valid event edge for timestamp event
      \arg        RTC_TIMESTAMP_FALLING_EDGE: falling edge is valid event edge for timestamp event
    \param[out] none
    \retval     none
*/
void hal_rtc_timestamp_enable(uint32_t edge)
{
    uint32_t reg_ctl = 0U;

    /* clear the bits to be configured in RTC_CTL */
    reg_ctl = (uint32_t)(RTC_CTL & (uint32_t)(~(RTC_CTL_TSEG | RTC_CTL_TSEN)));

    /* new configuration */
    reg_ctl |= (uint32_t)(edge | RTC_CTL_TSEN);

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL = (uint32_t)reg_ctl;

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      disable RTC time-stamp
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rtc_timestamp_disable(void)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* clear the TSEN bit */
    RTC_CTL &= ~(uint32_t)(RTC_CTL_TSEG | RTC_CTL_TSEN);

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      enable RTC tamper
    \param[in]  source: specify which tamper source to be disabled
                only one parameter can be selected which is shown as below:
      \arg        RTC_TAMPER0
      \arg        RTC_TAMPER1
    \param[out] none
    \retval     none
*/
void hal_rtc_tamper_enable(uint32_t source)
{
    /* enable tamper */
    RTC_TAMP |= (uint32_t)source;
}

/*!
    \brief      disable RTC tamper
    \param[in]  source: specify which tamper source to be disabled
                only one parameter can be selected which is shown as below:
      \arg        RTC_TAMPER0
      \arg        RTC_TAMPER1
    \param[out] none
    \retval     none
*/
void hal_rtc_tamper_disable(uint32_t source)
{
    /* disable tamper */
    RTC_TAMP &= (uint32_t)~source;
}

/*!
    \brief      enable RTC bypass shadow registers function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rtc_bypass_shadow_enable(void)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL |= (uint8_t)RTC_CTL_BPSHAD;

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      disable RTC bypass shadow registers function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rtc_bypass_shadow_disable(void)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL &= (uint8_t)~RTC_CTL_BPSHAD;

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      enable RTC reference clock detection function
    \param[in]  none
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hal_rtc_refclock_detection_enable(void)
{
    ErrStatus error_status = ERROR;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* enter init mode */
    error_status = hals_rtc_init_mode_enter();

    if(ERROR != error_status) {
        RTC_CTL |= (uint32_t)RTC_CTL_REFEN;
        /* exit init mode */
        hals_rtc_init_mode_exit();
    } else {
        /* do nothing */
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* return error status */
    return error_status;
}

/*!
    \brief      disable RTC reference clock detection function
    \param[in]  none
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hal_rtc_refclock_detection_disable(void)
{
    ErrStatus error_status = ERROR;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* enter init mode */
    error_status = hals_rtc_init_mode_enter();

    if(ERROR != error_status) {
        RTC_CTL &= (uint32_t)~RTC_CTL_REFEN;
        /* exit init mode */
        hals_rtc_init_mode_exit();
    } else {
        /* do nothing */
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* return error status */
    return error_status;
}

/*!
    \brief      disable RTC alarm
    \param[in]  rtc_alarm:RTC_ALARM0 or RTC_ALARM1
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hal_rtc_alarm_disable(uint32_t rtc_alarm)
{
    volatile uint32_t time_index = RTC_ALRMXWF_TIMEOUT;
    ErrStatus error_status       = ERROR;
    uint32_t flag_status         = RESET;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* clear the state of alarm */
    if(RTC_ALARM0 == rtc_alarm) {
        RTC_CTL &= (uint32_t)(~RTC_CTL_ALARM0EN);
        /* wait until ALRM0WF flag to be set after the alarm is disabled */
        do {
            flag_status = RTC_STAT & RTC_STAT_ALRM0WF;
        } while((((--time_index)) > 0U) && ((uint32_t)RESET == flag_status));
    } else {
        RTC_CTL &= (uint32_t)(~RTC_CTL_ALARM1EN);
        /* wait until ALRM1WF flag to be set after the alarm is disabled */
        do {
            flag_status = RTC_STAT & RTC_STAT_ALRM1WF;
        } while((((--time_index)) > 0U) && ((uint32_t)RESET == flag_status));
    }

    if(RESET != flag_status) {
        error_status = SUCCESS;
    } else {
        /* do nothing */
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* return error status */
    return error_status;
}

/*!
    \brief      enable RTC auto wakeup
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_rtc_wakeup_enable(void)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL |= RTC_CTL_WTEN;

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      disable RTC auto wakeup
    \param[in]  none
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hal_rtc_wakeup_disable(void)
{
    ErrStatus error_status       = ERROR;
    volatile uint32_t time_index = RTC_WTWF_TIMEOUT;
    uint32_t flag_status         = RESET;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL &= ~RTC_CTL_WTEN;
    /* wait until the WTWF flag to be set */
    do {
        flag_status = RTC_STAT & RTC_STAT_WTWF;
    } while((((--time_index)) > 0U) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_status = ERROR;
    } else {
        error_status = SUCCESS;
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* return error status */
    return error_status;
}

/*!
    \brief      write data to backup register
    \param[in]  rtc_dev: pointer to rtc device structure
    \param[in]  backup_index: the index of backup register
      \arg        0 - 31
    \param[in]  data: the data to be written to backup register
      \arg        0x0 - 0xFFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_backup_data_write(hal_rtc_dev_struct *rtc_dev, uint32_t backup_index, uint32_t data)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t reg       = 0U;
    /* check parameter backup_index */
#if (1U == HAL_PARAMETER_CHECK)
    if(backup_index > 31U) {
        HAL_DEBUGE("parameter [backup_index] value is invalid");
        return HAL_ERR_VAL;
    }
    if(rtc_dev == NULL) {
        HAL_DEBUGE("parameter [rtc_dev] is NULL");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);

    reg= ((RTC) + 0x00000050U + (backup_index << 2));
    *(__IO uint32_t *)reg = data;

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      read data from backup register
    \param[in]  rtc_dev: pointer to rtc device structure
    \param[in]  backup_index: the index of backup register
      \arg        0 - 31
    \param[out] data: the data read from backup register
      \arg        0x0 - 0xFFFFFFFF
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtc_backup_data_read(hal_rtc_dev_struct *rtc_dev, uint32_t backup_index, uint32_t *data)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t reg       = 0U;
    /* check parameter backup_index */
#if (1U == HAL_PARAMETER_CHECK)
    if(backup_index > 31U) {
        HAL_DEBUGE("parameter [backup_index] value is invalid");
        return HAL_ERR_VAL;
    }
    if(rtc_dev == NULL) {
        HAL_DEBUGE("parameter [rtc_dev] is NULL");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock rtc */
    HAL_LOCK(rtc_dev);

    reg   = (RTC) + 0x00000050U + (backup_index << 2);
    *data = (*(__IO uint32_t *)reg);

    /* unlock rtc */
    HAL_UNLOCK(rtc_dev);

    /* return error code */
    return error_code;
}

/*!
    \brief      configure RTC time-stamp internal event
    \param[in]  mode: specify which internal or external event to be detected
                only one parameter can be selected which is shown as below:
      \arg        RTC_ITSEN_DISABLE: disable RTC time-stamp internal event
      \arg        RTC_ITSEN_ENABLE: enable RTC time-stamp internal event
    \param[out] none
    \retval     none
*/
void hal_rtc_timestamp_internalevent_config(uint32_t mode)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    if(RTC_ITSEN_ENABLE == mode) {
        RTC_CTL |= RTC_CTL_ITSEN;
    } else {
        RTC_CTL &= ~(uint32_t)RTC_CTL_ITSEN;
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      get RTC time-stamp subsecond
    \param[in]  none
    \param[out] none
    \retval     RTC time-stamp subsecond value between 0x00000000 and 0xFFFFFFFF
*/
uint32_t hal_rtc_timestamp_subsecond_get(void)
{
    return ((uint32_t)RTC_SSTS);
}

/*!
    \brief      get RTC state
    \param[in]   rtc_dev: RTC device information structure
                 the structure is not necessary to be reconfigured after structure initialization,
                 the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_rtc_state_enum details refer to gd32h7xx_hal_rtc.h
*/
hal_rtc_state_enum hal_rtc_state_get(hal_rtc_dev_struct *rtc_dev)
{
    return (hal_rtc_state_enum)rtc_dev->state;
}

/*!
    \brief      get RTC error state
    \param[in]  rtc_dev: RTC device information structure
                 the structure is not necessary to be reconfigured after structure initialization,
                 the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_rtc_error_enum details refer to gd32h7xx_hal_rtc.h
*/
hal_rtc_error_enum hal_rtc_error_state_get(hal_rtc_dev_struct *rtc_dev)
{
    return (hal_rtc_error_enum)rtc_dev->error_state;
}

/*!
    \brief      wait until RTC_TIME and RTC_DATE registers are synchronized with APB clock, and the shadow
                registers are updated
    \param[in]  none
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hals_rtc_register_sync_wait(void)
{
    volatile uint32_t time_index = RTC_RSYNF_TIMEOUT;
    uint32_t flag_status         = RESET;
    ErrStatus error_status       = ERROR;

    if(RESET == (RTC_CTL & RTC_CTL_BPSHAD)) {
        /* disable the write protection */
        RTC_WPK = RTC_UNLOCK_KEY1;
        RTC_WPK = RTC_UNLOCK_KEY2;

        /* firstly clear RSYNF flag */
        RTC_STAT &= (uint32_t)(~RTC_STAT_RSYNF);

        /* wait until RSYNF flag to be set */
        do {
            flag_status = RTC_STAT & RTC_STAT_RSYNF;
        } while((((--time_index)) > 0x00U) && (RESET == flag_status));

        if(RESET != flag_status) {
            error_status = SUCCESS;
        } else {
            /* do nothing */
        }
        /* enable the write protection */
        RTC_WPK = RTC_LOCK_KEY;
    } else {
        error_status = SUCCESS;
    }

    /* return error status */
    return error_status;
}

/*!
    \brief      get RTC timestamp time and date
    \param[in]  none
    \param[out] rtc_timestamp: pointer to a hal_rtc_timestamp_config_struct structure which contains
                parameters for RTC time-stamp configuration
                members of the structure and the member values are shown as below:
                  rtc_timestamp_month: RTC_JAN, RTC_FEB, RTC_MAR, RTC_APR, RTC_MAY, RTC_JUN,
                                       RTC_JUL, RTC_AUG, RTC_SEP, RTC_OCT, RTC_NOV, RTC_DEC
                  rtc_timestamp_date: 0x1 - 0x31(BCD format)
                  rtc_timestamp_weekday: RTC_MONDAY, RTC_TUESDAY, RTC_WEDSDAY, RTC_THURSDAY, RTC_FRIDAY,
                                     RTC_SATURDAY, RTC_SUNDAY if RTC_ALARM_WEEKDAY_SELECTED is set
                  rtc_timestamp_hour: 0x0 - 0x12(BCD format) or 0x0 - 0x23(BCD format) depending on the
   rtc_display_format rtc_timestamp_minute: 0x0 - 0x59(BCD format) rtc_timestamp_second: 0x0 - 0x59(BCD format)
                  rtc_am_pm: RTC_AM, RTC_PM
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_rtc_timestamp_get(hal_rtc_timestamp_config_struct *rtc_timestamp)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t temp_tts  = 0U;
    uint32_t temp_dts  = 0U;
    uint32_t temp_ssts = 0U;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtc_timestamp) {
        HAL_DEBUGE("pointer [rtc_timestamp] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get the value of time_stamp registers */
    temp_tts  = (uint32_t)RTC_TTS;
    temp_dts  = (uint32_t)RTC_DTS;
    temp_ssts = (uint32_t)RTC_SSTS;

    /* get timestamp time and construct the rtc_timestamp_struct structure */
    rtc_timestamp->timestamp_am_pm     = (uint32_t)(temp_tts & RTC_TTS_PM);
    rtc_timestamp->timestamp_month     = hals_rtc_bcd_2_normal((uint8_t)GET_DTS_MON(temp_dts));
    rtc_timestamp->timestamp_date      = hals_rtc_bcd_2_normal((uint8_t)GET_DTS_DAY(temp_dts));
    rtc_timestamp->timestamp_weekday   = (uint8_t)GET_DTS_DOW(temp_dts);
    rtc_timestamp->timestamp_hour      = hals_rtc_bcd_2_normal((uint8_t)GET_TTS_HR(temp_tts));
    rtc_timestamp->timestamp_minute    = hals_rtc_bcd_2_normal((uint8_t)GET_TTS_MN(temp_tts));
    rtc_timestamp->timestamp_second    = hals_rtc_bcd_2_normal((uint8_t)GET_TTS_SC(temp_tts));
    rtc_timestamp->timestamp_subsecond = temp_ssts;

    /* return error code */
    return error_code;
}

/*!
    \brief      get current subsecond value
    \param[in]  none
    \param[out] none
    \retval     current subsecond value between 0x00000000 and 0xFFFFFFFF
*/
uint32_t hals_rtc_subsecond_get(void)
{
    uint32_t reg = 0U;
    /* if BPSHAD bit is reset, reading RTC_SS will lock RTC_TIME and RTC_DATE automatically */
    reg = (uint32_t)RTC_SS;
    /* read RTC_DATE to unlock the 3 shadow registers */
    (void)(RTC_DATE);

    return reg;
}

/*!
    \brief      adjust the daylight saving time by adding or subtracting one hour from the current time
    \param[in]  operation: hour adjustment operation
                only one parameter can be selected which is shown as below:
      \arg        RTC_DAYLIGHT_SAVING_ADD_1H: add one hour
      \arg        RTC_DAYLIGHT_SAVING_SUB_1H: subtract one hour
      \arg        RTC_DAYLIGHT_SAVING_NONE: no add or subtract one hour
    \param[in]  record: daylight saving mark operation
                only one parameter can be selected which is shown as below:
      \arg        RTC_RECORD_DAYLIGHTSAVING_SET: set daylight saving mark
      \arg        RTC_RECORD_DAYLIGHTSAVING_RESET: reset daylight saving mark
    \param[out] none
    \retval     none
*/
void hals_rtc_daylight_saving_time_adjust(uint32_t operation, uint32_t record)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL &= (uint32_t)(~(RTC_CTL_A1H | RTC_CTL_S1H | RTC_CTL_DSM));
    RTC_CTL |= (uint32_t)(operation | record);

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      configure rtc alternate output source
    \param[in]  source: specify signal to output
                only one parameter can be selected which is shown as below:
      \arg        RTC_CALIBRATION_512HZ: when the LSE frequency is 32768Hz and the RTC_PSC
                                         is the default value, output 512Hz signal
      \arg        RTC_CALIBRATION_1HZ: when the LSE frequency is 32768Hz and the RTC_PSC
                                       is the default value, output 1Hz signal
      \arg        RTC_ALARM_HIGH: when the  alarm flag is set, the output pin is high
      \arg        RTC_ALARM_LOW: when the  Alarm flag is set, the output pin is low
    \param[in]  mode: specify the output pin (PC13) mode when output alarm signal
                only one parameter can be selected which is shown as below:
      \arg        RTC_ALARM_OUTPUT_OD: open drain mode
      \arg        RTC_ALARM_OUTPUT_PP: push pull mode
    \param[out] none
    \retval     none
*/
void hals_rtc_alter_output_config(uint32_t source, uint32_t mode)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL &= (uint32_t)~(RTC_CTL_COEN | RTC_CTL_OS | RTC_CTL_OPOL | RTC_CTL_COS);

    RTC_CTL |= (uint32_t)(source);

    /* alarm output */
    if(RESET == (source & RTC_OS_DISABLE)) {
        RTC_CFG &= (uint32_t)~(RTC_CFG_ALRMOUTTYPE);
        RTC_CFG |= (uint32_t)(mode);
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      enter RTC init mode
    \param[in]  none
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hals_rtc_init_mode_enter(void)
{
    uint32_t time_index    = RTC_INITM_TIMEOUT;
    uint32_t flag_status   = RESET;
    ErrStatus error_status = ERROR;

    /* check whether it has been in init mode */
    if(RESET == (RTC_STAT & RTC_STAT_INITF)) {
        RTC_STAT |= RTC_STAT_INITM;

        /* wait until the INITF flag to be set */
        do {
            flag_status = RTC_STAT & RTC_STAT_INITF;
        } while((((--time_index) > 0x00U)) && (RESET == flag_status));

        if(RESET != flag_status) {
            error_status = SUCCESS;
        } else {
            /* do nothing */
        }
    } else {
        error_status = SUCCESS;
    }

    /* return error status */
    return error_status;
}

/*!
    \brief      exit RTC init mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_rtc_init_mode_exit(void)
{
    RTC_STAT &= (uint32_t)(~RTC_STAT_INITM);
}

/*!
    \brief      convert from  BCD format to binary format
    \param[in]  data: data to be converted
    \param[out] none
    \retval     converted data
*/
uint8_t hals_rtc_bcd_2_normal(uint8_t data)
{
    uint8_t temp = 0U;
    temp = ((uint8_t)(data & (uint8_t)0xF0) >> (uint8_t)0x4) * 10U;
    return (temp + (data & (uint8_t)0x0F));
}

/*!
    \brief      convert from  BCD format to binary format
    \param[in]  data: data to be converted
    \param[out] none
    \retval     converted data
*/
uint8_t hals_rtc_normal_2_bcd(uint8_t data)
{
    uint8_t bcd_high = 0U;

    while(data >= 10U) {
        bcd_high++;
        data -= 10U;
    }

    return ((uint8_t)(bcd_high << 4) | data);
}

/*!
    \brief      configure subsecond of RTC alarm
    \param[in]  alarm:RTC_ALARM0 or RTC_ALARM1
    \param[in]  mask_subsecond: alarm subsecond mask
                only one parameter can be selected which is shown as below:
      \arg        RTC_MASKSSC_0_14: mask alarm subsecond configuration
      \arg        RTC_MASKSSC_1_14: mask RTC_ALRM0SS_SSC[14:1], and RTC_ALRM0SS_SSC[0] is to be compared
      \arg        RTC_MASKSSC_2_14: mask RTC_ALRM0SS_SSC[14:2], and RTC_ALRM0SS_SSC[1:0] is to be compared
      \arg        RTC_MASKSSC_3_14: mask RTC_ALRM0SS_SSC[14:3], and RTC_ALRM0SS_SSC[2:0] is to be compared
      \arg        RTC_MASKSSC_4_14: mask RTC_ALRM0SS_SSC[14:4], and RTC_ALRM0SS_SSC[3:0] is to be compared
      \arg        RTC_MASKSSC_5_14: mask RTC_ALRM0SS_SSC[14:5], and RTC_ALRM0SS_SSC[4:0] is to be compared
      \arg        RTC_MASKSSC_6_14: mask RTC_ALRM0SS_SSC[14:6], and RTC_ALRM0SS_SSC[5:0] is to be compared
      \arg        RTC_MASKSSC_7_14: mask RTC_ALRM0SS_SSC[14:7], and RTC_ALRM0SS_SSC[6:0] is to be compared
      \arg        RTC_MASKSSC_8_14: mask RTC_ALRM0SS_SSC[14:8], and RTC_ALRM0SS_SSC[7:0] is to be compared
      \arg        RTC_MASKSSC_9_14: mask RTC_ALRM0SS_SSC[14:9], and RTC_ALRM0SS_SSC[8:0] is to be compared
      \arg        RTC_MASKSSC_10_14: mask RTC_ALRM0SS_SSC[14:10], and RTC_ALRM0SS_SSC[9:0] is to be compared
      \arg        RTC_MASKSSC_11_14: mask RTC_ALRM0SS_SSC[14:11], and RTC_ALRM0SS_SSC[10:0] is to be compared
      \arg        RTC_MASKSSC_12_14: mask RTC_ALRM0SS_SSC[14:12], and RTC_ALRM0SS_SSC[11:0] is to be compared
      \arg        RTC_MASKSSC_13_14: mask RTC_ALRM0SS_SSC[14:13], and RTC_ALRM0SS_SSC[12:0] is to be compared
      \arg        RTC_MASKSSC_14: mask RTC_ALRM0SS_SSC[14], and RTC_ALRM0SS_SSC[13:0] is to be compared
      \arg        RTC_MASKSSC_NONE: mask none, and RTC_ALRM0SS_SSC[14:0] is to be compared
    \param[in]  subsecond: alarm subsecond value(0x000 - 0x7FFF)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_rtc_alarm_subsecond_config(uint32_t alarm, uint32_t mask_subsecond, uint32_t subsecond)
{
    int32_t error_code = HAL_ERR_NONE;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    if(RTC_ALARM0 == alarm) {
        RTC_ALRM0SS = mask_subsecond | subsecond;
    } else if(RTC_ALARM1 == alarm) {
        RTC_ALRM1SS = mask_subsecond | subsecond;
    } else {
        HAL_DEBUGE("parameter [alarm] is invalid");
        error_code = HAL_ERR_VAL;
    }

    if(HAL_ERR_NONE == error_code) {
        /* enable the write protection */
        RTC_WPK = RTC_LOCK_KEY;
    } else {
        /* do nothing */
    }

    /* return error code */
    return error_code;
}

/*!
    \brief      get RTC alarm subsecond
    \param[in]  alarm:RTC_ALARM0 or RTC_ALARM1
    \param[out] none
    \retval     RTC alarm subsecond value
*/
uint32_t hals_rtc_alarm_subsecond_get(uint32_t alarm)
{
    uint32_t subsecond_value = 0U;

    if(RTC_ALARM0 == alarm) {
        subsecond_value = ((uint32_t)(RTC_ALRM0SS & RTC_ALRM0SS_SSC));
    } else {
        subsecond_value = ((uint32_t)(RTC_ALRM1SS & RTC_ALRM1SS_SSC));
    }

    return subsecond_value;
}

/*!
    \brief      enable RTC alarm
    \param[in]  rtc_alarm:RTC_ALARM0 or RTC_ALARM1
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_rtc_alarm_enable(uint32_t rtc_alarm)
{
    int32_t error_code = HAL_ERR_NONE;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    if(RTC_ALARM0 == rtc_alarm) {
        RTC_CTL |= RTC_CTL_ALARM0EN;
    } else if(RTC_ALARM1 == rtc_alarm) {
        RTC_CTL |= RTC_CTL_ALARM1EN;
    } else {
        HAL_DEBUGE("parameter [rtc_alarm] value is invalid");
        error_code = HAL_ERR_VAL;
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* return error code */
    return error_code;
}

/*!
    \brief      set RTC auto wakeup timer clock
    \param[in]  wakeup_clock:
      \arg        WAKEUP_RTCCK_DIV16: RTC auto wakeup timer clock is RTC clock divided by 16
      \arg        WAKEUP_RTCCK_DIV8: RTC auto wakeup timer clock is RTC clock divided by 8
      \arg        WAKEUP_RTCCK_DIV4: RTC auto wakeup timer clock is RTC clock divided by 4
      \arg        WAKEUP_RTCCK_DIV2: RTC auto wakeup timer clock is RTC clock divided by 2
      \arg        WAKEUP_CKSPRE: RTC auto wakeup timer clock is ckspre
      \arg        WAKEUP_CKSPRE_2EXP16: RTC auto wakeup timer clock is ckspre and wakeup timer add 2exp16
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hals_rtc_wakeup_clock_set(uint8_t wakeup_clock)
{
    ErrStatus error_status       = ERROR;
    volatile uint32_t time_index = RTC_WTWF_TIMEOUT;
    uint32_t flag_status         = RESET;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* only when RTC_CTL_WTEN=0 and RTC_STAT_WTWF=1 can write RTC_CTL[2:0] */
    /* wait until the WTWF flag to be set */
    do {
        flag_status = RTC_STAT & RTC_STAT_WTWF;
    } while((((--time_index) > 0U)) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_status = ERROR;
    } else {
        RTC_CTL &= (uint32_t)~RTC_CTL_WTCS;
        RTC_CTL |= (uint32_t)wakeup_clock;
        error_status = SUCCESS;
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* return error status */
    return error_status;
}

/*!
    \brief      set RTC auto wakeup timer value
    \param[in]  wakeup_timer: 0x0000-0xffff
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hals_rtc_wakeup_timer_set(uint16_t wakeup_timer)
{
    ErrStatus error_status       = ERROR;
    volatile uint32_t time_index = RTC_WTWF_TIMEOUT;
    uint32_t flag_status         = RESET;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* wait until the WTWF flag to be set */
    do {
        flag_status = RTC_STAT & RTC_STAT_WTWF;
    } while((((--time_index) > 0U)) && ((uint32_t)RESET == flag_status));

    if((uint32_t)RESET == flag_status) {
        error_status = ERROR;
    } else {
        RTC_WUT      = (uint32_t)wakeup_timer;
        error_status = SUCCESS;
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* return error status */
    return error_status;
}

/*!
    \brief      get RTC auto wakeup timer value
    \param[in]  none
    \param[out] none
    \retval     wakeup timer value
*/
uint16_t hals_rtc_wakeup_timer_get(void)
{
    return (uint16_t)RTC_WUT;
}

/*!
    \brief      adjust the daylight saving time by adding or subtracting one hour from the current time
    \param[in]  operation: hour adjustment operation
                only one parameter can be selected which is shown as below:
      \arg        RTC_CTL_A1H: add one hour
      \arg        RTC_CTL_S1H: subtract one hour
    \param[out] none
    \retval     none
*/
void hals_rtc_hour_adjust(uint32_t operation)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    RTC_CTL |= (uint32_t)(operation);

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      adjust RTC second or subsecond value of current time
    \param[in]  add: add 1s to current time or not
                only one parameter can be selected which is shown as below:
      \arg        RTC_SHIFT_ADD1S_RESET: no effect
      \arg        RTC_SHIFT_ADD1S_SET: add 1s to current time
    \param[in]  minus: number of subsecond to minus from current time(0x0 - 0x7FFF)
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hals_rtc_second_adjust(uint32_t add, uint32_t minus)
{
    uint32_t time_index    = RTC_SHIFTCTL_TIMEOUT;
    ErrStatus error_status = ERROR;
    uint32_t flag_status   = RESET;
    uint32_t temp          = 0U;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* check if a shift operation is ongoing */
    do {
        flag_status = RTC_STAT & RTC_STAT_SOPF;
    } while((((--time_index)) > 0x00U) && (RESET != flag_status));

    temp = RTC_CTL & RTC_CTL_REFEN;
    /* check if the function of reference clock detection is disabled */
    if((RESET == flag_status) && (RESET == temp)) {
        RTC_SHIFTCTL = (uint32_t)(add | SHIFTCTL_SFS(minus));
        error_status = hals_rtc_register_sync_wait();
    } else {
        /* do nothing */
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* return error status */
    return error_status;
}

/*!
    \brief      configure RTC smooth calibration
    \param[in]  window: select calibration window
      \arg        RTC_CALIBRATION_WINDOW_32S: 2exp20 RTCCLK cycles, 32s if RTCCLK = 32768 Hz
      \arg        RTC_CALIBRATION_WINDOW_16S: 2exp19 RTCCLK cycles, 16s if RTCCLK = 32768 Hz
      \arg        RTC_CALIBRATION_WINDOW_8S: 2exp18 RTCCLK cycles, 8s if RTCCLK = 32768 Hz
    \param[in]  plus: add RTC clock or not
      \arg        RTC_CALIBRATION_PLUS_SET: add one RTC clock every 2048 RTC clock
      \arg        RTC_CALIBRATION_PLUS_RESET: no effect
    \param[in]  minus: the RTC clock to minus during the calibration window(0x0 - 0x1FF)
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hals_rtc_smooth_calibration_config(uint32_t window, uint32_t plus, uint32_t minus)
{
    volatile uint32_t time_index = RTC_HRFC_TIMEOUT;
    ErrStatus error_status       = ERROR;
    uint32_t flag_status         = RESET;

    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* check if a smooth calibration operation is ongoing */
    do {
        flag_status = RTC_STAT & RTC_STAT_SCPF;
    } while((((--time_index)) > 0U) && ((uint32_t)RESET != flag_status));

    if((uint32_t)RESET == flag_status) {
        RTC_HRFC     = (uint32_t)(window | plus | HRFC_CMSK(minus));
        error_status = SUCCESS;
    } else {
        /* do nothing */
    }

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;

    /* return error status */
    return error_status;
}

/*!
    \brief      enable specified RTC interrupt
    \param[in]  interrupt: specify which interrupt source to be enabled
      \arg        RTC_INT_TIMESTAMP: timestamp interrupt
      \arg        RTC_INT_ALARM0: alarm0 interrupt
      \arg        RTC_INT_ALARM1: alarm1 interrupt
      \arg        RTC_INT_TAMP_ALL: tamper detection interrupt
      \arg        RTC_INT_WAKEUP: wakeup timer interrupt
    \param[out] none
    \retval     none
*/
void hals_rtc_interrupt_enable(uint32_t interrupt)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* enable the interrupts in RTC_CTL register */
    RTC_CTL |= (uint32_t)(interrupt & (uint32_t)~RTC_INT_TAMP_ALL);
    /* enable the interrupts in RTC_TAMP register */
    RTC_TAMP |= (uint32_t)(interrupt & RTC_INT_TAMP_ALL);

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      disable specified RTC interrupt
    \param[in]  interrupt: specify which interrupt source to be disabled
      \arg        RTC_INT_TIMESTAMP: timestamp interrupt
      \arg        RTC_INT_ALARM0: alarm0 interrupt
      \arg        RTC_INT_ALARM1: alarm1 interrupt
      \arg        RTC_INT_TAMP_ALL: tamper detection interrupt
      \arg        RTC_INT_WAKEUP: wakeup timer interrupt
    \param[out] none
    \retval     none
*/
void hals_rtc_interrupt_disable(uint32_t interrupt)
{
    /* disable the write protection */
    RTC_WPK = RTC_UNLOCK_KEY1;
    RTC_WPK = RTC_UNLOCK_KEY2;

    /* disable the interrupts in RTC_CTL register */
    RTC_CTL &= (uint32_t)~(interrupt & (uint32_t)~(RTC_TAMP_TPIE));
    /* disable the interrupts in RTC_TAMP register */
    RTC_TAMP &= (uint32_t)~(interrupt & (RTC_TAMP_TPIE));

    /* enable the write protection */
    RTC_WPK = RTC_LOCK_KEY;
}

/*!
    \brief      check specified flag
    \param[in]  flag: specify which flag to check
                only one parameter can be selected which is shown as below:
      \arg        RTC_FLAG_SCP: recalibration pending flag
      \arg        RTC_FLAG_TP0: tamper 1 event flag
      \arg        RTC_FLAG_TP1: tamper 0 event flag
      \arg        RTC_FLAG_TSOVR: time-stamp overflow event flag
      \arg        RTC_FLAG_TS: time-stamp event flag
      \arg        RTC_FLAG_ALARM0: alarm0 event flag
      \arg        RTC_FLAG_ALARM1: alarm1 event flag
      \arg        RTC_FLAG_INIT: init mode event flag
      \arg        RTC_FLAG_RSYN: time and date registers synchronized event flag
      \arg        RTC_FLAG_YCM: year parameter configured event flag
      \arg        RTC_FLAG_SOP: shift operation pending flag
      \arg        RTC_FLAG_ALARM0W: alarm0 written available flag
      \arg        RTC_FLAG_ALARM1W: alarm1 written available flag
      \arg        RTC_FLAG_WT: wakeup timer occurs flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_rtc_flag_get(uint32_t flag)
{
    FlagStatus flag_state = RESET;

    if(RESET != (RTC_STAT & flag)) {
        flag_state = SET;
    } else {
        /* do nothing */
    }

    /* return flag state */
    return flag_state;
}

/*!
    \brief      clear specified flag
    \param[in]  flag: specify which flag to clear
                only one parameter can be selected which is shown as below:
      \arg        RTC_FLAG_SCP: recalibration pending flag
      \arg        RTC_FLAG_TP0: tamper 1 event flag
      \arg        RTC_FLAG_TP1: tamper 0 event flag
      \arg        RTC_FLAG_TSOVR: time-stamp overflow event flag
      \arg        RTC_FLAG_TS: time-stamp event flag
      \arg        RTC_FLAG_ALARM0: alarm0 event flag
      \arg        RTC_FLAG_ALARM1: alarm1 event flag
      \arg        RTC_FLAG_INIT: init mode event flag
      \arg        RTC_FLAG_RSYN: time and date registers synchronized event flag
      \arg        RTC_FLAG_YCM: year parameter configured event flag
      \arg        RTC_FLAG_SOP: shift operation pending flag
      \arg        RTC_FLAG_ALARM0W: alarm0 written available flag
      \arg        RTC_FLAG_ALARM1W: alarm1 written available flag
      \arg        RTC_FLAG_WT: wakeup timer occurs flag
    \param[out] none
    \retval     none
*/
void hals_rtc_flag_clear(uint32_t flag)
{
    RTC_STAT &= (uint32_t)(~flag);
}

/*!
    \brief      alarm0 interrupt handler function
    \param[in]  rtc: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rtc_alarm0_handler(void *rtc)
{
    hal_rtc_dev_struct *rtc_dev = (hal_rtc_dev_struct *)rtc;
    hal_rtc_user_cb p_func      = (hal_rtc_user_cb)rtc_dev->alarm0_callback;

    /* check p_func valid */
    if(NULL != p_func) {
        p_func(rtc_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      alarm1 interrupt handler function
    \param[in]  rtc: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rtc_alarm1_handler(void *rtc)
{
    hal_rtc_dev_struct *rtc_dev = (hal_rtc_dev_struct *)rtc;
    hal_rtc_user_cb p_func      = (hal_rtc_user_cb)rtc_dev->alarm1_callback;

    /* check p_func valid */
    if(NULL != p_func) {
        p_func(rtc_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      tamper0 interrupt handler function
    \param[in]  rtc: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rtc_tamper0_handler(void *rtc)
{
    hal_rtc_dev_struct *rtc_dev = (hal_rtc_dev_struct *)rtc;
    hal_rtc_user_cb p_func      = (hal_rtc_user_cb)rtc_dev->tamper0_callback;

    /* check p_func valid */
    if(NULL != p_func) {
        p_func(rtc_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      tamper1 interrupt handler function
    \param[in]  rtc: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rtc_tamper1_handler(void *rtc)
{
    hal_rtc_dev_struct *rtc_dev = (hal_rtc_dev_struct *)rtc;
    hal_rtc_user_cb p_func      = (hal_rtc_user_cb)rtc_dev->tamper1_callback;

    /* check p_func valid */
    if(NULL != p_func) {
        p_func(rtc_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      timestamp interrupt handler function
    \param[in]  rtc: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rtc_timestamp_handler(void *rtc)
{
    hal_rtc_dev_struct *rtc_dev = (hal_rtc_dev_struct *)rtc;
    hal_rtc_user_cb p_func      = (hal_rtc_user_cb)rtc_dev->timestamp_callback;

    /* check p_func valid */
    if(NULL != p_func) {
        p_func(rtc_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      wakeup interrupt handler function
    \param[in]  rtc: RTC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _rtc_wakeup_handler(void *rtc)
{
    hal_rtc_dev_struct *rtc_dev = (hal_rtc_dev_struct *)rtc;
    hal_rtc_user_cb p_func      = (hal_rtc_user_cb)rtc_dev->wakeup_callback;

    /* check p_func valid */
    if(NULL != p_func) {
        p_func(rtc_dev);
    } else {
        /* do nothing*/
    }
}
