/*!
    \file    gd32h7xx_hal_ospim.c
    \brief   OSPIM driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/*!
    \brief      initialize the OSPIM peripheral parameters
    \param[in]  port: The port used by OSPIM, the values is OSPIM_PORT0, OSPIM_PORT1
    \param[in]  ospim_init: OSPIM init parameter struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_ospim_init(uint8_t port, hal_ospim_init_parameter_struct *ospim_init)
{
    uint32_t port0 = 0U;
    uint32_t port1 = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == ospim_init) {
        HAL_DEBUGE("pointer [ospim_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(OSPIM_PORT0 == port) {
        /* configuration port SCK */
        if(ospim_init->port_sck) {
            port1 |= (ospim_init->port_sck_en | OSPIM_SCK_SOURCE_OSPI0_SCK);
        } else {
            port0 |= (ospim_init->port_sck_en | OSPIM_SCK_SOURCE_OSPI0_SCK);
        }

        /* configuration port CSN */
        if(ospim_init->port_csn) {
            port1 |= (ospim_init->port_csn_en | OSPIM_CSN_SOURCE_OSPI0_CSN);
        } else {
            port0 |= (ospim_init->port_csn_en | OSPIM_CSN_SOURCE_OSPI0_CSN);
        }

        /* configuration OSPI IO0~3 */
        switch(ospim_init->io3_0) {
        case OSPIM_SRCPLIO_OSPI0_IO_LOW:
            port0 |= (ospim_init->io3_0en | OSPIM_SRCPHIO_OSPI0_IO_LOW);
            break;
        case OSPIM_SRCPLIO_OSPI1_IO_LOW:
            port1 |= (ospim_init->io3_0en | OSPIM_SRCPLIO_OSPI0_IO_LOW);
            break;
        case OSPIM_SRCPHIO_OSPI1_IO_LOW:
            port1 |= (ospim_init->io3_0en | OSPIM_SRCPHIO_OSPI0_IO_LOW);
            break;
        default:
            break;
        }

        /* configuration OSPI IO4~7 */
        switch(ospim_init->io7_4) {
        case OSPIM_SRCPLIO_OSPI0_IO_HIGH:
            port0 |= (ospim_init->io7_4en | OSPIM_SRCPLIO_OSPI0_IO_HIGH);
            break;
        case OSPIM_SRCPHIO_OSPI0_IO_HIGH:
            port0 |= (ospim_init->io7_4en | OSPIM_SRCPHIO_OSPI0_IO_HIGH);
            break;
        case OSPIM_SRCPLIO_OSPI1_IO_HIGH:
            port1 |= (ospim_init->io7_4en | OSPIM_SRCPLIO_OSPI0_IO_HIGH);
            break;
        case OSPIM_SRCPHIO_OSPI1_IO_HIGH:
            port1 |= (ospim_init->io7_4en | OSPIM_SRCPHIO_OSPI0_IO_HIGH);
            break;
        default:
            break;
        }
    } else {
        /* configuration port SCK */
        if(ospim_init->port_sck) {
            port1 |= (ospim_init->port_sck_en | OSPIM_SCK_SOURCE_OSPI1_SCK);
        } else {
            port0 |= (ospim_init->port_sck_en | OSPIM_SCK_SOURCE_OSPI1_SCK);
        }

        /* configuration port CSN */
        if(ospim_init->port_csn) {
            port1 |= (ospim_init->port_csn_en | OSPIM_CSN_SOURCE_OSPI1_CSN);
        } else {
            port0 |= (ospim_init->port_csn_en | OSPIM_CSN_SOURCE_OSPI1_CSN);
        }

        /* configuration OSPI IO0~3 */
        switch(ospim_init->io3_0) {
        case OSPIM_SRCPLIO_OSPI0_IO_LOW:
            port0 |= (ospim_init->io3_0en | OSPIM_SRCPLIO_OSPI1_IO_LOW);
            break;
        case OSPIM_SRCPLIO_OSPI1_IO_LOW:
            port1 |= (ospim_init->io3_0en | OSPIM_SRCPLIO_OSPI1_IO_LOW);
            break;
        case OSPIM_SRCPHIO_OSPI1_IO_LOW:
            port1 |= (ospim_init->io3_0en | OSPIM_SRCPHIO_OSPI1_IO_LOW);
            break;
        default:
            break;
        }

        /* configuration OSPI IO4~7 */
        switch(ospim_init->io7_4) {
        case OSPIM_SRCPLIO_OSPI0_IO_HIGH:
            port0 |= (ospim_init->io7_4en | OSPIM_SRCPLIO_OSPI1_IO_HIGH);
            break;
        case OSPIM_SRCPHIO_OSPI0_IO_HIGH:
            port0 |= (ospim_init->io7_4en | OSPIM_SRCPHIO_OSPI1_IO_HIGH);
            break;
        case OSPIM_SRCPLIO_OSPI1_IO_HIGH:
            port1 |= (ospim_init->io7_4en | OSPIM_SRCPLIO_OSPI1_IO_HIGH);
            break;
        case OSPIM_SRCPHIO_OSPI1_IO_HIGH:
            port1 |= (ospim_init->io7_4en | OSPIM_SRCPHIO_OSPI1_IO_HIGH);
            break;
        default:
            break;
        }
    }

    OSPIM_PCFG(OSPIM_PORT0) = port0;
    OSPIM_PCFG(OSPIM_PORT1) = port1;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the structure with default value
    \param[in]  ospim_init: OSPIM parameter struct,The structure of this parameter can be either hal_ospim_init_struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_ospim_struct_init(hal_ospim_init_parameter_struct *ospim_init)
{
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == ospim_init) {
        HAL_DEBUGE("pointer [ospim_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    ospim_init->port_sck_en = OSPIM_PORT_SCK_DISABLE;
    ospim_init->port_sck    = OSPIM_SCK_SOURCE_OSPI0_SCK;
    ospim_init->port_csn_en = OSPIM_PORT_CSN_DISABLE;
    ospim_init->port_csn    = OSPIM_CSN_SOURCE_OSPI0_CSN;
    ospim_init->io3_0en     = OSPIM_IO_LOW_DISABLE;
    ospim_init->io3_0       = OSPIM_SRCPLIO_OSPI0_IO_LOW;
    ospim_init->io7_4en     = OSPIM_IO_HIGH_DISABLE;
    ospim_init->io7_4       = OSPIM_SRCPHIO_OSPI0_IO_LOW;

    return HAL_ERR_NONE;
}

/*!
    \brief      reset the OSPIM peripheral
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_ospim_deinit(void)
{
    hal_rcu_periph_reset_enable(RCU_OSPIMRST);
    hal_rcu_periph_reset_disable(RCU_OSPIMRST);
}

/*!
    \brief      configurate SCK for port
    \param[in]  port: number of port
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT0: port 0
      \arg        OSPIM_PORT1: port 1
    \param[in]  sckconfg: enable or disable SCK
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT_SCK_DISABLE: disable SCK
      \arg        OSPIM_PORT_SCK_ENABLE: enable SCK
    \param[out] none
    \retval     none
*/
void hals_ospim_port_sck_config(uint8_t port, uint32_t sckconfg)
{
    OSPIM_PCFG(port) &= (uint32_t)(~OSPIM_PCFG_SCKEN);
    OSPIM_PCFG(port) |= (uint32_t)sckconfg;
}

/*!
    \brief      select source of SCK for port
    \param[in]  port: number of port
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT0: port 0
      \arg        OSPIM_PORT1: port 1
    \param[in]  sck_source: source of SCK
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_SCK_SOURCE_OSPI0_SCK: the source of SCK is OSPI0_SCK
      \arg        OSPIM_SCK_SOURCE_OSPI1_SCK: the source of SCK is OSPI1_SCK
    \param[out] none
    \retval     none
*/
void hals_ospim_port_sck_source_select(uint8_t port, uint32_t sck_source)
{
    OSPIM_PCFG(port) &= (uint32_t)(~OSPIM_PCFG_SRCPCK);
    OSPIM_PCFG(port) |= (uint32_t)sck_source;
}

/*!
    \brief      configurate CSN for port
    \param[in]  port: number of port
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT0: port 0
      \arg        OSPIM_PORT1: port 1
    \param[in]  csnconfig: enable or disable CSN
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT_CSN_DISABLE: disable CSN
      \arg        OSPIM_PORT_CSN_ENABLE: enable CSN
    \param[out] none
    \retval     none
*/
void hals_ospim_port_csn_config(uint8_t port, uint32_t csnconfig)
{
    OSPIM_PCFG(port) &= (uint32_t)(~OSPIM_PCFG_NCSEN);
    OSPIM_PCFG(port) |= (uint32_t)csnconfig;
}

/*!
    \brief      select source of CSN for port
    \param[in]  port: number of port
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT0: port 0
      \arg        OSPIM_PORT1: port 1
    \param[in]  csn_source: source of CSN
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_CSN_SOURCE_OSPI0_CSN: the source of CSN is OSPI0_CSN
      \arg        OSPIM_CSN_SOURCE_OSPI1_CSN: the source of CSN is OSPI1_CSN
    \param[out] none
    \retval     none
*/
void hals_ospim_port_csn_source_select(uint8_t port, uint32_t csn_source)
{
    OSPIM_PCFG(port) &= (uint32_t)(~OSPIM_PCFG_SRCPCS);
    OSPIM_PCFG(port) |= (uint32_t)csn_source;
}

/*!
    \brief      configurate IO[3:0] for port
    \param[in]  port: number of port
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT0: port 0
      \arg        OSPIM_PORT1: port 1
    \param[in]  ioconfig: enable or disable IO[3:0]
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_IO_LOW_DISABLE: disable IO[3:0]
      \arg        OSPIM_IO_LOW_ENABLE: enable IO[3:0]
    \param[out] none
    \retval     none
*/
void hals_ospim_port_io3_0_config(uint8_t port, uint32_t ioconfig)
{
    OSPIM_PCFG(port) &= (uint32_t)(~OSPIM_PCFG_POLEN);
    OSPIM_PCFG(port) |= (uint32_t)ioconfig;
}

/*!
    \brief      select source of IO[3:0] for port
    \param[in]  port: number of port
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT0: port 0
      \arg        OSPIM_PORT1: port 1
    \param[in]  io_source: source of IO[3:0]
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_SRCPLIO_OSPI0_IO_LOW: select OSPI0_IO[3:0]
      \arg        OSPIM_SRCPLIO_OSPI0_IO_HIGH: select OSPI0_IO[7:4]
      \arg        OSPIM_SRCPLIO_OSPI1_IO_LOW: select OSPI1_IO[3:0]
      \arg        OSPIM_SRCPLIO_OSPI1_IO_HIGH: select OSPI1_IO[7:4]
    \param[out] none
    \retval     none
*/
void hals_ospim_port_io3_0_source_select(uint8_t port, uint32_t io_source)
{
    OSPIM_PCFG(port) &= (uint32_t)(~OSPIM_PCFG_SRCPLIO);
    OSPIM_PCFG(port) |= (uint32_t)io_source;
}

/*!
    \brief      configurate IO[7:4] for port
    \param[in]  port: number of port
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT0: port 0
      \arg        OSPIM_PORT1: port 1
    \param[in]  ioconfig: enable or disable IO[7:4]
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_IO_HIGH_DISABLE: disable IO[7:4]
      \arg        OSPIM_IO_HIGH_ENABLE: enable IO[7:4]
    \param[out] none
    \retval     none
*/
void hals_ospim_port_io7_4_config(uint8_t port, uint32_t ioconfig)
{
    OSPIM_PCFG(port) &= (uint32_t)(~OSPIM_PCFG_POHEN);
    OSPIM_PCFG(port) |= (uint32_t)ioconfig;
}

/*!
    \brief      select source of IO[7:4] for port
    \param[in]  port: number of port
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_PORT0: port 0
      \arg        OSPIM_PORT1: port 1
    \param[in]  io_source: source of IO[7:4]
                only one parameter can be selected which is shown as below:
      \arg        OSPIM_SRCPHIO_OSPI0_IO_LOW: select OSPI0_IO[3:0]
      \arg        OSPIM_SRCPHIO_OSPI0_IO_HIGH: select OSPI0_IO[7:4]
      \arg        OSPIM_SRCPHIO_OSPI1_IO_LOW: select OSPI1_IO[3:0]
      \arg        OSPIM_SRCPHIO_OSPI1_IO_HIGH: select OSPI1_IO[7:4]
    \param[out] none
    \retval     none
*/
void hals_ospim_port_io7_4_source_select(uint8_t port, uint32_t io_source)
{
    OSPIM_PCFG(port) &= (uint32_t)(~OSPIM_PCFG_SRCPHIO);
    OSPIM_PCFG(port) |= (uint32_t)io_source;
}
