/*!
    \file    gd32h7xx_hal_nvic.c
    \brief   NVIC driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/*!
    \brief      set NVIC request priority group
    \param[in]  nvic_prigroup: the NVIC priority group
                only one parameter can be selected which is shown as below:
      \arg        NVIC_PRIGROUP_PRE0_SUB4:0 bit  for pre-emption priority 4 bits for subpriority
      \arg        NVIC_PRIGROUP_PRE1_SUB3:1 bit  for pre-emption priority 3 bits for subpriority
      \arg        NVIC_PRIGROUP_PRE2_SUB2:2 bits for pre-emption priority 2 bits for subpriority
      \arg        NVIC_PRIGROUP_PRE3_SUB1:3 bits for pre-emption priority 1 bit  for subpriority
      \arg        NVIC_PRIGROUP_PRE4_SUB0:4 bits for pre-emption priority 0 bit  for subpriority
    \param[out] none
    \retval     none
*/
void hal_nvic_irq_priority_group_set(uint32_t nvic_prigroup)
{
    /* set the priority group value */
    NVIC_SetPriorityGrouping(nvic_prigroup);
}

/*!
    \brief      get NVIC priority group
    \param[in]  none
    \param[out] none
    \retval     uin32_t: 0 - 0xFFFFFFFF
*/
uint32_t hal_nvic_irq_priority_group_get(void)
{
    return NVIC_GetPriorityGrouping();
}

/*!
    \brief      enable NVIC request
    \param[in]  nvic_irq: the NVIC interrupt request, detailed in IRQn_Type
    \param[in]  nvic_irq_pre_priority: the pre-emption priority needed to set
    \param[in]  nvic_irq_sub_priority: the subpriority needed to set
    \param[out] none
    \retval     none
*/
void hal_nvic_irq_enable(IRQn_Type nvic_irq, uint8_t nvic_irq_pre_priority, uint8_t nvic_irq_sub_priority)
{
    uint32_t temp_priority = 0U, temp_pre = 0U, temp_sub = 0U;
    int32_t exception_index = 0;
    /* System Exception Interrupt Handling */
    static const uint32_t exception_masks[14] = {
        0,                             /* NonMaskableInt_IRQn (non-configurable) */
        0,                             /* HardFault_IRQn (non-configurable) */
        SCB_SHCSR_MEMFAULTENA_Msk,     /* MemoryManagement_IRQn */
        SCB_SHCSR_BUSFAULTENA_Msk,     /* BusFault_IRQn */
        SCB_SHCSR_USGFAULTENA_Msk,     /* UsageFault_IRQn */
        0,                             /* reserve */
        0,                             /* reserve */
        0,                             /* reserve */
        0,                             /* reserve */
        0,                             /* SVCall_IRQn (No Enable Bit Required) */
        0,                             /* DebugMonitor_IRQn */
        0,                             /* reserve */
        0,                             /* PendSV_IRQn (No Enable Bit Required) */
        0                              /* SysTick_IRQn (Configured by SysTick_Config) */
    };

    /* use the priority group value to get the temp_pre and the temp_sub */
    switch(((SCB->AIRCR) & (uint32_t)0x00000700U) >> 8U) {
    case NVIC_PRIGROUP_PRE0_SUB4:
        temp_pre = 0U;
        temp_sub = 0x4U;
        break;
    case NVIC_PRIGROUP_PRE1_SUB3:
        temp_pre = 1U;
        temp_sub = 0x3U;
        break;
    case NVIC_PRIGROUP_PRE2_SUB2:
        temp_pre = 2U;
        temp_sub = 0x2U;
        break;
    case NVIC_PRIGROUP_PRE3_SUB1:
        temp_pre = 3U;
        temp_sub = 0x1U;
        break;
    case NVIC_PRIGROUP_PRE4_SUB0:
        temp_pre = 4U;
        temp_sub = 0x0U;
        break;
    default:
        hal_nvic_irq_priority_group_set(NVIC_PRIGROUP_PRE2_SUB2);
        temp_pre = 2U;
        temp_sub = 0x2U;
        break;
    }

    /* Setting the interrupt priority */
    if((int32_t)nvic_irq >= 0) {
        /* get the temp_priority to fill the NVIC->IP register */
        temp_priority = (uint32_t)nvic_irq_pre_priority << ((uint32_t)__NVIC_PRIO_BITS - temp_pre);
        temp_priority |= nvic_irq_sub_priority & (0x0FU >> ((uint32_t)__NVIC_PRIO_BITS - temp_sub));
        temp_priority = temp_priority << __NVIC_PRIO_BITS;

        #if(5U == __CM_CMSIS_VERSION_MAIN)
            NVIC->IP[nvic_irq] = (uint8_t)temp_priority;
        #else
            NVIC->IPR[nvic_irq] = (uint8_t)temp_priority;
        #endif

        /* Enable external interrupt */
        NVIC->ISER[(uint32_t)nvic_irq >> 5U] = (uint32_t)1U << ((uint32_t)nvic_irq & 0x1FU);
    } else {
        /* System Exception Interrupt Handling */
        if(((int32_t)(nvic_irq <= -4)) && ((int32_t)(nvic_irq >= -15))) {
            /* Skipping non-configurable exceptions (NMI and HardFault) */
            if((nvic_irq != NonMaskableInt_IRQn) && (nvic_irq != HardFault_IRQn)) {
                /* Setting the system exception priority */
                uint32_t shp_index = ((uint32_t)(nvic_irq) + 8U) & 0xFU;
                SCB->SHPR[shp_index] = (uint8_t)temp_priority;
            }
        }

        exception_index = (int32_t)nvic_irq + 14;
        if((uint32_t)exception_index < sizeof(exception_masks)/sizeof(exception_masks[0])) {
            if(0U != exception_masks[exception_index]) {
                uint32_t exception_irq = SCB->SHCSR;
                exception_irq |= exception_masks[exception_index];
                SCB->SHCSR = exception_irq;
            }
        }

        /* console DebugMonitor enable */
        if(nvic_irq == DebugMonitor_IRQn) {
            /* enable Debug Monitor */
            CoreDebug->DEMCR |= CoreDebug_DEMCR_MON_EN_Msk;

            /* config DebugMonitor priority */
            SCB->SHPR[11] = (uint8_t)((nvic_irq_pre_priority << (8U - (uint32_t)__NVIC_PRIO_BITS)) |
                                      (nvic_irq_sub_priority & 0x0FU));
        }
    }
}

/*!
    \brief      disable NVIC request
    \param[in]  nvic_irq: the NVIC interrupt request, detailed in IRQn_Type
    \param[out] none
    \retval     none
*/
void hal_nvic_irq_disable(IRQn_Type nvic_irq)
{
    NVIC_DisableIRQ(nvic_irq);
}

/*!
    \brief      set system NVIC priority
    \param[in]  nvic_irq: the NVIC interrupt request, only support negative number in IRQn_Type
    \param[in]  nvic_irq_pre_priority: the pre-emption priority needed to set
    \param[in]  nvic_irq_sub_priority: the subpriority needed to set
    \param[out] none
    \retval     none
*/
void hal_nvic_set_priority(IRQn_Type nvic_irq, uint32_t nvic_irq_pre_priority, uint32_t nvic_irq_sub_priority)
{
    uint32_t prioritygroup = 0U;
    uint32_t temp_priority = 0U;

    prioritygroup = NVIC_GetPriorityGrouping();
    temp_priority = NVIC_EncodePriority(prioritygroup, nvic_irq_pre_priority, nvic_irq_sub_priority);
    NVIC_SetPriority(nvic_irq, temp_priority);
}

/*!
    \brief      get system NVIC priority
    \param[in]  nvic_irq: the NVIC interrupt request, only support negative number in IRQn_Type
    \param[in]  nvic_prigroup: the NVIC priority group
                only one parameter can be selected which is shown as below:
      \arg        NVIC_PRIGROUP_PRE0_SUB4:0 bit for pre-emption priority 4 bits for subpriority
      \arg        NVIC_PRIGROUP_PRE1_SUB3:1 bit for pre-emption priority 3 bits for subpriority
      \arg        NVIC_PRIGROUP_PRE2_SUB2:2 bits for pre-emption priority 2 bits for subpriority
      \arg        NVIC_PRIGROUP_PRE3_SUB1:3 bits for pre-emption priority 1 bit for subpriority
      \arg        NVIC_PRIGROUP_PRE4_SUB0:4 bits for pre-emption priority 0 bit for subpriority
    \param[out] nvic_irq_pre_priority: Preemptive priority value (starting from 0).
    \param[out] nvic_irq_sub_priority: Subpriority value (starting from 0)
    \retval     none
*/
void hal_nvic_priority_get(IRQn_Type nvic_irq, uint32_t nvic_prigroup, \
                           uint32_t *nvic_irq_pre_priority, uint32_t *nvic_irq_sub_priority)
{
    NVIC_DecodePriority(NVIC_GetPriority(nvic_irq), nvic_prigroup, nvic_irq_pre_priority, nvic_irq_sub_priority);
}

/*!
    \brief      get nvic active interrupt
    \param[in]  nvic_irq: the NVIC interrupt request, detailed in IRQn_Type
    \param[out] none
    \retval     status: 0 - Interrupt status is not active; 1 - Interrupt status is active.
*/
uint32_t hal_nvic_active_get(IRQn_Type nvic_irq)
{
    return NVIC_GetActive(nvic_irq);
}

/*!
    \brief      system reset request
    \param[in]  none
    \param[out] none
    \retval     none
*/

void hal_nvic_system_reset(void)
{
    NVIC_SystemReset();
}

/*!
    \brief      set NVIC pend interrupt
    \param[in]  nvic_irq: the NVIC interrupt request, detailed in IRQn_Type
    \param[out] none
    \retval     none
*/
void hal_nvic_pend_irq_set(IRQn_Type nvic_irq)
{
    NVIC_SetPendingIRQ(nvic_irq);
}

/*!
    \brief      disable NVIC pend interrupt
    \param[in]  nvic_irq: the NVIC interrupt request, detailed in IRQn_Type
    \param[out] none
    \retval     none
*/
void hal_nvic_pend_irq_clear(IRQn_Type nvic_irq)
{
    NVIC_ClearPendingIRQ(nvic_irq);
}

/*!
    \brief      get NVIC pend interrupt
    \param[in]  nvic_irq: the NVIC interrupt request, detailed in IRQn_Type
    \param[out] none
    \retval     status: 0 - Function succeeded; 1 - Function failed.
*/
uint32_t hal_nvic_pend_irq_get(IRQn_Type nvic_irq)
{
  return NVIC_GetPendingIRQ(nvic_irq);
}

/*!
    \brief      set the NVIC vector table base address
    \param[in]  nvic_vict_tab: the RAM or FLASH base address
                only one parameter can be selected which is shown as below:
      \arg        NVIC_VECTTAB_RAM: RAM base address
      \arg        NVIC_VECTTAB_FLASH: FLASH base address
    \param[in]  offset: vector table offset
    \param[out] none
    \retval     none
*/
void hals_nvic_vector_table_set(uint32_t nvic_vict_tab, uint32_t offset)
{
    SCB->VTOR = nvic_vict_tab | (offset & NVIC_VECTTAB_OFFSET_MASK);
}
