/*!
    \file    gd32h7xx_hal_mdma.c
    \brief   MDMA driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define MDMA_ADDRESS_MASK           ((uint32_t)0x0000FFFFU)                 /*!< MDMA multi-block update address mask */
#define STAT1_FLAG_MASK             ((uint32_t)0x0000F000U)                 /*!< MDMA_STAT1 flag mask */
#define CHXMBADDRU_DADDRUV_OFFSET   (16U)                                   /*!< destination update address offset */
#define CHXCFG_BTLEN_OFFSET         (18U)                                   /*!< bit offset of BTLEN in MDMA_CHxCFG */
#define CHXBTCFG_BRNUM_OFFSET       (20U)                                   /*!< bit offset of BRNUM in MDMA_CHxBTCFG */

/*!
    \brief      initialize MDMA
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channelx: specify which MDMA channel is initialized
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  init_struct: the data needed to initialize MDMA single data mode
                             and the member values are shown as below:
                  request: mdma request
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_REQUEST_x x is the type of request
                  trans_trig_mode:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_BUFFER_TRANSFER: buffer transfer
      \arg          MDMA_BLOCK_TRANSFER: block transfer
      \arg          MDMA_MULTI_BLOCK_TRANSFER: multi block transfer
      \arg          MDMA_COMPLETE_TRANSFER: complete transfer
                  priority:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_PRIORITY_LOW: priority is low
      \arg          MDMA_PRIORITY_MEDIUM: priority is medium
      \arg          MDMA_PRIORITY_HIGH: priority is high
      \arg          MDMA_PRIORITY_ULTRA_HIGH: priority is ultra high
                  endianness:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_LITTLE_ENDIANNESS: little endianness
      \arg          MDMA_BYTE_ENDIANNESS_EXCHANGE: byte endianess swapping in half word
      \arg          MDMA_HALFWORD_ENDIANNESS_EXCHANGE:half word endianess swapping in word
      \arg          MDMA_WORD_ENDIANNESS_EXCHANGE: word endianess swapping in double word
                  source_inc:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOURCE_INCREASE_DISABLE: no increment
      \arg          MDMA_SOURCE_INCREASE_8BIT: source address pointer is incremented by a byte (8 bits)
      \arg          MDMA_SOURCE_INCREASE_16BIT: source address pointer is incremented by a half word (16 bits)
      \arg          MDMA_SOURCE_INCREASE_32BIT: source address pointer is incremented by a word (32 bits)
      \arg          MDMA_SOURCE_INCREASE_64BIT: source address pointer is incremented by a double word (64 bits)
      \arg          MDMA_SOURCE_DECREASE_8BIT: source address pointer is decremented by a byte (8 bits)
      \arg          MDMA_SOURCE_DECREASE_16BIT: source address pointer is decremented by a half word (16 bits)
      \arg          MDMA_SOURCE_DECREASE_32BIT: source address pointer is decremented by a word (32 bits)
      \arg          MDMA_SOURCE_DECREASE_64BIT: source address pointer is decremented by a double word (64 bits)
                  dest_inc:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DESTINATION_INCREASE_DISABLE: no increment
      \arg          MDMA_DESTINATION_INCREASE_8BIT: destination address pointer is incremented by a byte (8 bits)
      \arg          MDMA_DESTINATION_INCREASE_16BIT: destination address pointer is incremented by a half word (16 bits)
      \arg          MDMA_DESTINATION_INCREASE_32BIT: destination address pointer is incremented by a word (32 bits)
      \arg          MDMA_DESTINATION_INCREASE_64BIT: destination address pointer is incremented by a double word (64 bits)
      \arg          MDMA_DESTINATION_DECREASE_8BIT: destination address pointer is decremented by a byte (8 bits)
      \arg          MDMA_DESTINATION_DECREASE_16BIT: destination address pointer is decremented by a half word (16 bits)
      \arg          MDMA_DESTINATION_DECREASE_32BIT: destination address pointer is decremented by a word (32 bits)
      \arg          MDMA_DESTINATION_DECREASE_64BIT: destination address pointer is decremented by a double word (64 bits)
                  source_data_size:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOURCE_DATASIZE_8BIT: source data size is byte
      \arg          MDMA_SOURCE_DATASIZE_16BIT: source data size is half word
      \arg          MDMA_SOURCE_DATASIZE_32BIT: source data size is word
      \arg          MDMA_SOURCE_DATASIZE_64BIT: source data size is double word
                  dest_data_size:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DESTINATION_DATASIZE_8BIT: destination data size is byte
      \arg          MDMA_DESTINATION_DATASIZE_16BIT: destination data size is half word
      \arg          MDMA_DESTINATION_DATASIZE_32BIT: destination data size is word
      \arg          MDMA_DESTINATION_DATASIZE_64BIT: destination data size is double word
                  data_alignment:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DATAALIGN_PKEN: pack/unpack the source data to match the destination data size
      \arg          MDMA_DATAALIGN_RIGHT: right aligned, padded with 0s (default)
      \arg          MDMA_DATAALIGN_RIGHT_SIGNED: right aligned with sign extended, note: this mode is allowed only if the source data size is smaller than destination data size
      \arg          MDMA_DATAALIGN_LEFT: left aligned, padded with 0s in low bytes position when source data size \
                                       smaller than destination data size, and only high byte of source is written when source data size larger than destination data size
                  buff_trans_len: the number of bytes to be transferred is (buff_trans_len+1), buff_trans_len ranges from 0 to 127
                  source_burst:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOURCE_BURST_SINGLE: single transfer
      \arg          MDMA_SOURCE_BURST_2BEATS: burst 2 beats
      \arg          MDMA_SOURCE_BURST_4BEATS: burst 4 beats
      \arg          MDMA_SOURCE_BURST_8BEATS: burst 8 beats
      \arg          MDMA_SOURCE_BURST_16BEATS: burst 16 beats
      \arg          MDMA_SOURCE_BURST_32BEATS: burst 32 beats
      \arg          MDMA_SOURCE_BURST_64BEATS: burst 64 beats
      \arg          MDMA_SOURCE_BURST_128BEATS: burst 128 beats
                  dest_burst:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DESTINATION_BURST_SINGLE: single transfer
      \arg          MDMA_DESTINATION_BURST_2BEATS: burst 2 beats
      \arg          MDMA_DESTINATION_BURST_4BEATS: burst 4 beats
      \arg          MDMA_DESTINATION_BURST_8BEATS: burst 8 beats
      \arg          MDMA_DESTINATION_BURST_16BEATS: burst 16 beats
      \arg          MDMA_DESTINATION_BURST_32BEATS: burst 32 beats
      \arg          MDMA_DESTINATION_BURST_64BEATS: burst 64 beats
      \arg          MDMA_DESTINATION_BURST_128BEATS: burst 128 beats
                  mask_addr: mask address, ranges from 0x00000000 to 0xFFFFFFFF
                  mask_data: mask data, ranges from 0x00000000 to 0xFFFFFFFF
                  source_addr: source address, ranges from 0x00000000 to 0xFFFFFFFF
                  dest_addr: destination address, ranges from 0x00000000 to 0xFFFFFFFF
                  tbytes_num_in_block: transfer byte number of a block transfer, ranges from 0x00000000 to 0x0001FFFF
                  source_bus:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOURCE_AXI: source bus of channel x is the system bus or AXI bus
      \arg          MDMA_SOURCE_AHB_TCM: source bus of channel x is AHB bus or TCM
                  dest_bus:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DESTINATION_AXI: source bus of channel x is the system bus or AXI bus
      \arg          MDMA_DESTINATION_AHB_TCM: source bus of channel x is AHB bus or TCM
                  bufferable_write_mode:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_BUFFERABLE_WRITE_ENABLE: enable bufferable write mode
      \arg          MDMA_BUFFERABLE_WRITE_DISABLE：disable bufferable write mode
                  link_addr: link address register, ranges from 0x00000000 to 0xFFFFFFFF
                  secure_mode:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SECURE_MODE_DISABLE: secure mode disable
      \arg          MDMA_SECURE_MODE_ENABLE: secure mode enable
                  software_request:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOFTWARE_REQUEST_DISABLE: software request disable
      \arg          MDMA_SOFTWARE_REQUEST_ENABLE: software request enable
    \param[in]  block_init_struct: the data needed to initialize MDMA multi block mode,
                the member values are shown as below:
                  block_num: multi block number, ranges from 0x00000000 to 0x00000FFF
                  saddr_update_val:source address update value, ranges from 0x0000 to 0xFFFF
                  dstaddr_update_val: destination address update value, ranges from 0x0000 to 0xFFFF
                  saddr_update_dir: source address update direction
                  only one parameter can be selected which is shown as below:
      \arg          UPDATE_DIR_INCREASE: MDMA address update increase
      \arg          UPDATE_DIR_DECREASE: MDMA address update decrease
                  dstaddr_update_dir: destination address update direction
                  only one parameter can be selected which is shown as below:
      \arg          UPDATE_DIR_INCREASE: MDMA address update increase
      \arg          UPDATE_DIR_DECREASE: MDMA address update decrease
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL \
                details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_init(hal_mdma_dev_struct *mdma_dev, hal_mdma_channel_enum channelx, hal_mdma_init_struct *init_struct, \
                      hal_mdma_multi_block_parameter_struct *block_init_struct)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == mdma_dev) || (NULL == init_struct) || (NULL == block_init_struct)) {
        HAL_DEBUGE("pointer [mdma_dev] or [init_struct] or [block_init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    mdma_dev->state = HAL_MDMA_STATE_BUSY;

    hals_mdma_channel_disable(channelx);
    mdma_dev->channel = channelx;
    /* configure endianness and priority */
    MDMA_CHXCTL0(channelx) &= ~(MDMA_CHXCTL0_BES | MDMA_CHXCTL0_HWES | MDMA_CHXCTL0_WES | MDMA_CHXCTL0_PRIO | \
                                MDMA_CHXCTL0_SWREQ );
    MDMA_CHXCTL0(channelx) |= (init_struct->endianness | init_struct->priority | init_struct->secure_mode);

    /* configure MDMA transfer width, memory transfer width, channel priority */
    MDMA_CHXCFG(channelx) = (init_struct->data_alignment | init_struct->dest_burst | init_struct->dest_data_size | \
                             init_struct->dest_inc |init_struct->source_burst | init_struct->source_data_size | \
                             init_struct->source_inc | init_struct->trans_trig_mode | \
                             (((init_struct->buff_trans_len - 1U)<< CHXCFG_BTLEN_OFFSET) & MDMA_CHXCFG_BTLEN) | \
                             init_struct->bufferable_write_mode);
    /* configure mask address, mask data */
    MDMA_CHXMADDR(channelx) = init_struct->mask_addr;
    MDMA_CHXMDATA(channelx) = init_struct->mask_data;

    /* configure source address */
    MDMA_CHXSADDR(channelx) = init_struct->source_addr;
    /* configure destination address */
    MDMA_CHXDADDR(channelx) = init_struct->dest_addr;
    /* configure link address */
    MDMA_CHXLADDR(channelx) = init_struct->link_addr;

    /* configure block transfer byte number */
    if(MDMA_BUFFER_TRANSFER == init_struct->trans_trig_mode) {
        MDMA_CHXBTCFG(channelx) = (init_struct->tbytes_num_in_block & MDMA_CHXBTCFG_TBNUM);
    } else if(MDMA_BLOCK_TRANSFER == init_struct->trans_trig_mode) {
        MDMA_CHXBTCFG(channelx) = (init_struct->tbytes_num_in_block & MDMA_CHXBTCFG_TBNUM);
    } else if(MDMA_MULTI_BLOCK_TRANSFER == init_struct->trans_trig_mode) {
        MDMA_CHXBTCFG(channelx) &= ~MDMA_CHXBTCFG_TBNUM;
        MDMA_CHXBTCFG(channelx) |= (init_struct->tbytes_num_in_block & MDMA_CHXBTCFG_TBNUM);
    } else if(MDMA_COMPLETE_TRANSFER == init_struct->trans_trig_mode) {
        MDMA_CHXBTCFG(channelx) &= ~MDMA_CHXBTCFG_TBNUM;
        MDMA_CHXBTCFG(channelx) |= (init_struct->tbytes_num_in_block & MDMA_CHXBTCFG_TBNUM);
    } else {
        /* do nothing */
    }

    /* configure request source */
    MDMA_CHXCFG(channelx) &= ~MDMA_CHXCFG_SWREQMOD;
    if(MDMA_REQUEST_SW == init_struct->request) {
        MDMA_CHXCFG(channelx) |= MDMA_CHXCFG_SWREQMOD;
    } else {
        MDMA_CHXCTL1(channelx) &= ~MDMA_CHXCTL1_TRIGSEL;
        MDMA_CHXCTL1(channelx) |= init_struct->request;
    }

    /* configure bus type for source and destination */
    MDMA_CHXCTL1(channelx) &= ~(MDMA_CHXCTL1_SBSEL | MDMA_CHXCTL1_DBSEL);
    MDMA_CHXCTL1(channelx) |= (init_struct->source_bus | init_struct->dest_bus);

    /* configure multi block data */
    if (MDMA_MULTI_BLOCK_TRANSFER == init_struct->trans_trig_mode) {
        hals_mdma_multi_block_mode_config(channelx, init_struct->tbytes_num_in_block, block_init_struct);
    } else {
        /* do nothing */
    }

    /* change MDMA error state and state */
    mdma_dev->error_state = HAL_MDMA_ERROR_NONE;
    mdma_dev->state       = HAL_MDMA_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the MDMA link node configuration structure with the default values
    \param[in]  none
    \param[out] node: the initialization data needed to initialize link node
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_link_node_para_struct_init(hal_mdma_link_node_parameter_struct *node)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == node) {
        HAL_DEBUGE("pointer [*node] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    node->chxcfg_reg     = 0U;
    node->chxbtcfg_reg   = 0U;
    node->chxsaddr_reg   = 0U;
    node->chxdaddr_reg   = 0U;
    node->chxmbaddru_reg = 0U;
    node->chxladdr_reg   = 0U;
    node->chxctl1_reg    = 0U;
    node->reserved       = 0U;
    node->chxmaddr_reg   = 0U;
    node->chxmdata_reg   = 0U;

    return HAL_ERR_NONE;
}

/*!
    \brief      create MDMA link list node
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] node: link node structure to create, members of the structure are shown as below:
                  chxcfg_reg: channel configure register
                  chxbtcfg_reg: channel block transfer configure register
                  chxsaddr_reg: channel source address register
                  chxdaddr_reg: channel destination address register
                  chxmbaddru_reg: channel multi-block address update register
                  chxladdr_reg: channel link address register
                  chxctl1_reg: channel control register 1
                  chxmaddr_reg: channel mask address register
                  chxmdata_reg: channel mask data register
    \param[in]  block_init_struct: mdma muti block structure, members of the structure and the member values are shown as below:
                  block_num: multi-block number, ranges from 0x00000000 to 0x00000FFF
                  saddr_update_val: source address update value, ranges from 0x0000 to 0xFFFF
                  dstaddr_update_val: destination address update value, ranges from 0x0000 to 0xFFFF
                  saddr_update_dir: source address update direction
                  only one parameter can be selected which is shown as below:
      \arg          UPDATE_DIR_INCREASE: MDMA address update increase
      \arg          UPDATE_DIR_DECREASE: MDMA address update decrease
                  dstaddr_update_dir: destination address update direction
                  only one parameter can be selected which is shown as below:
      \arg          UPDATE_DIR_INCREASE: MDMA address update increase
      \arg          UPDATE_DIR_DECREASE: MDMA address update decrease
    \param[in]  init_struct: the data needed to initialize MDMA single data mode,
                members of the structure and the member values are shown as below:
                  request: mdma request
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_REQUEST_x x is the type of request
                  trans_trig_mode:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_BUFFER_TRANSFER: buffer transfer
      \arg          MDMA_BLOCK_TRANSFER: block transfer
      \arg          MDMA_MULTI_BLOCK_TRANSFER: multi block transfer
      \arg          MDMA_COMPLETE_TRANSFER: complete transfer
                  priority:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_PRIORITY_LOW: priority is low
      \arg          MDMA_PRIORITY_MEDIUM: priority is medium
      \arg          MDMA_PRIORITY_HIGH: priority is high
      \arg          MDMA_PRIORITY_ULTRA_HIGH: priority is ultra high
                  endianness:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_LITTLE_ENDIANNESS: little endianness
      \arg          MDMA_BYTE_ENDIANNESS_EXCHANGE: byte endianess swapping in half word
      \arg          MDMA_HALFWORD_ENDIANNESS_EXCHANGE:half word endianess swapping in word
      \arg          MDMA_WORD_ENDIANNESS_EXCHANGE: word endianess swapping in double word
                  source_inc:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOURCE_INCREASE_DISABLE: no increment
      \arg          MDMA_SOURCE_INCREASE_8BIT: source address pointer is incremented by a byte (8 bits)
      \arg          MDMA_SOURCE_INCREASE_16BIT: source address pointer is incremented by a half word (16 bits)
      \arg          MDMA_SOURCE_INCREASE_32BIT: source address pointer is incremented by a word (32 bits)
      \arg          MDMA_SOURCE_INCREASE_64BIT: source address pointer is incremented by a double word (64 bits)
      \arg          MDMA_SOURCE_DECREASE_8BIT: source address pointer is decremented by a byte (8 bits)
      \arg          MDMA_SOURCE_DECREASE_16BIT: source address pointer is decremented by a half word (16 bits)
      \arg          MDMA_SOURCE_DECREASE_32BIT: source address pointer is decremented by a word (32 bits)
      \arg          MDMA_SOURCE_DECREASE_64BIT: source address pointer is decremented by a double word (64 bits)
                  dest_inc:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DESTINATION_INCREASE_DISABLE: no increment
      \arg          MDMA_DESTINATION_INCREASE_8BIT: destination address pointer is incremented by a byte (8 bits)
      \arg          MDMA_DESTINATION_INCREASE_16BIT: destination address pointer is incremented by a half word (16 bits)
      \arg          MDMA_DESTINATION_INCREASE_32BIT: destination address pointer is incremented by a word (32 bits)
      \arg          MDMA_DESTINATION_INCREASE_64BIT: destination address pointer is incremented by a double word (64 bits)
      \arg          MDMA_DESTINATION_DECREASE_8BIT: destination address pointer is decremented by a byte (8 bits)
      \arg          MDMA_DESTINATION_DECREASE_16BIT: destination address pointer is decremented by a half word (16 bits)
      \arg          MDMA_DESTINATION_DECREASE_32BIT: destination address pointer is decremented by a word (32 bits)
      \arg          MDMA_DESTINATION_DECREASE_64BIT: destination address pointer is decremented by a double word (64 bits)
                  source_data_size:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOURCE_DATASIZE_8BIT: source data size is byte
      \arg          MDMA_SOURCE_DATASIZE_16BIT: source data size is half word
      \arg          MDMA_SOURCE_DATASIZE_32BIT: source data size is word
      \arg          MDMA_SOURCE_DATASIZE_64BIT: source data size is double word
                  dest_data_size:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DESTINATION_DATASIZE_8BIT: destination data size is byte
      \arg          MDMA_DESTINATION_DATASIZE_16BIT: destination data size is half word
      \arg          MDMA_DESTINATION_DATASIZE_32BIT: destination data size is word
      \arg          MDMA_DESTINATION_DATASIZE_64BIT: destination data size is double word
                  data_alignment:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DATAALIGN_PKEN: pack/unpack the source data to match the destination data size
      \arg          MDMA_DATAALIGN_RIGHT: right aligned, padded with 0s (default)
      \arg          MDMA_DATAALIGN_RIGHT_SIGNED: right aligned with sign extended, note: this mode is allowed only if
                                                 the source data size is smaller than destination data size
      \arg          MDMA_DATAALIGN_LEFT: left aligned, padded with 0s in low bytes position when source data size
                                       smaller than destination data size, and only high byte of source is written when
                                       source data size larger than destination data size
                  buff_trans_len: the number of bytes to be transferred is (buff_trans_len+1), buff_trans_len ranges from 0 to 127
                  source_burst:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOURCE_BURST_SINGLE: single transfer
      \arg          MDMA_SOURCE_BURST_2BEATS: burst 2 beats
      \arg          MDMA_SOURCE_BURST_4BEATS: burst 4 beats
      \arg          MDMA_SOURCE_BURST_8BEATS: burst 8 beats
      \arg          MDMA_SOURCE_BURST_16BEATS: burst 16 beats
      \arg          MDMA_SOURCE_BURST_32BEATS: burst 32 beats
      \arg          MDMA_SOURCE_BURST_64BEATS: burst 64 beats
      \arg          MDMA_SOURCE_BURST_128BEATS: burst 128 beats
                  dest_burst:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DESTINATION_BURST_SINGLE: single transfer
      \arg          MDMA_DESTINATION_BURST_2BEATS: burst 2 beats
      \arg          MDMA_DESTINATION_BURST_4BEATS: burst 4 beats
      \arg          MDMA_DESTINATION_BURST_8BEATS: burst 8 beats
      \arg          MDMA_DESTINATION_BURST_16BEATS: burst 16 beats
      \arg          MDMA_DESTINATION_BURST_32BEATS: burst 32 beats
      \arg          MDMA_DESTINATION_BURST_64BEATS: burst 64 beats
      \arg          MDMA_DESTINATION_BURST_128BEATS: burst 128 beats
                  mask_addr: mask address, ranges from 0x00000000 to 0xFFFFFFFF
                  mask_data: mask data, ranges from 0x00000000 to 0xFFFFFFFF
                  source_addr: source address, ranges from 0x00000000 to 0xFFFFFFFF
                  dest_addr: destination address, ranges from 0x00000000 to 0xFFFFFFFF
                  tbytes_num_in_block: transfer byte number of a block transfer, ranges from 0x00000000 to 0x0001FFFF
                  source_bus:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOURCE_AXI: source bus of channel x is the system bus or AXI bus
      \arg          MDMA_SOURCE_AHB_TCM: source bus of channel x is AHB bus or TCM
                  dest_bus:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_DESTINATION_AXI: source bus of channel x is the system bus or AXI bus
      \arg          MDMA_DESTINATION_AHB_TCM: source bus of channel x is AHB bus or TCM
                  bufferable_write_mode:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_BUFFERABLE_WRITE_ENABLE: enable bufferable write mode
      \arg          MDMA_BUFFERABLE_WRITE_DISABLE：disable bufferable write mode
                  link_addr: link address register, ranges from 0x00000000 to 0xFFFFFFFF
                  secure_mode:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SECURE_MODE_DISABLE: secure mode disable
      \arg          MDMA_SECURE_MODE_ENABLE: secure mode enable
                  software_request:
                  only one parameter can be selected which is shown as below:
      \arg          MDMA_SOFTWARE_REQUEST_DISABLE: software request disable
      \arg          MDMA_SOFTWARE_REQUEST_ENABLE: software request enable
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_node_create(hal_mdma_dev_struct *mdma_dev, hal_mdma_link_node_parameter_struct *node, \
                             hal_mdma_multi_block_parameter_struct *block_init_struct, hal_mdma_init_struct *init_struct)
{
    uint32_t cfg         = 0U;
    uint32_t blockoffset = 0U;

    /* check the parameter*/
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if((NULL == mdma_dev) || (NULL == node) || (NULL == block_init_struct) || (NULL == init_struct)) {
        HAL_DEBUGE("pointer [mdma_dev] or [node] or [block_init_struct] or [init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    mdma_dev->state = HAL_MDMA_STATE_BUSY;
    hal_mdma_link_node_para_struct_init(node);

    /* configure channel configure register */
    cfg = (init_struct->data_alignment | init_struct->dest_burst | init_struct->dest_data_size | init_struct->dest_inc | \
           init_struct->source_burst | init_struct->source_data_size | init_struct->source_inc | init_struct->trans_trig_mode | \
           (((init_struct->buff_trans_len - 1U) << CHXCFG_BTLEN_OFFSET) & MDMA_CHXCFG_BTLEN) | init_struct->bufferable_write_mode);
    node->chxcfg_reg = cfg;

    /* configure channel request source */
    if(MDMA_REQUEST_SW == init_struct->request) {
        node->chxcfg_reg |= MDMA_CHXCFG_SWREQMOD;
    } else {
        node->chxctl1_reg &= ~MDMA_CHXCTL1_TRIGSEL;
        node->chxctl1_reg |= init_struct->request;
    }
    /* configure bus type for source and destination */
    node->chxctl1_reg &= ~(MDMA_CHXCTL1_SBSEL | MDMA_CHXCTL1_DBSEL);
    node->chxctl1_reg |= (init_struct->source_bus | init_struct->dest_bus);

    /* configure channel block transfer configure register */
    cfg = (((block_init_struct->block_num << CHXBTCFG_BRNUM_OFFSET) & MDMA_CHXBTCFG_BRNUM) | \
            (init_struct->tbytes_num_in_block & MDMA_CHXBTCFG_TBNUM));
    node->chxbtcfg_reg = cfg;

    /* configure source address, destination address, mask address and mask data */
    node->chxsaddr_reg = init_struct->source_addr;
    node->chxdaddr_reg = init_struct->dest_addr;
    node->chxmaddr_reg = init_struct->mask_addr;
    node->chxmdata_reg = init_struct->mask_data;
    node->chxmbaddru_reg &= ~(MDMA_CHXMBADDRU_SADDRUV | MDMA_CHXMBADDRU_DADDRUV);

    /* if block source address offset is negative, set the block repeat source address update mode to decrement */
    if(UPDATE_DIR_DECREASE == block_init_struct->saddr_update_dir) {
        node->chxbtcfg_reg |= MDMA_CHXBTCFG_SADDRUM;
        /* write new chxmbaddru register value: source repeat block offset */
        blockoffset = (uint32_t)block_init_struct->saddr_update_val;
        node->chxmbaddru_reg |= (blockoffset & MDMA_ADDRESS_MASK);
    } else {
        node->chxbtcfg_reg &= ~MDMA_CHXBTCFG_SADDRUM;
        /* write new chxmbaddru register value: source repeat block offset */
        node->chxmbaddru_reg |= (((uint32_t)block_init_struct->saddr_update_val) & MDMA_ADDRESS_MASK);
    }

    if(UPDATE_DIR_DECREASE == block_init_struct->dstaddr_update_dir) {
        node->chxbtcfg_reg |= MDMA_CHXBTCFG_DADDRUM;
        /* write new chxmbaddru register value: destination repeat block offset */
        blockoffset = (uint32_t)block_init_struct->dstaddr_update_val;
        node->chxmbaddru_reg |= ((uint32_t)(blockoffset & MDMA_ADDRESS_MASK) << CHXMBADDRU_DADDRUV_OFFSET);
    } else {
        node->chxbtcfg_reg &= ~MDMA_CHXBTCFG_DADDRUM;
        /* write new chxmbaddru register value: destination repeat block offset */
        node->chxmbaddru_reg |= ((((uint32_t)block_init_struct->dstaddr_update_val) & MDMA_ADDRESS_MASK) << CHXMBADDRU_DADDRUV_OFFSET);
    }

    /* configure channel link address */
    node->chxladdr_reg = 0U;
    node->reserved     = 0U;
    mdma_dev->state    = HAL_MDMA_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      MDMA add node to link list
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  pre_node: previous structure node pointer, members of the structure and the member values are shown as below:
                  chxcfg_reg: channel configure register
                  chxbtcfg_reg: channel block transfer configure register
                  chxsaddr_reg: channel source address register
                  chxdaddr_reg: channel destination address register
                  chxmbaddru_reg: channel multi-block address update register
                  chxladdr_reg: channel link address register
                  chxctl1_reg: channel control register 1
                  chxmaddr_reg: channel mask address register
                  chxmdata_reg: channel mask data register
    \param[in]  new_node: new node pointer, members of the structure and the member values are shown as below:
                  chxcfg_reg: channel configure register
                  chxbtcfg_reg: channel block transfer configure register
                  chxsaddr_reg: channel source address register
                  chxdaddr_reg: channel destination address register
                  chxmbaddru_reg: channel multi-block address update register
                  chxladdr_reg: channel link address register
                  chxctl1_reg: channel control register 1
                  chxmaddr_reg: channel mask address register
                  chxmdata_reg: channel mask data register
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_node_add(hal_mdma_dev_struct *mdma_dev, hal_mdma_link_node_parameter_struct *pre_node, \
                          hal_mdma_link_node_parameter_struct *new_node)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t temp_node = 0U;

    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if((NULL == new_node) || (NULL == mdma_dev)) {
        HAL_DEBUGE("pointer [new_node] or [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_MDMA_STATE_READY == mdma_dev->state) {
        mdma_dev->state = HAL_MDMA_STATE_BUSY;
        if(NULL == mdma_dev->first_node) {
            mdma_dev->first_node = new_node;
            mdma_dev->last_node = new_node;
        } else if(mdma_dev->last_node == pre_node) {
            pre_node->chxladdr_reg = (uint32_t)new_node;
            mdma_dev->last_node = new_node;
        } else {
            temp_node = pre_node->chxladdr_reg;
            pre_node->chxladdr_reg = (uint32_t)new_node;
            new_node->chxladdr_reg = temp_node;
        }
    } else {
        error_code = HAL_ERR_BUSY;
    }

    mdma_dev->state = HAL_MDMA_STATE_READY;

    return error_code;
}

/*!
    \brief      make the linked list circular by connecting the last node to the first
    \param[in]  mdma_dev: mdma device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_list_circularmode_enable(hal_mdma_dev_struct *mdma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == mdma_dev)) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_MDMA_STATE_READY == mdma_dev->state) {
        mdma_dev->state = HAL_MDMA_STATE_BUSY;
        if(NULL != mdma_dev->first_node) {
            mdma_dev->last_node->chxladdr_reg = (uint32_t)mdma_dev->first_node;
        } else {
            mdma_dev->state = HAL_MDMA_STATE_READY;
            HAL_DEBUGE("there are no nodes in this list");
            error_code = HAL_ERR_ADDRESS;
        }
    } else {
        error_code = HAL_ERR_BUSY;
    }

    mdma_dev->state = HAL_MDMA_STATE_READY;

    return error_code;
}

/*!
    \brief      deinitialize MDMA
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_deinit(hal_mdma_dev_struct *mdma_dev)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == mdma_dev) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    mdma_dev->state = HAL_MDMA_STATE_BUSY;

    /* reset MDMA */
    hal_rcu_periph_reset_enable(RCU_MDMARST);
    hal_rcu_periph_reset_disable(RCU_MDMARST);

    /* change MDMA error state and state */
    mdma_dev->error_state = HAL_MDMA_ERROR_NONE;
    mdma_dev->state       = HAL_MDMA_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      MDMA disconnect link list node
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  pre_node: previous structure node pointer, members of the structure and the member values are shown as below:
                  chxcfg_reg: channel configure register
                  chxbtcfg_reg: channel block transfer configure register
                  chxsaddr_reg: channel source address register
                  chxdaddr_reg: channel destination address register
                  chxmbaddru_reg: channel multi-block address update register
                  chxladdr_reg: channel link address register
                  chxctl1_reg: channel control register 1
                  chxmaddr_reg: channel mask address register
                  chxmdata_reg: channel mask data register
    \param[in]  unused_node: unused link list node pointer, members of the structure and the member values are shown as below:
                  chxcfg_reg: channel configure register
                  chxbtcfg_reg: channel block transfer configure register
                  chxsaddr_reg: channel source address register
                  chxdaddr_reg: channel destination address register
                  chxmbaddru_reg: channel multi-block address update register
                  chxladdr_reg: channel link address register
                  chxctl1_reg: channel control register 1
                  chxmaddr_reg: channel mask address register
                  chxmdata_reg: channel mask data register
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_node_delete(hal_mdma_dev_struct *mdma_dev, hal_mdma_link_node_parameter_struct *pre_node, \
                             hal_mdma_link_node_parameter_struct *unused_node)
{
    int32_t error_code                        = HAL_ERR_NONE;
    uint32_t temp_node                        = 0U;
    hal_mdma_link_node_parameter_struct *node = NULL;

    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if((NULL == mdma_dev) || (NULL == unused_node)) {
        HAL_DEBUGE("pointer [mdma_dev] or [unused_node] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_MDMA_STATE_READY == mdma_dev->state) {
        mdma_dev->state = HAL_MDMA_STATE_BUSY;
        if(NULL == pre_node) {
            if((mdma_dev->first_node == unused_node) && (0U != mdma_dev->first_node->chxladdr_reg)) {
                node = (hal_mdma_link_node_parameter_struct*)mdma_dev->first_node->chxladdr_reg;
                mdma_dev->first_node = node;
            } else if((mdma_dev->first_node == unused_node) && (0U == mdma_dev->first_node->chxladdr_reg)) {
                mdma_dev->first_node = NULL;
                mdma_dev->last_node = NULL;
            } else {
                HAL_DEBUGE("when the [unused_node] is not first node, the [pre_node] must not be NULL");
                error_code = HAL_ERR_ADDRESS;
            }
        } else {
            temp_node = unused_node->chxladdr_reg;
            pre_node->chxladdr_reg = temp_node;
            if(0U == temp_node) {
                mdma_dev->last_node = NULL;
            } else {
                /* do nothing */
            }
        }
    } else {
        error_code = HAL_ERR_BUSY;
    }

    mdma_dev->state = HAL_MDMA_STATE_READY;

    return error_code;
}

/*!
    \brief      initialize the MDMA structure with the default values
    \param[in]  hal_struct_type: the argument could be selected from enumeration <hal_mdma_struct_type_enum>
    \param[out] p_struct: pointer to MDMA structure that contains the configuration information
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_struct_init(hal_mdma_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_MDMA_INIT_STRUCT:
        /* initialize MDMA initialization structure with the default values */
        ((hal_mdma_init_struct *)p_struct)->request                                     = 0U;
        ((hal_mdma_init_struct *)p_struct)->trans_trig_mode                             = MDMA_BUFFER_TRANSFER;
        ((hal_mdma_init_struct *)p_struct)->priority                                    = MDMA_PRIORITY_LOW;
        ((hal_mdma_init_struct *)p_struct)->endianness                                  = MDMA_LITTLE_ENDIANNESS;
        ((hal_mdma_init_struct *)p_struct)->source_inc                                  = MDMA_SOURCE_INCREASE_DISABLE;
        ((hal_mdma_init_struct *)p_struct)->dest_inc                                    = MDMA_DESTINATION_INCREASE_DISABLE;
        ((hal_mdma_init_struct *)p_struct)->source_data_size                            = MDMA_SOURCE_DATASIZE_8BIT;
        ((hal_mdma_init_struct *)p_struct)->dest_data_size                              = MDMA_DESTINATION_DATASIZE_8BIT;
        ((hal_mdma_init_struct *)p_struct)->data_alignment                              = MDMA_DATAALIGN_RIGHT;
        ((hal_mdma_init_struct *)p_struct)->buff_trans_len                              = 1U;
        ((hal_mdma_init_struct *)p_struct)->source_burst                                = MDMA_SOURCE_BURST_SINGLE;
        ((hal_mdma_init_struct *)p_struct)->dest_burst                                  = MDMA_DESTINATION_BURST_SINGLE;
        ((hal_mdma_init_struct *)p_struct)->mask_addr                                   = 0x00000000U;
        ((hal_mdma_init_struct *)p_struct)->mask_data                                   = 0x00000000U;
        ((hal_mdma_init_struct *)p_struct)->source_addr                                 = 0x00000000U;
        ((hal_mdma_init_struct *)p_struct)->dest_addr                                   = 0x00000000U;
        ((hal_mdma_init_struct *)p_struct)->tbytes_num_in_block                         = 0U;
        ((hal_mdma_init_struct *)p_struct)->source_bus                                  = MDMA_SOURCE_AXI;
        ((hal_mdma_init_struct *)p_struct)->dest_bus                                    = MDMA_DESTINATION_AXI;
        ((hal_mdma_init_struct *)p_struct)->bufferable_write_mode                       = MDMA_BUFFERABLE_WRITE_DISABLE;
        ((hal_mdma_init_struct *)p_struct)->link_addr                                   = 0U;
        ((hal_mdma_init_struct *)p_struct)->secure_mode                                 = MDMA_SECURE_MODE_DISABLE;
        ((hal_mdma_init_struct *)p_struct)->software_request                            = MDMA_SOFTWARE_REQUEST_DISABLE;
        break;
    case HAL_MDMA_LINK_NODE_STRUCT:
        /* initialize MDMA link node structure with the default values */
        ((hal_mdma_link_node_parameter_struct *)p_struct)->chxcfg_reg                   = 0U;
        ((hal_mdma_link_node_parameter_struct *)p_struct)->chxbtcfg_reg                 = 0U;
        ((hal_mdma_link_node_parameter_struct *)p_struct)->chxsaddr_reg                 = 0U;
        ((hal_mdma_link_node_parameter_struct *)p_struct)->chxdaddr_reg                 = 0U;
        ((hal_mdma_link_node_parameter_struct *)p_struct)->chxmbaddru_reg               = 0U;
        ((hal_mdma_link_node_parameter_struct *)p_struct)->chxladdr_reg                 = 0U;
        ((hal_mdma_link_node_parameter_struct *)p_struct)->chxctl1_reg                  = 0U;
        ((hal_mdma_link_node_parameter_struct *)p_struct)->reserved                     = 0U;
        ((hal_mdma_link_node_parameter_struct *)p_struct)->chxmaddr_reg                 = 0U;
        ((hal_mdma_link_node_parameter_struct *)p_struct)->chxmdata_reg                 = 0U;
        break;
    case HAL_MDMA_MULTI_BLOCK_STRUCT:
        /* initialize MDMA multi block structure with the default values */
        ((hal_mdma_multi_block_parameter_struct *)p_struct)->block_num                  = 0U;
        ((hal_mdma_multi_block_parameter_struct *)p_struct)->saddr_update_val           = 0U;
        ((hal_mdma_multi_block_parameter_struct *)p_struct)->dstaddr_update_val         = 0U;
        ((hal_mdma_multi_block_parameter_struct *)p_struct)->saddr_update_dir           = UPDATE_DIR_INCREASE;
        ((hal_mdma_multi_block_parameter_struct *)p_struct)->dstaddr_update_dir         = UPDATE_DIR_INCREASE;
        break;
    case HAL_MDMA_IRQ_STRUCT:
        /* initialize MDMA IRQ structure with the default values */
        ((hal_mdma_irq_struct *)p_struct)->mdma_error_handle                            = NULL;
        ((hal_mdma_irq_struct *)p_struct)->mdma_complete_handle                         = NULL;
        ((hal_mdma_irq_struct *)p_struct)->mdma_multi_block_complete_handle             = NULL;
        ((hal_mdma_irq_struct *)p_struct)->mdma_block_complete_handle                   = NULL;
        ((hal_mdma_irq_struct *)p_struct)->mdma_buffer_complete_handle                  = NULL;
        break;
    case HAL_MDMA_DEV_STRUCT:
        /* initialize MDMA device structure with the default values */
        ((hal_mdma_dev_struct *)p_struct)->channel                                      = MDMA_CH0;
        ((hal_mdma_dev_struct *)p_struct)->mdma_irq.mdma_error_handle                   = NULL;
        ((hal_mdma_dev_struct *)p_struct)->mdma_irq.mdma_complete_handle                = NULL;
        ((hal_mdma_dev_struct *)p_struct)->mdma_irq.mdma_multi_block_complete_handle    = NULL;
        ((hal_mdma_dev_struct *)p_struct)->mdma_irq.mdma_block_complete_handle          = NULL;
        ((hal_mdma_dev_struct *)p_struct)->mdma_irq.mdma_buffer_complete_handle         = NULL;
        ((hal_mdma_dev_struct *)p_struct)->first_node                                   = NULL;
        ((hal_mdma_dev_struct *)p_struct)->last_node                                    = NULL;
        ((hal_mdma_dev_struct *)p_struct)->error_state                                  = HAL_MDMA_ERROR_NONE;
        ((hal_mdma_dev_struct *)p_struct)->state                                        = HAL_MDMA_STATE_NONE;
        ((hal_mdma_dev_struct *)p_struct)->mutex                                        = HAL_MUTEX_UNLOCKED;
        ((hal_mdma_dev_struct *)p_struct)->priv                                         = NULL;
        ((hal_mdma_dev_struct *)p_struct)->p_periph                                     = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        error_code = HAL_ERR_ADDRESS;
        break;
    }

    return error_code;
}

/*!
    \brief      MDMA interrupt handler content function,which is merely used in mdma_handler
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_irq(hal_mdma_dev_struct *mdma_dev)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if(NULL == mdma_dev) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* transfer error interrupt handler */
    if(SET == hals_mdma_interrupt_flag_get(mdma_dev->channel, MDMA_INT_FLAG_ERR)) {
        hals_mdma_interrupt_disable(mdma_dev->channel, (MDMA_INT_ERR | MDMA_INT_CHTC | \
                                    MDMA_INT_MBTC | MDMA_INT_BTC | MDMA_INT_TC));
        hals_mdma_interrupt_flag_clear(mdma_dev->channel, MDMA_INT_FLAG_ERR);
        if(mdma_dev->mdma_irq.mdma_error_handle != NULL) {
            mdma_dev->mdma_irq.mdma_error_handle(mdma_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* multi-block transfer complete interrupt handler */
    if(SET == hals_mdma_interrupt_flag_get(mdma_dev->channel, MDMA_INT_FLAG_MBTCF)) {
        hals_mdma_interrupt_flag_clear(mdma_dev->channel, MDMA_INT_FLAG_MBTCF);
        if(mdma_dev->mdma_irq.mdma_multi_block_complete_handle != NULL) {
            mdma_dev->mdma_irq.mdma_multi_block_complete_handle(mdma_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* block transfer complete interrupt handler */
    if(SET == hals_mdma_interrupt_flag_get(mdma_dev->channel, MDMA_INT_FLAG_BTCF)) {
        hals_mdma_interrupt_flag_clear(mdma_dev->channel, MDMA_INT_FLAG_BTCF);
        if(mdma_dev->mdma_irq.mdma_block_complete_handle != NULL) {
            mdma_dev->mdma_irq.mdma_block_complete_handle(mdma_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* buffer transfer complete interrupt handler */
    if(SET == hals_mdma_interrupt_flag_get(mdma_dev->channel, MDMA_INT_FLAG_TCF)) {

        if(HAL_MDMA_STATE_ABORT ==  mdma_dev->state) {
            /* Change the DMA state */
            mdma_dev->state = HAL_MDMA_STATE_READY;

            if(mdma_dev->mdma_irq.mdma_abort_handle != NULL) {
                mdma_dev->mdma_irq.mdma_abort_handle(mdma_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        hals_mdma_interrupt_flag_clear(mdma_dev->channel, MDMA_INT_FLAG_TCF);
        if(mdma_dev->mdma_irq.mdma_buffer_complete_handle != NULL) {
            mdma_dev->mdma_irq.mdma_buffer_complete_handle(mdma_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* channel transfer complete interrupt handler */
    if(SET == hals_mdma_interrupt_flag_get(mdma_dev->channel, MDMA_INT_FLAG_CHTCF)) {
        hals_mdma_interrupt_flag_clear(mdma_dev->channel, MDMA_INT_FLAG_CHTCF);
        mdma_dev->state = HAL_MDMA_STATE_READY;
        if(mdma_dev->mdma_irq.mdma_complete_handle != NULL) {
            mdma_dev->mdma_irq.mdma_complete_handle(mdma_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock MDMA */
    mdma_dev->state = HAL_MDMA_STATE_READY;
    HAL_UNLOCK(mdma_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to MDMA interrupt callback function pointer structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_irq_handle_set(hal_mdma_dev_struct *mdma_dev, hal_mdma_irq_struct *p_irq)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if((NULL == mdma_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("parameter [mdma_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable MDMA channel */
    hals_mdma_channel_disable(mdma_dev->channel);

    /* set the handler pointer  */
    if(NULL != p_irq) {
        mdma_dev->mdma_irq.mdma_error_handle                =  p_irq->mdma_error_handle;
        mdma_dev->mdma_irq.mdma_complete_handle             = p_irq->mdma_complete_handle;
        mdma_dev->mdma_irq.mdma_multi_block_complete_handle = p_irq->mdma_multi_block_complete_handle;
        mdma_dev->mdma_irq.mdma_block_complete_handle       = p_irq->mdma_block_complete_handle;
        mdma_dev->mdma_irq.mdma_buffer_complete_handle      = p_irq->mdma_buffer_complete_handle;
        mdma_dev->mdma_irq.mdma_abort_handle                = p_irq->mdma_abort_handle;
    } else {
        /* do nothing */
    }

    /* enable channel transfer complete interrupt */
    if(NULL != mdma_dev->mdma_irq.mdma_complete_handle) {
        hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_CHTC);
    } else {
        hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_CHTC);
    }

    /* enable channel transfer error interrupt */
    if(NULL != mdma_dev->mdma_irq.mdma_error_handle) {
        hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_ERR);
    } else {
        hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_ERR);
    }

    /* set the multi-block transfer complete handler pointer and enable the channel multi-block transfer complete  interrupt */
    if(NULL != mdma_dev->mdma_irq.mdma_multi_block_complete_handle) {
        hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_MBTC);
    } else {
        hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_MBTC);
    }

    /* set block transfer complete handler pointer and enable the channel block transfer complete interrupt */
    if(NULL != mdma_dev->mdma_irq.mdma_block_complete_handle) {
        hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_BTC);
    } else {
        hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_BTC);
    }

    /* set the buffer transfer complete handler pointer and enable the channel buffer transfer complete interrupt */
    if(NULL != mdma_dev->mdma_irq.mdma_buffer_complete_handle) {
        hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_TC);
    } else {
        hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_TC);
    }

    /* set the abort callback */
    if(NULL != p_irq->mdma_abort_handle) {
        mdma_dev->mdma_irq.mdma_abort_handle = p_irq->mdma_abort_handle;
    } else {
        mdma_dev->mdma_irq.mdma_abort_handle = NULL;
    }

    /* enable MDMA channel */
    hals_mdma_channel_enable(mdma_dev->channel);

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_irq_handle_all_reset(hal_mdma_dev_struct *mdma_dev)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if(NULL == mdma_dev) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* MDMA interrupt handler reset */
    mdma_dev->mdma_irq.mdma_error_handle                = NULL;
    mdma_dev->mdma_irq.mdma_complete_handle             = NULL;
    mdma_dev->mdma_irq.mdma_multi_block_complete_handle = NULL;
    mdma_dev->mdma_irq.mdma_block_complete_handle       = NULL;
    mdma_dev->mdma_irq.mdma_buffer_complete_handle      = NULL;
    mdma_dev->mdma_irq.mdma_abort_handle                = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      starts the MDMA transfer with interrupts enabled
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  src_addr: source memory Buffer address
    \param[in]  dst_addr: destination memory Buffer address
    \param[in]  tbnum: the length of a block transfer in bytes
    \param[in]  block_num: the number of a blocks to be transfer
    \param[in]  p_irq: point to MDMA interrupt callback function pointer structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY \
                details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_start_interrupt(hal_mdma_dev_struct *mdma_dev, uint32_t src_addr, uint32_t dst_addr, \
                                 uint32_t tbnum, uint32_t block_num, hal_mdma_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if((NULL == mdma_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("parameter [mdma_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(65536U < tbnum) {
        HAL_DEBUGE("parameter [tbnum] value is illegal");
        return HAL_ERR_VAL;
    }
    if(4095U < block_num) {
        HAL_DEBUGE("parameter [block_num] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock MDMA */
    HAL_LOCK(mdma_dev);
    if(HAL_MDMA_STATE_READY == mdma_dev->state) {
        mdma_dev->state = HAL_MDMA_STATE_BUSY;
        mdma_dev->error_state = HAL_MDMA_ERROR_NONE;

        /* disable mdma channel */
        hals_mdma_channel_disable(mdma_dev->channel);

        /* enable channel transfer complete interrupt */
        if(NULL != p_irq->mdma_complete_handle) {
            mdma_dev->mdma_irq.mdma_complete_handle = p_irq->mdma_complete_handle;
            hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_CHTC);
        } else {
            hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_CHTC);
        }

        /* enable channel transfer error interrupt */
        if(NULL != p_irq->mdma_error_handle) {
            mdma_dev->mdma_irq.mdma_error_handle =  p_irq->mdma_error_handle;
            hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_ERR);
        } else {
            hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_ERR);
        }

        /* set the multi-block transfer complete handler pointer and enable the channel multi-block transfer complete  interrupt */
        if(NULL !=p_irq->mdma_multi_block_complete_handle) {
            mdma_dev->mdma_irq.mdma_multi_block_complete_handle = p_irq->mdma_multi_block_complete_handle;
            hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_MBTC);
        } else {
            hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_MBTC);
        }

        /* set block transfer complete handler pointer and enable the channel block transfer complete interrupt */
        if(NULL != p_irq->mdma_block_complete_handle) {
            mdma_dev->mdma_irq.mdma_block_complete_handle  = p_irq->mdma_block_complete_handle;
            hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_BTC);
        } else {
            hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_BTC);
        }

        /* set the buffer transfer complete handler pointer and enable the channel buffer transfer complete interrupt */
        if(NULL != p_irq->mdma_buffer_complete_handle) {
            mdma_dev->mdma_irq.mdma_buffer_complete_handle = p_irq->mdma_buffer_complete_handle;
            hals_mdma_interrupt_enable(mdma_dev->channel, MDMA_INT_TC);
        } else {
            hals_mdma_interrupt_disable(mdma_dev->channel, MDMA_INT_TC);
        }

        /* set the abort callback */
        if(NULL != p_irq->mdma_buffer_complete_handle) {
            mdma_dev->mdma_irq.mdma_abort_handle = p_irq->mdma_abort_handle;
        } else {
            mdma_dev->mdma_irq.mdma_abort_handle = NULL;
        }

        /* clear all flags */
        hals_mdma_flag_clear(mdma_dev->channel, (MDMA_FLAG_ERR | MDMA_FLAG_CHTCF | MDMA_FLAG_MBTCF | \
                             MDMA_FLAG_BTCF | MDMA_FLAG_TCF | MDMA_FLAG_REQAF | MDMA_FLAG_LDTERR | \
                             MDMA_FLAG_MDTERR | MDMA_FLAG_ASERR | MDMA_FLAG_BZERR));

        /* configure the source, destination address and the data length */
        hals_mdma_source_address_config(mdma_dev->channel, src_addr);
        hals_mdma_destination_address_config(mdma_dev->channel, dst_addr);
        hals_mdma_byte_block_config(mdma_dev->channel, tbnum);
        hals_mdma_block_count_config(mdma_dev->channel, block_num);

        /* enable MDMA channel */
        hals_mdma_channel_enable(mdma_dev->channel);
        if(MDMA_CHXCFG(mdma_dev->channel) & MDMA_CHXCFG_SWREQMOD) {
            MDMA_CHXCTL0(mdma_dev->channel) |= MDMA_CHXCTL0_SWREQ;
        } else {
            /* do nothing */
        }
    } else {
        error_code = HAL_ERR_BUSY;
    }

    /* unlock MDMA */
    HAL_UNLOCK(mdma_dev);

    return error_code;
}

/*!
    \brief      stop transferring amounts of data by interrupt method,the function is non-blocking
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_stop_interrupt(hal_mdma_dev_struct *mdma_dev)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check MDMA pointer address */
    if(NULL == mdma_dev) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock MDMA */
    HAL_LOCK(mdma_dev);
    mdma_dev->state = HAL_MDMA_STATE_BUSY;

    /* disable MDMA channel */
    hals_mdma_channel_disable(mdma_dev->channel);
    /* disable MDMA IT */
    hals_mdma_interrupt_disable(mdma_dev->channel, \
                               (MDMA_INT_ERR | MDMA_INT_CHTC | MDMA_INT_MBTC | MDMA_INT_BTC | MDMA_INT_TC));

    /* reset the interrupt handle */
    mdma_dev->mdma_irq.mdma_error_handle = NULL;
    mdma_dev->mdma_irq.mdma_complete_handle = NULL;
    mdma_dev->mdma_irq.mdma_multi_block_complete_handle = NULL;
    mdma_dev->mdma_irq.mdma_block_complete_handle = NULL;
    mdma_dev->mdma_irq.mdma_buffer_complete_handle = NULL;

    /* clear all flags */
    hals_mdma_flag_clear(mdma_dev->channel, (MDMA_FLAG_ERR | MDMA_FLAG_CHTCF | MDMA_FLAG_MBTCF | \
                         MDMA_FLAG_BTCF | MDMA_FLAG_TCF | MDMA_FLAG_REQAF | MDMA_FLAG_LDTERR | \
                         MDMA_FLAG_MDTERR | MDMA_FLAG_ASERR | MDMA_FLAG_BZERR));
    mdma_dev->state = HAL_MDMA_STATE_READY;

    /* unlock MDMA */
    HAL_UNLOCK(mdma_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      start MDMA transferring
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  src_addr: source memory Buffer address
    \param[in]  dst_addr: destination memory Buffer address
    \param[in]  tbnum: the length of a block transfer in bytes
    \param[in]  block_num: the number of a blocks to be transfer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_start(hal_mdma_dev_struct *mdma_dev, uint32_t src_addr, uint32_t dst_addr, \
                       uint32_t tbnum, uint32_t block_num)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if(NULL == mdma_dev) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(65536U < tbnum) {
        HAL_DEBUGE("parameter [tbnum] value is invalid");
        return HAL_ERR_VAL;
    }
    if(4095U < block_num) {
        HAL_DEBUGE("parameter [block_num] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock MDMA */
    HAL_LOCK(mdma_dev);

    if(HAL_MDMA_STATE_READY == mdma_dev->state) {
        mdma_dev->state = HAL_MDMA_STATE_BUSY ;
        mdma_dev->error_state =  HAL_MDMA_ERROR_NONE;
        /* disable MDMA channel */
        hals_mdma_channel_disable(mdma_dev->channel);

        /* configure the source, destination address and the data length */
        hals_mdma_source_address_config(mdma_dev->channel, src_addr);
        hals_mdma_destination_address_config(mdma_dev->channel, dst_addr);
        hals_mdma_byte_block_config(mdma_dev->channel, tbnum);
        hals_mdma_block_count_config(mdma_dev->channel, block_num);

        /* enable MDMA channel */
        hals_mdma_channel_enable(mdma_dev->channel);
        if((MDMA_CHXCFG(mdma_dev->channel) & MDMA_CHXCFG_SWREQMOD)) {
            hal_mdma_channel_software_request_enable(mdma_dev->channel);
        }
    } else {
        error_code = HAL_ERR_BUSY;
    }

    /* unlock MDMA */
    HAL_UNLOCK(mdma_dev);

    return error_code;
}

/*!
    \brief      stop MDMA transferring
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_ALREADY_DONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_stop(hal_mdma_dev_struct *mdma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check MDMA pointer address */
    if(NULL == mdma_dev) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock MDMA */
    HAL_LOCK(mdma_dev);

    if(HAL_MDMA_STATE_BUSY != mdma_dev->state) {
        /* dma not in transfer state */
        mdma_dev->error_state = HAL_MDMA_ERROR_NOTRANSFER;

        error_code = HAL_ERR_ALREADY_DONE;
    } else {
        /* disable MDMA channel */
        hals_mdma_channel_disable(mdma_dev->channel);

        /* clear all flags */
        hals_mdma_flag_clear(mdma_dev->channel, (MDMA_FLAG_ERR | MDMA_FLAG_CHTCF | MDMA_FLAG_MBTCF | \
                             MDMA_FLAG_BTCF | MDMA_FLAG_TCF | MDMA_FLAG_REQAF | MDMA_FLAG_LDTERR | \
                             MDMA_FLAG_MDTERR | MDMA_FLAG_ASERR | MDMA_FLAG_BZERR));
    }

    mdma_dev->state = HAL_MDMA_STATE_READY ;

    /* unlock MDMA */
    HAL_UNLOCK(mdma_dev);

    return error_code;
}

/*!
    \brief      abort MDMA transfer
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_ALREADY_DONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_abort(hal_mdma_dev_struct *mdma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check MDMA pointer address */
    if(NULL == mdma_dev) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock MDMA */
    HAL_LOCK(mdma_dev);

    if(HAL_MDMA_STATE_BUSY != mdma_dev->state) {
        /* dma not in transfer state */
        mdma_dev->error_state = HAL_MDMA_ERROR_NOTRANSFER;

        error_code = HAL_ERR_ALREADY_DONE;
    } else {
        /* Set Abort State  */
        mdma_dev->state = HAL_MDMA_STATE_ABORT;

        /* disable MDMA channel */
        hals_mdma_channel_disable(mdma_dev->channel);
    }

    /* unlock MDMA */
    HAL_UNLOCK(mdma_dev);

    return error_code;
}

/*!
    \brief      get the mdma state
    \param[in]  mdma_dev: mdma device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_mdma_state_enum
*/
hal_mdma_state_enum hal_mdma_state_get(hal_mdma_dev_struct *mdma_dev)
{
    return mdma_dev->state;
}

/*!
    \brief      get the mdma error state
    \param[in]  mdma_dev: mdma device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_mdma_error_enum
*/
hal_mdma_error_enum hal_mdma_error_get(hal_mdma_dev_struct *mdma_dev)
{
    return mdma_dev->error_state;
}

/*!
    \brief      disable the linked list circular by connecting the last node to the first
    \param[in]  mdma_dev: mdma device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_list_circularmode_disable(hal_mdma_dev_struct *mdma_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == mdma_dev) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_MDMA_STATE_READY == mdma_dev->state) {
        mdma_dev->state = HAL_MDMA_STATE_BUSY;
        if(NULL != mdma_dev->first_node) {
            mdma_dev->last_node->chxladdr_reg = 0U;
        } else {
            mdma_dev->state = HAL_MDMA_STATE_READY;
            HAL_DEBUGE("there are no nodes in this list");
            error_code = HAL_ERR_ADDRESS;
        }
    } else {
        error_code = HAL_ERR_BUSY;
    }

    mdma_dev->state = HAL_MDMA_STATE_READY;

    return error_code;
}

/*!
    \brief      mdma start polling for transfer complete
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  trans_trig_mode: MDMA transfer trigger mode
    \param[in]  timeout_ms: time out duration
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_HARDWARE , HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_ALREADY_DONE
                            details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_start_poll(hal_mdma_dev_struct *mdma_dev, uint32_t trans_trig_mode, uint32_t timeout_ms)
{
    int32_t error_code  = HAL_ERR_NONE;
    uint32_t flag_error = 0U;
    uint32_t flag_tr    = 0U;
    uint32_t tick_start = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if(NULL == mdma_dev) {
        HAL_DEBUGE("pointer [mdma_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(MDMA_COMPLETE_TRANSFER < trans_trig_mode) {
        HAL_DEBUGE("parameter [trans_trig_mode] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock MDMA */
    HAL_LOCK(mdma_dev);

    if(HAL_MDMA_STATE_BUSY == mdma_dev->state) {
        if(MDMA_COMPLETE_TRANSFER == trans_trig_mode) {
            flag_tr = MDMA_FLAG_CHTCF;
        } else if(MDMA_BUFFER_TRANSFER == trans_trig_mode) {
            flag_tr = MDMA_FLAG_TCF;
        } else if(MDMA_BLOCK_TRANSFER == trans_trig_mode) {
            flag_tr = MDMA_FLAG_BTCF;
        } else if(MDMA_MULTI_BLOCK_TRANSFER == trans_trig_mode) {
            flag_tr = MDMA_FLAG_MBTCF;
        } else {
            HAL_DEBUGE("parameter [trans_trig_mode] value is invalid");
            error_code = HAL_ERR_VAL;
        }

        if(HAL_ERR_NONE == error_code) {
            /* set timeout */
            tick_start = hal_sys_basetick_count_get();
            while(RESET == hals_mdma_flag_get(mdma_dev->channel, flag_tr)) {
                /* if transfer error */
                if(SET == hals_mdma_flag_get(mdma_dev->channel, MDMA_FLAG_ERR)) {
                    flag_error = MDMA_CHXSTAT1(mdma_dev->channel);
                    if(0U != (flag_error & MDMA_CHXSTAT1_ERRADDR)) {
                        mdma_dev->error_state = HAL_MDMA_ERROR_ERRADDR;
                    } else {
                        /* do nothing */
                    }

                    if(0U != (flag_error & MDMA_CHXSTAT1_TERRD)) {
                        mdma_dev->error_state = HAL_MDMA_ERROR_WRITE;
                    } else {
                        mdma_dev->error_state = HAL_MDMA_ERROR_READ;
                    }

                    if(0U != (flag_error & MDMA_CHXSTAT1_LDTERR)) {
                        mdma_dev->error_state = HAL_MDMA_ERROR_LINKED_LIST;
                    } else {
                        /* do nothing */
                    }

                    if(0U != (flag_error & MDMA_CHXSTAT1_MDTERR)) {
                        mdma_dev->error_state = HAL_MDMA_ERROR_MASK_DATA;
                    } else {
                        /* do nothing */
                    }

                    if(0U != (flag_error & MDMA_CHXSTAT1_ASERR)) {
                        mdma_dev->error_state = HAL_MDMA_ERROR_ALIGNMENT;
                    } else {
                        /* do nothing */
                    }

                    if(0U != (flag_error & MDMA_CHXSTAT1_BZERR)) {
                        mdma_dev->error_state = HAL_MDMA_ERROR_BLOCK_SIZE;
                    } else {
                        /* do nothing */
                    }

                    hal_mdma_stop(mdma_dev);
                    error_code = HAL_ERR_HARDWARE;
                    break;
                } else {
                    /* check for the timeout */
                    if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                        if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                            hal_mdma_stop(mdma_dev);
                            mdma_dev->error_state = HAL_MDMA_ERROR_TIMEOUT;

                            /* when timeout occurs, output timeout warning message */
                            HAL_DEBUGE("mdma transfer state poll timeout");

                            error_code = HAL_ERR_TIMEOUT;
                            break;
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }
            }

            /* clear all flags */
            hals_mdma_flag_clear(mdma_dev->channel, \
                                (MDMA_FLAG_ERR | MDMA_FLAG_CHTCF | MDMA_FLAG_MBTCF | MDMA_FLAG_BTCF | MDMA_FLAG_TCF | \
                                MDMA_FLAG_LDTERR | MDMA_FLAG_MDTERR | MDMA_FLAG_ASERR | MDMA_FLAG_BZERR));
        } else {
            /* do nothing */
        }
    } else {
        error_code = HAL_ERR_ALREADY_DONE;
    }

    mdma_dev->state = HAL_MDMA_STATE_READY;
    HAL_UNLOCK(mdma_dev);

    return error_code;
}

/*!
    \brief      configure the post request mask address and mask data
    \param[in]  mdma_dev: MDMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mask_address: configure mask address
    \param[in]  mask_data: configure mask data
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, HAL_ERR_BUSY, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_channel_post_request_mask_config(hal_mdma_dev_struct *mdma_dev, uint32_t mask_address, uint32_t mask_data)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == mdma_dev) {
        HAL_DEBUGE("parameter [mdma_dev] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock MDMA */
    HAL_LOCK(mdma_dev);

    if(HAL_MDMA_STATE_READY == mdma_dev->state) {
        if(RESET == (MDMA_CHXCFG(mdma_dev->channel) & MDMA_CHXCFG_SWREQMOD)) {
            /* configure mask address */
            MDMA_CHXMADDR(mdma_dev->channel) = mask_address;
            /* configure mask data */
            MDMA_CHXMDATA(mdma_dev->channel) = mask_data;

            if(0U == mask_address) {
                MDMA_CHXCFG(mdma_dev->channel) &= ~MDMA_CHXCFG_BWMOD;
            } else {
                MDMA_CHXCFG(mdma_dev->channel) |= ~MDMA_CHXCFG_BWMOD;
            }
        } else {
            error_code = HAL_ERR_NO_SUPPORT;
        }
    } else {
        error_code = HAL_ERR_BUSY;
    }

    /* unlock MDMA */
    HAL_UNLOCK(mdma_dev);

    return error_code;
}

/*!
    \brief      enable MDMA channel software request
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_mdma_channel_software_request_enable(hal_mdma_channel_enum channelx)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXCTL0(channelx) |= MDMA_CHXCTL0_SWREQ;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure MDMA multi block transfer mode
    \param[in]  channelx: specify which MDMA channel is initialized
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  tbnum: number of bytes to transfer in block, range from 0x00000000 to 0x00000FFF
    \param[in]  block_init_struct: the data needed to initialize MDMA multi block mode,
                the member values are shown as below:
                  block_num: multi block number, ranges from 0x00000000 to 0x00000FFF
                  saddr_update_val: source address update value, ranges from 0x0000 to 0xFFFF
                  dstaddr_update_val: destination address update value, ranges from 0x0000 to 0xFFFF
                  saddr_update_dir: source address update direction
                  only one parameter can be selected which is shown as below:
      \arg          UPDATE_DIR_INCREASE: MDMA address update increase
      \arg          UPDATE_DIR_DECREASE: MDMA address update decrease
                  dstaddr_update_dir: destination address update direction
                  only one parameter can be selected which is shown as below:
      \arg          UPDATE_DIR_INCREASE: MDMA address update increase
      \arg          UPDATE_DIR_DECREASE: MDMA address update decrease
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_multi_block_mode_config(hal_mdma_channel_enum channelx, \
                                          uint32_t tbnum, hal_mdma_multi_block_parameter_struct *block_init_struct)
{
    uint32_t blockoffset = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if(NULL == block_init_struct) {
        HAL_DEBUGE("parameter [block_init_struct] value is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if(65536U < tbnum) {
        HAL_DEBUGE("parameter [tbnum] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXBTCFG(channelx) = (((block_init_struct->block_num << CHXBTCFG_BRNUM_OFFSET) & MDMA_CHXBTCFG_BRNUM) | (tbnum & MDMA_CHXBTCFG_TBNUM));

    MDMA_CHXMBADDRU(channelx) &= ~(MDMA_CHXMBADDRU_SADDRUV | MDMA_CHXMBADDRU_DADDRUV);
    /* if block source address offset is negative, set the block repeat source address update mode to decrement */
    if(UPDATE_DIR_DECREASE == block_init_struct->saddr_update_dir) {
        MDMA_CHXBTCFG(channelx) |= MDMA_CHXBTCFG_SADDRUM;
        /* write new chxmbaddru register value: source repeat block offset */
        blockoffset = (uint32_t)block_init_struct->saddr_update_val;
        MDMA_CHXMBADDRU(channelx) |= (blockoffset & MDMA_ADDRESS_MASK);
    } else {
        MDMA_CHXBTCFG(channelx) &= ~MDMA_CHXBTCFG_SADDRUM;
        /* write new chxmbaddru register value: source repeat block offset */
        MDMA_CHXMBADDRU(channelx) |= (((uint32_t)block_init_struct->saddr_update_val) & MDMA_ADDRESS_MASK);
    }

    if(UPDATE_DIR_DECREASE == block_init_struct->dstaddr_update_dir) {
        MDMA_CHXBTCFG(channelx) |= MDMA_CHXBTCFG_DADDRUM;
        /* write new chxmbaddru register value: destination repeat block offset */
        blockoffset = (uint32_t)block_init_struct->dstaddr_update_val;
        MDMA_CHXMBADDRU(channelx) |= ((uint32_t)(blockoffset & MDMA_ADDRESS_MASK) << CHXMBADDRU_DADDRUV_OFFSET);
    } else {
        MDMA_CHXBTCFG(channelx) &= ~MDMA_CHXBTCFG_DADDRUM;
        /* write new chxmbaddru register value: destination repeat block offset */
        MDMA_CHXMBADDRU(channelx) |= ((((uint32_t)block_init_struct->dstaddr_update_val) & MDMA_ADDRESS_MASK) << CHXMBADDRU_DADDRUV_OFFSET);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      configure mask address
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  mask_address: mask address
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_channel_mask_address_config(hal_mdma_channel_enum channelx, uint32_t mask_address)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXMADDR(channelx) = mask_address;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure mask data
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  mask_data: mask data
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_channel_mask_data_config(hal_mdma_channel_enum channelx, uint32_t mask_data)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXMDATA(channelx) = mask_data;

    return HAL_ERR_NONE;
}

/*!
    \brief      deinitialize MDMA registers of a channel
    \param[in]  channelx: specify which MDMA channel is deinitialized
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_channel_deinit(hal_mdma_channel_enum channelx)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXSTATC(channelx)   = 0x0000001FU;
    MDMA_CHXCTL0(channelx)    = 0x00000000U;
    MDMA_CHXCFG(channelx)     = 0x00000000U;
    MDMA_CHXBTCFG(channelx)   = 0x00000000U;
    MDMA_CHXSADDR(channelx)   = 0x00000000U;
    MDMA_CHXDADDR(channelx)   = 0x00000000U;
    MDMA_CHXMBADDRU(channelx) = 0x00000000U;
    MDMA_CHXLADDR(channelx)   = 0x00000000U;
    MDMA_CHXCTL1(channelx)    = 0x00000000U;
    MDMA_CHXMADDR(channelx)   = 0x00000000U;
    MDMA_CHXMDATA(channelx)   = 0x00000000U;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure MDMA buffer/block transfer mode
    \param[in]  channelx: specify which MDMA channel is initialized
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  saddr: source address, ranges from 0x00000000 to 0xFFFFFFFF
    \param[in]  daddr: destination address, ranges from 0x00000000 to 0xFFFFFFFF
    \param[in]  tbnum: number of bytes to transfer, ranges from 0x00000000 to 0x00010000
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_buffer_block_mode_config(hal_mdma_channel_enum channelx, uint32_t saddr, uint32_t daddr, uint32_t tbnum)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check the MDMA pointer address and the number length parameter */
    if(65536U < tbnum) {
        HAL_DEBUGE("parameter [tbnum] value is illegal");
        return HAL_ERR_VAL;
    }
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXSADDR(channelx) = saddr;
    MDMA_CHXDADDR(channelx) = daddr;
    MDMA_CHXBTCFG(channelx) &= ~MDMA_CHXBTCFG_TBNUM;
    MDMA_CHXBTCFG(channelx) |= (tbnum & MDMA_CHXBTCFG_TBNUM);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure MDMA destination base address
    \param[in]  channelx: specify which MDMA channel to set peripheral base address
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  address: destination base address, ranges from 0x00000000 to 0xFFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_destination_address_config(hal_mdma_channel_enum channelx, uint32_t address)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXDADDR(channelx) = address;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure MDMA source base address
    \param[in]  channelx: specify which MDMA channel to set Memory base address
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  address: source base address, ranges from 0x00000000 to 0xFFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_source_address_config(hal_mdma_channel_enum channelx, uint32_t address)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXSADDR(channelx) = address;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure MDMA destination bus
    \param[in]  channelx: specify which MDMA channel to set bus
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  bus: destination bus
                only one parameter can be selected which is shown as below:
      \arg        MDMA_DESTINATION_AXI: destination bus is the system bus or AXI bus
      \arg        MDMA_DESTINATION_AHB_TCM: destination bus is AHB bus or TCM
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_destination_bus_config(hal_mdma_channel_enum channelx, uint32_t bus)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if((MDMA_DESTINATION_AXI != bus) && (MDMA_DESTINATION_AHB_TCM != bus)) {
        HAL_DEBUGE("parameter [bus] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXCTL1(channelx) &= ~MDMA_CHXCTL1_DBSEL;
    MDMA_CHXCTL1(channelx) |= bus;

    return HAL_ERR_NONE;
}


/*!
    \brief      configure MDMA source bus
    \param[in]  channelx: specify which MDMA channel to set bus
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  bus: source bus
                only one parameter can be selected which is shown as below:
      \arg        MDMA_SOURCE_AXI: source bus is the system bus or AXI bus
      \arg        MDMA_SOURCE_AHB_TCM: source bus is AHB bus or TCM
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_source_bus_config(hal_mdma_channel_enum channelx, uint32_t bus)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if((MDMA_SOURCE_AXI != bus) && (MDMA_SOURCE_AHB_TCM != bus)) {
        HAL_DEBUGE("parameter [bus] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXCTL1(channelx) &= ~MDMA_CHXCTL1_SBSEL;
    MDMA_CHXCTL1(channelx) |= bus;
    return HAL_ERR_NONE;
}

/*!
    \brief      configure priority level of MDMA channel
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  priority: priority level of this channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_PRIORITY_LOW: low priority
      \arg        MDMA_PRIORITY_MEDIUM: medium priority
      \arg        MDMA_PRIORITY_HIGH: high priority
      \arg        MDMA_PRIORITY_ULTRA_HIGH: ultra high priority
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_priority_config(hal_mdma_channel_enum channelx, uint32_t priority)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if(MDMA_PRIORITY_ULTRA_HIGH < priority) {
        HAL_DEBUGE("parameter [priority] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure priority level */
    MDMA_CHXCTL0(channelx) &= ~MDMA_CHXCTL0_PRIO;
    MDMA_CHXCTL0(channelx) |= priority;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure endianness of MDMA channel
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  endianness: MDMA endianness
                only one parameter can be selected which is shown as below:
      \arg        MDMA_LITTLE_ENDIANNESS: little endianness preserve
      \arg        MDMA_BYTE_ENDIANNESS_EXCHANGE: exchange the order of the bytes in a half-word
      \arg        MDMA_HALFWORD_ENDIANNESS_EXCHANGE: exchange the order of the half-words in a word
      \arg        MDMA_WORD_ENDIANNESS_EXCHANGE: exchange the order of the words in a double word
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_endianness_config(hal_mdma_channel_enum channelx, uint32_t endianness)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if(MDMA_WORD_ENDIANNESS_EXCHANGE < endianness) {
        HAL_DEBUGE("parameter [priority] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure MDMA endianness */
    MDMA_CHXCTL0(channelx) &= ~(MDMA_CHXCTL0_BES | MDMA_CHXCTL0_HWES | MDMA_CHXCTL0_WES);
    MDMA_CHXCTL0(channelx) |= endianness;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure data alignment of MDMA channel
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  alignment: MDMA data alignment
                only one parameter can be selected which is shown as below:
      \arg        MDMA_DATAALIGN_PKEN: pack/unpack the source data to match the destination data size
      \arg        MDMA_DATAALIGN_RIGHT: right aligned, padded with 0s (default)
      \arg        MDMA_DATAALIGN_RIGHT_SIGNED: right aligned with sign extended, note: this mode is allowed only if the source data size is smaller than destination data size
      \arg        MDMA_DATAALIGN_LEFT: left aligned, padded with 0s in low bytes position when source data size smaller than destination data size, and only high byte of source is written when source data size larger than destination data size
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_alignment_config(hal_mdma_channel_enum channelx, uint32_t alignment)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if((MDMA_DATAALIGN_LEFT < alignment) && (MDMA_DATAALIGN_PKEN != alignment)) {
        HAL_DEBUGE("parameter [alignment] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure MDMA endianness */
    MDMA_CHXCFG(channelx) &= ~(MDMA_CHXCFG_PKEN | MDMA_CHXCFG_PAMOD);
    MDMA_CHXCFG(channelx) |= alignment;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure transfer burst beats of source
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  sbeat: source transfer burst beats
                only one parameter can be selected which is shown as below:
      \arg        MDMA_SOURCE_BURST_SINGLE: source transfer single burst
      \arg        MDMA_SOURCE_BURST_2BEATS: source transfer 2-beat burst
      \arg        MDMA_SOURCE_BURST_4BEATS: source transfer 4-beat burst
      \arg        MDMA_SOURCE_BURST_8BEATS: source transfer 8-beat burst
      \arg        MDMA_SOURCE_BURST_16BEATS: source transfer 16-beat burst
      \arg        MDMA_SOURCE_BURST_32BEATS: source transfer 32-beat burst
      \arg        MDMA_SOURCE_BURST_64BEATS: source transfer 64-beat burst
      \arg        MDMA_SOURCE_BURST_128BEATS: source transfer 128-beat burst
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_source_burst_beats_config(hal_mdma_channel_enum channelx, uint32_t sbeat)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if(MDMA_SOURCE_BURST_128BEATS < sbeat) {
        HAL_DEBUGE("parameter [sbeat] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure transfer burst beats of source */
    MDMA_CHXCFG(channelx) &= ~MDMA_CHXCFG_SBURST;
    MDMA_CHXCFG(channelx) |= sbeat;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure transfer burst beats of destination
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in] dbeat: destination transfer burst beats
                only one parameter can be selected which is shown as below:
      \arg        MDMA_DESTINATION_BURST_SINGLE: destination transfer single burst
      \arg        MDMA_DESTINATION_BURST_2BEATS: destination transfer 2-beat burst
      \arg        MDMA_DESTINATION_BURST_4BEATS: destination transfer 4-beat burst
      \arg        MDMA_DESTINATION_BURST_8BEATS: destination transfer 8-beat burst
      \arg        MDMA_DESTINATION_BURST_16BEATS: destination transfer 16-beat burst
      \arg        MDMA_DESTINATION_BURST_32BEATS: destination transfer 32-beat burst
      \arg        MDMA_DESTINATION_BURST_64BEATS: destination transfer 64-beat burst
      \arg        MDMA_DESTINATION_BURST_128BEATS: destination transfer 128-beat burst
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_destination_burst_beats_config(hal_mdma_channel_enum channelx, uint32_t dbeat)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if(MDMA_DESTINATION_BURST_128BEATS < dbeat) {
        HAL_DEBUGE("parameter [dbeat] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure transfer burst beats of destination */
    MDMA_CHXCFG(channelx) &= ~MDMA_CHXCFG_DBURST;
    MDMA_CHXCFG(channelx) |= dbeat;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure data size of source
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  swidth: source data size
                only one parameter can be selected which is shown as below:
      \arg        MDMA_SOURCE_DATASIZE_8BIT: source data size is byte
      \arg        MDMA_SOURCE_DATASIZE_16BIT: source data size is half word
      \arg        MDMA_SOURCE_DATASIZE_32BIT: source data size is word
      \arg        MDMA_SOURCE_DATASIZE_64BIT: source data size is double word
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_source_width_config(hal_mdma_channel_enum channelx, uint32_t swidth)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if(MDMA_SOURCE_DATASIZE_64BIT < swidth) {
        HAL_DEBUGE("parameter [swidth] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure data size of source */
    MDMA_CHXCFG(channelx) &= ~MDMA_CHXCFG_SWIDTH;
    MDMA_CHXCFG(channelx) |= swidth;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure data size of destination
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  dwidth: destination data size
                only one parameter can be selected which is shown as below:
      \arg        MDMA_DESTINATION_DATASIZE_8BIT: destination data size is byte
      \arg        MDMA_DESTINATION_DATASIZE_16BIT: destination data size is half word
      \arg        MDMA_DESTINATION_DATASIZE_32BIT: destination data size is word
      \arg        MDMA_DESTINATION_DATASIZE_64BIT: destination data size is double word
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_destination_width_config(hal_mdma_channel_enum channelx, uint32_t dwidth)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if(MDMA_DESTINATION_DATASIZE_64BIT < dwidth) {
        HAL_DEBUGE("parameter [dwidth] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure data size of source */
    MDMA_CHXCFG(channelx) &= ~MDMA_CHXCFG_DWIDTH;
    MDMA_CHXCFG(channelx) |= dwidth;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure source address increment mode
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  sinc: source address increment mode
                only one parameter can be selected which is shown as below:
      \arg        MDMA_SOURCE_INCREASE_DISABLE: no increment
      \arg        MDMA_SOURCE_INCREASE_8BIT: source address pointer is incremented by a byte (8 bits)
      \arg        MDMA_SOURCE_INCREASE_16BIT: source address pointer is incremented by a half word (16 bits)
      \arg        MDMA_SOURCE_INCREASE_32BIT: source address pointer is incremented by a word (32 bits)
      \arg        MDMA_SOURCE_INCREASE_64BIT: source address pointer is incremented by a double word (64 bits)
      \arg        MDMA_SOURCE_DECREASE_8BIT: source address pointer is decremented by a byte (8 bits)
      \arg        MDMA_SOURCE_DECREASE_16BIT: source address pointer is decremented by a half word (16 bits)
      \arg        MDMA_SOURCE_DECREASE_32BIT: source address pointer is decremented by a word (32 bits)
      \arg        MDMA_SOURCE_DECREASE_64BIT: source address pointer is decremented by a double word (64 bits)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_source_increment_config(hal_mdma_channel_enum channelx, uint32_t sinc)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }

    if(MDMA_SOURCE_DECREASE_64BIT < sinc) {
        HAL_DEBUGE("parameter [sinc] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure address increment mode of source */
    MDMA_CHXCFG(channelx) &= ~(MDMA_CHXCFG_SIMOD | MDMA_CHXCFG_SIOS);
    MDMA_CHXCFG(channelx) |= sinc;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure destination address increment mode
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  dinc: destination address increment mode
                only one parameter can be selected which is shown as below:
      \arg        MDMA_DESTINATION_INCREASE_DISABLE: no increment
      \arg        MDMA_DESTINATION_INCREASE_8BIT: destination address pointer is incremented by a byte (8 bits)
      \arg        MDMA_DESTINATION_INCREASE_16BIT: destination address pointer is incremented by a half word (16 bits)
      \arg        MDMA_DESTINATION_INCREASE_32BIT: destination address pointer is incremented by a word (32 bits)
      \arg        MDMA_DESTINATION_INCREASE_64BIT: destination address pointer is incremented by a double word (64 bits))
      \arg        MDMA_DESTINATION_DECREASE_8BIT: destination address pointer is decremented by a byte (8 bits)
      \arg        MDMA_DESTINATION_DECREASE_16BIT: destination address pointer is decremented by a half word (16 bits)
      \arg        MDMA_DESTINATION_DECREASE_32BIT: destination address pointer is decremented by a word (32 bits)
      \arg        MDMA_DESTINATION_DECREASE_64BIT: destination address pointer is decremented by a double word (64 bits)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_destination_increment_config(hal_mdma_channel_enum channelx, uint32_t dinc)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if(MDMA_DESTINATION_DECREASE_64BIT < dinc) {
        HAL_DEBUGE("parameter [dinc] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure address increment mode of destination */
    MDMA_CHXCFG(channelx) &= ~(MDMA_CHXCFG_DIMOD | MDMA_CHXCFG_DIOS);
    MDMA_CHXCFG(channelx) |= dinc;

    return HAL_ERR_NONE;
}


/*!
    \brief      configure MDMA transfer byte number in block
    \param[in]  channelx: specify which MDMA channel to set Memory base address
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  tbnum: the length of a block transfer in bytes
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_byte_block_config(hal_mdma_channel_enum channelx, uint32_t tbnum)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }

    if(65536U < tbnum) {
        HAL_DEBUGE("parameter [tbnum] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXBTCFG(channelx) &= ~MDMA_CHXBTCFG_TBNUM;
    MDMA_CHXBTCFG(channelx) |= (tbnum & MDMA_CHXBTCFG_TBNUM);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure MDMA block count
    \param[in]  channelx: specify which MDMA channel to set Memory base address
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  block_num: the number of a blocks to be transfer
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_block_count_config(hal_mdma_channel_enum channelx, uint32_t block_num)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
    if(4095U < block_num) {
        HAL_DEBUGE("parameter [block_num] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXBTCFG(channelx) &= ~MDMA_CHXBTCFG_BRNUM;
    MDMA_CHXBTCFG(channelx) |= ((block_num << CHXBTCFG_BRNUM_OFFSET) & MDMA_CHXBTCFG_BRNUM);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable MDMA channel bufferable write mode
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_channel_bufferable_write_enable(hal_mdma_channel_enum channelx)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXCFG(channelx) |= MDMA_CHXCFG_BWMOD;

    return HAL_ERR_NONE;
}

/*!
    \brief      disable MDMA channel bufferable write mode
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_channel_bufferable_write_disable(hal_mdma_channel_enum channelx)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXCFG(channelx) &= ~MDMA_CHXCFG_BWMOD;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable MDMA channel
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_channel_enable(hal_mdma_channel_enum channelx)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXCTL0(channelx) |= MDMA_CHXCTL0_CHEN;

    return HAL_ERR_NONE;
}

/*!
    \brief      disable MDMA channel
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_channel_disable(hal_mdma_channel_enum channelx)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXCTL0(channelx) &= ~MDMA_CHXCTL0_CHEN;

    return HAL_ERR_NONE;
}

/*!
    \brief      get MDMA transfer error direction
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[out] none
    \retval     transfer error direction: MDMA_READ_ERROR or MDMA_WRITE_ERROR
*/
uint32_t hals_mdma_transfer_error_direction_get(hal_mdma_channel_enum channelx)
{
    return (MDMA_CHXSTAT1(channelx) & MDMA_CHXSTAT1_TERRD);
}

/*!
    \brief      get MDMA transfer error address
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[out] none
    \retval     uint32_t: 0x00-0x7F
*/
uint32_t hals_mdma_transfer_error_address_get(hal_mdma_channel_enum channelx)
{
    return (MDMA_CHXSTAT1(channelx) & MDMA_CHXSTAT1_ERRADDR);
}

/*!
    \brief      get MDMA flag
    \param[in]  channelx: specify which MDMA channel to clear flag
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  flag: specify get which flag
                only one parameter can be selected which is shown as below:
      \arg        MDMA_FLAG_ERR: transfer error flag
      \arg        MDMA_FLAG_CHTCF: channel transfer complete flag
      \arg        MDMA_FLAG_MBTCF: multi-block transfer complete flag
      \arg        MDMA_FLAG_BTCF: block transfer complete flag
      \arg        MDMA_FLAG_TCF: buffer transfer complete flag
      \arg        MDMA_FLAG_REQAF: request active flag
      \arg        MDMA_FLAG_LDTERR: link data transfer error flag in the last transfer of the channel
      \arg        MDMA_FLAG_MDTERR: mask data error flag
      \arg        MDMA_FLAG_ASERR: address and size error flag
      \arg        MDMA_FLAG_BZERR: block size error flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_mdma_flag_get(hal_mdma_channel_enum channelx, uint32_t flag)
{
    FlagStatus status = RESET;
    uint32_t flag_pos = 0U;

    if(STAT1_FLAG & flag) {
        /* get the flag in CHXSTAT1 */
        flag_pos = (flag & STAT1_FLAG_MASK);
        if(MDMA_CHXSTAT1(channelx) & flag_pos) {
            status = SET;
        } else {
            /* do nothing */
        }
    } else {
        /* get the flag in CHXSTAT0 */
        if(MDMA_CHXSTAT0(channelx) & flag) {
            status = SET;
        } else {
            /* do nothing */
        }
    }

    return status;
}

/*!
    \brief      clear MDMA flag
    \param[in]  channelx: specify which MDMA channel to clear flag
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  flag: specify clear which flag
                only one parameter can be selected which is shown as below:
      \arg        MDMA_FLAG_ERR: transfer error flag
      \arg        MDMA_FLAG_CHTCF: channel transfer complete flag
      \arg        MDMA_FLAG_MBTCF: multi-block transfer complete flag
      \arg        MDMA_FLAG_BTCF: block transfer complete flag
      \arg        MDMA_FLAG_TCF: buffer transfer complete flag
      \arg        MDMA_FLAG_LDTERR: link data transfer error flag in the last transfer of the channel
      \arg        MDMA_FLAG_MDTERR: mask data error flag
      \arg        MDMA_FLAG_ASERR: address and size error flag
      \arg        MDMA_FLAG_BZERR: block size error flag
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_flag_clear(hal_mdma_channel_enum channelx, uint32_t flag)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(STAT1_FLAG & flag) {
        MDMA_CHXSTATC(channelx) |= MDMA_CHXSTATC_ERRC;
    } else {
        MDMA_CHXSTATC(channelx) |= flag;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable MDMA interrupt
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  interrupt: specify which interrupt to enable
                one or more parameters can be selected which are shown as below:
      \arg        MDMA_INT_ERR: transfer error interrupt
      \arg        MDMA_INT_CHTC: channel transfer complete interrupt
      \arg        MDMA_INT_MBTC: multi-block transfer complete interrupt
      \arg        MDMA_INT_BTC: block transfer complete interrupt
      \arg        MDMA_INT_TC: buffer transfer complete interrupt
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_interrupt_enable(hal_mdma_channel_enum channelx, uint32_t interrupt)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXCTL0(channelx) |= interrupt;

    return HAL_ERR_NONE;
}

/*!
    \brief      disable MDMA interrupt
    \param[in]  channelx: specify which MDMA channel
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  interrupt: specify which interrupt to disable
                one or more parameters can be selected which are shown as below:
      \arg        MDMA_INT_ERR: transfer error interrupt
      \arg        MDMA_INT_CHTC: channel transfer complete interrupt
      \arg        MDMA_INT_MBTC: multi-block transfer complete interrupt
      \arg        MDMA_INT_BTC: block transfer complete interrupt
      \arg        MDMA_INT_TC: buffer transfer complete interrupt
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_interrupt_disable(hal_mdma_channel_enum channelx, uint32_t interrupt)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXCTL0(channelx) &= ~interrupt;

    return HAL_ERR_NONE;
}

/*!
    \brief      get MDMA interrupt flag
    \param[in]  channelx: specify which MDMA channel to get flag
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  int_flag: specify get which flag
                only one parameter can be selected which is shown as below:
      \arg        MDMA_INT_FLAG_ERR: transfer error interrupt flag
      \arg        MDMA_INT_FLAG_CHTCF: channel transfer complete interrupt flag
      \arg        MDMA_INT_FLAG_MBTCF: multi-block transfer complete interrupt flag
      \arg        MDMA_INT_FLAG_BTCF: block transfer complete interrupt flag
      \arg        MDMA_INT_FLAG_TCF: buffer transfer complete interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_mdma_interrupt_flag_get(hal_mdma_channel_enum channelx, uint32_t int_flag)
{
    FlagStatus status         = RESET;
    uint32_t interrupt_enable = 0U;
    uint32_t interrupt_flag   = 0U;

    switch(int_flag) {
    case MDMA_INT_FLAG_ERR:
        /* get error interrupt enable bit and flag bit */
        interrupt_enable = (MDMA_CHXCTL0(channelx) & MDMA_CHXCTL0_ERRIE);
        interrupt_flag = (MDMA_CHXSTAT0(channelx) & int_flag);
        break;
    case MDMA_INT_FLAG_CHTCF:
        /* get channel transfer complete interrupt enable bit and flag bit */
        interrupt_enable = (MDMA_CHXCTL0(channelx) & MDMA_CHXCTL0_CHTCIE);
        interrupt_flag = (MDMA_CHXSTAT0(channelx) & int_flag);
        break;
    case MDMA_INT_FLAG_MBTCF:
        /* get multi-block transfer complete interrupt enable bit and flag bit */
        interrupt_enable = (MDMA_CHXCTL0(channelx) & MDMA_CHXCTL0_MBTCIE);
        interrupt_flag = (MDMA_CHXSTAT0(channelx) & int_flag);
        break;
    case MDMA_INT_FLAG_BTCF:
        /* get block transfer complete interrupt enable bit and flag bit */
        interrupt_enable = (MDMA_CHXCTL0(channelx) & MDMA_CHXCTL0_BTCIE);
        interrupt_flag = (MDMA_CHXSTAT0(channelx) & int_flag);
        break;
    case MDMA_INT_FLAG_TCF:
        /* get buffer transfer complete interrupt enable bit and flag bit */
        interrupt_enable = (MDMA_CHXCTL0(channelx) & MDMA_CHXCTL0_TCIE);
        interrupt_flag = (MDMA_CHXSTAT0(channelx) & int_flag);
        break;
    default:
        break;
    }

    if(interrupt_flag && interrupt_enable) {
        status = SET;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      clear MDMA interrupt flag
    \param[in]  channelx: specify which MDMA channel to clear flag
                only one parameter can be selected which is shown as below:
      \arg        MDMA_CHx(x=0..15)
    \param[in]  int_flag: specify clear which flag
                only one parameter can be selected which is shown as below:
      \arg        MDMA_INT_FLAG_ERR: transfer error interrupt flag
      \arg        MDMA_INT_FLAG_CHTCF: channel transfer complete interrupt flag
      \arg        MDMA_INT_FLAG_MBTCF: multi-block transfer complete interrupt flag
      \arg        MDMA_INT_FLAG_BTCF: block transfer complete interrupt flag
      \arg        MDMA_INT_FLAG_TCF: buffer transfer complete interrupt flag
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_mdma_interrupt_flag_clear(hal_mdma_channel_enum channelx, uint32_t int_flag)
{
    /* check the parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(MDMA_CH15 < channelx) {
        HAL_DEBUGE("parameter [channelx] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MDMA_CHXSTATC(channelx) |= int_flag;

    return HAL_ERR_NONE;
}
