/*!
    \file    gd32h7xx_hal_mdio.c
    \brief   MDIO driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/*!
    \brief      initialize MDIO
    \param[in]  mdio_dev: MDIO device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_init: the pointer to MDIO init structure
                  dev_address: specifies the MDIO device address
                  phy_address: specifies the mdio port address
                  phy_mode:
                    only one parameter can be selected which is shown as below:
        \arg        MDIO_PHYADR_HARDWARE: sets expected PHYADR = PHYPIN[4:0]
        \arg        MDIO_PHYADR_SOFTWARE: sets expected PHYADR = PHYSW[4:0]
                  phy_length:
                    only one parameter can be selected which is shown as below:
        \arg        MDIO_PHY_BITS_5: PHY use 5 bits
        \arg        MDIO_PHY_BITS_3: PHY use 3 bits
                  timeout:
                    only one parameter can be selected which is shown as below:
        \arg        MDIO_TIMEOUT_DISABLE: timeout disable
        \arg        MDIO_TIMEOUT_ENABLE: timeout enable
                  timeout_value: specifies the mdio timeout value
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_mdio_init(hal_mdio_dev_struct *mdio_dev, hal_mdio_init_struct *p_init)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check mdio pointer and p_init address */
    if((NULL == mdio_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [mdio_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Change the MDIOS state */
    mdio_dev->state = HAL_MDIO_STATE_RESET;

    mdio_dev->periph = MDIO;
    mdio_dev->init   = p_init;

    hals_mdio_software_reset();

    /* configure MDIO phy bit length */
    MDIO_CTL &= ~MDIO_CTL_PHYB;
    MDIO_CTL |= p_init->phy_length;

    /* configure the PHYADR and DEVADD */
    MDIO_CFG &= ~(MDIO_CFG_PHYSW | MDIO_CFG_EPHYSEL | MDIO_CFG_EDEVADD);
    MDIO_CFG |= CFG_PHYSW(p_init->phy_address) | CFG_EPHYSEL(p_init->phy_mode) | CFG_EDEVADD(p_init->dev_address);

    MDIO_TO |= (p_init->timeout | TO_TOCNT(p_init->timeout_value));

    mdio_dev->state = HAL_MDIO_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      deinitialize MDIO
    \param[in]  mdio_dev: MDIO device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_mdio_deinit(hal_mdio_dev_struct *mdio_dev)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == mdio_dev) {
        HAL_DEBUGE("pointer [mdio_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(MDIO == mdio_dev->periph) {
        /* deinitialize the periph */
        hal_rcu_periph_reset_enable(RCU_MDIORST);
        hal_rcu_periph_reset_disable(RCU_MDIORST);

        /* reset MDIO state */
        mdio_dev->state = HAL_MDIO_STATE_RESET;
    } else {
        HAL_DEBUGE("parameter [mdio_dev->periph] value is invalid");
        ret = HAL_ERR_VAL;
    }

    return ret;
}

/*!
    \brief      initialize the MDIO structure with the default values
    \param[in]  hal_struct_type: type of MDIO structure for initialization
                only one  parameters can be selected which are shown as below:
      \arg        HAL_MDIO_INIT_STRUCT: initialization structure
      \arg        HAL_MDIO_DEV_STRUCT: device information structure
      \arg        HAL_MDIO_IRQ_INIT_STRUCT: interrupt callback initialization structure
    \param[out] p_struct: pointer to MDIO structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_mdio_struct_init(hal_mdio_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

    /* check parameter address */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_MDIO_INIT_STRUCT:
        /* initialize mdio initialization structure with the default values */
        ((hal_mdio_init_struct *)p_struct)->dev_address   = 0U;
        ((hal_mdio_init_struct *)p_struct)->phy_address   = 0U;
        ((hal_mdio_init_struct *)p_struct)->phy_mode      = MDIO_PHYADR_HARDWARE;
        ((hal_mdio_init_struct *)p_struct)->phy_length    = MDIO_PHY_BITS_5;
        ((hal_mdio_init_struct *)p_struct)->timeout       = MDIO_TIMEOUT_DISABLE;
        ((hal_mdio_init_struct *)p_struct)->timeout_value = 0U;
        break;
    case HAL_MDIO_DEV_STRUCT:
        /* initialize mdio device information structure with the default values */
        ((hal_mdio_dev_struct *)p_struct)->periph                                       = 0U;
        ((hal_mdio_dev_struct *)p_struct)->error_state                                  = 0U;
        ((hal_mdio_dev_struct *)p_struct)->init                                         = NULL;
        ((hal_mdio_dev_struct *)p_struct)->state                                        = HAL_MDIO_STATE_RESET;
        ((hal_mdio_dev_struct *)p_struct)->mutex                                        = HAL_MUTEX_UNLOCKED;
        ((hal_mdio_dev_struct *)p_struct)->priv                                         = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.data_receive_handle                 = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.address_frame_handle                = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.write_data_frame_handle             = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.read_data_frame_handle              = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.read_increment_address_frame_handle = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.devadd_match_frame_handle           = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.devadd_nomatch_frame_handle         = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.phyadr_match_frame_handle           = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.phyadr_nomatch_frame_handle         = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.ta_nomatch_frame_handle             = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.timeout_handle                      = NULL;
        ((hal_mdio_dev_struct *)p_struct)->mdio_irq.error_handle                        = NULL;
        break;
    case HAL_MDIO_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_mdio_irq_struct *)p_struct)->data_receive_handle                 = NULL;
        ((hal_mdio_irq_struct *)p_struct)->address_frame_handle                = NULL;
        ((hal_mdio_irq_struct *)p_struct)->write_data_frame_handle             = NULL;
        ((hal_mdio_irq_struct *)p_struct)->read_data_frame_handle              = NULL;
        ((hal_mdio_irq_struct *)p_struct)->read_increment_address_frame_handle = NULL;
        ((hal_mdio_irq_struct *)p_struct)->devadd_match_frame_handle           = NULL;
        ((hal_mdio_irq_struct *)p_struct)->devadd_nomatch_frame_handle         = NULL;
        ((hal_mdio_irq_struct *)p_struct)->phyadr_match_frame_handle           = NULL;
        ((hal_mdio_irq_struct *)p_struct)->phyadr_nomatch_frame_handle         = NULL;
        ((hal_mdio_irq_struct *)p_struct)->ta_nomatch_frame_handle             = NULL;
        ((hal_mdio_irq_struct *)p_struct)->timeout_handle                      = NULL;
        ((hal_mdio_irq_struct *)p_struct)->error_handle                        = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  mdio_dev: MDIO device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to MDIO interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg      NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_mdio_irq_handle_set(hal_mdio_dev_struct *mdio_dev, hal_mdio_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == mdio_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [mdio_dev] or pointer [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->data_receive_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_RBNE);
        mdio_dev->mdio_irq.data_receive_handle = p_irq->data_receive_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_RBNE);
        mdio_dev->mdio_irq.data_receive_handle = NULL;
    }

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->address_frame_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_ADDRFRM);
        mdio_dev->mdio_irq.address_frame_handle = p_irq->address_frame_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_ADDRFRM);
        mdio_dev->mdio_irq.address_frame_handle = NULL;
    }

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->write_data_frame_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_WRFRM);
        mdio_dev->mdio_irq.write_data_frame_handle = p_irq->write_data_frame_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_WRFRM);
        mdio_dev->mdio_irq.write_data_frame_handle = NULL;
    }

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->read_data_frame_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_RDFRM);
        mdio_dev->mdio_irq.read_data_frame_handle = p_irq->read_data_frame_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_RDFRM);
        mdio_dev->mdio_irq.read_data_frame_handle = NULL;
    }

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->read_increment_address_frame_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_RDINCFRM);
        mdio_dev->mdio_irq.read_increment_address_frame_handle = p_irq->read_increment_address_frame_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_RDINCFRM);
        mdio_dev->mdio_irq.read_increment_address_frame_handle = NULL;
    }

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->devadd_match_frame_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_DEVM);
        mdio_dev->mdio_irq.devadd_match_frame_handle = p_irq->devadd_match_frame_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_DEVM);
        mdio_dev->mdio_irq.devadd_match_frame_handle = NULL;
    }

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->devadd_nomatch_frame_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_DEVNM);
        mdio_dev->mdio_irq.devadd_nomatch_frame_handle = p_irq->devadd_nomatch_frame_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_DEVNM);
        mdio_dev->mdio_irq.devadd_nomatch_frame_handle = NULL;
    }

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->phyadr_match_frame_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_PHYM);
        mdio_dev->mdio_irq.phyadr_match_frame_handle = p_irq->phyadr_match_frame_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_PHYM);
        mdio_dev->mdio_irq.phyadr_match_frame_handle = NULL;
    }

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->phyadr_nomatch_frame_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_PHYNM);
        mdio_dev->mdio_irq.phyadr_nomatch_frame_handle = p_irq->phyadr_nomatch_frame_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_PHYNM);
        mdio_dev->mdio_irq.phyadr_nomatch_frame_handle = NULL;
    }

    /* configure the frame receive callback as the function implemented */
    if(NULL != p_irq->ta_nomatch_frame_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_TANM);
        mdio_dev->mdio_irq.ta_nomatch_frame_handle = p_irq->ta_nomatch_frame_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_TANM);
        mdio_dev->mdio_irq.ta_nomatch_frame_handle = NULL;
    }

    /* configure the timeout callback as the function implemented */
    if(NULL != p_irq->timeout_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_TIMEOUT);
        mdio_dev->mdio_irq.timeout_handle = p_irq->timeout_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_TIMEOUT);
        mdio_dev->mdio_irq.timeout_handle = NULL;
    }

    /* configure the error callback as the function implemented */
    if(NULL != p_irq->error_handle) {
        hals_mdio_interrupt_enable(MDIO_INT_TX_UNDERRUN);
        hals_mdio_interrupt_enable(MDIO_INT_RX_OVERRUN);
        mdio_dev->mdio_irq.error_handle = p_irq->error_handle;
    } else {
        hals_mdio_interrupt_disable(MDIO_INT_TX_UNDERRUN);
        hals_mdio_interrupt_disable(MDIO_INT_RX_OVERRUN);
        mdio_dev->mdio_irq.error_handle = NULL;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  mdio_dev: MDIO device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_mdio_irq_handle_all_reset(hal_mdio_dev_struct *mdio_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == mdio_dev) {
        HAL_DEBUGE("pointer [mdio_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure the frame receive callback as the function implemented */
    mdio_dev->mdio_irq.data_receive_handle                 = NULL;
    mdio_dev->mdio_irq.address_frame_handle                = NULL;
    mdio_dev->mdio_irq.write_data_frame_handle             = NULL;
    mdio_dev->mdio_irq.read_data_frame_handle              = NULL;
    mdio_dev->mdio_irq.read_increment_address_frame_handle = NULL;
    mdio_dev->mdio_irq.devadd_match_frame_handle           = NULL;
    mdio_dev->mdio_irq.devadd_nomatch_frame_handle         = NULL;
    mdio_dev->mdio_irq.phyadr_match_frame_handle           = NULL;
    mdio_dev->mdio_irq.phyadr_nomatch_frame_handle         = NULL;
    mdio_dev->mdio_irq.ta_nomatch_frame_handle             = NULL;
    mdio_dev->mdio_irq.timeout_handle                      = NULL;
    mdio_dev->mdio_irq.error_handle                        = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      MDIO interrupt handler content function,which is merely used in MDIO_IRQHandler
    \param[in]  mdio_dev: MDIO device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_mdio_irq(hal_mdio_dev_struct *mdio_dev)
{
    __IO uint32_t mdio_stat = MDIO_STAT;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == mdio_dev) {
        HAL_DEBUGE("pointer [mdio_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* receive data */
    if(MDIO_STAT_RBNE == (mdio_stat & MDIO_STAT_RBNE)) {
        if(NULL != mdio_dev->mdio_irq.data_receive_handle) {
            mdio_dev->mdio_irq.data_receive_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* an address frame */
    if(MDIO_STAT_ADDRFRM == (mdio_stat & MDIO_STAT_ADDRFRM)) {
        if(NULL != mdio_dev->mdio_irq.address_frame_handle) {
            mdio_dev->mdio_irq.address_frame_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* a write data frame */
    if(MDIO_STAT_WRFRM == (mdio_stat & MDIO_STAT_WRFRM)) {
        if(NULL != mdio_dev->mdio_irq.write_data_frame_handle) {
            mdio_dev->mdio_irq.write_data_frame_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* a post read increment address frame */
    if(MDIO_STAT_RDINCFRM == (mdio_stat & MDIO_STAT_RDINCFRM)) {
        if(NULL != mdio_dev->mdio_irq.read_increment_address_frame_handle) {
            mdio_dev->mdio_irq.read_increment_address_frame_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* a read data frame */
    if(MDIO_STAT_RDFRM == (mdio_stat & MDIO_STAT_RDFRM)) {
        if(NULL != mdio_dev->mdio_irq.read_data_frame_handle) {
            mdio_dev->mdio_irq.read_data_frame_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* a devadd match frame */
    if(MDIO_STAT_DEVM == (mdio_stat & MDIO_STAT_DEVM)) {
        if(NULL != mdio_dev->mdio_irq.devadd_match_frame_handle) {
            mdio_dev->mdio_irq.devadd_match_frame_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* a devadd not match frame */
    if(MDIO_STAT_DEVNM == (mdio_stat & MDIO_STAT_DEVNM)) {
        if(NULL != mdio_dev->mdio_irq.devadd_nomatch_frame_handle) {
            mdio_dev->mdio_irq.devadd_nomatch_frame_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* a PHYADR match frame */
    if(MDIO_STAT_PHYM == (mdio_stat & MDIO_STAT_PHYM)) {
        if(NULL != mdio_dev->mdio_irq.phyadr_match_frame_handle) {
            mdio_dev->mdio_irq.phyadr_match_frame_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* a PHYADR not match frame */
    if(MDIO_STAT_PHYNM == (mdio_stat & MDIO_STAT_PHYNM)) {
        if(NULL != mdio_dev->mdio_irq.phyadr_nomatch_frame_handle) {
            mdio_dev->mdio_irq.phyadr_nomatch_frame_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* a TA not match frame */
    if(MDIO_STAT_TANM == (mdio_stat & MDIO_STAT_TANM)) {
        if(NULL != mdio_dev->mdio_irq.ta_nomatch_frame_handle) {
            mdio_dev->mdio_irq.ta_nomatch_frame_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* a frame timeout */
    if(MDIO_STAT_TO == (mdio_stat & MDIO_STAT_TO)) {
        if(NULL != mdio_dev->mdio_irq.timeout_handle) {
            mdio_dev->mdio_irq.timeout_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* an overrun error callback */
    if(MDIO_STAT_OVR == (mdio_stat & MDIO_STAT_OVR)) {
        if(NULL != mdio_dev->mdio_irq.error_handle) {
            mdio_dev->mdio_irq.error_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* an underrun error callback */
    if(MDIO_STAT_UDR == (mdio_stat & MDIO_STAT_UDR)) {
        if(NULL != mdio_dev->mdio_irq.error_handle) {
            mdio_dev->mdio_irq.error_handle(mdio_dev);
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* clear address frame flag */
    if(RESET != hals_mdio_flag_get(MDIO_FLAG_ADDRFRM)) {
        hals_mdio_flag_clear(MDIO_FLAG_ADDRFRM);
    } else {
        /* nothing to do */
    }

    /* clear write data frame flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_WRFRM)) {
        hals_mdio_flag_clear(MDIO_FLAG_WRFRM);
    } else {
        /* nothing to do */
    }

    /* clear post read increment address frame flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_RDINCFRM)) {
        hals_mdio_flag_clear(MDIO_FLAG_RDINCFRM);
    } else {
        /* nothing to do */
    }

    /* clear read data frame flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_RDFRM)) {
        hals_mdio_flag_clear(MDIO_FLAG_RDFRM);
    } else {
        /* nothing to do */
    }

    /* clear devadd match frame flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_DEVM)) {
        hals_mdio_flag_clear(MDIO_FLAG_DEVM);
    } else {
        /* nothing to do */
    }

    /* clear devadd not match frame flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_DEVNM)) {
        hals_mdio_flag_clear(MDIO_FLAG_DEVNM);
    } else {
        /* nothing to do */
    }

    /* clear PHYADR match frame flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_PHYM)) {
        hals_mdio_flag_clear(MDIO_FLAG_PHYM);
    } else {
        /* nothing to do */
    }

    /* clear PHYADR not match frame flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_PHYNM)) {
        hals_mdio_flag_clear(MDIO_FLAG_PHYNM);
    } else {
        /* nothing to do */
    }

    /* clear TA not match frame flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_TANM)) {
        hals_mdio_flag_clear(MDIO_FLAG_TANM);
    } else {
        /* nothing to do */
    }

    /* clear frame timeout flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_TIMEOUT)) {
        hals_mdio_flag_clear(MDIO_FLAG_TIMEOUT);
    } else {
        /* nothing to do */
    }

    /* clear receive overrun flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_RX_OVERRUN)) {
        hals_mdio_flag_clear(MDIO_FLAG_RX_OVERRUN);
    } else {
        /* nothing to do */
    }

    /* clear transmit underrun flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_TX_UNDERRUN)) {
        hals_mdio_flag_clear(MDIO_FLAG_TX_UNDERRUN);
    } else {
        /* nothing to do */
    }

    /* clear read data and buffer not empty flag */
    if (RESET != hals_mdio_flag_get(MDIO_FLAG_RBNE)) {
        hals_mdio_flag_clear(MDIO_FLAG_RBNE);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      read the received frame field OP
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0-0x11
*/
uint16_t hal_mdio_op_receive(void)
{
    return (uint16_t)(GET_RFRM_ROP(MDIO_RFRM));
}

/*!
    \brief      read the received frame field PHYADR
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0-0x1F
*/
uint16_t hal_mdio_phyadr_receive(void)
{
    return (uint16_t)(GET_RFRM_RPHY(MDIO_RFRM));
}

/*!
    \brief      read the received frame field DEVADD
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0-0x1F
*/
uint16_t hal_mdio_devadd_receive(void)
{
    return (uint16_t)(GET_RFRM_RDEV(MDIO_RFRM));
}

/*!
    \brief      read the received frame field TA
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0-0x11
*/
uint16_t hal_mdio_ta_receive(void)
{
    return (uint16_t)(GET_RFRM_RTA(MDIO_RFRM));
}

/*!
    \brief      read the received frame field DATA
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0-0xFFFF
*/
uint16_t hal_mdio_data_receive(void)
{
    return (uint16_t)(GET_RDATA_RDATA(MDIO_RDATA));
}

/*!
    \brief      read the received frame field ADDRESS
    \param[in]  none
    \param[out] none
    \retval     uint16_t: 0x0-0xFFFF
*/
uint16_t hal_mdio_address_receive(void)
{
    return (uint16_t)(GET_RADDR_RADDR(MDIO_RADDR));
}

/*!
    \brief      transmit the frame field DATA
    \param[in]  data: data to put in a read or post read increment address frame for transmission (0x0-0xFFFF)
    \param[out] none
    \retval     none
*/
void hal_mdio_data_transmit(uint16_t data)
{
    MDIO_TDATA = (uint32_t)data;
}

/*!
    \brief      return MDIO state
    \param[in]  mdio_dev: MDIO device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_mdio_state_enum: HAL_MDIO_STATE_RESET, HAL_MDIO_STATE_READY,
                                     HAL_MDIO_STATE_BUSY, HAL_MDIO_STATE_ERROR
*/
hal_mdio_state_enum hal_mdio_state_get(hal_mdio_dev_struct *mdio_dev)
{
    return mdio_dev->state;
}

/*!
    \brief      return MDIO error code
    \param[in]  mdio_dev: MDIO device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: error state
*/
uint32_t hal_mdio_error_get(hal_mdio_dev_struct *mdio_dev)
{
    return mdio_dev->error_state;
}

/*!
    \brief      reset MDIO block
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_mdio_software_reset(void)
{
    MDIO_CTL |= MDIO_CTL_SWRST;
}

/*!
    \brief      configure MDIO phy bit length
    \param[in]  phy_bit: PHY bit length
                only one parameter can be selected which is shown as below:
      \arg        MDIO_PHY_BITS_3: PHY use 3 bits
      \arg        MDIO_PHY_BITS_5: PHY use 5 bits
    \param[out] none
    \retval     none
*/
void hals_mdio_phy_length_config(uint32_t phy_bit)
{
    MDIO_CTL &= ~MDIO_CTL_PHYB;
    MDIO_CTL |= phy_bit;
}

/*!
    \brief      set the software PHYADR value
    \param[in]  phy_soft: software provided PHYADR (0 - 31)
    \param[out] none
    \retval     none
*/
void hals_mdio_soft_phyadr_set(uint32_t phy_soft)
{
    MDIO_CFG &= ~MDIO_CFG_PHYSW;
    MDIO_CFG |= CFG_PHYSW(phy_soft);
}

/*!
    \brief      select the expected frame field PHYADR
    \param[in]  phy_sel: PHYADR select
                only one parameter can be selected which is shown as below:
      \arg        MDIO_PHYADR_HARDWARE: sets expected PHYADR = PHYPIN[4:0]
      \arg        MDIO_PHYADR_SOFTWARE: sets expected PHYADR = PHYSW[4:0]
      \arg        other user defined value: 1 - 30
    \param[out] none
    \retval     none
*/
void hals_mdio_framefield_phyadr_config(uint32_t phy_sel)
{
    MDIO_CFG &= ~MDIO_CFG_EPHYSEL;
    MDIO_CFG |= CFG_EPHYSEL(phy_sel);
}

/*!
    \brief      configure the expected frame field DEVADD
    \param[in]  type: device type
                only one parameter can be selected which is shown as below:
      \arg        DEVADD_PMA_PMD: device type PMA/PMD
      \arg        DEVADD_WIS: device type WIS
      \arg        DEVADD_PCS: device type PCS
      \arg        DEVADD_PHY_XS: device type PHY XS
      \arg        DEVADD_DTE_XS: device type DTE XS
    \param[out] none
    \retval     none
*/
void hals_mdio_framefield_devadd_config(uint16_t type)
{
    MDIO_CFG &= ~MDIO_CFG_EDEVADD;
    MDIO_CFG |= CFG_EDEVADD(type);
}

/*!
    \brief      read the hardware PRTADR[4:0] value
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0x0-0x1F
*/
uint32_t hals_mdio_phy_pin_read(void)
{
    return GET_PIN_PHYPIN(MDIO_PIN);
}

/*!
    \brief      configure the expected frame bit timeout
    \param[in]  timeout: timeout counter among frame bits (0 - 0xFFFF)
    \param[out] none
    \retval     none
*/
void hals_mdio_timeout_config(uint16_t timeout)
{
    MDIO_TO &= ~MDIO_TO_TOCNT;
    MDIO_TO |= TO_TOCNT(timeout);
}

/*!
    \brief      enable MDIO frame bit timeout
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_mdio_timeout_enable(void)
{
    MDIO_TO |= MDIO_TO_TOEN;
}

/*!
    \brief      disable MDIO frame bit timeout
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_mdio_timeout_disable(void)
{
    MDIO_TO &= ~MDIO_TO_TOEN;
}

/*!
    \brief      get the flag status of the frame
    \param[in]  flag: MDIO flag
                only one parameter can be selected which is shown as below:
      \arg        MDIO_FLAG_WRFRM: a write data frame flag status
      \arg        MDIO_FLAG_ADDRFRM: an address frame flag status
      \arg        MDIO_FLAG_RDINCFRM: a post read increment address frame flag status
      \arg        MDIO_FLAG_RDFRM: a read data frame flag status
      \arg        MDIO_FLAG_DEVM: a DEVADD match frame flag status
      \arg        MDIO_FLAG_DEVNM: a DEVADD nonmatch frame flag status
      \arg        MDIO_FLAG_PHYM: a PHYADR match frame flag status
      \arg        MDIO_FLAG_PHYNM: a PHYADR nonmatch frame flag status
      \arg        MDIO_FLAG_TANM: a TA nonmatch frame flag status
      \arg        MDIO_FLAG_TIMEOUT: timeout flag
      \arg        MDIO_FLAG_TX_UNDERRUN: transmit underrun flag
      \arg        MDIO_FLAG_RX_OVERRUN: receive overrun flag
      \arg        MDIO_FLAG_RBNE: read data buffer not empty flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_mdio_flag_get(uint32_t flag)
{
    FlagStatus status = RESET;
    __IO uint32_t reg = 0U;

    reg = MDIO_STAT;

    if(RESET != (reg & flag)) {
        status = SET;
    } else {
        status = RESET;
    }

    return status;
}

/*!
    \brief      clear MDIO flag status
    \param[in]  flag: MDIO flag
                one or more parameters can be selected which are shown as below:
      \arg        MDIO_FLAG_WRFRM: a write data frame flag status
      \arg        MDIO_FLAG_ADDRFRM: an address frame flag status
      \arg        MDIO_FLAG_RDINCFRM: a post read increment address frame flag status
      \arg        MDIO_FLAG_RDFRM: a read data frame flag status
      \arg        MDIO_FLAG_DEVM: a DEVADD match frame flag status
      \arg        MDIO_FLAG_DEVNM: a DEVADD nonmatch frame flag status
      \arg        MDIO_FLAG_PHYM: a PHYADR match frame flag status
      \arg        MDIO_FLAG_PHYNM: a PHYADR nonmatch frame flag status
      \arg        MDIO_FLAG_TANM: a TA nonmatch frame flag status
      \arg        MDIO_FLAG_TIMEOUT: timeout flag
      \arg        MDIO_FLAG_TX_UNDERRUN: transmit underrun flag
      \arg        MDIO_FLAG_RX_OVERRUN: receive overrun flag
      \arg        MDIO_FLAG_RBNE: read data buffer not empty flag
    \param[out] none
    \retval     none
*/
void hals_mdio_flag_clear(uint32_t flag)
{
    __IO uint32_t reg = 0U;

    reg = MDIO_TDATA;

    if((MDIO_FLAG_RX_OVERRUN | MDIO_FLAG_RBNE) & flag) {
        (void)(MDIO_RDATA);
    } else if(MDIO_FLAG_TX_UNDERRUN & flag) {
        MDIO_TDATA = reg;
    } else if((MDIO_FLAG_WRFRM | MDIO_FLAG_ADDRFRM | MDIO_FLAG_RDINCFRM | MDIO_FLAG_RDFRM | MDIO_FLAG_DEVM | \
               MDIO_FLAG_DEVNM | MDIO_FLAG_PHYM | MDIO_FLAG_PHYNM | MDIO_FLAG_TIMEOUT) & \
              flag) {
        (void)(MDIO_STAT);
    } else {
        /* illegal parameters */
    }
}

/*!
    \brief      enable MDIO interrupt
    \param[in]  interrupt: MDIO interrupt
                one or more parameters can be selected which are shown as below:
      \arg        MDIO_INT_WRFRM: a write data frame interrupt
      \arg        MDIO_INT_ADDRFRM: an address frame interrupt
      \arg        MDIO_INT_RDINCFRM: a post read increment address frame interrupt
      \arg        MDIO_INT_RDFRM: a read data frame interrupt
      \arg        MDIO_INT_DEVM: a DEVADD match frame interrupt
      \arg        MDIO_INT_DEVNM: a DEVADD nonmatch frame interrupt
      \arg        MDIO_INT_PHYM: a PHYADR match frame interrupt
      \arg        MDIO_INT_PHYNM: a PHYADR nonmatch frame interrupt
      \arg        MDIO_INT_TANM: a TA nonmatch frame flag interrupt
      \arg        MDIO_INT_TIMEOUT: a timeout interrupt
      \arg        MDIO_INT_TX_UNDERRUN: a transmit underrun interrupt
      \arg        MDIO_INT_RX_OVERRUN: a receive overrun interrupt
      \arg        MDIO_INT_RBNE: a read data buffer not empty interrupt
    \param[out] none
    \retval     none
*/
void hals_mdio_interrupt_enable(uint32_t interrupt)
{
    MDIO_INTEN |= interrupt;
}

/*!
    \brief      disable MDIO interrupt
    \param[in]  interrupt: MDIO interrupt
                one or more parameters can be selected which are shown as below:
      \arg        MDIO_INT_WRFRM: a write data frame interrupt
      \arg        MDIO_INT_ADDRFRM: an address frame interrupt
      \arg        MDIO_INT_RDINCFRM: a post read increment address frame interrupt
      \arg        MDIO_INT_RDFRM: a read data frame interrupt
      \arg        MDIO_INT_DEVM: a DEVADD match frame interrupt
      \arg        MDIO_INT_DEVNM: a DEVADD nonmatch frame interrupt
      \arg        MDIO_INT_PHYM: a PHYADR match frame interrupt
      \arg        MDIO_INT_PHYNM: a PHYADR nonmatch frame interrupt
      \arg        MDIO_INT_TANM: a TA nonmatch frame flag interrupt
      \arg        MDIO_INT_TIMEOUT: a timeout interrupt
      \arg        MDIO_INT_TX_UNDERRUN: a transmit underrun interrupt
      \arg        MDIO_INT_RX_OVERRUN: a receive overrun interrupt
      \arg        MDIO_INT_RBNE: a read data buffer not empty interrupt
    \param[out] none
    \retval     none
*/
void hals_mdio_interrupt_disable(uint32_t interrupt)
{
    MDIO_INTEN &= ~(interrupt);
}
