/*!
    \file    gd32h7xx_hal_lpdts.c
    \brief   LPDTS driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* LPDTS high threshold value offset macro */
#define LPDTS_IT_INTHT_OFFSET  ((uint32_t)16U)
/* sampling time offset macro */
#define LPDTS_CFG_SPT_OFFSET   ((uint32_t)16U)
/* engineering value offset macro */
#define LPDTS_SDATA_VAL_OFFSET ((uint32_t)16U)
/* the T0 temperature macro */
#define LPDTS_T0_TMP_VAL       ((uint32_t)25U)

/* lpdts async temperature end interrupt handle */
static void _lpdts_async_end_of_temperature_handle(void *lpdts_dev);
/* lpdts async low threshold interrupt handle */
static void _lpdts_async_low_threshold_handle(void *lpdts_dev);
/* lpdts async high threshold interrupt handle */
static void _lpdts_async_high_threshold_handle(void *lpdts_dev);
/* lpdts sync temperature end interrupt handle */
static void _lpdts_sync_end_of_temperature_handle(void *lpdts_dev);
/* lpdts sync low threshold interrupt handle */
static void _lpdts_sync_low_threshold_handle(void *lpdts_dev);
/* lpdts sync high threshold interrupt handle */
static void _lpdts_sync_high_threshold_handle(void *lpdts_dev);

/*!
    \brief      deinitialize LPDTS registers
    \param[in]  lpdts_dev: LPDTS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_lpdts_deinit(hal_lpdts_dev_struct *lpdts_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == lpdts_dev) {
        HAL_DEBUGE("pointer [lpdts_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(lpdts_dev);

    lpdts_dev->state = HAL_LPDTS_STATE_BUSY;

    hal_rcu_periph_reset_enable(RCU_LPDTSRST);
    hal_rcu_periph_reset_disable(RCU_LPDTSRST);

    lpdts_dev->state = HAL_LPDTS_STATE_RESET;

    HAL_UNLOCK(lpdts_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize LPDTS structure
    \param[in]  hal_struct_type: refer to <hal_lpdts_struct_type_enum>
    \param[in]  p_struct: pointer to LPDTS structure that contains the configuration information
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_lpdts_struct_init(hal_lpdts_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    /* initialize LPDTS initialization structure with the default values */
    case HAL_LPDTS_PARAMETER_STRUCT:
        ((hal_lpdts_parameter_struct *)p_struct)->ref_clock          = REF_PCLK;
        ((hal_lpdts_parameter_struct *)p_struct)->trigger_input      = NO_HARDWARE_TRIGGER;
        ((hal_lpdts_parameter_struct *)p_struct)->sampling_time      = 1U;
        ((hal_lpdts_parameter_struct *)p_struct)->high_threshold_set = 0U;
        ((hal_lpdts_parameter_struct *)p_struct)->low_threshold_set  = 0U;
        break;
    /* initialize LPDTS device information structure with the default values */
    case HAL_LPDTS_DEV_STRUCT:
        ((hal_lpdts_dev_struct *)p_struct)->lpdts_irq_sync.end_of_temperature_handle  = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->lpdts_irq_sync.low_threshold_handle       = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->lpdts_irq_sync.high_threshold_handle      = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->lpdts_irq_async.end_of_temperature_handle = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->lpdts_irq_async.low_threshold_handle      = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->lpdts_irq_async.high_threshold_handle     = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->syn_end_of_temperature_callback           = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->syn_low_threshold_callback                = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->syn_high_threshold_callback               = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->asyn_end_of_temperature_callback          = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->asyn_low_threshold_callback               = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->asyn_high_threshold_callback              = NULL;
        ((hal_lpdts_dev_struct *)p_struct)->mutex                                     = HAL_MUTEX_UNLOCKED;
        ((hal_lpdts_dev_struct *)p_struct)->state                                     = HAL_LPDTS_STATE_NONE;
        break;
    /* initialize LPDTS IRQ information structure with the default values */
    case HAL_LPDTS_IRQ_STRUCT:
        ((hal_lpdts_irq_struct *)p_struct)->end_of_temperature_handle = NULL;
        ((hal_lpdts_irq_struct *)p_struct)->low_threshold_handle      = NULL;
        ((hal_lpdts_irq_struct *)p_struct)->high_threshold_handle     = NULL;
        break;
    case HAL_LPDTS_USER_CALLBACK_STRUCT:
        ((hal_lpdts_irq_user_callback_struct *)p_struct)->syn_end_of_temperature_func  = NULL;
        ((hal_lpdts_irq_user_callback_struct *)p_struct)->syn_low_threshold_func       = NULL;
        ((hal_lpdts_irq_user_callback_struct *)p_struct)->syn_high_threshold_func      = NULL;
        ((hal_lpdts_irq_user_callback_struct *)p_struct)->asyn_end_of_temperature_func = NULL;
        ((hal_lpdts_irq_user_callback_struct *)p_struct)->asyn_low_threshold_func      = NULL;
        ((hal_lpdts_irq_user_callback_struct *)p_struct)->asyn_high_threshold_func     = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      initialize the LPDTS
    \param[in]  lpdts_dev: LPDTS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
    \param[in]  lpdts: the initialization data needed to initialize LPDTS_CFG
                  ref_clock: reference clock selection
                  only one parameter can be selected from the following:
      \arg:     REF_PCLK: high speed reference clock (PCLK)
      \arg:     REF_LXTAL: low speed reference clock (LXTAL)
                  trigger_input: trigger input selection
                  only one parameter can be selected from the following:
      \arg:     NO_HARDWARE_TRIGGER: no hardware trigger input
      \arg:     LPDTS_TRG: LPDTS trigger input
                  sampling_time: (1-15)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS,HAL_ERR_VAL,HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_lpdts_init(hal_lpdts_dev_struct *lpdts_dev, hal_lpdts_parameter_struct *lpdts)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == lpdts_dev) || (NULL == lpdts)) {
        HAL_DEBUGE("pointer [lpdts_dev] or [lpdts] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == lpdts->sampling_time) || (lpdts->sampling_time > 15U)) {
        HAL_DEBUGE("parameter [sampling_time] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    lpdts_dev->state = HAL_LPDTS_STATE_BUSY;

    /* reset clock selection, input trigger selection, sampling time*/
    LPDTS_CFG &= ~(uint32_t)(LPDTS_CFG_REFSEL | LPDTS_CFG_ITSEL | LPDTS_CFG_SPT);
    /* select clock */
    LPDTS_CFG |= lpdts->ref_clock;
    /* select input trigger source */
    LPDTS_CFG |= lpdts->trigger_input;
    /* set sampling time */
    LPDTS_CFG = (uint32_t)(lpdts->sampling_time << 16U);

    /* set LPDTS high threshold value */
    hals_lpdts_high_threshold_set(lpdts->high_threshold_set);
    /* set LPDTS low threshold value */
    hals_lpdts_low_threshold_set(lpdts->low_threshold_set);

    /* change LPDTS state */
    lpdts_dev->state = HAL_LPDTS_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      start LPDTS with interrupt
    \param[in]  lpdts_dev: LPDTS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_BUSY, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_lpdts_start_interrupt(hal_lpdts_dev_struct *lpdts_dev, hal_lpdts_irq_user_callback_struct *p_user_func)
{
    uint32_t tick_start = 0U;
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == lpdts_dev) {
        HAL_DEBUGE("pointer [lpdts_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock LPDTS */
    HAL_LOCK(lpdts_dev);

    if(HAL_LPDTS_STATE_READY == lpdts_dev->state) {
        lpdts_dev->state = HAL_LPDTS_STATE_BUSY;
        /* check clock speed */
        if(REF_LXTAL == (LPDTS_CFG & LPDTS_CFG_REFSEL)) {
            /* clear the LPDTS interrupt flag */
            hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_EMA | LPDTS_INT_FLAG_LTA | LPDTS_INT_FLAG_HTA);
            /* enable async interrupt */
            hals_lpdts_interrupt_enable(LPDTS_INT_EMA | LPDTS_INT_LTA | LPDTS_INT_HTA);

            /* configuration drive handle */
            lpdts_dev->lpdts_irq_async.end_of_temperature_handle = _lpdts_async_end_of_temperature_handle;
            lpdts_dev->lpdts_irq_async.low_threshold_handle      = _lpdts_async_low_threshold_handle;
            lpdts_dev->lpdts_irq_async.high_threshold_handle     = _lpdts_async_high_threshold_handle;

            /* clear user callback */
            lpdts_dev->asyn_end_of_temperature_callback = NULL;
            lpdts_dev->asyn_low_threshold_callback      = NULL;
            lpdts_dev->asyn_high_threshold_callback     = NULL;

            /* configuration user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->asyn_end_of_temperature_func) {
                    lpdts_dev->asyn_end_of_temperature_callback = (void *)p_user_func->asyn_end_of_temperature_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->asyn_low_threshold_func) {
                    lpdts_dev->asyn_low_threshold_callback = (void *)p_user_func->asyn_low_threshold_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->asyn_high_threshold_func) {
                    lpdts_dev->asyn_high_threshold_callback = (void *)p_user_func->asyn_high_threshold_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* clear the LPDTS interrupt flag */
            hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_EM | LPDTS_INT_FLAG_LT | LPDTS_INT_FLAG_HT);
            /* enable sync interrupt */
            hals_lpdts_interrupt_enable(LPDTS_INT_EM | LPDTS_INT_LT | LPDTS_INT_HT);

            /* configuration drive handle */
            lpdts_dev->lpdts_irq_sync.end_of_temperature_handle = _lpdts_sync_end_of_temperature_handle;
            lpdts_dev->lpdts_irq_sync.low_threshold_handle      = _lpdts_sync_low_threshold_handle;
            lpdts_dev->lpdts_irq_sync.high_threshold_handle     = _lpdts_sync_high_threshold_handle;

            /* clear user callback */
            lpdts_dev->syn_end_of_temperature_callback = NULL;
            lpdts_dev->syn_low_threshold_callback      = NULL;
            lpdts_dev->syn_high_threshold_callback     = NULL;

            /* configuration user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->syn_end_of_temperature_func) {
                    lpdts_dev->syn_end_of_temperature_callback = (void *)p_user_func->syn_end_of_temperature_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->syn_low_threshold_func) {
                    lpdts_dev->syn_low_threshold_callback = (void *)p_user_func->syn_low_threshold_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->syn_high_threshold_func) {
                    lpdts_dev->syn_high_threshold_callback = (void *)p_user_func->syn_high_threshold_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }

        hals_lpdts_enable();

        /* wait for the temperature transition to complete */
        tick_start = hal_sys_basetick_count_get();
        while(RESET == (LPDTS_STAT & LPDTS_STAT_TSRF)) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, LPDTS_TIMEOUT)) {
                HAL_DEBUGE(" LPDTS wait temperature sensor ready flag timeout !");
                ret = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        }

        /* change the state to ready */
        lpdts_dev->state = HAL_LPDTS_STATE_READY;
    } else {
        /* unlock LPDTS */
        HAL_UNLOCK(lpdts_dev);
        ret = HAL_ERR_BUSY;
    }

    /* unlock LPDTS */
    HAL_UNLOCK(lpdts_dev);

    return ret;
}

/*!
    \brief      stop LPDTS with interrupt
    \param[in]  lpdts_dev: LPDTS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_lpdts_stop_interrupt(hal_lpdts_dev_struct *lpdts_dev)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == lpdts_dev) {
        HAL_DEBUGE("pointer [lpdts_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(lpdts_dev);

    if(HAL_LPDTS_STATE_READY == lpdts_dev->state) {
        lpdts_dev->state = HAL_LPDTS_STATE_BUSY;
        /* check clock speed */
        if(REF_LXTAL == (LPDTS_CFG & LPDTS_CFG_REFSEL)) {
            /* clear the LPDTS interrupt flag */
            hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_EMA | LPDTS_INT_FLAG_LTA | LPDTS_INT_FLAG_HTA);
            /* disable async interrupt */
            hals_lpdts_interrupt_disable(LPDTS_INT_EMA | LPDTS_INT_LTA | LPDTS_INT_HTA);

            lpdts_dev->lpdts_irq_async.end_of_temperature_handle = NULL;
            lpdts_dev->lpdts_irq_async.low_threshold_handle      = NULL;
            lpdts_dev->lpdts_irq_async.high_threshold_handle     = NULL;
        } else {
            /* clear the LPDTS interrupt flag */
            hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_EM | LPDTS_INT_FLAG_LT | LPDTS_INT_FLAG_HT);
            /* disable sync interrupt */
            hals_lpdts_interrupt_disable(LPDTS_INT_EM | LPDTS_INT_LT | LPDTS_INT_HT);

            lpdts_dev->lpdts_irq_sync.end_of_temperature_handle = NULL;
            lpdts_dev->lpdts_irq_sync.low_threshold_handle      = NULL;
            lpdts_dev->lpdts_irq_sync.high_threshold_handle     = NULL;
        }

        hals_lpdts_disable();

        lpdts_dev->state = HAL_LPDTS_STATE_READY;
    } else {
        HAL_DEBUGE("LPDTS is busy");
        ret = HAL_ERR_BUSY;
    }

    HAL_UNLOCK(lpdts_dev);

    return ret;
}

/*!
    \brief      LPDTS interrupt handler content function,which is merely used in lpdts_handler
    \param[in]  lpdts_dev: LPDTS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_lpdts_irq(hal_lpdts_dev_struct *lpdts_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == lpdts_dev) {
        HAL_DEBUGE("pointer [lpdts_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check whether the end of measurement sync interrupt is set or not */
    if(SET == (hals_lpdts_interrupt_flag_get(LPDTS_INT_FLAG_EM))) {
        hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_EM);
        /* end of measurement sync interrupt handle */
        if(NULL != lpdts_dev->lpdts_irq_sync.end_of_temperature_handle) {
            lpdts_dev->lpdts_irq_sync.end_of_temperature_handle(lpdts_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the low threshold sync interrupt is set or not */
    if(SET == (hals_lpdts_interrupt_flag_get(LPDTS_INT_FLAG_LT))) {
        hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_LT);
        /* low threshold sync interrupt handle */
        if(NULL != lpdts_dev->lpdts_irq_sync.low_threshold_handle) {
            lpdts_dev->lpdts_irq_sync.low_threshold_handle(lpdts_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
    /* check whether the high threshold sync interrupt is set or not */
    if(SET == (hals_lpdts_interrupt_flag_get(LPDTS_INT_FLAG_HT))) {
        hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_HT);
        /* high threshold sync interrupt handle */
        if(NULL != lpdts_dev->lpdts_irq_sync.high_threshold_handle) {
            lpdts_dev->lpdts_irq_sync.high_threshold_handle(lpdts_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the end of measurement async interrupt is set or not */
    if(SET == (hals_lpdts_interrupt_flag_get(LPDTS_INT_FLAG_EMA))) {
        hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_EMA);
        /* end of measurement sync interrupt handle */
        if(NULL != lpdts_dev->lpdts_irq_async.end_of_temperature_handle) {
            lpdts_dev->lpdts_irq_async.end_of_temperature_handle(lpdts_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the low threshold async interrupt is set or not */
    if(SET == (hals_lpdts_interrupt_flag_get(LPDTS_INT_FLAG_LTA))) {
        hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_LTA);
        /* low threshold sync interrupt handle */
        if(NULL != lpdts_dev->lpdts_irq_async.low_threshold_handle) {
            lpdts_dev->lpdts_irq_async.low_threshold_handle(lpdts_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the high threshold async interrupt is set or not */
    if(SET == (hals_lpdts_interrupt_flag_get(LPDTS_INT_FLAG_HTA))) {
        hals_lpdts_interrupt_flag_clear(LPDTS_INT_FLAG_HTA);
        /* high threshold sync interrupt handle */
        if(NULL != lpdts_dev->lpdts_irq_async.high_threshold_handle) {
            lpdts_dev->lpdts_irq_async.high_threshold_handle(lpdts_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  lpdts_dev: LPDTS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  lpdts_irq_sync: point to LPDTS interrupt callback functions structure for sync interrupt
                  please refer to hal_lpdts_irq_struct
    \param[in]  lpdts_irq_async: point to LPDTS interrupt callback functions structure for async interrupt
                  please refer to hal_lpdts_irq_struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_lpdts_irq_handle_set(hal_lpdts_dev_struct *lpdts_dev, hal_lpdts_irq_struct *lpdts_irq_sync, \
                                 hal_lpdts_irq_struct *lpdts_irq_async)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == lpdts_dev) {
        HAL_DEBUGE("pointer [lpdts_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((NULL == lpdts_irq_sync) && (NULL == lpdts_irq_async)) {
        HAL_DEBUGE("pointer [lpdts_irq_sync] and [lpdts_irq_async] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((REF_LXTAL == (LPDTS_CFG & LPDTS_CFG_REFSEL)) && (NULL == lpdts_irq_async)) {
        HAL_DEBUGE("low speed clock must use async interrupt");
        return HAL_ERR_NO_SUPPORT;
    }

    if((REF_PCLK == (LPDTS_CFG & LPDTS_CFG_REFSEL)) && (NULL == lpdts_irq_sync)) {
        HAL_DEBUGE("high speed clock must use sync interrupt");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* LPDTS sync interrupt for end of measurement */
    if(NULL != lpdts_irq_sync->end_of_temperature_handle) {
        lpdts_dev->lpdts_irq_sync.end_of_temperature_handle = lpdts_irq_sync->end_of_temperature_handle;
        /* enable the lpdts interrupt */
        hals_lpdts_interrupt_enable(LPDTS_INT_EM);
    } else {
        lpdts_dev->lpdts_irq_sync.end_of_temperature_handle = NULL;
        /* disable the lpdts interrupt */
        hals_lpdts_interrupt_disable(LPDTS_INT_EM);
    }

    /* LPDTS sync interrupt for low threshold */
    if(NULL != lpdts_irq_sync->low_threshold_handle) {
        lpdts_dev->lpdts_irq_sync.low_threshold_handle = lpdts_irq_sync->low_threshold_handle;
        /* enable the lpdts interrupt */
        hals_lpdts_interrupt_enable(LPDTS_INT_LT);
    } else {
        lpdts_dev->lpdts_irq_sync.low_threshold_handle = NULL;
        /* disable the lpdts interrupt */
        hals_lpdts_interrupt_disable(LPDTS_INT_LT);
    }

    /* LPDTS sync interrupt for high threshold */
    if(NULL != lpdts_irq_sync->high_threshold_handle) {
        lpdts_dev->lpdts_irq_sync.high_threshold_handle = lpdts_irq_sync->high_threshold_handle;
        /* enable the lpdts interrupt */
        hals_lpdts_interrupt_enable(LPDTS_INT_HT);
    } else {
        lpdts_dev->lpdts_irq_sync.high_threshold_handle = NULL;
        /* disable the lpdts interrupt */
        hals_lpdts_interrupt_disable(LPDTS_INT_HT);
    }

    /* LPDTS async interrupt for end of measurement */
    if(NULL != lpdts_irq_async->end_of_temperature_handle) {
        lpdts_dev->lpdts_irq_async.end_of_temperature_handle = lpdts_irq_async->end_of_temperature_handle;
        /* enable the lpdts interrupt */
        hals_lpdts_interrupt_enable(LPDTS_INT_EMA);
    } else {
        lpdts_dev->lpdts_irq_async.end_of_temperature_handle = NULL;
        /* disable the lpdts interrupt */
        hals_lpdts_interrupt_disable(LPDTS_INT_EMA);
    }

    /* LPDTS async interrupt for low threshold */
    if(NULL != lpdts_irq_async->low_threshold_handle) {
        lpdts_dev->lpdts_irq_async.low_threshold_handle = lpdts_irq_async->low_threshold_handle;
        /* enable the lpdts interrupt */
        hals_lpdts_interrupt_enable(LPDTS_INT_LTA);
    } else {
        lpdts_dev->lpdts_irq_async.low_threshold_handle = NULL;
        /* disable the lpdts interrupt */
        hals_lpdts_interrupt_disable(LPDTS_INT_LTA);
    }

    /* LPDTS async interrupt for high threshold */
    if(NULL != lpdts_irq_async->high_threshold_handle) {
        lpdts_dev->lpdts_irq_async.high_threshold_handle = lpdts_irq_async->high_threshold_handle;
        /* enable the lpdts interrupt */
        hals_lpdts_interrupt_enable(LPDTS_INT_HTA);
    } else {
        lpdts_dev->lpdts_irq_async.high_threshold_handle = NULL;
        /* disable the lpdts interrupt */
        hals_lpdts_interrupt_disable(LPDTS_INT_HTA);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  lpdts_dev: LPDTS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_lpdts_irq_handle_all_reset(hal_lpdts_dev_struct *lpdts_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == lpdts_dev) {
        HAL_DEBUGE("pointer [lpdts_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset all irq handle */
    lpdts_dev->lpdts_irq_sync.end_of_temperature_handle  = NULL;
    lpdts_dev->lpdts_irq_sync.low_threshold_handle       = NULL;
    lpdts_dev->lpdts_irq_sync.high_threshold_handle      = NULL;
    lpdts_dev->lpdts_irq_async.end_of_temperature_handle = NULL;
    lpdts_dev->lpdts_irq_async.low_threshold_handle      = NULL;
    lpdts_dev->lpdts_irq_async.high_threshold_handle     = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      get LPDTS state
    \param[in]  lpdts_dev: LPDTS device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     LPDTS state: refer to hal_lpdts_state_enum
*/
hal_lpdts_state_enum hal_lpdts_state_get(hal_lpdts_dev_struct *lpdts_dev)
{
    return (lpdts_dev->state);
}

/*!
    \brief      start LPDTS temperature sensor and wait for the temperature transition to complete(Software trigger)
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_NONE， HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_lpdts_start(void)
{
    uint32_t tick_start = 0U;
    int32_t  ret        = HAL_ERR_NONE;

    /* enable LPDTS temperature sensor */
    LPDTS_CFG |= LPDTS_CFG_TSEN;

    /* wait for the temperature transition to complete */
    tick_start = hal_sys_basetick_count_get();
    while(RESET == (LPDTS_STAT & LPDTS_FLAG_TSR)) {
        if(SET == hal_sys_basetick_timeout_check(tick_start, LPDTS_TIMEOUT)) {
            HAL_DEBUGE(" LPDTS wait temperature sensor ready flag timeout !");
            ret = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* do nothing */
        }
    }

    /* check the ret value */
    if(HAL_ERR_NONE == ret) {
        /* enable software trigger */
        if(NO_HARDWARE_TRIGGER == (LPDTS_CFG & LPDTS_CFG_ITSEL)) {
            hals_lpdts_soft_trigger_enable();
        } else {
            /* do nothing */
        }

        /* wait for the temperature transition to complete */
        tick_start = hal_sys_basetick_count_get();
        while(RESET == (LPDTS_STAT & LPDTS_FLAG_TSR)) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, LPDTS_TIMEOUT)) {
                HAL_DEBUGE(" LPDTS wait temperature sensor ready flag timeout !");
                ret = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      stop LPDTS temperature sensor and Software trigger
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_lpdts_stop(void)
{
    LPDTS_CFG &= ~LPDTS_CFG_TSEN;

    /* disable software trigger */
    if(NO_HARDWARE_TRIGGER == (LPDTS_CFG & LPDTS_CFG_ITSEL)) {
        hals_lpdts_soft_trigger_disable();
    } else {
        /* do nothing */
    }
}

/*!
    \brief      get temperature from LPDTS
    \param[in]  none
    \param[out] none
    \retval     temperature: temperature value
*/
int32_t hal_lpdts_temperature_get(void)
{
    uint32_t freq;
    uint32_t count;
    uint32_t t0;
    uint32_t t0_freq;
    uint32_t ramp_coeff;
    uint32_t reg_cfg;
    int32_t temperature;

    /* get the total number of samples */
    count = (LPDTS_DATA & LPDTS_DATA_COVAL);
    /* get LPDTS_CFG configuration */
    reg_cfg = LPDTS_CFG;

    /* get the module frequency on Hz */
    if(LPDTS_CFG_REFSEL == (reg_cfg & LPDTS_CFG_REFSEL)) {
        freq = (LXTAL_VALUE * count) / (2U * ((reg_cfg & LPDTS_CFG_SPT) >> LPDTS_CFG_SPT_OFFSET));
    } else {
        freq = (2U * hal_rcu_clock_freq_get(CK_APB1) / count) * ((reg_cfg & LPDTS_CFG_SPT) >> LPDTS_CFG_SPT_OFFSET);
    }

    /* read factory settings */
    t0 = (LPDTS_SDATA & LPDTS_SDATA_VAL) >> LPDTS_SDATA_VAL_OFFSET;
    if(0U == t0) {
        t0 = LPDTS_T0_TMP_VAL;
    } else {
        /* do nothing */
    }

    /* get the T0 frequency on Hz */
    t0_freq = (LPDTS_SDATA & LPDTS_SDATA_FREQ) * 100U;
    /* get the ramp coefficient for the temperature sensor on deg C/Hz */
    ramp_coeff = LPDTS_RDATA & LPDTS_RDATA_RCVAL;

    /* figure out the temperature deg C */
    if(0U != ramp_coeff) {
        temperature = (int32_t)t0 + (((int32_t)freq - (int32_t)t0_freq) / (int32_t)ramp_coeff);
    } else {
        temperature = 0;
    }

    return temperature;
}

/*!
    \brief      enable LPDTS temperature sensor
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_lpdts_enable(void)
{
    LPDTS_CFG |= LPDTS_CFG_TSEN;
}

/*!
    \brief      disable LPDTS temperature sensor
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_lpdts_disable(void)
{
    LPDTS_CFG &= ~LPDTS_CFG_TSEN;
}

/*!
    \brief      enable LPDTS software trigger
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_lpdts_soft_trigger_enable(void)
{
    LPDTS_CFG |= LPDTS_CFG_TRGS;
}

/*!
    \brief      disable LPDTS software trigger
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_lpdts_soft_trigger_disable(void)
{
    LPDTS_CFG &= ~LPDTS_CFG_TRGS;
}

/*!
    \brief      configure LPDTS high threshold value
    \param[in]  value: high threshold value(0~65535)
    \param[out] none
    \retval     none
*/
void hals_lpdts_high_threshold_set(uint16_t value)
{
    uint32_t reg;

    /* configure the LPDTS_IT */
    reg = LPDTS_IT;
    reg &= ~LPDTS_IT_INTHT;
    reg |= (uint32_t)value << LPDTS_IT_INTHT_OFFSET;

    LPDTS_IT = reg;
}

/*!
    \brief      configure LPDTS low threshold value
    \param[in]  value: low threshold value(0~65535)
    \param[out] none
    \retval     none
*/
void hals_lpdts_low_threshold_set(uint16_t value)
{
    uint32_t reg;

    /* configure the LPDTS_IT */
    reg = LPDTS_IT;
    reg &= ~LPDTS_IT_INTLT;
    reg |= (uint32_t)value;

    LPDTS_IT = reg;
}

/*!
    \brief      get LPDTS flag
    \param[in]  flag: LPDTS ready flag
                only one parameter can be selected which is shown as below:
      \arg        LPDTS_FLAG_TSR
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_lpdts_flag_get(uint32_t flag)
{
    FlagStatus status = RESET;

    if(LPDTS_STAT & flag) {
        status = SET;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      enable LPDTS interrupt
    \param[in]  interrupt: the LPDTS interrupt
                one or more parameters can be selected which is shown as below:
      \arg        LPDTS_INT_EM: enable LPDTS interrupt on temperature sensor error
      \arg        LPDTS_INT_LT: enable LPDTS interrupt on temperature sensor low threshold
      \arg        LPDTS_INT_HT: enable LPDTS interrupt on temperature sensor high threshold
      \arg        LPDTS_INT_EMA: enable LPDTS interrupt on temperature sensor error active
      \arg        LPDTS_INT_LTA: enable LPDTS interrupt on temperature sensor low threshold active
      \arg        LPDTS_INT_HTA: enable LPDTS interrupt on temperature sensor high threshold active
    \param[out] none
    \retval     none
*/
void hals_lpdts_interrupt_enable(uint32_t interrupt)
{
    LPDTS_INTEN |= interrupt;
}

/*!
    \brief      disable LPDTS interrupt
    \param[in]  interrupt: the LPDTS interrupt
                one or more parameters can be selected which is shown as below:
      \arg        LPDTS_INT_EM: disable LPDTS interrupt on temperature sensor error
      \arg        LPDTS_INT_LT: disable LPDTS interrupt on temperature sensor low threshold
      \arg        LPDTS_INT_HT: disable LPDTS interrupt on temperature sensor high threshold
      \arg        LPDTS_INT_EMA: disable LPDTS interrupt on temperature sensor error active
      \arg        LPDTS_INT_LTA: disable LPDTS interrupt on temperature sensor low threshold active
      \arg        LPDTS_INT_HTA: disable LPDTS interrupt on temperature sensor high threshold active
    \param[out] none
    \retval     none
*/
void hals_lpdts_interrupt_disable(uint32_t interrupt)
{
    LPDTS_INTEN &= ~interrupt;
}

/*!
    \brief      get LPDTS interrupt flag
    \param[in]  flag: LPDTS interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        LPDTS_INT_FLAG_EM: get LPDTS interrupt flag on temperature sensor error
      \arg        LPDTS_INT_FLAG_LT: get LPDTS interrupt flag on temperature sensor low threshold
      \arg        LPDTS_INT_FLAG_HT: get LPDTS interrupt flag on temperature sensor high threshold
      \arg        LPDTS_INT_FLAG_EMA: get LPDTS interrupt flag on temperature sensor error active
      \arg        LPDTS_INT_FLAG_LTA: get LPDTS interrupt flag on temperature sensor low threshold active
      \arg        LPDTS_INT_FLAG_HTA: get LPDTS interrupt flag on temperature sensor high threshold active
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_lpdts_interrupt_flag_get(uint32_t flag)
{
    FlagStatus status = RESET;
    uint32_t state;

    state = LPDTS_STAT;
    if(state & flag) {
        state = LPDTS_INTEN;
        if(state & flag) {
            status = SET;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      clear the LPDTS interrupt flag
    \param[in]  flag: LPDTS flag
                one or more parameter can be selected which is shown as below:
      \arg        LPDTS_INT_FLAG_EM: clear LPDTS interrupt flag on temperature sensor error
      \arg        LPDTS_INT_FLAG_LT: clear LPDTS interrupt flag on temperature sensor low threshold
      \arg        LPDTS_INT_FLAG_HT: clear LPDTS interrupt flag on temperature sensor high threshold
      \arg        LPDTS_INT_FLAG_EMA: clear LPDTS interrupt flag on temperature sensor error active
      \arg        LPDTS_INT_FLAG_LTA: clear LPDTS interrupt flag on temperature sensor low threshold active
      \arg        LPDTS_INT_FLAG_HTA: clear LPDTS interrupt flag on temperature sensor high threshold active
    \param[out] none
    \retval     none
*/
void hals_lpdts_interrupt_flag_clear(uint32_t flag)
{
    LPDTS_INTC = (uint32_t)flag;
}

/*!
    \brief      lpdts async temperature end interrupt handle
    \param[in]  lpdts_dev: pointer to a lpdts device information structure
    \param[out] none
    \retval     none
*/
static void _lpdts_async_end_of_temperature_handle(void *lpdts_dev)
{
    hal_lpdts_dev_struct *p_lpdts = lpdts_dev;
    hal_lpdts_user_cb p_func      = (hal_lpdts_user_cb)p_lpdts->asyn_end_of_temperature_callback;

    if(NULL != p_func) {
        p_func(p_lpdts);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      lpdts low threshold interrupt handle
    \param[in]  lpdts_dev: pointer to a lpdts device information structure
    \param[out] none
    \retval     none
*/
static void _lpdts_async_low_threshold_handle(void *lpdts_dev)
{
    hal_lpdts_dev_struct *p_lpdts = lpdts_dev;
    hal_lpdts_user_cb p_func      = (hal_lpdts_user_cb)p_lpdts->asyn_low_threshold_callback;

    if(NULL != p_func) {
        p_func(p_lpdts);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      lpdts high threshold interrupt handle
    \param[in]  lpdts_dev: pointer to a lpdts device information structure
    \param[out] none
    \retval     none
*/
static void _lpdts_async_high_threshold_handle(void *lpdts_dev)
{
    hal_lpdts_dev_struct *p_lpdts = lpdts_dev;
    hal_lpdts_user_cb p_func      = (hal_lpdts_user_cb)p_lpdts->asyn_high_threshold_callback;

    if(NULL != p_func) {
        p_func(p_lpdts);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      lpdts sync temperature end interrupt handle
    \param[in]  lpdts_dev: pointer to a lpdts device information structure
    \param[out] none
    \retval     none
*/
static void _lpdts_sync_end_of_temperature_handle(void *lpdts_dev)
{
    hal_lpdts_dev_struct *p_lpdts = lpdts_dev;
    hal_lpdts_user_cb p_func      = (hal_lpdts_user_cb)p_lpdts->syn_end_of_temperature_callback;

    if(NULL != p_func) {
        p_func(p_lpdts);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      lpdts low threshold interrupt handle
    \param[in]  lpdts_dev: pointer to a lpdts device information structure
    \param[out] none
    \retval     none
*/
static void _lpdts_sync_low_threshold_handle(void *lpdts_dev)
{
    hal_lpdts_dev_struct *p_lpdts = lpdts_dev;
    hal_lpdts_user_cb p_func      = (hal_lpdts_user_cb)p_lpdts->syn_low_threshold_callback;

    if(NULL != p_func) {
        p_func(p_lpdts);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      lpdts high threshold interrupt handle
    \param[in]  lpdts_dev: pointer to a lpdts device information structure
    \param[out] none
    \retval     none
*/
static void _lpdts_sync_high_threshold_handle(void *lpdts_dev)
{
    hal_lpdts_dev_struct *p_lpdts = lpdts_dev;
    hal_lpdts_user_cb p_func      = (hal_lpdts_user_cb)p_lpdts->syn_high_threshold_callback;

    if(NULL != p_func) {
        p_func(p_lpdts);
    } else {
        /* do nothing */
    }
}
