/*!
    \file    gd32h7xx_hal_i2s.c
    \brief   I2S driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define I2S_FRAME_MCKOUT_DIV            (256U)                /*!< Divider for master clock output enable */
#define I2S_FRAME_CH16B_DIV             (32U)                 /*!< Divider for 16-bit channel frame format */
#define I2S_FRAME_CH32B_DIV             (64U)                 /*!< Divider for 32-bit channel frame format */
#define I2S_STOP_TRANS_TIMEOUT_VALUE    (100U)                /*!< Timeout value for I2S stop transfer */

/*!< Validate I2S peripheral */
#define IS_I2S_PERIPH(PERIPH) (((PERIPH) == SPI0) || ((PERIPH) == SPI1) || \
                               ((PERIPH) == SPI2) || ((PERIPH) == SPI5))

/*!< Validate I2S operation mode */
#define IS_I2S_MODE(MODE) (((MODE) == I2S_MODE_MASTERRX) || ((MODE) == I2S_MODE_MASTERTX) || \
                           ((MODE) == I2S_MODE_SLAVERX ) || ((MODE) == I2S_MODE_SLAVETX ))

/*!< Validate I2S standard */
#define IS_I2S_STANDARD(STANDARD) (((STANDARD) == I2S_STD_PHILIPS) || ((STANDARD) == I2S_STD_MSB     ) || \
                                   ((STANDARD) == I2S_STD_LSB    ) || ((STANDARD) == I2S_STD_PCMSHORT) || \
                                   ((STANDARD) == I2S_STD_PCMLONG))

/*!< Validate I2S frame format */
#define IS_I2S_FRAMEFORMAT(FORMAT) (((FORMAT) == I2S_FRAMEFORMAT_DT16B_CH16B) || \
                                    ((FORMAT) == I2S_FRAMEFORMAT_DT16B_CH32B) || \
                                    ((FORMAT) == I2S_FRAMEFORMAT_DT24B_CH32B) || \
                                    ((FORMAT) == I2S_FRAMEFORMAT_DT32B_CH32B))

/*!< Validate I2S master clock output */
#define IS_I2S_MCKOUT(MCKOUT) (((MCKOUT) == I2S_MCKOUT_ENABLE) || ((MCKOUT) == I2S_MCKOUT_DISABLE))

/*!< Validate I2S audio sample rate */
#define IS_I2S_AUDIOSAMPLE(SAMPLE) (((SAMPLE) == I2S_AUDIOSAMPLE_8K  ) || ((SAMPLE) == I2S_AUDIOSAMPLE_11K ) || \
                                    ((SAMPLE) == I2S_AUDIOSAMPLE_16K ) || ((SAMPLE) == I2S_AUDIOSAMPLE_22K ) || \
                                    ((SAMPLE) == I2S_AUDIOSAMPLE_32K ) || ((SAMPLE) == I2S_AUDIOSAMPLE_44K ) || \
                                    ((SAMPLE) == I2S_AUDIOSAMPLE_48K ) || ((SAMPLE) == I2S_AUDIOSAMPLE_96K ) || \
                                    ((SAMPLE) == I2S_AUDIOSAMPLE_192K) || ((SAMPLE) == I2S_AUDIOSAMPLE_USER))

/*!<Validate I2S clock polarity */
#define IS_I2S_CKPL(CKPL) (((CKPL) == I2S_CKPL_LOW) || ((CKPL) == I2S_CKPL_HIGH))

/*!<Validate I2S interrupt */
#define IS_I2S_INT(FLAG) (((FLAG) == SPI_I2S_INT_RP   ) || ((FLAG) == SPI_I2S_INT_TP   ) || \
                          ((FLAG) == SPI_I2S_INT_TXURE) || ((FLAG) == SPI_I2S_INT_RXORE) || \
                          ((FLAG) == SPI_I2S_INT_FE   ))

/* wait the flag status until timeout */
static int32_t _i2s_wait_flag_timeout(uint32_t periph, hal_i2s_flag_enum flag, FlagStatus status, uint32_t timeout);
/* transmit complete interrupt */
static void _i2s_transmit_interrupt(void *i2s);
/* receive complete interrupt */
static void _i2s_receive_interrupt(void *i2s);
/* I2S error interrupt handler */
static void _i2s_error_interrupt(void *i2s);
/* I2S DMA receive handler */
static void _i2s_complete_receive_dma(void *dma);
/* I2S DMA receive half complete handler */
static void _i2s_half_complete_receive_dma(void *dma);
/* I2S DMA transmit handler */
static void _i2s_complete_transmit_dma(void *dma);
/* I2S DMA transmit half complete handler */
static void _i2s_half_complete_transmit_dma(void *dma);
/* I2S DMA error handler */
static void _i2s_error_transmit_receive_dma(void *dma);
/* I2S stop transfer */
static void _i2s_stop_transfer(hal_i2s_dev_struct *i2s_dev);
/* I2S start transfer */
static void _i2s_start_transfer(hal_i2s_dev_struct *i2s_dev);

/*!
    \brief      deinitialize I2S
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_deinit(hal_i2s_dev_struct *i2s_dev)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2s_dev) {
        HAL_DEBUGE("pointer [i2s_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(!IS_I2S_PERIPH(i2s_dev->periph)) {
        HAL_DEBUGE("parameter [i2s_dev->periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    i2s_dev->state = HAL_I2S_STATE_BUSY;

    switch(i2s_dev->periph) {
    case SPI0:
        /* reset SPI0 and I2S0 */
        hal_rcu_periph_reset_enable(RCU_SPI0RST);
        hal_rcu_periph_reset_disable(RCU_SPI0RST);
        break;
    case SPI1:
        /* reset SPI1 and I2S1 */
        hal_rcu_periph_reset_enable(RCU_SPI1RST);
        hal_rcu_periph_reset_disable(RCU_SPI1RST);
        break;
    case SPI2:
        /* reset SPI2 and I2S2 */
        hal_rcu_periph_reset_enable(RCU_SPI2RST);
        hal_rcu_periph_reset_disable(RCU_SPI2RST);
        break;
    case SPI5:
        /* reset SPI5 and I2S5 */
        hal_rcu_periph_reset_enable(RCU_SPI5RST);
        hal_rcu_periph_reset_disable(RCU_SPI5RST);
        break;
    default:
        HAL_DEBUGE("I2S peripheral is invalid");
        retval = HAL_ERR_VAL;
        break;
    }

    i2s_dev->state = HAL_I2S_STATE_RESET;

    return retval;
}

/*!
    \brief      initialize I2S structure
    \param[in]  hal_struct_type: type of I2S structure for initialization
                only one parameters can be selected which are shown as below:
      \arg        HAL_I2S_INIT_STRUCT: initialization structure
      \arg        HAL_I2S_DEV_STRUCT: device structure
      \arg        HAL_I2S_IRQ_USER_CALLBACK_STRUCT: interrupt callback structure
    \param[out]  p_struct: point to I2S structure that contains the configuration information
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_struct_init(hal_i2s_struct_type_enum hal_struct_type, void *p_struct)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((HAL_I2S_INIT_STRUCT != hal_struct_type) && (HAL_I2S_DEV_STRUCT != hal_struct_type)) {
        HAL_DEBUGE("parameter [hal_struct_type] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_I2S_INIT_STRUCT:
        /* initialize I2S initialization structure with the default values */
        ((hal_i2s_init_struct *)p_struct)->mode            = I2S_MODE_SLAVETX;
        ((hal_i2s_init_struct *)p_struct)->standard        = I2S_STD_PHILIPS;
        ((hal_i2s_init_struct *)p_struct)->frameformat     = I2S_FRAMEFORMAT_DT16B_CH16B;
        ((hal_i2s_init_struct *)p_struct)->mckout          = I2S_MCKOUT_DISABLE;
        ((hal_i2s_init_struct *)p_struct)->audiosample     = I2S_AUDIOSAMPLE_8K;
        ((hal_i2s_init_struct *)p_struct)->ckpl            = I2S_CKPL_LOW;
        ((hal_i2s_init_struct *)p_struct)->user_input      = (uint32_t)I2S_AUDIOSAMPLE_8K;
        ((hal_i2s_init_struct *)p_struct)->word_access     = I2S_HALF_WORD_ACCESS;
        ((hal_i2s_init_struct *)p_struct)->keep_io_state   = I2S_KEEPIO_DISABLE;
        ((hal_i2s_init_struct *)p_struct)->first_bit       = I2S_FIRST_MSB;
        break;
    case HAL_I2S_DEV_STRUCT:
        /* initialize I2S device information structure with the default values */
        ((hal_i2s_dev_struct *)p_struct)->p_dma_rx                 = NULL;
        ((hal_i2s_dev_struct *)p_struct)->p_dma_tx                 = NULL;
        ((hal_i2s_dev_struct *)p_struct)->txbuffer.buffer          = NULL;
        ((hal_i2s_dev_struct *)p_struct)->txbuffer.length          = 0U;
        ((hal_i2s_dev_struct *)p_struct)->txbuffer.pos             = 0U;
        ((hal_i2s_dev_struct *)p_struct)->rxbuffer.buffer          = NULL;
        ((hal_i2s_dev_struct *)p_struct)->rxbuffer.length          = 0U;
        ((hal_i2s_dev_struct *)p_struct)->rxbuffer.pos             = 0U;
        ((hal_i2s_dev_struct *)p_struct)->rx_callback              = NULL;
        ((hal_i2s_dev_struct *)p_struct)->rx_half_callback         = NULL;
        ((hal_i2s_dev_struct *)p_struct)->tx_callback              = NULL;
        ((hal_i2s_dev_struct *)p_struct)->tx_half_callback         = NULL;
        ((hal_i2s_dev_struct *)p_struct)->error_callback           = NULL;
        ((hal_i2s_dev_struct *)p_struct)->state                    = HAL_I2S_STATE_READY;
        ((hal_i2s_dev_struct *)p_struct)->mutex                    = HAL_MUTEX_UNLOCKED;
        ((hal_i2s_dev_struct *)p_struct)->i2s_irq.receive_handler  = NULL;
        ((hal_i2s_dev_struct *)p_struct)->i2s_irq.transmit_handler = NULL;
        ((hal_i2s_dev_struct *)p_struct)->i2s_irq.error_handler    = NULL;
        break;
    case HAL_I2S_IRQ_USER_CALLBACK_STRUCT:
        /* initialize I2S interrupt callback structure with the default values */
        ((hal_i2s_user_callback_struct *)p_struct)->half_complete_func = NULL;
        ((hal_i2s_user_callback_struct *)p_struct)->complete_func      = NULL;
        ((hal_i2s_user_callback_struct *)p_struct)->error_func         = NULL;
        break;
    default:
        /* invalid struct type: log error and set return val */
        HAL_DEBUGE("parameter [hal_struct_type] value is invalid");
        retval = HAL_ERR_VAL;
        break;
    }

    return retval;
}

/*!
    \brief      initialize I2S
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[in]  p_init:
                mode: I2S operation mode
                only one parameter can be selected which is shown as below:
      \arg        I2S_MODE_SLAVETX: I2S slave transmit mode
      \arg        I2S_MODE_SLAVERX: I2S slave receive mode
      \arg        I2S_MODE_MASTERTX: I2S master transmit mode
      \arg        I2S_MODE_MASTERRX: I2S master receive mode
                standard: I2S standard mode
                only one parameter can be selected which is shown as below:
      \arg        I2S_STD_PHILIPS: I2S philips standard
      \arg        I2S_STD_MSB: I2S MSB standard
      \arg        I2S_STD_LSB: I2S LSB standard
      \arg        I2S_STD_PCMSHORT: I2S PCM short standard
      \arg        I2S_STD_PCMLONG: I2S PCM long standard
                ckpl: I2S clock polarity
                only one parameter can be selected which is shown as below:
      \arg        I2S_CKPL_LOW: I2S clock polarity low level
      \arg        I2S_CKPL_HIGH: I2S clock polarity high level
                mckout: I2S master clock output
                only one parameter can be selected which is shown as below:
      \arg        I2S_MCKOUT_ENABLE: I2S master clock output enable
      \arg        I2S_MCKOUT_DISABLE: I2S master clock output disable
                audiosample: I2S audio sample rate
                only one parameter can be selected which is shown as below:
      \arg        I2S_AUDIOSAMPLE_USER: audio sample user input frequency
      \arg        I2S_AUDIOSAMPLE_8K: audio sample rate is 8KHz
      \arg        I2S_AUDIOSAMPLE_11K: audio sample rate is 11KHz
      \arg        I2S_AUDIOSAMPLE_16K: audio sample rate is 16KHz
      \arg        I2S_AUDIOSAMPLE_22K: audio sample rate is 22KHz
      \arg        I2S_AUDIOSAMPLE_32K: audio sample rate is 32KHz
      \arg        I2S_AUDIOSAMPLE_44K: audio sample rate is 44KHz
      \arg        I2S_AUDIOSAMPLE_48K: audio sample rate is 48KHz
      \arg        I2S_AUDIOSAMPLE_96K: audio sample rate is 96KHz
      \arg        I2S_AUDIOSAMPLE_192K: audio sample rate is 192KHz
                frameformat: I2S frame format
                only one parameter can be selected which is shown as below:
      \arg        I2S_FRAMEFORMAT_DT16B_CH16B: I2S data length is 16 bit and channel length is 16 bit
      \arg        I2S_FRAMEFORMAT_DT16B_CH32B: I2S data length is 16 bit and channel length is 32 bit
      \arg        I2S_FRAMEFORMAT_DT24B_CH32B: I2S data length is 24 bit and channel length is 32 bit
      \arg        I2S_FRAMEFORMAT_DT32B_CH32B: I2S data length is 32 bit and channel length is 32 bit
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_init(hal_i2s_dev_struct *i2s_dev, uint32_t periph, hal_i2s_init_struct *p_init)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

    uint32_t i2sdiv = 2U, i2sof = 0U;
    uint32_t clks               = 0U;
    uint32_t i2sclock           = 0U;
    uint32_t i2s_clk_sel        = 0U;
    uint32_t spi0_2_clksel[5]   = {CK_PLL0Q, CK_PLL1P, CK_PLL2P, I2S_CKIN_VALUE, CK_PER};
    uint32_t spi3_5_clksel[7]   = {CK_APB2, CK_PLL1Q, CK_PLL2Q, CK_IRC64MDIV, CK_LPIRC4M, CK_HXTAL, I2S_CKIN_VALUE};

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
     /* check pin_init address */
    if((NULL == p_init) || (NULL == i2s_dev)) {
        HAL_DEBUGE("pointer [p_init] or [i2s_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph value */
    if(!IS_I2S_PERIPH(periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check mode value */
    if(!IS_I2S_MODE(p_init->mode)) {
        HAL_DEBUGE("parameter [mode] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check standard value */
    if(!IS_I2S_STANDARD(p_init->standard)) {
        HAL_DEBUGE("parameter [standard] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check frameformat value */
    if(!IS_I2S_FRAMEFORMAT(p_init->frameformat)) {
        HAL_DEBUGE("parameter [frameformat] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check mckout value */
    if(!IS_I2S_MCKOUT(p_init->mckout)) {
        HAL_DEBUGE("parameter [mckout] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check audiosample value */
    if(!IS_I2S_AUDIOSAMPLE(p_init->audiosample)) {
        HAL_DEBUGE("parameter [audiosample] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check ckpl value */
    if(!IS_I2S_CKPL(p_init->ckpl)) {
        HAL_DEBUGE("parameter [ckpl] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check user_input value (valid range: 8KHz ~ 192KHz) */
    if((p_init->user_input < (uint32_t)I2S_AUDIOSAMPLE_8K) || \
       (p_init->user_input > (uint32_t)I2S_AUDIOSAMPLE_192K)) {
        HAL_DEBUGE("parameter [user_input] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check word_access vs frameformat */
    if((p_init->frameformat != I2S_FRAMEFORMAT_DT16B_CH16B) && (p_init->frameformat != I2S_FRAMEFORMAT_DT16B_CH32B)) {
        if((I2S_WORD_ACCESS != p_init->word_access)) {
            HAL_DEBUGE("parameter [word_access] value is invalid");
            return HAL_ERR_VAL;
        }
    } else {
        if((I2S_HALF_WORD_ACCESS != p_init->word_access)) {
            HAL_DEBUGE("parameter [word_access] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    i2s_dev->periph = periph;
    i2s_dev->state  = HAL_I2S_STATE_BUSY;

    /* get the I2S clock source */
    switch(periph) {
    case SPI0:
        /* I2S0 clock source selection */
        i2s_clk_sel = RCU_CFG5 & RCU_CFG5_SPI0SEL;
        if(RCU_SPISRC_I2S_CKIN == i2s_clk_sel) {
            /* Select I2S_CKIN clock as the source clock for SPI0 / I2S0 */
            i2sclock = spi0_2_clksel[i2s_clk_sel];
        } else {
            i2sclock = hal_rcu_clock_freq_get((hal_rcu_clock_freq_enum)spi0_2_clksel[i2s_clk_sel]);
        }
        break;
    case SPI1:
        /* I2S1 clock source selection */
        i2s_clk_sel = (RCU_CFG5 & RCU_CFG5_SPI1SEL) >> 4U;
        if(RCU_SPISRC_I2S_CKIN == i2s_clk_sel) {
            /* Select I2S_CKIN clock as the source clock for SPI1 / I2S1 */
            i2sclock = spi0_2_clksel[i2s_clk_sel];
        } else {
            i2sclock = hal_rcu_clock_freq_get((hal_rcu_clock_freq_enum)spi0_2_clksel[i2s_clk_sel]);
        }
        break;
    case SPI2:
        /* I2S2 clock source selection */
        i2s_clk_sel = (RCU_CFG5 & RCU_CFG5_SPI2SEL) >> 8U;
        if(RCU_SPISRC_I2S_CKIN == i2s_clk_sel) {
            /* Select I2S_CKIN clock as the source clock for SPI2 / I2S2 */
            i2sclock = spi0_2_clksel[i2s_clk_sel];
        } else {
            i2sclock = hal_rcu_clock_freq_get((hal_rcu_clock_freq_enum)spi0_2_clksel[i2s_clk_sel]);
        }
        break;
    case SPI5:
        /* I2S5 clock source selection */
        i2s_clk_sel = (RCU_CFG5 & RCU_CFG5_SPI5SEL) >> 20U;
        if(RCU_SPISRC_I2S_CKIN > i2s_clk_sel) {
            /* Select I2S_CKIN clock as the source clock for SPI5 / I2S5 */
            i2sclock = hal_rcu_clock_freq_get((hal_rcu_clock_freq_enum)spi3_5_clksel[i2s_clk_sel]);
        } else {
            i2sclock = spi3_5_clksel[i2s_clk_sel];
        }
        break;
    default :
        HAL_DEBUGE("I2S peripheral is invalid");
        retval = HAL_ERR_VAL;
        break;
    }

    /* config the prescaler depending on the mclk output state, the frame format and audio sample rate */
    if((I2S_AUDIOSAMPLE_USER != p_init->audiosample) && (HAL_ERR_NONE == retval)) {
        if(I2S_MCKOUT_ENABLE == p_init->mckout) {
            clks = (uint32_t)(((i2sclock / I2S_FRAME_MCKOUT_DIV) * 10U) / (uint32_t)p_init->audiosample);
        } else if(I2S_FRAMEFORMAT_DT16B_CH16B == p_init->frameformat) {
            clks = (uint32_t)(((i2sclock / I2S_FRAME_CH16B_DIV) * 10U) / (uint32_t)p_init->audiosample);
        } else {
            clks = (uint32_t)(((i2sclock / I2S_FRAME_CH32B_DIV) * 10U) / (uint32_t)p_init->audiosample);
        }
    } else if(0U != p_init->user_input) {
        if(I2S_MCKOUT_ENABLE == p_init->mckout) {
            clks = (uint32_t)(((i2sclock / I2S_FRAME_MCKOUT_DIV) * 10U) / p_init->user_input);
        } else if(I2S_FRAMEFORMAT_DT16B_CH16B == p_init->frameformat) {
            clks = (uint32_t)(((i2sclock / I2S_FRAME_CH16B_DIV) * 10U) / p_init->user_input);
        } else {
            clks = (uint32_t)(((i2sclock / I2S_FRAME_CH32B_DIV) * 10U) / p_init->user_input);
        }
    } else {
        /* user_input value is 0 */
        HAL_DEBUGE("I2S user input is invalid");
        retval = HAL_ERR_VAL;
    }

    if(HAL_ERR_NONE == retval) {
        /* remove the floating point */
        clks   = (clks + 5U) / 10U;
        i2sof  = (clks & 0x00000001U);
        i2sdiv = ((clks - i2sof) / 2U);

        /* set the default values */
        if((2U > i2sdiv) || (255U < i2sdiv)) {
            i2sdiv = 2U;
            i2sof  = 0U;
        } else {
            /* Do nothing */
        }

        /* clear and set some bits in SPI_I2SCTL register*/
        SPI_I2SCTL(periph) &= (~(SPI_I2SCTL_DIV | SPI_I2SCTL_OF | SPI_I2SCTL_MCKOEN |SPI_I2SCTL_DTLEN | \
                                 SPI_I2SCTL_CHLEN | SPI_I2SCTL_I2SSEL | SPI_I2SCTL_I2SEN | SPI_I2SCTL_I2SSTD | \
                                 SPI_I2SCTL_I2SOPMOD | SPI_I2SCTL_PCMSMOD | SPI_I2SCTL_CKPL));
        SPI_I2SCTL(periph) |= ((i2sdiv << 16U) | (i2sof << 24U) | (p_init->mckout) | (p_init->frameformat) | \
                               (p_init->mode) | (p_init->standard) | (p_init->ckpl));

        /* clear and set SPI_CFG0_WORDEN bits in SPI_CFG0 register,clear and set LSB first mode and AF GPIOs control bits in SPI_CFG1*/
        SPI_CFG0(periph) &= ~(SPI_CFG0_WORDEN);
        SPI_CFG0(periph) |= p_init->word_access;
        SPI_CFG1(periph) &= ~(SPI_CFG1_LF | SPI_CFG1_AFCTL);
        SPI_CFG1(periph) |= (p_init->first_bit | p_init->keep_io_state);

        /* start I2S by placing the SPI_I2SCTL register*/
        SPI_I2SCTL(periph) |= (SPI_I2SCTL_I2SSEL | SPI_I2SCTL_I2SEN);

        /* clear master transfer start bit to indicate idle status */
        SPI_CTL0(periph) &= ~SPI_CTL0_MSTART;
    } else {
        /* Do nothing */
    }

    i2s_dev->state = HAL_I2S_STATE_READY;

    /* unlock I2S */
    HAL_UNLOCK(i2s_dev);

    return retval;
}

/*!
    \brief      transmit amounts of data
    \param[in]  i2s_dev:  I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to transmit data
    \param[in]  length: data size, When a 16-bit data frame or a 16-bit data frame extended is selected during the I2S
                        configuration phase, the Size parameter means the number of 16-bit data length
                        in the transaction and when a 24-bit data frame or a 32-bit data frame is selected
                        the Size parameter means the number of 16-bit data length
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT,
                            HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_transmit_poll(hal_i2s_dev_struct *i2s_dev,uint16_t *p_txbuffer, uint32_t length, uint32_t timeout)
{
    /* Initialize the function return value */
    int32_t retval  = HAL_ERR_NONE;
    uint32_t ctlreg = 0U;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL== i2s_dev)) {
        HAL_DEBUGE("pointer [i2s_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((NULL == p_txbuffer) || (0U == length)) {
        HAL_DEBUGE("parameter [p_txbuffer] or [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    if(HAL_I2S_STATE_READY != i2s_dev->state) {
        HAL_DEBUGE("I2S is busy");
        retval = HAL_ERR_BUSY;
    } else {
        /* set state*/
        i2s_dev->state           = HAL_I2S_STATE_BUSY_TX;
        i2s_dev->txbuffer.buffer = p_txbuffer;
        i2s_dev->txbuffer.length = length;
        i2s_dev->txbuffer.pos    = length;

        /* init field not used to zero */
        i2s_dev->rxbuffer.buffer = NULL;
        i2s_dev->rxbuffer.length = 0U;
        i2s_dev->rxbuffer.pos    = 0U;
        i2s_dev->tx_callback     = NULL;
        i2s_dev->rx_callback     = NULL;

        ctlreg = SPI_I2SCTL(i2s_dev->periph) & ((SPI_I2SCTL_DTLEN | SPI_I2SCTL_CHLEN));

        /* transfer size */
        SPI_CTL1(i2s_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(i2s_dev->periph) |= length;

        /* clear flag */
        if(SET == hals_i2s_flag_get(i2s_dev->periph, I2S_FLAG_TXURERR)) {
            hals_i2s_disable(i2s_dev->periph);
            hals_i2s_flag_clear(i2s_dev->periph, SPI_STATC_TXURERRC);
            SPI_TDATA(i2s_dev->periph) = 0U;
        } else {
            /* Do nothing */
        }

        /* Check if the I2S is already enabled */
        if(RESET == (SPI_I2SCTL(i2s_dev->periph) & SPI_I2SCTL_I2SEN)) {
            hals_i2s_enable(i2s_dev->periph);
        } else {
            /* Do nothing */
        }

        /* start transmit */
        _i2s_start_transfer(i2s_dev);
    }

    /*TP flag is set */
    while((0UL != i2s_dev->txbuffer.pos) && (HAL_ERR_NONE == retval)) {
        if(RESET != ((SPI_STAT(i2s_dev->periph) & SPI_STAT_TP))) {
            if((I2S_FRAMEFORMAT_DT32B_CH32B == ctlreg) || (I2S_FRAMEFORMAT_DT24B_CH32B == ctlreg)) {
                /* transmit data in 32 Bit mode */
                SPI_TDATA(i2s_dev->periph) = REG32(i2s_dev->txbuffer.buffer);
                i2s_dev->txbuffer.buffer += 2UL;
                i2s_dev->txbuffer.pos--;
            } else if((I2S_FRAMEFORMAT_DT16B_CH32B == ctlreg) || (I2S_FRAMEFORMAT_DT16B_CH16B == ctlreg)) {
                /* transmit data in 16 Bit mode */
                SPI_TDATA(i2s_dev->periph) = REG16(i2s_dev->txbuffer.buffer);
                i2s_dev->txbuffer.buffer++;
                i2s_dev->txbuffer.pos--;
            } else {
                /* Do nothing */
            }
        } else {
            if(HAL_ERR_NONE != _i2s_wait_flag_timeout(i2s_dev->periph, I2S_FLAG_TP, SET, timeout)) {
                i2s_dev->state = HAL_I2S_STATE_READY;
                HAL_DEBUGE("I2S transmit timeout");
                retval = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* Do nothing */
            }
        }
    }

    i2s_dev->state = HAL_I2S_STATE_READY;

    /* unlock I2S */
    HAL_UNLOCK(i2s_dev);

    return retval;
}

/*!
    \brief      receive data in a block
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  length: data size, when a 16-bit data frame or a 16-bit data frame extended is selected during the I2S
                        configuration phase, the Size parameter means the number of 16-bit data length
                        in the transaction and when a 24-bit data frame or a 32-bit data frame is selected
                        the Size parameter means the number of 16-bit data length
    \param[in]  timeout: timeout duration
    \param[out] p_rxbuffer: pointer to receive data
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT,
                            HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_receive_poll(hal_i2s_dev_struct *i2s_dev, uint16_t *p_rxbuffer, uint32_t length, uint32_t timeout)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;
    uint32_t ctlreg = 0U;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL== i2s_dev)) {
        HAL_DEBUGE("pointer [i2s_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((NULL == p_rxbuffer) || (0U == length)) {
        HAL_DEBUGE("parameter [p_txbuffer] or [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    if(HAL_I2S_STATE_READY != i2s_dev->state) {
        HAL_DEBUGE("I2S is busy");
        retval = HAL_ERR_BUSY;
    } else {
        /* set state*/
        i2s_dev->state           = HAL_I2S_STATE_BUSY_RX;
        i2s_dev->txbuffer.buffer = NULL;
        i2s_dev->txbuffer.length = 0U;
        i2s_dev->txbuffer.pos    = 0U;

        /* set the receiver information */
        i2s_dev->rxbuffer.buffer = p_rxbuffer;
        i2s_dev->rxbuffer.length = length;
        i2s_dev->rxbuffer.pos    = length;
        i2s_dev->tx_callback     = NULL;
        i2s_dev->rx_callback     = NULL;

        ctlreg = SPI_I2SCTL(i2s_dev->periph) & ((SPI_I2SCTL_DTLEN | SPI_I2SCTL_CHLEN));

        SPI_CTL1(i2s_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(i2s_dev->periph) |= length;

        /* check if the I2S is already enabled */
        if(RESET == (SPI_I2SCTL(i2s_dev->periph) & SPI_I2SCTL_I2SEN)) {
            hals_i2s_enable(i2s_dev->periph);
        } else {
            /* Do nothing */
        }
        /* start transmit */
        _i2s_start_transfer(i2s_dev);
    }

    /* RP flag is set */
    while((i2s_dev->rxbuffer.pos > 0UL) && (HAL_ERR_NONE == retval)) {
        if(RESET != (SPI_STAT(i2s_dev->periph) & SPI_STAT_RP)) {
            if((I2S_FRAMEFORMAT_DT32B_CH32B == ctlreg) || (I2S_FRAMEFORMAT_DT24B_CH32B == ctlreg)) {
                /* transmit data in 32 Bit mode */
                REG32(i2s_dev->rxbuffer.buffer) = SPI_RDATA(i2s_dev->periph);
                i2s_dev->rxbuffer.buffer += 2UL;
                i2s_dev->rxbuffer.pos--;
            } else if((I2S_FRAMEFORMAT_DT16B_CH32B == ctlreg) || (I2S_FRAMEFORMAT_DT16B_CH16B == ctlreg)) {
                /* transmit data in 16 Bit mode */
                REG16(i2s_dev->rxbuffer.buffer) = (uint16_t)SPI_RDATA(i2s_dev->periph);
                i2s_dev->rxbuffer.buffer++;
                i2s_dev->rxbuffer.pos--;
            } else {
                /* Do nothing */
            }
        } else {
            /* Wait for RP flag to be set, return error if timeout occurs */
            if(HAL_ERR_NONE != _i2s_wait_flag_timeout(i2s_dev->periph, I2S_FLAG_RP, SET, timeout)) {
                i2s_dev->state = HAL_I2S_STATE_READY;
                HAL_DEBUGE("I2S receive timeout");
                retval = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* Do nothing */
            }
        }
    }

    /* stop transmit */
    _i2s_stop_transfer(i2s_dev);

    i2s_dev->state = HAL_I2S_STATE_READY;

    /* unlock I2S */
    HAL_UNLOCK(i2s_dev);

    return retval;
}

/*!
    \brief      transmit data in an interrupted manner
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to transmit data
    \param[in]  length: data size, when a 16-bit data frame or a 16-bit data frame extended is selected during the I2S
                        configuration phase, the Size parameter means the number of 16-bit data length
                        in the transaction and when a 24-bit data frame or a 32-bit data frame is selected
                        the Size parameter means the number of 16-bit data length
    \param[in]  p_user_func:pointer to call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_transmit_interrupt(hal_i2s_dev_struct *i2s_dev, uint16_t *p_txbuffer, uint32_t length, \
                                   hal_i2s_user_callback_struct *p_user_func)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2s_dev) || (NULL == p_txbuffer)) {
        HAL_DEBUGE("pointer [i2s_dev] or [p_txbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    if(HAL_I2S_STATE_READY != i2s_dev->state) {
        HAL_DEBUGE("I2S is busy");
        retval = HAL_ERR_BUSY;
    } else {
        /* Set the current data transfer length for I2S transmission */
        SPI_CTL1(i2s_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(i2s_dev->periph) |= length;

        /* set state*/
        i2s_dev->state           = HAL_I2S_STATE_BUSY_TX;
        i2s_dev->txbuffer.buffer = p_txbuffer;
        i2s_dev->txbuffer.length = length;
        i2s_dev->txbuffer.pos    = length;

        /* init field not used to zero */
        i2s_dev->rxbuffer.buffer = NULL;
        i2s_dev->rxbuffer.length = 0U;
        i2s_dev->rxbuffer.pos    = 0U;

        i2s_dev->rx_callback     = NULL;
        i2s_dev->error_callback  = NULL;
    }

    if((NULL != p_user_func) && (HAL_ERR_NONE == retval)) {
        if(NULL != p_user_func->complete_func) {
            i2s_dev->tx_callback    = (void *)p_user_func->complete_func;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->error_func) {
            i2s_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    if(HAL_ERR_NONE == retval) {
        i2s_dev->i2s_irq.transmit_handler = _i2s_transmit_interrupt;
        i2s_dev->i2s_irq.receive_handler  = NULL;
        i2s_dev->i2s_irq.error_handler    = _i2s_error_interrupt;

        /* Check if the I2S is already enabled */
        if(RESET == (SPI_I2SCTL(i2s_dev->periph) & SPI_I2SCTL_I2SEN)) {
            hals_i2s_enable(i2s_dev->periph);
        } else {
            /* Do nothing */
        }
        /* start transmit */
        _i2s_start_transfer(i2s_dev);

        /* enable transmit complete interrupt */
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_TP);
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_TXURE);
    } else {
        /* Do nothing */
    }

    /* unlock I2S */
    HAL_UNLOCK(i2s_dev);

    return retval;
}

/*!
    \brief      receive data in an interrupted manner
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  length: data size, when a 16-bit data frame or a 16-bit data frame extended is selected during the I2S
                        configuration phase, the Size parameter means the number of 16-bit data length
                        in the transaction and when a 24-bit data frame or a 32-bit data frame is selected
                        the Size parameter means the number of 16-bit data length
    \param[in]  p_user_func:pointer to call back function for user
    \param[out] p_rxbuffer: pointer to receive data
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_receive_interrupt(hal_i2s_dev_struct *i2s_dev, uint16_t *p_rxbuffer, uint32_t length, \
                                  hal_i2s_user_callback_struct *p_user_func)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL== i2s_dev) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [i2s_dev] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    if(HAL_I2S_STATE_READY != i2s_dev->state) {
        HAL_DEBUGE("I2S is busy");
        retval = HAL_ERR_BUSY;
    } else {
        /* set state*/
        i2s_dev->state           = HAL_I2S_STATE_BUSY_RX;
        i2s_dev->txbuffer.buffer = NULL;
        i2s_dev->txbuffer.length = 0U;
        i2s_dev->txbuffer.pos    = 0U;

        /* set the receiver information */
        i2s_dev->rxbuffer.buffer = p_rxbuffer;
        i2s_dev->rxbuffer.length = length;
        i2s_dev->rxbuffer.pos    = length;
        i2s_dev->rx_callback     = NULL;
        i2s_dev->error_callback  = NULL;
    }

    /* set the callback function */
    if((NULL != p_user_func) && (HAL_ERR_NONE == retval)) {
        if(NULL != p_user_func->complete_func) {
            i2s_dev->rx_callback    = (void *)p_user_func->complete_func;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->error_func) {
            i2s_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    if(HAL_ERR_NONE == retval) {
        i2s_dev->i2s_irq.receive_handler  = _i2s_receive_interrupt;
        i2s_dev->i2s_irq.transmit_handler = NULL;
        i2s_dev->i2s_irq.error_handler    = _i2s_error_interrupt;

        /* check if the I2S is already enabled */
        if(RESET == (SPI_I2SCTL(i2s_dev->periph) & SPI_I2SCTL_I2SEN)) {
            hals_i2s_enable(i2s_dev->periph);
        } else {
            /* Do nothing */
        }

        /* enable receive complete interrupt */
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_RP);
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_RXORE);

        /* I2S start transmit */
        _i2s_start_transfer(i2s_dev);
    } else {
        /* Do nothing */
    }

    /* unlock I2S */
    HAL_UNLOCK(i2s_dev);

    return retval;
}

/*!
    \brief      transmit a piece of data in DMA mode
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to transmit data
    \param[in]  length: data size, when a 16-bit data frame or a 16-bit data frame extended is selected during the I2S
                        configuration phase, the Size parameter means the number of 16-bit data length
                        in the transaction and when a 24-bit data frame or a 32-bit data frame is selected
                        the Size parameter means the number of 16-bit data length
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_transmit_dma(hal_i2s_dev_struct *i2s_dev, uint16_t *p_txbuffer, uint16_t length, \
                             hal_i2s_user_callback_struct *p_user_func)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq = {0};

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL== i2s_dev) || (NULL == p_txbuffer)) {
        HAL_DEBUGE("pointer [i2s_dev] or [p_txbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK*/

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    if(HAL_I2S_STATE_READY != i2s_dev->state) {
        HAL_DEBUGE("I2S is busy");
        retval = HAL_ERR_BUSY;
    } else {
        /* set state*/
        i2s_dev->state                 = HAL_I2S_STATE_BUSY_TX;
        i2s_dev->txbuffer.buffer       = p_txbuffer;
        i2s_dev->txbuffer.length       = (uint32_t)length;
        i2s_dev->txbuffer.pos          = (uint32_t)length;

        /* init field not used to zero */
        i2s_dev->rxbuffer.buffer       = NULL;
        i2s_dev->rxbuffer.length       = 0U;
        i2s_dev->rxbuffer.pos          = 0U;

        i2s_dev->tx_half_callback      = NULL;
        i2s_dev->tx_callback           = NULL;
        i2s_dev->error_callback        = NULL;
        i2s_dev->i2s_irq.error_handler = _i2s_error_interrupt;
    }

    /* set the callback function */
    if((NULL != p_user_func) && (HAL_ERR_NONE == retval)) {
        if(NULL != p_user_func->half_complete_func) {
            i2s_dev->tx_half_callback = (void *)p_user_func->half_complete_func;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->complete_func) {
            i2s_dev->tx_callback = (void *)p_user_func->complete_func;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->error_func) {
            i2s_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    if(HAL_ERR_NONE == retval) {
        /* transmit size */
        SPI_CTL1(i2s_dev->periph) &= ~SPI_CTL1_TXSIZE;
        SPI_CTL1(i2s_dev->periph) |= (SPI_CTL1_TXSIZE & (uint32_t)length);

        /* clear flag */
        if(SET == hals_i2s_flag_get(i2s_dev->periph, I2S_FLAG_TXURERR)) {
            hals_i2s_disable(i2s_dev->periph);
            hals_i2s_flag_clear(i2s_dev->periph, SPI_STATC_TXURERRC);
            SPI_TDATA(i2s_dev->periph) = 0U;
        } else {
            /* Do nothing */
        }

        i2s_dev->p_dma_tx->dma_irq.half_finish_handle = (hal_irq_handle_cb)(uint32_t)_i2s_half_complete_transmit_dma;
        i2s_dev->p_dma_tx->dma_irq.full_finish_handle = (hal_irq_handle_cb)(uint32_t)_i2s_complete_transmit_dma;
        i2s_dev->p_dma_tx->dma_irq.error_handle       = (hal_irq_handle_cb)(uint32_t)_i2s_error_transmit_receive_dma;

        dma_irq.half_finish_handle                    = (hal_irq_handle_cb)(uint32_t)_i2s_half_complete_transmit_dma;
        dma_irq.full_finish_handle                    = (hal_irq_handle_cb)(uint32_t)_i2s_complete_transmit_dma;
        dma_irq.error_handle                          = (hal_irq_handle_cb)(uint32_t)_i2s_error_transmit_receive_dma;

        hal_dma_start_interrupt(i2s_dev->p_dma_tx, (uint32_t)i2s_dev->txbuffer.buffer, \
                                (uint32_t)&SPI_TDATA(i2s_dev->periph), \
                                (uint16_t)i2s_dev->txbuffer.length, &dma_irq);

        /* check if the I2S is already enabled */
        if(RESET == (SPI_I2SCTL(i2s_dev->periph) & SPI_I2SCTL_I2SEN)) {
            hals_i2s_enable(i2s_dev->periph);
        } else {
            /* Do nothing */
        }

        /* enable underrun error interrupt */
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_TXURE);

        /* start transmit */
        _i2s_start_transfer(i2s_dev);

        /* enable SPI DMA send */
        SPI_CFG0(i2s_dev->periph) |= SPI_CFG0_DMATEN;
    }

    /* unlock I2S */
    HAL_UNLOCK(i2s_dev);

    return retval;
}

/*!
    \brief      receive a piece of data in DMA mode
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  length: data size, when a 16-bit data frame or a 16-bit data frame extended is selected during the I2S
                        configuration phase, the Size parameter means the number of 16-bit data length
                        in the transaction and when a 24-bit data frame or a 32-bit data frame is selected
                        the Size parameter means the number of 16-bit data length
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] p_rxbuffer: pointer to receive data
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_receive_dma(hal_i2s_dev_struct *i2s_dev, uint16_t *p_rxbuffer, uint16_t length, \
                            hal_i2s_user_callback_struct *p_user_func)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq = {0};

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL== i2s_dev) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [i2s_dev] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    if(HAL_I2S_STATE_READY != i2s_dev->state) {
        HAL_DEBUGE("I2S is busy");
        retval = HAL_ERR_BUSY;
    } else {
        /* set state*/
        i2s_dev->state            = HAL_I2S_STATE_BUSY_RX;
        i2s_dev->txbuffer.buffer  = NULL;
        i2s_dev->txbuffer.length  = 0U;
        i2s_dev->txbuffer.pos     = 0U;

        /* set the receiver information */
        i2s_dev->rxbuffer.buffer  = p_rxbuffer;
        i2s_dev->rxbuffer.length  = (uint32_t)length;
        i2s_dev->rxbuffer.pos     = (uint32_t)length;
        i2s_dev->rx_half_callback = NULL;
        i2s_dev->rx_callback      = NULL;
        i2s_dev->error_callback   = NULL;
    }

    /* set the callback function */
    if((NULL != p_user_func) && (HAL_ERR_NONE == retval)) {
        if(NULL != p_user_func->half_complete_func) {
            i2s_dev->rx_half_callback = (void *)p_user_func->half_complete_func;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->complete_func) {
            i2s_dev->rx_callback = (void *)p_user_func->complete_func;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->error_func) {
            i2s_dev->error_callback = (void *)p_user_func->error_func;
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    if(HAL_ERR_NONE == retval) {
        /* clear RX data */
        SPI_RDATA(i2s_dev->periph);

        i2s_dev->p_dma_rx->dma_irq.half_finish_handle = (hal_irq_handle_cb)(uint32_t)_i2s_half_complete_receive_dma;
        i2s_dev->p_dma_rx->dma_irq.full_finish_handle = (hal_irq_handle_cb)(uint32_t)_i2s_complete_receive_dma;
        i2s_dev->p_dma_rx->dma_irq.error_handle       = (hal_irq_handle_cb)(uint32_t)_i2s_error_transmit_receive_dma;

        dma_irq.half_finish_handle                    = (hal_irq_handle_cb)(uint32_t)_i2s_half_complete_receive_dma;
        dma_irq.full_finish_handle                    = (hal_irq_handle_cb)(uint32_t)_i2s_complete_receive_dma;
        dma_irq.error_handle                          = (hal_irq_handle_cb)(uint32_t)_i2s_error_transmit_receive_dma;

        i2s_dev->i2s_irq.error_handler                = _i2s_error_interrupt;

        hal_dma_start_interrupt(i2s_dev->p_dma_rx, (uint32_t)&SPI_RDATA(i2s_dev->periph), \
                                (uint32_t)i2s_dev->rxbuffer.buffer, (uint16_t)i2s_dev->rxbuffer.length, &dma_irq);

        /* check if the I2S is already enabled */
        if(RESET == (SPI_I2SCTL(i2s_dev->periph) & SPI_I2SCTL_I2SEN)) {
            hals_i2s_enable(i2s_dev->periph);
        } else {
            /* Do nothing */
        }

        /* enable overrun error interrupt */
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_RXORE);

        /* enable SPI DMA receive */
        SPI_CFG0(i2s_dev->periph) |= SPI_CFG0_DMAREN;

        /* I2S start transmit */
        _i2s_start_transfer(i2s_dev);
    } else {
        /* Do nothing */
    }

    /* lock I2S */
    HAL_UNLOCK(i2s_dev);

    return retval;
}

/*!
    \brief      get I2S state
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_i2s_run_state_enum: I2S run state
*/
hal_i2s_run_state_enum hal_i2s_state_get(hal_i2s_dev_struct *i2s_dev)
{
    return i2s_dev->state;
}

/*!
    \brief      pause DMA channel data from the media
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY,
                            HAL_ERR_NONE, HAL_ERR_NO_SUPPORT details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_dma_pause(hal_i2s_dev_struct *i2s_dev)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2s_dev) {
        HAL_DEBUGE("pointer [i2s_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    if((I2S_MODE_MASTERTX == (SPI_I2SCTL(i2s_dev->periph) & I2S_MODE_MASTERTX)) || \
       (I2S_MODE_MASTERRX == (SPI_I2SCTL(i2s_dev->periph) & I2S_MODE_MASTERRX))) {
        /* suspend request when the master is in the transmission state */
        if(RESET != (SPI_CTL0(i2s_dev->periph) & SPI_CTL0_MSTART)) {
            /* disable the I2S DMA transmit request */
            SPI_CFG0(i2s_dev->periph) &= ~SPI_CFG0_DMAREN;
            SPI_CFG0(i2s_dev->periph) &= ~SPI_CFG0_DMATEN;
            i2s_dev->state = HAL_I2S_STATE_READY;
        } else {
            HAL_DEBUGE("The host is idle");
            retval = HAL_ERR_BUSY;
            i2s_dev->state = HAL_I2S_STATE_BUSY;
        }
    } else {
        HAL_DEBUGE("not suspend request");
        retval = HAL_ERR_NO_SUPPORT;
        i2s_dev->state = HAL_I2S_STATE_ERROR;
    }

    /* lock I2S */
    HAL_UNLOCK(i2s_dev);

    return retval;
}

/*!
    \brief      resume the DMA channel data from the media
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_dma_resume(hal_i2s_dev_struct *i2s_dev)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2s_dev) {
        HAL_DEBUGE("pointer [i2s_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    /* set state*/
    i2s_dev->state = HAL_I2S_STATE_BUSY;

    /* Determine whether it is receive or transmit mode */
    if(((SPI_I2SCTL(i2s_dev->periph) & SPI_I2SCTL_I2SOPMOD) == I2S_MODE_MASTERTX) || \
       ((SPI_I2SCTL(i2s_dev->periph) & SPI_I2SCTL_I2SOPMOD) == I2S_MODE_SLAVETX)) {
        /* enable I2S transmit dma buffer */
        SPI_CFG0(i2s_dev->periph) |= SPI_CFG0_DMATEN;
        /* disable I2S receive dma buffer*/
        SPI_CFG0(i2s_dev->periph) &= ~SPI_CFG0_DMAREN;
    } else {
        /* enable I2S receive dma buffer*/
        SPI_CFG0(i2s_dev->periph) |= SPI_CFG0_DMAREN;
        /* disable I2S transmit dma buffer*/
        SPI_CFG0(i2s_dev->periph) &= ~SPI_CFG0_DMATEN;
    }

    /* enable I2S peripheral */
    hals_i2s_enable(i2s_dev->periph);

    /* start the transfer */
    SPI_CTL0(i2s_dev->periph) &= ~SPI_CTL0_MSTART;
    SPI_CTL0(i2s_dev->periph) |= SPI_CTL0_MSTART;

    /* lock I2S */
    HAL_UNLOCK(i2s_dev);

    return retval;
}

/*!
    \brief      stop the DMA channel data from the media
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_dma_stop(hal_i2s_dev_struct *i2s_dev)
{
/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2s_dev) {
        HAL_DEBUGE("pointer [i2s_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock I2S */
    HAL_LOCK(i2s_dev);

    /* set state*/
    i2s_dev->state = HAL_I2S_STATE_BUSY;
    /* disenable I2S dma rx and tx stream */
    if(NULL != i2s_dev->p_dma_rx) {
        SPI_CFG0(i2s_dev->periph) &= ~SPI_CFG0_DMAREN;
        hal_dma_stop(i2s_dev->p_dma_rx);
    } else {
        /* Do nothing */
    }

    if(NULL != i2s_dev->p_dma_tx) {
        SPI_CFG0(i2s_dev->periph) &= ~SPI_CFG0_DMATEN;
        hal_dma_stop(i2s_dev->p_dma_tx);
    } else {
        /* Do nothing */
    }

    /* disable I2S peripheral */
    hals_i2s_disable(i2s_dev->periph);
    i2s_dev->state = HAL_I2S_STATE_READY;

    /* lock I2S */
    HAL_UNLOCK(i2s_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  i2s_dev: i2s_dev device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_irq_handle_all_reset(hal_i2s_dev_struct *i2s_dev)
{
/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2s_dev) {
        HAL_DEBUGE("pointer [i2s_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    i2s_dev->i2s_irq.error_handler    = NULL;
    i2s_dev->i2s_irq.transmit_handler = NULL;
    i2s_dev->i2s_irq.receive_handler  = NULL;

    /* disable underrun error interrupt */
    hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_TXURE);
    /* disable overrun error interrupt */
    hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_RXORE);
    /* disable receive complete interrupt */
    hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_RP);
    /* disable transmit complete interrupt */
    hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_TP);
    /* disable TI frame error interrupt */
    hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_FE);

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function, which will be registered and called
                when corresponding interrupt be triggered
    \param[in]  i2s_dev: I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to I2S interrupt callback functions structure
                  The structure member can be assigned as following parameters
     \arg       hal_irq_handle_cb: the function is user-defined,the corresponding callback mechanism is in use,
                  and enable corresponding interrupt
     \arg       NULL:The corresponding callback mechanism is out of use, and disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_irq_handle_set(hal_i2s_dev_struct *i2s_dev, hal_i2s_irq_struct *p_irq)
{
/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2s_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [i2s_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(NULL != p_irq->error_handler) {
        i2s_dev->i2s_irq.error_handler = p_irq->error_handler;
        /* enable underrun error interrupt */
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_TXURE);
        /* enable overrun error interrupt */
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_RXORE);
        /* enable TI frame error interrupt */
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_FE);
    } else {
        i2s_dev->i2s_irq.error_handler = NULL;
        /* disable underrun error interrupt */
        hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_TXURE);
        /* disable overrun error interrupt */
        hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_RXORE);
        /* disable TI frame error interrupt */
        hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_FE);
    }

    if(NULL != p_irq->receive_handler) {
        i2s_dev->i2s_irq.receive_handler = p_irq->receive_handler;
        /* enable receive complete interrupt */
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_RP);
    } else {
        i2s_dev->i2s_irq.receive_handler = NULL;
        /* disable receive complete interrupt */
        hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_RP);
    }

    if(NULL != p_irq->transmit_handler) {
        i2s_dev->i2s_irq.transmit_handler = p_irq->transmit_handler;
        /* enable transmit complete interrupt */
        hals_i2s_interrupt_enable(i2s_dev->periph, SPI_I2S_INT_TP);
    } else {
        i2s_dev->i2s_irq.transmit_handler = NULL;
        /* enable transmit complete interrupt */
        hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_TP);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      handles I2S interrupt request
    \param[in]  i2s_dev:  I2S device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2s_irq(hal_i2s_dev_struct *i2s_dev)
{
/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2s_dev) {
        HAL_DEBUGE("pointer [i2s_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((RESET == (SPI_STAT(i2s_dev->periph) & SPI_STAT_RXORERR)) && \
       (RESET != hals_i2s_interrupt_flag_get(i2s_dev->periph, I2S_INT_FLAG_RP))) {
        if(NULL != i2s_dev->i2s_irq.receive_handler) {
            i2s_dev->i2s_irq.receive_handler(i2s_dev);
        } else {
            /* Do nothing */
        }
    }

    if((RESET == (SPI_STAT(i2s_dev->periph) & SPI_STAT_TXURERR)) && \
       (RESET != hals_i2s_interrupt_flag_get(i2s_dev->periph, I2S_INT_FLAG_TP))) {
        if(NULL != i2s_dev->i2s_irq.transmit_handler) {
            i2s_dev->i2s_irq.transmit_handler(i2s_dev);
        } else {
            /* Do nothing */
        }
    }

    if(RESET != (SPI_STAT(i2s_dev->periph) & (SPI_I2S_INT_CONFE | SPI_I2S_INT_RXORE | \
                                              SPI_I2S_INT_FE | SPI_I2S_INT_CRCER | SPI_I2S_INT_TXURE))) {
        /* I2S configuration error interrupt occurred */
        if(hals_i2s_interrupt_flag_get(i2s_dev->periph, I2S_INT_FLAG_CONFERR) != RESET) {
            hals_i2s_flag_clear(i2s_dev->periph, SPI_STATC_CONFERRC);
        } else {
            /* Do nothing */
        }

        /* I2S receiver overrun error interrupt occurred */
        if(hals_i2s_interrupt_flag_get(i2s_dev->periph, I2S_INT_FLAG_RXORERR) != RESET) {
            hals_i2s_flag_clear(i2s_dev->periph, SPI_STATC_RXORERRC);
        } else {
            /* Do nothing */
        }

        /* I2S frame error interrupt occurred */
        if(hals_i2s_interrupt_flag_get(i2s_dev->periph, I2S_INT_FLAG_FERR) != RESET) {
            hals_i2s_flag_clear(i2s_dev->periph, SPI_STATC_FERRC);
        } else {
            /* Do nothing */
        }

        /* I2S CRC error interrupt occurred */
        if(hals_i2s_interrupt_flag_get(i2s_dev->periph, I2S_INT_FLAG_CRCERR) != RESET) {
            hals_i2s_flag_clear(i2s_dev->periph, SPI_STATC_CRCERRC);
        } else {
            /* Do nothing */
        }

        /* I2S transfer underrun interrupt occurred */
        if(hals_i2s_interrupt_flag_get(i2s_dev->periph, I2S_INT_FLAG_TXURERR) != RESET) {
            hals_i2s_flag_clear(i2s_dev->periph, SPI_STATC_TXURERRC);
            hals_i2s_disable(i2s_dev->periph);
            SPI_TDATA(i2s_dev->periph) = 0U;
        } else {
            /* Do nothing */
        }

        if(NULL != i2s_dev->i2s_irq.error_handler) {
            i2s_dev->i2s_irq.error_handler(i2s_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable I2S interrupt
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[in]  interrupt: i2s_dev interrupt
                only one parameter can be selected which is shown as below:
      \arg        SPI_I2S_INT_RP: RP interrupt
      \arg        SPI_I2S_INT_TP: TP interrupt
      \arg        SPI_I2S_INT_TXURE: underrun error interrupt
      \arg        SPI_I2S_INT_RXORE: overrun error interrupt
      \arg        SPI_I2S_INT_FE: TI frame error interrupt
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_i2s_interrupt_enable(uint32_t periph, uint8_t interrupt)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(!IS_I2S_INT(interrupt)) {
        HAL_DEBUGE("parameter [interrupt] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(interrupt) {
    /* SPI/I2S RP interrupt */
    case SPI_I2S_INT_RP:
        SPI_INT(periph) |= SPI_INT_RPIE;
        break;
    /* SPI/I2S TP interrupt */
    case SPI_I2S_INT_TP:
        SPI_INT(periph) |= SPI_INT_TPIE;
        break;
    /* SPI/I2S underrun error interrupt */
    case SPI_I2S_INT_TXURE:
        SPI_INT(periph) |= SPI_INT_TXUREIE;
        break;
    /* SPI/I2S overrun error interrupt*/
    case SPI_I2S_INT_RXORE:
        SPI_INT(periph) |= SPI_INT_RXOREIE;
        break;
    case SPI_I2S_INT_FE:
    /* SPI TI frame error interrupt */
        SPI_INT(periph) |= SPI_INT_FEIE;
        break;
    default:
        HAL_DEBUGE("parameter [interrupt] value is invalid");
        retval = HAL_ERR_VAL;
        break;
    }

    return retval;
}

/*!
    \brief      disable I2S interrupt
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[in]  interrupt: SPI/I2S interrupt
                only one parameter can be selected which is shown as below:
      \arg        SPI_I2S_INT_RP: RP interrupt
      \arg        SPI_I2S_INT_TP: TP interrupt
      \arg        SPI_I2S_INT_TXURE: underrun error interrupt
      \arg        SPI_I2S_INT_RXORE: overrun error interrupt
      \arg        SPI_I2S_INT_FE: TI frame error interrupt
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_i2s_interrupt_disable(uint32_t periph, uint8_t interrupt)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(!IS_I2S_PERIPH(periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if(!IS_I2S_INT(interrupt)) {
        HAL_DEBUGE("parameter [interrupt] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(interrupt) {
    /* SPI/I2S RP interrupt */
    case SPI_I2S_INT_RP:
        SPI_INT(periph) &= ~SPI_INT_RPIE;
        break;
    /* SPI/I2S TP interrupt */
    case SPI_I2S_INT_TP:
        SPI_INT(periph) &= ~SPI_INT_TPIE;
        break;
    /* SPI/I2S underrun error interrupt */
    case SPI_I2S_INT_TXURE:
        SPI_INT(periph) &= ~SPI_INT_TXUREIE;
        break;
    /* SPI/I2S overrun error interrupt*/
    case SPI_I2S_INT_RXORE:
        SPI_INT(periph) &= ~SPI_INT_RXOREIE;
        break;
    /* SPI TI frame error interrupt */
    case SPI_I2S_INT_FE:
        SPI_INT(periph) &= ~SPI_INT_FEIE;
        break;
    default:
        HAL_DEBUGE("parameter [interrupt] value is invalid");
        retval = HAL_ERR_VAL;
        break;
    }

    return retval;
}

/*!
    \brief      get I2S interrupt flag status
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[in]  interrupt: SPI/I2S interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        I2S_INT_FLAG_RP: RP interrupt flag
      \arg        I2S_INT_FLAG_TP: TP interrupt flag
      \arg        I2S_INT_FLAG_TXURERR: underrun error interrupt flag
      \arg        I2S_INT_FLAG_RXORERR: overrun error interrupt flag
      \arg        I2S_INT_FLAG_FERR: TI frame error interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_i2s_interrupt_flag_get(uint32_t periph, hal_i2s_interrupt_flag_enum interrupt)
{
    /* Initialize the function return value */
    FlagStatus retval = RESET;

    uint32_t reg1 =0U, reg2 = 0U;

    reg1 = SPI_STAT(periph);
    reg2 = SPI_INT(periph);

    switch(interrupt) {
    /* I2S RP interrupt */
    case I2S_INT_FLAG_RP:
        reg1 = reg1 & SPI_STAT_RP;
        reg2 = reg2 & SPI_INT_RPIE;
        break;
    /* I2S TP interrupt */
    case I2S_INT_FLAG_TP:
        reg1 = reg1 & SPI_STAT_TP;
        reg2 = reg2 & SPI_INT_TPIE;
        break;
    /* I2S underrun error interrupt */
    case I2S_INT_FLAG_TXURERR:
        reg1 = reg1 & SPI_STAT_TXURERR;
        reg2 = reg2 & SPI_INT_TXUREIE;
        break;
    /* I2S overrun error interrupt */
    case I2S_INT_FLAG_RXORERR:
        reg1 = reg1 & SPI_STAT_RXORERR;
        reg2 = reg2 & SPI_INT_RXOREIE;
        break;
    /* I2S TI frame error interrupt */
    case I2S_INT_FLAG_FERR:
        reg1 = reg1 & SPI_STAT_FERR;
        reg2 = reg2 & SPI_INT_FEIE;
        break;
    default:
        HAL_DEBUGE("parameter [interrupt] value is invalid");
        break;
    }

    /*get SPI/I2S interrupt flag status */
    if(reg1 && reg2) {
        retval = SET;
    } else {
        retval = RESET;
    }

    return retval;
}

/*!
    \brief      clear I2S interrupt flags
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[in]  interrupt: I2S interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        I2S_INT_FLAG_TXURERR: underrun error interrupt flag
      \arg        I2S_INT_FLAG_RXORERR: overrun error interrupt flag
      \arg        I2S_INT_FLAG_FERR: TI frame error interrupt flag
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_i2s_interrupt_flag_clear(uint32_t periph, hal_i2s_interrupt_flag_enum interrupt)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

    if(interrupt == I2S_INT_FLAG_TXURERR) {
        /* Clear TX underrun error interrupt flag */
        SPI_STATC(periph) |= SPI_STATC_TXURERRC;
    } else if (interrupt == I2S_INT_FLAG_RXORERR) {
        /* Clear RX overrun error interrupt flag */
        SPI_STATC(periph) |= SPI_STATC_RXORERRC;
    } else if (interrupt == I2S_INT_FLAG_FERR) {
        /* Clear TI frame error interrupt flag */
        SPI_STATC(periph) |= SPI_STATC_FERRC;
    } else {
        /* Invalid interrupt flag */
        HAL_DEBUGE("parameter [interrupt] value is invalid");
        retval = HAL_ERR_VAL;
    }

    return retval;
}

/*!
    \brief      get I2S flag status
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[in]  flag: I2S flag status
                only one parameter can be selected which is shown as below:
      \arg        I2S_FLAG_RP: RP flag
      \arg        I2S_FLAG_TP: TP flag
      \arg        I2S_FLAG_DP: DP flag
      \arg        I2S_FLAG_ET: end of transfer or receive flag
      \arg        I2S_FLAG_TXF: transmission filled flag
      \arg        I2S_FLAG_TXURERR: underrun error flag
      \arg        I2S_FLAG_RXORERR: overrun error flag
      \arg        I2S_FLAG_CRCERR: CRC error flag
      \arg        I2S_FLAG_FERR: TI frame error flag
      \arg        I2S_FLAG_CONFERR: mode error flag
      \arg        I2S_FLAG_TXSERF: TXSER reload flag
      \arg        I2S_FLAG_SPD: suspend flag
      \arg        I2S_FLAG_TC: TxFIFO clear flag
      \arg        I2S_FLAG_RWNE: the word of RXFIFO is not empty flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/

FlagStatus hals_i2s_flag_get(uint32_t periph, hal_i2s_flag_enum flag)
{
    /* Initialize the function return value */
    FlagStatus retval = RESET;

    if(SPI_STAT(periph) & (uint32_t)flag) {
        retval = SET;
    } else {
        retval = RESET;
    }

    return retval;
}

/*!
    \brief      clear I2S flag
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[in]  flag: SPI/I2S flag status
                only one parameter can be selected which is shown as below:
      \arg        SPI_STATC_ETC: clear the end of transfer flag
      \arg        SPI_STATC_TXFC: clear the send transmission filled flag
      \arg        SPI_STATC_TXURERRC: clear the transmission underrun error flag
      \arg        SPI_STATC_RXORERRC: clear the reception overrun error flag
      \arg        SPI_STATC_CRCERRC: clear the CRC error flag
      \arg        SPI_STATC_FERRC: clear the SPI TI format error flag
      \arg        SPI_STATC_CONFERRC: clear the configuration error flag
      \arg        SPI_STATC_TXSERFC: clear the TXSERF flag
      \arg        SPI_STATC_SPDC: clear the suspend flag
    \param[out] none
    \retval     none
*/
void hals_i2s_flag_clear(uint32_t periph, uint32_t flag)
{
    SPI_STATC(periph) |= flag;
}

/*!
    \brief      get I2S RXFIFO packing level
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hals_spi_i2s_rxfifo_plevel_get(uint32_t periph)
{
    return ((uint32_t)((SPI_STAT(periph) & SPI_STAT_RPLVL) >> 13U));
}

/*!
    \brief      get I2S remaining data frames number in the TXSIZE session
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hals_spi_i2s_remain_data_num_get(uint32_t periph)
{
    return ((uint32_t)((SPI_STAT(periph) & SPI_STAT_CTXSIZE) >> 16U));
}

/*!
    \brief      enable I2S
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[out] none
    \retval     none
*/
void hals_i2s_enable(uint32_t periph)
{
    SPI_I2SCTL(periph) |= SPI_I2SCTL_I2SEN;
}

/*!
    \brief      disable I2S
    \param[in]  periph: SPIx(x=0,1,2,5)
    \param[out] none
    \retval     none
*/
void hals_i2s_disable(uint32_t periph)
{
    SPI_I2SCTL(periph) &= ~SPI_I2SCTL_I2SEN;
}

/*!
    \brief      wait the flag status until timeout
    \param[in]  periph: I2Sx(x=0,1,2,5)
    \param[in]  flag: I2S flag status
                only one parameter can be selected which is shown as below:
      \arg        I2S_FLAG_RP: RP flag
      \arg        I2S_FLAG_TP: TP flag
      \arg        I2S_FLAG_DP: DP flag
      \arg        I2S_FLAG_ET: end of transfer or receive flag
      \arg        I2S_FLAG_TXF: transmission filled flag
      \arg        I2S_FLAG_TXURERR: underrun error flag
      \arg        I2S_FLAG_RXORERR: overrun error flag
      \arg        I2S_FLAG_CRCERR: CRC error flag
      \arg        I2S_FLAG_FERR: TI frame error flag
      \arg        I2S_FLAG_CONFERR: mode error flag
      \arg        I2S_FLAG_TXSERF: TXSER reload flag
      \arg        I2S_FLAG_SPD: suspend flag
      \arg        I2S_FLAG_TC: TxFIFO clear flag
      \arg        I2S_FLAG_RWNE: the word of RXFIFO is not empty flag
    \param[in]  status: SET or RESET
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _i2s_wait_flag_timeout(uint32_t periph, hal_i2s_flag_enum flag, FlagStatus status, uint32_t timeout)
{
    /* Initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

    __IO uint32_t tick_start = 0U;

    /* set timeout */
    tick_start = hal_sys_basetick_count_get();
    /* wait flag status RESET */
    while(status != hals_i2s_flag_get(periph, flag)) {
        if(HAL_TIMEOUT_FOREVER != timeout) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                HAL_DEBUGE("i2s_dev get flag timeout");
                retval = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* Do nothing */
            }
        } else {
            /* Do nothing */
        }
    }

    return retval;
}

/*!
    \brief      I2S transmit interrupt handler
    \param[in]  i2s: I2S device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2s_transmit_interrupt(void *i2s)
{
    uint32_t ctlreg             = 0U;
    hal_i2s_dev_struct *i2s_dev = (hal_i2s_dev_struct *)i2s;
    hal_i2s_user_cb p_func      = (hal_i2s_user_cb)i2s_dev->tx_callback;

    /* transmit data */
    ctlreg = SPI_I2SCTL(i2s_dev->periph) & ((SPI_I2SCTL_DTLEN | SPI_I2SCTL_CHLEN));
    if((I2S_FRAMEFORMAT_DT32B_CH32B == ctlreg) || (I2S_FRAMEFORMAT_DT24B_CH32B == ctlreg)) {
        /* transmit data in 32 Bit mode */
        SPI_TDATA(i2s_dev->periph) = REG32(i2s_dev->txbuffer.buffer);
        i2s_dev->txbuffer.buffer += 2UL;
        i2s_dev->txbuffer.pos--;
    } else if((I2S_FRAMEFORMAT_DT16B_CH32B == ctlreg) || (I2S_FRAMEFORMAT_DT16B_CH16B == ctlreg)) {
        /* transmit data in 16 Bit mode */
        SPI_TDATA(i2s_dev->periph) = REG16(i2s_dev->txbuffer.buffer);
        i2s_dev->txbuffer.buffer++;
        i2s_dev->txbuffer.pos--;
    } else {
        /* Do nothing */
    }

    if(0U == i2s_dev->txbuffer.pos) {
        /* disable TBE and ERR interrupt */
        hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_TP);

        if(NULL != p_func) {
            p_func(i2s_dev);
        } else {
            /* Do nothing */
        }

        i2s_dev->state = HAL_I2S_STATE_READY;
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      I2S receive interrupt handler
    \param[in]  i2s: I2S device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2s_receive_interrupt(void *i2s)
{
    uint32_t ctlreg             = 0U;
    hal_i2s_dev_struct *i2s_dev = (hal_i2s_dev_struct *)i2s;
    hal_i2s_user_cb p_func      = (hal_i2s_user_cb)i2s_dev->rx_callback;

    /* receive data */
    ctlreg = SPI_I2SCTL(i2s_dev->periph) & ((SPI_I2SCTL_DTLEN | SPI_I2SCTL_CHLEN));
    if((I2S_FRAMEFORMAT_DT32B_CH32B == ctlreg) || (I2S_FRAMEFORMAT_DT24B_CH32B == ctlreg)) {
        /* transmit data in 32 Bit mode */
        REG32(i2s_dev->rxbuffer.buffer) = SPI_RDATA(i2s_dev->periph);
        i2s_dev->rxbuffer.buffer += 2UL;
        i2s_dev->rxbuffer.pos--;
    } else if((I2S_FRAMEFORMAT_DT16B_CH32B == ctlreg) || (I2S_FRAMEFORMAT_DT16B_CH16B == ctlreg)) {
        /* transmit data in 16 Bit mode */
        REG16(i2s_dev->rxbuffer.buffer) = (uint16_t)SPI_RDATA(i2s_dev->periph);
        i2s_dev->rxbuffer.buffer++;
        i2s_dev->rxbuffer.pos--;
    } else {
        /* Do nothing */
    }

    if(0U == i2s_dev->rxbuffer.pos) {
        _i2s_stop_transfer(i2s_dev);

        /* disable RBNE and ERR interrupt */
        hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_RP);

        if(NULL != p_func) {
            p_func(i2s_dev);
        } else {
            /* Do nothing */
        }

        i2s_dev->state = HAL_I2S_STATE_READY;
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      I2S error interrupt handler
    \param[in]  i2s: I2S device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2s_error_interrupt(void *i2s)
{
    hal_i2s_dev_struct *i2s_dev = (hal_i2s_dev_struct *)i2s;
    hal_i2s_user_cb p_func      = (hal_i2s_user_cb)i2s_dev->error_callback;

    _i2s_stop_transfer(i2s_dev);

    /* disable TBE and ERR interrupt */
    hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_TP);
    /* disable RBNE and ERR interrupt */
    hals_i2s_interrupt_disable(i2s_dev->periph, SPI_I2S_INT_RP);

    if(NULL != p_func) {
        p_func(i2s_dev);
    } else {
        /* Do nothing */
    }

    i2s_dev->state = HAL_I2S_STATE_READY;
}

/*!
    \brief      I2S DMA receive handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _i2s_complete_receive_dma(void *dma)
{
    hal_i2s_user_cb p_func     = NULL;
    hal_i2s_user_cb p_func_err = NULL;
    hal_dma_dev_struct *p_dma  = NULL;
    hal_i2s_dev_struct *p_i2s  = NULL;

    p_dma      = (hal_dma_dev_struct *)dma;
    p_i2s      = (hal_i2s_dev_struct *)p_dma->p_periph;
    p_func     = (hal_i2s_user_cb)p_i2s->rx_callback;
    p_func_err = (hal_i2s_user_cb)p_i2s->error_callback;

    /* DMA Normal Mode */
    if(SET == ((DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) >> 8U) {
        /* stop transfer */
        _i2s_stop_transfer(p_i2s);

        /* disable Rx/Tx DMA Request */
        SPI_CFG0(p_i2s->periph) &= ~SPI_DMA_RECEIVE;
        p_i2s->rxbuffer.pos = 0U;

        if(((uint32_t)HAL_ERR_NONE != p_i2s->error_code) && (NULL != p_func_err)) {
            p_func_err(p_i2s);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    if(NULL != p_func) {
        p_func(p_i2s);
    } else {
        /* Do nothing */
    }

    p_i2s->state = HAL_I2S_STATE_READY;
}

/*!
    \brief      I2S DMA receive half complete handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _i2s_half_complete_receive_dma(void *dma)
{
    hal_i2s_user_cb p_func    = NULL;
    hal_dma_dev_struct *p_dma = NULL;
    hal_i2s_dev_struct *p_i2s = NULL;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_i2s  = (hal_i2s_dev_struct *)p_dma->p_periph;
    p_func = (hal_i2s_user_cb)p_i2s->rx_half_callback;

    /* check receive half complete is valid */
    if(NULL != p_func) {
        p_func(p_i2s);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      I2S DMA transmit handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _i2s_complete_transmit_dma(void *dma)
{
    hal_i2s_user_cb p_func      = NULL;
    hal_i2s_user_cb p_func_err  = NULL;
    hal_dma_dev_struct *p_dma   = NULL;
    hal_i2s_dev_struct *p_i2s   = NULL;

    p_dma      = (hal_dma_dev_struct *)dma;
    p_i2s      = (hal_i2s_dev_struct *)p_dma->p_periph;
    p_func     = (hal_i2s_user_cb)p_i2s->tx_callback;
    p_func_err = (hal_i2s_user_cb)p_i2s->error_callback;

    /* DMA normal Mode */
    if(SET != ((DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) >> 8U) {
        /* disable tx DMA Request */
        SPI_CFG0(p_i2s->periph) &= ~SPI_DMA_TRANSMIT;

        p_i2s->txbuffer.pos = 0U;
        if(((uint32_t)HAL_ERR_NONE != p_i2s->error_code) && (NULL != p_func_err)) {
            p_func_err(p_i2s);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    if(NULL != p_func) {
        p_func(p_i2s);
    } else {
        /* Do nothing */
    }

    p_i2s->state = HAL_I2S_STATE_READY;
}

/*!
    \brief      I2S DMA transmit half complete handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _i2s_half_complete_transmit_dma(void *dma)
{
    hal_i2s_user_cb p_func      = NULL;
    hal_dma_dev_struct *p_dma   = NULL;
    hal_i2s_dev_struct *p_i2s   = NULL;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_i2s  = (hal_i2s_dev_struct *)p_dma->p_periph;
    p_func = (hal_i2s_user_cb)p_i2s->tx_half_callback;

    /* check transmit half complete is valid */
    if(NULL != p_func) {
        p_func(p_i2s);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      I2S DMA error handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _i2s_error_transmit_receive_dma(void *dma)
{
    hal_i2s_user_cb p_func_err = NULL;
    hal_dma_dev_struct *p_dma  = NULL;
    hal_i2s_dev_struct *p_i2s  = NULL;

    p_dma      = (hal_dma_dev_struct *)dma;
    p_i2s      = (hal_i2s_dev_struct *)p_dma->p_periph;
    p_func_err = (hal_i2s_user_cb)p_i2s->error_callback;

    SPI_CFG0(p_i2s->periph) &= ~SPI_DMA_RECEIVE;
    SPI_CFG0(p_i2s->periph) &= ~SPI_DMA_TRANSMIT;

    p_i2s->error_code = HAL_I2S_ERROR_DMA;
    p_i2s->state      = HAL_I2S_STATE_READY;

    if(NULL != p_func_err) {
        p_func_err(p_i2s);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      I2S stop transfer
    \param[in]  i2s_dev: I2S device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2s_stop_transfer(hal_i2s_dev_struct *i2s_dev)
{
    __IO uint32_t tick_start = 0U;
    /* set timeout */
    tick_start = hal_sys_basetick_count_get();

    if((I2S_MODE_MASTERRX == (SPI_I2SCTL(i2s_dev->periph) & I2S_MODE_MASTERRX))) {
        /* suspend request */
        SPI_CTL0(i2s_dev->periph) |= SPI_CTL0_MSPDR;
        /* wait flag status RESET */
        while(RESET != (SPI_CTL0(i2s_dev->periph) & SPI_CTL0_MSTART)) {
            /* Wait timeout*/
            if(SET == hal_sys_basetick_timeout_check(tick_start, I2S_STOP_TRANS_TIMEOUT_VALUE)) {
                HAL_DEBUGE("i2s_dev stop transfer timeout");
                break;
            } else {
                /* Do nothing */
            }
        }
        /* disable I2S*/
        SPI_I2SCTL(i2s_dev->periph) &= ~SPI_I2SCTL_I2SEN;
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      I2S start transfer
    \param[in]  i2s_dev: I2S device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2s_start_transfer(hal_i2s_dev_struct *i2s_dev)
{
    SPI_CTL0(i2s_dev->periph) |= SPI_CTL0_MSTART;
}
