/*!
    \file    gd32h7xx_hal_i2c.c
    \brief   I2C driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define I2C_MAX_BYTE_SIZE               ((uint32_t)255U)
#define I2C_DATA_SEND_FINISH_LENGTH     (0U)
#define I2C_DATA_RECEIVED_FINISH_LENGTH (0U)

/* I2C register bit mask */
#define I2C_ALL_INT                     ((uint32_t)0x000000FEU)
#define SLAVE10_FIRST_BYTE(address)     ((uint32_t)((0xF0) | (uint8_t)((address & 0x0300U) >> 7U)))
#define SLAVE10_SECOND_BYTE(address)    ((uint8_t)(address & 0x00FFU))
#define I2C_MEMORY_ADDRESS_MSB(address) ((uint8_t)((address & 0xFF00U) >> 8U))
#define I2C_MEMORY_ADDRESS_LSB(address) ((uint8_t)(address & 0x00FFU))
#define I2C_ADDRESS_MASK                ((uint32_t)0x000003FFU)
#define I2C_ADDRESS2_MASK               ((uint32_t)0x000000FEU)

/* I2C Tx data register flush process */
static void _i2c_flush_tdata_register(uint32_t i2c_periph);
/* event handler in I2C master mode transmit and receive */
static void _i2c_master_interrupt(void *i2c_dev);
/* event handler in I2C slave transmit mode and receive */
static void _i2c_slave_interrupt(void *i2c_dev);
/* interrupt Sub-Routine which handle the interrupt Flags Memory Mode with interrupt */
static void _i2c_memory_interrupt(void *i2c_dev);
/* DMA transmit and receive complete in I2C master mode */
static void _i2c_dma_complete(void *dma);
/* handle the I2C DMA error process */
static void _i2c_dma_error(void *dma);
/* handle the I2C DMA abort process */
static void _i2c_dma_abort(void *dma);
/* master sends device address for memory read request */
static int32_t _i2c_memory_read(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                uint32_t timeout_ms);
/* master sends device address for memory write request */
static int32_t _i2c_memory_write(hal_i2c_dev_struct *i2c_dev, uint16_t mem_address, uint32_t timeout_ms);
/* I2C master transmit configuration */
static void _i2c_master_transmit_config(uint32_t i2c_periph, uint8_t length, uint32_t mode, uint32_t request);
/* wait the flag status until timeout */
static int32_t _i2c_wait_flag_timeout(uint32_t i2c_periph, hal_i2c_flag_enum flag, FlagStatus status, \
                                      uint32_t timeout_ms);
/* event handler for address listen in slave mode */
static void _i2c_address_listen_interrupt(void *i2c_dev);

/*!
    \brief      initialize the I2C structure with the default values
    \param[in]  struct_type: refer to <hal_i2c_struct_type_enum>
                only one parameter can be selected which is shown as below:
      \arg        HAL_I2C_INIT_STRUCT: I2C initialization structure
      \arg        HAL_I2C_DEV_STRUCT: I2C device information structure
      \arg        HAL_I2C_IRQ_STRUCT: I2C interrupt callback function structure
      \arg        HAL_I2C_BUFFER_STRUCT: I2C buffer structure
      \arg        HAL_I2C_IRQ_USER_CALLBACK_STRUCT: I2C interrupt user callback function structure
      \arg        HAL_I2C_SLAVE_ADDRESS_STRUCT: I2C slave address structure
    \param[out] p_struct: pointer to I2C structure that contains the configuration information
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_struct_init(hal_i2c_struct_type_enum struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(struct_type) {
        /* initialize I2C initialization structure with the default values */
    case HAL_I2C_INIT_STRUCT:
        ((hal_i2c_init_struct *)p_struct)->scl_stretch    = I2C_SCLSTRETCH_DISABLE;
        ((hal_i2c_init_struct *)p_struct)->address_format = I2C_ADDFORMAT_7BITS;
        ((hal_i2c_init_struct *)p_struct)->own_address1   = 0U;
        ((hal_i2c_init_struct *)p_struct)->dual_address   = I2C_DUADEN_DISABLE;
        ((hal_i2c_init_struct *)p_struct)->address2_mask  = 0U;
        ((hal_i2c_init_struct *)p_struct)->own_address2   = 0U;
        ((hal_i2c_init_struct *)p_struct)->time           = 0U;
        ((hal_i2c_init_struct *)p_struct)->general_call   = I2C_GCEN_DISABLE;
        ((hal_i2c_init_struct *)p_struct)->digital_filter = DIGITAL_FILTER_DISABLE;
        ((hal_i2c_init_struct *)p_struct)->analog_filter  = ANALOG_FILTER_DISABLE;
        break;
        /* initialize I2C device information structure with the default values */
    case HAL_I2C_DEV_STRUCT:
        ((hal_i2c_dev_struct *)p_struct)->periph                          = 0U;
        ((hal_i2c_dev_struct *)p_struct)->i2c_irq.event_handle            = NULL;
        ((hal_i2c_dev_struct *)p_struct)->i2c_irq.error_handle            = NULL;
        ((hal_i2c_dev_struct *)p_struct)->p_dma_rx                        = NULL;
        ((hal_i2c_dev_struct *)p_struct)->p_dma_tx                        = NULL;
        ((hal_i2c_dev_struct *)p_struct)->txbuffer.buffer                 = NULL;
        ((hal_i2c_dev_struct *)p_struct)->txbuffer.length                 = 0U;
        ((hal_i2c_dev_struct *)p_struct)->txbuffer.pos                    = 0U;
        ((hal_i2c_dev_struct *)p_struct)->rxbuffer.buffer                 = NULL;
        ((hal_i2c_dev_struct *)p_struct)->rxbuffer.length                 = 0U;
        ((hal_i2c_dev_struct *)p_struct)->rxbuffer.pos                    = 0U;
        ((hal_i2c_dev_struct *)p_struct)->slave_address.device_address    = 0U;
        ((hal_i2c_dev_struct *)p_struct)->slave_address.memory_address    = 0U;
        ((hal_i2c_dev_struct *)p_struct)->slave_address.address_size      = (uint16_t)I2C_ADDFORMAT_7BITS;
        ((hal_i2c_dev_struct *)p_struct)->slave_address.address_complete  = RESET;
        ((hal_i2c_dev_struct *)p_struct)->slave_address.address_count     = 0U;
        ((hal_i2c_dev_struct *)p_struct)->slave_address.second_addressing = RESET;
        ((hal_i2c_dev_struct *)p_struct)->transfer_option                 = I2C_NO_OPTION_TRANSFER;
        ((hal_i2c_dev_struct *)p_struct)->error_state                     = (uint16_t)HAL_I2C_ERROR_NONE;
        ((hal_i2c_dev_struct *)p_struct)->tx_state                        = HAL_I2C_STATE_READY;
        ((hal_i2c_dev_struct *)p_struct)->rx_state                        = HAL_I2C_STATE_READY;
        ((hal_i2c_dev_struct *)p_struct)->last_error                      = 0U;
        ((hal_i2c_dev_struct *)p_struct)->mode                            = HAL_I2C_MODE_NONE;
        ((hal_i2c_dev_struct *)p_struct)->previous_state                  = (uint32_t)HAL_I2C_PREVIOUS_STATE_NONE;
        ((hal_i2c_dev_struct *)p_struct)->rx_callback                     = NULL;
        ((hal_i2c_dev_struct *)p_struct)->tx_callback                     = NULL;
        ((hal_i2c_dev_struct *)p_struct)->abort_callback                  = NULL;
        ((hal_i2c_dev_struct *)p_struct)->addr_callback                   = NULL;
        ((hal_i2c_dev_struct *)p_struct)->mutex                           = HAL_MUTEX_UNLOCKED;
        break;
        /* initialize I2C irq structure with the default values */
    case HAL_I2C_IRQ_STRUCT:
        ((hal_i2c_irq_struct *)p_struct)->event_handle  = NULL;
        ((hal_i2c_irq_struct *)p_struct)->error_handle  = NULL;
        ((hal_i2c_irq_struct *)p_struct)->wakeup_handle = NULL;
        break;
    case HAL_I2C_BUFFER_STRUCT:
        ((hal_i2c_buffer_struct *)p_struct)->buffer = NULL;
        ((hal_i2c_buffer_struct *)p_struct)->length = 0U;
        ((hal_i2c_buffer_struct *)p_struct)->pos    = 0U;
        ((hal_i2c_buffer_struct *)p_struct)->size   = 0U;
        break;
    case HAL_I2C_IRQ_USER_CALLBACK_STRUCT:
        ((hal_i2c_irq_user_callback_struct *)p_struct)->rx_callback    = NULL;
        ((hal_i2c_irq_user_callback_struct *)p_struct)->tx_callback    = NULL;
        ((hal_i2c_irq_user_callback_struct *)p_struct)->addr_callback  = NULL;
        break;
    case HAL_I2C_SLAVE_ADDRESS_STRUCT:
        ((hal_i2c_slave_address_struct *)p_struct)->device_address    = 0U;
        ((hal_i2c_slave_address_struct *)p_struct)->memory_address    = 0U;
        ((hal_i2c_slave_address_struct *)p_struct)->address_size      = 0U;
        ((hal_i2c_slave_address_struct *)p_struct)->address_complete  = RESET;
        ((hal_i2c_slave_address_struct *)p_struct)->address_count     = 0U;
        ((hal_i2c_slave_address_struct *)p_struct)->second_addressing = RESET;
        ((hal_i2c_slave_address_struct *)p_struct)->address_count     = 0U;
    break;
    default:
        HAL_DEBUGE("parameter [struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      initialize I2C
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which I2C is initialized
    \param[in]  i2c_init:pointer of  the initialization data needed to initialize I2C
                  scl_stretch:
                  only one parameter can be selected which is shown as below:
        \arg      I2C_SCLSTRETCH_ENABLE: I2C SCL stretch is enabled
        \arg      I2C_SCLSTRETCH_DISABLE: I2C SCL stretch is disabled
                  address_format:
                  only one parameter can be selected which is shown as below:
       \arg       I2C_ADDFORMAT_7BITS: I2C address format is 7-bit
       \arg       I2C_ADDFORMAT_10BITS: I2C address format is 10-bit
                  dual_address:
                  only one parameter can be selected which is shown as below:
       \arg       I2C_DUADEN_ENABLE: I2C dual-address mode is enabled
       \arg       I2C_DUADEN_DISABLE: I2C dual-address mode is disabled
                  address2_mask:
                  only one parameter can be selected which is shown as below:
       \arg       ADDRESS2_NO_MASK: no mask for the second address
       \arg       ADDRESS2_MASK_BIT1: mask the second address bit 1
       \arg       ADDRESS2_MASK_BIT1_2: mask the second address bit 1 and 2
       \arg       ADDRESS2_MASK_BIT1_3: mask the second address bit 1, 2, and 3
       \arg       ADDRESS2_MASK_BIT1_4: mask the second address bit 1, 2, 3, and 4
       \arg       ADDRESS2_MASK_BIT1_5: mask the second address bit 1, 2, 3, 4, and 5
       \arg       ADDRESS2_MASK_BIT1_6: mask the second address bit 1, 2, 3, 4, 5, and 6
       \arg       ADDRESS2_MASK_ALL: mask all the second address bits
                  general_call:
                  only one parameter can be selected which is shown as below:
       \arg       I2C_GCEN_ENABLE: enable general call response
       \arg       I2C_GCEN_DISABLE: disable general call response
                  analog_filter:
                  only one parameter can be selected which is shown as below:
       \arg       ANALOG_FILTER_ENABLE: enable analog filter
       \arg       ANALOG_FILTER_DISABLE: disable analog filter
                  time: the I2C clock timing register value
                  own_address1: the I2C own address
                  own_address2: the second address in dual-address mode
                  digital_filter:
                  only one parameter can be selected which is shown as below:
       \arg       DIGITAL_FILTER_DISABLE: disable digital filter
       \arg       DIGITAL_FILTER_LESSTHAN_1CLK: filter out spikes with pulse width less than 1 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_2CLK: filter out spikes with pulse width less than 2 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_3CLK: filter out spikes with pulse width less than 3 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_4CLK: filter out spikes with pulse width less than 4 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_5CLK: filter out spikes with pulse width less than 5 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_6CLK: filter out spikes with pulse width less than 6 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_7CLK: filter out spikes with pulse width less than 7 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_8CLK: filter out spikes with pulse width less than 8 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_9CLK: filter out spikes with pulse width less than 9 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_10CLK: filter out spikes with pulse width less than 10 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_11CLK: filter out spikes with pulse width less than 11 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_12CLK: filter out spikes with pulse width less than 12 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_13CLK: filter out spikes with pulse width less than 13 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_14CLK: filter out spikes with pulse width less than 14 tI2CCLK
       \arg       DIGITAL_FILTER_LESSTHAN_15CLK: filter out spikes with pulse width less than 15 tI2CCLK
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_init(hal_i2c_dev_struct *i2c_dev, uint32_t periph, hal_i2c_init_struct *i2c_init)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == i2c_init)) {
        HAL_DEBUGE("pointer [i2c_dev] or [i2c_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    i2c_dev->periph = periph;

    HAL_I2C_DISABLE(i2c_dev->periph);

    /* configure I2C clock */
    I2C_TIMING(periph) = i2c_init->time;

    /* configure analog noise filter */
    if(ANALOG_FILTER_ENABLE == i2c_init->analog_filter) {
        I2C_CTL0(periph) &= ~I2C_CTL0_ANOFF;
    } else {
        I2C_CTL0(periph) |= I2C_CTL0_ANOFF;
    }

    /* configure digital noise filter */
    if(DIGITAL_FILTER_DISABLE != i2c_init->digital_filter) {
        I2C_CTL0(periph) |= (uint32_t)((uint32_t)i2c_init->digital_filter << CTL0_DNF_OFFSET);
    } else {
        I2C_CTL0(periph) |= (uint32_t)((uint32_t)DIGITAL_FILTER_DISABLE << CTL0_DNF_OFFSET);
    }

    if(I2C_ADDFORMAT_10BITS == i2c_init->address_format) {
        hals_i2c_address10_enable(periph);
    } else {
        /* nothing to do */
    }

    /* configure I2C address */
    if(HAL_ERR_NONE != hals_i2c_device_address_config(periph, (uint32_t)i2c_init->own_address1, i2c_init->address_format)) {
        HAL_DEBUGE("parameter [i2c_init->own_address1] value is invalid");
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* configure dual-address mode */
    if(I2C_DUADEN_ENABLE == i2c_init->dual_address) {
        if(HAL_ERR_NONE != hals_i2c_device_second_address_enable(periph, (uint32_t)i2c_init->own_address2, i2c_init->address2_mask)) {
            HAL_DEBUGE("parameter [i2c_init->own_address2] value is invalid");
            ret = HAL_ERR_VAL;
        } else {
            /* nothing to do */
        }
    } else {
        hals_i2c_device_second_address_disable(periph);
    }

    /* configure whether to stretch SCL low when data is not ready in slave mode */
    hals_i2c_stretch_scl_low_config(periph, i2c_init->scl_stretch);
    /* whether or not to response to a general call */
    hals_i2c_slave_response_to_gcall_config(periph, i2c_init->general_call);

    /* enable I2C */
    HAL_I2C_ENABLE(periph);

    /* enable acknowledge */
    if(RESET == ((I2C_SADDR0(periph) & I2C_SADDR0_ADDFORMAT) >> 10)) {
        hals_i2c_ack_config(periph, I2C_ACK_ENABLE);
    } else {
        /* nothing to do */
    }

    i2c_dev->tx_state = HAL_I2C_STATE_READY;
    i2c_dev->rx_state = HAL_I2C_STATE_READY;

    return ret;
}

/*!
    \brief      deinitialize I2C
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_deinit(hal_i2c_dev_struct *i2c_dev)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2c_dev) {
        HAL_DEBUGE("pointer [i2c_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(i2c_dev->periph) {
    case I2C0:
        hal_rcu_periph_reset_enable(RCU_I2C0RST);
        hal_rcu_periph_reset_disable(RCU_I2C0RST);
        i2c_dev->tx_state       = HAL_I2C_STATE_READY;
        i2c_dev->rx_state       = HAL_I2C_STATE_READY;
        i2c_dev->error_state    = (uint32_t)HAL_I2C_ERROR_NONE;
        i2c_dev->previous_state = HAL_I2C_PREVIOUS_STATE_NONE;
        break;
    case I2C1:
        hal_rcu_periph_reset_enable(RCU_I2C1RST);
        hal_rcu_periph_reset_disable(RCU_I2C1RST);
        i2c_dev->tx_state       = HAL_I2C_STATE_READY;
        i2c_dev->rx_state       = HAL_I2C_STATE_READY;
        i2c_dev->error_state    = (uint32_t)HAL_I2C_ERROR_NONE;
        i2c_dev->previous_state = HAL_I2C_PREVIOUS_STATE_NONE;
        break;
    case I2C2:
        hal_rcu_periph_reset_enable(RCU_I2C2RST);
        hal_rcu_periph_reset_disable(RCU_I2C2RST);
        i2c_dev->tx_state       = HAL_I2C_STATE_READY;
        i2c_dev->rx_state       = HAL_I2C_STATE_READY;
        i2c_dev->error_state    = (uint32_t)HAL_I2C_ERROR_NONE;
        i2c_dev->previous_state = HAL_I2C_PREVIOUS_STATE_NONE;
        break;
    case I2C3:
        hal_rcu_periph_reset_enable(RCU_I2C3RST);
        hal_rcu_periph_reset_disable(RCU_I2C3RST);
        i2c_dev->tx_state       = HAL_I2C_STATE_READY;
        i2c_dev->rx_state       = HAL_I2C_STATE_READY;
        i2c_dev->error_state    = (uint32_t)HAL_I2C_ERROR_NONE;
        i2c_dev->previous_state = HAL_I2C_PREVIOUS_STATE_NONE;
        break;
    default:
        HAL_DEBUGE("parameter [i2c_dev->periph] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      generate a START condition on I2C bus
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hal_i2c_start_on_bus(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) |= I2C_CTL1_START;
}

/*!
    \brief      generate a STOP condition on I2C bus
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hal_i2c_stop_on_bus(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) |= I2C_CTL1_STOP;
}

/*!
    \brief      transmit amounts of data in master mode, poll transmit process and completed status,
                the function is blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  timeout_ms: time out configuration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL
                details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_master_transmit_poll(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                     uint32_t length, uint32_t timeout_ms)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout_ms)) {
        HAL_DEBUGE("parameter [length] or [timeout_ms] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the tx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, timeout_ms)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;

    i2c_dev->tx_state = HAL_I2C_STATE_BUSY;
    i2c_dev->error_state |= (uint32_t)HAL_I2C_ERROR_NONE;

    /* configuration slave address */
    if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
        HAL_DEBUGE("I2C slave address is invalid value");
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* set BYTENUM to write and reload if i2c_dev->txbuffer.length > I2C_MAX_BYTE_SIZE and generate RESTART */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
        i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_WRITE);
    } else {
        i2c_dev->txbuffer.pos = length;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_AUTOEND_MODE, \
                                    I2C_GENERATE_START_WRITE);
    }

    /* send data */
    while(I2C_DATA_SEND_FINISH_LENGTH < i2c_dev->txbuffer.pos) {
        /* wait until TI flag is set */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TI, SET, timeout_ms)) {
            HAL_DEBUGE("I2C get TI flag timeout");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* nothing to do */
        }

        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)*i2c_dev->txbuffer.buffer);
        i2c_dev->txbuffer.buffer++;
        i2c_dev->txbuffer.length--;
        i2c_dev->txbuffer.pos--;

        /* send data not empty */
        if((I2C_DATA_SEND_FINISH_LENGTH != i2c_dev->txbuffer.length) && (0U == i2c_dev->txbuffer.pos)) {
            /* wait until TCR flag is set */
            if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TCR, SET, timeout_ms)) {
                HAL_DEBUGE("I2C get TCR flag timeout");
                HAL_UNLOCK(i2c_dev);
                ret = HAL_ERR_TIMEOUT;
            } else {
                /* nothing to do */
            }

            if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
                i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_RELOAD_MODE, \
                                            I2C_NO_STARTSTOP);
            } else {
                i2c_dev->txbuffer.pos = i2c_dev->txbuffer.length;
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_AUTOEND_MODE, \
                                            I2C_NO_STARTSTOP);
            }
        }
    }

    /* wait until stop flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_STPDET, SET, timeout_ms)) {
        HAL_DEBUGE("I2C get stop signal timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* clear stop flag */
        hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_STPDET);
    }

    HAL_UNLOCK(i2c_dev);
    i2c_dev->tx_state = HAL_I2C_STATE_READY;

    return ret;
}

/*!
    \brief      receive amounts of data in master mode, poll receive process and completed
                status, the function is blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  length: length of data to be read
    \param[in]  timeout_ms: time out configuration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY,HAL_ERR_VAL
                details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_master_receive_poll(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                    uint32_t length, uint32_t timeout_ms)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout_ms)) {
        HAL_DEBUGE("parameter [length] or [timeout_ms] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the tx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, timeout_ms)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize received parameters */
    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;

    i2c_dev->rx_state = HAL_I2C_STATE_BUSY;
    i2c_dev->error_state |= (uint32_t)HAL_I2C_ERROR_NONE;

    /* configuration slave address */
    if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
        HAL_DEBUGE("I2C slave address is invalid value");
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* set BYTENUM to write and reload if i2c_dev->txbuffer.length > I2C_MAX_BYTE_SIZE and generate RESTART */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->rxbuffer.length) {
        i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_READ);
    } else {
        i2c_dev->rxbuffer.pos = length;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_AUTOEND_MODE, \
                                    I2C_GENERATE_START_READ);
    }

    while(I2C_DATA_RECEIVED_FINISH_LENGTH < i2c_dev->rxbuffer.pos) {
        /* wait until RBNE flag is set */
        if(HAL_ERR_NONE == _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_RBNE, SET, timeout_ms)) {
            *i2c_dev->rxbuffer.buffer = hals_i2c_data_receive(i2c_dev->periph);
            i2c_dev->rxbuffer.buffer++;
            i2c_dev->rxbuffer.length--;
            i2c_dev->rxbuffer.pos--;
        } else {
            /* nothing to do */
        }

        /* received data not empty */
        if((0U == i2c_dev->rxbuffer.pos) && (0U != i2c_dev->rxbuffer.length)) {
            /* wait until TCR flag is set */
            if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TCR, SET, timeout_ms)) {
                HAL_UNLOCK(i2c_dev);
                HAL_DEBUGE("I2C get TCR flag timeout");
                ret = HAL_ERR_TIMEOUT;
            } else {
                /* nothing to do */
            }

            if(I2C_MAX_BYTE_SIZE <= i2c_dev->rxbuffer.length) {
                i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_RELOAD_MODE, \
                                            I2C_NO_STARTSTOP);
            } else {
                i2c_dev->rxbuffer.pos = i2c_dev->rxbuffer.length;
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_AUTOEND_MODE, \
                                            I2C_NO_STARTSTOP);
            }
        }
    }

    /* no need to check TC flag, with AUTOEND mode the stop is automatically generated */
    /* wait until stop flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_STPDET, SET, timeout_ms)) {
        HAL_UNLOCK(i2c_dev);
        HAL_DEBUGE("I2C get stop signal timeout");
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* clear stop flag */
        hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_STPDET);
    }

    HAL_UNLOCK(i2c_dev);
    i2c_dev->rx_state = HAL_I2C_STATE_READY;

    return ret;
}

/*!
    \brief      transmit amounts of data in slave mode, poll transmit process and completed status, the function is
   blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  timeout_ms: time out configuration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL
                details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_slave_transmit_poll(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                    uint32_t timeout_ms)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout_ms)) {
        HAL_DEBUGE("parameter [length] or [timeout_ms] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the tx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = length;

    i2c_dev->tx_state = HAL_I2C_STATE_BUSY;
    i2c_dev->error_state |= (uint32_t)HAL_I2C_ERROR_NONE;

    /* enable acknowledge */
    if(RESET == ((I2C_SADDR0(i2c_dev->periph) & I2C_SADDR0_ADDFORMAT) >> 10)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_ENABLE);
    } else {
        /* nothing to do */
    }

    if(I2C_CTL0(i2c_dev->periph) & I2C_CTL0_SS) {
        /* transmit data form TDATA register */
        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)*i2c_dev->txbuffer.buffer);
        i2c_dev->txbuffer.buffer++;
        i2c_dev->txbuffer.length--;
        i2c_dev->txbuffer.pos--;
    } else {
        /* nothing to do */
    }

    if(RESET != (I2C_SADDR0(i2c_dev->periph) & I2C_SADDR0_ADDFORMAT)) {
        /* wait until ADDSEND flag is set */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_ADDSEND, SET, timeout_ms)) {
            HAL_DEBUGE("I2C addsend timeout");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* nothing to do */
        }

        /* clear ADDSEND flag */
        hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_ADDSEND);
    } else {
        /* nothing to do */
    }

    /* wait until ADDSEND flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_ADDSEND, SET, timeout_ms)) {
        HAL_DEBUGE("I2C addsend timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* wait until TR flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TR, SET, timeout_ms)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);
        HAL_DEBUGE("I2C get TR timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* clear ADDSEND flag */
    hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_ADDSEND);

    _i2c_flush_tdata_register(i2c_dev->periph);
    while(I2C_DATA_SEND_FINISH_LENGTH < i2c_dev->txbuffer.pos) {
        /* wait until TI flag is set */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TI, SET, timeout_ms)) {
            hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);
            HAL_DEBUGE("I2C get TI flag timeout");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* nothing to do */
        }

        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)*i2c_dev->txbuffer.buffer);
        i2c_dev->txbuffer.buffer++;
        i2c_dev->txbuffer.length--;
        i2c_dev->txbuffer.pos--;
    }

    /* wait until nack flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_NACK, SET, timeout_ms)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);
        HAL_DEBUGE("I2C get nack timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* clear the bit of nack */
    hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_NACK);

    /* wait until stop flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_STPDET, SET, timeout_ms)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);
        HAL_DEBUGE("I2C get stop signal timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* clear stop flag */
        hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_STPDET);
    }

    /* disable acknowledge */
    hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);

    HAL_UNLOCK(i2c_dev);
    i2c_dev->tx_state = HAL_I2C_STATE_READY;

    return ret;
}

/*!
    \brief      receive amounts of data in slave mode, poll receive process and completed
                status, the function is blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be read
    \param[in]  timeout_ms: configuration time out
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL,
                            HAL_ERR_TIMEOUT, HAL_ERR_BUSY, details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_slave_receive_poll(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout_ms)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout_ms)) {
        HAL_DEBUGE("parameter [length] or [timeout_ms] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* initialize received parameters */
    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = length;

    i2c_dev->rx_state = HAL_I2C_STATE_BUSY;
    i2c_dev->error_state |= (uint32_t)HAL_I2C_ERROR_NONE;

    /* enable acknowledge */
    if(RESET == ((I2C_SADDR0(i2c_dev->periph) & I2C_SADDR0_ADDFORMAT) >> 10)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_ENABLE);
    } else {
        /* nothing to do */
    }

    /* wait until ADDSEND bit is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_ADDSEND, SET, timeout_ms)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);
        HAL_DEBUGE("I2C get ADDSEND timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* clear ADDSEND bit */
    hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_ADDSEND);

    /* wait until TR bit is reset */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TR, RESET, timeout_ms)) {
        /* disable address acknowledge */
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);
        HAL_DEBUGE("I2C get TR timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    while(I2C_DATA_SEND_FINISH_LENGTH < ((uint32_t)i2c_dev->rxbuffer.pos)) {
        /* wait until RBNE flag is set */
        if(HAL_ERR_NONE == _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_RBNE, SET, timeout_ms)) {
            *i2c_dev->rxbuffer.buffer = hals_i2c_data_receive(i2c_dev->periph);
            i2c_dev->rxbuffer.buffer++;
            i2c_dev->rxbuffer.length--;
            i2c_dev->rxbuffer.pos--;
        } else {
            /* nothing to do */
        }
    }

    /* wait until stop flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_STPDET, SET, timeout_ms)) {
        HAL_DEBUGE("I2C get stop signal timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* clear stop flag */
    hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_STPDET);

    /* disable address acknowledge */
    hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);

    HAL_UNLOCK(i2c_dev);

    i2c_dev->rx_state = HAL_I2C_STATE_READY;

    return ret;
}

/*!
    \brief      write amounts of data to memory, poll transmit process and completed status, the function is blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  mem_address: 0-0xFF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  timeout_ms: configuration time out
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT,
                            HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_memory_write_poll(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                  uint8_t *p_buffer, uint32_t length, uint32_t timeout_ms)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout_ms)) {
        HAL_DEBUGE("parameter [length] or [timeout_ms] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xffU < mem_address) {
        HAL_DEBUGE("parameter [mem_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the tx_state whether it is busy or not */
    if((HAL_I2C_STATE_MEMORY_BUSY_TX == i2c_dev->tx_state) || (HAL_I2C_STATE_BUSY == i2c_dev->tx_state)) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, timeout_ms)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize received parameters */
    i2c_dev->tx_state = HAL_I2C_STATE_MEMORY_BUSY_TX;
    i2c_dev->error_state |= (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;
    i2c_dev->slave_address.memory_address = mem_address;

    if(0U == I2C_MEMORY_ADDRESS_MSB(i2c_dev->slave_address.memory_address)) {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_8BIT;
    } else {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_16BIT;
    }

    /* configuration slave address */
    if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
        HAL_DEBUGE("I2C slave address is invalid value");
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* set nbyte to write and reload i2c_dev->txbuffer.length > I2C_MAX_BYTE_SIZE */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
        i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_WRITE);
    } else {
        i2c_dev->txbuffer.pos = i2c_dev->txbuffer.length;
        if(I2C_MEMORY_ADDRESS_8BIT == i2c_dev->slave_address.memory_size) {
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos + 1U, I2C_AUTOEND_MODE, \
                                        I2C_GENERATE_START_WRITE);
        } else {
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos + 2U, I2C_AUTOEND_MODE, \
                                        I2C_GENERATE_START_WRITE);
        }
    }

    /* master sends device address for memory write request */
    if(HAL_ERR_NONE != _i2c_memory_write(i2c_dev, (uint16_t)i2c_dev->slave_address.memory_address, timeout_ms)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    while(I2C_DATA_SEND_FINISH_LENGTH < i2c_dev->txbuffer.pos) {
        /* wait until the transmission data register is empty */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TI, SET, timeout_ms)) {
            HAL_DEBUGE("I2C get TBE timeout");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* nothing to do */
        }

        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)*i2c_dev->txbuffer.buffer);
        i2c_dev->txbuffer.buffer++;
        i2c_dev->txbuffer.length--;
        i2c_dev->txbuffer.pos--;

        /* send data not empty */
        if((I2C_DATA_SEND_FINISH_LENGTH != i2c_dev->txbuffer.length) && (0U == i2c_dev->txbuffer.pos)) {
            /* wait until TCR flag is set */
            if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TCR, SET, timeout_ms)) {
                HAL_DEBUGE("I2C get TCR flag timeout");
                HAL_UNLOCK(i2c_dev);
                ret = HAL_ERR_TIMEOUT;
            } else {
                /* nothing to do */
            }

            if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
                i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_RELOAD_MODE, \
                                            I2C_NO_STARTSTOP);
            } else {
                i2c_dev->txbuffer.pos = i2c_dev->txbuffer.length;
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_AUTOEND_MODE, \
                                            I2C_NO_STARTSTOP);
            }
        } else {
            /* nothing to do */
        }
    }

    /* wait until stop flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_STPDET, SET, timeout_ms)) {
        HAL_DEBUGE("I2C get stop signal timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* clear stop flag */
    hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_STPDET);

    HAL_UNLOCK(i2c_dev);

    i2c_dev->tx_state = HAL_I2C_STATE_READY;

    return ret;
}

/*!
    \brief      read amounts of data from memory, poll receive process and completed status, the function is blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0x0-0xff
    \param[in]  mem_address: 0x0-0x3ff
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be read
    \param[in]  timeout_ms: configuration time out
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT,
                            HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_memory_read_poll(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                 uint8_t *p_buffer, uint32_t length, uint32_t timeout_ms)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout_ms)) {
        HAL_DEBUGE("parameter [length] or [timeout_ms] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xffU < mem_address) {
        HAL_DEBUGE("pointer [mem_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, timeout_ms)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize received parameters */
    i2c_dev->rx_state    = HAL_I2C_STATE_MEMORY_BUSY_RX;
    i2c_dev->error_state = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;
    i2c_dev->slave_address.memory_address = mem_address;

    /* master sends device address for memory write request */
    if(HAL_ERR_NONE != _i2c_memory_read(i2c_dev, i2c_dev->slave_address.device_address, \
                                        (uint16_t)i2c_dev->slave_address.memory_address, timeout_ms)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* set nbyte to write and reload i2c_dev->rxbuffer.length > I2C_MAX_BYTE_SIZE */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->rxbuffer.length) {
        i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_READ);
    } else {
        i2c_dev->rxbuffer.pos = length;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_AUTOEND_MODE, \
                                    I2C_GENERATE_START_READ);
    }

    while(I2C_DATA_RECEIVED_FINISH_LENGTH < i2c_dev->rxbuffer.pos) {
        /* wait until RBNE flag is set */
        if(HAL_ERR_NONE == _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_RBNE, SET, timeout_ms)) {
            *i2c_dev->rxbuffer.buffer = hals_i2c_data_receive(i2c_dev->periph);
            i2c_dev->rxbuffer.buffer++;
            i2c_dev->rxbuffer.length--;
            i2c_dev->rxbuffer.pos--;
        } else {
            /* nothing to do */
        }

        /* check if the reload mode is enabled or not */
        if((0U != i2c_dev->rxbuffer.length) && (0U == i2c_dev->rxbuffer.pos)) {
            /* wait until I2C TCR flag set */
            if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TCR, SET, timeout_ms)) {
                HAL_DEBUGE("I2C busy timeout");
                HAL_UNLOCK(i2c_dev);
                ret = HAL_ERR_TIMEOUT;
            } else {
                /* nothing to do */
            }

            if(I2C_MAX_BYTE_SIZE <= i2c_dev->rxbuffer.length) {
                i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_RELOAD_MODE, \
                                            I2C_NO_STARTSTOP);
            } else {
                i2c_dev->rxbuffer.pos = i2c_dev->rxbuffer.length;
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_AUTOEND_MODE, \
                                            I2C_NO_STARTSTOP);
            }
        } else {
            /* nothing to do */
        }
    }

    /* wait until stop flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_STPDET, SET, timeout_ms)) {
        HAL_DEBUGE("I2C get stop signal timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* clear stop flag */
    hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_STPDET);

    HAL_UNLOCK(i2c_dev);

    i2c_dev->rx_state = HAL_I2C_STATE_READY;

    return ret;
}

/*!
    \brief      transmit amounts of data in master mode by interrupt method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS,
                            HAL_ERR_TIMEOUT, HAL_ERR_BUSY, details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_master_transmit_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                          uint32_t length, hal_i2c_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->tx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;

    i2c_dev->transfer_option      = I2C_NO_OPTION_TRANSFER;
    i2c_dev->i2c_irq.event_handle = _i2c_master_interrupt;
    i2c_dev->tx_callback          = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->tx_callback) {
            i2c_dev->tx_callback = (void *)p_user_func->tx_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* configuration slave address */
    if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
        HAL_DEBUGE("I2C slave address is invalid value");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* configuration nbyte and data length */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
        i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_WRITE);
    } else {
        i2c_dev->txbuffer.pos = i2c_dev->txbuffer.length;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_AUTOEND_MODE, \
                                    I2C_GENERATE_START_WRITE);
    }

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TX);

    HAL_UNLOCK(i2c_dev);
    return ret;
}

/*!
    \brief      received amounts of data in master mode by interrupt method, the function is
                non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_TIMEOUT
                details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_master_receive_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                         uint32_t length, hal_i2c_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->rx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;

    i2c_dev->transfer_option      = I2C_NO_OPTION_TRANSFER;
    i2c_dev->i2c_irq.event_handle = _i2c_master_interrupt;
    i2c_dev->rx_callback          = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->rx_callback) {
            i2c_dev->rx_callback = (void *)p_user_func->rx_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* configuration slave address */
    if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
        HAL_DEBUGE("I2C slave address is invalid value");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* configuration nbyte and data length */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->rxbuffer.length) {
        i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_READ);
    } else {
        i2c_dev->rxbuffer.pos = i2c_dev->rxbuffer.length;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_AUTOEND_MODE, \
                                    I2C_GENERATE_START_READ);
    }

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_RX);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data in slave mode by interrupt method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT,
                            HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_slave_transmit_interrupt(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                         hal_i2c_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    }

    /* initialize transmit parameters */
    i2c_dev->tx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = length;

    i2c_dev->transfer_option      = I2C_NO_OPTION_TRANSFER;
    i2c_dev->i2c_irq.event_handle = _i2c_slave_interrupt;

    i2c_dev->tx_callback   = NULL;
    i2c_dev->addr_callback = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->tx_callback) {
            i2c_dev->tx_callback = (void *)p_user_func->tx_callback;
        } else {
            /* nothing to do */
        }

        if(NULL != p_user_func->addr_callback) {
            i2c_dev->addr_callback = (void *)p_user_func->addr_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    if(I2C_CTL0(i2c_dev->periph) & I2C_CTL0_SS) {
        /* transmit data form TDATA register */
        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)*i2c_dev->txbuffer.buffer);
        i2c_dev->txbuffer.buffer++;
        i2c_dev->txbuffer.length--;
        i2c_dev->txbuffer.pos--;
    } else {
        /* nothing to do */
    }

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TX);
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_LISTEN);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      receive amounts of data in slave mode by interrupt method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be receive
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL, HAL_ERR_TIMEOUT
                details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_slave_receive_interrupt(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                        hal_i2c_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->rx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = 0U;

    i2c_dev->transfer_option      = I2C_NO_OPTION_TRANSFER;
    i2c_dev->i2c_irq.event_handle = _i2c_slave_interrupt;

    i2c_dev->rx_callback   = NULL;
    i2c_dev->addr_callback = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->rx_callback) {
            i2c_dev->rx_callback = (void *)p_user_func->rx_callback;
        } else {
            /* nothing to do */
        }

        if(NULL != p_user_func->addr_callback) {
            i2c_dev->addr_callback = (void *)p_user_func->addr_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_RX);
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_LISTEN);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      write amounts of data to memory by interrupt method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  mem_address: 0-0xFF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be write
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_VAL,
                            HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_memory_write_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                       uint8_t *p_buffer, uint32_t length, \
                                       hal_i2c_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xffU < mem_address) {
        HAL_DEBUGE("parameter [mem_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the tx_state whether it is busy or not */
    if(HAL_I2C_STATE_MEMORY_BUSY_TX == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->tx_state   = HAL_I2C_STATE_MEMORY_BUSY_TX;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;
    i2c_dev->slave_address.memory_address = mem_address;

    i2c_dev->transfer_option      = I2C_NO_OPTION_TRANSFER;
    i2c_dev->tx_callback          = NULL;
    i2c_dev->i2c_irq.event_handle = _i2c_memory_interrupt;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->tx_callback) {
            i2c_dev->tx_callback = (void *)p_user_func->tx_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    if(0U == I2C_MEMORY_ADDRESS_MSB(i2c_dev->slave_address.memory_address)) {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_8BIT;
    } else {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_16BIT;
    }

    /* configuration slave address */
    if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
        HAL_DEBUGE("I2C slave address is invalid value");
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* set nbyte to write and reload i2c_dev->txbuffer.length > I2C_MAX_BYTE_SIZE */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
        i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_WRITE);
    } else {
        i2c_dev->txbuffer.pos = i2c_dev->txbuffer.length;
        if(I2C_MEMORY_ADDRESS_8BIT == i2c_dev->slave_address.memory_size) {
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos + 1U, I2C_AUTOEND_MODE, \
                                        I2C_GENERATE_START_WRITE);
        } else {
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos + 2U, I2C_AUTOEND_MODE, \
                                        I2C_GENERATE_START_WRITE);
        }
    }

    /* master sends device address for memory write request */
    if(HAL_ERR_NONE != _i2c_memory_write(i2c_dev, (uint16_t)i2c_dev->slave_address.memory_address, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TX);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      read amounts of data to memory by interrupt method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  mem_address: 0-0xFF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be write
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_VAL, HAL_ERR_BUSY details refer to
   gd32h7xx_hal.h
*/
int32_t hal_i2c_memory_read_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                      uint8_t *p_buffer, uint32_t length, \
                                      hal_i2c_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xffU < mem_address) {
        HAL_DEBUGE("parameter [mem_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_MEMORY_BUSY_RX == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->rx_state   = HAL_I2C_STATE_MEMORY_BUSY_RX;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;
    i2c_dev->slave_address.memory_address = mem_address;

    i2c_dev->transfer_option      = I2C_NO_OPTION_TRANSFER;
    i2c_dev->rx_callback          = NULL;
    i2c_dev->i2c_irq.event_handle = _i2c_memory_interrupt;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->rx_callback) {
            i2c_dev->rx_callback = (void *)p_user_func->rx_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    if(0U == I2C_MEMORY_ADDRESS_MSB(i2c_dev->slave_address.memory_address)) {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_8BIT;
    } else {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_16BIT;
    }

    /* master sends device address for memory write request */
    if(HAL_ERR_NONE != _i2c_memory_read(i2c_dev, i2c_dev->slave_address.device_address, \
                                        (uint16_t)i2c_dev->slave_address.memory_address, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* set nbyte to write and reload i2c_dev->rxbuffer.length > I2C_MAX_BYTE_SIZE */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->rxbuffer.length) {
        i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_READ);
    } else {
        i2c_dev->rxbuffer.pos = length;
        if(I2C_MEMORY_ADDRESS_8BIT == i2c_dev->slave_address.memory_size) {
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos + 1U, I2C_AUTOEND_MODE, \
                                        I2C_GENERATE_START_READ);
        } else {
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos + 2U, I2C_AUTOEND_MODE, \
                                        I2C_GENERATE_START_READ);
        }
    }

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_RX);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      serial transmit amounts of data in master mode by interrupt method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  transfer_option: transfer option,
                only one parameter can be selected which is shown as below:
      \arg        I2C_FIRST_FRAME: I2C master sends the first frame of data to the slave.
      \arg        I2C_NEXT_FRAME: I2C master sends the next frame of data to the slave.
      \arg        I2C_LAST_FRAME: I2C master sends the last frame of data to the slave.
      \arg        I2C_NO_OPTION_TRANSFER: I2C master sends the data to the slave without any option.
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to
   gd32h7xx_hal.h
*/
int32_t hal_i2c_master_serial_transmit_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                                 uint32_t length, uint8_t transfer_option, \
                                                 hal_i2c_irq_user_callback_struct *p_user_func)
{
    uint32_t tmp_mode    = 0U;
    uint32_t tmp_request = I2C_GENERATE_START_WRITE;
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if((I2C_FIRST_FRAME != transfer_option) && (I2C_NEXT_FRAME != transfer_option) && \
       (I2C_LAST_FRAME != transfer_option) && (I2C_NO_OPTION_TRANSFER != transfer_option)) {
        HAL_DEBUGE("parameter [transfer_option] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the tx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    }

    if((I2C_FIRST_FRAME == transfer_option) || (I2C_NO_OPTION_TRANSFER == transfer_option)) {
        /* wait until I2C bus is idle */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
            HAL_DEBUGE("I2C busy timeout");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_TIMEOUT;
        }
    }

    /* initialize transmit parameters */
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;
    i2c_dev->tx_state   = HAL_I2C_STATE_BUSY;

    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;

    i2c_dev->tx_callback          = NULL;
    i2c_dev->transfer_option      = transfer_option;
    i2c_dev->i2c_irq.event_handle = _i2c_master_interrupt;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->tx_callback) {
            i2c_dev->tx_callback = (void *)p_user_func->tx_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* If i2c_dev->txbuffer.length > I2C_MAX_BYTE_SIZE, use reload mode */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
        tmp_mode              = I2C_RELOAD_MODE;
        i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
    } else {
        if((I2C_LAST_FRAME != transfer_option) && (I2C_NO_OPTION_TRANSFER != transfer_option)) {
            tmp_mode = I2C_RELOAD_MODE;
        } else {
            tmp_mode = I2C_AUTOEND_MODE;
        }

        i2c_dev->txbuffer.pos = i2c_dev->txbuffer.length;
    }

    if((HAL_I2C_PREVIOUS_STATE_TX == i2c_dev->previous_state) && (I2C_FIRST_FRAME != i2c_dev->transfer_option)) {
        tmp_request = I2C_NO_STARTSTOP;
    } else {
        /* nothing to do */
    }

    /* configuration slave address */
    if((I2C_FIRST_FRAME == i2c_dev->transfer_option) || (I2C_NO_OPTION_TRANSFER == i2c_dev->transfer_option)) {
        /* configuration slave address */
        if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
            HAL_DEBUGE("I2C slave address is invalid value");
            ret = HAL_ERR_VAL;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* send slave address and set nbyte to read */
    _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, tmp_mode, tmp_request);

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TX);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      serial received amounts of data in master mode by interrupt method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be sent
    \param[in]  transfer_option: transfer option
                only one parameter can be selected which is shown as below:
      \arg        I2C_FIRST_FRAME: I2C master sends the first frame of data to the slave.
      \arg        I2C_NEXT_FRAME: I2C master sends the next frame of data to the slave.
      \arg        I2C_LAST_FRAME: I2C master sends the last frame of data to the slave.
      \arg        I2C_NO_OPTION_TRANSFER: I2C master sends the data to the slave without any option.
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to
   gd32h7xx_hal.h
*/
int32_t hal_i2c_master_serial_receive_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                                uint32_t length, uint8_t transfer_option, \
                                                hal_i2c_irq_user_callback_struct *p_user_func)
{
    uint32_t tmp_mode    = 0U;
    uint32_t tmp_request = I2C_GENERATE_START_READ;
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if((I2C_FIRST_FRAME != transfer_option) && (I2C_NEXT_FRAME != transfer_option) && \
       (I2C_LAST_FRAME != transfer_option)) {
        HAL_DEBUGE("parameter [transfer_option] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    }

    if((I2C_FIRST_FRAME == transfer_option)) {
        /* wait until I2C bus is idle */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
            HAL_DEBUGE("I2C busy timeout");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->rx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;

    i2c_dev->transfer_option      = transfer_option;
    i2c_dev->rx_callback          = NULL;
    i2c_dev->i2c_irq.event_handle = _i2c_master_interrupt;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->rx_callback) {
            i2c_dev->rx_callback = (void *)p_user_func->rx_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* Ifi2c_dev->rxbuffer.length > I2C_MAX_BYTE_SIZE, use reload mode */
    if(I2C_MAX_BYTE_SIZE <= (i2c_dev->rxbuffer.length)) {
        tmp_mode              = I2C_RELOAD_MODE;
        i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
    } else {
        if((I2C_LAST_FRAME != transfer_option) && (I2C_NO_OPTION_TRANSFER != transfer_option)) {
            tmp_mode = I2C_RELOAD_MODE;
        } else {
            tmp_mode = I2C_AUTOEND_MODE;
        }
        i2c_dev->rxbuffer.pos = i2c_dev->rxbuffer.length;
    }

    if((HAL_I2C_PREVIOUS_STATE_RX == i2c_dev->previous_state) && (I2C_FIRST_FRAME != i2c_dev->transfer_option)) {
        tmp_request = I2C_NO_STARTSTOP;
    } else {
        /* nothing to do */
    }

    /* configuration slave address */
    if((I2C_FIRST_FRAME == i2c_dev->transfer_option)) {
        /* configuration slave address */
        if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
            HAL_DEBUGE("I2C slave address is invalid value");
            ret = HAL_ERR_VAL;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* configuration master read data length and send start signal */
    _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, tmp_mode, tmp_request);

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_RX);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      serial transmit amounts of data in slave mode by interrupt method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  transfer_option: transfer option,
                only one parameter can be selected which is shown as below:
      \arg        I2C_FIRST_FRAME: I2C master sends the first frame of data to the slave.
      \arg        I2C_NEXT_FRAME: I2C master sends the next frame of data to the slave.
      \arg        I2C_LAST_FRAME: I2C master sends the last frame of data to the slave.
      \arg        I2C_NO_OPTION_TRANSFER: I2C master sends the data to the slave without any option.
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to
   gd32h7xx_hal.h
*/
int32_t hal_i2c_slave_serial_transmit_interrupt(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                                uint8_t transfer_option, hal_i2c_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((I2C_FIRST_FRAME != transfer_option) && (I2C_NEXT_FRAME != transfer_option) && \
       (I2C_LAST_FRAME != transfer_option) && (I2C_NO_OPTION_TRANSFER != transfer_option)) {
        HAL_DEBUGE("parameter [transfer_option] value is invalid");
        return HAL_ERR_VAL;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    if((I2C_FIRST_FRAME == transfer_option) || (I2C_NO_OPTION_TRANSFER == transfer_option)) {
        /* wait until I2C bus is idle */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
            HAL_DEBUGE("I2C busy timeout");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->tx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = 0U;

    i2c_dev->transfer_option      = transfer_option;
    i2c_dev->i2c_irq.event_handle = _i2c_slave_interrupt;

    i2c_dev->tx_callback   = NULL;
    i2c_dev->addr_callback = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->tx_callback) {
            i2c_dev->tx_callback = (void *)p_user_func->tx_callback;
        } else {
            /* nothing to do */
        }

        if(NULL != p_user_func->addr_callback) {
            i2c_dev->addr_callback = (void *)p_user_func->addr_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    if(I2C_CTL0(i2c_dev->periph) & I2C_CTL0_SS) {
        /* transmit data form TDATA register */
        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)*i2c_dev->txbuffer.buffer);
        i2c_dev->txbuffer.buffer++;
        i2c_dev->txbuffer.length--;
        i2c_dev->txbuffer.pos--;
    } else {
        /* nothing to do */
    }

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TX);
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_LISTEN);

    HAL_UNLOCK(i2c_dev);
    return ret;
}

/*!
    \brief      serial received amounts of data in slave mode by interrupt method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be sent
    \param[in]  transfer_option: transfer option,
                only one parameter can be selected which is shown as below:
      \arg        I2C_FIRST_FRAME: I2C master sends the first frame of data to the slave.
      \arg        I2C_NEXT_FRAME: I2C master sends the next frame of data to the slave.
      \arg        I2C_LAST_FRAME: I2C master sends the last frame of data to the slave.
      \arg        I2C_NO_OPTION_TRANSFER: I2C master sends the data to the slave without any option.
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to
   gd32h7xx_hal.h
*/
int32_t hal_i2c_slave_serial_receive_interrupt(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                               uint8_t transfer_option, hal_i2c_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((I2C_FIRST_FRAME != transfer_option) && (I2C_NEXT_FRAME != transfer_option) && \
       (I2C_LAST_FRAME != transfer_option) && (I2C_NO_OPTION_TRANSFER != transfer_option)) {
        HAL_DEBUGE("parameter [transfer_option] value is invalid");
        return HAL_ERR_VAL;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize transmit parameters */
    i2c_dev->rx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = length;

    i2c_dev->transfer_option      = transfer_option;
    i2c_dev->i2c_irq.event_handle = _i2c_slave_interrupt;

    i2c_dev->rx_callback   = NULL;
    i2c_dev->addr_callback = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->rx_callback) {
            i2c_dev->rx_callback = (void *)p_user_func->rx_callback;
        } else {
            /* nothing to do */
        }

        if(NULL != p_user_func->addr_callback) {
            i2c_dev->addr_callback = (void *)p_user_func->addr_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* enable the I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_RX);
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_LISTEN);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      I2C interrupt handler content function,which is merely used in I2C_ERROR_handler
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_error_irq(hal_i2c_dev_struct *i2c_dev)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2c_dev) {
        HAL_DEBUGE("pointer [i2c_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* over-run or under-run when SCL stretch is disabled */
    if(hals_i2c_interrupt_flag_get(i2c_dev->periph, I2C_INT_FLAG_OUERR)) {
        hals_i2c_interrupt_flag_clear(i2c_dev->periph, I2C_INT_FLAG_OUERR);
        i2c_dev->error_state = (uint32_t)HAL_I2C_ERROR_OUERR;
        i2c_dev->last_error  = (uint32_t)HAL_I2C_ERROR_OUERR;
    } else {
        /* nothing to do */
    }

    /* arbitration lost */
    if(hals_i2c_interrupt_flag_get(i2c_dev->periph, I2C_INT_FLAG_LOSTARB)) {
        hals_i2c_interrupt_flag_clear(i2c_dev->periph, I2C_INT_FLAG_LOSTARB);
        i2c_dev->error_state = (uint32_t)HAL_I2C_ERROR_LOSTARB;
        i2c_dev->last_error  = (uint32_t)HAL_I2C_ERROR_LOSTARB;
    } else {
        /* nothing to do */
    }

    /* bus error */
    if(hals_i2c_interrupt_flag_get(i2c_dev->periph, I2C_INT_FLAG_BERR)) {
        hals_i2c_interrupt_flag_clear(i2c_dev->periph, I2C_INT_FLAG_BERR);
        i2c_dev->error_state = (uint32_t)HAL_I2C_ERROR_BERR;
        i2c_dev->last_error  = (uint32_t)HAL_I2C_ERROR_BERR;
    } else {
        /* nothing to do */
    }

    /* CRC value doesn't match */
    if(hals_i2c_interrupt_flag_get(i2c_dev->periph, I2C_INT_FLAG_PECERR)) {
        hals_i2c_interrupt_flag_clear(i2c_dev->periph, I2C_INT_FLAG_PECERR);
        i2c_dev->error_state = (uint32_t)HAL_I2C_ERROR_PECERR;
        i2c_dev->last_error  = (uint32_t)HAL_I2C_ERROR_PECERR;
    } else {
        /* nothing to do */
    }

    if((uint32_t)HAL_I2C_ERROR_NONE != i2c_dev->error_state) {
        if(NULL != i2c_dev->i2c_irq.error_handle) {
            i2c_dev->i2c_irq.error_handle(i2c_dev);
            i2c_dev->error_state = (uint32_t)HAL_I2C_ERROR_NONE;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_ALL);

    return ret;
}

/*!
    \brief      I2C interrupt handler content function,which is merely used in I2C_EVENT_handler
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_event_irq(hal_i2c_dev_struct *i2c_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2c_dev) {
        HAL_DEBUGE("pointer [i2c_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(NULL != i2c_dev->i2c_irq.event_handle) {
        i2c_dev->i2c_irq.event_handle(i2c_dev);
    } else {
        /* nothing to do */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      I2C interrupt handler content function,which is merely used in I2C_WAKEUP_handler
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_wake_up_irq(hal_i2c_dev_struct *i2c_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2c_dev) {
        HAL_DEBUGE("pointer [i2c_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(SET == hals_i2c_interrupt_flag_get(i2c_dev->periph, I2C_INT_FLAG_ADDSEND)) {
        /* enable wake up from power saving mode */
        I2C_CTL0(i2c_dev->periph) |= I2C_CTL0_WUEN;

        /* I2C clock source select irc64m */
        if(I2C0 == i2c_dev->periph) {
            hals_rcu_i2c_clock_config(INDEX_I2C0, RCU_I2C_CKSRC_CK_IRC64MDIV);
        } else if(I2C1 == i2c_dev->periph) {
            hals_rcu_i2c_clock_config(INDEX_I2C1, RCU_I2C_CKSRC_CK_IRC64MDIV);
        } else if(I2C2 == i2c_dev->periph) {
            hals_rcu_i2c_clock_config(INDEX_I2C2, RCU_I2C_CKSRC_CK_IRC64MDIV);
        } else if(I2C3 == i2c_dev->periph){
            hals_rcu_i2c_clock_config(INDEX_I2C3, RCU_I2C_CKSRC_CK_IRC64MDIV);
        } else {
            /* nothing to do */
        }

        if(NULL != i2c_dev->i2c_irq.wakeup_handle) {
            i2c_dev->i2c_irq.wakeup_handle(i2c_dev);
        } else {
            /* nothing to do */
        }

        hals_i2c_interrupt_flag_clear(i2c_dev->periph, I2C_INT_FLAG_ADDSEND);
    } else {
        /* nothing to do */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: the structure that contains callback handlers of I2C interrupt
                The structure member can be assigned as following parameters:
      \arg      hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg      NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_irq_handle_set(hal_i2c_dev_struct *i2c_dev, hal_i2c_irq_struct *p_irq)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* event interrupt handler set */
    if(NULL != p_irq->event_handle) {
        i2c_dev->i2c_irq.event_handle = p_irq->event_handle;
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TX);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_RX);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_LISTEN);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TCIE);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_STPDETIE);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_NACKIE);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_ADDMIE);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_RBNEIE);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TIE);
    } else {
        i2c_dev->i2c_irq.event_handle = NULL;
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_TX);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_RX);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_LISTEN);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_TCIE);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_STPDETIE);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_NACKIE);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_ADDMIE);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_RBNEIE);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_TIE);
    }

    /* error interrupt handler set */
    if(NULL != p_irq->error_handle) {
        i2c_dev->i2c_irq.error_handle = p_irq->error_handle;
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_ERR);
    } else {
        i2c_dev->i2c_irq.error_handle = NULL;
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_ERR);
    }

    /* wakeup interrupt handler set */
    if(NULL != p_irq->wakeup_handle) {
        i2c_dev->i2c_irq.wakeup_handle = p_irq->wakeup_handle;
        I2C_CTL0(i2c_dev->periph) |= I2C_CTL0_WUEN;
    } else {
        i2c_dev->i2c_irq.wakeup_handle = NULL;
        I2C_CTL0(i2c_dev->periph) &= ~I2C_CTL0_WUEN;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_irq_handle_all_reset(hal_i2c_dev_struct *i2c_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2c_dev) {
        HAL_DEBUGE("pointer [i2c_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    i2c_dev->i2c_irq.event_handle  = NULL;
    i2c_dev->i2c_irq.error_handle  = NULL;
    i2c_dev->i2c_irq.wakeup_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable address listen in slave mode by interrupt method
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: call back function for user
                The structure contains the following members:
      \arg      tx_callback: transmit complete callback function
                  if not NULL, the callback will be executed after transmission is completed
                  if NULL, no callback will be executed
      \arg      rx_callback: receive complete callback function
                  if not NULL, the callback will be executed after reception is completed
                  if NULL, no callback will be executed
      \arg      addr_callback: address match callback function
                  if not NULL, the callback will be executed when slave address is matched
                  if NULL, no callback will be executed
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_address_listen_interrupt_enable(hal_i2c_dev_struct *i2c_dev, \
                                                hal_i2c_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2c_dev) {
        HAL_DEBUGE("pointer [i2c_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the tx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    i2c_dev->tx_state = HAL_I2C_STATE_LISTEN;
    i2c_dev->rx_state = HAL_I2C_STATE_LISTEN;

    /* check if I2C is already enabled */
    if(I2C_CTL0_I2CEN != (I2C_CTL0(i2c_dev->periph) & I2C_CTL0_I2CEN)) {
        HAL_I2C_ENABLE(i2c_dev->periph);
    } else {
        /* nothing to do */
    }

    /* enable acknowledge */
    if(RESET == ((I2C_SADDR0(i2c_dev->periph) & I2C_SADDR0_ADDFORMAT) >> 10)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_ENABLE);
    } else {
        /* nothing to do */
    }

    i2c_dev->rx_callback          = NULL;
    i2c_dev->i2c_irq.event_handle = _i2c_address_listen_interrupt;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->rx_callback) {
            i2c_dev->rx_callback = (void *)p_user_func->rx_callback;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* enable I2C interrupt */
    hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_LISTEN);

    return ret;
}

/*!
    \brief      disable address listen in slave mode by interrupt method
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_address_listen_interrupt_disable(hal_i2c_dev_struct *i2c_dev)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2c_dev) {
        HAL_DEBUGE("pointer [i2c_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((HAL_I2C_STATE_BUSY == i2c_dev->tx_state) || (HAL_I2C_STATE_BUSY_LISTEN == i2c_dev->tx_state)) {
        i2c_dev->tx_state = HAL_I2C_STATE_READY;

        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_TX);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_LISTEN);
        i2c_dev->previous_state = HAL_I2C_PREVIOUS_STATE_TX;

        ret = HAL_ERR_NONE;
    } else if((HAL_I2C_STATE_BUSY == i2c_dev->rx_state) || (HAL_I2C_STATE_BUSY_LISTEN == i2c_dev->rx_state)) {
        i2c_dev->rx_state = HAL_I2C_STATE_READY;

        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_RX);
        hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_LISTEN);
        i2c_dev->previous_state = HAL_I2C_PREVIOUS_STATE_RX;

        ret = HAL_ERR_NONE;
    } else {
        ret = HAL_ERR_BUSY;
    }

    return ret;
}

/*!
    \brief      I2C master abort transmit in interrupt mode
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_master_abort_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == i2c_dev) {
        HAL_DEBUGE("parameter [i2c_dev] value is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(SET == ((I2C_CTL1(i2c_dev->periph) & I2C_CTL1_TRDIR) >> 10U)) {
        /* disable interrupt and store previous state */
        if((HAL_I2C_STATE_BUSY == i2c_dev->rx_state)) {
            hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_RX);
            hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_LISTEN);
            i2c_dev->previous_state = HAL_I2C_PREVIOUS_STATE_RX;
        } else {
            /* nothing to do */
        }

        i2c_dev->rx_state = HAL_I2C_STATE_READY;

        /* configuration slave address */
        if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)slave_address)) {
            HAL_DEBUGE("I2C slave address is invalid value");
            ret = HAL_ERR_VAL;
        } else {
            /* nothing to do */
        }

        /* set nbyte to 1 to generate a dummy read on I2C peripheral
           set AUTOEND mode, this will generate a NACK then STOP condition to abort the current transfer */
        _i2c_master_transmit_config(i2c_dev->periph, 1U, I2C_AUTOEND_MODE, I2C_GENERATE_STOP);

        /* I2C interrupts must be enabled to avoid the risk of I2C interrupt handle execution */
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_STPDETIE);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TCIE);
    } else {
        /* disable interrupt and store previous state */
        if((HAL_I2C_STATE_BUSY == i2c_dev->tx_state)) {
            hals_i2c_interrupt_disable(i2c_dev->periph, I2C_INT_TX);
            i2c_dev->previous_state = HAL_I2C_PREVIOUS_STATE_TX;
        } else {
            /* nothing to do */
        }

        i2c_dev->tx_state = HAL_I2C_STATE_READY;

        /* configuration slave address */
        if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)slave_address)) {
            HAL_DEBUGE("I2C slave address is invalid value");
            ret = HAL_ERR_VAL;
        } else {
            /* nothing to do */
        }

        /* set nbyte to 1 to generate a dummy read on I2C peripheral
           set AUTOEND mode, this will generate a NACK then STOP condition to abort the current transfer */
        _i2c_master_transmit_config(i2c_dev->periph, 1U, I2C_AUTOEND_MODE, I2C_GENERATE_STOP);

        /* I2C interrupts must be enabled to avoid the risk of I2C interrupt handle execution */
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_STPDETIE);
        hals_i2c_interrupt_enable(i2c_dev->periph, I2C_INT_TCIE);
    }

    return ret;
}

/*!
    \brief      transmit amounts of data in master mode by dma method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to
   gd32h7xx_hal.h
*/
int32_t hal_i2c_master_transmit_dma(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                    uint32_t length, hal_i2c_user_cb p_user_func)
{
    hal_dma_irq_struct dma_irq;

    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the tx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize DMA interrupt callback function structure with default value */
    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &dma_irq);

    /* initialize transmit parameters */
    i2c_dev->tx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;

    i2c_dev->transfer_option = I2C_NO_OPTION_TRANSFER;
    i2c_dev->tx_callback     = (void *)p_user_func;

    dma_irq.full_finish_handle = _i2c_dma_complete;
    dma_irq.error_handle       = _i2c_dma_error;
    dma_irq.abort_handle       = _i2c_dma_abort;

    /* configure DMA interrupt callback function */
    if(NULL != i2c_dev->p_dma_tx) {
        dma_irq.half_finish_handle = i2c_dev->p_dma_tx->dma_irq.half_finish_handle;
    } else {
        /* nothing to do */
    }

    /* enable I2CX transmit DMA */
    hals_i2c_dma_enable(i2c_dev->periph, (uint8_t)I2C_DMA_TRANSMIT);

    /* configuration slave address */
    if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
        HAL_DEBUGE("I2C slave address is invalid value");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* set nbyte to write and reload i2c_dev->txbuffer.length > I2C_MAX_BYTE_SIZE */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
        i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_WRITE);
    } else {
        i2c_dev->txbuffer.pos = i2c_dev->txbuffer.length;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_AUTOEND_MODE, \
                                    I2C_GENERATE_START_WRITE);
    }

    /* start DMA interrupt mode transfer */
    hal_dma_start_interrupt(i2c_dev->p_dma_tx, (uint32_t)i2c_dev->txbuffer.buffer, \
                            (uint32_t)&I2C_TDATA(i2c_dev->periph), (uint16_t)i2c_dev->txbuffer.pos, &dma_irq);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      received amounts of data in master mode by dma method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to
   gd32h7xx_hal.h
*/
int32_t hal_i2c_master_receive_dma(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                   uint32_t length, hal_i2c_user_cb p_user_func)
{
    hal_dma_irq_struct dma_irq;

    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize DMA interrupt callback function structure with default value */
    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &dma_irq);

    /* initialize transmit parameters */
    i2c_dev->rx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;

    i2c_dev->transfer_option = I2C_NO_OPTION_TRANSFER;
    i2c_dev->rx_callback     = (void *)p_user_func;

    dma_irq.full_finish_handle = _i2c_dma_complete;
    dma_irq.error_handle       = _i2c_dma_error;
    dma_irq.abort_handle       = _i2c_dma_abort;

    /* configure DMA interrupt callback function */
    if(NULL != i2c_dev->p_dma_rx->dma_irq.half_finish_handle) {
        dma_irq.half_finish_handle = i2c_dev->p_dma_rx->dma_irq.half_finish_handle;
    } else {
        /* nothing to do */
    }

    /* configuration dma tx */
    hals_i2c_dma_enable(i2c_dev->periph, (uint8_t)I2C_DMA_RECEIVE);

    /* configuration slave address */
    if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
        HAL_DEBUGE("I2C slave address is invalid value");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* set nbyte to write and reload i2c_dev->rxbuffer.length > I2C_MAX_BYTE_SIZE */
    if(I2C_MAX_BYTE_SIZE <= i2c_dev->rxbuffer.length) {
        i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_RELOAD_MODE, \
                                    I2C_GENERATE_START_READ);
    } else {
        i2c_dev->rxbuffer.pos = length;
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_AUTOEND_MODE, \
                                    I2C_GENERATE_START_READ);
    }

    /* start DMA interrupt mode transfer */
    hal_dma_start_interrupt(i2c_dev->p_dma_rx, (uint32_t)&I2C_RDATA(i2c_dev->periph), \
                            (uint32_t)i2c_dev->rxbuffer.buffer, (uint16_t)i2c_dev->rxbuffer.pos, &dma_irq);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data in slave mode by dma method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to
                gd32h7xx_hal.h
*/
int32_t hal_i2c_slave_transmit_dma(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                   hal_i2c_user_cb p_user_func)
{
    hal_dma_irq_struct dma_irq;

    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize DMA interrupt callback function structure with default value */
    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &dma_irq);

    /* initialize transmit parameters */
    i2c_dev->tx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = length;

    i2c_dev->transfer_option = I2C_NO_OPTION_TRANSFER;
    i2c_dev->tx_callback     = (void *)p_user_func;

    /* enable I2CX transmit DMA */
    hals_i2c_dma_enable(i2c_dev->periph, (uint8_t)I2C_DMA_TRANSMIT);

    dma_irq.full_finish_handle = _i2c_dma_complete;
    dma_irq.error_handle       = _i2c_dma_error;
    dma_irq.abort_handle       = _i2c_dma_abort;

    /* configure DMA interrupt callback function */
    if(NULL != i2c_dev->p_dma_tx) {
        dma_irq.half_finish_handle = i2c_dev->p_dma_tx->dma_irq.half_finish_handle;
    } else {
        /* nothing to do */
    }

    if(RESET != (I2C_SADDR0(i2c_dev->periph) & I2C_SADDR0_ADDFORMAT)) {
        /* wait until ADDSEND flag is set */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_ADDSEND, SET, I2C_BUSY_TIMEOUT)) {
            HAL_DEBUGE("I2C addsend timeout");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* nothing to do */
        }

        /* clear addsend flag */
        hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_ADDSEND);
    } else {
        /* nothing to do */
    }

    /* wait until ADDSEND flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_ADDSEND, SET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C addsend timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* clear addsend flag */
    hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_ADDSEND);

    /* wait until TR flag is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TR, SET, I2C_BUSY_TIMEOUT)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);
        HAL_DEBUGE("I2C get TR timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* start DMA interrupt mode transfer */
    hal_dma_start_interrupt(i2c_dev->p_dma_tx, (uint32_t)i2c_dev->txbuffer.buffer, \
                            (uint32_t)&I2C_TDATA(i2c_dev->periph), (uint16_t)i2c_dev->txbuffer.pos, &dma_irq);

    HAL_UNLOCK(i2c_dev);

    i2c_dev->tx_state = HAL_I2C_STATE_READY;

    return ret;
}

/*!
    \brief      received amounts of data in slave mode by dma method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be sent
    \param[in]  p_user_func: call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL details refer to
   gd32h7xx_hal.h
*/
int32_t hal_i2c_slave_receive_dma(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                  hal_i2c_user_cb p_user_func)
{
    hal_dma_irq_struct dma_irq;

    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the tx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize DMA interrupt callback function structure with default value */
    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &dma_irq);

    /* initialize transmit parameters */
    i2c_dev->rx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = length;

    i2c_dev->transfer_option = I2C_NO_OPTION_TRANSFER;
    i2c_dev->rx_callback     = (void *)p_user_func;

    /* configure DMA interrupt callback function */
    dma_irq.full_finish_handle = _i2c_dma_complete;
    dma_irq.error_handle       = _i2c_dma_error;
    dma_irq.abort_handle       = _i2c_dma_abort;

    /* configure DMA interrupt callback function */
    if(NULL != i2c_dev->p_dma_rx) {
        dma_irq.half_finish_handle = i2c_dev->p_dma_rx->dma_irq.half_finish_handle;
    }

    /* enable I2CX transmit DMA */
    hals_i2c_dma_enable(i2c_dev->periph, (uint8_t)I2C_DMA_RECEIVE);

    /* enable acknowledge */
    if(RESET == ((I2C_SADDR0(i2c_dev->periph) & I2C_SADDR0_ADDFORMAT) >> 10)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_ENABLE);
    } else {
        /* nothing to do */
    }

    /* wait until ADDSEND bit is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_ADDSEND, SET, I2C_BUSY_TIMEOUT)) {
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);
        HAL_DEBUGE("I2C get ADDSEND timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* clear ADDSEND bit */
    hals_i2c_flag_clear(i2c_dev->periph, I2C_FLAG_ADDSEND);

    /* start DMA interrupt mode transfer */
    hal_dma_start_interrupt(i2c_dev->p_dma_rx, (uint32_t)&I2C_RDATA(i2c_dev->periph), \
                            (uint32_t)i2c_dev->rxbuffer.buffer, (uint16_t)i2c_dev->rxbuffer.pos, &dma_irq);

    /* wait until TR bit is reset */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TR, RESET, I2C_BUSY_TIMEOUT)) {
        /* disable address acknowledge */
        hals_i2c_ack_config(i2c_dev->periph, I2C_ACK_DISABLE);
        HAL_DEBUGE("I2C get TR timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    HAL_UNLOCK(i2c_dev);

    i2c_dev->rx_state = HAL_I2C_STATE_READY;

    return ret;
}

/*!
    \brief      write amounts of data to memory by dma method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  mem_address: 0-0xFF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to txbuffer
    \param[in]  length: length of data to be write
    \param[in]  p_user_func: call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_VAL, HAL_ERR_BUSY
                details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_memory_write_dma(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                 uint8_t *p_buffer, uint32_t length, hal_i2c_user_cb p_user_func)
{
    hal_dma_irq_struct dma_irq;

    int32_t ret = HAL_ERR_NONE;

/* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xffU < mem_address) {
        HAL_DEBUGE("parameter [mem_address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the tx_state whether it is busy or not */
    if(HAL_I2C_STATE_BUSY == i2c_dev->tx_state) {
        HAL_DEBUGE("I2C tx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* wait until I2C bus is idle */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* initialize DMA interrupt callback function structure with default value */
    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &dma_irq);

    /* initialize transmit parameters */
    i2c_dev->tx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->txbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->txbuffer.length = length;
    i2c_dev->txbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;
    i2c_dev->slave_address.memory_address = mem_address;

    if(0U == I2C_MEMORY_ADDRESS_MSB(i2c_dev->slave_address.memory_address)) {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_8BIT;
    } else {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_16BIT;
    }

    i2c_dev->transfer_option = I2C_NO_OPTION_TRANSFER;
    i2c_dev->tx_callback     = (void *)p_user_func;

    dma_irq.full_finish_handle = _i2c_dma_complete;
    dma_irq.error_handle       = _i2c_dma_error;
    dma_irq.abort_handle       = _i2c_dma_abort;

    /* configure DMA interrupt callback function */
    if(NULL != i2c_dev->p_dma_tx) {
        dma_irq.half_finish_handle = i2c_dev->p_dma_tx->dma_irq.half_finish_handle;
    } else {
        /* nothing to do */
    }

    /* enable I2CX transmit DMA */
    hals_i2c_dma_enable(i2c_dev->periph, (uint8_t)I2C_DMA_TRANSMIT);

    /* configuration slave address */
    if(RESET == hals_i2c_flag_get(i2c_dev->periph, I2C_FLAG_TCR)) {
        /* configuration slave address */
        if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)i2c_dev->slave_address.device_address)) {
            HAL_DEBUGE("I2C slave address is invalid value");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_VAL;
        } else {
            /* nothing to do */
        }

        /* set nbyte to write and reload i2c_dev->txbuffer.length > I2C_MAX_BYTE_SIZE */
        if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
            i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_RELOAD_MODE, \
                                        I2C_GENERATE_START_WRITE);
        } else {
            i2c_dev->txbuffer.pos = i2c_dev->txbuffer.length;
            if(I2C_MEMORY_ADDRESS_8BIT == i2c_dev->slave_address.memory_size) {
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos + 1U, I2C_AUTOEND_MODE, \
                                            I2C_GENERATE_START_WRITE);
            } else {
                _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos + 2U, I2C_AUTOEND_MODE, \
                                            I2C_GENERATE_START_WRITE);
            }
        }
    } else {
        /* set nbyte to write and reload i2c_dev->txbuffer.length > I2C_MAX_BYTE_SIZE */
        if(I2C_MAX_BYTE_SIZE <= i2c_dev->txbuffer.length) {
            i2c_dev->txbuffer.pos = I2C_MAX_BYTE_SIZE;
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_RELOAD_MODE, \
                                        I2C_NO_STARTSTOP);
        } else {
            i2c_dev->txbuffer.pos = i2c_dev->txbuffer.length;
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->txbuffer.pos, I2C_AUTOEND_MODE, \
                                        I2C_NO_STARTSTOP);
        }
    }

    /* master sends device address for memory write request */
    if(HAL_ERR_NONE != _i2c_memory_write(i2c_dev, (uint16_t)i2c_dev->slave_address.memory_address, I2C_BUSY_TIMEOUT)) {
        HAL_DEBUGE("I2C busy timeout");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* start DMA interrupt mode transfer */
    hal_dma_start_interrupt(i2c_dev->p_dma_tx, (uint32_t)i2c_dev->txbuffer.buffer, \
                            (uint32_t)&I2C_TDATA(i2c_dev->periph), (uint16_t)i2c_dev->txbuffer.pos, &dma_irq);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      read amounts of data to memory by dma method, the function is non-blocking
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  mem_address: 0-0xFF except reserved address, I2C slave address to be sent
    \param[in]  p_buffer: pointer to rxbuffer
    \param[in]  length: length of data to be write
    \param[in]  p_user_func: call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_VAL, HAL_ERR_BUSY
                details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_memory_read_dma(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                uint8_t *p_buffer, uint32_t length, hal_i2c_user_cb p_user_func)
{
    hal_dma_irq_struct dma_irq;

    int32_t ret = HAL_ERR_NONE;

/* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [i2c_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3ffU < slave_address) {
        HAL_DEBUGE("parameter [slave_address] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xffU < mem_address) {
        HAL_DEBUGE("parameter [mem_address] value is invalid");
        return HAL_ERR_VAL;
    }

#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(i2c_dev);

    /* check the rx_state whether it is busy or not */
    if(HAL_I2C_STATE_MEMORY_BUSY_RX == i2c_dev->rx_state) {
        HAL_DEBUGE("I2C rx has already been used, please wait until run_state change to free ");
        HAL_UNLOCK(i2c_dev);
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do */
    }

    /* initialize DMA interrupt callback function structure with default value */
    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &dma_irq);

    /* initialize transmit parameters */
    i2c_dev->rx_state   = HAL_I2C_STATE_BUSY;
    i2c_dev->last_error = (uint32_t)HAL_I2C_ERROR_NONE;

    i2c_dev->rxbuffer.buffer = (uint8_t *)p_buffer;
    i2c_dev->rxbuffer.length = length;
    i2c_dev->rxbuffer.pos    = 0U;

    i2c_dev->slave_address.device_address = slave_address;
    i2c_dev->slave_address.memory_address = mem_address;

    i2c_dev->transfer_option = I2C_NO_OPTION_TRANSFER;
    i2c_dev->rx_callback     = (void *)p_user_func;

    if(0U == I2C_MEMORY_ADDRESS_MSB(i2c_dev->slave_address.memory_address)) {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_8BIT;
    } else {
        i2c_dev->slave_address.memory_size = I2C_MEMORY_ADDRESS_16BIT;
    }

    dma_irq.full_finish_handle = _i2c_dma_complete;
    dma_irq.error_handle       = _i2c_dma_error;
    dma_irq.abort_handle       = _i2c_dma_abort;

    /* configure DMA interrupt callback function */
    if(NULL != i2c_dev->p_dma_rx) {
        dma_irq.half_finish_handle = i2c_dev->p_dma_rx->dma_irq.half_finish_handle;
    } else {
        /* nothing to do */
    }

    /* configuration dma rx */
    hals_i2c_dma_enable(i2c_dev->periph, (uint8_t)I2C_DMA_RECEIVE);

    /* configuration slave address */
    if(RESET == hals_i2c_flag_get(i2c_dev->periph, I2C_FLAG_TCR)) {
        /* master sends device address for memory write request */
        if(HAL_ERR_NONE != _i2c_memory_read(i2c_dev, i2c_dev->slave_address.device_address, \
                                            (uint16_t)i2c_dev->slave_address.memory_address, I2C_BUSY_TIMEOUT)) {
            HAL_DEBUGE("I2C busy timeout");
            HAL_UNLOCK(i2c_dev);
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* nothing to do */
        }

        /* set nbyte to write and reload i2c_dev->rxbuffer.length > I2C_MAX_BYTE_SIZE */
        if(I2C_MAX_BYTE_SIZE <= i2c_dev->rxbuffer.length) {
            i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_RELOAD_MODE, \
                                        I2C_GENERATE_START_READ);
        } else {
            i2c_dev->rxbuffer.pos = i2c_dev->rxbuffer.length;
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_AUTOEND_MODE, \
                                        I2C_GENERATE_START_READ);
        }
    } else {
        /* set nbyte to write and reload i2c_dev->rxbuffer.length > I2C_MAX_BYTE_SIZE */
        if(I2C_MAX_BYTE_SIZE <= i2c_dev->rxbuffer.length) {
            i2c_dev->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_RELOAD_MODE, \
                                        I2C_NO_STARTSTOP);
        } else {
            i2c_dev->rxbuffer.pos = i2c_dev->txbuffer.length;
            _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)i2c_dev->rxbuffer.pos, I2C_AUTOEND_MODE, \
                                        I2C_NO_STARTSTOP);
        }
    }

    /* start DMA interrupt mode transfer */
    hal_dma_start_interrupt(i2c_dev->p_dma_rx, (uint32_t)&I2C_RDATA(i2c_dev->periph), \
                            (uint32_t)i2c_dev->rxbuffer.buffer, (uint16_t)i2c_dev->rxbuffer.pos, &dma_irq);

    HAL_UNLOCK(i2c_dev);

    return ret;
}

/*!
    \brief      check whether the device is ready for access
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_device_ready_check(hal_i2c_dev_struct *i2c_dev)
{
    int32_t ret = HAL_ERR_NONE;

    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == i2c_dev)) {
        HAL_DEBUGE("pointer [i2c_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    while(1) {
        if(RESET == (I2C_CTL1(i2c_dev->periph) & I2C_CTL1_RELOAD)) {
            /* wait until I2C bus is idle */
            while(hals_i2c_flag_get(i2c_dev->periph, I2C_FLAG_I2CBUSY)) {}
            if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_I2CBUSY, RESET, I2C_BUSY_TIMEOUT)) {
                HAL_DEBUGE("I2C busy timeout");
                ret = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* nothing to do */
            }

            /* check if I2C is already enabled */
            if(I2C_CTL0_I2CEN != (I2C_CTL0(i2c_dev->periph) & I2C_CTL0_I2CEN)) {
                HAL_I2C_ENABLE(i2c_dev->periph);
            } else {
                /* nothing to do */
            }

            if((HAL_I2C_STATE_READY != i2c_dev->tx_state) || (HAL_I2C_STATE_READY != i2c_dev->rx_state)) {
                i2c_dev->error_state = (uint32_t)HAL_I2C_ERROR_BERR;
                ret = HAL_ERR_BUSY;
                break;
            } else {
                /* nothing to do */
            }
        } else {
            if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TCR, SET, I2C_BUSY_TIMEOUT)) {
                HAL_DEBUGE("I2C busy timeout");
                ret = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* nothing to do */
            }
        }
        break;
    }

    return ret;
}

/*!
    \brief      configure the scl high and low period of clock in master mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  sclh: 0-0xFF, SCL high period
    \param[in]  scll: 0-0xFF, SCL low period
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_master_clock_config(uint32_t i2c_periph, uint32_t sclh, uint32_t scll)
{
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(0xFF < sclh) {
        HAL_DEBUGE("parameter [sclh] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xFF < scll) {
        HAL_DEBUGE("parameter [scll] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear SCLH, SCLL bits in I2C_TIMING register */
    I2C_TIMING(i2c_periph) &= ~I2C_TIMING_SCLH;
    I2C_TIMING(i2c_periph) &= ~I2C_TIMING_SCLL;

    /* mask SCLH, SCLL bits in I2C_TIMING register */
    sclh = (uint32_t)(sclh << TIMING_SCLH_OFFSET) & I2C_TIMING_SCLH;
    scll = (uint32_t)(scll << TIMING_SCLL_OFFSET) & I2C_TIMING_SCLL;

    /* write SCLH, SCLL bits in I2C_TIMING register */
    I2C_TIMING(i2c_periph) |= (sclh | scll);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure I2C slave address and transfer direction in master mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  trans_direction: I2C transfer direction in master mode
                only one parameter can be selected which is shown as below:
      \arg        I2C_MASTER_TRANSMIT: master transmit
      \arg        I2C_MASTER_RECEIVE: master receive
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_i2c_mastermode_slave_address_config(uint32_t i2c_periph, uint32_t address, uint32_t trans_direction)
{
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x3FF < address) {
        HAL_DEBUGE("parameter [address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure slave address */
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_SADDRESS;
    I2C_CTL1(i2c_periph) |= address;

    /* configure transfer direction */
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_TRDIR;
    I2C_CTL1(i2c_periph) |= trans_direction;

    return HAL_ERR_NONE;
}

/*!
    \brief      start I2C wake-up
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hal_i2c_wakeup_start(uint32_t i2c_periph)
{
    I2C_CTL0(i2c_periph) |= I2C_CTL0_WUEN;
    hals_i2c_interrupt_enable(i2c_periph, I2C_INT_ERR);
    hals_i2c_interrupt_enable(i2c_periph, I2C_INT_STPDETIE);
    hals_i2c_interrupt_enable(i2c_periph, I2C_INT_ADDMIE);
    hals_i2c_interrupt_enable(i2c_periph, I2C_INT_RBNEIE);
}

/*!
    \brief      stop I2C wake-up
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hal_i2c_wakeup_stop(uint32_t i2c_periph)
{
    I2C_CTL0(i2c_periph) &= ~I2C_CTL0_WUEN;
    hals_i2c_interrupt_disable(i2c_periph, I2C_INT_ERR);
    hals_i2c_interrupt_disable(i2c_periph, I2C_INT_ADDMIE);
    hals_i2c_interrupt_disable(i2c_periph, I2C_INT_RBNEIE);

}

/*!
    \brief      configure digital noise filter
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  filter_length: the length of filter spikes
                only one parameter can be selected which is shown as below:
      \arg        DIGITAL_FILTER_DISABLE: digital filter is disabled
      \arg        DIGITAL_FILTER_LESSTHAN_1CLK: digital filter is enabled and filter spikes with a length of up to 1
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_2CLK: digital filter is enabled and filter spikes with a length of up to 2
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_3CLK: digital filter is enabled and filter spikes with a length of up to 3
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_4CLK: digital filter is enabled and filter spikes with a length of up to 4
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_5CLK: digital filter is enabled and filter spikes with a length of up to 5
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_6CLK: digital filter is enabled and filter spikes with a length of up to 6
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_7CLK: digital filter is enabled and filter spikes with a length of up to 7
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_8CLK: digital filter is enabled and filter spikes with a length of up to 8
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_9CLK: digital filter is enabled and filter spikes with a length of up to 9
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_10CLK: digital filter is enabled and filter spikes with a length of up to 10
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_11CLK: digital filter is enabled and filter spikes with a length of up to 11
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_12CLK: digital filter is enabled and filter spikes with a length of up to 12
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_13CLK: digital filter is enabled and filter spikes with a length of up to 13
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_14CLK: digital filter is enabled and filter spikes with a length of up to 14
                    tI2CCLK
      \arg        DIGITAL_FILTER_LESSTHAN_15CLK: digital filter is enabled and filter spikes with a length of up to 15
                    tI2CCLK
    \param[out] none
    \retval     none
*/
void hal_i2c_digital_noise_filter_config(uint32_t i2c_periph, hal_i2c_digital_filter_enum filter_length)
{
    I2C_CTL0(i2c_periph) &= (uint32_t)(~I2C_CTL0_DNF);
    I2C_CTL0(i2c_periph) |= (uint32_t)((uint32_t)filter_length << CTL0_DNF_OFFSET);
}

/*!
    \brief      enable analog noise filter
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hal_i2c_analog_noise_filter_enable(uint32_t i2c_periph)
{
    I2C_CTL0(i2c_periph) &= ~I2C_CTL0_ANOFF;
}

/*!
    \brief      disable analog noise filter
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hal_i2c_analog_noise_filter_disable(uint32_t i2c_periph)
{
    I2C_CTL0(i2c_periph) |= I2C_CTL0_ANOFF;
}

/*!
    \brief      enable I2C wake up from power saving mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hal_i2c_wakeup_enable(uint32_t i2c_periph)
{
    if(0U == (I2C_CTL0(i2c_periph) & I2C_CTL0_DNF)) {
        I2C_CTL0(i2c_periph) |= I2C_CTL0_WUEN;
    }
}

/*!
    \brief      disable wake up from power saving mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hal_i2c_wakeup_disable(uint32_t i2c_periph)
{
    I2C_CTL0(i2c_periph) &= ~I2C_CTL0_WUEN;
}

/*!
    \brief      get I2C state
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  state_type: hal_i2c_state_type_enum
                only one parameter can be selected which is shown as below:
      \arg        HAL_I2C_STATE_TYPE_TX: I2C transmit state type
      \arg        HAL_I2C_STATE_TYPE_RX: I2C receive state type
    \param[out] none
    \retval     int32_t: I2C running state, refer to hal_i2c_run_state_enum for details
*/
hal_i2c_run_state_enum hal_i2c_state_get(hal_i2c_dev_struct *i2c_dev, hal_i2c_state_type_enum state_type)
{
    hal_i2c_run_state_enum state = HAL_I2C_STATE_READY;

    /* return I2C state */
    /* return the specified state */
    if(state_type == HAL_I2C_STATE_TYPE_RX) {
        state = i2c_dev->rx_state;
    } else if(state_type == HAL_I2C_STATE_TYPE_TX) {
        state = i2c_dev->tx_state;
    } else {
        /* nothing to do */
    }

    return state;
}

/*!
    \brief      get I2C mode
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     int32_t: current I2C mode, values from hal_i2c_mode_enum for details
*/
int32_t hal_i2c_mode_get(hal_i2c_dev_struct *i2c_dev)
{
    return (int32_t)i2c_dev->mode;
}


/*!
    \brief      configure the timing parameters
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  psc: 0-0xF, timing prescaler
    \param[in]  scl_delay: 0-0xF, data setup time
    \param[in]  sda_delay: 0-0xF, data hold time
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_i2c_timing_config(uint32_t i2c_periph, uint32_t psc, uint32_t scl_delay, uint32_t sda_delay)
{
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(0xF < psc) {
        HAL_DEBUGE("parameter [psc] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xF < scl_delay) {
        HAL_DEBUGE("parameter [scl_delay] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xF < sda_delay) {
        HAL_DEBUGE("parameter [sda_delay] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear PSC, SCLDELY, SDADELY bits in I2C_TIMING register */
    I2C_TIMING(i2c_periph) &= ~I2C_TIMING_PSC;
    I2C_TIMING(i2c_periph) &= ~I2C_TIMING_SCLDELY;
    I2C_TIMING(i2c_periph) &= ~I2C_TIMING_SDADELY;

    /* mask PSC, SCLDELY, SDADELY bits in I2C_TIMING register */
    psc       = (uint32_t)(psc << TIMING_PSC_OFFSET) & I2C_TIMING_PSC;
    scl_delay = (uint32_t)(scl_delay << TIMING_SCLDELY_OFFSET) & I2C_TIMING_SCLDELY;
    sda_delay = (uint32_t)(sda_delay << TIMING_SDADELY_OFFSET) & I2C_TIMING_SDADELY;

    /* write PSC, SCLDELY, SDADELY bits in I2C_TIMING register */
    I2C_TIMING(i2c_periph) |= (psc | scl_delay | sda_delay);

    return HAL_ERR_NONE;
}

/*!
    \brief      I2C transmit data function
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  data: data of transmission
    \param[out] none
    \retval     none
*/
void hals_i2c_data_transmit(uint32_t i2c_periph, uint32_t data)
{
    I2C_TDATA(i2c_periph) = (I2C_TDATA_TDATA & data);
}

/*!
    \brief      I2C receive data function
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     uint8_t: 0 - 0xF
*/
uint8_t hals_i2c_data_receive(uint32_t i2c_periph)
{
    return (uint8_t)(I2C_RDATA(i2c_periph) & I2C_RDATA_RDATA);
}

/*!
    \brief      I2C master status function
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  trans_direction: I2C transfer direction in master mode
                only one parameter can be selected which is shown as below:
      \arg        I2C_MASTER_TRANSMIT: master transmit
      \arg        I2C_MASTER_RECEIVE: master receive
    \param[out] none
    \retval     none
*/
void hals_i2c_master_status(uint32_t i2c_periph, uint32_t trans_direction)
{
    /* configure transfer direction */
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_TRDIR;
    I2C_CTL1(i2c_periph) |= trans_direction;
}

/*!
    \brief      configure I2C slave address and transfer direction in master mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_i2c_master_addressing(uint32_t i2c_periph, uint32_t address)
{
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x3FF < address) {
        HAL_DEBUGE("parameter [address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure slave address */
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_SADDRESS;
    I2C_CTL1(i2c_periph) |= address;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure number of bytes to be transferred
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  byte_number: number of bytes to be transferred : 0x0-0xFF,
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_i2c_transfer_byte_number_config(uint32_t i2c_periph, uint32_t byte_number)
{
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(0xFF < byte_number) {
        HAL_DEBUGE("parameter [byte_number] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    I2C_CTL1(i2c_periph) &= (uint32_t)(~I2C_CTL1_BYTENUM);
    I2C_CTL1(i2c_periph) |= (uint32_t)(byte_number << CTL1_BYTENUM_OFFSET);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure I2C address
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  address: 7 bit: 0-0x7f; 10 bit: 0- 0x3FF
    \param[in]  addr_format: 7 bits or 10 bits
                only one parameter can be selected which is shown as below:
      \arg        I2C_ADDFORMAT_7BITS: address format is 7 bits
      \arg        I2C_ADDFORMAT_10BITS: address format is 10 bits
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_i2c_device_address_config(uint32_t i2c_periph, uint32_t address, uint32_t addr_format)
{
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x3FF < address) {
        HAL_DEBUGE("parameter [address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure ADDRESS[7:1] and address format */
    address                = address & I2C_ADDRESS_MASK;
    I2C_SADDR0(i2c_periph) = (addr_format | address);
    /* enable I2C address in slave mode */
    I2C_SADDR0(i2c_periph) |= I2C_SADDR0_ADDRESSEN;

    return HAL_ERR_NONE;
}

/*!
    \brief      disable I2C second address in slave mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_device_second_address_disable(uint32_t i2c_periph)
{
    I2C_SADDR1(i2c_periph) &= ~I2C_SADDR1_ADDRESS2EN;
}

/*!
    \brief      configure I2C second slave address
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  address: 7 bit: 0-0x7f
    \param[in]  addr_mask: the bits not need to compare
                only one parameter can be selected which is shown as below:
      \arg        ADDRESS2_NO_MASK: no mask, all the bits must be compared
      \arg        ADDRESS2_MASK_BIT1: ADDRESS2[1] is masked, only ADDRESS2[7:2] are compared
      \arg        ADDRESS2_MASK_BIT1_2: ADDRESS2[2:1] is masked, only ADDRESS2[7:3] are compared
      \arg        ADDRESS2_MASK_BIT1_3: ADDRESS2[3:1] is masked, only ADDRESS2[7:4] are compared
      \arg        ADDRESS2_MASK_BIT1_4: ADDRESS2[4:1] is masked, only ADDRESS2[7:5] are compared
      \arg        ADDRESS2_MASK_BIT1_5: ADDRESS2[5:1] is masked, only ADDRESS2[7:6] are compared
      \arg        ADDRESS2_MASK_BIT1_6: ADDRESS2[6:1] is masked, only ADDRESS2[7] are compared
      \arg        ADDRESS2_MASK_ALL: all the ADDRESS2[7:1] bits are masked
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_i2c_device_second_address_enable(uint32_t i2c_periph, uint32_t address, uint32_t addr_mask)
{
    /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x7F < address) {
        HAL_DEBUGE("parameter [address] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure ADDRESS2[7:1] */
    address = address & I2C_ADDRESS2_MASK;
    I2C_SADDR1(i2c_periph) |= address;

    /* configure ADDRESS2[7:1] mask */
    I2C_SADDR1(i2c_periph) &= ~I2C_SADDR1_ADDMSK2;
    I2C_SADDR1(i2c_periph) |= (uint32_t)(addr_mask << SADDR1_ADDMSK_OFFSET);

    /* enable I2C second address in slave mode */
    I2C_SADDR1(i2c_periph) |= I2C_SADDR1_ADDRESS2EN;

    return HAL_ERR_NONE;
}

/*!
    \brief      whether or not to response to a general call
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  gcallpara: slave response to a general call enable or disable
                only one parameter can be selected which is shown as below:
      \arg        I2C_GCEN_ENABLE: slave will response to a general call
      \arg        I2C_GCEN_DISABLE: slave will not response to a general call
    \param[out] none
    \retval     none
*/
void hals_i2c_slave_response_to_gcall_config(uint32_t i2c_periph, uint32_t gcallpara)
{
    /* configure slave response to a general call enable or disable */
    uint32_t ctl = 0U;

    ctl = I2C_CTL0(i2c_periph);
    ctl &= ~(I2C_CTL0_GCEN);
    ctl |= gcallpara;
    I2C_CTL0(i2c_periph) = ctl;
}

/*!
    \brief      whether to stretch SCL low when data is not ready in slave mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  stretchpara: SCL stretching enable or disable
                only one parameter can be selected which is shown as below:
      \arg        I2C_SCLSTRETCH_ENABLE: enable SCL stretching
      \arg        I2C_SCLSTRETCH_DISABLE: disable SCL stretching
    \param[out] none
    \retval     none
*/
void hals_i2c_stretch_scl_low_config(uint32_t i2c_periph, uint32_t stretchpara)
{
    /* configure I2C SCL stretching */
    uint32_t ctl = 0U;

    ctl = I2C_CTL0(i2c_periph);
    ctl &= ~(I2C_CTL0_SS);
    ctl |= stretchpara;
    I2C_CTL0(i2c_periph) = ctl;
}

/*!
    \brief      enable I2C interrupt
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  interrupt: I2C interrupts, refer to hal_i2c_interrupt_enum
                only one parameter can be selected which is shown as below:
      \arg        I2C_INT_ALL : I2C all interrupt
      \arg        I2C_INT_TX  : I2C transfer relation interrupt
      \arg        I2C_INT_RX  : I2C received relation interrupt
      \arg        I2C_INT_LISTEN :I2C address listen relation interrupt
      \arg        I2C_INT_ERR  :error interrupt
      \arg        I2C_INT_TCIE :transfer complete interrupt
      \arg        I2C_INT_STPDETIE:stop signal detection interrupt
      \arg        I2C_INT_NACKIE :nack received respond interrupt
      \arg        I2C_INT_ADDMIE :address matching in slave mode interrupt
      \arg        I2C_INT_RBNEIE :received interrupt
      \arg        I2C_INT_TIE :transmit interrupt
    \param[out] none
    \retval     none
*/
void hals_i2c_interrupt_enable(uint32_t i2c_periph, hal_i2c_interrupt_enum interrupt)
{
    if(I2C_INT_ALL == interrupt) {
        I2C_CTL0(i2c_periph) |= I2C_ALL_INT;
    } else {
        /* nothing to do */
    }

    if(I2C_INT_TX == interrupt) {
        I2C_CTL0(i2c_periph) |= (I2C_INT_TCIE | I2C_INT_TIE | I2C_INT_ERR | I2C_INT_STPDETIE);
    } else {
        /* nothing to do */
    }

    if(I2C_INT_RX == interrupt) {
        I2C_CTL0(i2c_periph) |= (I2C_INT_TCIE | I2C_INT_RBNEIE | I2C_INT_ERR | I2C_INT_STPDETIE);
    } else {
        /* nothing to do */
    }

    if(I2C_INT_LISTEN == interrupt) {
        I2C_CTL0(i2c_periph) |= (I2C_INT_ADDMIE | I2C_INT_ERR | I2C_INT_STPDETIE);
    } else {
        /* nothing to do */
    }

    I2C_CTL0(i2c_periph) |= interrupt;
}

/*!
    \brief      disable I2C interrupt
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  interrupt: I2C interrupts, refer to hal_i2c_interrupt_enum
                only one parameter can be selected which is shown as below:
      \arg        I2C_INT_ALL : I2C all interrupt
      \arg        I2C_INT_TX  : I2C transfer relation interrupt
      \arg        I2C_INT_RX  : I2C received relation interrupt
      \arg        I2C_INT_LISTEN :I2C address listen relation interrupt
      \arg        I2C_INT_ERR  :error interrupt
      \arg        I2C_INT_TCIE :transfer complete interrupt
      \arg        I2C_INT_STPDETIE:stop signal detection interrupt
      \arg        I2C_INT_NACKIE :nack received respond interrupt
      \arg        I2C_INT_ADDMIE :address matching in slave mode interrupt
      \arg        I2C_INT_RBNEIE :received interrupt
      \arg        I2C_INT_TIE :transmit interrupt
    \param[out] none
    \retval     none
*/
void hals_i2c_interrupt_disable(uint32_t i2c_periph, hal_i2c_interrupt_enum interrupt)
{
    if(I2C_INT_ALL == interrupt) {
        I2C_CTL0(i2c_periph) &= ~I2C_ALL_INT;
    } else {
        /* nothing to do */
    }

    if(I2C_INT_TX == interrupt) {
        I2C_CTL0(i2c_periph) &= ~(uint32_t)(I2C_INT_TCIE | I2C_INT_TIE | I2C_INT_ERR | \
                                            I2C_INT_NACKIE | I2C_INT_STPDETIE);
    } else {
        /* nothing to do */
    }

    if(I2C_INT_RX == interrupt) {
        I2C_CTL0(i2c_periph) &= ~(uint32_t)(I2C_INT_TCIE | I2C_INT_RBNEIE | I2C_INT_ERR | \
                                            I2C_INT_NACKIE | I2C_INT_STPDETIE);
    } else {
        /* nothing to do */
    }

    if(I2C_INT_LISTEN == interrupt) {
        I2C_CTL0(i2c_periph) &= ~(uint32_t)(I2C_INT_ADDMIE | I2C_INT_ERR | I2C_INT_NACKIE | I2C_INT_STPDETIE);
    } else {
        /* nothing to do */
    }

    I2C_CTL0(i2c_periph) &= ~(uint32_t)interrupt;
}

/*!
    \brief      enable I2C DMA for transmission or reception
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  dma: I2C DMA transfer or receive
                only one parameter can be selected which is shown as below:
      \arg        I2C_DMA_TRANSMIT: transmit data using DMA
      \arg        I2C_DMA_RECEIVE: receive data using DMA
    \param[out] none
    \retval     none
*/
void hals_i2c_dma_enable(uint32_t i2c_periph, uint8_t dma)
{
    if(I2C_DMA_TRANSMIT == dma) {
        I2C_CTL0(i2c_periph) |= I2C_CTL0_DENT;
    } else {
        I2C_CTL0(i2c_periph) |= I2C_CTL0_DENR;
    }
}

/*!
    \brief      disable I2C DMA for transmission or reception
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  dma: I2C DMA transfer or receive
                only one parameter can be selected which is shown as below:
      \arg        I2C_DMA_TRANSMIT: transmit data using DMA
      \arg        I2C_DMA_RECEIVE: receive data using DMA
    \param[out] none
    \retval     none
*/
void hals_i2c_dma_disable(uint32_t i2c_periph, uint8_t dma)
{
    if(I2C_DMA_TRANSMIT == dma) {
        I2C_CTL0(i2c_periph) &= ~I2C_CTL0_DENT;
    } else {
        I2C_CTL0(i2c_periph) &= ~I2C_CTL0_DENR;
    }
}

/*!
    \brief      whether or not to send an ack
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  ack: ACK or NACK
                only one parameter can be selected which is shown as below:
      \arg        I2C_ACK_ENABLE: ACK will be sent
      \arg        I2C_ACK_DISABLE: ACK will not be sent
    \param[out] none
    \retval     none
*/
void hals_i2c_ack_config(uint32_t i2c_periph, uint32_t ack)
{
    uint32_t ctl = 0U;

    ctl = I2C_CTL1(i2c_periph);
    ctl &= ~(I2C_CTL1_NACKEN);
    ctl |= ack;
    I2C_CTL1(i2c_periph) = ctl;
}

/*!
    \brief      generate a nack in slave mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_nack_enable(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) |= I2C_CTL1_NACKEN;
}

/*!
    \brief      generate an ack in slave mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_nack_disable(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_NACKEN;
}

/*!
    \brief      enable I2C automatic end mode in master mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_automatic_end_enable(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_RELOAD;
    I2C_CTL1(i2c_periph) |= I2C_CTL1_AUTOEND;
}

/*!
    \brief      disable I2C automatic end mode in master mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_automatic_end_disable(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_AUTOEND;
}

/*!
    \brief      enable I2C reload mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_reload_enable(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_AUTOEND;
    I2C_CTL1(i2c_periph) |= I2C_CTL1_RELOAD;
}

/*!
    \brief      disable I2C reload mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_reload_disable(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_RELOAD;
}

/*!
    \brief      enable 10-bit addressing mode in master mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_address10_enable(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) |= I2C_CTL1_ADD10EN;
}

/*!
    \brief      disable 10-bit addressing mode in master mode
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
void hals_i2c_address10_disable(uint32_t i2c_periph)
{
    I2C_CTL1(i2c_periph) &= ~I2C_CTL1_ADD10EN;
}

/*!
    \brief      check I2C flag is set or not
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  flag: I2C flags
                only one parameter can be selected which is shown as below:
      \arg        I2C_FLAG_TBE: I2C_TDATA is empty during transmitting
      \arg        I2C_FLAG_TI: transmit interrupt
      \arg        I2C_FLAG_RBNE: I2C_RDATA is not empty during receiving
      \arg        I2C_FLAG_ADDSEND: address received matches in slave mode
      \arg        I2C_FLAG_NACK: not acknowledge flag
      \arg        I2C_FLAG_STPDET: STOP condition detected in slave mode
      \arg        I2C_FLAG_TC: transfer complete in master mode
      \arg        I2C_FLAG_TCR: transfer complete reload
      \arg        I2C_FLAG_BERR: bus error
      \arg        I2C_FLAG_LOSTARB: arbitration Lost
      \arg        I2C_FLAG_OUERR: overrun/underrun error in slave mode
      \arg        I2C_FLAG_PECERR: PEC error
      \arg        I2C_FLAG_TIMEOUT: timeout flag
      \arg        I2C_FLAG_SMBALT: SMBus alert
      \arg        I2C_FLAG_I2CBSY: busy flag
      \arg        I2C_FLAG_TR: whether the I2C is a transmitter or a receiver in slave mode
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_i2c_flag_get(uint32_t i2c_periph, hal_i2c_flag_enum flag)
{
    FlagStatus ret = RESET;

    if(RESET != (I2C_STAT(i2c_periph) & (uint32_t)flag)) {
        ret = SET;
    } else {
        ret = RESET;
    }

    return ret;
}

/*!
    \brief      clear I2C flag status
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  flag: I2C flags
                only one parameter can be selected which is shown as below:
      \arg        I2C_FLAG_TBE: I2C_TDATA is empty during transmitting
      \arg        I2C_FLAG_TI: I2C_TDATA is empty during transmitting and ready to transfer data
      \arg        I2C_FLAG_RBNE: I2C_RDATA is not empty during receiving
      \arg        I2C_FLAG_ADDSEND: address is sent in master mode or received and matches in slave mode
      \arg        I2C_FLAG_NACK: received nack request
      \arg        I2C_FLAG_STPDET: stop condition detected in slave mode
      \arg        I2C_FLAG_TC: transfer end in master mode
      \arg        I2C_FLAG_TCR: transfer reload complete
      \arg        I2C_FLAG_BERR: a bus error occurs indication a unexpected start or stop condition on I2C bus
      \arg        I2C_FLAG_LOSTARB: arbitration lost in master mode
      \arg        I2C_FLAG_OUERR: over-run or under-run situation occurs in slave mode
      \arg        I2C_FLAG_PECERR: PEC error when receiving data
      \arg        I2C_FLAG_TIMEOUT: timeout signal
      \arg        I2C_FLAG_SMBALT: SMBus alert status
      \arg        I2C_FLAG_I2CBUSY: busy flag
      \arg        I2C_FLAG_TR: whether the I2C is a transmitter or a receiver
    \param[out] none
    \retval     none
*/
void hals_i2c_flag_clear(uint32_t i2c_periph, hal_i2c_flag_enum flag)
{
    I2C_STATC(i2c_periph) |= (uint32_t)flag;
}

/*!
    \brief      get I2C interrupt flag status
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  int_flag: I2C interrupt flags
                only one parameter can be selected which is shown as below:
      \arg        I2C_INT_FLAG_TI: transmit interrupt flag
      \arg        I2C_INT_FLAG_RBNE: I2C_RDATA is not empty during receiving interrupt flag
      \arg        I2C_INT_FLAG_ADDSEND: address received matches in slave mode interrupt flag
      \arg        I2C_INT_FLAG_NACK: not acknowledge interrupt flag
      \arg        I2C_INT_FLAG_STPDET: stop condition detected in slave mode interrupt flag
      \arg        I2C_INT_FLAG_TC: transfer complete in master mode interrupt flag
      \arg        I2C_INT_FLAG_TCR: transfer complete reload interrupt flag
      \arg        I2C_INT_FLAG_BERR: bus error interrupt flag
      \arg        I2C_INT_FLAG_LOSTARB: arbitration lost interrupt flag
      \arg        I2C_INT_FLAG_OUERR: overrun/underrun error in slave mode interrupt flag
      \arg        I2C_INT_FLAG_PECERR: PEC error interrupt flag
      \arg        I2C_INT_FLAG_TIMEOUT: timeout interrupt flag
      \arg        I2C_INT_FLAG_SMBALT: SMBus Alert interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_i2c_interrupt_flag_get(uint32_t i2c_periph, hal_i2c_interrupt_flag_enum int_flag)
{
    FlagStatus ret = RESET;
    uint32_t ret1 = RESET;
    uint32_t ret2 = RESET;

    /* get the status of interrupt enable bit */
    ret1 = (I2C_REG_VAL(i2c_periph, int_flag) & BIT(I2C_BIT_POS(int_flag)));
    /* get the status of interrupt flag */
    ret2 = (I2C_REG_VAL2(i2c_periph, int_flag) & BIT(I2C_BIT_POS2(int_flag)));

    if(ret1 && ret2) {
        ret = SET;
    } else {
        ret = RESET;
    }

    return ret;
}

/*!
    \brief      clear I2C interrupt flag status
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  int_flag: I2C interrupt flags
                only one parameter can be selected which is shown as below:
      \arg        I2C_INT_FLAG_ADDSEND: address received matches in slave mode interrupt flag
      \arg        I2C_INT_FLAG_NACK: not acknowledge interrupt flag
      \arg        I2C_INT_FLAG_STPDET: stop condition detected in slave mode interrupt flag
      \arg        I2C_INT_FLAG_BERR: bus error interrupt flag
      \arg        I2C_INT_FLAG_LOSTARB: arbitration lost interrupt flag
      \arg        I2C_INT_FLAG_OUERR: overrun/underrun error in slave mode interrupt flag
      \arg        I2C_INT_FLAG_PECERR: PEC error interrupt flag
      \arg        I2C_INT_FLAG_TIMEOUT: timeout interrupt flag
      \arg        I2C_INT_FLAG_SMBALT: SMBus Alert interrupt flag
    \param[out] none
    \retval     none
*/
void hals_i2c_interrupt_flag_clear(uint32_t i2c_periph, hal_i2c_interrupt_flag_enum int_flag)
{
    I2C_STATC(i2c_periph) |= BIT(I2C_BIT_POS2(int_flag));
}

/*!
    \brief      I2C Tx data register flush process.
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[out] none
    \retval     none
*/
static void _i2c_flush_tdata_register(uint32_t i2c_periph)
{
    /* write a dummy data in TDATA to clear it */
    if(SET == hals_i2c_flag_get(i2c_periph, I2C_FLAG_TI)) {
        hals_i2c_data_transmit(i2c_periph, 0x00000000U);
    } else {
        /* nothing to do */
    }

    /* clear TBE flag if it's not already cleared */
    if(RESET == hals_i2c_flag_get(i2c_periph, I2C_FLAG_TBE)) {
        I2C_STAT(i2c_periph) |= I2C_STAT_TBE;
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      interrupt Sub-Routine which handle the interrupt Flags Memory Mode with interrupt
                for both transmit and receive operations
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2c_memory_interrupt(void *i2c_dev)
{
    hal_i2c_dev_struct *p_i2c = i2c_dev;
    hal_i2c_user_cb p_func_tx = (hal_i2c_user_cb)p_i2c->tx_callback;
    hal_i2c_user_cb p_func_rx = (hal_i2c_user_cb)p_i2c->rx_callback;

    if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_STPDET)) {
        /* Determine whether we were in transmit or receive mode */
        p_i2c->tx_state = HAL_I2C_STATE_READY;
        p_i2c->rx_state = HAL_I2C_STATE_READY;

        /* clear stop signal Flag */
        hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_STPDET);

        /* disable interrupt */
        hals_i2c_interrupt_disable(p_i2c->periph, I2C_INT_ALL);

        /* user configuration callback based on current mode */
        if (NULL != p_func_tx) {
            p_func_tx(p_i2c);
        } else if (NULL != p_func_rx) {
            p_func_rx(p_i2c);
        } else {
            /* nothing to do */
        }
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_TI)) {
        /* Transmit data */
        hals_i2c_data_transmit(p_i2c->periph, (uint32_t)*p_i2c->txbuffer.buffer);
        p_i2c->txbuffer.buffer++;
        p_i2c->txbuffer.pos--;
        p_i2c->txbuffer.length--;
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_RBNE)) {
        /* received data */
        *p_i2c->rxbuffer.buffer = hals_i2c_data_receive(p_i2c->periph);
        p_i2c->rxbuffer.buffer++;
        p_i2c->rxbuffer.pos--;
        p_i2c->rxbuffer.length--;
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_TCR)) {
        /* received reload complete interrupt */
        if((0U != p_i2c->rxbuffer.length) && (0U == p_i2c->rxbuffer.pos)) {
            if(I2C_MAX_BYTE_SIZE <= p_i2c->rxbuffer.length) {
                p_i2c->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->rxbuffer.pos, I2C_RELOAD_MODE, \
                                            I2C_NO_STARTSTOP);
            } else {
                p_i2c->rxbuffer.pos = p_i2c->rxbuffer.length;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->rxbuffer.pos, I2C_AUTOEND_MODE, \
                                            I2C_NO_STARTSTOP);
            }
        }
        /* transmit reload complete interrupt */
        if((0U != p_i2c->txbuffer.length) && (0U == p_i2c->txbuffer.pos)) {
            if(I2C_MAX_BYTE_SIZE <= p_i2c->txbuffer.length) {
                p_i2c->txbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->txbuffer.pos, I2C_RELOAD_MODE, \
                                            I2C_NO_STARTSTOP);
            } else {
                p_i2c->txbuffer.pos = p_i2c->txbuffer.length;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->txbuffer.pos, I2C_AUTOEND_MODE, \
                                            I2C_NO_STARTSTOP);
            }
        }
    } else if((SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_NACK))) {
        if((I2C_DATA_SEND_FINISH_LENGTH != p_i2c->txbuffer.length)) {
            p_i2c->error_state = (uint32_t)HAL_I2C_ERROR_MASTER_TX;
        }
        if((I2C_DATA_SEND_FINISH_LENGTH != p_i2c->txbuffer.length)) {
            p_i2c->error_state = (uint32_t)HAL_I2C_ERROR_MASTER_RX;
        }

        /* clear NACK flag */
        hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_NACK);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      event handler in I2C master mode for both transmit and receive
    \param[in]  i2c_dev: I2C device information structure
    \param[out] none
    \retval     none
*/
static void _i2c_master_interrupt(void *i2c_dev)
{
    hal_i2c_dev_struct *p_i2c = i2c_dev;
    hal_i2c_user_cb p_func_tx = (hal_i2c_user_cb)p_i2c->tx_callback;
    hal_i2c_user_cb p_func_rx = (hal_i2c_user_cb)p_i2c->rx_callback;

    if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_STPDET)) {
        /* Clear stop signal flag */
        hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_STPDET);

        /* Check if we are in transmit or receive state */
        p_i2c->tx_state = HAL_I2C_STATE_READY;
        p_i2c->rx_state = HAL_I2C_STATE_READY;

        /* Disable all interrupts */
        hals_i2c_interrupt_disable(p_i2c->periph, I2C_INT_ALL);

        /* Call appropriate callback */
        if (NULL != p_func_tx) {
            p_func_tx(p_i2c);
        } else {
            /* nothing to do */
        }

        if (NULL != p_func_rx) {
            p_func_rx(p_i2c);
        } else {
            /* nothing to do*/
        }
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_TI)) {
        /* Transmit data from tdata */
        I2C_TDATA(p_i2c->periph) = *p_i2c->txbuffer.buffer;
        p_i2c->txbuffer.buffer++;
        p_i2c->txbuffer.pos--;
        p_i2c->txbuffer.length--;
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_RBNE)) {
        /* received data */
        *p_i2c->rxbuffer.buffer = hals_i2c_data_receive(p_i2c->periph);
         p_i2c->rxbuffer.buffer++;
         p_i2c->rxbuffer.pos--;
         p_i2c->rxbuffer.length--;
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_TC)) {
        /* Transfer complete */
        if (HAL_I2C_STATE_BUSY == p_i2c->tx_state) {
            p_i2c->tx_state = HAL_I2C_STATE_READY;
            p_i2c->previous_state = HAL_I2C_PREVIOUS_STATE_TX;

            /* Generate stop signal */
            hal_i2c_stop_on_bus(p_i2c->periph);

            /* Call transmit callback */
            if(NULL != p_func_tx) {
                p_func_tx(p_i2c);
            } else {
                /* nothing to do*/
            }
        } else {
            /* nothing to do */
        }

        if (HAL_I2C_STATE_BUSY == p_i2c->rx_state) {
            p_i2c->rx_state = HAL_I2C_STATE_READY;
            p_i2c->previous_state = HAL_I2C_PREVIOUS_STATE_RX;

            /* Generate stop signal */
            hal_i2c_stop_on_bus(p_i2c->periph);

            /* Call receive callback */
            if(NULL != p_func_rx) {
                p_func_rx(p_i2c);
            } else {
                /* nothing to do*/
            }
        } else {
            /* nothing to do */
        }
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_TCR)) {
        /* transmit reload complete interrupt */
        if((I2C_DATA_SEND_FINISH_LENGTH != p_i2c->txbuffer.length) && (0U == p_i2c->txbuffer.pos)) {
            if(I2C_MAX_BYTE_SIZE <= p_i2c->txbuffer.length) {
                p_i2c->txbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->txbuffer.pos, \
                                            I2C_RELOAD_MODE, I2C_NO_STARTSTOP);
            } else {
                if(I2C_NO_OPTION_TRANSFER == p_i2c->transfer_option) {
                    p_i2c->txbuffer.pos = p_i2c->txbuffer.length;
                    _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->txbuffer.pos, \
                                                I2C_AUTOEND_MODE, I2C_NO_STARTSTOP);
                } else {
                    p_i2c->txbuffer.pos = p_i2c->txbuffer.length;
                    _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->txbuffer.pos, \
                                                I2C_SOFTEND_MODE, I2C_NO_STARTSTOP);
                }
            }
        } else {
            p_i2c->tx_state = HAL_I2C_STATE_READY;
            p_i2c->previous_state = HAL_I2C_PREVIOUS_STATE_TX;

            /* disable interrupt */
            hals_i2c_interrupt_disable(p_i2c->periph, I2C_INT_TCIE);

            /* user configuration rx_callback*/
            if(NULL != p_func_tx) {
                p_func_tx(p_i2c);
            }
        }

        /* received reload complete interrupt */
        if((I2C_DATA_RECEIVED_FINISH_LENGTH != p_i2c->rxbuffer.length) && (0U == p_i2c->rxbuffer.pos)) {
            if(I2C_MAX_BYTE_SIZE <= p_i2c->rxbuffer.length) {
                p_i2c->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->rxbuffer.pos, \
                                            I2C_RELOAD_MODE, I2C_NO_STARTSTOP);
            } else {
                if(I2C_NO_OPTION_TRANSFER == p_i2c->transfer_option) {
                    p_i2c->rxbuffer.pos = p_i2c->rxbuffer.length;
                    _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->rxbuffer.pos, \
                                                I2C_AUTOEND_MODE, I2C_NO_STARTSTOP);
                } else {
                    p_i2c->rxbuffer.pos = p_i2c->rxbuffer.length;
                    _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->rxbuffer.pos, \
                                                I2C_SOFTEND_MODE, I2C_NO_STARTSTOP);
                }
            }
        } else {
            p_i2c->rx_state = HAL_I2C_STATE_READY;
            p_i2c->previous_state = HAL_I2C_PREVIOUS_STATE_RX;

            /* disable interrupt */
            hals_i2c_interrupt_disable(p_i2c->periph, I2C_INT_TCIE);

            /* user configuration rx_callback*/
            if(NULL != p_func_rx) {
                p_func_rx(p_i2c);
            }
        }
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_NACK)) {
        if((I2C_DATA_SEND_FINISH_LENGTH != p_i2c->txbuffer.length)) {
            p_i2c->error_state = (uint32_t)HAL_I2C_ERROR_MASTER_TX;
        }

        if((I2C_DATA_RECEIVED_FINISH_LENGTH != p_i2c->rxbuffer.length)) {
            p_i2c->error_state = (uint32_t)HAL_I2C_ERROR_MASTER_RX;
        }

        /* Clear NACK flag */
        hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_NACK);

        /* Clear tdata register if in transmit mode */
        _i2c_flush_tdata_register(p_i2c->periph);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      event handler in I2C slave mode for both transmit and receive
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2c_slave_interrupt(void *i2c_dev)
{
    hal_i2c_dev_struct *p_i2c = i2c_dev;
    hal_i2c_user_cb p_func_tx = (hal_i2c_user_cb)p_i2c->tx_callback;
    hal_i2c_user_cb p_func_rx = (hal_i2c_user_cb)p_i2c->rx_callback;
    hal_i2c_user_cb p_func_addr = (hal_i2c_user_cb)p_i2c->addr_callback;

    if((SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_STPDET))) {
        p_i2c->rx_state = HAL_I2C_STATE_READY;
        p_i2c->tx_state = HAL_I2C_STATE_READY;

        /* clear the STOP bit */
        hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_STPDET);
        /* disable interrupt */
        hals_i2c_interrupt_disable(p_i2c->periph, I2C_INT_ALL);

        /* user configuration callback based on current mode */
        if (NULL != p_func_tx) {
            p_func_tx(p_i2c);
        } else {
            /* nothing to do*/
        }

        if (NULL != p_func_rx) {
            p_func_rx(p_i2c);
        } else {
            /* nothing to do*/
        }
    } else if((SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_ADDSEND))) {
        /* clear the ADDSEND bit */
        hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_ADDSEND);

        /* clear I2C_TDATA register */
        _i2c_flush_tdata_register(p_i2c->periph);

        /* user configuration addr_callback*/
        if(NULL != p_func_addr) {
            p_func_addr(p_i2c);
        }
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_TI)) {
        /* Transmit data from TDATA */
        hals_i2c_data_transmit(p_i2c->periph, (uint32_t)*p_i2c->txbuffer.buffer);
        p_i2c->txbuffer.buffer++;
        p_i2c->txbuffer.length--;
        p_i2c->txbuffer.pos--;
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_RBNE)) {
        /* received data */
        *p_i2c->rxbuffer.buffer = hals_i2c_data_receive(p_i2c->periph);
         p_i2c->rxbuffer.buffer++;
         p_i2c->rxbuffer.pos--;
         p_i2c->rxbuffer.length--;
    } else if(SET == hals_i2c_interrupt_flag_get(p_i2c->periph, I2C_INT_FLAG_NACK)) {
        if((I2C_DATA_SEND_FINISH_LENGTH != p_i2c->txbuffer.length)) {
            p_i2c->error_state = (uint32_t)HAL_I2C_ERROR_SLAVE_TX;
        }

        if((I2C_DATA_RECEIVED_FINISH_LENGTH != p_i2c->rxbuffer.length)) {
            p_i2c->error_state = (uint32_t)HAL_I2C_ERROR_SLAVE_RX;
        }
        /* clear the ADDSEND bit */
        hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_NACK);
    } else {
        /* nothing to do*/
    }
}

/*!
    \brief      master sends device address for memory write request
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mem_address: 0-0xFF except reserved address, I2C slave address to be sent
    \param[in]  timeout_ms: timeout duration value: 0x0 - 0xFFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
static int32_t _i2c_memory_write(hal_i2c_dev_struct *i2c_dev, uint16_t mem_address, uint32_t timeout_ms)
{
    int32_t ret = HAL_ERR_NONE;

    /* wait until TI bit is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TI, SET, timeout_ms)) {
        HAL_DEBUGE("I2C get TI flag timeout");
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* send memory address to write to */
    if(0U == I2C_MEMORY_ADDRESS_MSB(mem_address)) {
        /* one byte address */
        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)I2C_MEMORY_ADDRESS_LSB(mem_address));
    } else {
        /* two bytes address */
        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)I2C_MEMORY_ADDRESS_MSB(mem_address));

        /* wait for TI set */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TI, SET, timeout_ms)) {
            HAL_DEBUGE("I2C get TBE timeout");
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* nothing to do */
        }

        /* two bytes address */
        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)I2C_MEMORY_ADDRESS_LSB(mem_address));
    }

    /* wait for TI set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TI, SET, timeout_ms)) {
        HAL_DEBUGE("I2C get TBE timeout");
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      DMA transfer complete in I2C mode for both transmit and receive
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2c_dma_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_i2c_dev_struct *p_i2c;
    hal_i2c_user_cb p_func_tx;
    hal_i2c_user_cb p_func_rx;
    uint32_t pos;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_i2c  = (hal_i2c_dev_struct *)p_dma->p_periph;
    p_func_tx = (hal_i2c_user_cb)p_i2c->tx_callback;
    p_func_rx = (hal_i2c_user_cb)p_i2c->rx_callback;

    if(HAL_I2C_STATE_BUSY == p_i2c->tx_state){
        pos = p_i2c->txbuffer.pos;
        p_i2c->txbuffer.length -= pos;
        if(I2C_DATA_SEND_FINISH_LENGTH < p_i2c->txbuffer.length) {
            p_i2c->txbuffer.buffer += I2C_MAX_BYTE_SIZE;

            /* set nbyte to write and reload i2c_dev->txbuffer.length > I2C_MAX_BYTE_SIZE */
            if(I2C_MAX_BYTE_SIZE <= p_i2c->txbuffer.length) {
                p_i2c->txbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->txbuffer.pos, \
                                            I2C_RELOAD_MODE, I2C_GENERATE_START_READ);
            } else {
                p_i2c->txbuffer.pos = p_i2c->txbuffer.length;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->txbuffer.pos, \
                                            I2C_AUTOEND_MODE, I2C_GENERATE_START_READ);
            }

            /* check the direction of the data transfer */
            if(DMA_MEMORY_TO_PERIPH == (DMA_CHCTL(p_i2c->p_dma_tx->dma_periph, \
                                                p_i2c->p_dma_tx->channel) & DMA_CHXCTL_TM)) {
                /* configure the transfer destination and source address */
                hals_dma_periph_address_config(p_i2c->p_dma_tx->dma_periph, \
                                            p_i2c->p_dma_tx->channel, \
                                            (uint32_t)p_i2c->txbuffer.buffer);
            } else {
                /* configure the transfer destination and source address */
                hals_dma_memory_address_config(p_i2c->p_dma_tx->dma_periph, \
                                            p_i2c->p_dma_tx->channel, (uint8_t)DMA_MEMORY_0, \
                                            (uint32_t)p_i2c->txbuffer.buffer);
            }
            /* configure the transfer number */
            hals_dma_transfer_number_config(p_i2c->p_dma_tx->dma_periph, \
                                            p_i2c->p_dma_tx->channel, p_i2c->txbuffer.pos);

            /* enable DMA channel */
            hals_dma_channel_enable(p_i2c->p_dma_tx->dma_periph, p_i2c->p_dma_tx->channel);
        } else {
            /* DMA normal mode */
            if(RESET == (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
                p_i2c->txbuffer.pos = p_i2c->txbuffer.length;
                if(RESET == hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_TCR)) {
                    hals_i2c_dma_disable(p_i2c->periph, (uint8_t)I2C_DMA_TRANSMIT);
                }
                p_i2c->tx_state = HAL_I2C_STATE_READY;
            }

            /* master transmit complete flag */
            if((SET == ((I2C_CTL1(p_i2c->periph) & I2C_CTL1_AUTOEND) >> 25))) {
                while(!hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_STPDET));

                /* clear the stop signal bit */
                hals_i2c_flag_clear(p_i2c->periph, I2C_FLAG_STPDET);
                HAL_I2C_ENABLE(p_i2c->periph);

                /* clear parameter */
                I2C_CTL1(p_i2c->periph) &= ~(I2C_CTL1_AUTOEND | I2C_CTL1_BYTENUM | I2C_CTL1_SADDRESS);
            }

            /* master transmit complete flag */
            if(SET == hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_TC)) {
                hal_i2c_stop_on_bus(p_i2c->periph);
                while(I2C_CTL1(p_i2c->periph) & 0x00004000U);

                /* clear parameter */
                I2C_CTL1(p_i2c->periph) &= ~(I2C_CTL1_AUTOEND | I2C_CTL1_BYTENUM | I2C_CTL1_SADDRESS);
            }

            /* slave transmit complete flag */
            if((SET == hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_TR)) && \
            (SET == hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_TI))) {
                while(!hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_STPDET));
                /* clear the stop signal bit */
                hals_i2c_flag_clear(p_i2c->periph, I2C_FLAG_STPDET);
                HAL_I2C_ENABLE(p_i2c->periph);
            }

            if(NULL != p_func_tx) {
                p_func_tx(p_i2c);
            }
        }
    }

    if(HAL_I2C_STATE_BUSY == p_i2c->rx_state){
        pos = p_i2c->rxbuffer.pos;
        p_i2c->rxbuffer.length -= pos;
        if(I2C_DATA_RECEIVED_FINISH_LENGTH < p_i2c->rxbuffer.length) {
            p_i2c->rxbuffer.buffer += I2C_MAX_BYTE_SIZE;

            /* set nbyte to write and reload i2c_dev->rxbuffer.length > I2C_MAX_BYTE_SIZE */
            if(I2C_MAX_BYTE_SIZE <= p_i2c->rxbuffer.length) {
                p_i2c->rxbuffer.pos = I2C_MAX_BYTE_SIZE;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->rxbuffer.pos, \
                                            I2C_RELOAD_MODE, I2C_GENERATE_START_READ);
            } else {
                p_i2c->rxbuffer.pos = p_i2c->rxbuffer.length;
                _i2c_master_transmit_config(p_i2c->periph, (uint8_t)p_i2c->rxbuffer.pos, \
                                            I2C_AUTOEND_MODE, I2C_GENERATE_START_READ);
            }

            /* check the direction of the data transfer */
            if(DMA_MEMORY_TO_PERIPH == (DMA_CHCTL(p_i2c->p_dma_rx->dma_periph, \
                                                p_i2c->p_dma_rx->channel) & DMA_CHXCTL_TM)) {
                /* configure the transfer destination and source address */
                hals_dma_periph_address_config(p_i2c->p_dma_rx->dma_periph, \
                                            p_i2c->p_dma_rx->channel, \
                                            (uint32_t)p_i2c->rxbuffer.buffer);
            } else {
                /* configure the transfer destination and source address */
                hals_dma_memory_address_config(p_i2c->p_dma_rx->dma_periph, \
                                            p_i2c->p_dma_rx->channel, (uint8_t)DMA_MEMORY_0, \
                                            (uint32_t)p_i2c->rxbuffer.buffer);
            }
            /* configure the transfer number */
            hals_dma_transfer_number_config(p_i2c->p_dma_rx->dma_periph, \
                                            p_i2c->p_dma_rx->channel, p_i2c->rxbuffer.pos);

            /* enable DMA channel */
            hals_dma_channel_enable(p_i2c->p_dma_rx->dma_periph, p_i2c->p_dma_rx->channel);
        } else {
            /* DMA normal mode */
            if(RESET == (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN)) {
                p_i2c->rxbuffer.pos = p_i2c->rxbuffer.length;
                if(RESET == hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_TCR)) {
                    hals_i2c_dma_disable(p_i2c->periph, (uint8_t)I2C_DMA_RECEIVE);
                }
                hals_i2c_dma_disable(p_i2c->periph, (uint8_t)I2C_DMA_RECEIVE);
                p_i2c->rx_state = HAL_I2C_STATE_READY;
            }

            /* master received complete flag */
            if(SET == hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_TC)) {
                hal_i2c_stop_on_bus(p_i2c->periph);
                while(I2C_CTL1(p_i2c->periph) & 0x00004000U);
            } else if(RESET == (I2C_CTL1(p_i2c->periph) & I2C_CTL1_RELOAD)) {
                while(!hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_STPDET));
                hals_i2c_flag_clear(p_i2c->periph, I2C_FLAG_STPDET);
            } else {
                /* nothing to do */
            }

            /* slave received complete flag */
            if((RESET == hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_TR)) && \
            (SET == hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_STPDET))) {
                while(!hals_i2c_flag_get(p_i2c->periph, I2C_FLAG_STPDET));
                /* clear the stop signal bit */
                HAL_I2C_ENABLE(p_i2c->periph);
            }

            if(NULL != p_func_rx) {
                p_func_rx(p_i2c);
            }
        }
    }
}

/*!
    \brief      handle the I2C DMA error process
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2c_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_i2c_dev_struct *p_i2c;
    p_dma = (hal_dma_dev_struct *)dma;
    p_i2c = (hal_i2c_dev_struct *)p_dma->p_periph;

    if(HAL_I2C_STATE_BUSY == p_i2c->tx_state) {
        /* transmit state is busy */
        p_i2c->error_state  = (uint32_t)HAL_I2C_ERROR_DMATX;
        p_i2c->last_error   = (uint32_t)HAL_I2C_ERROR_DMATX;
        p_i2c->txbuffer.pos = p_i2c->txbuffer.length;

        /* disable DMA transmit and reset tx_state */
        hals_i2c_dma_disable(p_i2c->periph, (uint8_t)I2C_DMA_TRANSMIT);
        hals_i2c_interrupt_disable(p_i2c->periph, I2C_INT_TX);
        p_i2c->tx_state = HAL_I2C_STATE_READY;
    } else if(HAL_I2C_STATE_BUSY == p_i2c->rx_state) {
        /* receive state is busy */
        p_i2c->error_state  = (uint32_t)HAL_I2C_ERROR_DMARX;
        p_i2c->last_error   = (uint32_t)HAL_I2C_ERROR_DMARX;
        p_i2c->rxbuffer.pos = p_i2c->rxbuffer.length;

        /* disable DMA receive, PERR, ERR interrupt */
        hals_i2c_dma_disable(p_i2c->periph, (uint8_t)I2C_DMA_RECEIVE);
        hals_i2c_interrupt_disable(p_i2c->periph, I2C_INT_RX);
        /* reset rx_state */
        p_i2c->rx_state = HAL_I2C_STATE_READY;
    } else {
        HAL_DEBUGE("I2C processor fatal error: dma error exception due to run state");
    }

    if(p_i2c->i2c_irq.error_handle != NULL) {
        /* if there is a user error callback */
        p_i2c->i2c_irq.error_handle(p_i2c);
        p_i2c->error_state = (uint32_t)HAL_I2C_ERROR_NONE;
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      handle the I2C DMA abort process
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2c_dma_abort(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_i2c_dev_struct *p_i2c;
    hal_i2c_user_cb p_func;
    uint8_t dma_direction;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_i2c  = (hal_i2c_dev_struct *)p_dma->p_periph;
    p_func = (hal_i2c_user_cb)p_i2c->abort_callback;

    /* disable DMA transmit */
    dma_direction = 0U;
    if(p_dma == p_i2c->p_dma_tx) {
        dma_direction = (uint8_t)I2C_DMA_TRANSMIT;
    } else if(p_dma == p_i2c->p_dma_rx) {
        dma_direction = (uint8_t)I2C_DMA_RECEIVE;
    } else {
        /* nothing to do */
    }

    /* disable DMA */
    if(dma_direction != 0U) {
        hals_i2c_dma_disable(p_i2c->periph, dma_direction);
    } else {
        /* nothing to do */
    }

    /* clear all I2C interrupt flags */
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_TI);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_RBNE);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_NACK);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_STPDET);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_TC);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_TCR);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_OUERR);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_LOSTARB);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_BERR);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_PECERR);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_ADDSEND);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_TIMEOUT);
    hals_i2c_interrupt_flag_clear(p_i2c->periph, I2C_INT_FLAG_SMBALT);

    /* reset the corresponding status */
    if(dma_direction == I2C_DMA_TRANSMIT) {
        p_i2c->tx_state = HAL_I2C_STATE_READY;
        p_i2c->error_state = (uint32_t)HAL_I2C_ERROR_DMATX;
    } else if(dma_direction == I2C_DMA_RECEIVE) {
        p_i2c->rx_state = HAL_I2C_STATE_READY;
        p_i2c->error_state = (uint32_t)HAL_I2C_ERROR_DMARX;
    } else {
        /* nothing to do */
    }

    if(NULL != p_func) {
        p_func(p_i2c);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      master sends device address for memory read request
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  slave_address: 0-0x3FF except reserved address, I2C slave address to be sent
    \param[in]  mem_address: 0-0xFF except reserved address, I2C slave address to be sent
    \param[in]  timeout_ms: timeout duration value: 0x0 - 0xFFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
static int32_t _i2c_memory_read(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                uint32_t timeout_ms)
{
    int32_t ret = HAL_ERR_NONE;

    /* clear i2c_tdata register */
    I2C_STAT(i2c_dev->periph) |= I2C_STAT_TBE;

    /* configuration slave address */
    if(HAL_ERR_NONE != hals_i2c_master_addressing(i2c_dev->periph, (uint32_t)slave_address)) {
        HAL_DEBUGE("I2C slave address is invalid value");
        ret = HAL_ERR_VAL;
    } else {
        /* nothing to do */
    }

    /* send memory address to write to */
    if(0U == I2C_MEMORY_ADDRESS_MSB(mem_address)) {
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)I2C_MEMORY_ADDRESS_8BIT, I2C_SOFTEND_MODE, \
                                    I2C_GENERATE_START_WRITE);
    } else {
        _i2c_master_transmit_config(i2c_dev->periph, (uint8_t)I2C_MEMORY_ADDRESS_16BIT, I2C_SOFTEND_MODE, \
                                    I2C_GENERATE_START_WRITE);
    }

    /* wait until TBE bit is set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TBE, SET, timeout_ms)) {
        HAL_DEBUGE("I2C get TI flag timeout");
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    /* send memory address to write to */
    if(0U == I2C_MEMORY_ADDRESS_MSB(mem_address)) {
        /* one byte address */
        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)I2C_MEMORY_ADDRESS_LSB(mem_address));
    } else {
        /* two bytes address */
        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)I2C_MEMORY_ADDRESS_MSB(mem_address));

        /* wait until TI bit is set */
        if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TI, SET, timeout_ms)) {
            HAL_DEBUGE("I2C busy timeout");
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* nothing to do */
        }

        /* two bytes address */
        hals_i2c_data_transmit(i2c_dev->periph, (uint32_t)I2C_MEMORY_ADDRESS_LSB(mem_address));
    }

    /* wait for TC set */
    if(HAL_ERR_NONE != _i2c_wait_flag_timeout(i2c_dev->periph, I2C_FLAG_TC, SET, timeout_ms)) {
        HAL_DEBUGE("I2C get TC flag timeout");
        ret = HAL_ERR_TIMEOUT;
    } else {
        /* nothing to do */
    }

    return ret;
}

/*!
    \brief      I2C master transmit configuration
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  length: the length of send data
    \param[in]  mode: I2C transfer mode,
                only one parameters can be selected which are shown as below:
     \arg         I2C_RELOAD_MODE: reload mode
     \arg         I2C_AUTOEND_MODE: automatic end mode
     \arg         I2C_SOFTEND_MODE: soft end mode
    \param[in]  request: I2C Master request type,
                only one parameters can be selected which are shown as below:
     \arg         I2C_GENERATE_START_WRITE: generate a start condition and transmit the slave address in write mode
     \arg         I2C_GENERATE_START_READ: generate a start condition and transmit the slave address in read mode
     \arg         I2C_NO_STARTSTOP: no stop condition is generated
     \arg         I2C_GENERATE_STOP: generate a stop condition
    \param[out] none
    \retval     none
*/
static void _i2c_master_transmit_config(uint32_t i2c_periph, uint8_t length, uint32_t mode, uint32_t request)
{
    hals_i2c_transfer_byte_number_config(i2c_periph, (uint32_t)length);

    if(I2C_RELOAD_MODE == mode) {
        hals_i2c_reload_enable(i2c_periph);
    } else if(I2C_AUTOEND_MODE == mode) {
        hals_i2c_automatic_end_enable(i2c_periph);
    } else if(I2C_SOFTEND_MODE == mode) {
        hals_i2c_reload_disable(i2c_periph);
        hals_i2c_automatic_end_disable(i2c_periph);
    } else {
        /* nothing to do */
    }

    switch(request) {
    case I2C_NO_STARTSTOP:
        I2C_CTL1(i2c_periph) &= ~(I2C_CTL1_START | I2C_CTL1_STOP);
        break;
    case I2C_GENERATE_STOP:
        hal_i2c_stop_on_bus(i2c_periph);
        break;
    case I2C_GENERATE_START_READ:
        hals_i2c_master_status(i2c_periph, I2C_MASTER_RECEIVE);
        hal_i2c_start_on_bus(i2c_periph);
        break;
    case I2C_GENERATE_START_WRITE:
        hals_i2c_master_status(i2c_periph, I2C_MASTER_TRANSMIT);
        hal_i2c_start_on_bus(i2c_periph);
        break;
    default:
        break;
    }
}

/*!
    \brief      wait the flag status until timeout
    \param[in]  i2c_periph: I2Cx(x=0,1,2,3)
    \param[in]  flag: I2C flags, refer to i2c_flag_enum
    \param[in]  status: the status of I2C flag to wait
    \param[in]  timeout_ms: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
static int32_t _i2c_wait_flag_timeout(uint32_t i2c_periph, hal_i2c_flag_enum flag, FlagStatus status, \
                                      uint32_t timeout_ms)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t tick_start;

    /* set timeout */
    tick_start = hal_sys_basetick_count_get();

    /* wait flag status RESET */
    if(RESET == status) {
        while(SET == hals_i2c_flag_get(i2c_periph, flag)) {
            if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                    HAL_DEBUGE("I2C get flag timeout");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* nothing to do */
                }
            } else {
                /* nothing to do */
            }
        }
    } else {
        /* wait flag status SET */
        while(RESET == hals_i2c_flag_get(i2c_periph, flag)) {
            if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                    HAL_DEBUGE("I2C get flag timeout");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* nothing to do */
                }
            } else {
                /* nothing to do */
            }
        }
    }

    return ret;
}

/*!
    \brief      event handler for address listen in slave mode
    \param[in]  i2c_dev: I2C device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _i2c_address_listen_interrupt(void *i2c_dev)
{
    hal_i2c_dev_struct *p_i2c = i2c_dev;
    hal_i2c_user_cb p_func    = (hal_i2c_user_cb)p_i2c->rx_callback;

    hals_i2c_interrupt_disable(p_i2c->periph, I2C_INT_LISTEN);

    /* check transfer direction in p_func, so that the user knows
    which function to call next(hal_i2c_slave_serial_transmit_interrupt
    or hal_i2c_slave_serial_receive_interrupt) */
    if(NULL != p_func) {
        p_func(p_i2c);
    } else {
        /* nothing to do */
    }
}
