/*!
    \file    gd32h7xx_hal_hwsem.c
    \brief   HWSEM driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/*!
    \brief      initialize the HWSEM structure with the default values
    \param[in]  struct_type: The type of the struct to initialize
                only one parameters can be selected which are shown as below:
      \arg        HAL_HWSEM_DEV_STRUCT: HWSEM device information structure
      \arg        HAL_HWSEM_IRQ_STRUCT: HWSEM device interrupt callback function pointer structure
    \param[out] p_struct: pointer to HWSEM structure that contains the configuration information
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hwsem_struct_init(hal_hwsem_struct_type_enum struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
    uint8_t i;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(struct_type) {
        /* initialize HWSEM device information structure with the default values */
    case HAL_HWSEM_DEV_STRUCT:
        ((hal_hwsem_dev_struct *)p_struct)->state = HAL_HWSEM_STATE_NONE;
        for(i = 0U; i < 32U; i++) {
            ((hal_hwsem_dev_struct *)p_struct)->hwsem_irq.semaphorex_handle[i] = NULL;
        }
        break;
        /* initialize HWSEM irq structure with the default values */
    case HAL_HWSEM_IRQ_STRUCT:
        for(i = 0U; i < 32U; i++) {
            ((hal_hwsem_irq_struct *)p_struct)->semaphorex_handle[i] = NULL;
        }
        break;
    default:
        HAL_DEBUGE("parameter [struct_type] value is undefined");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize HWSEM
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_hwsem_deinit(hal_hwsem_dev_struct *hwsem_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hwsem_dev) {
        HAL_DEBUGE("pointer [hwsem_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Acquire peripheral lock */
    HAL_LOCK(hwsem_dev);

    hwsem_dev->state = HAL_HWSEM_STATE_BUSY;

    hal_rcu_periph_reset_enable(RCU_HWSEMRST);
    hal_rcu_periph_reset_disable(RCU_HWSEMRST);

    hwsem_dev->state = HAL_HWSEM_STATE_RESET;

    /* Acquire peripheral unlock */
    HAL_UNLOCK(hwsem_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      try to lock the specific semaphore by writing process ID
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  process: the process to lock the semaphore
                only one parameter can be selected which is shown as below:
      \arg        0 - 0xFF
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hal_hwsem_lock_set(hal_hwsem_dev_struct *hwsem_dev, uint8_t process)
{
    uint32_t temp_mid = 0U, temp_pid = 0U;
    ErrStatus ret = ERROR;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hwsem_dev) {
        HAL_DEBUGE("pointer [hwsem_dev] address is invalid");
        return ERROR;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* try to lock the semaphore */
    HWSEM_CTL((uint32_t)hwsem_dev->semaphore) = (uint32_t)(HWSEM_LOCK | CTL_MID(HWSEM_MASTER_ID) | CTL_PID(process));

    /* read the control register to confirm the semaphore is locked by target process or not */
    temp_mid = hal_hwsem_master_id_get(hwsem_dev);
    temp_pid = hal_hwsem_process_id_get(hwsem_dev);

    if((HWSEM_MASTER_ID == temp_mid) && (process == temp_pid)) {
        ret = SUCCESS;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      try to release the lock of the semaphore by writing process ID
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  process: the process to unlock the semaphore
                only one parameter can be selected which is shown as below:
      \arg        0 - 0xFF
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hal_hwsem_lock_release(hal_hwsem_dev_struct *hwsem_dev, uint8_t process)
{
    uint32_t lock_state = 0U;
    ErrStatus ret = ERROR;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hwsem_dev) {
        HAL_DEBUGE("pointer [hwsem_dev] address is invalid");
        return ERROR;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* release the lock of the semaphore */
    HWSEM_CTL((uint32_t)hwsem_dev->semaphore) = (uint32_t)(CTL_MID(HWSEM_MASTER_ID) | CTL_PID(process));

    lock_state = HWSEM_CTL((uint32_t)hwsem_dev->semaphore) & HWSEM_CTL_LK;
    if(0U == lock_state) {
        ret = SUCCESS;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      try to lock the semaphore by reading
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hal_hwsem_lock_by_reading(hal_hwsem_dev_struct *hwsem_dev)
{
    ErrStatus ret = ERROR;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hwsem_dev) {
        HAL_DEBUGE("pointer [hwsem_dev] address is invalid");
        return ERROR;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((uint32_t)(HWSEM_LOCK | CTL_MID(HWSEM_MASTER_ID)) == HWSEM_RLK((uint32_t)hwsem_dev->semaphore)) {
        ret = SUCCESS;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      unlock all semaphores of the master ID
    \param[in]  key: key value
      \arg        0 - 0xFFFF
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hal_hwsem_unlock_all(uint16_t key)
{
    ErrStatus ret = ERROR;

    HWSEM_UNLK = UNLK_KEY(key) | UNLK_MID(HWSEM_MASTER_ID);

    if(key == hal_hwsem_key_get()) {
        ret = SUCCESS;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      get process ID of the specific semaphore
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: process ID of semaphore
*/
uint32_t hal_hwsem_process_id_get(hal_hwsem_dev_struct *hwsem_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hwsem_dev) {
        HAL_DEBUGE("pointer [hwsem_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    return (uint32_t)(GET_CTL_PID(HWSEM_CTL((uint32_t)hwsem_dev->semaphore)));
}

/*!
    \brief      get master ID of the specific semaphore
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: master ID of semaphore
*/
uint32_t hal_hwsem_master_id_get(hal_hwsem_dev_struct *hwsem_dev)
{
    return (uint32_t)(GET_CTL_MID(HWSEM_CTL((uint32_t)hwsem_dev->semaphore)));
}

/*!
    \brief      get the lock status of the semaphore
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hal_hwsem_lock_status_get(hal_hwsem_dev_struct *hwsem_dev)
{
    FlagStatus ret = RESET;

    if(0U != (HWSEM_CTL((uint32_t)hwsem_dev->semaphore) & HWSEM_LOCK)) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      set the key
    \param[in]  key: key value
      \arg        0 - 0xFFFF
    \param[out] none
    \retval     none
*/
void hal_hwsem_key_set(uint16_t key)
{
    HWSEM_KEY = KEY_KEY(key);
}

/*!
    \brief      get the key
    \param[in]  none
    \param[out] none
    \retval     uint16_t: key to unlock all semaphores
*/
uint16_t hal_hwsem_key_get(void)
{
    return ((uint16_t)GET_KEY_KEY(HWSEM_KEY));
}

/*!
    \brief      HWSEM interrupt handler content function,which is merely used in hwsem_handle
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_LOCK details refer to gd32h7xx_hal.h
*/
int32_t hal_hwsem_irq(hal_hwsem_dev_struct *hwsem_dev)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t sem_index = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == hwsem_dev) {
        HAL_DEBUGE("pointer [hwsem_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check semaphore x lock state (x = 0...31) */
    for(sem_index = 0U; sem_index < 32U; sem_index++) {
        if(SET == hals_hwsem_interrupt_flag_get(sem_index)) {
            hals_hwsem_interrupt_flag_clear(sem_index);

            /* check semaphore x lock status (x = 0...31) */
            if(SET != (hal_hwsem_lock_status_get(hwsem_dev))) {
                if(NULL != hwsem_dev->hwsem_irq.semaphorex_handle[sem_index]) {
                    hwsem_dev->hwsem_irq.semaphorex_handle[sem_index](hwsem_dev);
                } else {
                    /* do nothing */
                }
            } else {
                ret = HAL_ERR_LOCK;
                break;
            }
        } else {
            /* do nothing */
        }
    }

    return ret;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to HWSEM interrupt callback functions structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hwsem_irq_handle_set(hal_hwsem_dev_struct *hwsem_dev, hal_hwsem_irq_struct *p_irq)
{
    uint8_t sem_index = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == hwsem_dev) {
        HAL_DEBUGE("pointer [hwsem_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_irq) {
        HAL_DEBUGE("pointer [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* semaphore x interrupt (x = 0...31) */
    for(sem_index = 0U; sem_index < 32U; sem_index++) {
        if(NULL != p_irq->semaphorex_handle[sem_index]) {
            hwsem_dev->hwsem_irq.semaphorex_handle[sem_index] = p_irq->semaphorex_handle[sem_index];
            hals_hwsem_interrupt_enable((uint32_t)sem_index);
        } else {
            hwsem_dev->hwsem_irq.semaphorex_handle[sem_index] = NULL;
            hals_hwsem_interrupt_disable((uint32_t)sem_index);
        }
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  hwsem_dev: HWSEM device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hwsem_irq_handle_all_reset(hal_hwsem_dev_struct *hwsem_dev)
{
    uint8_t i = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == hwsem_dev) {
        HAL_DEBUGE("pointer [hwsem_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    for(i = 0U; i < 32U; i++) {
        hwsem_dev->hwsem_irq.semaphorex_handle[i] = NULL;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      get the HWSEM flag status
    \param[in]  semaphore: semaphore index, refer to hal_hwsem_semaphore_enum
                only one parameter can be selected which is shown as below:
      \arg        SEMx (x=0..31): semaphore x
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_hwsem_flag_get(uint32_t semaphore)
{
    FlagStatus ret = RESET;

    if(RESET != ((HWSEM_STAT >> semaphore) & 0x1U)) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      clear HWSEM flag status
    \param[in]  semaphore: semaphore index, refer to hal_hwsem_semaphore_enum
                only one parameter can be selected which is shown as below:
      \arg        SEMx (x=0..31): semaphore x
    \param[out] none
    \retval     none
*/
void hals_hwsem_flag_clear(uint32_t semaphore)
{
    HWSEM_INTC = ((uint32_t)1U << semaphore);
}

/*!
    \brief      get HWSEM interrupt flag status
    \param[in]  semaphore: semaphore index, refer to hal_hwsem_semaphore_enum
                only one parameter can be selected which is shown as below:
      \arg        SEMx (x=0..31): semaphore x
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_hwsem_interrupt_flag_get(uint32_t semaphore)
{
    FlagStatus ret = RESET;

    if(RESET != ((HWSEM_INTF >> semaphore) & 0x1U)) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      clear HWSEM interrupt flag
    \param[in]  semaphore: semaphore index, refer to hal_hwsem_semaphore_enum
                only one parameter can be selected which is shown as below:
      \arg        SEMx (x=0..31): semaphore x
    \param[out] none
    \retval     none
*/
void hals_hwsem_interrupt_flag_clear(uint32_t semaphore)
{
    HWSEM_INTC = ((uint32_t)1U << semaphore);
}

/*!
    \brief      enable HWSEM interrupt
    \param[in]  semaphore: semaphore index, refer to hal_hwsem_semaphore_enum
                only one parameter can be selected which is shown as below:
      \arg        SEMx (x=0..31): semaphore x
    \param[out] none
    \retval     none
*/
void hals_hwsem_interrupt_enable(uint32_t semaphore)
{
    HWSEM_INTEN |= ((uint32_t)1U << semaphore);
}

/*!
    \brief      disable HWSEM interrupt
    \param[in]  semaphore: semaphore index, refer to hal_hwsem_semaphore_enum
                only one parameter can be selected which is shown as below:
      \arg        SEMx (x=0..31): semaphore x
    \param[out] none
    \retval     none
*/
void hals_hwsem_interrupt_disable(uint32_t semaphore)
{
    HWSEM_INTEN &= (uint32_t)(~((uint32_t)1U << semaphore));
}
