/*!
    \file    gd32h7xx_hal_gpio.c
    \brief   GPIO driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define GPIO_GET_OUTPUT_MODE(mode)       ((uint32_t)((mode >> 4) & 0x00000001U))
#define GPIO_AF_VALUE_MASK               ((uint32_t)0xFFFFFFF8U)
#define GPIO_PIN_VALUE_MASK              ((uint32_t)0xFFFF0000U)
#define GPIOA_PIN_VALUE_MASK             ((uint32_t)0xFFFF1800U)
#define GPIOJ_PIN_VALUE_MASK             ((uint32_t)0xFFFFF0FFU)
#define GPIOK_PIN_VALUE_MASK             ((uint32_t)0xFFFFFFF8U)

#define GPIO_PIN_NUMBER                   (16U)

/*!
    \brief      initialize gpio
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL(GPIO_PIN_ALL can be selected only
                                                    when gpio_periph is GPIOB,GPIOC,GPIOD,GPIOE,GPIOF,GPIOG,GPIOH)
    \param[in]  p_init: pointer of GPIO initialization structure
                  mode: GPIO_MODE_ANALOG, GPIO_MODE_INPUT, GPIO_MODE_OUTPUT_PP
                        GPIO_MODE_OUTPUT_OD, GPIO_MODE_AF_PP, GPIO_MODE_AF_OD
                        GPIO_MODE_IT_RISING, GPIO_MODE_IT_FALLING, GPIO_MODE_IT_BOTH
                        GPIO_MODE_EVT_RISING, GPIO_MODE_EVT_FALLING, GPIO_MODE_EVT_BOTH
                  pull: GPIO_PULL_NONE, GPIO_PULL_UP, GPIO_PULL_DOWN
                  ospeed: GPIO_OSPEED_12MHZ, GPIO_OSPEED_60MHZ, GPIO_OSPEED_85MHZ
                          GPIO_OSPEED_100_220MHZ
                  af: GPIO_AF_x(x=0...15)
                  filter_type:GPIO_IFTP_Synchronization,GPIO_IFTP_Filter_3
                              GPIO_IFTP_Filter_6,GPIO_IFTP_Asynchronization
                  filter:GPIO_IFL_CK_AHB,GPIO_IFL_CK_AHB_x(x=2,4,6,8...512)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_ALREADY_DONE, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_gpio_init(uint32_t gpio_periph, uint32_t pin, hal_gpio_init_struct *p_init)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t i         = 0U;
    uint32_t reg_temp  = 0U;
    uint32_t gpio_mode = 0U;
    uint32_t exti_mode = 0U;

  /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check pin_init address */
    if(NULL == p_init) {
        HAL_DEBUGE("pointer [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/
            PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

/*--------------------- GPIO Mode Configure ------------------------*/
    for(i = 0U; i < GPIO_PIN_NUMBER; i++) {
        if((1U << i) & pin) {
            /* check exti mode */
            gpio_mode = p_init->mode & 0xffU;
            exti_mode = (p_init->mode & 0xff00U) >> 8U;

            if(((EXTI_EVENT_TRIG_RISING == exti_mode) || (EXTI_EVENT_TRIG_FALLING == exti_mode) || \
            (EXTI_EVENT_TRIG_BOTH == exti_mode) || (EXTI_INTERRUPT_TRIG_RISING == exti_mode) || \
            (EXTI_INTERRUPT_TRIG_FALLING == exti_mode) || (EXTI_INTERRUPT_TRIG_BOTH == exti_mode))&&
            (HAL_ERR_NONE != hals_exti_register_gpio_line(gpio_periph, pin))) {
                error_code = HAL_ERR_ALREADY_DONE;
            } else {
                /* set GPIO_CTL register */
                reg_temp = GPIO_CTL(gpio_periph);
                reg_temp &= ~GPIO_MODE_MASK(i);
                reg_temp |= GPIO_MODE_SET(i, (CTL_CLTR(3) & gpio_mode));
                GPIO_CTL(gpio_periph) = reg_temp;

                /* set GPIO_OMODE register */
                if(GPIO_OTYPE_OD == GPIO_GET_OUTPUT_MODE(gpio_mode)) {
                    GPIO_OMODE(gpio_periph) |= (uint32_t)pin;
                } else {
                    GPIO_OMODE(gpio_periph) &= (uint32_t)(~pin);
                }

                /* set GPIO_OSPD register */
                reg_temp = GPIO_OSPD(gpio_periph);
                reg_temp &= ~GPIO_OSPEED_MASK(i);
                reg_temp |= GPIO_OSPEED_SET(i, p_init->ospeed);
                GPIO_OSPD(gpio_periph) = reg_temp;

                /* set GPIO_PUD register */
                reg_temp = GPIO_PUD(gpio_periph);
                reg_temp &= ~GPIO_PUPD_MASK(i);
                reg_temp |= GPIO_PUPD_SET(i, p_init->pull);
                GPIO_PUD(gpio_periph) = reg_temp;

                /* set GPIO_AFSELx register */
                if(((uint32_t)GPIO_MODE_AF_PP == gpio_mode) || ((uint32_t)GPIO_MODE_AF_OD == gpio_mode)) {
                    if(i < 8U) {
                        /* set pin0 to pin7 alternate function */
                        reg_temp = GPIO_AFSEL0(gpio_periph);
                        reg_temp &= ~GPIO_AFR_MASK(i);
                        reg_temp |= GPIO_AFR_SET(i, p_init->af);
                        GPIO_AFSEL0(gpio_periph) = reg_temp;
                    } else {
                        /* set pin8 to pin15 alternate function */
                        reg_temp = GPIO_AFSEL1(gpio_periph);
                        reg_temp &= ~GPIO_AFR_MASK(i - 8U);
                        reg_temp |= GPIO_AFR_SET(i - 8U, p_init->af);
                        GPIO_AFSEL1(gpio_periph) = reg_temp;
                    }
                }

                /* set GPIO_IFTP register */
                if((uint32_t)GPIO_MODE_INPUT == gpio_mode) {
                    reg_temp = GPIO_IFTP(gpio_periph);
                    reg_temp &= ~GPIO_IFTP_MASK(i);
                    reg_temp |= GPIO_IFTP_SET(i, p_init->filter_type);
                    GPIO_IFTP(gpio_periph) = reg_temp;
                } else {
                    /* do nothing */
                }

                /* set GPIO_IFL register */
                if((uint32_t)GPIO_MODE_INPUT == gpio_mode) {
                    if(i < 8U) {
                        reg_temp = GPIO_IFL(gpio_periph);
                        reg_temp &= ~GPIO_IFL_FLPRD0;
                        reg_temp |= (uint32_t)p_init->filter;
                        GPIO_IFL(gpio_periph) = reg_temp;
                    } else {
                        reg_temp = GPIO_IFL(gpio_periph);
                        reg_temp &= ~GPIO_IFL_FLPRD1;
                        reg_temp |= ((uint32_t)p_init->filter << 8);
                        GPIO_IFL(gpio_periph) = reg_temp;
                    }
                } else {
                    /* do nothing */
                }

                if((EXTI_EVENT_TRIG_RISING == exti_mode) || (EXTI_EVENT_TRIG_FALLING == exti_mode) || \
                (EXTI_EVENT_TRIG_BOTH == exti_mode) || (EXTI_INTERRUPT_TRIG_RISING == exti_mode) || \
                (EXTI_INTERRUPT_TRIG_FALLING == exti_mode) || (EXTI_INTERRUPT_TRIG_BOTH == exti_mode)) {
                    hals_exti_config_gpio_line(gpio_periph, pin, exti_mode);
                } else {
                    /* do nothing */
                }
            }
        } else {
            /* do nothing */
        }
    }
    return error_code;
}

/*!
    \brief      deinitialize gpio
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL(GPIO_PIN_ALL can be selected only
                                                    when gpio_periph is GPIOB,GPIOC,GPIOD,GPIOE,GPIOF,GPIOG,GPIOH)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_gpio_deinit(uint32_t gpio_periph, uint32_t pin)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t i         = 0U;

  /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

/*------------------------------- GPIO Mode Configure -------------------------------*/
    if(GPIO_PIN_ALL == pin) {
        switch(gpio_periph) {
        /* reset GPIOA */
        case GPIOA:
            hal_rcu_periph_reset_enable(RCU_GPIOARST);
            hal_rcu_periph_reset_disable(RCU_GPIOARST);
            break;
        /* reset GPIOB */
        case GPIOB:
            hal_rcu_periph_reset_enable(RCU_GPIOBRST);
            hal_rcu_periph_reset_disable(RCU_GPIOBRST);
            break;
        /* reset GPIOC */
        case GPIOC:
            hal_rcu_periph_reset_enable(RCU_GPIOCRST);
            hal_rcu_periph_reset_disable(RCU_GPIOCRST);
            break;
        /* reset GPIOD */
        case GPIOD:
            hal_rcu_periph_reset_enable(RCU_GPIODRST);
            hal_rcu_periph_reset_disable(RCU_GPIODRST);
            break;
        /* reset GPIOE */
        case GPIOE:
            hal_rcu_periph_reset_enable(RCU_GPIOERST);
            hal_rcu_periph_reset_disable(RCU_GPIOERST);
            break;
        /* reset GPIOF */
        case GPIOF:
            hal_rcu_periph_reset_enable(RCU_GPIOFRST);
            hal_rcu_periph_reset_disable(RCU_GPIOFRST);
            break;
        /* reset GPIOG */
        case GPIOG:
            hal_rcu_periph_reset_enable(RCU_GPIOGRST);
            hal_rcu_periph_reset_disable(RCU_GPIOGRST);
            break;
        /* reset GPIOH */
        case GPIOH:
            hal_rcu_periph_reset_enable(RCU_GPIOHRST);
            hal_rcu_periph_reset_disable(RCU_GPIOHRST);
            break;
        /* reset GPIOJ */
        case GPIOJ:
            hal_rcu_periph_reset_enable(RCU_GPIOJRST);
            hal_rcu_periph_reset_disable(RCU_GPIOJRST);
            break;
        /* reset GPIOK */
        case GPIOK:
            hal_rcu_periph_reset_enable(RCU_GPIOKRST);
            hal_rcu_periph_reset_disable(RCU_GPIOKRST);
            break;
        default:
            HAL_DEBUGE("parameter [gpio_periph] value is invalid");
            error_code = HAL_ERR_VAL;
            break;
        }
    } else {
        for(i = 0U;i < GPIO_PIN_NUMBER; i++) {
            if((1U << i) & pin) {

            /* clear the specified pin mode bits */
            GPIO_CTL(gpio_periph) &= ~GPIO_MODE_MASK(i);
            /* clear the specified pin pupd bits */
            GPIO_PUD(gpio_periph) &= ~GPIO_PUPD_MASK(i);
            /* clear the specified pin output speed bits */
            GPIO_OSPD(gpio_periph) &= ~GPIO_OSPEED_MASK(i);

            /* clear the specified pin alternate function bits */
            if(i < 8U) {
                GPIO_AFSEL0(gpio_periph) &= ~GPIO_AFR_MASK(i);
            } else {
                GPIO_AFSEL1(gpio_periph) &= ~GPIO_AFR_MASK(i - 8U);
            }

            /* clear the specified pin input filter type bits */
            GPIO_IFTP(gpio_periph) &= ~GPIO_IFTP_MASK(i);
            /* clear the specified pin input filter type bits */
            GPIO_IFL(gpio_periph)  &= ~GPIO_IFL_MASK(i);
            } else {
                /* do nothing */
            }
        }
    }

    return error_code;
}

/*!
    \brief      initialize the gpio structure with the default values
    \param[in]  none
    \param[out] p_init: pointer of GPIO init parameter struct
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_gpio_struct_init(hal_gpio_init_struct *p_init)
{
    int32_t error_code = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_init) {
        HAL_DEBUGE("pointer [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set the GPIO initialization structure with the default values */
    p_init->mode        = GPIO_MODE_INPUT;
    p_init->pull        = GPIO_PULL_NONE;
    p_init->ospeed      = GPIO_OSPEED_12MHZ;
    p_init->af          = GPIO_AF_0;
    p_init->filter      = GPIO_IFL_CK_AHB;
    p_init->filter_type = GPIO_IFTP_Synchronization;

    return error_code;
}

/*!
    \brief      set GPIO alternate function
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL(GPIO_PIN_ALL can be selected only
                                                    when gpio_periph is GPIOB,GPIOC,GPIOD,GPIOE,GPIOF,GPIOG,GPIOH)
    \param[in]  alt_func_num: GPIO pin af function, please refer to specific device datasheet
                only one parameter can be selected which is shown as below:
      \arg        GPIO_AF_0: SYSTEM, TIMER40, TIMER41, TIMER42, TIMER43, TIMER44
      \arg        GPIO_AF_1: TIMER0, TIMER1, TIMER15, TIMER16, EXMC, SAI1, SAI2
      \arg        GPIO_AF_2: TIMER2, TIMER3, TIMER4, TIMER7, TIMER14, TLI, CAN2, SAI0, EXMC
      \arg        GPIO_AF_3: TIMER7, TIMER9, EDOUT, EXMC, TLI, HPDF, OSPIM
      \arg        GPIO_AF_4: TIMER14, TIMER30, TIMER31, I2C0, I2C1, I2C2, I2C3, USART0, HPDF, OSPIM, TLI
      \arg        GPIO_AF_5: SPI0, SPI1, SPI2, SPI3, SPI4, SPI5, CAN2
      \arg        GPIO_AF_6: UART3, SPI2, I2C3, HPDF, SAI0, ETH1, EDOUT, OSPIM
      \arg        GPIO_AF_7: USART0, USART1, USART2, USART5, UART6, TIMER40, TIMER41, TIMER42, TIMER43,
                             SPI1, SPI2, SPI5, SDIO0, USBHS1
      \arg        GPIO_AF_8: UART3, UART4, UART7, SPI5, SDIO0, RSPDIF, TIMER44, USBHS1, SAI1, SAI2
      \arg        GPIO_AF_9: SDIO1, TRGSEL, CAN0, CAN1, TLI, OPSIM, EXMC, RSPDIF, SAI2
      \arg        GPIO_AF_10: SAI1, SAI2, OTG0, SDIO1, CMP, USBHS0, OPSIM, EXMC
      \arg        GPIO_AF_11: ETH0, MDIO, CMP, UART6, EXMC, HPDF, I2C3, TLI, SDIO1, OPSIM
      \arg        GPIO_AF_12: TIMER0, MDIOS, SDIO0, EXMC, OPSIM, CMP, TLI, USBHS1
      \arg        GPIO_AF_13: TRGSEL, DCI, COMP0, CMP, TIMER22
      \arg        GPIO_AF_14: TLI, UART4, TIMER23
      \arg        GPIO_AF_15: EVENTOUT
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_gpio_af_set(uint32_t gpio_periph, uint32_t pin, hal_gpio_af_enum alt_func_num)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t i         = 0U;
    uint32_t reg_afrl  = 0U;
    uint32_t reg_afrh  = 0U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check alt_func_num value */
    if((GPIO_AF_0 != alt_func_num) && (GPIO_AF_1 != alt_func_num) && (GPIO_AF_2 != alt_func_num) && \
       (GPIO_AF_3 != alt_func_num) && ((GPIO_AF_4 != alt_func_num)) && (GPIO_AF_5 != alt_func_num) && \
       (GPIO_AF_6 != alt_func_num) && (GPIO_AF_7 != alt_func_num) && (GPIO_AF_8 != alt_func_num) && \
       (GPIO_AF_9 != alt_func_num) && ((GPIO_AF_10 != alt_func_num)) && (GPIO_AF_11 != alt_func_num) && \
       (GPIO_AF_12 != alt_func_num) && (GPIO_AF_13 != alt_func_num) && (GPIO_AF_14 != alt_func_num) && \
       (GPIO_AF_15 != alt_func_num) ) {
        HAL_DEBUGE("parameter [alt_func_num] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    reg_afrl = GPIO_AFSEL0(gpio_periph);
    reg_afrh = GPIO_AFSEL1(gpio_periph);

    for(i = 0U; i < 8U; i++) {
        if((1U << i) & pin) {
            /* Configure the specified pin alternate function bits */
            reg_afrl &= ~GPIO_AFR_MASK(i);
            reg_afrl |= GPIO_AFR_SET(i, alt_func_num);
        } else {
            /* do nothing */
        }
    }
    for(i = 8U; i < 16U; i++) {
        if((1U << i) & pin) {
            /* clear the specified pin alternate function bits */
            reg_afrh &= ~GPIO_AFR_MASK(i - 8U);
            reg_afrh |= GPIO_AFR_SET(i - 8U, alt_func_num);
        } else {
            /* do nothing */
        }

        GPIO_AFSEL0(gpio_periph) = reg_afrl;
        GPIO_AFSEL1(gpio_periph) = reg_afrh;
    }

    return error_code;
}

/*!
    \brief      set GPIO mode
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL(GPIO_PIN_ALL can be selected only
                                                    when gpio_periph is GPIOB,GPIOC,GPIOD,GPIOE,GPIOF,GPIOG,GPIOH)
    \param[in]  mode: GPIO pin mode
                only one parameter can be selected which is shown as below:
      \arg        GPIO_MODE_INPUT: input mode
      \arg        GPIO_MODE_OUTPUT: output mode
      \arg        GPIO_MODE_AF: alternate function mode
      \arg        GPIO_MODE_ANALOG: analog mode
    \param[in]  pull_up_down: GPIO pin with pull-up or pull-down resistor
                only one parameter can be selected which is shown as below:
      \arg        GPIO_PULL_NONE: floating mode, no pull-up and pull-down resistors
      \arg        GPIO_PULL_UP: with pull-up resistor
      \arg        GPIO_PULL_DOWN:with pull-down resistor
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_gpio_mode_set(uint32_t gpio_periph, uint32_t pin, hal_gpio_mode_enum mode, hal_gpio_pull_enum pull_up_down)
{
    int32_t error_code     = HAL_ERR_NONE;
    uint32_t i             = 0U;
    uint32_t reg_ctl_temp  = 0U;
    uint32_t reg_pupd_temp = 0U;

  /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check mode value */
    if((GPIO_MODE_INPUT != mode) && (GPIO_MODE_OUTPUT != (uint32_t)mode) && \
        (GPIO_MODE_AF != (uint32_t)mode)   && (GPIO_MODE_ANALOG != mode)) {
        HAL_DEBUGE("parameter [mode] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check pull_up_down value */
    if((GPIO_PULL_NONE != pull_up_down) && (GPIO_PULL_UP != pull_up_down) && \
       (GPIO_PULL_DOWN != pull_up_down)) {
        HAL_DEBUGE("parameter [pull_up_down] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/
            PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    reg_ctl_temp   = GPIO_CTL(gpio_periph);
    reg_pupd_temp  = GPIO_PUD(gpio_periph);

    for(i = 0U;i < 16U;i++) {
        if((1U << i) & pin) {
            /* Configure the specified pin mode bits */
            reg_ctl_temp &= ~GPIO_MODE_MASK(i);
            reg_ctl_temp |= GPIO_MODE_SET(i, mode);

            /* Configure the specified pin pupd bits */
            reg_pupd_temp &= ~GPIO_PUPD_MASK(i);
            reg_pupd_temp |= GPIO_PUPD_SET(i, pull_up_down);
        } else {
            /* do nothing */
        }
    }

    GPIO_CTL(gpio_periph) = reg_ctl_temp;
    GPIO_PUD(gpio_periph) = reg_pupd_temp;

    return error_code;
}

/*!
    \brief      set GPIO output type and speed
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL(GPIO_PIN_ALL can be selected only
                                                    when gpio_periph is GPIOB,GPIOC,GPIOD,GPIOE,GPIOF,GPIOG,GPIOH)
    \param[in]  otype: GPIO pin output mode
                only one parameter can be selected which is shown as below:
      \arg        GPIO_OTYPE_PP: push pull mode
      \arg        GPIO_OTYPE_OD: open drain mode
    \param[in]  speed: GPIO pin output max speed
                only one parameter can be selected which is shown as below:
      \arg        GPIO_OSPEED_12MHZ: output max speed 12MHz
      \arg        GPIO_OSPEED_60MHZ: output max speed 60MHz
      \arg        GPIO_OSPEED_85MHZ: output max speed 85MHz
      \arg        GPIO_OSPEED_100_220MHZ: output max speed 100/220MHz
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_gpio_output_options_set(uint32_t gpio_periph, uint32_t pin, uint8_t otype, hal_gpio_ospeed_enum speed)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t i         = 0U;
    uint32_t ospeed    = 0U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check otype value */
    if((GPIO_OTYPE_PP != otype) && (GPIO_OTYPE_OD != otype)) {
        HAL_DEBUGE("parameter [otype] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check speed value */
    if((GPIO_OSPEED_12MHZ != speed) && (GPIO_OSPEED_60MHZ != speed) && \
       (GPIO_OSPEED_85MHZ != speed) && (GPIO_OSPEED_100_220MHZ != speed)) {
        HAL_DEBUGE("parameter [speed] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Configure the specified pin output type bits */
    if(GPIO_OTYPE_OD == otype) {
        GPIO_OMODE(gpio_periph) |= (uint32_t)pin;
    } else {
        GPIO_OMODE(gpio_periph) &= (uint32_t)(~pin);
    }

    /* get the specified pin output speed bits value */
    ospeed = GPIO_OSPD(gpio_periph);

    for(i = 0U;i < 16U;i++) {
        if((1U << i) & pin) {
            /* Configure the specified pin output speed bits */
            ospeed &= ~GPIO_OSPEED_MASK(i);
            ospeed |= GPIO_OSPEED_SET(i, speed);
        } else {
            /* do nothing */
        }
    }
    GPIO_OSPD(gpio_periph) = ospeed;

    return error_code;
}

/*!
    \brief      set GPIO input filter
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[in]  filter: GPIO pin input sample period,
                the argument could be selected from enumeration <hal_gpio_filter_value_enum>
    \param[in]  filter_type: GPIO pin input filtering type
                only one parameter can be selected which is shown as below:
      \arg        GPIO_IFTP_Synchronization: input filter type synchronization
      \arg        GPIO_IFTP_Filter_3: input filter type filter 3 samples
      \arg        GPIO_IFTP_Filter_6: input filter type filter 6 samples
      \arg        GPIO_IFTP_Asynchronization: input filter type asynchronous
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_gpio_input_filter_set(uint32_t gpio_periph, uint32_t pin, uint8_t filter, hal_gpio_filter_type_enum filter_type)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t i         = 0U;
    uint32_t isperiod  = 0U;
    uint32_t iftp      = 0U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check filter_type value */
    if((GPIO_IFTP_Synchronization != filter_type) && (GPIO_IFTP_Filter_3 != filter_type) && \
       (GPIO_IFTP_Filter_6 != filter_type) && (GPIO_IFTP_Asynchronization != filter_type)) {
        HAL_DEBUGE("parameter [filter] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check filter value */
    if((GPIO_IFL_CK_AHB >= filter) && (GPIO_IFL_CK_AHB_510 <= filter)) {
        HAL_DEBUGE("parameter [filter] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Configure GPIO pin input sample period */
    isperiod = GPIO_IFL(gpio_periph);
    if(pin & 0x000000FFU) {
        isperiod &= ~GPIO_IFL_FLPRD0;
        isperiod |= (uint32_t)filter;
    } else {
        /* do nothing */
    }

    if(pin & 0x0000FF00U) {
        isperiod &= ~GPIO_IFL_FLPRD1;
        isperiod |= ((uint32_t)filter << 8U);
    } else {
        /* do nothing */
    }

    GPIO_IFL(gpio_periph) = isperiod;

    /* Configure GPIO pin input filtering type */
    iftp = GPIO_IFTP(gpio_periph);
    for(i = 0U;i < 16U;i++) {
        if((1U << i) & pin) {
            /* clear the specified pin input filtering type */
            iftp &= ~GPIO_IFTYPE_MASK(i);
            /* set the specified pin input filtering type */
            iftp |= GPIO_IFTYPE_SET(i, filter_type);
        } else {
            /* do nothing */
        }
    }

    GPIO_IFTP(gpio_periph) = iftp;

    return error_code;
}

/*!
    \brief      set GPIO pin bit
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void hal_gpio_bit_set(uint32_t gpio_periph, uint32_t pin)
{
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return ;
    }

    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    GPIO_BOP(gpio_periph) = (uint32_t)pin;
}

/*!
    \brief      reset GPIO pin bit
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,F)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void hal_gpio_bit_reset(uint32_t gpio_periph, uint32_t pin)
{
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return ;
    }
    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    GPIO_BC(gpio_periph) = (uint32_t)pin;
}

/*!
    \brief      write data to the specified GPIO pin
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
                GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL(GPIO_PIN_ALL can be selected only
                                                    when gpio_periph is GPIOB,GPIOC,GPIOD,GPIOE,GPIOF,GPIOG,GPIOH)
    \param[in]  bit_value: SET or RESET
      \arg        RESET: clear the port pin
      \arg        SET: set the port pin
    \param[out] none
    \retval     none
*/
void hal_gpio_bit_write(uint32_t gpio_periph, uint32_t pin, bit_status bit_value)
{
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return ;
    }
    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(RESET != bit_value) {
        GPIO_BOP(gpio_periph) |= (uint32_t)pin;
    } else {
        GPIO_BC(gpio_periph) |= (uint32_t)pin;
    }
}

/*!
    \brief      write data to the specified GPIO port
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  data: specify the value to be written to the port output control register
    \param[out] none
    \retval     none
*/
void hal_gpio_port_write(uint32_t gpio_periph, uint16_t data)
{
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return ;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    GPIO_OCTL(gpio_periph) = (uint32_t)data;
}

/*!
    \brief      get GPIO pin input status
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                only one parameter can be selected which is shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL(GPIO_PIN_ALL can be selected only
                                                    when gpio_periph is GPIOB,GPIOC,GPIOD,GPIOE,GPIOF,GPIOG,GPIOH)
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hal_gpio_input_bit_get(uint32_t gpio_periph, uint32_t pin)
{
    FlagStatus status = RESET;

    if((uint32_t)RESET != (GPIO_ISTAT(gpio_periph) & (pin))) {
        status = SET;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      get GPIO port input status
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[out] none
    \retval     uint16_t: 0 - 0xFFFF
*/
uint16_t hal_gpio_input_port_get(uint32_t gpio_periph)
{
    return (uint16_t)GPIO_ISTAT(gpio_periph);
}

/*!
    \brief      get GPIO pin output status
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                only one parameter can be selected which is shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hal_gpio_output_bit_get(uint32_t gpio_periph, uint32_t pin)
{
    FlagStatus status = RESET;

    if((uint32_t)RESET != (GPIO_OCTL(gpio_periph) & (pin))) {
        status = SET;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      get GPIO port output status
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[out] none
    \retval     uint16_t: 0 - 0xFFFF
*/
uint16_t hal_gpio_output_port_get(uint32_t gpio_periph)
{
    return (uint16_t)GPIO_OCTL(gpio_periph);
}

/*!
    \brief      lock GPIO pin bit
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL(GPIO_PIN_ALL can be selected only
                                                    when gpio_periph is GPIOB,GPIOC,GPIOD,GPIOE,GPIOF,GPIOG,GPIOH)
    \param[out] none
    \retval     none
*/
void hal_gpio_pin_lock(uint32_t gpio_periph, uint32_t pin)
{
    uint32_t lock = 0x00010000U;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return ;
    }

    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    lock |= pin;

    /* lock key writing sequence: write 1->write 0->write 1->read 0->read 1 */
    GPIO_LOCK(gpio_periph) = (uint32_t)lock;
    GPIO_LOCK(gpio_periph) = (uint32_t)pin;
    GPIO_LOCK(gpio_periph) = (uint32_t)lock;

    lock = GPIO_LOCK(gpio_periph);
    lock = GPIO_LOCK(gpio_periph);
}

/*!
    \brief      toggle GPIO pin status
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL(GPIO_PIN_ALL can be selected only
                                                    when gpio_periph is GPIOB,GPIOC,GPIOD,GPIOE,GPIOF,GPIOG,GPIOH)
    \param[out] none
    \retval     none
*/
void hal_gpio_bit_toggle(uint32_t gpio_periph, uint32_t pin)
{    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return ;
    }

    /* check if pin is PB0 ~ PB15/PC0 ~ PC15/PD0 ~ PD15/PE0 ~ PE15/PF0 ~ PF15/PG0 ~ PG15/PH0 ~ PH15 or not*/
    if((GPIOB == gpio_periph) || (GPIOC == gpio_periph) || (GPIOD == gpio_periph) || \
       (GPIOE == gpio_periph) || (GPIOF == gpio_periph) || (GPIOG == gpio_periph) || \
       (GPIOH == gpio_periph)) {
        if(0U != (pin & GPIO_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PA0 ~ PA10/PA13 ~ PA15 or not*/
    if(GPIOA == gpio_periph) {
        if(0U != (pin & GPIOA_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PJ8 ~ PJ11 or not*/
    if(GPIOJ == gpio_periph) {
        if(0U != (pin & GPIOJ_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
    /* check if pin is PK0 ~ PK2 or not*/
    if(GPIOK == gpio_periph) {
        if(0U != (pin & GPIOK_PIN_VALUE_MASK)) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return ;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    GPIO_TG(gpio_periph) = (uint32_t)pin;
}

/*!
    \brief      toggle GPIO port status
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
                only one parameter can be selected which is shown as below:
      \arg        GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[out] none
    \retval     none
*/
void hal_gpio_port_toggle(uint32_t gpio_periph)
{
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && \
       (GPIOD != gpio_periph) && (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && \
       (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && (GPIOJ != gpio_periph) && \
       (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return ;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    GPIO_TG(gpio_periph) = 0x0000FFFFU;
}
