/*!
    \file    gd32h7xx_hal_exti.c
    \brief   EXTI driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define EXTI_GPIO_MAX_NUM               ((uint32_t)16U)
#define EXTI_RISING                     ((uint32_t)0x00000001U)
#define EXTI_FALLING                    ((uint32_t)0x00000002U)
#define EXTI_EVENT                      ((uint32_t)0x00000004U)
#define EXTI_INTERRUPT                  ((uint32_t)0x00000008U)

#define EXTI_GPIO_PORT_GET(x)           ((x >> 4U) & (uint8_t)0x0F)
#define EXTI_GPIO_PIN_GET(x)            (x & (uint8_t)0x0F)
#define EXTI_REG_RESET_VALUE            ((uint32_t)0x00000000U)

#define GPIO_PIN_VALUE_MASK             ((uint32_t)0xFFFF0000U)
#define GPIOJ_PIN_VALUE_MASK            ((uint32_t)0xFFFFF0FFU)
#define GPIOK_PIN_VALUE_MASK            ((uint32_t)0xFFFFFFF8U)
#define EXTI_GPIO_SOURCE_MASK           ((uint32_t)0x00003C00U)

volatile uint32_t _exti_gpio_used = 0U;

/* EXTI IRQ HANDLE callback */
static hal_gpio_irq_handle_cb s_gpio_irq_handle = NULL;
static volatile uint8_t s_exti_gpio_info[EXTI_GPIO_MAX_NUM];

static int32_t _exti_type_config(uint32_t exti_group, uint32_t pin, hal_exti_type_enum exti_type);
static int32_t _exti_type_get(uint32_t exti_group, uint32_t pin, hal_exti_type_enum *p_exti_type);
static void _exti_gpio_info_set(uint32_t gpio_periph, uint32_t pin);

/*!
    \brief      deinitialize the EXTI gpio
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin, one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
void hal_exti_gpio_deinit(uint32_t gpio_periph, uint32_t pin)
{
    hal_gpio_deinit(gpio_periph, pin);
    _exti_gpio_used &= ~pin;

    /* reset the EXTI gpio pin */
    EXTI_INTEN(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET)) &= (uint32_t)(~pin);
    EXTI_EVEN(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET))  &= (uint32_t)(~pin);
    EXTI_RTEN(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET))  &= (uint32_t)(~pin);
    EXTI_FTEN(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET))  &= (uint32_t)(~pin);
    EXTI_SWIEV(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET)) &= (uint32_t)(~pin);
}

/*!
    \brief      deinitialize the EXTI internal line
    \param[in]  line: the argument could be selected from enumeration <hal_exti_internal_line_enum>
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exti_internal_deinit(hal_exti_internal_line_enum line)
{
     uint32_t line_mask = 0U;
    /* check parameter value */
#if (1U == HAL_PARAMETER_CHECK)
    if((EXTI_LINE_16_LVD != line) && (EXTI_LINE_17_RTC_ALARM != line) && \
       (EXTI_LINE_18_RTC_TAMPER_TIMESTAMP != line) && (EXTI_LINE_19_RTC_WAKEUP != line) && \
       (EXTI_LINE_20_CMP0_OUTPUT != line) && (EXTI_LINE_21_CMP1_OUTPUT != line) && \
       (EXTI_LINE_22_ENET1_WAKEUP != line) && (EXTI_LINE_23_ENET0_WAKEUP != line) && \
       (EXTI_LINE_24_CAN0_WAKEUP != line) && (EXTI_LINE_25_CAN1_WAKEUP != line) && \
       (EXTI_LINE_26_CAN2_WAKEUP != line) && (EXTI_LINE_27_USART0_WAKEUP != line) && \
       (EXTI_LINE_28_USART1_WAKEUP != line) && (EXTI_LINE_29_USART2_WAKEUP != line) && \
       (EXTI_LINE_30_USART5_WAKEUP != line) && (EXTI_LINE_31_USBHS0_WAKEUP != line) && \
       (EXTI_LINE_32_USBHS1_WAKEUP != line) && (EXTI_LINE_33_I2C0_WAKEUP != line) && \
       (EXTI_LINE_34_I2C1_WAKEUP != line) && (EXTI_LINE_35_I2C2_WAKEUP != line) && \
       (EXTI_LINE_36_I2C3_WAKEUP != line) && (EXTI_LINE_37_LPDTS_WAKEUP != line)) {
        HAL_DEBUGE("parameter [line] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    line_mask = 1U << line;
    if(0U != (line_mask & (uint32_t)0xFFFF0000U)) {
        EXTI_INTEN(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET)) &= (uint32_t)(~line_mask);
        EXTI_EVEN(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET))  &= (uint32_t)(~line_mask);
        EXTI_RTEN(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET))  &= (uint32_t)(~line_mask);
        EXTI_FTEN(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET))  &= (uint32_t)(~line_mask);
        EXTI_SWIEV(EXTI_REG_VAL_GROUP(EXTI_GROUP0_OFFSET)) &= (uint32_t)(~line_mask);
    } else {
        EXTI_INTEN(EXTI_REG_VAL_GROUP(EXTI_GROUP1_OFFSET)) &= (uint32_t)(~line_mask);
        EXTI_EVEN(EXTI_REG_VAL_GROUP(EXTI_GROUP1_OFFSET))  &= (uint32_t)(~line_mask);
        EXTI_RTEN(EXTI_REG_VAL_GROUP(EXTI_GROUP1_OFFSET))  &= (uint32_t)(~line_mask);
        EXTI_FTEN(EXTI_REG_VAL_GROUP(EXTI_GROUP1_OFFSET))  &= (uint32_t)(~line_mask);
        EXTI_SWIEV(EXTI_REG_VAL_GROUP(EXTI_GROUP1_OFFSET)) &= (uint32_t)(~line_mask);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set the configuration of EXTI line
    \param[in]  line: EXTI Line
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x(x=0 ~ 37)
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  exti_type: the argument could be selected from enumeration <hal_exti_type_enum>
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ALREADY_DONE, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exti_line_set_config(hal_exti_line_enum line, uint32_t gpio_periph, hal_exti_type_enum exti_type)
{
    uint32_t exti_group     = 0U;
    uint32_t exti_line_mask = 0U;
    int32_t  error_code     = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && (GPIOD != gpio_periph) && \
       (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && \
       (GPIOJ != gpio_periph) && (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check if pin is 0 ~ 15 GPIOx(x = A,B,C,D,E,F,G,H) */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && (GPIOD != gpio_periph) && \
       (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && (GPIOG != gpio_periph) && (GPIOH != gpio_periph)) {
        if((0U != (((uint32_t)line & 0x1fU) & GPIO_PIN_VALUE_MASK))) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PJx(8..11) or not*/
    if((GPIOJ == gpio_periph)) {
        if((0U != (((uint32_t)line & 0x1fU) & GPIOJ_PIN_VALUE_MASK))) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
    /* check if pin is PKx(0..2) or not*/
    if((GPIOK == gpio_periph)) {
        if((0U != (((uint32_t)line & 0x1fU) & GPIOK_PIN_VALUE_MASK))) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    exti_group = ((uint32_t)line >> 8U);
    exti_line_mask = EXTI_BIT_POS(line);

    /* Configure gpio exti line */
    if((15U >= ((uint16_t)line & 0x1FU)) && (EXTI_GROUP0_OFFSET == exti_group)) {
        /* check the pin is use or not */
        if(HAL_ERR_NONE != hals_exti_register_gpio_line(gpio_periph, exti_line_mask)) {
            error_code = HAL_ERR_ALREADY_DONE;
        } else {
            hals_exti_config_gpio_line(gpio_periph, exti_line_mask, (uint32_t)exti_type);
        }
    } else {
        _exti_type_config(exti_group, exti_line_mask, exti_type);
    }

    return error_code;
}

/*!
    \brief      get the configuration of EXTI line
    \param[in]  line: EXTI Line
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x(x=0 ~ 37)
    \param[out]  p_gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[out]  p_exti_type: the argument from enumeration <hal_exti_type_enum>
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exit_line_get_config(hal_exti_line_enum line, uint32_t *p_gpio_periph, hal_exti_type_enum *p_exti_type)
{
    uint32_t exti_group      = 0U;
    uint32_t exti_line_mask  = 0U;
    uint32_t temp            = 0U;
    int32_t error_code       = HAL_ERR_NONE;
    uint16_t line_mask       = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == p_gpio_periph) || (NULL == p_exti_type)) {
        HAL_DEBUGE("parameter [p_gpio_periph] or [p_exti_type] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    exti_group = (uint32_t)line >> 8U;
    exti_line_mask = EXTI_BIT_POS(line);
    line_mask = (uint16_t)line & 0x1FU;

    /* get gpio exti line configure */
    if((15U >= line_mask) && (EXTI_GROUP0_OFFSET == exti_group)) {
        /* get EXTI GPIO port */
        temp = EXTI_GPIO_PORT_GET((uint32_t)s_exti_gpio_info[line_mask]);
        *p_gpio_periph = GPIO_BASE + (temp << 10U);
    } else {
       /* do nothing */
    }
    _exti_type_get(exti_group, exti_line_mask, p_exti_type);

    return error_code;
}

/*!
    \brief      clear the configuration of EXTI line
    \param[in]  line: EXTI Line
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x(x=0 ~ 37)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exit_line_clear_config(hal_exti_line_enum line)
{
    uint32_t exti_group      = 0U;
    uint32_t exti_line_mask  = 0U;
    int32_t  error_code      = HAL_ERR_NONE;

    exti_group = (uint32_t)line >> 8U;
    exti_line_mask = EXTI_BIT_POS(line);

    /* clear gpio exti line configure */
    if((15U >= ((uint16_t)line & 0x1fU)) && (EXTI_GROUP0_OFFSET == exti_group)) {
        /* clear exti info */
        _exti_gpio_used &= ~exti_line_mask;
    }

    /* reset the EXTI line */
    EXTI_INTEN(EXTI_REG_VAL_GROUP(exti_group)) &= (uint32_t)(~exti_line_mask);
    EXTI_EVEN(EXTI_REG_VAL_GROUP(exti_group))  &= (uint32_t)(~exti_line_mask);
    EXTI_RTEN(EXTI_REG_VAL_GROUP(exti_group))  &= (uint32_t)(~exti_line_mask);
    EXTI_FTEN(EXTI_REG_VAL_GROUP(exti_group))  &= (uint32_t)(~exti_line_mask);
    EXTI_SWIEV(EXTI_REG_VAL_GROUP(exti_group)) &= (uint32_t)(~exti_line_mask);

    return error_code;
}

/*!
    \brief      initialize the configuration of EXTI internal
    \param[in]  line: the argument could be selected from enumeration <hal_exti_internal_line_enum>
    \param[in]  exti_type: the argument could be selected from enumeration <hal_exti_type_enum>
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exti_internal_init(hal_exti_internal_line_enum line, hal_exti_type_enum exti_type)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t line_mask = 0U;
    /* check parameter value */
#if (1U == HAL_PARAMETER_CHECK)
    if((EXTI_LINE_16_LVD != line) && (EXTI_LINE_17_RTC_ALARM != line) && \
       (EXTI_LINE_18_RTC_TAMPER_TIMESTAMP != line) && (EXTI_LINE_19_RTC_WAKEUP != line) && \
       (EXTI_LINE_20_CMP0_OUTPUT != line) && (EXTI_LINE_21_CMP1_OUTPUT != line) && \
       (EXTI_LINE_22_ENET1_WAKEUP != line) && (EXTI_LINE_23_ENET0_WAKEUP != line) && \
       (EXTI_LINE_24_CAN0_WAKEUP != line) && (EXTI_LINE_25_CAN1_WAKEUP != line) && \
       (EXTI_LINE_26_CAN2_WAKEUP != line) && (EXTI_LINE_27_USART0_WAKEUP != line) && \
       (EXTI_LINE_28_USART1_WAKEUP != line) && (EXTI_LINE_29_USART2_WAKEUP != line) && \
       (EXTI_LINE_30_USART5_WAKEUP != line) && (EXTI_LINE_31_USBHS0_WAKEUP != line) && \
       (EXTI_LINE_32_USBHS1_WAKEUP != line) && (EXTI_LINE_33_I2C0_WAKEUP != line) && \
       (EXTI_LINE_34_I2C1_WAKEUP != line) && (EXTI_LINE_35_I2C2_WAKEUP != line) && \
       (EXTI_LINE_36_I2C3_WAKEUP != line) && (EXTI_LINE_37_LPDTS_WAKEUP != line)) {
        HAL_DEBUGE("parameter [line] value is invalid");
        return HAL_ERR_VAL;
    }
    if((EXTI_EVENT_TRIG_RISING != exti_type) && (EXTI_EVENT_TRIG_FALLING != exti_type) && \
       (EXTI_EVENT_TRIG_BOTH != exti_type) && (EXTI_INTERRUPT_TRIG_RISING != exti_type) && \
       (EXTI_INTERRUPT_TRIG_FALLING != exti_type) && (EXTI_INTERRUPT_TRIG_BOTH != exti_type)) {
        HAL_DEBUGE("parameter [exti_type] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    line_mask = 1U << line;

    if(0U != (line_mask & (uint32_t)0xFFFF0000U)) {
        _exti_type_config(EXTI_GROUP0_OFFSET, line_mask, exti_type);
    } else {
        _exti_type_config(EXTI_GROUP1_OFFSET, line_mask, exti_type);
    }

    return error_code;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  irq_handler: configuration the EXTI callback function
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exti_gpio_irq_handle_set(hal_gpio_irq_handle_cb irq_handler)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check parameter value */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == irq_handler)) {
        HAL_DEBUGE("parameter [irq_handler] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    s_gpio_irq_handle = irq_handler;

    return error_code;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_exti_gpio_irq_handle_all_reset(void)
{
    s_gpio_irq_handle = NULL;
}

/*!
    \brief      EXTI_GPIO interrupt handler content function,which is merely used in EXTI_GPIO_handler
    \param[in]  index: indicate this function will be called by which interrupt handler entry,
                       the argument could be selected from enumeration <hal_exti_irq_index_enum>
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exti_gpio_irq(hal_exti_irq_index_enum index)
{
    uint32_t i          = 0U;
    uint32_t gpio_port  = 0U;
    uint32_t gpio_pin   = 0U;
    uint32_t temp       = 0U;
    uint8_t start_pin   = 0U;
    uint8_t end_pin     = 0U;
    uint8_t exti_pin    = 0U;
    uint8_t exti_line   = 0U;
    int32_t error_code  = HAL_ERR_NONE;

    /* check parameter value */
#if (1U == HAL_PARAMETER_CHECK)
    if((EXTI_0_IRQHandler_USED != index) && (EXTI_1_IRQHandler_USED != index) && \
       (EXTI_2_IRQHandler_USED != index) && (EXTI_3_IRQHandler_USED != index) && \
       (EXTI_4_IRQHandler_USED != index) && (EXTI_5_9_IRQHandler_USED != index) && \
       (EXTI_10_15_IRQHandler_USED != index)) {
        HAL_DEBUGE("parameter [index] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((EXTI_0_IRQHandler_USED == index) || (EXTI_1_IRQHandler_USED == index) || \
       (EXTI_2_IRQHandler_USED == index) || (EXTI_3_IRQHandler_USED == index) || \
       (EXTI_4_IRQHandler_USED == index)) {

        exti_pin = (uint8_t)index & 0x0FU;

        /* get EXTI GPIO port and pin */
        temp = EXTI_GPIO_PORT_GET((uint32_t)s_exti_gpio_info[exti_pin]);
        gpio_port = GPIO_BASE + (temp << 10);
        temp = EXTI_GPIO_PIN_GET((uint32_t)s_exti_gpio_info[exti_pin]);
        gpio_pin = ((uint32_t)1 << temp);

        exti_line = EXTI_GPIO_PIN_GET(s_exti_gpio_info[exti_pin]);
        /* get EXTI lines flag when the interrupt flag is set */
        if(RESET != hals_exti_interrupt_flag_get((hal_exti_line_enum)exti_line)) {
            hals_exti_interrupt_flag_clear((hal_exti_line_enum)exti_line);
            if(NULL == s_gpio_irq_handle) {
                HAL_DEBUGE("parameter [EXTI irq handle] is invalid");
                error_code = HAL_ERR_VAL;
            } else {
                /* get EXTI GPIO pin input status */
                if(RESET == hal_gpio_input_bit_get(gpio_port, gpio_pin)) {
                    s_gpio_irq_handle(gpio_pin, EXTI_IRQ_EVENT_FALLING);
                } else {
                    s_gpio_irq_handle(gpio_pin, EXTI_IRQ_EVENT_RISING);
                }
            }
        } else {
            /* do nothing */
        }
    } else {
        start_pin = (index & 0xF0U) >> 4;
        end_pin = index & 0x0FU;

        /* get EXTI GPIO port and pin */
        for(i = start_pin; i <= end_pin; i++) {
            if(0U != (_exti_gpio_used & (1U << i))) {
                temp = EXTI_GPIO_PORT_GET((uint32_t)s_exti_gpio_info[i]);
                gpio_port = GPIO_BASE + (temp << 10);
                temp = EXTI_GPIO_PIN_GET((uint32_t)s_exti_gpio_info[i]);
                gpio_pin = ((uint32_t)1U << temp);

                exti_line = EXTI_GPIO_PIN_GET(s_exti_gpio_info[i]);
                /* get EXTI lines flag when the interrupt flag is set */
                if(RESET != hals_exti_interrupt_flag_get((hal_exti_line_enum)exti_line)) {
                    hals_exti_interrupt_flag_clear((hal_exti_line_enum)exti_line);
                    if(NULL != s_gpio_irq_handle) {
                        /* get EXTI GPIO pin input status */
                        if(RESET == hal_gpio_input_bit_get(gpio_port, gpio_pin)) {
                            s_gpio_irq_handle(gpio_pin, EXTI_IRQ_EVENT_FALLING);
                        } else {
                            s_gpio_irq_handle(gpio_pin, EXTI_IRQ_EVENT_RISING);
                        }
                    }
                }
            } else {
                /* do nothing */
            }
        }
    }

    return error_code;
}

/*!
    \brief      enable the interrupts from EXTI line x
    \param[in]  linex: EXTI line number, refer to hal_exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     none
*/
void hals_exti_interrupt_enable(hal_exti_line_enum linex)
{
    EXTI_INTEN(EXTI_REG_VAL_LINE(linex)) |= EXTI_BIT_POS(linex);
}

/*!
    \brief      disable the interrupts from EXTI line x
    \param[in]  linex: EXTI line number, refer to hal_exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     none
*/
void hals_exti_interrupt_disable(hal_exti_line_enum linex)
{
    EXTI_INTEN(EXTI_REG_VAL_LINE(linex)) &= ~(EXTI_BIT_POS(linex));
}

/*!
    \brief      enable the events from EXTI line x
    \param[in]  linex: EXTI line number, refer to hal_exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     none
*/
void hals_exti_event_enable(hal_exti_line_enum linex)
{
    EXTI_EVEN(EXTI_REG_VAL_LINE(linex)) |= EXTI_BIT_POS(linex);
}

/*!
    \brief      disable the events from EXTI line x
    \param[in]  linex: EXTI line number, refer to hal_exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     none
*/
void hals_exti_event_disable(hal_exti_line_enum linex)
{
    EXTI_EVEN(EXTI_REG_VAL_LINE(linex)) &= ~(EXTI_BIT_POS(linex));
}

/*!
    \brief      enable the software interrupt event from EXTI line x
    \param[in]  linex: EXTI line number, refer to hal_exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     none
*/
void hal_exti_software_interrupt_enable(hal_exti_line_enum linex)
{
    EXTI_SWIEV(EXTI_REG_VAL_LINE(linex)) |= EXTI_BIT_POS(linex);
}

/*!
    \brief      disable the software interrupt event from EXTI line x
    \param[in]  linex: EXTI line number, refer to hal_exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     none
*/
void hals_exti_software_interrupt_disable(hal_exti_line_enum linex)
{
    EXTI_SWIEV(EXTI_REG_VAL_LINE(linex)) &= ~(EXTI_BIT_POS(linex));
}

/*!
    \brief      get EXTI line x interrupt pending flag
    \param[in]  linex: EXTI line number, refer to exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     FlagStatus: RESET or SET
*/
FlagStatus hals_exti_flag_get(hal_exti_line_enum linex)
{
    FlagStatus state = RESET;
    if(RESET != (EXTI_PD(EXTI_REG_VAL_LINE(linex)) & EXTI_BIT_POS(linex))) {
        state = SET;
    } else {
        /* do nothing */
    }
    return state;
}

/*!
    \brief      clear EXTI line x interrupt pending flag
    \param[in]  linex: EXTI line number, refer to exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     none
*/
void hals_exti_flag_clear(hal_exti_line_enum linex)
{
    EXTI_PD(EXTI_REG_VAL_LINE(linex)) = EXTI_BIT_POS(linex);
}

/*!
    \brief      get EXTI line x interrupt pending flag
    \param[in]  linex: EXTI line number, refer to hal_exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     FlagStatus: RESET or SET
*/
FlagStatus hals_exti_interrupt_flag_get(hal_exti_line_enum linex)
{
    FlagStatus state = RESET;
    if(RESET != (EXTI_PD(EXTI_REG_VAL_LINE(linex)) & EXTI_BIT_POS(linex))) {
        state = SET;
    } else {
        /* do nothing */
    }
    return state;
}

/*!
    \brief      clear EXTI line x interrupt pending flag
    \param[in]  linex: EXTI line number, refer to hal_exti_line_enum
                only one parameter can be selected which is shown as below:
      \arg        EXTI_x (x=0..37): EXTI line x
    \param[out] none
    \retval     none
*/
void hals_exti_interrupt_flag_clear(hal_exti_line_enum linex)
{
    EXTI_PD(EXTI_REG_VAL_LINE(linex)) = EXTI_BIT_POS(linex);
}

/*!
    \brief      register EXTI GPIO line
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     error code: HAL_ERR_ALREADY_DONE, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_exti_register_gpio_line(uint32_t gpio_periph, uint32_t pin)
{
    int32_t error_code = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && (GPIOD != gpio_periph) && \
       (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && \
       (GPIOJ != gpio_periph) && (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check if pin is 0 ~ 15 GPIOx(x = A,B,C,D,E,F,G,H) */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && (GPIOD != gpio_periph) && \
       (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && (GPIOG != gpio_periph) && (GPIOH != gpio_periph)) {
        if((0U != (pin & GPIO_PIN_VALUE_MASK))) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }

    /* check if pin is PJx(8..11) or not*/
    if((GPIOJ == gpio_periph)) {
        if((0U != (pin & GPIOJ_PIN_VALUE_MASK))) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }

    /* check if pin is PKx(0..2) or not*/
    if((GPIOK == gpio_periph)) {
        if((0U != (pin & GPIOK_PIN_VALUE_MASK))) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(0U != (_exti_gpio_used & pin)) {
        HAL_DEBUGE("EXTI gpio init fail, this pin is already used");
        error_code = HAL_ERR_ALREADY_DONE;
    } else {
        _exti_gpio_used |= pin;
        _exti_gpio_info_set(gpio_periph, pin);
    }

    return error_code;
}

/*!
    \brief      configure EXTI GPIO line
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[in]  exti_type: the argument could be selected from enumeration <hal_exti_type_enum>
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_exti_config_gpio_line(uint32_t gpio_periph, uint32_t pin, uint32_t exti_type)
{
    uint32_t i                  = 0U;
    uint32_t exti_gpio_port     = 0U;
    uint32_t exti_gpio_pin_port = 0U;
    int32_t error_code          = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check gpio_periph value */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && (GPIOD != gpio_periph) && \
       (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && (GPIOG != gpio_periph) && (GPIOH != gpio_periph) && \
       (GPIOJ != gpio_periph) && (GPIOK != gpio_periph)) {
        HAL_DEBUGE("parameter [gpio_periph] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check if pin is 0 ~ 15 GPIOx(x = A,B,C,D,E,F,G,H) */
    if((GPIOA != gpio_periph) && (GPIOB != gpio_periph) && (GPIOC != gpio_periph) && (GPIOD != gpio_periph) && \
       (GPIOE != gpio_periph) && (GPIOF != gpio_periph) && (GPIOG != gpio_periph) && (GPIOH != gpio_periph)) {
        if((0U != (pin & GPIO_PIN_VALUE_MASK))) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }

    /* check if pin is PJx(8..11) or not*/
    if((GPIOJ == gpio_periph)) {
        if((0U != (pin & GPIOJ_PIN_VALUE_MASK))) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }

    /* check if pin is PKx(0..2) or not*/
    if((GPIOK == gpio_periph)) {
        if((0U != (pin & GPIOK_PIN_VALUE_MASK))) {
            HAL_DEBUGE("parameter [pin] value is invalid");
            return HAL_ERR_VAL;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get EXTI GPIO port */
    exti_gpio_port = (((gpio_periph & EXTI_GPIO_SOURCE_MASK) >> 10U) & (0x0000000FU));

    /* get EXTI GPIO pin */
    for(i = 0U; i < 15U; i++) {
        if(pin == BIT(i)) {
            exti_gpio_pin_port = (uint32_t)i;
        } else {
            /* do nothing */
        }
    }

    /* configure EXTI GPIO */
    hal_syscfg_exti_config(exti_gpio_port, exti_gpio_pin_port);
    _exti_type_config(EXTI_GROUP0_OFFSET, pin, (hal_exti_type_enum)exti_type);

    return error_code;
}

/*!
    \brief      enable the configuration of EXTI type
    \param[in]  exti_group: EXTI group offset
                only one parameter can be selected which is shown as below:
      \arg        EXTI_GROUP0_OFFSET: the index offset of EXTI group 0
      \arg        EXTI_GROUP1_OFFSET: the index offset of EXTI group 1
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[in]  exti_type: the argument could be selected from enumeration <hal_exti_type_enum>
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
static int32_t _exti_type_config(uint32_t exti_group, uint32_t pin, hal_exti_type_enum exti_type)
{
    uint32_t reg_temp   = 0U;
    int32_t  error_code = HAL_ERR_NONE;

    /* check parameter value */
#if (1U == HAL_PARAMETER_CHECK)
    if((EXTI_EVENT_TRIG_RISING != exti_type) && (EXTI_EVENT_TRIG_FALLING != exti_type) && \
       (EXTI_EVENT_TRIG_BOTH != exti_type) && (EXTI_INTERRUPT_TRIG_RISING != exti_type) && \
       (EXTI_INTERRUPT_TRIG_FALLING != exti_type) && (EXTI_INTERRUPT_TRIG_BOTH != exti_type)) {
        HAL_DEBUGE("parameter [exti_type] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset the EXTI gpio pin */
    EXTI_INTEN(EXTI_REG_VAL_GROUP(exti_group)) &= ~pin;
    EXTI_EVEN(EXTI_REG_VAL_GROUP(exti_group)) &= ~pin;
    EXTI_RTEN(EXTI_REG_VAL_GROUP(exti_group)) &= ~pin;
    EXTI_FTEN(EXTI_REG_VAL_GROUP(exti_group)) &= ~pin;
    EXTI_PD(EXTI_REG_VAL_GROUP(exti_group)) = pin;

    /* set the EXTI trigger type as the rising edge trigger */
    reg_temp = EXTI_RTEN(EXTI_REG_VAL_GROUP(exti_group));
    if(0U != (exti_type & EXTI_RISING)) {
        reg_temp |= pin;
    } else {
        reg_temp &= ~pin;
    }
    EXTI_RTEN(EXTI_REG_VAL_GROUP(exti_group)) = reg_temp;

    /* set the EXTI trigger type as the falling edge trigger */
    reg_temp = EXTI_FTEN(EXTI_REG_VAL_GROUP(exti_group));
    if(0U != (exti_type & EXTI_FALLING)) {
        reg_temp |= pin;
    } else {
        reg_temp &= ~pin;
    }

    EXTI_FTEN(EXTI_REG_VAL_GROUP(exti_group)) = reg_temp;

    /* set the EXTI trigger type as the event trigger */
    reg_temp = EXTI_EVEN(EXTI_REG_VAL_GROUP(exti_group));
    if(0U != (exti_type & EXTI_EVENT)) {
        reg_temp |= pin;
    } else {
        reg_temp &= ~pin;
    }

    EXTI_EVEN(EXTI_REG_VAL_GROUP(exti_group)) = reg_temp;

    /* set the EXTI trigger type as the interrupt trigger */
    reg_temp = EXTI_INTEN(EXTI_REG_VAL_GROUP(exti_group));
    if(0U != (exti_type & EXTI_INTERRUPT)) {
        reg_temp |= pin;
    } else {
        reg_temp &= ~pin;
    }
    EXTI_INTEN(EXTI_REG_VAL_GROUP(exti_group)) = reg_temp;

    return error_code;
}

/*!
    \brief      get the configuration of EXTI type
    \param[in]  exti_group: EXTI group offset
                only one parameter can be selected which is shown as below:
      \arg        EXTI_GROUP0_OFFSET: the index offset of EXTI group 0
      \arg        EXTI_GROUP1_OFFSET: the index offset of EXTI group 1
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[in]  exti_type: the argument could be selected from enumeration <hal_exti_type_enum>
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
static int32_t _exti_type_get(uint32_t exti_group, uint32_t pin, hal_exti_type_enum *p_exti_type)
{
    uint32_t reg_value  = 0U;
    int32_t  error_code = HAL_ERR_NONE;

    /* check parameter value */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_exti_type) {
        HAL_DEBUGE("parameter [p_exti_type] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear value */
    *p_exti_type = EXTI_EVENT_TRIG_RISING;

    /* get the EXTI trigger type as the rising edge trigger */
    reg_value = EXTI_RTEN(EXTI_REG_VAL_GROUP(exti_group));
    if(0U != (reg_value & pin)) {
        *p_exti_type |= (hal_exti_type_enum)EXTI_RISING;
    } else {
        /* do nothing */
    }

    /* get the EXTI trigger type as the falling edge trigger */
    reg_value = EXTI_FTEN(EXTI_REG_VAL_GROUP(exti_group));
    if(0U != (reg_value & pin)) {
        *p_exti_type |= (hal_exti_type_enum)EXTI_FALLING;
    } else {
        /* do nothing */
    }

    /* get the EXTI trigger type as the event trigger */
    reg_value = EXTI_EVEN(EXTI_REG_VAL_GROUP(exti_group));
    if(0U != (reg_value & pin)) {
        *p_exti_type |= (hal_exti_type_enum)EXTI_EVENT;
    } else {
        /* do nothing */
    }

    /* get the EXTI trigger type as the interrupt trigger */
    reg_value = EXTI_INTEN(EXTI_REG_VAL_GROUP(exti_group));
    if(0U != (reg_value & pin)) {
        *p_exti_type |= (hal_exti_type_enum)EXTI_INTERRUPT;
    } else {
        /* do nothing */
    }

    return error_code;
}

/*!
    \brief      set the EXTI gpio port and pin
    \param[in]  gpio_periph: GPIOx(x = A,B,C,D,E,F,G,H,J,K)
    \param[in]  pin: GPIO pin
                one or more parameters can be selected which are shown as below:
      \arg        GPIO_PIN_x(x=0..15), GPIO_PIN_ALL
    \param[out] none
    \retval     none
*/
static void _exti_gpio_info_set(uint32_t gpio_periph, uint32_t pin)
{
    uint32_t gpio_port;
    uint8_t gpio_pin;

    /* set the EXTI gpio port */
    gpio_port = gpio_periph - GPIO_BASE;
    gpio_port >>= 10U;

    /* set the EXTI gpio pin */
    for(gpio_pin = 0U; gpio_pin < 16U; gpio_pin++) {
        if((1U << gpio_pin) & pin) {
            s_exti_gpio_info[gpio_pin] = (uint8_t)((gpio_port << 4) | gpio_pin);
        } else {
            /* do nothing */
        }
    }
}
