/*!
    \file    gd32h7xx_hal_exmc_sdram.c
    \brief   EXMC SDRAM driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* EXMC SRAM private function */
static void _sdram_dma_complete(void *dma);
static void _sdram_dma_complete_protected(void *dma);
static void _sdram_dma_error(void *dma);

/*!
    \brief      initialize EXMC SDRAM device
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sdram_init: initialize structure the EXMC SDRAM device parameter
                This parameter contains following fields:
                sdram_device: select the SDRAM device
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_DEVICE0: SDRAM device0
      \arg        EXMC_SDRAM_DEVICE1: SDRAM device1
                bank_remap: select EXMC NOR/PSRAM remap region
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK_REMAP_DEFAULT: default bank remap configuration
      \arg        EXMC_BANK_NORPSRAM_SDRAM_SWAP: NOR/PSRAM and SDRAM bank swap
                column_address_width: specify the bit width of a column address
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_COW_ADDRESS_8: column address bit width is 8 bits
      \arg        EXMC_SDRAM_COW_ADDRESS_9: column address bit width is 9 bits
      \arg        EXMC_SDRAM_COW_ADDRESS_10: column address bit width is 10 bits
      \arg        EXMC_SDRAM_COW_ADDRESS_11: column address bit width is 11 bits
                row_address_width: specify the bit width of a row address
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_ROW_ADDRESS_11: row address bit width is 11 bits
      \arg        EXMC_SDRAM_ROW_ADDRESS_12: row address bit width is 12 bits
      \arg        EXMC_SDRAM_ROW_ADDRESS_13: row address bit width is 13 bits
                data_width: specify the data bus width of SDRAM memory
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_DATABUS_WIDTH_8B: SDRAM data width 8 bits
      \arg        EXMC_SDRAM_DATABUS_WIDTH_16B: SDRAM data width 16 bits
      \arg        EXMC_SDRAM_DATABUS_WIDTH_32B: SDRAM data width 32 bits
                internal_bank_number: specify the number of internal bank
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_2_INTER_BANK: 2 internal banks
      \arg        EXMC_SDRAM_4_INTER_BANK: 4 internal banks
                cas_latency: configure the SDRAM CAS latency
                only one parameter can be selected which is shown as below:
      \arg        EXMC_CAS_LATENCY_1_SDCLK: CAS latency is 1 memory clock cycle
      \arg        EXMC_CAS_LATENCY_2_SDCLK: CAS latency is 2 memory clock cycle
      \arg        EXMC_CAS_LATENCY_3_SDCLK: CAS latency is 3 memory clock cycle
                write_protection: enable or disable SDRAM device write protection function
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the write protection function
      \arg        DISABLE: disable the write protection function
                sdclock_config: configure the SDCLK memory clock
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDCLK_PERIODS_2_CK_EXMC: SDCLK memory period = 2CK_EXMC
      \arg        EXMC_SDCLK_PERIODS_3_CK_EXMC: SDCLK memory period = 3CK_EXMC
      \arg        EXMC_SDCLK_PERIODS_4_CK_EXMC: SDCLK memory period = 4CK_EXMC
      \arg        EXMC_SDCLK_PERIODS_5_CK_EXMC: SDCLK memory period = 5*CK_EXMC
                burst_read_switch: enable or disable the burst read
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the burst read
      \arg        DISABLE: disable the burst read
                pipeline_read_delay: configure the delay for reading data after CAS latency
                only one parameter can be selected which is shown as below:
      \arg        EXMC_PIPELINE_DELAY_0_CK_EXMC: 0 CK_EXMC clock cycle delay
      \arg        EXMC_PIPELINE_DELAY_1_CK_EXMC: 1 CK_EXMC clock cycle delay
      \arg        EXMC_PIPELINE_DELAY_2_CK_EXMC: 2 CK_EXMC clock cycle delay
                load_mode_register_delay: 1 - 16, configure the load mode register delay
                exit_selfrefresh_delay: 1 - 16, configure the exit self-refresh delay
                row_address_select_delay: 1 - 16, configure the row address select delay
                auto_refresh_delay: 1 - 16, configure the auto refresh delay
                write_recovery_delay: 1 - 16, configure the write recovery delay
                row_precharge_delay: 1 - 16, configure the row precharge delay
                row_to_column_delay: 1- 16, configure the row to column delay
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_init(hal_exmc_sdram_dev_struct *exmc_dev, hal_exmc_sdram_init_struct *sdram_init)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == sdram_init)) {
        HAL_DEBUGE("parameter [exmc_dev] or [sdram_init ] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(exmc_dev);
    exmc_dev->state = HAL_EXMC_SDRAM_STATE_RESET;
    exmc_dev->sdram_device = sdram_init->sdram_device;

    if(EXMC_SDRAM_DEVICE0 == sdram_init->sdram_device) {
        /* configure control register */
        EXMC_SDCTL(EXMC_SDRAM_DEVICE0) = (uint32_t)((sdram_init->column_address_width & EXMC_SDCTL_CAW) | \
                                                    (sdram_init->row_address_width & EXMC_SDCTL_RAW) | \
                                                    (sdram_init->data_width & EXMC_SDCTL_SDW) | \
                                                    (sdram_init->internal_bank_number & EXMC_SDCTL_NBK) | \
                                                    ((sdram_init->write_protection << SDCTL_WPEN_OFFSET) & EXMC_SDCTL_WPEN) | \
                                                    (sdram_init->sdclock_config & (EXMC_SDCTL_SDCLK | EXMC_SDCTL_SDCLK_2)) | \
                                                    (sdram_init->pipeline_read_delay & EXMC_SDCTL_PIPED) | \
                                                    ((sdram_init->burst_read_switch << SDCTL_BRSTRD_OFFSET) & EXMC_SDCTL_BRSTRD) | \
                                                    (sdram_init->cas_latency & EXMC_SDCTL_CL));

        /* configure timing configuration register */
        EXMC_SDTCFG(EXMC_SDRAM_DEVICE0) = (uint32_t)((((sdram_init->load_mode_register_delay) - 1U) & EXMC_SDTCFG_LMRD)| \
                                                     ((((sdram_init->exit_selfrefresh_delay) - 1U) << SDTCFG_XSRD_OFFSET) & EXMC_SDTCFG_XSRD) | \
                                                     ((((sdram_init->row_address_select_delay) - 1U) << SDTCFG_RASD_OFFSET) & EXMC_SDTCFG_RASD)| \
                                                     ((((sdram_init->auto_refresh_delay) - 1U) << SDTCFG_ARFD_OFFSET) & EXMC_SDTCFG_ARFD) | \
                                                     ((((sdram_init->write_recovery_delay) - 1U) << SDTCFG_WRD_OFFSET) & EXMC_SDTCFG_WRD) | \
                                                     ((((sdram_init->row_precharge_delay) - 1U) << SDTCFG_RPD_OFFSET) & EXMC_SDTCFG_RPD) | \
                                                     ((((sdram_init->row_to_column_delay) - 1U) << SDTCFG_RCD_OFFSET) & EXMC_SDTCFG_RCD));

        /* configure bank remapping register */
        EXMC_SNCTL(EXMC_BANK0_NORSRAM_REGION0) &= (uint32_t)(~EXMC_SNCTL_BKREMAP);
        EXMC_SNCTL(EXMC_BANK0_NORSRAM_REGION0) |= (sdram_init->bank_remap & EXMC_SNCTL_BKREMAP);
    } else {
        /* configure control register */
        /* some bits in the EXMC_SDCTL1 register are reserved */
        EXMC_SDCTL(EXMC_SDRAM_DEVICE1) &= (~(EXMC_SDCTL_PIPED | EXMC_SDCTL_BRSTRD | EXMC_SDCTL_SDCLK | \
        EXMC_SDCTL_SDCLK_2));

        EXMC_SDCTL(EXMC_SDRAM_DEVICE1) |= (uint32_t)((sdram_init->column_address_width & EXMC_SDCTL_CAW) | \
                                                     (sdram_init->row_address_width & EXMC_SDCTL_RAW) | \
                                                     (sdram_init->data_width & EXMC_SDCTL_SDW) | \
                                                     (sdram_init->internal_bank_number & EXMC_SDCTL_NBK) | \
                                                     (sdram_init->cas_latency & EXMC_SDCTL_CL) | \
                                                     ((sdram_init->write_protection << SDCTL_WPEN_OFFSET) & EXMC_SDCTL_WPEN));

        /* configure timing configuration register */
        /* some bits in the EXMC_SDTCFG1 register are reserved */
        EXMC_SDTCFG(EXMC_SDRAM_DEVICE1) &= (~(EXMC_SDTCFG_RPD | EXMC_SDTCFG_WRD | EXMC_SDTCFG_ARFD));

        EXMC_SDTCFG(EXMC_SDRAM_DEVICE1) = (uint32_t)((((sdram_init->load_mode_register_delay) - 1U) & EXMC_SDTCFG_LMRD) | \
                                                     ((((sdram_init->exit_selfrefresh_delay) - 1U) << SDTCFG_XSRD_OFFSET) & EXMC_SDTCFG_XSRD) | \
                                                     ((((sdram_init->row_address_select_delay) - 1U) << SDTCFG_RASD_OFFSET) & EXMC_SDTCFG_RASD) | \
                                                     ((((sdram_init->row_to_column_delay) - 1U) << SDTCFG_RCD_OFFSET) & EXMC_SDTCFG_RCD));
    }

    /* configure auto-refresh interval register */
    EXMC_SDARI &= ~(uint32_t)EXMC_SDARI_ARINTV;
    EXMC_SDARI |= (uint32_t)((sdram_init->auto_refresh_interval << SDARI_ARINTV_OFFSET) & EXMC_SDARI_ARINTV);

    /* configure read sample control register */
    EXMC_SDRSCTL = (uint32_t)((sdram_init->read_sample_enable & EXMC_SDRSCTL_RSEN) | (sdram_init->read_sample_cycle & EXMC_SDRSCTL_SSCR) | \
                              ((sdram_init->read_sample_delay << SDRSCTL_SDSC_OFFSET) & EXMC_SDRSCTL_SDSC));

    exmc_dev->state = HAL_EXMC_SDRAM_STATE_READY;
    HAL_UNLOCK(exmc_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the SDRAM structure with default values
    \param[in]  hal_struct_type: The type of the struct to initialize
                only one parameters can be selected which are shown as below:
      \arg        HAL_EXMC_SDRAM_INIT_STRUCT: SDRAM initialization structure
      \arg        HAL_EXMC_SDRAM_DEV_STRUCT: SDRAM device structure
      \arg        HAL_EXMC_SDRAM_IRQ_INIT_STRUCT: SDRAM interrupt callback initialization structure
    \param[out] p_struct: pointer to SDRAM structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_struct_init(hal_exmc_sdram_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    switch(hal_struct_type) {
    case HAL_EXMC_SDRAM_INIT_STRUCT:
        ((hal_exmc_sdram_init_struct *)p_struct)->sdram_device        = EXMC_SDRAM_DEVICE0;
        ((hal_exmc_sdram_init_struct *)p_struct)->pipeline_read_delay = EXMC_PIPELINE_DELAY_0_CK_EXMC;
        ((hal_exmc_sdram_init_struct *)p_struct)->burst_read_switch   = DISABLE;
        ((hal_exmc_sdram_init_struct *)p_struct)->sdclock_config      = EXMC_SDCLK_DISABLE;
        ((hal_exmc_sdram_init_struct *)p_struct)->write_protection    = DISABLE;
        ((hal_exmc_sdram_init_struct *)p_struct)->cas_latency         = EXMC_CAS_LATENCY_1_SDCLK;

        ((hal_exmc_sdram_init_struct *)p_struct)->internal_bank_number  = EXMC_SDRAM_4_INTER_BANK;
        ((hal_exmc_sdram_init_struct *)p_struct)->data_width            = EXMC_SDRAM_DATABUS_WIDTH_16B;
        ((hal_exmc_sdram_init_struct *)p_struct)->row_address_width     = EXMC_SDRAM_ROW_ADDRESS_11;
        ((hal_exmc_sdram_init_struct *)p_struct)->column_address_width  = EXMC_SDRAM_COW_ADDRESS_8;
        ((hal_exmc_sdram_init_struct *)p_struct)->auto_refresh_interval = 0U;
        ((hal_exmc_sdram_init_struct *)p_struct)->read_sample_enable    = DISABLE;
        ((hal_exmc_sdram_init_struct *)p_struct)->read_sample_cycle     = EXMC_SDRAM_READSAMPLE_0_EXTRACK;
        ((hal_exmc_sdram_init_struct *)p_struct)->read_sample_delay     = 0U;

        ((hal_exmc_sdram_init_struct *)p_struct)->row_to_column_delay      = 15U;
        ((hal_exmc_sdram_init_struct *)p_struct)->row_precharge_delay      = 15U;
        ((hal_exmc_sdram_init_struct *)p_struct)->write_recovery_delay     = 15U;
        ((hal_exmc_sdram_init_struct *)p_struct)->auto_refresh_delay       = 15U;
        ((hal_exmc_sdram_init_struct *)p_struct)->row_address_select_delay = 15U;
        ((hal_exmc_sdram_init_struct *)p_struct)->exit_selfrefresh_delay   = 15U;
        ((hal_exmc_sdram_init_struct *)p_struct)->load_mode_register_delay = 15U;
        break;
    case HAL_EXMC_SDRAM_DEV_STRUCT:
        ((hal_exmc_sdram_dev_struct *)p_struct)->sdram_device               = EXMC_SDRAM_DEVICE0;
        ((hal_exmc_sdram_dev_struct *)p_struct)->p_dma_rx                   = NULL;
        ((hal_exmc_sdram_dev_struct *)p_struct)->p_dma_tx                   = NULL;
        ((hal_exmc_sdram_dev_struct *)p_struct)->exmc_irq.refresh_error_handle = NULL;
        ((hal_exmc_sdram_dev_struct *)p_struct)->dma_transfer_complete      = NULL;
        ((hal_exmc_sdram_dev_struct *)p_struct)->dma_transfer_error         = NULL;
        ((hal_exmc_sdram_dev_struct *)p_struct)->mutex                      = HAL_MUTEX_UNLOCKED;
        ((hal_exmc_sdram_dev_struct *)p_struct)->state                      = HAL_EXMC_SDRAM_STATE_RESET;
        ((hal_exmc_sdram_dev_struct *)p_struct)->priv                       = NULL;
        break;
    case HAL_EXMC_SDRAM_IRQ_INIT_STRUCT:
        ((hal_exmc_sdram_irq_struct *)p_struct)->refresh_error_handle = NULL;
        break;
    case HAL_EXMC_SDRAM_COMMAND_PARAMETER_STRUCT:
        ((hal_exmc_sdram_command_parameter_struct *)p_struct)->mode_register_content = 0U;
        ((hal_exmc_sdram_command_parameter_struct *)p_struct)->auto_refresh_number   = 0U;
        ((hal_exmc_sdram_command_parameter_struct *)p_struct)->bank_select           = 0U;
        ((hal_exmc_sdram_command_parameter_struct *)p_struct)->command               = 0U;
        break;
    case HAL_EXMC_SDRAM_USER_CALLBACK_STRUCT:
        ((hal_exmc_sdram_user_callback_struct *)p_struct)->dma_transfer_complete = NULL;
        ((hal_exmc_sdram_user_callback_struct *)p_struct)->dma_transfer_error    = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize EXMC SDRAM device
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_deinit(hal_exmc_sdram_dev_struct *exmc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_dev) {
        HAL_DEBUGE("pointer [exmc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset the registers */
    EXMC_SDCTL(exmc_dev->sdram_device)  = SDRAM_DEVICE_SDCTL_RESET;
    EXMC_SDTCFG(exmc_dev->sdram_device) = SDRAM_DEVICE_SDTCFG_RESET;
    EXMC_SDCMD                          = SDRAM_DEVICE_SDCMD_RESET;
    EXMC_SDARI                          = SDRAM_DEVICE_SDARI_RESET;
    EXMC_SDRSCTL                        = SDRAM_DEVICE_SDRSCTL_RESET;
    exmc_dev->state                     = HAL_EXMC_SDRAM_STATE_RESET;
    exmc_dev->mutex                     = HAL_MUTEX_UNLOCKED;

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to EXMC interrupt callback functions structure
                  hal_irq_handle_cb: the function is user-defined,the corresponding callback mechanism is in use,
                  and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_irq_handle_set(hal_exmc_sdram_dev_struct *exmc_dev, hal_exmc_sdram_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [exmc_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(exmc_dev);
    if(NULL != p_irq->refresh_error_handle) {
        exmc_dev->exmc_irq.refresh_error_handle = p_irq->refresh_error_handle;
        hals_exmc_sdram_interrupt_enable();
    } else {
        hals_exmc_sdram_interrupt_disable();
    }
    HAL_UNLOCK(exmc_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_irq_handle_all_reset(hal_exmc_sdram_dev_struct *exmc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_dev) {
        HAL_DEBUGE("pointer [exmc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(exmc_dev);
    exmc_dev->exmc_irq.refresh_error_handle = NULL;
    hals_exmc_sdram_interrupt_disable();
    HAL_UNLOCK(exmc_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      EXMC SDRAM interrupt handler content function, which is merely used in EXMC_IRQHandler
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_exmc_sdram_irq(hal_exmc_sdram_dev_struct *exmc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_dev) {
        HAL_DEBUGE("pointer [exmc_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /*!< refresh error interrupt handler */
    if(SET == hals_exmc_sdram_interrupt_flag_get()) {
        hals_exmc_sdram_interrupt_flag_clear();
        if(NULL != exmc_dev->exmc_irq.refresh_error_handle) {
            exmc_dev->exmc_irq.refresh_error_handle(exmc_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      write 8 bit data to sdram memory
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  write_address: pointer write address
    \param[in]  write_buffer: pointer write buffer address
    \param[in]  length: write byte length
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_write_8bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *write_address, uint8_t *write_buffer, \
                                  uint32_t length, hal_exmc_sdram_irq_struct *p_irq)
{
    uint32_t size;
    __IO uint8_t *psdram_address = (uint8_t *)write_address;
    uint8_t *pwrite_buff = write_buffer;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == write_buffer) || (NULL == write_address)) {
        HAL_DEBUGE("pointer [exmc_dev] or [write_buffer] or [write_address] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the tx_state whether it is busy or not */
    if(HAL_EXMC_SDRAM_STATE_READY == exmc_dev->state) {

        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->refresh_error_handle) {
                exmc_dev->exmc_irq.refresh_error_handle = p_irq->refresh_error_handle;
            } else {
                /* do nothing */
            }
        } else {
                /* do nothing */
        }


        /* Write data to memory */
        for (size = length; 0U != size; size--) {
            *(__IO uint8_t *)psdram_address = *pwrite_buff;
            pwrite_buff++;
            psdram_address++;
        }

        exmc_dev->state = HAL_EXMC_SDRAM_STATE_READY;
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read 8 bit data to sdram memory
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  read_address: pointer read address
    \param[out] read_buffer: pointer read buffer address
    \param[in]  length: write byte length
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_read_8bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *read_address, uint8_t *read_buffer, \
                                 uint32_t length, hal_exmc_sdram_irq_struct *p_irq)
{
    uint32_t size, state = exmc_dev->state;
    __IO uint8_t *psdram_address = (uint8_t *)read_address;
    uint8_t *pread_buff = read_buffer;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == read_buffer) || (NULL == read_address)) {
        HAL_DEBUGE("pointer [exmc_dev] or read_buffer] or [read_address] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the tx_state whether it is busy or not */
    if((HAL_EXMC_SDRAM_STATE_READY == state) || (HAL_EXMC_SDRAM_STATE_PROTECTED == state)) {

        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->refresh_error_handle) {
                exmc_dev->exmc_irq.refresh_error_handle = p_irq->refresh_error_handle;
            } else {
                /* do nothing */
            }
        } else {
                /* do nothing */
        }
        /* read data to memory */
        for (size = length; 0U != size; size--) {
            *pread_buff = *(__IO uint8_t *)psdram_address;
            pread_buff++;
            psdram_address++;
        }

        exmc_dev->state = (hal_exmc_sdram_state_enum)state;
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      write 16 bit data to sdram memory
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  write_address: write address
    \param[in]  write_buffer: pointer to write buffer address
    \param[in]  length: write half-word length
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_write_16bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *write_address, uint16_t *write_buffer, \
                                   uint32_t length, hal_exmc_sdram_irq_struct *p_irq)
{
    uint32_t size;
    __IO uint32_t *psdram_address = (uint32_t *)write_address;
    uint16_t *pwrite_buff = write_buffer;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == write_buffer) || (NULL == write_address)) {
        HAL_DEBUGE("pointer [exmc_dev] or [write_buffer] or [write_address] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("pointer [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the tx_state whether it is busy or not */
    if(HAL_EXMC_SDRAM_STATE_READY == exmc_dev->state) {

        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->refresh_error_handle) {
                exmc_dev->exmc_irq.refresh_error_handle = p_irq->refresh_error_handle;
            } else {
                /* do nothing */
            }
        } else {
                /* do nothing */
        }
        /* Write data to memory */
        for (size = length; size >= 2U; size -= 2U) {
            *psdram_address = (uint32_t)(*pwrite_buff);
            pwrite_buff++;
            *psdram_address |= ((uint32_t)(*pwrite_buff) << 16U);
            pwrite_buff++;
            psdram_address++;
        }

        /* write last 16-bits if size is not 32-bits multiple */
        if(0U != (length % 2U)) {
            *psdram_address = ((uint32_t)(*pwrite_buff) & 0x0000FFFFU) | ((*psdram_address) & 0xFFFF0000U);
        } else {
            /* do nothing */
        }

        exmc_dev->state = HAL_EXMC_SDRAM_STATE_READY;
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read 16 bit data to sdram memory
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  read_address: pointer read address
    \param[out] read_buffer: pointer read buffer address
    \param[in]  length: read half-word length
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_read_16bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *read_address, uint16_t *read_buffer, \
                                  uint32_t length, hal_exmc_sdram_irq_struct *p_irq)
{
    uint32_t size, state = exmc_dev->state;
    __IO uint32_t *psdram_address = (uint32_t *)read_address;
    uint16_t *pread_buff = read_buffer;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == read_buffer) || (NULL == read_address)) {
        HAL_DEBUGE("pointer [exmc_dev] or [read_buffer] or [read_address] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the tx_state whether it is busy or not */
    if((HAL_EXMC_SDRAM_STATE_READY == state) || (HAL_EXMC_SDRAM_STATE_PROTECTED == state)) {

        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->refresh_error_handle) {
                exmc_dev->exmc_irq.refresh_error_handle = p_irq->refresh_error_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* read data to memory */
        for (size = length; size >= 2U; size-= 2U) {
            *pread_buff = (uint16_t)((*psdram_address) & 0x0000FFFFU);
            pread_buff++;
            *pread_buff = (uint16_t)(((*psdram_address) & 0xFFFF0000U) >> 16U);
            pread_buff++;
            psdram_address++;
        }

        /* read last 16-bits if size is not 32-bits multiple */
        if(0U != (length % 2U)) {
            *pread_buff = (uint16_t)((*psdram_address) & 0x0000FFFFU);
        } else {
            /* do nothing */
        }

        exmc_dev->state = (hal_exmc_sdram_state_enum)state;
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      write 32 bit data to sdram memory
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  write_address: pointer write address
    \param[in]  write_buffer: pointer write buffer address
    \param[in]  length: write word length
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_write_32bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *write_address, uint32_t *write_buffer, \
                                   uint32_t length, hal_exmc_sdram_irq_struct *p_irq)
{
    uint32_t size;
    __IO uint32_t *psdram_address = (uint32_t *)write_address;
    uint32_t *pwrite_buff = write_buffer;
    uint32_t state;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == write_buffer) || (NULL == write_address)) {
        HAL_DEBUGE("pointer [exmc_dev] or [write_buffer] or [write_address] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    state = exmc_dev->state;
    /* check the tx_state whether it is busy or not */
    if((HAL_EXMC_SDRAM_STATE_READY == state) || (HAL_EXMC_SDRAM_STATE_PROTECTED == state)) {

        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->refresh_error_handle) {
                exmc_dev->exmc_irq.refresh_error_handle = p_irq->refresh_error_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* Write data to memory */
        for (size = length; 0U != size; size--) {
            *psdram_address = *pwrite_buff;
            pwrite_buff++;
            psdram_address++;
        }

        exmc_dev->state = (hal_exmc_sdram_state_enum)state;
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read 32 bit data to sdram memory
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  read_address: pointer read address
    \param[out] read_buffer: pointer read buffer address
    \param[in]  length: read word length
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_read_32bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *read_address, uint32_t *read_buffer, \
                                  uint32_t length, hal_exmc_sdram_irq_struct *p_irq)
{
    uint32_t size, state = exmc_dev->state;
    __IO uint32_t *psdram_address = (uint32_t *)read_address;
    uint32_t *pread_buff = read_buffer;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == read_buffer) || (NULL == read_address)) {
        HAL_DEBUGE("pointer [exmc_dev] or [read_buffer] or [read_address] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the tx_state whether it is busy or not */
    if(HAL_EXMC_SDRAM_STATE_READY == state) {

        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->refresh_error_handle) {
                exmc_dev->exmc_irq.refresh_error_handle = p_irq->refresh_error_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* read data to memory */
        for (size = length; 0U != size; size--) {
            *pread_buff = *(__IO uint32_t *)psdram_address;
            pread_buff++;
            psdram_address++;
        }

        exmc_dev->state = HAL_EXMC_SDRAM_STATE_READY;
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      write a block of 32-bit data from the EXMC SRAM memory using DMA transfer.
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  in_addr: pointer to source buffer to write
    \param[in]  sram_addr: pointer to write start address
    \param[in]  size: write length in byte
    \param[in]  p_user_func: pointer to usercallback function struct
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_write_dma(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *in_addr, uint32_t *sram_addr, \
                                 uint32_t size, hal_exmc_sdram_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == in_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_dev || in_addr || read_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("pointer [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the SRAM controller state */
    if(HAL_EXMC_SDRAM_STATE_READY == exmc_dev->state) {
        /* process Locked */
        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        /* configure DMA user callbacks */
        if(NULL != p_user_func->dma_transfer_complete) {
            exmc_dev->dma_transfer_complete = (hal_irq_handle_cb)p_user_func->dma_transfer_complete;
        } else {
            /* do nothing */
        }
        if(NULL != p_user_func->dma_transfer_error) {
            exmc_dev->dma_transfer_error = (hal_irq_handle_cb)p_user_func->dma_transfer_error;
        } else {
            /* do nothing */
        }
        /* configure DMA interrupt callback function */
        dma_irq.full_finish_handle = &_sdram_dma_complete;
        dma_irq.error_handle = &_sdram_dma_error;

        /* start DMA interrupt mode transfer */
        if(NULL != exmc_dev->p_dma_tx) {
            if(HAL_ERR_NONE != hal_dma_start_interrupt(exmc_dev->p_dma_tx, (uint32_t)in_addr, \
                    (uint32_t)sram_addr, (uint16_t)(size * 4u), &dma_irq)) {
                exmc_dev->state = HAL_EXMC_SDRAM_STATE_READY;
            } else {
                /* do nothing */
            }
        } else {
                /* do nothing */
        }

        /* process unlocked */
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read a block of 32-bit data from the EXMC SRAM memory using DMA transfer.
    \param[in]  exmc_dev: EXMC device information structure the EXMC NOR/SRAM parameter
    \param[in]  sram_addr: pointer to read start address
    \param[in]  size: read length in byte
    \param[out] out_addr: pointer to source buffer to read
    \param[in]  p_user_func: pointer to user callback function struct
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_read_dma(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *out_addr, uint32_t *sram_addr, \
                                uint32_t size, hal_exmc_sdram_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev) || (NULL == out_addr) || (NULL == sram_addr)) {
        HAL_DEBUGE("pointer [exmc_dev || out_addr || read_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == size)) {
        HAL_DEBUGE("pointer [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the SRAM controller state */
    if((HAL_EXMC_SDRAM_STATE_READY == exmc_dev->state) || (HAL_EXMC_SDRAM_STATE_PROTECTED == exmc_dev->state))
    {
        /* process Locked */
        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        /* configure DMA user callbacks */
        if(NULL != p_user_func->dma_transfer_complete) {
            exmc_dev->dma_transfer_complete = (hal_irq_handle_cb)p_user_func->dma_transfer_complete;
        } else {
            /* do nothing */
        }
        if(NULL != p_user_func->dma_transfer_error) {
            exmc_dev->dma_transfer_error =(hal_irq_handle_cb)p_user_func->dma_transfer_error;
        } else {
            /* do nothing */
        }
        /* configure DMA interrupt callback function */
        if(HAL_EXMC_SDRAM_STATE_READY == exmc_dev->state) {
            dma_irq.full_finish_handle = &_sdram_dma_complete;
        } else {
            dma_irq.full_finish_handle = &_sdram_dma_complete_protected;
        }
        dma_irq.error_handle = &_sdram_dma_error;

        /* start DMA interrupt mode transfer */
        if(NULL != exmc_dev->p_dma_rx) {
            if(HAL_ERR_NONE != hal_dma_start_interrupt(exmc_dev->p_dma_rx, (uint32_t)sram_addr, \
                    (uint32_t)out_addr, (uint16_t)(size * 4u), &dma_irq)) {
                exmc_dev->state = HAL_EXMC_SDRAM_STATE_READY;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* process unlocked */
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      enable nor write operation
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_write_operation_enable(hal_exmc_sdram_dev_struct *exmc_dev)
{
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev)) {
        HAL_DEBUGE("pointer exmc_dev address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_SDRAM_STATE_READY == exmc_dev->state) {
        /* process unlocked */
        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        EXMC_SDCTL(exmc_dev->sdram_device) |= EXMC_SDCTL_WPEN;

        exmc_dev->state = HAL_EXMC_SDRAM_STATE_PROTECTED;
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_ABORT;
    }

    return retval;
}

/*!
    \brief      disable nor write operation
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_ABORT details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_write_operation_disable(hal_exmc_sdram_dev_struct *exmc_dev)
{
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev)) {
        HAL_DEBUGE("pointer exmc_dev address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_SDRAM_STATE_PROTECTED == exmc_dev->state) {
        /* Process unlocked */
        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        EXMC_SDCTL(exmc_dev->sdram_device) &= ~((uint32_t)EXMC_SDCTL_WPEN);

        exmc_dev->state = HAL_EXMC_SDRAM_STATE_READY;
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_ABORT;
    }

    return retval;
}

/*!
    \brief      configure the SDRAM memory command
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  exmc_sdram_command: pointer to EXMC SDRAM command parameter structure
                  mode_register_content:
                  auto_refresh_number: EXMC_SDRAM_AUTO_REFLESH_x_SDCLK, x=1~15
                  bank_select: EXMC_SDRAM_DEVICE0_SELECT, EXMC_SDRAM_DEVICE1_SELECT, EXMC_SDRAM_DEVICE0_1_SELECT
                  command: EXMC_SDRAM_NORMAL_OPERATION, EXMC_SDRAM_CLOCK_ENABLE, EXMC_SDRAM_PRECHARGE_ALL,
                           EXMC_SDRAM_AUTO_REFRESH, EXMC_SDRAM_LOAD_MODE_REGISTER, EXMC_SDRAM_SELF_REFRESH,
                           EXMC_SDRAM_POWERDOWN_ENTRY
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_command_send(hal_exmc_sdram_dev_struct *exmc_dev, \
                                    hal_exmc_sdram_command_parameter_struct *exmc_sdram_command)
{
    int32_t retval = HAL_ERR_NONE;
    uint32_t timeout = 0xFFFFU;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_dev)) {
        HAL_DEBUGE("pointer exmc_dev address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((HAL_EXMC_SDRAM_STATE_READY == exmc_dev->state) || (HAL_EXMC_SDRAM_STATE_PRECHARGED == exmc_dev->state)) {

        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        while((hals_exmc_sdram_flag_get(EXMC_SDRAM_FLAG_NREADY) != RESET) && (timeout > 0U)) {
            timeout--;
        }

        /* Send SDRAM command */
        (void)hals_exmc_sdram_command_config(exmc_sdram_command);

        /* Update the SDRAM controller state state */
        if(exmc_sdram_command->command == EXMC_SDRAM_PRECHARGE_ALL) {
            exmc_dev->state = HAL_EXMC_SDRAM_STATE_PRECHARGED;
        } else {
            exmc_dev->state = HAL_EXMC_SDRAM_STATE_READY;
        }
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      set auto-refresh interval
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  exmc_count: the number SDRAM clock cycles unit between two successive auto-refresh commands,0x00000000~0x00001FFF
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_refresh_count_set(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t exmc_count)
{
    uint32_t sdari;
    uint32_t timeout = SDRAM_FREASH_TIMEOUT;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_dev) {
        HAL_DEBUGE("pointer [exmc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_SDRAM_STATE_READY == exmc_dev->state) {
        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        while((hals_exmc_sdram_flag_get(EXMC_SDRAM_FLAG_NREADY) != RESET) && (timeout > 0U)) {
            timeout--;
        }

        /* SDRAM controller timeout */
        if(0U == timeout) {
            retval = HAL_ERR_TIMEOUT;
        } else {
            sdari      = EXMC_SDARI & (~EXMC_SDARI_ARINTV);
            EXMC_SDARI = sdari | (uint32_t)((exmc_count << SDARI_ARINTV_OFFSET) & EXMC_SDARI_ARINTV);
        }

        exmc_dev->state = HAL_EXMC_SDRAM_STATE_READY;
        HAL_UNLOCK(exmc_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      set the number of successive auto-refresh command
    \param[in]  exmc_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  exmc_number: the number of successive Auto-refresh cycles will be send, 1~15
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_sdram_autorefresh_number_set(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t exmc_number)
{
    uint32_t sdcmd;
    uint32_t timeout = SDRAM_FREASH_TIMEOUT;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_dev) {
        HAL_DEBUGE("pointer [exmc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_SDRAM_STATE_READY == exmc_dev->state) {
        HAL_LOCK(exmc_dev);
        exmc_dev->state = HAL_EXMC_SDRAM_STATE_BUSY;

        sdcmd      = EXMC_SDCMD & (~EXMC_SDCMD_NARF);
        EXMC_SDCMD = sdcmd | (uint32_t)((exmc_number << SDCMD_NARF_OFFSET) & EXMC_SDCMD_NARF);

        while((hals_exmc_sdram_flag_get(EXMC_SDRAM_FLAG_NREADY) != RESET) && (timeout > 0U)) {
            timeout--;
        }

        /* SDRAM controller timeout */
        if(0U == timeout) {
            retval = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      return the sdram state
    \param[in]  exmc_dev: specify the region of NOR/PSRAM bank
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_DEVICEx(x=0,1)
    \param[out] none
    \retval     hal_exmc_sdram_state_enum details refer to gd32h7xx_hal_sdio_sdram.h
*/
hal_exmc_sdram_state_enum hal_exmc_sdram_state_get(hal_exmc_sdram_dev_struct *exmc_dev)
{
  return exmc_dev->state;
}

/*!
    \brief      enable read sample function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_exmc_sdram_readsample_enable(void)
{
    EXMC_SDRSCTL |= EXMC_SDRSCTL_RSEN;
}

/*!
    \brief      disable read sample function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_exmc_sdram_readsample_disable(void)
{
    EXMC_SDRSCTL &= (uint32_t)(~EXMC_SDRSCTL_RSEN);
}

/*!
    \brief      configure the delayed sample clock of read data
    \param[in]  delay_cell: SDRAM the delayed sample clock of read data
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_x_DELAY_CELL(x=0..15): select the clock after x delay cell
    \param[in]  extra_clk: sample cycle of read data
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_READSAMPLE_0_EXTRACK: add 0 extra CK_EXMC cycle to the read data sample clock besides the delay chain
      \arg        EXMC_SDRAM_READSAMPLE_1_EXTRACK: add 1 extra CK_EXMC cycle to the read data sample clock besides the delay chain
    \param[out] none
    \retval     none
*/
void hals_exmc_sdram_readsample_config(uint32_t delay_cell, uint32_t extra_clk)
{
    uint32_t sdrsctl = 0U;

    /* reset the bits */
    sdrsctl = EXMC_SDRSCTL & (~(EXMC_SDRSCTL_SDSC | EXMC_SDRSCTL_SSCR));
    /* set the bits */
    sdrsctl |= (uint32_t)(delay_cell | extra_clk);
    EXMC_SDRSCTL = sdrsctl;
}

/*!
    \brief      configure the SDRAM memory command
    \param[in]  exmc_sdram_command: pointer to EXMC SDRAM command parameter structure
                  mode_register_content:
                  auto_refresh_number: EXMC_SDRAM_AUTO_REFLESH_x_SDCLK, x=1~15
                  bank_select: EXMC_SDRAM_DEVICE0_SELECT, EXMC_SDRAM_DEVICE1_SELECT, EXMC_SDRAM_DEVICE0_1_SELECT
                  command: EXMC_SDRAM_NORMAL_OPERATION, EXMC_SDRAM_CLOCK_ENABLE, EXMC_SDRAM_PRECHARGE_ALL,
                           EXMC_SDRAM_AUTO_REFRESH, EXMC_SDRAM_LOAD_MODE_REGISTER, EXMC_SDRAM_SELF_REFRESH,
                           EXMC_SDRAM_POWERDOWN_ENTRY
    \param[out] none
    \retval     none
*/
void hals_exmc_sdram_command_config(hal_exmc_sdram_command_parameter_struct *exmc_sdram_command)
{
    /* configure command register */
    EXMC_SDCMD = (uint32_t)(exmc_sdram_command->command |\
                            exmc_sdram_command->bank_select |\
                            exmc_sdram_command->auto_refresh_number |\
                            (exmc_sdram_command->mode_register_content << SDCMD_MRC_OFFSET));
}

/*!
    \brief      get the status of SDRAM device0 or device1
    \param[in]  exmc_sdram_device: specify the SDRAM device
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_DEVICEx(x=0,1)
    \param[out] none
    \retval     uint32_t:the status of SDRAM device
*/
uint32_t hals_exmc_sdram_bankstatus_get(uint32_t exmc_sdram_device)
{
    uint32_t sdstat = 0U;

    if(EXMC_SDRAM_DEVICE0 == exmc_sdram_device) {
        sdstat = ((uint32_t)(EXMC_SDSTAT & EXMC_SDSDAT_STA0) >> SDSTAT_STA0_OFFSET);
    } else {
        sdstat = ((uint32_t)(EXMC_SDSTAT & EXMC_SDSDAT_STA1) >> SDSTAT_STA1_OFFSET);
    }

    return sdstat;
}

/*!
    \brief      configure NOR/PSRAM and SDRAM remap
    \param[in]  bank_remap: NOR/PSRAM and SDRAM map address
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK_REMAP_DEFAULT: default mapping
      \arg        EXMC_BANK_NORPSRAM_SDRAM_SWAP: NOR/PSRAM bank and SDRAM device 0 swapped
    \param[out] none
    \retval     none
*/
void hals_exmc_norsram_sdram_remap_config(uint32_t bank_remap)
{
    /* reset BKREMAP bits */
    EXMC_SNCTL(EXMC_BANK0_NORSRAM_REGION0) &= (uint32_t)(~EXMC_SNCTL_BKREMAP);

    EXMC_SNCTL(EXMC_BANK0_NORSRAM_REGION0) |= bank_remap;
}

/*!
    \brief      get EXMC flag status
    \param[in]  flag: EXMC status and flag
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SDRAM_FLAG_REFRESH: refresh error interrupt flag
      \arg        EXMC_SDRAM_FLAG_NREADY: not ready status
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_exmc_sdram_flag_get(uint32_t flag)
{
    uint32_t status = EXMC_SDSTAT;
    FlagStatus retval = RESET;
    if((status & flag) != (uint32_t)flag) {
        /* flag is reset */
        retval = RESET;
    } else {
        /* flag is set */
        retval = SET;
    }
    return retval;
}

/*!
    \brief      clear EXMC flag status
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_exmc_sdram_flag_clear(void)
{
    /* SDRAM device0 or device1 */
    EXMC_SDARI |= EXMC_SDARI_REC;

}

/*!
    \brief      enable EXMC interrupt
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_exmc_sdram_interrupt_enable(void)
{
    /* SDRAM device0 or device1 */
    EXMC_SDARI |= EXMC_SDARI_REIE;
}

/*!
    \brief      disable EXMC interrupt
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_exmc_sdram_interrupt_disable(void)
{
    /* SDRAM device0 or device1 */
    EXMC_SDARI &= ~EXMC_SDARI_REIE;
}

/*!
    \brief      get EXMC interrupt flag
    \param[in]  none
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_exmc_sdram_interrupt_flag_get(void)
{
    uint32_t reg_value = 0x00000000U;
    uint32_t interrupt_enable = 0x00000000U;
    uint32_t interrupt_status = 0x00000000U;
    FlagStatus retval = RESET;
    /* SDRAM device0 or device1 */
    reg_value = EXMC_SDARI;
    interrupt_status = (EXMC_SDSTAT & EXMC_SDSDAT_REIF);
    interrupt_enable = (reg_value & EXMC_SDRAM_INT_FLAG_REFRESH);

    if((interrupt_enable) && (interrupt_status)) {
        /* interrupt flag is set */
        retval = SET;
    } else {
        /* interrupt flag is reset */
        retval = RESET;
    }
    return retval;
}

/*!
    \brief      clear EXMC interrupt flag
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_exmc_sdram_interrupt_flag_clear(void)
{
    /* SDRAM device0 or device1 */
    EXMC_SDARI |= EXMC_SDARI_REC;
}

/*!
    \brief      EXMC SDRAM process complete callback.
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _sdram_dma_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_exmc_sdram_dev_struct *p_sdram;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sdram = (hal_exmc_sdram_dev_struct *)p_dma->p_periph;

    /* Disable the MDMA channel */
    hals_dma_channel_disable(p_dma->dma_periph, p_dma->channel);

    /* Update the SRAM controller state */
    p_sdram->state = HAL_EXMC_SDRAM_STATE_READY;

    if(NULL != p_sdram->dma_transfer_complete) {
        p_sdram->dma_transfer_complete(p_sdram);
    } else {
            /* do nothing */
    }
}

/*!
    \brief      EXMC SDRAM  process complete callback.
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _sdram_dma_complete_protected(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_exmc_sdram_dev_struct *p_sdram;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sdram = (hal_exmc_sdram_dev_struct *)p_dma->p_periph;

    /* Disable the MDMA channel */
    hals_dma_channel_disable(p_dma->dma_periph, p_dma->channel);

    /* Update the SRAM controller state */
    p_sdram->state = HAL_EXMC_SDRAM_STATE_PROTECTED;

    if(NULL != p_sdram->dma_transfer_complete) {
        p_sdram->dma_transfer_complete(p_sdram);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      EXMC SDRAM error callback.
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _sdram_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_exmc_sdram_dev_struct *p_sdram;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sdram = (hal_exmc_sdram_dev_struct *)p_dma->p_periph;

    /* Disable the MDMA channel */
    hals_dma_channel_disable(p_dma->dma_periph, p_dma->channel);

    /* Update the SRAM controller state */
    p_sdram->state = HAL_EXMC_SDRAM_STATE_ERROR;

    if(NULL != p_sdram->dma_transfer_error) {
        p_sdram->dma_transfer_error(p_sdram);
    } else {
            /* do nothing */
    }
}
