/*!
    \file    gd32h7xx_hal_exmc_nor.c
    \brief   EXMC NOR driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* get EXMC NOR/PSRAM address */
static uint32_t _exmc_nor_address_get(uint32_t norsram_region);

/*!
    \brief      initialize EXMC NOR/SRAM region
    \param[in]  nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  nor_init: pointer to EXMC NOR initialization structure
                This parameter contains following fields:
                norsram_region: select the region of EXMC NOR/PSRAM region
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK0_NORSRAM_REGION0: EXMC bank0 NOR/SRAM region0
      \arg        EXMC_BANK0_NORSRAM_REGION1: EXMC bank0 NOR/SRAM region1
      \arg        EXMC_BANK0_NORSRAM_REGION2: EXMC bank0 NOR/SRAM region2
      \arg        EXMC_BANK0_NORSRAM_REGION3: EXMC bank0 NOR/SRAM region3
                bank_remap: select EXMC NOR/PSRAM remap region
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK_REMAP_DEFAULT: default bank remap configuration
      \arg        EXMC_BANK_NORPSRAM_SDRAM_SWAP: NOR/PSRAM and SDRAM bank swap
                consecutive_clock:configure consecutive clock mode
                only one parameter can be selected which is shown as below:
      \arg        EXMC_CLOCK_SYN_MODE: the clock is generated only during synchronous access
      \arg        EXMC_CLOCK_UNCONDITIONALLY: the clock is generated unconditionally
                asyn_access_mode: asynchronous access mode
                only one parameter can be selected which is shown as below:
      \arg        EXMC_ACCESS_MODE_A: access mode A
      \arg        EXMC_ACCESS_MODE_B: access mode B
      \arg        EXMC_ACCESS_MODE_C: access mode C
      \arg        EXMC_ACCESS_MODE_D: access mode D
                syn_data_latency: configure the data latency
                only one parameter can be selected which is shown as below:
      \arg        EXMC_DATALAT_x_CLK(x=2..17): data latency of first burst access is x EXMC_CLK
                syn_clk_division: configure the clock divide ratio
                only one parameter can be selected which is shown as below:
      \arg        EXMC_SYN_CLOCK_RATIO_DISABLE: clock divide ratio disable
      \arg        EXMC_SYN_CLOCK_RATIO_x_CLK(x=2..16): EXMC_CLK period = x*CK_EXMC period
                asyn_address_holdtime: 0 - 15, configure the address hold time
                bus_latency: 0 - 15, configure the bus latency
                asyn_address_setuptime: 1 - 15, configure the address setup time
                asyn_data_setuptime: 1 - 255, configure the data setup time
                write_asyn_access_mode: write asynchronous access mode
                only one parameter can be selected which is shown as below:
      \arg        EXMC_ACCESS_MODE_A: access mode A
      \arg        EXMC_ACCESS_MODE_B: access mode B
      \arg        EXMC_ACCESS_MODE_C: access mode C
      \arg        EXMC_ACCESS_MODE_D: access mode D
                write_asyn_address_setuptime: 1 - 15, write configure the address setup time
                write_bus_latency: 0 - 15, write configure the bus latency
                write_asyn_data_setuptime: 1 - 255, write configure the data setup time
                write_asyn_address_holdtime: 0 - 15, write configure the address hold time
                write_mode: the write mode
                only one parameter can be selected which is shown as below:
      \arg        EXMC_ASYN_WRITE: asynchronous write mode
      \arg        EXMC_SYN_WRITE: synchronous write mode
                extended_mode: enable or disable the extended mode
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the extended mode
      \arg        DISABLE: disable the extended mode
                asyn_wait: enable or disable the asynchronous wait function
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the asynchronous wait function
      \arg        DISABLE: disable the asynchronous wait function
                nwait_signal: enable or disable the NWAIT signal while in synchronous bust mode
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the NWAIT signal
      \arg        DISABLE: disable the NWAIT signal
                memory_write: enable or disable the write operation
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the write operation
      \arg        DISABLE: disable the write operation
                nwait_config: ENWAIT signal configuration
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NWAIT_CONFIG_BEFORE: NWAIT signal is active one clock cycle before the wait state
      \arg        EXMC_NWAIT_CONFIG_DURING: NWAIT signal is active during wait state
                nwait_polarity: specify the polarity of NWAIT signal from memory
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NWAIT_POLARITY_LOW: NWAIT active low
      \arg        XMC_NWAIT_POLARITY_HIGH: NWAIT active high
                burst_mode: enable or disable the burst mode
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable the burst mode
      \arg        DISABLE: disable the burst mode
                databus_width: specify the data bus width of external memory
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NOR_DATABUS_WIDTH_8B: NOR data width is 8 bits
      \arg        EXMC_NOR_DATABUS_WIDTH_16B: NOR data width is 16 bits
      \arg        EXMC_NOR_DATABUS_WIDTH_32B: NOR data width is 32 bits
                memory_type: specify the type of external memory
                only one parameter can be selected which is shown as below:
      \arg        EXMC_MEMORY_TYPE_SRAM: SRAM memory type
      \arg        EXMC_MEMORY_TYPE_PSRAM: PSRAM memory type
      \arg        EXMC_MEMORY_TYPE_NOR: NOR memory type
                address_data_mux: specify whether the data bus and address bus are multiplexed
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: address/data multiplexed
      \arg        DISABLE: address/data not multiplexed
                cram_page_size: cram page size
                only one parameter can be selected which is shown as below:
      \arg        EXMC_CRAM_AUTO_SPLIT: the clock is generated only during synchronous access
      \arg        EXMC_CRAM_PAGE_SIZE_128_BYTES: page size is 128 bytes
      \arg        EXMC_CRAM_PAGE_SIZE_256_BYTES: page size is 256 bytes
      \arg        EXMC_CRAM_PAGE_SIZE_512_BYTES: page size is 512 bytes
      \arg        EXMC_CRAM_PAGE_SIZE_1024_BYTES: page size is 1024 bytes
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_init(hal_exmc_nor_dev_struct *nor_dev, hal_exmc_nor_init_struct *nor_init)
{
    uint32_t snctl = 0U, sntcfg = 0U, snwtcfg = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == nor_dev) || (NULL == nor_init)) {
        HAL_DEBUGE("parameter [nor_dev] or [nor_init ] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(nor_dev);
    nor_dev->state = HAL_EXMC_NOR_STATE_RESET;
    nor_dev->nor_region = nor_init->norsram_region;

    /* disable exmc nor region */
    EXMC_SNCTL(nor_init->norsram_region) &= (uint32_t)(~EXMC_SNCTL_NRBKEN);

    snctl   = EXMC_SNCTL(nor_init->norsram_region);
    sntcfg  = EXMC_SNTCFG(nor_init->norsram_region);
    snwtcfg = EXMC_SNWTCFG(nor_init->norsram_region);

    if(EXMC_BANK0_NORSRAM_REGION0 == nor_init->norsram_region) {
        snctl &= (uint32_t)(~(EXMC_SNCTL_BKREMAP | EXMC_SNCTL_CCK));
    }
    snctl &= (uint32_t)(~(EXMC_SNCTL_NREN | EXMC_SNCTL_NRTP | EXMC_SNCTL_NRW | EXMC_SNCTL_SBRSTEN | EXMC_SNCTL_NRWTPOL | \
                          EXMC_SNCTL_NRWTCFG | EXMC_SNCTL_WEN | EXMC_SNCTL_NRWTEN | EXMC_SNCTL_EXMODEN | \
                          EXMC_SNCTL_ASYNCWTEN | EXMC_SNCTL_SYNCWR | EXMC_SNCTL_NRMUX | EXMC_SNCTL_CPS));

    /* configure control register */
    snctl = (uint32_t)((nor_init->bank_remap & EXMC_SNCTL_BKREMAP) | (nor_init->consecutive_clock & EXMC_SNCTL_CCK) | \
                      ((nor_init->address_data_mux << SNCTL_NRMUX_OFFSET) & EXMC_SNCTL_NRMUX) | \
                      (nor_init->memory_type & EXMC_SNCTL_NRTP) | ((nor_init->databus_width) & EXMC_SNCTL_NRW) | \
                      ((nor_init->burst_mode << SNCTL_SBRSTEN_OFFSET) & EXMC_SNCTL_SBRSTEN) | \
                      (nor_init->nwait_polarity & EXMC_SNCTL_NRWTPOL) | (nor_init->nwait_config & EXMC_SNCTL_NRWTCFG) | \
                      ((nor_init->memory_write << SNCTL_WREN_OFFSET) & EXMC_SNCTL_WEN) | \
                      ((nor_init->nwait_signal << SNCTL_NRWTEN_OFFSET) & EXMC_SNCTL_NRWTEN) | \
                      ((nor_init->extended_mode << SNCTL_EXMODEN_OFFSET) & EXMC_SNCTL_EXMODEN) | \
                      ((nor_init->asyn_wait << SNCTL_ASYNCWAITEN_OFFSET) & EXMC_SNCTL_ASYNCWTEN) | \
                      (nor_init->cram_page_size & EXMC_SNCTL_CPS) | (nor_init->write_mode & EXMC_SNCTL_SYNCWR));

    /* nor flash access enable */
    if(EXMC_MEMORY_TYPE_NOR == nor_init->memory_type) {
        snctl |= (uint32_t)EXMC_SNCTL_NREN;
    }

    /* configure timing register */
    sntcfg = (uint32_t)((nor_init->asyn_access_mode & EXMC_SNTCFG_ASYNCMOD) | SNTCFG_SYN_DATA_LATENCY(nor_init->syn_data_latency) | \
                        SNTCFG_SYN_CLK_DIVISION(nor_init->syn_clk_division) | \
                        ((nor_init->asyn_address_holdtime << SNTCFG_AHLD_OFFSET) & EXMC_SNTCFG_AHLD) | \
                        ((nor_init->bus_latency << SNTCFG_BUSLAT_OFFSET) & EXMC_SNTCFG_BUSLAT) | \
                        SNTCFG_ASYN_ADDR_SETUPTIME(nor_init->asyn_address_setuptime) | \
                        ((nor_init->asyn_data_setuptime << SNTCFG_DSET_OFFSET) & EXMC_SNTCFG_DSET));

    /* for extended mode, configure write timing */
    if(ENABLE == nor_init->extended_mode) {
        snwtcfg = (uint32_t)((nor_init->write_asyn_access_mode & EXMC_SNWTCFG_WASYNCMOD) | \
                             SNTCFG_WRITE_ASYN_ADDR_SETUPTIME(nor_init->write_asyn_address_setuptime) | \
                             ((nor_init->write_bus_latency << SNTCFG_BUSLAT_OFFSET) & EXMC_SNWTCFG_WBUSLAT) | \
                             ((nor_init->write_asyn_data_setuptime << SNTCFG_DSET_OFFSET) & EXMC_SNWTCFG_WDSET) | \
                             ((nor_init->write_asyn_address_holdtime << SNTCFG_AHLD_OFFSET) & EXMC_SNWTCFG_WAHLD));
    } else {
        snwtcfg = BANK0_SNWTCFG_RESET;
    }

    EXMC_SNCTL(nor_init->norsram_region)   = snctl;
    EXMC_SNTCFG(nor_init->norsram_region)  = sntcfg;
    EXMC_SNWTCFG(nor_init->norsram_region) = snwtcfg;

    /* enable exmc nor region */
    EXMC_SNCTL(nor_init->norsram_region) |= (uint32_t)EXMC_SNCTL_NRBKEN;

    nor_dev->state = HAL_EXMC_NOR_STATE_READY;
    HAL_UNLOCK(nor_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the NOR structure with default values
    \param[in]  hal_struct_type: The type of the struct to initialize
                only one parameters can be selected which are shown as below:
      \arg        HAL_EXMC_NOR_INIT_STRUCT: NOR initialization structure
      \arg        HAL_EXMC_NOR_DEV_STRUCT: NOR device structure
    \param[out] p_struct: pointer to NOR structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_struct_init(hal_exmc_nor_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_EXMC_NOR_INIT_STRUCT:
        ((hal_exmc_nor_init_struct *)p_struct)->norsram_region    = EXMC_BANK0_NORSRAM_REGION0;
        ((hal_exmc_nor_init_struct *)p_struct)->bank_remap        = EXMC_BANK_REMAP_DEFAULT;
        ((hal_exmc_nor_init_struct *)p_struct)->consecutive_clock = EXMC_CLOCK_SYN_MODE;
        ((hal_exmc_nor_init_struct *)p_struct)->address_data_mux  = ENABLE;
        ((hal_exmc_nor_init_struct *)p_struct)->memory_type       = EXMC_MEMORY_TYPE_NOR;
        ((hal_exmc_nor_init_struct *)p_struct)->databus_width     = EXMC_NOR_DATABUS_WIDTH_16B;
        ((hal_exmc_nor_init_struct *)p_struct)->burst_mode        = DISABLE;
        ((hal_exmc_nor_init_struct *)p_struct)->nwait_polarity    = EXMC_NWAIT_POLARITY_LOW;
        ((hal_exmc_nor_init_struct *)p_struct)->nwait_config      = EXMC_NWAIT_CONFIG_BEFORE;
        ((hal_exmc_nor_init_struct *)p_struct)->memory_write      = ENABLE;
        ((hal_exmc_nor_init_struct *)p_struct)->nwait_signal      = ENABLE;
        ((hal_exmc_nor_init_struct *)p_struct)->extended_mode     = DISABLE;
        ((hal_exmc_nor_init_struct *)p_struct)->asyn_wait         = DISABLE;
        ((hal_exmc_nor_init_struct *)p_struct)->cram_page_size    = EXMC_CRAM_AUTO_SPLIT;
        ((hal_exmc_nor_init_struct *)p_struct)->write_mode        = EXMC_ASYN_WRITE;

        ((hal_exmc_nor_init_struct *)p_struct)->asyn_access_mode       = EXMC_ACCESS_MODE_A;
        ((hal_exmc_nor_init_struct *)p_struct)->syn_data_latency       = 15U;
        ((hal_exmc_nor_init_struct *)p_struct)->syn_clk_division       = 15U;
        ((hal_exmc_nor_init_struct *)p_struct)->bus_latency            = 15U;
        ((hal_exmc_nor_init_struct *)p_struct)->asyn_data_setuptime    = 255U;
        ((hal_exmc_nor_init_struct *)p_struct)->asyn_address_holdtime  = 15U;
        ((hal_exmc_nor_init_struct *)p_struct)->asyn_address_setuptime = 15U;

        ((hal_exmc_nor_init_struct *)p_struct)->write_asyn_access_mode       = EXMC_WACCESS_MODE_A;
        ((hal_exmc_nor_init_struct *)p_struct)->write_bus_latency            = 15U;
        ((hal_exmc_nor_init_struct *)p_struct)->write_asyn_data_setuptime    = 255U;
        ((hal_exmc_nor_init_struct *)p_struct)->write_asyn_address_holdtime  = 15U;
        ((hal_exmc_nor_init_struct *)p_struct)->write_asyn_address_setuptime = 15U;
        break;
    case HAL_EXMC_NOR_DEV_STRUCT:
        ((hal_exmc_nor_dev_struct *)p_struct)->nor_region = EXMC_BANK0_NORSRAM_REGION0;
        ((hal_exmc_nor_dev_struct *)p_struct)->mutex      = HAL_MUTEX_UNLOCKED;
        ((hal_exmc_nor_dev_struct *)p_struct)->state      = HAL_EXMC_NOR_STATE_RESET;
        ((hal_exmc_nor_dev_struct *)p_struct)->priv       = NULL;
        break;
    case HAL_EXMC_NOR_CONFIG_STRUCT:
        ((hal_exmc_nor_cfg_struct *)p_struct)->cfi_1      = 0U;
        ((hal_exmc_nor_cfg_struct *)p_struct)->cfi_2      = 0U;
        ((hal_exmc_nor_cfg_struct *)p_struct)->cfi_3      = 0U;
        ((hal_exmc_nor_cfg_struct *)p_struct)->cfi_4      = 0U;
        break;
    case HAL_EXMC_NOR_ID_STRUCT:
        ((hal_exmc_nor_id_struct *)p_struct)->manufacturer_code = 0U;
        ((hal_exmc_nor_id_struct *)p_struct)->device_code1      = 0U;
        ((hal_exmc_nor_id_struct *)p_struct)->device_code2      = 0U;
        ((hal_exmc_nor_id_struct *)p_struct)->device_code3      = 0U;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize EXMC NOR/SRAM region
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_deinit(hal_exmc_nor_dev_struct *exmc_nor_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_nor_dev) {
        HAL_DEBUGE("pointer [nor_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset the registers */
    EXMC_SNCTL(exmc_nor_dev->nor_region)   = BANK0_SNCTL_RESET;
    EXMC_SNTCFG(exmc_nor_dev->nor_region)  = BANK0_SNTCFG_RESET;
    EXMC_SNWTCFG(exmc_nor_dev->nor_region) = BANK0_SNWTCFG_RESET;
    exmc_nor_dev->mutex                    = HAL_MUTEX_UNLOCKED;
    exmc_nor_dev->state                    = HAL_EXMC_NOR_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      read the NOR memory id
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  nor_id: nor id
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_id_read(hal_exmc_nor_dev_struct *exmc_nor_dev, hal_exmc_nor_id_struct *nor_id)
{
    uint32_t nor_addr = 0U;
    uint32_t state;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_nor_dev) || (NULL == nor_id)) {
        HAL_DEBUGE("pointer [exmc_nor_dev] or [nor_id] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    state = exmc_nor_dev->state;
    if(HAL_EXMC_NOR_STATE_READY == state) {
        HAL_LOCK(exmc_nor_dev);
        nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_READID_ADD_1ST), CMD_READID_DATA_1ST);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_READID_ADD_2ND), CMD_READID_DATA_2ND);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_READID_ADD_3RD), CMD_READID_DATA_3RD);
        /* read NOR Flash ID */
        nor_id->manufacturer_code = *(__IO uint16_t *) ADDR_SHIFT(nor_addr, 0x0000U);
        nor_id->device_code1 = *(__IO uint16_t *) ADDR_SHIFT(nor_addr, 0x0001U);
        nor_id->device_code2 = *(__IO uint16_t *) ADDR_SHIFT(nor_addr, 0x000EU);
        nor_id->device_code3 = *(__IO uint16_t *) ADDR_SHIFT(nor_addr, 0x000FU);
        HAL_UNLOCK(exmc_nor_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      NOR memory return to read mode
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_readmode_return(hal_exmc_nor_dev_struct *exmc_nor_dev)
{
    uint32_t nor_addr = 0U;
    uint32_t state;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_nor_dev) {
        HAL_DEBUGE("pointer [exmc_nor_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    state = exmc_nor_dev->state;
    if(HAL_EXMC_NOR_STATE_READY == state) {
        HAL_LOCK(exmc_nor_dev);
        nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);
        NOR_WRITE(nor_addr, CMD_RESET_DATA_3RD);
        HAL_UNLOCK(exmc_nor_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      NOR memory read
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  read_addr: NOR flash internal address to read from
    \param[out] data: pointer to read data
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_read(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t read_addr, uint16_t *data)
{
    uint32_t nor_addr = 0U;
    uint32_t state;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_nor_dev) || (NULL == data)) {
        HAL_DEBUGE("pointer [exmc_nor_dev]  or [data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    state = exmc_nor_dev->state;
    if(HAL_EXMC_NOR_STATE_BUSY == state) {
        return HAL_ERR_BUSY;
    } else if(HAL_EXMC_NOR_STATE_READY == state) {
        /* process locked */
        HAL_LOCK(exmc_nor_dev);
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_BUSY;

        nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_READ_ADD_1ST), CMD_READ_DATA_1ST);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_READ_ADD_2ND), CMD_READ_DATA_2ND);
        NOR_WRITE((uint32_t)(nor_addr + read_addr), CMD_READ_DATA_3RD);

        *data = (uint16_t)(*(__IO uint16_t *)(uint32_t)(nor_addr + read_addr));

        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_READY;
        HAL_UNLOCK(exmc_nor_dev);
    } else {
        return HAL_ERR_VAL;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      NOR memory write
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  write_addr: NOR flash internal address to write from
    \param[out] data: pointer to the data to write
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_write(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t write_addr, uint16_t data)
{
    uint32_t nor_addr = 0U;
    uint32_t state = exmc_nor_dev->state;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_nor_dev)) {
        HAL_DEBUGE("pointer [exmc_nor_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_NOR_STATE_READY == state) {
        /* process locked */
        HAL_LOCK(exmc_nor_dev);
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_BUSY;
        nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_WRITE_ADD_1ST), CMD_WRITE_DATA_1ST);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_WRITE_ADD_2ND), CMD_WRITE_DATA_2ND);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_WRITE_ADD_3RD), CMD_WRITE_DATA_3RD);
        NOR_WRITE((uint32_t)(nor_addr + write_addr), data);

        if(HAL_NOR_STATUS_SUCCESS != hal_exmc_nor_status_get(exmc_nor_dev, NOR_PROGRAM_TIMEOUT)) {
            retval = HAL_ERR_HARDWARE;
        } else {
            /* do nothing */
        }

        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_READY;
        /* process unlocked */
        HAL_UNLOCK(exmc_nor_dev);
    } else {
        retval =  HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      NOR memory read buffer
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  read_addr: read data address for nor
    \param[in]  len: read data length
    \param[out] data: read data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_buffer_read(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t read_addr, uint16_t *data, uint32_t len)
{
    uint32_t nor_addr = 0U;
    uint32_t state;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_nor_dev) || (NULL == data)) {
        HAL_DEBUGE("pointer [exmc_nor_dev] or [data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    state = exmc_nor_dev->state;
    if(HAL_EXMC_NOR_STATE_READY == state) {
        HAL_LOCK(exmc_nor_dev);
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_BUSY;
        nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_READ_ADD_1ST), CMD_READ_DATA_1ST);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_READ_ADD_2ND), CMD_READ_DATA_2ND);
        NOR_WRITE((uint32_t)(nor_addr + read_addr), CMD_READ_DATA_3RD);

        for(; len != 0x00U; len--) {
            /* read a halfword from the nor flash */
            *data++ = (uint16_t)(*(__IO uint16_t *)(uint32_t)(nor_addr + read_addr));
            read_addr += 2U;
        }

        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_READY;
        HAL_UNLOCK(exmc_nor_dev);
    } else {
        retval = HAL_ERR_VAL;
    }

    return retval;
}

/*!
    \brief      NOR memory write buffer
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  write_addr: write data address for nor
    \param[in]  data: write data buffer
    \param[in]  len: write data length
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_buffer_write(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t write_addr, uint16_t *data, uint32_t len)
{
    uint32_t nor_addr = 0U;
    uint16_t *p_currentaddress;
    const uint16_t *p_endaddress;
    uint32_t state;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_nor_dev) || (NULL == data)) {
        HAL_DEBUGE("pointer [exmc_nor_dev] or [data] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(len == 0U) {
        HAL_DEBUGE("parameter [len] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    state = exmc_nor_dev->state;
    if(HAL_EXMC_NOR_STATE_READY == state) {
        HAL_LOCK(exmc_nor_dev);
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_BUSY;
        nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);

        /* Initialize variables */
        p_currentaddress = (uint16_t *)(uint32_t)(nor_addr + write_addr);
        p_endaddress = (uint16_t *)(uint32_t)(nor_addr + write_addr + (2U * (len - 1U)));

        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_WRITE_ADD_1ST), CMD_WRITE_DATA_1ST);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_WRITE_ADD_2ND), CMD_WRITE_DATA_2ND);

        /* Write Buffer Load Command */
        NOR_WRITE((uint32_t)(nor_addr + write_addr), (uint16_t)0x0025);
        NOR_WRITE((uint32_t)(nor_addr + write_addr), (uint16_t)(len - 1U));

        while(p_currentaddress <= p_endaddress) {
            NOR_WRITE(p_currentaddress, *data);
            data++;
            p_currentaddress++;
        }

        NOR_WRITE((uint32_t)(nor_addr + write_addr), (uint16_t)0x0029);

        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_READY;
        HAL_UNLOCK(exmc_nor_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      NOR memory erase block
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  block_addr: erase block data address for nor
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_block_erase(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t block_addr)
{
    uint32_t nor_addr = 0U;
    uint32_t state;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_nor_dev) {
        HAL_DEBUGE("pointer [exmc_nor_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    state = exmc_nor_dev->state;

    if(HAL_EXMC_NOR_STATE_READY == state) {
        HAL_LOCK(exmc_nor_dev);
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_BUSY;
        nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_BLOCKERASE_ADD_1ST), CMD_BLOCKERASE_DATA_1ST);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_BLOCKERASE_ADD_2ND), CMD_BLOCKERASE_DATA_2ND);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_BLOCKERASE_ADD_3RD), CMD_BLOCKERASE_DATA_3RD);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_BLOCKERASE_ADD_4TH), CMD_BLOCKERASE_DATA_4TH);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_BLOCKERASE_ADD_5TH), CMD_BLOCKERASE_DATA_5TH);
        NOR_WRITE((uint32_t)(nor_addr  + block_addr), CMD_BLOCKERASE_DATA_6TH);

        if(HAL_NOR_STATUS_SUCCESS != hal_exmc_nor_status_get(exmc_nor_dev, NOR_BLOCKERASE_TIMEOUT)) {
            retval = HAL_ERR_HARDWARE;
        } else {
            /* do nothing */
        }

        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_READY;
        HAL_UNLOCK(exmc_nor_dev);
    } else {
        retval = HAL_ERR_VAL;
    }

    return retval;
}

/*!
    \brief      NOR memory erase chip
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_chip_erase(hal_exmc_nor_dev_struct *exmc_nor_dev)
{
    uint32_t nor_addr = 0U;
    uint32_t state;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_nor_dev) {
        HAL_DEBUGE("pointer [exmc_nor_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    state = exmc_nor_dev->state;
    if(HAL_EXMC_NOR_STATE_READY == state) {
        HAL_LOCK(exmc_nor_dev);
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_BUSY;
        nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_CHIPERASE_ADD_1ST), CMD_CHIPERASE_DATA_1ST);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_CHIPERASE_ADD_2ND), CMD_CHIPERASE_DATA_2ND);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_CHIPERASE_ADD_3RD), CMD_CHIPERASE_DATA_3RD);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_CHIPERASE_ADD_4TH), CMD_CHIPERASE_DATA_4TH);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_CHIPERASE_ADD_5TH), CMD_CHIPERASE_DATA_5TH);
        NOR_WRITE(ADDR_SHIFT(nor_addr, CMD_CHIPERASE_ADD_6TH), CMD_CHIPERASE_DATA_6TH);

        if(HAL_NOR_STATUS_SUCCESS != hal_exmc_nor_status_get(exmc_nor_dev, NOR_CHIPERASE_TIMEOUT)) {
            retval = HAL_ERR_HARDWARE;
        } else {
            /* do nothing */
        }

        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_READY;
        HAL_UNLOCK(exmc_nor_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read NOR flash CFI IDs
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] p_cfi: pointer to NOR CFI IDs structure
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_cfi_read(hal_exmc_nor_dev_struct *exmc_nor_dev, hal_exmc_nor_cfg_struct *p_cfi)
{
    uint32_t nor_addr;
    uint32_t state = exmc_nor_dev->state;
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_nor_dev) || (NULL == p_cfi)) {
        HAL_DEBUGE("pointer [exmc_nor_dev] or [p_cfi] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NOR controller state */
    if(HAL_EXMC_NOR_STATE_READY == state) {
        /* process locked */
        HAL_LOCK(exmc_nor_dev);
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_BUSY;

        /* select the NOR device address */
        nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);

        /* send read CFI query command */
        NOR_WRITE(ADDR_SHIFT(nor_addr, (uint16_t)0x0055), (uint16_t)0x0098);

        /* read the NOR CFI information */
        p_cfi->cfi_1 = *(__IO uint16_t *)ADDR_SHIFT(nor_addr, (uint16_t)0x0061);
        p_cfi->cfi_2 = *(__IO uint16_t *)ADDR_SHIFT(nor_addr, (uint16_t)0x0062);
        p_cfi->cfi_3 = *(__IO uint16_t *)ADDR_SHIFT(nor_addr, (uint16_t)0x0063);
        p_cfi->cfi_4 = *(__IO uint16_t *)ADDR_SHIFT(nor_addr, (uint16_t)0x0064);

        /* check the NOR controller state */
        exmc_nor_dev->state = (hal_exmc_nor_state_enum)state;

        /* process unlocked */
        HAL_UNLOCK(exmc_nor_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      enable nor write operation
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_write_operation_enable(hal_exmc_nor_dev_struct *exmc_nor_dev)
{
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_nor_dev)) {
        HAL_DEBUGE("pointer [exmc_nor_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_NOR_STATE_PROTECTED == exmc_nor_dev->state) {
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_BUSY;
        /* enable write operation */
        EXMC_SNCTL(exmc_nor_dev->nor_region) |= EXMC_SNCTL_WEN;
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_READY;
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      disable nor write operation
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nor_write_operation_disable(hal_exmc_nor_dev_struct *exmc_nor_dev)
{
    int32_t retval = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == exmc_nor_dev)) {
        HAL_DEBUGE("pointer [exmc_nor_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_EXMC_NOR_STATE_READY == exmc_nor_dev->state) {
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_BUSY;
        /* disable write operation */
        EXMC_SNCTL(exmc_nor_dev->nor_region) &= ~(EXMC_SNCTL_WEN);
        exmc_nor_dev->state = HAL_EXMC_NOR_STATE_PROTECTED;
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      return the nor flash state
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_exmc_nor_state_enum details refer to gd32h7xx_hal_exmc_nor.h
*/
hal_exmc_nor_state_enum hal_exmc_nor_state_get(hal_exmc_nor_dev_struct *exmc_nor_dev)
{
    return exmc_nor_dev->state;
}

/*!
    \brief      return the nor flash operation status
    \param[in]  exmc_nor_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  time_out: NOR flash programming timeout
    \param[out] none
    \retval     hal_exmc_nor_status_enum details refer to gd32h7xx_hal_exmc_nor.h
*/
hal_exmc_nor_status_enum hal_exmc_nor_status_get(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t time_out)
{
    hal_exmc_nor_status_enum status = HAL_NOR_STATUS_ONGOING;
    uint16_t val1 = 0x00U, val2 = 0x00U;
    uint32_t tickstart = 0U;
    uint32_t nor_addr = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == exmc_nor_dev) {
        HAL_DEBUGE("pointer [exmc_nor_dev] address is invalid");
        return HAL_NOR_STATUS_ERROR;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    nor_addr = _exmc_nor_address_get(exmc_nor_dev->nor_region);

    tickstart = hal_sys_basetick_count_get();
    /* get the nor flash operation status */
    while((HAL_NOR_STATUS_TIMEOUT != status) && (HAL_NOR_STATUS_SUCCESS != status)) {
        if(SET == hal_sys_basetick_timeout_check(tickstart, time_out)) {
            status = HAL_NOR_STATUS_TIMEOUT;
        }
        /* read DQ6 and DQ5 */
        val1 = *(__IO uint16_t *)(nor_addr);
        val2 = *(__IO uint16_t *)(nor_addr);

        if((val1 & 0x0040U) == (val2 & 0x0040U)) {
            status = HAL_NOR_STATUS_SUCCESS;
        } else {
            if((val1 & 0x0020U) != 0x0020U) {
                status = HAL_NOR_STATUS_ONGOING;
            } else {
                /* do nothing */
            }

            val1 = *(__IO uint16_t *)(nor_addr);
            val2 = *(__IO uint16_t *)(nor_addr);

            if((val1 & 0x0040U) == (val2 & 0x0040U)) {
                status = HAL_NOR_STATUS_SUCCESS;
            } else if((val1 & 0x0020U) == 0x0020U) {
                status = HAL_NOR_STATUS_ERROR;
            } else {
            /* do nothing */
            }
        }
    }

    return status;
}

/*!
    \brief      get EXMC NOR/PSRAM address
    \param[in]  norsram_region: specify the region of NOR/PSRAM Bank
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK0_NORSRAM_REGIONx(x=0..3): EXMC BANK0 REGIONx
    \param[out] none
    \retval     nor address
*/
static uint32_t _exmc_nor_address_get(uint32_t norsram_region)
{
    uint32_t nor_addr = 0U;

    switch (norsram_region) {
    case EXMC_BANK0_NORSRAM_REGION0:
        nor_addr = BANK0_NOR0_ADDR;
        break;
    case EXMC_BANK0_NORSRAM_REGION1:
        nor_addr = BANK0_NOR1_ADDR;
        break;
    case EXMC_BANK0_NORSRAM_REGION2:
        nor_addr = BANK0_NOR2_ADDR;
        break;
    case EXMC_BANK0_NORSRAM_REGION3:
    default:
        nor_addr = BANK0_NOR3_ADDR;
        break;
    }

    return nor_addr;
}
