/*!
    \file    gd32h7xx_hal_edout.c
    \brief   EDOUT driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* EDOUT register bit offset */
#define LOC_LOCMAX_MIN             ((uint32_t)0x0000000FU) /*!< LOCMAX fields minimum value */
#define LOC_LOCMAX_STEP            ((uint32_t)0x00000004U) /*!< LOCMAX fields step value */
#define OCNT_PDC_OFFSET            ((uint32_t)0x00000010U) /*!< bit offset of PDC in EDOUT_OCNT */
#define ZCR_ZOWH_OFFSET            ((uint32_t)0x00000010U) /*!< bit offset of ZOWH in EDOUT_ZCR */
#define PHASE_DIFFERENCE_COUNT_MIN ((uint32_t)0x00000002U) /*!< The minimum of the phase difference */

/*!
    \brief      initialize EDOUT
    \param[in]  edout_dev: EDOUT device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_edout_init: EDOUT pointer to a hal_edout_init_struct
                  b_phase_pol:
                  only one parameter can be selected which is shown as below:
      \arg          EDOUT_POL_POSITIVE
      \arg          EDOUT_POL_NEGATIVE
                  z_phase_mode:
                  only one parameter can be selected which is shown as below:
      \arg          EDOUT_Z_OUTPUT_MODE0
      \arg          EDOUT_Z_OUTPUT_MODE1
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_edout_init(hal_edout_dev_struct *edout_dev, hal_edout_init_struct *p_edout_init)
{
/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == p_edout_init) || (NULL == edout_dev)) {
        HAL_DEBUGE("pointer [p_edout_init] or [edout_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    edout_dev->state = HAL_EDOUT_STATE_BUSY;

    /* set B-phase active polarity */
    hals_edout_polarity_config(p_edout_init->b_phase_pol);

    /* set the maximum location value for one rotation */
    hals_edout_max_location_value_config(p_edout_init->max_location - 1U);

    /* set the current location value */
    hals_edout_current_location_config(p_edout_init->current_location);

    /* update the output counter */
    hal_edout_output_counter_update(p_edout_init->edge_count, (uint16_t)p_edout_init->phase_difference);

    /* reset EDOUT output */
    EDOUT_ENABLE &= ~EDOUT_ENABLE_EDOUTEN;

    /* configure EDOUT output */
    EDOUT_ENABLE |= EDOUT_ENABLE_EDOUTEN;

    /* configure Z-phase output Mode */
    hal_edout_z_output_mode_config(p_edout_init->z_phase_mode);

    /* configure Z-phase output start location and width */
    hal_edout_z_output_start_loc_and_width_config(p_edout_init->z_phase_start_position, p_edout_init->z_phase_width);

    /*set EDOUT state*/
    edout_dev->state = HAL_EDOUT_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      deinitialize EDOUT
    \param[in]  edout_dev: EDOUT device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_edout_deinit(hal_edout_dev_struct *edout_dev)
{
/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == edout_dev) {
        HAL_DEBUGE("pointer [edout_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    hal_rcu_periph_reset_enable(RCU_EDOUTRST);
    hal_rcu_periph_reset_disable(RCU_EDOUTRST);
    edout_dev->state = HAL_EDOUT_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the EDOUT structure
    \param[in]  hal_struct_type: refer to <hal_edout_struct_type_enum>
                    only one parameter can be selected which is shown as below:
        \arg        HAL_EDOUT_INIT_STRUCT
        \arg        HAL_EDOUT_DEV_STRUCT
    \param[out] p_struct: point to EDOUT structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_edout_struct_init(hal_edout_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_EDOUT_INIT_STRUCT:
        /* initialize EDOUT initialization structure with the default values */
        ((hal_edout_init_struct *)p_struct)->b_phase_pol            = EDOUT_POL_POSITIVE;
        ((hal_edout_init_struct *)p_struct)->max_location           = LOC_LOCMAX_MIN;
        ((hal_edout_init_struct *)p_struct)->phase_difference       = PHASE_DIFFERENCE_COUNT_MIN;
        ((hal_edout_init_struct *)p_struct)->edge_count             = 0;
        ((hal_edout_init_struct *)p_struct)->resolution             = EDOUT_USER_SET_RESOLUTION;
        ((hal_edout_init_struct *)p_struct)->user_resolution        = 0U;
        ((hal_edout_init_struct *)p_struct)->current_location       = 0U;
        ((hal_edout_init_struct *)p_struct)->z_phase_mode           = EDOUT_Z_OUTPUT_MODE0;
        ((hal_edout_init_struct *)p_struct)->z_phase_width          = 0U;
        ((hal_edout_init_struct *)p_struct)->z_phase_start_position = 0U;
        break;
    case HAL_EDOUT_DEV_STRUCT:
        /* initialize EDOUT device information structure with the default values */
        ((hal_edout_dev_struct *)p_struct)->state = HAL_EDOUT_STATE_READY;
        ((hal_edout_dev_struct *)p_struct)->mutex = HAL_MUTEX_UNLOCKED;
        break;
    default:
        HAL_DEBUGE("parameter [struct_type] value is undefine");
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}

/*!
    \brief      update the output counter, used to set the phase difference and the number of edges for the next update period
    \param[in]  edge_count:value range is -32768~32767, positive means clockwise rotation, negative means
                           counter-clockwise rotation
    \param[in]  phase_difference:value range is 2~65535, in units of PCLK
    \param[out] none
    \retval     none
*/
void hal_edout_output_counter_update(int16_t edge_count, uint16_t phase_difference)
{
    EDOUT_OCNT = ((uint32_t)edge_count & EDOUT_OCNT_EDGC) | ((uint32_t)phase_difference << OCNT_PDC_OFFSET);
}

/*!
    \brief      get the current location value
    \param[in]  none
    \param[out] none
    \retval     current location value, 0~locmax (locmax is the LOCMAX bit fields value of EDOUT_LOC register)
*/
uint16_t hal_edout_current_location_get(void)
{
    return (uint16_t)EDOUT_LCNT;
}

/*!
    \brief      get the maximum location value
    \param[in]  none
    \param[out] none
    \retval     maximum location value
*/
uint16_t hal_edout_max_location_value_get(void)
{
    return (uint16_t)EDOUT_LOC;
}

/*!
    \brief      configure Z-phase output mode
    \param[in]  z_phase_mode: Z-phase output Mode
                only one parameter can be selected which is shown as below:
      \arg        EDOUT_Z_OUTPUT_MODE0: output according to the current location
      \arg        EDOUT_Z_OUTPUT_MODE1: output according to the number of edges
    \param[out] none
    \retval     none
*/
void hal_edout_z_output_mode_config(uint32_t z_phase_mode)
{
    /* reset and set the Z-phase output z_phase_mode */
    EDOUT_ZCR &= ~EDOUT_ZCR_ZOMD;
    EDOUT_ZCR |= z_phase_mode;
}

/*!
    \brief      configure Z-phase output start location and width
    \param[in]  z_phase_start_position: Z-phase output start location
    \param[in]  z_phase_width: Z-phase output width
    \param[out] none
    \retval     none
*/
void hal_edout_z_output_start_loc_and_width_config(uint32_t z_phase_start_position, uint32_t z_phase_width)
{
    /* reset the Z-phase output start location and output width */
    EDOUT_ZCR &= ~(EDOUT_ZCR_ZOSP | EDOUT_ZCR_ZOWH);
    /* set the Z-phase output start location and output width */
    EDOUT_ZCR |= (z_phase_start_position & EDOUT_ZCR_ZOSP) | ((z_phase_width << ZCR_ZOWH_OFFSET) & EDOUT_ZCR_ZOWH);
}

/*!
    \brief      enable EDOUT
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_edout_enable(void)
{
    EDOUT_ENABLE |= EDOUT_ENABLE_EDOUTEN;
}

/*!
    \brief      disable EDOUT
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_edout_disable(void)
{
    EDOUT_ENABLE &= ~EDOUT_ENABLE_EDOUTEN;
}

/*!
    \brief      get EDOUT state
    \param[in]  edout_dev: EDOUT device information structure
    \param[out] none
    \retval     EDOUT state: refer to hal_edout_state_enum
*/
hal_edout_state_enum hal_edout_state_get(hal_edout_dev_struct *edout_dev)
{
    return edout_dev->state;
}

/*!
    \brief      set B-phase active polarity
    \param[in]  b_phase_pol: the active polarity of the B-phase output signal selection
                only one parameter can be selected which is shown as below:
      \arg        EDOUT_POL_POSITIVE: active polarity is positive
      \arg        EDOUT_POL_NEGATIVE: active polarity is negative
    \param[out] none
    \retval     none
*/
void hals_edout_polarity_config(uint32_t b_phase_pol)
{
    /* setting the polarity is required when edout is disabled */
    if(SET == (EDOUT_ENABLE & EDOUT_ENABLE_EDOUTEN)) {
        EDOUT_ENABLE &= ~EDOUT_ENABLE_EDOUTEN;
    }
    EDOUT_CTL &= ~EDOUT_CTL_POL;
    EDOUT_CTL |= b_phase_pol;
}

/*!
    \brief      set the maximum location value for one rotation
    \param[in]  max_location: (max_location+1) must be a multiple of four between 16~65536
    \param[out] none
    \retval     none
*/
void hals_edout_max_location_value_config(uint32_t max_location)
{
    EDOUT_LOC &= (uint32_t)(~EDOUT_LOC_LOCMAX);

    /* check the maximum location value */
    if(LOC_LOCMAX_MIN > max_location) {
        max_location = LOC_LOCMAX_MIN;
    }

    while(0U != ((max_location + 1U) % LOC_LOCMAX_STEP)) {
        max_location++;
    }

    EDOUT_LOC |= max_location;
}

/*!
    \brief      set the current location value
    \param[in]  current_location: current location value, 0~locmax
                (locmax is the LOCMAX bit fields value of EDOUT_LOC register)
    \param[out] none
    \retval     none
*/
void hals_edout_current_location_config(uint32_t current_location)
{
    /* setting the current location is required when edout is disabled */
    if(SET == (EDOUT_ENABLE & EDOUT_ENABLE_EDOUTEN)) {
        EDOUT_ENABLE &= ~EDOUT_ENABLE_EDOUTEN;
    }
    EDOUT_LCNT &= (uint32_t)(~EDOUT_LCNT_LOCCNT);
    EDOUT_LCNT |= current_location;
}
