/*!
    \file    gd32h7xx_hal_dci.c
    \brief   DCI driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define DCI_WINDOW_HORIZONTAL_START_SIZE (0x3FFFU)   /* DCI window horizontal start size */
#define DCI_WINDOW_VERTICAL_START_SIZE   (0x1FFFU)   /* DCI window vertical start size */
#define DCI_WINDOW_HORIZONTAL_MAX_SIZE   (0x3FFFU)   /* DCI window horizontal max size */
#define DCI_WINDOW_VERTICAL_MAX_SIZE     (0x3FFFU)   /* DCI window vertical max size */

/* DCI DMA full transfer complete callback function */
static void _dci_dma_full_transfer_complete(void *dma);
/* DCI DMA half transfer complete callback function */
static void _dci_dma_half_transfer_complete(void *dma);
/* DCI DMA error callback function */
static void _dci_dma_error(void *dma);
/* DCI vsync event callback function */
static void __vsync_event_callback(hal_dci_dev_struct *dci_dev);
/* DCI line event callback function */
static void __line_event_callback(hal_dci_dev_struct *dci_dev);
/* DCI ccir error callback function */
static void __ccir_error_callback(hal_dci_dev_struct *dci_dev);
/* DCI frame event callback function */
static void __frame_event_callback(hal_dci_dev_struct *dci_dev);
/* DCI error callback function */
static void __error_callback(hal_dci_dev_struct *dci_dev);

/*!
    \brief      reset the DCI peripheral
    \param[in]  dci_dev: DCI device information structure
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_deinit(hal_dci_dev_struct *dci_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    dci_dev->state = HAL_DCI_STATE_BUSY;

    /*reset dci*/
    hal_rcu_periph_reset_enable(RCU_DCIRST);
    hal_rcu_periph_reset_disable(RCU_DCIRST);

    /* change dci state */
    dci_dev->state = HAL_DCI_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the DCI structure with the default values
    \param[in]  hal_struct_type: type of DCI structure for initialization
                only one  parameters can be selected which are shown as below:
      \arg        HAL_DCI_INIT_STRUCT: initialization structure
      \arg        HAL_DCI_DEV_STRUCT: device structure
      \arg        HAL_DCI_IRQ_STRUCT: interrupt structure
      \arg        HAL_DCI_BUFFER_STRUCT: buffer structure
      \arg        HAL_DCI_IRQ_USER_CALLBACK_STRUCT: user callback structure
      \arg        HAL_DCI_DMA_HANDLE_CB_STRUCT: DMA handle callback structure
    \param[out]  p_struct: point to the structure to be initialized
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_struct_init(hal_dci_struct_type_enum hal_struct_type, void *p_struct)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* initialize the structure based on type: handle different DCI struct init cases */
    switch(hal_struct_type) {
    case HAL_DCI_INIT_STRUCT:
        /* set default values for DCI init config: operating mode, interface format, etc. */
        ((hal_dci_init_struct *)p_struct)->operating_mode                    = DCI_SYNCHRO_HARDWARE;
        ((hal_dci_init_struct *)p_struct)->interface_format                  = DCI_INTERFACE_FORMAT_8BITS;
        ((hal_dci_init_struct *)p_struct)->clock_polarity                    = DCI_CK_POLARITY_FALLING;
        ((hal_dci_init_struct *)p_struct)->vsync_polarity                    = DCI_VSYNC_POLARITY_LOW;
        ((hal_dci_init_struct *)p_struct)->hsync_polarity                    = DCI_HSYNC_POLARITY_LOW;
        ((hal_dci_init_struct *)p_struct)->capture_mode                      = DCI_CAPTURE_MODE_CONTINUOUS;
        ((hal_dci_init_struct *)p_struct)->frame_rate                        = DCI_FRAME_RATE_ALL;
        ((hal_dci_init_struct *)p_struct)->jpeg_mode                         = DCI_JPEG_MODE_DISABLE;
        ((hal_dci_init_struct *)p_struct)->window_mode                       = DCI_WINDOW_MODE_DISABLE;
        ((hal_dci_init_struct *)p_struct)->frame_start                       = 0U;
        ((hal_dci_init_struct *)p_struct)->frame_end                         = 0U;
        ((hal_dci_init_struct *)p_struct)->line_start                        = 0U;
        ((hal_dci_init_struct *)p_struct)->line_end                          = 0U;
        ((hal_dci_init_struct *)p_struct)->ccir_mode                         = CCIR_PROGRESSIVE_MODE;
        ((hal_dci_init_struct *)p_struct)->ex_synchro                        = DCI_EX_VSYNC_DISABLE;
        ((hal_dci_init_struct *)p_struct)->auto_correct                      = DCI_AUTO_CORRECT_DISABLE;
        ((hal_dci_init_struct *)p_struct)->vertical_pos                      = 0U;
        ((hal_dci_init_struct *)p_struct)->horizon_pos                       = 0U;
        ((hal_dci_init_struct *)p_struct)->vertical_size                     = 0U;
        ((hal_dci_init_struct *)p_struct)->horizon_size                      = 0U;
        break;
    case HAL_DCI_DEV_STRUCT:
        /* set default values for DCI device struct: periph, irq handles, buffer, state */
        ((hal_dci_dev_struct *)p_struct)->periph                             = DCI;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.frame_event_handle         = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.vsync_event_handle         = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.line_event_handle          = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.embedded_sync_handle       = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.f0_handle                  = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.overrun_handle             = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.f1_handle                  = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.charge_failed_handle       = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.ccir_error_handle          = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_irq.error_handle               = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_dma.full_transcom_handle       = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_dma.error_handle               = NULL;
        ((hal_dci_dev_struct *)p_struct)->dci_dma.half_transcom_handle       = NULL;
        ((hal_dci_dev_struct *)p_struct)->rx_buffer.buffer                   = NULL;
        ((hal_dci_dev_struct *)p_struct)->rx_buffer.length                   = 0U;
        ((hal_dci_dev_struct *)p_struct)->rx_buffer.remain                   = 0U;
        ((hal_dci_dev_struct *)p_struct)->state                              = HAL_DCI_STATE_READY;
        ((hal_dci_dev_struct *)p_struct)->error_state                        = HAL_DCI_ERROR_NONE;
        ((hal_dci_dev_struct *)p_struct)->mutex                              = HAL_MUTEX_UNLOCKED;
        ((hal_dci_dev_struct *)p_struct)->vsync_event_callback               = NULL;
        ((hal_dci_dev_struct *)p_struct)->line_event_callback                = NULL;
        ((hal_dci_dev_struct *)p_struct)->ccir_error_callback                = NULL;
        ((hal_dci_dev_struct *)p_struct)->frame_event_callback               = NULL;
        ((hal_dci_dev_struct *)p_struct)->error_callback                     = NULL;
        break;
    case HAL_DCI_IRQ_INIT_STRUCT:
        /* set default (NULL) for DCI IRQ handlers */
        ((hal_dci_irq_struct *)p_struct)->frame_event_handle                 = NULL;
        ((hal_dci_irq_struct *)p_struct)->vsync_event_handle                 = NULL;
        ((hal_dci_irq_struct *)p_struct)->line_event_handle                  = NULL;
        ((hal_dci_irq_struct *)p_struct)->embedded_sync_handle               = NULL;
        ((hal_dci_irq_struct *)p_struct)->f0_handle                          = NULL;
        ((hal_dci_irq_struct *)p_struct)->overrun_handle                     = NULL;
        ((hal_dci_irq_struct *)p_struct)->f1_handle                          = NULL;
        ((hal_dci_irq_struct *)p_struct)->charge_failed_handle               = NULL;
        ((hal_dci_irq_struct *)p_struct)->ccir_error_handle                  = NULL;
        ((hal_dci_irq_struct *)p_struct)->error_handle                       = NULL;
        break;
    case HAL_DCI_BUFFER_STRUCT:
        /* set default values for DCI buffer struct: buffer, length, remain */
        ((hal_dci_buffer_struct *)p_struct)->buffer                          = NULL;
        ((hal_dci_buffer_struct *)p_struct)->length                          = 0U;
        ((hal_dci_buffer_struct *)p_struct)->remain                          = 0U;
        break;
    case HAL_DCI_DMA_HANDLE_CB_STRUCT:
    /* set default values for DCI DMA handle callback struct: dma_handle_cb */
        ((hal_dci_dma_handle_cb_struct *)p_struct)->full_transcom_handle     = NULL;
        ((hal_dci_dma_handle_cb_struct *)p_struct)->half_transcom_handle     = NULL;
        ((hal_dci_dma_handle_cb_struct *)p_struct)->error_handle             = NULL;
        break;
    case HAL_DCI_IRQ_USER_CALLBACK_STRUCT:
        /* set default values for DCI user callback struct */
        ((hal_dci_irq_user_callback_struct *)p_struct)->vsync_event_callback = NULL;
        ((hal_dci_irq_user_callback_struct *)p_struct)->line_event_callback  = NULL;
        ((hal_dci_irq_user_callback_struct *)p_struct)->ccir_error_callback  = NULL;
        ((hal_dci_irq_user_callback_struct *)p_struct)->frame_event_callback = NULL;
        ((hal_dci_irq_user_callback_struct *)p_struct)->error_callback       = NULL;
        break;
    default:
        /* print prompt information */
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        /* invalid struct type: log error and set return val */
        retval = HAL_ERR_VAL;
        break;
    }

    return retval;
}

/*!
    \brief      initialize DCI registers
    \param[in]  dci_dev: DCI device information structure
    \param[in]  p_init: DCI parameter initialization structure
                  operating_mode:
                  only one parameter can be selected which is shown as below:
      \arg          DCI_SYNCHRO_HARDWARE, DCI_SYNCHRO_EMBEDDED, DCI_SYNCHRO_CCIR656
                  interface_format:
                  only one parameter can be selected which is shown as below:
      \arg          DCI_INTERFACE_FORMAT_8BITS, DCI_INTERFACE_FORMAT_10BITS,
      \arg          DCI_INTERFACE_FORMAT_12BITS, DCI_INTERFACE_FORMAT_14BITS
                  clock_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          DCI_CK_POLARITY_FALLING, DCI_CK_POLARITY_RISING
                  vsync_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          DCI_VSYNC_POLARITY_LOW, DCI_VSYNC_POLARITY_HIGH
                  hsync_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          DCI_HSYNC_POLARITY_LOW, DCI_HSYNC_POLARITY_HIGH
                  frame_rate:
                  only one parameter can be selected which is shown as below:
      \arg          DCI_FRAME_RATE_ALL, DCI_FRAME_RATE_1_2, DCI_FRAME_RATE_1_4
                  jpeg_mode :
                  only one parameter can be selected which is shown as below:
      \arg          DCI_JPEG_MODE_DISABLE, DCI_JPEG_MODE_ENABLE
                  window_mode:
                  only one parameter can be selected which is shown as below:
      \arg          DCI_WINDOW_MODE_DISABLE, DCI_WINDOW_MODE_ENABLE
      \arg        frame_start: frame start code in embedded synchronous mode, ranges from 0 to 255
      \arg        frame_end: frame end code in embedded synchronous mode, ranges from 0 to 255
      \arg        line_start: line start code in embedded synchronous mode, ranges from 0 to 255
      \arg        line_end: line end code in embedded synchronous mode, ranges from 0 to 255
                  ccir_mode:
                  only one parameter can be selected which is shown as below:
      \arg        CCIR_PROGRESSIVE_MODE, CCIR_INTERLACE_MODE
                  ex_synchro:
                  only one parameter can be selected which is shown as below:
      \arg        DCI_EX_VSYNC_DISABLE, DCI_EX_VSYNC_ENABLE
                  auto_correct:
                  only one parameter can be selected which is shown as below:
      \arg        DCI_AUTO_CORRECT_DISABLE, DCI_AUTO_CORRECT_ENABLE
      \arg        vertical_pos: window vertical start position, ranges from 0 to 8191
      \arg        horizon_pos: window horizontal start position, ranges from 0 to 16383
      \arg        vertical_size: window vertical size position, ranges from 0 to 16383
      \arg        horizon_size: window horizontal size position, ranges from 0 to 16383
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_init(hal_dci_dev_struct *dci_dev, hal_dci_init_struct *p_init)
{
    uint32_t reg   = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == dci_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [dci_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    dci_dev->state = HAL_DCI_STATE_BUSY;

    /* disable capture function and DCI */
    DCI_CTL &= ~(DCI_CTL_CAP | DCI_CTL_DCIEN);

    /* configure DCI parameter */
    reg |= p_init->operating_mode;
    reg |= p_init->interface_format;
    reg |= p_init->clock_polarity;
    reg |= p_init->vsync_polarity;
    reg |= p_init->hsync_polarity;
    reg |= p_init->capture_mode;
    reg |= p_init->frame_rate;
    reg |= p_init->jpeg_mode;
    reg |= p_init->window_mode;
    reg |= p_init->ccir_mode;
    reg |= p_init->ex_synchro;
    reg |= p_init->auto_correct;

    DCI_CTL = reg;

    /* configure synchronous codes unmask in embedded synchronous mode */
    DCI_SC = ((uint32_t)p_init->frame_start | ((uint32_t)p_init->line_start << 8U) | \
              ((uint32_t)p_init->line_end << 16U) | ((uint32_t)p_init->frame_end << 24U));

    /* configure DCI cropping window */
    DCI_CWSPOS = (p_init->horizon_pos | (p_init->vertical_pos << 16U));
    DCI_CWSZ   = (p_init->horizon_size | (p_init->vertical_size << 16U));

    dci_dev->state = HAL_DCI_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      start DCI request
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dci_mode: capture mode enable or disable
                only one parameter can be selected which is shown as below:
      \arg        CAPTURE_MODE_ENABLE: enable capture
      \arg        CAPTURE_MODE_DISABLE: disable capture
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_start(hal_dci_dev_struct *dci_dev, uint32_t dci_mode)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == dci_dev)) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((CAPTURE_MODE_ENABLE != dci_mode) && (CAPTURE_MODE_DISABLE != dci_mode)) {
        HAL_DEBUGE("param [dci_mode] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DCI */
    HAL_LOCK(dci_dev);

    dci_dev->state = HAL_DCI_STATE_BUSY;

    if(CAPTURE_MODE_ENABLE == dci_mode) {
        /* enable capture */
        DCI_CTL |= DCI_CTL_CAP;
    } else {
        /* disable capture */
        DCI_CTL &= ~DCI_CTL_CAP;
    }

    hals_dci_enable();

    dci_dev->state = HAL_DCI_STATE_READY;

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      start DCI DMA request
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dci_mode: capture mode snapshot or continuous grab
                only one parameter can be selected which is shown as below:
      \arg        DCI_CAPTURE_MODE_CONTINUOUS: continuous capture mode
      \arg        DCI_CAPTURE_MODE_SNAPSHOT: snapshot capture mode
    \param[in]  pdata: the source memory buffer address1
    \param[in]  length: the number of data to be transferred from source to destination
    \param[in]  dmacb: DCI DMA transfer complete interrupt callback function structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY,
                            HAL_ERR_HARDWARE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_start_dma(hal_dci_dev_struct *dci_dev, uint32_t dci_mode, \
                          uint32_t *pdata, uint16_t length, \
                          hal_dci_dma_handle_cb_struct *dmacb)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == dci_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [dci_dev] or [pdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DCI_CAPTURE_MODE_SNAPSHOT != dci_mode) && (DCI_CAPTURE_MODE_CONTINUOUS != dci_mode)) {
        HAL_DEBUGE("param [dci_mode] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DCI */
    HAL_LOCK(dci_dev);

    if(HAL_DCI_STATE_BUSY == dci_dev->state) {
        /* state busy */
        HAL_DEBUGE("DCI is busy");
        retval = HAL_ERR_BUSY;
    } else {
        /* set busy state */
        dci_dev->state = HAL_DCI_STATE_BUSY;

        if(DCI_CAPTURE_MODE_SNAPSHOT == dci_mode) {
            /* set snapshot mode */
            DCI_CTL |= DCI_CTL_SNAP;
        } else {
            /* set continuous mode */
            DCI_CTL &= ~DCI_CTL_SNAP;
        }

        /* clear interrupt callback */
        dci_dev->dci_dma.full_transcom_handle = NULL;
        dci_dev->dci_dma.half_transcom_handle = NULL;
        dci_dev->dci_dma.error_handle         = NULL;
    }

    /* register the DMA  callback */
    if((NULL != dmacb) && (HAL_ERR_NONE == retval)) {
        /* full transfer complete */
        if(NULL != dmacb->full_transcom_handle) {
            dci_dev->dci_dma.full_transcom_handle = dmacb->full_transcom_handle;
        } else {
            /* Do nothing */
        }

        /* half transfer complete */
        if(NULL != dmacb->half_transcom_handle) {
            dci_dev->dci_dma.half_transcom_handle = dmacb->half_transcom_handle;
        } else {
            /* Do nothing */
        }

        /* dma underflow error */
        if(NULL != dmacb->error_handle) {
            dci_dev->dci_dma.error_handle = dmacb->error_handle;
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    if(HAL_ERR_NONE == retval) {
        /* set dci dma irq callback */
        dma_irq.full_finish_handle = _dci_dma_full_transfer_complete;
        dma_irq.half_finish_handle = _dci_dma_half_transfer_complete;
        dma_irq.error_handle       = _dci_dma_error;

        /* start the DMA channel */
        retval = hal_dma_start_interrupt(dci_dev->p_dma_dci, (uint32_t)&DCI_DATA, (uint32_t)pdata, length, &dma_irq);

        if (HAL_ERR_NONE != retval) {
            HAL_DEBUGE("start DCI DMA failed");
            retval = HAL_ERR_HARDWARE;
            dci_dev->error_state = HAL_DCI_ERROR_DMA;
        } else {
            /* enable DCI */
            hals_dci_enable();
            /* enable DCI capture */
            hals_dci_capture_enable();
        }
    } else {
        /* Do nothing */
    }

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return retval;
}

/*!
    \brief      stop DCI request
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_stop(hal_dci_dev_struct *dci_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable DCI DMA mode */
    HAL_LOCK(dci_dev);

    dci_dev->state = HAL_DCI_STATE_BUSY;

    /* disable DCI */
    hals_dci_disable();
    /* disable DCI capture */
    hals_dci_capture_disable();

    dci_dev->state = HAL_DCI_STATE_READY;

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop DCI DMA request
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_stop_dma(hal_dci_dev_struct *dci_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable DCI DMA mode */
    HAL_LOCK(dci_dev);

    dci_dev->state = HAL_DCI_STATE_BUSY;

    /* disable the DMA channel */
    hal_dma_stop_interrupt(dci_dev->p_dma_dci);
    /* disable DCI */
    hals_dci_disable();
    /* disable DCI capture */
    hals_dci_capture_disable();

    dci_dev->state = HAL_DCI_STATE_READY;

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      start DCI under error interrupt
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: DCI under error interrupt callback function structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[in]  p_user_func: user function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_start_interrupt(hal_dci_dev_struct *dci_dev, hal_dci_irq_struct *p_irq, \
                                hal_dci_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == dci_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [dci_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DCI */
    HAL_LOCK(dci_dev);

    dci_dev->state = HAL_DCI_STATE_BUSY;

    /* default callback*/
    dci_dev->dci_irq.vsync_event_handle = (hal_irq_handle_cb)__vsync_event_callback;
    dci_dev->dci_irq.line_event_handle  = (hal_irq_handle_cb)__line_event_callback;
    dci_dev->dci_irq.ccir_error_handle  = (hal_irq_handle_cb)__ccir_error_callback;
    dci_dev->dci_irq.frame_event_handle = (hal_irq_handle_cb)__frame_event_callback;
    dci_dev->dci_irq.error_handle       = (hal_irq_handle_cb)__error_callback;

    dci_dev->vsync_event_callback       = NULL;
    dci_dev->line_event_callback        = NULL;
    dci_dev->ccir_error_callback        = NULL;
    dci_dev->frame_event_callback       = NULL;
    dci_dev->error_callback             = NULL;

    /* set user interrupt callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->vsync_event_callback) {
            dci_dev->vsync_event_callback = (void *)p_user_func->vsync_event_callback;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->line_event_callback) {
            dci_dev->line_event_callback = (void *)p_user_func->line_event_callback;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->ccir_error_callback) {
            dci_dev->ccir_error_callback = (void *)p_user_func->ccir_error_callback;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->frame_event_callback) {
            dci_dev->frame_event_callback = (void *)p_user_func->frame_event_callback;
        } else {
            /* Do nothing */
        }

        if(NULL != p_user_func->error_callback) {
            dci_dev->error_callback = (void *)p_user_func->error_callback;
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* end of frame interrupt */
    if(NULL != p_irq->frame_event_handle) {
        dci_dev->dci_irq.frame_event_handle = p_irq->frame_event_handle;
        hals_dci_interrupt_enable(DCI_INT_EF);
    } else {
        hals_dci_interrupt_disable(DCI_INT_EF);
    }

    /* vsync error interrupt */
    if(NULL != p_irq->vsync_event_handle) {
        dci_dev->dci_irq.vsync_event_handle = p_irq->vsync_event_handle;
        hals_dci_interrupt_enable(DCI_INT_VSYNC);
    } else {
        hals_dci_interrupt_disable(DCI_INT_VSYNC);
    }

    /* end of line  interrupt */
    if(NULL != p_irq->line_event_handle) {
        dci_dev->dci_irq.line_event_handle = p_irq->line_event_handle;
        hals_dci_interrupt_enable(DCI_INT_EL);
    } else {
        hals_dci_interrupt_disable(DCI_INT_EL);
    }

    /* set overrun handle interrupt */
    if(NULL != p_irq->overrun_handle) {
        dci_dev->dci_irq.overrun_handle = p_irq->overrun_handle;
        hals_dci_interrupt_enable(DCI_INT_OVR);
    } else {
        hals_dci_interrupt_disable(DCI_INT_OVR);
    }

    /* set embedded synchronous error interrupt */
    if(NULL != p_irq->embedded_sync_handle) {
        dci_dev->dci_irq.embedded_sync_handle = p_irq->embedded_sync_handle;
        hals_dci_interrupt_enable(DCI_INT_ESE);
    } else {
        hals_dci_interrupt_disable(DCI_INT_ESE);
    }

    /* set CCIR field 0 interrupt */
    if(NULL != p_irq->f0_handle) {
        dci_dev->dci_irq.f0_handle = p_irq->f0_handle;
        hals_dci_interrupt_enable(DCI_INT_F0);
    } else {
        hals_dci_interrupt_disable(DCI_INT_F0);
    }

    /* set CCIR field 1 interrupt */
    if(NULL != p_irq->f1_handle) {
        dci_dev->dci_irq.f1_handle = p_irq->f1_handle;
        hals_dci_interrupt_enable(DCI_INT_F1);
    } else {
        hals_dci_interrupt_disable(DCI_INT_F1);
    }

    /* set charge of field interrupt */
    if(NULL != p_irq->charge_failed_handle) {
        dci_dev->dci_irq.charge_failed_handle = p_irq->charge_failed_handle;
        hals_dci_interrupt_enable(DCI_INT_COF);
    } else {
        hals_dci_interrupt_disable(DCI_INT_COF);
    }

    /* set ccir error interrupt */
    if(NULL != p_irq->ccir_error_handle) {
        dci_dev->dci_irq.ccir_error_handle = p_irq->ccir_error_handle;
        hals_dci_interrupt_enable(DCI_INT_CCE);
    } else {
        hals_dci_interrupt_disable(DCI_INT_CCE);
    }

    /* clear all flags */
    hals_dci_interrupt_flag_clear(DCI_INT_FLAG_CCE | DCI_INT_FLAG_COF | DCI_INT_FLAG_F1 | DCI_INT_FLAG_F0 | \
                                  DCI_INT_FLAG_ESE | DCI_INT_FLAG_OVR | DCI_INT_FLAG_EF | DCI_INT_FLAG_VSYNC | \
                                  DCI_INT_FLAG_EL);

    hals_dci_capture_enable();
    hals_dci_enable();

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    /* change DCI state */
    dci_dev->state = HAL_DCI_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      stop DCI under error interrupt
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_stop_interrupt(hal_dci_dev_struct *dci_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DCI */
    HAL_LOCK(dci_dev);

    /* disable DCI */
    hals_dci_disable();
    hals_dci_capture_disable();

    /* disable DCI IT */
    hals_dci_interrupt_disable(DCI_INT_CCE | DCI_INT_COF | DCI_INT_F1 | DCI_INT_F0 | DCI_INT_ESE | DCI_INT_OVR | \
                               DCI_INT_EF | DCI_INT_VSYNC | DCI_INT_EL);
    /* change DCI state */
    dci_dev->state = HAL_DCI_STATE_READY;

    /* reset the interrupt handle */
    dci_dev->dci_irq.frame_event_handle   = NULL;
    dci_dev->dci_irq.vsync_event_handle   = NULL;
    dci_dev->dci_irq.line_event_handle    = NULL;
    dci_dev->dci_irq.embedded_sync_handle = NULL;
    dci_dev->dci_irq.f0_handle            = NULL;
    dci_dev->dci_irq.overrun_handle       = NULL;
    dci_dev->dci_irq.f1_handle            = NULL;
    dci_dev->dci_irq.charge_failed_handle = NULL;
    dci_dev->dci_irq.ccir_error_handle    = NULL;
    /* clear all flags */
    hals_dci_interrupt_flag_clear(DCI_INT_FLAG_CCE | DCI_INT_FLAG_COF | DCI_INT_FLAG_F1 | DCI_INT_FLAG_F0 | \
                                  DCI_INT_FLAG_ESE | DCI_INT_FLAG_OVR | DCI_INT_FLAG_EF | DCI_INT_FLAG_VSYNC | \
                                  DCI_INT_FLAG_EL);

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function and control interrupt enable/disable
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to DCI interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_irq_handle_set(hal_dci_dev_struct *dci_dev, hal_dci_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == dci_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [dci_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DCI */
    HAL_LOCK(dci_dev);

    /* disable DCI channel */
    hals_dci_disable();

    /* set the handler pointer and enable cframe event interrupt */
    if(NULL != p_irq->frame_event_handle) {
        /* set frame event handle */
        dci_dev->dci_irq.frame_event_handle = p_irq->frame_event_handle;
        /* enable frame event interrupt */
        hals_dci_interrupt_enable(DCI_INT_EF);
    } else {
        /* disable frame event interrupt */
        hals_dci_interrupt_disable(DCI_INT_EF);
    }

    /* enable vsync event error interrupt */
    if(NULL != p_irq->vsync_event_handle) {
        /* set vsync event handle */
        dci_dev->dci_irq.vsync_event_handle = p_irq->vsync_event_handle;
        /* enable vsync event interrupt */
        hals_dci_interrupt_enable(DCI_INT_VSYNC);
    } else {
        /* disable vsync event interrupt */
        hals_dci_interrupt_disable(DCI_INT_VSYNC);
    }

    /* set the line event interrupt */
    if(NULL != p_irq->line_event_handle) {
        /* set line event handle */
        dci_dev->dci_irq.line_event_handle = p_irq->line_event_handle;
        /* enable line event interrupt */
        hals_dci_interrupt_enable(DCI_INT_EL);
    } else {
        /* disable line event interrupt */
        hals_dci_interrupt_disable(DCI_INT_EL);
    }

    /* set overrun handle interrupt */
    if(NULL != p_irq->overrun_handle) {
        /* set overrun handle */
        dci_dev->dci_irq.overrun_handle = p_irq->overrun_handle;
        /* enable overrun interrupt */
        hals_dci_interrupt_enable(DCI_INT_OVR);
    } else {
        /* disable overrun interrupt */
        hals_dci_interrupt_disable(DCI_INT_OVR);
    }

    /* set embedded synchronous error interrupt */
    if(NULL != p_irq->embedded_sync_handle) {
        /* set embedded synchronous error handle */
        dci_dev->dci_irq.embedded_sync_handle = p_irq->embedded_sync_handle;
        /* enable embedded synchronous error interrupt */
        hals_dci_interrupt_enable(DCI_INT_ESE);
    } else {
        /* disable embedded synchronous error interrupt */
        hals_dci_interrupt_disable(DCI_INT_ESE);
    }

    /* set CCIR field 0 interrupt */
    if(NULL != p_irq->f0_handle) {
        /* set CCIR field 0 handle */
        dci_dev->dci_irq.f0_handle = p_irq->f0_handle;
        /* enable CCIR field 0 interrupt */
        hals_dci_interrupt_enable(DCI_INT_F0);
    } else {
        /* disable CCIR field 0 interrupt */
        hals_dci_interrupt_disable(DCI_INT_F0);
    }

    /* set CCIR field 1 interrupt */
    if(NULL != p_irq->f1_handle) {
        /* set CCIR field 1 handle */
        dci_dev->dci_irq.f1_handle = p_irq->f1_handle;
        /* enable CCIR field 1 interrupt */
        hals_dci_interrupt_enable(DCI_INT_F1);
    } else {
        /* disable CCIR field 1 interrupt */
        hals_dci_interrupt_disable(DCI_INT_F1);
    }

    /* set charge of field interrupt */
    if(NULL != p_irq->charge_failed_handle) {
        /* set charge of field handle */
        dci_dev->dci_irq.charge_failed_handle = p_irq->charge_failed_handle;
        /* enable charge of field interrupt */
        hals_dci_interrupt_enable(DCI_INT_COF);
    } else {
        /* disable charge of field interrupt */
        hals_dci_interrupt_disable(DCI_INT_COF);
    }

    /* set ccir error interrupt */
    if(NULL != p_irq->ccir_error_handle) {
        /* set ccir error handle */
        dci_dev->dci_irq.ccir_error_handle = p_irq->ccir_error_handle;
        /* enable ccir error interrupt */
        hals_dci_interrupt_enable(DCI_INT_CCE);
    } else {
        /* disable ccir error interrupt */
        hals_dci_interrupt_disable(DCI_INT_CCE);
    }

    /* set error handle */
    if (NULL != p_irq->error_handle) {
        dci_dev->dci_irq.error_handle = p_irq->error_handle;
    } else {
        /* Do nothing */
    }

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      DCI interrupt handler content function,which is merely used in dci_handler
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_irq(hal_dci_dev_struct *dci_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* end of frame interrupt handler */
    if(SET == hals_dci_interrupt_flag_get(DCI_INT_FLAG_EF)) {
        hals_dci_interrupt_flag_clear(DCI_INT_FLAG_EF);
        if(NULL != dci_dev->dci_irq.frame_event_handle) {
            dci_dev->dci_irq.frame_event_handle(dci_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* vsync interrupt handler */
    if(SET == hals_dci_interrupt_flag_get(DCI_INT_FLAG_VSYNC)) {
        hals_dci_interrupt_flag_clear(DCI_INT_FLAG_VSYNC);
        if(NULL != dci_dev->dci_irq.vsync_event_handle) {
            dci_dev->dci_irq.vsync_event_handle(dci_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* end of line interrupt handler */
    if(SET == hals_dci_interrupt_flag_get(DCI_INT_FLAG_EL)) {
        hals_dci_interrupt_flag_clear(DCI_INT_FLAG_EL);
        if(NULL != dci_dev->dci_irq.line_event_handle) {
            dci_dev->dci_irq.line_event_handle(dci_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* set overrun handle interrupt */
    if(SET == hals_dci_interrupt_flag_get(DCI_INT_FLAG_OVR)) {
        hals_dci_interrupt_flag_clear(DCI_INT_FLAG_OVR);
        if(NULL != dci_dev->dci_irq.overrun_handle) {
            dci_dev->dci_irq.overrun_handle(dci_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* set embedded synchronous error interrupt */
    if(SET == hals_dci_interrupt_flag_get(DCI_INT_FLAG_ESE)) {
        hals_dci_interrupt_flag_clear(DCI_INT_FLAG_ESE);
        if(NULL != dci_dev->dci_irq.embedded_sync_handle) {
            dci_dev->dci_irq.embedded_sync_handle(dci_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* set CCIR field 0 interrupt */
    if(SET == hals_dci_interrupt_flag_get(DCI_INT_FLAG_F0)) {
        hals_dci_interrupt_flag_clear(DCI_INT_FLAG_F0);
        if(NULL != dci_dev->dci_irq.f0_handle) {
            dci_dev->dci_irq.f0_handle(dci_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* set CCIR field 1 interrupt */
    if(SET == hals_dci_interrupt_flag_get(DCI_INT_FLAG_F1)) {
        hals_dci_interrupt_flag_clear(DCI_INT_FLAG_F1);
        if(NULL != dci_dev->dci_irq.f1_handle) {
            dci_dev->dci_irq.f1_handle(dci_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* set change of field interrupt */
    if(SET == hals_dci_interrupt_flag_get(DCI_INT_FLAG_COF)) {
        hals_dci_interrupt_flag_clear(DCI_INT_FLAG_COF);
        if(NULL != dci_dev->dci_irq.charge_failed_handle) {
            dci_dev->dci_irq.charge_failed_handle(dci_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* set ccir error interrupt */
    if(SET == hals_dci_interrupt_flag_get(DCI_INT_FLAG_CCE)) {
        hals_dci_interrupt_flag_clear(DCI_INT_FLAG_CCE);
        if(NULL != dci_dev->dci_irq.ccir_error_handle) {
            dci_dev->dci_irq.ccir_error_handle(dci_dev);
        } else {
            /* Do nothing */
        }
    } else {
        /* Do nothing */
    }

    /* set DCI state */
    dci_dev->state = HAL_DCI_STATE_READY;

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_irq_handle_all_reset(hal_dci_dev_struct *dci_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DCI */
    HAL_LOCK(dci_dev);

    /* DCI interrupt handler reset */
    dci_dev->dci_irq.frame_event_handle   = NULL;
    dci_dev->dci_irq.vsync_event_handle   = NULL;
    dci_dev->dci_irq.line_event_handle    = NULL;
    dci_dev->dci_irq.embedded_sync_handle = NULL;
    dci_dev->dci_irq.overrun_handle       = NULL;
    dci_dev->dci_irq.f0_handle            = NULL;
    dci_dev->dci_irq.f1_handle            = NULL;
    dci_dev->dci_irq.charge_failed_handle = NULL;
    dci_dev->dci_irq.ccir_error_handle    = NULL;
    dci_dev->dci_irq.error_handle         = NULL;

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      suspend DCI
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_suspend(hal_dci_dev_struct *dci_dev)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DCI */
    HAL_LOCK(dci_dev);

    if(HAL_DCI_STATE_BUSY == dci_dev->state) {
        dci_dev->state = HAL_DCI_STATE_SUSPENDED;
        /* DCI capture disable */
        hals_dci_capture_disable();
    } else {
        HAL_DEBUGE("DCI state is not busy");
        retval = HAL_ERR_BUSY;
    }

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return retval;
}

/*!
    \brief      resume DCI
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_resume(hal_dci_dev_struct *dci_dev)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock DCI */
    HAL_LOCK(dci_dev);

    if(HAL_DCI_STATE_SUSPENDED == dci_dev->state) {
        dci_dev->state = HAL_DCI_STATE_BUSY;
        /* DCI capture enable */
        hals_dci_capture_enable();
    } else {
        HAL_DEBUGE("DCI state is not suspended");
        retval = HAL_ERR_BUSY;
    }

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return retval;
}

/*!
    \brief      get DCI state
    \param[in]  dci_dev: dci device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_dci_state_enum: state of DCI, details refer to gd32h7xx_hal_dci.h
*/
hal_dci_state_enum hal_dci_state_get(hal_dci_dev_struct *dci_dev)
{
    return dci_dev->state;
}

/*!
    \brief      get DCI error state
    \param[in]  dci_dev: dci device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     HAL_DCI_ERROR_NONE, HAL_DCI_ERROR_OVR, HAL_DCI_ERROR_SYNC, HAL_DCI_ERROR_TIMEOUT, HAL_DCI_ERROR_DMA
*/
uint32_t hal_dci_error_get(hal_dci_dev_struct *dci_dev)
{
    return dci_dev->error_state;
}

/*!
    \brief      enable DCI external vsync in CCIR progressive mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_dci_external_vsync_enable(void)
{
    DCI_CTL |= DCI_CTL_EVSEN;
}

/*!
    \brief      disable DCI external vsync in CCIR progressive mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_dci_external_vsync_disable(void)
{
    DCI_CTL &= ~DCI_CTL_EVSEN;
}

/*!
    \brief      CCIR mode select
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ccir_mode: specify which mode to select
                only one parameter can be selected which is shown as below:
      \arg        CCIR_PROGRESSIVE_MODE: CCIR progressive mode
      \arg        CCIR_INTERLACE_MODE: CCIR interlace mode
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_ccir_mode_select(hal_dci_dev_struct *dci_dev, uint32_t ccir_mode)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((CCIR_PROGRESSIVE_MODE != ccir_mode) && (CCIR_INTERLACE_MODE != ccir_mode)) {
        HAL_DEBUGE("parameter [ccir_mode] value is illegal");
        return HAL_ERR_VAL;
    }

    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(dci_dev);

    if(HAL_DCI_STATE_BUSY == dci_dev->state) {
        HAL_DEBUGE("DCI state busy");
        retval = HAL_ERR_BUSY;
    } else {
        if(CCIR_INTERLACE_MODE == ccir_mode) {
            DCI_CTL |= DCI_CTL_CCMOD;
        } else {
            DCI_CTL &= ~DCI_CTL_CCMOD;
        }
    }

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return retval;
}

/*!
    \brief      configure DCI cropping window
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  start_x: window horizontal start position
    \param[in]  start_y: window vertical start position
    \param[in]  size_width: window horizontal size
    \param[in]  size_height: window vertical size
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_crop_window_config(hal_dci_dev_struct *dci_dev, \
                                   uint16_t start_x, uint16_t start_y, \
                                   uint16_t size_width, uint16_t size_height)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(DCI_WINDOW_VERTICAL_START_SIZE < start_y) {
        HAL_DEBUGE("parameter [start_y] value is invalid");
        return HAL_ERR_VAL;
    }

    if(DCI_WINDOW_HORIZONTAL_START_SIZE < start_x) {
        HAL_DEBUGE("parameter [start_x] value is invalid");
        return HAL_ERR_VAL;
    }

    if(DCI_WINDOW_HORIZONTAL_MAX_SIZE < size_width) {
        HAL_DEBUGE("parameter [size_width] value is invalid");
        return HAL_ERR_VAL;
    }

    if(DCI_WINDOW_VERTICAL_MAX_SIZE < size_height) {
        HAL_DEBUGE("parameter [size_height] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(dci_dev);

    if(HAL_DCI_STATE_BUSY == dci_dev->state) {
        HAL_DEBUGE("DCI state busy");
        retval = HAL_ERR_BUSY;
    } else {
        DCI_CWSPOS = ((uint32_t)start_x    | ((uint32_t)start_y    << 16U));
        DCI_CWSZ   = ((uint32_t)size_width | ((uint32_t)size_height << 16U));
    }

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return retval;
}

/*!
    \brief      config synchronous codes in embedded synchronous mode
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  frame_start: frame start code in embedded synchronous mode
    \param[in]  line_start: line start code in embedded synchronous mode
    \param[in]  line_end: line end code in embedded synchronous mode
    \param[in]  frame_end: frame end code in embedded synchronous mode
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_sync_codes_config(hal_dci_dev_struct *dci_dev, \
                                  uint8_t frame_start, uint8_t line_start, \
                                  uint8_t line_end, uint8_t frame_end)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(dci_dev);

    if(HAL_DCI_STATE_BUSY == dci_dev->state) {
        HAL_DEBUGE("DCI state busy");
        retval = HAL_ERR_BUSY;
    } else {
        DCI_SC = ((uint32_t)frame_start | ((uint32_t)line_start << 8U) | \
                  ((uint32_t)line_end << 16U) | ((uint32_t)frame_end << 24U));
    }

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return retval;
}

/*!
    \brief      config synchronous codes unmask in embedded synchronous mode
    \param[in]  dci_dev: DCI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  frame_start: frame start code unmask bits in embedded synchronous mode
    \param[in]  line_start: line start code unmask bits in embedded synchronous mode
    \param[in]  line_end: line end code unmask bits in embedded synchronous mode
    \param[in]  frame_end: frame end code unmask bits in embedded synchronous mode
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_dci_sync_codes_unmask_config(hal_dci_dev_struct *dci_dev, \
                                         uint8_t frame_start, uint8_t line_start, \
                                         uint8_t line_end, uint8_t frame_end)
{
    /* initialize the function return value */
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == dci_dev) {
        HAL_DEBUGE("pointer [dci_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(dci_dev);

    if(HAL_DCI_STATE_BUSY == dci_dev->state) {
        HAL_DEBUGE("DCI state busy");
        retval = HAL_ERR_BUSY;
    } else {
        DCI_SCUMSK = ((uint32_t)frame_start | ((uint32_t)line_start << 8U) | \
                      ((uint32_t)line_end << 16U) | ((uint32_t)frame_end << 24U));
    }

    /* unlock DCI */
    HAL_UNLOCK(dci_dev);

    return retval;
}

/*!
    \brief      enable cropping window function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_dci_crop_window_enable(void)
{
    DCI_CTL |= DCI_CTL_WDEN;
}

/*!
    \brief      disable cropping window function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_dci_crop_window_disable(void)
{
    DCI_CTL &= ~DCI_CTL_WDEN;
}

/*!
    \brief      enable DCI function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_enable(void)
{
    DCI_CTL |= DCI_CTL_DCIEN;
}

/*!
    \brief      disable DCI function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_disable(void)
{
    DCI_CTL &= ~DCI_CTL_DCIEN;
}

/*!
    \brief      enable DCI capture
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_capture_enable(void)
{
    DCI_CTL |= DCI_CTL_CAP;
}

/*!
    \brief      disable DCI capture
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_capture_disable(void)
{
    DCI_CTL &= ~DCI_CTL_CAP;
}

/*!
    \brief      enable DCI automatic error correction in CCIR interlaced mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_automatic_error_correction_enable(void)
{
    DCI_CTL |= DCI_CTL_AECEN;
}

/*!
    \brief      disable DCI automatic error correction in CCIR interlaced mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_automatic_error_correction_disable(void)
{
    DCI_CTL &= ~DCI_CTL_AECEN;
}

/*!
    \brief      enable DCI jpeg mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_jpeg_enable(void)
{
    DCI_CTL |= DCI_CTL_JM;
}

/*!
    \brief      disable DCI jpeg mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_jpeg_disable(void)
{
    DCI_CTL &= ~DCI_CTL_JM;
}

/*!
    \brief      enable embedded synchronous mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_embedded_sync_enable(void)
{
    DCI_CTL |= DCI_CTL_ESM;
}

/*!
    \brief      disable embedded synchronous mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_embedded_sync_disable(void)
{
    DCI_CTL &= ~DCI_CTL_ESM;
}

/*!
    \brief      CCIR mode enable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_ccir_enable(void)
{
    DCI_CTL |= DCI_CTL_CCEN;
}

/*!
    \brief      CCIR mode disable
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dci_ccir_disable(void)
{
    DCI_CTL &= ~DCI_CTL_CCEN;
}

/*!
    \brief      read DCI data register
    \param[in]  none
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hals_dci_data_read(void)
{
    return DCI_DATA;
}

/*!
    \brief      get specified flag
    \param[in]  flag: specify which flag to get
                only one parameter can be selected which is shown as below:
      \arg         DCI_FLAG_HS: HS line status
      \arg         DCI_FLAG_VS: VS line status
      \arg         DCI_FLAG_FV: FIFO valid
      \arg         DCI_FLAG_EF: end of frame flag
      \arg         DCI_FLAG_OVR: FIFO overrun flag
      \arg         DCI_FLAG_ESE: embedded synchronous error flag
      \arg         DCI_FLAG_VSYNC: vsync flag
      \arg         DCI_FLAG_EL: end of line flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_dci_flag_get(uint32_t flag)
{
    uint32_t stat = 0U;
    FlagStatus retval;

    if(flag >> 31U) {
        /* get flag status from DCI_STAT1 register */
        stat = DCI_STAT1;
    } else {
        /* get flag status from DCI_STAT0 register */
        stat = DCI_STAT0;
    }

    if(flag & stat) {
        retval = SET;
    } else {
        retval = RESET;
    }

    return retval;
}

/*!
    \brief      enable specified DCI interrupt
    \param[in]  interrupt: specify which interrupt to enable
                one or more parameter can be selected which is shown as below:
      \arg         DCI_INT_EF: end of frame interrupt
      \arg         DCI_INT_OVR: FIFO overrun interrupt
      \arg         DCI_INT_ESE: embedded synchronous error interrupt
      \arg         DCI_INT_VSYNC: vsync interrupt
      \arg         DCI_INT_EL: end of line interrupt
      \arg         DCI_INT_F0: CCIR field 0 interrupt
      \arg         DCI_INT_F1: CCIR field 1 interrupt
      \arg         DCI_INT_COF: CCIR change of field interrupt
      \arg         DCI_INT_CCE: CCIR error interrupt
    \param[out] none
    \retval     none
*/
void hals_dci_interrupt_enable(uint32_t interrupt)
{
    DCI_INTEN |= interrupt;
}

/*!
    \brief      disable specified DCI interrupt
    \param[in]  interrupt: specify which interrupt to disable
                one or more parameter can be selected which is shown as below:
      \arg         DCI_INT_EF: end of frame interrupt
      \arg         DCI_INT_OVR: FIFO overrun interrupt
      \arg         DCI_INT_ESE: embedded synchronous error interrupt
      \arg         DCI_INT_VSYNC: vsync interrupt
      \arg         DCI_INT_EL: end of line interrupt
      \arg         DCI_INT_F0: CCIR field 0 interrupt
      \arg         DCI_INT_F1: CCIR field 1 interrupt
      \arg         DCI_INT_COF: CCIR change of field interrupt
      \arg         DCI_INT_CCE: CCIR error interrupt
    \param[out] none
    \retval     none
*/
void hals_dci_interrupt_disable(uint32_t interrupt)
{
    DCI_INTEN &= ~interrupt;
}

/*!
    \brief      get specified interrupt flag
    \param[in]  int_flag: specify which flag to get
                one or more parameter can be selected which is shown as below:
      \arg         DCI_INT_FLAG_EF: end of frame interrupt flag
      \arg         DCI_INT_FLAG_OVR: FIFO overrun interrupt flag
      \arg         DCI_INT_FLAG_ESE: embedded synchronous error interrupt flag
      \arg         DCI_INT_FLAG_VSYNC: vsync interrupt flag
      \arg         DCI_INT_FLAG_EL: end of line interrupt flag
      \arg         DCI_INT_FLAG_F0: CCIR field 0 interrupt flag
      \arg         DCI_INT_FLAG_F1: CCIR field 1 interrupt flag
      \arg         DCI_INT_FLAG_COF: CCIR change of field interrupt flag
      \arg         DCI_INT_FLAG_CCE: CCIR error interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_dci_interrupt_flag_get(uint32_t int_flag)
{
    FlagStatus retval;

    if (RESET != (DCI_INTF & int_flag)) {
        retval = SET;
    } else {
        retval = RESET;
    }

    return retval;
}

/*!
    \brief      clear specified interrupt flag
    \param[in]  int_flag: specify which flag to clear
                one or more parameter can be selected which is shown as below:
      \arg         DCI_INT_FLAG_EF: end of frame interrupt flag
      \arg         DCI_INT_FLAG_OVR: FIFO overrun interrupt flag
      \arg         DCI_INT_FLAG_ESE: embedded synchronous error interrupt flag
      \arg         DCI_INT_FLAG_VSYNC: vsync interrupt flag
      \arg         DCI_INT_FLAG_EL: end of line interrupt flag
      \arg         DCI_INT_FLAG_COF: CCIR change of field interrupt flag
      \arg         DCI_INT_FLAG_CCE: CCIR error interrupt flag
    \param[out] none
    \retval     none
*/
void hals_dci_interrupt_flag_clear(uint32_t int_flag)
{
    DCI_INTC |= int_flag;
}

/*!
    \brief      DCI DMA full transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _dci_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_dci_dev_struct *dci_dev;
    hal_dci_user_cb p_func;

    p_dma   = (hal_dma_dev_struct *)dma;
    dci_dev = (hal_dci_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_dci_user_cb)(dci_dev->dci_dma.full_transcom_handle);

    if (NULL != p_func) {
        p_func(dci_dev);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      DCI DMA half transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _dci_dma_half_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_dci_dev_struct *dci_dev;
    hal_dci_user_cb p_func;

    p_dma   = (hal_dma_dev_struct *)dma;
    dci_dev = (hal_dci_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_dci_user_cb)(dci_dev->dci_dma.half_transcom_handle);

    if (NULL != p_func) {
        p_func(dci_dev);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      DCI DMA error callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _dci_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_dci_dev_struct *dci_dev;
    hal_dci_user_cb p_func;

    p_dma   = (hal_dma_dev_struct *)dma;
    dci_dev = (hal_dci_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_dci_user_cb)(dci_dev->dci_dma.error_handle);

    if (NULL != p_func) {
        p_func(dci_dev);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      DCI vsync event callback
    \param[in]  dci_dev: DCI device information structure
    \param[out] none
    \retval     none
*/
static void __vsync_event_callback(hal_dci_dev_struct *dci_dev)
{
    /* Create a pointer p_dci to the DCI device information structure, initialized with the passed dci_dev */
    hal_dci_dev_struct *p_dci = dci_dev;
    /* Define a HAL DCI user callback function pointer p_func */
    hal_dci_user_cb p_func;

    /* Assign the vsync event callback function in the dci_dev structure to p_func */
    p_func = (hal_dci_user_cb)(dci_dev->vsync_event_callback);

    /* Check if p_func is not a null pointer. If it's not null, call the callback function */
    if(NULL != p_func) {
        p_func(p_dci);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      DCI line event callback
    \param[in]  dci_dev: DCI device information structure
    \param[out] none
    \retval     none
*/
static void __line_event_callback(hal_dci_dev_struct *dci_dev)
{
    /* Create a pointer p_dci to the DCI device information structure, initialized with the passed dci_dev */
    hal_dci_dev_struct *p_dci = dci_dev;
    /* Define a HAL DCI user callback function pointer p_func */
    hal_dci_user_cb p_func;

    /* Assign the line event callback function in the dci_dev structure to p_func */
    p_func = (hal_dci_user_cb)(dci_dev->line_event_callback);

    /* Check if p_func is not a null pointer. If it's not null, call the callback function */
    if(NULL != p_func) {
        p_func(p_dci);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      DCI ccir error callback
    \param[in]  dci_dev: DCI device information structure
    \param[out] none
    \retval     none
*/
static void __ccir_error_callback(hal_dci_dev_struct *dci_dev)
{
    /* Create a pointer p_dci to the DCI device information structure, initialized with the passed dci_dev */
    hal_dci_dev_struct *p_dci = dci_dev;
    /* Define a HAL DCI user callback function pointer p_func */
    hal_dci_user_cb p_func;

    /* Assign the ccir error callback function in the dci_dev structure to p_func */
    p_func = (hal_dci_user_cb)(dci_dev->ccir_error_callback);

    /* Check if p_func is not a null pointer. If it's not null, call the callback function */
    if(NULL != p_func) {
        p_func(p_dci);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      DCI frame event callback
    \param[in]  dci_dev: DCI device information structure
    \param[out] none
    \retval     none
*/
static void __frame_event_callback(hal_dci_dev_struct *dci_dev)
{
    /* Create a pointer p_dci to the DCI device information structure, initialized with the passed dci_dev */
    hal_dci_dev_struct *p_dci = dci_dev;
    /* Define a HAL DCI user callback function pointer p_func */
    hal_dci_user_cb p_func;

    /* Assign the frame event callback function in the dci_dev structure to p_func */
    p_func = (hal_dci_user_cb)(dci_dev->frame_event_callback);

    /* Check if p_func is not a null pointer. If it's not null, call the callback function */
    if(NULL != p_func) {
        p_func(p_dci);
    } else {
        /* Do nothing */
    }
}

/*!
    \brief      DCI error callback
    \param[in]  dci_dev: DCI device information structure
    \param[out] none
    \retval     none
*/
static void __error_callback(hal_dci_dev_struct *dci_dev)
{
    /* Create a pointer p_dci to the DCI device information structure, initialized with the passed dci_dev */
    hal_dci_dev_struct *p_dci = dci_dev;
    /* Define a HAL DCI user callback function pointer p_func */
    hal_dci_user_cb p_func;

    /* Assign the error callback function in the dci_dev structure to p_func */
    p_func = (hal_dci_user_cb)(dci_dev->error_callback);

    /* Check if p_func is not a null pointer. If it's not null, call the callback function */
    if(NULL != p_func) {
        p_func(p_dci);
    } else {
        /* Do nothing */
    }
}
