/*!
    \file    gd32h7xx_hal_crc.c
    \brief   CRC driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define CRC_IDATA_RESET_VALUE ((uint32_t)0xFFFFFFFFU)
#define CRC_DATA_RESET_VALUE  ((uint32_t)0xFFFFFFFFU)
#define CRC_FDATA_RESET_VALUE ((uint32_t)0x00000000U)
#define CRC_POLY_RESET_VALUE  ((uint32_t)0x04C11DB7U)

/*!
    \brief      initialize the CRC structure with the default values
    \param[in]  hal_struct_type: the type of the structure
      \arg        HAL_CRC_INIT_STRUCT:the CRC initialization structure
      \arg        HAL_CRC_DEV_STRUCT:the CRC device information structure
    \param[out] p_struct: point to the structure to be initialized
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_crc_struct_init(hal_crc_struct_type_enum hal_struct_type, void *p_struct)
{
    int ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_CRC_INIT_STRUCT:
        ((hal_crc_init_struct *)p_struct)->input_data_reverse_mode = CRC_INPUT_REVERSE_NONE;
        ((hal_crc_init_struct *)p_struct)->output_data_reverse     = CRC_OUTPUT_REVERSE_DISABLE;
        ((hal_crc_init_struct *)p_struct)->polynomial_size         = CRC_POLYNOMIAL_SIZE_32BIT;
        ((hal_crc_init_struct *)p_struct)->polynomial_value        = CRC_DEFAULT_POLYNOMIAL_VALUE;
        ((hal_crc_init_struct *)p_struct)->init_data_value         = CRC_DEFAULT_INIT_DATA_VALUE;
        break;
    case HAL_CRC_DEV_STRUCT:
        ((hal_crc_dev_struct *)p_struct)->state                    = HAL_CRC_STATE_RESET;
        ((hal_crc_dev_struct *)p_struct)->mutex                    = HAL_MUTEX_UNLOCKED;
        ((hal_crc_dev_struct *)p_struct)->priv                     = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      initialize CRC
    \param[in]  crc_dev: CRC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_crc_init: CRC initialize structure
                  output_data_reverse:
                  only one parameter can be selected which is shown as below:
      \arg          CRC_OUTPUT_REVERSE_DISABLE
      \arg          CRC_OUTPUT_REVERSE_ENABLE
                  input_data_reverse_mode:
                  only one parameter can be selected which is shown as below:
      \arg          CRC_INPUT_REVERSE_NONE
      \arg          CRC_INPUT_REVERSE_BYTE
      \arg          CRC_INPUT_REVERSE_HALFWORD
      \arg          CRC_INPUT_REVERSE_WORD
                  polynomial_size:
                  only one parameter can be selected which is shown as below:
      \arg          CRC_POLYNOMIAL_SIZE_32BIT
      \arg          CRC_POLYNOMIAL_SIZE_16BIT
      \arg          CRC_POLYNOMIAL_SIZE_8BIT
      \arg          CRC_POLYNOMIAL_SIZE_7BIT
                  polynomial_value:
      \arg          polynomial_value or CRC_DEFAULT_POLYNOMIAL_VALUE(if don't care this parameter)
                  init_data_value:
      \arg          crc_init_value or CRC_DEFAULT_INIT_DATA_VALUE(if don't care this parameter)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_crc_init(hal_crc_dev_struct *crc_dev, hal_crc_init_struct *p_crc_init)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == crc_dev) || (NULL == p_crc_init)) {
        HAL_DEBUGE("pointer [p_crc_init] or pointer [crc_dev] address is invalid");

        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(crc_dev);
    crc_dev->state = HAL_CRC_STATE_BUSY;

    /* reset crc data register */
    CRC_CTL |= CRC_CTL_RST;

    CRC_IDATA = p_crc_init->init_data_value;
    CRC_POLY  = p_crc_init->polynomial_value;

    /* enable the reverse operation of output data */
    if(CRC_OUTPUT_REVERSE_ENABLE == p_crc_init->output_data_reverse) {
        CRC_CTL |= CRC_CTL_REV_O;
    } else {
        CRC_CTL &= ~CRC_CTL_REV_O;
    }

    /* configure the CRC input data function */
    CRC_CTL &= (uint32_t)(~CRC_CTL_REV_I);
    CRC_CTL |= (uint32_t)p_crc_init->input_data_reverse_mode;

    /* configure the CRC size of polynomial function */
    CRC_CTL &= (uint32_t)(~(CRC_CTL_PS));
    CRC_CTL |= (uint32_t)p_crc_init->polynomial_size;

    /* change CRC state */
    crc_dev->state = HAL_CRC_STATE_READY;
    HAL_UNLOCK(crc_dev);

    /* return function state */
    return HAL_ERR_NONE;
}

/*!
    \brief      deinitialize CRC
    \param[in]  crc_dev: CRC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_crc_deinit(hal_crc_dev_struct *crc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == crc_dev) {
        HAL_DEBUGE("pointer [crc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock CRC */
    HAL_LOCK(crc_dev);
    crc_dev->state = HAL_CRC_STATE_BUSY;

    hal_rcu_periph_reset_enable(RCU_CRCRST);
    hal_rcu_periph_reset_disable(RCU_CRCRST);

    /* set CRC state */
    crc_dev->state = HAL_CRC_STATE_RESET;

    /* unlock CRC */
    HAL_UNLOCK(crc_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      CRC calculate single data
    \param[in]  crc_dev: CRC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  sdata: specify input data
    \param[in]  data_format: input data format
                only one parameter can be selected which is shown as below:
      \arg        INPUT_FORMAT_WORD: input data in word format
      \arg        INPUT_FORMAT_HALFWORD: input data in half-word format
      \arg        INPUT_FORMAT_BYTE: input data in byte format
    \param[out] none
    \retval     uint32_t: 0-0xFFFFFFFF
*/
uint32_t hal_crc_single_data_calculate(hal_crc_dev_struct *crc_dev, uint32_t sdata, uint8_t data_format)
{
    crc_dev->state = HAL_CRC_STATE_BUSY;

    /* calculate data */
    if(INPUT_FORMAT_WORD == data_format) {
        REG32(CRC) = sdata;
    } else if(INPUT_FORMAT_HALFWORD == data_format) {
        REG16(CRC) = (uint16_t)sdata;
    } else {
        REG8(CRC) = (uint8_t)sdata;
    }

    /* set CRC state */
    crc_dev->state = HAL_CRC_STATE_READY;

    return (CRC_DATA);
}

/*!
    \brief      Calculates the 7-, 8-, 16-, or 32-bit CRC value for the 8-, 16-, or 32-bit data buffer
                using the value in the CRC_IDATA register as the initialization value.
    \param[in]  crc_dev: CRC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  array: pointer to the input data array
    \param[in]  size: size of the array
    \param[in]  data_format: input data format
                only one parameter can be selected which is shown as below:
      \arg        INPUT_FORMAT_WORD: input data in word format
      \arg        INPUT_FORMAT_HALFWORD: input data in half-word format
      \arg        INPUT_FORMAT_BYTE: input data in byte format
    \param[out] none
    \retval     uint32_t: 0-0xFFFFFFFF
*/
uint32_t hal_crc_block_data_calculate(hal_crc_dev_struct *crc_dev, void *array, uint32_t size, uint8_t data_format)
{
    uint8_t *data8;
    uint16_t *data16;
    uint32_t *data32;
    uint32_t index;
    /* update CRC state */
    crc_dev->state = HAL_CRC_STATE_BUSY;

    /* reset CRC DATA value */
    CRC_CTL |= CRC_CTL_RST;

    /* calculate data */
    if(INPUT_FORMAT_WORD == data_format) {
        data32 = (uint32_t *)array;
        for(index = 0U; index < size; index++) {
            REG32(CRC) = data32[index];
        }
    } else if(INPUT_FORMAT_HALFWORD == data_format) {
        data16 = (uint16_t *)array;
        for(index = 0U; index < size; index++) {
            REG16(CRC) = data16[index];
        }
    } else {
        data8 = (uint8_t *)array;
        for(index = 0U; index < size; index++) {
            REG8(CRC) = data8[index];
        }
    }

    /* set CRC state */
    crc_dev->state = HAL_CRC_STATE_READY;

    return (CRC_DATA);
}

/*!
    \brief      Calculates the 7-, 8-, 16-, or 32-bit CRC value for the 8-, 16-, or 32-bit data buffer
                using the value in the CRC_DATA register as the initialization value.
    \param[in]  crc_dev: CRC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  array: pointer to the input data array
    \param[in]  size: size of the array
    \param[in]  data_format: input data format
                only one parameter can be selected which is shown as below:
      \arg        INPUT_FORMAT_WORD: input data in word format
      \arg        INPUT_FORMAT_HALFWORD: input data in half-word format
      \arg        INPUT_FORMAT_BYTE: input data in byte format
    \param[out] none
    \retval     uint32_t: 0-0xFFFFFFFF
*/
uint32_t hal_crc_block_data_accumulate(hal_crc_dev_struct *crc_dev, void *array, uint32_t size, uint8_t data_format)
{
    uint8_t *data8;
    uint16_t *data16;
    uint32_t *data32;
    uint32_t index;

    /* lock CRC */
    crc_dev->state = HAL_CRC_STATE_BUSY;

    /* calculate data */
    if(INPUT_FORMAT_WORD == data_format) {
        data32 = (uint32_t *)array;
        for(index = 0U; index < size; index++) {
            REG32(CRC) = data32[index];
        }
    } else if(INPUT_FORMAT_HALFWORD == data_format) {
        data16 = (uint16_t *)array;
        for(index = 0U; index < size; index++) {
            REG16(CRC) = data16[index];
        }
    } else {
        data8 = (uint8_t *)array;
        for(index = 0U; index < size; index++) {
            REG8(CRC) = data8[index];
        }
    }

    /* set CRC state */
    crc_dev->state = HAL_CRC_STATE_READY;

    return (CRC_DATA);
}

/*!
    \brief      configure the CRC input data function
    \param[in]  data_reverse: specify input data reverse function
                only one parameter can be selected which is shown as below:
      \arg        CRC_INPUT_REVERSE_NONE: input data is not reversed
      \arg        CRC_INPUT_REVERSE_BYTE: input data is reversed on 8 bits
      \arg        CRC_INPUT_REVERSE_HALFWORD: input data is reversed on 16 bits
      \arg        CRC_INPUT_REVERSE_WORD: input data is reversed on 32 bits
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_crc_input_data_reverse_config(uint32_t data_reverse)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((CRC_INPUT_REVERSE_NONE != data_reverse) && (CRC_INPUT_REVERSE_BYTE != data_reverse) && \
       (CRC_INPUT_REVERSE_HALFWORD != data_reverse) && (CRC_INPUT_REVERSE_WORD != data_reverse)) {
        HAL_DEBUGE("parameter [data_reverse] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    CRC_CTL &= (uint32_t)(~CRC_CTL_REV_I);
    CRC_CTL |= (uint32_t)data_reverse;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable the reverse operation of output data
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_crc_reverse_output_data_enable(void)
{
    CRC_CTL |= CRC_CTL_REV_O;
}

/*!
    \brief      disable the reverse operation of output data
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_crc_reverse_output_data_disable(void)
{
    CRC_CTL &= ~CRC_CTL_REV_O;
}

/*!
    \brief      configure the CRC size of polynomial function
    \param[in]  poly_size: size of polynomial
                only one parameter can be selected which is shown as below:
      \arg        CRC_CTL_PS_32: 32-bit polynomial for CRC calculation
      \arg        CRC_CTL_PS_16: 16-bit polynomial for CRC calculation
      \arg        CRC_CTL_PS_8: 8-bit polynomial for CRC calculation
      \arg        CRC_CTL_PS_7: 7-bit polynomial for CRC calculation
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_crc_polynomial_size_set(uint32_t poly_size)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((CRC_CTL_PS_32 != poly_size) && (CRC_CTL_PS_16 != poly_size) && (CRC_CTL_PS_8 != poly_size) && \
       (CRC_CTL_PS_7 != poly_size)) {
        HAL_DEBUGE("parameter [poly_size] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    CRC_CTL &= (uint32_t)(~(CRC_CTL_PS));
    CRC_CTL |= poly_size;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the CRC polynomial value function
    \param[in]  poly: configurable polynomial value
    \param[out] none
    \retval     none
*/
void hal_crc_polynomial_set(uint32_t poly)
{
    CRC_POLY = poly;
}

/*!
    \brief      CRC state get function
    \param[in]  crc_dev: CRC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_crc_state_enum: CRC state, details refer to gd32h7xx_hal_crc.h
*/
hal_crc_state_enum hal_crc_state_get(hal_crc_dev_struct *crc_dev)
{
    return (crc_dev->state);
}

/*!
    \brief      reset data register to the value of initialization data register
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_crc_data_register_reset(void)
{
    CRC_CTL |= CRC_CTL_RST;
}

/*!
    \brief      read the data register
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0-0xFFFFFFFF
*/
uint32_t hals_crc_data_register_read(void)
{
    return (CRC_DATA);
}

/*!
    \brief      read the free data register
    \param[in]  none
    \param[out] none
    \retval     uint8_t: 0-0xFF
*/
uint8_t hals_crc_free_data_register_read(void)
{
    return ((uint8_t)CRC_FDATA);
}

/*!
    \brief      write the free data register
    \param[in]  free_data: specify 8-bit data
    \param[out] none
    \retval     none
*/
void hals_crc_free_data_register_write(uint8_t free_data)
{
    CRC_FDATA = (uint32_t)free_data;
}

/*!
    \brief      write the initial value register
    \param[in]  init_data:specify 32-bit data
    \param[out] none
    \retval     none
*/
void hals_crc_init_data_register_write(uint32_t init_data)
{
    CRC_IDATA = init_data;
}

/*!
    \brief      reset CRC polynomial register to the default value
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_crc_calculate_reset(void)
{
    CRC_POLY = CRC_POLY_RESET_VALUE;
}
