/*!
    \file    gd32h7xx_hal_cau.c
    \brief   CAU driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define FLAG_MASK         ((uint32_t)0x00000020U)
#define STAT0_AESDES_MASK ((uint32_t)0x00000015U)
#define STAT0_TDES_MASK   ((uint32_t)0x00000014U)

/* AES key structure parameter config */
static void _cau_aes_key_config(uint32_t *key, hal_cau_aes_key_length_enum keysize, \
                                hal_cau_key_parameter_struct *cau_key_initpara);
/* AES vectors structure parameter config */
static void _cau_aes_iv_config(uint32_t *iv, hal_cau_iv_parameter_struct *cau_iv_initpara);
/* encrypt and decrypt using TDES or DES handler in interrupt */
static void _cau_des_tdes_interrupt(hal_cau_dev_struct *cau_dev);
/* encrypt and decrypt using AES handler in interrupt */
static void _cau_aes_interrupt(hal_cau_dev_struct *cau_dev);
/* encrypt and decrypt using AES handler in interrupt for ccm */
static ErrStatus _cau_aes_ccm_gcm_interrupt(hal_cau_dev_struct *cau_dev);
/* CAU dma transfer input complete callback */
static void _cau_in_dma_complete_callback(void *dma);
/* CAU dma transfer output complete callback */
static void _cau_out_dma_complete_callback(void *dma);
/* CAU dma error callback */
static void _cau_dma_error_callback(void *dma);

/*!
    \brief      deinitialize CAU
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_cau_deinit(hal_cau_dev_struct *cau_dev)
{
      int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == cau_dev) {
        HAL_DEBUGE("pointer [cau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_CAU_STATE_BUSY == cau_dev->state) {
        HAL_DEBUGE("CAU is busy!");
        ret = HAL_ERR_BUSY;
    } else {
        /* nothing to do*/
    }

    cau_dev->state = HAL_CAU_STATE_BUSY;
    /* reset CAU */
    hal_rcu_periph_reset_enable(RCU_CAURST);
    hal_rcu_periph_reset_disable(RCU_CAURST);

    cau_dev->state = HAL_CAU_STATE_RESET;

    return ret;
}

/*!
    \brief      initialize CAU
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_init: CAU init parameter struct
                  mode:cau mode
                  only one parameter can be selected which is shown as below:
      \arg          CAU_TDES_ECB: TDES-ECB (3DES Electronic codebook)
      \arg          CAU_TDES_CBC: TDES-CBC (3DES Cipher block chaining)
      \arg          CAU_DES_ECB: DES-ECB (simple DES Electronic codebook)
      \arg          CAU_DES_CBC: DES-CBC (simple DES Cipher block chaining)
      \arg          CAU_AES_ECB: AES-ECB (AES Electronic codebook)
      \arg          CAU_AES_CBC: AES-CBC (AES Cipher block chaining)
      \arg          CAU_AES_CTR: AES-CTR (AES counter mode)
      \arg          CAU_AES_GCM: AES-GCM (AES Galois/counter mode)
      \arg          CAU_AES_CCM: AES-CCM (AES combined cipher machine mode)
      \arg          CAU_AES_CFB: AES-CFB (cipher feedback mode)
      \arg          CAU_AES_OFB: AES-OFB (output feedback mode)
                  cau_dir:
                  only one parameter can be selected which is shown as below:
     \arg           CAU_ENCRYPTION_DIRECTION： encrypt
     \arg           CAU_DECRYPTION_DIRECTION： decrypt
                  key_source:
                  only one parameter can be selected which is shown as below:
     \arg           CAU_KEY_FROM_REGISTER:use the key from CAU register
     \arg           CAU_KEY_FROM_EFUSE:use the key from EFUSE
                  aes_key_length:aes key length
                  only one parameter can be selected which is shown as below:
      \arg          CAU_AES_KEY_LENGTH_128BIT: 128 bit key length
      \arg          CAU_AES_KEY_LENGTH_192BIT: 192 bit key length
      \arg          CAU_AES_KEY_LENGTH_256BIT: 256 bit key length
                  key：encryption/decryption Key
                  init_vector: initialization vector
                  aes_verify_key: aes verify key
                  data_exchange_mode: data exchange mode
                  only one parameters can be selected which are shown as below:
      \arg          CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg          CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg          CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg          CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_cau_init(hal_cau_dev_struct *cau_dev, hal_cau_init_struct *p_init)
{
    hal_cau_iv_parameter_struct iv_initpara;
    hal_cau_key_parameter_struct key_initpara;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == cau_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [cau_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    cau_dev->mutex = HAL_MUTEX_LOCKED;
    cau_dev->state = HAL_CAU_STATE_BUSY;

    CAU_CTL &= (~(uint32_t)(CAU_CTL_KEY_SEL | CAU_CTL_CAUDIR | CAU_CTL_ALGM | \
                CAU_CTL_DATAM | CAU_CTL_KEYM | CAU_CTL_FFLUSH));
    /* encrypt */
    if(CAU_ENCRYPTION_DIRECTION == p_init->cau_dir) {
        if(CAU_KEY_FROM_REGISTER == p_init->key_source) {
            CAU_CTL |= (uint32_t)p_init->aes_key_length;
            /* init key */
            hals_cau_key_struct_para_init(&key_initpara);
            _cau_aes_key_config((uint32_t *)p_init->key, p_init->aes_key_length, &key_initpara);
            hals_cau_key_init(&key_initpara);
        } else {
            /* nothing to do */
        }
        CAU_CTL |= p_init->key_source;
        /* set data swapping mode*/
        CAU_CTL |= (p_init->data_exchange_mode);
        /* decrypt  */
    } else if(CAU_DECRYPTION_DIRECTION == p_init->cau_dir) {
        CAU_CTL |= (uint32_t)p_init->aes_key_length;
        CAU_CTL |= p_init->key_source;
        CAU_CTL |= p_init->data_exchange_mode;
        if(CAU_KEY_FROM_REGISTER == p_init->key_source) {
            /* init key */
            hals_cau_key_struct_para_init(&key_initpara);
            _cau_aes_key_config((uint32_t *)p_init->key, p_init->aes_key_length, &key_initpara);
            hals_cau_key_init(&key_initpara);
        } else {
            /* nothing to do */
        }
        /* Configure the prepared key for decryption */
        CAU_CTL |= CAU_MODE_AES_KEY;
        CAU_CTL |= CAU_CTL_CAUEN;
        /* wait busy is 0, wait cau disable */
        while((0U != (CAU_CTL & CAU_STAT0_BUSY)) && (0U != (CAU_CTL & CAU_CTL_CAUEN))) {}
    } else {
        /* nothing to do */
    }

    CAU_CTL &= (~(uint32_t)CAU_CTL_ALGM);
    CAU_CTL |= ((uint32_t)p_init->mode | p_init->cau_dir);

    /* init Vector */
    hals_cau_iv_struct_para_init(&iv_initpara);
    _cau_aes_iv_config((uint32_t *)p_init->init_vector, &iv_initpara);
    hals_cau_iv_init(&iv_initpara);

    /* Refresh the input FIFO and output FIFO */
    CAU_CTL |= CAU_CTL_FFLUSH;
    cau_dev->mutex = HAL_MUTEX_UNLOCKED;
    cau_dev->state = HAL_CAU_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the CAU structure with default values
    \param[in]  hal_struct_type: The type of the struct to initialize
                only one parameters can be selected which are shown as below:
      \arg        HAL_CAU_INIT_STRUCT: CAU initialization structure
      \arg        HAL_CAU_IRQ_STRUCT: CAU interrupt structure
      \arg        HAL_CAU_DMA_HANDLE_CB_STRUCT: CAU dma handle structure
      \arg        HAL_CAU_DEV_STRUCT: CAU device information structure
      \arg        HAL_CAU_CONTEXT_PARAMETER_STRUCT: CAU context parameter structure
      \arg        HAL_CAU_IRQ_USER_CALLBACK_STRUCT: CAU irq user callback structure
      \arg        HAL_CAU_IV_PARAMETER_STRUCT: CAU IV parameter structure
      \arg        HAL_CAU_KEY_PARAMETER_STRUCT: CAU key parameter structure
      \arg        HAL_CAU_PARAMETER_STRUCT: CAU parameter structure
    \param[out] p_struct: pointer to CAU structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_cau_struct_init(hal_cau_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
    int i = 0;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_CAU_INIT_STRUCT:
        /* code */
        ((hal_cau_init_struct *)p_struct)->mode                              = CAU_TDES_ECB;
        ((hal_cau_init_struct *)p_struct)->cau_dir                           = CAU_ENCRYPTION_DIRECTION;
        ((hal_cau_init_struct *)p_struct)->key_source                        = CAU_KEY_FROM_REGISTER;
        ((hal_cau_init_struct *)p_struct)->aes_key_length                    = CAU_AES_KEY_LENGTH_128BIT;
        ((hal_cau_init_struct *)p_struct)->key                               = NULL;
        ((hal_cau_init_struct *)p_struct)->init_vector                       = NULL;
        ((hal_cau_init_struct *)p_struct)->aes_verify_key                    = NULL;
        ((hal_cau_init_struct *)p_struct)->data_exchange_mode                = CAU_DATA_EXCHANGE_32BIT;
        break;
    case HAL_CAU_DEV_STRUCT:
        ((hal_cau_dev_struct *)p_struct)->fifo_input_callback                = NULL;
        ((hal_cau_dev_struct *)p_struct)->fifo_output_callback               = NULL;
        ((hal_cau_dev_struct *)p_struct)->in_full_transcom_callback          = NULL;
        ((hal_cau_dev_struct *)p_struct)->in_half_transcom_callback          = NULL;
        ((hal_cau_dev_struct *)p_struct)->out_full_transcom_callback         = NULL;
        ((hal_cau_dev_struct *)p_struct)->out_half_transcom_callback         = NULL;
        ((hal_cau_dev_struct *)p_struct)->error_callback                     = NULL;
        ((hal_cau_dev_struct *)p_struct)->cau_irq.fifo_input_handle          = NULL;
        ((hal_cau_dev_struct *)p_struct)->cau_irq.fifo_output_handle         = NULL;
        ((hal_cau_dev_struct *)p_struct)->cau_dma.in_full_transcom_handle    = NULL;
        ((hal_cau_dev_struct *)p_struct)->cau_dma.in_half_transcom_handle    = NULL;
        ((hal_cau_dev_struct *)p_struct)->cau_dma.out_full_transcom_handle   = NULL;
        ((hal_cau_dev_struct *)p_struct)->cau_dma.out_half_transcom_handle   = NULL;
        ((hal_cau_dev_struct *)p_struct)->cau_dma.error_handle               = NULL;
        ((hal_cau_dev_struct *)p_struct)->p_dma_cau_in                       = NULL;
        ((hal_cau_dev_struct *)p_struct)->p_dma_cau_out                      = NULL;
        ((hal_cau_dev_struct *)p_struct)->error_code                         = HAL_CAU_ERROR_NONE;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->alg_dir                = CAU_ENCRYPTION_DIRECTION;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->key                    = NULL;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->key_size               = 0U;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->iv                     = NULL;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->iv_size                = 0U;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->input                  = NULL;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->in_length              = 0U;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->aad                    = NULL;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->aad_size               = 0U;
        ((hal_cau_dev_struct *)p_struct)->p_irq_para->data_change_mode       = CAU_DATA_EXCHANGE_32BIT;
        ((hal_cau_dev_struct *)p_struct)->in_buffer                          = NULL;
        ((hal_cau_dev_struct *)p_struct)->in_size                            = 0U;
        ((hal_cau_dev_struct *)p_struct)->out_buffer                         = NULL;
        ((hal_cau_dev_struct *)p_struct)->out_size                           = 0U;
        ((hal_cau_dev_struct *)p_struct)->aad_buffer                         = NULL;
        ((hal_cau_dev_struct *)p_struct)->aad_size                           = 0U;
        ((hal_cau_dev_struct *)p_struct)->tag                                = NULL;
        ((hal_cau_dev_struct *)p_struct)->tag_size                           = 0U;
        ((hal_cau_dev_struct *)p_struct)->mutex                              = HAL_MUTEX_UNLOCKED;
        ((hal_cau_dev_struct *)p_struct)->state                              = HAL_CAU_STATE_RESET;
        ((hal_cau_dev_struct *)p_struct)->phase                              = HAL_CAU_PHASE_PREPARE;
        break;
    case HAL_CAU_IRQ_STRUCT:
        ((hal_cau_irq_struct *)p_struct)->fifo_input_handle                  = NULL;
        ((hal_cau_irq_struct *)p_struct)->fifo_output_handle                 = NULL;
        break;
    case HAL_CAU_DMA_HANDLE_CB_STRUCT:
        ((hal_cau_dma_handle_cb_struct *)p_struct)->in_full_transcom_handle  = NULL;
        ((hal_cau_dma_handle_cb_struct *)p_struct)->in_half_transcom_handle  = NULL;
        ((hal_cau_dma_handle_cb_struct *)p_struct)->out_full_transcom_handle = NULL;
        ((hal_cau_dma_handle_cb_struct *)p_struct)->out_half_transcom_handle = NULL;
        ((hal_cau_dma_handle_cb_struct *)p_struct)->error_handle             = NULL;
        break;
    case HAL_CAU_CONTEXT_PARAMETER_STRUCT:
        ((hal_cau_context_parameter_struct *)p_struct)->ctl_config           = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->iv_0_high            = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->iv_0_low             = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->iv_1_high            = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->iv_1_low             = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->key_0_high           = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->key_0_low            = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->key_1_high           = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->key_1_low            = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->key_2_high           = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->key_2_low            = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->key_3_high           = 0U;
        ((hal_cau_context_parameter_struct *)p_struct)->key_3_low            = 0U;
        for(i = 0; i < 8; i++) {
            ((hal_cau_context_parameter_struct *)p_struct)->gcmccmctxs[i]    = 0U;
            ((hal_cau_context_parameter_struct *)p_struct)->gcmctxs[i]       = 0U;
        }
        break;
    case HAL_CAU_IRQ_USER_CALLBACK_STRUCT:
        ((hal_cau_irq_user_callback_struct *)p_struct)->fifo_input_callback   = NULL;
        ((hal_cau_irq_user_callback_struct *)p_struct)->fifo_output_callback  = NULL;
        break;
    case HAL_CAU_IV_PARAMETER_STRUCT:
        ((hal_cau_iv_parameter_struct *)p_struct)->iv_0_high                  = 0U;
        ((hal_cau_iv_parameter_struct *)p_struct)->iv_0_low                   = 0U;
        ((hal_cau_iv_parameter_struct *)p_struct)->iv_1_high                  = 0U;
        ((hal_cau_iv_parameter_struct *)p_struct)->iv_1_low                   = 0U;
        break;
    case HAL_CAU_KEY_PARAMETER_STRUCT:
        ((hal_cau_key_parameter_struct *)p_struct)->key_0_high                = 0U;
        ((hal_cau_key_parameter_struct *)p_struct)->key_0_low                 = 0U;
        ((hal_cau_key_parameter_struct *)p_struct)->key_1_high                = 0U;
        ((hal_cau_key_parameter_struct *)p_struct)->key_1_low                 = 0U;
        ((hal_cau_key_parameter_struct *)p_struct)->key_2_high                = 0U;
        ((hal_cau_key_parameter_struct *)p_struct)->key_2_low                 = 0U;
        ((hal_cau_key_parameter_struct *)p_struct)->key_3_high                = 0U;
        ((hal_cau_key_parameter_struct *)p_struct)->key_3_low                 = 0U;
        break;
    case HAL_CAU_PARAMETER_STRUCT:
        ((hal_cau_parameter_struct *)p_struct)->alg_dir                      = 0U;
        ((hal_cau_parameter_struct *)p_struct)->key                          = NULL;
        ((hal_cau_parameter_struct *)p_struct)->key_size                     = 0U;
        ((hal_cau_parameter_struct *)p_struct)->iv                           = NULL;
        ((hal_cau_parameter_struct *)p_struct)->iv_size                      = 0U;
        ((hal_cau_parameter_struct *)p_struct)->input                        = NULL;
        ((hal_cau_parameter_struct *)p_struct)->in_length                    = 0U;
        ((hal_cau_parameter_struct *)p_struct)->aad                          = NULL;
        ((hal_cau_parameter_struct *)p_struct)->aad_size                     = 0U;
        ((hal_cau_parameter_struct *)p_struct)->data_change_mode             = CAU_DATA_EXCHANGE_32BIT;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      CAU interrupt handler content function, which is merely used in CAU_IRQHandler
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_cau_irq(hal_cau_dev_struct *cau_dev)
{
    uint32_t algo = 0U;

    algo = (CAU_CTL & CAU_CTL_ALGM);

    /* if state */
    if((HAL_CAU_STATE_BUSY == cau_dev->state)) {
        /*!< fifo input interrupt handler */
        if((algo == CAU_MODE_TDES_ECB) || (algo == CAU_MODE_TDES_CBC) || (algo == CAU_MODE_DES_CBC) || \
           (algo == CAU_MODE_DES_ECB)) {
            /* TDES or DES */
            _cau_des_tdes_interrupt(cau_dev);
        } else if((algo == CAU_MODE_AES_ECB) || (algo == CAU_MODE_AES_CBC) || (algo == CAU_MODE_AES_CTR) || \
                  (algo == CAU_MODE_AES_CFB) || (algo == CAU_MODE_AES_OFB)) {
            /* AES(ECB,CBC,CTR,CFB,OFB) */
            _cau_aes_interrupt(cau_dev);
        } else if((algo == CAU_MODE_AES_CCM) || (algo == CAU_MODE_AES_GCM)) {
            /* AES(CCM) */
            _cau_aes_ccm_gcm_interrupt(cau_dev);
        } else {
            /* nothing to do */
        }

        /* input FIFO call back handler*/
        if(SET == hals_cau_flag_get(CAU_FLAG_INFIFO)) {
            if(NULL != cau_dev->cau_irq.fifo_input_handle) {
                cau_dev->cau_irq.fifo_input_handle(cau_dev);
            } else {
                /* nothing to do */
            }
            /* disable interrupt */
            hals_cau_interrupt_disable(CAU_INT_INFIFO);
        } else {
            /* nothing to do */
        }

        /* output FIFO call back handler*/
        if(SET == hals_cau_flag_get(CAU_FLAG_OUTFIFO)) {
            /* callback */
            if(NULL != cau_dev->cau_irq.fifo_output_handle) {
                cau_dev->cau_irq.fifo_output_handle(cau_dev);
            } else {
                /* nothing to do */
            }
            /* disable CAU */
            hals_cau_disable();
            /* disable interrupt */
            hals_cau_interrupt_disable(CAU_INT_INFIFO);
            /* set state */
            cau_dev->state = HAL_CAU_STATE_READY;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      set user-defined interrupt callback function, which will be registered and
                called when corresponding interrupt be triggered
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: pointer to CAU interrupt callback functions structure
                  The structure contains the following members:
      \arg        fifo_input_handle: input FIFO interrupt callback function
                  if not NULL, the callback mechanism is in use and corresponding interrupt is enabled
                  if NULL, the callback mechanism is out of use and corresponding interrupt is disabled
      \arg        fifo_output_handle: output FIFO interrupt callback function
                  if not NULL, the callback mechanism is in use and corresponding interrupt is enabled
                  if NULL, the callback mechanism is out of use and corresponding interrupt is disabled
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_cau_irq_handle_set(hal_cau_dev_struct *cau_dev, hal_cau_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == cau_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [cau_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1 = HAL_PARAMETER_CHECK */
    cau_dev->mutex = HAL_MUTEX_LOCKED;

    /* CAU fifo input interrupt call */
    if(NULL != p_irq->fifo_input_handle) {
        cau_dev->cau_irq.fifo_input_handle = p_irq->fifo_input_handle;
        /* enable CAU fifo input interrupt */
        hals_cau_interrupt_enable(CAU_INT_INFIFO);
    } else {
        cau_dev->cau_irq.fifo_input_handle = NULL;
        /* disable CAU fifo input interrupt */
        hals_cau_interrupt_disable(CAU_INT_INFIFO);
    }

    /* CAU fifo output interrupt call */
    if(NULL != p_irq->fifo_output_handle) {
        cau_dev->cau_irq.fifo_output_handle = p_irq->fifo_output_handle;
        /* enable CAU fifo output interrupt */
        hals_cau_interrupt_enable(CAU_INT_OUTFIFO);
    } else {
        cau_dev->cau_irq.fifo_output_handle = NULL;
        /* disable CAU fifo output interrupt */
        hals_cau_interrupt_disable(CAU_INT_OUTFIFO);
    }

    cau_dev->mutex = HAL_MUTEX_UNLOCKED;

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function, which will be registered and
                called when corresponding interrupt be triggered
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_cau_irq_handle_all_reset(hal_cau_dev_struct *cau_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == cau_dev) {
        HAL_DEBUGE("pointer [cau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1 = HAL_PARAMETER_CHECK */
    cau_dev->mutex = HAL_MUTEX_LOCKED;

    cau_dev->cau_irq.fifo_input_handle  = NULL;
    cau_dev->cau_irq.fifo_output_handle = NULL;

    cau_dev->mutex = HAL_MUTEX_UNLOCKED;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the CAU encrypt and decrypt parameter struct with the default values
    \param[in]  none
    \param[in]  cau_parameter: pointer to the input structure
                  alg_dir: algorithm directory
                  only one parameters can be selected which are shown as below:
      \arg          CAU_ENCRYPT: encrypt
      \arg          CAU_DECRYPT: decrypt
                  key: key
                  key_size: key size in bits, must be either 128, 192 or 256
                  iv: initialization vector
                  iv_size: iv size in bytes
                  input: input data
                  in_length: input data length in bytes, must be a multiple of 16 bytes
                  aad: additional authentication data
                  aad_size: aad size in bytes, must be a multiple of 16 bytes
                  data_change_mode:
                  only one parameters can be selected which are shown as below:
      \arg          CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg          CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg          CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg          CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] none
    \retval     none
*/
void hal_cau_encrypt_decrypt_para_init(hal_cau_parameter_struct *cau_parameter)
{
    /* set the CAU encrypt and decrypt parameters struct with the default values */
    cau_parameter->alg_dir   = CAU_ENCRYPT;
    cau_parameter->key       = 0U;
    cau_parameter->key_size  = 0U;
    cau_parameter->iv        = 0U;
    cau_parameter->iv_size   = 0U;
    cau_parameter->input     = 0U;
    cau_parameter->in_length = 0U;
    cau_parameter->aad       = 0U;
    cau_parameter->aad_size  = 0U;
}

/*!
    \brief      save context before context switching
    \param[in]  key_initpara: key init parameter struct
                  key_0_high: key 0 high
                  key_0_low:  key 0 low
                  key_1_high: key 1 high
                  key_1_low:  key 1 low
                  key_2_high: key 2 high
                  key_2_low:  key 2 low
                  key_3_high: key 3 high
                  key_3_low:  key 3 low
    \param[out] cau_context:
                  ctl_config: current configuration
                  iv_0_high: init vector 0 high
                  iv_0_low:  init vector 0 low
                  iv_1_high: init vector 1 high
                  iv_1_low:  init vector 1 low
                  key_0_high: key 0 high
                  key_0_low:  key 0 low
                  key_1_high: key 1 high
                  key_1_low:  key 1 low
                  key_2_high: key 2 high
                  key_2_low:  key 2 low
                  key_3_high: key 3 high
                  key_3_low:  key 3 low
                  gcmccmctxs[8]: GCM or CCM mode context switch
                  gcmctxs[8]: GCM mode context switch
    \retval     none
*/
void hal_cau_context_save(hal_cau_context_parameter_struct *cau_context, hal_cau_key_parameter_struct *key_initpara)
{
    uint32_t checkmask = 0U;
    uint32_t checkbits = 0U;
    uint32_t algm_reg  = 0U;

    /* stop DMA transfers on the in fifo by clearing the DMAIEN bit in the CAU_DMAEN */
    CAU_DMAEN &= ~CAU_DMA_INFIFO;

    algm_reg = CAU_CTL & CAU_CTL_ALGM;
    /* AES or DES */
    if(algm_reg & (~CAU_MODE_TDES_CBC)) {
        /* wait until both the IN and out fifo are empty (IEM=1 and ONE=0 in the CAU_STAT0 register) and BUSY=0 */
        checkbits = CAU_STAT0_IEM;
        checkmask = STAT0_AESDES_MASK;
        /* TDES */
    } else {
        /* wait until out fifo is empty (ONE=0 in the CAU_STAT0 register) and BUSY=0 */
        checkbits = 0U;
        checkmask = STAT0_TDES_MASK;
    }

    while((CAU_STAT0 & checkmask) != checkbits) {}

    /* stop DMA transfers on the out fifo by clear CAU_DMAEN_DMAOEN=0 */
    CAU_DMAEN &= ~CAU_DMAEN_DMAOEN;
    /* disable CAU */
    CAU_CTL &= ~CAU_CTL_CAUEN;

    /* save the current configuration (bit 19, bit[17:16] and bit[9:2] in the CAU_CTL register) */
    cau_context->ctl_config = CAU_CTL & (CAU_CTL_GCM_CCMPH | CAU_CTL_KEYM | CAU_CTL_DATAM | \
                                         CAU_CTL_ALGM | CAU_CTL_CAUDIR | CAU_CTL_NBPILB);

    /* save the key value */
    cau_context->key_0_high = key_initpara->key_0_high;
    cau_context->key_0_low  = key_initpara->key_0_low;
    cau_context->key_1_high = key_initpara->key_1_high;
    cau_context->key_1_low  = key_initpara->key_1_low;
    cau_context->key_2_high = key_initpara->key_2_high;
    cau_context->key_2_low  = key_initpara->key_2_low;
    cau_context->key_3_high = key_initpara->key_3_high;
    cau_context->key_3_low  = key_initpara->key_3_low;

    if((CAU_MODE_TDES_ECB != algm_reg) && (CAU_MODE_DES_ECB != algm_reg) && (CAU_MODE_AES_ECB != algm_reg)) {
        /* if not in ECB mode, save the initialization vectors */
        cau_context->iv_0_high = CAU_IV0H;
        cau_context->iv_0_low  = CAU_IV0L;
        cau_context->iv_1_high = CAU_IV1H;
        cau_context->iv_1_low  = CAU_IV1L;
    } else {
        /* nothing to do */
    }

    /* if in GCM/CCM mode, save the context switch registers */
    if((CAU_MODE_AES_GCM == algm_reg) || (CAU_MODE_AES_CCM == algm_reg)) {
        cau_context->gcmccmctxs[0U] = CAU_GCMCCMCTXSx(0U);
        cau_context->gcmccmctxs[1U] = CAU_GCMCCMCTXSx(1U);
        cau_context->gcmccmctxs[2U] = CAU_GCMCCMCTXSx(2U);
        cau_context->gcmccmctxs[3U] = CAU_GCMCCMCTXSx(3U);
        cau_context->gcmccmctxs[4U] = CAU_GCMCCMCTXSx(4U);
        cau_context->gcmccmctxs[5U] = CAU_GCMCCMCTXSx(5U);
        cau_context->gcmccmctxs[6U] = CAU_GCMCCMCTXSx(6U);
        cau_context->gcmccmctxs[7U] = CAU_GCMCCMCTXSx(7U);
    } else {
        /* nothing to do */
    }

    /* if in GCM mode, save the context switch registers */
    if(CAU_MODE_AES_GCM == algm_reg) {
        cau_context->gcmctxs[0U] = CAU_GCMCTXSx(0U);
        cau_context->gcmctxs[1U] = CAU_GCMCTXSx(1U);
        cau_context->gcmctxs[2U] = CAU_GCMCTXSx(2U);
        cau_context->gcmctxs[3U] = CAU_GCMCTXSx(3U);
        cau_context->gcmctxs[4U] = CAU_GCMCTXSx(4U);
        cau_context->gcmctxs[5U] = CAU_GCMCTXSx(5U);
        cau_context->gcmctxs[6U] = CAU_GCMCTXSx(6U);
        cau_context->gcmctxs[7U] = CAU_GCMCTXSx(7U);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      restore context after context switching
    \param[in]  cau_context:
                  ctl_config: current configuration
                  iv_0_high: init vector 0 high
                  iv_0_low:  init vector 0 low
                  iv_1_high: init vector 1 high
                  iv_1_low:  init vector 1 low
                  key_0_high: key 0 high
                  key_0_low:  key 0 low
                  key_1_high: key 1 high
                  key_1_low:  key 1 low
                  key_2_high: key 2 high
                  key_2_low:  key 2 low
                  key_3_high: key 3 high
                  key_3_low:  key 3 low
                  gcmccmctxs[8]: GCM or CCM mode context switch
                  gcmctxs[8]: GCM mode context switch
    \param[out] none
    \retval     none
*/
void hal_cau_context_restore(hal_cau_context_parameter_struct *cau_context)
{
    uint32_t algm_reg, aes_decrypt;
    uint32_t alg_dir, algo_mode;
    hal_cau_data_exchange_mode_enum swapping;

    /* configure the processor with the saved configuration */
    CAU_CTL = cau_context->ctl_config;

    algm_reg = CAU_CTL & CAU_CTL_ALGM;

    /* restore the key value */
    CAU_KEY0H = cau_context->key_0_high;
    CAU_KEY0L = cau_context->key_0_low;
    CAU_KEY1H = cau_context->key_1_high;
    CAU_KEY1L = cau_context->key_1_low;
    CAU_KEY2H = cau_context->key_2_high;
    CAU_KEY2L = cau_context->key_2_low;
    CAU_KEY3H = cau_context->key_3_high;
    CAU_KEY3L = cau_context->key_3_low;

    if((CAU_MODE_TDES_ECB != algm_reg) && (CAU_MODE_DES_ECB != algm_reg) && (CAU_MODE_AES_ECB != algm_reg)) {
        /* restore the initialization vectors */
        CAU_IV0H = cau_context->iv_0_high;
        CAU_IV0L = cau_context->iv_0_low;
        CAU_IV1H = cau_context->iv_1_high;
        CAU_IV1L = cau_context->iv_1_low;
    } else {
        /* nothing to do */
    }

    /* if in GCM/CCM mode, restore the context switch registers */
    if((CAU_MODE_AES_GCM == algm_reg) || (CAU_MODE_AES_CCM == algm_reg)) {
        CAU_GCMCCMCTXSx(0U) = cau_context->gcmccmctxs[0U];
        CAU_GCMCCMCTXSx(1U) = cau_context->gcmccmctxs[1U];
        CAU_GCMCCMCTXSx(2U) = cau_context->gcmccmctxs[2U];
        CAU_GCMCCMCTXSx(3U) = cau_context->gcmccmctxs[3U];
        CAU_GCMCCMCTXSx(4U) = cau_context->gcmccmctxs[4U];
        CAU_GCMCCMCTXSx(5U) = cau_context->gcmccmctxs[5U];
        CAU_GCMCCMCTXSx(6U) = cau_context->gcmccmctxs[6U];
        CAU_GCMCCMCTXSx(7U) = cau_context->gcmccmctxs[7U];
    } else {
        /* nothing to do */
    }

    /* if in GCM mode, restore the context switch registers */
    if(CAU_MODE_AES_GCM == algm_reg) {
        CAU_GCMCTXSx(0U) = cau_context->gcmctxs[0U];
        CAU_GCMCTXSx(1U) = cau_context->gcmctxs[1U];
        CAU_GCMCTXSx(2U) = cau_context->gcmctxs[2U];
        CAU_GCMCTXSx(3U) = cau_context->gcmctxs[3U];
        CAU_GCMCTXSx(4U) = cau_context->gcmctxs[4U];
        CAU_GCMCTXSx(5U) = cau_context->gcmctxs[5U];
        CAU_GCMCTXSx(6U) = cau_context->gcmctxs[6U];
        CAU_GCMCTXSx(7U) = cau_context->gcmctxs[7U];
    } else {
        /* nothing to do */
    }

    /* if it is AES ECB/CBC decryption, then first prepare key */
    aes_decrypt = CAU_CTL & (CAU_CTL_ALGM | CAU_CTL_CAUDIR);
    if(((CAU_MODE_AES_ECB | CAU_DECRYPT) == aes_decrypt) || ((CAU_MODE_AES_CBC | CAU_DECRYPT) == aes_decrypt)) {
        /* flush IN/out fifo */
        CAU_CTL |= CAU_CTL_FFLUSH;
        /* parameters for key preparation for AES decryption */
        alg_dir   = CAU_DECRYPT;
        algo_mode = CAU_MODE_AES_KEY;
        swapping  = CAU_DATA_EXCHANGE_32BIT;
        hals_cau_init_param(alg_dir, algo_mode, swapping);

        hals_cau_enable();

        /* wait until BUSY=0 */
        while((uint32_t)0U != hals_cau_flag_get(CAU_FLAG_BUSY)) {}

        /* parameters for decryption */
        CAU_CTL = cau_context->ctl_config;
    } else {
        /* nothing to do */
    }

    hals_cau_enable();
}

/*!
    \brief      cau stop encrypt and decrypt in interrupt mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_cau_interrupt_stop(hal_cau_dev_struct *cau_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == cau_dev) {
        HAL_DEBUGE("pointer [cau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1 = HAL_PARAMETER_CHECK */

    HAL_LOCK(cau_dev);

    hals_cau_interrupt_disable(CAU_INT_INFIFO | CAU_INT_OUTFIFO);
    cau_dev->phase = HAL_CAU_PHASE_PREPARE;
    cau_dev->state = HAL_CAU_STATE_READY;

    HAL_UNLOCK(cau_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      cau stop encrypt and decrypt in dma mode
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_cau_dma_stop(hal_cau_dev_struct *cau_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == cau_dev) {
        HAL_DEBUGE("pointer [cau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1 = HAL_PARAMETER_CHECK */

    HAL_LOCK(cau_dev);

    /* enable cau in dma and out dma transfer */
    if(NULL != cau_dev->p_dma_cau_in) {
        hal_dma_stop(cau_dev->p_dma_cau_in);
    } else {
        /* nothing to do */
    }

    if(NULL != cau_dev->p_dma_cau_out) {
        hal_dma_stop(cau_dev->p_dma_cau_out);
    } else {
        /* nothing to do */
    }

    CAU_DMAEN &= ~(CAU_DMA_INFIFO | CAU_DMA_OUTFIFO);
    cau_dev->phase = HAL_CAU_PHASE_PREPARE;
    cau_dev->state = HAL_CAU_STATE_READY;

    HAL_UNLOCK(cau_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      cau encrypt and decrypt for poll
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
    \param[in]  cau_mode: The parameters you can set are as follows:
                only one parameters can be selected which are shown as below:
      \arg        CAU_MODE_TDES_ECB: TDES-ECB (3DES Electronic codebook)
      \arg        CAU_MODE_TDES_CBC: TDES-CBC (3DES Cipher block chaining)
      \arg        CAU_MODE_DES_ECB: DES-ECB (simple DES Electronic codebook)
      \arg        CAU_MODE_DES_CBC: DES-CBC (simple DES Cipher block chaining)
      \arg        CAU_MODE_AES_ECB: AES-ECB (AES Electronic codebook)
      \arg        CAU_MODE_AES_CBC: AES-CBC (AES Cipher block chaining)
      \arg        CAU_MODE_AES_CTR: AES-CTR (AES counter mode)
      \arg        CAU_MODE_AES_GCM: AES-GCM (AES Galois/counter mode)
      \arg        CAU_MODE_AES_CCM: AES-CCM (AES combined cipher machine mode)
      \arg        CAU_MODE_AES_CFB: AES-CFB (cipher feedback mode)
      \arg        CAU_MODE_AES_OFB: AES-OFB (output feedback mode)
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hal_cau_encrypt_decrypt_poll(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                       uint8_t *output, uint32_t cau_mode)
{
    ErrStatus ret = SUCCESS;

    switch(cau_mode) {
    case CAU_MODE_TDES_ECB:
        ret = hals_cau_tdes_ecb(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_TDES_CBC:
        ret = hals_cau_tdes_cbc(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_DES_ECB:
        ret = hals_cau_des_ecb(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_DES_CBC:
        ret = hals_cau_des_cbc(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_AES_ECB:
        ret = hals_cau_aes_ecb(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_AES_CBC:
        ret = hals_cau_aes_cbc(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_AES_CTR:
        ret = hals_cau_aes_ctr(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_AES_CFB:
        ret = hals_cau_aes_cfb(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_AES_OFB:
        ret = hals_cau_aes_ofb(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_AES_CCM:
        ret = hals_cau_aes_ccm(cau_dev, cau_parameter, output);
        break;
    case CAU_MODE_AES_GCM:
        ret = hals_cau_aes_gcm(cau_dev, cau_parameter, output);
        break;
    default:
        HAL_DEBUGE("cau_mode parameter invalid!");
        ret = ERROR;
        break;
    }

    return ret;
}

/*!
    \brief      cau encrypt and decrypt for interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
    \param[in]  cau_mode: The parameters you can set are as follows:
                only one parameters can be selected which are shown as below:
      \arg        CAU_MODE_TDES_ECB: TDES-ECB (3DES Electronic codebook)
      \arg        CAU_MODE_TDES_CBC: TDES-CBC (3DES Cipher block chaining)
      \arg        CAU_MODE_DES_ECB: DES-ECB (simple DES Electronic codebook)
      \arg        CAU_MODE_DES_CBC: DES-CBC (simple DES Cipher block chaining)
      \arg        CAU_MODE_AES_ECB: AES-ECB (AES Electronic codebook)
      \arg        CAU_MODE_AES_CBC: AES-CBC (AES Cipher block chaining)
      \arg        CAU_MODE_AES_CTR: AES-CTR (AES counter mode)
      \arg        CAU_MODE_AES_GCM: AES-GCM (AES Galois/counter mode)
      \arg        CAU_MODE_AES_CCM: AES-CCM (AES combined cipher machine mode)
      \arg        CAU_MODE_AES_CFB: AES-CFB (cipher feedback mode)
      \arg        CAU_MODE_AES_OFB: AES-OFB (output feedback mode)
    \param[in]  p_user_func: The structure contains the following members:
      \arg        fifo_input_callback: input data transfer complete callback function
      \arg        fifo_output_callback: output data transfer complete callback function
    \param[out] output: pointer to the returned buffer
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hal_cau_encrypt_decrypt_interrupt(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                            hal_cau_irq_user_callback_struct *p_user_func, uint8_t *output, \
                                            uint32_t cau_mode)
{
    ErrStatus ret = SUCCESS;

    switch(cau_mode) {
    case CAU_MODE_TDES_ECB:
        hals_cau_tdes_ecb_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    case CAU_MODE_TDES_CBC:
        hals_cau_tdes_cbc_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    case CAU_MODE_DES_ECB:
        hals_cau_des_ecb_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    case CAU_MODE_DES_CBC:
        hals_cau_des_cbc_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    case CAU_MODE_AES_ECB:
        hals_cau_aes_ecb_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    case CAU_MODE_AES_CBC:
        hals_cau_aes_cbc_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    case CAU_MODE_AES_CTR:
        hals_cau_aes_ctr_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    case CAU_MODE_AES_CFB:
        hals_cau_aes_cfb_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    case CAU_MODE_AES_OFB:
        hals_cau_aes_ofb_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    case CAU_MODE_AES_CCM:
    case CAU_MODE_AES_GCM:
        hals_cau_aes_ccm_gcm_interrupt(cau_dev, cau_parameter, p_user_func, output);
        break;
    default:
        HAL_DEBUGE("cau_mode parameter invalid!");
        ret = ERROR;
        break;
    }

    return ret;
}

/*!
    \brief      cau encrypt and decrypt for dma
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cau_parameter: pointer to the input structure
    \param[in]  cau_mode: The parameters you can set are as follows:
                only one parameters can be selected which are shown as below:
      \arg        CAU_MODE_TDES_ECB: TDES-ECB (3DES Electronic codebook)
      \arg        CAU_MODE_TDES_CBC: TDES-CBC (3DES Cipher block chaining)
      \arg        CAU_MODE_DES_ECB: DES-ECB (simple DES Electronic codebook)
      \arg        CAU_MODE_DES_CBC: DES-CBC (simple DES Cipher block chaining)
      \arg        CAU_MODE_AES_ECB: AES-ECB (AES Electronic codebook)
      \arg        CAU_MODE_AES_CBC: AES-CBC (AES Cipher block chaining)
      \arg        CAU_MODE_AES_CTR: AES-CTR (AES counter mode)
      \arg        CAU_MODE_AES_CFB: AES-CFB (cipher feedback mode)
      \arg        CAU_MODE_AES_OFB: AES-OFB (output feedback mode)
    \param[in]  dmacb: CAU DMA callback function pointer structure
                only one parameter can be selected which is shown as below:
      \arg        in_full_transcom_handle: CAU DMA input full transfer complete interrupt handler
      \arg        in_half_transcom_handle: CAU DMA input half transfer complete interrupt handler
      \arg        out_full_transcom_handle: CAU DMA output full transfer complete interrupt handler
      \arg        out_half_transcom_handle: CAU DMA output half transfer complete interrupt handler
      \arg        error_handle:CAU DMA error interrupt handler
    \param[out] output: pointer to the returned buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_cau_encrypt_decrypt_dma(hal_cau_dev_struct *cau_dev, hal_cau_parameter_struct *cau_parameter, \
                                    hal_cau_dma_handle_cb_struct *dmacb, uint8_t *output, uint32_t cau_mode)
{
    int32_t ret = HAL_ERR_NONE;

    switch(cau_mode) {
    case CAU_MODE_TDES_ECB:
        hals_cau_tdes_ecb_dma(cau_dev, cau_parameter, dmacb, output);
        break;
    case CAU_MODE_TDES_CBC:
        hals_cau_tdes_cbc_dma(cau_dev, cau_parameter, dmacb, output);
        break;
    case CAU_MODE_DES_ECB:
        hals_cau_des_ecb_dma(cau_dev, cau_parameter, dmacb, output);
        break;
    case CAU_MODE_DES_CBC:
        hals_cau_des_cbc_dma(cau_dev, cau_parameter, dmacb, output);
        break;
    case CAU_MODE_AES_ECB:
        hals_cau_aes_ecb_dma(cau_dev, cau_parameter, dmacb, output);
        break;
    case CAU_MODE_AES_CBC:
        hals_cau_aes_cbc_dma(cau_dev, cau_parameter, dmacb, output);
        break;
    case CAU_MODE_AES_CTR:
        hals_cau_aes_ctr_dma(cau_dev, cau_parameter, dmacb, output);
        break;
    case CAU_MODE_AES_CFB:
        hals_cau_aes_cfb_dma(cau_dev, cau_parameter, dmacb, output);
        break;
    case CAU_MODE_AES_OFB:
        hals_cau_aes_ofb_dma(cau_dev, cau_parameter, dmacb, output);
        break;

    default:
        HAL_DEBUGE("parameter [cau_mode] value is undefine!");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      return whether CAU peripheral is enabled or disabled
    \param[in]  none
    \param[out] none
    \retval     ControlStatus: ENABLE or DISABLE
*/
ControlStatus hal_cau_enable_state_get(void)
{
    ControlStatus ret = DISABLE;
    if(RESET != (CAU_CTL & CAU_CTL_CAUEN)) {
        ret = ENABLE;
    } else {
        /* nothing to do */
    }
    return ret;
}

/*!
    \brief      get CAU state
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     CAU state
*/
hal_cau_state_enum hal_cau_state_get(hal_cau_dev_struct *cau_dev)
{
    return cau_dev->state;
}

/*!
    \brief      get CAU error code
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     CAU error code
*/
hal_cau_error_enum hal_cau_error_code_get(hal_cau_dev_struct *cau_dev)
{
    return cau_dev->error_code;
}

/*!
    \brief      configure key selection
    \param[in]  key_selection: key source selection when aes mode
                only one parameter can be selected which is shown as below:
      \arg        CAU_KEY:       use the key from CAU register
      \arg        CAU_EFUSE_KEY: use the key from EFUSE
    \param[out] none
    \retval     none
*/
void hals_cau_aes_key_select(uint32_t key_selection)
{
    CAU_CTL &= ~CAU_CTL_KEY_SEL;
    CAU_CTL |= key_selection;
}

/*!
    \brief      initialize the vectors parameter struct with the default values
    \param[in]  none
    \param[out] iv_initpara:
                  iv_0_high: init vector 0 high
                  iv_0_low:  init vector 0 low
                  iv_1_high: init vector 1 high
                  iv_1_low:  init vector 1 low
    \retval     none
*/
void hals_cau_iv_struct_para_init(hal_cau_iv_parameter_struct *iv_initpara)
{
    /* set the vectors parameters struct with the default values */
    iv_initpara->iv_0_high = 0U;
    iv_initpara->iv_0_low  = 0U;
    iv_initpara->iv_1_high = 0U;
    iv_initpara->iv_1_low  = 0U;
}

/*!
    \brief      initialize the context parameter struct with the default values
    \param[in]  none
    \param[out] cau_context:
                  ctl_config: current configuration
                  iv_0_high: init vector 0 high
                  iv_0_low:  init vector 0 low
                  iv_1_high: init vector 1 high
                  iv_1_low:  init vector 1 low
                  key_0_high: key 0 high
                  key_0_low:  key 0 low
                  key_1_high: key 1 high
                  key_1_low:  key 1 low
                  key_2_high: key 2 high
                  key_2_low:  key 2 low
                  key_3_high: key 3 high
                  key_3_low:  key 3 low
                  gcmccmctxs[8]: GCM or CCM mode context switch
                  gcmctxs[8]: GCM mode context switch
    \retval     none
*/
void hals_cau_context_struct_para_init(hal_cau_context_parameter_struct *cau_context)
{
    cau_context->ctl_config = 0U;

    /* set the vectors parameters with the default values */
    cau_context->iv_0_high = 0U;
    cau_context->iv_0_low  = 0U;
    cau_context->iv_1_high = 0U;
    cau_context->iv_1_low  = 0U;

    /* set the key parameters with the default values */
    cau_context->key_0_high = 0U;
    cau_context->key_0_low  = 0U;
    cau_context->key_1_high = 0U;
    cau_context->key_1_low  = 0U;
    cau_context->key_2_high = 0U;
    cau_context->key_2_low  = 0U;
    cau_context->key_3_high = 0U;
    cau_context->key_3_low  = 0U;

    /* set the context switch with the default values */
    cau_context->gcmccmctxs[0] = 0U;
    cau_context->gcmccmctxs[1] = 0U;
    cau_context->gcmccmctxs[2] = 0U;
    cau_context->gcmccmctxs[3] = 0U;
    cau_context->gcmccmctxs[4] = 0U;
    cau_context->gcmccmctxs[5] = 0U;
    cau_context->gcmccmctxs[6] = 0U;
    cau_context->gcmccmctxs[7] = 0U;

    cau_context->gcmctxs[0] = 0U;
    cau_context->gcmctxs[1] = 0U;
    cau_context->gcmctxs[2] = 0U;
    cau_context->gcmctxs[3] = 0U;
    cau_context->gcmctxs[4] = 0U;
    cau_context->gcmctxs[5] = 0U;
    cau_context->gcmctxs[6] = 0U;
    cau_context->gcmctxs[7] = 0U;
}

/*!
    \brief      initialize the CAU peripheral
    \param[in]  alg_dir: algorithm direction
                only one parameter can be selected which is shown as below:
      \arg        CAU_ENCRYPT: encrypt
      \arg        CAU_DECRYPT: decrypt
    \param[in]  algo_mode: algorithm mode selection
                only one parameter can be selected which is shown as below:
      \arg        CAU_MODE_TDES_ECB: TDES-ECB (3DES Electronic codebook)
      \arg        CAU_MODE_TDES_CBC: TDES-CBC (3DES Cipher block chaining)
      \arg        CAU_MODE_DES_ECB: DES-ECB (simple DES Electronic codebook)
      \arg        CAU_MODE_DES_CBC: DES-CBC (simple DES Cipher block chaining)
      \arg        CAU_MODE_AES_ECB: AES-ECB (AES Electronic codebook)
      \arg        CAU_MODE_AES_CBC: AES-CBC (AES Cipher block chaining)
      \arg        CAU_MODE_AES_CTR: AES-CTR (AES counter mode)
      \arg        CAU_MODE_AES_KEY: AES decryption key preparation mode
      \arg        CAU_MODE_AES_GCM: AES-GCM (AES Galois/counter mode)
      \arg        CAU_MODE_AES_CCM: AES-CCM (AES combined cipher machine mode)
      \arg        CAU_MODE_AES_CFB: AES-CFB (cipher feedback mode)
      \arg        CAU_MODE_AES_OFB: AES-OFB (output feedback mode)
    \param[in]  swapping: data swapping selection
                only one parameter can be selected which is shown as below:
      \arg         CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg         CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg         CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg         CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] none
    \retval     none
*/
void hals_cau_init_param(uint32_t alg_dir, uint32_t algo_mode, hal_cau_data_exchange_mode_enum swapping)
{
    /* select algorithm mode */
    CAU_CTL &= ~CAU_CTL_ALGM;
    CAU_CTL |= algo_mode;

    /* select data swapping */
    CAU_CTL &= ~CAU_CTL_DATAM;
    CAU_CTL |= swapping;

    /* select algorithm direction */
    CAU_CTL &= ~CAU_CTL_CAUDIR;
    CAU_CTL |= alg_dir;
}

/*!
    \brief      get the current CAU parameters
    \param[in]  alg_dir: pointer to store algorithm direction
                  The value returned can be one of the following:
      \arg        CAU_ENCRYPT: encrypt
      \arg        CAU_DECRYPT: decrypt
    \param[in]  algo_mode: pointer to store algorithm mode
                  The value returned can be one of the following:
      \arg        CAU_MODE_TDES_ECB: TDES-ECB (3DES Electronic codebook)
      \arg        CAU_MODE_TDES_CBC: TDES-CBC (3DES Cipher block chaining)
      \arg        CAU_MODE_DES_ECB: DES-ECB (simple DES Electronic codebook)
      \arg        CAU_MODE_DES_CBC: DES-CBC (simple DES Cipher block chaining)
      \arg        CAU_MODE_AES_ECB: AES-ECB (AES Electronic codebook)
      \arg        CAU_MODE_AES_CBC: AES-CBC (AES Cipher block chaining)
      \arg        CAU_MODE_AES_CTR: AES-CTR (AES counter mode)
      \arg        CAU_MODE_AES_KEY: AES decryption key preparation mode
      \arg        CAU_MODE_AES_GCM: AES-GCM (AES Galois/counter mode)
      \arg        CAU_MODE_AES_CCM: AES-CCM (AES combined cipher machine mode)
      \arg        CAU_MODE_AES_CFB: AES-CFB (cipher feedback mode)
      \arg        CAU_MODE_AES_OFB: AES-OFB (output feedback mode)
    \param[in]  swapping: pointer to store data swapping selection
                  The value returned can be one of the following:
      \arg        CAU_DATA_EXCHANGE_32BIT: no swapping
      \arg        CAU_DATA_EXCHANGE_16BIT: half-word swapping
      \arg        CAU_DATA_EXCHANGE_8BIT: bytes swapping
      \arg        CAU_DATA_EXCHANGE_1BIT: bit swapping
    \param[out] none
    \retval     none
*/
void hals_cau_get_current_param(uint32_t *alg_dir, uint32_t *algo_mode, hal_cau_data_exchange_mode_enum *swapping)
{
    uint32_t ctl_value = CAU_CTL;

    /* extract algorithm mode */
    *algo_mode = ctl_value & CAU_CTL_ALGM;

    /* extract data swapping */
    *swapping = (hal_cau_data_exchange_mode_enum)(ctl_value & CAU_CTL_DATAM);

    /* extract algorithm direction */
    *alg_dir = ctl_value & CAU_CTL_CAUDIR;
}

/*!
    \brief      initialize the key parameters
    \param[in]  key_initpara: key init parameter struct
                  key_0_high: key 0 high
                  key_0_low:  key 0 low
                  key_1_high: key 1 high
                  key_1_low:  key 1 low
                  key_2_high: key 2 high
                  key_2_low:  key 2 low
                  key_3_high: key 3 high
                  key_3_low:  key 3 low
    \param[out] none
    \retval     none
*/
void hals_cau_key_init(hal_cau_key_parameter_struct *key_initpara)
{
    CAU_KEY0H = key_initpara->key_0_high;
    CAU_KEY0L = key_initpara->key_0_low;
    CAU_KEY1H = key_initpara->key_1_high;
    CAU_KEY1L = key_initpara->key_1_low;
    CAU_KEY2H = key_initpara->key_2_high;
    CAU_KEY2L = key_initpara->key_2_low;
    CAU_KEY3H = key_initpara->key_3_high;
    CAU_KEY3L = key_initpara->key_3_low;
}

/*!
    \brief      initialize the vectors parameters
    \param[in]  iv_initpara: vectors init parameter struct
                  iv_0_high: init vector 0 high
                  iv_0_low:  init vector 0 low
                  iv_1_high: init vector 1 high
                  iv_1_low:  init vector 1 low
    \param[out] none
    \retval     none
*/
void hals_cau_iv_init(hal_cau_iv_parameter_struct *iv_initpara)
{
    CAU_IV0H = iv_initpara->iv_0_high;
    CAU_IV0L = iv_initpara->iv_0_low;
    CAU_IV1H = iv_initpara->iv_1_high;
    CAU_IV1L = iv_initpara->iv_1_low;
}

/*!
    \brief      configure phase
    \param[in]  phase: gcm or ccm phase
                only one parameter can be selected which is shown as below:
      \arg        CAU_PREPARE_PHASE: prepare phase
      \arg        CAU_AAD_PHASE: AAD phase
      \arg        CAU_ENCRYPT_DECRYPT_PHASE: encryption/decryption phase
      \arg        CAU_TAG_PHASE: tag phase
    \param[out] none
    \retval     none
*/
void hals_cau_phase_config(uint32_t phase)
{
    uint32_t temp;
    /* Get the CTL register */
    temp = CAU_CTL;
    /* Reset the phase configuration bits */
    temp &= ~CAU_CTL_GCM_CCMPH;
    /* Set the selected phase */
    temp |= phase;
    /* Set the CTL register */
    CAU_CTL = temp;
}

/*!
    \brief      CAU dma transfer param config
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dmacb: CAU DMA callback function pointer structure
                only one parameter can be selected which is shown as below:
      \arg        in_full_transcom_handle: CAU DMA input full transfer complete interrupt handler
      \arg        in_half_transcom_handle: CAU DMA input half transfer complete interrupt handler
      \arg        out_full_transcom_handle: CAU DMA output full transfer complete interrupt handler
      \arg        out_half_transcom_handle: CAU DMA output half transfer complete interrupt handler
      \arg        error_handle:CAU DMA error interrupt handler
    \param[out] none
    \retval     none
*/
void hals_cau_dma_transfer_config(hal_cau_dev_struct *cau_dev, hal_cau_dma_handle_cb_struct *dmacb)
{
    hal_dma_irq_struct dma_in_irq;
    hal_dma_irq_struct dma_out_irq;

    memset(&dma_in_irq, 0, sizeof(dma_in_irq));
    memset(&dma_out_irq, 0, sizeof(dma_out_irq));

    cau_dev->in_full_transcom_callback  = NULL;
    cau_dev->in_half_transcom_callback  = NULL;
    cau_dev->out_full_transcom_callback = NULL;
    cau_dev->out_half_transcom_callback = NULL;
    cau_dev->error_callback             = NULL;

    /* user callback */
    if(NULL != dmacb) {
        if(NULL != dmacb->in_full_transcom_handle) {
            cau_dev->in_full_transcom_callback = (void *)dmacb->in_full_transcom_handle;
        } else {
            /* nothing to do */
        }

        if(NULL != dmacb->in_half_transcom_handle) {
            cau_dev->in_half_transcom_callback = (void *)dmacb->in_half_transcom_handle;
        } else {
            /* nothing to do */
        }

        if(NULL != dmacb->out_full_transcom_handle) {
            cau_dev->out_full_transcom_callback = (void *)dmacb->out_full_transcom_handle;
        } else {
            /* nothing to do */
        }

        if(NULL != dmacb->out_half_transcom_handle) {
            cau_dev->out_half_transcom_callback = (void *)dmacb->out_half_transcom_handle;
        } else {
            /* nothing to do */
        }

        if(NULL != dmacb->error_handle) {
            cau_dev->error_callback = (void *)dmacb->error_handle;
        } else {
            /* nothing to do */
        }
    } else {
        /* nothing to do */
    }

    /* callback register */
    dma_in_irq.full_finish_handle   = _cau_in_dma_complete_callback;
    dma_in_irq.error_handle         = _cau_dma_error_callback;

    dma_out_irq.full_finish_handle   = _cau_out_dma_complete_callback;
    dma_out_irq.error_handle         = _cau_dma_error_callback;

    /* enable CAU */
    hals_cau_enable();

    /* config input DMA */
    hal_dma_start_interrupt(cau_dev->p_dma_cau_in, (uint32_t)cau_dev->in_buffer, (uint32_t)&CAU_DI, \
                            (uint16_t)cau_dev->in_size, &dma_in_irq);
    hal_dma_start_interrupt(cau_dev->p_dma_cau_out, (uint32_t)&CAU_DO, (uint32_t)cau_dev->out_buffer, \
                            (uint16_t)cau_dev->in_size, &dma_out_irq);

    /* enable cau in dma and out dma transfer */
    CAU_DMAEN |= (CAU_DMA_INFIFO | CAU_DMA_OUTFIFO);
}

/*!
    \brief      write data to the in fifo
    \param[in]  data: data to write (0 - 0xFFFFFFFF)
    \param[out] none
    \retval     none
*/
void hals_cau_data_write(uint32_t data)
{
    CAU_DI = data;
}

/*!
    \brief      return the last data entered into the output FIFO
    \param[in]  none
    \param[out] none
    \retval     last data entered into the output FIFO
*/
uint32_t hals_cau_data_read(void)
{
    return CAU_DO;
}

/*!
    \brief      enable the CAU peripheral
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_cau_enable(void)
{
    /* enable the CAU processor */
    CAU_CTL |= CAU_CTL_CAUEN;
}

/*!
    \brief      disable the CAU peripheral
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_cau_disable(void)
{
    /* disable the CAU processor */
    CAU_CTL &= ~CAU_CTL_CAUEN;
}

/*!
    \brief      initialize the key parameter struct with the default values
    \param[in]  none
    \param[out] key_initpara:
                  key_0_high: key 0 high
                  key_0_low:  key 0 low
                  key_1_high: key 1 high
                  key_1_low:  key 1 low
                  key_2_high: key 2 high
                  key_2_low:  key 2 low
                  key_3_high: key 3 high
                  key_3_low:  key 3 low
    \retval     none
*/
void hals_cau_key_struct_para_init(hal_cau_key_parameter_struct *key_initpara)
{
    /* set the key parameters struct with the default values */
    key_initpara->key_0_high = 0U;
    key_initpara->key_0_low  = 0U;
    key_initpara->key_1_high = 0U;
    key_initpara->key_1_low  = 0U;
    key_initpara->key_2_high = 0U;
    key_initpara->key_2_low  = 0U;
    key_initpara->key_3_high = 0U;
    key_initpara->key_3_low  = 0U;
}

/*!
    \brief      get the CAU flag status
    \param[in]  flag: CAU flag status
                only one parameter can be selected which is shown as below:
      \arg        CAU_FLAG_INFIFO_EMPTY: input FIFO empty
      \arg        CAU_FLAG_INFIFO_NO_FULL: input FIFO is not full
      \arg        CAU_FLAG_OUTFIFO_NO_EMPTY: output FIFO not empty
      \arg        CAU_FLAG_OUTFIFO_FULL: output FIFO is full
      \arg        CAU_FLAG_BUSY: the CAU core is busy
      \arg        CAU_FLAG_INFIFO: input FIFO flag status
      \arg        CAU_FLAG_OUTFIFO: output FIFO flag status
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_cau_flag_get(uint32_t flag)
{
    uint32_t reg        = 0U;
    FlagStatus ret_flag = RESET;

    /* check if the flag is in CAU_STAT1 register */
    if(RESET != (flag & FLAG_MASK)) {
        reg = CAU_STAT1;
    } else {
        /* the flag is in CAU_STAT0 register */
        reg = CAU_STAT0;
    }

    /* check the status of the specified CAU flag */
    if(RESET != (reg & flag)) {
        ret_flag = SET;
    } else {
        /* nothing to do */
    }

    return ret_flag;
}

/*!
    \brief      enable the CAU interrupts
    \param[in]  interrupt: specify the CAU interrupt source to be enabled
                one or more parameters can be selected which are shown as below:
      \arg        CAU_INT_INFIFO: input FIFO interrupt
      \arg        CAU_INT_OUTFIFO: output FIFO interrupt
    \param[out] none
    \retval     none
*/
void hals_cau_interrupt_enable(uint32_t interrupt)
{
    /* enable the selected CAU interrupt */
    CAU_INTEN |= interrupt;
}

/*!
    \brief      disable the CAU interrupts
    \param[in]  interrupt: specify the CAU interrupt source to be disabled
                one or more parameters can be selected which are shown as below:
      \arg        CAU_INT_INFIFO: input FIFO interrupt
      \arg        CAU_INT_OUTFIFO: output FIFO interrupt
    \param[out] none
    \retval     none
*/
void hals_cau_interrupt_disable(uint32_t interrupt)
{
    /* disable the selected CAU interrupt */
    CAU_INTEN &= ~(interrupt);
}

/*!
    \brief      get the interrupt flag
    \param[in]  int_flag: CAU interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        CAU_INT_FLAG_INFIFO: input FIFO interrupt
      \arg        CAU_INT_FLAG_OUTFIFO: output FIFO interrupt
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_cau_interrupt_flag_get(uint32_t int_flag)
{
    FlagStatus flag = RESET;

    /* check the status of the specified CAU interrupt */
    if(RESET != (CAU_INTF & int_flag)) {
        flag = SET;
    } else {
        /* nothing to do */
    }

    return flag;
}

/*!
    \brief      AES key structure parameter config
    \param[in]  key: key used for AES algorithm
    \param[in]  keysize: length of the key in bits, must be either 128, 192 or 256
                only one parameter can be selected which is shown as below:
      \arg        CAU_AES_KEY_LENGTH_128BIT: 128 bit key length
      \arg        CAU_AES_KEY_LENGTH_192BIT: 192 bit key length
      \arg        CAU_AES_KEY_LENGTH_256BIT: 256 bit key length
    \param[out] cau_key_initpara: key init parameter struct
                  key_0_high: key 0 high
                  key_0_low:  key 0 low
                  key_1_high: key 1 high
                  key_1_low:  key 1 low
                  key_2_high: key 2 high
                  key_2_low:  key 2 low
                  key_3_high: key 3 high
                  key_3_low:  key 3 low
    \retval     none
*/
static void _cau_aes_key_config(uint32_t *key, hal_cau_aes_key_length_enum keysize, \
                                hal_cau_key_parameter_struct *cau_key_initpara)
{
    uint32_t keyaddr = (uint32_t)key;

    /* clear key size register */
    CAU_CTL &= ~CAU_CTL_KEYM;

    switch(keysize) {
    /* 128-bit key initialization */
    case CAU_AES_KEY_LENGTH_128BIT:
        CAU_CTL |= CAU_KEYSIZE_128BIT;
        cau_key_initpara->key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_low  = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_low  = __REV(*(uint32_t *)(keyaddr));
        break;
    /* 192-bit key initialization */
    case CAU_AES_KEY_LENGTH_192BIT:
        CAU_CTL |= CAU_KEYSIZE_192BIT;
        cau_key_initpara->key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_1_low  = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_low  = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_low  = __REV(*(uint32_t *)(keyaddr));
        break;
    /* 256-bit key initialization */
    case CAU_AES_KEY_LENGTH_256BIT:
        CAU_CTL |= CAU_KEYSIZE_256BIT;
        cau_key_initpara->key_0_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_0_low  = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_1_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_1_low  = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_2_low  = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_high = __REV(*(uint32_t *)(keyaddr));
        keyaddr += 4U;
        cau_key_initpara->key_3_low  = __REV(*(uint32_t *)(keyaddr));
        break;
    default:
        break;
    }
}

/*!
    \brief      AES vectors structure parameter config
    \param[in]  iv: vectors used for AES algorithm
    \param[out] cau_iv_initpara: iv init parameter struct
                  iv_0_high: iv 0 high
                  iv_0_low:  iv 0 low
                  iv_1_high: iv 1 high
                  iv_1_low:  iv 1 low
    \retval     none
*/
static void _cau_aes_iv_config(uint32_t *iv, hal_cau_iv_parameter_struct *cau_iv_initpara)
{
    uint32_t ivaddr = (uint32_t)iv;

    cau_iv_initpara->iv_0_high = __REV(*(uint32_t *)(ivaddr));
    ivaddr += 4U;
    cau_iv_initpara->iv_0_low  = __REV(*(uint32_t *)(ivaddr));
    ivaddr += 4U;
    cau_iv_initpara->iv_1_high = __REV(*(uint32_t *)(ivaddr));
    ivaddr += 4U;
    cau_iv_initpara->iv_1_low  = __REV(*(uint32_t *)(ivaddr));
}

/*!
    \brief      encrypt and decrypt using TDES or DES handler in interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _cau_des_tdes_interrupt(hal_cau_dev_struct *cau_dev)
{
    __IO uint32_t counter;
    uint32_t busystatus;
    /* flush the IN and out fifo */
    CAU_CTL |= CAU_CTL_FFLUSH;
    counter = 0U;
    busystatus = 0U;

    hals_cau_data_write(*(cau_dev->in_buffer));
    cau_dev->in_buffer += 1U;
    hals_cau_data_write(*(cau_dev->in_buffer));
    cau_dev->in_buffer += 1U;

    counter = 0U;
    do {
        busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
        counter++;
    } while((DESBUSY_TIMEOUT != counter) && (RESET != busystatus));

    /* read data in out fifo */
    *(cau_dev->out_buffer) = hals_cau_data_read();
    cau_dev->out_buffer += 1U;
    *(cau_dev->out_buffer) = hals_cau_data_read();
    cau_dev->out_buffer += 1U;
}

/*!
    \brief      encrypt and decrypt using AES handler in interrupt
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _cau_aes_interrupt(hal_cau_dev_struct *cau_dev)
{
    __IO uint32_t counter = 0U;
    uint32_t busystatus   = 0U;
    uint32_t i = 0U;

    for(i = 0U; i < cau_dev->in_size; i += BLOCK_DATA_SIZE) {
        /* wait until the IEM flag is set */
        while(RESET == hals_cau_flag_get(CAU_FLAG_INFIFO_EMPTY)) {}

        /* check if the last input data block */
        if(i + BLOCK_DATA_SIZE > cau_dev->in_size) {
            /* the last block data number is less than 128bit */
            uint32_t block_data_temp[4] = {0};

            /* fill the remaining bits with zero */
            memcpy(block_data_temp, cau_dev->in_buffer, cau_dev->in_size - i);
            cau_dev->in_buffer = (uint32_t *)block_data_temp;

            /* if GCM encryption or CCM decryption, then configurate NBPILB bits in CTL register */
            if(((CAU_CTL & (CAU_CTL_ALGM | CAU_CTL_CAUDIR)) == (CAU_MODE_AES_GCM | CAU_ENCRYPT)) || \
               ((CAU_CTL & (CAU_CTL_ALGM | CAU_CTL_CAUDIR)) == (CAU_MODE_AES_CCM | CAU_DECRYPT))) {
                CAU_CTL |= CAU_PADDING_BYTES(i + BLOCK_DATA_SIZE - cau_dev->in_size);
            } else {
                /* nothing to do */
            }
        } else {
            /* nothing to do */
        }

        /* write data to the in fifo */
        hals_cau_data_write(*(cau_dev->in_buffer));
        cau_dev->in_buffer += 1U;
        hals_cau_data_write(*(cau_dev->in_buffer));
        cau_dev->in_buffer += 1U;
        hals_cau_data_write(*(cau_dev->in_buffer));
        cau_dev->in_buffer += 1U;
        hals_cau_data_write(*(cau_dev->in_buffer));
        cau_dev->in_buffer += 1U;
        counter = 0U;

        do {
            busystatus = hals_cau_flag_get(CAU_FLAG_BUSY);
            counter++;
        } while((AESBUSY_TIMEOUT != counter) && (RESET != busystatus));

        /* read data in out fifo */
        *(cau_dev->out_buffer) = hals_cau_data_read();
        cau_dev->out_buffer += 1U;
        *(cau_dev->out_buffer) = hals_cau_data_read();
        cau_dev->out_buffer += 1U;
        *(cau_dev->out_buffer) = hals_cau_data_read();
        cau_dev->out_buffer += 1U;
        *(cau_dev->out_buffer) = hals_cau_data_read();
        cau_dev->out_buffer += 1U;
    }
}

/*!
    \brief      encrypt and decrypt using AES handler in interrupt for ccm
    \param[in]  cau_dev: CAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static ErrStatus _cau_aes_ccm_gcm_interrupt(hal_cau_dev_struct *cau_dev)
{
    uint32_t algo = 0U;
    ErrStatus status = SUCCESS;

    algo           = (CAU_CTL & CAU_CTL_ALGM);
    cau_dev->state = HAL_CAU_STATE_READY;
    if(algo == CAU_MODE_AES_CCM) {
        status = hals_cau_aes_ccm(cau_dev, cau_dev->p_irq_para, (uint8_t *)cau_dev->out_buffer);
    } else {
        status = hals_cau_aes_gcm(cau_dev, cau_dev->p_irq_para, (uint8_t *)cau_dev->out_buffer);
    }

    return status;
}

/*!
    \brief      CAU dma transfer input complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _cau_in_dma_complete_callback(void *dma)
{
    hal_dma_dev_struct *p_dma_dev = (hal_dma_dev_struct *)dma;
    hal_cau_dev_struct *cau_dev   = (hal_cau_dev_struct *)(p_dma_dev->p_periph);

    /* disable cau in dma transfer */
    CAU_DMAEN &= ~(uint32_t)CAU_DMA_INFIFO;

    if(NULL != cau_dev->in_half_transcom_callback) {
        ((hal_cau_dma_handle_cb)cau_dev->in_half_transcom_callback)(cau_dev);
    } else {
        /* nothing to do */
    }

    if(NULL != cau_dev->in_full_transcom_callback) {
        ((hal_cau_dma_handle_cb)cau_dev->in_full_transcom_callback)(cau_dev);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      CAU dma transfer output complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _cau_out_dma_complete_callback(void *dma)
{
    hal_dma_dev_struct *p_dma_dev = (hal_dma_dev_struct *)dma;
    hal_cau_dev_struct *cau_dev   = (hal_cau_dev_struct *)(p_dma_dev->p_periph);

    /* disable cau out dma transfer */
    CAU_DMAEN &= ~(uint32_t)CAU_DMA_OUTFIFO;
    hals_cau_disable();
    cau_dev->state = HAL_CAU_STATE_READY;

    if(NULL != cau_dev->out_half_transcom_callback) {
        ((hal_cau_dma_handle_cb)cau_dev->out_half_transcom_callback)(cau_dev);
    } else {
        /* nothing to do */
    }

    if(NULL != cau_dev->out_full_transcom_callback) {
        ((hal_cau_dma_handle_cb)cau_dev->out_full_transcom_callback)(cau_dev);
    } else {
        /* nothing to do */
    }
}

/*!
    \brief      CAU dma error callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _cau_dma_error_callback(void *dma)
{
    hal_dma_dev_struct *p_dma_dev = (hal_dma_dev_struct *)dma;
    hal_cau_dev_struct *cau_dev   = (hal_cau_dev_struct *)(p_dma_dev->p_periph);

    /* disable cau out dma transfer */
    CAU_DMAEN &= ~(uint32_t)(CAU_DMA_OUTFIFO | CAU_DMA_INFIFO);

    hals_cau_disable();
    cau_dev->state = HAL_CAU_STATE_READY;

    if(NULL != cau_dev->error_callback) {
        ((hal_cau_dma_handle_cb)cau_dev->error_callback)(cau_dev);
    } else {
        /* nothing to do */
    }
}
