/*!
    \file    gd32h7xx_hal_adc.c
    \brief   ADC driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define ADC_SYSTEM_CLOCK_DIVISOR 1000000U

/* enable ADC */
/* enable the ADC */
static void _adc_deinit(uint32_t adc_periph);
static int32_t _adc_enable(hal_adc_dev_struct *adc_dev);
static int32_t _adc_disable(hal_adc_dev_struct *adc_dev);
static FlagStatus _adc_enable_state_get(uint32_t adc_periph);

/* DMA callback function */
/* ADC DMA full transmission complete callback */
static void _adc_dma_full_transfer_complete(void *dma);
/* ADC DMA half transmission complete callback */
static void _adc_dma_half_transfer_complete(void *dma);
/* ADC DMA error callback */
static void _adc_dma_error(void *dma);
/* ADC EOC interrupt callback function */
static void _adc_eoc_interrupt_callback(void *adc_dev);
/* ADC ROVF interrupt callback function */
static void _adc_rovf_interrupt_callback(void *adc_dev);

/* get ADC state and error */
/* set ADC state */
static void _adc_state_set(hal_adc_dev_struct *adc_dev, hal_adc_state_enum adc_state);
/* clear ADC state */
static void _adc_state_clear(hal_adc_dev_struct *adc_dev, hal_adc_state_enum adc_state);
/* set ADC error */
static void _adc_error_set(hal_adc_dev_struct *adc_dev, hal_adc_error_enum adc_error);

/*!
    \brief      initialize the ADC structure with the default values
    \param[in]  hal_struct_type: the type of ADC parameter struct enumeration
                the argument could be selected from enumeration <hal_adc_struct_type_enum>
    \param[out] p_struct: pointer to ADC structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_struct_init(hal_adc_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
	
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_ADC_INIT_STRUCT:
        /* initialize ADC initialization structure with the default values */
        ((hal_adc_init_struct *)p_struct)->mode                            = ADC_ALL_INDEPENDENT_MODE;
        ((hal_adc_init_struct *)p_struct)->dma_sync_mode                   = ADC_SYNC_DMA_DISABLE;
        ((hal_adc_init_struct *)p_struct)->dma_sync_cont_req               = ADC_DMA_SYNC_CONT_REQ_DISABLE;
        ((hal_adc_init_struct *)p_struct)->delay_between_2_sampling_phases = ADC_DELAY_5_CYCLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in0                      = ADC_DIFCTL_IN0_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in1                      = ADC_DIFCTL_IN1_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in2                      = ADC_DIFCTL_IN2_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in3                      = ADC_DIFCTL_IN3_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in4                      = ADC_DIFCTL_IN4_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in5                      = ADC_DIFCTL_IN5_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in6                      = ADC_DIFCTL_IN6_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in7                      = ADC_DIFCTL_IN7_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in8                      = ADC_DIFCTL_IN8_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in9                      = ADC_DIFCTL_IN9_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in10                     = ADC_DIFCTL_IN10_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in11                     = ADC_DIFCTL_IN11_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in12                     = ADC_DIFCTL_IN12_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in13                     = ADC_DIFCTL_IN13_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in14                     = ADC_DIFCTL_IN14_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in15                     = ADC_DIFCTL_IN15_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in16                     = ADC_DIFCTL_IN16_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in17                     = ADC_DIFCTL_IN17_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in18                     = ADC_DIFCTL_IN18_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in19                     = ADC_DIFCTL_IN19_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in20                     = ADC_DIFCTL_IN20_DISABLE;
        ((hal_adc_init_struct *)p_struct)->difctl_in21                     = ADC_DIFCTL_IN21_DISABLE;
        ((hal_adc_init_struct *)p_struct)->data_alignment                  = ADC_LSB_ALIGNMENT;
        ((hal_adc_init_struct *)p_struct)->calibration                     = ADC_CALIBRATION_DISABLE;
        ((hal_adc_init_struct *)p_struct)->calibration_mode                = ADC_CAL_MODE_OFFSET_MISMATCH;
        ((hal_adc_init_struct *)p_struct)->calibration_number              = ADC_CALIBRATION_NUMBER_1;
        ((hal_adc_init_struct *)p_struct)->clock_prescaler                 = ADC_ASYNCHRONOUS_CLOCK_DIVIDED_BY_1;
        ((hal_adc_init_struct *)p_struct)->resolution                      = ADC01_RESOLUTION_14B;
        ((hal_adc_init_struct *)p_struct)->scan_mode                       = ADC_SCAN_MODE_DISABLE;
        ((hal_adc_init_struct *)p_struct)->hardware_oversampling           = ADC_OVERSAMPLING_DISABLE;
        ((hal_adc_init_struct *)p_struct)->oversampling_right_shift        = ADC_OVERSAMPLE_SHIFT_NONE;
        ((hal_adc_init_struct *)p_struct)->oversampling_ratio              = 1U;
        ((hal_adc_init_struct *)p_struct)->triggered_oversampling          = ADC_OVERSAMPLING_ALL_CONVERT;
        ((hal_adc_init_struct *)p_struct)->hpdf_mode                       = ADC_HPDF_MODE_DISABLE;
        break;
    case HAL_ADC_ROUTINE_CONFIG_STRUCT:
        /* initialize ADC routine channel initialization structure with the default values */
        ((hal_adc_routine_config_struct *)p_struct)->routine_conversions                         = DISABLE;
        ((hal_adc_routine_config_struct *)p_struct)->number_of_conversions                       = ADC_NUMBER_OF_CONVERSIONS_0;
        ((hal_adc_routine_config_struct *)p_struct)->routine_external_trigger_source             = ADC_ROUTINE_SOFTWARE_TRIGGER;
        ((hal_adc_routine_config_struct *)p_struct)->routine_external_trigger_edge               = ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_RISING;
        ((hal_adc_routine_config_struct *)p_struct)->continuous_mode                             = ADC_CONTINUOUS_MODE_DISABLE;
        ((hal_adc_routine_config_struct *)p_struct)->discontinuous_mode                          = ADC_DISCONTINUOUS_MODE_DISABLE;
        ((hal_adc_routine_config_struct *)p_struct)->number_of_conversions_in_discontinuous_mode = 0U;
        ((hal_adc_routine_config_struct *)p_struct)->dma_cont_req                                = ADC_DMA_CONT_REQ_DISABLE;
        break;
    case HAL_ADC_ROUTINE_RANK_CONFIG_STRUCT:
        /* initialize ADC routine channel configuration structure with the default values */
        ((hal_adc_routine_rank_config_struct *)p_struct)->channel          = ADC_CHANNEL_0;
        ((hal_adc_routine_rank_config_struct *)p_struct)->sampling_time    = ADC01_SAMPLETIME_3POINT5;
        ((hal_adc_routine_rank_config_struct *)p_struct)->routine_sequence = ADC_ROUTINE_SEQUENCE_0;
        break;
    case HAL_ADC_WATCHDOG0_CONFIG_STRUCT:
        /* initialize ADC watchdog0 configuration structure with the default values */
        ((hal_adc_watchdog0_config_struct *)p_struct)->watchdog0_mode           = ADC_WATCHDOG0_MONITOR_ALL_CHANNELS;
        ((hal_adc_watchdog0_config_struct *)p_struct)->watchdog0_channel        = ADC_CHANNEL_0;
        ((hal_adc_watchdog0_config_struct *)p_struct)->watchdog0_high_threshold = 4095U;
        ((hal_adc_watchdog0_config_struct *)p_struct)->watchdog0_low_threshold  = 0U;
        break;
    case HAL_ADC_WATCHDOG1_CONFIG_STRUCT:
        /* initialize ADC watchdog1 configuration structure with the default values */
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel0       = ADC_WDT1_CHANNEL0_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel1       = ADC_WDT1_CHANNEL1_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel2       = ADC_WDT1_CHANNEL2_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel3       = ADC_WDT1_CHANNEL3_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel4       = ADC_WDT1_CHANNEL4_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel5       = ADC_WDT1_CHANNEL5_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel6       = ADC_WDT1_CHANNEL6_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel7       = ADC_WDT1_CHANNEL7_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel8       = ADC_WDT1_CHANNEL8_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel9       = ADC_WDT1_CHANNEL9_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel10      = ADC_WDT1_CHANNEL10_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel11      = ADC_WDT1_CHANNEL11_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel12      = ADC_WDT1_CHANNEL12_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel13      = ADC_WDT1_CHANNEL13_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel14      = ADC_WDT1_CHANNEL14_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel15      = ADC_WDT1_CHANNEL15_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel16      = ADC_WDT1_CHANNEL16_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel17      = ADC_WDT1_CHANNEL17_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel18      = ADC_WDT1_CHANNEL18_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel19      = ADC_WDT1_CHANNEL19_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_channel20      = ADC_WDT1_CHANNEL20_DISABLE;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_high_threshold = 4095U;
        ((hal_adc_watchdog1_config_struct *)p_struct)->watchdog1_low_threshold  = 0U;
        break;
    case HAL_ADC_WATCHDOG2_CONFIG_STRUCT:
        /* initialize ADC watchdog2 configuration structure with the default values */
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel0       = ADC_WDT2_CHANNEL0_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel1       = ADC_WDT2_CHANNEL1_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel2       = ADC_WDT2_CHANNEL2_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel3       = ADC_WDT2_CHANNEL3_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel4       = ADC_WDT2_CHANNEL4_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel5       = ADC_WDT2_CHANNEL5_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel6       = ADC_WDT2_CHANNEL6_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel7       = ADC_WDT2_CHANNEL7_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel8       = ADC_WDT2_CHANNEL8_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel9       = ADC_WDT2_CHANNEL9_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel10      = ADC_WDT2_CHANNEL10_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel11      = ADC_WDT2_CHANNEL11_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel12      = ADC_WDT2_CHANNEL12_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel13      = ADC_WDT2_CHANNEL13_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel14      = ADC_WDT2_CHANNEL14_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel15      = ADC_WDT2_CHANNEL15_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel16      = ADC_WDT2_CHANNEL16_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel17      = ADC_WDT2_CHANNEL17_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel18      = ADC_WDT2_CHANNEL18_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel19      = ADC_WDT2_CHANNEL19_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_channel20      = ADC_WDT2_CHANNEL20_DISABLE;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_high_threshold = 4095U;
        ((hal_adc_watchdog2_config_struct *)p_struct)->watchdog2_low_threshold  = 0U;
        break;
    case HAL_ADC_DEV_STRUCT:
        /* initialize ADC device information structure with the default values */
        ((hal_adc_dev_struct *)p_struct)->periph                               = ADC0;
        ((hal_adc_dev_struct *)p_struct)->adc_irq.adc_eoc_handle               = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_irq.adc_watchdog0_handle         = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_irq.adc_watchdog1_handle         = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_irq.adc_watchdog2_handle         = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_irq.adc_dma_full_complete_handle = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_irq.adc_dma_half_complete_handle = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_irq.adc_dma_error_handle         = NULL;
        ((hal_adc_dev_struct *)p_struct)->p_dma_adc                            = NULL;
        ((hal_adc_dev_struct *)p_struct)->error_state                          = HAL_ADC_ERROR_NONE;
        ((hal_adc_dev_struct *)p_struct)->state                                = HAL_ADC_STATE_RESET;
        ((hal_adc_dev_struct *)p_struct)->adc_dma_error_callback               = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_dma_full_complete_callback       = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_dma_half_complete_callback       = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_eoc_callback                     = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_rovf_callback                    = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_watchdog0_callback               = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_watchdog1_callback               = NULL;
        ((hal_adc_dev_struct *)p_struct)->adc_watchdog2_callback               = NULL;
        ((hal_adc_dev_struct *)p_struct)->mutex                                = HAL_MUTEX_UNLOCKED;
        break;
    case HAL_ADC_IRQ_STRUCT:
        /* initialize ADC device interrupt callback function pointer structure with the default values */
        ((hal_adc_irq_struct *)p_struct)->adc_eoc_handle       = NULL;
        ((hal_adc_irq_struct *)p_struct)->adc_rovf_handle      = NULL;
        ((hal_adc_irq_struct *)p_struct)->adc_watchdog0_handle = NULL;
        ((hal_adc_irq_struct *)p_struct)->adc_watchdog1_handle = NULL;
        ((hal_adc_irq_struct *)p_struct)->adc_watchdog2_handle = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize ADC
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_deinit(hal_adc_dev_struct *adc_dev)
{
    int32_t ret = HAL_ERR_NONE;
	
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    _adc_state_set(adc_dev, HAL_ADC_STATE_BUSY_SYSTEM);

    /* if ADC is disabled */
    if(HAL_ERR_NONE == _adc_disable(adc_dev)) {
        /* deinit ADC */
        _adc_deinit(adc_dev->periph);
        adc_dev->error_state = HAL_ADC_ERROR_NONE;
        adc_dev->state       = HAL_ADC_STATE_READY;
    } else {
        ret = HAL_ERR_HARDWARE;
    }

    return ret;
}

/*!
    \brief      initialize ADC
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  p_init: the pointer of ADC init structure
                  mode: the argument could be selected from enumeration <hal_adc_mode_enum>
                  dma_sync_mode:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_DMA_SYNC_MODE_DISABLE: ADC sync DMA disabled
      \arg          ADC_DMA_SYNC_MODE0 : ADC sync DMA mode 0
      \arg          ADC_DMA_SYNC_MODE1 : ADC sync DMA mode 1
                  dma_sync_cont_req:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_DMA_SYNC_CONT_REQ_DISABLE: ADC sync DMA continuous mode disable
      \arg          ADC_DMA_SYNC_CONT_REQ_ENABLE: ADC sync DMA continuous mode enable
                  delay_between_2_sampling_phases:
                  the argument could be selected from enumeration <hal_adc_delay_between_2_sampling_phases_enum>
                  data_alignment:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_LSB_ALIGNMENT: LSB alignment
      \arg          ADC_MSB_ALIGNMENT: MSB alignment
                  calibration:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_CALIBRATION_DISABLE: ADC calibration disable
      \arg          ADC_CALIBRATION_ENABLE: ADC calibration enable
                  calibration_mode:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_CAL_MODE_OFFSET_MISMATCH: ADC calibration offset and mismatch mode
      \arg          ADC_CAL_MODE_OFFSET: ADC calibration offset mode
                  calibration_number:
                  only one parameter can be selected details refer to <hal_adc_calibration_number_enum> in gd32h7xx_hal_adc.h
                  clock_prescaler:
                  only one parameter can be selected details refer to <hal_adc_clock_prescaler_enum> in gd32h7xx_hal_adc.h
                  resolution:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_RESOLUTION_14B: 14-bit ADC resolution, only for ADC0/1
      \arg          ADC_RESOLUTION_12B: 12-bit ADC resolution
      \arg          ADC_RESOLUTION_10B: 10-bit ADC resolution
      \arg          ADC_RESOLUTION_8B : 8-bit ADC resolution
      \arg          ADC_RESOLUTION_6B : 6-bit ADC resolution, only for ADC2
                  scan_mode:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_SCAN_MODE_DISABLE: disable scan mode
      \arg          ADC_SCAN_MODE_ENABLE: enable scan mode
                  hardware_oversampling:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_OVERSAMPLING_DISABLE: disable hardware oversampling
      \arg          ADC_OVERSAMPLING_ENABLE: enable hardware oversampling
                  oversampling_right_shift:
                    the argument could be selected from enumeration <hal_adc_oversampling_right_shift_enum>
                  triggered_oversampling:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_OVERSAMPLING_ALL_CONVERT: all oversampled conversions
                     for a channel are done consecutively after a trigger
      \arg          ADC_OVERSAMPLING_ONE_CONVERT: each oversampled conversion for a channel needs a trigger
                  hpdf_mode:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_HPDF_MODE_DISABLE: HPDF mode disable
      \arg          ADC_HPDF_MODE_ENABLE: HPDF mode enable
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_init(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, hal_adc_init_struct *p_init)
{
    uint32_t reg_temp;

    int32_t ret = HAL_ERR_NONE;
	
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == adc_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [adc_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check periph parameter */
    if((ADC0 != adc_periph) && (ADC1 != adc_periph) && (ADC2 != adc_periph)) {
        HAL_DEBUGE("parameter [adc_periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    adc_dev->periph = adc_periph;

    if(HAL_ADC_STATE_RESET == (hal_adc_state_get(adc_dev))) {
        adc_dev->error_state = HAL_ADC_ERROR_NONE;
    } else {
        /* do nothing */
    }

    if((HAL_ERR_NONE == _adc_disable(adc_dev)) && (RESET == ((hal_adc_error_get(adc_dev)) & HAL_ADC_ERROR_SYSTEM))) {
        /* set ADC state  */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);
        _adc_state_set(adc_dev, HAL_ADC_STATE_BUSY_SYSTEM);

        /* init ADC */
        if(ADC0 == adc_periph) {
            /* configure the ADC sync mode */
            hals_adc_sync_mode_config(p_init->mode);

            /* configure ADC sync DMA mode selection */
            hals_adc_sync_dma_config(p_init->dma_sync_mode);

            /* configure ADC sync DMA mode selection */
            if(ADC_DMA_SYNC_CONT_REQ_DISABLE != p_init->dma_sync_cont_req) {
                ADC_SYNCCTL(adc_periph) |= (uint32_t)ADC_SYNCCTL_SYNCDDM;
            } else if(ADC_DMA_SYNC_CONT_REQ_ENABLE != p_init->dma_sync_cont_req) {
                ADC_SYNCCTL(adc_periph) &= ~((uint32_t)ADC_SYNCCTL_SYNCDDM);
            } else {
                /* do nothing */
            }

            /* configure the delay between 2 sampling phases in ADC sync modes */
            hals_adc_sync_delay_config(p_init->delay_between_2_sampling_phases);
        } else {
            /* do nothing */
        }

        /* ADC data alignment config */
        hals_adc_data_alignment_config(adc_periph, p_init->data_alignment);

        /* configure ADC calibration */
        if((ADC0 == adc_periph) || (ADC1 == adc_periph)) {
            /* configure ADC calibration mode */
            hals_adc_calibration_mode_config(adc_periph, p_init->calibration_mode);
            /* configure ADC calibration number */
            hals_adc_calibration_number(adc_periph, p_init->calibration_number);
        } else {
            /* do nothing */
        }

        /* configure differential mode for ADC channel */
        reg_temp = p_init->difctl_in0 | p_init->difctl_in1 | p_init->difctl_in2 | p_init->difctl_in3 | \
                   p_init->difctl_in4 | p_init->difctl_in5 | p_init->difctl_in6 | p_init->difctl_in7 | \
                   p_init->difctl_in8 | p_init->difctl_in9 | p_init->difctl_in10 | p_init->difctl_in11 | \
                   p_init->difctl_in12 | p_init->difctl_in13 | p_init->difctl_in14 | p_init->difctl_in15 | \
                   p_init->difctl_in16 | p_init->difctl_in17 | p_init->difctl_in18 | p_init->difctl_in19 | \
                   p_init->difctl_in20 | p_init->difctl_in21;
        ADC_DIFCTL(adc_periph) = reg_temp;

        /* configure the ADC clock */
        /* the ADC1 clock shares the ADC0 clock. when using the ADC1, ADC0 clock needs to
            be opened, and clock prescaler can only be configured through ADC0 */
        hals_adc_clock_config(adc_periph, (hal_adc_clock_prescaler_enum)p_init->clock_prescaler);

        /* ADC data resolution select */
        /* ADC data resolution select can be updated only when ADC is disabled*/
        if(RESET == _adc_enable_state_get(adc_periph)) {
            hals_adc_resolution_config(adc_periph, p_init->resolution);
        } else {
            /* do nothing */
        }

        /* ADC scan mode config */
        hals_adc_special_function_config(adc_periph, ADC_SCAN_MODE, p_init->scan_mode);

        /* ADC oversampling mode config */
        if(ADC_OVERSAMPLING_ENABLE == p_init->hardware_oversampling) {
            hals_adc_oversample_mode_config(adc_periph, p_init->triggered_oversampling, \
                                            p_init->oversampling_right_shift, \
                                            p_init->oversampling_ratio);
            /* enable ADC oversample mode */
            hals_adc_oversample_mode_enable(adc_periph);
        } else {
            if(SET == HAL_ADC_GET_OVERSAMPLE_ENABLE(adc_periph)) {
                hals_adc_oversample_mode_config(adc_periph, ADC_OVERSAMPLING_ALL_CONVERT,
                                                ADC_OVERSAMPLE_SHIFT_NONE, 1U);
                /* disable ADC oversample mode */
                hals_adc_oversample_mode_disable(adc_periph);
            } else {
                /* do nothing */
            }
        }

        /* HPDF mode config */
        if(ADC_HPDF_MODE_ENABLE == p_init->hpdf_mode) {
            hals_adc_hpdf_mode_enable(adc_periph);
        } else if(ADC_HPDF_MODE_DISABLE == p_init->hpdf_mode) {
            hals_adc_hpdf_mode_disable(adc_periph);
        } else {
            /* do nothing */
        }
    } else {
        ret = HAL_ERR_HARDWARE;
    }

    return ret;
}

/*!
    \brief      start ADC calibration function
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_calibration_start(hal_adc_dev_struct *adc_dev)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* if ADC is disabled */
    if(HAL_ERR_NONE == _adc_disable(adc_dev)) {
        /* set ADC state  */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);
        _adc_state_set(adc_dev, HAL_ADC_STATE_BUSY_SYSTEM);

        /* enable the ADC */
        _adc_enable(adc_dev);
        /* ADC calibration and reset calibration */
        hals_adc_calibration_enable(adc_dev->periph);

        /* set ADC state  */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_BUSY_SYSTEM);
        _adc_state_set(adc_dev, HAL_ADC_STATE_READY);
    } else {
        ret = HAL_ERR_HARDWARE;
    }

    return ret;
}

/*!
    \brief      ADC interrupt handler content function, which is merely used in ADC_IRQHandler
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_adc_irq(hal_adc_dev_struct *adc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* EOC interrupt handler */
    if(SET == (hals_adc_interrupt_flag_get(adc_dev->periph, ADC_INT_FLAG_EOC))) {
        /* clear EOC flag */
        hals_adc_interrupt_flag_clear(adc_dev->periph, ADC_INT_FLAG_EOC);
        if(NULL != (adc_dev->adc_irq.adc_eoc_handle)) {
            adc_dev->adc_irq.adc_eoc_handle(adc_dev);
        } else {
            /* do nothing */
        }

        /* if not in error state */
        if(SET == ((hal_adc_error_get(adc_dev)) & HAL_ADC_ERROR_SYSTEM)) {
            _adc_state_set(adc_dev, HAL_ADC_STATE_ROUTINE_EOC);
        } else {
            /* do nothing */
        }

        if((RESET == HAL_ADC_GET_ROUTINECH_EXTTRIGGER(adc_dev->periph)) && \
           (RESET == HAL_ADC_GET_CONTINUOUS_MODE(adc_dev->periph))) {
            hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_EOC);
            /* set ADC state */
            _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ROVF interrupt handler */
    if(SET == (hals_adc_interrupt_flag_get(adc_dev->periph, ADC_INT_FLAG_ROVF))) {
        /* clear ROVF flag */
        hals_adc_interrupt_flag_clear(adc_dev->periph, ADC_INT_FLAG_ROVF);
        if(NULL != (adc_dev->adc_irq.adc_rovf_handle)) {
            adc_dev->adc_irq.adc_rovf_handle(adc_dev);
        } else {
            /* do nothing */
        }

        /* set ROVF state */
        _adc_state_set(adc_dev, HAL_ADC_STATE_ROVF);
    } else {
        /* do nothing */
    }

    /* watchdog0 interrupt handler */
    if(SET == (hals_adc_interrupt_flag_get(adc_dev->periph, ADC_INT_FLAG_WDE0))) {
        /* clear watchdog flag */
        hals_adc_interrupt_flag_clear(adc_dev->periph, ADC_INT_FLAG_WDE0);
        if(NULL != (adc_dev->adc_irq.adc_watchdog0_handle)) {
            adc_dev->adc_irq.adc_watchdog0_handle(adc_dev);
        } else {
            /* do nothing */
        }

        /* set ADC state */
        _adc_state_set(adc_dev, HAL_ADC_STATE_WATCHDOG0);
    } else {
        /* do nothing */
    }

    /* watchdog1 interrupt handler */
    if(SET == (hals_adc_interrupt_flag_get(adc_dev->periph, ADC_INT_FLAG_WDE1))) {
        /* clear watchdog flag */
        hals_adc_interrupt_flag_clear(adc_dev->periph, ADC_INT_FLAG_WDE1);
        if(NULL != (adc_dev->adc_irq.adc_watchdog1_handle)) {
            adc_dev->adc_irq.adc_watchdog1_handle(adc_dev);
        } else {
            /* do nothing */
        }

        /* set ADC state */
        _adc_state_set(adc_dev, HAL_ADC_STATE_WATCHDOG1);
    } else {
        /* do nothing */
    }

    /* watchdog2 interrupt handler */
    if(SET == (hals_adc_interrupt_flag_get(adc_dev->periph, ADC_INT_FLAG_WDE2))) {
        /* clear watchdog flag */
        hals_adc_interrupt_flag_clear(adc_dev->periph, ADC_INT_FLAG_WDE2);
        if(NULL != (adc_dev->adc_irq.adc_watchdog2_handle)) {
            adc_dev->adc_irq.adc_watchdog2_handle(adc_dev);
        } else {
            /* do nothing */
        }

        /* set ADC state */
        _adc_state_set(adc_dev, HAL_ADC_STATE_WATCHDOG2);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to ADC interrupt callback functions structure
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_irq_handle_set(hal_adc_dev_struct *adc_dev, hal_adc_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == adc_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [adc_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* EOC interrupt handler set */
    if(NULL != p_irq->adc_eoc_handle) {
        adc_dev->adc_irq.adc_eoc_handle = p_irq->adc_eoc_handle;
        hals_adc_interrupt_enable(adc_dev->periph, ADC_INT_EOC);
    } else {
        adc_dev->adc_irq.adc_eoc_handle = NULL;
        hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_EOC);
    }

    /* rovf interrupt handler set */
    if(NULL != p_irq->adc_rovf_handle) {
        adc_dev->adc_irq.adc_rovf_handle = p_irq->adc_rovf_handle;
        hals_adc_interrupt_enable(adc_dev->periph, ADC_INT_ROVF);
    } else {
        adc_dev->adc_irq.adc_rovf_handle = NULL;
        hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_ROVF);
    }

    /* watchdog0 interrupt handler set */
    if(NULL != p_irq->adc_watchdog0_handle) {
        adc_dev->adc_irq.adc_watchdog0_handle = p_irq->adc_watchdog0_handle;
        hals_adc_interrupt_enable(adc_dev->periph, ADC_INT_WDE0);
    } else {
        adc_dev->adc_irq.adc_watchdog0_handle = NULL;
        hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_WDE0);
    }

    /* watchdog1 interrupt handler set */
    if(NULL != p_irq->adc_watchdog1_handle) {
        adc_dev->adc_irq.adc_watchdog1_handle = p_irq->adc_watchdog1_handle;
        hals_adc_interrupt_enable(adc_dev->periph, ADC_INT_WDE1);
    } else {
        adc_dev->adc_irq.adc_watchdog1_handle = NULL;
        hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_WDE1);
    }

    /* watchdog2 interrupt handler set */
    if(NULL != p_irq->adc_watchdog2_handle) {
        adc_dev->adc_irq.adc_watchdog2_handle = p_irq->adc_watchdog2_handle;
        hals_adc_interrupt_enable(adc_dev->periph, ADC_INT_WDE2);
    } else {
        adc_dev->adc_irq.adc_watchdog2_handle = NULL;
        hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_WDE2);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_irq_handle_all_reset(hal_adc_dev_struct *adc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* ADC interrupt handler reset */
    adc_dev->adc_irq.adc_eoc_handle       = NULL;
    adc_dev->adc_irq.adc_rovf_handle      = NULL;
    adc_dev->adc_irq.adc_watchdog0_handle = NULL;
    adc_dev->adc_irq.adc_watchdog1_handle = NULL;
    adc_dev->adc_irq.adc_watchdog2_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize ADC routine channel
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  p_routine: the pointer of ADC routine sequence init structure
                  routine_sequence_conversions:
                  only one parameter can be selected which is shown as below:
      \arg          ENABLE: enable external trigger
      \arg          DISABLE: disable external trigger
                  number_of_conversions:hal_adc_number_of_conversions_enum
                    the argument could be selected from enumeration <hal_adc_number_of_conversions_enum>
                  routine_external_trigger_source:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_ROUTINE_SOFTWARE_TRIGGER: trigger source is software
      \arg          ADC_ROUTINE_TRIGSEL_TRIGGER: trigger source is TRIGSEL
                  routine_external_trigger_edge:
                    the argument could be selected from enumeration <hal_adc_routine_external_trigger_edge_enum>
                  continuous_mode:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_CONTINUOUS_MODE_DISABLE: continuous mode disable
      \arg          ADC_CONTINUOUS_MODE_ENABLE: continuous mode enable
                  discontinuous_mode:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_DISCONTINUOUS_MODE_DISABLE: discontinuous mode disable
      \arg          ADC_DISCONTINUOUS_MODE_ENABLE: discontinuous mode enable
                  number_of_conversions_in_discontinuous_mode: 1-8
                  dma_cont_req:
                  only one parameter can be selected which is shown as below:
      \arg          ADC_DMA_CONT_REQ_DISABLE: DMA continuous requests disabled
      \arg          ADC_DMA_CONT_REQ_ENABLE: DMA continuous requests enable
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                               hal_adc_routine_config_struct *p_routine)
{
    uint32_t reg_temp;

    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == adc_dev) || (NULL == p_routine)) {
        HAL_DEBUGE("pointer [adc_dev] or [p_routine] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check periph parameter */
    if((ADC0 != adc_periph) && (ADC1 != adc_periph) && (ADC2 != adc_periph)) {
        HAL_DEBUGE("parameter [adc_periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_ADC_STATE_RESET == (hal_adc_state_get(adc_dev))) {
        adc_dev->error_state = HAL_ADC_ERROR_NONE;
    } else {
        /* do nothing */
    }

    if((HAL_ERR_NONE == _adc_disable(adc_dev)) && (RESET == ((hal_adc_error_get(adc_dev)) & HAL_ADC_ERROR_SYSTEM)) && \
       (ENABLE == (p_routine->routine_conversions))) {
        /* set ADC state  */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);
        _adc_state_set(adc_dev, HAL_ADC_STATE_BUSY_SYSTEM);
        /* init ADC routine channel */
        /* ADC channel length config */
        hals_adc_channel_length_config(adc_periph, ADC_ROUTINE_CHANNEL, \
                                       p_routine->number_of_conversions);

        /* ADC routine channel external trigger config */
        if(ADC_ROUTINE_SOFTWARE_TRIGGER == p_routine->routine_external_trigger_source) {
            hals_adc_routine_external_trigger_config(adc_periph, ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_NONE);
        } else {
            hals_adc_routine_external_trigger_config(adc_periph, (hal_adc_routine_external_trigger_edge_enum)p_routine->routine_external_trigger_edge);
        }

        /* ADC continuous mode config */
        hals_adc_special_function_config(adc_periph, ADC_CONTINUOUS_MODE, (uint32_t)p_routine->continuous_mode);

        /* ADC dma request */
        if(ADC_DMA_CONT_REQ_ENABLE == p_routine->dma_cont_req) {
            ADC_CTL1(adc_periph) |= (uint32_t)ADC_CTL1_DDM;
        } else if(ADC_DMA_CONT_REQ_DISABLE == p_routine->dma_cont_req) {
            ADC_CTL1(adc_periph) &= ~(uint32_t)ADC_CTL1_DDM;
        } else {
            /* do nothing */
        }

        /* ADC discontinuous mode config */
        if(ADC_DISCONTINUOUS_MODE_ENABLE == p_routine->discontinuous_mode) {
            if(ADC_CONTINUOUS_MODE_DISABLE == p_routine->continuous_mode) {
                reg_temp = ADC_CTL0(adc_periph);
                /* clear DISRC bit and DISNUM bit */
                reg_temp &= ~((uint32_t)ADC_CTL0_DISRC);
                reg_temp &= ~((uint32_t)ADC_CTL0_DISNUM);
                reg_temp |= CTL0_DISNUM(((uint32_t)(p_routine->number_of_conversions_in_discontinuous_mode) - 1U));
                reg_temp |= (uint32_t)ADC_CTL0_DISRC;
                ADC_CTL0(adc_periph) = reg_temp;
            } else {
                ADC_CTL0(adc_periph) &= ~((uint32_t)ADC_CTL0_DISRC);
                _adc_error_set(adc_dev, HAL_ADC_ERROR_CONFIG);
                _adc_error_set(adc_dev, HAL_ADC_ERROR_SYSTEM);
            }
        } else {
            reg_temp = ADC_CTL0(adc_periph);
            reg_temp &= ~(uint32_t)ADC_CTL0_DISRC;
            ADC_CTL0(adc_periph) = reg_temp;
        }
    } else {
        ret = HAL_ERR_HARDWARE;
    }

    return ret;
}

/*!
    \brief      configure ADC routine channel
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  p_routine_rank: the pointer of ADC routine channel configuration structure
                  channel: ADC_CHANNEL_x(x=0..20)
                  sampling_time:
                    range from 0-807: for ADC0/1, the final sampling time will +3.5Cycles
                    range from 0-638: for ADC2, the final sampling time will +2.5Cycles
                  routine_sequence:
                    the argument could be selected from enumeration <hal_adc_routine_sequence_enum>
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_rank_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                                    hal_adc_routine_rank_config_struct *p_routine_rank)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == adc_dev) || (NULL == p_routine_rank)) {
        HAL_DEBUGE("pointer [adc_dev] or [p_routine_rank] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check periph parameter */
    if((ADC0 != adc_periph) && (ADC1 != adc_periph) && (ADC2 != adc_periph)) {
        HAL_DEBUGE("parameter [adc_periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    adc_dev->periph = adc_periph;

    /* configure ADC routine channel */
    hals_adc_routine_channel_config(adc_periph, p_routine_rank->routine_sequence, p_routine_rank->channel, \
                                    p_routine_rank->sampling_time);

    /* configure ADC internal channel */
    hals_adc_internal_channel_config(adc_periph, p_routine_rank->channel);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable ADC and start the conversion of routine sequence
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_start(hal_adc_dev_struct *adc_dev)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* if ADC is enabled */
    if(HAL_ERR_NONE == _adc_enable(adc_dev)) {
        _adc_state_clear(adc_dev, HAL_ADC_STATE_READY);
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_EOC);
        _adc_state_set(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);

        adc_dev->error_state = HAL_ADC_ERROR_NONE;

        /* clear ADC flag */
        hals_adc_flag_clear(adc_dev->periph, ADC_FLAG_EOC);
    } else {
        HAL_UNLOCK(adc_dev);
        ret = HAL_ERR_HARDWARE;
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      stop the conversion of routine sequence and disable ADC
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_stop(hal_adc_dev_struct *adc_dev)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* if ADC is disabled */
    if(HAL_ERR_NONE == _adc_disable(adc_dev)) {
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);
        _adc_state_set(adc_dev, HAL_ADC_STATE_READY);
    } else {
        HAL_UNLOCK(adc_dev);
        ret = HAL_ERR_HARDWARE;
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      enable ADC and start the conversion of routine sequence
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: the pointer of ADC interrupt user callback structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_start_interrupt(hal_adc_dev_struct *adc_dev, hal_adc_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* if ADC is enabled */
    if(HAL_ERR_NONE == _adc_enable(adc_dev)) {
        /* set ADC state */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_READY);
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_EOC);
        _adc_state_set(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);

        adc_dev->error_state = HAL_ADC_ERROR_NONE;

		adc_dev->adc_eoc_callback  = NULL;
        adc_dev->adc_rovf_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->adc_eoc_callback) {
                adc_dev->adc_eoc_callback = (void *)p_user_func->adc_eoc_callback;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->adc_rovf_callback) {
                adc_dev->adc_rovf_callback = (void *)p_user_func->adc_rovf_callback;
            } else {
                /* do nothing */
            }
        }

        adc_dev->adc_irq.adc_eoc_handle  = _adc_eoc_interrupt_callback;
        adc_dev->adc_irq.adc_rovf_handle = _adc_rovf_interrupt_callback;

        /* clear ADC interrupt flag */
        hals_adc_interrupt_flag_clear(adc_dev->periph, ADC_INT_FLAG_EOC);
        /* enable ADC end of sequence conversion flag */
        hals_adc_interrupt_enable(adc_dev->periph, ADC_INT_EOC);
    } else {
        /* unlock ADC */
        HAL_UNLOCK(adc_dev);

        ret = HAL_ERR_HARDWARE;
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      stop the conversion of routine sequence and disable ADC
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_stop_interrupt(hal_adc_dev_struct *adc_dev)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* if ADC is disabled */
    if(HAL_ERR_NONE == _adc_disable(adc_dev)) {
        /* end of sequence conversion interrupt */
        hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_EOC);
        adc_dev->adc_irq.adc_eoc_handle = NULL;

        /* set ADC state */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);
        _adc_state_set(adc_dev, HAL_ADC_STATE_READY);
    } else {
        HAL_UNLOCK(adc_dev);
        ret = HAL_ERR_HARDWARE;
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      enable ADC and start the conversion of routine sequence with DMA
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: call back function for user
    \param[in]  length: the number of data to be transferred
    \param[out] pdata: the destination memory buffer address
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_start_dma(hal_adc_dev_struct *adc_dev, hal_adc_irq_user_callback_struct *p_user_func, \
                                  uint16_t length, uint32_t *pdata)
{
    hal_dma_irq_struct dma_irq = {0};

    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == adc_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [adc_dev] or [pdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check the parameter */
    if(NULL == adc_dev->p_dma_adc) {
        HAL_DEBUGE("parameter [adc_dev->p_dma_adc] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* if ADC is enabled */
    if(HAL_ERR_NONE == _adc_enable(adc_dev)) {
        /* set ADC state */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_READY);
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_EOC);
        _adc_state_set(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);

        adc_dev->error_state = HAL_ADC_ERROR_NONE;

        /* clear ADC flag */
        hals_adc_flag_clear(adc_dev->periph, ADC_FLAG_EOC);

        /* set the DMA transfer complete callback */
        dma_irq.full_finish_handle   = _adc_dma_full_transfer_complete;
        dma_irq.half_finish_handle   = _adc_dma_half_transfer_complete;
        dma_irq.error_handle         = _adc_dma_error;

		adc_dev->adc_dma_full_complete_callback = NULL;
        adc_dev->adc_dma_half_complete_callback = NULL;
        adc_dev->adc_dma_error_callback         = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->adc_dma_full_complete_callback) {
                adc_dev->adc_dma_full_complete_callback = (void *)p_user_func->adc_dma_full_complete_callback;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->adc_dma_half_complete_callback) {
                adc_dev->adc_dma_half_complete_callback = (void *)p_user_func->adc_dma_half_complete_callback;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->adc_dma_error_callback) {
                adc_dev->adc_dma_error_callback = (void *)p_user_func->adc_dma_error_callback;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable ADC DMA mode */
        hals_adc_dma_mode_enable(adc_dev->periph);
        /* start the DMA channel */
        hal_dma_start_interrupt(adc_dev->p_dma_adc, (uint32_t)&ADC_RDATA(adc_dev->periph), (uint32_t)pdata, \
                                (uint16_t)length, &dma_irq);
    } else {
        /* unlock ADC */
        HAL_UNLOCK(adc_dev);
        ret = HAL_ERR_HARDWARE;
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      stop the conversion of routine sequence, disable ADC DMA mode and disable ADC
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_stop_dma(hal_adc_dev_struct *adc_dev)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* if ADC is enabled */
    if(HAL_ERR_NONE == _adc_disable(adc_dev)) {
        /* disable ADC DMA mode */
        hals_adc_dma_mode_disable(adc_dev->periph);
        /* disable the DMA channel */
        hal_dma_stop(adc_dev->p_dma_adc);

        /* set ADC state */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);
        _adc_state_set(adc_dev, HAL_ADC_STATE_READY);
    } else {
        HAL_UNLOCK(adc_dev);
        ret = HAL_ERR_HARDWARE;
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      enable ADC and start the conversion of routine sequence with sync DMA
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: the pointer of ADC interrupt user callback structure
    \param[in]  length: the number of data to be transferred
    \param[out] pdata: the destination memory buffer address
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_sync_start_dma(hal_adc_dev_struct *adc_dev, hal_adc_irq_user_callback_struct *p_user_func, \
                                       uint16_t length, uint32_t *pdata)
{
    hal_dma_irq_struct dma_irq = {0};

    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == adc_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [adc_dev] or [pdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
    /* check the parameter */
    if(NULL == adc_dev->p_dma_adc) {
        HAL_DEBUGE("parameter [adc_dev->p_dma_adc] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* if ADC is enabled */
    if(HAL_ERR_NONE == _adc_enable(adc_dev)) {
        /* set ADC state */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_READY);
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_EOC);
        _adc_state_set(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);

        adc_dev->error_state = HAL_ADC_ERROR_NONE;

        /* clear ADC flag */
        hals_adc_flag_clear(adc_dev->periph, ADC_FLAG_EOC);

        /* set the DMA transfer complete callback */
        dma_irq.full_finish_handle   = _adc_dma_full_transfer_complete;
        dma_irq.half_finish_handle   = _adc_dma_half_transfer_complete;
        dma_irq.error_handle         = _adc_dma_error;

		adc_dev->adc_dma_full_complete_callback = NULL;
        adc_dev->adc_dma_half_complete_callback = NULL;
        adc_dev->adc_dma_error_callback         = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->adc_dma_full_complete_callback) {
                adc_dev->adc_dma_full_complete_callback = (void *)p_user_func->adc_dma_full_complete_callback;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->adc_dma_half_complete_callback) {
                adc_dev->adc_dma_half_complete_callback = (void *)p_user_func->adc_dma_half_complete_callback;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->adc_dma_error_callback) {
                adc_dev->adc_dma_error_callback = (void *)p_user_func->adc_dma_error_callback;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable ADC DMA mode */
        hals_adc_dma_mode_enable(adc_dev->periph);
        /* start the DMA channel */
        hal_dma_start_interrupt(adc_dev->p_dma_adc, (uint32_t)&ADC_SYNCDATA0, (uint32_t)pdata, (uint16_t)length, \
                                &dma_irq);
    } else {
        /* unlock ADC */
        HAL_UNLOCK(adc_dev);

        ret = HAL_ERR_HARDWARE;
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      stop the conversion of routine sequence, disable ADC DMA mode and disable ADC
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_sync_stop_dma(hal_adc_dev_struct *adc_dev)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* if ADC is enabled */
    if(HAL_ERR_NONE == _adc_disable(adc_dev)) {
        /* disable ADC DMA mode */
        hals_adc_dma_mode_disable(adc_dev->periph);
        /* disable the DMA channel */
        hal_dma_stop(adc_dev->p_dma_adc);

        /* set ADC state */
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);
        _adc_state_set(adc_dev, HAL_ADC_STATE_READY);
    } else {
        HAL_UNLOCK(adc_dev);
        ret = HAL_ERR_HARDWARE;
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      polling for ADC routine sequence conversion
                (scan mode disabled or scan mode enabled with the length of routine channel is 1)
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timeout: time out duration in millisecond
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_conversion_poll(hal_adc_dev_struct *adc_dev, uint32_t timeout)
{
    uint32_t tick_start = 0U;

    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* if ADC configured in DMA mode */
    if(SET == HAL_ADC_GET_DMA_MODE(adc_dev->periph)) {
        _adc_error_set(adc_dev, HAL_ADC_ERROR_CONFIG);

        /* unlock ADC */
        HAL_UNLOCK(adc_dev);

        ret = HAL_ERR_NO_SUPPORT;
    } else {
        /*do nothing */
    }

    /* set timeout */
    tick_start = hal_sys_basetick_count_get();

    /* if single conversion: scan mode disabled or enabled with length = 1 */
    if((RESET == HAL_ADC_GET_SCAN_MODE(adc_dev->periph)) && \
       (RESET == HAL_ADC_GET_ROUTINECH_LENGTH(adc_dev->periph))) {
        /* wait until end of conversion flag is raised */
        while(RESET == hals_adc_flag_get(adc_dev->periph, ADC_FLAG_EOC)) {
            /* check for the timeout */
            if(HAL_TIMEOUT_FOREVER != timeout) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                    _adc_state_set(adc_dev, HAL_ADC_STATE_TIMEOUT);
                    /* when timeout occurs, output timeout warning message */
                    HAL_DEBUGE("adc_dev routine conversion poll timeout");

                    /* unlock ADC */
                    HAL_UNLOCK(adc_dev);

                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    } else {
        /* unlock ADC */
        HAL_UNLOCK(adc_dev);

        ret = HAL_ERR_NO_SUPPORT;
    }

    /* clear ADC EOC flag and STRC flag */
    hals_adc_flag_clear(adc_dev->periph, ADC_FLAG_EOC);
    hals_adc_flag_clear(adc_dev->periph, ADC_FLAG_STRC);

    /* set ADC states */
    _adc_state_set(adc_dev, HAL_ADC_STATE_ROUTINE_EOC);
    if((EXTERNAL_TRIGGER_DISABLE == HAL_ADC_GET_ROUTINECH_EXTTRIGGER(adc_dev->periph)) && \
       (RESET == HAL_ADC_GET_CONTINUOUS_MODE(adc_dev->periph))) {
        _adc_state_clear(adc_dev, HAL_ADC_STATE_ROUTINE_BUSY);
    } else {
        /* do nothing */
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      enable ADC routine sequence software trigger
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_routine_software_trigger_enable(hal_adc_dev_struct *adc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    hals_adc_software_trigger_enable(adc_dev->periph, ADC_ROUTINE_CHANNEL);

    return HAL_ERR_NONE;
}

/*!
    \brief      polling for analog watchdog event
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  event_type: the ADC event type
      \arg          ADC_EVENT_WDE0: ADC analog watchdog 0 event
      \arg          ADC_EVENT_WDE1: ADC analog watchdog 1 event
      \arg          ADC_EVENT_WDE2: ADC analog watchdog 2 event
    \param[in]  timeout: time out duration in millisecond
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog_event_poll(hal_adc_dev_struct *adc_dev, uint32_t event_type, uint32_t timeout)
{
    uint32_t tick_start = 0U;

    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if((ADC_EVENT_WDE0 != event_type) || (ADC_EVENT_WDE1 != event_type) || (ADC_EVENT_WDE2 != event_type)) {
        HAL_DEBUGE("parameter [event_type] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ADC */
    HAL_LOCK(adc_dev);

    /* set timeout */
    tick_start = hal_sys_basetick_count_get();

    /* check watchdog event flag */
    while(RESET == hals_adc_flag_get(adc_dev->periph, event_type)) {
        /* check for the timeout */
        if(HAL_TIMEOUT_FOREVER != timeout) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                _adc_state_set(adc_dev, HAL_ADC_STATE_TIMEOUT);
                /* when timeout occurs, output timeout warning message */
                HAL_DEBUGE("adc_dev watchdog event poll timeout");
                HAL_UNLOCK(adc_dev);
                ret = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }

    /* clear analog watchdog event */
    switch(event_type) {
    case ADC_EVENT_WDE0:
        /* set ADC state */
        _adc_state_set(adc_dev, HAL_ADC_STATE_WATCHDOG0);

        /* clear ADC analog watchdog0 event flag */
        hals_adc_flag_clear(adc_dev->periph, ADC_FLAG_WDE0);
        break;
    case ADC_EVENT_WDE1:
        /* set ADC state */
        _adc_state_set(adc_dev, HAL_ADC_STATE_WATCHDOG1);

        /* clear ADC analog watchdog1 event flag */
        hals_adc_flag_clear(adc_dev->periph, ADC_FLAG_WDE1);
        break;
    case ADC_EVENT_WDE2:
        /* set ADC state */
        _adc_state_set(adc_dev, HAL_ADC_STATE_WATCHDOG2);

        /* clear ADC analog watchdog2 event flag */
        hals_adc_flag_clear(adc_dev->periph, ADC_FLAG_WDE2);
        break;
    default:
        HAL_UNLOCK(adc_dev);
        ret = HAL_ERR_VAL;
        break;
    }

    /* unlock ADC */
    HAL_UNLOCK(adc_dev);

    return ret;
}

/*!
    \brief      configure watchdog0
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  p_watchdog0: the pointer of ADC watchdog configuration structure
                  routine_seq_watchdog0:
                    only one parameter can be selected which is shown as below:
      \arg          ADC_ROUTINE_SEQ_WATCHDOG0_DISABLE: disable routine sequence analog watchdog 0
      \arg          ADC_ROUTINE_SEQ_WATCHDOG0_ENABLE: enable routine sequence analog watchdog 0
                  inserted_seq_watchdog0:
                    only one parameter can be selected which is shown as below:
      \arg          ADC_INSERTED_SEQ_WATCHDOG0_DISABLE: disable inserted sequence analog watchdog
      \arg          ADC_INSERTED_SEQ_WATCHDOG0_ENABLE: enable inserted sequence analog watchdog
                  watchdog0_mode:
                    only one parameter can be selected which is shown as below:
      \arg          ADC_WATCHDOG_MODE_ALL_CHANNELS: analog watchdog is effective on all channels
      \arg          ADC_WATCHDOG_MODE_SINGLE_CHANNEL: analog watchdog is effective on a single channel
                  watchdog0_channel: ADC_WDT0_CHANNEL_x(x=0..20)
                  watchdog0_high_threshold: analog watchdog 0 low threshold, range from 0x0-0x00FFFFFF(for ADC0/1),
                    range from 0x0-0x00000FFF(for ADC2)
                  watchdog0_low_threshold: analog watchdog 0 high threshold, range from 0x0-0x00FFFFFF(for ADC0/1),
                    range from 0x0-0x00000FFF(for ADC2)
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog0_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                                 hal_adc_watchdog0_config_struct *p_watchdog0)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == adc_dev) || (NULL == p_watchdog0)) {
        HAL_DEBUGE("pointer [adc_dev] or [p_watchdog0] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check periph parameter */
    if((ADC0 != adc_periph) && (ADC1 != adc_periph) && (ADC2 != adc_periph)) {
        HAL_DEBUGE("parameter [adc_periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    adc_dev->periph = adc_periph;

    switch(p_watchdog0->watchdog0_mode) {
    case ADC_WATCHDOG0_MONITOR_SINGLE_CHANNEL:
        hals_adc_watchdog0_single_channel_enable(adc_periph, (uint8_t)p_watchdog0->watchdog0_channel);
        break;
    case ADC_WATCHDOG0_MONITOR_ALL_CHANNELS:
        hals_adc_watchdog0_sequence_channel_enable(adc_periph, ADC_ROUTINE_CHANNEL);
        break;
    default:
        break;
    }

    /* configure ADC analog watchdog 0 threshold */
    hals_adc_watchdog0_threshold_config(adc_periph, p_watchdog0->watchdog0_low_threshold, \
                                        p_watchdog0->watchdog0_high_threshold);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable ADC watchdog0 interrupt
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog0_interrupt_enable(hal_adc_dev_struct *adc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear ADC watchdog interrupt flag */
    hals_adc_interrupt_flag_clear(adc_dev->periph, ADC_INT_FLAG_WDE0);
    /* enable ADC watchdog interrupt */
    hals_adc_interrupt_enable(adc_dev->periph, ADC_INT_WDE0);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable ADC watchdog0 interrupt
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog0_interrupt_disable(hal_adc_dev_struct *adc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable ADC watchdog interrupt */
    hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_WDE0);
    /* ADC analog watchdog interrupt handle config */
    adc_dev->adc_irq.adc_watchdog0_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure watchdog1
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  p_watchdog1: the pointer of ADC watchdog configuration structure
                  watchdog1_channelx(x=0..20)
      \arg          ENABLE: enable analog watchdog 1 monitor
      \arg          DISABLE: disable analog watchdog 1 monitor
                  watchdog1_high_threshold: analog watchdog 1 low threshold, range from 0x0-0x00FFFFFF(for ADC0/1),
                    range from 0x0-0x000000FF(for ADC2)
                  watchdog1_low_threshold: analog watchdog 1 high threshold, range from 0x0-0x00FFFFFF(for ADC0/1),
                    range from 0x0-0x000000FF(for ADC2)
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog1_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                                 hal_adc_watchdog1_config_struct *p_watchdog1)
{
    uint32_t reg_temp = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == adc_dev) || (NULL == p_watchdog1)) {
        HAL_DEBUGE("pointer [adc_dev] or [p_watchdog1] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check periph parameter */
    if((ADC0 != adc_periph) && (ADC1 != adc_periph) && (ADC2 != adc_periph)) {
        HAL_DEBUGE("parameter [adc_periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    adc_dev->periph = adc_periph;

    reg_temp |= p_watchdog1->watchdog1_channel0;
    reg_temp |= p_watchdog1->watchdog1_channel1;
    reg_temp |= p_watchdog1->watchdog1_channel2;
    reg_temp |= p_watchdog1->watchdog1_channel3;
    reg_temp |= p_watchdog1->watchdog1_channel4;
    reg_temp |= p_watchdog1->watchdog1_channel5;
    reg_temp |= p_watchdog1->watchdog1_channel6;
    reg_temp |= p_watchdog1->watchdog1_channel7;
    reg_temp |= p_watchdog1->watchdog1_channel8;
    reg_temp |= p_watchdog1->watchdog1_channel9;
    reg_temp |= p_watchdog1->watchdog1_channel10;
    reg_temp |= p_watchdog1->watchdog1_channel11;
    reg_temp |= p_watchdog1->watchdog1_channel12;
    reg_temp |= p_watchdog1->watchdog1_channel13;
    reg_temp |= p_watchdog1->watchdog1_channel14;
    reg_temp |= p_watchdog1->watchdog1_channel15;
    reg_temp |= p_watchdog1->watchdog1_channel16;
    reg_temp |= p_watchdog1->watchdog1_channel17;
    reg_temp |= p_watchdog1->watchdog1_channel18;
    reg_temp |= p_watchdog1->watchdog1_channel19;
    reg_temp |= p_watchdog1->watchdog1_channel20;

    if(RESET == _adc_enable_state_get(adc_periph)) {
        ADC_WD1SR(adc_periph) &= ~(uint32_t)ADC_WD1SR_AWD1CS;
        ADC_WD1SR(adc_periph) |= reg_temp;
    } else {
        /* do nothing */
    }

    /* configure ADC analog watchdog 1 threshold */
    hals_adc_watchdog1_threshold_config(adc_periph, p_watchdog1->watchdog1_low_threshold, \
                                        p_watchdog1->watchdog1_high_threshold);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable ADC watchdog1 interrupt
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog1_interrupt_enable(hal_adc_dev_struct *adc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear ADC watchdog interrupt flag */
    hals_adc_interrupt_flag_clear(adc_dev->periph, ADC_INT_FLAG_WDE1);
    /* enable ADC watchdog interrupt */
    hals_adc_interrupt_enable(adc_dev->periph, ADC_INT_WDE1);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable ADC watchdog1 interrupt
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog1_interrupt_disable(hal_adc_dev_struct *adc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable ADC watchdog interrupt */
    hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_WDE1);
    /* ADC analog watchdog interrupt handle config */
    adc_dev->adc_irq.adc_watchdog1_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure watchdog2
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  p_watchdog2: the pointer of ADC watchdog configuration structure
                  watchdog2_channelx(x=0..20)
      \arg          ENABLE: enable analog watchdog 2 monitor
      \arg          DISABLE: disable analog watchdog 2 monitor
                  watchdog2_high_threshold: analog watchdog 2 low threshold, range from 0x0-0x00FFFFFF(for ADC0/1),
                    range from 0x0-0x000000FF(for ADC2)
				  watchdog2_low_threshold: analog watchdog 2 high threshold, range from 0x0-0x00FFFFFF(for ADC0/1),
                    range from 0x0-0x000000FF(for ADC2)
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog2_config(hal_adc_dev_struct *adc_dev, uint32_t adc_periph, \
                                 hal_adc_watchdog2_config_struct *p_watchdog2)
{
    uint32_t reg_temp = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == adc_dev) || (NULL == p_watchdog2)) {
        HAL_DEBUGE("pointer [adc_dev] or [p_watchdog2] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check periph parameter */
    if((ADC0 != adc_periph) && (ADC1 != adc_periph) && (ADC2 != adc_periph)) {
        HAL_DEBUGE("parameter [adc_periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    adc_dev->periph = adc_periph;

    reg_temp |= p_watchdog2->watchdog2_channel0;
    reg_temp |= p_watchdog2->watchdog2_channel1;
    reg_temp |= p_watchdog2->watchdog2_channel2;
    reg_temp |= p_watchdog2->watchdog2_channel3;
    reg_temp |= p_watchdog2->watchdog2_channel4;
    reg_temp |= p_watchdog2->watchdog2_channel5;
    reg_temp |= p_watchdog2->watchdog2_channel6;
    reg_temp |= p_watchdog2->watchdog2_channel7;
    reg_temp |= p_watchdog2->watchdog2_channel8;
    reg_temp |= p_watchdog2->watchdog2_channel9;
    reg_temp |= p_watchdog2->watchdog2_channel10;
    reg_temp |= p_watchdog2->watchdog2_channel11;
    reg_temp |= p_watchdog2->watchdog2_channel12;
    reg_temp |= p_watchdog2->watchdog2_channel13;
    reg_temp |= p_watchdog2->watchdog2_channel14;
    reg_temp |= p_watchdog2->watchdog2_channel15;
    reg_temp |= p_watchdog2->watchdog2_channel16;
    reg_temp |= p_watchdog2->watchdog2_channel17;
    reg_temp |= p_watchdog2->watchdog2_channel18;
    reg_temp |= p_watchdog2->watchdog2_channel19;
    reg_temp |= p_watchdog2->watchdog2_channel20;

    if(RESET == _adc_enable_state_get(adc_periph)) {
        ADC_WD2SR(adc_periph) &= ~(uint32_t)ADC_WD2SR_AWD2CS;
        ADC_WD2SR(adc_periph) |= reg_temp;
    } else {
        /* do nothing */
    }

    /* configure ADC analog watchdog 2 threshold */
    hals_adc_watchdog2_threshold_config(adc_periph, p_watchdog2->watchdog2_low_threshold, \
                                        p_watchdog2->watchdog2_high_threshold);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable ADC watchdog2 interrupt
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog2_interrupt_enable(hal_adc_dev_struct *adc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear ADC watchdog interrupt flag */
    hals_adc_interrupt_flag_clear(adc_dev->periph, ADC_INT_FLAG_WDE2);
    /* enable ADC watchdog interrupt */
    hals_adc_interrupt_enable(adc_dev->periph, ADC_INT_WDE2);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable ADC watchdog2 interrupt
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_adc_watchdog2_interrupt_disable(hal_adc_dev_struct *adc_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable ADC watchdog interrupt */
    hals_adc_interrupt_disable(adc_dev->periph, ADC_INT_WDE2);
    /* ADC analog watchdog interrupt handle config */
    adc_dev->adc_irq.adc_watchdog2_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      get routine sequence conversion result
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hal_adc_routine_value_get(hal_adc_dev_struct *adc_dev)
{
    /* return ADC routine sequence converted value */
    return (hals_adc_routine_data_read(adc_dev->periph));
}

/*!
    \brief      get ADC error
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_adc_error_enum details refer to gd32h7xx_hal_adc.h
*/
hal_adc_error_enum hal_adc_error_get(hal_adc_dev_struct *adc_dev)
{
    return (hal_adc_error_enum)(adc_dev->error_state);
}

/*!
    \brief      get ADC state
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_adc_state_enum details refer to gd32h7xx_hal_adc.h
*/
hal_adc_state_enum hal_adc_state_get(hal_adc_dev_struct *adc_dev)
{
    return (hal_adc_state_enum)(adc_dev->state);
}

/*!
    \brief      enable ADC interface
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
void hals_adc_enable(uint32_t adc_periph)
{
    if(RESET == (ADC_CTL1(adc_periph) & ADC_CTL1_ADCON)) {
        /* enable ADC */
        ADC_CTL1(adc_periph) |= (uint32_t)ADC_CTL1_ADCON;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      disable ADC interface
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
void hals_adc_disable(uint32_t adc_periph)
{
    /* disable ADC */
    ADC_CTL1(adc_periph) &= ~((uint32_t)ADC_CTL1_ADCON);
}

/*!
    \brief      ADC calibration and reset calibration
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
void hals_adc_calibration_enable(uint32_t adc_periph)
{
    uint32_t tick_start = 0U;

    /* reset the selected ADC calibration registers */
    ADC_CTL1(adc_periph) |= (uint32_t)ADC_CTL1_RSTCLB;

    tick_start = hal_sys_basetick_count_get();

    /* check the RSTCLB bit state */
    while(RESET != (ADC_CTL1(adc_periph) & ADC_CTL1_RSTCLB)) {
        /* check for the timeout */
        if(SET == hal_sys_basetick_timeout_check(tick_start, ADC_ENABLE_DELAYTIME)) {
            /* timeout occurred, exit loop */
            break;
        } else {
            /* do nothing */
        }
    }
    /* enable ADC calibration process */
    ADC_CTL1(adc_periph) |= (uint32_t)ADC_CTL1_CLB;
    /* check the CLB bit state */
    while(RESET != (ADC_CTL1(adc_periph) & ADC_CTL1_CLB)) {
        /* check for the timeout */
        if(SET == hal_sys_basetick_timeout_check(tick_start, ADC_ENABLE_DELAYTIME)) {
            /* timeout occurred, exit loop */
            break;
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      configure ADC calibration mode
    \param[in]  adc_periph: ADCx, x=0,1
    \param[in]  clb_mode: calibration mode
                only one parameter can be selected which is shown as below:
      \arg        ADC_CALIBRATION_OFFSET_MISMATCH: ADC calibration offset and mismatch mode
      \arg        ADC_CALIBRATION_OFFSET: ADC calibration offset mode
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hals_adc_calibration_mode_config(uint32_t adc_periph, uint32_t clb_mode)
{
    int32_t ret = HAL_ERR_NONE;

    if(ADC_CALIBRATION_OFFSET_MISMATCH == clb_mode) {
        /* offset and mismatch mode */
        ADC_CTL1(adc_periph) &= ~((uint32_t)ADC_CTL1_CALMOD);
    } else if(ADC_CALIBRATION_OFFSET == clb_mode) {
        /* offset mode */
        ADC_CTL1(adc_periph) |= (uint32_t)ADC_CTL1_CALMOD;
    } else {
        /* illegal parameters */
        ret = HAL_ERR_VAL;
    }

    return ret;
}

/*!
    \brief      configure ADC calibration number
    \param[in]  adc_periph: ADCx, x=0,1
    \param[in]  clb_num: calibration number
                only one parameter can be selected which is shown as below:
      \arg        ADC_CALIBRATION_NUMBER_1: calibrate once
      \arg        ADC_CALIBRATION_NUMBER_2: calibrate twice
      \arg        ADC_CALIBRATION_NUMBER_4: calibrate 4 times
      \arg        ADC_CALIBRATION_NUMBER_8: calibrate 8 times
      \arg        ADC_CALIBRATION_NUMBER_16: calibrate 16 times
      \arg        ADC_CALIBRATION_NUMBER_32: calibrate 32 times
    \param[out] none
    \retval     none
*/
void hals_adc_calibration_number(uint32_t adc_periph, hal_adc_calibration_number_enum clb_num)
{
    ADC_CTL1(adc_periph) &= ~((uint32_t)ADC_CTL1_CALNUM);
    ADC_CTL1(adc_periph) |= (uint32_t)clb_num;
}

/*!
    \brief      enable DMA request
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
void hals_adc_dma_mode_enable(uint32_t adc_periph)
{
    /* enable DMA request */
    ADC_CTL1(adc_periph) |= (uint32_t)ADC_CTL1_DMA;
}

/*!
    \brief      disable DMA request
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
void hals_adc_dma_mode_disable(uint32_t adc_periph)
{
    /* disable DMA request */
    ADC_CTL1(adc_periph) &= ~((uint32_t)ADC_CTL1_DMA);
}

/*!
    \brief      configure the ADC sync mode
    \param[in]  sync_mode: ADC sync mode
                only one parameter can be selected which is shown as below:
      \arg        ADC_SYNC_MODE_INDEPENDENT: all the ADCs work independently
      \arg        ADC_DAUL_REGULAL_PARALLEL: routine parallel mode
      \arg        ADC_DAUL_REGULAL_FOLLOW_UP: follow-up mode
    \param[out] none
    \retval     none
*/
void hals_adc_sync_mode_config(hal_adc_mode_enum sync_mode)
{
    ADC_SYNCCTL(ADC0) &= ~((uint32_t)ADC_SYNCCTL_SYNCM);
    ADC_SYNCCTL(ADC0) |= (uint32_t)sync_mode;
}

/*!
    \brief      configure ADC sync DMA mode selection
    \param[in]  dma_mode:  ADC sync DMA mode
                only one parameter can be selected which is shown as below:
      \arg        ADC_SYNC_DMA_DISABLE: ADC sync DMA disabled
      \arg        ADC_SYNC_DMA_MODE0: ADC sync DMA mode 0
      \arg        ADC_SYNC_DMA_MODE1: ADC sync DMA mode 1
    \param[out] none
    \retval     none
*/
void hals_adc_sync_dma_config(uint32_t dma_mode)
{
    ADC_SYNCCTL(ADC0) &= ~((uint32_t)ADC_SYNCCTL_SYNCDMA);
    ADC_SYNCCTL(ADC0) |= (uint32_t)dma_mode;
}

/*!
    \brief      configure the delay between 2 sampling phases in ADC sync modes
    \param[in]  sample_delay:  the delay between 2 sampling phases in ADC sync modes
                only one parameter can be selected which is shown as below:
      \arg        ADC_SYNC_DELAY_xCYCLE: x=5..20,the delay between 2 sampling phases in ADC sync modes is x ADC clock cycles
    \param[out] none
    \retval     none
*/
void hals_adc_sync_delay_config(hal_adc_delay_between_2_sampling_phases_enum sample_delay)
{
    ADC_SYNCCTL(ADC0) &= ~((uint32_t)ADC_SYNCCTL_SYNCDLY);
    ADC_SYNCCTL(ADC0) |= (uint32_t)sample_delay;
}

/*!
    \brief      configure ADC data alignment
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  data_alignment: data alignment select
                only one parameter can be selected which is shown as below:
      \arg        ADC_DATAALIGN_RIGHT: LSB alignment
      \arg        ADC_DATAALIGN_LEFT: MSB alignment
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_adc_data_alignment_config(uint32_t adc_periph, uint32_t data_alignment)
{
    int32_t ret = HAL_ERR_NONE;

    if(ADC_DATAALIGN_RIGHT == data_alignment) {
        /* LSB alignment */
        ADC_CTL1(adc_periph) &= ~((uint32_t)ADC_CTL1_DAL);
    } else if(ADC_DATAALIGN_LEFT == data_alignment) {
        /* MSB alignment */
        ADC_CTL1(adc_periph) |= (uint32_t)ADC_CTL1_DAL;
    } else {
        /* illegal parameters */
        ret = HAL_ERR_VAL;
    }

    return ret;
}

/*!
    \brief      configure differential mode for ADC channel
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  adc_channel: the channel use differential mode
                one or more parameters can be selected which is shown as below:
      \arg        ADC_DIFFERENTIAL_MODE_CHANNEL_x(x=0..21), ADC_DIFFERENTIAL_MODE_CHANNEL_ALL: ADC channel for differential mode
    \param[in]  newvalue: ENABLE or DISABLE
    \param[out] none
    \retval     none
*/
void hals_adc_channel_differential_mode_config(uint32_t adc_periph, uint32_t adc_channel, ControlStatus newvalue)
{
    if(ENABLE == newvalue) {
        ADC_DIFCTL(adc_periph) |= (uint32_t)adc_channel;
    } else {
        ADC_DIFCTL(adc_periph) &= ~((uint32_t)adc_channel);
    }
}

/*!
    \brief      configure the ADC clock
    \param[in]  adc_periph: ADCx, x=0,2
    \param[in]  prescaler: configure ADCs prescaler ratio
                only one parameter can be selected which is shown as below:
      \arg        ADC_CLK_SYNC_HCLK_DIV2: ADC sync clock mode HCLK div2
      \arg        ADC_CLK_SYNC_HCLK_DIV4: ADC sync clock mode HCLK div4
      \arg        ADC_CLK_SYNC_HCLK_DIV8: ADC sync clock mode HCLK div8
      \arg        ADC_CLK_SYNC_HCLK_DIV10: ADC sync clock mode HCLK div10
      \arg        ADC_CLK_SYNC_HCLK_DIV12: ADC sync clock mode HCLK div12
      \arg        ADC_CLK_SYNC_HCLK_DIV14: ADC sync clock mode HCLK div14
      \arg        ADC_CLK_SYNC_HCLK_DIV16: ADC sync clock mode HCLK div16
      \arg        ADC_CLK_ASYNC_DIV1: ADC async clock mode div1
      \arg        ADC_CLK_ASYNC_DIV2: ADC async clock mode div2
      \arg        ADC_CLK_ASYNC_DIV4: ADC async clock mode div4
      \arg        ADC_CLK_ASYNC_DIV6: ADC async clock mode div6
      \arg        ADC_CLK_ASYNC_DIV8: ADC async clock mode div8
      \arg        ADC_CLK_ASYNC_DIV10: ADC async clock mode div10
      \arg        ADC_CLK_ASYNC_DIV12: ADC async clock mode div12
      \arg        ADC_CLK_ASYNC_DIV16: ADC async clock mode div16
      \arg        ADC_CLK_ASYNC_DIV32: ADC async clock mode div32
      \arg        ADC_CLK_ASYNC_DIV64: ADC async clock mode div64
      \arg        ADC_CLK_ASYNC_DIV128: ADC async clock mode div128
      \arg        ADC_CLK_ASYNC_DIV256: ADC async clock mode div256
    \param[out]  none
    \retval      none
*/
void hals_adc_clock_config(uint32_t adc_periph, hal_adc_clock_prescaler_enum prescaler)
{
    if(ADC2 == adc_periph) {
        ADC_SYNCCTL(ADC2) &= ~((uint32_t)(ADC_SYNCCTL_ADCCK | ADC_SYNCCTL_ADCSCK));
        ADC_SYNCCTL(ADC2) |= (uint32_t)prescaler;
    } else {
        /* the ADC1 clock shares the ADC0 clock. when using the ADC1, ADC0 clock needs to
            be opened, and clock prescaler can only be configured through ADC0 */
        ADC_SYNCCTL(ADC0) &= ~((uint32_t)(ADC_SYNCCTL_ADCCK | ADC_SYNCCTL_ADCSCK));
        ADC_SYNCCTL(ADC0) |= (uint32_t)prescaler;
    }
}

/*!
    \brief      configure ADC resolution
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  resolution: ADC resolution
                only one parameter can be selected which is shown as below:
      \arg        ADC_RESOLUTION_14B: 14-bit ADC resolution, for ADC0/ADC1
      \arg        ADC_RESOLUTION_12B: 12-bit ADC resolution, for all ADCs
      \arg        ADC_RESOLUTION_10B: 10-bit ADC resolution, for all ADCs
      \arg        ADC_RESOLUTION_8B: 8-bit ADC resolution, for all ADCs
      \arg        ADC_RESOLUTION_6B: 6-bit ADC resolution, only for ADC2
    \param[out] none
    \retval      error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_adc_resolution_config(uint32_t adc_periph, uint32_t resolution)
{
    int32_t ret = HAL_ERR_NONE;

    ADC_CTL0(adc_periph) &= ~((uint32_t)ADC_CTL0_DRES);

    if(ADC2 == adc_periph) {
        if(ADC_RESOLUTION_14B == resolution) {
            /* illegal parameters */
            ret = HAL_ERR_VAL;
        } else {
            ADC_CTL0(adc_periph) |= (uint32_t)CTL0_DRES(resolution - 1U);
        }
    } else {
        if(ADC_RESOLUTION_6B == resolution) {
            /* illegal parameters */
            ret = HAL_ERR_VAL;
        } else {
            ADC_CTL0(adc_periph) |= (uint32_t)CTL0_DRES(resolution);
        }
    }

    return ret;
}

/*!
    \brief      enable or disable ADC special function
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  function: the function to config
                only one parameter can be selected which is shown as below:
      \arg        ADC_SCAN_MODE: scan mode select
      \arg        ADC_CONTINUOUS_MODE: continuous mode select
    \param[in]  newvalue: ENABLE or DISABLE
    \param[out] none
    \retval     none
*/
void hals_adc_special_function_config(uint32_t adc_periph, uint32_t function, uint32_t newvalue)
{
    if(DISABLE != newvalue) {
        if(RESET != (function & ADC_SCAN_MODE)) {
            /* enable scan mode */
            ADC_CTL0(adc_periph) |= (uint32_t)ADC_SCAN_MODE;
        } else {
            /* do nothing */
        }

        if(RESET != (function & ADC_CONTINUOUS_MODE)) {
            /* enable continuous mode */
            ADC_CTL1(adc_periph) |= (uint32_t)ADC_CONTINUOUS_MODE;
        } else {
            /* do nothing */
        }

    } else {
        if(RESET != (function & ADC_SCAN_MODE)) {
            /* disable scan mode */
            ADC_CTL0(adc_periph) &= ~((uint32_t)ADC_SCAN_MODE);
        } else {
            /* do nothing */
        }

        if(RESET != (function & ADC_CONTINUOUS_MODE)) {
            /* disable continuous mode */
            ADC_CTL1(adc_periph) &= ~((uint32_t)ADC_CONTINUOUS_MODE);
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      configure ADC oversample mode
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  mode: ADC oversampling mode
                only one parameter can be selected which is shown as below:
      \arg        ADC_OVERSAMPLING_ALL_CONVERT: all oversampled conversions for
                     a channel are done consecutively after a trigger
      \arg        ADC_OVERSAMPLING_ONE_CONVERT: each oversampled conversion for a channel needs a trigger
    \param[in]  shift: ADC oversampling shift
                only one parameter can be selected which is shown as below:
      \arg        ADC_OVERSAMPLE_SHIFT_NONE: no oversampling shift
      \arg        ADC_OVERSAMPLE_SHIFT_1B: 1-bit oversampling shift
      \arg        ADC_OVERSAMPLE_SHIFT_2B: 2-bit oversampling shift
      \arg        ADC_OVERSAMPLE_SHIFT_3B: 3-bit oversampling shift
      \arg        ADC_OVERSAMPLE_SHIFT_4B: 4-bit oversampling shift
      \arg        ADC_OVERSAMPLE_SHIFT_5B: 5-bit oversampling shift
      \arg        ADC_OVERSAMPLE_SHIFT_6B: 6-bit oversampling shift
      \arg        ADC_OVERSAMPLE_SHIFT_7B: 7-bit oversampling shift
      \arg        ADC_OVERSAMPLE_SHIFT_8B: 8-bit oversampling shift
      \arg        ADC_OVERSAMPLE_SHIFT_9B: 9-bit oversampling shift, available for ADC0/ADC1
      \arg        ADC_OVERSAMPLE_SHIFT_10B: 10-bit oversampling shift, available for ADC0/ADC1
      \arg        ADC_OVERSAMPLE_SHIFT_11B: 11-bit oversampling shift, available for ADC0/ADC1
    \param[in]  ratio: ADC oversampling ratio, 0..1023 for ADC0/ADC1, 0..255 for ADC2
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_adc_oversample_mode_config(uint32_t adc_periph, uint32_t mode, hal_adc_oversampling_right_shift_enum shift, uint16_t ratio)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t regval = 0U;

    if(ADC_OVERSAMPLING_ALL_CONVERT == mode) {
        /* all oversampled conversions for a channel are done consecutively after a trigger */
        ADC_OVSAMPCTL(adc_periph) &= ~((uint32_t)ADC_OVSAMPCTL_TOVS);
    } else if(ADC_OVERSAMPLING_ONE_CONVERT == mode) {
        /* each oversampled conversion for a channel needs a trigger */
        ADC_OVSAMPCTL(adc_periph) |= (uint32_t)ADC_OVSAMPCTL_TOVS;
    } else {
        /* illegal parameters */
        ret = HAL_ERR_VAL;
    }

    /* config the shift and ratio */
    ADC_OVSAMPCTL(adc_periph) &= ~((uint32_t)(ADC_OVSAMPCTL_OVSR | ADC_OVSAMPCTL_OVSS));

    if(0U == ratio) {
        /* illegal parameters */
        ret = HAL_ERR_VAL;
    } else {
        regval = OVSCR_OVSR(((uint32_t)ratio) - 1U);
        regval |= (uint32_t)shift;
        ADC_OVSAMPCTL(adc_periph) |= regval;
    }

    return ret;
}

/*!
    \brief      enable ADC oversample mode
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
void hals_adc_oversample_mode_enable(uint32_t adc_periph)
{
    ADC_OVSAMPCTL(adc_periph) |= (uint32_t)ADC_OVSAMPCTL_OVSEN;
}

/*!
    \brief      disable ADC oversample mode
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
void hals_adc_oversample_mode_disable(uint32_t adc_periph)
{
    ADC_OVSAMPCTL(adc_periph) &= ~((uint32_t)ADC_OVSAMPCTL_OVSEN);
}

/*!
    \brief      enable hpdf mode
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
void hals_adc_hpdf_mode_enable(uint32_t adc_periph)
{
    /* enable hpdf mode */
    ADC_CTL1(adc_periph) |= (uint32_t)ADC_CTL1_HPDFCFG;
}

/*!
    \brief      disable hpdf mode
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
void hals_adc_hpdf_mode_disable(uint32_t adc_periph)
{
    /* disable hpdf mode */
    ADC_CTL1(adc_periph) &= ~((uint32_t)ADC_CTL1_HPDFCFG);
}

/*!
    \brief      configure the length of routine channel sequence
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  adc_sequence: select the channel sequence
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine channel sequence
    \param[in]  length: the length of the channel
                        routine channel 1-16
    \param[out] none
    \retval     none
*/
void hals_adc_channel_length_config(uint32_t adc_periph, uint8_t adc_sequence, hal_adc_number_of_conversions_enum length)
{
    switch(adc_sequence) {
    case ADC_ROUTINE_CHANNEL:
        if(((uint32_t)length >= 1U) && ((uint32_t)length <= 16U)) {
            ADC_RSQ0(adc_periph) &= ~((uint32_t)ADC_RSQ0_RL);
            ADC_RSQ0(adc_periph) |= RSQ0_RL(((uint32_t)length - 1U));
        } else {
            /* do nothing */
        }
        break;
    default:
        break;
    }
}

/*!
    \brief      configure ADC routine channel
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  rank: the routine group sequencer rank,this parameter must be between 0 to 15
    \param[in]  adc_channel: the selected ADC channel
                only one parameter can be selected which is shown as below:
      \arg        ADC_CHANNEL_x(x=0..20): ADC Channelx
    \param[in]  sample_time: the sample time value, 0..809 for ADC0/ADC1, 0..638 for ADC2
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_adc_routine_channel_config(uint32_t adc_periph, uint8_t rank, uint8_t adc_channel, uint32_t sample_time)
{
    uint32_t rsq;
    int32_t ret = HAL_ERR_NONE;

    /* configure ADC routine sequence */
    if(rank < ADC_ROUTINE_CHANNEL_RANK_ONE) {
        /* the routine group sequence rank is smaller than one */
        rsq = ADC_RSQ8(adc_periph);
        rsq &= ~((uint32_t)((ADC_RSQX_RSMPN | ADC_RSQX_RSQN) << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * rank)));
        rsq |= ((uint32_t)(SQX_SMP(sample_time) | adc_channel) << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * rank));
        ADC_RSQ8(adc_periph) = rsq;
    } else if(rank < ADC_ROUTINE_CHANNEL_RANK_THREE) {
        /* the routine group sequence rank is smaller than three */
        rsq = ADC_RSQ7(adc_periph);
        rsq &= ~((uint32_t)((ADC_RSQX_RSMPN | ADC_RSQX_RSQN) \
                            << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_ONE))));
        rsq |= ((uint32_t)(SQX_SMP(sample_time) | adc_channel) \
                << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_ONE)));
        ADC_RSQ7(adc_periph) = rsq;
    } else if(rank < ADC_ROUTINE_CHANNEL_RANK_FIVE) {
        /* the routine group sequence rank is smaller than five */
        rsq = ADC_RSQ6(adc_periph);
        rsq &= ~((uint32_t)((ADC_RSQX_RSMPN | ADC_RSQX_RSQN) \
                            << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_THREE))));
        rsq |= ((uint32_t)(SQX_SMP(sample_time) | adc_channel) \
                << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_THREE)));
        ADC_RSQ6(adc_periph) = rsq;
    } else if(rank < ADC_ROUTINE_CHANNEL_RANK_SEVEN) {
        /* the routine group sequence rank is smaller than seven */
        rsq = ADC_RSQ5(adc_periph);
        rsq &= ~((uint32_t)((ADC_RSQX_RSMPN | ADC_RSQX_RSQN) \
                            << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_FIVE))));
        rsq |= ((uint32_t)(SQX_SMP(sample_time) | adc_channel) \
                << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_FIVE)));
        ADC_RSQ5(adc_periph) = rsq;
    } else if(rank < ADC_ROUTINE_CHANNEL_RANK_NINE) {
        /* the routine group sequence rank is smaller than nine */
        rsq = ADC_RSQ4(adc_periph);
        rsq &= ~((uint32_t)((ADC_RSQX_RSMPN | ADC_RSQX_RSQN) \
                            << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_SEVEN))));
        rsq |= ((uint32_t)(SQX_SMP(sample_time) | adc_channel) \
                << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_SEVEN)));
        ADC_RSQ4(adc_periph) = rsq;
    } else if(rank < ADC_ROUTINE_CHANNEL_RANK_ELEVEN) {
        /* the routine group sequence rank is smaller than eleven */
        rsq = ADC_RSQ3(adc_periph);
        rsq &= ~((uint32_t)((ADC_RSQX_RSMPN | ADC_RSQX_RSQN) \
                            << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_NINE))));
        rsq |= ((uint32_t)(SQX_SMP(sample_time) | adc_channel) \
                << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_NINE)));
        ADC_RSQ3(adc_periph) = rsq;
    } else if(rank < ADC_ROUTINE_CHANNEL_RANK_THIRTEEN) {
        /* the routine group sequence rank is smaller than thirteen */
        rsq = ADC_RSQ2(adc_periph);
        rsq &= ~((uint32_t)((ADC_RSQX_RSMPN | ADC_RSQX_RSQN) \
                            << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_ELEVEN))));
        rsq |= ((uint32_t)(SQX_SMP(sample_time) | adc_channel) \
                << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_ELEVEN)));
        ADC_RSQ2(adc_periph) = rsq;
    } else if(rank < ADC_ROUTINE_CHANNEL_RANK_FIFTEEN) {
        /* the routine group sequence rank is smaller than fifteen */
        rsq = ADC_RSQ1(adc_periph);
        rsq &= ~((uint32_t)((ADC_RSQX_RSMPN | ADC_RSQX_RSQN) \
                            << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_THIRTEEN))));
        rsq |= ((uint32_t)(SQX_SMP(sample_time) | adc_channel) \
                << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_THIRTEEN)));
        ADC_RSQ1(adc_periph) = rsq;
    } else if(rank < ADC_ROUTINE_CHANNEL_RANK_SIXTEEN) {
        /* the routine group sequence rank is smaller than sixteen */
        rsq = ADC_RSQ0(adc_periph);
        rsq &= ~((uint32_t)((ADC_RSQX_RSMPN | ADC_RSQX_RSQN) \
                            << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_FIFTEEN))));
        rsq |= ((uint32_t)(SQX_SMP(sample_time) | adc_channel) \
                << (ADC_ROUTINE_CHANNEL_SHIFT_LENGTH * (rank - ADC_ROUTINE_CHANNEL_RANK_FIFTEEN)));
        ADC_RSQ0(adc_periph) = rsq;
    } else {
        /* illegal parameters */
        ret = HAL_ERR_VAL;
    }

    return ret;
}

/*!
    \brief      enable ADC external trigger
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  trigger_mode: external trigger mode
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_NONE: external trigger disable
      \arg        ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_RISING: rising edge of external trigger
      \arg        ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_FALLING: falling edge of external trigger
      \arg        ADC_ROUTINE_EXTERNAL_TRIGGER_EDGE_RISING_FALLING: rising and falling edge of external trigger
    \param[out] none
    \retval     none
*/
void hals_adc_routine_external_trigger_config(uint32_t adc_periph, hal_adc_routine_external_trigger_edge_enum trigger_mode)
{
    /* configure ADC routine channel group external trigger mode */
    ADC_CTL1(adc_periph) &= ~((uint32_t)ADC_CTL1_ETMRC);
    ADC_CTL1(adc_periph) |= (uint32_t)((uint32_t)trigger_mode << ROUTINE_TRIGGER_MODE);
}

/*!
    \brief      enable ADC software trigger
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  adc_sequence: select the channel sequence
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine channel sequence
    \param[out] none
    \retval     none
*/
void hals_adc_software_trigger_enable(uint32_t adc_periph, uint8_t adc_sequence)
{
    /* enable routine group channel software trigger */
    if(RESET != (adc_sequence & ADC_ROUTINE_CHANNEL)) {
        ADC_CTL1(adc_periph) |= (uint32_t)ADC_CTL1_SWRCST;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      read ADC routine sequence data register
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  none
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hals_adc_routine_data_read(uint32_t adc_periph)
{
    return (uint32_t)(ADC_RDATA(adc_periph));
}

/*!
    \brief      configure ADC internal channel
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  adc_channel: ADC_CHANNEL_x(x=0..20)
    \param[out] none
    \retval     ret: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_NO_SUPPORT details refer to gd32h7xx_hal.h
*/
int32_t hals_adc_internal_channel_config(uint32_t adc_periph, uint8_t adc_channel)
{
    int32_t ret = HAL_ERR_NONE;

    __IO uint32_t wait_loop  = 0U;
    uint32_t systemcoreclock = hal_rcu_system_core_clock_update();

    if(0U == systemcoreclock) {
        ret = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    if(ADC0 == adc_periph) {
        if(ADC0_CHANNEL_DAC0_OUT0 == adc_channel) {
            /* DAC0 OUT0 */
            ret = HAL_ERR_NO_SUPPORT;
        } else {
            /* do nothing */
        }
    }
    if(HAL_ERR_NONE == ret) {
        if(ADC1 == adc_periph) {
            switch(adc_channel) {
            /* enable backup battery voltage measurement */
            case ADC1_CHANNEL_BATTERY:
                if(RESET == HAL_ADC_GET_BATTERY_ENABLE(adc_periph)) {
                    /* enable backup battery voltage measurement */
                    ADC_CTL1(adc_periph) |= ADC_CTL1_VBATEN;
                } else {
                    /* do nothing */
                }
                break;
            /* enable INVREFINT measurement */
            case ADC1_CHANNEL_VREFINT:
                if(RESET == HAL_ADC_GET_INVREFINT_ENABLE(adc_periph)) {
                    /* enable INVREFINT measurement */
                    ADC_CTL1(adc_periph) |= ADC_CTL1_INREFEN;
                } else {
                    /* do nothing */
                }
                break;
            case ADC1_CHANNEL_DAC0_OUT1:
                /* DAC0 OUT1 */
                break;
            default:
                break;
            }
        } else {
            /* do nothing */
        }

        if(ADC2 == adc_periph) {
            switch(adc_channel) {
            /* enable backup battery voltage measurement */
            case ADC2_CHANNEL_BATTERY:
                if(RESET == HAL_ADC_GET_BATTERY_ENABLE(adc_periph)) {
                    /* enable backup battery voltage measurement */
                    ADC_CTL1(adc_periph) |= ADC_CTL1_VBATEN;
                } else {
                    /* do nothing */
                }
                break;
            /* enable temperature sensor measurement */
            case ADC2_CHANNEL_TEMPSENSOR:
                /* enable temperature sensor measurement */
                if(RESET == HAL_ADC_GET_TEMPSENSOR_ENABLE(adc_periph)) {
                    ADC_CTL1(adc_periph) |= ADC_CTL1_TSVEN1;
                } else {
                    /* do nothing */
                }

                /* compute number of CPU cycles to wait for */
                wait_loop = (ADC_TEMPSENSOR_DELAYTIME * (systemcoreclock / ADC_SYSTEM_CLOCK_DIVISOR));
                /* delay for temperature sensor stabilization time */
                while(0U != wait_loop) {
                    wait_loop--;
                }
                break;
            /* enable INVREFINT measurement */
            case ADC2_CHANNEL_VREFINT:
                if(RESET == HAL_ADC_GET_INVREFINT_ENABLE(adc_periph)) {
                    /* enable INVREFINT measurement */
                    ADC_CTL1(adc_periph) |= ADC_CTL1_INREFEN;
                } else {
                    /* do nothing */
                }
                break;
            /* enable high precision temperature sensor measurement */
            case ADC2_CHANNEL_HIGH_PRECISION_TEMPSENSOR:
                /* enable high precision temperature sensor measurement */
                if(RESET == HAL_ADC_GET_HPTEMPSENSOR_ENABLE(adc_periph)) {
                    ADC_CTL1(adc_periph) |= ADC_CTL1_TSVEN2;
                } else {
                    /* do nothing */
                }
                /* compute number of CPU cycles to wait for */
                wait_loop = (ADC_TEMPSENSOR_DELAYTIME * (systemcoreclock / ADC_SYSTEM_CLOCK_DIVISOR));
                /* delay for temperature sensor stabilization time */
                while(0U != wait_loop) {
                    wait_loop--;
                }
                break;
            default:
                break;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      configure ADC analog watchdog 0 single channel
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  adc_channel: the selected ADC channel
                only one parameter can be selected which is shown as below:
      \arg        ADC_CHANNEL_x: ADC Channelx(x=0..20)
    \param[out] none
    \retval     none
*/
void hals_adc_watchdog0_single_channel_enable(uint32_t adc_periph, uint8_t adc_channel)
{
    ADC_CTL0(adc_periph) &= ~((uint32_t)(ADC_CTL0_RWD0EN | ADC_CTL0_WD0SC | ADC_CTL0_WD0CHSEL));

    ADC_CTL0(adc_periph) |= (uint32_t)adc_channel;
    ADC_CTL0(adc_periph) |= (uint32_t)(ADC_CTL0_RWD0EN | ADC_CTL0_WD0SC);
}

/*!
    \brief      configure ADC analog watchdog 0 sequence channel
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  adc_sequence: the channel sequence use analog watchdog 0
                only one parameter can be selected which is shown as below:
      \arg        ADC_ROUTINE_CHANNEL: routine channel sequence
      \arg        ADC_INSERTED_CHANNEL: inserted channel sequence
      \arg        ADC_ROUTINE_INSERTED_CHANNEL: both routine and inserted sequence
    \param[out] none
    \retval     none
*/
void hals_adc_watchdog0_sequence_channel_enable(uint32_t adc_periph, uint8_t adc_sequence)
{
    ADC_CTL0(adc_periph) &= ~((uint32_t)(ADC_CTL0_RWD0EN | ADC_CTL0_IWD0EN | ADC_CTL0_WD0SC));
    /* select the sequence */
    switch(adc_sequence) {
    case ADC_ROUTINE_CHANNEL:
        ADC_CTL0(adc_periph) |= (uint32_t)ADC_CTL0_RWD0EN;
        break;
    default:
        break;
    }
}

/*!
    \brief      configure ADC analog watchdog 0 threshold
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  low_threshold: analog watchdog 0 low threshold, 0..0xFFFFFF for ADC0/ADC1, 0..0xFFF for ADC2
    \param[in]  high_threshold: analog watchdog 0 high threshold, 0..0xFFFFFF for ADC0/ADC1, 0..0xFFF for ADC2
    \param[out] none
    \retval     none
*/
void hals_adc_watchdog0_threshold_config(uint32_t adc_periph, uint32_t low_threshold, uint32_t high_threshold)
{
    ADC_WDLT0(adc_periph) = (uint32_t)WDLT0_WDLT0(low_threshold);
    ADC_WDHT0(adc_periph) = (uint32_t)WDHT0_WDHT0(high_threshold);
}

/*!
    \brief      configure ADC analog watchdog 1 threshold
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  low_threshold: analog watchdog 1 low threshold, 0..0xFFFFFF for ADC0/ADC1, 0..0xFF for ADC2
    \param[in]  high_threshold: analog watchdog 1 high threshold, 0..0xFFFFFF for ADC0/ADC1, 0..0xFF for ADC2
    \param[out] none
    \retval     none
*/
void hals_adc_watchdog1_threshold_config(uint32_t adc_periph, uint32_t low_threshold, uint32_t high_threshold)
{
    ADC_WDLT1(adc_periph) = (uint32_t)WDLT1_WDLT1(low_threshold);
    ADC_WDHT1(adc_periph) = (uint32_t)WDHT1_WDHT1(high_threshold);
}

/*!
    \brief      configure ADC analog watchdog 2 threshold
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  low_threshold: analog watchdog 2 low threshold, 0..0xFFFFFF for ADC0/ADC1, 0..0xFF for ADC2
    \param[in]  high_threshold: analog watchdog 2 high threshold, 0..0xFFFFFF for ADC0/ADC1, 0..0xFF for ADC2
    \param[out] none
    \retval     none
*/
void hals_adc_watchdog2_threshold_config(uint32_t adc_periph, uint32_t low_threshold, uint32_t high_threshold)
{
    ADC_WDLT2(adc_periph) = (uint32_t)WDLT2_WDLT2(low_threshold);
    ADC_WDHT2(adc_periph) = (uint32_t)WDHT2_WDHT2(high_threshold);
}

/*!
    \brief      get the ADC flag bits
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  flag: the adc flag
                only one parameter can be selected which is shown as below:
      \arg        ADC_FLAG_WDE0: analog watchdog 0 event flag
      \arg        ADC_FLAG_EOC: end of group conversion flag
      \arg        ADC_FLAG_STRC: start flag of routine channel group
      \arg        ADC_FLAG_ROVF: routine data register overflow flag
      \arg        ADC_FLAG_WDE1: analog watchdog 1 event flag
      \arg        ADC_FLAG_WDE2: analog watchdog 2 event flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_adc_flag_get(uint32_t adc_periph, uint32_t flag)
{
    FlagStatus retval = RESET;
    if(ADC_STAT(adc_periph) & flag) {
        retval = SET;
    } else {
        /* do nothing */
    }
    return retval;
}

/*!
    \brief      clear the ADC flag bits
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  flag: the adc flag
                only one parameter can be selected which is shown as below:
      \arg        ADC_FLAG_WDE0: analog watchdog 0 event flag
      \arg        ADC_FLAG_EOC: end of group conversion flag
      \arg        ADC_FLAG_STRC: start flag of routine channel group
      \arg        ADC_FLAG_ROVF: routine data register overflow flag
      \arg        ADC_FLAG_WDE1: analog watchdog 1 event flag
      \arg        ADC_FLAG_WDE2: analog watchdog 2 event flag
    \param[out] none
    \retval     none
*/
void hals_adc_flag_clear(uint32_t adc_periph, uint32_t flag)
{
    ADC_STAT(adc_periph) = ~((uint32_t)flag);
}

/*!
    \brief      enable ADC interrupt
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  interrupt: the adc interrupt
                only one parameter can be selected which is shown as below:
      \arg        ADC_INT_WDE0: analog watchdog 0 interrupt
      \arg        ADC_INT_EOC: end of group conversion interrupt
      \arg        ADC_INT_ROVF: routine data register overflow interrupt
      \arg        ADC_INT_WDE1: analog watchdog 1 interrupt
      \arg        ADC_INT_WDE2: analog watchdog 2 interrupt
    \param[out] none
    \retval     none
*/
void hals_adc_interrupt_enable(uint32_t adc_periph, uint32_t interrupt)
{
    ADC_CTL0(adc_periph) |= (uint32_t)interrupt;
}

/*!
    \brief      disable ADC interrupt
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  interrupt: the adc interrupt
                only one parameter can be selected which is shown as below:
      \arg        ADC_INT_WDE0: analog watchdog 0 interrupt
      \arg        ADC_INT_EOC: end of group conversion interrupt
      \arg        ADC_INT_ROVF: routine data register overflow interrupt
      \arg        ADC_INT_WDE1: analog watchdog 1 interrupt
      \arg        ADC_INT_WDE2: analog watchdog 2 interrupt
    \param[out] none
    \retval     none
*/
void hals_adc_interrupt_disable(uint32_t adc_periph, uint32_t interrupt)
{
    ADC_CTL0(adc_periph) &= ~(uint32_t)interrupt;
}

/*!
    \brief      clear the ADC flag
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  int_flag: the adc interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        ADC_INT_FLAG_WDE0: analog watchdog 0 interrupt flag
      \arg        ADC_INT_FLAG_EOC: end of group conversion interrupt flag
      \arg        ADC_INT_FLAG_ROVF: routine data register overflow interrupt flag
      \arg        ADC_INT_FLAG_WDE1: analog watchdog 1 interrupt flag
      \arg        ADC_INT_FLAG_WDE2: analog watchdog 2 interrupt flag
    \param[out] none
    \retval     none
*/
void hals_adc_interrupt_flag_clear(uint32_t adc_periph, uint32_t int_flag)
{
    ADC_STAT(adc_periph) = ~((uint32_t)int_flag);
}

/*!
    \brief      get the ADC interrupt bits
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[in]  int_flag: the adc interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        ADC_INT_FLAG_WDE0: analog watchdog 0 interrupt
      \arg        ADC_INT_FLAG_EOC: end of group conversion interrupt
      \arg        ADC_INT_FLAG_ROVF: routine data register overflow interrupt
      \arg        ADC_INT_FLAG_WDE1: analog watchdog 1 interrupt
      \arg        ADC_INT_FLAG_WDE2: analog watchdog 2 interrupt
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_adc_interrupt_flag_get(uint32_t adc_periph, uint32_t int_flag)
{
    FlagStatus retval = RESET;
    uint32_t state;
    /* check the interrupt bits */
    switch(int_flag) {
    case ADC_INT_FLAG_WDE0:
        /* get the ADC analog watchdog 0 interrupt bits */
        state = ADC_STAT(adc_periph) & ADC_STAT_WDE0;
        if((ADC_CTL0(adc_periph) & ADC_CTL0_WDE0IE) && state) {
            retval = SET;
        } else {
            /* do nothing */
        }
        break;
    case ADC_INT_FLAG_EOC:
        /* get the ADC end of group conversion interrupt bits */
        state = ADC_STAT(adc_periph) & ADC_STAT_EOC;
        if((ADC_CTL0(adc_periph) & ADC_CTL0_EOCIE) && state) {
            retval = SET;
        } else {
            /* do nothing */
        }
        break;
    case ADC_INT_FLAG_ROVF:
        /* get the ADC routine data register overflow interrupt bits */
        state = ADC_STAT(adc_periph) & ADC_STAT_ROVF;
        if((ADC_CTL0(adc_periph) & ADC_CTL0_ROVFIE) && state) {
            retval = SET;
        } else {
            /* do nothing */
        }
        break;
    case ADC_INT_FLAG_WDE1:
        /* get the ADC analog watchdog 1 interrupt bits */
        state = ADC_STAT(adc_periph) & ADC_STAT_WDE1;
        if((ADC_CTL0(adc_periph) & ADC_CTL0_WDE1IE) && state) {
            retval = SET;
        } else {
            /* do nothing */
        }
        break;
    case ADC_INT_FLAG_WDE2:
        /* get the ADC analog watchdog 2 interrupt bits */
        state = ADC_STAT(adc_periph) & ADC_STAT_WDE2;
        if((ADC_CTL0(adc_periph) & ADC_CTL0_WDE2IE) && state) {
            retval = SET;
        } else {
            /* do nothing */
        }
        break;
    default:
        break;
    }

    return retval;
}

/*!
    \brief      enable the ADC
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
static int32_t _adc_enable(hal_adc_dev_struct *adc_dev)
{
    uint32_t tick_start = 0U;
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* if ADC is disabled */
    if(RESET == _adc_enable_state_get(adc_dev->periph)) {
        /* enable ADC */
        hals_adc_enable(adc_dev->periph);

        /* wait for ADC enable */
        tick_start = hal_sys_basetick_count_get();

        /* wait for ADC actually enabled */
        while(RESET == _adc_enable_state_get(adc_dev->periph)) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, ADC_ENABLE_DELAYTIME)) {
                _adc_error_set(adc_dev, HAL_ADC_ERROR_SYSTEM);
                ret = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      disable the ADC
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
static int32_t _adc_disable(hal_adc_dev_struct *adc_dev)
{
    uint32_t tick_start = 0U;
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == adc_dev) {
        HAL_DEBUGE("pointer [adc_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* if ADC is not already disabled */
    if(RESET != _adc_enable_state_get(adc_dev->periph)) {
        /* disable ADC */
        hals_adc_disable(adc_dev->periph);

        /* wait for ADC disable */
        tick_start = hal_sys_basetick_count_get();

        /* wait for ADC actually disabled */
        while(RESET != _adc_enable_state_get(adc_dev->periph)) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, ADC_ENABLE_DELAYTIME)) {
                _adc_error_set(adc_dev, HAL_ADC_ERROR_SYSTEM);
                ret = HAL_ERR_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      get ADC enable state
    \param[in]    adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     FlagStatus:SET or RESET
*/
static FlagStatus _adc_enable_state_get(uint32_t adc_periph)
{
    FlagStatus retval = RESET;
    if(0U != (ADC_CTL1(adc_periph) & ADC_CTL1_ADCON)) {
        retval = SET;
    } else {
        /* do nothing */
    }

    return retval;
}

/*!
    \brief      ADC DMA full transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _adc_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_adc_dev_struct *p_adc;

    p_dma = (hal_dma_dev_struct *)dma;
    p_adc = (hal_adc_dev_struct *)(p_dma->p_periph);

    if(RESET == ((hal_adc_error_get(p_adc)) & (HAL_ADC_ERROR_DMA | HAL_ADC_ERROR_SYSTEM))) {
        /* set ADC state */
        _adc_state_set(p_adc, HAL_ADC_STATE_ROUTINE_EOC);

        if((EXTERNAL_TRIGGER_DISABLE == HAL_ADC_GET_ROUTINECH_EXTTRIGGER(p_adc->periph)) && \
           (DISABLE == HAL_ADC_GET_CONTINUOUS_MODE(p_adc->periph))) {
            /* set ADC state */
            _adc_state_clear(p_adc, HAL_ADC_STATE_ROUTINE_BUSY);
        } else {
            /* do nothing */
        }

        if(NULL != (p_adc->adc_dma_full_complete_callback)) {
            ((hal_irq_handle_cb)p_adc->adc_dma_full_complete_callback)(p_adc);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      ADC DMA half transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _adc_dma_half_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_adc_dev_struct *p_adc;

    p_dma = (hal_dma_dev_struct *)dma;
    p_adc = (hal_adc_dev_struct *)(p_dma->p_periph);

    if(RESET == ((hal_adc_error_get(p_adc)) & (HAL_ADC_ERROR_DMA | HAL_ADC_ERROR_SYSTEM))) {
        /* set ADC state */
        _adc_state_set(p_adc, HAL_ADC_STATE_ROUTINE_EOC);

        if((EXTERNAL_TRIGGER_DISABLE == HAL_ADC_GET_ROUTINECH_EXTTRIGGER(p_adc->periph)) && \
           (DISABLE == HAL_ADC_GET_CONTINUOUS_MODE(p_adc->periph))) {
            /* set ADC state */
            _adc_state_clear(p_adc, HAL_ADC_STATE_ROUTINE_BUSY);
        } else {
            /* do nothing */
        }

        if(NULL != (p_adc->adc_dma_half_complete_callback)) {
            ((hal_irq_handle_cb)p_adc->adc_dma_half_complete_callback)(p_adc);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      ADC DMA error callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _adc_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_adc_dev_struct *p_adc;

    p_dma = (hal_dma_dev_struct *)dma;
    p_adc = (hal_adc_dev_struct *)p_dma->p_periph;

    if(NULL != (p_adc->adc_dma_error_callback)) {
        ((hal_irq_handle_cb)p_adc->adc_dma_error_callback)(p_adc);
    } else {
        /* do nothing */
    }

    /* set ADC state and error */
    _adc_error_set(p_adc, HAL_ADC_ERROR_DMA);
}

/*!
    \brief      ADC device eoc interrupt callback
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _adc_eoc_interrupt_callback(void *adc_dev)
{
    hal_adc_dev_struct *p_adc;

    p_adc = (hal_adc_dev_struct *)adc_dev;

    if(NULL != (p_adc->adc_eoc_callback)) {
        ((hal_irq_handle_cb)p_adc->adc_eoc_callback)(p_adc);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      ADC device rovf interrupt callback
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _adc_rovf_interrupt_callback(void *adc_dev)
{
    hal_adc_dev_struct *p_adc;

    p_adc = (hal_adc_dev_struct *)adc_dev;

    if(NULL != (p_adc->adc_rovf_callback)) {
        ((hal_irq_handle_cb)p_adc->adc_rovf_callback)(p_adc);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      set ADC state
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  adc_state: the state of ADC
                  the argument could be selected from enumeration <hal_adc_state_enum>
    \param[out] none
    \retval     none
*/
static void _adc_state_set(hal_adc_dev_struct *adc_dev, hal_adc_state_enum adc_state)
{
    /* set ADC state */
    adc_dev->state = adc_state;
}

/*!
    \brief      clear ADC state
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  adc_state: the state of ADC
                  the argument could be selected from enumeration <hal_adc_state_enum>
    \param[out] none
    \retval     none
*/
static void _adc_state_clear(hal_adc_dev_struct *adc_dev, hal_adc_state_enum adc_state)
{
    uint32_t state = (uint32_t)adc_dev->state;

    /* clear ADC state */
    state &= ~((uint32_t)adc_state);
    adc_dev->state = (hal_adc_state_enum)state;
}

/*!
    \brief      set ADC error
    \param[in]  adc_dev: ADC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  adc_error: the state of ADC
                  the argument could be selected from enumeration <hal_adc_error_enum>
    \param[out] none
    \retval     none
*/
static void _adc_error_set(hal_adc_dev_struct *adc_dev, hal_adc_error_enum adc_error)
{
    /* set ADC error */
    adc_dev->error_state |= (adc_error);
}

/*!
    \brief      reset ADC
    \param[in]  adc_periph: ADCx, x=0,1,2
    \param[out] none
    \retval     none
*/
static void _adc_deinit(uint32_t adc_periph)
{
    switch(adc_periph) {
    case ADC0:
        hal_rcu_periph_reset_enable(RCU_ADC0RST);
        hal_rcu_periph_reset_disable(RCU_ADC0RST);
        break;
    case ADC1:
        hal_rcu_periph_reset_enable(RCU_ADC1RST);
        hal_rcu_periph_reset_disable(RCU_ADC1RST);
        break;
    case ADC2:
        hal_rcu_periph_reset_enable(RCU_ADC2RST);
        hal_rcu_periph_reset_disable(RCU_ADC2RST);
        break;
    default:
        break;
    }
}
