/*!
    \file    gd32h7xx_hal_uart.h
    \brief   definitions for the UART

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_UART_H
#define GD32H7XX_HAL_UART_H

#include "gd32h7xx_hal.h"

/* uart error code */
#define HAL_UART_ERROR_NONE                    BIT(0)                           /*!< no error */
#define HAL_UART_ERROR_PERR                    BIT(1)                           /*!< parity error */
#define HAL_UART_ERROR_NERR                    BIT(2)                           /*!< noise error */
#define HAL_UART_ERROR_FERR                    BIT(3)                           /*!< frame error */
#define HAL_UART_ERROR_ORERR                   BIT(4)                           /*!< overrun error */
#define HAL_UART_ERROR_DMA                     BIT(5)                           /*!< DMA transfer error */
#define HAL_UART_ERROR_RTO                     BIT(6)                           /*!< receiver timeout error */
#define HAL_UART_ERROR_TTO                     BIT(7)                           /*!< receiver timeout error */
#define HAL_UART_ERROR_BUSY                    BIT(8)                           /*!< busy error */

/* uart struct initialization type enum */
typedef enum {
    HAL_UART_INIT_STRUCT = 0U,  /*!< initialization structure */
    HAL_UART_DEV_STRUCT,        /*!< device information structure */
    HAL_UART_IRQ_INIT_STRUCT,   /*!< interrupt callback initialization structure */
    HAL_UART_IRQ_USER_STRUCT    /*!< interrupt callback user structure */
} hal_uart_struct_type_enum;

/* uart run state enum */
typedef enum {
    HAL_UART_STATE_RESET = 0U,  /*!< not initialized state */
    HAL_UART_STATE_READY,       /*!< initialized and ready for use */
    HAL_UART_STATE_BUSY,        /*!< internal process is ongoing */
    HAL_UART_STATE_BUSY_TX,     /*!< data transmission process is ongoing */
    HAL_UART_STATE_BUSY_RX      /*!< data reception process is ongoing */
} hal_uart_run_state_enum;

/* uart device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb transmit_fifo_empty_handle;  /*!< transmit fifo empty callback function */
    __IO hal_irq_handle_cb transmit_complete_handle;    /*!< transmit complete callback function */
    __IO hal_irq_handle_cb receive_complete_handle;     /*!< receive complete callback function */
    __IO hal_irq_handle_cb receive_fifo_full_handle;    /*!< receive fifo full callback function */
    __IO hal_irq_handle_cb receive_timeout_handle;      /*!< receive timeout callback function */
    __IO hal_irq_handle_cb error_handle;                /*!< error callback function */
    __IO hal_irq_handle_cb wakeup_handle;               /*!< wakeup callback function */
    __IO hal_irq_handle_cb idle_line_detected_handle;   /*!< idle line detected callback function */
    __IO hal_irq_handle_cb address_match_handle;        /*!< address match callback function */
    __IO hal_irq_handle_cb lin_break_detected_handle;   /*!< LIN break detected callback function */
    __IO hal_irq_handle_cb cts_change_handle;           /*!< CTS change callback function */
} hal_uart_irq_struct;

/* uart transfer buffer structure */
typedef struct {
    __IO uint8_t *buffer;   /*!< pointer to transfer buffer */
    __IO uint32_t length;   /*!< transfer byte length */
    __IO uint32_t remain;   /*!< transfer remain byte */
} hal_uart_buffer_struct;

/* @PARA: p_uart */
/* @STRUCT: UART Init structure definition */
typedef struct {
    uint32_t baudrate;              /*!< configures the UART communication baud rate */
    uint32_t wordlength;            /*!< specifies the number of data bits transmitted or received in a frame */
    uint32_t stopbits;              /*!< specifies the number of stop bits transmitted */
    uint32_t parity;                /*!< specifies the parity mode */
    uint32_t mode;                  /*!< specifies whether the receive or transmit mode is enabled or disabled */
    uint32_t hwflowctl;             /*!< specifies whether the hardware flow control mode is enabled or disabled */
    uint32_t oversampling;          /*!< specifies whether the over sampling 8 is enabled or disabled */
    uint32_t samplingmethod;        /*!< specifies whether a single sample or three samples' majority vote is selected */
    uint32_t timeoutenable;         /*!< specifies whether the receiver timeout is enabled */
    uint32_t timeoutvalue;          /*!< specifies the receiver time out value in number of baud blocks */
    uint32_t fifomode;              /*!< specifies whether the receive or transmit fifo is enabled or disabled*/
    uint32_t txthreshold;           /*!< specifies UART transmit FIFO threshold */
    uint32_t rxthreshold;           /*!< specifies UART receive FIFO threshold */
    /* Advanced Feature */
    uint32_t txpinlevelinvert;      /*!< specifies whether the TX pin active level is inverted */
    uint32_t rxpinlevelinvert;      /*!< specifies whether the RX pin active level is inverted */
    uint32_t datainvert;            /*!< specifies whether data are inverted */
    uint32_t swap;                  /*!< specifies whether TX and RX pins are swapped */
    uint32_t overrundisable;        /*!< specifies whether the reception overrun detection is disabled */
    uint32_t dmadisableonrxerror;   /*!< specifies whether the DMA is disabled in case of reception error */
    uint32_t msbfirst;              /*!< specifies whether MSB is sent first on UART line */
} hal_uart_init_struct;

/* @PARA: uart_dev */
/* @STRUCT: UART device information definition */
typedef struct {
    uint32_t periph;                       /*!< UART periph */
    hal_uart_init_struct init;             /*!< UART init parameter */
    hal_uart_irq_struct uart_irq;          /*!< device interrupt callback function pointer */
    hal_dma_dev_struct *p_dma_rx;          /*!< DMA receive pointer */
    hal_dma_dev_struct *p_dma_tx;          /*!< DMA transmit pointer */
    hal_uart_buffer_struct txbuffer;       /*!< transmit buffer */
    hal_uart_buffer_struct rxbuffer;       /*!< receive buffer */
    uint16_t data_bit_mask;                /*!< mask bit of data */
    __IO uint16_t last_error;              /*!< the last error code */
    __IO uint16_t error_state;             /*!< error state */
    __IO hal_uart_run_state_enum tx_state; /*!< transmit state */
    __IO hal_uart_run_state_enum rx_state; /*!< receive state */
    void *transmit_half_complete_callback; /*!< transmit half complete interrupt callback */
    void *transmit_complete_callback;      /*!< transmit complete interrupt callback */
    void *transmit_fifo_empty_callback;    /*!< transmit fifo empty interrupt callback */
    void *receive_half_complete_callback;  /*!< receive half complete interrupt callback */
    void *receive_complete_callback;       /*!< receive complete interrupt callback */
    void *receive_fifo_full_callback;      /*!< receive fifo full interrupt callback */
    void *abort_complete_callback;         /*!< abort complete interrupt callback */
    void *abort_tx_complete_callback;      /*!< abort transmit complete interrupt callback */
    void *abort_rx_complete_callback;      /*!< abort received complete interrupt callback */
    void *idle_receive_callback;           /*!< idle received interrupt callback */
    void *wakeup_callback;                 /*!< wake up interrupt callback */
    void *error_callback;                  /*!< error interrupt callback */
    hal_mutex_enum mutex;                  /*!< lock */
    void *priv;                            /*!< private pointer */
} hal_uart_dev_struct;

typedef void (*hal_uart_user_cb)(hal_uart_dev_struct *uart_dev);

typedef struct {
    __IO hal_uart_user_cb transmit_half_complete_func; /*!< transmit half complete interrupt callback */
    __IO hal_uart_user_cb transmit_complete_func;      /*!< transmit complete interrupt callback */
    __IO hal_uart_user_cb transmit_fifo_empty_func;    /*!< transmit fifo empty interrupt callback */
    __IO hal_uart_user_cb abort_tx_complete_func;      /*!< abort transmit complete interrupt callback */
    __IO hal_uart_user_cb receive_half_complete_func;  /*!< receive half interrupt callback */
    __IO hal_uart_user_cb receive_complete_func;       /*!< receive complete interrupt callback */
    __IO hal_uart_user_cb receive_fifo_full_func;      /*!< receive fifo full interrupt callback */
    __IO hal_uart_user_cb abort_rx_complete_func;      /*!< abort receive complete interrupt callback */
    __IO hal_uart_user_cb abort_complete_func;         /*!< abort complete interrupt callback */
    __IO hal_uart_user_cb idle_receive_func;           /*!< idle received interrupt callback */
    __IO hal_uart_user_cb error_func;                  /*!< error interrupt callback */
} hal_uart_irq_user_callback_struct;

/* @STRUCT_MEMBER: baudrate */
/* @=NULL */

/* @STRUCT_MEMBER: timeoutvalue */
/* @=NULL */

/* hal function declarations */
/* Initialization and de-initialization functions */
/* @FUNCTION: initialize UART */
/* initialize uart asynchronous mode */
int32_t hal_uart_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart);
/* initialize uart rs485 mode */
int32_t hal_rs485_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart, \
                       uint32_t polarity, uint32_t assertion_time, uint32_t deassertion_time);
/* initialize uart single wire(half-duplex) mode */
int32_t hal_halfduplex_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart);
/* initialize uart lin mode */
int32_t hal_lin_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart, \
                     uint32_t break_length);
/* initialize uart multiprocessor communication mode */
int32_t hal_multiprocessor_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart, \
                                uint32_t wakeup_method);
/* initialize uart modbus mode */
int32_t hal_modbus_init(hal_uart_dev_struct *uart_dev, uint32_t periph, hal_uart_init_struct *p_uart);

/* Configuration functions*/
/* configure the address of the uart in wake up by address match mode */
int32_t hal_uart_address_config(hal_uart_dev_struct *uart_dev, uint8_t address0, uint32_t address0length, \
                                uint8_t address1, uint32_t address1length);
/* @END */
/* set wakeup from sleep mode interrupt flag selection */
int32_t hal_uart_deepsleep_wakeupsource_config(hal_uart_dev_struct *uart_dev, uint32_t wake_mode);
/* initialize the uart structure with the default values */
int32_t hal_uart_struct_init(hal_uart_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize uart */
int32_t hal_uart_deinit(hal_uart_dev_struct *uart_dev);

/* uart interrupt handle functions */
/* uart interrupt handler content function,which is merely used in USART_IRQHandler */
int32_t hal_uart_irq(hal_uart_dev_struct *uart_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_uart_irq_handle_set(hal_uart_dev_struct *uart_dev, hal_uart_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_uart_irq_handle_all_reset(hal_uart_dev_struct *uart_dev);

/* transmit or receive functions */
/* transmit amounts of data, poll transmit process and completed status */
/* the function is blocking */
int32_t hal_uart_transmit_poll(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout);
/* receive amounts of data, poll receive process and completed status */
/* the function is blocking */
int32_t hal_uart_receive_poll(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout);
/* receive amounts of data, poll receive process and completed status */
/* the function is blocking */
int32_t hal_uart_receive_idle_poll(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout);
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_uart_transmit_interrupt(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, \
                                    hal_uart_irq_user_callback_struct *p_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_uart_receive_interrupt(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, \
                                   hal_uart_irq_user_callback_struct *p_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_uart_receive_idle_interrupt(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint32_t length, \
                                        hal_uart_irq_user_callback_struct *p_func);
/* transmit amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_uart_transmit_dma(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint16_t length, \
                              hal_uart_irq_user_callback_struct *p_func);
/* receive amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_uart_receive_dma(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint16_t length, \
                             hal_uart_irq_user_callback_struct *p_func);
/* receive amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_uart_receive_idle_dma(hal_uart_dev_struct *uart_dev, uint8_t *p_buffer, uint16_t length, \
                                  hal_uart_irq_user_callback_struct *p_func);

/* transfer control functions */
/* pause uart DMA transfer during transmission process */
int32_t hal_uart_dma_pause(hal_uart_dev_struct *uart_dev);
/* resume uart DMA transfer during transmission process */
int32_t hal_uart_dma_resume(hal_uart_dev_struct *uart_dev);
/* stop uart DMA transfer during transmission process */
int32_t hal_uart_dma_stop(hal_uart_dev_struct *uart_dev);
/* abort ongoing transfers */
int32_t hal_uart_abort(hal_uart_dev_struct *uart_dev);
/* abort ongoing transmit transfer */
int32_t hal_uart_transmit_abort(hal_uart_dev_struct *uart_dev);
/* abort ongoing receive transfer */
int32_t hal_uart_receive_abort(hal_uart_dev_struct *uart_dev);

/* abort ongoing transmit transfer in interrupt method */
int32_t hal_uart_transmit_abort_interrupt(hal_uart_dev_struct *uart_dev, hal_uart_irq_user_callback_struct *p_func);
/* abort ongoing receive transfer in interrupt method */
int32_t hal_uart_receive_abort_interrupt(hal_uart_dev_struct *uart_dev, hal_uart_irq_user_callback_struct *p_func);
/* abort ongoing transfers in interrupt method */
int32_t hal_uart_abort_interrupt(hal_uart_dev_struct *uart_dev, hal_uart_irq_user_callback_struct *p_func);

/* UART FIFO */
/* enable FIFO */
int32_t hal_uart_fifo_enable(hal_uart_dev_struct *uart_dev);
/* disable FIFO */
int32_t hal_uart_fifo_disable(hal_uart_dev_struct *uart_dev);
/* configure transmit FIFO threshold */
int32_t hal_uart_transmit_fifo_threshold_config(hal_uart_dev_struct *uart_dev, uint32_t txthreshold);
/* configure receive FIFO threshold */
int32_t hal_uart_receive_fifo_threshold_config(hal_uart_dev_struct *uart_dev, uint32_t rxthreshold);

/*configure receiver timeout threshold */
int32_t hal_uart_receiver_timeout_config(hal_uart_dev_struct *uart_dev, uint32_t rtimeout);
/* enable receiver timeout */
int32_t hal_uart_receiver_timeout_enable(hal_uart_dev_struct *uart_dev);
/* disable receiver timeout */
int32_t hal_uart_receiver_timeout_disable(hal_uart_dev_struct *uart_dev);

/* enable mute mode */
int32_t hal_uart_mute_mode_enable(hal_uart_dev_struct *uart_dev);
/* disable mute mode */
int32_t hal_uart_mute_mode_disable(hal_uart_dev_struct *uart_dev);
/* enable UART command */
int32_t hal_uart_command_enable(hal_uart_dev_struct *uart_dev, uint32_t cmdtype);
/* configure UART transmitter */
int32_t hal_uart_transmit_config(hal_uart_dev_struct *uart_dev, uint32_t txconfig);
/* configure UART receiver */
int32_t hal_uart_receive_config(hal_uart_dev_struct *uart_dev, uint32_t rxconfig);
/* enable UART to wakeup the mcu from deep-sleep mode */
int32_t hal_uart_wakeup_enable(hal_uart_dev_struct *uart_dev);
/* disable UART to wakeup the mcu from deep-sleep mode */
int32_t hal_uart_wakeup_disable(hal_uart_dev_struct *uart_dev);

/* return the uart tx state */
hal_uart_run_state_enum hal_uart_tx_state_get(hal_uart_dev_struct *uart_dev);
/* return the uart rx state */
hal_uart_run_state_enum hal_uart_rx_state_get(hal_uart_dev_struct *uart_dev);
/* return the uart error code */
uint16_t hal_uart_error_get(hal_uart_dev_struct *uart_dev);

#endif /* GD32H7XX_HAL_UART_H */
