/*!
    \file    gd32h7xx_hal_tli.h
    \brief   definitions for the TLI

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/
#ifndef GD32H7XX_HAL_TLI_H
#define GD32H7XX_HAL_TLI_H

#include "gd32h7xx_hal.h"

/* TLI definitions */
#define TLI                               TLI_BASE                              /*!< TLI base address */

/* @PARA: layer */
/* @DEFINE: TLI layer definitions */
#define LAYER0                            (TLI + 0x00000084U)                   /*!< TLI layer0 base address */
#define LAYER1                            (TLI + 0x00000104U)                   /*!< TLI layer1 base address */

/* registers definitions */
#define TLI_SPSZ                          REG32(TLI + 0x00000008U)              /*!< TLI synchronous pulse size register */
#define TLI_BPSZ                          REG32(TLI + 0x0000000CU)              /*!< TLI back-porch size register */
#define TLI_ASZ                           REG32(TLI + 0x00000010U)              /*!< TLI active size register */
#define TLI_TSZ                           REG32(TLI + 0x00000014U)              /*!< TLI total size register */
#define TLI_CTL                           REG32(TLI + 0x00000018U)              /*!< TLI control register */
#define TLI_RL                            REG32(TLI + 0x00000024U)              /*!< TLI reload Layer register */
#define TLI_BGC                           REG32(TLI + 0x0000002CU)              /*!< TLI background color register */
#define TLI_INTEN                         REG32(TLI + 0x00000034U)              /*!< TLI interrupt enable register */
#define TLI_INTF                          REG32(TLI + 0x00000038U)              /*!< TLI interrupt flag register */
#define TLI_INTC                          REG32(TLI + 0x0000003CU)              /*!< TLI interrupt flag clear register */
#define TLI_LM                            REG32(TLI + 0x00000040U)              /*!< TLI line mark register */
#define TLI_CPPOS                         REG32(TLI + 0x00000044U)              /*!< TLI current pixel position register */
#define TLI_STAT                          REG32(TLI + 0x00000048U)              /*!< TLI status register */
#define TLI_LXCTL(layerx)                 REG32((layerx) + 0x00000000U)         /*!< TLI layer x control register */
#define TLI_LXHPOS(layerx)                REG32((layerx) + 0x00000004U)         /*!< TLI layer x horizontal position parameters register */
#define TLI_LXVPOS(layerx)                REG32((layerx) + 0x00000008U)         /*!< TLI layer x vertical position parameters register */
#define TLI_LXCKEY(layerx)                REG32((layerx) + 0x0000000CU)         /*!< TLI layer x color key register */
#define TLI_LXPPF(layerx)                 REG32((layerx) + 0x00000010U)         /*!< TLI layer x packeted pixel format register */
#define TLI_LXSA(layerx)                  REG32((layerx) + 0x00000014U)         /*!< TLI layer x specified alpha register */
#define TLI_LXDC(layerx)                  REG32((layerx) + 0x00000018U)         /*!< TLI layer x default color register */
#define TLI_LXBLEND(layerx)               REG32((layerx) + 0x0000001CU)         /*!< TLI layer x blending register */
#define TLI_LXFBADDR(layerx)              REG32((layerx) + 0x00000028U)         /*!< TLI layer x frame base address register */
#define TLI_LXFLLEN(layerx)               REG32((layerx) + 0x0000002CU)         /*!< TLI layer x frame line length register */
#define TLI_LXFTLN(layerx)                REG32((layerx) + 0x00000030U)         /*!< TLI layer x frame total line number register */
#define TLI_LXLUT(layerx)                 REG32((layerx) + 0x00000040U)         /*!< TLI layer x look up table register */

/* bits definitions */
/* TLI_SPSZ */
#define TLI_SPSZ_VPSZ                     BITS(0,11)                            /*!< size of the vertical synchronous pulse */
#define TLI_SPSZ_HPSZ                     BITS(16,27)                           /*!< size of the horizontal synchronous pulse */

/* TLI_BPSZ */
#define TLI_BPSZ_VBPSZ                    BITS(0,11)                            /*!< size of the vertical back porch plus synchronous pulse */
#define TLI_BPSZ_HBPSZ                    BITS(16,27)                           /*!< size of the horizontal back porch plus synchronous pulse */

/* TLI_ASZ */
#define TLI_ASZ_VASZ                      BITS(0,11)                            /*!< size of the vertical active area width plus back porch and synchronous pulse */
#define TLI_ASZ_HASZ                      BITS(16,27)                           /*!< size of the horizontal active area width plus back porch and synchronous pulse */

/* TLI_SPSZ */
#define TLI_TSZ_VTSZ                      BITS(0,11)                            /*!< vertical total size of the display, including active area, back porch, synchronous pulse and front porch */
#define TLI_TSZ_HTSZ                      BITS(16,27)                           /*!< horizontal total size of the display, including active area, back porch, synchronous pulse and front porch */

/* TLI_CTL */
#define TLI_CTL_TLIEN                     BIT(0)                                /*!< TLI enable bit */
#define TLI_CTL_BDB                       BITS(4,6)                             /*!< blue channel dither bits number */
#define TLI_CTL_GDB                       BITS(8,10)                            /*!< green channel dither bits number */
#define TLI_CTL_RDB                       BITS(12,14)                           /*!< red channel dither bits number */
#define TLI_CTL_DFEN                      BIT(16)                               /*!< dither function enable */
#define TLI_CTL_CLKPS                     BIT(28)                               /*!< pixel clock polarity selection */
#define TLI_CTL_DEPS                      BIT(29)                               /*!< data enable polarity selection */
#define TLI_CTL_VPPS                      BIT(30)                               /*!< vertical pulse polarity selection */
#define TLI_CTL_HPPS                      BIT(31)                               /*!< horizontal pulse polarity selection */

/* TLI_RL */
#define TLI_RL_RQR                        BIT(0)                                /*!< request reload */
#define TLI_RL_FBR                        BIT(1)                                /*!< frame blank reload */

/* TLI_BGC */
#define TLI_BGC_BVB                       BITS(0,7)                             /*!< background value blue */
#define TLI_BGC_BVG                       BITS(8,15)                            /*!< background value green */
#define TLI_BGC_BVR                       BITS(16,23)                           /*!< background value red */

/* TLI_INTEN */
#define TLI_INTEN_LMIE                    BIT(0)                                /*!< line mark interrupt enable */
#define TLI_INTEN_FEIE                    BIT(1)                                /*!< FIFO error interrupt enable */
#define TLI_INTEN_TEIE                    BIT(2)                                /*!< transaction error interrupt enable */
#define TLI_INTEN_LCRIE                   BIT(3)                                /*!< layer configuration reloaded interrupt enable */

/* TLI_INTF */
#define TLI_INTF_LMF                      BIT(0)                                /*!< line mark flag */
#define TLI_INTF_FEF                      BIT(1)                                /*!< FIFO error flag */
#define TLI_INTF_TEF                      BIT(2)                                /*!< transaction error flag */
#define TLI_INTF_LCRF                     BIT(3)                                /*!< layer configuration reloaded flag */

/* TLI_INTC */
#define TLI_INTC_LMC                      BIT(0)                                /*!< line mark flag clear */
#define TLI_INTC_FEC                      BIT(1)                                /*!< FIFO error flag clear */
#define TLI_INTC_TEC                      BIT(2)                                /*!< transaction error flag clear */
#define TLI_INTC_LCRC                     BIT(3)                                /*!< layer configuration reloaded flag clear */

/* TLI_LM */
#define TLI_LM_LM                         BITS(0,10)                            /*!< line mark value */

/* TLI_CPPOS */
#define TLI_CPPOS_VPOS                    BITS(0,15)                            /*!< vertical position */
#define TLI_CPPOS_HPOS                    BITS(16,31)                           /*!< horizontal position */

/* TLI_STAT */
#define TLI_STAT_VDE                      BIT(0)                                /*!< current VDE status */
#define TLI_STAT_HDE                      BIT(1)                                /*!< current HDE status */
#define TLI_STAT_VS                       BIT(2)                                /*!< current VS status of the TLI */
#define TLI_STAT_HS                       BIT(3)                                /*!< current HS status of the TLI */

/* TLI_LXCTL */
#define TLI_LXCTL_LEN                     BIT(0)                                /*!< layer enable */
#define TLI_LXCTL_CKEYEN                  BIT(1)                                /*!< color keying enable */
#define TLI_LXCTL_LUTEN                   BIT(4)                                /*!< LUT enable */

/* TLI_LXHPOS */
#define TLI_LXHPOS_WLP                    BITS(0,11)                            /*!< window left position */
#define TLI_LXHPOS_WRP                    BITS(16,27)                           /*!< window right position */

/* TLI_LXVPOS */
#define TLI_LXVPOS_WTP                    BITS(0,11)                            /*!< window top position */
#define TLI_LXVPOS_WBP                    BITS(16,27)                           /*!< window bottom position */

/* TLI_LXCKEY */
#define TLI_LXCKEY_CKEYB                  BITS(0,7)                             /*!< color key blue */
#define TLI_LXCKEY_CKEYG                  BITS(8,15)                            /*!< color key green */
#define TLI_LXCKEY_CKEYR                  BITS(16,23)                           /*!< color key red */

/* TLI_LXPPF */
#define TLI_LXPPF_PPF                     BITS(0,2)                             /*!< packeted pixel format */

/* TLI_LXSA */
#define TLI_LXSA_SA                       BITS(0,7)                             /*!< specified alpha */

/* TLI_LXDC */
#define TLI_LXDC_DCB                      BITS(0,7)                             /*!< the default color blue */
#define TLI_LXDC_DCG                      BITS(8,15)                            /*!< the default color green */
#define TLI_LXDC_DCR                      BITS(16,23)                           /*!< the default color red */
#define TLI_LXDC_DCA                      BITS(24,31)                           /*!< the default color alpha */

/* TLI_LXBLEND */
#define TLI_LXBLEND_ACF2                  BITS(0,2)                             /*!< alpha calculation factor 2 of blending method */
#define TLI_LXBLEND_ACF1                  BITS(8,10)                            /*!< alpha calculation factor 1 of blending method */

/* TLI_LXFBADDR */
#define TLI_LXFBADDR_FBADD                BITS(0,31)                            /*!< frame buffer base address */

/* TLI_LXFLLEN */
#define TLI_LXFLLEN_FLL                   BITS(0,13)                            /*!< frame line length */
#define TLI_LXFLLEN_STDOFF                BITS(16,29)                           /*!< frame buffer stride offset */

/* TLI_LXFTLN */
#define TLI_LXFTLN_FTLN                   BITS(0,10)                            /*!< frame total line number */

/* TLI_LXLUT */
#define TLI_LXLUT_TB                      BITS(0,7)                             /*!< blue channel of a LUT entry */
#define TLI_LXLUT_TG                      BITS(8,15)                            /*!< green channel of a LUT entry */
#define TLI_LXLUT_TR                      BITS(16,23)                           /*!< red channel of a LUT entry */
#define TLI_LXLUT_TADD                    BITS(24,31)                           /*!< look up table write address */

/* TLI error code */
#define HAL_TLI_ERROR_NONE                (0U)                                  /*!< no error */
#define HAL_TLI_ERROR_TEF                 BIT(0)                                /*!< transfer error */
#define HAL_TLI_ERROR_FEF                 BIT(1)                                /*!< fifo error */

/* TLI flags */
#define TLI_FLAG_VDE                      TLI_STAT_VDE                          /*!< current VDE status */
#define TLI_FLAG_HDE                      TLI_STAT_HDE                          /*!< current HDE status */
#define TLI_FLAG_VS                       TLI_STAT_VS                           /*!< current VS status of the TLI */
#define TLI_FLAG_HS                       TLI_STAT_HS                           /*!< current HS status of the TLI */
#define TLI_FLAG_LM                       BIT(0) | BIT(31)                      /*!< line mark interrupt flag */
#define TLI_FLAG_FE                       BIT(1) | BIT(31)                      /*!< FIFO error interrupt flag */
#define TLI_FLAG_TE                       BIT(2) | BIT(31)                      /*!< transaction error interrupt flag */
#define TLI_FLAG_LCR                      BIT(3) | BIT(31)                      /*!< layer configuration reloaded interrupt flag */

/* TLI interrupt enable or disable */
#define TLI_INT_LM                        BIT(0)                                /*!< line mark interrupt */
#define TLI_INT_FE                        BIT(1)                                /*!< FIFO error interrupt */
#define TLI_INT_TE                        BIT(2)                                /*!< transaction error interrupt */
#define TLI_INT_LCR                       BIT(3)                                /*!< layer configuration reloaded interrupt */

/* TLI interrupt flag */
#define TLI_INT_FLAG_LM                   BIT(0)                                /*!< line mark interrupt flag */
#define TLI_INT_FLAG_FE                   BIT(1)                                /*!< FIFO error interrupt flag */
#define TLI_INT_FLAG_TE                   BIT(2)                                /*!< transaction error interrupt flag */
#define TLI_INT_FLAG_LCR                  BIT(3)                                /*!< layer configuration reloaded interrupt flag */

/* layer reload configure */
#define TLI_FRAME_BLANK_RELOAD_EN         ((uint8_t)0x00U)                      /*!< the layer configuration will be reloaded at frame blank */
#define TLI_REQUEST_RELOAD_EN             ((uint8_t)0x01U)                      /*!< the layer configuration will be reloaded after this bit sets */

/* dither function */
#define TLI_DITHER_DISABLE                ((uint8_t)0x00U)                      /*!< dither function disable */
#define TLI_DITHER_ENABLE                 ((uint8_t)0x01U)                      /*!< dither function enable */

/* @STRUCT_MEMBER:signalpolarity_hs */
/* @DEFINE:horizontal pulse polarity selection */
#define TLI_HSYN_ACTLIVE_LOW              ((uint32_t)0x00000000U)               /*!< horizontal synchronous pulse active low */
#define TLI_HSYN_ACTLIVE_HIGH             TLI_CTL_HPPS                          /*!< horizontal synchronous pulse active high */

/* @STRUCT_MEMBER:signalpolarity_vs */
/* @DEFINE:vertical pulse polarity selection */
#define TLI_VSYN_ACTLIVE_LOW              ((uint32_t)0x00000000U)               /*!< vertical synchronous pulse active low */
#define TLI_VSYN_ACTLIVE_HIGH             TLI_CTL_VPPS                          /*!< vertical synchronous pulse active high */

/* @STRUCT_MEMBER:signalpolarity_pixelck */
/* @DEFINE:pixel clock polarity selection */
#define TLI_PIXEL_CLOCK_TLI               ((uint32_t)0x00000000U)               /*!< pixel clock is TLI clock */
#define TLI_PIXEL_CLOCK_INVERTEDTLI       TLI_CTL_CLKPS                         /*!< pixel clock is inverted TLI clock */

/* @STRUCT_MEMBER:signalpolarity_de */
/* @DEFINE:data enable polarity selection */
#define TLI_DE_ACTLIVE_LOW                ((uint32_t)0x00000000U)               /*!< data enable active low */
#define TLI_DE_ACTLIVE_HIGH               TLI_CTL_DEPS                          /*!< data enable active high */

/* @STRUCT_MEMBER:Dither_function */
/* @DEFINE:Dither function enable selection */
#define TLI_JITTER_ENABLE                 TLI_CTL_DFEN                          /*!< TLI jitter enable */
#define TLI_JITTER_DISABLE                ((uint32_t)0x00000000U)               /*!< TLI jitter disable */

/* @STRUCT_MEMBER:layer_acf1 */
/* @DEFINE:alpha calculation factor 1 of blending method */
#define LXBLEND_ACF1(regval)              (BITS(8,10) & ((uint32_t)(regval) << 8U))
#define LAYER_ACF1_SA                     LXBLEND_ACF1(4)                       /*!< normalization specified alpha */
#define LAYER_ACF1_PASA                   LXBLEND_ACF1(6)                       /*!< normalization pixel alpha * normalization specified alpha */

/* @STRUCT_MEMBER:layer_acf2 */
/* @DEFINE:alpha calculation factor 2 of blending method */
#define LXBLEND_ACF2(regval)              (BITS(0,2) & ((uint32_t)(regval)))
#define LAYER_ACF2_SA                     LXBLEND_ACF2(5)                       /*!< normalization specified alpha */
#define LAYER_ACF2_PASA                   LXBLEND_ACF2(7)                       /*!< normalization pixel alpha * normalization specified alpha */

/* @STRUCT_MEMBER:layer_function */
/* @DEFINE:alpha calculation factor 2 of blending method */
#define TLI_LAYER_ENABLE                  TLI_LXCTL_LEN                         /*!< layer enable */
#define TLI_LAYER_DISABLE                 ((uint32_t)0x00000000U)               /*!< layer disable */

/* @STRUCT_MEMBER:color_key_function */
/* @DEFINE:color key function enable */
#define TLI_CKEY_ENABLE                   TLI_LXCTL_CKEYEN                      /*!< color keying enable */
#define TLI_CKEY_DISABLE                  ((uint32_t)0x00000000U)               /*!< color keying disable */

/* @STRUCT_MEMBER:lut_function */
/* @DEFINE: lut function enable */
#define TLI_LUT_ENABLE                    TLI_LXCTL_LUTEN                       /*!< LUT enable */
#define TLI_LUT_DISABLE                   ((uint32_t)0x00000000U)               /*!< LUT disable */

/* @PARA: layer_ppf */
/* @ENUM: packeted pixel format */
typedef enum {
    LAYER_PPF_ARGB8888 = 0U, /*!< layerx pixel format ARGB8888 */
    LAYER_PPF_RGB888,        /*!< layerx pixel format RGB888 */
    LAYER_PPF_RGB565,        /*!< layerx pixel format RGB565 */
    LAYER_PPF_ARGB1555,      /*!< layerx pixel format ARGB1555 */
    LAYER_PPF_ARGB4444,      /*!< layerx pixel format ARGB4444 */
    LAYER_PPF_L8,            /*!< layerx pixel format L8 */
    LAYER_PPF_AL44,          /*!< layerx pixel format AL44 */
    LAYER_PPF_AL88           /*!< layerx pixel format AL88 */
} hal_tli_layer_ppf_enum;

/* @PARA: hal_struct_type */
/* @ENUM: SPI structure type enumeration */
typedef enum {
    HAL_TLI_INIT_STRUCT = 0U,             /*!< TLI initialization structure */
    HAL_TLI_DEV_STRUCT,                   /*!< TLI device information structure */
    HAL_TLI_IRQ_STRUCT,                   /*!< TLI interrupt structure */
    HAL_TLI_USER_CALLBACK_STRUCT,         /*!< TLI user callback structure */
    HAL_TLI_LUT_PARAMETER_STRUCT,         /*!< TLI LUT parameter structure */
    HAL_TLI_LAYER_PARAMETER_STRUCT        /*!< TLI layer parameter structure */
} hal_tli_struct_type_enum;

/* TLI run state enumeration */
typedef enum {
    HAL_TLI_STATE_READY    = 0x01U, /*!< peripheral Initialized and ready for use */
    HAL_TLI_STATE_BUSY     = 0x02U, /*!< an internal process is ongoing */
    HAL_TLI_STATE_ERROR    = 0x03U, /*!< TLI error state */
    HAL_TLI_STATE_TIMEROUT = 0x04U  /*!< TLI timeout state */
} hal_tli_run_state_enum;

/* @PARA: layer_parameter */
/* @STRUCT: TLI layer parameter struct definitions */
typedef struct {
    uint32_t layer_frame_bufaddr;           /*!< frame buffer base address */
    uint16_t layer_function;                /*!< layer function enable */
    uint16_t lut_function;                  /*!< lut function enable */
    uint16_t color_key_function;            /*!< color key function enable */
    hal_tli_layer_ppf_enum layer_ppf;       /*!< packeted pixel format */
    uint16_t layer_window_rightpos;         /*!< window right position */
    uint16_t layer_window_leftpos;          /*!< window left position */
    uint16_t layer_window_bottompos;        /*!< window bottom position */
    uint16_t layer_window_toppos;           /*!< window top position */
    uint16_t layer_frame_buf_stride_offset; /*!< frame buffer stride offset */
    uint16_t layer_frame_line_length;       /*!< frame line length */
    uint16_t layer_frame_total_line_number; /*!< frame total line number */
    uint8_t layer_default_red;              /*!< the default color red */
    uint8_t layer_default_green;            /*!< the default color green */
    uint8_t layer_default_blue;             /*!< the default color blue */
    uint8_t layer_sa;                       /*!< specified alpha */
    uint8_t layer_default_alpha;            /*!< the default color alpha */
    uint16_t layer_acf1;                    /*!< alpha calculation factor 1 of blending method */
    uint8_t layer_acf2;                     /*!< alpha calculation factor 2 of blending method */
} hal_tli_layer_parameter_struct;

/* @PARA: lut */
/* @STRUCT: TLI layer LUT parameter struct definitions */
typedef struct {
    uint32_t layer_table_addr;       /*!< look up table write address */
    uint8_t layer_lut_channel_red;   /*!< red channel of a LUT entry */
    uint8_t layer_lut_channel_green; /*!< green channel of a LUT entry */
    uint8_t layer_lut_channel_blue;  /*!< blue channel of a LUT entry */
} hal_tli_layer_lut_parameter_struct;

/* @PARA: tli_init */
/* @STRUCT:TLI parameter struct definitions */
typedef struct {
    uint32_t signalpolarity_hs;                     /*!< horizontal pulse polarity selection */
    uint32_t signalpolarity_vs;                     /*!< vertical pulse polarity selection */
    uint32_t signalpolarity_de;                     /*!< data enable polarity selection */
    uint32_t signalpolarity_pixelck;                /*!< pixel clock polarity selection */
    hal_tli_layer_parameter_struct layer_parameter; /*!< layer parameter struct */
    hal_tli_layer_lut_parameter_struct lut;         /*!< lut parameter struct */
    uint16_t synpsz_vpsz;                           /*!< size of the vertical synchronous pulse */
    uint16_t synpsz_hpsz;                           /*!< size of the horizontal synchronous pulse */
    uint16_t backpsz_vbpsz;                         /*!< size of the vertical back porch plus synchronous pulse */
    uint16_t backpsz_hbpsz;                         /*!< size of the horizontal back porch plus synchronous pulse */
    uint16_t activesz_vasz;                        /*!< size of the vertical active area width plus back porch and synchronous pulse */
    uint16_t activesz_hasz;                        /*!< size of the horizontal active area width plus back porch and synchronous pulse */
    uint16_t totalsz_vtsz;                         /*!< vertical total size of the display */
    uint16_t totalsz_htsz;                         /*!< horizontal total size of the display */
    uint8_t backcolor_red;                         /*!< background value red */
    uint8_t backcolor_green;                       /*!< background value green */
    uint8_t backcolor_blue;                        /*!< background value blue */
    uint32_t Dither_function;                      /*!< dither function enable selection */
    uint8_t color_key_blue;                        /*!< color key blue value */
    uint8_t color_key_red;                         /*!< color key red value */
    uint8_t color_key_green;                       /*!< color key green value */
} hal_tli_init_struct;

/* TLI device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb line_mark_handle;                  /*!< TLI link complete callback function */
    __IO hal_irq_handle_cb layer_configuration_reload_handle; /*!< TLI Layer configuration reload callback function */
    __IO hal_irq_handle_cb fifo_error_handle;                 /*!< TLI FIFO error callback function */
    __IO hal_irq_handle_cb transmit_error_handle;             /*!< TLI transmit error callback function */
} hal_tli_irq_struct;

/* @PARA: tli_struct */
/* @STRUCT: TLI device information structure */
typedef struct {
    uint32_t layer_num;                        /*!< layer num */
    hal_tli_irq_struct tli_irq;                /*!< TLI device interrupt callback function pointer */
    void *line_mark_callback;                  /*!< line_mark callback function pointer */
    void *layer_configuration_reload_callback; /*!< Layer configuration reload callback function pointer */
    void *error_callback;                      /*!< error callback function pointer */
    __IO hal_tli_run_state_enum state;         /*!< TLI communication state */
    __IO uint32_t error_code;                  /*!< TLI error code */
    hal_mutex_enum mutex;                      /*!< lock */
} hal_tli_dev_struct;

/* TLI device user callback function pointer */
typedef void (*hal_tli_user_cb)(hal_tli_dev_struct *tli);

/* TLI callback structure */
typedef struct {
    hal_tli_user_cb layer_configuration_reload_func; /*!< TLI user complete callback function */
    hal_tli_user_cb fifo_error_func;                 /*!< TLI fifo error callback function */
    hal_tli_user_cb transmit_error_func;             /*!< TLI transmit error callback function */
    hal_tli_user_cb line_mark_func;                  /*!< TLI line mark callback function */
} hal_tli_user_callback_struct;

/* @STRUCT_MEMBER: synpsz_vpsz */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: synpsz_hpsz */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: backpsz_vbpsz */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: backpsz_hbpsz */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: activesz_hasz */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: activesz_vasz */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: totalsz_vtsz */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: totalsz_htsz */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: backcolor_red */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: backcolor_green */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: backcolor_blue */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_window_rightpos */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_window_leftpos */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_window_bottompos */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_window_toppos */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_default_red */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_default_green */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_default_blue */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_default_alpha */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_sa */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_frame_bufaddr */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_frame_buf_stride_offset */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_frame_line_length */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_frame_total_line_number */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_table_addr */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_lut_channel_red */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_lut_channel_green */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: layer_lut_channel_blue */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: color_key_blue */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: color_key_red */
/* @DEFINE: NULL*/

/* @STRUCT_MEMBER: color_key_green */
/* @DEFINE: NULL*/

/* function declarations */
/* initialization functions, TLI enable or disable, TLI reload mode configuration */
/* @FUNCTION: deinitialize TLI registers */
int32_t hal_tli_deinit(hal_tli_dev_struct *tli_dev);
/* @FUNCTION:  initialize the xxx structure with the default values */
/* deinitialize the TLI structure */
int32_t hal_tli_struct_init(hal_tli_struct_type_enum hal_struct_type, void *p_struct);
/* @FUNCTION: initialize TLI */
int32_t hal_tli_init(hal_tli_dev_struct *tli_dev, uint32_t layer, hal_tli_init_struct *tli_init);
/* @END */

/* initialize TLI layer */
int32_t hal_tli_layer_init(uint32_t layerx, hal_tli_layer_parameter_struct *layer_struct);
/* initialize TLI layer reload */
int32_t hal_tli_layer_init_reload(uint32_t layerx, hal_tli_layer_parameter_struct *layer_struct);

/* initialize TLI layer LUT */
int32_t hal_tli_lut_init(uint32_t layerx, hal_tli_layer_lut_parameter_struct *lut_struct);
/* initialize TLI layer LUT reload */
int32_t hal_tli_lut_init_reload(uint32_t layerx, hal_tli_layer_lut_parameter_struct *lut_struct);

/* configuration layer window size */
int32_t hal_tli_layer_window_size_config(uint32_t layerx, hal_tli_layer_parameter_struct *layer_struct, \
                                         uint16_t x_size, uint16_t y_size);

/* reconfigure window position */
void hal_tli_layer_window_offset_modify(uint32_t layerx, uint16_t offset_x, uint16_t offset_y);
/* reconfigure window position reload */
void hal_tli_layer_window_offset_modify_reload(uint32_t layerx, uint16_t offset_x, uint16_t offset_y);

/* tli interrupt handler content function,which is merely used in tsi_handler */
int32_t hal_tli_irq(hal_tli_dev_struct *tli_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_tli_irq_handle_set(hal_tli_dev_struct *tli_dev, hal_tli_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_tli_irq_handle_all_reset(hal_tli_dev_struct *tli_dev);
/* TLI interrupt handler content function,which is merely used in tli_error_handler */
int32_t hal_tli_error_irq(hal_tli_dev_struct *tli_dev);
/* enable TLI */
void hal_tli_enable(void);
/* disable TLI */
void hal_tli_disable(void);
/* configure TLI reload mode */
void hal_tli_reload_config(uint8_t reload_mod);

/* enable TLI layer */
void hal_tli_layer_enable(uint32_t layerx);
/* disable TLI layer */
void hal_tli_layer_disable(uint32_t layerx);

/* enable TLI layer color keying */
void hal_tli_color_key_enable(uint32_t layerx);
/* enable TLI layer color keying reload */
void hal_tli_color_key_enable_reload(uint32_t layerx);
/* disable TLI layer color keying */
void hal_tli_color_key_disable(uint32_t layerx);
/* disable TLI layer color keying reload */
void hal_tli_color_key_disable_reload(uint32_t layerx);
/* enable TLI layer LUT */
void hal_tli_lut_enable(uint32_t layerx);
/* enable TLI layer LUT reload */
void hal_tli_lut_enable_reload(uint32_t layerx);
/* disable TLI layer LUT */
void hal_tli_lut_disable(uint32_t layerx);
/* disable TLI layer LUT reload */
void hal_tli_lut_disable_reload(uint32_t layerx);

/* configure TLI layer reload buffer address configuration */
void hal_tli_layer_reload_bufaddress_config(uint32_t layerx, uint32_t bufaddr);

/* configure TLI dither function */
void hal_tli_dither_config(uint8_t dither_stat);
/* configure TLI dither function reload */
void hal_tli_dither_config_reload(uint8_t dither_stat);
/* set line mark value */
void hal_tli_line_mark_set(uint16_t line_num);
/* set line mark value reload */
void hal_tli_line_mark_set_reload(uint16_t line_num);
/* initialize TLI layer color key */
void hal_tli_color_key_init(uint32_t layerx, uint8_t redkey, uint8_t greenkey, uint8_t bluekey);
/* initialize TLI layer color key */
void hal_tli_color_key_init_reload(uint32_t layerx, uint8_t redkey, uint8_t greenkey, uint8_t bluekey);

/* TLI state get */
hal_tli_run_state_enum hal_tli_state_get(hal_tli_dev_struct *tli_dev);
/* TLI error state get */
uint32_t hal_tli_error_state_get(hal_tli_dev_struct *tli_dev);

/* synchronous pulse size configuration */
int32_t hals_tli_synchronous_pulse_config(uint16_t synpsz_vpsz, uint16_t synpsz_hpsz);
/* back-porch size configuration */
int32_t hals_tli_back_porch_config(uint16_t backpsz_vbpsz, uint16_t backpsz_hbpsz);
/* active size configuration */
int32_t hals_tli_active_size_config(uint16_t activesz_vasz, uint16_t activesz_hasz);
/* configure total size */
int32_t hals_tli_total_size_config(uint16_t totalsz_vtsz, uint16_t totalsz_htsz);
/* background color configuration */
void hals_tli_background_color_config(uint8_t backcolor_red, uint8_t backcolor_green, uint8_t backcolor_blue);

/* get current displayed position */
uint32_t hals_tli_current_pos_get(void);

/* flag and interrupt functions */
/* enable TLI interrupt */
void hals_tli_interrupt_enable(uint32_t interrupt);
/* disable TLI interrupt */
void hals_tli_interrupt_disable(uint32_t interrupt);
/* get TLI interrupt flag */
FlagStatus hals_tli_interrupt_flag_get(uint32_t int_flag);
/* clear TLI interrupt flag */
void hals_tli_interrupt_flag_clear(uint32_t int_flag);
/* get TLI flag or state in TLI_INTF register or TLI_STAT register */
FlagStatus hals_tli_flag_get(uint32_t flag);

#endif /* GD32H7XX_HAL_TLI_H */
