/*!
    \file    gd32h7xx_hal_timer.h
    \brief   definitions for the TIMER

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/
#ifndef GD32H7XX_HAL_TIMER_H
#define GD32H7XX_HAL_TIMER_H

#include "gd32h7xx_hal.h"

/* TIMER definitions */
#define TIMER0                                          (TIMER_BASE + 0x00010000U)                  /*!< TIMER0 base address */
#define TIMER1                                          (TIMER_BASE + 0x00000000U)                  /*!< TIMER1 base address */
#define TIMER2                                          (TIMER_BASE + 0x00000400U)                  /*!< TIMER2 base address */
#define TIMER3                                          (TIMER_BASE + 0x00000800U)                  /*!< TIMER3 base address */
#define TIMER4                                          (TIMER_BASE + 0x00000C00U)                  /*!< TIMER4 base address */
#define TIMER5                                          (TIMER_BASE + 0x00001000U)                  /*!< TIMER5 base address */
#define TIMER6                                          (TIMER_BASE + 0x00001400U)                  /*!< TIMER6 base address */
#define TIMER7                                          (TIMER_BASE + 0x00010400U)                  /*!< TIMER7 base address */
#define TIMER14                                         (TIMER_BASE + 0x00014000U)                  /*!< TIMER14 base address */
#define TIMER15                                         (TIMER_BASE + 0x00014400U)                  /*!< TIMER15 base address */
#define TIMER16                                         (TIMER_BASE + 0x00014800U)                  /*!< TIMER16 base address */
#define TIMER22                                         (TIMER_BASE + 0x0000E000U)                  /*!< TIMER22 base address */
#define TIMER23                                         (TIMER_BASE + 0x0000E400U)                  /*!< TIMER23 base address */
#define TIMER30                                         (TIMER_BASE + 0x0000E800U)                  /*!< TIMER30 base address */
#define TIMER31                                         (TIMER_BASE + 0x0000EC00U)                  /*!< TIMER31 base address */
#define TIMER40                                         (TIMER_BASE + 0x0001D000U)                  /*!< TIMER40 base address */
#define TIMER41                                         (TIMER_BASE + 0x0001D400U)                  /*!< TIMER41 base address */
#define TIMER42                                         (TIMER_BASE + 0x0001D800U)                  /*!< TIMER42 base address */
#define TIMER43                                         (TIMER_BASE + 0x0001DC00U)                  /*!< TIMER43 base address */
#define TIMER44                                         (TIMER_BASE + 0x0001F000U)                  /*!< TIMER44 base address */
#define TIMER50                                         (TIMER_BASE + 0x0000F000U)                  /*!< TIMER50 base address */
#define TIMER51                                         (TIMER_BASE + 0x0000F400U)                  /*!< TIMER51 base address */

/* registers definitions */
#define TIMER_CTL0(timerx)                              REG32((timerx) + 0x00000000U)               /*!< TIMER control register 0 */
#define TIMER_CTL1(timerx)                              REG32((timerx) + 0x00000004U)               /*!< TIMER control register 1 */
#define TIMER_SMCFG(timerx)                             REG32((timerx) + 0x00000008U)               /*!< TIMER slave mode configuration register */
#define TIMER_DMAINTEN(timerx)                          REG32((timerx) + 0x0000000CU)               /*!< TIMER DMA and interrupt enable register */
#define TIMER_INTF(timerx)                              REG32((timerx) + 0x00000010U)               /*!< TIMER interrupt flag register */
#define TIMER_SWEVG(timerx)                             REG32((timerx) + 0x00000014U)               /*!< TIMER software event generation register */
#define TIMER_CHCTL0(timerx)                            REG32((timerx) + 0x00000018U)               /*!< TIMER channel control register 0 */
#define TIMER_CHCTL1(timerx)                            REG32((timerx) + 0x0000001CU)               /*!< TIMER channel control register 1 */
#define TIMER_CHCTL2(timerx)                            REG32((timerx) + 0x00000020U)               /*!< TIMER channel control register 2 */
#define TIMER_CNT(timerx)                               REG32((timerx) + 0x00000024U)               /*!< TIMER counter register */
#define TIMER_CNTL(timerx)                              REG32((timerx) + 0x00000024U)               /*!< TIMER counter low register (only for TIMERx, x=50,51) */
#define TIMER_PSC(timerx)                               REG32((timerx) + 0x00000028U)               /*!< TIMER prescaler register */
#define TIMER_CAR(timerx)                               REG32((timerx) + 0x0000002CU)               /*!< TIMER counter auto reload register */
#define TIMER_CARL(timerx)                              REG32((timerx) + 0x0000002CU)               /*!< TIMER counter auto reload low register (only for TIMERx, x=50,51) */
#define TIMER_CREP0(timerx)                             REG32((timerx) + 0x00000030U)               /*!< TIMER counter repetition register 0 */
#define TIMER_CH0CV(timerx)                             REG32((timerx) + 0x00000034U)               /*!< TIMER channel 0 capture or compare value register */
#define TIMER_CH1CV(timerx)                             REG32((timerx) + 0x00000038U)               /*!< TIMER channel 1 capture or compare value register */
#define TIMER_CH2CV(timerx)                             REG32((timerx) + 0x0000003CU)               /*!< TIMER channel 2 capture or compare value register */
#define TIMER_CH3CV(timerx)                             REG32((timerx) + 0x00000040U)               /*!< TIMER channel 3 capture or compare value register */
#define TIMER_CCHP(timerx)                              REG32((timerx) + 0x00000044U)               /*!< TIMER channel complementary protection register */
#define TIMER_MCHCTL0(timerx)                           REG32((timerx) + 0x00000048U)               /*!< TIMER multi mode channel control register 0 */
#define TIMER_MCHCTL1(timerx)                           REG32((timerx) + 0x0000004CU)               /*!< TIMER multi mode channel control register 1 */
#define TIMER_MCHCTL2(timerx)                           REG32((timerx) + 0x00000050U)               /*!< TIMER multi mode channel control register 2 */
#define TIMER_MCH0CV(timerx)                            REG32((timerx) + 0x00000054U)               /*!< TIMER multi mode channel 0 capture or compare value register */
#define TIMER_MCH1CV(timerx)                            REG32((timerx) + 0x00000058U)               /*!< TIMER multi mode channel 1 capture or compare value register */
#define TIMER_MCH2CV(timerx)                            REG32((timerx) + 0x0000005CU)               /*!< TIMER multi mode channel 2 capture or compare value register */
#define TIMER_MCH3CV(timerx)                            REG32((timerx) + 0x00000060U)               /*!< TIMER multi mode channel 3 capture or compare value register */
#define TIMER_CH0COMV_ADD(timerx)                       REG32((timerx) + 0x00000064U)               /*!< TIMER channel 0 additional compare value register */
#define TIMER_CH1COMV_ADD(timerx)                       REG32((timerx) + 0x00000068U)               /*!< TIMER channel 1 additional compare value register */
#define TIMER_CH2COMV_ADD(timerx)                       REG32((timerx) + 0x0000006CU)               /*!< TIMER channel 2 additional compare value register */
#define TIMER_CH3COMV_ADD(timerx)                       REG32((timerx) + 0x00000070U)               /*!< TIMER channel 3 additional compare value register */
#define TIMER_CTL2(timerx)                              REG32((timerx) + 0x00000074U)               /*!< TIMER control register 2 */
#define TIMER_FCCHP0(timerx)                            REG32((timerx) + 0x0000007CU)               /*!< TIMER free complementary channel protection register 0 */
#define TIMER_FCCHP1(timerx)                            REG32((timerx) + 0x00000080U)               /*!< TIMER free complementary channel protection register 1 */
#define TIMER_FCCHP2(timerx)                            REG32((timerx) + 0x00000084U)               /*!< TIMER free complementary channel protection register 2 */
#define TIMER_FCCHP3(timerx)                            REG32((timerx) + 0x00000088U)               /*!< TIMER free complementary channel protection register 3 */
#define TIMER_AFCTL0(timerx)                            REG32((timerx) + 0x0000008CU)               /*!< TIMER alternate function control register 0 */
#define TIMER_AFCTL1(timerx)                            REG32((timerx) + 0x00000090U)               /*!< TIMER alternate function control register 1 */
#define TIMER_WDGPER(timerx)                            REG32((timerx) + 0x00000094U)               /*!< TIMER watchdog counter period register */
#define TIMER_CREP1(timerx)                             REG32((timerx) + 0x00000098U)               /*!< TIMER counter repetition register 1 */
#define TIMER_CNTH(timerx)                              REG32((timerx) + 0x000000D0U)               /*!< TIMER counter high register (only for TIMERx, x=50,51) */
#define TIMER_CARH(timerx)                              REG32((timerx) + 0x000000D4U)               /*!< TIMER counter auto reload high register (only for TIMERx, x=50,51) */
#define TIMER_DMACFG(timerx)                            REG32((timerx) + 0x000000E0U)               /*!< TIMER DMA configuration register */
#define TIMER_DMATB(timerx)                             REG32((timerx) + 0x000000E4U)               /*!< TIMER DMA transfer buffer register */
#define TIMER_CFG(timerx)                               REG32((timerx) + 0x000000FCU)               /*!< TIMER configuration register */

/* bits definitions */
/* TIMER_CTL0 */
#define TIMER_CTL0_CEN                                  BIT(0)                                      /*!< TIMER counter enable */
#define TIMER_CTL0_UPDIS                                BIT(1)                                      /*!< update disable */
#define TIMER_CTL0_UPS                                  BIT(2)                                      /*!< update source */
#define TIMER_CTL0_SPM                                  BIT(3)                                      /*!< single pulse mode */
#define TIMER_CTL0_DIR                                  BIT(4)                                      /*!< timer counter direction */
#define TIMER_CTL0_CAM                                  BITS(5,6)                                   /*!< counter aligned mode selection */
#define TIMER_CTL0_ARSE                                 BIT(7)                                      /*!< auto-reload shadow enable */
#define TIMER_CTL0_CKDIV                                BITS(8,9)                                   /*!< clock division */
#define TIMER_CTL0_UPIFBUEN                             BIT(11)                                     /*!< UPIF bit backup enable */

/* TIMER_CTL1 */
#define TIMER_CTL1_CCSE                                 BIT(0)                                      /*!< commutation control shadow enable */
#define TIMER_CTL1_CCUC                                 (BIT(2) | BITS(30,31))                      /*!< commutation control shadow register update control */
#define TIMER_CTL1_DMAS                                 BIT(3)                                      /*!< DMA request source selection */
#define TIMER_CTL1_MMC0                                 BITS(4,6)                                   /*!< master mode control 0 */
#define TIMER_CTL1_TI0S                                 BIT(7)                                      /*!< channel 0 trigger input selection(hall mode selection) */
#define TIMER_CTL1_ISO0                                 BIT(8)                                      /*!< idle state of channel 0 output */
#define TIMER_CTL1_ISO0N                                BIT(9)                                      /*!< idle state of channel 0 complementary output */
#define TIMER_CTL1_ISO1                                 BIT(10)                                     /*!< idle state of channel 1 output */
#define TIMER_CTL1_ISO1N                                BIT(11)                                     /*!< idle state of channel 1 complementary output */
#define TIMER_CTL1_ISO2                                 BIT(12)                                     /*!< idle state of channel 2 output */
#define TIMER_CTL1_ISO2N                                BIT(13)                                     /*!< idle state of channel 2 complementary output */
#define TIMER_CTL1_ISO3                                 BIT(14)                                     /*!< idle state of channel 3 output */
#define TIMER_CTL1_ISO3N                                BIT(15)                                     /*!< idle state of channel 3 complementary output */
#define TIMER_CTL1_MMC1                                 BITS(20,22)                                 /*!< master mode control 1 */

/* TIMER_SMCFG */
#define TIMER_SMCFG_MSM                                 BIT(7)                                      /*!< master-slave mode */
#define TIMER_SMCFG_ETFC                                BITS(8,11)                                  /*!< external trigger filter control */
#define TIMER_SMCFG_ETPSC                               BITS(12,13)                                 /*!< external trigger prescaler */
#define TIMER_SMCFG_SMC1                                BIT(14)                                     /*!< part of SMC for enable external clock mode 1 */
#define TIMER_SMCFG_ETP                                 BIT(15)                                     /*!< external trigger polarity */

/* TIMER_DMAINTEN */
#define TIMER_DMAINTEN_UPIE                             BIT(0)                                      /*!< update interrupt enable */
#define TIMER_DMAINTEN_CH0IE                            BIT(1)                                      /*!< channel 0 capture or compare interrupt enable */
#define TIMER_DMAINTEN_CH1IE                            BIT(2)                                      /*!< channel 1 capture or compare interrupt enable */
#define TIMER_DMAINTEN_CH2IE                            BIT(3)                                      /*!< channel 2 capture or compare interrupt enable */
#define TIMER_DMAINTEN_CH3IE                            BIT(4)                                      /*!< channel 3 capture or compare interrupt enable */
#define TIMER_DMAINTEN_CMTIE                            BIT(5)                                      /*!< commutation interrupt request enable */
#define TIMER_DMAINTEN_TRGIE                            BIT(6)                                      /*!< trigger interrupt enable */
#define TIMER_DMAINTEN_BRKIE                            BIT(7)                                      /*!< break interrupt enable */
#define TIMER_DMAINTEN_UPDEN                            BIT(8)                                      /*!< update DMA request enable */
#define TIMER_DMAINTEN_CH0DEN                           BIT(9)                                      /*!< channel 0 capture or compare DMA request enable */
#define TIMER_DMAINTEN_CH1DEN                           BIT(10)                                     /*!< channel 1 capture or compare DMA request enable */
#define TIMER_DMAINTEN_CH2DEN                           BIT(11)                                     /*!< channel 2 capture or compare DMA request enable */
#define TIMER_DMAINTEN_CH3DEN                           BIT(12)                                     /*!< channel 3 capture or compare DMA request enable */
#define TIMER_DMAINTEN_CMTDEN                           BIT(13)                                     /*!< commutation DMA request enable */
#define TIMER_DMAINTEN_TRGDEN                           BIT(14)                                     /*!< trigger DMA request enable */
#define TIMER_DMAINTEN_DECJIE                           BIT(16)                                     /*!< quadrature decoder signal jump (the two signals jump at the same time) interrupt enable */
#define TIMER_DMAINTEN_DECDISIE                         BIT(17)                                     /*!< quadrature decoder signal disconnection interrupt enable */
#define TIMER_DMAINTEN_MCH0IE                           BIT(20)                                     /*!< multi mode channel 0 capture or compare interrupt enable */
#define TIMER_DMAINTEN_MCH1IE                           BIT(21)                                     /*!< multi mode channel 1 capture or compare interrupt enable */
#define TIMER_DMAINTEN_MCH2IE                           BIT(22)                                     /*!< multi mode channel 2 capture or compare interrupt enable */
#define TIMER_DMAINTEN_MCH3IE                           BIT(23)                                     /*!< multi mode channel 3 capture or compare interrupt enable */
#define TIMER_DMAINTEN_MCH0DEN                          BIT(24)                                     /*!< multi mode channel 0 capture or compare DMA request enable */
#define TIMER_DMAINTEN_MCH1DEN                          BIT(25)                                     /*!< multi mode channel 1 capture or compare DMA request enable */
#define TIMER_DMAINTEN_MCH2DEN                          BIT(26)                                     /*!< multi mode channel 2 capture or compare DMA request enable */
#define TIMER_DMAINTEN_MCH3DEN                          BIT(27)                                     /*!< multi mode channel 3 capture or compare DMA request enable */
#define TIMER_DMAINTEN_CH0COMADDIE                      BIT(28)                                     /*!< channel 0 additional compare interrupt enable */
#define TIMER_DMAINTEN_CH1COMADDIE                      BIT(29)                                     /*!< channel 1 additional compare interrupt enable */
#define TIMER_DMAINTEN_CH2COMADDIE                      BIT(30)                                     /*!< channel 2 additional compare interrupt enable */
#define TIMER_DMAINTEN_CH3COMADDIE                      BIT(31)                                     /*!< channel 3 additional compare interrupt enable */

/* TIMER_INTF */
#define TIMER_INTF_UPIF                                 BIT(0)                                      /*!< update interrupt flag */
#define TIMER_INTF_CH0IF                                BIT(1)                                      /*!< channel 0 capture or compare interrupt flag */
#define TIMER_INTF_CH1IF                                BIT(2)                                      /*!< channel 1 capture or compare interrupt flag */
#define TIMER_INTF_CH2IF                                BIT(3)                                      /*!< channel 2 capture or compare interrupt flag */
#define TIMER_INTF_CH3IF                                BIT(4)                                      /*!< channel 3 capture or compare interrupt flag */
#define TIMER_INTF_CMTIF                                BIT(5)                                      /*!< channel commutation interrupt flag */
#define TIMER_INTF_TRGIF                                BIT(6)                                      /*!< trigger interrupt flag */
#define TIMER_INTF_BRK0IF                               BIT(7)                                      /*!< BREAK0 interrupt flag */
#define TIMER_INTF_BRK1IF                               BIT(8)                                      /*!< BREAK1 interrupt flag */
#define TIMER_INTF_CH0OF                                BIT(9)                                      /*!< channel 0 over capture flag */
#define TIMER_INTF_CH1OF                                BIT(10)                                     /*!< channel 1 over capture flag */
#define TIMER_INTF_CH2OF                                BIT(11)                                     /*!< channel 2 over capture flag */
#define TIMER_INTF_CH3OF                                BIT(12)                                     /*!< channel 3 over capture flag */
#define TIMER_INTF_SYSBIF                               BIT(13)                                     /*!< system source break interrupt flag */
#define TIMER_INTF_DECJIF                               BIT(16)                                     /*!< quadrature decoder signal jump (the two signals jump at the same time) interrupt flag */
#define TIMER_INTF_DECDISIF                             BIT(17)                                     /*!< quadrature decoder signal disconnection interrupt flag */
#define TIMER_INTF_MCH0IF                               BIT(20)                                     /*!< multi mode channel 0 capture or compare interrupt flag */
#define TIMER_INTF_MCH1IF                               BIT(21)                                     /*!< multi mode channel 1 capture or compare interrupt flag */
#define TIMER_INTF_MCH2IF                               BIT(22)                                     /*!< multi mode channel 2 capture or compare interrupt flag */
#define TIMER_INTF_MCH3IF                               BIT(23)                                     /*!< multi mode channel 3 capture or compare interrupt flag */
#define TIMER_INTF_MCH0OF                               BIT(24)                                     /*!< multi mode channel 0 over capture flag */
#define TIMER_INTF_MCH1OF                               BIT(25)                                     /*!< multi mode channel 1 over capture flag */
#define TIMER_INTF_MCH2OF                               BIT(26)                                     /*!< multi mode channel 2 over capture flag */
#define TIMER_INTF_MCH3OF                               BIT(27)                                     /*!< multi mode channel 3 over capture flag */
#define TIMER_INTF_CH0COMADDIF                          BIT(28)                                     /*!< channel 0 additional compare interrupt flag */
#define TIMER_INTF_CH1COMADDIF                          BIT(29)                                     /*!< channel 1 additional compare interrupt flag */
#define TIMER_INTF_CH2COMADDIF                          BIT(30)                                     /*!< channel 2 additional compare interrupt flag */
#define TIMER_INTF_CH3COMADDIF                          BIT(31)                                     /*!< channel 3 additional compare interrupt flag */

/* TIMER_SWEVG */
#define TIMER_SWEVG_UPG                                 BIT(0)                                      /*!< update event generate */
#define TIMER_SWEVG_CH0G                                BIT(1)                                      /*!< channel 0 capture or compare event generation */
#define TIMER_SWEVG_CH1G                                BIT(2)                                      /*!< channel 1 capture or compare event generation */
#define TIMER_SWEVG_CH2G                                BIT(3)                                      /*!< channel 2 capture or compare event generation */
#define TIMER_SWEVG_CH3G                                BIT(4)                                      /*!< channel 3 capture or compare event generation */
#define TIMER_SWEVG_CMTG                                BIT(5)                                      /*!< channel commutation event generation */
#define TIMER_SWEVG_TRGG                                BIT(6)                                      /*!< trigger event generation */
#define TIMER_SWEVG_BRK0G                               BIT(7)                                      /*!< BREAK0 event generation */
#define TIMER_SWEVG_BRK1G                               BIT(8)                                      /*!< BREAK1 event generation */
#define TIMER_SWEVG_MCH0G                               BIT(20)                                     /*!< multi mode channel 0 capture or compare event generation */
#define TIMER_SWEVG_MCH1G                               BIT(21)                                     /*!< multi mode channel 1 capture or compare event generation */
#define TIMER_SWEVG_MCH2G                               BIT(22)                                     /*!< multi mode channel 2 capture or compare event generation */
#define TIMER_SWEVG_MCH3G                               BIT(23)                                     /*!< multi mode channel 3 capture or compare event generation */
#define TIMER_SWEVG_CH0COMADDG                          BIT(28)                                     /*!< channel 0 additional compare event generation */
#define TIMER_SWEVG_CH1COMADDG                          BIT(29)                                     /*!< channel 1 additional compare event generation */
#define TIMER_SWEVG_CH2COMADDG                          BIT(30)                                     /*!< channel 2 additional compare event generation */
#define TIMER_SWEVG_CH3COMADDG                          BIT(31)                                     /*!< channel 3 additional compare event generation */

/* TIMER_CHCTL0 */
/* output compare mode */
#define TIMER_CHCTL0_CH0MS                              (BITS(0,1) | BIT(30))                       /*!< channel 0 mode selection */
#define TIMER_CHCTL0_CH0COMSEN                          BIT(3)                                      /*!< channel 0 output compare shadow enable */
#define TIMER_CHCTL0_CH0COMCTL                          (BITS(4,6) | BIT(16))                       /*!< channel 0 output compare control */
#define TIMER_CHCTL0_CH0COMCEN                          BIT(7)                                      /*!< channel 0 output compare clear enable */
#define TIMER_CHCTL0_CH1MS                              (BITS(8,9) | BIT(31))                       /*!< channel 1 mode selection */
#define TIMER_CHCTL0_CH1COMSEN                          BIT(11)                                     /*!< channel 1 output compare shadow enable */
#define TIMER_CHCTL0_CH1COMCTL                          (BITS(12,14) | BIT(24))                     /*!< channel 1 output compare control */
#define TIMER_CHCTL0_CH1COMCEN                          BIT(15)                                     /*!< channel 1 output compare clear enable */
#define TIMER_CHCTL0_CH0COMADDSEN                       BIT(28)                                     /*!< channel 0 additional compare output shadow enable */
#define TIMER_CHCTL0_CH1COMADDSEN                       BIT(29)                                     /*!< channel 1 additional compare output shadow enable */
/* input capture mode */
#define TIMER_CHCTL0_CH0CAPPSC                          BITS(2,3)                                   /*!< channel 0 input capture prescaler */
#define TIMER_CHCTL0_CH0CAPFLT                          BITS(4,7)                                   /*!< channel 0 input capture filter control */
#define TIMER_CHCTL0_CH1CAPPSC                          BITS(10,11)                                 /*!< channel 1 input capture prescaler */
#define TIMER_CHCTL0_CH1CAPFLT                          BITS(12,15)                                 /*!< channel 1 input capture filter control */

/* TIMER_CHCTL1 */
/* output compare mode */
#define TIMER_CHCTL1_CH2MS                              (BITS(0,1) | BIT(30))                       /*!< channel 2 mode selection */
#define TIMER_CHCTL1_CH2COMSEN                          BIT(3)                                      /*!< channel 2 output compare shadow enable */
#define TIMER_CHCTL1_CH2COMCTL                          (BITS(4,6) | BIT(16))                       /*!< channel 2 output compare control */
#define TIMER_CHCTL1_CH2COMCEN                          BIT(7)                                      /*!< channel 2 output compare clear enable */
#define TIMER_CHCTL1_CH3MS                              (BITS(8,9) | BIT(31))                       /*!< channel 3 mode selection */
#define TIMER_CHCTL1_CH3COMSEN                          BIT(11)                                     /*!< channel 3 output compare shadow enable */
#define TIMER_CHCTL1_CH3COMCTL                          (BITS(12,14) | BIT(24))                     /*!< channel 3 output compare control */
#define TIMER_CHCTL1_CH3COMCEN                          BIT(15)                                     /*!< channel 3 output compare clear enable */
#define TIMER_CHCTL1_CH2COMADDSEN                       BIT(28)                                     /*!< channel 2 additional compare output shadow enable */
#define TIMER_CHCTL1_CH3COMADDSEN                       BIT(29)                                     /*!< channel 3 additional compare output shadow enable */
/* input capture mode */
#define TIMER_CHCTL1_CH2CAPPSC                          BITS(2,3)                                   /*!< channel 2 input capture prescaler */
#define TIMER_CHCTL1_CH2CAPFLT                          BITS(4,7)                                   /*!< channel 2 input capture filter control */
#define TIMER_CHCTL1_CH3CAPPSC                          BITS(10,11)                                 /*!< channel 3 input capture prescaler */
#define TIMER_CHCTL1_CH3CAPFLT                          BITS(12,15)                                 /*!< channel 3 input capture filter control */

/* TIMER_CHCTL2 */
#define TIMER_CHCTL2_CH0EN                              BIT(0)                                      /*!< channel 0 capture or compare function enable */
#define TIMER_CHCTL2_CH0P                               BIT(1)                                      /*!< channel 0 capture or compare function polarity */
#define TIMER_CHCTL2_CH0NEN                             BIT(2)                                      /*!< channel 0 complementary output enable */
#define TIMER_CHCTL2_CH0NP                              BIT(3)                                      /*!< channel 0 complementary output polarity */
#define TIMER_CHCTL2_CH1EN                              BIT(4)                                      /*!< channel 1 capture or compare function enable */
#define TIMER_CHCTL2_CH1P                               BIT(5)                                      /*!< channel 1 capture or compare function polarity */
#define TIMER_CHCTL2_CH1NEN                             BIT(6)                                      /*!< channel 1 complementary output enable */
#define TIMER_CHCTL2_CH1NP                              BIT(7)                                      /*!< channel 1 complementary output polarity */
#define TIMER_CHCTL2_CH2EN                              BIT(8)                                      /*!< channel 2 capture or compare function enable */
#define TIMER_CHCTL2_CH2P                               BIT(9)                                      /*!< channel 2 capture or compare function polarity */
#define TIMER_CHCTL2_CH2NEN                             BIT(10)                                     /*!< channel 2 complementary output enable */
#define TIMER_CHCTL2_CH2NP                              BIT(11)                                     /*!< channel 2 complementary output polarity */
#define TIMER_CHCTL2_CH3EN                              BIT(12)                                     /*!< channel 3 capture or compare function enable */
#define TIMER_CHCTL2_CH3P                               BIT(13)                                     /*!< channel 3 capture or compare function polarity */
#define TIMER_CHCTL2_CH3NEN                             BIT(14)                                     /*!< channel 3 complementary output enable */
#define TIMER_CHCTL2_CH3NP                              BIT(15)                                     /*!< channel 3 complementary output polarity */

#define TIMER_CHCTL2_MCH0EN                             BIT(2)                                      /*!< multi mode channel 0 capture or compare function enable */
#define TIMER_CHCTL2_MCH0P                              BIT(3)                                      /*!< multi mode channel 0 complementary output polarity */
#define TIMER_CHCTL2_MCH1EN                             BIT(6)                                      /*!< multi mode channel 1 capture or compare function enable */
#define TIMER_CHCTL2_MCH1P                              BIT(7)                                      /*!< multi mode channel 1 complementary output polarity */
#define TIMER_CHCTL2_MCH2EN                             BIT(10)                                     /*!< multi mode channel 2 capture or compare function enable */
#define TIMER_CHCTL2_MCH2P                              BIT(11)                                     /*!< multi mode channel 2 complementary output polarity */
#define TIMER_CHCTL2_MCH3EN                             BIT(14)                                     /*!< multi mode channel 3 capture or compare function enable */
#define TIMER_CHCTL2_MCH3P                              BIT(15)                                     /*!< multi mode channel 3 complementary output polarity */

/* TIMER_CNT */
#define TIMER_CNT_UPIFBU                                BIT(31)                                     /*!< UPIF bit backup */
#define TIMER_CNT_CNT                                   BITS(0,15)                                  /*!< 16 bit timer counter */

/* TIMER_PSC */
#define TIMER_PSC_PSC                                   BITS(0,15)                                  /*!< prescaler value of the counter clock */

/* TIMER_CAR */
#define TIMER_CAR_CARL                                  BITS(0,15)                                  /*!< 16 bit counter auto reload value */

/* TIMER_CARL */
#define TIMER_CARL_CARL                                 BITS(0,31)                                  /*!< 32 bit counter auto reload value */

/* TIMER_CARH */
#define TIMER_CARH_CARH                                 BITS(0,31)                                  /*!< 32 bit counter auto reload value */

/* TIMER_CREP0 */
#define TIMER_CREP0_CREP0                               BITS(0,7)                                   /*!< counter repetition value 0 */

/* TIMER_CH0CV */
#define TIMER_CH0CV_CH0VAL                              BITS(0,15)                                  /*!< 16 bit capture or compare value of channel 0 */

/* TIMER_CH1CV */
#define TIMER_CH1CV_CH1VAL                              BITS(0,15)                                  /*!< 16 bit capture or compare value of channel 1 */

/* TIMER_CH2CV */
#define TIMER_CH2CV_CH2VAL                              BITS(0,15)                                  /*!< 16 bit capture or compare value of channel 2 */

/* TIMER_CH3CV */
#define TIMER_CH3CV_CH3VAL                              BITS(0,15)                                  /*!< 16 bit capture or compare value of channel 3 */

/* TIMER_CCHP */
#define TIMER_CCHP_DTCFG                                BITS(0,7)                                   /*!< dead time configure */
#define TIMER_CCHP_PROT                                 BITS(8,9)                                   /*!< complementary register protect control */
#define TIMER_CCHP_IOS                                  BIT(10)                                     /*!< idle mode off-state configure */
#define TIMER_CCHP_ROS                                  BIT(11)                                     /*!< run mode off-state configure */
#define TIMER_CCHP_BRK0EN                               BIT(12)                                     /*!< BREAK0 input signal enable */
#define TIMER_CCHP_BRK0P                                BIT(13)                                     /*!< BREAK0 input signal polarity */
#define TIMER_CCHP_OAEN                                 BIT(14)                                     /*!< output automatic enable */
#define TIMER_CCHP_POEN                                 BIT(15)                                     /*!< primary output enable */
#define TIMER_CCHP_BRK0F                                BITS(16,19)                                 /*!< BREAK0 input signal filter */
#define TIMER_CCHP_BRK1F                                BITS(20,23)                                 /*!< BREAK1 input signal filter */
#define TIMER_CCHP_BRK1EN                               BIT(24)                                     /*!< BREAK1 input signal enable */
#define TIMER_CCHP_BRK1P                                BIT(25)                                     /*!< BREAK1 input signal polarity */
#define TIMER_CCHP_BRK0REL                              BIT(26)                                     /*!< BREAK0 input released */
#define TIMER_CCHP_BRK1REL                              BIT(27)                                     /*!< BREAK1 input released */
#define TIMER_CCHP_BRK0LK                               BIT(28)                                     /*!< BREAK0 input locked */
#define TIMER_CCHP_BRK1LK                               BIT(29)                                     /*!< BREAK1 input locked */

/* TIMER BREAK0/BREAK1 */
#define TIMER_BREAK0                                    ((uint16_t)0x0000U)                         /*!< TIMER BREAK0 */
#define TIMER_BREAK1                                    ((uint16_t)0x0001U)                         /*!< TIMER BREAK1 */

/* TIMER_MCHCTL0 */
/* output compare mode */
#define TIMER_MCHCTL0_MCH0MS                            (BITS(0,1) | BIT(30))                       /*!< multi mode channel 0 I/O mode selection */
#define TIMER_MCHCTL0_MCH0COMSEN                        BIT(3)                                      /*!< multi mode channel 0 output compare shadow enable */
#define TIMER_MCHCTL0_MCH0COMCTL                        (BITS(4,6) | BIT(16))                       /*!< multi mode channel 0 compare output control */
#define TIMER_MCHCTL0_MCH0COMCEN                        BIT(7)                                      /*!< multi mode channel 0 output compare clear enable */
#define TIMER_MCHCTL0_MCH1MS                            (BITS(8,9) | BIT(31))                       /*!< multi mode channel 1 I/O mode selection */
#define TIMER_MCHCTL0_MCH1COMSEN                        BIT(11)                                     /*!< multi mode channel 1 output compare shadow enable */
#define TIMER_MCHCTL0_MCH1COMCTL                        (BITS(12,14) | BIT(24))                     /*!< multi mode channel 1 compare output control */
#define TIMER_MCHCTL0_MCH1COMCEN                        BIT(15)                                     /*!< multi mode channel 1 output compare clear enable */

/* input capture mode */
#define TIMER_MCHCTL0_MCH0CAPPSC                        BITS(2,3)                                   /*!< multi mode channel 0 input capture prescaler */
#define TIMER_MCHCTL0_MCH0CAPFLT                        BITS(4,7)                                   /*!< multi mode channel 0 input capture filter control */
#define TIMER_MCHCTL0_MCH1CAPPSC                        BITS(10,11)                                 /*!< multi mode channel 1 input capture prescaler */
#define TIMER_MCHCTL0_MCH1CAPFLT                        BITS(12,15)                                 /*!< multi mode channel 1 input capture filter control */

/* TIMER_MCHCTL1 */
/* output compare mode */
#define TIMER_MCHCTL1_MCH2MS                            (BITS(0,1) | BIT(30))                       /*!< multi mode channel 2 I/O mode selection */
#define TIMER_MCHCTL1_MCH2COMSEN                        BIT(3)                                      /*!< multi mode channel 2 output compare shadow enable */
#define TIMER_MCHCTL1_MCH2COMCTL                        (BITS(4,6) | BIT(16))                       /*!< multi mode channel 2 compare output control */
#define TIMER_MCHCTL1_MCH2COMCEN                        BIT(7)                                      /*!< multi mode channel 2 output compare clear enable */
#define TIMER_MCHCTL1_MCH3MS                            (BITS(8,9) | BIT(31))                       /*!< multi mode channel 3 I/O mode selection */
#define TIMER_MCHCTL1_MCH3COMSEN                        BIT(11)                                     /*!< multi mode channel 3 output compare shadow enable */
#define TIMER_MCHCTL1_MCH3COMCTL                        (BITS(12,14) | BIT(24))                     /*!< multi mode channel 3 compare output control */
#define TIMER_MCHCTL1_MCH3COMCEN                        BIT(15)                                     /*!< multi mode channel 3 output compare clear enable */
/* input capture mode */
#define TIMER_MCHCTL1_MCH2CAPPSC                        BITS(2,3)                                   /*!< multi mode channel 2 input capture prescaler */
#define TIMER_MCHCTL1_MCH2CAPFLT                        BITS(4,7)                                   /*!< multi mode channel 2 input capture filter control */
#define TIMER_MCHCTL1_MCH3CAPPSC                        BITS(10,11)                                 /*!< multi mode channel 3 input capture prescaler */
#define TIMER_MCHCTL1_MCH3CAPFLT                        BITS(12,15)                                 /*!< multi mode channel 3 input capture filter control */

/* TIMER_MCHCTL2 */
#define TIMER_MCHCTL2_MCH0FP                            BITS(0,1)                                   /*!< multi mode channel 0 capture or compare function polarity */
#define TIMER_MCHCTL2_MCH1FP                            BITS(2,3)                                   /*!< multi mode channel 1 capture or compare function polarity */
#define TIMER_MCHCTL2_MCH2FP                            BITS(4,5)                                   /*!< multi mode channel 2 capture or compare function polarity */
#define TIMER_MCHCTL2_MCH3FP                            BITS(6,7)                                   /*!< multi mode channel 3 capture or compare function polarity */

/* TIMER_MCH0CV */
#define TIMER_MCH0CV_MCH0VAL                            BITS(0,15)                                  /*!< 16 bit capture or compare value of multi mode channel 0 */

/* TIMER_MCH1CV */
#define TIMER_MCH1CV_MCH1VAL                            BITS(0,15)                                  /*!< 16 bit capture or compare value of multi mode channel 1 */

/* TIMER_MCH2CV */
#define TIMER_MCH2CV_MCH2VAL                            BITS(0,15)                                  /*!< 16 bit capture or compare value of multi mode channel 2 */

/* TIMER_MCH3CV */
#define TIMER_MCH3CV_MCH3VAL                            BITS(0,15)                                  /*!< 16 bit capture or compare value of multi mode channel 3 */

/* TIMER_CH0COMV_ADD */
#define TIMER_CH0COMV_ADD_CH0COMVAL                     BITS(0,15)                                  /*!< additional compare value of channel 0 */

/* TIMER_CH1COMV_ADD */
#define TIMER_CH1COMV_ADD_CH1COMVAL                     BITS(0,15)                                  /*!< additional compare value of channel 1 */

/* TIMER_CH2COMV_ADD */
#define TIMER_CH2COMV_ADD_CH2COMVAL                     BITS(0,15)                                  /*!< additional compare value of channel 2 */

/* TIMER_CH3COMV_ADD */
#define TIMER_CH3COMV_ADD_CH3COMVAL                     BITS(0,15)                                  /*!< additional compare value of channel 3 */

/* TIMER_CTL2 */
#define TIMER_CTL2_DTIENCH0                             BIT(0)                                      /*!< dead time inserted enable for channel 0 and channel 0N */
#define TIMER_CTL2_DTIENCH1                             BIT(1)                                      /*!< dead time inserted enable for channel 1 and channel 1N */
#define TIMER_CTL2_DTIENCH2                             BIT(2)                                      /*!< dead time inserted enable for channel 2 and channel 2N */
#define TIMER_CTL2_DTIENCH3                             BIT(3)                                      /*!< dead time inserted enable for channel 3 and channel 3N */
#define TIMER_CTL2_BRKENCH0                             BIT(4)                                      /*!< break control enable for channel 0 and multi mode channel 0 */
#define TIMER_CTL2_BRKENCH1                             BIT(5)                                      /*!< break control enable for channel 1 and multi mode channel 1 */
#define TIMER_CTL2_BRKENCH2                             BIT(6)                                      /*!< break control enable for channel 2 and multi mode channel 2 */
#define TIMER_CTL2_BRKENCH3                             BIT(7)                                      /*!< break control enable for channel 3 and multi mode channel 3 */
#define TIMER_CTL2_CH0OMPSEL                            BITS(8,9)                                   /*!< channel 0 output match pulse select */
#define TIMER_CTL2_CH1OMPSEL                            BITS(10,11)                                 /*!< channel 1 output match pulse select */
#define TIMER_CTL2_CH2OMPSEL                            BITS(12,13)                                 /*!< channel 2 output match pulse select */
#define TIMER_CTL2_CH3OMPSEL                            BITS(14,15)                                 /*!< channel 3 output match pulse select */
#define TIMER_CTL2_DECJDEN                              BIT(18)                                     /*!< quadrature decoder signal jump (the two signals jump at the same time) detection enable */
#define TIMER_CTL2_DECDISDEN                            BIT(19)                                     /*!< quadrature decoder signal disconnection detection enable */
#define TIMER_CTL2_MCH0MSEL                             BITS(20,21)                                 /*!< multi mode channel 0 mode select */
#define TIMER_CTL2_MCH1MSEL                             BITS(22,23)                                 /*!< multi mode channel 1 mode select */
#define TIMER_CTL2_MCH2MSEL                             BITS(24,25)                                 /*!< multi mode channel 2 mode select */
#define TIMER_CTL2_MCH3MSEL                             BITS(26,27)                                 /*!< multi mode channel 3 mode select */
#define TIMER_CTL2_CH0CPWMEN                            BIT(28)                                     /*!< channel 0 composite PWM mode enable */
#define TIMER_CTL2_CH1CPWMEN                            BIT(29)                                     /*!< channel 1 composite PWM mode enable */
#define TIMER_CTL2_CH2CPWMEN                            BIT(30)                                     /*!< channel 2 composite PWM mode enable */
#define TIMER_CTL2_CH3CPWMEN                            BIT(31)                                     /*!< channel 3 composite PWM mode enable */

/* TIMER_FCCHP0 */
#define TIMER_FCCHP0_DTCFG                              BITS(0,7)                                   /*!< dead time configure */
#define TIMER_FCCHP0_IOS                                BIT(10)                                     /*!< idle mode off-state configure */
#define TIMER_FCCHP0_ROS                                BIT(11)                                     /*!< run mode off-state configure */
#define TIMER_FCCHP0_FCCHP0EN                           BIT(31)                                     /*!< free complementary channel protection register 0 enable */

/* TIMER_FCCHP1 */
#define TIMER_FCCHP1_DTCFG                              BITS(0,7)                                   /*!< dead time configure */
#define TIMER_FCCHP1_IOS                                BIT(10)                                     /*!< idle mode off-state configure */
#define TIMER_FCCHP1_ROS                                BIT(11)                                     /*!< run mode off-state configure */
#define TIMER_FCCHP1_FCCHP1EN                           BIT(31)                                     /*!< free complementary channel protection register 1 enable */

/* TIMER_FCCHP2 */
#define TIMER_FCCHP2_DTCFG                              BITS(0,7)                                   /*!< dead time configure */
#define TIMER_FCCHP2_IOS                                BIT(10)                                     /*!< idle mode off-state configure */
#define TIMER_FCCHP2_ROS                                BIT(11)                                     /*!< run mode off-state configure */
#define TIMER_FCCHP2_FCCHP2EN                           BIT(31)                                     /*!< free complementary channel protection register 2 enable */

/* TIMER_FCCHP3 */
#define TIMER_FCCHP3_DTCFG                              BITS(0,7)                                   /*!< dead time configure */
#define TIMER_FCCHP3_IOS                                BIT(10)                                     /*!< idle mode off-state configure */
#define TIMER_FCCHP3_ROS                                BIT(11)                                     /*!< run mode off-state configure */
#define TIMER_FCCHP3_FCCHP3EN                           BIT(31)                                     /*!< free complementary channel protection register 3 enable */

/* TIMER_AFCTL0 */
#define TIMER_AFCTL0_BRK0IN0EN                          BIT(0)                                      /*!< BREAK0 BRKIN0 alternate function input enable */
#define TIMER_AFCTL0_BRK0IN1EN                          BIT(1)                                      /*!< BREAK0 BRKIN1 alternate function input enable */
#define TIMER_AFCTL0_BRK0IN2EN                          BIT(2)                                      /*!< BREAK0 BRKIN2 alternate function input enable */
#define TIMER_AFCTL0_BRK0HPDFEN                         BIT(8)                                      /*!< BREAK0 HPDF input enable */
#define TIMER_AFCTL0_BRK0CMP0EN                         BIT(9)                                      /*!< BREAK0 CMP0 enable */
#define TIMER_AFCTL0_BRK0CMP1EN                         BIT(10)                                     /*!< BREAK0 CMP1 enable */
#define TIMER_AFCTL0_BRK0IN0P                           BIT(16)                                     /*!< BREAK0 BRKIN0 alternate function input polarity */
#define TIMER_AFCTL0_BRK0IN1P                           BIT(17)                                     /*!< BREAK0 BRKIN1 alternate function input polarity */
#define TIMER_AFCTL0_BRK0IN2P                           BIT(18)                                     /*!< BREAK0 BRKIN2 alternate function input polarity */
#define TIMER_AFCTL0_BRK0CMP0P                          BIT(25)                                     /*!< BREAK0 CMP0 input polarity */
#define TIMER_AFCTL0_BRK0CMP1P                          BIT(26)                                     /*!< BREAK0 CMP1 input polarity */

/* TIMER_AFCTL1 */
#define TIMER_AFCTL1_BRK1IN0EN                          BIT(0)                                      /*!< BREAK1 BRKIN0 alternate function input enable */
#define TIMER_AFCTL1_BRK1IN1EN                          BIT(1)                                      /*!< BREAK1 BRKIN1 alternate function input enable */
#define TIMER_AFCTL1_BRK1IN2EN                          BIT(2)                                      /*!< BREAK1 BRKIN2 alternate function input enable */
#define TIMER_AFCTL1_BRK1HPDFEN                         BIT(8)                                      /*!< BREAK1 HPDF input enable */
#define TIMER_AFCTL1_BRK1CMP0EN                         BIT(9)                                      /*!< BREAK1 CMP0 enable */
#define TIMER_AFCTL1_BRK1CMP1EN                         BIT(10)                                     /*!< BREAK1 CMP1 enable */
#define TIMER_AFCTL1_BRK1IN0P                           BIT(16)                                     /*!< BREAK1 BRKIN0 alternate function input polarity */
#define TIMER_AFCTL1_BRK1IN1P                           BIT(17)                                     /*!< BREAK1 BRKIN1 alternate function input polarity */
#define TIMER_AFCTL1_BRK1IN2P                           BIT(18)                                     /*!< BREAK1 BRKIN2 alternate function input polarity */
#define TIMER_AFCTL1_BRK1CMP0P                          BIT(25)                                     /*!< BREAK1 CMP0 input polarity */
#define TIMER_AFCTL1_BRK1CMP1P                          BIT(26)                                     /*!< BREAK1 CMP1 input polarity */

/* TIMER_WDGPER */
#define TIMER_WDGPER_WDGPER                             BITS(0,31)                                  /*!< watchdog counter period value */

/* TIMER_CREP1 */
#define TIMER_CREP1_CREP1                               BITS(0,31)                                  /*!< counter repetition value 1 */

/* TIMER_DMACFG */
#define TIMER_DMACFG_DMATA                              BITS(0,5)                                   /*!< DMA transfer access start address */
#define TIMER_DMACFG_DMATC                              BITS(8,13)                                  /*!< DMA transfer count */

/* TIMER_DMATB */
#define TIMER_DMATB_DMATB                               BITS(0,31)                                  /*!< DMA transfer buffer */

/* TIMER_CFG */
#define TIMER_CFG_OUTSEL                                BIT(0)                                      /*!< the output value selection */
#define TIMER_CFG_CHVSEL                                BIT(1)                                      /*!< write CHxVAL register selection */
#define TIMER_CFG_CREPSEL                               BIT(2)                                      /*!< the counter repetition register selection */
#define TIMER_CFG_CCUSEL                                BIT(3)                                      /*!< commutation control shadow register update select */

/* constants definitions */
/* DMA access base address */
#define DMACFG_DMATA(regval)                            (BITS(0,5) & ((uint32_t)(regval) << 0U))

/* DMA access burst length */
#define DMACFG_DMATC(regval)                            (BITS(8,13) & ((uint32_t)(regval) << 8U))

/* TIMER counter alignment mode */
#define CTL0_CAM(regval)                                ((uint16_t)(BITS(5,6) & ((uint32_t)(regval) << 5U)))

/* TIMER clock division */
#define CTL0_CKDIV(regval)                              ((uint16_t)(BITS(8,9) & ((uint32_t)(regval) << 8U)))

/* TIMER protect_mode */
#define CCHP_PROT(regval)                               ((uint16_t)(BITS(8,9) & ((uint32_t)(regval) << 8U)))

/* TIMER external trigger filter */
#define SMCFG_ETFC(regval)                              (BITS(8,11) & ((uint32_t)(regval) << 11U))

/* TIMER external trigger prescaler */
#define SMCFG_ETPSC(regval)                             (BITS(12,13) & ((uint32_t)(regval) << 12U))

/* TIMER interrupt enable or disable */
#define TIMER_INT_UP                                    TIMER_DMAINTEN_UPIE                         /*!< update interrupt */
#define TIMER_INT_CH0                                   TIMER_DMAINTEN_CH0IE                        /*!< channel 0 capture or compare interrupt */
#define TIMER_INT_CH1                                   TIMER_DMAINTEN_CH1IE                        /*!< channel 1 capture or compare interrupt */
#define TIMER_INT_CH2                                   TIMER_DMAINTEN_CH2IE                        /*!< channel 2 capture or compare interrupt */
#define TIMER_INT_CH3                                   TIMER_DMAINTEN_CH3IE                        /*!< channel 3 capture or compare interrupt */
#define TIMER_INT_CMT                                   TIMER_DMAINTEN_CMTIE                        /*!< channel commutation interrupt flag */
#define TIMER_INT_TRG                                   TIMER_DMAINTEN_TRGIE                        /*!< trigger interrupt */
#define TIMER_INT_BRK                                   TIMER_DMAINTEN_BRKIE                        /*!< break interrupt */
#define TIMER_INT_DECJ                                  TIMER_DMAINTEN_DECJIE                       /*!< quadrature decoder signal jump (the two signals jump at the same time) interrupt */
#define TIMER_INT_DECDIS                                TIMER_DMAINTEN_DECDISIE                     /*!< quadrature decoder signal disconnection interrupt */
#define TIMER_INT_MCH0                                  TIMER_DMAINTEN_MCH0IE                       /*!< multi mode channel 0 capture or compare interrupt */
#define TIMER_INT_MCH1                                  TIMER_DMAINTEN_MCH1IE                       /*!< multi mode channel 1 capture or compare interrupt */
#define TIMER_INT_MCH2                                  TIMER_DMAINTEN_MCH2IE                       /*!< multi mode channel 2 capture or compare interrupt */
#define TIMER_INT_MCH3                                  TIMER_DMAINTEN_MCH3IE                       /*!< multi mode channel 3 capture or compare interrupt */
#define TIMER_INT_CH0COMADD                             TIMER_DMAINTEN_CH0COMADDIE                  /*!< channel 0 additional compare interrupt */
#define TIMER_INT_CH1COMADD                             TIMER_DMAINTEN_CH1COMADDIE                  /*!< channel 1 additional compare interrupt */
#define TIMER_INT_CH2COMADD                             TIMER_DMAINTEN_CH2COMADDIE                  /*!< channel 2 additional compare interrupt */
#define TIMER_INT_CH3COMADD                             TIMER_DMAINTEN_CH3COMADDIE                  /*!< channel 3 additional compare interrupt */

/* TIMER interrupt flag */
#define TIMER_INT_FLAG_UP                               TIMER_INTF_UPIF                             /*!< update interrupt flag */
#define TIMER_INT_FLAG_CH0                              TIMER_INTF_CH0IF                            /*!< channel 0 capture or compare interrupt flag */
#define TIMER_INT_FLAG_CH1                              TIMER_INTF_CH1IF                            /*!< channel 1 capture or compare interrupt flag */
#define TIMER_INT_FLAG_CH2                              TIMER_INTF_CH2IF                            /*!< channel 2 capture or compare interrupt flag */
#define TIMER_INT_FLAG_CH3                              TIMER_INTF_CH3IF                            /*!< channel 3 capture or compare interrupt flag */
#define TIMER_INT_FLAG_CMT                              TIMER_INTF_CMTIF                            /*!< channel commutation interrupt flag */
#define TIMER_INT_FLAG_TRG                              TIMER_INTF_TRGIF                            /*!< trigger interrupt flag */
#define TIMER_INT_FLAG_BRK0                             TIMER_INTF_BRK0IF                           /*!< BREAK0 interrupt flag */
#define TIMER_INT_FLAG_BRK1                             TIMER_INTF_BRK1IF                           /*!< BREAK1 interrupt flag */
#define TIMER_INT_FLAG_SYSB                             TIMER_INTF_SYSBIF                           /*!< system source break interrupt flag */
#define TIMER_INT_FLAG_DECJ                             TIMER_INTF_DECJIF                           /*!< quadrature decoder signal jump (the two signals jump at the same time) interrupt flag */
#define TIMER_INT_FLAG_DECDIS                           TIMER_INTF_DECDISIF                         /*!< quadrature decoder signal disconnection interrupt flag */
#define TIMER_INT_FLAG_MCH0                             TIMER_INTF_MCH0IF                           /*!< multi mode channel 0 capture or compare interrupt flag */
#define TIMER_INT_FLAG_MCH1                             TIMER_INTF_MCH1IF                           /*!< multi mode channel 1 capture or compare interrupt flag */
#define TIMER_INT_FLAG_MCH2                             TIMER_INTF_MCH2IF                           /*!< multi mode channel 2 capture or compare interrupt flag */
#define TIMER_INT_FLAG_MCH3                             TIMER_INTF_MCH3IF                           /*!< multi mode channel 3 capture or compare interrupt flag */
#define TIMER_INT_FLAG_CH0COMADD                        TIMER_INTF_CH0COMADDIF                      /*!< channel 0 additional compare interrupt flag */
#define TIMER_INT_FLAG_CH1COMADD                        TIMER_INTF_CH1COMADDIF                      /*!< channel 1 additional compare interrupt flag */
#define TIMER_INT_FLAG_CH2COMADD                        TIMER_INTF_CH2COMADDIF                      /*!< channel 2 additional compare interrupt flag */
#define TIMER_INT_FLAG_CH3COMADD                        TIMER_INTF_CH3COMADDIF                      /*!< channel 3 additional compare interrupt flag */

/* TIMER flag */
#define TIMER_FLAG_UP                                   TIMER_INTF_UPIF                             /*!< update flag */
#define TIMER_FLAG_CH0                                  TIMER_INTF_CH0IF                            /*!< channel 0 capture or compare flag */
#define TIMER_FLAG_CH1                                  TIMER_INTF_CH1IF                            /*!< channel 1 capture or compare flag */
#define TIMER_FLAG_CH2                                  TIMER_INTF_CH2IF                            /*!< channel 2 capture or compare flag */
#define TIMER_FLAG_CH3                                  TIMER_INTF_CH3IF                            /*!< channel 3 capture or compare flag */
#define TIMER_FLAG_CMT                                  TIMER_INTF_CMTIF                            /*!< channel commutation flag */
#define TIMER_FLAG_TRG                                  TIMER_INTF_TRGIF                            /*!< trigger flag */
#define TIMER_FLAG_BRK0                                 TIMER_INTF_BRK0IF                           /*!< BREAK0 flag */
#define TIMER_FLAG_BRK1                                 TIMER_INTF_BRK1IF                           /*!< BREAK1 flag */
#define TIMER_FLAG_CH0O                                 TIMER_INTF_CH0OF                            /*!< channel 0 overcapture flag */
#define TIMER_FLAG_CH1O                                 TIMER_INTF_CH1OF                            /*!< channel 1 overcapture flag */
#define TIMER_FLAG_CH2O                                 TIMER_INTF_CH2OF                            /*!< channel 2 overcapture flag */
#define TIMER_FLAG_CH3O                                 TIMER_INTF_CH3OF                            /*!< channel 3 overcapture flag */
#define TIMER_FLAG_SYSB                                 TIMER_INTF_SYSBIF                           /*!< system source break flag */
#define TIMER_FLAG_DECJ                                 TIMER_INTF_DECJIF                           /*!< quadrature decoder signal jump (the two signals jump at the same time) flag */
#define TIMER_FLAG_DECDIS                               TIMER_INTF_DECDISIF                         /*!< quadrature decoder signal disconnection flag */
#define TIMER_FLAG_MCH0                                 TIMER_INTF_MCH0IF                           /*!< multi mode channel 0 capture or compare flag */
#define TIMER_FLAG_MCH1                                 TIMER_INTF_MCH1IF                           /*!< multi mode channel 1 capture or compare flag */
#define TIMER_FLAG_MCH2                                 TIMER_INTF_MCH2IF                           /*!< multi mode channel 2 capture or compare flag */
#define TIMER_FLAG_MCH3                                 TIMER_INTF_MCH3IF                           /*!< multi mode channel 3 capture or compare flag */
#define TIMER_FLAG_MCH0O                                TIMER_INTF_MCH0OF                           /*!< multi mode channel 0 overcapture flag */
#define TIMER_FLAG_MCH1O                                TIMER_INTF_MCH1OF                           /*!< multi mode channel 1 overcapture flag */
#define TIMER_FLAG_MCH2O                                TIMER_INTF_MCH2OF                           /*!< multi mode channel 2 overcapture flag */
#define TIMER_FLAG_MCH3O                                TIMER_INTF_MCH3OF                           /*!< multi mode channel 3 overcapture flag */
#define TIMER_FLAG_CH0COMADD                            TIMER_INTF_CH0COMADDIF                      /*!< channel 0 additional compare flag */
#define TIMER_FLAG_CH1COMADD                            TIMER_INTF_CH1COMADDIF                      /*!< channel 1 additional compare flag */
#define TIMER_FLAG_CH2COMADD                            TIMER_INTF_CH2COMADDIF                      /*!< channel 2 additional compare flag */
#define TIMER_FLAG_CH3COMADD                            TIMER_INTF_CH3COMADDIF                      /*!< channel 3 additional compare flag */

/* TIMER DMA source */
#define TIMER_DMA_UPD                                   TIMER_DMAINTEN_UPDEN                        /*!< update DMA request */
#define TIMER_DMA_CH0D                                  TIMER_DMAINTEN_CH0DEN                       /*!< channel 0 capture or compare DMA request */
#define TIMER_DMA_CH1D                                  TIMER_DMAINTEN_CH1DEN                       /*!< channel 1 capture or compare DMA request */
#define TIMER_DMA_CH2D                                  TIMER_DMAINTEN_CH2DEN                       /*!< channel 2 capture or compare DMA request */
#define TIMER_DMA_CH3D                                  TIMER_DMAINTEN_CH3DEN                       /*!< channel 3 capture or compare DMA request */
#define TIMER_DMA_CMTD                                  TIMER_DMAINTEN_CMTDEN                       /*!< commutation DMA request */
#define TIMER_DMA_TRGD                                  TIMER_DMAINTEN_TRGDEN                       /*!< trigger DMA request */
#define TIMER_DMA_MCH0D                                 TIMER_DMAINTEN_MCH0DEN                      /*!< multi mode channel 0 capture or compare DMA request */
#define TIMER_DMA_MCH1D                                 TIMER_DMAINTEN_MCH1DEN                      /*!< multi mode channel 1 capture or compare DMA request */
#define TIMER_DMA_MCH2D                                 TIMER_DMAINTEN_MCH2DEN                      /*!< multi mode channel 2 capture or compare DMA request */
#define TIMER_DMA_MCH3D                                 TIMER_DMAINTEN_MCH3DEN                      /*!< multi mode channel 3 capture or compare DMA request */

/* DMA device information index */
#define TIMER_DMA_ID_UP                                 ((uint8_t)0x0U)                             /*!< specify update DMA request */
#define TIMER_DMA_ID_CH0                                ((uint8_t)0x1U)                             /*!< specify channel 0 DMA request */
#define TIMER_DMA_ID_CH1                                ((uint8_t)0x2U)                             /*!< specify channel 1 DMA request */
#define TIMER_DMA_ID_CH2                                ((uint8_t)0x3U)                             /*!< specify channel 2 DMA request */
#define TIMER_DMA_ID_CH3                                ((uint8_t)0x4U)                             /*!< specify channel 3 DMA request */
#define TIMER_DMA_ID_CMT                                ((uint8_t)0x5U)                             /*!< specify commutation DMA request */
#define TIMER_DMA_ID_TRG                                ((uint8_t)0x6U)                             /*!< specify trigger DMA request */
#define TIMER_DMA_ID_MCH0                               ((uint8_t)0x7U)                             /*!< specify multi mode channel 0 DMA request */
#define TIMER_DMA_ID_MCH1                               ((uint8_t)0x8U)                             /*!< specify multi mode channel 1 DMA request */
#define TIMER_DMA_ID_MCH2                               ((uint8_t)0x9U)                             /*!< specify multi mode channel 2 DMA request */
#define TIMER_DMA_ID_MCH3                               ((uint8_t)0xAU)                             /*!< specify multi mode channel 3 DMA request */

/* channel DMA request source selection */
#define TIMER_DMAREQUEST_UPDATEEVENT                    TIMER_CTL1_DMAS                             /*!< DMA request of channel n is sent when update event occurs */
#define TIMER_DMAREQUEST_CHANNELEVENT                   ((uint32_t)0x00000000U)                     /*!< DMA request of channel n is sent when channel n event occurs */

/* TIMER software event generation source */
#define TIMER_EVENT_SRC_UPG                             TIMER_SWEVG_UPG                             /*!< update event generation */
#define TIMER_EVENT_SRC_CH0G                            TIMER_SWEVG_CH0G                            /*!< channel 0 capture or compare event generation */
#define TIMER_EVENT_SRC_CH1G                            TIMER_SWEVG_CH1G                            /*!< channel 1 capture or compare event generation */
#define TIMER_EVENT_SRC_CH2G                            TIMER_SWEVG_CH2G                            /*!< channel 2 capture or compare event generation */
#define TIMER_EVENT_SRC_CH3G                            TIMER_SWEVG_CH3G                            /*!< channel 3 capture or compare event generation */
#define TIMER_EVENT_SRC_CMTG                            TIMER_SWEVG_CMTG                            /*!< channel commutation event generation */
#define TIMER_EVENT_SRC_TRGG                            TIMER_SWEVG_TRGG                            /*!< trigger event generation */
#define TIMER_EVENT_SRC_BRK0G                           TIMER_SWEVG_BRK0G                           /*!< BREAK0 event generation */
#define TIMER_EVENT_SRC_BRK1G                           TIMER_SWEVG_BRK1G                           /*!< BREAK1 event generation */
#define TIMER_EVENT_SRC_MCH0G                           TIMER_SWEVG_MCH0G                           /*!< multi mode channel 0 capture or compare event generation */
#define TIMER_EVENT_SRC_MCH1G                           TIMER_SWEVG_MCH1G                           /*!< multi mode channel 1 capture or compare event generation */
#define TIMER_EVENT_SRC_MCH2G                           TIMER_SWEVG_MCH2G                           /*!< multi mode channel 2 capture or compare event generation */
#define TIMER_EVENT_SRC_MCH3G                           TIMER_SWEVG_MCH3G                           /*!< multi mode channel 3 capture or compare event generation */
#define TIMER_EVENT_SRC_CH0COMADDG                      TIMER_SWEVG_CH0COMADDG                      /*!< channel 0 additional compare event generation */
#define TIMER_EVENT_SRC_CH1COMADDG                      TIMER_SWEVG_CH1COMADDG                      /*!< channel 1 additional compare event generation */
#define TIMER_EVENT_SRC_CH2COMADDG                      TIMER_SWEVG_CH2COMADDG                      /*!< channel 2 additional compare event generation */
#define TIMER_EVENT_SRC_CH3COMADDG                      TIMER_SWEVG_CH3COMADDG                      /*!< channel 3 additional compare event generation */

/* channel mask */
#define TIMER_CHX_EN_MASK                               ((uint32_t)(TIMER_CHCTL2_CH0EN | TIMER_CHCTL2_CH1EN | TIMER_CHCTL2_CH2EN | TIMER_CHCTL2_CH3EN | \
                                                                    TIMER_CHCTL2_MCH0EN | TIMER_CHCTL2_MCH1EN | TIMER_CHCTL2_MCH2EN | TIMER_CHCTL2_MCH3EN))
                                                                                                    /*!< channel enable mask */
#define TIMER_CHNX_EN_MASK                              ((uint32_t)(TIMER_CHCTL2_CH0NEN | TIMER_CHCTL2_CH1NEN| TIMER_CHCTL2_CH2NEN))
                                                                                                    /*!< complementary channel enable mask */
/* registers address definitions */
#define TIMER_CTL0_ADDRESS(timerx)                      (uint32_t)((timerx) + 0x00U)                /*!< TIMER control register 0 */
#define TIMER_CTL1_ADDRESS(timerx)                      (uint32_t)((timerx) + 0x04U)                /*!< TIMER control register 1 */
#define TIMER_SMCFG_ADDRESS(timerx)                     (uint32_t)((timerx) + 0x08U)                /*!< TIMER slave mode configuration register */
#define TIMER_DMAINTEN_ADDRESS(timerx)                  (uint32_t)((timerx) + 0x0CU)                /*!< TIMER DMA and interrupt enable register */
#define TIMER_INTF_ADDRESS(timerx)                      (uint32_t)((timerx) + 0x10U)                /*!< TIMER interrupt flag register */
#define TIMER_SWEVG_ADDRESS(timerx)                     (uint32_t)((timerx) + 0x14U)                /*!< TIMER software event generation register */
#define TIMER_CHCTL0_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x18U)                /*!< TIMER channel control register 0 */
#define TIMER_CHCTL1_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x1CU)                /*!< TIMER channel control register 1 */
#define TIMER_CHCTL2_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x20U)                /*!< TIMER channel control register 2 */
#define TIMER_CNT_ADDRESS(timerx)                       (uint32_t)((timerx) + 0x24U)                /*!< TIMER counter register */
#define TIMER_PSC_ADDRESS(timerx)                       (uint32_t)((timerx) + 0x28U)                /*!< TIMER prescaler register */
#define TIMER_CAR_ADDRESS(timerx)                       (uint32_t)((timerx) + 0x2CU)                /*!< TIMER counter auto reload register */
#define TIMER_CREP_ADDRESS(timerx)                      (uint32_t)((timerx) + 0x30U)                /*!< TIMER counter repetition register */
#define TIMER_CH0CV_ADDRESS(timerx)                     (uint32_t)((timerx) + 0x34U)                /*!< TIMER channel 0 capture/compare value register */
#define TIMER_CH1CV_ADDRESS(timerx)                     (uint32_t)((timerx) + 0x38U)                /*!< TIMER channel 1 capture/compare value register */
#define TIMER_CH2CV_ADDRESS(timerx)                     (uint32_t)((timerx) + 0x3CU)                /*!< TIMER channel 2 capture/compare value register */
#define TIMER_CH3CV_ADDRESS(timerx)                     (uint32_t)((timerx) + 0x40U)                /*!< TIMER channel 3 capture/compare value register */
#define TIMER_CCHP_ADDRESS(timerx)                      (uint32_t)((timerx) + 0x44U)                /*!< TIMER complementary channel protection register */
#define TIMER_MCHCTL0_ADDRESS(timerx)                   (uint32_t)((timerx) + 0x48U)                /*!< TIMER multi-mode channel control register 0 */
#define TIMER_MCHCTL1_ADDRESS(timerx)                   (uint32_t)((timerx) + 0x4CU)                /*!< TIMER multi-mode channel control register 1 */
#define TIMER_MCHCTL2_ADDRESS(timerx)                   (uint32_t)((timerx) + 0x50U)                /*!< TIMER multi-mode channel control register 2 */
#define TIMER_MCH0CV_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x54U)                /*!< TIMER multi-mode channel 0 capture/compare value register */
#define TIMER_MCH1CV_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x58U)                /*!< TIMER multi-mode channel 1 capture/compare value register */
#define TIMER_MCH2CV_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x5CU)                /*!< TIMER multi-mode channel 2 capture/compare value register */
#define TIMER_MCH3CV_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x60U)                /*!< TIMER multi-mode channel 3 capture/compare value register */
#define TIMER_CH0COMV_ADDRESS(timerx)                   (uint32_t)((timerx) + 0x64U)                /*!< TIMER channel 0 additional compare value register */
#define TIMER_CH1COMV_ADDRESS(timerx)                   (uint32_t)((timerx) + 0x68U)                /*!< TIMER channel 1 additional compare value register */
#define TIMER_CH2COMV_ADDRESS(timerx)                   (uint32_t)((timerx) + 0x6CU)                /*!< TIMER channel 2 additional compare value register */
#define TIMER_CH3COMV_ADDRESS(timerx)                   (uint32_t)((timerx) + 0x70U)                /*!< TIMER channel 3 additional compare value register */
#define TIMER_CTL2_ADDRESS(timerx)                      (uint32_t)((timerx) + 0x74U)                /*!< TIMER control register 2 */
#define TIMER_FCCHP0_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x7CU)                /*!< TIMER free complementary channel protection register 0 */
#define TIMER_FCCHP1_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x80U)                /*!< TIMER free complementary channel protection register 1 */
#define TIMER_FCCHP2_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x84U)                /*!< TIMER free complementary channel protection register 2 */
#define TIMER_FCCHP3_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x88U)                /*!< TIMER free complementary channel protection register 3 */
#define TIMER_AFCTL0_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x8CU)                /*!< TIMER alternate function control register 0 */
#define TIMER_AFCTL1_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x90U)                /*!< TIMER alternate function control register 1 */
#define TIMER_WDGPER_ADDRESS(timerx)                    (uint32_t)((timerx) + 0x94U)                /*!< TIMER watchdog counter period register */
#define TIMER_CREP1_ADDRESS(timerx)                     (uint32_t)((timerx) + 0x98U)                /*!< TIMER counter repetition register 1 */
#define TIMER_CNTH_ADDRESS(timerx)                      (uint32_t)((timerx) + 0xD0U)                /*!< TIMER counter high register (only for TIMERx, x=50,51) */
#define TIMER_CARH_ADDRESS(timerx)                      (uint32_t)((timerx) + 0xD4U)                /*!< TIMER counter auto reload high register (only for TIMERx, x=50,51) */
#define TIMER_DMACFG_ADDRESS(timerx)                    (uint32_t)((timerx) + 0xE0U)                /*!< TIMER DMA configuration register */
#define TIMER_DMATB_ADDRESS(timerx)                     (uint32_t)((timerx) + 0xE4U)                /*!< TIMER DMA transfer buffer register */
#define TIMER_CFG_ADDRESS(timerx)                       (uint32_t)((timerx) + 0xFCU)                /*!< TIMER configuration register */

/* TIMER prescaler reload mode */
#define TIMER_PSC_RELOAD_NOW                            TIMER_SWEVG_UPG                             /*!< the prescaler is loaded right now */
#define TIMER_PSC_RELOAD_UPDATE                         ((uint32_t)0x00000000U)                     /*!< the prescaler is loaded at the next update event */

/* master mode control 0 */
#define CTL1_MMC0(regval)                               (BITS(4,6) & ((uint32_t)(regval) << 4U))

/* master mode control 1 */
#define CTL1_MMC1(regval)                               (BITS(20,22) & ((uint32_t)(regval) << 20U))

/* BREAK input filter */
#define CCHP_BRK0F(regval)                              (BITS(16,19) & ((uint32_t)(regval) << 16U))
#define CCHP_BRK1F(regval)                              (BITS(20,23) & ((uint32_t)(regval) << 20U))

/* auto reload state */
#define TIMER_UPIF_BACKUP_ENABLE                        TIMER_CTL0_UPIFBUEN                         /*!< upif backup enable */
#define TIMER_UPIF_BACKUP_DISABLE                       ((uint32_t)0x00000000U)                     /*!< upif backup disable */

/* update source */
#define TIMER_UPDATE_SRC_REGULAR                        TIMER_CTL0_UPS                              /*!< update generate only by counter overflow/underflow */
#define TIMER_UPDATE_SRC_GLOBAL                         ((uint32_t)0x00000000U)                     /*!< update generate by setting of UPG bit or the counter overflow/underflow,or the slave mode controller trigger */

/* TIMER channel */
#define TIMER_CH_0_1                                    ((uint16_t)0x0004U)                         /*!< TIMER channel 0 and 1 for decoder function parameter */

/* @DEFINE:TIMER break0 input enable */
#define TIMER_BRK0IN0                                   TIMER_AFCTL0_BRK0IN0EN                      /*!< BREAKx BRK0IN0 alternate function input enable */
#define TIMER_BRK0IN1                                   TIMER_AFCTL0_BRK0IN1EN                      /*!< BREAKx BRK0IN1 alternate function input enable */
#define TIMER_BRK0IN2                                   TIMER_AFCTL0_BRK0IN2EN                      /*!< BREAKx BRK0IN2 alternate function input enable */
#define TIMER_BRK0CMP0                                  TIMER_AFCTL0_BRK0CMP0EN                     /*!< BREAKx CMP0 input enable */
#define TIMER_BRK0CMP1                                  TIMER_AFCTL0_BRK0CMP1EN                     /*!< BREAKx CMP1 input enable */
#define TIMER_BRK0HPDF                                  TIMER_AFCTL0_BRK0HPDFEN                     /*!< BREAKx HPDF input enable */

/* @DEFINE:TIMER break1 input enable */
#define TIMER_BRK1IN0                                   TIMER_AFCTL1_BRK1IN0EN                      /*!< BREAKx BRK1IN0 alternate function input enable */
#define TIMER_BRK1IN1                                   TIMER_AFCTL1_BRK1IN1EN                      /*!< BREAKx BRK1IN1 alternate function input enable */
#define TIMER_BRK1IN2                                   TIMER_AFCTL1_BRK1IN2EN                      /*!< BREAKx BRK1IN2 alternate function input enable */
#define TIMER_BRK1CMP0                                  TIMER_AFCTL1_BRK0CMP0EN                     /*!< BREAKx CMP0 input enable */
#define TIMER_BRK1CMP1                                  TIMER_AFCTL1_BRK0CMP1EN                     /*!< BREAKx CMP1 input enable */
#define TIMER_BRK1HPDF                                  TIMER_AFCTL1_BRK0HPDFEN                     /*!< BREAKx HPDF input enable */

/* @DEFINE:TIMER external break0 input polarity */
#define TIMER_BRK0IN0P                                  TIMER_AFCTL0_BRK0IN0P                       /*!< BREAKx BRK0IN0 polarity */
#define TIMER_BRK0IN1P                                  TIMER_AFCTL0_BRK0IN1P                       /*!< BREAKx BRK0IN1 polarity */
#define TIMER_BRK0IN2P                                  TIMER_AFCTL0_BRK0IN2P                       /*!< BREAKx BRK0IN2 polarity */
#define TIMER_BRK0CMP0P                                 TIMER_AFCTL0_BRK0CMP0P                      /*!< BREAKx CMP0 polarity */
#define TIMER_BRK0CMP1P                                 TIMER_AFCTL0_BRK0CMP1P                      /*!< BREAKx CMP1 polarity */

/* @DEFINE:TIMER external break1 input polarity */
#define TIMER_BRK1IN0P                                  TIMER_AFCTL1_BRK1IN0P                       /*!< BREAKx BRK0IN0 polarity */
#define TIMER_BRK1IN1P                                  TIMER_AFCTL1_BRK1IN1P                       /*!< BREAKx BRK0IN1 polarity */
#define TIMER_BRK1IN2P                                  TIMER_AFCTL1_BRK1IN2P                       /*!< BREAKx BRK0IN2 polarity */
#define TIMER_BRK1CMP0P                                 TIMER_AFCTL1_BRK1CMP0P                      /*!< BREAKx CMP0 polarity */
#define TIMER_BRK1CMP1P                                 TIMER_AFCTL1_BRK1CMP1P                      /*!< BREAKx CMP1 polarity */

/* @DEFINE:multi mode channel output polarity */
#define TIMER_OMC_POLARITY_HIGH                         ((uint16_t)0x0000U)                         /*!< multi mode channel output polarity is high */
#define TIMER_OMC_POLARITY_LOW                          ((uint16_t)0x0001U)                         /*!< multi mode channel output polarity is low */

/* @ENUM:channel control shadow register update control */
#define CTL1_CCUC(regval)                               ((BIT(2) & ((uint32_t)(regval) << 2U)) | \
                                                        (BITS(30,31) & ((uint32_t)(regval) << 29U)))

#define CTL0_CH0CAPFLT(regval)                          (BITS(0,3) & ((uint32_t)(regval) << 0U))

/* TIMER write CHxVAL register selection */
#define TIMER_CHVSEL_ENABLE                             ((uint16_t)TIMER_CFG_CHVSEL)                /*!< write CHxVAL register selection enable */
#define TIMER_CHVSEL_DISABLE                            ((uint16_t)0x0000U)                         /*!< write CHxVAL register selection disable */

/* TIMER output value selection enable */
#define TIMER_OUTSEL_ENABLE                             ((uint16_t)TIMER_CFG_OUTSEL)                /*!< output value selection enable */
#define TIMER_OUTSEL_DISABLE                            ((uint16_t)0x0000U)                         /*!< output value selection disable */

/* TIMER counter repetition register selection */
#define TIMER_CREP1_ENABLE                              ((uint16_t)TIMER_CFG_CREPSEL)               /*!< the update event rate is depended to TIMERx_CREP1 register */
#define TIMER_CREP0_ENABLE                              ((uint16_t)0x0000U)                         /*!< the update event rate is depended to TIMERx_CREP0 register */

/* TIMER output value selection */
#define TIMER_PULSE_OUTPUT_NORMAL                       ((uint16_t)0x0000U)                         /*!< channel output normal */
#define TIMER_PULSE_OUTPUT_CNT_UP                       ((uint16_t)0x0001U)                         /*!< pulse output only when counting up */
#define TIMER_PULSE_OUTPUT_CNT_DOWN                     ((uint16_t)0x0002U)                         /*!< pulse output only when counting down */
#define TIMER_PULSE_OUTPUT_CNT_BOTH                     ((uint16_t)0x0003U)                         /*!< pulse output when counting up or down */
#define TIMER_PULSE_OUTPUT_MASK                         ((uint16_t)0x0003U)                         /*!< pulse output mode mask */

/* multi mode channel input capture polarity */
#define TIMER_IMC_POLARITY_RISING                       ((uint16_t)0x0000U)                         /*!< multi mode channel input capture rising edge */
#define TIMER_IMC_POLARITY_FALLING                      ((uint16_t)0x0001U)                         /*!< multi mode channel input capture falling edge */
#define TIMER_IMC_POLARITY_BOTH_EDGE                    ((uint16_t)0x0003U)                         /*!< multi mode channel input capture both edge */

/* channel output polarity */
#define TIMER_OC_MCH_POLARITY_HIGH                      ((uint16_t)0x0000U)                         /*!< channel output polarity is high */
#define TIMER_OC_MCH_POLARITY_LOW                       ((uint16_t)0x0001U)                         /*!< channel output polarity is low */

/* TIMER commutation control shadow register update selection enable */
#define TIMER_CCUSEL_ENABLE                             ((uint16_t)TIMER_CFG_CCUSEL)                /*!< commutation control shadow register update selection enable */
#define TIMER_CCUSEL_DISABLE                            ((uint16_t)0x0000U)                         /*!< commutation control shadow register update selection disable */

/* TIMER timebase initialize */
/* @STRUCT_MEMBER: prescaler */
/* @=NULL */
/* @STRUCT_MEMBER: alignedmode */
/* @DEFINE: counter alignment mode */
#define TIMER_COUNTER_EDGE                              CTL0_CAM(0)                                 /*!< edge-aligned mode */
#define TIMER_COUNTER_CENTER_DOWN                       CTL0_CAM(1)                                 /*!< center-aligned and counting down assert mode */
#define TIMER_COUNTER_CENTER_UP                         CTL0_CAM(2)                                 /*!< center-aligned and counting up assert mode */
#define TIMER_COUNTER_CENTER_BOTH                       CTL0_CAM(3)                                 /*!< center-aligned and counting up/down assert mode */

/* @STRUCT_MEMBER: counter_direction */
/* @DEFINE: counter direction */
#define TIMER_COUNTER_UP                                ((uint16_t)0x0000U)                         /*!< counter up direction */
#define TIMER_COUNTER_DOWN                              ((uint16_t)0x0010U)                         /*!< counter down direction */

/* @STRUCT_MEMBER: period */
/* @=NULL */
/* @STRUCT_MEMBER: clock_division */
/* @DEFINE: specify division ratio between TIMER clock and dead-time and sampling clock */
#define TIMER_CKDIV_DIV1                                CTL0_CKDIV(0)                               /*!< clock division value is 1, fDTS = fTIMER_CK */
#define TIMER_CKDIV_DIV2                                CTL0_CKDIV(1)                               /*!< clock division value is 2, fDTS = fTIMER_CK/2 */
#define TIMER_CKDIV_DIV4                                CTL0_CKDIV(2)                               /*!< clock division value is 4, fDTS = fTIMER_CK/4 */

/* @STRUCT_MEMBER: repetition_counter */
/* @=NULL */
/* @STRUCT_MEMBER: auto_reload_state */
/* @DEFINE:  auto reload state */
#define TIMER_AUTO_RELOAD_ENABLE                        TIMER_CTL0_ARSE                             /*!< counter auto reload enable */
#define TIMER_AUTO_RELOAD_DISABLE                       ((uint32_t)0x00000000U)                     /*!< counter auto reload disable */

/* @STRUCT_MEMBER: master_slave_mode */
/* @SP: ENABLE/DISABLE */
/* @STRUCT_MEMBER:repetition_crep_selection */
/* @DEFINE: crep selection */
#define TIMER_REPETITION_0                              ((uint32_t)0x00000000U)                     /*!< selection crep0 */
#define TIMER_REPETITION_1                              ((uint32_t)0x00000001U)                     /*!< selection crep1 */

/* TIMER input capture mode configuration */
/* @PARA:channel */
/* @DEFINE:TIMER channel n/Multi-Mode Channel n (n=0,1,2,3) */
#define TIMER_CH_0                                      ((uint16_t)0x0000U)                         /*!< TIMER channel 0 */
#define TIMER_CH_1                                      ((uint16_t)0x0001U)                         /*!< TIMER channel 1 */
#define TIMER_CH_2                                      ((uint16_t)0x0002U)                         /*!< TIMER channel 2 */
#define TIMER_CH_3                                      ((uint16_t)0x0003U)                         /*!< TIMER channel 3 */
#define TIMER_MCH_0                                     ((uint16_t)0x0010U)                         /*!< TIMER multi mode channel 0 */
#define TIMER_MCH_1                                     ((uint16_t)0x0011U)                         /*!< TIMER multi mode channel 1 */
#define TIMER_MCH_2                                     ((uint16_t)0x0012U)                         /*!< TIMER multi mode channel 2 */
#define TIMER_MCH_3                                     ((uint16_t)0x0013U)                         /*!< TIMER multi mode channel 3 */

/* @STRUCT_MEMBER: ic_polarity */
/* @DEFINE:channel input capture polarity */
#define TIMER_IC_POLARITY_RISING                        ((uint16_t)0x0000U)                         /*!< input capture rising edge */
#define TIMER_IC_POLARITY_FALLING                       ((uint16_t)0x0002U)                         /*!< input capture falling edge */
#define TIMER_IC_POLARITY_BOTH_EDGE                     ((uint16_t)0x000AU)                         /*!< input capture both edge */

/* @STRUCT_MEMBER: ic_selection */
/* @DEFINE:TIMER input capture selection */
#define TIMER_IC_SELECTION_DIRECTTI                     ((uint16_t)0x0001U)                         /*!< channel n is configured as input and icy is mapped on CIy/CINy */
#define TIMER_IC_SELECTION_INDIRECTTI                   ((uint16_t)0x0002U)                         /*!< channel n is configured as input and icy is mapped on opposite input */
#define TIMER_IC_SELECTION_ITS                          ((uint16_t)0x0003U)                         /*!< channel n is configured as input and icy is mapped on ITS */
#define TIMER_IC_SELECTION_PAIR                         ((uint16_t)0x0004U)                         /*!< channel n is configured as input and icy is mapped on the other channel of same pair */

/* @STRUCT_MEMBER: ic_prescaler */
/* @DEFINE:channel input capture prescaler */
#define TIMER_IC_PSC_DIV1                               ((uint16_t)0x0000U)                         /*!< no prescaler */
#define TIMER_IC_PSC_DIV2                               ((uint16_t)0x0004U)                         /*!< divided by 2 */
#define TIMER_IC_PSC_DIV4                               ((uint16_t)0x0008U)                         /*!< divided by 4 */
#define TIMER_IC_PSC_DIV8                               ((uint16_t)0x000CU)                         /*!< divided by 8 */

/* channel enable state */
#define TIMER_CCX_ENABLE                                ((uint16_t)0x0001U)                         /*!< channel enable */
#define TIMER_CCX_DISABLE                               ((uint16_t)0x0000U)                         /*!< channel disable */

/* channel complementary output enable state */
#define TIMER_CCXN_ENABLE                               ((uint16_t)0x0005U)                         /*!< channel complementary enable */
#define TIMER_CCXN_DISABLE                              ((uint16_t)0x0000U)                         /*!< channel complementary disable */

/* @STRUCT_MEMBER: oc_outputmode */
/* @DEFINE:  TIMER multi mode channel mode selection */
#define TIMER_MCH_MODE_INDEPENDENTLY                    ((uint16_t)0x0000U)                         /*!< multi mode channel work in independently mode */
#define TIMER_MCH_MODE_COMPLEMENTARY                    ((uint16_t)0x0003U)                         /*!< multi mode channel work in complementary output mode */
#define TIMER_MCH_MODE_MASK                             ((uint16_t)0x0003U)                         /*!< multi mode channel mode mask */

/* @STRUCT_MEMBER: oc_pulsevalue */
/* @DEFINE: NULL */
/* @STRUCT_MEMBER: oc_polarity */
/* @DEFINE: channel output polarity */
#define TIMER_OC_POLARITY_HIGH                          ((uint16_t)0x0000U)                         /*!< channel output polarity is high */
#define TIMER_OC_POLARITY_LOW                           ((uint16_t)0x0002U)                         /*!< channel output polarity is low */

/* @STRUCT_MEMBER: oc_npolarity */
/* @DEFINE: channel complementary output polarity */
#define TIMER_OCN_POLARITY_HIGH                         ((uint16_t)0x0000U)                         /*!< channel complementary output polarity is high */
#define TIMER_OCN_POLARITY_LOW                          ((uint16_t)0x0008U)                         /*!< channel complementary output polarity is low */

/* @STRUCT_MEMBER: oc_idlestate */
/* @DEFINE: idle state of channel output */
#define TIMER_OC_IDLE_STATE_HIGH                        ((uint16_t)0x0100U)                         /*!< idle state of channel output is high */
#define TIMER_OC_IDLE_STATE_LOW                         ((uint16_t)0x0000U)                         /*!< idle state of channel output is low */

/* @STRUCT_MEMBER: oc_nidlestate */
/* @DEFINE: idle state of channel complementary output */
#define TIMER_OCN_IDLE_STATE_HIGH                       ((uint16_t)0x0200U)                         /*!< idle state of channel complementary output is high */
#define TIMER_OCN_IDLE_STATE_LOW                        ((uint16_t)0x0000U)                         /*!< idle state of channel complementary output is low */

/* @STRUCT_MEMBER:oc_deadtime_insert */
/* @SP: ENABLE/DISABLE */
/* @STRUCT_MEMBER: oc_shadow */
/* @DEFINE: channel output compare shadow enable */
#define TIMER_OC_SHADOW_ENABLE                          ((uint16_t)0x0008U)                         /*!< channel output compare shadow enable */
#define TIMER_OC_SHADOW_DISABLE                         ((uint16_t)0x0000U)                         /*!< channel output compare shadow disable */

/* @STRUCT_MEMBER: oc_clearmode */
/* @DEFINE:channel output compare clear enable */
#define TIMER_OC_CLEAR_ENABLE                           ((uint16_t)0x0080U)                         /*!< channel output clear function enable */
#define TIMER_OC_CLEAR_DISABLE                          ((uint16_t)0x0000U)                         /*!< channel output clear function disable */

/* channel free complementary parameter struct definitions */
/* @STRUCT_MEMBER: freecomstate */
/* @DEFINE: free complementary channel protection configure */
#define TIMER_FCCHP_STATE_ENABLE                        ((uint32_t)0x80000000U)                     /*!< free complementary channel protection enable */
#define TIMER_FCCHP_STATE_DISABLE                       ((uint32_t)0x00000000U)                     /*!< free complementary channel protection disable */

/* BREAK and complementary channel protection configuration */
/* @STRUCT_MEMBER: run_offstate */
/* @DEFINE: run mode off-state configure */
#define TIMER_ROS_STATE_ENABLE                          ((uint32_t)0x00000800U)                     /*!< when POEN bit is set, the channel output signals(CHx_O/MCHx_O) are enabled, with relationship to CHxEN/CHxNEN bits */
#define TIMER_ROS_STATE_DISABLE                         ((uint32_t)0x00000000U)                     /*!< when POEN bit is set, the channel output signals(CHx_O/MCHx_O) are disabled */

/* @STRUCT_MEMBER: idle_offstate */
/* @DEFINE: idle mode off-state configure */
#define TIMER_IOS_STATE_ENABLE                          ((uint32_t)0x00000400U)                     /*!< when POEN bit is reset, the channel output signals(CHx_O/MCHx_O) are enabled, with relationship to CHxEN/CHxNEN bits */
#define TIMER_IOS_STATE_DISABLE                         ((uint32_t)0x00000000U)                     /*!< when POEN bit is reset, the channel output signals(CHx_O/MCHx_O) are disabled */

/* @STRUCT_MEMBER: deadtime */
/* @DEFINE: NULL */
/* @STRUCT_MEMBER: output_autostate */
/* @DEFINE: output automatic enable */
#define TIMER_OUTAUTO_ENABLE                            ((uint32_t)TIMER_CCHP_OAEN)                 /*!< output automatic enable */
#define TIMER_OUTAUTO_DISABLE                           ((uint32_t)0x00000000U)                     /*!< output automatic disable */

/* @STRUCT_MEMBER: protectmode */
/* @DEFINE:complementary register protect control */
#define TIMER_CCHP_PROT_OFF                             CCHP_PROT(0)                                /*!< protect disable */
#define TIMER_CCHP_PROT_0                               CCHP_PROT(1)                                /*!< PROT mode 0 */
#define TIMER_CCHP_PROT_1                               CCHP_PROT(2)                                /*!< PROT mode 1 */
#define TIMER_CCHP_PROT_2                               CCHP_PROT(3)                                /*!< PROT mode 2 */

/* @STRUCT_MEMBER: break0_state */
/* @DEFINE: BREAK0 input signal enable */
#define TIMER_BREAK0_ENABLE                             ((uint32_t)TIMER_CCHP_BRK0EN)               /*!< BREAK0 input signal enable */
#define TIMER_BREAK0_DISABLE                            ((uint32_t)0x00000000U)                     /*!< BREAK0 input signal disable */

/* @STRUCT_MEMBER: break0_polarity */
/* @DEFINE: BREAK0 input signal polarity */
#define TIMER_BREAK0_POLARITY_LOW                       ((uint32_t)0x00000000U)                     /*!< BREAK0 input signal polarity is low */
#define TIMER_BREAK0_POLARITY_HIGH                      ((uint32_t)TIMER_CCHP_BRK0P)                /*!< BREAK0 input signal polarity is high */

/* @STRUCT_MEMBER: break0_lock */
/* @DEFINE: BREAK0 input locked */
#define TIMER_BREAK0_LK_ENABLE                          ((uint32_t)TIMER_CCHP_BRK0LK)               /*!< BREAK0 input locked enable */
#define TIMER_BREAK0_LK_DISABLE                         ((uint32_t)0x00000000U)                     /*!< BREAK0 input locked disable */

/* @STRUCT_MEMBER: break1_state */
/* @DEFINE: BREAK1 input signal enable */
#define TIMER_BREAK1_ENABLE                             ((uint32_t)TIMER_CCHP_BRK1EN)               /*!< BREAK1 input signal enable */
#define TIMER_BREAK1_DISABLE                            ((uint32_t)0x00000000U)                     /*!< BREAK1 input signal disable */

/* @STRUCT_MEMBER: break1_polarity */
/* @DEFINE: BREAK1 input signal polarity */
#define TIMER_BREAK1_POLARITY_LOW                       ((uint32_t)0x00000000U)                     /*!< BREAK1 input signal polarity is low */
#define TIMER_BREAK1_POLARITY_HIGH                      ((uint32_t)TIMER_CCHP_BRK1P)                /*!< BREAK1 input signal polarity is high */

/* @STRUCT_MEMBER: break1_lock */
/* @DEFINE: BREAK1 input locked */
#define TIMER_BREAK1_LK_ENABLE                          ((uint32_t)TIMER_CCHP_BRK1LK)               /*!< BREAK1 input locked enable */
#define TIMER_BREAK1_LK_DISABLE                         ((uint32_t)0x00000000U)                     /*!< BREAK1 input locked disable */

/* BREAK0 or BREAK1 input source config */
/* @STRUCT_MEMBER: break_input_source_state */
/* @SP:ENABLE/DISABLE */
/* @STRUCT_MEMBER: break_input_source */
/* @DEFINE:BREAK Input source */
#define TIMER_BREAK_INPUT_SOURCE_BRK0IN0                ((uint8_t)0x00U)                            /*!< BREAK0 input source BRK0IN0 */
#define TIMER_BREAK_INPUT_SOURCE_BRK0IN1                ((uint8_t)0x01U)                            /*!< BREAK0 input source BRK0IN1 */
#define TIMER_BREAK_INPUT_SOURCE_BRK0IN2                ((uint8_t)0x02U)                            /*!< BREAK0 input source BRK0IN2 */
#define TIMER_BREAK_INPUT_SOURCE_BRK0HPDF               ((uint8_t)0x03U)                            /*!< BREAK0 input source BRK0HPDF */
#define TIMER_BREAK_INPUT_SOURCE_BRK0CMP0               ((uint8_t)0x04U)                            /*!< BREAK0 input source BRK0CMP0 */
#define TIMER_BREAK_INPUT_SOURCE_BRK0CMP1               ((uint8_t)0x05U)                            /*!< BREAK0 input source BRK0CMP1 */
#define TIMER_BREAK_INPUT_SOURCE_BRK1IN0                ((uint8_t)0x06U)                            /*!< BREAK1 input source BRK1IN0 */
#define TIMER_BREAK_INPUT_SOURCE_BRK1IN1                ((uint8_t)0x07U)                            /*!< BREAK1 input source BRK1IN1 */
#define TIMER_BREAK_INPUT_SOURCE_BRK1IN2                ((uint8_t)0x08U)                            /*!< BREAK1 input source BRK1IN2 */
#define TIMER_BREAK_INPUT_SOURCE_BRK1HPDF               ((uint8_t)0x09U)                            /*!< BREAK1 input source BRK0HPDF */
#define TIMER_BREAK_INPUT_SOURCE_BRK1CMP0               ((uint8_t)0x0AU)                            /*!< BREAK1 input source BRK1CMP0 */
#define TIMER_BREAK_INPUT_SOURCE_BRK1CMP1               ((uint8_t)0x0BU)                            /*!< BREAK1 input source BRK1CMP1 */
#define TIMER_BREAK_INPUT_SOURCE_DISABLE                ((uint8_t)0x0FU)                            /*!< BREAK1 input source DISABLE */

/* @STRUCT_MEMBER: break_input_source_polarity */
/* @DEFINE:break input source polarity */
#define TIMER_BRKIN_POLARITY_LOW                        ((uint16_t)0x0000U)                         /*!< TIMER break external input signal will not be inverted */
#define TIMER_BRKIN_POLARITY_HIGH                       ((uint16_t)0x0001U)                         /*!< TIMER break external input signal will be inverted */

/* TIMER OxCPRE clear source configuration */
/* @STRUCT_MEMBER: exttrigger_polarity */
/* @DEFINE:external trigger polarity */
#define TIMER_ETP_FALLING                               TIMER_SMCFG_ETP                             /*!< active low or falling edge active */
#define TIMER_ETP_RISING                                ((uint32_t)0x00000000U)                     /*!< active high or rising edge active */

/* @STRUCT_MEMBER: exttrigger_prescaler */
/* @DEFINE:external trigger prescaler */
#define TIMER_EXT_TRI_PRESCALER_OFF                     SMCFG_ETPSC(0)                              /*!< no divided */
#define TIMER_EXT_TRI_PRESCALER_DIV2                    SMCFG_ETPSC(1)                              /*!< divided by 2 */
#define TIMER_EXT_TRI_PRESCALER_DIV4                    SMCFG_ETPSC(2)                              /*!< divided by 4 */
#define TIMER_EXT_TRI_PRESCALER_DIV8                    SMCFG_ETPSC(3)                              /*!< divided by 8 */

/* TIMER ci0 trigger input configuration */
/* @PARA: ci0_select */
/* @DEFINE: ci0 trigger input selection */
#define TIMER_CI0_XOR_CH012                             (TIMER_CTL1_TI0S)                           /*!< TIMER hall sensor mode enable */
#define TIMER_CI0_CH0IN                                 ((uint32_t)0x00000000U)                     /*!< TIMER hall sensor mode disable */

/* @STRUCT_MEMBER: clock_polarity */
/* @DEFINE: clock input source polarity */
#define TIMER_CLOCK_TRIGGER_ETI_POLARITY_RISING         TIMER_ETP_FALLING                           /*!< clock or trigger input source is ETI, active high or rising edge active */
#define TIMER_CLOCK_TRIGGER_ETI_POLARITY_FALLING        TIMER_ETP_RISING                            /*!< clock or trigger input source is ETI, active low or falling edge active */
#define TIMER_CLOCK_TRIGGER_POLARITY_RISING             (uint32_t)TIMER_IC_POLARITY_RISING          /*!< clock or trigger input source is CIx(x=0,1), rising edge active */
#define TIMER_CLOCK_TRIGGER_POLARITY_FALLING            (uint32_t)TIMER_IC_POLARITY_FALLING         /*!< clock or trigger input source is CIx(x=0,1), falling edge active */
#define TIMER_CLOCK_TRIGGER_POLARITY_BOTH_EDGE          (uint32_t)TIMER_IC_POLARITY_BOTH_EDGE       /*!< clock or trigger input source is CI0F_ED, both rising and falling edge active */

/* @STRUCT_MEMBER:signal_disconnect_detection */
/* @DEFINE: quadrature decoder signal disconnect detection */
#define TIMER_DECDISCONNECTDEN                          TIMER_CTL2_DECDISDEN                        /*!< quadrature decoder signal disconnection detection enable */
#define TIMER_DECDISCONNECTDISABLE                      ((uint32_t)0x00000000U)                     /*!< quadrature decoder signal disconnection detection disable */

/* @STRUCT_MEMBER: signal_jump_detection */
/* @DEFINE: quadrature decoder signal disconnect detection */
#define TIMER_DECJUMPDEN                                TIMER_CTL2_DECJDEN                          /*!< quadrature decoder signal jump (the two signals jump at the same time) detection enable */
#define TIMER_DECJUNPDISABLE                            ((uint32_t)0x00000000U)                     /*!< quadrature decoder signal jump (the two signals jump at the same time) detection disable */

#define TIMER_CHANNEL_STATE_SET(ptr, channel, state) do { \
    if      ((channel) == (TIMER_CH_0))  {(ptr)->channel_state[0]       = (state);} \
    else if ((channel) == (TIMER_CH_1))  {(ptr)->channel_state[1]       = (state);} \
    else if ((channel) == (TIMER_CH_2))  {(ptr)->channel_state[2]       = (state);} \
    else if ((channel) == (TIMER_CH_3))  {(ptr)->channel_state[3]       = (state);} \
    else if ((channel) == (TIMER_MCH_0)) {(ptr)->multi_channel_state[0] = (state);} \
    else if ((channel) == (TIMER_MCH_1)) {(ptr)->multi_channel_state[1] = (state);} \
    else if ((channel) == (TIMER_MCH_2)) {(ptr)->multi_channel_state[2] = (state);} \
    else if ((channel) == (TIMER_MCH_3)) {(ptr)->multi_channel_state[3] = (state);} \
    else { /* do nothing */ } \
} while (0)                                                                                         /*!< TIMER set channel state */

#define TIMER_CHANNEL_STATE_SETALL(ptr, state) do {   \
        ((ptr)->channel_state[0]       = (state));    \
        ((ptr)->channel_state[1]       = (state));    \
        ((ptr)->channel_state[2]       = (state));    \
        ((ptr)->channel_state[3]       = (state));    \
        ((ptr)->multi_channel_state[0] = (state));    \
        ((ptr)->multi_channel_state[1] = (state));    \
        ((ptr)->multi_channel_state[2] = (state));    \
    } while (0)                                                                                     /*!< TIMER set all channel state */

/* TIMER state enumeration */
typedef enum {
    HAL_TIMER_STATE_NONE = 0x00,    /*!< NONE(default value) */
    HAL_TIMER_STATE_RESET,          /*!< RESET */
    HAL_TIMER_STATE_BUSY,           /*!< BUSY */
    HAL_TIMER_STATE_TIMEOUT,        /*!< TIMEOUT */
    HAL_TIMER_STATE_ERROR,          /*!< ERROR */
    HAL_TIMER_STATE_READY           /*!< READY */
} hal_timer_state_enum;

/* TIMER error type enumeration */
typedef enum {
    HAL_TIMER_ERROR_NONE    = (uint32_t)0x00000000U,    /*!< no error */
    HAL_TIMER_ERROR_SYSTEM  = (uint32_t)0x00000001U,    /*!< TIMER internal error: if problem of clocking, enable/disable, wrong state */
    HAL_TIMER_ERROR_DMA     = (uint32_t)0x00000002U,    /*!< DMA transfer error */
    HAL_TIMER_ERROR_CONFIG  = (uint32_t)0x00000003U     /*!< configuration error occurs */
} hal_timer_error_enum;

/* the service channel x(x=0..3) */
/* channel number */
typedef enum {
    HAL_TIMER_SERVICE_CHANNEL_NONE = 0x00,    /*!< none channel service */
    HAL_TIMER_SERVICE_CHANNEL_0,              /*!< service channel 0 */
    HAL_TIMER_SERVICE_CHANNEL_1,              /*!< service channel 1 */
    HAL_TIMER_SERVICE_CHANNEL_2,              /*!< service channel 2 */
    HAL_TIMER_SERVICE_CHANNEL_3,              /*!< service channel 3 */
    HAL_TIMER_SERVICE_MULCHANNEL_0,           /*!< service mulchannel 0 */
    HAL_TIMER_SERVICE_MULCHANNEL_1,           /*!< service mulchannel 1 */
    HAL_TIMER_SERVICE_MULCHANNEL_2,           /*!< service mulchannel 2 */
    HAL_TIMER_SERVICE_MULCHANNEL_3            /*!< service mulchannel 3 */
} hal_timer_service_channel_enum;

/* the channel state */
typedef enum {
    HAL_TIMER_CHANNEL_STATE_RESET = 0U,       /*!< channel state init */
    HAL_TIMER_CHANNEL_STATE_READY,            /*!< channel state ready */
    HAL_TIMER_CHANNEL_STATE_BUSY              /*!< channel state busy */
} hal_timer_channel_state_enum;

/* the timer dma state */
typedef enum {
    HAL_TIMER_DMA_STATE_RESET = 0U,           /*!< timer dma state init */
    HAL_TIMER_DMA_STATE_READY,                /*!< timer dma state ready */
    HAL_TIMER_DMA_STATE_BUSY                  /*!< timer dma state busy */
} hal_timer_dma_state_enum;

/* TIMER init structure type enumeration */
typedef enum
{
    HAL_TIMER_INIT_STRUCT = 0x00,                    /*!< TIMER basic initialize configuration structure */
    HAL_TIMER_INPUT_CAPTURE_STRUCT,                  /*!< TIMER input capture configuration structure */
    HAL_TIMER_OUTPUT_COMPARE_STRUCT,                 /*!< TIMER output compare configuration structure */
    HAL_TIMER_BREAK_STRUCT,                          /*!< TIMER break and complementary channel protection configuration structure */
    HAL_TIMER_BREAK_INPUT_SOURCE_STRUCT,             /*!< TIMER break input source configuration structure */
    HAL_TIMER_CLEAR_SOURCE_STRUCT,                   /*!< TIMER OCPRE clear source configuration structure */
    HAL_TIMER_CLOCK_SOURCE_STRUCT,                   /*!< TIMER clock source configuration structure */
    HAL_TIMER_SLAVE_MODE_STRUCT,                     /*!< TIMER slave mode configuration structure */
    HAL_TIMER_DECODER_STRUCT,                        /*!< TIMER decoder mode configuration structure */
    HAL_TIMER_DECODER_DMA_CONFIG_STRUCT,             /*!< TIMER decoder mode DMA transfer configuration structure */
    HAL_TIMER_HALL_SENSOR_STRUCT,                    /*!< TIMER hall sensor mode configuration structure */
    HAL_TIMER_SINGLE_PULSE_STRUCT,                   /*!< TIMER single pulse mode configuration structure */
    HAL_TIMER_OUTPUT_COMPARE_ADDITIONAL_STRUCT,      /*!< TIMER output compare additional configuration structure */
    HAL_TIMER_DMA_TRANSFER_CONFIG_STRUCT,            /*!< TIMER DMA transfer configuration structure */
    HAL_TIMER_IRQ_STRUCT,                            /*!< TIMER interrupt user callback function pointer structure */
    HAL_TIMER_DMA_HANDLE_CB_STRUCT,                  /*!< TIMER DMA interrupt user callback function pointer structure */
    HAL_TIMER_DEV_STRUCT,                            /*!< TIMER device information structure */
    HAL_TIMER_FREE_COMPLEMENTARY_PARAMETER_STRUCT,   /*!< TIMER free complementary structure */
    HAL_TIMER_IRQ_USER_CALLBACK_STRUCT               /*!< TIMER interrupt callback structure */
} hal_timer_struct_type_enum;

/* DMA transfer access start address enumeration */
typedef enum {
    TIMER_DMACFG_DMATA_CTL0             = DMACFG_DMATA(0),     /*!< DMA transfer address is TIMER_CTL0 */
    TIMER_DMACFG_DMATA_CTL1             = DMACFG_DMATA(1),     /*!< DMA transfer address is TIMER_CTL1 */
    TIMER_DMACFG_DMATA_SMCFG            = DMACFG_DMATA(2),     /*!< DMA transfer address is TIMER_SMCFG */
    TIMER_DMACFG_DMATA_DMAINTEN         = DMACFG_DMATA(3),     /*!< DMA transfer address is TIMER_DMAINTEN */
    TIMER_DMACFG_DMATA_INTF             = DMACFG_DMATA(4),     /*!< DMA transfer address is TIMER_INTF */
    TIMER_DMACFG_DMATA_SWEVG            = DMACFG_DMATA(5),     /*!< DMA transfer address is TIMER_SWEVG */
    TIMER_DMACFG_DMATA_CHCTL0           = DMACFG_DMATA(6),     /*!< DMA transfer address is TIMER_CHCTL0 */
    TIMER_DMACFG_DMATA_CHCTL1           = DMACFG_DMATA(7),     /*!< DMA transfer address is TIMER_CHCTL1 */
    TIMER_DMACFG_DMATA_CHCTL2           = DMACFG_DMATA(8),     /*!< DMA transfer address is TIMER_CHCTL2 */
    TIMER_DMACFG_DMATA_CNT              = DMACFG_DMATA(9),     /*!< DMA transfer address is TIMER_CNT */
    TIMER_DMACFG_DMATA_PSC              = DMACFG_DMATA(10),    /*!< DMA transfer address is TIMER_PSC */
    TIMER_DMACFG_DMATA_CAR              = DMACFG_DMATA(11),    /*!< DMA transfer address is TIMER_CAR */
    TIMER_DMACFG_DMATA_CREP0            = DMACFG_DMATA(12),    /*!< DMA transfer address is TIMER_CREP0 */
    TIMER_DMACFG_DMATA_CH0CV            = DMACFG_DMATA(13),    /*!< DMA transfer address is TIMER_CH0CV */
    TIMER_DMACFG_DMATA_CH1CV            = DMACFG_DMATA(14),    /*!< DMA transfer address is TIMER_CH1CV */
    TIMER_DMACFG_DMATA_CH2CV            = DMACFG_DMATA(15),    /*!< DMA transfer address is TIMER_CH2CV */
    TIMER_DMACFG_DMATA_CH3CV            = DMACFG_DMATA(16),    /*!< DMA transfer address is TIMER_CH3CV */
    TIMER_DMACFG_DMATA_CCHP             = DMACFG_DMATA(17),    /*!< DMA transfer address is TIMER_CCHP */
    TIMER_DMACFG_DMATA_MCHCTL0          = DMACFG_DMATA(18),    /*!< DMA transfer address is TIMER_MCHCTL0 */
    TIMER_DMACFG_DMATA_MCHCTL1          = DMACFG_DMATA(19),    /*!< DMA transfer address is TIMER_MCHCTL1 */
    TIMER_DMACFG_DMATA_MCHCTL2          = DMACFG_DMATA(20),    /*!< DMA transfer address is TIMER_MCHCTL2 */
    TIMER_DMACFG_DMATA_MCH0CV           = DMACFG_DMATA(21),    /*!< DMA transfer address is TIMER_MCH0CV */
    TIMER_DMACFG_DMATA_MCH1CV           = DMACFG_DMATA(22),    /*!< DMA transfer address is TIMER_MCH1CV */
    TIMER_DMACFG_DMATA_MCH2CV           = DMACFG_DMATA(23),    /*!< DMA transfer address is TIMER_MCH2CV */
    TIMER_DMACFG_DMATA_MCH3CV           = DMACFG_DMATA(24),    /*!< DMA transfer address is TIMER_MCH3CV */
    TIMER_DMACFG_DMATA_CH0COMV_ADD      = DMACFG_DMATA(25),    /*!< DMA transfer address is TIMER_CH0COMV_ADD */
    TIMER_DMACFG_DMATA_CH1COMV_ADD      = DMACFG_DMATA(26),    /*!< DMA transfer address is TIMER_CH1COMV_ADD */
    TIMER_DMACFG_DMATA_CH2COMV_ADD      = DMACFG_DMATA(27),    /*!< DMA transfer address is TIMER_CH2COMV_ADD */
    TIMER_DMACFG_DMATA_CH3COMV_ADD      = DMACFG_DMATA(28),    /*!< DMA transfer address is TIMER_CH3COMV_ADD */
    TIMER_DMACFG_DMATA_CTL2             = DMACFG_DMATA(29),    /*!< DMA transfer address is TIMER_CTL2 */
    TIMER_DMACFG_DMATA_FCCHP0           = DMACFG_DMATA(30),    /*!< DMA transfer address is TIMER_FCCHP0 */
    TIMER_DMACFG_DMATA_FCCHP1           = DMACFG_DMATA(31),    /*!< DMA transfer address is TIMER_FCCHP1 */
    TIMER_DMACFG_DMATA_FCCHP2           = DMACFG_DMATA(32),    /*!< DMA transfer address is TIMER_FCCHP2 */
    TIMER_DMACFG_DMATA_FCCHP3           = DMACFG_DMATA(33),    /*!< DMA transfer address is TIMER_FCCHP3 */
    TIMER_DMACFG_DMATA_AFCTL0           = DMACFG_DMATA(34),    /*!< DMA transfer address is TIMER_AFCTL0 */
    TIMER_DMACFG_DMATA_AFCTL1           = DMACFG_DMATA(35),    /*!< DMA transfer address is TIMER_AFCTL1 */
    TIMER_DMACFG_DMATA_WDGPER           = DMACFG_DMATA(36),    /*!< DMA transfer address is TIMER_WDGPER */
    TIMER_DMACFG_DMATA_CREP1            = DMACFG_DMATA(37)     /*!< DMA transfer address is TIMER_CREP1 */
} hal_timer_dma_transfer_start_address_enum;

/* DMA transfer access length enumeration */
typedef enum {
    TIMER_DMACFG_DMATC_1TRANSFER        = DMACFG_DMATC(0),     /*!< DMA transfer 1 time */
    TIMER_DMACFG_DMATC_2TRANSFER        = DMACFG_DMATC(1),     /*!< DMA transfer 2 times */
    TIMER_DMACFG_DMATC_3TRANSFER        = DMACFG_DMATC(2),     /*!< DMA transfer 3 times */
    TIMER_DMACFG_DMATC_4TRANSFER        = DMACFG_DMATC(3),     /*!< DMA transfer 4 times */
    TIMER_DMACFG_DMATC_5TRANSFER        = DMACFG_DMATC(4),     /*!< DMA transfer 5 times */
    TIMER_DMACFG_DMATC_6TRANSFER        = DMACFG_DMATC(5),     /*!< DMA transfer 6 times */
    TIMER_DMACFG_DMATC_7TRANSFER        = DMACFG_DMATC(6),     /*!< DMA transfer 7 times */
    TIMER_DMACFG_DMATC_8TRANSFER        = DMACFG_DMATC(7),     /*!< DMA transfer 8 times */
    TIMER_DMACFG_DMATC_9TRANSFER        = DMACFG_DMATC(8),     /*!< DMA transfer 9 times */
    TIMER_DMACFG_DMATC_10TRANSFER       = DMACFG_DMATC(9),     /*!< DMA transfer 10 times */
    TIMER_DMACFG_DMATC_11TRANSFER       = DMACFG_DMATC(10),    /*!< DMA transfer 11 times */
    TIMER_DMACFG_DMATC_12TRANSFER       = DMACFG_DMATC(11),    /*!< DMA transfer 12 times */
    TIMER_DMACFG_DMATC_13TRANSFER       = DMACFG_DMATC(12),    /*!< DMA transfer 13 times */
    TIMER_DMACFG_DMATC_14TRANSFER       = DMACFG_DMATC(13),    /*!< DMA transfer 14 times */
    TIMER_DMACFG_DMATC_15TRANSFER       = DMACFG_DMATC(14),    /*!< DMA transfer 15 times */
    TIMER_DMACFG_DMATC_16TRANSFER       = DMACFG_DMATC(15),    /*!< DMA transfer 16 times */
    TIMER_DMACFG_DMATC_17TRANSFER       = DMACFG_DMATC(16),    /*!< DMA transfer 17 times */
    TIMER_DMACFG_DMATC_18TRANSFER       = DMACFG_DMATC(17),    /*!< DMA transfer 18 times */
    TIMER_DMACFG_DMATC_19TRANSFER       = DMACFG_DMATC(18),    /*!< DMA transfer 19 times */
    TIMER_DMACFG_DMATC_20TRANSFER       = DMACFG_DMATC(19),    /*!< DMA transfer 20 times */
    TIMER_DMACFG_DMATC_21TRANSFER       = DMACFG_DMATC(20),    /*!< DMA transfer 21 times */
    TIMER_DMACFG_DMATC_22TRANSFER       = DMACFG_DMATC(21),    /*!< DMA transfer 22 times */
    TIMER_DMACFG_DMATC_23TRANSFER       = DMACFG_DMATC(22),    /*!< DMA transfer 23 times */
    TIMER_DMACFG_DMATC_24TRANSFER       = DMACFG_DMATC(23),    /*!< DMA transfer 24 times */
    TIMER_DMACFG_DMATC_25TRANSFER       = DMACFG_DMATC(24),    /*!< DMA transfer 25 times */
    TIMER_DMACFG_DMATC_26TRANSFER       = DMACFG_DMATC(25),    /*!< DMA transfer 26 times */
    TIMER_DMACFG_DMATC_27TRANSFER       = DMACFG_DMATC(26),    /*!< DMA transfer 27 times */
    TIMER_DMACFG_DMATC_28TRANSFER       = DMACFG_DMATC(27),    /*!< DMA transfer 28 times */
    TIMER_DMACFG_DMATC_29TRANSFER       = DMACFG_DMATC(28),    /*!< DMA transfer 29 times */
    TIMER_DMACFG_DMATC_30TRANSFER       = DMACFG_DMATC(29),    /*!< DMA transfer 30 times */
    TIMER_DMACFG_DMATC_31TRANSFER       = DMACFG_DMATC(30),    /*!< DMA transfer 31 times */
    TIMER_DMACFG_DMATC_32TRANSFER       = DMACFG_DMATC(31),    /*!< DMA transfer 32 times */
    TIMER_DMACFG_DMATC_33TRANSFER       = DMACFG_DMATC(32),    /*!< DMA transfer 33 times */
    TIMER_DMACFG_DMATC_34TRANSFER       = DMACFG_DMATC(33),    /*!< DMA transfer 34 times */
    TIMER_DMACFG_DMATC_35TRANSFER       = DMACFG_DMATC(34),    /*!< DMA transfer 35 times */
    TIMER_DMACFG_DMATC_36TRANSFER       = DMACFG_DMATC(35),    /*!< DMA transfer 36 times */
    TIMER_DMACFG_DMATC_37TRANSFER       = DMACFG_DMATC(36),    /*!< DMA transfer 37 times */
    TIMER_DMACFG_DMATC_38TRANSFER       = DMACFG_DMATC(37)     /*!< DMA transfer 38 times */
} hal_timer_dma_transfer_length_enum;

/* TIMER decoder mode DMA transfer configuration structure */
typedef struct {
    uint32_t *mem_addr0;    /*!< TIMER decoder mode DMA transfer memory address 0 */
    uint32_t *mem_addr1;    /*!< TIMER decoder mode DMA transfer memory address 1 */
    uint16_t length;        /*!< TIMER DMA transfer length */
} hal_timer_decoder_dma_config_struct;

/* TIMER DMA transfer configuration structure */
typedef struct {
    hal_timer_dma_transfer_start_address_enum start_addr;    /*!< TIMER DMA transfer access start address */
    uint32_t *mem_addr;                                      /*!< TIMER DMA transfer memory address */
    hal_timer_dma_transfer_length_enum length;               /*!< TIMER DMA transfer access length */
} hal_timer_dma_transfer_config_struct;

/* TIMER interrupt user callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb update_handle;                  /*!< TIMER update interrupt callback function */
    __IO hal_irq_handle_cb channelx_capture_handle;        /*!< TIMER channel interrupt for input capture full callback function */
    __IO hal_irq_handle_cb channelx_compare_handle;        /*!< TIMER channel interrupt for output compare full callback function */
    __IO hal_irq_handle_cb channelx_add_compare_handle;    /*!< TIMER channel interrupt for additional output compare callback function */
    __IO hal_irq_handle_cb commutation_handle;             /*!< TIMER commutation interrupt callback function */
    __IO hal_irq_handle_cb signal_jump_handle;             /*!< TIMER quadrature decoder signal jump interrupt callback function */
    __IO hal_irq_handle_cb signal_disconnect_handle;       /*!< TIMER quadrature decoder signal disconnection interrupt callback function */
    __IO hal_irq_handle_cb trigger_handle;                 /*!< TIMER trigger interrupt callback function */
    __IO hal_irq_handle_cb break0_handle;                  /*!< TIMER break interrupt callback function */
    __IO hal_irq_handle_cb break1_handle;                  /*!< TIMER break interrupt callback function */
    __IO hal_irq_handle_cb break_sys_handle;               /*!< TIMER break interrupt callback function */
    __IO hal_irq_handle_cb error_handle;                   /*!< TIMER error interrupt callback function */
} hal_timer_irq_struct;

/* TIMER DMA interrupt callback function pointer structure */
typedef void (*hal_timer_dma_handle_cb)(void *ptr);

typedef struct {
    __IO hal_timer_dma_handle_cb update_dma_full_transcom_handle;              /*!< TIMER DMA transfer complete interrupt handler for TIMER update DMA request */
    __IO hal_timer_dma_handle_cb update_dma_half_transcom_handle;              /*!< TIMER DMA transfer half interrupt handler for TIMER update DMA request */
    __IO hal_timer_dma_handle_cb channelx_capture_dma_full_transcom_handle;    /*!< TIMER DMA transfer complete interrupt handler for TIMER channel input capture DMA request */
    __IO hal_timer_dma_handle_cb channelx_capture_dma_half_transcom_handle;    /*!< TIMER DMA transfer half interrupt handler for TIMER channel input capture DMA request */
    __IO hal_timer_dma_handle_cb channelx_compare_dma_full_transcom_handle;    /*!< TIMER DMA transfer complete interrupt handler for TIMER channel output compare DMA request */
    __IO hal_timer_dma_handle_cb channelx_compare_dma_half_transcom_handle;    /*!< TIMER DMA transfer half interrupt handler for TIMER channel output compare DMA request */
    __IO hal_timer_dma_handle_cb commutation_dma_full_transcom_handle;         /*!< TIMER DMA transfer complete interrupt handler for TIMER commutation DMA request */
    __IO hal_timer_dma_handle_cb trigger_dma_full_transcom_handle;             /*!< TIMER DMA transfer complete interrupt handler for TIMER trigger DMA request */
    __IO hal_timer_dma_handle_cb trigger_dma_half_transcom_handle;             /*!< TIMER DMA transfer half interrupt handler for TIMER trigger DMA request */
    __IO hal_timer_dma_handle_cb error_handle;                                 /*!< TIMER DMA transfer error interrupt handler */
} hal_timer_dma_handle_cb_struct;

/* TIMER device information structure */
typedef struct {
    uint32_t periph;                                        /*!< the timerx (x=0~4,7,14~16,22,23,30,31,40~44) */
    hal_timer_service_channel_enum service_channel;         /*!< the service channel x */
    hal_timer_irq_struct timer_irq;                         /*!< TIMER interrupt user callback function pointer structure */
    hal_dma_dev_struct *p_dma_timer[11];                    /*!< DMA device information structure */
    hal_timer_dma_handle_cb_struct timer_dma;               /*!< TIMER DMA interrupt user callback function pointer structure */
    hal_timer_error_enum error_state;                       /*!< TIMER error state */
    hal_timer_channel_state_enum channel_state[4];          /*!< TIMER channel state */
    hal_timer_channel_state_enum multi_channel_state[4];    /*!< TIMER multi-channel state */
    hal_timer_dma_state_enum timer_dma_state;               /*!< TIMER DMA state */
    hal_timer_state_enum state;                             /*!< TIMER state */
    hal_mutex_enum mutex;                                   /*!< TIMER locked */
    void *priv;                                             /*!< priv data */
    void *trigger_callback;                                 /*!< TIMER trigger callback function pointer */
    void *trigger_dma_half_callback;                        /*!< TIMER trigger DMA half callback function pointer */
    void *dma_error_callback;                               /*!< TIMER error callback function pointer */
    void *update_callback;                                  /*!< TIMER update callback function pointer */
    void *half_update_callback;                             /*!< TIMER half update callback function pointer */
    void *break0_callback;                                  /*!< TIMER break0 callback function pointer */
    void *break1_callback;                                  /*!< TIMER break1 callback function pointer */
    void *channelx_capture_callback;                        /*!< TIMER channel input capture callback function pointer */
    void *channelx_half_capture_callback;                   /*!< TIMER channel input capture callback function pointer */
    void *channelx_compare_callback;                        /*!< TIMER channel output compare callback function pointer */
    void *channelx_half_compare_callback;                   /*!< TIMER channel output compare callback function pointer */
    void *full_commutation_callback;                        /*!< TIMER commutation callback function pointer */
    void *half_commutation_callback;                        /*!< TIMER half commutation callback function pointer */
    void *calculate_complete_callback;                      /*!< TIMER calculate complete callback function pointer */
    void *error_handle;                                     /*!< TIMER error callback function pointer */
} hal_timer_dev_struct;

/* channel free complementary parameter struct definitions */
typedef struct {
    uint32_t freecomstate;     /*!< free complementary channel protection enable */
    uint32_t run_offstate;     /*!< run mode off-state */
    uint32_t idle_offstate;    /*!< idle mode off-state */
    uint32_t deadtime;         /*!< dead time */
} hal_timer_free_complementary_parameter_struct;

/* UPIFBU bit state */
typedef enum {
    VALID_RESET = 0,  /*!< The queried flag/state is currently in the reset condition */
    VALID_SET   = 1,  /*!< The queried flag/state is currently in the set condition */
    INVALID     = 2   /*!< The queried flag/state is not in a valid domain */
} UPIFBUStatus;

/* @STRUCT_MEMBER:  shadow_update */
typedef enum {
    TIMER_UPDATECTL_CCU             = 0x00000000U,     /*!< the shadow registers update by when CMTG bit is set */
    TIMER_UPDATECTL_CCUTRI          = 0x00000001U,     /*!< the shadow registers update by when CMTG bit is set or an rising edge of TRGI occurs */
    TIMER_UPDATECTL_CCUOVER         = 0x00000002U,     /*!< the shadow registers update by when the overflow event occurs */
    TIMER_UPDATECTL_CCUUNDER        = 0x00000003U,     /*!< the shadow registers update by when the underflow event occurs */
    TIMER_UPDATECTL_CCUOVERUNDER    = 0x00000004U      /*!< the shadow registers update by when the overflow or underflow event occurs */
} hal_timer_shadow_update_enum;

/* @STRUCT_MEMBER: upif_backup_state */
/* @SP: ENABLE/DISABLE */
/* @STRUCT_MEMBER: trgo0_selection */
/* @ENUM: master mode control */
typedef enum {
    TIMER_TRI_OUT0_SRC_RESET        = CTL1_MMC0(0),    /*!< the UPG bit as trigger output 0 */
    TIMER_TRI_OUT0_SRC_ENABLE       = CTL1_MMC0(1),    /*!< the counter enable signal as trigger output 0 */
    TIMER_TRI_OUT0_SRC_UPDATE       = CTL1_MMC0(2),    /*!< update event as trigger output 0 */
    TIMER_TRI_OUT0_SRC_CH0          = CTL1_MMC0(3),    /*!< a capture or a compare match occurred in channel 0 as trigger output 0 */
    TIMER_TRI_OUT0_SRC_O0CPRE       = CTL1_MMC0(4),    /*!< O0CPRE as trigger output 0 */
    TIMER_TRI_OUT0_SRC_O1CPRE       = CTL1_MMC0(5),    /*!< O1CPRE as trigger output 0 */
    TIMER_TRI_OUT0_SRC_O2CPRE       = CTL1_MMC0(6),    /*!< O2CPRE as trigger output 0 */
    TIMER_TRI_OUT0_SRC_O3CPRE       = CTL1_MMC0(7)     /*!< O3CPRE as trigger output 0 */
} hal_timer_trgo0_selection_enum;

/* @STRUCT_MEMBER: trgo1_selection */
/* @ENUM: master mode control */
typedef enum {
    TIMER_TRI_OUT1_SRC_RESET        = CTL1_MMC1(0),    /*!< the UPG bit as trigger output 1 */
    TIMER_TRI_OUT1_SRC_ENABLE       = CTL1_MMC1(1),    /*!< the counter enable signal TIMER_CTL0_CEN as trigger output 1 */
    TIMER_TRI_OUT1_SRC_UPDATE       = CTL1_MMC1(2),    /*!< update event as trigger output 1 */
    TIMER_TRI_OUT1_SRC_CH0          = CTL1_MMC1(3),    /*!< a capture or a compare match occurred in channel 0 as trigger output 1 */
    TIMER_TRI_OUT1_SRC_O0CPRE       = CTL1_MMC1(4),    /*!< O0CPRE as trigger output 1 */
    TIMER_TRI_OUT1_SRC_O1CPRE       = CTL1_MMC1(5),    /*!< O1CPRE as trigger output 1 */
    TIMER_TRI_OUT1_SRC_O2CPRE       = CTL1_MMC1(6),    /*!< O2CPRE as trigger output 1 */
    TIMER_TRI_OUT1_SRC_O3CPRE       = CTL1_MMC1(7)     /*!< O3CPRE as trigger output 1 */
} hal_timer_trgo1_selection_enum;

/* @PARA: timer */
/* @STRUCT: hal_timer_init_struct */
/* TIMER init parameter struct definitions */
typedef struct {
    uint16_t prescaler;                                /*!< prescaler value */
    uint16_t alignedmode;                              /*!< aligned mode */
    uint16_t counter_direction;                        /*!< counter direction */
    uint64_t period;                                   /*!< period value, TIMERx_CAR */
    uint16_t clock_division;                           /*!< clock division value */
    uint32_t repetition_counter;                       /*!< the counter repetition value */
    uint16_t auto_reload_state;                        /*!< counter auto reload state */
    uint32_t master_slave_mode;                        /*!< master-slave mode enable or disable, TIMERx_SMCFG:bit7(MSM) */
    uint16_t repetition_crep_selection;                /*!< selection crep0 or crep1 */
    uint16_t upif_backup_state;                        /*!< upif backup state */
    hal_timer_trgo0_selection_enum trgo0_selection;    /*!< master mode control 0, TIMERx_CTL1:bit4~6(MMC) */
    hal_timer_trgo1_selection_enum trgo1_selection;    /*!< master mode control 1, TIMERx_CTL1:bit4~6(MMC) */
} hal_timer_init_struct;

/* @STRUCT_MEMBER: ic_filter */
/* @=NULL */
/* @PARA: timer_inputcapture */
/* @STRUCT: channel input parameter struct definitions */
typedef struct {
    uint16_t ic_polarity;     /*!< channel input polarity */
    uint16_t ic_selection;    /*!< channel input mode selection */
    uint16_t ic_prescaler;    /*!< channel input capture prescaler */
    uint16_t ic_filter;       /*!< channel input capture filter control */
} hal_timer_input_capture_struct;

/* TIMER output compare mode configuration */
/* @PARA: channel */
/* @DEFINE: TIMER channel n(n=0,1,2,3) */
/* @REFER: TIMER_CH_0 */
/* @STRUCT_MEMBER: compare_mode */
/* @ENUM: TIMER channel compare output control */
typedef enum {
    TIMER_OC_MODE_TIMING   = ((uint32_t)0x00000000U),    /*!< timing mode */
    TIMER_OC_MODE_ACTIVE   = ((uint32_t)0x00000010U),    /*!< active mode, set on match */
    TIMER_OC_MODE_INACTIVE = ((uint32_t)0x00000020U),    /*!< inactive mode, clear on match */
    TIMER_OC_MODE_TOGGLE   = ((uint32_t)0x00000030U),    /*!< toggle mode */
    TIMER_OC_MODE_LOW      = ((uint32_t)0x00000040U),    /*!< force low mode */
    TIMER_OC_MODE_HIGH     = ((uint32_t)0x00000050U),    /*!< force high mode */
    TIMER_OC_MODE_PWM0     = ((uint32_t)0x00000060U),    /*!< PWM0 mode */
    TIMER_OC_MODE_PWM1     = ((uint32_t)0x00000070U),    /*!< PWM1 mode */
    TIMER_OC_MODE_DSPM0    = ((uint32_t)0x00010000U),    /*!< delayable SPM mode 0 */
    TIMER_OC_MODE_DSPM1    = ((uint32_t)0x00010010U)     /*!< delayable SPM mode 1 */
} hal_timer_output_compare_enum;

/* @STRUCT_MEMBER:match_mode */
/* @ENUM: output pulse match */
typedef enum {
    HAL_OC_MATCH_NORMAL    = (uint16_t)0x0000U,    /*!< output normal */
    HAL_OC_MATCH_UP        = (uint16_t)0x0100U,    /*!< only when the counter counts upwards and a matching event occurs , output one pulse */
    HAL_OC_MATCH_DOWN      = (uint16_t)0x0200U,    /*!< only when the counter counts downwards and a matching event occurs , output one pulse */
    HAL_OC_MATCH_BOTH      = (uint16_t)0x0300U     /*!< when the counter counts up or down, when a match event occurs , output one pulse */
} hal_timer_output_match_pulse_enum;

/* @STRUCT_MEMBER:channel_mode */
/* @ENUM: output compare channel mode */
typedef enum {
    HAL_OC_WITHOUT_PINS     = (uint16_t)0x0000U,   /*!< channelx output compare without pins */
    HAL_OC_ON_CHX           = (uint16_t)0x0001U,   /*!< channelx output compare on channelx */
    HAL_OC_ON_CHX_AND_MCH   = (uint16_t)0x0002U,   /*!< channelx output compare on channelx and mul-mode channelx */
    HAL_OC_ON_MCHX          = (uint16_t)0x0003U    /*!< channelx output compare on mul-mode channelx */
} hal_timer_output_compare_channel_enum;

/* @STRUCT_MEMBER:composite_pwm_mode */
/* @SP: ENABLE/DISABLE */
/* @STRUCT_MEMBER:additional_oc_shadow */
/* @SP: ENABLE/DISABLE */
/* @STRUCT_MEMBER:additional_oc_value */
/* @=NULL */

/* @PARA: timer_outputcompare */
/* @STRUCT: timer output compare struct definitions */
/* channel output parameter struct definitions */
typedef struct {
    hal_timer_output_compare_enum compare_mode;             /*!< channel output compare mode control */
    hal_timer_output_match_pulse_enum match_mode;           /*!< channel output match selection */
    hal_timer_output_compare_channel_enum channel_mode;     /*!< channel output on channelx mode selection */
    uint16_t oc_outputmode;                                 /*!< channel output complementary mode or independent mode */
    uint32_t oc_pulsevalue;                                 /*!< channel capture or compare value */
    uint16_t oc_polarity;                                   /*!< channel output polarity */
    uint16_t oc_npolarity;                                  /*!< channel complementary output polarity */
    uint16_t oc_idlestate;                                  /*!< idle state of channel output */
    uint16_t oc_nidlestate;                                 /*!< idle state of channel complementary output */
    uint16_t oc_deadtime_insert;                            /*!< channel output compare dead time insert state */
    uint16_t oc_shadow;                                     /*!< channel output compare shadow */
    uint16_t oc_clearmode;                                  /*!< channel output compare clear mode */
    uint32_t freecomstate;                                  /*!< free complementary channel protection enable */
    uint32_t run_offstate;                                  /*!< run mode off-state */
    uint32_t idle_offstate;                                 /*!< idle mode off-state */
    uint32_t deadtime;                                      /*!< dead time */
    uint16_t composite_pwm_mode;                            /*!< composite PWM mode enable */
    uint16_t additional_oc_shadow;                          /*!< additional output compare shadow register enable */
    uint32_t additional_oc_value;                           /*!< additional compare value */
} hal_timer_output_compare_struct;

/* @STRUCT_MEMBER: break0_filter */
/* @=NULL */
/* @STRUCT_MEMBER: break1_filter */
/* @=NULL */
/* @PARA: timer_break */
/* @STRUCT:break config */
/* break parameter struct definitions */
typedef struct {
    uint32_t run_offstate;        /*!< run mode off-state */
    uint32_t idle_offstate;       /*!< idle mode off-state */
    uint32_t deadtime;            /*!< dead time */
    uint32_t output_autostate;    /*!< output automatic enable */
    uint32_t protectmode;         /*!< complementary register protect control */
    uint32_t break0_state;        /*!< BREAK0 input enable */
    uint32_t break0_polarity;     /*!< BREAK0 input polarity */
    uint32_t break0_lock;         /*!< BREAK0 input lock */
    uint32_t break0_filter;       /*!< BREAK0 input filter */
    uint32_t break1_state;        /*!< BREAK1 input enable */
    uint32_t break1_polarity;     /*!< BREAK1 input polarity */
    uint32_t break1_lock;         /*!< BREAK1 input lock */
    uint32_t break1_filter;       /*!< BREAK1 input filter */
} hal_timer_break_struct;

/* @PARA:timer_break_input_source */
/* @STRUCT:break input source config */
typedef struct {                             /*!< BREAK Number.Break1 for Timer0,7 */
    uint16_t break_input_source_state;       /*!< BREAK input source disable or enable */
    uint16_t break_input_source;             /*!< BREAK input source config Break1 for Timer0,7 */
    uint16_t break_input_source_polarity;    /*!< BREAK input source polarity Break1 for Timer0,7 */
} hal_timer_break_input_source_struct;

/* @STRUCT_MEMBER: exttrigger_filter */
/* @=NULL */
/* @PARA: timer_clearsource */
/* @STRUCT: TIMER OCPRE clear source configuration structure */
typedef struct {
    uint32_t exttrigger_polarity;     /*!< external trigger polarity */
    uint32_t exttrigger_prescaler;    /*!< external trigger prescaler */
    uint32_t exttrigger_filter;       /*!< external trigger filter, 0x00~0x0F */
} hal_timer_clear_source_struct;

/* TIMER single pulse mode configuration */
/* @PARA: single_pulse */
/* @SP: ENABLE/DISABLE */

/* clock source configuration */
/* @STRUCT_MEMBER: clock_source */
/* @ENUM: timer clock source select */
typedef enum {
    TIMER_CLOCK_SOURCE_CK_TIMER = (uint32_t)0x00000000U,           /*!< clock input source selection:CK_TIMER, internal clock */
    TIMER_CLOCK_SOURCE_ITI0     = TIMER_SMCFG_TRGSEL_ITI0,         /*!< clock input source selection:ITI0,external clock mode 0 */
    TIMER_CLOCK_SOURCE_ITI1     = TIMER_SMCFG_TRGSEL_ITI1,         /*!< clock input source selection:ITI1, external clock mode 0 */
    TIMER_CLOCK_SOURCE_ITI2     = TIMER_SMCFG_TRGSEL_ITI2,         /*!< clock input source selection:ITI2, external clock mode 0 */
    TIMER_CLOCK_SOURCE_ITI3     = TIMER_SMCFG_TRGSEL_ITI3,         /*!< clock input source selection:ITI3, external clock mode 0 */
    TIMER_CLOCK_SOURCE_CI0FE0   = TIMER_SMCFG_TRGSEL_CI0FE0,       /*!< clock input source selection:CI0FE0, external clock mode 0 */
    TIMER_CLOCK_SOURCE_CI1FE1   = TIMER_SMCFG_TRGSEL_CI1FE1,       /*!< clock input source selection:CI1FE1, external clock mode 0 */
    TIMER_CLOCK_SOURCE_CI0FED   = TIMER_SMCFG_TRGSEL_CI0F_ED,      /*!< clock input source selection:CI0FED, external clock mode 0 */
    TIMER_CLOCK_SOURCE_CI2FE2   = TIMER_SMCFG_TRGSEL_CI2FE2,       /*!< clock input source selection:CI2FE2, external clock mode 0 */
    TIMER_CLOCK_SOURCE_CI3FE3   = TIMER_SMCFG_TRGSEL_CI3FE3,       /*!< clock input source selection:CI3FE3, external clock mode 0 */
    TIMER_CLOCK_SOURCE_MCI0FEM0 = TIMER_SMCFG_TRGSEL_MCI0FEM0,     /*!< clock input source selection:MCI0FEM0, external clock mode 0 */
    TIMER_CLOCK_SOURCE_MCI1FEM1 = TIMER_SMCFG_TRGSEL_MCI1FEM1,     /*!< clock input source selection:MCI1FEM1, external clock mode 0 */
    TIMER_CLOCK_SOURCE_MCI2FEM2 = TIMER_SMCFG_TRGSEL_MCI2FEM2,     /*!< clock input source selection:MCI2FEM2, external clock mode 0 */
    TIMER_CLOCK_SOURCE_MCI3FEM3 = TIMER_SMCFG_TRGSEL_MCI3FEM3,     /*!< clock input source selection:MCI3FEM3, external clock mode 0 */
    TIMER_CLOCK_SOURCE_ITI12    = TIMER_SMCFG_TRGSEL_ITI12,        /*!< clock input source selection:ITI12, external clock mode 0 */
    TIMER_CLOCK_SOURCE_ITI13    = TIMER_SMCFG_TRGSEL_ITI13,        /*!< clock input source selection:ITI13, external clock mode 0 */
    TIMER_CLOCK_SOURCE_ITI14    = TIMER_SMCFG_TRGSEL_ITI14,        /*!< clock input source selection:ITI14, external clock mode 0 */
    TIMER_CLOCK_SOURCE_ETIMODE0 = TIMER_SMCFG_TRGSEL_ETIFP,        /*!< clock input source selection:ETI in external clock mode0 */
    TIMER_CLOCK_SOURCE_ETIMODE1 = TIMER_SMCFG_SMC1,                /*!< clock input source selection:ETI in external clock mode1 */
    TIMER_CLOCK_SOURCE_ITI4     = TIMER_L0_SMCFG_TRGSEL_ITI4,      /*!< clock input source selection:ITI4 for General-L0 timer */
    TIMER_CLOCK_SOURCE_ITI5     = TIMER_L0_SMCFG_TRGSEL_ITI5,      /*!< clock input source selection:ITI5 for General-L0 timer */
    TIMER_CLOCK_SOURCE_ITI7     = TIMER_L0_SMCFG_TRGSEL_ITI7,      /*!< clock input source selection:ITI7 for General-L0 timer */
    TIMER_CLOCK_SOURCE_ITI9     = TIMER_L0_SMCFG_TRGSEL_ITI9,      /*!< clock input source selection:ITI9 for General-L0 timer */
    TIMER_CLOCK_SOURCE_ITI10    = TIMER_L0_SMCFG_TRGSEL_ITI10,     /*!< clock input source selection:ITI10 for General-L0 timer */
    TIMER_CLOCK_SOURCE_ITI11    = TIMER_L0_SMCFG_TRGSEL_ITI11      /*!< clock input source selection:ITI11 for General-L0 timer */
} hal_timer_clock_source_enum;

/* clock source configuration */
/* @STRUCT_MEMBER: clock_mode */
/* @ENUM: timer clock mode select */
typedef enum {
    TIMER_CLOCK_MODE_INTERNAL = (uint32_t)0x00000000U,    /*!< clock internal mode */
    TIMER_CLOCK_MODE_EXTERNAL0,                           /*!< clock external 0 mode */
    TIMER_CLOCK_MODE_EXTERNAL1                            /*!< clock external 1 mode */
} hal_timer_clock_mode_enum;

/* @STRUCT_MEMBER: clock_prescaler */
/* @DEFINE: clock input source prescaler for ETI */
/* @REFER: TIMER_EXT_TRI_PRESCALER_OFF */
/* @STRUCT_MEMBER: clock_filter */
/* @=NULL */
/* @PARA: timer_clocksource */
/* @STRUCT: TIMER clock source configuration structure */
typedef struct {
    hal_timer_clock_source_enum clock_source;    /*!< select clock input source */
    hal_timer_clock_mode_enum clock_mode;        /*!< clock mode selection */
    uint32_t clock_polarity;                     /*!< clock input source polarity */
    uint32_t clock_filter;                       /*!< clock input source filter */
    uint32_t clock_prescaler;                    /*!< clock input source prescaler */
} hal_timer_clock_source_struct;

/* TIMER slave mode configuration */
/* @STRUCT_MEMBER: slave_mode */
/* @ENUM: slave mode control */
typedef enum {
    TIMER_SLAVE_DISABLE_MODE            = TIMER_SLAVE_MODE_DISABLE,         /*!< slave mode disable */
    TIMER_SLAVE_MODE_RESTART_MODE       = TIMER_SLAVE_MODE_RESTART,         /*!< restart mode */
    TIMER_SLAVE_MODE_PAUSE_MODE         = TIMER_SLAVE_MODE_PAUSE,           /*!< pause mode */
    TIMER_SLAVE_MODE_EVENT_MODE         = TIMER_SLAVE_MODE_EVENT,           /*!< event mode */
    TIMER_SLAVE_MODE_EXTERNAL0_MODE     = TIMER_SLAVE_MODE_EXTERNAL0,       /*!< external 0 mode */
    TIMER_SLAVE_MODE_RESTART_EVENT_MODE = TIMER_SLAVE_MODE_RESTART_EVENT    /*!< restart + event mode */
} hal_timer_slave_mode_enum;

/* @STRUCT_MEMBER: trigger_selection */
/* @ENUM: trigger input source */
typedef enum {
    TIMER_TRIGGER_SOURCE_DISABLE  = (uint32_t)0x00000000U,          /*!< trigger input source selection:none */
    TIMER_TRIGGER_SOURCE_ITI0     = TIMER_SMCFG_TRGSEL_ITI0,        /*!< trigger input source selection:ITI0 */
    TIMER_TRIGGER_SOURCE_ITI1     = TIMER_SMCFG_TRGSEL_ITI1,        /*!< trigger input source selection:ITI1 */
    TIMER_TRIGGER_SOURCE_ITI2     = TIMER_SMCFG_TRGSEL_ITI2,        /*!< trigger input source selection:ITI2 */
    TIMER_TRIGGER_SOURCE_ITI3     = TIMER_SMCFG_TRGSEL_ITI3,        /*!< trigger input source selection:ITI3 */
    TIMER_TRIGGER_SOURCE_CI0FE0   = TIMER_SMCFG_TRGSEL_CI0FE0,      /*!< trigger input source selection:CI0FE0 */
    TIMER_TRIGGER_SOURCE_CI1FE1   = TIMER_SMCFG_TRGSEL_CI1FE1,      /*!< trigger input source selection:CI1FE1 */
    TIMER_TRIGGER_SOURCE_CI0FED   = TIMER_SMCFG_TRGSEL_CI0F_ED,     /*!< trigger input source selection:CI0FED */
    TIMER_TRIGGER_SOURCE_CI2FE2   = TIMER_SMCFG_TRGSEL_CI2FE2,      /*!< trigger input source selection:CI2FE2 */
    TIMER_TRIGGER_SOURCE_CI3FE3   = TIMER_SMCFG_TRGSEL_CI3FE3,      /*!< trigger input source selection:CI3FE3 */
    TIMER_TRIGGER_SOURCE_MCI0FEM0 = TIMER_SMCFG_TRGSEL_MCI0FEM0,    /*!< trigger input source selection:MCI0FEM0 */
    TIMER_TRIGGER_SOURCE_MCI1FEM1 = TIMER_SMCFG_TRGSEL_MCI1FEM1,    /*!< trigger input source selection:MCI1FEM1 */
    TIMER_TRIGGER_SOURCE_MCI2FEM2 = TIMER_SMCFG_TRGSEL_MCI2FEM2,    /*!< trigger input source selection:MCI2FEM2 */
    TIMER_TRIGGER_SOURCE_MCI3FEM3 = TIMER_SMCFG_TRGSEL_MCI3FEM3,    /*!< trigger input source selection:MCI3FEM3 */
    TIMER_TRIGGER_SOURCE_ITI12    = TIMER_SMCFG_TRGSEL_ITI12,       /*!< trigger input source selection:ITI12 */
    TIMER_TRIGGER_SOURCE_ITI13    = TIMER_SMCFG_TRGSEL_ITI13,       /*!< trigger input source selection:ITI13 */
    TIMER_TRIGGER_SOURCE_ITI14    = TIMER_SMCFG_TRGSEL_ITI14,       /*!< trigger input source selection:ITI14 */
    TIMER_TRIGGER_SOURCE_ETIFP    = TIMER_SMCFG_TRGSEL_ETIFP,       /*!< trigger input source selection:ETI */
    TIMER_TRIGGER_SOURCE_ITI4     = TIMER_L0_SMCFG_TRGSEL_ITI4,     /*!< internal trigger 4 for General-L0 timer */
    TIMER_TRIGGER_SOURCE_ITI5     = TIMER_L0_SMCFG_TRGSEL_ITI5,     /*!< internal trigger 5 for General-L0 timer */
    TIMER_TRIGGER_SOURCE_ITI7     = TIMER_L0_SMCFG_TRGSEL_ITI7,     /*!< internal trigger 7 for General-L0 timer */
    TIMER_TRIGGER_SOURCE_ITI9     = TIMER_L0_SMCFG_TRGSEL_ITI9,     /*!< internal trigger 9 for General-L0 timer */
    TIMER_TRIGGER_SOURCE_ITI10    = TIMER_L0_SMCFG_TRGSEL_ITI10,    /*!< internal trigger 10 for General-L0 timer */
    TIMER_TRIGGER_SOURCE_ITI11    = TIMER_L0_SMCFG_TRGSEL_ITI11     /*!< internal trigger 11 for General-L0 timer */
} hal_timer_input_trigger_source_enum;

/* @STRUCT_MEMBER: trigger_polarity */
/* @DEFINE: trigger input source polarity */
/* @REFER: TIMER_CLOCK_TRIGGER_ETI_POLARITY_RISING */
/* @STRUCT_MEMBER: trigger_prescaler */
/* @DEFINE: trigger input source prescaler */
/* @REFER: TIMER_EXT_TRI_PRESCALER_OFF */
/* @STRUCT_MEMBER: trigger_filter */
/* @=NULL */
/* @PARA: timer_slavemode */
/* @STRUCT: TIMER slave mode configuration structure */
typedef struct {
    hal_timer_slave_mode_enum slave_mode;                     /*!< slave mode selection */
    hal_timer_input_trigger_source_enum trigger_selection;    /*!< trigger selection */
    uint32_t trigger_polarity;                                /*!< trigger input source polarity */
    uint32_t trigger_filter;                                  /*!< trigger input source filter, 0x00~0x0F */
    uint32_t trigger_prescaler;                               /*!< trigger input source prescaler */
} hal_timer_slave_mode_struct;

/* TIMER decoder mode configuration */
/* @STRUCT_MEMBER: decoder_mode */
/* @ENUM: timer decoder mode */
typedef enum {
    TIMER_QUADRATURE_DECODER_MODE0      = TIMER_QUAD_DECODER_MODE0,           /*!< quadrature decoder mode 0 */
    TIMER_QUADRATURE_DECODER_MODE1      = TIMER_QUAD_DECODER_MODE1,           /*!< quadrature decoder mode 1 */
    TIMER_QUADRATURE_DECODER_MODE2      = TIMER_QUAD_DECODER_MODE2,           /*!< quadrature decoder mode 2 */
    TIMER_NON_QUADRATURE_DECODER_MODE0  = TIMER_NONQUAD_DECODER_MODE0,        /*!< non-quadrature decoder mode 0 */
    TIMER_NON_QUADRATURE_DECODER_MODE1  = TIMER_NONQUAD_DECODER_MODE1         /*!< non-quadrature decoder mode 1 */
} hal_timer_decoder_mode_enum;

/* @STRUCT_MEMBER: ci0_polarity */
/* @DEFINE: channel0 input capture polarity */
/* @REFER: TIMER_IC_POLARITY_RISING */
/* @STRUCT_MEMBER: ci0_selection */
/* @DEFINE: channel0 input capture source selection */
/* @REFER: TIMER_IC_SELECTION_DIRECTTI */
/* @STRUCT_MEMBER: ci0_prescaler */
/* @DEFINE: channel0 input capture prescaler */
/* @REFER: TIMER_IC_PSC_DIV1 */
/* @STRUCT_MEMBER: ci0_filter */
/* @=NULL */
/* @STRUCT_MEMBER: ci1_polarity */
/* @DEFINE: channel1 input capture polarity */
/* @REFER: TIMER_IC_POLARITY_RISING */
/* @STRUCT_MEMBER: ci1_selection */
/* @DEFINE: channel1 input capture source selection */
/* @REFER: TIMER_IC_SELECTION_DIRECTTI */
/* @STRUCT_MEMBER: ci1_prescaler */
/* @DEFINE: channel1 input capture prescaler */
/* @REFER: TIMER_IC_PSC_DIV1 */
/* @STRUCT_MEMBER: ci1_filter */
/* @=NULL */
/* @STRUCT_MEMBER: watchdog_counter_period */
/* @=NULL */
/* @PARA: timer_decoder */
/* @STRUCT: decoder mode configuration structure */
typedef struct {
    hal_timer_decoder_mode_enum decoder_mode;    /*!< decoder mode */
    uint32_t signal_disconnect_detection;        /*!< decoder signal disconnection detection */
    uint32_t signal_jump_detection;              /*!< decoder signal jump detection */
    uint16_t ci0_polarity;                       /*!< channel0 input capture polarity */
    uint16_t ci0_selection;                      /*!< channel0 input capture source selection */
    uint16_t ci0_prescaler;                      /*!< channel0 input capture prescaler */
    uint16_t ci0_filter;                         /*!< channel0 input capture filter */
    uint16_t ci1_polarity;                       /*!< channel1 input capture polarity */
    uint16_t ci1_selection;                      /*!< channel1 input capture source selection */
    uint16_t ci1_prescaler;                      /*!< channel1 input capture prescaler */
    uint16_t ci1_filter;                         /*!< channel1 input capture filter */
    uint32_t watchdog_counter_period;            /*!< decoder signal disconnect detection watchdog counter value */
} hal_timer_decoder_struct;

/* HALL sensor mode configuration */
/* @STRUCT_MEMBER: cmt_delay */
/* @=NULL */
/* @STRUCT_MEMBER: ci0_polarity */
/* @DEFINE: channel0 input capture polarity */
/* @REFER: TIMER_IC_POLARITY_RISING */
/* @STRUCT_MEMBER: ci0_selection */
/* @DEFINE: channel0 input capture source selection */
/* @REFER: TIMER_IC_SELECTION_DIRECTTI */
/* @STRUCT_MEMBER: ci0_prescaler */
/* @DEFINE: channel0 input capture prescaler */
/* @REFER: TIMER_IC_PSC_DIV1 */
/* @STRUCT_MEMBER: ci0_filter */
/* @=NULL: ci0 filter value */
/* @PARA: timer_hallsensor */
/* @STRUCT: hall sensor mode configuration structure */
typedef struct {
    uint32_t cmt_delay;        /*!< commutation delay(channel 1 compare value) */
    uint16_t ci0_polarity;     /*!< channel0 input capture polarity */
    uint16_t ci0_selection;    /*!< channel0 input capture source selection */
    uint16_t ci0_prescaler;    /*!< channel0 input capture prescaler */
    uint16_t ci0_filter;       /*!< channel0 input capture filter */
} hal_timer_hall_sensor_struct;

/* TIMER single pulse mode channel configuration structure */
typedef struct {
    hal_timer_output_compare_enum sp_compare_mode;     /*!< channel_out compare output control */
    uint32_t sp_oc_pulse_value;                        /*!< channel_out compare value */
    uint16_t sp_oc_polarity;                           /*!< channel_out output polarity */
    uint16_t sp_ocn_polarity;                          /*!< channel_out complementary output polarity */
    uint16_t sp_oc_idlestate;                          /*!< idle state of channel_out output */
    uint16_t sp_ocn_idlestate;                         /*!< idle state of channel_out complementary output */
    uint16_t sp_oc_fastmode;                           /*!< channel_out output compare fast mode */
    uint16_t sp_oc_clearmode;                          /*!< channel_out output compare clear mode */
    uint16_t sp_ic_polarity;                           /*!< channel_in input capture polarity */
    uint16_t sp_ic_selection;                          /*!< channel_in I/O mode selection */
    uint16_t sp_ic_filter;                             /*!< channel_in input capture filter */
} hal_timer_single_pulse_struct;

typedef struct {
    uint16_t oca_pwm_mode;    /*!< composite PWM mode enable */
    uint16_t oca_shadow;      /*!< additional output compare shadow register enable */
    uint32_t oca_value;       /*!< additional compare value */
} hal_timer_output_compare_additional_struct;

typedef void (*hal_timer_user_cb)(hal_timer_dev_struct *timer_dev);

/* TIMER user callback function */
typedef struct {
    __IO hal_timer_user_cb update_callback;                   /*!< TIMER update interrupt callback function */
    __IO hal_timer_user_cb half_update_callback;              /*!< TIMER update interrupt callback function */
    __IO hal_timer_user_cb channelx_capture_callback;         /*!< TIMER channel interrupt for input capture callback function */
    __IO hal_timer_user_cb channelx_half_capture_callback;    /*!< TIMER channel interrupt for input capture callback function */
    __IO hal_timer_user_cb channelx_compare_callback;         /*!< TIMER channel interrupt for output compare callback function */
    __IO hal_timer_user_cb channelx_half_compare_callback;    /*!< TIMER channel interrupt for output compare callback function */
    __IO hal_timer_user_cb full_commutation_callback;         /*!< TIMER commutation interrupt callback function */
    __IO hal_timer_user_cb half_commutation_callback;         /*!< TIMER half complete interrupt callback function */
    __IO hal_timer_user_cb calculate_complete_callback;       /*!< calculate complete interrupt handle */
    __IO hal_timer_user_cb dma_error_callback;                /*!< DMA error callback */
    __IO hal_timer_user_cb trigger_callback;                  /*!< TIMER trigger interrupt callback function */
    __IO hal_timer_user_cb break0_callback;                   /*!< TIMER break0 interrupt callback function */
    __IO hal_timer_user_cb break1_callback;                   /*!< TIMER break1 interrupt callback function */
} hal_timer_irq_user_callback_struct;

/* function declarations */
/* TIMER timebase */
/* deinitialize TIMER and device structure */
int32_t hal_timer_deinit(hal_timer_dev_struct *timer_dev);
/* initialize TIMER structure with the default values */
int32_t hal_timer_struct_init(hal_timer_struct_type_enum hal_struct_type, void *p_struct);
/* @FUNCTION: initialize TIMER base registers */
int32_t hal_timer_init(hal_timer_dev_struct *timer_dev, uint32_t periph, hal_timer_init_struct *timer);
/* configure TIMER input capture mode */
int32_t hal_timer_input_capture_config(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                       hal_timer_input_capture_struct *timer_inputcapture);
/* configure TIMER output compare mode */
int32_t hal_timer_output_compare_config(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                        hal_timer_output_compare_struct *timer_outputcompare);
/* configure TIMER break function */
int32_t hal_timer_break_config(hal_timer_dev_struct *timer_dev, hal_timer_break_struct *timer_break);
/* configure TIMER break input source function */
int32_t hal_timer_break_input_source_config(hal_timer_dev_struct *timer_dev, \
                                            hal_timer_break_input_source_struct *timer_break_input_source);
/* configure TIMER OCPRE clear source */
int32_t hal_timer_ocpre_clear_source_config(hal_timer_dev_struct *timer_dev, \
                                            hal_timer_clear_source_struct *timer_clearsource);
/* configure TIMER ci0 trigger input */
int32_t hal_timer_ci0_input_select(hal_timer_dev_struct *timer_dev, uint16_t ci0_select);
/* configure TIMER hall sensor mode */
int32_t hal_timer_hall_sensor_config(hal_timer_dev_struct *timer_dev, hal_timer_hall_sensor_struct *timer_hallsensor);
/* configure TIMER decoder mode */
int32_t hal_timer_decoder_config(hal_timer_dev_struct *timer_dev, hal_timer_decoder_struct *timer_decoder);
/* configure TIMER single pulse mode */
int32_t hal_timer_single_pulse_mode_config(hal_timer_dev_struct *timer_dev, uint32_t single_pulse);
/* configure TIMER clock source */
int32_t hal_timer_clock_source_config(hal_timer_dev_struct *timer_dev, \
                                      hal_timer_clock_source_struct *timer_clocksource);
/* configure TIMER slave mode */
int32_t hal_timer_slave_mode_config(hal_timer_dev_struct *timer_dev, \
                                    hal_timer_slave_mode_struct *timer_slavemode);
/* @END */

/* TIMER timebase */
/* start TIMER counter */
int32_t hal_timer_counter_start(hal_timer_dev_struct *timer_dev);
/* stop TIMER counter */
int32_t hal_timer_counter_stop(hal_timer_dev_struct *timer_dev);
/* start TIMER counter and update interrupt */
int32_t hal_timer_counter_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                          hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER counter and update interrupt */
int32_t hal_timer_counter_stop_interrupt(hal_timer_dev_struct *timer_dev);
/* start TIMER counter and update DMA request */
int32_t hal_timer_counter_start_dma(hal_timer_dev_struct *timer_dev, \
                                    uint32_t *mem_addr, uint16_t dma_length, \
                                    hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER counter and update DMA request */
int32_t hal_timer_counter_stop_dma(hal_timer_dev_struct *timer_dev);

/* TIMER input capture */
/* start TIMER channel input capture mode */
int32_t hal_timer_input_capture_start(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* stop TIMER channel input capture mode */
int32_t hal_timer_input_capture_stop(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* start TIMER channel input capture mode and channel interrupt */
int32_t hal_timer_input_capture_start_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER channel input capture mode and channel interrupt */
int32_t hal_timer_input_capture_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* start TIMER channel input capture mode and channel DMA request */
int32_t hal_timer_input_capture_start_dma(hal_timer_dev_struct *timer_dev, uint16_t channel, uint32_t *mem_addr, \
                                          uint16_t dma_length, hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER channel input capture mode and channel DMA request */
int32_t hal_timer_input_capture_stop_dma(hal_timer_dev_struct *timer_dev, uint16_t channel);

/* TIMER output compare */
/* start TIMER channel output compare mode */
int32_t hal_timer_output_compare_start(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* stop TIMER channel output compare mode */
int32_t hal_timer_output_compare_stop(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* start TIMER channel output compare mode and channel interrupt */
int32_t hal_timer_output_compare_start_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                 hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER channel output compare mode and channel interrupt */
int32_t hal_timer_output_compare_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* start TIMER channel output compare mode and channel DMA request */
int32_t hal_timer_output_compare_start_dma(hal_timer_dev_struct *timer_dev, \
                                           uint16_t channel, uint32_t *mem_addr, uint16_t dma_length, \
                                           hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER channel output compare mode and channel DMA request */
int32_t hal_timer_output_compare_stop_dma(hal_timer_dev_struct *timer_dev, uint16_t channel);

/* TIMER output compare additional */
/* start TIMER output compare additional mode */
int32_t hal_timer_output_compare_additional_start(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                  hal_timer_output_compare_additional_struct *ocapara);
/* stop TIMER output compare additional mode */
int32_t hal_timer_output_compare_additional_stop(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* start TIMER output compare additional mode with interrupt */
int32_t hal_timer_output_compare_additional_start_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                            hal_timer_output_compare_additional_struct *ocapara, \
                                                            hal_timer_irq_struct *p_irq);
/* stop TIMER output compare additional mode with interrupt */
int32_t hal_timer_output_compare_additional_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel);

/* TIMER output compare and compare additional */
/* start TIMER output compare and compare additional mode */
int32_t hal_timer_output_compare_mix_additional_start(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                      hal_timer_output_compare_struct *ocpara);
/* stop TIMER output compare and compare additional mode */
int32_t hal_timer_output_compare_mix_additional_stop(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* start TIMER output compare and compare additional mode with interrupt */
int32_t hal_timer_output_compare_mix_additional_start_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                                hal_timer_output_compare_struct *ocpara, \
                                                                hal_timer_irq_user_callback_struct *p_user_func);

/* stop TIMER output compare and compare additional mode with interrupt */
int32_t hal_timer_output_compare_mix_additional_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel);

/* TIMER complementary channel output compare */
/* start TIMER complementary channel output compare mode */
int32_t hal_timer_output_compare_complementary_channel_start(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* stop TIMER complementary channel output compare mode */
int32_t hal_timer_output_compare_complementary_channel_stop(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* start TIMER complementary channel output compare mode and channel interrupt */
int32_t hal_timer_output_compare_complementary_channel_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                       uint16_t channel, \
                                                                       hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER complementary channel output compare mode and channel interrupt */
int32_t hal_timer_output_compare_complementary_channel_stop_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                      uint16_t channel);
/* start TIMER complementary channel output compare mode and channel DMA request */
int32_t hal_timer_output_compare_complementary_channel_start_dma(hal_timer_dev_struct *timer_dev, \
                                                                 uint16_t channel, \
                                                                 uint32_t *mem_addr, uint16_t dma_length, \
                                                                 hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER complementary channel output compare mode and channel DMA request */
int32_t hal_timer_output_compare_complementary_channel_stop_dma(hal_timer_dev_struct *timer_dev, uint16_t channel);

/* TIMER single pulse mode configure */
int32_t hal_timer_single_pulse_mode_channel_config(hal_timer_dev_struct *timer_dev, \
                                                   hal_timer_single_pulse_struct *timer_singlepulse, \
                                                   uint16_t channel_out, uint16_t channel_in);
/* start TIMER single pulse mode */
int32_t hal_timer_single_pulse_start(hal_timer_dev_struct *timer_dev);
/* stop TIMER single pulse mode */
int32_t hal_timer_single_pulse_stop(hal_timer_dev_struct *timer_dev);
/* start TIMER single pulse mode and channel interrupt */
int32_t hal_timer_single_pulse_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                               hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER single pulse mode and channel interrupt */
int32_t hal_timer_single_pulse_stop_interrupt(hal_timer_dev_struct *timer_dev);
/* start TIMER complementary channel single pulse mode */
int32_t hal_timer_single_pulse_complementary_channel_start(hal_timer_dev_struct *timer_dev, \
                                                           uint16_t channel_out);
/* stop TIMER complementary channel single pulse mode */
int32_t hal_timer_single_pulse_complementary_channel_stop(hal_timer_dev_struct *timer_dev, \
                                                          uint16_t channel_out);
/* start TIMER complementary channel single pulse mode and channel interrupt */
int32_t hal_timer_single_pulse_complementary_channel_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                     uint16_t channel_out, \
                                                                     hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER complementary channel single pulse mode and channel interrupt */
int32_t hal_timer_single_pulse_complementary_channel_stop_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                    uint16_t channel_out);
/* start TIMER independent channel single pulse mode */
int32_t hal_timer_single_pulse_independent_channel_start(hal_timer_dev_struct *timer_dev, uint16_t channel_out);
/* stop TIMER independent channel single pulse mode */
int32_t hal_timer_single_pulse_independent_channel_stop(hal_timer_dev_struct *timer_dev, uint16_t channel_out);
/* start TIMER independent channel single pulse mode and channel interrupt */
int32_t hal_timer_single_pulse_independent_channel_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                   uint16_t channel_out, \
                                                                   hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER independent channel single pulse mode and channel interrupt */
int32_t hal_timer_single_pulse_independent_channel_stop_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                  uint16_t channel_out);

/* TIMER break input lock release config */
int32_t hal_timer_break_lock_release_config(hal_timer_dev_struct *timer_dev, uint16_t break_num, ControlStatus newvalue);

/* TIMER slave mode */
/* configure TIMER slave mode and interrupt */
int32_t hal_timer_slave_mode_interrupt_config(hal_timer_dev_struct *timer_dev, \
                                              hal_timer_slave_mode_struct *timer_slavemode, \
                                              hal_timer_irq_struct *p_irq);

/* TIMER decoder mode */
/* start TIMER decoder mode */
int32_t hal_timer_decoder_start(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* stop TIMER decoder mode */
int32_t hal_timer_decoder_stop(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* start TIMER decoder mode and channel interrupt */
int32_t hal_timer_decoder_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                          uint16_t channel, \
                                          hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER decoder mode and channel interrupt */
int32_t hal_timer_decoder_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel);
/* start TIMER decoder mode and channel interrupt and jump detection interrupt */
int32_t hal_timer_decoder_jump_detection_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                         hal_timer_irq_struct *p_irq);
/* stop TIMER decoder mode and channel interrupt and jump detection interrupt */
int32_t hal_timer_decoder_jump_detection_stop_interrupt(hal_timer_dev_struct *timer_dev);
/* start TIMER decoder mode and channel interrupt and disconnect detection interrupt */
int32_t hal_timer_decoder_disconnect_detection_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                               hal_timer_irq_struct *p_irq);
/* stop TIMER decoder mode and channel interrupt and disconnect detection interrupt */
int32_t hal_timer_decoder_disconnect_detection_stop_interrupt(hal_timer_dev_struct *timer_dev);
/* start TIMER decoder mode and channel DMA request */
int32_t hal_timer_decoder_start_dma(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                    hal_timer_decoder_dma_config_struct *decoder_dma, \
                                    hal_timer_irq_user_callback_struct *p_user_func);

/* stop TIMER decoder mode and channel DMA request */
int32_t hal_timer_decoder_stop_dma(hal_timer_dev_struct *timer_dev, uint16_t channel);

/* TIMER hall sensor mode */
/* start TIMER hall sensor mode */
int32_t hal_timer_hall_sensor_start(hal_timer_dev_struct *timer_dev);
/* stop TIMER hall sensor mode */
int32_t hal_timer_hall_sensor_stop(hal_timer_dev_struct *timer_dev);
/* start TIMER hall sensor mode and channel interrupt */
int32_t hal_timer_hall_sensor_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                              hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER hall sensor mode and channel interrupt */
int32_t hal_timer_hall_sensor_stop_interrupt(hal_timer_dev_struct *timer_dev);
/* start TIMER hall sensor mode and channel DMA request */
int32_t hal_timer_hall_sensor_start_dma(hal_timer_dev_struct *timer_dev, \
                                        uint32_t *mem_addr, uint16_t dma_length, \
                                        hal_timer_irq_user_callback_struct *p_user_func);
/* stop TIMER hall sensor mode and channel DMA request */
int32_t hal_timer_hall_sensor_stop_dma(hal_timer_dev_struct *timer_dev);

/* DMA mode settings */
/* start TIMER DMA transfer mode for writing data to TIMER */
int32_t hal_timer_dma_transfer_write_start(hal_timer_dev_struct *timer_dev, \
                                           uint32_t dmareq, \
                                           hal_timer_dma_transfer_config_struct *dmatcfg, \
                                           hal_timer_dma_handle_cb_struct *dmacb);
/* stop TIMER DMA transfer mode for writing data to TIMER */
int32_t hal_timer_dma_transfer_write_stop(hal_timer_dev_struct *timer_dev, uint32_t dmareq);
/* start TIMER DMA transfer mode for read data from TIMER */
int32_t hal_timer_dma_transfer_read_start(hal_timer_dev_struct *timer_dev, uint32_t dmareq, \
                                          hal_timer_dma_transfer_config_struct *dmatcfg, \
                                          hal_timer_dma_handle_cb_struct *dmacb);
/* stop TIMER DMA transfer mode for read data from TIMER */
int32_t hal_timer_dma_transfer_read_stop(hal_timer_dev_struct *timer_dev, uint32_t dmareq);
/* TIMER start with dma */
int32_t hal_timer_dma_transfer_start(hal_timer_dev_struct *timer_dev, uint32_t dmareq, \
                                     hal_timer_dma_transfer_config_struct *dmatcfg, \
                                     hal_dma_dev_struct *dma_info);

/* commutation control settings */
/* configure TIMER commutation event */
int32_t hal_timer_commutation_event_config(hal_timer_dev_struct *timer_dev, \
                                           hal_timer_input_trigger_source_enum trigger_source, \
                                           hal_timer_shadow_update_enum com_source);
/* configure TIMER commutation event and enable CMT interrupt */
int32_t hal_timer_commutation_event_interrupt_config(hal_timer_dev_struct *timer_dev, \
                                                     hal_timer_input_trigger_source_enum trigger_source, \
                                                     hal_timer_shadow_update_enum com_source, \
                                                     hal_timer_irq_struct *p_irq);
/* configure TIMER commutation event and enable CMT DMA request */
int32_t hal_timer_commutation_event_dma_config(hal_timer_dev_struct *timer_dev, \
                                               hal_timer_input_trigger_source_enum trigger_source, \
                                               hal_timer_shadow_update_enum com_source, \
                                               hal_timer_dma_handle_cb_struct *dmacb);

/* software generate events */
void hal_timer_event_software_generate(uint32_t timer_periph, uint32_t event);
/* read TIMER channel capture compare register value */
uint32_t hal_timer_channel_capture_value_register_read(uint32_t timer_periph, uint16_t channel);

/* TIMER interrupt callback */
/* set user-defined interrupt callback function,
   which will be registered and called when corresponding interrupt be triggered */
int32_t hal_timer_irq_handle_set(hal_timer_dev_struct *timer_dev, hal_timer_irq_struct *p_irq);
/* reset all user-defined interrupt callback function
   which will be registered and called when corresponding interrupt be triggered */
int32_t hal_timer_irq_handle_all_reset(hal_timer_dev_struct *timer_dev);
/* TIMER interrupt handler content function,which is merely used in timer_handler */
void hal_timer_irq(hal_timer_dev_struct *timer_dev);

/* configure TIMER channel output polarity */
int32_t hal_timer_channel_output_polarity_config(uint32_t timer_periph, uint16_t channel, uint16_t ocpolarity);
/* configure TIMER primary output function */
void hal_timer_primary_output_config(uint32_t timer_periph, ControlStatus newvalue);

/* TIMER get active channel */
hal_timer_service_channel_enum hal_timer_get_active_channel(hal_timer_dev_struct *timer_dev);
/* TIMER get base state */
hal_timer_state_enum hal_timer_get_base_state(hal_timer_dev_struct *timer_dev);
/* TIMER get input capture state */
hal_timer_state_enum hal_timer_get_input_capture_state(hal_timer_dev_struct *timer_dev);
/* TIMER get output compare state */
hal_timer_state_enum hal_timer_get_output_compare_state(hal_timer_dev_struct *timer_dev);
/* TIMER get PWM state */
hal_timer_state_enum hal_timer_get_pwm_state(hal_timer_dev_struct *timer_dev);
/* TIMER get single pulse state */
hal_timer_state_enum hal_timer_get_single_pulse_state(hal_timer_dev_struct *timer_dev);
/* TIMER get decoder mode state */
hal_timer_state_enum hal_timer_get_decoder_mode_state(hal_timer_dev_struct *timer_dev);
/* TIMER get hall sensor mode state */
hal_timer_state_enum hal_timer_get_hall_sensor_mode_state(hal_timer_dev_struct *timer_dev);
/* TIMER get channel state */
hal_timer_channel_state_enum hal_timer_get_channel_state(hal_timer_dev_struct *timer_dev, uint32_t channel);
/* TIMER get DMA state */
hal_timer_dma_state_enum hal_timer_get_dma_state(hal_timer_dev_struct *timer_dev);

/* configure commutation control shadow register update selection */
void hals_timer_commutation_control_shadow_register_config(uint32_t timer_periph, uint16_t ccssel);
/* configure TIMER channel output pulse value */
int32_t hals_timer_channel_output_pulse_value_config(uint32_t timer_periph, uint16_t channel, uint32_t pulse);
/* configure TIMER channel output shadow function */
int32_t hals_timer_channel_output_shadow_config(uint32_t timer_periph, uint16_t channel, uint16_t ocshadow);
/* configure TIMER channel output composite PWM function */
int32_t hals_timer_channel_output_compare_additional_config(uint32_t timer_periph, uint16_t channel, \
                                                            hal_timer_output_compare_additional_struct *ocapare);
/* configure TIMER channel free complementary protection */
int32_t hals_timer_free_complementary_config(uint32_t timer_periph, uint16_t channel, \
                                             hal_timer_free_complementary_parameter_struct *fcpara);
/* configure TIMER channel output match config */
int32_t hals_timer_channel_output_match_config(uint32_t timer_periph, uint16_t channel, \
                                               hal_timer_output_match_pulse_enum ocmatch);
/* configure TIMER channel break input enable */
void hals_timer_channel_break_control_config(uint32_t timer_periph, uint16_t channel, ControlStatus newvalue);
/* configure TIMER internal clock mode */
void hals_timer_internal_clock_config(uint32_t timer_periph, uint32_t intrigger);
/* configure TIMER the internal trigger as external clock input */
void hals_timer_internal_trigger_as_external_clock_config(uint32_t timer_periph, uint32_t intrigger);
/* configure TIMER the external trigger as external clock input */
void hals_timer_external_trigger_as_external_clock_config(uint32_t timer_periph, uint32_t exttrigger, \
                                                          uint32_t extpolarity, uint32_t extfilter);
/* configure TIMER external trigger input */
void hals_timer_external_trigger_config(uint32_t timer_periph, uint32_t extprescaler, \
                                        uint32_t extpolarity, uint32_t extfilter);
/* configure TIMER the external clock mode0 */
void hals_timer_external_clock_mode0_config(uint32_t timer_periph, uint32_t extprescaler, \
                                            uint32_t extpolarity, uint32_t extfilter);
/* configure TIMER the external clock mode1 */
void hals_timer_external_clock_mode1_config(uint32_t timer_periph, uint32_t extprescaler, \
                                            uint32_t extpolarity, uint32_t extfilter);
/* multi mode channel mode select */
void hals_timer_multi_mode_channel_mode_config(uint32_t timer_periph, uint32_t channel, uint32_t multi_mode_sel);

/* configure TIMER break input source */
int32_t hals_timer_break_external_source_config(uint32_t timer_periph, uint16_t brkin_state, \
                                                uint16_t brkin_source, uint16_t brkin_polarity);
/* configure TIMER dead timer insert */
int32_t hals_timer_dead_time_insert_config(uint32_t timer_periph, uint16_t channel, uint16_t deadtime_insert);
/* enable TIMER channel */
int32_t hals_timer_channel_state_config(uint32_t timer_periph, uint16_t channel, uint32_t state);
/* configure TIMER channel complementary output enable state */
int32_t hals_timer_channel_complementary_output_state_config(uint32_t timer_periph, \
                                                             uint16_t channel, uint16_t ocnstate);

/* enable a TIMER */
void hals_timer_enable(uint32_t timer_periph);
/* disable a TIMER */
void hals_timer_disable(uint32_t timer_periph);
/* enable the TIMER DMA */
void hals_timer_dma_enable(uint32_t timer_periph, uint32_t dma);
/* disable the TIMER DMA */
void hals_timer_dma_disable(uint32_t timer_periph, uint32_t dma);
/* TIMER slave mode trigsel source detect */
int32_t hals_timer_slave_mode_detect(hal_timer_dev_struct *timer_dev, \
                                     hal_timer_slave_mode_struct *timer_slavemode);

/* configure TIMER master slave mode */
void hals_timer_master_slave_mode_config(uint32_t timer_periph, uint32_t masterslave);
/* configure TIMER single pulse mode */
void hals_timer_single_pulse_mode_configuration(uint32_t timer_periph, uint32_t single_pulse);
/* configure TIMER channel output clear function */
int32_t hals_timer_channel_output_clear_config(uint32_t timer_periph, uint16_t channel, uint16_t occlear);
/* configure TIMER TGRGO0 */
void hals_timer_trgo0_config(uint32_t periph, hal_timer_trgo0_selection_enum trgo_selection);
/* configure TIMER TGRGO1 */
void hals_timer_trgo1_config(uint32_t periph, hal_timer_trgo1_selection_enum trgo_selection);
/* input capture config */
int32_t hals_timer_channel_input_capture_config(uint32_t timer_periph, uint16_t channel, \
                                                hal_timer_input_capture_struct *timer_inputcapture);
/* output compare config */
int32_t hals_timer_channel_output_compare_config(uint32_t timer_periph, uint16_t channel, \
                                                 hal_timer_output_compare_struct *timer_outputcompare);
/* configure TIMER channel output compare mode */
void hals_timer_channel_output_mode_config(uint32_t timer_periph, uint16_t channel, \
                                           uint16_t oc_mode, hal_timer_output_compare_enum ocmode);

/* enable the update event */
void hals_timer_update_event_enable(uint32_t timer_periph);
/* disable the update event */
void hals_timer_update_event_disable(uint32_t timer_periph);
/* enable the auto-reload shadow register */
void hals_timer_auto_reload_shadow_enable(uint32_t timer_periph);
/* disable the auto-reload shadow register */
void hals_timer_auto_reload_shadow_disable(uint32_t timer_periph);
/* configure TIMER prescaler */
void hals_timer_prescaler_config(uint32_t timer_periph, uint16_t prescaler, uint8_t pscreload);
/* set TIMER counter alignment mode */
void hals_timer_counter_alignment(uint32_t timer_periph, uint16_t aligned);
/* set TIMER counter up direction */
void hals_timer_counter_up_direction(uint32_t timer_periph);
/* set TIMER counter down direction */
void hals_timer_counter_down_direction(uint32_t timer_periph);
/* configure TIMER autoreload register value */
void hals_timer_autoreload_value_config(uint32_t timer_periph, uint64_t autoreload);
/* get TIMER autoreload register value */
uint64_t hals_timer_autoreload_value_read(uint32_t timer_periph);
/* configure TIMER runtime repetition counter value */
void hals_timer_repetition_value_config(uint32_t timer_periph, uint16_t ccsel, uint32_t repetition);
/* get TIMER runtime repetition counter value */
uint32_t hals_timer_repetition_counter_read(uint32_t timer_periph);
/* read quadrature decoder signal disconnection detection watchdog value */
uint32_t hals_timer_watchdog_value_read(uint32_t timer_periph);
/* configure TIMER counter register value */
void hals_timer_counter_value_config(uint32_t timer_periph, uint64_t counter);
/* read TIMER counter value */
uint64_t hals_timer_counter_read(uint32_t timer_periph);
/* read TIMER prescaler value */
uint16_t hals_timer_prescaler_read(uint32_t timer_periph);
/* read TIMER counter direction mode */
uint8_t hals_timer_counter_direction_read(uint32_t timer_periph);
/* configure timer delayable single pulse mode */
void hals_timer_channel_output_delayable_single_pulse_mode_config(uint32_t timer_periph, uint16_t channel, uint16_t cnt_dir);
/* configure TIMER update source */
void hals_timer_update_source_config(uint32_t timer_periph, uint32_t update);
/* channel DMA request source selection */
void hals_timer_channel_dma_request_source_select(uint32_t timer_periph, uint32_t dma_request);
/* configure the TIMER DMA transfer */
void hals_timer_dma_transfer_config(uint32_t timer_periph, hal_timer_dma_transfer_start_address_enum dma_baseaddr, \
                                    hal_timer_dma_transfer_length_enum dma_length);

/* enable TIMER break function */
void hals_timer_break_enable(uint32_t timer_periph, uint16_t break_num);
/* disable TIMER break function */
void hals_timer_break_disable(uint32_t timer_periph, uint16_t break_num);
/* enable TIMER output automatic function */
void hals_timer_automatic_output_enable(uint32_t timer_periph);
/* disable TIMER output automatic function */
void hals_timer_automatic_output_disable(uint32_t timer_periph);
/* configure channel commutation control shadow register */
void hals_timer_channel_control_shadow_config(uint32_t timer_periph, ControlStatus newvalue);
/* configure TIMER channel control shadow register update control */
void hals_timer_channel_control_shadow_update_config(uint32_t timer_periph, hal_timer_shadow_update_enum ccuctl);
/* configure TIMER write CHxVAL register selection */
void hals_timer_write_chxval_register_config(uint32_t timer_periph, uint16_t ccsel);
/* configure TIMER output value selection */
void hals_timer_output_value_selection_config(uint32_t timer_periph, uint16_t outsel);
/* read TIMER channel additional compare value */
uint32_t hals_timer_channel_additional_compare_value_read(uint32_t timer_periph, uint16_t channel);
/* configure quadrature decoder mode */
void hals_timer_quadrature_decoder_mode_config(uint32_t timer_periph, uint32_t decomode, uint16_t ic0polarity, uint16_t ic1polarity);
/* configure non-quadrature decoder mode */
void hals_timer_non_quadrature_decoder_mode_config(uint32_t timer_periph, uint32_t decomode, uint16_t ic1polarity);
/* configure quadrature decoder signal disconnection detection function */
void hals_timer_decoder_disconnection_detection_config(uint32_t timer_periph, \
                                                       uint32_t signal_disconnect_state);
/* configure quadrature decoder signal jump detection function */
void hals_timer_decoder_jump_detection_config(uint32_t timer_periph, uint32_t signal_jump_state);
/* get the UPIFBU bit in the TIMERx_CNT register */
UPIFBUStatus hals_timer_upifbu_bit_get(uint32_t timer_periph);

/* get TIMER flags */
FlagStatus hals_timer_flag_get(uint32_t timer_periph, uint32_t flag);
/* clear TIMER flags */
void hals_timer_flag_clear(uint32_t timer_periph, uint32_t flag);
/* enable the TIMER interrupt */
void hals_timer_interrupt_enable(uint32_t timer_periph, uint32_t interrupt);
/* disable the TIMER interrupt */
void hals_timer_interrupt_disable(uint32_t timer_periph, uint32_t interrupt);
/* get timer interrupt flags */
FlagStatus hals_timer_interrupt_flag_get(uint32_t timer_periph, uint32_t int_flag);
/* clear TIMER interrupt flags */
void hals_timer_interrupt_flag_clear(uint32_t timer_periph, uint32_t int_flag);

#endif /* GD32H7XX_HAL_TIMER_H */
