/*!
    \file    gd32h7xx_hal_sys.h
    \brief   definitions for the SYS

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_SYS_H
#define GD32H7XX_HAL_SYS_H

#include "gd32h7xx_hal.h"

/* DBG definitions */
#define SYS_DBG                                 DBG_BASE

/* registers definitions */
#define SYS_DBG_ID                              REG32(SYS_DBG + 0x00000000U)                       /*!< SYS_DBG_ID code register */
#define SYS_DBG_CTL0                            REG32(SYS_DBG + 0x00000004U)                       /*!< DBG control register 0 */
#define SYS_DBG_CTL1                            REG32(SYS_DBG + 0x00000034U)                       /*!< DBG control register 1 */
#define SYS_DBG_CTL2                            REG32(SYS_DBG + 0x0000003CU)                       /*!< DBG control register 2 */
#define SYS_DBG_CTL3                            REG32(SYS_DBG + 0x0000004CU)                       /*!< DBG control register 3 */
#define SYS_DBG_CTL4                            REG32(SYS_DBG + 0x00000054U)                       /*!< DBG control register 4 */

/* bits definitions */
/* SYS_DBG_ID */
#define SYS_DBG_ID_ID_CODE                      BITS(0,31)                                         /*!< DBG ID code values */

/* SYS_DBG_CTL0 */
#define SYS_DBG_CTL0_SLP_HOLD                   BIT(0)                                             /*!< keep debugger connection during sleep mode */
#define SYS_DBG_CTL0_DSLP_HOLD                  BIT(1)                                             /*!< keep debugger connection during deepsleep mode */
#define SYS_DBG_CTL0_STB_HOLD                   BIT(2)                                             /*!< keep debugger connection during standby mode */
#define SYS_DBG_CTL0_TRACE_MODE                 BITS(18,19)                                        /*!< trace pin mode selection */
#define SYS_DBG_CTL0_TRACECLKEN                 BIT(20)                                            /*!< enable trace pin assignment */

/* SYS_DBG_CTL1 */
#define SYS_DBG_CTL1_WWDGT_HOLD                 BIT(6)                                             /*!< debug WWDGT kept when core is halted */

/* SYS_DBG_CTL2 */
#define SYS_DBG_CTL2_TIMER1_HOLD                BIT(0)                                             /*!< debug TIMER1 kept when core is halted */
#define SYS_DBG_CTL2_TIMER2_HOLD                BIT(1)                                             /*!< debug TIMER2 kept when core is halted */
#define SYS_DBG_CTL2_TIMER3_HOLD                BIT(2)                                             /*!< debug TIMER3 kept when core is halted */
#define SYS_DBG_CTL2_TIMER4_HOLD                BIT(3)                                             /*!< debug TIMER4 kept when core is halted */
#define SYS_DBG_CTL2_TIMER5_HOLD                BIT(4)                                             /*!< debug TIMER5 kept when core is halted */
#define SYS_DBG_CTL2_TIMER6_HOLD                BIT(5)                                             /*!< debug TIMER6 kept when core is halted */
#define SYS_DBG_CTL2_TIMER22_HOLD               BIT(6)                                             /*!< debug TIMER22 kept when core is halted */
#define SYS_DBG_CTL2_TIMER23_HOLD               BIT(7)                                             /*!< debug TIMER23 kept when core is halted */
#define SYS_DBG_CTL2_TIMER30_HOLD               BIT(8)                                             /*!< debug TIMER30 kept when core is halted */
#define SYS_DBG_CTL2_TIMER31_HOLD               BIT(9)                                             /*!< debug TIMER31 kept when core is halted */
#define SYS_DBG_CTL2_TIMER50_HOLD               BIT(10)                                            /*!< debug TIMER50 kept when core is halted */
#define SYS_DBG_CTL2_TIMER51_HOLD               BIT(11)                                            /*!< debug TIMER51 kept when core is halted */
#define SYS_DBG_CTL2_I2C0_HOLD                  BIT(21)                                            /*!< debug I2C0 kept when core is halted */
#define SYS_DBG_CTL2_I2C1_HOLD                  BIT(22)                                            /*!< debug I2C1 kept when core is halted */
#define SYS_DBG_CTL2_I2C2_HOLD                  BIT(23)                                            /*!< debug I2C2 kept when core is halted */
#define SYS_DBG_CTL2_I2C3_HOLD                  BIT(24)                                            /*!< debug I2C3 kept when core is halted */

/* SYS_DBG_CTL3 */
#define SYS_DBG_CTL3_TIMER0_HOLD                BIT(0)                                             /*!< debug TIMER0 kept when core is halted */
#define SYS_DBG_CTL3_TIMER7_HOLD                BIT(1)                                             /*!< debug TIMER7 kept when core is halted */
#define SYS_DBG_CTL3_CAN0_HOLD                  BIT(2)                                             /*!< debug CAN0 kept when core is halted */
#define SYS_DBG_CTL3_CAN1_HOLD                  BIT(3)                                             /*!< debug CAN1 kept when core is halted */
#define SYS_DBG_CTL3_CAN2_HOLD                  BIT(4)                                             /*!< debug CAN2 kept when core is halted */
#define SYS_DBG_CTL3_TIMER14_HOLD               BIT(16)                                            /*!< debug TIMER14 kept when core is halted */
#define SYS_DBG_CTL3_TIMER15_HOLD               BIT(17)                                            /*!< debug TIMER15 kept when core is halted */
#define SYS_DBG_CTL3_TIMER16_HOLD               BIT(18)                                            /*!< debug TIMER16 kept when core is halted */
#define SYS_DBG_CTL3_TIMER40_HOLD               BIT(19)                                            /*!< debug TIMER40 kept when core is halted */
#define SYS_DBG_CTL3_TIMER41_HOLD               BIT(20)                                            /*!< debug TIMER41 kept when core is halted */
#define SYS_DBG_CTL3_TIMER42_HOLD               BIT(21)                                            /*!< debug TIMER42 kept when core is halted */
#define SYS_DBG_CTL3_TIMER43_HOLD               BIT(22)                                            /*!< debug TIMER43 kept when core is halted */
#define SYS_DBG_CTL3_TIMER44_HOLD               BIT(23)                                            /*!< debug TIMER44 kept when core is halted */

/* SYS_DBG_CTL4 */
#define SYS_DBG_CTL4_RTC_HOLD                   BIT(16)                                            /*!< debug RTC kept when core is halted */
#define SYS_DBG_CTL4_FWDGT_HOLD                 BIT(18)                                            /*!< debug FWDGT kept when core is halted */

/* constants definitions */
#define SYS_DBG_LOW_POWER_SLEEP                 SYS_DBG_CTL0_SLP_HOLD                              /*!< keep debugger connection during sleep mode */
#define SYS_DBG_LOW_POWER_DEEPSLEEP             SYS_DBG_CTL0_DSLP_HOLD                             /*!< keep debugger connection during deepsleep mode */
#define SYS_DBG_LOW_POWER_STANDBY               SYS_DBG_CTL0_STB_HOLD                              /*!< keep debugger connection during standby mode */

/* Bit definition for SYS_DBG_CTL1 SYS_DBG_CTL2 SYS_DBG_CTL3 SYS_DBG_CTL4 register */
/* define the peripheral debug hold bit position and its register index offset */
#define SYS_DBG_REGIDX_BIT(regidx, bitpos)      (((regidx) << 6) | (bitpos))
#define SYS_DBG_REG_VAL(periph)                 (REG32(SYS_DBG + ((uint32_t)(periph) >> 6)))
#define SYS_DBG_BIT_POS(val)                    ((uint32_t)(val) & 0x0000001FU)

/* Bit definition for SYS_DBG_CTL0 register */
#define CTL0_TRACE_MODE(regval)                 (BITS(18,19) & ((uint32_t)(regval) << 18U))

/* @PARA: trace_mode */
/* @DEFINE: DBG Trace Mode */
#define SYS_DBG_TRACE_ASYNC                     CTL0_TRACE_MODE(0)                                 /*!< trace pin used for async mode */
#define SYS_DBG_TRACE_SYNC_DATASIZE_1           CTL0_TRACE_MODE(1)                                 /*!< trace pin used for sync mode and data size is 1 */
#define SYS_DBG_TRACE_SYNC_DATASIZE_2           CTL0_TRACE_MODE(2)                                 /*!< trace pin used for sync mode and data size is 2 */
#define SYS_DBG_TRACE_SYNC_DATASIZE_4           CTL0_TRACE_MODE(3)                                 /*!< trace pin used for sync mode and data size is 4 */

/* register offset */
#define SYS_DBG_IDX_CTL1_REG_OFFSET             ((uint32_t)0x00000034U)	                           /*!< DBG control register 1 offset */
#define SYS_DBG_IDX_CTL2_REG_OFFSET             ((uint32_t)0x0000003CU)                            /*!< DBG control register 2 offset */
#define SYS_DBG_IDX_CTL3_REG_OFFSET             ((uint32_t)0x0000004CU)                            /*!< DBG control register 3 offset */
#define SYS_DBG_IDX_CTL4_REG_OFFSET             ((uint32_t)0x00000054U)                            /*!< DBG control register 4 offset */

/* tick freq value */
#define HAL_TICK_FREQ_1KHZ                      (1000U)                                            /*!< tick freq value 1KHz */
#define HAL_TICK_FREQ_100HZ                     (100U)                                             /*!< tick freq value 100Hz */

/* choose the Systick clock source */
#define SYSTICK_CLKSOURCE_CKSYS_DIV8            ((uint32_t)0xFFFFFFFBU)                            /*!< Systick clock source is from CK_SYS/8 */
#define SYSTICK_CLKSOURCE_CKSYS                 ((uint32_t)0x00000004U)                            /*!< Systick clock source is from CK_SYS */

/* @STRUCT_MEMBER: region_number */
/* @DEFINE: MPU region number */
#define MPU_REGION_NUMBER0                      ((uint8_t)0x00U)                                   /*!< MPU region number 0 */
#define MPU_REGION_NUMBER1                      ((uint8_t)0x01U)                                   /*!< MPU region number 1 */
#define MPU_REGION_NUMBER2                      ((uint8_t)0x02U)                                   /*!< MPU region number 2 */
#define MPU_REGION_NUMBER3                      ((uint8_t)0x03U)                                   /*!< MPU region number 3 */
#define MPU_REGION_NUMBER4                      ((uint8_t)0x04U)                                   /*!< MPU region number 4 */
#define MPU_REGION_NUMBER5                      ((uint8_t)0x05U)                                   /*!< MPU region number 5 */
#define MPU_REGION_NUMBER6                      ((uint8_t)0x06U)                                   /*!< MPU region number 6 */
#define MPU_REGION_NUMBER7                      ((uint8_t)0x07U)                                   /*!< MPU region number 7 */
#define MPU_REGION_NUMBER8                      ((uint8_t)0x08U)                                   /*!< MPU region number 8 */
#define MPU_REGION_NUMBER9                      ((uint8_t)0x09U)                                   /*!< MPU region number 9 */
#define MPU_REGION_NUMBER10                     ((uint8_t)0x0AU)                                   /*!< MPU region number 10 */
#define MPU_REGION_NUMBER11                     ((uint8_t)0x0BU)                                   /*!< MPU region number 11 */
#define MPU_REGION_NUMBER12                     ((uint8_t)0x0CU)                                   /*!< MPU region number 12 */
#define MPU_REGION_NUMBER13                     ((uint8_t)0x0DU)                                   /*!< MPU region number 13 */
#define MPU_REGION_NUMBER14                     ((uint8_t)0x0EU)                                   /*!< MPU region number 14 */
#define MPU_REGION_NUMBER15                     ((uint8_t)0x0FU)                                   /*!< MPU region number 15 */

/* @STRUCT_MEMBER: region_size */
/* @DEFINE: MPU region size */
#define MPU_REGION_SIZE_32B                     ARM_MPU_REGION_SIZE_32B                            /*!< MPU region size is 32 bytes, the smallest supported region size is 32 bytes */
#define MPU_REGION_SIZE_64B                     ARM_MPU_REGION_SIZE_64B                            /*!< MPU region size is 64 bytes */
#define MPU_REGION_SIZE_128B                    ARM_MPU_REGION_SIZE_128B                           /*!< MPU region size is 128 bytes */
#define MPU_REGION_SIZE_256B                    ARM_MPU_REGION_SIZE_256B                           /*!< MPU region size is 256 bytes */
#define MPU_REGION_SIZE_512B                    ARM_MPU_REGION_SIZE_512B                           /*!< MPU region size is 512 bytes */
#define MPU_REGION_SIZE_1KB                     ARM_MPU_REGION_SIZE_1KB                            /*!< MPU region size is 1K bytes */
#define MPU_REGION_SIZE_2KB                     ARM_MPU_REGION_SIZE_2KB                            /*!< MPU region size is 2K bytes */
#define MPU_REGION_SIZE_4KB                     ARM_MPU_REGION_SIZE_4KB                            /*!< MPU region size is 4K bytes */
#define MPU_REGION_SIZE_8KB                     ARM_MPU_REGION_SIZE_8KB                            /*!< MPU region size is 8K bytes */
#define MPU_REGION_SIZE_16KB                    ARM_MPU_REGION_SIZE_16KB                           /*!< MPU region size is 16K bytes */
#define MPU_REGION_SIZE_32KB                    ARM_MPU_REGION_SIZE_32KB                           /*!< MPU region size is 32K bytes */
#define MPU_REGION_SIZE_64KB                    ARM_MPU_REGION_SIZE_64KB                           /*!< MPU region size is 64K bytes */
#define MPU_REGION_SIZE_128KB                   ARM_MPU_REGION_SIZE_128KB                          /*!< MPU region size is 128K bytes */
#define MPU_REGION_SIZE_256KB                   ARM_MPU_REGION_SIZE_256KB                          /*!< MPU region size is 256K bytes */
#define MPU_REGION_SIZE_512KB                   ARM_MPU_REGION_SIZE_512KB                          /*!< MPU region size is 512K bytes */
#define MPU_REGION_SIZE_1MB                     ARM_MPU_REGION_SIZE_1MB                            /*!< MPU region size is 1M bytes */
#define MPU_REGION_SIZE_2MB                     ARM_MPU_REGION_SIZE_2MB                            /*!< MPU region size is 2M bytes */
#define MPU_REGION_SIZE_4MB                     ARM_MPU_REGION_SIZE_4MB                            /*!< MPU region size is 4M bytes */
#define MPU_REGION_SIZE_8MB                     ARM_MPU_REGION_SIZE_8MB                            /*!< MPU region size is 8M bytes */
#define MPU_REGION_SIZE_16MB                    ARM_MPU_REGION_SIZE_16MB                           /*!< MPU region size is 16M bytes */
#define MPU_REGION_SIZE_32MB                    ARM_MPU_REGION_SIZE_32MB                           /*!< MPU region size is 32M bytes */
#define MPU_REGION_SIZE_64MB                    ARM_MPU_REGION_SIZE_64MB                           /*!< MPU region size is 64M bytes */
#define MPU_REGION_SIZE_128MB                   ARM_MPU_REGION_SIZE_128MB                          /*!< MPU region size is 128M bytes */
#define MPU_REGION_SIZE_256MB                   ARM_MPU_REGION_SIZE_256MB                          /*!< MPU region size is 256M bytes */
#define MPU_REGION_SIZE_512MB                   ARM_MPU_REGION_SIZE_512MB                          /*!< MPU region size is 512M bytes */
#define MPU_REGION_SIZE_1GB                     ARM_MPU_REGION_SIZE_1GB                            /*!< MPU region size is 1G bytes */
#define MPU_REGION_SIZE_2GB                     ARM_MPU_REGION_SIZE_2GB                            /*!< MPU region size is 2G bytes */
#define MPU_REGION_SIZE_4GB                     ARM_MPU_REGION_SIZE_4GB                            /*!< MPU region size is 4G bytes */

/* @STRUCT_MEMBER: subregion_disable */
/* @DEFINE: subregion status */
#define MPU_SUBREGION_ENABLE                    ((uint8_t)0x00U)                                   /*!< Subregion enable */
#define MPU_SUBREGION_DISABLE                   ((uint8_t)0x01U)                                   /*!< Subregion disable */

/* @STRUCT_MEMBER: tex_type */
/* @DEFINE: MPU TEX type */
#define MPU_TEX_TYPE0                           ((uint8_t)0x00U)                                   /*!< MPU TEX type 0 */
#define MPU_TEX_TYPE1                           ((uint8_t)0x01U)                                   /*!< MPU TEX type 1 */
#define MPU_TEX_TYPE2                           ((uint8_t)0x02U)                                   /*!< MPU TEX type 2 */

/* @STRUCT_MEMBER: access_permission */
/* @DEFINE: MPU access permission */
#define MPU_AP_NO_ACCESS                        ARM_MPU_AP_NONE                                    /*!< MPU access permission no access */
#define MPU_AP_PRIV_RW                          ARM_MPU_AP_PRIV                                    /*!< MPU access permission privileged access only */
#define MPU_AP_PRIV_RW_UNPRIV_RO                ARM_MPU_AP_URO                                     /*!< MPU access permission unprivileged access read-only */
#define MPU_AP_FULL_ACCESS                      ARM_MPU_AP_FULL                                    /*!< MPU access permission full access */
#define MPU_AP_PRIV_RO                          ARM_MPU_AP_PRO                                     /*!< MPU access permission privileged access read-only */
#define MPU_AP_PRIV_UNPRIV_RO                   ARM_MPU_AP_RO                                      /*!< MPU access permission privileged and unprivileged read-only access */

/* @STRUCT_MEMBER: access_shareable */
/* @DEFINE:  MPU access shareable */
#define MPU_ACCESS_SHAREABLE                    ((uint8_t)0x01U)                                   /*!< MPU access shareable */
#define MPU_ACCESS_NON_SHAREABLE                ((uint8_t)0x00U)                                   /*!< MPU access non-shareable */

/* @STRUCT_MEMBER: access_cacheable */
/* @DEFINE:  MPU access cacheable */
#define MPU_ACCESS_CACHEABLE                    ((uint8_t)0x01U)                                   /*!< MPU access cacheable */
#define MPU_ACCESS_NON_CACHEABLE                ((uint8_t)0x00U)                                   /*!< MPU access non-cacheable */

/* @STRUCT_MEMBER: access_bufferable */
/* @DEFINE:  MPU access bufferable */
#define MPU_ACCESS_BUFFERABLE                   ((uint8_t)0x01U)                                   /*!< MPU access bufferable */
#define MPU_ACCESS_NON_BUFFERABLE               ((uint8_t)0x00U)                                   /*!< MPU access non-bufferable */

/* @STRUCT_MEMBER: instruction_exec */
/* @DEFINE:  MPU execution of an instruction */
#define MPU_INSTRUCTION_EXEC_PERMIT             ((uint8_t)0x00U)                                   /*!< execution of an instruction fetched from this region permitted */
#define MPU_INSTRUCTION_EXEC_NOT_PERMIT         ((uint8_t)0x01U)                                   /*!< execution of an instruction fetched from this region not permitted */

/* @PARA: mpu_control */
/* @DEFINE:  MPU control mode */
#define MPU_MODE_HFNMI_PRIVDEF_NONE             ((uint32_t)0x00000000U)                            /*!< HFNMIENA and PRIVDEFENA are 0 */
#define MPU_MODE_HARDFAULT_NMI                  MPU_CTRL_HFNMIENA_Msk                              /*!< use the MPU for memory accesses by HardFault and NMI handlers only */
#define MPU_MODE_PRIV_DEFAULT                   MPU_CTRL_PRIVDEFENA_Msk                            /*!< enables the default memory map as a background region for privileged access only */
#define MPU_MODE_HFNMI_PRIVDEF                  ((uint32_t)MPU_CTRL_HFNMIENA_Msk | \
                                                 MPU_CTRL_PRIVDEFENA_Msk)                          /*!< HFNMIENA and PRIVDEFENA are 1 */

#define BASETICK_SOURCE_TIMERX_RST  \
{   RCU_TIMER0RST,  RCU_TIMER1RST,  RCU_TIMER2RST,  RCU_TIMER3RST,  \
    RCU_TIMER4RST,  RCU_TIMER5RST,  RCU_TIMER6RST,  RCU_TIMER7RST,  \
    RCU_TIMER14RST,  RCU_TIMER15RST,  RCU_TIMER16RST,  RCU_TIMER22RST,  \
    RCU_TIMER23RST,  RCU_TIMER30RST,  RCU_TIMER31RST,  RCU_TIMER40RST,  \
    RCU_TIMER41RST,  RCU_TIMER42RST,  RCU_TIMER43RST,  RCU_TIMER44RST,  \
    RCU_TIMER50RST,  RCU_TIMER51RST }

#define BASETICK_SOURCE_TIMERX_CLK  \
{   RCU_TIMER0,  RCU_TIMER1,  RCU_TIMER2,  RCU_TIMER3,  \
    RCU_TIMER4,  RCU_TIMER5,  RCU_TIMER6,  RCU_TIMER7,  \
    RCU_TIMER14,  RCU_TIMER15,  RCU_TIMER16,  RCU_TIMER22,  \
    RCU_TIMER23,  RCU_TIMER30,  RCU_TIMER31,  RCU_TIMER40,  \
    RCU_TIMER41,  RCU_TIMER42,  RCU_TIMER43,  RCU_TIMER44,  \
    RCU_TIMER50,  RCU_TIMER51 }

#define BASETICK_SOURCE_TIMERX_PERIPH  \
{   TIMER0,  TIMER1,  TIMER2,  TIMER3,  \
    TIMER4,  TIMER5,  TIMER6,  TIMER7,  \
    TIMER14, TIMER15, TIMER16, TIMER22,  \
    TIMER23, TIMER30, TIMER31, TIMER40,  \
    TIMER41, TIMER42, TIMER43, TIMER44,  \
    TIMER50, TIMER51 }

#define BASETICK_SOURCE_TIMERX_IRQN  \
{   TIMER0_UP_IRQn, TIMER1_IRQn, TIMER2_IRQn, TIMER3_IRQn, \
    TIMER4_IRQn, TIMER5_DAC_UDR_IRQn, TIMER6_IRQn, TIMER7_UP_IRQn, \
    TIMER14_IRQn, TIMER15_IRQn, TIMER16_IRQn, TIMER22_IRQn, \
    TIMER23_IRQn, TIMER30_IRQn, TIMER31_IRQn, TIMER40_IRQn, \
    TIMER41_IRQn, TIMER42_IRQn, TIMER43_IRQn, TIMER44_IRQn, \
    TIMER50_IRQn, TIMER51_IRQn}

/* @PARA: debug_cfg */
/* @ENUM: debug selection */
typedef enum {
    SYS_DEBUG_DISABLE = 0U,                                                                        /*!< Debug disable */
    SYS_DEBUG_SERIAL_WIRE                                                                          /*!< Serial wire debug */
} hal_sys_debug_cfg_enum;

/* peripherals hold bit */
typedef enum {
    SYS_DBG_WWDGT_HOLD   = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL1_REG_OFFSET, 6U),                    /*!< debug WWDGT kept when core is halted */
    SYS_DBG_I2C3_HOLD    = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 24U),                   /*!< debug I2C3 kept when core is halted */
    SYS_DBG_I2C2_HOLD    = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 23U),                   /*!< debug I2C2 kept when core is halted */
    SYS_DBG_I2C1_HOLD    = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 22U),                   /*!< debug I2C1 kept when core is halted */
    SYS_DBG_I2C0_HOLD    = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 21U),                   /*!< debug I2C0 kept when core is halted */
    SYS_DBG_TIMER51_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 11U),                   /*!< debug TIMER51 kept when core is halted */
    SYS_DBG_TIMER50_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 10U),                   /*!< debug TIMER50 kept when core is halted */
    SYS_DBG_TIMER31_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 9U),                    /*!< debug TIMER31 kept when core is halted */
    SYS_DBG_TIMER30_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 8U),                    /*!< debug TIMER30 kept when core is halted */
    SYS_DBG_TIMER23_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 7U),                    /*!< debug TIMER23 kept when core is halted */
    SYS_DBG_TIMER22_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 6U),                    /*!< debug TIMER22 kept when core is halted */
    SYS_DBG_TIMER6_HOLD  = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 5U),                    /*!< debug TIMER6 kept when core is halted */
    SYS_DBG_TIMER5_HOLD  = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 4U),                    /*!< debug TIMER5 kept when core is halted */
    SYS_DBG_TIMER4_HOLD  = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 3U),                    /*!< debug TIMER4 kept when core is halted */
    SYS_DBG_TIMER3_HOLD  = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 2U),                    /*!< debug TIMER3 kept when core is halted */
    SYS_DBG_TIMER2_HOLD  = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 1U),                    /*!< debug TIMER2 kept when core is halted */
    SYS_DBG_TIMER1_HOLD  = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL2_REG_OFFSET, 0U),                    /*!< debug TIMER1 kept when core is halted */
    SYS_DBG_TIMER44_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 23U),                   /*!< debug TIMER44 kept when core is halted */
    SYS_DBG_TIMER43_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 22U),                   /*!< debug TIMER43 kept when core is halted */
    SYS_DBG_TIMER42_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 21U),                   /*!< debug TIMER42 kept when core is halted */
    SYS_DBG_TIMER41_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 20U),                   /*!< debug TIMER41 kept when core is halted */
    SYS_DBG_TIMER40_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 19U),                   /*!< debug TIMER40 kept when core is halted */
    SYS_DBG_TIMER16_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 18U),                   /*!< debug TIMER16 kept when core is halted */
    SYS_DBG_TIMER15_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 17U),                   /*!< debug TIMER15 kept when core is halted */
    SYS_DBG_TIMER14_HOLD = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 16U),                   /*!< debug TIMER14 kept when core is halted */
    SYS_DBG_CAN2_HOLD    = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 4U),                    /*!< debug CAN2 kept when core is halted */
    SYS_DBG_CAN1_HOLD    = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 3U),                    /*!< debug CAN1 kept when core is halted */
    SYS_DBG_CAN0_HOLD    = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 2U),                    /*!< debug CAN0 kept when core is halted */
    SYS_DBG_TIMER7_HOLD  = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 1U),                    /*!< debug TIMER7 kept when core is halted */
    SYS_DBG_TIMER0_HOLD  = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL3_REG_OFFSET, 0U),                    /*!< debug TIMER0 kept when core is halted */
    SYS_DBG_FWDGT_HOLD   = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL4_REG_OFFSET, 18U),                   /*!< debug FWDGT kept when core is halted */
    SYS_DBG_RTC_HOLD     = SYS_DBG_REGIDX_BIT(SYS_DBG_IDX_CTL4_REG_OFFSET, 16U)                    /*!< debug RTC kept when core is halted */
} hal_sys_dbg_periph_enum;

/* @PARA: timebase_source */
/* @ENUM: SYS timebase source */
typedef enum {
    SYS_TIMEBASE_SOURCE_SYSTICK = 0U,                                                              /*!< Select SysTick as timebase source */
    SYS_TIMEBASE_SOURCE_SYSTICK_DIV8,                                                              /*!< Select SysTick/8 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER0,                                                                    /*!< Select TIMER0  as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER1,                                                                    /*!< Select TIMER1  as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER2,                                                                    /*!< Select TIMER2  as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER3,                                                                    /*!< Select TIMER3  as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER4,                                                                    /*!< Select TIMER4  as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER5,                                                                    /*!< Select TIMER5  as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER6,                                                                    /*!< Select TIMER6  as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER7,                                                                    /*!< Select TIMER7  as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER14,                                                                   /*!< Select TIMER14 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER15,                                                                   /*!< Select TIMER15 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER16,                                                                   /*!< Select TIMER16 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER22,                                                                   /*!< Select TIMER22 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER23,                                                                   /*!< Select TIMER23 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER30,                                                                   /*!< Select TIMER30 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER31,                                                                   /*!< Select TIMER31 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER40,                                                                   /*!< Select TIMER40 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER41,                                                                   /*!< Select TIMER41 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER42,                                                                   /*!< Select TIMER42 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER43,                                                                   /*!< Select TIMER43 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER44,                                                                   /*!< Select TIMER44 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER50,                                                                   /*!< Select TIMER50 as timebase source */
    SYS_TIMEBASE_SOURCE_TIMER51                                                                    /*!< Select TIMER51 as timebase source */
} hal_sys_timebase_source_enum;

/* @PARA: mpu_init_struct */
/* @STRUCT: MPU region init parameter struct definitions */
typedef struct {
    uint32_t region_base_address;                                                                  /*!< region base address */
    uint8_t  region_number;                                                                        /*!< region number */
    uint8_t  region_size;                                                                          /*!< region size */
    uint8_t  subregion_disable;                                                                    /*!< subregion disable */
    uint8_t  tex_type;                                                                             /*!< tex type */
    uint8_t  access_permission;                                                                    /*!< access permissions(AP) field */
    uint8_t  access_shareable;                                                                     /*!< shareable */
    uint8_t  access_cacheable;                                                                     /*!< cacheable */
    uint8_t  access_bufferable;                                                                    /*!< bufferable */
    uint8_t  instruction_exec;                                                                     /*!< execute never */
} hal_mpu_region_init_struct;

/* the callback of basetick interrupt declaration */
typedef void (*hal_sys_basetick_irq_handle_cb)(void);
/* the callback of system reset periph clock */
typedef void (*hal_system_cb)(void);

/* function declarations */
/* deinitialize SYS */
int32_t hal_sys_deinit(void);
/* @FUNCTION: initialize SYS debug configuration */
int32_t hal_sys_debug_init(hal_sys_debug_cfg_enum debug_cfg);
/* @FUNCTION: initialize SYS timebase source */
int32_t hal_sys_timesource_init(hal_sys_timebase_source_enum timebase_source);
/* @FUNCTION: initialize mpu_region_init_struct with the default values */
/* deinitialize MPU struct */
int32_t hal_mpu_region_struct_init(hal_mpu_region_init_struct *mpu_init_struct);
/* @FUNCTION: configure the MPU region */
int32_t hal_mpu_region_config(hal_mpu_region_init_struct *mpu_init_struct);
/* @FUNCTION:  enable the MPU region */
void hal_mpu_region_enable(uint32_t region_number);
/* @FUNCTION:  disable the MPU region */
void hal_mpu_region_disable(uint32_t region_number);
/* @FUNCTION:  enable the MPU */
void hal_mpu_enable(uint32_t mpu_control);
/* @FUNCTION:  disable the MPU */
void hal_mpu_disable(void);
/* @FUNCTION: configure the DBG */
void hal_sys_dbg_trace_pin_mode_set(uint32_t trace_mode);
/* enable trace pin assignment */
void hal_sys_dbg_trace_pin_enable(void);
/* disable trace pin assignment */
void hal_sys_dbg_trace_pin_disable(void);
/*@FUNCTION: enable i-cache */
void hal_sys_icache_enable(void);
/* disable i-cache */
void hal_sys_icache_disable(void);
/*@FUNCTION: enable d-cache */
void hal_sys_dcache_enable(void);
/* disable d-cache */
void hal_sys_dcache_disable(void);
/* @END */

/* timeout and delay functions */
/* get the basetick count */
uint32_t hal_sys_basetick_count_get(void);
/* check whether the delay is finished */
FlagStatus hal_sys_basetick_timeout_check(uint32_t time_start, uint32_t delay);
/* set the basetick delay */
void hal_sys_basetick_delay_ms(uint32_t time_ms);

/*set the systick clock source */
int32_t hal_sys_tick_clksource_set(uint32_t systick_clksource);
/* set new tick freq value */
int32_t hal_sys_tick_freq_set(uint32_t tick_freq);
/* get priority of basetick timer */
uint8_t hal_sys_basetick_prio_get(void);
/* get current tick freq value */
uint32_t hal_sys_tick_freq_get(void);

/* control functions */
/* suspend the basetick timer */
void hal_sys_basetick_suspend(void);
/* resume the basetick timer */
void hal_sys_basetick_resume(void);

/* system deinit */
void hal_system_deinit(hal_system_cb system_handler);

/* interrupt functions */
/* basetick interrupt handler content function, which is merely used in SysTick_Handler or TIMERx_UP_IRQHandler */
void hal_sys_basetick_irq(void);
/* set user-defined interrupt callback function, which will be registered and called when corresponding interrupt be triggered */
int32_t hal_sys_basetick_irq_handle_set(hal_sys_basetick_irq_handle_cb irq_handler);
/* reset all user-defined interrupt callback function, which will be registered and called when corresponding interrupt be triggered */
void hal_sys_basetick_irq_handle_all_reset(void);

/* Initialization and de-initialization functions */
/* deinitialize the DBG */
void hals_sys_dbg_deinit(void);

/* read DBG_ID code register */
uint32_t hals_sys_dbg_id_get(void);

/* enable low power behavior when the mcu is in debug mode */
void hals_sys_dbg_low_power_enable(uint32_t dbg_low_power);
/* disable low power behavior when the mcu is in debug mode */
void hals_sys_dbg_low_power_disable(uint32_t dbg_low_power);

/* enable peripheral behavior when the mcu is in debug mode */
void hals_sys_dbg_periph_enable(hal_sys_dbg_periph_enum dbg_periph);
/* disable peripheral behavior when the mcu is in debug mode */
void hals_sys_dbg_periph_disable(hal_sys_dbg_periph_enum dbg_periph);

#endif /* GD32H7XX_HAL_SYS_H */
