/*!
    \file    gd32h7xx_hal_smbus.h
    \brief   definitions for the SMBUS

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_SMBUS_H
#define GD32H7XX_HAL_SMBUS_H

#include "gd32h7xx_hal.h"

/* constants definitions */
/* @STRUCT_MEMBER: scl_stretch */
/* @DEFINE: whether to stretch SCL low when data is not ready in slave mode */
#define SMBUS_SCLSTRETCH_ENABLE                   I2C_SCLSTRETCH_ENABLE                                         /*!< SCL stretching is enabled */
#define SMBUS_SCLSTRETCH_DISABLE                  I2C_SCLSTRETCH_DISABLE                                        /*!< SCL stretching is disabled */

/* @STRUCT_MEMBER: address_format */
/* @DEFINE: SMBUS addformat, 7bits or 10bits */
#define SMBUS_ADDFORMAT_7BITS                     I2C_ADDFORMAT_7BITS                                           /*!< address:7 bits */
#define SMBUS_ADDFORMAT_10BITS                    I2C_ADDFORMAT_10BITS                                          /*!< address:10 bits */

/* @STRUCT_MEMBER: own_address1 */
/* @=NULL */

/* @STRUCT_MEMBER: dual_address */
/* @DEFINE: dual-address mode switch */
#define SMBUS_DUADEN_DISABLE                      I2C_DUADEN_DISABLE                                            /*!< dual-address mode disabled */
#define SMBUS_DUADEN_ENABLE                       I2C_DUADEN_ENABLE                                             /*!< dual-address mode enabled */

/* @STRUCT_MEMBER: own_address2 */
/* @=NULL */

/* @STRUCT_MEMBER: address2_mask */
/* defines which bits of ADDRESS2[7:1] are compared with an incoming address byte, and which bits are masked (do not care) */
#define ADDRESS2_NO_MASK                          ((uint32_t)0x00000000U)                                       /*!< no mask, all the bits must be compared */
#define ADDRESS2_MASK_BIT1                        ((uint32_t)0x00000001U)                                       /*!< ADDRESS2[1] is masked, only ADDRESS2[7:2] are compared */
#define ADDRESS2_MASK_BIT1_2                      ((uint32_t)0x00000002U)                                       /*!< ADDRESS2[2:1] is masked, only ADDRESS2[7:3] are compared */
#define ADDRESS2_MASK_BIT1_3                      ((uint32_t)0x00000003U)                                       /*!< ADDRESS2[3:1] is masked, only ADDRESS2[7:4] are compared */
#define ADDRESS2_MASK_BIT1_4                      ((uint32_t)0x00000004U)                                       /*!< ADDRESS2[4:1] is masked, only ADDRESS2[7:5] are compared */
#define ADDRESS2_MASK_BIT1_5                      ((uint32_t)0x00000005U)                                       /*!< ADDRESS2[5:1] is masked, only ADDRESS2[7:6] are compared */
#define ADDRESS2_MASK_BIT1_6                      ((uint32_t)0x00000006U)                                       /*!< ADDRESS2[6:1] is masked, only ADDRESS2[7] are compared */
#define ADDRESS2_MASK_ALL                         ((uint32_t)0x00000007U)                                       /*!< all the ADDRESS2[7:1] bits are masked */

/* @STRUCT_MEMBER: general_call */
/* @DEFINE: whether or not to response to a general call */
#define SMBUS_GCEN_ENABLE                         I2C_GCEN_ENABLE                                               /*!< slave will response to a general call */
#define SMBUS_GCEN_DISABLE                        I2C_GCEN_DISABLE                                              /*!< slave will not response to a general call */

/* @STRUCT_MEMBER: smbus_type */
/* @DEFINE: SMBus type */
#define SMBUS_HOST                                I2C_CTL0_SMBHAEN                                              /*!< SMBus mode host type */
#define SMBUS_SLAVE                               I2C_CTL0_I2CEN                                                /*!< SMBus mode slave type */
#define SMBUS_SLAVE_ARP                           I2C_CTL0_SMBDAEN                                              /*!< SMBus mode slave arp type */

/* PEC enable */
#define I2C_PEC_ENABLE                            I2C_CTL0_PECEN                                                /*!< PEC calculation on */
#define I2C_PEC_DISABLE                           ((uint32_t)0x00000000U)                                       /*!< PEC calculation off */

/* @STRUCT_MEMBER: smbus_pec */
/* @DEFINE: SMBus PEC calculation enable */
#define SMBUS_PEC_ENABLE                          I2C_PEC_ENABLE                                                /*!< I2C is under reset */
#define SMBUS_PEC_DISABLE                         I2C_PEC_DISABLE                                               /*!< I2C is not under reset */

/* SMBus Extended clock timeout enable */
#define SMBUS_EXCLK_TIMEOUT_ENABLE                I2C_TIMEOUT_EXTOEN                                            /*!< SMBus Extended clock timeout enable */
#define SMBUS_EXCLK_TIMEOUT_DISABLE               ((uint32_t)0x00000000U)                                       /*!< SMBus Extended clock timeout disable */

/* SMBus timeout enable */
#define SMBUS_CLK_TIMEOUT_ENABLE                  I2C_TIMEOUT_TOEN                                              /*!< SMBus timeout enable */
#define SMBUS_CLK_TIMEOUT_DISABLE                 ((uint32_t)0x00000000U)                                       /*!< SMBus timeout disable */

/* SMBus timeout enable */
#define SMBUS_IDLE_CLK_TIMEOUT_ENABLE             I2C_TIMEOUT_TOIDLE                                            /*!< SMBus idle clock timeout detect enable */
#define SMBUS_IDLE_CLK_TIMEOUT_DISABLE            ((uint32_t)0x00000000U)                                       /*!< SMBus idle clock timeout detect disable */

/* PEC transfer */
#define I2C_PECTRANS_ENABLE                       I2C_CTL0_PECTRANS                                             /*!< transfer PEC */
#define I2C_PECTRANS_DISABLE                      ((uint32_t)0x00000000U)                                       /*!< not transfer PEC value */

/* PEC transfer */
#define SMBUS_PECTRANS_ENABLE                     I2C_PECTRANS_ENABLE                                           /*!< transfer PEC */
#define SMBUS_PECTRANS_DISABLE                    I2C_PECTRANS_DISABLE                                          /*!< not transfer PEC value */

/* whether or not to send an ACK */
#define SMBUS_ACK_DISABLE                         I2C_ACK_DISABLE                                               /*!< ACK will be not sent */
#define SMBUS_ACK_ENABLE                          I2C_ACK_ENABLE                                                /*!< ACK will be sent */

/* SMBUS error code */
#define HAL_SMBUS_ERROR_NONE                      ((uint32_t)0x00000000U)                                       /*!< no error */
#define HAL_SMBUS_ERROR_BERR                      BIT(0)                                                        /*!< bus error */
#define HAL_SMBUS_ERROR_LOSTARB                   BIT(1)                                                        /*!< arbitration lost in master mode */
#define HAL_SMBUS_ERROR_OUERR                     BIT(2)                                                        /*!< over-run error */
#define HAL_SMBUS_ERROR_PECERR                    BIT(3)                                                        /*!< PEC error */
#define HAL_SMBUS_ERROR_SMBTO                     BIT(4)                                                        /*!< bus timeout */
#define HAL_SMBUS_ERROR_SMBALT                    BIT(5)                                                        /*!< alert */
#define HAL_SMBUS_ERROR_DMATX                     BIT(6)                                                        /*!< DMA TX error */
#define HAL_SMBUS_ERROR_DMARX                     BIT(7)                                                        /*!< DMA RX error */
#define HAL_SMBUS_ERROR_ADDRESS                   BIT(8)                                                        /*!< address error */

/* smbus start/stop mode */
#define SMBUS_NO_STARTSTOP                       ((uint32_t)0x00000000U)                                        /* smbus no start and stop signal */
#define SMBUS_GENERATE_STOP                      (uint32_t)(0x80000000U | I2C_CTL1_STOP)                        /* smbus generate stop signal */
#define SMBUS_GENERATE_START_READ                (uint32_t)(0x80000000U | I2C_CTL1_START | I2C_CTL1_TRDIR)      /* smbus generate start signal in master read mode */
#define SMBUS_GENERATE_START_WRITE               (uint32_t)(0x80000000U | I2C_CTL1_START)                       /* smbus generate start signal in master write mode */

/* smbus relation option mode */
#define SMBUS_SOFTEND_MODE                       ((uint32_t)0x00000000U)                                        /* smbus software mode */
#define SMBUS_RELOAD_MODE                        I2C_CTL1_RELOAD                                                /* smbus reload mode */
#define SMBUS_AUTOEND_MODE                       I2C_CTL1_AUTOEND                                               /* smbus auto_finish mode */
#define SMBUS_SENDPEC_MODE                       I2C_CTL1_PECTRANS                                              /* smbus transfer pec mode */

/* smbus timeout value */
#define SMBUS_BUSY_TIMEOUT                        ((uint32_t)10000U)                                            /* 10s timeout */
#define SMBUS_TIMEOUT                             ((uint32_t)100000U)                                           /* 100s timeout */
#define SMBUS_ADDR_TIMEOUT                        ((uint32_t)10000U)                                            /* 10s timeout */

/* SMBUS Extended Fast Mode Plus */
#define SMBUS_FMP_NOT_SUPPORTED                   ((uint32_t)0xAAAA0000U)                                       /*!< Fast Mode Plus not supported */
#define SMBUS_FASTMODEPLUS_PB6                    SYSCFG_I2C_FMP_PB6                                            /*!< Enable Fast Mode Plus on PB6 */
#define SMBUS_FASTMODEPLUS_PB7                    SYSCFG_I2C_FMP_PB7                                            /*!< Enable Fast Mode Plus on PB7 */
#define SMBUS_FASTMODEPLUS_PB8                    SYSCFG_I2C_FMP_PB8                                            /*!< Enable Fast Mode Plus on PB8 */
#define SMBUS_FASTMODEPLUS_PB9                    SYSCFG_I2C_FMP_PB8                                            /*!< Enable Fast Mode Plus on PB9 */
#define SMBUS_FASTMODEPLUS_I2C1                   SYSCFG_I2C0_FMP                                               /*!< Enable Fast Mode Plus on I2C1 pins */
#define SMBUS_FASTMODEPLUS_I2C2                   SYSCFG_I2C1_FMP                                               /*!< Enable Fast Mode Plus on I2C2 pins */
#define SMBUS_FASTMODEPLUS_I2C3                   SYSCFG_I2C2_FMP                                               /*!< Enable Fast Mode Plus on I2C3 pins */
#define SMBUS_FASTMODEPLUS_I2C4                   SYSCFG_I2C3_FMP                                               /*!< Enable Fast Mode Plus on I2C4 pins */

typedef enum {
    HAL_SMBUS_STATE_READY = 0,                    /*!< smbus ready state */
    HAL_SMBUS_STATE_FREE,                         /*!< smbus free state */
    HAL_SMBUS_STATE_BUSY,                         /*!< smbus busy state */
    HAL_SMBUS_STATE_LISTEN                        /*!< smbus listen state */
} hal_smbus_run_state_enum;

typedef enum {
    HAL_SMBUS_INIT_STRUCT = 0,                    /*!< SMBUS initialize structure */
    HAL_SMBUS_DEV_STRUCT,                         /*!< SMBUS device information structure */
    HAL_SMBUS_IRQ_STRUCT,                         /*!< SMBUS device interrupt callback function pointer structure */
    HAL_SMBUS_BUFFER_STRUCT,                      /*!< SMBUS buffer structure */
    HAL_SMBUS_IRQ_USER_CALLBACK_STRUCT            /*!< SMBUS user callback function pointer structure */
} hal_smbus_struct_type_enum;

typedef struct {
    __IO hal_irq_handle_cb event_handle;          /*!< event handle */
    __IO hal_irq_handle_cb error_handle;          /*!< error handle */
} hal_smbus_irq_struct;

typedef struct {
    __IO uint8_t *buffer;                         /*!< buffer address */
    __IO uint32_t length;                         /*!< buffer length */
    __IO uint32_t pos;                            /*!< buffer position */
} hal_smbus_buffer_struct;

typedef struct {
    uint32_t                         periph;                    /*!< smbus port */
    hal_i2c_slave_address_struct     slave_address;             /*!< slave address */
    hal_smbus_irq_struct             smbus_irq;                 /*!< smbus interrupt request structure */
    hal_smbus_buffer_struct          txbuffer;                  /*!< smbus transmit buffer structure */
    hal_smbus_buffer_struct          rxbuffer;                  /*!< smbus receive buffer structure */
    uint32_t                         smbus_pec_transfer;        /*!< SMBus PEC transfer */
    __IO uint16_t                    last_error;                /*!< smbus last error */
    __IO uint32_t                    error_state;               /*!< smbus error state */
    __IO hal_smbus_run_state_enum    tx_state;                  /*!< smbus transmit state enum */
    __IO hal_smbus_run_state_enum    rx_state;                  /*!< smbus receive state enum */
    uint8_t                          transfer_option;           /*!< transfer option */
    void                             *rx_callback;              /*!< smbus receive callback function */
    void                             *tx_callback;              /*!< smbus transmit callback function */
    void                             *addr_callback;            /*!< address match callback function */
    void                             *priv;                     /*!< priv data */
    hal_mutex_enum                    mutex;                    /*!< mutex locked and unlocked state */
} hal_smbus_dev_struct;

/* @PARA: smbus_init */
/* @STRUCT: SMBUS init struct */
typedef struct {
    uint32_t                        scl_stretch;                /*!< whether to stretch SCL low when data is not ready in slave mode */
    uint32_t                        address_format;             /*!< SMBUS addformat, 7bits or 10bits */
    uint32_t                        own_address1;               /*!< SMBUS own address */
    uint32_t                        dual_address;               /*!< dual-address mode switch */
    uint32_t                        address2_mask;              /*!< Second Slave address mask */
    uint32_t                        own_address2;               /*!< I2C own address2 in dual-address mode */
    uint32_t                        general_call;               /*!< whether or not to response to a general call */
    uint32_t                        time;                       /*!< time register value */
    hal_i2c_digital_filter_enum     digital_filter;             /*!< digital filter */
    uint32_t                        analog_filter;              /*!< analog filter */
    uint32_t                        smbus_pec;                  /*!< SMBus PEC calculation enable */
    uint32_t                        smbus_type;                 /*!< SMBus type */
    uint32_t                        smbus_timeout;              /*!< SMBus timeout value */
} hal_smbus_init_struct;

/* SMBUS callback function */
typedef void (*hal_smbus_user_cb)(hal_smbus_dev_struct *smbus_dev);

/* SMBUS device interrupt user callback function pointer structure */
typedef struct {
    __IO hal_smbus_user_cb rx_callback;                         /*!< receive callback function */
    __IO hal_smbus_user_cb tx_callback;                         /*!< transmit callback function */
    __IO hal_smbus_user_cb addr_callback;                       /*!< address match callback function */
} hal_smbus_irq_user_callback_struct;

/* function declarations */
/* @FUNCTION: initialize the smbus structure with the default values */
/* deinitialize SMBUS */
int32_t hal_smbus_struct_init(hal_smbus_struct_type_enum struct_type, void *p_struct);
/* @FUNCTION: initialize smbus */
int32_t hal_smbus_init(hal_smbus_dev_struct *smbus_dev, uint32_t periph, hal_smbus_init_struct *smbus_init);
/* @FUNCTION: deinitialize smbus */
int32_t hal_smbus_deinit(hal_smbus_dev_struct *smbus_dev);
/* @END */

/* SMBUS interrupt handler content function,which is merely used in i2c_event_handler */
int32_t hal_smbus_event_irq(hal_smbus_dev_struct *smbus_dev);
/* SMBUS interrupt handler content function,which is merely used in i2c_error_handler */
int32_t hal_smbus_error_irq(hal_smbus_dev_struct *smbus_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_smbus_irq_handle_set(hal_smbus_dev_struct *smbus_dev, hal_smbus_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_smbus_irq_handle_all_reset(hal_smbus_dev_struct *smbus_dev);
/* config the SMBUS alert mode in master mode */
void hals_smbus_alert_master_config(uint32_t smbus_periph);
/* config the SMBUS alert mode in slave mode */
void hals_smbus_alert_slave_config(uint32_t smbus_periph);

/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_smbus_master_transmit_interrupt(hal_smbus_dev_struct *smbus_dev, uint16_t slave_address, \
                                            uint8_t *p_buffer, uint32_t length, \
                                            hal_smbus_irq_user_callback_struct *p_user_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_smbus_master_receive_interrupt(hal_smbus_dev_struct *smbus_dev, uint16_t slave_address, \
                                           uint8_t *p_buffer, uint32_t length, \
                                           hal_smbus_irq_user_callback_struct *p_user_func);
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_smbus_slave_transmit_interrupt(hal_smbus_dev_struct *smbus_dev, \
                                           uint8_t *p_buffer, uint32_t length, \
                                           hal_smbus_irq_user_callback_struct *p_user_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_smbus_slave_receive_interrupt(hal_smbus_dev_struct *smbus_dev, \
                                          uint8_t *p_buffer, uint32_t length, \
                                          hal_smbus_irq_user_callback_struct *p_user_func);
/* abort a master/host SMBUS process communication in interrupt mode.*/
int32_t hal_smbus_master_abort_interrupt(hal_smbus_dev_struct *smbus_dev, uint16_t slave_address);

/* check whether the device is ready for access */
int32_t hal_smbus_device_ready_check(hal_smbus_dev_struct *smbus_dev);

/* enable the address listen mode with Interrupt */
int32_t hal_smbus_address_listen_interrupt_enable(hal_smbus_dev_struct *smbus_dev, \
                                                  hal_smbus_irq_user_callback_struct *p_user_func);
/* disable the address listen mode with Interrupt */
int32_t hal_smbus_address_listen_interrupt_disable(hal_smbus_dev_struct *smbus_dev);

/* configure digital noise filter */
void hal_smbus_digital_noise_filter_config(uint32_t smbus_periph, hal_i2c_digital_filter_enum filter_length);
/* configure analog noise filter */
void hal_smbus_analog_noise_filter_config(uint32_t smbus_periph, ControlStatus state);

/* return the smbus tx state */
hal_smbus_run_state_enum hal_smbus_tx_state_get(hal_smbus_dev_struct *smbus_dev);
/* return the smbus rx state */
hal_smbus_run_state_enum hal_smbus_rx_state_get(hal_smbus_dev_struct *smbus_dev);
/* return the smbus error code */
uint32_t hal_smbus_error_code_get(hal_smbus_dev_struct *smbus_dev);

/* select SMBus type */
void hals_smbus_type_config(uint32_t smbus_periph, uint32_t type);
/* enable SMBus device default address */
void hals_smbus_default_addr_enable(uint32_t smbus_periph);
/* disable SMBus device default address */
void hals_smbus_default_addr_disable(uint32_t smbus_periph);
/* enable SMBus host address */
void hals_smbus_alert_master_device_config(uint32_t smbus_periph);
/* disable SMBus host address */
void hals_smbus_alert_slave_device_config(uint32_t smbus_periph);
/* enable SMBus alert */
void hals_smbus_alert_enable(uint32_t smbus_periph);
/* disable SMBus alert */
void hals_smbus_alert_disable(uint32_t smbus_periph);

/* configure I2C PEC calculation enable or disable */
void hals_i2c_smbus_pec_config(uint32_t smbus_periph, uint32_t pec_state);
/* enable the SMBUS fast mode plus driving capability */
void hals_smbus_enable_fastplus_mode(uint32_t config_fastmode);
/* disable the SMBUS fast mode plus driving capability */
void hals_smbus_disable_fastplus_mode(uint32_t config_fastmode);

/* enable extended clock timeout detection */
void hals_i2c_extented_clock_timeout_enable(uint32_t smbus_periph);
/* disable extended clock timeout detection */
void hals_i2c_extented_clock_timeout_disable(uint32_t smbus_periph);
/* enable clock timeout detection */
void hals_i2c_clock_timeout_enable(uint32_t smbus_periph);
/* disable clock timeout detection */
void hals_i2c_clock_timeout_disable(uint32_t smbus_periph);
/* configure bus timeout B */
void hals_i2c_bus_timeout_b_config(uint32_t smbus_periph, uint32_t timeout);
/* configure bus timeout A */
void hals_i2c_bus_timeout_a_config(uint32_t smbus_periph, uint32_t timeout);
/* configure idle clock timeout detection */
void hals_i2c_idle_clock_timeout_config(uint32_t smbus_periph, ControlStatus timeout_detection);

#endif /* GD32H7XX_HAL_SMBUS_H_*/
