/*!
    \file    gd32h7xx_hal_smartcard.h
    \brief   definitions for the SMARTCARD

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_SMARTCARD_H
#define GD32H7XX_HAL_SMARTCARD_H

#include "gd32h7xx_hal.h"

/* smartcard error code */
#define HAL_SMARTCARD_ERROR_NONE               BIT(0)                           /*!< no error */
#define HAL_SMARTCARD_ERROR_PERR               BIT(1)                           /*!< parity error */
#define HAL_SMARTCARD_ERROR_NERR               BIT(2)                           /*!< noise error */
#define HAL_SMARTCARD_ERROR_FERR               BIT(3)                           /*!< frame error */
#define HAL_SMARTCARD_ERROR_ORERR              BIT(4)                           /*!< overrun error */
#define HAL_SMARTCARD_ERROR_DMA                BIT(5)                           /*!< DMA transfer error */
#define HAL_SMARTCARD_ERROR_RTO                BIT(6)                           /*!< receiver timeout error */
#define HAL_SMARTCARD_ERROR_TTO                BIT(7)                           /*!< transmitter timeout error */
#define HAL_SMARTCARD_ERROR_BUSY               BIT(8)                           /*!< busy error */

/* smartcard structure type enum */
typedef enum {
    HAL_SMARTCARD_INIT_STRUCT = 0U,    /*!< smartcard initialization structure */
    HAL_SMARTCARD_DEV_STRUCT,          /*!< smartcard device information structure */
    HAL_SMARTCARD_IRQ_INIT_STRUCT,     /*!< smartcard interrupt callback initialization structure */
    HAL_SMARTCARD_USER_CALLBACK_STRUCT /*!< smartcard user callback structure */
} hal_smartcard_struct_type_enum;

/* smartcard run state enum */
typedef enum {
    HAL_SMARTCARD_STATE_RESET = 0U,   /*!< not initialized state */
    HAL_SMARTCARD_STATE_READY,        /*!< initialized and ready for use */
    HAL_SMARTCARD_STATE_BUSY,         /*!< internal process is ongoing */
    HAL_SMARTCARD_STATE_BUSY_TX,      /*!< data transmission process is ongoing */
    HAL_SMARTCARD_STATE_BUSY_RX       /*!< data reception process is ongoing */
} hal_smartcard_run_state_enum;

/* smartcard transfer buffer structure */
typedef struct {
    __IO uint8_t *buffer;   /*!< pointer to transfer buffer */
    __IO uint32_t length;   /*!< transfer byte length */
    __IO uint32_t remain;   /*!< transfer remain byte */
} hal_smartcard_buffer_struct;

/* smartcard device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb transmit_complete_handle;    /*!< transmit complete callback function */
    __IO hal_irq_handle_cb transmit_fifo_empty_handle;  /*!< transmit fifo empty callback function */
    __IO hal_irq_handle_cb receive_complete_handle;     /*!< receive timeout callback function */
    __IO hal_irq_handle_cb receive_fifo_full_handle;    /*!< receive fifo full callback function */
    __IO hal_irq_handle_cb error_handle;                /*!< error callback function */
} hal_smartcard_irq_struct;

/* @PARA: p_smartcard */
/* @STRUCT: UART Init structure definition */
typedef struct {
    uint32_t baudrate;              /*!< configures the smartcard communication baud rate */
    uint32_t wordlength;            /*!< specifies the number of data bits transmitted or received in a frame */
    uint32_t stopbits;              /*!< specifies the number of stop bits */
    uint32_t parity;                /*!< specifies the parity mode */
    uint32_t mode;                  /*!< specifies whether the receive or transmit mode is enabled or disabled */
    uint32_t prescaler;             /*!< specifies the SmartCard prescaler */
    uint32_t clkpolarity;           /*!< specifies the steady state of the serial clock */
    uint32_t clkphase;              /*!< specifies the clock transition on which the bit capture is made */
    uint32_t clklastbit;            /*!< specifies whether the clock pulse corresponding to the last transmitted
                                      data bit (MSB) has to be output on the SCLK pin in synchronous mode */
    uint32_t samplingmethod;        /*!< specifies whether a single sample or three samples' majority vote is selected */
    uint32_t guardtime;             /*!< specifies the smartcard guard time applied after stop bits */
    uint32_t nackenable;            /*!< specifies whether the smartcard nack transmission is enabled in case of parity error */
    uint32_t earlynack;             /*!< specifies whether early NACK when smartcard mode is selected */
    uint32_t timeoutenable;         /*!< specifies whether the receiver timeout is enabled */
    uint32_t timeoutvalue;          /*!< specifies the receiver time out value in number of baud blocks */
    uint32_t blocklength;           /*!< specifies the smartcard block length in T=1 Reception mode */
    uint32_t autoretrycount;        /*!< specifies the smartcard auto-retry count (number of retries in receive and transmit mode) */
    uint32_t fifomode;              /*!< specifies whether the receive or transmit fifo is enabled or disabled*/
    uint32_t txthreshold;           /*!< specifies smartcard transmit FIFO threshold */
    uint32_t rxthreshold;           /*!< specifies smartcard receive FIFO threshold */

    uint32_t txpinlevelinvert;      /*!< specifies whether the TX pin active level is inverted */
    uint32_t rxpinlevelinvert;      /*!< specifies whether the RX pin active level is inverted */
    uint32_t datainvert;            /*!< specifies whether data are inverted */
    uint32_t swap;                  /*!< specifies whether TX and RX pins are swapped */
    uint32_t overrundisable;        /*!< specifies whether the reception overrun detection is disabled */
    uint32_t dmadisableonrxerror;   /*!< specifies whether the DMA is disabled in case of reception error */
    uint32_t msbfirst;              /*!< specifies whether MSB is sent first on UART line */
} hal_smartcard_init_struct;

/* @PARA: smartcard_dev */
/* @STRUCT: smartcard device information definition */
typedef struct {
    uint32_t periph;                                /*!< smartcard periph */
    hal_smartcard_init_struct init;                 /*!< smartcard init parameter */
    hal_smartcard_irq_struct smartcard_irq;         /*!< smartcard device interrupt callback function pointer */
    hal_dma_dev_struct *p_dma_rx;                   /*!< DMA receive pointer */
    hal_dma_dev_struct *p_dma_tx;                   /*!< DMA transmit pointer */
    hal_smartcard_buffer_struct txbuffer;           /*!< transmit buffer */
    hal_smartcard_buffer_struct rxbuffer;           /*!< receive buffer */
    __IO uint16_t last_error;                       /*!< the last error code */
    __IO uint16_t error_state;                      /*!< smartcard error state */
    __IO hal_smartcard_run_state_enum tx_state;     /*!< transmit state */
    __IO hal_smartcard_run_state_enum rx_state;     /*!< receive state */
    void *transmit_complete_callback;               /*!< transmit complete interrupt callback */
    void *transmit_fifo_empty_callback;             /*!< transmit fifo empty interrupt callback */
    void *receive_complete_callback;                /*!< receive complete interrupt callback */
    void *receive_fifo_full_callback;               /*!< receive fifo full interrupt callback */
    void *error_callback;                           /*!< error interrupt callback */
    void *abort_complete_callback;                  /*!< abort complete interrupt callback */
    void *abort_tx_complete_callback;               /*!< abort transmit complete interrupt callback */
    void *abort_rx_complete_callback;               /*!< abort received complete interrupt callback */
    hal_mutex_enum mutex;                           /*!< lock */
    void *priv;                                     /*!< private pointer */
} hal_smartcard_dev_struct;

typedef void (*hal_smartcard_user_cb)(hal_smartcard_dev_struct *smartcard_dev);

typedef struct {
    __IO hal_smartcard_user_cb transmit_complete_func;   /*!< transmit complete interrupt callback */
    __IO hal_smartcard_user_cb transmit_fifo_empty_func; /*!< transmit fifo empty interrupt callback */
    __IO hal_smartcard_user_cb abort_tx_complete_func;   /*!< abort transmit complete interrupt callback */
    __IO hal_smartcard_user_cb receive_complete_func;    /*!< receive complete interrupt callback */
    __IO hal_smartcard_user_cb receive_fifo_full_func;   /*!< receive fifo full interrupt callback */
    __IO hal_smartcard_user_cb abort_rx_complete_func;   /*!< abort receive complete interrupt callback */
    __IO hal_smartcard_user_cb abort_complete_func;      /*!< abort complete interrupt callback */
    __IO hal_smartcard_user_cb error_func;               /*!< error interrupt callback */
} hal_smartcard_irq_user_callback_struct;

/* @STRUCT_MEMBER: baudrate */
/* @=NULL */

/* @STRUCT_MEMBER: prescaler */
/* @=NULL */

/* @STRUCT_MEMBER: guardtime */
/* @=NULL */

/* @STRUCT_MEMBER: timeoutvalue */
/* @=NULL */

/* @STRUCT_MEMBER: blocklength */
/* @=NULL*/

/* @STRUCT_MEMBER: autoretrycount */
/* @=NULL*/

/* hal function declarations */
/* Initialization and de-initialization functions */
/* @FUNCTION: initialize the smartcard with specified values */
/* init smartcard mode */
int32_t hal_smartcard_init(hal_smartcard_dev_struct *smartcard_dev, uint32_t periph, \
                           hal_smartcard_init_struct *p_smartcard);
/* @END */

/* initialize the smartcard structure with the default values */
int32_t hal_smartcard_struct_init(hal_smartcard_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize smartcard */
int32_t hal_smartcard_deinit(hal_smartcard_dev_struct *smartcard_dev);

/* smartcard interrupt handle functions */
/* smartcard interrupt handler content function,which is merely used in USART_IRQHandler */
int32_t hal_smartcard_irq(hal_smartcard_dev_struct *smartcard_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_smartcard_irq_handle_set(hal_smartcard_dev_struct *smartcard_dev, hal_smartcard_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_smartcard_irq_handle_all_reset(hal_smartcard_dev_struct *smartcard_dev);

/* transmit or receive functions */
/* transmit amounts of data, poll transmit process and completed status */
/* the function is blocking */
int32_t hal_smartcard_transmit_poll(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint32_t length, \
                                    uint32_t timeout);
/* receive amounts of data, poll receive process and completed status */
/* the function is blocking */
int32_t hal_smartcard_receive_poll(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint32_t length, \
                                   uint32_t timeout);
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_smartcard_transmit_interrupt(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint32_t length, \
                                         hal_smartcard_irq_user_callback_struct *p_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_smartcard_receive_interrupt(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint32_t length, \
                                        hal_smartcard_irq_user_callback_struct *p_func);
/* transmit amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_smartcard_transmit_dma(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint16_t length, \
                                   hal_smartcard_irq_user_callback_struct *p_func);
/* receive amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_smartcard_receive_dma(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint16_t length, \
                                  hal_smartcard_irq_user_callback_struct *p_func);

/* transfer control functions */
/* abort ongoing transmit transfer */
int32_t hal_smartcard_transmit_abort(hal_smartcard_dev_struct *smartcard_dev);
/* abort ongoing receive transfer */
int32_t hal_smartcard_receive_abort(hal_smartcard_dev_struct *smartcard_dev);
/* abort ongoing transfers */
int32_t hal_smartcard_abort(hal_smartcard_dev_struct *smartcard_dev);

/* abort ongoing transmit transfer in interrupt method */
int32_t hal_smartcard_transmit_abort_interrupt(hal_smartcard_dev_struct *smartcard_dev, \
                                               hal_smartcard_irq_user_callback_struct *p_func);
/* abort ongoing receive transfer in interrupt method */
int32_t hal_smartcard_receive_abort_interrupt(hal_smartcard_dev_struct *smartcard_dev, \
                                              hal_smartcard_irq_user_callback_struct *p_func);
/* abort ongoing transfers in interrupt method */
int32_t hal_smartcard_abort_interrupt(hal_smartcard_dev_struct *smartcard_dev, \
                                      hal_smartcard_irq_user_callback_struct *p_func);

/* return the smartcard tx state */
hal_smartcard_run_state_enum hal_smartcard_tx_state_get(hal_smartcard_dev_struct *smartcard_dev);
/* return the smartcard rx state */
hal_smartcard_run_state_enum hal_smartcard_rx_state_get(hal_smartcard_dev_struct *smartcard_dev);
/* return the smartcard error code */
uint16_t hal_smartcard_error_get(hal_smartcard_dev_struct *smartcard_dev);

/* USART FIFO */
/* enable FIFO */
int32_t hal_smartcard_fifo_enable(hal_smartcard_dev_struct *smartcard_dev);
/* disable FIFO */
int32_t hal_smartcard_fifo_disable(hal_smartcard_dev_struct *smartcard_dev);
/* configure transmit FIFO threshold */
int32_t hal_smartcard_transmit_fifo_threshold_config(hal_smartcard_dev_struct *smartcard_dev, uint32_t txthreshold);
/* configure receive FIFO threshold */
int32_t hal_smartcard_receive_fifo_threshold_config(hal_smartcard_dev_struct *smartcard_dev, uint32_t rxthreshold);
/* configure block length */
int32_t hal_smartcard_block_length_config(hal_smartcard_dev_struct *smartcard_dev, uint32_t bl);
/* configure receiver timeout threshold */
int32_t hal_smartcard_receiver_timeout_config(hal_smartcard_dev_struct *smartcard_dev, uint32_t rtimeout);
/* enable receiver timeout */
int32_t hal_smartcard_receiver_timeout_enable(hal_smartcard_dev_struct *smartcard_dev);
/* disable receiver timeout */
int32_t hal_smartcard_receiver_timeout_disable(hal_smartcard_dev_struct *smartcard_dev);

#endif /* GD32H7XX_HAL_SMARTCARD_H */
