/*!
    \file    gd32h7xx_hal_sdio_emmc.h
    \brief   definitions for the eMMC

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_SDIO_EMMC_H
#define GD32H7XX_HAL_SDIO_EMMC_H

#include "gd32h7xx_hal.h"

/* eMMC memory card bus commands index */
#define EMMC_CMD_GO_IDLE_STATE                  ((uint8_t)0)                    /* CMD0, GO_IDLE_STATE */
#define EMMC_CMD_SEND_OP_COND                   ((uint8_t)1)                    /* CMD1, SEND_OP_COND */
#define EMMC_CMD_ALL_SEND_CID                   ((uint8_t)2)                    /* CMD2, ALL_SEND_CID */
#define EMMC_CMD_SEND_RELATIVE_ADDR             ((uint8_t)3)                    /* CMD3, SEND_RELATIVE_ADDR */
#define EMMC_CMD_SET_DSR                        ((uint8_t)4)                    /* CMD4, SET_DSR */
#define EMMC_CMD_SLEEP_AWAKE                    ((uint8_t)5)                    /* CMD5, SLEEP_AWAKE */
#define EMMC_CMD_SWITCH_FUNC                    ((uint8_t)6)                    /* CMD6, SWITCH */
#define EMMC_CMD_SELECT_DESELECT_CARD           ((uint8_t)7)                    /* CMD7, SELECT_DESELECT_CARD */
#define EMMC_CMD_SEND_EXT_CSD                   ((uint8_t)8)                    /* CMD8, SEND_EXT_CSD */
#define EMMC_CMD_SEND_CSD                       ((uint8_t)9)                    /* CMD9, SEND_CSD */
#define EMMC_CMD_SEND_CID                       ((uint8_t)10)                   /* CMD10, SEND_CID */
#define EMMC_CMD_STOP_TRANSMISSION              ((uint8_t)12)                   /* CMD12, STOP_TRANSMISSION */
#define EMMC_CMD_SEND_STATUS                    ((uint8_t)13)                   /* CMD13, SEND_STATUS */
#define EMMC_CMD_GO_INACTIVE_STATE              ((uint8_t)15)                   /* CMD15, GO_INACTIVE_STATE */
#define EMMC_CMD_SET_BLOCKLEN                   ((uint8_t)16)                   /* CMD16, SET_BLOCKLEN */
#define EMMC_CMD_READ_SINGLE_BLOCK              ((uint8_t)17)                   /* CMD17, READ_SINGLE_BLOCK */
#define EMMC_CMD_READ_MULTIPLE_BLOCK            ((uint8_t)18)                   /* CMD18, READ_MULTIPLE_BLOCK */
#define EMMC_SET_BLOCK_COUNT                    ((uint8_t)23)                   /* CMD23, SET_BLOCK_COUNT */
#define EMMC_CMD_WRITE_BLOCK                    ((uint8_t)24)                   /* CMD24, WRITE_BLOCK */
#define EMMC_CMD_WRITE_MULTIPLE_BLOCK           ((uint8_t)25)                   /* CMD25, WRITE_MULTIPLE_BLOCK */
#define EMMC_CMD_PROG_CSD                       ((uint8_t)27)                   /* CMD27, PROG_CSD */
#define EMMC_CMD_SET_WRITE_PROT                 ((uint8_t)28)                   /* CMD28, SET_WRITE_PROT */
#define EMMC_CMD_CLR_WRITE_PROT                 ((uint8_t)29)                   /* CMD29, CLR_WRITE_PROT */
#define EMMC_CMD_SEND_WRITE_PROT                ((uint8_t)30)                   /* CMD30, SEND_WRITE_PROT */
#define EMMC_CMD_SEND_WRITE_PROT_TYPE           ((uint8_t)30)                   /* CMD30, SEND_WRITE_PROT_TYPE */
#define EMMC_CMD_ERASE_GROUP_START              ((uint8_t)35)                   /* CMD35, ERASE_GROUP_START */
#define EMMC_CMD_ERASE_GROUP_END                ((uint8_t)36)                   /* CMD36, ERASE_GROUP_END */
#define EMMC_CMD_ERASE                          ((uint8_t)38)                   /* CMD38, ERASE */
#define EMMC_CMD_LOCK_UNLOCK                    ((uint8_t)42)                   /* CMD42, LOCK_UNLOCK */
#define EMMC_CMD_APP_CMD                        ((uint8_t)55)                   /* CMD55, APP_CMD */
#define EMMC_CMD_GEN_CMD                        ((uint8_t)56)                   /* CMD56, GEN_CMD */

/* lock unlock status */
#define EMMC_LOCK                               ((uint8_t)0x05)                 /* lock the eMMC */
#define EMMC_UNLOCK                             ((uint8_t)0x02)                 /* unlock the eMMC */

/* card status of R1 definitions */
#define EMMC_R1_OUT_OF_RANGE                    BIT(31)                         /* command's argument was out of the allowed range */
#define EMMC_R1_ADDRESS_ERROR                   BIT(30)                         /* misaligned address which did not match the block length */
#define EMMC_R1_BLOCK_LEN_ERROR                 BIT(29)                         /* transferred block length is not allowed */
#define EMMC_R1_ERASE_SEQ_ERROR                 BIT(28)                         /* an error in the sequence of erase commands occurred */
#define EMMC_R1_ERASE_PARAM                     BIT(27)                         /* an invalid selection of write-blocks for erase occurred */
#define EMMC_R1_WP_VIOLATION                    BIT(26)                         /* the host attempts to write to a protected block or to the temporary or permanent write protected card */
#define EMMC_R1_CARD_IS_LOCKED                  BIT(25)                         /* the card is locked by the host */
#define EMMC_R1_LOCK_UNLOCK_FAILED              BIT(24)                         /* a sequence or password error has been detected in lock/unlock card command */
#define EMMC_R1_COM_CRC_ERROR                   BIT(23)                         /* CRC check of the previous command failed */
#define EMMC_R1_ILLEGAL_COMMAND                 BIT(22)                         /* command not legal for the card state */
#define EMMC_R1_CARD_ECC_FAILED                 BIT(21)                         /* card internal ECC was applied but failed to correct the data */
#define EMMC_R1_CC_ERROR                        BIT(20)                         /* internal card controller error */
#define EMMC_R1_GENERAL_UNKNOWN_ERROR           BIT(19)                         /* a general or an unknown error occurred during the operation */
#define EMMC_R1_CSD_OVERWRITE                   BIT(16)                         /* read only section of the CSD does not match or attempt to reverse the copy or permanent WP bits */
#define EMMC_R1_WP_ERASE_SKIP                   BIT(15)                         /* partial address space was erased */
#define EMMC_R1_CARD_ECC_DISABLED               BIT(14)                         /* command has been executed without using the internal ECC */
#define EMMC_R1_ERASE_RESET                     BIT(13)                         /* an erase sequence was cleared before executing */
#define EMMC_R1_READY_FOR_DATA                  BIT(8)                          /* correspond to buffer empty signaling on the bus */
#define EMMC_R1_APP_CMD                         BIT(5)                          /* card will expect ACMD */
#define EMMC_R1_AKE_SEQ_ERROR                   BIT(3)                          /* error in the sequence of the authentication process */
#define EMMC_R1_ERROR_BITS                      ((uint32_t)0xFDF9E008U)         /* all the R1 error bits */

/* card status of R6 definitions */
#define EMMC_R6_COM_CRC_ERROR                   BIT(15)                         /* CRC check of the previous command failed */
#define EMMC_R6_ILLEGAL_COMMAND                 BIT(14)                         /* command not legal for the card state */
#define EMMC_R6_GENERAL_UNKNOWN_ERROR           BIT(13)                         /* a general or an unknown error occurred during the operation */

/* card state */
#define EMMC_CARDSTATE_IDLE                     ((uint8_t)0x00)                 /* eMMC is in idle state */
#define EMMC_CARDSTATE_READY                    ((uint8_t)0x01)                 /* eMMC is in ready state */
#define EMMC_CARDSTATE_IDENTIFICAT              ((uint8_t)0x02)                 /* eMMC is in identification state */
#define EMMC_CARDSTATE_STANDBY                  ((uint8_t)0x03)                 /* eMMC is in standby state */
#define EMMC_CARDSTATE_TRANSFER                 ((uint8_t)0x04)                 /* eMMC is in transfer state */
#define EMMC_CARDSTATE_DATA                     ((uint8_t)0x05)                 /* eMMC is in data sending state */
#define EMMC_CARDSTATE_RECEIVING                ((uint8_t)0x06)                 /* eMMC is in receiving state */
#define EMMC_CARDSTATE_PROGRAMMING              ((uint8_t)0x07)                 /* eMMC is in programming state */
#define EMMC_CARDSTATE_DISCONNECT               ((uint8_t)0x08)                 /* eMMC is in disconnect state */
#define EMMC_CARDSTATE_LOCKED                   ((uint32_t)0x02000000U)         /* eMMC is in locked state */

/* eMMC bus width, check SCR register */
#define EMMC_BUS_WIDTH_8BIT                     ((uint32_t)0x00000002U)         /* 8-bit width bus mode */
#define EMMC_BUS_WIDTH_4BIT                     ((uint32_t)0x00000001U)         /* 4-bit width bus mode */
#define EMMC_BUS_WIDTH_1BIT                     ((uint32_t)0x00000000U)         /* 1-bit width bus mode */

/* masks for SCR register */
#define EMMC_MASK_0_7BITS                       ((uint32_t)0x000000FFU)         /* mask [7:0] bits */
#define EMMC_MASK_8_15BITS                      ((uint32_t)0x0000FF00U)         /* mask [15:8] bits */
#define EMMC_MASK_16_23BITS                     ((uint32_t)0x00FF0000U)         /* mask [23:16] bits */
#define EMMC_MASK_24_31BITS                     ((uint32_t)0xFF000000U)         /* mask [31:24] bits */

#define SDIO_FIFO_ADDR                          SDIO_FIFO(SDIO)                 /* address of SDIO_FIFO */
#define EMMC_FIFOHALF_WORDS                     ((uint32_t)0x00000008U)         /* words of FIFO half full/empty */
#define EMMC_FIFOHALF_BYTES                     ((uint32_t)0x00000020U)         /* bytes of FIFO half full/empty */

#define EMMC_DATATIMEOUT                        ((uint32_t)0xFFFFFFFFU)         /* DSM data timeout */
#define EMMC_MAX_VOLT_VALIDATION                ((uint32_t)0x0000FFFFU)         /* the maximum times of voltage validation */
#define EMMC_MAX_DATA_LENGTH                    ((uint32_t)0x01FFFFFFU)         /* the maximum length of data */
#define EMMC_ALLZERO                            ((uint32_t)0x00000000U)         /* all zero */
#define EMMC_RCA_SHIFT                          ((uint8_t)0x10U)                /* RCA shift bits */

#define EMMC_CLK_DIV_INIT                       ((uint16_t)0x01F4)              /* eMMC clock division in initialization phase */
#define EMMC_CLK_DIV_TRANS_DSPEED               ((uint16_t)0x0008)              /* eMMC clock division in default speed transmission phase */
#define EMMC_CLK_DIV_TRANS_HSPEED               ((uint16_t)0x0002)              /* eMMC clock division in high speed transmission phase */

#define EMMC_TRANSFER_NONE                      ((uint32_t)0x00000000U)         /*!< none */
#define EMMC_TRANSFER_READ_SINGLE_BLOCK         ((uint32_t)0x00000001U)         /*!< read single block operation */
#define EMMC_TRANSFER_READ_MULTIPLE_BLOCK       ((uint32_t)0x00000002U)         /*!< read multiple blocks operation */
#define EMMC_TRANSFER_WRITE_SINGLE_BLOCK        ((uint32_t)0x00000010U)         /*!< write single block operation */
#define EMMC_TRANSFER_WRITE_MULTIPLE_BLOCK      ((uint32_t)0x00000020U)         /*!< write multiple blocks operation */
#define EMMC_TRANSFER_IT                        ((uint32_t)0x00000008U)         /*!< process in interrupt mode */
#define EMMC_TRANSFER_DMA                       ((uint32_t)0x00000080U)         /*!< process in DMA mode */

#define MMC_HIGH_VOLTAGE_RANGE                  ((uint32_t)0x80FF8000U)         /*!< high voltage in byte mode */
#define MMC_DUAL_VOLTAGE_RANGE                  ((uint32_t)0x80FF8080U)         /*!< dual voltage in byte mode */
#define MMC_LOW_VOLTAGE_RANGE                   ((uint32_t)0x80000080U)         /*!< low voltage in byte mode */
#define EMMC_HIGH_VOLTAGE_RANGE                 ((uint32_t)0xC0FF8000U)         /*!< high voltage in sector mode */
#define EMMC_DUAL_VOLTAGE_RANGE                 ((uint32_t)0xC0FF8080U)         /*!< dual voltage in sector mode */
#define EMMC_LOW_VOLTAGE_RANGE                  ((uint32_t)0xC0000080U)         /*!< low voltage in sector mode */
#define MMC_INVALID_VOLTAGE_RANGE               ((uint32_t)0x0001FF01U)

#define EMMC_LOW_CAPACITY_CARD                  ((uint32_t)0x00000000U)         /*!< MMC card capacity <=2Gbytes */
#define EMMC_HIGH_CAPACITY_CARD                 ((uint32_t)0x00000001U)         /*!< MMC card capacity >2Gbytes and <2Tbytes */

#define HAL_EMMC_ERASE                          ((uint32_t)0x00000000U)         /*!< erase the erase groups identified by CMD35 & 36 */
#define HAL_EMMC_TRIM                           ((uint32_t)0x00000001U)         /*!< erase the write blocks identified by CMD35 & 36 */
#define HAL_EMMC_DISCARD                        ((uint32_t)0x00000003U)         /*!< discard the write blocks identified by CMD35 & 36 */
#define HAL_EMMC_SECURE_ERASE                   ((uint32_t)0x80000000U)         /*!< perform a secure purge according SRT on the erase groups identified by CMD35 & 36 */
#define HAL_EMMC_SECURE_TRIM_STEP1              ((uint32_t)0x80000001U)         /*!< mark the write blocks identified by CMD35 & 36 for secure erase */
#define HAL_EMMC_SECURE_TRIM_STEP2              ((uint32_t)0x80008000U)         /*!< perform a secure purge according SRT on the write blocks previously identified */

#define HAL_EMMC_CARD_IDLE                      ((uint32_t)0x00000000U)         /*!< Card is in idle state (can't be checked by CMD13) */
#define HAL_EMMC_CARD_READY                     ((uint32_t)0x00000001U)         /*!< Card state is ready (can't be checked by CMD13) */
#define HAL_EMMC_CARD_IDENTIFICATION            ((uint32_t)0x00000002U)         /*!< Card is in identification state (can't be checked by CMD13) */
#define HAL_EMMC_CARD_STANDBY                   ((uint32_t)0x00000003U)         /*!< Card is in standby state */
#define HAL_EMMC_CARD_TRANSFER                  ((uint32_t)0x00000004U)         /*!< Card is in transfer state */
#define HAL_EMMC_CARD_SENDING                   ((uint32_t)0x00000005U)         /*!< Card is sending an operation */
#define HAL_EMMC_CARD_RECEIVING                 ((uint32_t)0x00000006U)         /*!< Card is receiving operation information */
#define HAL_EMMC_CARD_PROGRAMMING               ((uint32_t)0x00000007U)         /*!< Card is in programming state */
#define HAL_EMMC_CARD_DISCONNECTED              ((uint32_t)0x00000008U)         /*!< Card is disconnected */
#define HAL_EMMC_CARD_BUSTEST                   ((uint32_t)0x00000009U)         /*!< Card is in bus test state */
#define HAL_EMMC_CARD_SLEEP                     ((uint32_t)0x0000000AU)         /*!< Card is in sleep state (can't be checked by CMD13) */
#define HAL_EMMC_CARD_ERROR                     ((uint32_t)0x000000FFU)         /*!< Card response Error (can't be checked by CMD13) */

/* eMMC struct initialization type enum */
typedef enum {
    HAL_EMMC_INIT_STRUCT = 0U,         /*!< initialization structure */
    HAL_EMMC_DEV_STRUCT,               /*!< device information structure */
    HAL_EMMC_IRQ_INIT_STRUCT,          /*!< interrupt callback initialization structure */
    HAL_EMMC_CARDID_STRUCT,            /*!< card identification structure */
    HAL_EMMC_CARDSD_STRUCT,            /*!< card identification structure */
    HAL_EMMC_CARD_INFO_STRUCT,         /*!< card information structure */
    HAL_EMMC_IRQ_USER_CALLBACK_STRUCT  /*!< interrupt callback user structure */
} hal_sdio_emmc_struct_type_enum;

/* eMMC state enum */
typedef enum {
    HAL_EMMC_STATE_RESET       = ((uint32_t)0x00000000U), /*!< MMC not yet initialized or disabled */
    HAL_EMMC_STATE_READY       = ((uint32_t)0x00000001U), /*!< MMC initialized and ready for use */
    HAL_EMMC_STATE_TIMEOUT     = ((uint32_t)0x00000002U), /*!< MMC timeout state */
    HAL_EMMC_STATE_BUSY        = ((uint32_t)0x00000003U), /*!< MMC process ongoing */
    HAL_EMMC_STATE_PROGRAMMING = ((uint32_t)0x00000004U), /*!< MMC programming state */
    HAL_EMMC_STATE_RECEIVING   = ((uint32_t)0x00000005U), /*!< MMC receiving state */
    HAL_EMMC_STATE_TRANSFER    = ((uint32_t)0x00000006U), /*!< MMC transfer state */
    HAL_EMMC_STATE_ERROR       = ((uint32_t)0x0000000FU)  /*!< MMC is in error state */
} hal_sdio_emmc_state_enum;

/* eMMC error flags */
typedef enum {
    HAL_EMMC_OUT_OF_RANGE = 0U,       /* command's argument was out of range */
    HAL_EMMC_ADDRESS_ERROR,           /* misaligned address which did not match the block length */
    HAL_EMMC_BLOCK_LEN_ERROR,         /* transferred block length is not allowed for the card or the number of transferred bytes does not match the block length */
    HAL_EMMC_ERASE_SEQ_ERROR,         /* an error in the sequence of erase command occurs */
    HAL_EMMC_ERASE_PARAM,             /* an invalid selection of write-blocks for erase occurred */
    HAL_EMMC_WP_VIOLATION,            /* attempt to program a write protect block or permanent write protected card */
    HAL_EMMC_LOCK_UNLOCK_FAILED,      /* sequence or password error has been detected in lock/unlock card command */
    HAL_EMMC_COM_CRC_ERROR,           /* CRC check of the previous command failed */
    HAL_EMMC_ILLEGAL_COMMAND,         /* command not legal for the card state */
    HAL_EMMC_CARD_ECC_FAILED,         /* card internal ECC was applied but failed to correct the data */
    HAL_EMMC_CC_ERROR,                /* internal card controller error */
    HAL_EMMC_GENERAL_UNKNOWN_ERROR,   /* general or unknown error occurred during the operation */
    HAL_EMMC_CSD_OVERWRITE,           /* read only section of the CSD does not match the card content or an attempt to reverse the copy or permanent WP bits was made */
    HAL_EMMC_WP_ERASE_SKIP,           /* only partial address space was erased or the temporary or permanent write protected card was erased */
    HAL_EMMC_CARD_ECC_DISABLED,       /* command has been executed without using internal ECC */
    HAL_EMMC_ERASE_RESET,             /* erase sequence was cleared before executing because an out of erase sequence command was received */
    HAL_EMMC_AKE_SEQ_ERROR,           /* error in the sequence of the authentication process */
    HAL_EMMC_CMD_CRC_ERROR,           /* command response received (CRC check failed) */
    HAL_EMMC_DATA_CRC_ERROR,          /* data block sent/received (CRC check failed) */
    HAL_EMMC_CMD_RESP_TIMEOUT,        /* command response timeout */
    HAL_EMMC_DATA_TIMEOUT,            /* data timeout */
    HAL_EMMC_TX_UNDERRUN_ERROR,       /* transmit FIFO underrun error occurs */
    HAL_EMMC_RX_OVERRUN_ERROR,        /* received FIFO overrun error occurs */
    HAL_EMMC_START_BIT_ERROR,         /* start bit error in the bus */
    HAL_EMMC_VOLTRANGE_INVALID,       /* the voltage range is invalid */
    HAL_EMMC_PARAMETER_INVALID,       /* the parameter is invalid */
    HAL_EMMC_OPERATION_IMPROPER,      /* the operation is improper */
    HAL_EMMC_FUNCTION_UNSUPPORTED,    /* the function is unsupported */
    HAL_EMMC_ERROR,                   /* an error occurred */
    HAL_EMMC_DMA_ERROR,               /* an error when idma transfer */
    HAL_EMMC_OK                       /* no error occurred */
} hal_sdio_emmc_error_enum;

/* eMMC card csd data */
typedef struct {
    uint8_t  csdstruct;                  /*!< csd structure */
    uint8_t  sysspecversion;             /*!< system specification version */
    uint8_t  reserved1;                  /*!< reserved */
    uint8_t  taac;                       /*!< data read access time 1 */
    uint8_t  nsac;                       /*!< data read access time 2 in clk cycles */
    uint8_t  maxbusclkfrec;              /*!< max. bus clock frequency */
    uint16_t cardcomdclasses;            /*!< card command classes */
    uint8_t  rdblocklen;                 /*!< max. read data block length */
    uint8_t  partblockread;              /*!< partial blocks for read allowed */
    uint8_t  wrblockmisalign;            /*!< write block misalignment */
    uint8_t  rdblockmisalign;            /*!< read block misalignment */
    uint8_t  dsrimpl;                    /*!< dsr implemented */
    uint8_t  reserved2;                  /*!< reserved */
    uint32_t devicesize;                 /*!< device size */
    uint8_t  maxrdcurrentvddmin;         /*!< max. read current @ vdd min */
    uint8_t  maxrdcurrentvddmax;         /*!< max. read current @ vdd max */
    uint8_t  maxwrcurrentvddmin;         /*!< max. write current @ vdd min */
    uint8_t  maxwrcurrentvddmax;         /*!< max. write current @ vdd max */
    uint8_t  devicesizemul;              /*!< device size multiplier */
    uint8_t  erasegrsize;                /*!< erase group size */
    uint8_t  erasegrmul;                 /*!< erase group size multiplier */
    uint8_t  wrprotectgrsize;            /*!< write protect group size */
    uint8_t  wrprotectgrenable;          /*!< write protect group enable */
    uint8_t  mandeflecc;                 /*!< manufacturer default ecc */
    uint8_t  wrspeedfact;                /*!< write speed factor  */
    uint8_t  maxwrblocklen;              /*!< max. write data block length */
    uint8_t  writeblockpapartial;        /*!< partial blocks for write allowed */
    uint8_t  reserved3;                  /*!< reserved */
    uint8_t  contentprotectappli;        /*!< content protection application */
    uint8_t  fileformatgroup;            /*!< file format group */
    uint8_t  copyflag;                   /*!< copy flag (otp) */
    uint8_t  permwrprotect;              /*!< permanent write protection */
    uint8_t  tempwrprotect;              /*!< temporary write protection */
    uint8_t  fileformat;                 /*!< file format */
    uint8_t  ecc;                        /*!< ecc code */
    uint8_t  csd_crc;                    /*!< csd crc */
    uint8_t  reserved4;                  /*!< always 1 */
} hal_sdio_emmc_cardcsd_struct;

/* eMMC card cid data */
typedef struct {
    uint8_t  manufacturerid;             /*!< manufacturer id */
    uint16_t oem_appliid;                /*!< oem/application id */
    uint32_t prodname1;                  /*!< product name part1 */
    uint8_t  prodname2;                  /*!< product name part2 */
    uint8_t  prodrev;                    /*!< product revision */
    uint32_t prodsn;                     /*!< product serial number */
    uint8_t  reserved1;                  /*!< reserved1 */
    uint16_t manufactdate;               /*!< manufacturing date */
    uint8_t  cid_crc;                    /*!< cid crc */
    uint8_t  reserved2;                  /*!< always 1 */
} hal_sdio_emmc_cardcid_struct;

/* information of eMMC */
typedef struct {
    uint32_t card_type;                  /* card type */
    uint32_t card_speed;                 /* card speed */
    uint16_t card_rca;                   /* card relative card address */
    uint32_t card_blocknum;              /* card capacity */
    uint32_t card_blocksize;             /* card block size */
    uint32_t logic_blocknum;             /* card logic capacity */
    uint32_t logic_blocksize;            /* card logic block size */
} hal_sdio_emmc_cardinfo_struct;

/* SDIO device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb transmit_fifo_half_empty_handle;     /*!< SDIO transmit fifo half empty callback function */
    __IO hal_irq_handle_cb receive_fifo_half_empty_handle;      /*!< SDIO receive fifo half empty callback function */
    __IO hal_irq_handle_cb transfer_complete_handle;            /*!< SDIO transmit receive complete callback function */
    __IO hal_irq_handle_cb idma_complete_handle;                /*!< SDIO idma double buffer transfer complete callback function */
    __IO hal_irq_handle_cb transfer_error_handle;               /*!< SDIO transfer error callback function */
} hal_sdio_emmc_irq_struct;

/* @PARA: p_emmc_init */
/* @STRUCT: emmc init structure */
typedef struct {
    uint32_t bus_width;        /*!< SDIO bus width mode */
    uint32_t clock_division;   /*!< SDIO clock division value */
    uint32_t clock_edge;       /*!< SDIO clock edge value */
    uint32_t clock_powersave;  /*!< SDIO clock dynamic switch on/off value */
    uint32_t bus_speed_mode;   /*!< SDIO card bus speed control value */
    uint32_t data_rate;        /*!< SDIO data rate value */
    uint32_t hardware_control; /*!< SDIO hardware control */
} hal_sdio_emmc_init_struct;

/* @PARA: emmc_dev */
/* @STRUCT: sdio device information structure */
typedef struct {
    uint32_t periph;                            /*!< SDIO periph */
    hal_mutex_enum mutex;                       /*!< mutex lock */
    uint32_t *p_txbuffer;                       /*!< SDIO transmit buffer */
    uint32_t tx_length;                         /*!< SDIO transmit length */
    uint32_t *p_rxbuffer;                       /*!< SDIO receive buffer */
    uint32_t rx_length;                         /*!< SDIO receive length */
    uint32_t transfer;                          /*!< SDIO transfer context */
    hal_sdio_emmc_state_enum state;             /*!< SDIO state */
    hal_sdio_emmc_error_enum error_state;       /*!< SDIO error state */
    hal_sdio_emmc_cardinfo_struct card_info;    /*!< SDIO card information */
    uint32_t csd_data[4];                       /*!< content of CSD register */
    uint32_t cid_data[4];                       /*!< content of CID register */
    uint32_t ext_csd[128];                      /*!< content of EXT_CSD register */
    hal_sdio_emmc_irq_struct emmc_irq;          /*!< SDIO device interrupt callback function pointer */
    void *transmit_complete_callback;           /*!< SDIO transmit complete callback function */
    void *receive_complete_callback;            /*!< SDIO receive complete callback function */
    void *transfer_error_callback;              /*!< SDIO transfer error callback function */
    void *write_dma_buffer0_complete_callback;  /*!< SDIO write DMA buffer complete callback function */
    void *write_dma_buffer1_complete_callback;  /*!< SDIO write DMA buffer complete callback function */
    void *read_dma_buffer0_complete_callback;   /*!< SDIO write DMA buffer complete callback function */
    void *read_dma_buffer1_complete_callback;   /*!< SDIO write DMA buffer complete callback function */
    void *abort_complete_callback;              /*!< SDIO abort complete callback function */
} hal_sdio_emmc_dev_struct;

typedef void (*hal_emmc_user_cb)(hal_sdio_emmc_dev_struct *emmc_dev);

typedef struct {
    __IO hal_emmc_user_cb transmit_complete_func;           /*!< SDIO transmit complete callback function */
    __IO hal_emmc_user_cb receive_complete_func;            /*!< SDIO receive complete callback function */
    __IO hal_emmc_user_cb read_dma_buffer0_complete_func;   /*!< SDIO read DMA buffer complete callback function */
    __IO hal_emmc_user_cb read_dma_buffer1_complete_func;   /*!< SDIO read DMA buffer complete callback function */
    __IO hal_emmc_user_cb write_dma_buffer0_complete_func;  /*!< SDIO write DMA buffer complete callback function */
    __IO hal_emmc_user_cb write_dma_buffer1_complete_func;  /*!< SDIO write DMA buffer complete callback function */
    __IO hal_emmc_user_cb abort_complete_func;              /*!< SDIO abort complete callback function */
    __IO hal_emmc_user_cb transfer_error_func;              /*!< SDIO transfer error callback function */
} hal_sdio_emmc_irq_user_callback_struct;

/* function declarations */
/* Initialization and de-initialization functions */
/* @FUNCTION: initialize eMMC */
hal_sdio_emmc_error_enum hal_sdio_emmc_init(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t periph, hal_sdio_emmc_init_struct *p_emmc_init);
/* @END */
/* initialize the eMMC structure with the default values */
int32_t hal_sdio_emmc_struct_init(hal_sdio_emmc_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize the eMMC */
int32_t hal_sdio_emmc_deinit(hal_sdio_emmc_dev_struct *emmc_dev);

/* SDIO interrupt handle functions */
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_sdio_emmc_irq_handle_set(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_sdio_emmc_irq_handle_all_reset(hal_sdio_emmc_dev_struct *emmc_dev);
/* eMMC interrupt handler content function,which is merely used in SDIO_IRQHandler */
void hal_sdio_emmc_irq(hal_sdio_emmc_dev_struct *emmc_dev);

/* configure the bus mode */
int32_t hal_sdio_emmc_bus_mode_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t busmode, uint32_t speed);
/* configure dma dual buffer mode. the data transfer is managed by an internal dma */
int32_t hal_sdio_emmc_dma_buffer_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *data_buffer0, uint32_t *data_buffer1, \
                                   uint32_t buffer_size);
/* change the dma buffer0 or buffer1 address on the fly */
int32_t hal_sdio_emmc_change_dma_buffer(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t buffer_select, uint32_t *data_buffer);
/* read data into a buffer from the specified address of a card */
int32_t hal_sdio_emmc_readblocks_poll(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, \
                                 uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber);
/* read data into a buffer from the specified address of a card by interrupt method */
int32_t hal_sdio_emmc_readblocks_interrupt(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata,\
                                      uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                      hal_sdio_emmc_irq_user_callback_struct *p_user_func);
/* read data into a buffer from the specified address of a card by dma method */
int32_t hal_sdio_emmc_readblocks_dma(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, \
                                uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                hal_sdio_emmc_irq_user_callback_struct *p_user_func);
/* read data into a buffer from the specified address of a card by dma double buffer method */
int32_t hal_sdio_emmc_readblocks_multibuffer_dma(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pbuf0, uint32_t *pbuf1, \
                                            uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                            hal_sdio_emmc_irq_user_callback_struct *p_user_func);
/* write data to the specified address of a card */
int32_t hal_sdio_emmc_writeblocks_poll(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, \
                                  uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber);
/* write data to the specified address of a card by interrupt method */
int32_t hal_sdio_emmc_writeblocks_interrupt(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, \
                                       uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                       hal_sdio_emmc_irq_user_callback_struct *p_user_func);
/* write data to the specified address of a card by dma method */
int32_t hal_sdio_emmc_writeblocks_dma(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pdata, \
                                 uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                 hal_sdio_emmc_irq_user_callback_struct *p_user_func);
/* write data to the specified address of a card by dma double buffer method */
int32_t hal_sdio_emmc_writeblocks_multibuffer_dma(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pbuf0, uint32_t *pbuf1, \
                                             uint32_t blockaddr, uint16_t blocksize, uint16_t blocknumber, \
                                             hal_sdio_emmc_irq_user_callback_struct *p_user_func);
/* erase the eMMC device */
int32_t hal_sdio_emmc_erase(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t startaddr, uint32_t endaddr);
/* abort an ongoing data transfer */
int32_t hal_sdio_emmc_abort(hal_sdio_emmc_dev_struct *emmc_dev);
/* abort an ongoing data transfer in interrupt */
int32_t hal_sdio_emmc_abort_interrupt(hal_sdio_emmc_dev_struct *emmc_dev);
/* perform specific commands sequence for the different type of erase */
int32_t hal_sdio_emmc_sequence_erase(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t erase_type,
                                uint32_t startaddr, uint32_t endaddr);
/* perform sanitize operation on the device */
int32_t hal_sdio_emmc_sanitize(hal_sdio_emmc_dev_struct *emmc_dev);
/* configure the SRT in the Extended CSD register */
int32_t hal_sdio_emmc_srt_config(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t srt_mode);
/* gets the supported values of the the SRT */
int32_t hal_sdio_emmc_supported_srt_get(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *srt);
/* switch the device from standby state to sleep state */
int32_t hal_sdio_emmc_device_sleep(hal_sdio_emmc_dev_struct *emmc_dev);
/* switch the device from sleep State to standby state */
int32_t hal_sdio_emmc_device_awake(hal_sdio_emmc_dev_struct *emmc_dev);
/* get the information of the card which are stored on */
int32_t hal_sdio_emmc_card_cid_get(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_cardcid_struct *p_cid);
/* get the information of the card which are stored on */
int32_t hal_sdio_emmc_card_csd_get(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_cardcsd_struct *p_csd);
/* get the EXT_CSD register detailed information of the eMMC */
int32_t hal_sdio_emmc_card_extcsd_get(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *ext_csd);
/* get the detailed information of the SD card based on received CID and CSD */
int32_t hal_sdio_emmc_card_information_get(hal_sdio_emmc_dev_struct *emmc_dev, hal_sdio_emmc_cardinfo_struct *p_cardinfo);
/* get the state which the card is in */
int32_t hal_sdio_emmc_cardstate_get(hal_sdio_emmc_dev_struct *emmc_dev, uint8_t *pcardstate);
/* get the card status whose response format R1 contains a 32-bit field */
int32_t hal_sdio_emmc_cardstatus_get(hal_sdio_emmc_dev_struct *emmc_dev, uint32_t *pcardstatus);

/* return the emmc state */
hal_sdio_emmc_state_enum hal_sdio_emmc_state_get(hal_sdio_emmc_dev_struct *emmc_dev);
/* return the emmc error code */
hal_sdio_emmc_error_enum hal_sdio_emmc_error_get(hal_sdio_emmc_dev_struct *emmc_dev);

#endif /* GD32H7XX_HAL_SDIO_EMMC_H */
