/*!
    \file    gd32h7xx_hal_rspdif.h
    \brief   definitions for the RSPDIF

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_RSPDIF_H
#define GD32H7XX_HAL_RSPDIF_H

#include "gd32h7xx_hal.h"

/* RSPDIF definitions */
#define RSPDIF                              RSPDIF_BASE                              /*!< RSPDIF base address */

/* registers definitions */
#define RSPDIF_CTL                          REG32(RSPDIF + 0x00000000U)              /*!< RSPDIF control register */
#define RSPDIF_INTEN                        REG32(RSPDIF + 0x00000004U)              /*!< RSPDIF interrupt enable register */
#define RSPDIF_STAT                         REG32(RSPDIF + 0x00000008U)              /*!< RSPDIF status register */
#define RSPDIF_STATC                        REG32(RSPDIF + 0x0000000CU)              /*!< RSPDIF status flag clear register */
#define RSPDIF_DATA                         REG32(RSPDIF + 0x00000010U)              /*!< RSPDIF rx data register */
#define RSPDIF_CHSTAT                       REG32(RSPDIF + 0x00000014U)              /*!< RSPDIF rx channel status register */
#define RSPDIF_DTH                          REG32(RSPDIF + 0x00000018U)              /*!< RSPDIF rx data threshold register */

#define RSPDIF_DATA_F0                      REG32(RSPDIF + 0x00000010U)              /*!< RSPDIF rx data register format 0 */
#define RSPDIF_DATA_F1                      REG32(RSPDIF + 0x00000010U)              /*!< RSPDIF rx data register format 1 */
#define RSPDIF_DATA_F2                      REG32(RSPDIF + 0x00000010U)              /*!< RSPDIF rx data register format 2 */

/* bits definitions */
/* RSPDIF_CTL */
#define RSPDIF_CTL_RXCFG                    BITS(0,1)                                /*!< RSPDIF configuration */
#define RSPDIF_CTL_DMAREN                   BIT(2)                                   /*!< RSPDIF receiver DMA enable for data flow */
#define RSPDIF_CTL_RXSTEOMEN                BIT(3)                                   /*!< RSPDIF rx stereo mode enable */
#define RSPDIF_CTL_RXDF                     BITS(4,5)                                /*!< RSPDIF rx data format selection */
#define RSPDIF_CTL_PNCPEN                   BIT(6)                                   /*!< RSPDIF parity error bit no copy enable bit */
#define RSPDIF_CTL_VNCPEN                   BIT(7)                                   /*!< RSPDIF validity bit no copy enable bit */
#define RSPDIF_CTL_CUNCPEN                  BIT(8)                                   /*!< RSPDIF channel status and user bits no copy enable bit */
#define RSPDIF_CTL_PTNCPEN                  BIT(9)                                   /*!< RSPDIF preamble type bits no copy enable bit */
#define RSPDIF_CTL_DMACBEN                  BIT(10)                                  /*!< RSPDIF control buffer DMA enable for control flow */
#define RSPDIF_CTL_CFCHSEL                  BIT(11)                                  /*!< RSPDIF the control flow acquires channel state source selection */
#define RSPDIF_CTL_MAXRT                    BITS(12,13)                              /*!< RSPDIF maximum number of retries allowed during the RSPDIF synchronization phase */
#define RSPDIF_CTL_WFRXA                    BIT(14)                                  /*!< RSPDIF wait for the four valid transition signal of the selected RSPDIF channel */
#define RSPDIF_CTL_RXCHSEL                  BITS(16,18)                              /*!< RSPDIF input channel selection */
#define RSPDIF_CTL_SCKEN                    BIT(20)                                  /*!< RSPDIF symbol clock enable */
#define RSPDIF_CTL_BKSCKEN                  BIT(21)                                  /*!< RSPDIF backup symbol clock enable */

/* RSPDIF_INTEN */
#define RSPDIF_INTEN_RBNEIE                 BIT(0)                                   /*!< RSPDIF_DATA register no empty interrupt enable */
#define RSPDIF_INTEN_CBNEIE                 BIT(1)                                   /*!< RSPDIF_CHSTAT register no empty interrupt enable */
#define RSPDIF_INTEN_PERRIE                 BIT(2)                                   /*!< RSPDIF parity error interrupt enable */
#define RSPDIF_INTEN_RXORERRIE              BIT(3)                                   /*!< RSPDIF rx overrun error interrupt enable */
#define RSPDIF_INTEN_SYNDBIE                BIT(4)                                   /*!< RSPDIF synchronization block detected interrupt enable */
#define RSPDIF_INTEN_SYNDOIE                BIT(5)                                   /*!< RSPDIF synchronization done interrupt enable */
#define RSPDIF_INTEN_RXDCERRIE              BIT(6)                                   /*!< RSPDIF data decoding error interrupt enable */

/* RSPDIF_STAT */
#define RSPDIF_STAT_RBNE                    BIT(0)                                   /*!< RSPDIF_DATA register is not empty */
#define RSPDIF_STAT_CBNE                    BIT(1)                                   /*!< RSPDIF_CHSTAT control buffer is not empty */
#define RSPDIF_STAT_PERR                    BIT(2)                                   /*!< RSPDIF parity error */
#define RSPDIF_STAT_RXORERR                 BIT(3)                                   /*!< RSPDIF rx overrun error */
#define RSPDIF_STAT_SYNDB                   BIT(4)                                   /*!< RSPDIF synchronization block detected */
#define RSPDIF_STAT_SYNDO                   BIT(5)                                   /*!< RSPDIF synchronization done */
#define RSPDIF_STAT_FRERR                   BIT(6)                                   /*!< RSPDIF frame error */
#define RSPDIF_STAT_SYNERR                  BIT(7)                                   /*!< RSPDIF synchronization error */
#define RSPDIF_STAT_TMOUTERR                BIT(8)                                   /*!< RSPDIF timeout error */
#define RSPDIF_STAT_CKCNT5                  BITS(16,30)                              /*!< the number of consecutive time clock cycles */

/* RSPDIF_STATC */
#define RSPDIF_STATC_PERRC                  BIT(2)                                   /*!< RSPDIF clears the parity error flag */
#define RSPDIF_STATC_RXORERRC               BIT(3)                                   /*!< RSPDIF clears the rx overrun error flag */
#define RSPDIF_STATC_SYNDBC                 BIT(4)                                   /*!< RSPDIF clears the synchronization block detected flag  */
#define RSPDIF_STATC_SYNDOC                 BIT(5)                                   /*!< RSPDIF clears the synchronization done flag  */

/* RSPDIF_DATA */
/* RSPDIF_DATA_F0 */
#define RSPDIF_DATA_F0_DATA0                BITS(0,23)                               /*!< data value */
#define RSPDIF_DATA_F0_P                    BIT(24)                                  /*!< parity error bit */
#define RSPDIF_DATA_F0_V                    BIT(25)                                  /*!< validity bit */
#define RSPDIF_DATA_F0_U                    BIT(26)                                  /*!< user bit */
#define RSPDIF_DATA_F0_C                    BIT(27)                                  /*!< channel status bit */
#define RSPDIF_DATA_F0_PREF                 BITS(28,29)                              /*!< preamble type */

/* RSPDIF_DATA_F1 */
#define RSPDIF_DATA_F1_P                    BIT(0)                                   /*!< parity error bit */
#define RSPDIF_DATA_F1_V                    BIT(1)                                   /*!< validity bit */
#define RSPDIF_DATA_F1_U                    BIT(2)                                   /*!< user bit */
#define RSPDIF_DATA_F1_C                    BIT(3)                                   /*!< channel status bit */
#define RSPDIF_DATA_F1_PREF                 BITS(4,5)                                /*!< preamble type */
#define RSPDIF_DATA_F1_DATA0                BITS(8,31)                               /*!< data value */

/* RSPDIF_DATA_F2 */
#define RSPDIF_DATA_F2_DATA1                BITS(0,15)                               /*!< stereo mode: this field contains the channel B data mono mode: this field contains the more recent value */
#define RSPDIF_DATA_F2_DATA2                BITS(16,31)                              /*!< stereo mode: this field contains the channel A data mono mode: this field contains the oldest value */

/* RSPDIF_CHSTAT */
#define RSPDIF_CHSTAT_USER                  BITS(0,15)                               /*!< user data information */
#define RSPDIF_CHSTAT_CHS                   BITS(16,23)                              /*!< channel A status information */
#define RSPDIF_CHSTAT_SOB                   BIT(24)                                  /*!< start of block */

/* RSPDIF_DTH */
#define RSPDIF_DTH_THHI                     BITS(0,12)                               /*!< high threshold */
#define RSPDIF_DTH_THLO                     BITS(16,28)                              /*!< low threshold */

/* RSPDIF configuration */
/* @PARA: rspdif_cfg */
/* @DEFINE: the RSPDIF configuration*/
#define CTL_RXCFG(regval)                   (BITS(0,1) & ((uint32_t)(regval)))
#define RSPDIF_STATE_IDLE                   CTL_RXCFG(0)                             /*!< disable RSPDIF */
#define RSPDIF_STATE_SYNC                   CTL_RXCFG(1)                             /*!< enable RSPDIF synchronization only */
#define RSPDIF_STATE_RCV                    CTL_RXCFG(3)                             /*!< enable RSPDIF */

/* rx stereo mode enable */
/* @STRUCT_MEMBER: sound_mode */
/* @DEFINE: the RSPDIF sound mode selection */
#define RSPDIF_STEREOMODE_DISABLE           ((uint32_t)0x00000000U)                  /*!< MONO mode */
#define RSPDIF_STEREOMODE_ENABLE            ((uint32_t)RSPDIF_CTL_RXSTEOMEN)         /*!< STEREO mode */

/* rx data format selection */
/* @STRUCT_MEMBER: sample_format */
/* @DEFINE: the RSPDIF receive data format */
#define CTL_RXDF(regval)                    (BITS(4,5) & ((uint32_t)(regval) << 4))
#define RSPDIF_DATAFORMAT_LSB               CTL_RXDF(0)                              /*!< the data format is described in the RSPDIF_DATA_F0 register, audio data is right aligned (LSB) */
#define RSPDIF_DATAFORMAT_MSB               CTL_RXDF(1)                              /*!< the data format is described in the RSPDIF_DATA_F1 register, audio data is left aligned (MSB) */
#define RSPDIF_DATAFORMAT_32BITS            CTL_RXDF(2)                              /*!< the data format is described in the RSPDIF_DATA_F2 register, which packs two 16-bit audio data into one 32-bit data */

/* whether copy preamble type bit into RSPDIF_DATA */
/* @STRUCT_MEMBER: pre_type */
/* @DEFINE: copy the preamble type */
#define RSPDIF_PREAMBLE_TYPE_MASK_OFF       ((uint32_t)0x00000000U)                  /*!< copy the preamble type bit into the RSPDIF_DATA */
#define RSPDIF_PREAMBLE_TYPE_MASK_ON        ((uint32_t)RSPDIF_CTL_PTNCPEN)           /*!< do not copy the preamble type bit into the RSPDIF_DATA, but write 0 instead */

/* @STRUCT_MEMBER: channel_status_bit */
/* @DEFINE: whether copy channel status and user bits into RSPDIF_DATA */
#define RSPDIF_CHANNEL_STATUS_MASK_OFF      ((uint32_t)0x00000000U)                  /*!< copy the channel status and user bit into the RSPDIF_DATA */
#define RSPDIF_CHANNEL_STATUS_MASK_ON       ((uint32_t)RSPDIF_CTL_CUNCPEN)           /*!< do not copy the channel status and user bit into the RSPDIF_DATA, but write 0 instead */

/* @STRUCT_MEMBER: validity_bit */
/* @DEFINE: whether copy validity bit into RSPDIF_DATA */
#define RSPDIF_VALIDITY_MASK_OFF            ((uint32_t)0x00000000U)                  /*!< copy the validity bit into the RSPDIF_DATA */
#define RSPDIF_VALIDITY_MASK_ON             ((uint32_t)RSPDIF_CTL_VNCPEN)            /*!< do not copy the validity bit into the RSPDIF_DATA, but write 0 instead */

/* @STRUCT_MEMBER: parity_error_bit */
/* @DEFINE: whether copy parity error bit into RSPDIF_DATA */
#define RSPDIF_PERROR_MASK_OFF              ((uint32_t)0x00000000U)                  /*!< copy the parity error bit into the RSPDIF_DATA */
#define RSPDIF_PERROR_MASK_ON               ((uint32_t)RSPDIF_CTL_PNCPEN)            /*!< do not copy the parity error bit into the RSPDIF_DATA, but write 0 instead */

/* @STRUCT_MEMBER: max_retrie */
/* @DEFINE: maximum number of retries allowed during the RSPDIF synchronization */
#define CTL_MAXRT(regval)                   (BITS(12,13) & ((uint32_t)(regval) << 12))
#define RSPDIF_MAXRETRIES_NONE              CTL_MAXRT(0)                             /*!< no retry */
#define RSPDIF_MAXRETRIES_3                 CTL_MAXRT(1)                             /*!< allow up to 3 retries */
#define RSPDIF_MAXRETRIES_15                CTL_MAXRT(2)                             /*!< allow up to 15 retries */
#define RSPDIF_MAXRETRIES_63                CTL_MAXRT(3)                             /*!< allow up to 63 retries */

/* @STRUCT_MEMBER: wait_activity */
/* @DEFINE: wait for the four valid transition signal of the selected RSPDIF channel */
#define RSPDIF_WAIT_FOR_ACTIVITY_OFF        ((uint32_t)0x00000000U)                  /*!< RSPDIF does not wait for the valid conversion signal from the selected RSPDIF channel */
#define RSPDIF_WAIT_FOR_ACTIVITY_ON         ((uint32_t)RSPDIF_CTL_WFRXA)             /*!< RSPDIF wait for the valid conversion signal from the selected RSPDIF channel */

/* @STRUCT_MEMBER: input_sel */
/* @DEFINE: input channel selection */
#define CTL_RXCHSEL(regval)                 (BITS(16,18) & ((uint32_t)(regval) << 16))
#define RSPDIF_INPUT_IN0                    CTL_RXCHSEL(0)                           /*!< RSPDIF_CH0 selected */
#define RSPDIF_INPUT_IN1                    CTL_RXCHSEL(1)                           /*!< RSPDIF_CH1 selected */
#define RSPDIF_INPUT_IN2                    CTL_RXCHSEL(2)                           /*!< RSPDIF_CH2 selected */
#define RSPDIF_INPUT_IN3                    CTL_RXCHSEL(3)                           /*!< RSPDIF_CH3 selected */

/* @STRUCT_MEMBER: symbol_clk */
/* @DEFINE: symbol clock enable */
#define RSPDIF_SYMBOL_CLK_OFF               ((uint32_t)0x00000000U)                  /*!< the RSPDIF does not generate a symbol clock */
#define RSPDIF_SYMBOL_CLK_ON                RSPDIF_CTL_SCKEN                         /*!< the RSPDIF generates a symbol clock */

/* @STRUCT_MEMBER: bak_symbol_clk */
/* @DEFINE: backup symbol clock enable */
#define RSPDIF_BACKUP_SYMBOL_CLK_OFF        ((uint32_t)0x00000000U)                  /*!< the RSPDIF does not generate a backup symbol clock */
#define RSPDIF_BACKUP_SYMBOL_CLK_ON         RSPDIF_CTL_BKSCKEN                       /*!< the RSPDIF generates a backup symbol clock if SCKEN = 1 */

/* @STRUCT_MEMBER: channel_sel */
/* @DEFINE: the control flow acquires channel state source selection */
#define RSPDIF_CHANNEL_A                    ((uint32_t)0x00000000U)                  /*!< gets channel status from channel A */
#define RSPDIF_CHANNEL_B                    ((uint32_t)RSPDIF_CTL_CFCHSEL)           /*!< gets channel status from channel B */

/* RSPDIF preamble type */
#define RSPDIF_PREAMBLE_NONE                ((uint32_t)0x00000000U)                  /*!< RSPDIF preamble received not used */
#define RSPDIF_PREAMBLE_B                   ((uint32_t)0x00000001U)                  /*!< RSPDIF preamble B received */
#define RSPDIF_PREAMBLE_M                   ((uint32_t)0x00000002U)                  /*!< RSPDIF preamble M received */
#define RSPDIF_PREAMBLE_W                   ((uint32_t)0x00000003U)                  /*!< RSPDIF preamble W received */

/* RSPDIF interrupt enable or disable constants definitions */
#define RSPDIF_INT_RBNE                     RSPDIF_INTEN_RBNEIE                      /*!< RSPDIF rx buffer no empty interrupt enable */
#define RSPDIF_INT_CBNE                     RSPDIF_INTEN_CBNEIE                      /*!< RSPDIF rx control buffer no empty interrupt enable */
#define RSPDIF_INT_PERR                     RSPDIF_INTEN_PERRIE                      /*!< RSPDIF parity error interrupt enable */
#define RSPDIF_INT_RXORERR                  RSPDIF_INTEN_RXORERRIE                   /*!< RSPDIF rx overrun error interrupt enable */
#define RSPDIF_INT_SYNDB                    RSPDIF_INTEN_SYNDBIE                     /*!< RSPDIF synchronization block detected interrupt enable */
#define RSPDIF_INT_SYNDO                    RSPDIF_INTEN_SYNDOIE                     /*!< RSPDIF synchronization done interrupt enable */
#define RSPDIF_INT_RXDCERR                  RSPDIF_INTEN_RXDCERRIE                   /*!< RSPDIF data decoding error interrupt enable */

/* RSPDIF flags */
#define RSPDIF_FLAG_RBNE                    RSPDIF_STAT_RBNE                         /*!< RSPDIF rx buffer is not empty */
#define RSPDIF_FLAG_CBNE                    RSPDIF_STAT_CBNE                         /*!< RSPDIF rx control buffer is not empty */
#define RSPDIF_FLAG_PERR                    RSPDIF_STAT_PERR                         /*!< RSPDIF parity error */
#define RSPDIF_FLAG_RXORERR                 RSPDIF_STAT_RXORERR                      /*!< RSPDIF rx overrun error */
#define RSPDIF_FLAG_SYNDB                   RSPDIF_STAT_SYNDB                        /*!< RSPDIF synchronization block detected */
#define RSPDIF_FLAG_SYNDO                   RSPDIF_STAT_SYNDO                        /*!< RSPDIF synchronization done */
#define RSPDIF_FLAG_FRERR                   RSPDIF_STAT_FRERR                        /*!< RSPDIF frame error */
#define RSPDIF_FLAG_SYNERR                  RSPDIF_STAT_SYNERR                       /*!< RSPDIF synchronization error */
#define RSPDIF_FLAG_TMOUTERR                RSPDIF_STAT_TMOUTERR                     /*!< RSPDIF time out error */

/* RSPDIF interrupt flags */
#define RSPDIF_INT_FLAG_RBNE                RSPDIF_STAT_RBNE                         /*!< RSPDIF rx buffer no empty interrupt flag */
#define RSPDIF_INT_FLAG_CBNE                RSPDIF_STAT_CBNE                         /*!< RSPDIF rx control buffer no empty interrupt flag */
#define RSPDIF_INT_FLAG_PERR                RSPDIF_STAT_PERR                         /*!< RSPDIF parity error interrupt flag */
#define RSPDIF_INT_FLAG_RXORERR             RSPDIF_STAT_RXORERR                      /*!< RSPDIF rx overrun error interrupt flag */
#define RSPDIF_INT_FLAG_SYNDB               RSPDIF_STAT_SYNDB                        /*!< RSPDIF synchronization block detected interrupt flag */
#define RSPDIF_INT_FLAG_SYNDO               RSPDIF_STAT_SYNDO                        /*!< RSPDIF synchronization done interrupt flag */
#define RSPDIF_INT_FLAG_FRERR               RSPDIF_STAT_FRERR                        /*!< RSPDIF frame error interrupt flag */
#define RSPDIF_INT_FLAG_SYNERR              RSPDIF_STAT_SYNERR                       /*!< RSPDIF synchronization error interrupt flag */
#define RSPDIF_INT_FLAG_TMOUTERR            RSPDIF_STAT_TMOUTERR                     /*!< RSPDIF time out error interrupt flag */

/* RSPDIF error type enum */
typedef enum {
    HAL_RSPDIF_ERROR_NONE     = (uint32_t)0x00000000U,                               /*!< no error */
    HAL_RSPDIF_ERROR_FRERR    = (uint32_t)0x00000001U,                               /*!< frame error */
    HAL_RSPDIF_ERROR_SYNERR   = (uint32_t)0x00000002U,                               /*!< synchronization error */
    HAL_RSPDIF_ERROR_TMOUTERR = (uint32_t)0x00000003U,                               /*!< timeout error */
    HAL_RSPDIF_ERROR_DMA      = (uint32_t)0x00000004U,                               /*!< DMA error */
    HAL_RSPDIF_ERROR_UNKNOWN  = (uint32_t)0x00000005U                                /*!< unknown error */
} hal_rspdif_error_enum;

/* RSPDIF state enum */
typedef enum {
    HAL_RSPDIF_STATE_NONE = 0,                                                       /*!< NONE(default value) */
    HAL_RSPDIF_STATE_RESET,                                                          /*!< RESET */
    HAL_RSPDIF_STATE_BUSY,                                                           /*!< BUSY */
    HAL_RSPDIF_STATE_BUSY_RX,                                                        /*!< RSPDIF internal Data Flow RX process is ongoing */
    HAL_RSPDIF_STATE_BUSY_CX,                                                        /*!< RSPDIF internal Control Flow RX process is ongoing */
    HAL_RSPDIF_STATE_ERROR,                                                          /*!< ERROR */
    HAL_RSPDIF_STATE_READY,                                                          /*!< READY */
} hal_rspdif_state_enum;

/* RSPDIF structure type enum */
typedef enum {
    HAL_RSPDIF_INIT_STRUCT = 0,                                                      /*!< RSPDIF initialization structure */
    HAL_RSPDIF_DATA_STRUCT,                                                          /*!< RSPDIF data structure */
    HAL_RSPDIF_IRQ_STRUCT,                                                           /*!< RSPDIF interrupt handle callback structure */
    HAL_RSPDIF_DEV_STRUCT,                                                           /*!< RSPDIF device information structure */
    HAL_RSPDIF_IRQ_USER_CALLBACK_STRUCT,                                             /*!< RSPDIF user interrupt callback structure */
    HAL_RSPDIF_DATAFORMAT_STRUCT                                                     /*!< RSPDIF data format structure */
} hal_rspdif_struct_type_enum;

/* constants definitions */

/* @PARA: rspdif_struct */
/* @STRUCT:  RSPDIF initialization parameter structure definitions */
typedef struct {
    uint32_t input_sel;                                                              /*!< the RSPDIF input selection */
    uint32_t max_retrie;                                                             /*!< the RSPDIF maximum allowed re-tries during synchronization phase */
    uint32_t wait_activity;                                                          /*!< the RSPDIF wait for activity on the selected input */
    uint32_t channel_sel;                                                            /*!< whether swapping the channel status from channel A or B */
    uint32_t sample_format;                                                          /*!< the RSPDIF data samples format (LSB, MSB, ...) */
    uint32_t sound_mode;                                                             /*!< the RSPDIF is in stereo or mono mode */
    uint32_t pre_type;                                                               /*!< whether opt the preamble type value into the RSPDIF_DATA */
    uint32_t channel_status_bit;                                                     /*!< whether the channel status and user bits are copied or not into the received frame */
    uint32_t validity_bit;                                                           /*!< whether the validity bit is copied or not into the received frame */
    uint32_t parity_error_bit;                                                       /*!< whether the parity error bit is copied or not into the received frame */
    uint32_t symbol_clk;                                                             /*!< the RSPDIF symbol clock generation */
    uint32_t bak_symbol_clk;                                                         /*!< the RSPDIF backup symbol clock generation */
} hal_rspdif_init_struct;

/* @STRUCT: RSPDIF set data format structure definition */
typedef struct
{
    uint32_t DataFormat;                                                             /*!< specifies the Data samples format (LSB, MSB, ...) */
    uint32_t StereoMode;                                                             /*!< specifies whether the peripheral is in stereo or mono mode */
    uint32_t PreambleTypeMask;                                                       /*!< specifies whether The preamble type bits are copied or not into the received frame */
    uint32_t ChannelStatusMask;                                                      /*!< specifies whether the channel status and user bits are copied or not into the received frame */
    uint32_t ValidityBitMask;                                                        /*!< specifies whether the validity bit is copied or not into the received frame */
    uint32_t ParityErrorMask;                                                        /*!< specifies whether the parity error bit is copied or not into the received frame */
} hal_rspdif_dataformat_struct;

/* RSPDIF data parameter structure definitions */
typedef struct {
    uint32_t format;                                                                 /*!< the data format */
    uint32_t preamble;                                                               /*!< the preamble type */
    uint32_t channel_status;                                                         /*!< channel status bit */
    uint32_t user_bit;                                                               /*!< user bit */
    uint32_t validity;                                                               /*!< validity bit */
    uint32_t parity_err;                                                             /*!< parity error bit */
    uint32_t data0;                                                                  /*!< data value 0 */
    uint32_t data1;                                                                  /*!< data value 1 */
} hal_rspdif_data_struct;

/* RSPDIF interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb rspdif_rx_cpl_handle;                                     /*!< RSPDIF data flow completed interrupt */
    __IO hal_irq_handle_cb rspdif_rx_half_handle;                                    /*!< RSPDIF data flow half completed interrupt */
    __IO hal_irq_handle_cb rspdif_cx_cpl_handle;                                     /*!< RSPDIF control flow completed interrupt */
    __IO hal_irq_handle_cb rspdif_cx_half_handle;                                    /*!< RSPDIF control flow half completed interrupt */
    __IO hal_irq_handle_cb rspdif_error_handle;                                      /*!< data decoding error interrupt */
} hal_rspdif_irq_struct;

/* RSPDIF DMA callback function pointer structure */
typedef void (*hal_rspdif_dma_handle_cb)(void *ptr);

/* RSPDIF device information structure */
typedef struct {
    hal_rspdif_irq_struct rspdif_irq;                                                /*!< RSPDIF device interrupt callback function pointer structure */
    __IO uint32_t *p_rx_buffer;                                                      /*!< RSPDIF rx transfer buffer */
    __IO uint16_t rx_xfer_size;                                                      /*!< RSPDIF rx transfer size */
    __IO uint16_t rx_xfer_count;                                                     /*!< RSPDIF rx transfer counter */
    __IO uint32_t *p_cs_buffer;                                                      /*!< RSPDIF rx transfer buffer */
    __IO uint16_t cs_xfer_size;                                                      /*!< RSPDIF rx transfer size */
    __IO uint16_t cs_xfer_count;                                                     /*!< RSPDIF rx transfer counter */
    hal_dma_dev_struct *p_dma_rspdif;                                                /*!< DMA device information structure */
    hal_rspdif_error_enum error_state;                                               /*!< RSPDIF error state */
    hal_rspdif_state_enum state;                                                     /*!< RSPDIF state */
    hal_mutex_enum mutex;                                                            /*!< lock */
    void *priv;                                                                      /*!< priv data */
    void *rspdif_rx_cpl_callback;                                                    /*!< RSPDIF data flow completed callback */
    void *rspdif_rx_half_callback;                                                   /*!< RSPDIF data flow half completed callback */
    void *rspdif_cx_cpl_callback;                                                    /*!< RSPDIF control flow completed callback */
    void *rspdif_cx_half_callback;                                                   /*!< RSPDIF control flow half completed callback */
    void *rspdif_error_callback;                                                     /*!< data decoding error interrupt */
} hal_rspdif_dev_struct;

/* RSPDIF interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb rspdif_rx_cpl_callback;                                   /*!< RSPDIF data flow completed callback */
    __IO hal_irq_handle_cb rspdif_rx_half_callback;                                  /*!< RSPDIF data flow half completed callback */
    __IO hal_irq_handle_cb rspdif_cx_cpl_callback;                                   /*!< RSPDIF control flow completed callback */
    __IO hal_irq_handle_cb rspdif_cx_half_callback;                                  /*!< RSPDIF control flow half completed callback */
    __IO hal_irq_handle_cb rspdif_error_callback;                                    /*!< data decoding error interrupt */
} hal_rspdif_irq_user_callback_struct;

/* function declarations */

/* reset the RSPDIF */
int32_t hal_rspdif_deinit(hal_rspdif_dev_struct *rspdif_dev);

/* @FUNCTION: initialize the RSPDIF parameters */
int32_t hal_rspdif_init(hal_rspdif_dev_struct *rspdif_dev, hal_rspdif_init_struct *rspdif_struct);
/* @FUNCTION: specifies the RSPDIF peripheral state */
int32_t hal_rspdif_enable(hal_rspdif_dev_struct *rspdif_dev, uint32_t rspdif_cfg);
/* @END */

/* disable RSPDIF */
void hal_rspdif_disable(void);
/* initialize the RSPDIF device structure with the default values */
int32_t hal_rspdif_struct_init(hal_rspdif_struct_type_enum hal_struct_type, void *p_struct);
/* initialize the parameters of RSPDIF structure with the default values */
int32_t hals_rspdif_struct_para_init(hal_rspdif_init_struct *rspdif_struct);
/*  enable RSPDIF and start the conversion of routine sequence with DMA */
int32_t hal_rspdif_start_dma(hal_rspdif_dev_struct *rspdif_dev, uint32_t *src_addr, uint32_t *dst_addr, uint16_t length, \
                             hal_rspdif_irq_user_callback_struct *p_user_func);
/* stop RSPDIF */
int32_t hal_rspdif_stop(hal_rspdif_dev_struct *rspdif_dev);
/* stop RSPDIF DMA request*/
int32_t hal_rspdif_stop_dma(hal_rspdif_dev_struct *rspdif_dev);
/* stop RSPDIF DMA request interrupt */
int32_t hal_rspdif_stop_dma_interrupt(hal_rspdif_dev_struct *rspdif_dev);
/* RSPDIF interrupt handler content function,which is merely used in rspdif_handler */
int32_t hal_rspdif_irq(hal_rspdif_dev_struct *rspdif_dev);
/* set user-defined interrupt callback function, which will be registered and called when corresponding interrupt be triggered */
int32_t hal_rspdif_irq_handle_set(hal_rspdif_dev_struct *rspdif_dev, hal_rspdif_irq_struct *p_irq);
/* reset all user-defined interrupt callback function, which will be registered and called when corresponding interrupt be triggered */
int32_t hal_rspdif_irq_handle_all_reset(hal_rspdif_dev_struct *rspdif_dev);
/* start RSPDIF with interrupt */
int32_t hal_rspdif_start_interrupt(hal_rspdif_dev_struct *rspdif_dev, hal_rspdif_irq_user_callback_struct *p_user_func);
/* stop RSPDIF with interrupt */
int32_t hal_rspdif_stop_interrupt(hal_rspdif_dev_struct *rspdif_dev);
/* RSPDIF receives an amount of data (data flow) in blocking mode */
int32_t hal_rspdif_data_flow_receive(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size, \
                                     uint32_t timeout);
/* RSPDIF receives an amount of data (control flow) in blocking mode */
int32_t hal_rspdif_ctrl_flow_receive(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size, \
                                     uint32_t timeout);
/* RSPDIF receives an amount of data (data flow) in blocking mode with interrupt */
int32_t hal_rspdif_data_flow_receive_interrupt(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size);
/* RSPDIF receives an amount of data (control flow) in blocking mode with interrupt */
int32_t hal_rspdif_ctrl_flow_receive_interrupt(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size);
/* RSPDIF receives an amount of data (data flow) in blocking mode with dma */
int32_t hal_rspdif_data_flow_receive_dma(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size);
/* RSPDIF receives an amount of data (control flow) in blocking mode with dma */
int32_t hal_rspdif_ctrl_flow_receive_dma(hal_rspdif_dev_struct *rspdif_dev, uint32_t *p_data, uint16_t size);
/* get the RSPDIF state */
hal_rspdif_state_enum hal_rspdif_state_get(hal_rspdif_dev_struct *rspdif_dev);
/* get the RSPDIF error state */
hal_rspdif_error_enum hal_rspdif_error_get(hal_rspdif_dev_struct *rspdif_dev);
/* RSPDIF read data */
int32_t hal_rspdif_data_read(hal_rspdif_data_struct *data_struct);
/* configure the RSPDIF receive data format */
int32_t hal_rspdif_dataformat_set(hal_rspdif_dataformat_struct *rspdif_format_struct);

/* enable RSPDIF symbol clock */
void hals_rspdif_symbol_clock_enable(void);
/* disable RSPDIF symbol clock */
void hals_rspdif_symbol_clock_disable(void);
/* enable RSPDIF backup symbol clock */
void hals_rspdif_backup_symbol_clock_enable(void);
/* disable RSPDIF backup symbol clock */
void hals_rspdif_backup_symbol_clock_disable(void);

/* DMA functions */
/* enable the RSPDIF receiver DMA */
void hals_rspdif_dma_enable(void);
/* disable the RSPDIF receiver DMA */
void hals_rspdif_dma_disable(void);
/* enable the RSPDIF control buffer DMA */
void hals_rspdif_control_buffer_dma_enable(void);
/* disable the RSPDIF control buffer DMA */
void hals_rspdif_control_buffer_dma_disable(void);

/* information acquisition functions */
/* get duration of 5 symbols counted using rspdif_ck */
uint16_t hals_rspdif_duration_of_symbols_get(void);
/* get user data information */
uint16_t hals_rspdif_user_data_get(void);
/* get channel status information */
uint8_t hals_rspdif_channel_status_get(void);
/* get start of block */
FlagStatus hals_rspdif_start_block_status_get(void);
/* get threshold low estimation */
uint16_t hals_rspdif_low_threshold_get(void);
/* get threshold high estimation */
uint16_t hals_rspdif_high_threshold_get(void);

/* flag and interrupt functions */
/* get RSPDIF flag status */
FlagStatus hals_rspdif_flag_get(uint32_t flag);
/* clear RSPDIF flag */
void hals_rspdif_flag_clear(uint32_t flag);
/* enable RSPDIF interrupt */
void hals_rspdif_interrupt_enable(uint32_t interrupt);
/* disable RSPDIF interrupt */
void hals_rspdif_interrupt_disable(uint32_t interrupt);
/* get RSPDIF interrupt flag status */
FlagStatus hals_rspdif_interrupt_flag_get(uint32_t int_flag);
/* clear RSPDIF interrupt flag status */
void hals_rspdif_interrupt_flag_clear(uint32_t int_flag);

#endif /* GD32H7XX_HAL_RSPDIF_H */
