/*!
    \file    gd32h7xx_hal_nvic.h
    \brief   definitions for the NVIC

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_NVIC_H
#define GD32H7XX_HAL_NVIC_H

#include "gd32h7xx_hal.h"

/* constants definitions */
/* set the RAM and FLASH base address */
#define NVIC_VECTTAB_RAM                     ((uint32_t)0x24000000U)                     /*!< RAM base address */
#define NVIC_VECTTAB_FLASH                   ((uint32_t)0x08000000U)                     /*!< Flash base address */

/* set the NVIC vector table offset mask */
#define NVIC_VECTTAB_OFFSET_MASK             ((uint32_t)0x1FFFFF80U)                     /*!< NVIC vector table offset mask */

/* the register key mask, if you want to do the write operation, you should write 0x5FA to VECTKEY bits */
#define NVIC_AIRCR_VECTKEY_MASK              ((uint32_t)0x05FA0000U)                     /*!< NVIC VECTKEY mask */

/* priority group - define the pre-emption priority and the subpriority */
#define NVIC_PRIGROUP_PRE0_SUB4              ((uint32_t)0x00000007U)                     /*!< 0 bit for pre-emption priority 4 bits for subpriority */
#define NVIC_PRIGROUP_PRE1_SUB3              ((uint32_t)0x00000006U)                     /*!< 1 bit for pre-emption priority 3 bits for subpriority */
#define NVIC_PRIGROUP_PRE2_SUB2              ((uint32_t)0x00000005U)                     /*!< 2 bits for pre-emption priority 2 bits for subpriority */
#define NVIC_PRIGROUP_PRE3_SUB1              ((uint32_t)0x00000004U)                     /*!< 3 bits for pre-emption priority 1 bit for subpriority */
#define NVIC_PRIGROUP_PRE4_SUB0              ((uint32_t)0x00000003U)                     /*!< 4 bits for pre-emption priority 0 bit for subpriority */

/* choose the method to enter or exit the low power mode */
#define SCB_SCR_SLEEPONEXIT                  ((uint8_t)0x02U)                            /*!< choose the the system whether enter low power mode by exiting from ISR */
#define SCB_SCR_SLEEPDEEP                    ((uint8_t)0x04U)                            /*!< choose the the system enter the DEEPSLEEP mode or SLEEP mode */
#define SCB_SCR_SEVONPEND                    ((uint8_t)0x10U)                            /*!< choose the interrupt source that can wake up the low power mode */

#define SCB_LPM_SLEEP_EXIT_ISR               SCB_SCR_SLEEPONEXIT                         /*!< low power mode by exiting from ISR */
#define SCB_LPM_DEEPSLEEP                    SCB_SCR_SLEEPDEEP                           /*!< DEEPSLEEP mode or SLEEP mode */
#define SCB_LPM_WAKE_BY_ALL_INT              SCB_SCR_SEVONPEND                           /*!< wakeup by all interrupt */

/* function declarations */
/* nvic priority function */
/* set NVIC request priority group */
void hal_nvic_irq_priority_group_set(uint32_t nvic_prigroup);
/* get NVIC priority group */
uint32_t hal_nvic_irq_priority_group_get(void);
/* enable NVIC request */
void hal_nvic_irq_enable(IRQn_Type nvic_irq, uint8_t nvic_irq_pre_priority, uint8_t nvic_irq_sub_priority);
/* disable NVIC request */
void hal_nvic_irq_disable(IRQn_Type nvic_irq);
/* set system NVIC priority */
void hal_nvic_set_priority(IRQn_Type nvic_irq, uint32_t nvic_irq_pre_priority, uint32_t nvic_irq_sub_priority);
/* get system nvic priority */
void hal_nvic_priority_get(IRQn_Type nvic_irq, uint32_t nvic_prigroup, \
                           uint32_t *nvic_irq_pre_priority, uint32_t *nvic_irq_sub_priority);
/* get nvic active interrupt */
uint32_t hal_nvic_active_get(IRQn_Type nvic_irq);

/* system function */
/* system reset request */
void hal_nvic_system_reset(void);

/* pend interrupt function */
/* set NVIC pend interrupt */
void hal_nvic_pend_irq_set(IRQn_Type nvic_irq);
/* disable NVIC pend interrupt */
void hal_nvic_pend_irq_clear(IRQn_Type nvic_irq);
/* get NVIC pend interrupt */
uint32_t hal_nvic_pend_irq_get(IRQn_Type nvic_irq);

/* set the NVIC vector table base address */
void hals_nvic_vector_table_set(uint32_t nvic_vict_tab, uint32_t offset);

#endif /* GD32H7XX_HAL_NVIC_H */

