/*!
    \file    gd32h7xx_hal_hau.h
    \brief   definitions for the HAU

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_HAU_H
#define GD32H7XX_HAL_HAU_H

#include "gd32h7xx_hal.h"

/* HAU definitions */
#define HAU                                 HAU_BASE                                   /*!< HAU base address */

/* registers definitions */
#define HAU_CTL                             REG32(HAU + 0x00000000U)                   /*!< control register */
#define HAU_DI                              REG32(HAU + 0x00000004U)                   /*!< data input register */
#define HAU_CFG                             REG32(HAU + 0x00000008U)                   /*!< configuration register */
#define HAU_DO0                             REG32(HAU + 0x0000000CU)                   /*!< data output register 0 */
#define HAU_DO1                             REG32(HAU + 0x00000010U)                   /*!< data output register 1 */
#define HAU_DO2                             REG32(HAU + 0x00000014U)                   /*!< data output register 2 */
#define HAU_DO3                             REG32(HAU + 0x00000018U)                   /*!< data output register 3 */
#define HAU_DO4                             REG32(HAU + 0x0000001CU)                   /*!< data output register 4 */
#define HAU_DO5                             REG32(HAU + 0x00000324U)                   /*!< data output register 5 */
#define HAU_DO6                             REG32(HAU + 0x00000328U)                   /*!< data output register 6 */
#define HAU_DO7                             REG32(HAU + 0x0000032CU)                   /*!< data output register 7 */
#define HAU_INTEN                           REG32(HAU + 0x00000020U)                   /*!< interrupt enable register */
#define HAU_STAT                            REG32(HAU + 0x00000024U)                   /*!< status and interrupt flag register */
#define HAU_CTXS(x)                         REG32(HAU + 0x000000F8U + 0x04U * (x))     /*!< context switch register x, x <= 53 */

/* bits definitions */
/* HAU_CTL */
#define HAU_CTL_START                       BIT(2)                                     /*!< set to 1 to reset the HAU processor core, so that it is ready to start the digest calculation */
#define HAU_CTL_DMAE                        BIT(3)                                     /*!< DMA enable */
#define HAU_CTL_DATAM                       BITS(4,5)                                  /*!< data type mode */
#define HAU_CTL_HMS                         BIT(6)                                     /*!< HAU mode selection */
#define HAU_CTL_ALGM_0                      BIT(7)                                     /*!< algorithm selection bit 0 */
#define HAU_CTL_NWIF                        BITS(8,11)                                 /*!< number of words in the input FIFO */
#define HAU_CTL_DINE                        BIT(12)                                    /*!< DI register not empty */
#define HAU_CTL_MDS                         BIT(13)                                    /*!< multiple DMA selection */
#define HAU_CTL_KLM                         BIT(16)                                    /*!< key length mode */
#define HAU_CTL_ALGM_1                      BIT(18)                                    /*!< algorithm selection bit 1 */

/* HAU_DI */
#define HAU_DI_DI                           BITS(0,31)                                 /*!< message data input */

/* HAU_CFG */
#define HAU_CFG_VBL                         BITS(0,4)                                  /*!< valid bits length in the last word */
#define HAU_CFG_CALEN                       BIT(8)                                     /*!< digest calculation enable */

/* HAU_DOx x=0..7 */
#define HAU_DOX_DOX                         BITS(0,31)                                 /*!< message digest result of hash algorithm */

/* HAU_INTEN */
#define HAU_INTEN_DIIE                      BIT(0)                                     /*!< data input interrupt enable */
#define HAU_INTEN_CCIE                      BIT(1)                                     /*!< calculation completion interrupt enable */

/* HAU_STAT */
#define HAU_STAT_DIF                        BIT(0)                                     /*!< data input interrupt flag */
#define HAU_STAT_CCF                        BIT(1)                                     /*!< digest calculation completion interrupt flag */
#define HAU_STAT_DMAS                       BIT(2)                                     /*!< DMA status */
#define HAU_STAT_BUSY                       BIT(3)                                     /*!< busy bit */

/* constants definitions */
/* hau_ctl register value */
/* @STRUCT_MEMBER: algo */
/* @DEFINE: algorithm */
#define HAU_ALGO_SHA1                       ((uint32_t)0x00000000U)                    /*!< HAU function is SHA1 */
#define HAU_ALGO_MD5                        HAU_CTL_ALGM_0                             /*!< HAU function is MD5 */
#define HAU_ALGO_SHA224                     HAU_CTL_ALGM_1                             /*!< HAU function is SHA224 */
#define HAU_ALGO_SHA256                     (HAU_CTL_ALGM_1 | HAU_CTL_ALGM_0)          /*!< HAU function is SHA256 */

/* @STRUCT_MEMBER: mode */
/* @DEFINE: algorithm mode */
#define HAU_MODE_HASH                       ((uint32_t)0x00000000U)                    /*!< HAU mode is HASH */
#define HAU_MODE_HMAC                       HAU_CTL_HMS                                /*!< HAU mode is HMAC */

/* @STRUCT_MEMBER: datatype */
#define CTL_DATAM_1(regval)                 (BITS(4,5) & ((uint32_t)(regval) << 4U))   /*!< write value to HAU_CTL_DATAM bit field */
/* @DEFINE: hash data type in HAU_DI */
#define HAU_SWAPPING_32BIT                  CTL_DATAM_1(0)                             /*!< no swapping */
#define HAU_SWAPPING_16BIT                  CTL_DATAM_1(1)                             /*!< half-word swapping */
#define HAU_SWAPPING_8BIT                   CTL_DATAM_1(2)                             /*!< bytes swapping */
#define HAU_SWAPPING_1BIT                   CTL_DATAM_1(3)                             /*!< bit swapping */
/* @STOP_ANALYSIS: */

/* HMAC key length */
#define HAU_KEY_SHORTER_64                  ((uint32_t)0x00000000U)                    /*!< HMAC key is <= 64 bytes */
#define HAU_KEY_LONGER_64                  HAU_CTL_KLM                                /*!< HMAC key is > 64 bytes  */

#define GET_CTL_NWIF(regval)                GET_BITS((regval),8,11)                    /*!< get value of HAU_CTL_NWIF bit field */

#define SINGLE_DMA_AUTO_DIGEST              ((uint32_t)0x00000000U)                    /*!< message padding and message digest calculation at the end of a DMA transfer */
#define MULTIPLE_DMA_NO_DIGEST              HAU_CTL_MDS                                /*!< multiple DMA transfers needed and CALEN bit is not automatically set at the end of a DMA transfer */

#define CFG_VBL(regval)                     (BITS(0,4) & (((uint32_t)(regval))))       /*!< write value to HAU_CFG_VBL bit field */

/* hau_inten register value */
#define HAU_INT_DATA_INPUT                  HAU_INTEN_DIIE                             /*!< a new block can be entered into the IN buffer */
#define HAU_INT_CALCULATION_COMPLETE        HAU_INTEN_CCIE                             /*!< calculation complete */

#define HAU_FLAG_DATA_INPUT                 HAU_STAT_DIF                               /*!< there is enough space (16 bytes) in the input FIFO */
#define HAU_FLAG_CALCULATION_COMPLETE       HAU_STAT_CCF                               /*!< digest calculation is completed */
#define HAU_FLAG_DMA                        HAU_STAT_DMAS                              /*!< DMA is enabled (DMAE =1) or a transfer is processing */
#define HAU_FLAG_BUSY                       HAU_STAT_BUSY                              /*!< data block is in process */
#define HAU_FLAG_INFIFO_NO_EMPTY            HAU_CTL_DINE                               /*!< the input FIFO is not empty */

#define HAU_INT_FLAG_DATA_INPUT             HAU_STAT_DIF                               /*!< there is enough space (16 bytes) in the input FIFO */
#define HAU_INT_FLAG_CALCULATION_COMPLETE   HAU_STAT_CCF                               /*!< digest calculation is completed */

#define HAU_BUSY_TIMEOUT                    ((uint32_t)0x00010000U)                    /*!< busy timeout */
#define SHA_CCF_TIMEOUT                     ((uint32_t)0x000FFFFFU)                    /*!< digest calculation complete timeout */
#define HAU_SUSPEND_WORD_LIMIT              (20U)                                      /*!< HAU suspend word limit */

#define HAU_GET_DATA_TYPE                   ((HAU_CTL & HAU_CTL_DATAM) >> 4U)          /*!< HAU get date type */
#define HAU_GET_NUMBER_DATE                 ((HAU_CTL & HAU_CTL_NWIF)  >> 8U)          /*!< HAU get transfer number of data */
#define HAU_GET_MULTIPLE_DMA_TRANSFER       ((HAU_CTL & HAU_CTL_MDS)   >> 12U)         /*!< HAU get multiple DMA transfer state */

#define HAU_MULTIPLE_TRANSFER_NONE          ((uint32_t)0x00000000U)                    /*!< multiple transfer none */
#define HAU_MULTIPLE_TRANSFER_STEP_1        ((uint32_t)0x00000001U)                    /*!< multiple transfer step 1 */
#define HAU_MULTIPLE_TRANSFER_STEP_2        ((uint32_t)0x00000002U)                    /*!< multiple transfer step 2 */
#define HAU_MULTIPLE_TRANSFER_STEP_3        ((uint32_t)0x00000003U)                    /*!< multiple transfer step 3 */

#define HAU_CONTEXT_NONE                    ((uint32_t)0x00000000U)                     /*!< context switch none */
#define HAU_CONTEXT_STEP_1                  ((uint32_t)0x00000001U)                     /*!< context switch step 1 */
#define HAU_CONTEXT_STEP_2                  ((uint32_t)0x00000002U)                     /*!< context switch step 2 */
#define HAU_CONTEXT_STEP_3                  ((uint32_t)0x00000003U)                     /*!< context switch step 3 */
#define HAU_CONTEXT_STEP_END                ((uint32_t)0x00000004U)                     /*!< context switch step end */

#define HAU_MAX_CONSOLE_DATA                64U                                         /*!< max console data */
/* HAL structure type enum */
typedef enum {
    HAL_HAU_INIT_STRUCT = 0U,         /*!< HAU initialization structure */
    HAL_HAU_DEV_STRUCT,               /*!< HAU device structure */
    HAL_HAU_IRQ_STRUCT,               /*!< HAU interrupt structure */
    HAL_HAU_USER_CALLBACK_STRUCT,     /*!< HAU user callback structure */
    HAL_HAU_CONTEXT_PARAMETER_STRUCT, /*!< HAU context parameter structure */
    HAL_HAU_CONTEXT_STRUCT,           /*!< HAU context structure */
    HAL_HAU_DIGEST_PARAMETER_STRUCT   /*!< HAU digest parameter structure */
} hal_hau_struct_type_enum;

/* HAL suspend state */
typedef enum {
    HAL_HAU_SUSPEND_STATE_NONE,     /*!< no suspend */
    HAL_HAU_SUSPEND_STATE_PROCESS   /*!< process suspend */
} hal_hau_suspend_state_enum;

/* HAU state enum */
typedef enum {
    HAL_HAU_STATE_RESET = 0U, /*!< reset */
    HAL_HAU_STATE_READY,      /*!< ready */
    HAL_HAU_STATE_BUSY,       /*!< busy */
    HAL_HAU_STATE_SUSPENDED   /*!< suspend state */
} hal_hau_state_enum;

/* HAU state error enum */
typedef enum {
    HAL_HAU_ERROR_STATE_NONE = 0U, /*!< none error */
    HAL_HAU_ERROR_STATE_BUSY,      /*!< busy error */
    HAL_HAU_ERROR_STATE_SUSPEND,   /*!< suspend error */
    HAL_HAU_ERROR_STATE_TIMEOUT,   /*!< timeout error */
    HAL_HAU_ERROR_STATE_HARDWARE,  /*!< hardware error */
    HAL_HAU_ERROR_STATE_SOFTWARE   /*!< software error */
} hal_hau_state_error_enum;

/* HAU step enum */
typedef enum {
    HAL_HAU_STEP_READY     = 0U,   /*!< HAU is ready to start */
    HAL_HAU_STEP_PROCESS   = 1U,   /*!< HAU is in HASH processing */
    HAL_HAU_STEP_HMAC_1    = 2U,   /*!< HAU is in HMAC step 1 processing
                                     (step 1 consists in entering the inner hash function key) */
    HAL_HAU_STEP_HMAC_2    = 3U,   /*!< HAU is in HMAC step 2 processing
                                      (step 2 consists in entering the message text) */
    HAL_HAU_STEP_HMAC_3    = 4U,   /*!< HAU is in HMAC step 3 processing
                                      (step 3 consists in entering the outer hash function key) */
    HAL_HAU_STEP_ACCMULATE = 5U    /*!< HAU is in accumulation processing */
} hal_hau_step_enum;

typedef enum {
    HAL_HAU_SUSPEND_NONE = 0U, /*!< hau suspension not requested */
    HAL_HAU_SUSPEND      = 1U  /*!< hau suspension is requested */
} hal_hau_suspend_type_enum;

/* structure for message digest result of the hau */
typedef struct {
    uint32_t out[8]; /*!< message digest result 0-7 */
} hal_hau_digest_parameter_struct;

/* structure for context switch */
typedef struct {
    uint32_t hau_ctl_bak;      /*!< backup of HAU_CTL register */
    uint32_t hau_cfg_bak;      /*!< backup of HAU_CFG register */
    uint32_t hau_inten_bak;    /*!< backup of HAU_INTEN register */
    uint32_t hau_ctxs_bak[54]; /*!< backup of HAU_CTXSx registers */
} hal_hau_context_parameter_struct;

/* HAU device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb input_fifo_handle;         /*!< hau fifo input callback */
    __IO hal_irq_handle_cb calculate_complete_handle; /*!< calculate complete callback */
} hal_hau_irq_struct;

/* @PARA: p_init */
/* @STRUCT: HAU Init structure definition */
typedef struct {
    uint32_t mode;      /*!< HAU mode selection */
    uint32_t algo;      /*!< hash algorithm selection */
    uint8_t *key;       /*!< key value */
    uint32_t keylength; /*!< key length */
    uint32_t datatype;  /*!< data type mode, data change */
    uint32_t keytype;   /*!< key length mode */
} hal_hau_init_struct;

typedef struct {
    uint32_t algo_step1;                              /*!< hash algorithm selection step 1 */
    __IO uint8_t const *key_1;                        /*!< key value */
    __IO uint32_t key_length_1;                       /*!< key length */
    __IO uint8_t const *input_buffer_1;               /*!< input data buffer */
    __IO uint32_t input_length_1;                     /*!< input data length */
    hal_hau_digest_parameter_struct *digeset_read1;   /*!< digest result read */
    uint32_t algo_step2;                               /*!< hash algorithm selection step 2 */
    __IO uint8_t const *key_2;                        /*!< key value */
    __IO uint32_t key_length_2;                       /*!< key length */
    __IO uint8_t const *input_buffer_2;               /*!< input data buffer */
    __IO uint32_t input_length_2;                     /*!< input data length */
    __IO uint32_t transfer_step_size;                 /* switch size */
    hal_hau_context_parameter_struct *context;        /*!< context switch parameter */
    hal_hau_digest_parameter_struct *digeset_read2;   /*!< digest result read */
} hal_hau_context_struct;

/* HAU device structure definition (For compatibility with standard library interfaces) */
typedef struct {
    hal_hau_irq_struct hau_irq;                     /*!< interrupt callback */
    hal_dma_dev_struct *p_dma_hau;                  /*!< DMA transmit pointer */
    hal_mutex_enum mutex;                           /*!< lock object */
    __IO hal_hau_state_enum state;                  /*!< hau device state */
    __IO hal_hau_state_error_enum error_state;      /*!< hau error state */
    __IO hal_hau_step_enum step;                    /*!< HAU step in processing */
    __IO hal_hau_suspend_state_enum suspend_state;  /*!< suspend state */
    uint32_t algo;                                  /*!< hash algorithm selection */
    __IO FlagStatus digest_calculation_disable;     /*!< HMAC messages for large files require multiple DMA transfers
                                                         to skip the digest calculation */
    hal_hau_context_struct *context;
    uint32_t multiple_transmit_flag;                /*!< multiple transmit buffer */
    uint32_t context_flag;                          /*!< context switch flag */
    uint32_t already_push_words;                    /*!< numbers of words already pushed in FIFO before inputting new block */
    __IO uint8_t const *key;                        /*!< key value */
    __IO uint32_t key_length;                       /*!< key length */
    __IO uint8_t const *input_buffer;               /*!< input data buffer */
    __IO uint32_t input_length;                     /*!< input data length */
    uint8_t *output_buffer;                         /*!< the result digest */
    void *input_fifo_callback;                      /*!< input fifo callback */
    void *calculate_complete_callback;              /*!< calculate complete callback */
    void *dma_error_callback;                       /*!< dma error callback */
} hal_hau_dev_struct;

typedef void (*hal_hau_user_cb)(hal_hau_dev_struct *hau_dev);

typedef struct {
    __IO hal_hau_user_cb input_fifo_complete_func; /*!< input fifo complete interrupt handle */
    __IO hal_hau_user_cb calculate_complete_func;  /*!< calculate complete interrupt handle */
    __IO hal_hau_user_cb dma_error_func;           /*!< DMA error callback */
} hal_hau_irq_user_callback_struct;

/* hal function declarations */
/* initialization functions */
/* @FUNCTION: initialize HAU */
int32_t hal_hau_init(hal_hau_dev_struct *hau_dev, hal_hau_init_struct *p_init);
/* @FUNCTION: initialize the HAU structure with default values */
int32_t hal_hau_struct_init(hal_hau_struct_type_enum struct_type, void *p_struct);
/*  @FUNCTION: deinitialize HAU */
int32_t hal_hau_deinit(hal_hau_dev_struct *hau_dev);
/* @END */

/* read the message digest result */
int32_t hal_hau_digest_read(hal_hau_digest_parameter_struct *digestpara);

/* HAU interrupt handler content function, which is merely used in  HAU_TRNG_IRQHandler */
int32_t hal_hau_irq(hal_hau_dev_struct *hau_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_hau_irq_handle_set(hal_hau_dev_struct *hau_dev, hal_hau_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_hau_irq_handle_all_reset(hal_hau_dev_struct *hau_dev);

/* initialize the struct context */
int32_t hal_hau_context_struct_para_init(hal_hau_context_parameter_struct *context);
/* save the HAU peripheral context */
int32_t hal_hau_context_save(hal_hau_context_parameter_struct *context_save);
/* restore the HAU peripheral context */
int32_t hal_hau_context_restore(hal_hau_context_parameter_struct *context_restore);
/* get calculate digest result */
int32_t hal_hau_hash_finish(hal_hau_dev_struct *hau_dev, uint8_t output[]);

/* context switch in DMA mode */
int32_t hal_hau_context_switch_dma(hal_hau_dev_struct *hau_dev, hal_hau_context_struct *context, \
                                   uint32_t transfer_step_size, hal_hau_irq_user_callback_struct *p_user_func);

/* HAU suspend state set in poll or interrupt mode */
int32_t hal_hau_poll_interrupt_suspend_config(hal_hau_dev_struct *hau_dev);
/* HAU suspend state set in DMA mode */
int32_t hal_hau_dma_suspend_config(hal_hau_dev_struct *hau_dev);

/* calculate digest using SHA1 in HASH mode */
int32_t hal_hau_hash_sha_1(hal_hau_dev_struct *hau_dev, \
                           const uint8_t *const input, uint32_t in_length, \
                           uint8_t *output, uint32_t timeout);
/* accumulate data using SHA1 in HASH mode */
int32_t hal_hau_hash_sha_1_accmulate(hal_hau_dev_struct *hau_dev, const uint8_t *const input, uint32_t in_length);
/* accumulate data end using SHA1 in HASH mode */
int32_t hal_hau_hash_sha_1_accmulate_end(hal_hau_dev_struct *hau_dev, \
                                         const uint8_t *const input, uint32_t in_length, \
                                         uint8_t *output, uint32_t timeout);
/* calculate digest using SHA1 in HASH mode for DMA */
int32_t hal_hau_hash_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                               const uint8_t *const input, uint32_t in_length, \
                               hal_hau_irq_user_callback_struct *p_user_func);
/* calculate digest using SHA1 in HASH mode for interrupt */
int32_t hal_hau_hash_sha_1_interrupt(hal_hau_dev_struct *hau_dev, \
                                     const uint8_t *const input, uint32_t in_length, uint8_t *output, \
                                     hal_hau_irq_user_callback_struct *p_user_func);
/* accumulate data using SHA1 in HASH mode for interrupt */
int32_t hal_hau_hash_sha_1_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                               const uint8_t *const input, uint32_t in_length, \
                                               hal_hau_irq_user_callback_struct *p_user_func);
/* accumulate data end using SHA1 in HASH mode for interrupt */
int32_t hal_hau_hash_sha_1_accmulate_end_interrupt(hal_hau_dev_struct *hau_dev, \
                                                   hal_hau_irq_user_callback_struct *p_user_func);

/* calculate digest using SHA224 in HASH mode */
int32_t hal_hau_hash_sha_224(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const input, uint32_t in_length, \
                             uint8_t *output, uint32_t timeout);
/* accumulate data using SHA224 in HASH mode */
int32_t hal_hau_hash_sha_224_accmulate(hal_hau_dev_struct *hau_dev, const uint8_t *const input, uint32_t in_length);
/* accumulate data end using SHA224 in HASH mode */
int32_t hal_hau_hash_sha_224_accmulate_end(hal_hau_dev_struct *hau_dev, \
                                           const uint8_t *const input, uint32_t in_length, \
                                           uint8_t *output, uint32_t timeout);
/* calculate digest using SHA224 in HASH mode for DMA */
int32_t hal_hau_hash_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                 const uint8_t *const input, uint32_t in_length, \
                                 hal_hau_irq_user_callback_struct *p_user_func);
/* calculate digest using SHA224 in HASH mode for interrupt */
int32_t hal_hau_hash_sha_224_interrupt(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const input, uint32_t in_length, \
                                       uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func);
/* accumulate data using SHA224 in HASH mode for interrupt */
int32_t hal_hau_hash_sha_224_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                                 const uint8_t *const input, uint32_t in_length, \
                                                 hal_hau_irq_user_callback_struct *p_user_func);
/* accumulate data end using SHA224 in HASH mode for interrupt */
int32_t hal_hau_hash_sha_224_accmulate_end_interrupt(hal_hau_dev_struct *hau_dev, \
                                                     hal_hau_irq_user_callback_struct *p_user_func);

/* calculate digest using SHA256 in HASH mode */
int32_t hal_hau_hash_sha_256(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const input, uint32_t in_length, \
                             uint8_t *output, uint32_t timeout);
/* accumulate data using SHA256 in HASH mode */
int32_t hal_hau_hash_sha_256_accmulate(hal_hau_dev_struct *hau_dev, const uint8_t *const input, uint32_t in_length);
/* accumulate data end using SHA256 in HASH mode */
int32_t hal_hau_hash_sha_256_accmulate_end(hal_hau_dev_struct *hau_dev, \
                                           const uint8_t *const input, uint32_t in_length, \
                                           uint8_t *output, uint32_t timeout);
/* calculate digest using SHA256 in HASH mode DMA */
int32_t hal_hau_hash_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                 const uint8_t *const input, uint32_t in_length, \
                                 hal_hau_irq_user_callback_struct *p_user_func);
/* calculate digest using SHA256 in HASH mode interrupt */
int32_t hal_hau_hash_sha_256_interrupt(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const input, uint32_t in_length, uint8_t *output, \
                                       hal_hau_irq_user_callback_struct *p_user_func);
/* accumulate data using SHA256 in HASH mode for interrupt */
int32_t hal_hau_hash_sha_256_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                                 const uint8_t *const input, uint32_t in_length, \
                                                 hal_hau_irq_user_callback_struct *p_user_func);
/* accumulate data end using SHA256 in HASH mode for interrupt */
int32_t hal_hau_hash_sha_256_accmulate_end_interrupt(hal_hau_dev_struct *hau_dev, \
                                                     hal_hau_irq_user_callback_struct *p_user_func);

/* calculate digest using MD5 in HASH mode */
int32_t hal_hau_hash_md5(hal_hau_dev_struct *hau_dev, \
                         const uint8_t *const input, uint32_t in_length, \
                         uint8_t *output, uint32_t timeout);
/* accumulate data using MD5 in HASH mode */
int32_t hal_hau_hash_md5_accmulate(hal_hau_dev_struct *hau_dev, \
                                   const uint8_t *const input, uint32_t in_length);
/* accumulate data end using MD5 in HASH mode */
int32_t hal_hau_hash_md5_accmulate_end(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const input, uint32_t in_length, \
                                       uint8_t *output, uint32_t timeout);
/* calculate digest using MD5 in HASH mode for DMA */
int32_t hal_hau_hash_md5_dma(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const input, uint32_t in_length, \
                             hal_hau_irq_user_callback_struct *p_user_func);
/* calculate digest using MD5 in HASH mode for interrupt */
int32_t hal_hau_hash_md5_interrupt(hal_hau_dev_struct *hau_dev, \
                                   const uint8_t *const input, uint32_t in_length, \
                                   uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func);
/* accumulate data using MD5 in HASH mode for interrupt */
int32_t hal_hau_hash_md5_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                             const uint8_t *const input, uint32_t in_length, \
                                             hal_hau_irq_user_callback_struct *p_user_func);
/* accumulate data end using MD5 in HASH mode for interrupt */
int32_t hal_hau_hash_md5_accmulate_end_interrupt(hal_hau_dev_struct *hau_dev, \
                                                 hal_hau_irq_user_callback_struct *p_user_func);

/* calculate digest using SHA1 in HMAC mode */
int32_t hal_hau_hmac_sha_1(hal_hau_dev_struct *hau_dev, \
                           const uint8_t *const key, uint32_t keysize, \
                           const uint8_t *const input, uint32_t in_length, \
                           uint8_t *output, uint32_t timeout);
/* computing the digest of DMA using SHA-1 in HMAC mode, with data length less than 64 bytes. */
int32_t hal_hau_hmac_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                               const uint8_t *const key, uint32_t keysize, \
                               const uint8_t *const input, uint32_t in_length, \
                               hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 1 calculation complement and step 2 start using SHA1 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step1_complete_step2_multi_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                                                          const uint8_t *const key, uint32_t keysize, \
                                                          hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 2 start using SHA1 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step2_multi_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                                           const uint8_t *const input, uint32_t in_length, \
                                           hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 2 complete and step 3 start using SHA1 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step2_complete_step3_multi_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                                                          const uint8_t *const key, uint32_t keysize, \
                                                          hal_hau_irq_user_callback_struct *p_user_func);
/* calculate digest using SHA1 in HMAC mode for interrupt */
int32_t hal_hau_hmac_sha_1_interrupt(hal_hau_dev_struct *hau_dev, \
                                     const uint8_t *const key, uint32_t keysize, \
                                     const uint8_t *const input, uint32_t in_length, \
                                     uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func);

/* calculate digest using SHA224 in HMAC mode */
int32_t hal_hau_hmac_sha_224(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const key, uint32_t keysize, \
                             const uint8_t *const input, uint32_t in_length, \
                             uint8_t *output, uint32_t timeout);
/* calculate digest using SHA224 in HMAC mode for DMA */
int32_t hal_hau_hmac_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                 const uint8_t *const key, uint32_t keysize, \
                                 const uint8_t *const input, uint32_t in_length, \
                                 hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 1 calculation complement and step 2 start using SHA224 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step1_complete_step2_multi_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                                            const uint8_t *const key, uint32_t keysize, \
                                                            hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 2 start using SHA224 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step2_multi_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                             const uint8_t *const input, uint32_t in_length, \
                                             hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 2 complete and step 3 start using SHA224 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step2_complete_step3_multi_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                                            const uint8_t *const key, uint32_t keysize, \
                                                            hal_hau_irq_user_callback_struct *p_user_func);
/* calculate digest using SHA224 in HMAC mode for interrupt */
int32_t hal_hau_hmac_sha_224_interrupt(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const key, uint32_t keysize, \
                                       const uint8_t *const input, uint32_t in_length, \
                                       uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func);

/* calculate digest using SHA256 in HMAC mode */
int32_t hal_hau_hmac_sha_256(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const key, uint32_t keysize, \
                             const uint8_t *const input, uint32_t in_length, \
                             uint8_t *output, uint32_t timeout);
/* calculate digest using SHA256 in HMAC mode for DMA */
int32_t hal_hau_hmac_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                 const uint8_t *const key, uint32_t keysize, \
                                 const uint8_t *const input, uint32_t in_length, \
                                 hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 1 calculation complement and step 2 start using SHA256 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step1_complete_step2_multi_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                                            const uint8_t *const key, uint32_t keysize, \
                                                            hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 2 start using SHA256 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step2_multi_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                             const uint8_t *const input, uint32_t in_length, \
                                             hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 2 complete and step 3 start using SHA256 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step2_complete_step3_multi_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                                            const uint8_t *const key, uint32_t keysize, \
                                                            hal_hau_irq_user_callback_struct *p_user_func);
/* calculate digest using SHA256 in HMAC mode for interrupt */
int32_t hal_hau_hmac_sha_256_interrupt(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const key, uint32_t keysize, \
                                       const uint8_t *const input, uint32_t in_length, \
                                       uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func);

/* calculate digest using MD5 in HMAC mode */
int32_t hal_hau_hmac_md5(hal_hau_dev_struct *hau_dev, \
                         const uint8_t *const key, uint32_t keysize, \
                         const uint8_t *const input, uint32_t in_length, \
                         uint8_t *output, uint32_t timeout);
/* calculate digest using MD5 in HMAC mode for DMA */
int32_t hal_hau_hmac_md5_dma(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const key, uint32_t keysize, \
                             const uint8_t *const input, uint32_t in_length, \
                             hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 1 calculation complement and step 2 start using MD5 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step1_complete_step2_multi_md5_dma(hal_hau_dev_struct *hau_dev, \
                                                        const uint8_t *const key, uint32_t keysize, \
                                                        hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 2 start using MD5 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step2_multi_md5_dma(hal_hau_dev_struct *hau_dev, \
                                         const uint8_t *const input, uint32_t in_length, \
                                         hal_hau_irq_user_callback_struct *p_user_func);
/* HMAC step 2 complete and step 3 start using MD5 HAU mode in multi-buffer DMA mode */
int32_t hal_hau_hmac_step2_complete_step3_multi_md5_dma(hal_hau_dev_struct *hau_dev, \
                                                        const uint8_t *const key, uint32_t keysize, \
                                                        hal_hau_irq_user_callback_struct *p_user_func);
/* calculate digest using MD5 in HMAC mode for interrupt */
int32_t hal_hau_hmac_md5_interrupt(hal_hau_dev_struct *hau_dev, \
                                     const uint8_t *const key, uint32_t keysize, \
                                     const uint8_t *const input, uint32_t in_length, \
                                     uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func);

/* get HAU state*/
hal_hau_state_enum hal_hau_state_get(hal_hau_dev_struct *hau_dev);
/* get HAU step */
hal_hau_step_enum hal_hau_step_get(hal_hau_dev_struct *hau_dev);
/* get HAU error state */
hal_hau_state_error_enum hal_hau_error_state_get(hal_hau_dev_struct *hau_dev);

/* reset the HAU processor core */
void hals_hau_reset(void);
/* enable digest calculation */
void hals_hau_digest_calculation_enable(void);
/* configure single or multiple DMA is used, and digest calculation at the end of a DMA transfer or not */
int32_t hals_hau_multiple_single_dma_config(uint32_t multi_single);
/* enable the HAU DMA interface */
void hals_hau_dma_enable(void);
/* disable the HAU DMA interface */
void hals_hau_dma_disable(void);

/* HAU SHA/MD5 digest read */
int32_t hals_hau_sha_md5_digest_read(uint32_t algo, uint8_t output[]);
/* configure the number of valid bits in last word of the message */
int32_t hals_hau_last_word_validbits_num_config(uint32_t valid_num);
/* write data to the in fifo */
void hals_hau_data_write(uint32_t data);
/* return the number of words already written into the in fifo */
uint32_t hals_hau_infifo_words_num_get(void);

/* get the HAU flag status */
FlagStatus hals_hau_flag_get(uint32_t flag);
/* clear the HAU flag status */
void hals_hau_flag_clear(uint32_t flag);
/* enable the HAU interrupts */
void hals_hau_interrupt_enable(uint32_t interrupt);
/* disable the HAU interrupts */
void hals_hau_interrupt_disable(uint32_t interrupt);
/* get the HAU interrupt flag status */
FlagStatus hals_hau_interrupt_flag_get(uint32_t int_flag);
/* clear the HAU interrupt flag status */
void hals_hau_interrupt_flag_clear(uint32_t int_flag);

#endif /* GD32H7XX_HAL_HAU_H */
