/*!
    \file    gd32h7xx_hal_fac.h
    \brief   definitions for the FAC

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_FAC_H
#define GD32H7XX_HAL_FAC_H

#include "gd32h7xx_hal.h"

/* FAC definitions */
#define FAC                              FAC_BASE                               /*!< FAC base address */

#define REG16_INT(addr)                  (*(volatile int16_t *)(uint32_t)(addr))/*!< FAC int16 */
#define REG32_FLOAT(addr)                (*(volatile float *)(uint32_t)(addr))  /*!< FAC float */

/* registers definitions */
#define FAC_X0BCFG                       REG32((FAC_BASE) + 0x00000000U)        /*!< FAC X0 buffer configure register */
#define FAC_X1BCFG                       REG32((FAC_BASE) + 0x00000004U)        /*!< FAC X1 buffer configure register */
#define FAC_YBCFG                        REG32((FAC_BASE) + 0x00000008U)        /*!< FAC Y buffer configure register */
#define FAC_PARACFG                      REG32((FAC_BASE) + 0x0000000CU)        /*!< FAC Parameter configure register */
#define FAC_CTL                          REG32((FAC_BASE) + 0x00000010U)        /*!< FAC Control register */
#define FAC_STAT                         REG32((FAC_BASE) + 0x00000014U)        /*!< FAC Status register */
#define FAC_WDATA                        REG32((FAC_BASE) + 0x00000018U)        /*!< FAC write data register */
#define FAC_RDATA                        REG32((FAC_BASE) + 0x0000001CU)        /*!< FAC read data register */

#define FAC_WDATA_INT                    REG16_INT((FAC_BASE) + 0x00000018U)    /*!< FAC write data register */
#define FAC_WDATA_FLOAT                  REG32_FLOAT((FAC_BASE) + 0x00000018U)  /*!< FAC write data register */

#define FAC_RDATA_INT                    REG16_INT((FAC_BASE) + 0x0000001CU)    /*!< FAC read data register */
#define FAC_RDATA_FLOAT                  REG32_FLOAT((FAC_BASE) + 0x0000001CU)  /*!< FAC read data register */

/* bits definitions */
/* FAC_X0BCFG */
#define FAC_X0BCFG_X0B_ADDR              BITS(0,7)                              /*!< X0 buffer base address */
#define FAC_X0BCFG_X0B_SIZE              BITS(8,15)                             /*!< X0 buffer allocated size */
#define FAC_X0BCFG_X0_WBFF               BITS(24,25)                            /*!< watermark for buffer full flag */

/* FAC_X1BCFG */
#define FAC_X1BCFG_X1B_ADDR              BITS(0,7)                              /*!< X1 buffer base address */
#define FAC_X1BCFG_X1B_SIZE              BITS(8,15)                             /*!< X1 buffer allocated size */

/* FAC_YBCFG */
#define FAC_YBCFG_YB_ADDR                BITS(0,7)                              /*!< Y buffer base address */
#define FAC_YBCFG_YB_SIZE                BITS(8,15)                             /*!< Y buffer allocated size */
#define FAC_YBCFG_Y_WBEF                 BITS(24,25)                            /*!< watermark for buffer empty flag */

/* FAC_PARACFG */
#define FAC_PARACFG_IPP                  BITS(0,7)                              /*!< input parameter IPP */
#define FAC_PARACFG_IPQ                  BITS(8,15)                             /*!< input parameter IPQ */
#define FAC_PARACFG_IPR                  BITS(16,23)                            /*!< input parameter IPR */
#define FAC_PARACFG_FUN                  BITS(24,30)                            /*!< function */
#define FAC_PARACFG_EXE                  BIT(31)                                /*!< execution */

/* FAC_CTL */
#define FAC_CTL_RIE                      BIT(0)                                 /*!< read interrupt enable */
#define FAC_CTL_WIE                      BIT(1)                                 /*!< write interrupt enable */
#define FAC_CTL_OFEIE                    BIT(2)                                 /*!< overflow error interrupt enable */
#define FAC_CTL_UFEIE                    BIT(3)                                 /*!< underflow error interrupt enable */
#define FAC_CTL_STEIE                    BIT(4)                                 /*!< saturation error interrupt enable */
#define FAC_CTL_GSTEIE                   BIT(5)                                 /*!< gain saturation error interrupt enable */
#define FAC_CTL_DREN                     BIT(8)                                 /*!< DMA read channel enable */
#define FAC_CTL_DWEN                     BIT(9)                                 /*!< DMA write channel enable */
#define FAC_CTL_FLTEN                    BIT(14)                                /*!< floating point format enable */
#define FAC_CTL_CPEN                     BIT(15)                                /*!< clipping enable */
#define FAC_CTL_RST                      BIT(16)                                /*!< reset FAC unit */

/* FAC_STAT */
#define FAC_STAT_YBEF                    BIT(0)                                 /*!< Y buffer empty flag */
#define FAC_STAT_X0BFF                   BIT(1)                                 /*!< X0 buffer full flag */
#define FAC_STAT_OFEF                    BIT(8)                                 /*!< overflow error flag */
#define FAC_STAT_UFEF                    BIT(9)                                 /*!< underflow error flag */
#define FAC_STAT_STEF                    BIT(10)                                /*!< saturation error flag */
#define FAC_STAT_GSTEF                   BIT(11)                                /*!< gain saturation error flag */

/* FAC_WDATA */
#define FAC_WDATA_WDATA                  BITS(0,15)                             /*!< write data */

/* FAC_RDATA */
#define FAC_RDATA_RDATA                  BITS(0,15)                             /*!< read data */

/* constants definitions */
/* FAC function execution definitions */
#define PARACFG_EXE(regval)              (FAC_PARACFG_EXE & ((uint32_t)(regval) << 31U))
#define FAC_FUNC_START                   PARACFG_EXE(0)                         /*!< start execution function */
#define FAC_FUNC_STOP                    PARACFG_EXE(1)                         /*!< stop execution function */

/* FAC DMA mode definitions */
#define FAC_DMA_READ                     FAC_CTL_DREN                           /*!< enable dma read */
#define FAC_DMA_WRITE                    FAC_CTL_DWEN                           /*!< enable dma write */

/* FAC interrupt flag definitions */
#define FAC_INT_FLAG_YBEF                ((uint8_t)0x00U)                       /*!< Y buffer empty interrupt flag */
#define FAC_INT_FLAG_X0BFF               ((uint8_t)0x01U)                       /*!< X0 buffer full interrupt flag */
#define FAC_INT_FLAG_OFEF                ((uint8_t)0x02U)                       /*!< Overflow error interrupt flag */
#define FAC_INT_FLAG_UFEF                ((uint8_t)0x03U)                       /*!< underflow error interrupt flag */
#define FAC_INT_FLAG_STEF                ((uint8_t)0x04U)                       /*!< saturation error interrupt flag */
#define FAC_INT_FLAG_GSTEF               ((uint8_t)0x05U)                       /*!< gain saturation error interrupt flag */

/* FAC flag definitions */
#define FAC_FLAG_YBEF                    FAC_STAT_YBEF                          /*!< Y buffer empty flag */
#define FAC_FLAG_X0BFF                   FAC_STAT_X0BFF                         /*!< X0 buffer full flag */
#define FAC_FLAG_OFEF                    FAC_STAT_OFEF                          /*!< overflow error flag */
#define FAC_FLAG_UFEF                    FAC_STAT_UFEF                          /*!< underflow error flag */
#define FAC_FLAG_STEF                    FAC_STAT_STEF                          /*!< saturation error flag */
#define FAC_FLAG_GSTEF                   FAC_STAT_GSTEF                         /*!< gain saturation error flag */

/* @STRUCT_MEMBER: func */
/* @DEFINE: FAC function select definitions */
#define CFG_FUN(regval)                  (BITS(24,30) & ((uint32_t)(regval) << 24U))
#define FAC_LOAD_X0                      CFG_FUN(1)                             /*!< load_X0_buffer */
#define FAC_LOAD_X1                      CFG_FUN(2)                             /*!< load_X1_buffer */
#define FAC_LOAD_Y                       CFG_FUN(3)                             /*!< load_Y_buffer */
#define FAC_CONVO_FIR                    CFG_FUN(8)                             /*!< convolution (FIR filter) */
#define FAC_IIR_DIRECT_FORM_1            CFG_FUN(9)                             /*!< IIR filter (direct form 1) */

/* @STRUCT_MEMBER: input_threshold */
/* @DEFINE: FAC x0 watermark setting definitions */
#define X0BCFG_X0_WBFF(regval)           (BITS(24,25) & ((uint32_t)(regval) << 24U))
#define FAC_X0_THRESHOLD_1               X0BCFG_X0_WBFF(0)                      /*!< full/empty flag when buffer less than 1 */
#define FAC_X0_THRESHOLD_2               X0BCFG_X0_WBFF(1)                      /*!< full/empty flag when buffer less than 2 */
#define FAC_X0_THRESHOLD_4               X0BCFG_X0_WBFF(2)                      /*!< full/empty flag when buffer less than 4 */
#define FAC_X0_THRESHOLD_8               X0BCFG_X0_WBFF(3)                      /*!< full/empty flag when buffer less than 8 */

/* @STRUCT_MEMBER: output_threshold */
/* @DEFINE: FAC y watermark setting definitions */
#define YBCFG_Y_WBEF(regval)             (BITS(24,25) & ((uint32_t)(regval) << 24U))
#define FAC_Y_THRESHOLD_1                YBCFG_Y_WBEF(0)                        /*!< full/empty flag when buffer less than 1 */
#define FAC_Y_THRESHOLD_2                YBCFG_Y_WBEF(1)                        /*!< full/empty flag when buffer less than 2 */
#define FAC_Y_THRESHOLD_4                YBCFG_Y_WBEF(2)                        /*!< full/empty flag when buffer less than 4 */
#define FAC_Y_THRESHOLD_8                YBCFG_Y_WBEF(3)                        /*!< full/empty flag when buffer less than 8 */

/* @STRUCT_MEMBER: clip */
/* @DEFINE: FAC clip function definitions */
#define FAC_CP_DISABLE                   ((uint32_t)0x00000000U)                /*!< clipping disabled */
#define FAC_CP_ENABLE                    FAC_CTL_CPEN                           /*!< clipping enabled */

/* @STRUCT_MEMBER: fpoint */
/* @DEFINE: FAC float data support definitions */
#define FAC_FLOAT_DISABLE                ((uint32_t)0x00000000U)                /*!< disable FAC float data support */
#define FAC_FLOAT_ENABLE                 FAC_CTL_FLTEN                          /*!< enable FAC float data support */

#define FAC_BUFFER_ACCESS_NONE           0x00U                                  /*!< buffer handled by an external */
#define FAC_BUFFER_ACCESS_DMA            0x01U                                  /*!< buffer accessed through DMA */
#define FAC_BUFFER_ACCESS_POLLING        0x02U                                  /*!< buffer accessed through polling */
#define FAC_BUFFER_ACCESS_IT             0x03U                                  /*!< buffer accessed through interruptions */

#define HAL_FAC_ERROR_NONE               0x00000000U                            /*!< no error */
#define HAL_FAC_ERROR_SAT                0x00000001U                            /*!< saturation error */
#define HAL_FAC_ERROR_UNFL               0x00000002U                            /*!< underflow error */
#define HAL_FAC_ERROR_OVFL               0x00000004U                            /*!< overflow error */
#define HAL_FAC_ERROR_DMA                0x00000008U                            /*!< DMA error */
#define HAL_FAC_ERROR_RESET              0x00000010U                            /*!< reset error */
#define HAL_FAC_ERROR_PARAM              0x00000020U                            /*!< parameter error */
#define HAL_FAC_ERROR_TIMEOUT            0x00000080U                            /*!< timeout error */

/* FAC structure type enum */
typedef enum {
    HAL_FAC_INIT_STRUCT = 0U,                                                   /*!< FAC initialization structure */
    HAL_FAC_DEV_STRUCT,                                                         /*!< FAC device structure */
    HAL_FAC_IRQ_INIT_STRUCT,                                                    /*!< interrupt callback initialization structure */
    HAL_FAC_IRQ_USER_CALLBACK_STRUCT,                                           /*!< interrupt callback user structure */
    HAL_FAC_FIXED_DATA_PRELOAD_STRUCT,                                          /*!< FAC fixed data preload structure */
    HAL_FAC_FLOAT_DATA_PRELOAD_STRUCT,                                          /*!< FAC float data preload structure */
    HAL_FAC_CONFIG_STRUCT                                                       /*!< FAC config struct */
} hal_fac_struct_type_enum;

typedef enum {
    HAL_FAC_STATE_RESET   = 0x00U,                                              /*!< FAC not yet initialized or disabled */
    HAL_FAC_STATE_READY   = 0x20U,                                              /*!< FAC initialized and ready for use */
    HAL_FAC_STATE_BUSY    = 0x24U,                                              /*!< FAC internal process is ongoing */
    HAL_FAC_STATE_BUSY_RD = 0x25U,                                              /*!< FAC reading configuration is ongoing */
    HAL_FAC_STATE_BUSY_WR = 0x26U,                                              /*!< FAC writing configuration is ongoing */
    HAL_FAC_STATE_TIMEOUT = 0xA0U,                                              /*!< FAC in timeout state */
    HAL_FAC_STATE_ERROR   = 0xE0U                                               /*!< FAC in error state */
} hal_fac_state_enum;

/* FAC device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb error_handle;                                        /*!< FAC error callback */
    __IO hal_irq_handle_cb half_get_data_handle;                                /*!< FAC get half data callback */
    __IO hal_irq_handle_cb get_data_handle;                                     /*!< FAC get data callback */
    __IO hal_irq_handle_cb half_output_data_ready_handle;                       /*!< FAC half output data ready callback */
    __IO hal_irq_handle_cb output_data_ready_handle;                            /*!< FAC output data ready callback */
    __IO hal_irq_handle_cb filter_config_handle;                                /*!< FAC filter configuration callback */
    __IO hal_irq_handle_cb filter_preload_handle;                               /*!< FAC filter preload callback */
} hal_fac_irq_struct;

/* @PARA: p_init */
/* @STRUCT: FAC init structure definition */
typedef struct {
    uint8_t input_addr;                                                         /*!< base address of the input buffer (X0) */
    uint8_t input_size;                                                         /*!< size of input buffer */
    uint32_t input_threshold;                                                   /*!< threshold of input buffer full */
    uint8_t coeff_addr;                                                         /*!< base address of the coefficient buffer (X1) */
    uint8_t coeff_size;                                                         /*!< size of coefficient buffer */
    uint8_t output_addr;                                                        /*!< base address of the output buffer (Y) */
    uint8_t output_size;                                                        /*!< size of output buffer */
    uint32_t output_threshold;                                                  /*!< threshold of output buffer empty */
    uint32_t clip;                                                              /*!< enable or disable the clipping feature */
    uint32_t fpoint;                                                            /*!< enable or disable the float point data feature */
    uint32_t func;                                                              /*!< FAC functions select */
} hal_fac_init_struct;

/* @STRUCT_MEMBER: input_size */
/* @=NULL*/

/* @STRUCT_MEMBER: coeff_size */
/* @=NULL*/

/* @STRUCT_MEMBER: output_size */
/* @=NULL*/

/* @PARA: fac_dev */
/* @STRUCT: FAC device structure definition */
typedef struct {
    hal_fac_init_struct   init;                                                 /*!< FAC init parameters */
    hal_fac_irq_struct    fac_irq;                                              /*!< FAC interrupt callback */
    uint32_t              filter_param;                                         /*!< filter configuration (operation and parameters) */
    uint8_t               input_access;                                         /*!< access to the input buffer (internal memory area) */
    uint8_t               output_access;                                        /*!< access to the output buffer (internal memory area) */
    void                 *input_ctx;                                            /*!< pointer to fac input data buffer */
    uint16_t              input_size;                                           /*!< number of input elements to write (memory allocated to input) */
    uint16_t              input_current_size;                                   /*!< number of the input elements already written into FAC */
    void                 *output_ctx;                                           /*!< pointer to fac output data buffer */
    uint16_t              output_size;                                          /*!< number of output elements to read (memory allocated to output) */
    uint16_t              output_current_size;                                  /*!< number of the output elements already read from FAC */
    hal_dma_dev_struct    *p_dma_preload;                                       /*!< DMA transmit pointer */
    hal_dma_dev_struct    *p_dma_in;                                            /*!< DMA receive pointer */
    hal_dma_dev_struct    *p_dma_out;                                           /*!< DMA transmit pointer */
    hal_fac_state_enum    state;                                                /*!< FAC device state */
    hal_fac_state_enum    rd_state;                                             /*!< FAC device read state */
    hal_fac_state_enum    wr_state;                                             /*!< FAC device write state */
    uint32_t              error_state;                                          /*!< FAC device error state */
    hal_mutex_enum        mutex;                                                /*!< FAC lock */
    void                  *priv;                                                /*!< private pointer */
    void                  *error_callback;                                        /*!< FAC user error callback */
    void                  *half_get_data_callback;                                /*!< FAC user get half data callback */
    void                  *get_data_callback;                                     /*!< FAC user get data callback */
    void                  *half_output_data_ready_callback;                       /*!< FAC user half output data ready callback */
    void                  *output_data_ready_callback;                            /*!< FAC user output data ready callback */
    void                  *filter_config_callback;                                /*!< FAC user filter configuration callback */
    void                  *filter_preload_callback;                               /*!< FAC user filter preload callback */
} hal_fac_dev_struct;

/* structure for FAC configuration parameters */
typedef struct {
    uint32_t func;                                                              /*!< functions select */
    uint8_t ipp;                                                                /*!< value P (vector length, number of filter taps, etc.) */
    uint8_t ipq;                                                                /*!< value Q (vector length, etc.) */
    uint8_t ipr;                                                                /*!< value R (gain, etc.) */
    void    *coeffa_ctx;                                                        /*!< initialization of the coefficient vector A */
    uint8_t coeffa_size;                                                        /*!< size of the coefficient vector A */
    void    *coeffb_ctx;                                                        /*!< initialization of the coefficient vector B */
    uint8_t coeffb_size;                                                        /*!< size of the coefficient vector B */
    uint8_t input_access;                                                       /*!< access to the input buffer */
    uint8_t output_access;                                                      /*!< access to the output buffer */
} hal_fac_config_struct;

/* structure for FAC fixed data preload parameters */
typedef struct {
    uint8_t coeffa_size;                                                        /*!< size of the coefficient vector A */
    int16_t *coeffa_ctx;                                                        /*!< content of the coefficient vector A */
    uint8_t coeffb_size;                                                        /*!< size of the coefficient vector B */
    int16_t *coeffb_ctx;                                                        /*!< content of the coefficient vector B */
    uint8_t input_size;                                                         /*!< Size of the input data */
    int16_t *input_ctx;                                                         /*!< content of the input data */
    uint8_t output_size;                                                        /*!< size of the output data */
    int16_t *output_ctx;                                                        /*!< content of the output data */
} hal_fac_fixed_data_preload_struct;

/* structure for FAC float data preload parameters */
typedef struct {
    uint8_t coeffa_size;                                                        /*!< size of the coefficient vector A */
    float *coeffa_ctx;                                                          /*!< content of the coefficient vector A */
    uint8_t coeffb_size;                                                        /*!< size of the coefficient vector B */
    float *coeffb_ctx;                                                          /*!< content of the coefficient vector B */
    uint8_t input_size;                                                         /*!< size of the input data */
    float *input_ctx;                                                           /*!< content of the input data */
    uint8_t  output_size;                                                       /*!< size of the output data */
    float *output_ctx;                                                          /*!< content of the output data */
} hal_fac_float_data_preload_struct;

typedef void (*hal_fac_user_cb)(hal_fac_dev_struct *fac_dev);
/* FAC device interrupt user callback function pointer structure */
typedef struct {
    __IO hal_fac_user_cb error_callback;                                        /*!< FAC user error callback */
    __IO hal_fac_user_cb half_get_data_callback;                                /*!< FAC user get half data callback */
    __IO hal_fac_user_cb get_data_callback;                                     /*!< FAC user get data callback */
    __IO hal_fac_user_cb half_output_data_ready_callback;                       /*!< FAC user half output data ready callback */
    __IO hal_fac_user_cb output_data_ready_callback;                            /*!< FAC user output data ready callback */
    __IO hal_fac_user_cb filter_config_callback;                                /*!< FAC user filter configuration callback */
    __IO hal_fac_user_cb filter_preload_callback;                               /*!< FAC user filter preload callback */
} hal_fac_irq_user_callback_struct;

/* hal function declarations */
/* initialization functions */
/* @FUNCTION: initialize FAC */
int32_t hal_fac_init(hal_fac_dev_struct *fac_dev, hal_fac_init_struct *p_init);
/* @FUNCTION: initialize the FAC structure with default values */
int32_t hal_fac_struct_init(hal_fac_struct_type_enum hal_struct_type, void *p_struct);
/* @FUNCTION: deinitialize FAC */
int32_t hal_fac_deinit(hal_fac_dev_struct *fac_dev);
/* @END */
/* configuration the FAC filter peripheral */
int32_t hal_fac_filter_config(hal_fac_dev_struct *fac_dev, hal_fac_config_struct *p_config);
/* configuration the FAC filter peripheral with DMA */
int32_t hal_fac_filter_config_dma(hal_fac_dev_struct *fac_dev, hal_fac_config_struct *p_config, \
                                  hal_fac_irq_user_callback_struct *p_user_func);
/* preload the FAC filter peripheral */
int32_t hal_fac_filter_preload(hal_fac_dev_struct *fac_dev, void *input_ctx, uint8_t input_size, \
                               void *output_ctx, uint8_t output_size);
/* preload the FAC filter peripheral with DMA */
int32_t hal_fac_filter_preload_dma(hal_fac_dev_struct *fac_dev, void *input_ctx, uint8_t input_size, \
                                   void *output_ctx, uint8_t output_size, \
                                   hal_fac_irq_user_callback_struct *p_user_func);
/* provide a new input buffer that will be loaded into the FAC input memory area */
int32_t hal_fac_append_filter_data(hal_fac_dev_struct *fac_dev, void *input_ctx, uint16_t input_size, \
                                   hal_fac_irq_user_callback_struct *p_user_func);
/* provide a new output buffer to be filled with the data computed by FAC unit */
int32_t hal_fac_config_filter_output_buffer(hal_fac_dev_struct *fac_dev, void *output_ctx, uint16_t output_size);
/* handle the input and/or output data in polling mode */
int32_t hal_fac_poll_filter_data(hal_fac_dev_struct *fac_dev);
/* start FAC module function */
int32_t hal_fac_filter_start(hal_fac_dev_struct *fac_dev, void *output_ctx, uint16_t output_size);
/* stop FAC module function */
int32_t hal_fac_filter_stop(hal_fac_dev_struct *fac_dev);

/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_fac_irq_handle_set(hal_fac_dev_struct *fac_dev, hal_fac_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_fac_irq_handle_all_reset(hal_fac_dev_struct *fac_dev);
/* FAC interrupt handler content function, which is merely used in FAC_IRQHandler */
void hal_fac_irq(hal_fac_dev_struct *fac_dev);

/* return the FAC state */
hal_fac_state_enum hal_fac_state_get(hal_fac_dev_struct *fac_dev);
/* return the FAC error code */
uint32_t hal_fac_error_get(hal_fac_dev_struct *fac_dev);

/* enable the FAC interrupt */
void hals_fac_interrupt_enable(uint32_t interrupt);
/* disable the FAC interrupt */
void hals_fac_interrupt_disable(uint32_t interrupt);
/* get the FAC interrupt flag status */
FlagStatus hals_fac_interrupt_flag_get(uint8_t interrupt);
/* get the FAC flag status */
FlagStatus hals_fac_flag_get(uint32_t flag);

#endif /* GD32H7XX_HAL_FAC_H */
