/*!
    \file    gd32h7xx_hal_exmc_sram.h
    \brief   definitions for the EXMC SRAM

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_EXMC_SRAM_H
#define GD32H7XX_HAL_EXMC_SRAM_H

#include "gd32h7xx_hal.h"

/* EXMC bank0 register reset value */
#define SRAM_BANK0_SNCTL_RESET   ((uint32_t)0x000030DAU)                        /* SNCTL register reset value */
#define SRAM_BANK0_SNTCFG_RESET  ((uint32_t)0x0FFFFFFFU)                        /* SNTCFG register reset value */
#define SRAM_BANK0_SNWTCFG_RESET ((uint32_t)0x0FFFFFFFU)                        /* SNWTCFG register reset value */

/* EXMC SRAM structure type enum */
typedef enum {
    HAL_EXMC_SRAM_INIT_STRUCT = 0x00U,                                          /*!< EXMC sram initialization structure */
    HAL_EXMC_SRAM_DEV_STRUCT,                                                   /*!< EXMC sram device structure */
    HAL_EXMC_SRAM_USER_CALLBACK_STRUCT                                          /*!< EXMC sram user callback structure */
} hal_exmc_sram_struct_type_enum;

/* EXMC sram state type enum */
typedef enum {
    HAL_EXMC_SRAM_STATE_RESET         = (uint32_t)0x00000000U,                  /*!< EXMC sram is not initialized or disabled */
    HAL_EXMC_SRAM_STATE_READY         = (uint32_t)0x00000001U,                  /*!< EXMC sram is ready */
    HAL_EXMC_SRAM_STATE_BUSY          = (uint32_t)0x00000002U,                  /*!< EXMC sram is busy */
    HAL_EXMC_SRAM_STATE_ERROR         = (uint32_t)0x00000004U,                  /*!< EXMC sram timeout occurs */
    HAL_EXMC_SRAM_STATE_PROTECTED     = (uint32_t)0x00000005U                   /*!< EXMC sram device write protected*/
} hal_exmc_sram_state_enum;

/* @PARA: sram_init */
/* @STRUCT: EXMC NOR/SRAM initialize structure */
typedef struct {
    uint32_t norsram_region;                                                    /*!< select the region of EXMC NOR/PSRAM region */
    uint32_t bank_remap;                                                        /*!< select EXMC NOR/PSRAM remap region */
    uint32_t consecutive_clock;                                                 /*!< configure consecutive clock mode */
    uint32_t address_data_mux;                                                  /*!< specify whether the data bus and address bus are multiplexed */
    uint32_t memory_type;                                                       /*!< specify the type of external memory */
    uint32_t databus_width;                                                     /*!< specify the data bus width of external memory */
    uint32_t burst_mode;                                                        /*!< enable or disable the burst mode */
    uint32_t nwait_polarity;                                                    /*!< specify the polarity of NWAIT signal from memory */
    uint32_t nwait_config;                                                      /*!< NWAIT signal configuration */
    uint32_t memory_write;                                                      /*!< enable or disable the write operation */
    uint32_t nwait_signal;                                                      /*!< enable or disable the NWAIT signal while in synchronous bust mode */
    uint32_t extended_mode;                                                     /*!< enable or disable the extended mode */
    uint32_t asyn_wait;                                                         /*!< enable or disable the asynchronous wait function */
    uint32_t cram_page_size;                                                    /*!< specify CRAM page size */
    uint32_t write_mode;                                                        /*!< the write mode, synchronous mode or asynchronous mode */
    /* common timing parameters */
    uint32_t asyn_access_mode;                                                  /*!< asynchronous access mode */
    uint32_t syn_data_latency;                                                  /*!< configure the data latency */
    uint32_t syn_clk_division;                                                  /*!< configure the clock divide ratio */
    uint32_t bus_latency;                                                       /*!< configure the bus latency */
    uint32_t asyn_data_setuptime;                                               /*!< configure the data setup time, asynchronous access mode valid */
    uint32_t asyn_address_holdtime;                                             /*!< configure the address hold time, asynchronous access mode valid */
    uint32_t asyn_address_setuptime;                                            /*!< configure the address setup time, asynchronous access mode valid */
    /* write timing parameters */
    uint32_t write_asyn_access_mode;                                            /*!< write asynchronous access mode */
    uint32_t write_bus_latency;                                                 /*!< write configure the bus latency */
    uint32_t write_asyn_data_setuptime;                                         /*!< write configure the data setup time, asynchronous access mode valid */
    uint32_t write_asyn_address_holdtime;                                       /*!< write configure the address hold time, asynchronous access mode valid */
    uint32_t write_asyn_address_setuptime;                                      /*!< write configure the address setup time, asynchronous access mode valid */
} hal_exmc_sram_init_struct;

/* @STRUCT_MEMBER: extended_mode */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: asyn_wait */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: nwait_signal */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: memory_write */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: burst_mode */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: address_data_mux */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: syn_data_latency */
/* @=NULL */

/* @STRUCT_MEMBER: syn_clk_division */
/* @=NULL */

/* @STRUCT_MEMBER: bus_latency */
/* @=NULL */

/* @STRUCT_MEMBER: asyn_data_setuptime */
/* @=NULL */

/* @STRUCT_MEMBER: asyn_address_holdtime */
/* @=NULL */

/* @STRUCT_MEMBER: asyn_address_setuptime */
/* @=NULL */

/* @STRUCT_MEMBER: write_bus_latency */
/* @=NULL */

/* @STRUCT_MEMBER: write_asyn_data_setuptime */
/* @=NULL */

/* @STRUCT_MEMBER: write_asyn_address_holdtime */
/* @=NULL */

/* @STRUCT_MEMBER: write_asyn_address_setuptime */
/* @=NULL */

/* @PARA: exmc_sram_dev */
/* @STRUCT: EXMC sram device structure definition */
typedef struct {
    uint32_t sram_region;                                                       /*!< EXMC sram region */
    hal_dma_dev_struct *p_dma_rx;                                               /*!< DMA receive pointer */
    hal_dma_dev_struct *p_dma_tx;                                               /*!< DMA transmit pointer */
    __IO hal_irq_handle_cb dma_transfer_complete;                               /*!< dma transfer complete interrupt handler */
    __IO hal_irq_handle_cb dma_transfer_error;                                  /*!< dma transfer error interrupt handler */
    hal_exmc_sram_state_enum state;                                             /*!< EXMC sram state */
    hal_mutex_enum mutex;                                                       /*!< EXMC sram mutex set */
    void *priv;                                                                 /*!< EXMC sram private data*/
} hal_exmc_sram_dev_struct;

/* EXMC device user callback function pointer */
typedef void (*hal_sram_user_cb)(hal_exmc_sram_dev_struct *sram_dev);

/* EXMC callback structure */
typedef struct {
    hal_sram_user_cb dma_transfer_complete;                                     /*!< EXMC user complete callback function */
    hal_sram_user_cb dma_transfer_error;                                        /*!< EXMC user error callback function */
} hal_exmc_sram_user_callback_struct;

/* hal function declarations */
/* @FUNCTION:initialization functions */
/* initialize EXMC NOR/SRAM region */
int32_t hal_exmc_sram_init(hal_exmc_sram_dev_struct *exmc_sram_dev, hal_exmc_sram_init_struct *sram_init);
/* @END */
/* initialize the SRAM structure with default value */
int32_t hal_exmc_sram_struct_init(hal_exmc_sram_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize EXMC NOR/SRAM region */
int32_t hal_exmc_sram_deinit(hal_exmc_sram_dev_struct *exmc_sram_dev);

/* write and read data function */
/* write a Byte buffer(data is 8 bits ) to the EXMC SRAM memory */
int32_t hal_exmc_sram_write_8bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint8_t *in_addr, uint32_t *sram_addr, uint32_t size);
/* read a block of 8-bit data from the EXMC SRAM memory */
int32_t hal_exmc_sram_read_8bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint8_t *out_addr, uint32_t *sram_addr, uint32_t size);
/* write a Half-word buffer(data is 16 bits) to the EXMC SRAM memory */
int32_t hal_exmc_sram_write_16bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint16_t *in_addr, uint32_t *sram_addr, uint32_t size);
/* read a block of 16-bit data from the EXMC SRAM memory */
int32_t hal_exmc_sram_read_16bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint16_t *out_addr, uint32_t *sram_addr, uint32_t size);
/* write a word buffer(data is 32 bits) to the EXMC SRAM memory */
int32_t hal_exmc_sram_write_32bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint32_t *in_addr, uint32_t *sram_addr, uint32_t size);
/* read a block of 32-bit data from the EXMC SRAM memory */
int32_t hal_exmc_sram_read_32bit(hal_exmc_sram_dev_struct *exmc_sram_dev, uint32_t *out_addr, uint32_t *sram_addr, uint32_t size);
/* write a block of 32-bit data from the EXMC SRAM memory using DMA transfer */
int32_t hal_exmc_sram_write_dma(hal_exmc_sram_dev_struct *exmc_sram_dev, uint32_t *in_addr, uint32_t *sram_addr, uint32_t size, \
                                hal_exmc_sram_user_callback_struct *p_user_func);
/* read a block of 32-bit data from the EXMC SRAM memory using DMA transfer */
int32_t hal_exmc_sram_read_dma(hal_exmc_sram_dev_struct *exmc_sram_dev, uint32_t *out_addr, uint32_t *sram_addr, uint32_t size, \
                               hal_exmc_sram_user_callback_struct *p_user_func);

/* configuration functions */
/* EXMC sram write operation enable */
int32_t hal_exmc_sram_write_operation_enable(hal_exmc_sram_dev_struct *exmc_sram_dev);
/* EXMC sram write operation disable */
int32_t hal_exmc_sram_write_operation_disable(hal_exmc_sram_dev_struct *exmc_sram_dev);
/* get NOR/PSRAM and SDRAM remap configuration */
uint32_t hal_exmc_sram_sdram_remap_get(void);
/* return the EXMC SRAM state */
hal_exmc_sram_state_enum hal_exmc_sram_state_get(hal_exmc_sram_dev_struct *exmc_sram_dev);

/* configuration functions */
/* configure NOR/PSRAM and SDRAM remap */
void hals_exmc_sram_sdram_remap_config(uint32_t bank_remap);
/* configure consecutive clock mode (only supported in EXMC BANK0 REGION0) */
void hals_exmc_sram_consecutive_clock_config(uint32_t clock_mode);
/* configure CRAM page size */
void hals_exmc_sram_page_size_config(uint32_t exmc_sram_region, uint32_t page_size);

#endif /* GD32H7XX_HAL_EXMC_SRAM_H */
