/*!
    \file    gd32h7xx_hal_exmc_sdram.h
    \brief   definitions for the EXMC SDRAM

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_EXMC_SDRAM_H
#define GD32H7XX_HAL_EXMC_SDRAM_H

#include "gd32h7xx_hal.h"

/* constants definitions */
/* EXMC SDRAM device register reset value */
#define SDRAM_DEVICE_SDCTL_RESET          ((uint32_t)0x000002D0U)               /* SDCTL register reset value */
#define SDRAM_DEVICE_SDTCFG_RESET         ((uint32_t)0x0FFFFFFFU)               /* SDTCFG register reset value */
#define SDRAM_DEVICE_SDCMD_RESET          ((uint32_t)0x00000000U)               /* SDCMD register reset value */
#define SDRAM_DEVICE_SDARI_RESET          ((uint32_t)0x00000000U)               /* SDARI register reset value */
#define SDRAM_DEVICE_SDRSCTL_RESET        ((uint32_t)0x00000000U)               /* SDRSCTL register reset value */

/* EXMC_SDCTL register bit offset */
#define SDCTL_WPEN_OFFSET                 ((uint32_t)0x00000009U)               /* bit offset of WPEN */
#define SDCTL_BRSTRD_OFFSET               ((uint32_t)0x0000000CU)               /* bit offset of BRSTRD */

/* EXMC_SDTCFG register bit offset */
#define SDTCFG_XSRD_OFFSET                ((uint32_t)0x00000004U)               /* bit offset of XSRD */
#define SDTCFG_RASD_OFFSET                ((uint32_t)0x00000008U)               /* bit offset of RASD */
#define SDTCFG_ARFD_OFFSET                ((uint32_t)0x0000000CU)               /* bit offset of ARFD */
#define SDTCFG_WRD_OFFSET                 ((uint32_t)0x00000010U)               /* bit offset of WRD */
#define SDTCFG_RPD_OFFSET                 ((uint32_t)0x00000014U)               /* bit offset of RPD */
#define SDTCFG_RCD_OFFSET                 ((uint32_t)0x00000018U)               /* bit offset of RCD */

/* EXMC_SDCMD register bit offset */
#define SDCMD_NARF_OFFSET                 ((uint32_t)0x00000005U)               /* bit offset of NARF */
#define SDCMD_MRC_OFFSET                  ((uint32_t)0x00000009U)               /* bit offset of MRC */

/* EXMC_SDARI register bit offset */
#define SDARI_ARINTV_OFFSET               ((uint32_t)0x00000001U)               /* bit offset of ARINTV */

/* EXMC_SDRSCTL register bit offset */
#define SDRSCTL_SSCR_OFFSET               ((uint32_t)0x00000001U)               /* bit offset of SSCR */
#define SDRSCTL_SDSC_OFFSET               ((uint32_t)0x00000004U)               /* bit offset of SDSC */

/* EXMC_SDSTAT register bit offset */
#define SDSTAT_STA0_OFFSET                ((uint32_t)0x00000001U)               /* bit offset of STA0 */
#define SDSTAT_STA1_OFFSET                ((uint32_t)0x00000003U)               /* bit offset of STA1 */

/* SDRAM device address */
#define SDRAM_DEVICE0_ADDR                ((uint32_t)0xC0000000U)               /* sdram0 device base address */
#define SDRAM_DEVICE1_ADDR                ((uint32_t)0xD0000000U)               /* sdram1 device base address */

#define SDRAM_FREASH_TIMEOUT             ((uint32_t)0x0000FFFFU)                /* SDRAM device fresh timeout */

/* EXMC structure type enum */
typedef enum {
    HAL_EXMC_SDRAM_INIT_STRUCT = 0x00U,                                         /*!< SDRAM initialization structure */
    HAL_EXMC_SDRAM_DEV_STRUCT,                                                  /*!< SDRAM device structure */
    HAL_EXMC_SDRAM_IRQ_INIT_STRUCT,                                             /*!< SDRAM interrupt callback initialization structure */
    HAL_EXMC_SDRAM_COMMAND_PARAMETER_STRUCT,                                    /*!< SDRAM command parameter structure */
    HAL_EXMC_SDRAM_USER_CALLBACK_STRUCT                                         /*!< SDRAM user callback structure */
} hal_exmc_sdram_struct_type_enum;

/* EXMC state type enum */
typedef enum {
    HAL_EXMC_SDRAM_STATE_RESET         = (uint32_t)0x00000000U,                 /*!< SDRAM is not initialized or disabled */
    HAL_EXMC_SDRAM_STATE_READY         = (uint32_t)0x00000001U,                 /*!< SDRAM is ready */
    HAL_EXMC_SDRAM_STATE_BUSY          = (uint32_t)0x00000002U,                 /*!< SDRAM is busy */
    HAL_EXMC_SDRAM_STATE_ERROR         = (uint32_t)0x00000003U,                 /*!< SDRAM error state */
    HAL_EXMC_SDRAM_STATE_PROTECTED     = (uint32_t)0x00000004U,                 /*!< SDRAM write protected */
    HAL_EXMC_SDRAM_STATE_PRECHARGED    = (uint32_t)0x00000005U                  /*!< SDRAM device precharged */
} hal_exmc_sdram_state_enum;

/* EXMC device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb refresh_error_handle;                                /*!< refresh error interrupt handler */
} hal_exmc_sdram_irq_struct;

/* @PARA: sdram_init */
/* @STRUCT: EXMC SDRAM initialize structure */
typedef struct {
    uint32_t sdram_device;                                                      /*!< device of SDRAM */
    uint32_t bank_remap;                                                        /*!< select EXMC NOR/PSRAM remap region */
    uint32_t pipeline_read_delay;                                               /*!< the delay for reading data after CAS latency in CK_EXMC clock cycles */
    uint32_t burst_read_switch;                                                 /*!< enable or disable the burst read */
    uint32_t sdclock_config;                                                    /*!< the SDCLK memory clock for both SDRAM devices */
    uint32_t write_protection;                                                  /*!< enable or disable SDRAM device write protection function */
    uint32_t cas_latency;                                                       /*!< configure the SDRAM CAS latency */
    uint32_t internal_bank_number;                                              /*!< the number of internal bank */
    uint32_t data_width;                                                        /*!< the data bus width of SDRAM memory */
    uint32_t row_address_width;                                                 /*!< the bit width of a row address */
    uint32_t column_address_width;                                              /*!< the bit width of a column address */
    uint32_t auto_refresh_interval;                                             /*!< auto-refresh interval */
    uint32_t read_sample_enable;                                                /*!< read sample enable bit */
    uint32_t read_sample_cycle;                                                 /*!< select sample cycle of read data */
    uint32_t read_sample_delay;                                                 /*!< select the delayed sample clock of read data */
    /* timing parameters */
    uint32_t row_to_column_delay;                                               /*!< configure the row to column delay */
    uint32_t row_precharge_delay;                                               /*!< configure the row precharge delay */
    uint32_t write_recovery_delay;                                              /*!< configure the write recovery delay */
    uint32_t auto_refresh_delay;                                                /*!< configure the auto refresh delay */
    uint32_t row_address_select_delay;                                          /*!< configure the row address select delay */
    uint32_t exit_selfrefresh_delay;                                            /*!< configure the exit self-refresh delay */
    uint32_t load_mode_register_delay;                                          /*!< configure the load mode register delay */
} hal_exmc_sdram_init_struct;

/* @STRUCT_MEMBER: write_protection */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: burst_read_switch */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: auto_refresh_interval */
/* @=NULL */

/* @STRUCT_MEMBER: read_sample_enable */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: read_sample_delay */
/* @=NULL */

/* @STRUCT_MEMBER: row_to_column_delay */
/* @=NULL */

/* @STRUCT_MEMBER: row_precharge_delay */
/* @=NULL */

/* @STRUCT_MEMBER: write_recovery_delay */
/* @=NULL */

/* @STRUCT_MEMBER: auto_refresh_delay */
/* @=NULL */

/* @STRUCT_MEMBER: row_address_select_delay */
/* @=NULL */

/* @STRUCT_MEMBER: exit_selfrefresh_delay */
/* @=NULL */

/* @STRUCT_MEMBER: load_mode_register_delay */
/* @=NULL */

/* EXMC SDRAM command initialize structure */
typedef struct {
    uint32_t mode_register_content;                                             /*!< the SDRAM mode register content */
    uint32_t auto_refresh_number;                                               /*!< the number of successive auto-refresh cycles will be send when CMD = 011 */
    uint32_t bank_select;                                                       /*!< the bank which command will be sent to */
    uint32_t command;                                                           /*!< the commands that will be sent to SDRAM */
} hal_exmc_sdram_command_parameter_struct;

/* @PARA: exmc_dev */
/* @STRUCT: EXMC device structure definition */
typedef struct {
    uint32_t sdram_device;                                                      /*!< SDRAM device */
    hal_dma_dev_struct *p_dma_rx;                                               /*!< DMA receive pointer */
    hal_dma_dev_struct *p_dma_tx;                                               /*!< DMA transmit pointer */
    hal_exmc_sdram_irq_struct exmc_irq;                                         /*!< EXMC device interrupt callback function pointer structure */
    __IO hal_irq_handle_cb dma_transfer_complete;                               /*!< dma transfer complete interrupt handler */
    __IO hal_irq_handle_cb dma_transfer_error;                                  /*!< dma transfer error interrupt handler */
    hal_exmc_sdram_state_enum state;                                            /*!< EXMC state */
    hal_mutex_enum mutex;                                                       /*!< EXMC mutex set */
    void *priv;                                                                 /*!< EXMC private data */
} hal_exmc_sdram_dev_struct;

/* EXMC device user callback function pointer */
typedef void (*hal_sdram_user_cb)(hal_exmc_sdram_dev_struct *sdram_dev);

/* EXMC callback structure */
typedef struct {
    hal_sdram_user_cb dma_transfer_complete;                                     /*!< EXMC user complete callback function */
    hal_sdram_user_cb dma_transfer_error;                                        /*!< EXMC user error callback function */
} hal_exmc_sdram_user_callback_struct;

/* hal function declarations */
/* @FUNCTION:initialization functions */
/* initialize EXMC SDRAM device */
int32_t hal_exmc_sdram_init(hal_exmc_sdram_dev_struct *exmc_dev, hal_exmc_sdram_init_struct *sdram_init);
/* @END */
/* initialize exmc parameter struct with the default values */
int32_t hal_exmc_sdram_struct_init(hal_exmc_sdram_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize EXMC SDRAM device */
int32_t hal_exmc_sdram_deinit(hal_exmc_sdram_dev_struct *exmc_dev);

/* EXMC interrupt handle functions */
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_exmc_sdram_irq_handle_set(hal_exmc_sdram_dev_struct *exmc_dev, hal_exmc_sdram_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_exmc_sdram_irq_handle_all_reset(hal_exmc_sdram_dev_struct *exmc_dev);
/* EXMC SDRAM interrupt handler content function, which is merely used in EXMC_IRQHandler */
void hal_exmc_sdram_irq(hal_exmc_sdram_dev_struct *exmc_dev);

/* write 8 bit data to sdram memory */
int32_t hal_exmc_sdram_write_8bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *write_address, uint8_t *write_buffer, \
                                  uint32_t length, hal_exmc_sdram_irq_struct *p_irq);
/* read 8 bit data to sdram memory */
int32_t hal_exmc_sdram_read_8bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *read_address, uint8_t *read_buffer, \
                                uint32_t length, hal_exmc_sdram_irq_struct *p_irq);
/* write 16 bit data to sdram memory */
int32_t hal_exmc_sdram_write_16bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *write_address, uint16_t *write_buffer, \
                                   uint32_t length, hal_exmc_sdram_irq_struct *p_irq);
/* read 16 bit data to sdram memory */
int32_t hal_exmc_sdram_read_16bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *read_address, uint16_t *read_buffer, \
                                  uint32_t length, hal_exmc_sdram_irq_struct *p_irq);
/* write 32 bit data to sdram memory */
int32_t hal_exmc_sdram_write_32bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *write_address, uint32_t *write_buffer, \
                                   uint32_t length, hal_exmc_sdram_irq_struct *p_irq);
/* read 32 bit data to sdram memory */
int32_t hal_exmc_sdram_read_32bit(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *read_address, uint32_t *read_buffer, \
                                  uint32_t length, hal_exmc_sdram_irq_struct *p_irq);
/* write 32 bit data to sdram memory */
int32_t hal_exmc_sdram_write_dma(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *in_addr, uint32_t *sram_addr, \
                                uint32_t size,  hal_exmc_sdram_user_callback_struct *p_user_func);
/* read 32 bit data to sdram memory */
int32_t hal_exmc_sdram_read_dma(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t *out_addr, uint32_t *sram_addr, \
                                uint32_t size,  hal_exmc_sdram_user_callback_struct *p_user_func);

/* enable the write protection function */
int32_t hal_exmc_sdram_write_operation_enable(hal_exmc_sdram_dev_struct *exmc_dev);
/* disable the write protection function */
int32_t hal_exmc_sdram_write_operation_disable(hal_exmc_sdram_dev_struct *exmc_dev);
/* configure the SDRAM memory command */
int32_t hal_exmc_sdram_command_send(hal_exmc_sdram_dev_struct *exmc_dev, \
                                    hal_exmc_sdram_command_parameter_struct *exmc_sdram_command);
/* set auto-refresh interval */
int32_t hal_exmc_sdram_refresh_count_set(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t exmc_count);
/* set the number of successive auto-refresh command */
int32_t hal_exmc_sdram_autorefresh_number_set(hal_exmc_sdram_dev_struct *exmc_dev, uint32_t exmc_number);
/* return the sdram state */
hal_exmc_sdram_state_enum hal_exmc_sdram_state_get(hal_exmc_sdram_dev_struct *exmc_dev);

/* configuration functions */
/* enable read sample function */
void hals_exmc_sdram_readsample_enable(void);
/* disable read sample function */
void hals_exmc_sdram_readsample_disable(void);
/* configure the delayed sample clock of read data */
void hals_exmc_sdram_readsample_config(uint32_t delay_cell, uint32_t extra_clk);
/* configure the SDRAM memory command */
void hals_exmc_sdram_command_config(hal_exmc_sdram_command_parameter_struct *exmc_sdram_command);
/* get the status of SDRAM device0 or device1 */
uint32_t hals_exmc_sdram_bankstatus_get(uint32_t exmc_sdram_device);
/* configure NOR/PSRAM and SDRAM remap */
void hals_exmc_norsram_sdram_remap_config(uint32_t bank_remap);

/* interrupt & flag functions */
/* get EXMC flag status */
FlagStatus hals_exmc_sdram_flag_get(uint32_t flag);
/* clear EXMC flag status */
void hals_exmc_sdram_flag_clear(void);
/* enable EXMC interrupt */
void hals_exmc_sdram_interrupt_enable(void);
/* disable EXMC interrupt */
void hals_exmc_sdram_interrupt_disable(void);
/* get EXMC interrupt flag */
FlagStatus hals_exmc_sdram_interrupt_flag_get(void);
/* clear EXMC interrupt flag */
void hals_exmc_sdram_interrupt_flag_clear(void);

#endif /* GD32H7XX_HAL_EXMC_SDRAM_H */
