/*!
    \file    gd32h7xx_hal_exmc_nor.h
    \brief   definitions for the EXMC NOR

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_EXMC_NOR_H
#define GD32H7XX_HAL_EXMC_NOR_H

#include "gd32h7xx_hal.h"

/* read id command define */
#define CMD_READID_ADD_1ST                      ((uint16_t)0x0555U)             /*!< CMD_READID_ADD_1ST */
#define CMD_READID_DATA_1ST                     ((uint16_t)0x00AAU)             /*!< CMD_READID_DATA_1ST */
#define CMD_READID_ADD_2ND                      ((uint16_t)0x02AAU)             /*!< CMD_READID_ADD_2ND */
#define CMD_READID_DATA_2ND                     ((uint16_t)0x0055U)             /*!< CMD_READID_DATA_2ND */
#define CMD_READID_ADD_3RD                      ((uint16_t)0x0555U)             /*!< CMD_READID_ADD_3RD */
#define CMD_READID_DATA_3RD                     ((uint16_t)0x0090U)             /*!< CMD_READID_DATA_3RD */

/* Block Erase Command define */
#define CMD_BLOCKERASE_ADD_1ST                  ((uint16_t)0x0555U)             /*!< CMD_BLOCKERASE_ADD_1ST */
#define CMD_BLOCKERASE_DATA_1ST                 ((uint16_t)0x00AAU)             /*!< CMD_BLOCKERASE_DATA_1ST */
#define CMD_BLOCKERASE_ADD_2ND                  ((uint16_t)0x02AAU)             /*!< CMD_BLOCKERASE_ADD_2ND */
#define CMD_BLOCKERASE_DATA_2ND                 ((uint16_t)0x0055U)             /*!< CMD_BLOCKERASE_DATA_2ND */
#define CMD_BLOCKERASE_ADD_3RD                  ((uint16_t)0x0555U)             /*!< CMD_BLOCKERASE_ADD_3RD */
#define CMD_BLOCKERASE_DATA_3RD                 ((uint16_t)0x0080U)             /*!< CMD_BLOCKERASE_DATA_3RD */
#define CMD_BLOCKERASE_ADD_4TH                  ((uint16_t)0x0555U)             /*!< CMD_BLOCKERASE_ADD_4TH */
#define CMD_BLOCKERASE_DATA_4TH                 ((uint16_t)0x00AAU)             /*!< CMD_BLOCKERASE_DATA_4TH */
#define CMD_BLOCKERASE_ADD_5TH                  ((uint16_t)0x02AAU)             /*!< CMD_BLOCKERASE_ADD_5TH */
#define CMD_BLOCKERASE_DATA_5TH                 ((uint16_t)0x0055U)             /*!< CMD_BLOCKERASE_DATA_5TH */
#define CMD_BLOCKERASE_DATA_6TH                 ((uint16_t)0x0030U)             /*!< CMD_BLOCKERASE_DATA_6TH */

/* chip erase command define */
#define CMD_CHIPERASE_ADD_1ST                   ((uint16_t)0x0555U)             /*!< CMD_CHIPERASE_ADD_1ST */
#define CMD_CHIPERASE_DATA_1ST                  ((uint16_t)0x00AAU)             /*!< CMD_CHIPERASE_DATA_1ST */
#define CMD_CHIPERASE_ADD_2ND                   ((uint16_t)0x02AAU)             /*!< CMD_CHIPERASE_ADD_2ND */
#define CMD_CHIPERASE_DATA_2ND                  ((uint16_t)0x0055U)             /*!< CMD_CHIPERASE_DATA_2ND */
#define CMD_CHIPERASE_ADD_3RD                   ((uint16_t)0x0555U)             /*!< CMD_CHIPERASE_ADD_3RD */
#define CMD_CHIPERASE_DATA_3RD                  ((uint16_t)0x0080U)             /*!< CMD_CHIPERASE_DATA_3RD */
#define CMD_CHIPERASE_ADD_4TH                   ((uint16_t)0x0555U)             /*!< CMD_CHIPERASE_ADD_4TH */
#define CMD_CHIPERASE_DATA_4TH                  ((uint16_t)0x00AAU)             /*!< CMD_CHIPERASE_DATA_4TH */
#define CMD_CHIPERASE_ADD_5TH                   ((uint16_t)0x02AAU)             /*!< CMD_CHIPERASE_ADD_5TH */
#define CMD_CHIPERASE_DATA_5TH                  ((uint16_t)0x0055U)             /*!< CMD_CHIPERASE_DATA_5TH */
#define CMD_CHIPERASE_ADD_6TH                   ((uint16_t)0x0555U)             /*!< CMD_CHIPERASE_ADD_6TH */
#define CMD_CHIPERASE_DATA_6TH                  ((uint16_t)0x0010U)             /*!< CMD_CHIPERASE_DATA_6TH */

/* reset command define */
#define CMD_RESET_ADD_1ST                       ((uint16_t)0x0555U)             /*!< CMD_RESET_ADD_1ST */
#define CMD_RESET_DATA_1ST                      ((uint16_t)0x00AAU)             /*!< CMD_RESET_DATA_1ST */
#define CMD_RESET_ADD_2ND                       ((uint16_t)0x02AAU)             /*!< CMD_RESET_ADD_2ND */
#define CMD_RESET_DATA_2ND                      ((uint16_t)0x0055U)             /*!< CMD_RESET_DATA_2ND */
#define CMD_RESET_DATA_3RD                      ((uint16_t)0x00F0U)             /*!< CMD_RESET_DATA_3RD */

/* read command define */
#define CMD_READ_ADD_1ST                        ((uint16_t)0x0555U)             /*!< CMD_READ_ADD_1ST */
#define CMD_READ_DATA_1ST                       ((uint16_t)0x00AAU)             /*!< CMD_READ_DATA_1ST */
#define CMD_READ_ADD_2ND                        ((uint16_t)0x02AAU)             /*!< CMD_READ_ADD_2ND */
#define CMD_READ_DATA_2ND                       ((uint16_t)0x0055U)             /*!< CMD_READ_DATA_2ND */
#define CMD_READ_DATA_3RD                       ((uint16_t)0x00F0U)             /*!< CMD_READ_DATA_3RD */

/* write command define */
#define CMD_WRITE_ADD_1ST                       ((uint16_t)0x0555U)             /*!< CMD_WRITE_ADD_1ST */
#define CMD_WRITE_DATA_1ST                      ((uint16_t)0x00AAU)             /*!< CMD_WRITE_DATA_1ST */
#define CMD_WRITE_ADD_2ND                       ((uint16_t)0x02AAU)             /*!< CMD_WRITE_ADD_2ND */
#define CMD_WRITE_DATA_2ND                      ((uint16_t)0x0055U)             /*!< CMD_WRITE_DATA_2ND */
#define CMD_WRITE_ADD_3RD                       ((uint16_t)0x0555U)             /*!< CMD_WRITE_ADD_3RD */
#define CMD_WRITE_DATA_3RD                      ((uint16_t)0x00A0U)             /*!< CMD_WRITE_DATA_3RD */

/* max read and write address */
#define NOR_MAX_ADDRESS                         ((uint32_t)0x01000000U)

#define EXMC_NOR_REGION0                        (0U)                            /*!< NOR device region0 */
#define EXMC_NOR_REGION1                        (1U)                            /*!< NOR device region1 */
#define EXMC_NOR_REGION2                        (2U)                            /*!< NOR device region2 */
#define EXMC_NOR_REGION3                        (3U)                            /*!< NOR device region3 */

/* NOR memory device read/write start address */
#define BANK0_NOR0_ADDR                         ((uint32_t)0x60000000U)         /*!< NOR device bank0 address */
#define BANK0_NOR1_ADDR                         ((uint32_t)0x64000000U)         /*!< NOR device bank1 address */
#define BANK0_NOR2_ADDR                         ((uint32_t)0x68000000U)         /*!< NOR device bank2 address */
#define BANK0_NOR3_ADDR                         ((uint32_t)0x6C000000U)         /*!< NOR device bank3 address */

/* NOR memory data width */
#define NOR_MEMORY_8B                           ((uint8_t)0x00)                 /*!< NOR memory data width */
#define NOR_MEMORY_16B                          ((uint8_t)0x01)                 /*!< NOR memory data width */

#define ADDR_SHIFT8(NORx_ADDR, ADDR)            ((NORx_ADDR) + (ADDR))
#define ADDR_SHIFT(NORx_ADDR, ADDR)             ((uint32_t)((NORx_ADDR) + ((uint32_t)(ADDR) << 1U)))

#define NOR_WRITE(Address, Data)                do {                                        \
                                                    (*(__IO uint16_t *)(Address) = (Data)); \
                                                    __DSB();                                \
                                                } while(0)

#define NOR_BLOCKERASE_TIMEOUT             ((uint32_t)0x00A00000U)              /*!< NOR memory block erase timeout */
#define NOR_CHIPERASE_TIMEOUT              ((uint32_t)0x30000000U)              /*!< NOR memory chip erase timeout */
#define NOR_PROGRAM_TIMEOUT                ((uint32_t)0x00001400U)              /*!< NOR memory program timeout */

/* EXMC structure type enum */
typedef enum {
    HAL_EXMC_NOR_INIT_STRUCT = 0x00U,                                           /*!< NOR initialization structure */
    HAL_EXMC_NOR_DEV_STRUCT,                                                    /*!< NOR device structure */
    HAL_EXMC_NOR_CONFIG_STRUCT,                                                 /*!< NOR configuration structure */
    HAL_EXMC_NOR_ID_STRUCT                                                      /*!< NOR id structure */
} hal_exmc_nor_struct_type_enum;

/* NOR status */
typedef enum {
    HAL_NOR_STATUS_SUCCESS = 0,                                                 /*!< NOR nor device status success */
    HAL_NOR_STATUS_ONGOING,                                                     /*!< NOR nor device status ongoing */
    HAL_NOR_STATUS_ERROR,                                                       /*!< NOR nor device status error */
    HAL_NOR_STATUS_TIMEOUT                                                      /*!< NOR nor device status timeout */
} hal_exmc_nor_status_enum;

/* NOR state type enum */
typedef enum {
    HAL_EXMC_NOR_STATE_RESET             = 0x00U,                               /*!< NOR not yet initialized or disabled  */
    HAL_EXMC_NOR_STATE_READY             = 0x01U,                               /*!< NOR initialized and ready for use    */
    HAL_EXMC_NOR_STATE_BUSY              = 0x02U,                               /*!< NOR internal processing is ongoing   */
    HAL_EXMC_NOR_STATE_ERROR             = 0x03U,                               /*!< NOR error state                      */
    HAL_EXMC_NOR_STATE_PROTECTED         = 0x04U                                /*!< NOR nor device write protected   */
} hal_exmc_nor_state_enum;

/* NOR id structure */
typedef struct {
    uint16_t manufacturer_code;                                                 /*!< NOR device manufacturer_code */
    uint16_t device_code1;                                                      /*!< NOR device device_code1 */
    uint16_t device_code2;                                                      /*!< NOR device device_code2 */
    uint16_t device_code3;                                                      /*!< NOR device device_code3 */
} hal_exmc_nor_id_struct;

/* NOR cfi structure */
typedef struct {
    uint16_t cfi_1;                                                             /*!< NOR device common flash interface 1 */
    uint16_t cfi_2;                                                             /*!< NOR device common flash interface 2 */
    uint16_t cfi_3;                                                             /*!< NOR device common flash interface 3 */
    uint16_t cfi_4;                                                             /*!< NOR device common flash interface 3 */
} hal_exmc_nor_cfg_struct;

/* @PARA: nor_init */
/* @STRUCT: EXMC NOR initialize structure */
typedef struct {
    uint32_t norsram_region;                                                    /*!< select the region of EXMC NOR/PSRAM region */
    uint32_t bank_remap;                                                        /*!< select EXMC NOR/PSRAM remap region */
    uint32_t consecutive_clock;                                                 /*!< configure consecutive clock mode */
    uint32_t address_data_mux;                                                  /*!< specify whether the data bus and address bus are multiplexed */
    uint32_t memory_type;                                                       /*!< specify the type of external memory */
    uint32_t databus_width;                                                     /*!< specify the data bus width of external memory */
    uint32_t burst_mode;                                                        /*!< enable or disable the burst mode */
    uint32_t nwait_polarity;                                                    /*!< specify the polarity of NWAIT signal from memory */
    uint32_t nwait_config;                                                      /*!< NWAIT signal configuration */
    uint32_t memory_write;                                                      /*!< enable or disable the write operation */
    uint32_t nwait_signal;                                                      /*!< enable or disable the NWAIT signal while in synchronous bust mode */
    uint32_t extended_mode;                                                     /*!< enable or disable the extended mode */
    uint32_t asyn_wait;                                                         /*!< enable or disable the asynchronous wait function */
    uint32_t cram_page_size;                                                    /*!< specify CRAM page size */
    uint32_t write_mode;                                                        /*!< the write mode, synchronous mode or asynchronous mode */
    /* common timing parameters */
    uint32_t asyn_access_mode;                                                  /*!< asynchronous access mode */
    uint32_t syn_data_latency;                                                  /*!< configure the data latency */
    uint32_t syn_clk_division;                                                  /*!< configure the clock divide ratio */
    uint32_t bus_latency;                                                       /*!< configure the bus latency */
    uint32_t asyn_data_setuptime;                                               /*!< configure the data setup time, asynchronous access mode valid */
    uint32_t asyn_address_holdtime;                                             /*!< configure the address hold time, asynchronous access mode valid */
    uint32_t asyn_address_setuptime;                                            /*!< configure the address setup time, asynchronous access mode valid */
    /* write timing parameters */
    uint32_t write_asyn_access_mode;                                            /*!< write asynchronous access mode */
    uint32_t write_bus_latency;                                                 /*!< write configure the bus latency */
    uint32_t write_asyn_data_setuptime;                                         /*!< write configure the data setup time, asynchronous access mode valid */
    uint32_t write_asyn_address_holdtime;                                       /*!< write configure the address hold time, asynchronous access mode valid */
    uint32_t write_asyn_address_setuptime;                                      /*!< write configure the address setup time, asynchronous access mode valid */
} hal_exmc_nor_init_struct;

/* @STRUCT_MEMBER: extended_mode */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: asyn_wait */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: nwait_signal */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: memory_write */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: burst_mode */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: address_data_mux */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: syn_data_latency */
/* @=NULL */

/* @STRUCT_MEMBER: syn_clk_division */
/* @=NULL */

/* @STRUCT_MEMBER: bus_latency */
/* @=NULL */

/* @STRUCT_MEMBER: asyn_data_setuptime */
/* @=NULL */

/* @STRUCT_MEMBER: asyn_address_holdtime */
/* @=NULL */

/* @STRUCT_MEMBER: asyn_address_setuptime */
/* @=NULL */

/* @STRUCT_MEMBER: write_bus_latency */
/* @=NULL */

/* @STRUCT_MEMBER: write_asyn_data_setuptime */
/* @=NULL */

/* @STRUCT_MEMBER: write_asyn_address_holdtime */
/* @=NULL */

/* @STRUCT_MEMBER: write_asyn_address_setuptime */
/* @=NULL */

/* @PARA: nor_dev */
/* @STRUCT: EXMC NOR device structure definition */
typedef struct {
    uint32_t nor_region;                                                        /*!< EXMC NOR region */
    hal_exmc_nor_init_struct *init;                                              /*!< EXMC NOR init parameters */
    hal_exmc_nor_state_enum state;                                              /*!< EXMC NOR state */
    hal_mutex_enum mutex;                                                       /*!< EXMC NOR mutex set */
    void *priv;                                                                 /*!< EXMC NOR private data */
} hal_exmc_nor_dev_struct;

/* hal function declarations */
/* @FUNCTION:initialization functions */
/* initialize EXMC NOR/SRAM region */
int32_t hal_exmc_nor_init(hal_exmc_nor_dev_struct *nor_dev, hal_exmc_nor_init_struct *nor_init);
/* @END */
/* initialize the NOR structure with default values */
int32_t hal_exmc_nor_struct_init(hal_exmc_nor_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize EXMC NOR/SRAM region */
int32_t hal_exmc_nor_deinit(hal_exmc_nor_dev_struct *exmc_nor_dev);

/* read the NOR memory id */
int32_t hal_exmc_nor_id_read(hal_exmc_nor_dev_struct *exmc_nor_dev, hal_exmc_nor_id_struct *nor_id);
/* NOR memory return to read mode */
int32_t hal_exmc_nor_readmode_return(hal_exmc_nor_dev_struct *exmc_nor_dev);
/* NOR memory read */
int32_t hal_exmc_nor_read(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t read_addr, uint16_t *data);
/* NOR memory write */
int32_t hal_exmc_nor_write(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t write_addr, uint16_t data);
/* NOR memory read buffer */
int32_t hal_exmc_nor_buffer_read(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t read_addr, uint16_t *data, uint32_t len);
/* NOR memory write buffer */
int32_t hal_exmc_nor_buffer_write(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t write_addr, uint16_t *data, uint32_t len);
/* NOR memory erase block */
int32_t hal_exmc_nor_block_erase(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t block_addr);
/* NOR memory erase chip */
int32_t hal_exmc_nor_chip_erase(hal_exmc_nor_dev_struct *exmc_nor_dev);
/* read NOR memory cfi id */
int32_t hal_exmc_nor_cfi_read(hal_exmc_nor_dev_struct *exmc_nor_dev, hal_exmc_nor_cfg_struct *p_cfi);
/* enable nor write operation */
int32_t hal_exmc_nor_write_operation_enable(hal_exmc_nor_dev_struct *exmc_nor_dev);
/* disable nor write operation */
int32_t hal_exmc_nor_write_operation_disable(hal_exmc_nor_dev_struct *exmc_nor_dev);
/* return the nor flash state */
hal_exmc_nor_state_enum hal_exmc_nor_state_get(hal_exmc_nor_dev_struct *exmc_nor_dev);
/* return the nor flash operation status */
hal_exmc_nor_status_enum hal_exmc_nor_status_get(hal_exmc_nor_dev_struct *exmc_nor_dev, uint32_t time_out);

#endif /* GD32H7XX_HAL_EXMC_NOR_H */
