/*!
    \file    gd32h7xx_hal_edout.h
    \brief   definitions for the EDOUT

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_EDOUT_H
#define GD32H7XX_HAL_EDOUT_H

#include "gd32h7xx_hal.h"

/* EDOUT definition */
#define EDOUT                            EDOUT_BASE                        /*!< EDOUT base address */

/* registers definitions */
#define EDOUT_CTL                        REG32((EDOUT) + 0x00000000U)      /*!< EDOUT control register */
#define EDOUT_ENABLE                     REG32((EDOUT) + 0x00000004U)      /*!< EDOUT enable register */
#define EDOUT_LOC                        REG32((EDOUT) + 0x00000008U)      /*!< EDOUT location register */
#define EDOUT_OCNT                       REG32((EDOUT) + 0x0000000CU)      /*!< EDOUT output counter register */
#define EDOUT_LCNT                       REG32((EDOUT) + 0x00000010U)      /*!< EDOUT location counter register */
#define EDOUT_ZCR                        REG32((EDOUT) + 0x00000014U)      /*!< EDOUT Z-phase configure register */

/* bits definitions */
/* EDOUT_CTL */
#define EDOUT_CTL_POL                    BIT(0)                            /*!< active polarity of the B-phase output signal selection */

/* EDOUT_ENABLE */
#define EDOUT_ENABLE_EDOUTEN             BIT(0)                            /*!< EDOUT enable bit */

/* EDOUT_LOC */
#define EDOUT_LOC_LOCMAX                 BITS(0,15)                        /*!< maximum location value for one rotation */

/* EDOUT_OCNT */
#define EDOUT_OCNT_EDGC                  BITS(0,15)                        /*!< number of edges of the A-phase and the B-phase signal for the next update period */
#define EDOUT_OCNT_PDC                   BITS(16,31)                       /*!< phase difference between the A-phase and the B-phase signal for the next update period */

/* EDOUT_LCNT */
#define EDOUT_LCNT_LOCCNT                BITS(0,15)                        /*!< current location value */

/* EDOUT_ZCR */
#define EDOUT_ZCR_ZOSP                   BITS(0,15)                        /*!< Z-phase output start location */
#define EDOUT_ZCR_ZOWH                   BITS(16,23)                       /*!< Z-phase output width */
#define EDOUT_ZCR_ZOMD                   BIT(24)                           /*!< Z-phase output mode */

/* constants definitions */
/* @PARA: b_phase_pol */
/* @DEFINE: b_phase_pol */
/* EDOUT_CTL */
#define EDOUT_POL_POSITIVE               ((uint32_t)0x00000000U)           /*!< active polarity is positive */
#define EDOUT_POL_NEGATIVE               EDOUT_CTL_POL                     /*!< active polarity is negative */

/* @PARA: z_phase */
/* @DEFINE: z_phase */
/* EDOUT_ENABLE */
#define EDOUT_STATE_DISABLE              ((uint32_t)0x00000000U)           /*!< disable EDOUT */
#define EDOUT_STATE_ENABLE               EDOUT_ENABLE_EDOUTEN              /*!< enable EDOUT */

/* @PARA: z_phase_mode */
/* @DEFINE: z_phase_mode */
/* EDOUT_ZCR */
#define EDOUT_Z_OUTPUT_MODE0             ((uint32_t)0x00000000U)           /*!< Z-phase output according to the current location */
#define EDOUT_Z_OUTPUT_MODE1             EDOUT_ZCR_ZOMD                    /*!< Z-phase output according to the number of edges */

/* enumerator definition */
/* EDOUT state enum */
typedef enum {
    HAL_EDOUT_STATE_NONE = 0,                                              /*!< NONE(default value) */
    HAL_EDOUT_STATE_RESET,                                                 /*!< RESET */
    HAL_EDOUT_STATE_READY,                                                 /*!< READY */
    HAL_EDOUT_STATE_BUSY                                                   /*!< BUSY */
} hal_edout_state_enum;

/* EDDOUT structure type enum */
typedef enum {
    HAL_EDOUT_INIT_STRUCT = 0,                                             /*!< EDOUT initialization structure */
    HAL_EDOUT_DEV_STRUCT                                                   /*!< EDOUT device information structure */
} hal_edout_struct_type_enum;

/* @STRUCT_MEMBER: resolution */
/* @ENUM: EDOUT resolution enumeration */
typedef enum {
    EDOUT_USER_SET_RESOLUTION            = ((uint64_t)0U),                 /*!< user input encoder resolution */
    EDOUT_RESOLUTION_12BIT               = ((uint64_t)4096U),              /*!< the resolution of the encoder is 12 bits */
    EDOUT_RESOLUTION_14BIT               = ((uint64_t)16384U),             /*!< the resolution of the encoder is 14 bits */
    EDOUT_RESOLUTION_16BIT               = ((uint64_t)65536U),             /*!< the resolution of the encoder is 16 bits */
    EDOUT_RESOLUTION_18BIT               = ((uint64_t)262144U),            /*!< the resolution of the encoder is 18 bits */
    EDOUT_RESOLUTION_20BIT               = ((uint64_t)1048576U)            /*!< the resolution of the encoder is 20 bits*/
} hal_edout_resolution_enum;

/* structure definition */
/* @PARA: p_edout_init */
/* @STRUCT: edout init structure */
typedef struct {
    uint32_t                             b_phase_pol;                      /*!< EDOUT B-phase working polarity */
    uint32_t                             max_location;                     /*!< EDOUT maximum location */
    uint32_t                             phase_difference;                 /*!< EDOUT phase difference between phase A and B */
    int16_t                              edge_count;                       /*!< EDOUT edge count between phase A and B */
    hal_edout_resolution_enum            resolution;                       /*!< EDOUT resolution of encoder location value*/
    uint64_t                             user_resolution;                  /*!< EDOUT user input encoder resolution */
    uint32_t                             current_location;                 /*!< EDOUT current Location */
    uint32_t                             z_phase_mode;                     /*!< EDOUT Z-Phase Output Mode */
    uint32_t                             z_phase_width;                    /*!< EDOUT Z-phase output width */
    uint32_t                             z_phase_start_position;           /*!< EDOUT z-phase output starting position */
} hal_edout_init_struct;

/* @PARA: edout_dev */
/* @STRUCT: edout device information structure */
typedef struct {
    hal_edout_state_enum                 state;                            /*!< EDOUT state */
    hal_mutex_enum                       mutex;                            /*!< mutex lock */
} hal_edout_dev_struct;

/* hal function declarations */
/* @FUNCTION: initialize EDOUT  */
int32_t hal_edout_init(hal_edout_dev_struct *edout_dev, hal_edout_init_struct *p_edout_init);
/* deinitialize EDOUT */
int32_t hal_edout_deinit(hal_edout_dev_struct *edout_dev);
/* initialize the EDOUT structure*/
int32_t hal_edout_struct_init(hal_edout_struct_type_enum hal_struct_type, void *p_struct);
/* @END */

/* update the output counter, used to set the phase difference and the number of edges for the next update period */
void hal_edout_output_counter_update(int16_t edge_count, uint16_t phase_difference);
/* get the current location value */
uint16_t hal_edout_current_location_get(void);
/* get the maximum location value */
uint16_t hal_edout_max_location_value_get(void);
/* configure Z-phase output mode */
void hal_edout_z_output_mode_config(uint32_t z_phase_mode);
/* configure Z-phase output start location and width */
void hal_edout_z_output_start_loc_and_width_config(uint32_t z_phase_start_position, uint32_t z_phase_width);
/* enable EDOUT */
void hal_edout_enable(void);
/* disable EDOUT */
void hal_edout_disable(void);

/* get EDOUT state */
hal_edout_state_enum hal_edout_state_get(hal_edout_dev_struct *edout_dev);

/* set B-phase active polarity */
void hals_edout_polarity_config(uint32_t b_phase_pol);
/* set the maximum location value for one rotation */
void hals_edout_max_location_value_config(uint32_t max_location);
/* set the current location value */
void hals_edout_current_location_config(uint32_t current_location);

#endif /* GD32H7XX_HAL_EDOUT_H */
